/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2014 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief         System Interface, PVR
 * @file          pvr.c
 * @date          January 2014
 * @author        Sergio Panseri
 */

//#define DEBUG

/*---includes for this file--------------------------------------------------*/

/* compiler library header files */
#include <stdio.h>
#include <string.h>

/* third party header files */

#include <techtype.h>
#include <dbgfuncs.h>
#include "stbhwos.h"
#include "stbhwdsk.h"

#include "stbheap.h"
#include "stbgc.h"
#include "stbpvr.h"
#include "stbvtc.h"
#include "ap_dbacc.h"
#include "ap_tmr.h"
#include "ap_pvr.h"

#include "hbbtv.h"
#include "hbbtv_sif_types.h"
#include "hbbtv_sif_service.h"
#include "hbbtv_sif_pvr.h"
#include "common.h"


/*---constant definitions for this file--------------------------------------*/

#ifdef DEBUG
#define DBG(string, vars ...) STB_SPDebugWrite("[%s:%d] "string, __FUNCTION__, __LINE__, ##vars)
#else
#define DBG(string, vars ...)
#endif


/*---local typedef structs for this file-------------------------------------*/

typedef union
{
   void *pointer;
   U32BIT u32bit;
} U_DATA;

typedef struct s_llist
{
   U_DATA data;
   struct s_llist *next;
} S_LLIST;

typedef struct
{
   U32BIT timer_handle;
   U32BIT rec_handle;
   U32BIT rec_id;
} S_REC;


/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local)                 */

static U32BIT recording_player_id = 0;
static S_LLIST *recording_list = NULL;
static void *rec_list_mutex = NULL;


/*---local function prototypes for this file---------------------------------*/
/*   (internal functions declared static to make them local)                 */

static U8BIT* CreateString(S_HBBTV_STRING *hbbtv_string);
static S_REC *CreateNewRecording(U32BIT timer, U32BIT recording);
static S_REC *FindRecByTimerHandle(U32BIT timer);
static S_REC *FindRecByRecordingHandle(U32BIT recording);
static S_REC *FindRecByRecordingID(U32BIT rec_id);
static BOOLEAN RetrieveSchedRecDetails(U32BIT timer_id, S_HBBTV_SCHEDULED_RECORDING_DETAILS *details);
static void ReleaseSchedRecDetails(S_HBBTV_SCHEDULED_RECORDING_DETAILS *details);
static void RetrieveRecordingDetails(S_REC *rec, BOOLEAN in_progress, S_HBBTV_RECORDING_DETAILS *details);
static void ReleaseRecordingDetails(S_HBBTV_RECORDING_DETAILS *details);


/*---local function definitions----------------------------------------------*/

static U8BIT* CreateString(S_HBBTV_STRING *hbbtv_string)
{
   U8BIT *string;

   FUNCTION_START(CreateString);

   string = STB_AppGetMemory(hbbtv_string->zlen + 1);
   if (string != NULL)
   {
      memcpy(string, hbbtv_string->zptr, hbbtv_string->zlen);
      string[hbbtv_string->zlen] = 0;
   }

   FUNCTION_FINISH(CreateString);

   return string;
}

/**
 * @brief   Creates a new entry in the recording lists and allocates a recording ID for the HbbTV
 *          engine. Either 'timer' or 'recording' can be 0 if they are not known at the time of
 *          creation. If a recording structure exist with either the specified timer handle or
 *          recording handler, this function returns it.
 * @param   timer Timer handle
 * @param   recording Recording handle
 * @return  Recording structure
 */
static S_REC *CreateNewRecording(U32BIT timer, U32BIT recording)
{
   S_LLIST *new_entry;
   S_REC *new_rec;

   FUNCTION_START(CreateNewRecording);

   STB_OSMutexLock(rec_list_mutex);

   /* Before creating a new recording structure, check if it's already present */
   if (timer != 0)
   {
      new_rec = FindRecByTimerHandle(timer);
   }
   else
   {
      new_rec = 0;
   }

   if (new_rec == NULL)
   {
      if (recording != 0)
      {
         new_rec = FindRecByRecordingHandle(recording);
      }
#ifdef DEBUG
      if (new_rec != NULL)
      {
         DBG("Found recording structure with recording=0x%x", recording);
      }
#endif
   }
#ifdef DEBUG
   else
   {
      DBG("Found recording structure with timer=0x%x", timer);
   }
#endif

   if (new_rec == NULL)
   {
      DBG("Creating a new recording structure with timer=0x%x and recording=0x%x", timer, recording);
      new_entry = STB_AppGetMemory(sizeof(S_LLIST));
      if (new_entry != NULL)
      {
         new_rec = STB_AppGetMemory(sizeof(S_REC));
         if (new_rec != NULL)
         {
            new_rec->timer_handle = timer;
            new_rec->rec_handle = recording;
            if (recording_list == NULL)
            {
               /* First recording */
               new_rec->rec_id = 1;
            }
            else
            {
               new_rec->rec_id = ((S_REC *)recording_list->data.pointer)->rec_id + 1;
            }

            new_entry->data.pointer = new_rec;
            new_entry->next = recording_list;
            recording_list = new_entry;
         }
#ifdef DEBUG
         else
         {
            DBG("Failed allocating %d bytes", sizeof(S_REC));
         }
#endif
      }
#ifdef DEBUG
      else
      {
         DBG("Failed allocating %d bytes", sizeof(S_LLIST));
      }
#endif
   }
   STB_OSMutexUnlock(rec_list_mutex);

   FUNCTION_FINISH(CreateNewRecording);

   return new_rec;
}

/**
 * @brief   Finds the recording structure associated with the specified timer handle
 * @param   timer Timer handle
 * @return  Recording structure
 */
static S_REC *FindRecByTimerHandle(U32BIT timer)
{
   S_LLIST *entry;
   S_REC *rec;

   FUNCTION_START(FindRecByTimerHandle);

   STB_OSMutexLock(rec_list_mutex);
   rec = NULL;
   entry = recording_list;
   while (entry != NULL)
   {
      rec = entry->data.pointer;
      if ((rec != NULL) && (rec->timer_handle == timer))
      {
         break;
      }
      else
      {
         rec = NULL;
      }

      entry = entry->next;
   }
   STB_OSMutexUnlock(rec_list_mutex);

   FUNCTION_FINISH(FindRecByTimerHandle);

   return rec;
}

/**
 * @brief   Finds the recording structure associated with the specified recording handle
 * @param   timer Recording handle
 * @return  Recording structure
 */
static S_REC *FindRecByRecordingHandle(U32BIT recording)
{
   S_LLIST *entry;
   S_REC *rec;

   FUNCTION_START(FindRecByRecordingHandle);

   STB_OSMutexLock(rec_list_mutex);
   rec = NULL;
   entry = recording_list;
   while (entry != NULL)
   {
      rec = entry->data.pointer;
      if ((rec != NULL) && (rec->rec_handle == recording))
      {
         break;
      }
      else
      {
         rec = NULL;
      }

      entry = entry->next;
   }
   STB_OSMutexUnlock(rec_list_mutex);

   FUNCTION_FINISH(FindRecByRecordingHandle);

   return rec;
}

/**
 * @brief   Finds the recording structure associated with the specified recording ID
 * @param   rec_id HbbTV engine recording ID
 * @return  Recording structure
 */
static S_REC *FindRecByRecordingID(U32BIT rec_id)
{
   S_LLIST *entry;
   S_REC *rec;

   FUNCTION_START(FindRecByRecordingID);

   STB_OSMutexLock(rec_list_mutex);
   rec = NULL;
   entry = recording_list;
   while (entry != NULL)
   {
      rec = entry->data.pointer;
      if ((rec != NULL) && (rec->rec_id == rec_id))
      {
         break;
      }
      else
      {
         rec = NULL;
      }

      entry = entry->next;
   }
   STB_OSMutexUnlock(rec_list_mutex);

   FUNCTION_FINISH(FindRecByRecordingID);

   return rec;
}

static BOOLEAN RetrieveSchedRecDetails(U32BIT timer_id, S_HBBTV_SCHEDULED_RECORDING_DETAILS *details)
{
   BOOLEAN retval;
   S_TIMER_INFO timer_info;
   U32DHMS duration;
   U16BIT onid, tsid, sid;
   void *service_ptr, *event_ptr;
   U8BIT *string;

   FUNCTION_START(RetrieveSchedRecDetails);

   retval = ATMR_GetTimerInfo(timer_id, &timer_info);
   if (retval)
   {
      details->timer_id = timer_id;
      details->description.zptr = NULL; /* For now not implemented */
      details->description.zlen = 0;

      duration = ATMR_GetDuration(details->timer_id);

      /* Cannot use STB_GCConvertToTimestamp for duration, because of different meaning
       * of U32DHMS type between date and duration */
      details->duration = DHMS_DAYS(duration) * 24;
      details->duration = (details->duration + DHMS_HOUR(duration)) * 60;
      details->duration = (details->duration + DHMS_MINS(duration)) * 60;
      details->duration += DHMS_SECS(duration);
      DBG("duration=%d", details->duration);

      onid = ATMR_GetOriginalNetworkId(details->timer_id);
      tsid = ATMR_GetTransportId(details->timer_id);
      sid = ATMR_GetServiceId(details->timer_id);
      service_ptr = ADB_FindServiceByIds(onid, tsid, sid);
      details->lcn = ADB_GetServiceLcn(service_ptr);
      details->start_time = STB_GCConvertToTimestamp(ATMR_GetStartDateTime(details->timer_id));

      details->start_padding = ATMR_GetStartPadding(details->timer_id);
      details->end_padding = ATMR_GetEndPadding(details->timer_id);

      string = ATMR_GetName(details->timer_id);
      if (string != NULL)
      {
         details->name.zlen = strlen((char *)string);
         details->name.zptr = STB_AppGetMemory(details->name.zlen + 1);
         strcpy((char *)details->name.zptr, (char *)string);
      }
      else
      {
         details->name.zlen = 0;
         details->name.zptr = NULL;
      }

      event_ptr = ADB_FindEventFromTime(details->start_time, service_ptr);
      details->parental_rating = ADB_GetEventParentalAge(event_ptr);
   }

   FUNCTION_FINISH(RetrieveSchedRecDetails);

   return retval;
}

static void ReleaseSchedRecDetails(S_HBBTV_SCHEDULED_RECORDING_DETAILS *details)
{
   FUNCTION_START(ReleaseSchedRecDetails);

   if (details->description.zptr != NULL)
   {
      STB_AppFreeMemory(details->description.zptr);
   }
   if (details->name.zptr != NULL)
   {
      STB_AppFreeMemory(details->name.zptr);
   }

   FUNCTION_FINISH(ReleaseSchedRecDetails);
}

static void RetrieveRecordingDetails(S_REC *rec, BOOLEAN in_progress, S_HBBTV_RECORDING_DETAILS *details)
{
   void *service_ptr;
   U16BIT serv_id, ts_id, orig_net_id;
   U8BIT *string;
   U32BIT size;
   U8BIT rec_hours, rec_mins, rec_secs;
   U32DHMS time;
   U16BIT rec_date;

   FUNCTION_START(RetrieveRecordingDetails);

   details->recording_id = rec->rec_id;
   details->in_progress = in_progress;

   if (STB_PVRRecordingGetTriplet(rec->rec_handle, &serv_id, &ts_id, &orig_net_id) == TRUE)
   {
      service_ptr = ADB_FindServiceByIds(orig_net_id, ts_id, serv_id);
   }
   else
   {
      service_ptr = NULL;
   }

   if (service_ptr != NULL)
   {
      details->lcn = ADB_GetServiceLcn(service_ptr);
   }

   details->start_padding = STB_PVRRecordingGetStartPadding(rec->rec_handle);
   details->end_padding = STB_PVRRecordingGetEndPadding(rec->rec_handle);
   details->parental_rating = STB_PVRRecordingGetParentalRating(rec->rec_handle);

   string = STB_PVRRecordingGetName(rec->rec_handle);
   if (string != NULL)
   {
      details->name.zlen = strlen((char *)string);
      details->name.zptr = STB_AppGetMemory(details->name.zlen + 1);
      strcpy((char *)details->name.zptr, (char *)string);
   }
   else
   {
      details->name.zlen = 0;
      details->name.zptr = NULL;
   }

   size = STB_PVRRecordingGetDescriptionLen(rec->rec_handle);
   if (size != 0)
   {
      string = STB_AppGetMemory(size);
      if (string != NULL)
      {
         if (!STB_PVRRecordingGetDescription(rec->rec_handle, string, size))
         {
            STB_AppFreeMemory(string);
            string = NULL;
         }
      }
      details->description.zlen = size - 1;
      details->description.zptr = string;
   }

   if (STB_PVRRecordingGetLength(rec->rec_handle, &rec_hours, &rec_mins, &rec_secs, &size) == TRUE)
   {
      details->recording_duration = rec_hours * 60;
      details->recording_duration = (details->recording_duration + rec_mins) * 60 + rec_secs;
   }

   if (STB_PVRRecordingGetDateTime(rec->rec_handle, &rec_date, &rec_hours, &rec_mins, &rec_secs) == TRUE)
   {
      time = DHMS_CREATE(rec_date, rec_hours, rec_mins, 0);
      details->recording_start_time = STB_GCConvertToTimestamp(time);
   }
   details->sched_duration = details->recording_duration;
   details->sched_start_time = details->recording_start_time;

   FUNCTION_FINISH(RetrieveRecordingDetails);
}

static void ReleaseRecordingDetails(S_HBBTV_RECORDING_DETAILS *details)
{
   FUNCTION_START(ReleaseRecordingDetails);

   if (details->description.zptr != NULL)
   {
      STB_AppFreeMemory(details->description.zptr);
   }
   if (details->name.zptr != NULL)
   {
      STB_AppFreeMemory(details->name.zptr);
   }

   FUNCTION_FINISH(ReleaseRecordingDetails);
}


/*---global function definitions---------------------------------------------*/

/**
 * @brief   Initialises the PVR resources needed by the HbbTV external interface
 */
void EXT_HbbtvPVRInitialise(void)
{
   FUNCTION_START(EXT_HbbtvPVRInitialise);
   rec_list_mutex = STB_OSCreateMutex();
   FUNCTION_FINISH(EXT_HbbtvPVRInitialise);
}

/**
 * @brief   Releases the PVR resources allocated by the HbbTV external interface
 */
void EXT_HbbtvPVRTerminate(void)
{
   S_LLIST *entry, *next;

   FUNCTION_START(EXT_HbbtvPVRTerminate);

   STB_OSMutexLock(rec_list_mutex);
   entry = recording_list;
   while (entry != NULL)
   {
      next = entry->next;
      if (entry->data.pointer != NULL)
      {
         STB_AppFreeMemory(entry->data.pointer);
      }
      STB_AppFreeMemory(entry);
      entry = next;
   }
   STB_OSMutexUnlock(rec_list_mutex);

   STB_OSDeleteMutex(rec_list_mutex);
   rec_list_mutex = NULL;

   FUNCTION_FINISH(EXT_HbbtvPVRTerminate);
}

/**
 * @brief   Schedules an event to be recorded
 * @param   crid Pointer to S_HBBTV_STRING containing the CRID for the event to be scheduled for
 *          recording. This function makes a copy of it as the engine does not guarantee the pointer
 *          will be valid after the function returns.
 * @param   fqdn Pointer to a utf-8 string containing the application Full Qualified Domain Name to
 *          be stored with the scheduled recording and the resulting recording. This function makes
 *          a copy of it as the engine does not guarantee the pointer will be valid after the
 *          function returns.
 * @param   service_lcn Pointer to a U16BIT representing the service LCN in the current channel list
 *          associated with the event to be recorded
 * @param   id Pointer to a U32BIT representing the scheduled recording unique identifier
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER if programCrid is invalid
 *          - HBBTV_ERR_OTHER if the event couldn't be scheduled for recording
 */
E_HBBTV_ERR HBBTV_PVRRecordCRID(S_HBBTV_STRING *crid, S_HBBTV_STRING *fqdn, U16BIT *service_lcn,
   U32BIT *id)
{
   E_HBBTV_ERR result;
   U8BIT *crid_str, *fqdn_str;
   void *service_ptr;
   void *event_ptr;
   U16BIT onid, tsid, sid, eid;

   FUNCTION_START(HBBTV_PVRRecordCRID);

   crid_str = NULL;
   fqdn_str = NULL;
   event_ptr = NULL;

   DBG("crid: %s, fqdn: %s", ((crid != NULL) && (crid->zptr != NULL)) ? (char *)crid->zptr : "NULL",
      ((fqdn != NULL) && (fqdn->zptr != NULL)) ? (char *)fqdn->zptr : "NULL");
   if ((crid != NULL) && (crid->zptr != NULL) && (fqdn != NULL) && (fqdn->zptr != NULL) &&
       (service_lcn != NULL) && (id != NULL))
   {
      crid_str = CreateString(crid);
      fqdn_str = CreateString(fqdn);
      if ((crid_str != NULL) && (fqdn_str != NULL))
      {
         event_ptr = ADB_FindEventFromCrid(crid_str, NULL, &service_ptr);
         if (event_ptr != NULL)
         {
            event_ptr = ATMR_RecordEvent(service_ptr, event_ptr, crid_str, NULL, FALSE, TRUE, FALSE);
            if (event_ptr != NULL)
            {
               /* Save the fqdn in the timer */
               ATMR_SetAdditionalInfo(*id, fqdn_str, fqdn->zlen + 1);
               ADB_SaveDatabase();

               *service_lcn = ADB_GetServiceLcn(service_ptr);
               ADB_GetServiceIds(service_ptr, &onid, &tsid, &sid);
               eid = ADB_GetEventId(event_ptr);

               *id = ATMR_FindTimerFromEvent(onid, tsid, sid, eid);
               CreateNewRecording(*id, 0);

               result = HBBTV_OK;
            }
         }
      }
      if (event_ptr == NULL)
      {
         result = HBBTV_ERR_OTHER;
      }
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   if (crid_str != NULL)
   {
      STB_AppFreeMemory(crid_str);
   }
   if (fqdn_str != NULL)
   {
      STB_AppFreeMemory(fqdn_str);
   }

   FUNCTION_FINISH(HBBTV_PVRRecordCRID);

   return result;
}

/**
 * @brief   Schedules an event to be recorded, the event is identified by its ID and the service
 *          LCN.
 * @param   event_id DVB event ID representing the event to be recorded.
 * @param   service_lcn service LCN in the current channel list associated with the event to be
 *          recorded.
 * @param   fqdn Pointer to a utf-8 string containing the application Full Qualified Domain Name to
 *          be stored with the scheduled recording and the resulting recording. This function makes
 *          a copy of it as the engine does not guarantee the pointer will be valid after the
 *          function returns.
 * @param   id Pointer to a U32BIT representing the scheduled recording unique identifier
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER if programCrid is invalid
 *          - HBBTV_ERR_OTHER if the event couldn't be scheduled for recording
 */
E_HBBTV_ERR HBBTV_PVRRecordEventID(U16BIT event_id, U16BIT service_lcn, S_HBBTV_STRING *fqdn,
   U32BIT *id)
{
   E_HBBTV_ERR result;
   U8BIT *fqdn_str;
   void *service_ptr;
   void *event_ptr;
   U16BIT onid, tsid, sid;

   FUNCTION_START(HBBTV_PVRRecordEventID);

   fqdn_str = NULL;
   event_ptr = NULL;

   DBG("event ID: 0x%x, LCN: %d. fqdn: %s", event_id, service_lcn,
      ((fqdn != NULL) && (fqdn->zptr != NULL)) ? (char *)fqdn->zptr : "NULL");
   if ((fqdn != NULL) && (fqdn->zptr != NULL) && (id != NULL))
   {
      fqdn_str = CreateString(fqdn);
      if (fqdn_str != NULL)
      {
         service_ptr = ADB_FindServiceByLcn(ADB_SERVICE_LIST_ALL, service_lcn, TRUE);
         event_ptr = ADB_GetEvent(service_ptr, event_id);
         if (event_ptr != NULL)
         {
            event_ptr = ATMR_RecordEvent(service_ptr, event_ptr, NULL, NULL, FALSE, FALSE, FALSE);
            if (event_ptr != NULL)
            {
               /* Save the fqdn in the timer */
               ATMR_SetAdditionalInfo(*id, fqdn_str, fqdn->zlen + 1);
               ADB_SaveDatabase();

               ADB_GetServiceIds(service_ptr, &onid, &tsid, &sid);

               *id = ATMR_FindTimerFromEvent(onid, tsid, sid, event_id);
               CreateNewRecording(*id, 0);

               result = HBBTV_OK;
            }
         }
#ifdef DEBUG
         else
         {
            DBG("Could not find event, serviceptr=%p, event ID=0x%x", service_ptr, event_id);
         }
#endif
      }
      if (event_ptr == NULL)
      {
         result = HBBTV_ERR_OTHER;
      }
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   if (fqdn_str != NULL)
   {
      STB_AppFreeMemory(fqdn_str);
   }

   FUNCTION_FINISH(HBBTV_PVRRecordEventID);

   return result;
}

/**
 * @brief   Schedules a recording based on start time and duration
 * @param   service_lcn LCN in the current channel list of the service to be recorded
 * @param   start_time Start of the time period of the recording measured in seconds since midnight
 *          (GMT) on 1/1/1970. If the start time occurs in the past and the current time is within
 *          the specified duration of the recording, the recording starts immediately.
 * @param   duration Duration of the recording in seconds.
 * @param   repeat_days Bitfield indicating which days of the week the recording should be repeated.
 *          The meaning of each bit is the following:
 *          - b0: Sunday
 *          - b1: Monday
 *          - b2: Tuesday
 *          - b3: Wednesday
 *          - b4: Thursday
 *          - b5: Friday
 *          - b6: Saturday
 * @param   fqdn Pointer to a utf-8 string containing the application Full Qualified Domain Name to
 *          be stored with the scheduled recording and the resulting recording. This function makes
 *          a copy of it as the engine does not guarantee the pointer will be valid after the
 *          function returns.
 * @param   details Pointer to the details of the event to be recorded. The engine will call
 *          HBBTV_ReleaseEventDetails to release allocated data on the same task and immediately
 *          after it has copied the data.
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_OTHER if the recording couldn't be scheduled
 */
E_HBBTV_ERR HBBTV_PVRRecordAt(U16BIT service_lcn, U32BIT start_time, U32BIT duration,
   U8BIT repeat_days, S_HBBTV_STRING *fqdn, S_HBBTV_EVENT_DETAILS *details, U32BIT *id)
{
   E_HBBTV_ERR result;
   U8BIT *fqdn_str;
   U8BIT hr, mn, sc;
   void *service_ptr;
   void *event_ptr;
   U32BIT timer;
   S_TIMER_INFO s_timer;
   U32DHMS start, end, now, dur;
   U8BIT max_recordings;
   U8BIT num_recordings;

   FUNCTION_START(HBBTV_PVRRecordAt);
   USE_UNWANTED_PARAM(repeat_days);

   fqdn_str = NULL;
   timer = INVALID_TIMER_HANDLE;
   DBG("lcn: %d, start: %d, duration: %d, fqdn: %s", service_lcn, start_time, duration,
      ((fqdn != NULL) && (fqdn->zptr != NULL)) ? (char *)fqdn->zptr : "NULL");
   if ((fqdn != NULL) && (fqdn->zptr != NULL) && (details != NULL) && (id != NULL))
   {
      fqdn_str = CreateString(fqdn);

      now = STB_GCNowDHMSGmt();
      start = STB_GCConvertTimestamp(start_time);
      if (start < now)
      {
         start = now;
      }

      mn = duration / 60;
      sc = duration % 60;
      hr = mn / 60;
      mn %= 60;
      dur = DHMS_CREATE(0, hr, mn, sc);

      end = STB_GCCalculateDHMS(start, dur, CALC_ADD);

      service_ptr = ADB_FindServiceByLcn(ADB_SERVICE_LIST_ALL, service_lcn, TRUE);
      if ((service_ptr != NULL) && (end > now))
      {
         event_ptr = ADB_FindEventFromTime(start, service_ptr);

         ATMR_InitialiseTimer(&s_timer, TIMER_TYPE_PVR_RECORD, service_ptr, event_ptr);

         s_timer.start_time = start;
         s_timer.u.record.duration = dur;
         s_timer.u.record.disk_id = STB_PVRGetDefaultDisk();
         s_timer.u.record.event_triggered= FALSE;

         max_recordings = STB_HWGetNumRecorders();
         num_recordings = ATMR_GetNumSimultaneousRecordings(max_recordings, start, end, NULL);

         if ((num_recordings < max_recordings) &&
             (APVR_CheckSpaceDuration(s_timer.u.record.disk_id, dur) == TRUE))
         {
            timer = ATMR_AddTimer(&s_timer);
            if (timer != INVALID_TIMER_HANDLE)
            {
               /* Save the fqdn in the timer */
               ATMR_SetAdditionalInfo(timer, fqdn->zptr, fqdn->zlen + 1);
               ADB_SaveDatabase();

               *id = timer;
               CreateNewRecording(*id, 0);

               EXT_HbbtvGetEvent(event_ptr, service_ptr, details);
               ADB_ReleaseEventData(event_ptr);

               result = HBBTV_OK;
            }
         }
#ifdef DEBUG
         else
         {
            DBG("Could not create timer, num_recordings: %d, max_recordings: %d", num_recordings,
               max_recordings);
         }
#endif
      }
#ifdef DEBUG
      else
      {
         DBG("service_ptr: %p, end: %s %s, now: %s %s", service_ptr,
            STB_GCGetDateStringDHMS(end, DATE_DMY), STB_GCGetTimeStringDHMS(end, TIME_24H),
            STB_GCGetDateStringDHMS(now, DATE_DMY), STB_GCGetTimeStringDHMS(now, TIME_24H));
      }
#endif
      if (timer == INVALID_TIMER_HANDLE)
      {
         result = HBBTV_ERR_OTHER;
      }
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   if (fqdn_str != NULL)
   {
      STB_AppFreeMemory(fqdn_str);
   }

   FUNCTION_FINISH(HBBTV_PVRRecordAt);

   return result;
}

/**
 * @brief   Returns the recording ID (as in S_HBBTV_RECORDING_DETAILS) for an in progress scheduled
 *          recording represented by scheduled_recording_id.
 * @param   scheduled_recording_id Scheduled recording unique identifier, as returned by
 *          HBBTV_PVRRecordAt, HBBTV_PVRRecordEventID, HBBTV_PVRRecordEventID or
 *          HBBTV_PVRRecordCRID.
 * @param   record_id Pointer to the returned recording ID.
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER otherwise
 */
E_HBBTV_ERR HBBTV_PVRGetRecordingId(U32BIT scheduled_recording_id, U32BIT *recording_id)
{
   E_HBBTV_ERR result;
   S_REC *rec;

   FUNCTION_START(HBBTV_PVRGetRecordingId);

   rec = FindRecByTimerHandle(scheduled_recording_id);
   *recording_id = rec->rec_id;
   if (*recording_id == 0)
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }
   else
   {
      result = HBBTV_OK;
   }

   FUNCTION_FINISH(HBBTV_PVRGetRecordingId);

   return result;
}

/**
 * @brief   Stops an in progress recording
 * @param   recording id recordig ID
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER otherwise
 */
E_HBBTV_ERR HBBTV_PVRStopRecording(U32BIT recording_id)
{
   E_HBBTV_ERR result;
   S_REC *rec;

   FUNCTION_START(HBBTV_PVRStopRecording);

   rec = FindRecByRecordingID(recording_id);
   if (APVR_StopRecording(rec->rec_handle))
   {
      result = HBBTV_OK;
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_PVRStopRecording);

   return result;
}

/**
 * @brief   Sets the name for the specified scheduled recording
 * @param   id Scheduled recording unique identifier as returned by HBBTV_PVRRecordCRID,
 *          HBBTV_PVRRecordEventID or HBBTV_PVRRecordAt.
 * @param   Name name to be associated with the specified scheduled recording
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_OTHER if the name cannot be set
 */
E_HBBTV_ERR HBBTV_PVRSetScheduledRecordingName(U32BIT id, S_HBBTV_STRING *name)
{
   E_HBBTV_ERR result;
   U8BIT *name_str;

   FUNCTION_START(HBBTV_PVRSetScheduledRecordingName);

   if ((name != NULL) && (name->zptr != NULL))
   {
      if (name->zptr[0] == 0x15)
      {
         /* String is already in DVB format */
         name_str = CreateString(name);
      }
      else
      {
         /* To let the DVB know it's a UTF8 string, 0x15 must be prepended */
         name_str = STB_AppGetMemory(name->zlen + 2); /* +1 for 0x15 and +1 for \0 */
         name_str[0] = 0x15;
         memcpy(&(name_str[1]), name->zptr, name->zlen);
         name_str[name->zlen + 1] = 0;
      }
      if (name_str != NULL)
      {
         ATMR_SetName(id, name_str);
         result = HBBTV_OK;
         STB_AppFreeMemory(name_str);
      }
      else
      {
         result = HBBTV_ERR_OTHER;
      }
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_PVRSetScheduledRecordingName);

   return result;
}

/**
 * @brief   Sets the description for the specified scheduled recording
 * @param   id Scheduled recording unique identifier as returned by HBBTV_PVRRecordCRID,
 *          HBBTV_PVRRecordEventID or HBBTV_PVRRecordAt.
 * @param   description Description to be associated with the specified scheduled recording
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_OTHER if the name cannot be set
 */
E_HBBTV_ERR HBBTV_PVRSetScheduledRecordingDescription(U32BIT id, S_HBBTV_STRING *description)
{
   E_HBBTV_ERR result;

   FUNCTION_START(HBBTV_PVRSetScheduledRecordingDescription);
   USE_UNWANTED_PARAM(id);

   if ((description != NULL) && (description->zptr != NULL))
   {
      /* Not implemented yet */
      result = HBBTV_ERR_OTHER;
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_PVRSetScheduledRecordingDescription);

   return result;
}

/**
 * @brief   Retrieves the details of the scheduled recording identified by 'id'. Provided
 *          HBBTV_OK is returned, the HbbTV engine will call
 *          HBBTV_PVRReleaseScheduledRecordingDetails to release allocated data on the same task
 *          immediately after it has copied the data.
 * @param   id Scheduled recording unique identifier as returned by HBBTV_PVRRecordCRID or
 *          HBBTV_PVRRecordAt.
 * @param   details Pointer where the details will be returned.
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_ALLOCATING_MEMORY
 *          - HBBTV_ERR_OTHER
 */
E_HBBTV_ERR HBBTV_PVRGetScheduledRecordingDetails(U32BIT id,
   S_HBBTV_SCHEDULED_RECORDING_DETAILS *details)
{
   E_HBBTV_ERR result;

   FUNCTION_START(HBBTV_PVRGetScheduledRecordingDetails);

   if (RetrieveSchedRecDetails(id, details))
   {
      result = HBBTV_OK;
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_PVRGetScheduledRecordingDetails);

   return result;
}

/**
 * @brief   Release data allocated by HBBTV_PVRGetScheduledRecordingDetails (e.g. String data).
 *          Always called immediately after the HbbTV engine has copied data.
 * @param   details Pointer to the S_HBBTV_SCHEDULED_RECORDING_DETAILS filled by
 *          HBBTV_PVRGetScheduledRecordingDetails.
 */
void HBBTV_PVRReleaseScheduledRecordingDetails(S_HBBTV_SCHEDULED_RECORDING_DETAILS *details)
{
   FUNCTION_START(HBBTV_PVRReleaseScheduledRecordingDetails);
   ReleaseSchedRecDetails(details);
   FUNCTION_FINISH(HBBTV_PVRReleaseScheduledRecordingDetails);
}

/**
 * @brief   Retrieves the list of recordingins scheduled with the given Full Qualified Domain Name.
 *          Provided HBBTV_OK is returned, the HbbTV engine will call
 *          HBBTV_PVRReleaseScheduledRecordingList to release allocated data on the same task
 *          immediately after it has copied the data.
 * @param   fqdn Pointer to a utf-8 string containing the application Full Qualified Domain Name to
 *          be stored with the scheduled recording and the resulting recording. This function makes
 *          a copy of it as the engine does not guarantee the pointer will be valid after the
 *          function returns.
 * @param   num_ptr Pointer to number of events (Not NULL)
 * @param   list_ptr Pointer to S_HBBTV_EVENT_DETAILS Pointer (Not NULL)
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_ALLOCATING_MEMORY
 *          - HBBTV_ERR_OTHER
 */
E_HBBTV_ERR HBBTV_PVRGetScheduledRecordingList(S_HBBTV_STRING *fqdn, U32BIT *num_ptr,
   S_HBBTV_SCHEDULED_RECORDING_DETAILS **list_ptr)
{
   E_HBBTV_ERR result;
   U32BIT *timer_list;
   U16BIT num, i;
   U8BIT *tmr_fqdn;
   U32BIT tmr_fqdn_size;
   S_LLIST *timer_ll = NULL;
   S_LLIST *timer_entry, *prev_timer;
   S_HBBTV_SCHEDULED_RECORDING_DETAILS *details;

   FUNCTION_START(HBBTV_PVRGetScheduledRecordingList);

   if ((fqdn != NULL) && (fqdn->zptr != NULL) && (num_ptr != NULL) && (list_ptr != NULL))
   {
      *num_ptr = 0;
      if (ATMR_GetTimerList(&timer_list, &num, TIMER_TYPE_PVR_RECORD, FALSE) == TRUE)
      {
         for (i = 0; i < num; i++)
         {
            tmr_fqdn = ATMR_GetAdditionalInfo(timer_list[i], &tmr_fqdn_size);
            if (tmr_fqdn != NULL)
            {
               if ((tmr_fqdn_size == (fqdn->zlen + 1)) && (memcmp(tmr_fqdn, fqdn->zptr, fqdn->zlen) == 0))
               {
                  (*num_ptr)++;
                  timer_entry = STB_AppGetMemory(sizeof(S_LLIST));
                  timer_entry->next = timer_ll;
                  timer_entry->data.u32bit = timer_list[i];
                  timer_ll = timer_entry;
               }
               STB_AppFreeMemory(tmr_fqdn);
            }
         }

         *list_ptr = STB_AppGetMemory(*num_ptr * sizeof(S_HBBTV_SCHEDULED_RECORDING_DETAILS));
         if (*list_ptr != NULL)
         {
            details = *list_ptr;
            timer_entry = timer_ll;
            while (timer_entry != NULL)
            {
               RetrieveSchedRecDetails(timer_entry->data.u32bit, details);
               prev_timer = timer_entry;
               timer_entry = timer_entry->next;
               details++;

               STB_AppFreeMemory(prev_timer);
            }
         }

         ATMR_ReleaseTimerList(timer_list, num);
      }

      if ((*list_ptr != NULL) || (*num_ptr == 0))
      {
         result = HBBTV_OK;
      }
      else
      {
         *num_ptr = 0;
         result = HBBTV_ERR_OTHER;
      }
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_PVRGetScheduledRecordingList);

   return result;
}

/**
 * @brief   Release data allocated by HBBTV_PVRGetScheduledRecordingList (e.g. String data). Always
 *          called immediately after the HbbTV engine has copied data.
 * @param   num number of elements in the list
 * @param   list_ptr Pointer to S_HBBTV_SCHEDULED_RECORDING_DETAILS list
 */
void HBBTV_PVRReleaseScheduledRecordingList(U32BIT num, S_HBBTV_SCHEDULED_RECORDING_DETAILS *list)
{
   U32BIT i;
   S_HBBTV_SCHEDULED_RECORDING_DETAILS *rec;

   FUNCTION_START(HBBTV_PVRReleaseScheduledRecordingList);

   rec = list;
   for (i = 0; i < num; i++, rec++)
   {
      ReleaseSchedRecDetails(rec);
   }
   STB_AppFreeMemory(list);

   FUNCTION_FINISH(HBBTV_PVRReleaseScheduledRecordingList);
}

/**
 * @brief   Removes a scheduled recording created with HBBTV_PVRRecordCRID, HBBTV_PVRRecordEventID
 *          or HBBTV_PVRRecordAt.
 * @param   id Scheduled recording unique identifier as returned by HBBTV_PVRRecordCRID,
 *          HBBTV_PVRRecordEventID or HBBTV_PVRRecordAt.
 * @param   fqdn Pointer to a utf-8 string containing the application Full Qualified Domain Name to
 *          be stored with the scheduled recording and the resulting recording. This function makes
 *          a copy of it as the engine does not guarantee the pointer will be valid after the
 *          function returns.
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_OTHER
 */
E_HBBTV_ERR HBBTV_PVRRemoveScheduledRecording(U32BIT id, S_HBBTV_STRING *fqdn)
{
   E_HBBTV_ERR result;
   U8BIT *tmr_fqdn;
   U32BIT tmr_fqdn_size;

   FUNCTION_START(HBBTV_PVRRemoveScheduledRecording);

   if ((fqdn != NULL) && (fqdn->zptr != NULL))
   {
      result = HBBTV_ERR_OTHER;
      tmr_fqdn = ATMR_GetAdditionalInfo(id, &tmr_fqdn_size);
      if (tmr_fqdn != NULL)
      {
         if ((tmr_fqdn_size == fqdn->zlen + 1) && (memcmp(tmr_fqdn, fqdn->zptr, fqdn->zlen) == 0))
         {
            if (ATMR_DeleteTimer(id))
            {
               ADB_SaveDatabase();
               result = HBBTV_OK;
            }
         }

         STB_AppFreeMemory(tmr_fqdn);
      }
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_PVRRemoveScheduledRecording);

   return result;
}

/**
 * @brief   Changes the start padding value for the specified scheduled recording
 * @param   id Scheduled recording unique identifier as returned by HBBTV_PVRRecordCRID,
 *          HBBTV_PVRRecordEventID or HBBTV_PVRRecordAt.
 * @param   start_padding new value of start padding
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_OTHER
 */
E_HBBTV_ERR HBBTV_PVRSetScheduledRecordingStartPadding(U32BIT id, S32BIT start_padding)
{
   E_HBBTV_ERR result;

   FUNCTION_START(HBBTV_PVRSetScheduledRecordingStartPadding);

   ATMR_SetStartPadding(id, start_padding);
   result = HBBTV_OK;

   FUNCTION_FINISH(HBBTV_PVRSetScheduledRecordingStartPadding);

   return result;
}

/**
 * @brief   Changes the end padding value for the specified scheduled recording
 * @param   id Scheduled recording unique identifier as returned by HBBTV_PVRRecordCRID,
 *          HBBTV_PVRRecordEventID or HBBTV_PVRRecordAt.
 * @param   end_padding new value of end padding
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_OTHER
 */
E_HBBTV_ERR HBBTV_PVRSetScheduledRecordingEndPadding(U32BIT id, S32BIT end_padding)
{
   E_HBBTV_ERR result;

   FUNCTION_START(HBBTV_PVRSetScheduledRecordingEndPadding);

   ATMR_SetEndPadding(id, end_padding);
   result = HBBTV_OK;

   FUNCTION_FINISH(HBBTV_PVRSetScheduledRecordingEndPadding);

   return result;
}

/**
 * @brief   Retrieves the details of the recording identified by 'id'. Provided  HBBTV_OK is
 *          returned, the HbbTV engine will call HBBTV_PVRReleaseRecordingDetails to release
 *          allocated data on the same task immediately after it has copied the data.
 * @param   id Recording unique identifier.
 * @param   details Pointer where the details will be returned.
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_ALLOCATING_MEMORY
 *          - HBBTV_ERR_OTHER
 */
E_HBBTV_ERR HBBTV_PVRGetRecordingDetails(U32BIT id, S_HBBTV_RECORDING_DETAILS *details)
{
   E_HBBTV_ERR result;
   S_REC *rec;

   FUNCTION_START(HBBTV_PVRGetRecordingDetails);

   rec = FindRecByRecordingID(id);
   if ((rec != NULL) && STB_PVRIsValidHandle(rec->rec_handle))
   {
      RetrieveRecordingDetails(rec, STB_PVRIsBeingRecorded(rec->rec_handle), details);
      result = HBBTV_OK;
   }
   else
   {
      DBG("Could not find recording with id 0x%x", id);
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_PVRGetRecordingDetails);

   return result;
}

/**
 * @brief   Release data allocated by HBBTV_PVRGetRecordingDetails (e.g. String data).
 *          Always called immediately after the HbbTV engine has copied data.
 * @param   details Pointer to the S_HBBTV_RECORDING_DETAILS filled by HBBTV_PVRGetRecordingDetails.
 */
void HBBTV_PVRReleaseRecordingDetails(S_HBBTV_RECORDING_DETAILS *details)
{
   FUNCTION_START(HBBTV_PVRReleaseRecordingDetails);

   ReleaseRecordingDetails(details);

   FUNCTION_FINISH(HBBTV_PVRReleaseRecordingDetails);
}

/**
 * @brief   Retrieves the list of recordings completed or in progress with the given Fully Qualified
 *          Domain Name. Provided HBBTV_OK is returned, the HbbTV engine will call
 *          HBBTV_PVRReleaseScheduledRecordingList to release allocated data on the same task
 *          immediately after it has copied the data.
 * @param   fqdn Pointer to a utf-8 string containing the application Full Qualified Domain Name to
 *          be stored with the scheduled recording and the resulting recording. This function makes
 *          a copy of it as the engine does not guarantee the pointer will be valid after the
 *          function returns.
 * @param   num_ptr Pointer to number of recordings (Not NULL)
 * @param   list_ptr Pointer to S_HBBTV_RECORDING_DETAILS (Not NULL)
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_ALLOCATING_MEMORY
 *          - HBBTV_ERR_OTHER
 */
E_HBBTV_ERR HBBTV_PVRGetRecordingList(S_HBBTV_STRING *fqdn, U32BIT *num_ptr,
   S_HBBTV_RECORDING_DETAILS **list_ptr)
{
   E_HBBTV_ERR result;
   U8BIT **name_list;
   U32BIT *list, *status_list, *locked_list, *selected_list, *split_list;
   U16BIT num, i;
   U8BIT *rec_fqdn;
   U32BIT size;
   S_LLIST *rec_ll = NULL;
   S_LLIST *entry, *prev_entry;
   S_HBBTV_RECORDING_DETAILS *details;
   S_REC *rec;

   FUNCTION_START(HBBTV_PVRGetRecordingList);

   if ((fqdn != NULL) && (fqdn->zptr != NULL) && (num_ptr != NULL) && (list_ptr != NULL))
   {
      DBG("fqdn: %.*s", fqdn->zlen, fqdn->zptr);
      *num_ptr = 0;
      num = APVR_GetPlayList(&list, &name_list, &status_list, &locked_list, &selected_list, &split_list);
      DBG("APVR_GetPlayList returned %d recording(s)", num);
      if (num > 0)
      {
         for (i = 0; i < num; i++)
         {
            rec_fqdn = STB_PVRRecordingGetAdditionalInfo(list[i]);
            if (rec_fqdn != NULL)
            {
               DBG("recording %d, fqdn: %s", i, rec_fqdn);
               size = strlen((char *)rec_fqdn);
               if ((size == fqdn->zlen) && (memcmp(rec_fqdn, fqdn->zptr, fqdn->zlen) == 0))
               {
                  (*num_ptr)++;
                  entry = STB_AppGetMemory(sizeof(S_LLIST));
                  entry->next = rec_ll;
                  entry->data.u32bit = i;
                  rec_ll = entry;
               }
            }
         }

         *list_ptr = STB_AppGetMemory(*num_ptr * sizeof(S_HBBTV_RECORDING_DETAILS));
         if (*list_ptr != NULL)
         {
            details = *list_ptr;
            entry = rec_ll;
            while (entry != NULL)
            {
               i = entry->data.u32bit;
               rec = CreateNewRecording(0, list[i]);
               RetrieveRecordingDetails(rec, (status_list[i] == 1), details);
               prev_entry = entry;
               entry = entry->next;
               details++;

               STB_AppFreeMemory(prev_entry);
            }
         }

         APVR_ReleasePlayList(list, name_list, status_list, locked_list, selected_list, split_list);
      }
      else
      {
         *list_ptr = NULL;
      }

      if ((*list_ptr != NULL) || (*num_ptr == 0))
      {
         result = HBBTV_OK;
      }
      else
      {
         *num_ptr = 0;
         result = HBBTV_ERR_OTHER;
      }
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_PVRGetRecordingList);

   return result;
}

/**
 * @brief   Release data allocated by HBBTV_PVRGetRecordingList (e.g. String data). Always called
 *          immediately after the HbbTV engine has copied data.
 * @param   num number of elements in the list
 * @param   list_ptr Pointer to S_HBBTV_RECORDING_DETAILS list
 */
void HBBTV_PVRReleaseRecordingList(U32BIT num, S_HBBTV_RECORDING_DETAILS *list)
{
   U32BIT i;
   S_HBBTV_RECORDING_DETAILS *rec;

   FUNCTION_START(HBBTV_PVRReleaseRecordingList);

   rec = list;
   for (i = 0; i < num; i++, rec++)
   {
      ReleaseRecordingDetails(rec);
   }
   STB_AppFreeMemory(list);

   FUNCTION_FINISH(HBBTV_PVRReleaseRecordingList);
}

/**
 * @brief   Initialises an instance of PVR player for the specified recording id and returns its
 *          handle.
 * @param   recording_id Recording id as in S_HBBTV_RECORDING_DETAILS structure
 * @param   handle Handle
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_ALLOCATING_MEMORY
 *          - HBBTV_ERR_OTHER
 */
E_HBBTV_ERR HBBTV_PVRPlayInit(U32BIT recording_id, void **handle)
{
   E_HBBTV_ERR result;
   S_REC *rec;

   FUNCTION_START(HBBTV_PVRPlayInit);

   if (handle != NULL)
   {
      rec = FindRecByRecordingID(recording_id);
      result = HBBTV_OK;
      recording_player_id = rec->rec_handle;
      *handle = &recording_player_id;
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_PVRPlayInit);

   return result;
}

/**
 * @brief   Releases the specified instance of PVR playe
 * @param   handle Handle
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_OTHER
 */
E_HBBTV_ERR HBBTV_PVRPlayExit(void *handle)
{
   E_HBBTV_ERR result;

   FUNCTION_START(HBBTV_PVRPlayExit);

   if (handle != NULL)
   {
      recording_player_id = 0;
      result = HBBTV_OK;
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_PVRPlayExit);

   return result;
}

/**
 * @brief   Starts the presentation of the specified PVR player
 * @param   handle Handle
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_ALLOCATING_MEMORY
 *          - HBBTV_ERR_OTHER
 */
E_HBBTV_ERR HBBTV_PVRPlayStart(void *handle)
{
   E_HBBTV_ERR result;

   FUNCTION_START(HBBTV_PVRPlayStart);

   if (handle != NULL)
   {
      if (APVR_PlayRecording(recording_player_id, FALSE, NULL) == TRUE)
      {
         result = HBBTV_OK;
      }
      else
      {
         result = HBBTV_ERR_OTHER;
      }
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_PVRPlayStart);

   return result;
}

/**
 * @brief   Stops the presentation of the specified PVR player
 * @param   handle Handle
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_OTHER
 */
E_HBBTV_ERR HBBTV_PVRPlayStop(void *handle)
{
   E_HBBTV_ERR result;

   FUNCTION_START(HBBTV_PVRPlayStop);

   if (handle != NULL)
   {
      APVR_StopPlay(FALSE);
      result = HBBTV_OK;
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_PVRPlayStop);

   return result;
}

/**
 * @brief   Pauses the PVR player
 * @param   handle Handle
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_OTHER
 */
E_HBBTV_ERR HBBTV_PVRPlayPause(void *handle)
{
   E_HBBTV_ERR result;

   FUNCTION_START(HBBTV_PVRPlayPause);

   if (handle != NULL)
   {
      APVR_PausePlay();
      result = HBBTV_OK;
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_PVRPlayPause);

   return result;
}

/**
 * @brief   Resumes the PVR player
 * @param   handle Handle
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_OTHER
 */
E_HBBTV_ERR HBBTV_PVRPlayResume(void *handle)
{
   E_HBBTV_ERR result;

   FUNCTION_START(HBBTV_PVRPlayResume);

   if (handle != NULL)
   {
      APVR_NormalPlay();
      result = HBBTV_OK;
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_PVRPlayResume);

   return result;
}

/**
 * @brief   Changes the position of the PVR player
 * @param   handle Handle
 * @param   position position in in milliseconds in the recording
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_OTHER
 */
E_HBBTV_ERR HBBTV_PVRPlaySeek(void *handle, U32BIT position)
{
   E_HBBTV_ERR result;

   FUNCTION_START(HBBTV_PVRPlaySeek);
   USE_UNWANTED_PARAM(position);

   if (handle != NULL)
   {
      result = HBBTV_OK;
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_PVRPlaySeek);

   return result;
}

/**
 * @brief   Returns the current position in the recording
 * @param   handle Handle
 * @param   position Current position in in milliseconds in the recording
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_OTHER
 */
E_HBBTV_ERR HBBTV_PVRPlayGetPosition(void *handle, U32BIT *position)
{
   E_HBBTV_ERR result;
   U8BIT h, m, s, percent;

   FUNCTION_START(HBBTV_PVRPlayGetPosition);

   if ((handle != NULL) && (position != NULL))
   {
      if (APVR_GetPlaybackElapsedTime(recording_player_id, &h, &m, &s, &percent) == TRUE)
      {
         *position = h * 60 + m;
         *position = *position * 60 + s;
         *position *= 1000;
         result = HBBTV_OK;
      }
      else
      {
         result = HBBTV_ERR_OTHER;
      }
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_PVRPlayGetPosition);

   return result;
}

/**
 * @brief   Returns the length of the recording
 * @param   handle Handle
 * @param   length Length of the recording in milliseconds
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_OTHER
 */
E_HBBTV_ERR HBBTV_PVRPlayGetLength(void *handle, U32BIT *length)
{
   E_HBBTV_ERR result;
   U8BIT h, m, s;
   U32BIT kb;

   FUNCTION_START(HBBTV_PVRPlayGetLength);

   if ((handle != NULL) && (length != NULL))
   {
      if (STB_PVRRecordingGetLength(recording_player_id, &h, &m, &s, &kb) == TRUE)
      {
         result = HBBTV_OK;
         *length = h * 60 + m;
         *length = *length * 60 + s;
         *length *= 1000;
      }
      else
      {
         result = HBBTV_ERR_OTHER;
      }
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_PVRPlayGetLength);

   return result;
}

/**
 * @brief   Retrieves the list of available components in the recording. Provided HBBTV_OK is
 *          returned, the HbbTV engine will call HBBTV_PVRPlayReleaseComponentList to release
 *          allocated data on the same task immediately after the HbbTV engine has copied the data.
 * @param   handle Handle
 * @param   type Required component type
 * @param   num_ptr Pointer to number of components (Not NULL)
 * @param   list_ptr Pointer to S_HBBTV_COMPONENT_DETAILS Pointer (Not NULL)
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_ALLOCATING_MEMORY
 *          - HBBTV_ERR_OTHER
 */
E_HBBTV_ERR HBBTV_PVRPlayGetComponentList(void *handle, E_HBBTV_COMPONENT_TYPE type,
   U32BIT *num_ptr, S_HBBTV_COMPONENT_DETAILS **list_ptr)
{
   E_HBBTV_ERR result;

   FUNCTION_START(HBBTV_PVRPlayGetComponentList);
   USE_UNWANTED_PARAM(type);

   if ((handle != NULL) && (num_ptr != NULL) && (list_ptr != NULL))
   {
      result = HBBTV_OK;
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_PVRPlayGetComponentList);

   return result;
}

/**
 * @brief   Release data allocated by HBBTV_PVRPlayGetComponentList. Always called immediately after
 *          the HbbTV engine has copied data.
 * @param   recording_id Recording id as in S_HBBTV_RECORDING_DETAILS structure
 * @param   handle Handle
 * @param   list_ptr Pointer to S_HBBTV_COMPONENT_DETAILS
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 */
void HBBTV_PVRPlayReleaseComponentList(void *handle, S_HBBTV_COMPONENT_DETAILS *list_ptr)
{
   FUNCTION_START(HBBTV_PVRPlayReleaseComponentList);
   USE_UNWANTED_PARAM(handle);
   USE_UNWANTED_PARAM(list_ptr);
   FUNCTION_FINISH(HBBTV_PVRPlayReleaseComponentList);
}

/**
 * @brief   Replace any component of the same type with the specified one. This function can be
 *          called before or after the media player has been started.
 * @param   handle Handle
 * @param   component Pointer to the component to be rendered
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_OTHER
 */
E_HBBTV_ERR HBBTV_PVRPlaySelectComponent(void *handle, S_HBBTV_COMPONENT_DETAILS *component)
{
   E_HBBTV_ERR result;

   FUNCTION_START(HBBTV_PVRPlaySelectComponent);

   if ((handle != NULL) && (component != NULL))
   {
      result = HBBTV_OK;
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_PVRPlaySelectComponent);

   return result;
}

/**
 * @brief   Stops rendering the specified component.
 * @param   handle Handle
 * @param   component Pointer to the component to be stopped
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_OTHER
 */
E_HBBTV_ERR HBBTV_PVRPlayUnselectComponent(void *handle, S_HBBTV_COMPONENT_DETAILS *component)
{
   E_HBBTV_ERR result;

   FUNCTION_START(HBBTV_PVRPlayUnselectComponent);

   if ((handle != NULL) && (component != NULL))
   {
      result = HBBTV_OK;
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_PVRPlayUnselectComponent);

   return result;
}

/**
 * @brief   Changes position and size of the video presentation window
 * @param   handle Handle
 * @param   rect Pointer to the rectangle defining the presentation window. Values are in HbbTV
 *          1280x720 coordinates.
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_OTHER
 */
E_HBBTV_ERR HBBTV_PVRPlaySetRectangle(void *handle, S_HBBTV_RECT *rect)
{
   E_HBBTV_ERR result;
   S_RECTANGLE output;

   FUNCTION_START(HBBTV_PVRPlaySetRectangle);

   if ((handle != NULL) && (rect != NULL))
   {
      result = HBBTV_OK;
      output.left = rect->left;
      output.top = rect->top;
      output.width = rect->width;
      output.height = rect->height;

      STB_VTSetHbbtvVideoWindow(&output);
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_PVRPlaySetRectangle);

   return result;
}

/**
 * @brief   Retrieves the error code from the PVR player
 * @param   handle Handle
 * @return  E_HBBTV_RECPLAYER_ERROR error code
 */
E_HBBTV_RECPLAYER_ERROR HBBTV_PVRPlayGetError(void *handle)
{
   E_HBBTV_RECPLAYER_ERROR result;

   FUNCTION_START(HBBTV_PVRPlayGetError);
   USE_UNWANTED_PARAM(handle);

   result = HBBTV_RECPLAYER_UNDEFINED;

   FUNCTION_FINISH(HBBTV_PVRPlayGetError);

   return result;
}

/**
 * @brief   Called by the HbbTV engine when a recording starts so the recording handle can be saved
 * @param   timer timer handle associated with the scheduled recording
 */
void HBBTV_PVRRecordingStarted(U32BIT timer)
{
   S_REC *rec;

   rec = FindRecByTimerHandle(timer);
   if (rec != NULL)
   {
      rec->rec_handle = ATMR_GetRecordingHandle(timer);;
   }
   else
   {
      DBG("Could not find recording structure for timer=0x%x", timer);
   }
}

/**
 * @brief   Returns the recording ID associated with the specified timer handle and recording handle
 * @param   timer Timer handle
 * @param   recording Recording handle
 * @return  Recording ID
 */
U32BIT EXT_HbbtvPVRGetRecID(U32BIT timer, U32BIT recording)
{
   U32BIT rec_id;
   S_REC *rec;

   FUNCTION_START(EXT_HbbtvPVRGetRecID);

   rec = CreateNewRecording(timer, recording);
   if (rec != NULL)
   {
      rec_id = rec->rec_id;
   }
   else
   {
      DBG("CreateNewRecording returned NULL");
      rec_id = 0;
   }

   FUNCTION_FINISH(EXT_HbbtvPVRGetRecID);

   return rec_id;
}
