/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2014 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief         System Interface, Service
 * @file          service.c
 * @date          January 2014
 * @author        Sergio Panseri
 */

//#define DEBUG

/*---includes for this file--------------------------------------------------*/

/* compiler library header files */
#include <stdio.h>
#include <string.h>

/* third party header files */
#include <techtype.h>
#include <dbgfuncs.h>

#include "stbhwos.h"
#include "stbheap.h"
#include "stbuni.h"
#include "stbvtc.h"
#include "stbllist.h"
#include "stbpvr.h"
#include "ap_dbacc.h"
#include "ap_cntrl.h"
#include "ap_tmr.h"

#include "hbbtv_sif_types.h"
#include "hbbtv_sif_service.h"
#include "common.h"


/*---constant definitions for this file--------------------------------------*/
#ifdef DEBUG
#define DBG(string, vars ...) STB_SPDebugWrite("[%s:%d] "string, __FUNCTION__, __LINE__, ##vars)
#else
#define DBG(...)
#endif

/*---local typedef structs for this file-------------------------------------*/
typedef struct
{
   LINK_LIST_PTR_BLK ptrs;
   U8BIT lang_code[3];
} S_LANG_CODE;


/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local)                 */
CREATE_LINK_LIST_HEADER(audio_lang_list);
static void *service_mutex = NULL;


/*---local function prototypes for this file---------------------------------*/
/*   (internal functions declared static to make them local)                 */

static U16BIT GetServiceNID(void *service);


/*---local function definitions----------------------------------------------*/

static U16BIT GetServiceNID(void *service)
{
   U16BIT nid = 0;
   void *transport, *network;

   network = NULL;
   transport = ADB_GetServiceTransportPtr(service);
   if (transport != NULL)
   {
      network = ADB_GetTransportNetworkPtr(transport);
      if (network != NULL)
      {
         nid = ADB_GetNetworkId(network);
      }
#ifdef DEBUG
      else
      {
         DBG("Could not find network");
      }
#endif
   }
#ifdef DEBUG
   else
   {
      DBG("Could not find transport");
   }
#endif

   return nid;
}

static void AddLangCode(U8BIT *code)
{
   S_LANG_CODE *entry;

   FUNCTION_START(AddLangCode);

   entry = STB_AppGetMemory(sizeof(S_LANG_CODE));
   if (entry != NULL)
   {
      memcpy(entry->lang_code, code, 3);
      STB_LLAddBlockToEnd(&audio_lang_list, (LINK_LIST_PTR_BLK *)entry);
   }

   FUNCTION_FINISH(AddLangCode);
}

static BOOLEAN IsLangCodePresent(U8BIT *code)
{
   S_LANG_CODE *entry;
   BOOLEAN found;

   FUNCTION_START(IsLangCodePresent);

   found = FALSE;
   entry = (S_LANG_CODE *)STB_LLGetFirstBlock(&audio_lang_list);
   while (entry != NULL)
   {
      if (memcmp((char *)entry->lang_code, code, 3) == 0)
      {
         found = TRUE;
         break;
      }
      entry = (S_LANG_CODE *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)entry);
   }

   FUNCTION_FINISH(IsLangCodePresent);

   return found;
}

static void EmptyLangCodeList(void)
{
   S_LANG_CODE *entry, *next;

   FUNCTION_START(EmptyLangCodeList);

   entry = (S_LANG_CODE *)STB_LLGetFirstBlock(&audio_lang_list);
   while (entry != NULL)
   {
      next = (S_LANG_CODE *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)entry);
      STB_LLRemoveBlock((LINK_LIST_PTR_BLK *)entry);
      STB_AppFreeMemory(entry);
      entry = next;
   }

   FUNCTION_FINISH(EmptyLangCodeList);
}

static BOOLEAN IsLangCodeValid(U8BIT *code)
{
   BOOLEAN valid;

   FUNCTION_START(IsLangCodeValid);

   if (((code[0] >= 0x61) && (code[0] <= 0x7a) &&
      (code[1] >= 0x61) && (code[1] <= 0x7a) &&
      (code[2] >= 0x61) && (code[2] <= 0x7a)) ||
      ((code[0] >= 0x41) && (code[0] <= 0x5a) &&
      (code[1] >= 0x41) && (code[1] <= 0x5a) &&
      (code[2] >= 0x41) && (code[2] <= 0x5a)))
   {
      valid = TRUE;
   }
   else
   {
      STB_SPDebugWrite("code %c%c%c (0x%x 0x%x 0x%x) is not valid", code[0], code[1], code[2], 
         code[0], code[1], code[2]);
      valid = FALSE;
   }

   FUNCTION_FINISH(IsLangCodeValid);

   return valid;
}

static void ReleaseEventDetails(S_HBBTV_EVENT_DETAILS *details)
{
   U8BIT i;
   S_HBBTV_STRING *string;

   FUNCTION_START(ReleaseEventDetails);

   if (details->event_name.zptr != NULL)
   {
      STB_ReleaseUnicodeString(details->event_name.zptr);
   }

   if (details->short_description.zptr != NULL)
   {
      STB_ReleaseUnicodeString(details->short_description.zptr);
   }

   if (details->long_description.zptr != NULL)
   {
      STB_ReleaseUnicodeString(details->long_description.zptr);
   }

   if (details->crid.zptr != NULL)
   {
      STB_AppFreeMemory(details->crid.zptr);
   }

   if (details->audio_language_array != NULL)
   {
      for (i = 0; i < details->num_audio_languages; i++)
      {
         string = &(details->audio_language_array[i]);
         if (string->zptr != NULL)
         {
            STB_AppFreeMemory(string->zptr);
         }
      }
      STB_AppFreeMemory(details->audio_language_array);
   }

   FUNCTION_FINISH(ReleaseEventDetails);
}

/*---global function definitions---------------------------------------------*/

/**
 * @brief   Retrieve demux resource path reference
 * @return  U8BIT demux resource reference
 */
U8BIT HBBTV_GetDemuxPath(void)
{
   U8BIT path;
   U8BIT dmxref;
   FUNCTION_START(HBBTV_GetDemuxPath);
   path = STB_DPGetLivePath();
   if (path == INVALID_RES_ID)
   {
      dmxref = INVALID_RES_ID;
   }
   else
   {
      dmxref = STB_DPGetPathDemux(path);
   }
   FUNCTION_FINISH(HBBTV_GetDemuxPath);
   return dmxref;
}

/**
 * @brief   Retrieves information about the present or following event on a service.
 * @param   service_index Receiver dependent identifier for the required service
 * @param   porf Indicates whether the present or following event should be retrieved, TRUE = present,
 *          FALSE = following
 * @param   details Pointer to S_HBBTV_EVENT_DETAILS structure to place required event information
 *          (this will NOT be NULL)
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER if service_index is invalid
 *          - HBBTV_ERR_OTHER otherwise.
 */
E_HBBTV_ERR HBBTV_GetEventDetails(S32BIT service_index, BOOLEAN porf,
   S_HBBTV_EVENT_DETAILS *details)
{
   E_HBBTV_ERR result;
   void *service, *event;

   FUNCTION_START(HBBTV_GetEventDetails);

   service = ADB_FindServiceByLcn(ADB_SERVICE_LIST_ALL, service_index, TRUE);
   if (service != NULL)
   {
      if (porf)
      {
         ADB_GetNowNextEvents(service, &event, NULL);
      }
      else
      {
         ADB_GetNowNextEvents(service, NULL, &event);
      }

      if (event != NULL)
      {
         result = HBBTV_OK;

         EXT_HbbtvGetEvent(event, service, details);

         /* No need to call ADB_ReleaseEventData as we do ADB_ReleaseEventData below */
         details->dvb_user_data = NULL;

         ADB_ReleaseEventData(event);
      }
      else
      {
         result = HBBTV_ERR_OTHER;
      }
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_GetEventDetails);

   return result;
}

/**
 * @brief   Releases data allocated by HBBTV_GetEventDetails (e.g. String data).
 * @param   details Pointer to S_HBBTV_EVENT_DETAILS
 */
void HBBTV_ReleaseEventDetails(S_HBBTV_EVENT_DETAILS *details)
{
   FUNCTION_START(HBBTV_ReleaseEventDetails);

   ReleaseEventDetails(details);

   FUNCTION_FINISH(HBBTV_ReleaseEventDetails);
}

/**
 * @brief   Retrieves the DVB locator information for a given LCN.
 * @param   lcn The LCN for the channel in question
 * @param   p_dvb_locator The DVB locator structure
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER invalid parameter
 *          - HBBTV_ERR_OTHER controlling application specific error
 */
E_HBBTV_ERR HBBTV_GetLCNLocator(U32BIT lcn, S_HBBTV_DVB_LOCATOR *p_dvb_locator)
{
   E_HBBTV_ERR result;
   void *service;

   FUNCTION_START(HBBTV_GetLCNLocator);

   service = ADB_FindServiceByLcn(ADB_SERVICE_LIST_ALL, lcn, TRUE);
   if (service != NULL)
   {
      result = HBBTV_OK;
      ADB_GetServiceIds(service, &(p_dvb_locator->original_network_id),
         &(p_dvb_locator->transport_stream_id), &(p_dvb_locator->service_id));
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_GetLCNLocator);

   return result;
}

/**
 * @brief   Retrieves the list of components available for a service. Provided HBBTV_OK is returned,
 *          the HbbTV engine will call HBBTV_ReleaseComponentList to release allocated data on the
 *          same task and immediately after it has copied the data.
 * @param   locator DVB locator identifying a service (Not NULL)
 * @param   type Required component type
 * @param   num_ptr Pointer to number of components (Not NULL)
 * @param   list_ptr Pointer to S_HBBTV_COMPONENT_DETAILS Pointer (Not NULL)
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER invalid parameter
 *          - HBBTV_ERR_OTHER controlling application specific error
 */
E_HBBTV_ERR HBBTV_ObtainComponentList(S_HBBTV_DVB_LOCATOR *locator, E_HBBTV_COMPONENT_TYPE type,
   U32BIT *num_ptr, S_HBBTV_COMPONENT_DETAILS **list_ptr)
{
   E_HBBTV_ERR result;
   void *service;
   ADB_STREAM_LIST_TYPE list_type;
   void **list;
   U32BIT num_components = 0, i;
   U8BIT path;
   BOOLEAN service_has_ca_descriptor;
   S_HBBTV_COMPONENT_DETAILS *comp;
   U8BIT component_type;
   ADB_STREAM_TYPE stream_type;
   E_STB_DP_DECODE_STATUS decode_status;
   U32BIT lang_code;

   FUNCTION_START(HBBTV_ObtainComponentList);

   service = ADB_FindServiceByIds(locator->original_network_id, locator->transport_stream_id,
         locator->service_id);
   if (service != NULL)
   {
      DBG("Requested type=%s", (type == HBBTV_COMPONENT_VIDEO) ? "HBBTV_COMPONENT_VIDEO" :
         (type == HBBTV_COMPONENT_AUDIO) ? "HBBTV_COMPONENT_AUDIO" :
         (type == HBBTV_COMPONENT_SUBTITLE) ? "HBBTV_COMPONENT_SUBTITLE" : "HBBTV_COMPONENT_ALL");

      switch (type)
      {
         case HBBTV_COMPONENT_VIDEO:
         {
            list_type = ADB_VIDEO_LIST_STREAM;
            break;
         }
         case HBBTV_COMPONENT_AUDIO:
         {
            list_type = ADB_AUDIO_LIST_STREAM;
            break;
         }
         case HBBTV_COMPONENT_SUBTITLE:
         {
            list_type = ADB_SUBTITLE_LIST_STREAM | ADB_TTEXT_SUBT_LIST_STREAM;;
            break;
         }
         case HBBTV_COMPONENT_ALL:
         {
            list_type = ADB_VIDEO_LIST_STREAM | ADB_AUDIO_LIST_STREAM | ADB_SUBTITLE_LIST_STREAM |
               ADB_TTEXT_SUBT_LIST_STREAM;
            break;
         }
      }

      ADB_GetStreamList(service, list_type, &list, (U16BIT *)&num_components);
      DBG("ADB_GetStreamList returned %d components", num_components);

      if (num_components > 0)
      {
         *list_ptr = STB_AppGetMemory(num_components * sizeof(S_HBBTV_COMPONENT_DETAILS));
         if (*list_ptr != NULL)
         {
            if ((path = ACTL_GetActivePath()) == INVALID_RES_ID)
            {
               /* Ignore the decoding state and just set the 'active' flag on
                * whether they're marked as 'in_use' */
            }
            else
            {
               if (service != STB_DPGetTunedService(path))
               {
                  /* Getting components for a different service, so set the 'active'
                   * flag on whether they're marked as 'in_use' */
                  path = INVALID_RES_ID;
               }
            }

            service_has_ca_descriptor = ADB_GetServiceHasCaDesc(service);
            comp = *list_ptr;
            for (i = 0; i < num_components; i++, comp++)
            {
               stream_type = ADB_GetStreamType(list[i]);

               /* Overwrite the variable type in case it was HBBTV_COMPONENT_ALL */
               switch (stream_type)
               {
                  case ADB_VIDEO_STREAM:
                  case ADB_H264_VIDEO_STREAM:
#if 0
                  /* H265 isn't defined as a supported format in the OIPFs specs */
                  case ADB_H265_VIDEO_STREAM:
#endif
                  {
                     type = HBBTV_COMPONENT_VIDEO;
                     break;
                  }

                  case ADB_AUDIO_STREAM:
                  case ADB_AAC_AUDIO_STREAM:
                  case ADB_HEAAC_AUDIO_STREAM:
                  case ADB_AC3_AUDIO_STREAM:
                  case ADB_EAC3_AUDIO_STREAM:
                  {
                     type = HBBTV_COMPONENT_AUDIO;
                     break;
                  }

                  case ADB_SUBTITLE_STREAM:
                  case ADB_TTEXT_STREAM:
                  {
                     type = HBBTV_COMPONENT_SUBTITLE;
                     break;
                  }

                  default:
                     DBG("Unexpected stream type %d", stream_type);
                     break;
               }
               if (ADB_GetStreamNumTags(list[i]) > 0)
               {
                  comp->component_tag = ADB_GetStreamTag(list[i], 0);
#ifdef DEBUG
                  {
                     U8BIT t;

                     for (t = 0; t < ADB_GetStreamNumTags(list[i]); t++)
                     {
                        DBG("index=%d tag=%d", t, ADB_GetStreamTag(list[i], t));
                     }
                  }
#endif
               }
               else
               {
                  DBG("ADB_GetStreamNumTags returned 0 tags");
                  comp->component_tag = -1;
               }

               comp->pid = ADB_GetStreamPID(list[i]);
               comp->type = type;
               component_type = ADB_GetStreamComponentType(list[i]);

               /* Initial value, it will be overwritten if needed */
               comp->active = FALSE;

               if (type == HBBTV_COMPONENT_VIDEO)
               {
                  decode_status = STB_DPGetVideoStatus(path);
                  if ((path != INVALID_RES_ID) && (decode_status != DECODE_STOPPING) &&
                      (decode_status != DECODE_STOPPED))
                  {
                     if (comp->pid == STB_DPGetVideoPID(path))
                     {
                        comp->active = TRUE;
                     }
                  }

                  /* if the stream is currently being decoded, ask the decoder, otherwise rely on
                   * stream_type and component_type */
                  if (comp->active == TRUE)
                  {
                     U16BIT w, h;

                     if (STB_DPGetVideoCodec(path) == VIDEO_CODEC_H264)
                     {
                        comp->av.video.encoding = HBBTV_VIDEO_H_264;
                     }
                     else
                     {
                        comp->av.video.encoding = HBBTV_VIDEO_MPEG2;
                     }
                     if (STB_VTGetVideoAspectRatio() == ASPECT_RATIO_4_3)
                     {
                        comp->av.video.aspect_ratio = HBBTV_VIDEO_ASPECT_RATIO_4x3;
                     }
                     else
                     {
                        comp->av.video.aspect_ratio = HBBTV_VIDEO_ASPECT_RATIO_16x9;
                     }
                     DBG("aspect_ratio=%s", (comp->av.video.aspect_ratio == HBBTV_VIDEO_ASPECT_RATIO_16x9) ?
                        "HBBTV_VIDEO_ASPECT_RATIO_16x9" : "HBBTV_VIDEO_ASPECT_RATIO_4x3");
                     comp->av.video.frame_rate = STB_AVGetVideoFrameRate(STB_DPGetPathVideoDecoder(path));
                     STB_VTGetVideoResolution(&w, &h);
                     if (w > 720)
                     {
                        comp->av.video.hd = TRUE;
                     }
                     else
                     {
                        comp->av.video.hd = FALSE;
                     }
                  }
                  else
                  {
                     switch (stream_type)
                     {
#if 0
                        /* H265 isn't defined as a supported format in the OIPFs specs */
                        case ADB_H265_VIDEO_STREAM:
                            /* as yet undefined */
#endif
                        case ADB_H264_VIDEO_STREAM:
                        {
                           comp->av.video.encoding = HBBTV_VIDEO_H_264;
                           if (((component_type > 4) && (component_type < 9)) ||
                               ((component_type > 0xE) && (component_type < 0x11)) ||
                               ((component_type > 0x81) && (component_type < 0x84)))
                           {
                              comp->av.video.frame_rate = 30;
                           }
                           else
                           {
                              comp->av.video.frame_rate = 25;
                           }

                           if ((component_type == 1) || (component_type == 5))
                           {
                              comp->av.video.aspect_ratio = HBBTV_VIDEO_ASPECT_RATIO_4x3;
                           }
                           else
                           {
                              comp->av.video.aspect_ratio = HBBTV_VIDEO_ASPECT_RATIO_16x9;
                           }
                           break;
                        }
                        default:
                        {
                           comp->av.video.encoding = HBBTV_VIDEO_MPEG2;
                           if (((component_type > 4) && (component_type < 9)) ||
                               ((component_type > 0xC) && (component_type < 0x11)))
                           {
                              comp->av.video.frame_rate = 30;
                           }
                           else
                           {
                              comp->av.video.frame_rate = 25;
                           }

                           if ((component_type == 0) || ((component_type % 4) == 1))
                           {
                              comp->av.video.aspect_ratio = HBBTV_VIDEO_ASPECT_RATIO_4x3;
                           }
                           else
                           {
                              comp->av.video.aspect_ratio = HBBTV_VIDEO_ASPECT_RATIO_16x9;
                           }
                           break;
                        }
                     }

                     if (component_type > 8)
                     {
                        comp->av.video.hd = TRUE;
                     }
                     else
                     {
                        comp->av.video.hd = FALSE;
                     }
                  }
               }
               else if (type == HBBTV_COMPONENT_AUDIO)
               {
                  E_STB_DP_AUDIO_MODE audio_mode;
                  ADB_AUDIO_TYPE audio_type;

                  switch (stream_type)
                  {
                     case ADB_HEAAC_AUDIO_STREAM:
                     {
                        comp->av.audio.encoding = HBBTV_AUDIO_HEAAC;
                        break;
                     }
                     case ADB_EAC3_AUDIO_STREAM:
                     {
                        comp->av.audio.encoding = HBBTV_AUDIO_E_AC3;
                        break;
                     }
                     default:
                     {
                        comp->av.audio.encoding = HBBTV_AUDIO_MPEG2;
                        break;
                     }
                  }

                  lang_code = ADB_GetAudioStreamLangCode(list[i]);
                  comp->av.audio.lang_code[0] = (lang_code >> 16) & 0xFF;
                  comp->av.audio.lang_code[1] = (lang_code >> 8) & 0xFF;
                  comp->av.audio.lang_code[2] = lang_code & 0xFF;
                  comp->av.audio.lang_code[3] = 0;
                  DBG("lang_code=%s", comp->av.audio.lang_code);
                  audio_type = ADB_GetAudioStreamType(list[i]);
                  if (audio_type == ADB_AUDIO_TYPE_FOR_VISUALLY_IMPAIRED)
                  {
                     comp->av.audio.audio_description = TRUE;
                  }
                  else
                  {
                     comp->av.audio.audio_description = FALSE;
                  }

                  audio_mode = ADB_GetAudioStreamMode(list[i]);
                  switch (audio_mode)
                  {
                     case AUDIO_STEREO:
                     {
                        comp->av.audio.audio_channels = 1;
                        break;
                     }
                     case AUDIO_MULTICHANNEL:
                     {
                        comp->av.audio.audio_channels = 5;
                        break;
                     }
                     default:
                     {
                        comp->av.audio.audio_channels = 2;
                        break;
                     }
                  }

                  decode_status = STB_DPGetAudioStatus(path);
                  if ((path != INVALID_RES_ID) && (decode_status != DECODE_STOPPING) &&
                      (decode_status != DECODE_STOPPED))
                  {
                     if (comp->pid == STB_DPGetAudioPID(path))
                     {
                        comp->active = TRUE;
                     }
                  }
               }
               else if (type == HBBTV_COMPONENT_SUBTITLE)
               {
                  if (ADB_GetStreamType(list[i]) == ADB_TTEXT_STREAM)
                  {
                     U8BIT ttx_type;
                     U8BIT magazine, page;

                     lang_code = ADB_GetTtextStreamLangCode(list[i]);
                     comp->av.subtitle.lang_code[0] = (lang_code >> 16) & 0xFF;
                     comp->av.subtitle.lang_code[1] = (lang_code >> 8) & 0xFF;
                     comp->av.subtitle.lang_code[2] = lang_code & 0xFF;
                     comp->av.subtitle.lang_code[3] = 0;
                     DBG("lang_code=%s", comp->av.subtitle.lang_code);
                     ttx_type = ADB_GetTtextStreamType(list[i]);
                     if (ttx_type == ADB_AUDIO_TYPE_FOR_HEARING_IMPAIRED)
                     {
                        comp->av.subtitle.hearing_impaired = TRUE;
                     }
                     else
                     {
                        comp->av.subtitle.hearing_impaired = FALSE;
                     }

                     if (ACTL_AreSubtitlesDisplayed() == TRUE)
                     {
                        if (comp->pid == ADB_GetRequiredTtextPid(STB_DPGetTunedService(path), TRUE,
                           &magazine, &page))
                        {
                           comp->active = TRUE;
                        }
                     }
                  }
                  else
                  {
                     ADB_SUBTITLE_TYPE subt_type;
                     U16BIT cpage, apage;

                     lang_code = ADB_GetSubtitleStreamLangCode(list[i]);
                     comp->av.subtitle.lang_code[0] = (lang_code >> 16) & 0xFF;
                     comp->av.subtitle.lang_code[1] = (lang_code >> 8) & 0xFF;
                     comp->av.subtitle.lang_code[2] = lang_code & 0xFF;
                     comp->av.subtitle.lang_code[3] = 0;

                     subt_type = ADB_GetSubtitleStreamType(list[i]);
                     if ((subt_type == ADB_SUBTITLE_TYPE_DVB_HARD_HEARING) ||
                         (subt_type == ADB_SUBTITLE_TYPE_DVB_HARD_HEARING_4_3) ||
                         (subt_type == ADB_SUBTITLE_TYPE_DVB_HARD_HEARING_16_9) ||
                         (subt_type == ADB_SUBTITLE_TYPE_DVB_HARD_HEARING_221_1) ||
                         (subt_type == ADB_SUBTITLE_TYPE_DVB_HARD_HEARING_HD))
                     {
                        comp->av.subtitle.hearing_impaired = TRUE;
                     }
                     else
                     {
                        comp->av.subtitle.hearing_impaired = FALSE;
                     }

                     if (ACTL_AreSubtitlesDisplayed() == TRUE)
                     {
                        if (comp->pid == ADB_GetRequiredSubtitleParams(STB_DPGetTunedService(path),
                           &cpage, &apage))
                        {
                           comp->active = TRUE;
                        }
                     }
                  }
               }

               if (service_has_ca_descriptor == TRUE)
               {
                  comp->encrypted = TRUE;
               }
               else
               {
                  comp->encrypted = ADB_GetStreamHasCaDesc(list[i]);
               }
               DBG("type=%d PID=%d tag=%d active=%d", comp->type, comp->pid, comp->component_tag, comp->active);
            }
            result = HBBTV_OK;
         }
         else
         {
            DBG("Failed allocating %d bytes", num_components * sizeof(S_HBBTV_COMPONENT_DETAILS));
            result = HBBTV_ERR_OTHER;
            num_components = 0;
         }

         ADB_ReleaseStreamList(list, (U16BIT)num_components);
      }
   }
   else
   {
      DBG("Service not found");
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   *num_ptr = num_components;

   FUNCTION_FINISH(HBBTV_ObtainComponentList);

   return result;
}

/**
 * @brief   Releases data allocated by HBBTV_ObtainComponentList (e.g. String data). Always called
 *          immediately after the HbbTV engine has copied data.
 * @param   num Number of elements in the list
 * @param   list_ptr Pointer to S_HBBTV_COMPONENT_DETAILS
 */
void HBBTV_ReleaseComponentList(U32BIT num, S_HBBTV_COMPONENT_DETAILS *list_ptr)
{
   FUNCTION_START(HBBTV_ReleaseComponentList);
   USE_UNWANTED_PARAM(num);
   STB_AppFreeMemory(list_ptr);
   FUNCTION_FINISH(HBBTV_ReleaseComponentList);
}

/**
 * @ brief  Retrieves Event list information for a Service. Provided HBBTV_OK is returned, the HbbTV
 *          engine will call HBBTV_ReleaseEventList to release allocated data on the same task and
 *          immediately after it has copied the data.
 * @param   service_index Receiver dependent identifier for the required service
 * @param   num_ptr Pointer to number of events (Not NULL)
 * @param   list_ptr Pointer to S_HBBTV_EVENT_DETAILS Pointer (Not NULL)
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER invalid parameter
 *          - HBBTV_ERR_OTHER controlling application specific error
 */
E_HBBTV_ERR HBBTV_ObtainEventList(U32BIT service_index, U32BIT *num_ptr,
   S_HBBTV_EVENT_DETAILS **list_ptr)
{
   E_HBBTV_ERR result;
   void *s_ptr, **elist, *e_ptr;
   U16BIT num_events, i;
   S_HBBTV_EVENT_DETAILS *p_evnt;

   FUNCTION_START(HBBTV_ObtainEventList);

   s_ptr = ADB_FindServiceByLcn(ADB_SERVICE_LIST_ALL, (U16BIT)service_index, FALSE );
   if (s_ptr == NULL)
   {
      result = HBBTV_ERR_OTHER;
      num_events = 0;
      *list_ptr = NULL;
   }
   else
   {
      ADB_GetEventSchedule(FALSE, s_ptr, &elist, &num_events);
      DBG("ADB_GetEventSchedule returned %d events (service_index=%d, s_ptr=%p)", num_events, 
         service_index, s_ptr);
      if ((elist == NULL) || (num_events == 0))
      {
         DBG("ADB_GetEventSchedule returned: elist=%p num=%d", elist, num_events);
         num_events = 0;
         *list_ptr = NULL;
         result = HBBTV_ERR_OTHER;
      }
      else
      {
         p_evnt = STB_AppGetMemory(num_events * sizeof(S_HBBTV_EVENT_DETAILS));
         if (p_evnt != NULL)
         {
            *list_ptr = p_evnt;
            for (i = 0; i < num_events; i++, p_evnt++)
            {
               e_ptr = elist[i];

               EXT_HbbtvGetEvent(e_ptr, s_ptr, p_evnt);

               /* No need to call ADB_ReleaseEventData as we do ADB_ReleaseEventList below */
               p_evnt->dvb_user_data = NULL;
            }

            ADB_ReleaseEventList(elist, num_events);
            result = HBBTV_OK;
         }
         else
         {
            num_events = 0;
            *list_ptr = NULL;
            result = HBBTV_ERR_OTHER;
         }
      }
   }

   *num_ptr = num_events;

   FUNCTION_FINISH(HBBTV_ObtainEventList);

   return result;
}

/**
 * @brief   Release data allocated by HBBTV_ObtainEventList(e.g. String data). Always called
 *          immediately after the HbbTV engine has copied the data.
 * @param   num Number of elements in the list
 * @param   list_ptr Pointer to S_HBBTV_EVENT_DETAILS
 */
void HBBTV_ReleaseEventList(U32BIT num, S_HBBTV_EVENT_DETAILS *list_ptr)
{
   U32BIT i;
   S_HBBTV_EVENT_DETAILS *event;

   FUNCTION_START(HBBTV_ReleaseEventList);

   for (i = 0, event = list_ptr; i < num; i++, event++)
   {
      ReleaseEventDetails(event);
   }

   STB_AppFreeMemory(list_ptr);

   FUNCTION_FINISH(HBBTV_ReleaseEventList);
}

/**
 * @brief   Retrieves information about the indicated service. Provided HBBTV_OK is returned, the
 *          HbbTV engine will call HBBTV_ReleaseServiceDetails to release allocated data on the same
 *          task and immediately after it has copied the data.
 * @param   service_lcn Service LCN
 * @param   details Pointer to S_HBBTV_SERVICE_DETAILS structure to place required service
 *          information (this will NOT be NULL)
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_OTHER otherwise.
 */
E_HBBTV_ERR HBBTV_GetServiceDetails(U16BIT service_lcn, S_HBBTV_SERVICE_DETAILS *details)
{
   E_HBBTV_ERR result;
   void *service;

   FUNCTION_START(HBBTV_GetServiceDetails);

   service = ADB_FindServiceByLcn(ADB_SERVICE_LIST_ALL, service_lcn, TRUE);
   if (service != NULL)
   {
      details->service_index = service_lcn;
      details->service_lcn = service_lcn;
      details->service_name.zptr = ADB_GetServiceShortName(service, TRUE);
      details->service_name.zlen = STB_GetNumBytesInString(details->service_name.zptr) - 1;
      details->service_type = ADB_GetServiceType(service);
      details->nid = GetServiceNID(service);
      details->hidden = ADB_GetServiceHiddenFlag(service);

      result = HBBTV_OK;
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_GetServiceDetails);

   return result;
}

/**
 * @brief   Releases data allocated by HBBTV_ReleaseServiceDetails (e.g. String data). Always called
 *          immediately after the HbbTV engine has copied data.
 * @param   details Pointer to S_HBBTV_SERVICE_DETAILS
 */
void HBBTV_ReleaseServiceDetails(S_HBBTV_SERVICE_DETAILS *details)
{
   FUNCTION_START(HBBTV_ReleaseServiceDetails);

   if (details->service_name.zptr != NULL)
   {
      STB_ReleaseUnicodeString(details->service_name.zptr);
   }

   FUNCTION_FINISH(HBBTV_ReleaseServiceDetails);
}

/**
 * @brief   Retrieves raw SI descriptor data with the defined descriptor tag id, and optionally the
 *          extended descriptor tag id, for an event on a service. The data must be freed using
 *          HBBTV_ReleaseEventSIDescriptorData.
 * @param   dvb_locator Pointer to S_HBBTV_DVB_LOCATOR containing the original network id, transport
 *          id and service id of the service to be queried
 * @param   event_id Id of the event to be queried
 * @param   desc_tag_id Descriptor tag id to be returned
 * @param   ext_desc_tag_id Extended descriptor tag id to be returned, or negative if not relevant
 * @param   desc_len Length of the returned data, or 0 if no data is returned
 * @return  Pointer to an allocated buffer containing a copy of the data for the descriptor. If
 *          there are multiple descriptors with the same tag id then they will all be returned. NULL
 *          is returned if the descriptor isn't found.
 */
U8BIT* HBBTV_GetEventSIDescriptorData(S_HBBTV_DVB_LOCATOR *dvb_locator, U16BIT event_id,
   U8BIT desc_tag_id, S32BIT ext_desc_tag_id, U16BIT *desc_len)
{
   U8BIT *result;
   void *service;

   FUNCTION_START(HBBTV_GetEventSIDescriptorData);

   result = NULL;

   DBG("Service: (%d, %d, %d); ID: %d, tag:%d ext_tag:%d", dvb_locator->original_network_id, dvb_locator->transport_stream_id,
      dvb_locator->service_id, event_id);

   service = ADB_FindServiceByIds(dvb_locator->original_network_id,
         dvb_locator->transport_stream_id, dvb_locator->service_id);
   if (service != NULL)
   {
      result = ADB_GetEventSIDescriptorData(service, event_id, desc_tag_id, ext_desc_tag_id,
            desc_len);
#ifdef DEBUG
      {
         U16BIT i;

         DBG("Result: (0x%x)", result);
         for (i = 0; (i < *desc_len) && (result != NULL); i++)
         {
            STB_SPDebugNoCnWrite("0x%02x ", result[i]);
            if ((i % 16) == 15)
            {
               STB_SPDebugNoCnWrite("\n");
            }
         }
         if ((i % 16) != 0)
         {
            STB_SPDebugNoCnWrite("\n");
         }
      }
#endif
   }
#ifdef DEBUG
   else
   {
      DBG("Could not find service with locator (%d, %d, %d)", dvb_locator->original_network_id,
         dvb_locator->transport_stream_id, dvb_locator->service_id);
   }
#endif

   FUNCTION_FINISH(HBBTV_GetEventSIDescriptorData);

   return result;
}

/**
 * @brief   Frees event SI descriptor data returned through a call to HBBTV_GetEventSIDescriptorData.
 * @param   desc_data Pointer to data to be freed
 * @param   desc_len Length of the descriptor data
 */
void HBBTV_ReleaseEventSIDescriptorData(U8BIT *desc_data, U16BIT desc_len)
{
   FUNCTION_START(HBBTV_ReleaseEventSIDescriptorData);

   ADB_ReleaseEventSIDescriptorData(desc_data, desc_len);

   FUNCTION_FINISH(HBBTV_ReleaseEventSIDescriptorData);
}

/**
 * @brief   Retrieves the locked status of the specified service. The correct implementation of this
 *          function is not mandatory for HbbTV 1.2.1. It is used to implement the channel's locked
 *          property as defined in OIPF vol. 5, section 7.13.11.2.
 * @param   service_lcn Service LCN
 * @return  TRUE if the parental control system prevents the channel from being viewed (e.g. when
 *          a pin needs to be entered by the user).
 */
BOOLEAN HBBTV_GetServiceLocked(U16BIT service_lcn)
{
   BOOLEAN locked = FALSE;
   void *service;

   FUNCTION_START(HBBTV_GetServiceLocked);

   service = ADB_FindServiceByLcn(ADB_SERVICE_LIST_ALL, service_lcn, TRUE);
   if (service != NULL)
   {
      locked = ADB_GetServiceLockedFlag(service);
   }
#ifdef DEBUG
   else
   {
      DBG("service NULL for LCN %d", service_lcn);
   }
#endif

   FUNCTION_FINISH(HBBTV_GetServiceLocked);

   return locked;
}

/**
 * @brief   Fills in the event details structure given the event handle
 * @param   event Event handle
 * @param   service Service handle
 * @param   details Pointer to the structure to fill in. HBBTV_ReleaseEventDetails can be called
 *          passing the structure pointer to ensure any allocated data relative to the event is
 *          freed
 */
void EXT_HbbtvGetEvent(void *event, void *service, S_HBBTV_EVENT_DETAILS *details)
{
   U8BIT *temp;
   U32DHMS event_start_time, event_duration, event_end_time;
   ADB_EVENT_COMPONENT_INFO *component_list;
   U8BIT num, i;
   S_HBBTV_STRING *string;
   S_LANG_CODE *lang_code_entry;
   U32BIT *timers;

   FUNCTION_START(EXT_HbbtvGetEvent);

   STB_OSMutexLock(service_mutex);

   details->dvb_user_data = event;

   details->event_name.zptr = ADB_GetEventName(event);
   details->event_name.zlen = STB_GetNumBytesInString(details->event_name.zptr) - 1;
   details->short_description.zptr = ADB_GetEventDescription(event);
   details->short_description.zlen = STB_GetNumBytesInString(details->short_description.zptr) - 1;
   
   details->parental_rating = ADB_GetEventParentalAge(event);

   event_start_time = ADB_GetEventStartDateTime(event);
   details->start_date = DHMS_DATE(event_start_time);

   details->start_time.hours = DHMS_HOUR(event_start_time);
   details->start_time.minutes = DHMS_MINS(event_start_time);
   details->start_time.seconds = DHMS_SECS(event_start_time);

   event_duration = ADB_GetEventDuration(event);
   details->duration.hours = DHMS_HOUR(event_duration);
   details->duration.minutes = DHMS_MINS(event_duration);
   details->duration.seconds = DHMS_SECS(event_duration);

   details->event_id = ADB_GetEventId(event);

   temp = ADB_GetEventFullProgrammeCrid(service, event);
   if (temp != NULL)
   {
      details->crid.zptr = temp;
      details->crid.zlen = STB_GetNumBytesInString(temp) - 1;
      details->use_crid = TRUE;
   }
   else
   {
      details->crid.zptr = NULL;
      details->crid.zlen = 0;
      details->use_crid = FALSE;
   }
   
   details->long_description.zptr = ADB_GetEventExtendedDescription(event);
   details->long_description.zlen = STB_GetNumBytesInString(details->long_description.zptr) - 1;

   details->audio_type = 0;
   details->is_hd = FALSE;

   details->num_audio_languages = 0;

   num = ADB_GetEventComponentList(event, &component_list);
   if ((num > 0) && (component_list != NULL))
   {
      BOOLEAN audio;

      for (i = 0; i < num; i++)
      {
         audio = FALSE;
         if (component_list[i].stream_content == 1)
         {
            if ((component_list[i].component_type >= 9) && 
               (component_list[i].component_type <= 0x10))
            {
               details->is_hd = TRUE;
            }
         }
         else if (component_list[i].stream_content == 2)
         {
            audio = TRUE;
            if ((component_list[i].component_type == 1) || (component_list[i].component_type == 2))
            {
               details->audio_type |= 1;
            }
            else if ((component_list[i].component_type == 3))
            {
               details->audio_type |= 2;
            }
            else if ((component_list[i].component_type == 4) || (component_list[i].component_type == 5))
            {
               details->audio_type |= 4;
            }
         }
         else if (component_list[i].stream_content == 4)
         {
            audio = TRUE;
            if (((component_list[i].component_type & 7) == 0) || 
               ((component_list[i].component_type & 7) == 1) ||
               ((component_list[i].component_type & 7) == 6))
            {
               details->audio_type |= 1;
            }
            else if (((component_list[i].component_type & 7) == 2) || 
               ((component_list[i].component_type & 7) == 3))
            {
               details->audio_type |= 2;
            }
            else if (((component_list[i].component_type & 7) == 4) ||
               ((component_list[i].component_type & 7) == 5))
            {
               details->audio_type |= 8;
            }
         }
         else if (component_list[i].stream_content == 5)
         {
            if ((component_list[i].component_type == 0xb) || 
               (component_list[i].component_type == 0xc) ||
               (component_list[i].component_type == 0xf) ||
               (component_list[i].component_type == 0x10) ||
               ((component_list[i].component_type >= 0x80) && 
               (component_list[i].component_type <= 0x83)))
            {
               details->is_hd = TRUE;
            }
         }
         else if (component_list[i].stream_content == 6)
         {
            audio = TRUE;
            if (component_list[i].component_type == 1)
            {
               details->audio_type |= 1;
            }
            else if ((component_list[i].component_type == 3) || 
               (component_list[i].component_type == 0x43))

            {
               details->audio_type |= 2;
            }
            else if (component_list[i].component_type == 5)
            {
               details->audio_type |= 4;
            }
         }

         if (audio)
         {
            if(IsLangCodeValid(component_list[i].language_code) && 
               !IsLangCodePresent(component_list[i].language_code))
            {
               DBG("Adding %c%c%c", component_list[i].language_code[0], 
                  component_list[i].language_code[1], component_list[i].language_code[2]);
               AddLangCode(component_list[i].language_code);
               details->num_audio_languages++;
            }
         }
      }

      STB_AppFreeMemory(component_list);
   }

   if (details->num_audio_languages > 0)
   {
      details->audio_language_array = STB_AppGetMemory(details->num_audio_languages * 
         sizeof(S_HBBTV_STRING));
      if (details->audio_language_array != NULL)
      {
         memset(details->audio_language_array, 0, details->num_audio_languages * 
            sizeof(S_HBBTV_STRING));
         i = 0;
         lang_code_entry = (S_LANG_CODE *)STB_LLGetFirstBlock(&audio_lang_list);
         while ((lang_code_entry != NULL) && (i < details->num_audio_languages))
         {
            string = &(details->audio_language_array[i]);
            string->zptr = STB_AppGetMemory(4);
            if (string->zptr != NULL)
            {
               memcpy(string->zptr, lang_code_entry->lang_code, 3);
               string->zptr[3] = 0;
               string->zlen = 3;
            }
#ifdef DEBUG
            else
            {
               DBG("Failed allocating 4 bytes");
            }
#endif
            lang_code_entry = (S_LANG_CODE *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)lang_code_entry);
            i++;
         }
      }
#ifdef DEBUG
      else
      {
         DBG("Failed allocating %d bytes", details->num_audio_languages * 
            sizeof(S_HBBTV_STRING));
      }
#endif

      EmptyLangCodeList();
   }

   if (details->num_audio_languages > 1)
   {
      details->is_multilingual = TRUE;
   }
   else
   {
      details->is_multilingual = FALSE;
   }

   details->num_genres = 0;

   details->has_recording = FALSE;
   details->recording_id = 0;
   details->timer_id = 0;
   event_end_time = STB_GCCalculateDHMS(event_start_time, event_duration, CALC_ADD);
   num = ATMR_GetNumSimultaneousRecordings(1, event_start_time, event_end_time, &timers);
   if (num > 0)
   {
      for (i = 0; i < num; i++)
      {
         if (timers[i] != INVALID_TIMER_HANDLE)
         {
            if (ATMR_GetServiceId(timers[i]) == ADB_GetServiceId(service))
            {
               details->has_recording = TRUE;
               details->timer_id = timers[i];
               details->recording_id = ATMR_GetRecordingHandle(timers[i]);
               break;
            }
         }
      }
   }

   if (timers != NULL)
   {
      STB_AppFreeMemory(timers);
   }

   if (details->recording_id == 0)
   {
      U32BIT handle;

      /* No in progress recording, try to find a complete recording using CRID */
      if ((details->crid.zptr != NULL) && STB_PVRFindRecordingFromCrid(details->crid.zptr, &handle))
      {
         details->recording_id = EXT_HbbtvPVRGetRecID(details->timer_id, handle);
      }
   }

   STB_OSMutexUnlock(service_mutex);

   FUNCTION_FINISH(EXT_HbbtvGetEvent);
}

void EXT_HbbtvServiceInitialise(void)
{
   if (service_mutex == NULL)
   {
      service_mutex = STB_OSCreateMutex();
   }
}

void EXT_HbbtvServiceTerminate(void)
{
   if (service_mutex != NULL)
   {
      STB_OSDeleteMutex(service_mutex);
      service_mutex = NULL;
   }
}
