/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2014 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief      DVB Interface for MHEG5 engine - Audio
 * @file       audio.c
 * @date       Janauary 2014
 * @author     Adam Sturtridge
 */
//#define DEBUG_PRINTING_ENABLED

/*---includes for this file--------------------------------------------------*/

/* compiler library header files */
#include <stdio.h>

/* third party header files */

#include <techtype.h>
#include <dbgfuncs.h>

#include "stberc.h"
#include "stbhwav.h"
#include "stbdpc.h"

#include "ap_dbacc.h"
#include "ap_cntrl.h"
#include "app.h"
#include "ap_cfg.h"
#include "ap_uiinfo.h"
#include "dvbmh_int.h"

#include "dvb_audio.h"


/*---constant definitions for this file--------------------------------------*/

/*---local typedef structs for this file-------------------------------------*/

/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local)                 */

/*---local function prototypes for this file---------------------------------*/
/*   (internal functions declared static to make them local)                 */

/*---local function definitions----------------------------------------------*/

static E_STB_DP_AUDIO_CODEC EvalAudioCodec(void *service, ADB_STREAM_TYPE stream_type)
{
   E_STB_DP_AUDIO_CODEC codec;
   DBGPRINT("stream_type=%x ServiceType=%x",stream_type,ADB_GetServiceType(service))
   switch (stream_type)
   {
      case ADB_AAC_AUDIO_STREAM:
      {
         codec = AUDIO_CODEC_AAC;
         break;
      }
      case ADB_HEAAC_AUDIO_STREAM:
      {
         codec = AUDIO_CODEC_HEAAC;
         break;
      }
      case ADB_AC3_AUDIO_STREAM:
      {
         codec = AUDIO_CODEC_AC3;
         break;
      }
      case ADB_EAC3_AUDIO_STREAM:
      {
         codec = AUDIO_CODEC_EAC3;
         break;
      }
      case ADB_AUDIO_STREAM:
      {
         codec = AUDIO_CODEC_MP2;
         break;
      }
      default:
      case ADB_DATA_STREAM:
      {
         switch (ADB_GetServiceType(service))
         {
            default:
            {
               DBGPRINT("ServiceType=%x",ADB_GetServiceType(service))
               /* Fall tho */
            }
            case ADB_SERVICE_TYPE_TV:
            case ADB_SERVICE_TYPE_MPEG2_HD:
            {
               codec = AUDIO_CODEC_MP2;
               break;
            }
            case ADB_SERVICE_TYPE_AVC_SD_TV:
            case ADB_SERVICE_TYPE_HD_TV:
            {
               codec = AUDIO_CODEC_AAC;
               break;
            }
            case ADB_SERVICE_TYPE_UHD_TV:
            {
               codec = AUDIO_CODEC_EAC3;
               break;
            }
         }
         break;
      }
   }
   return codec;
}

/*---global function definitions---------------------------------------------*/


void DVBMH_PlayAudio(U8BIT path,U16BIT audio_pid, E_STB_DP_AUDIO_CODEC codec, U16BIT ad_pid)
{
   E_STB_DP_AUDIO_CODEC prev_codec;
   U16BIT prev_pid;
   E_STB_DP_DECODE_STATUS decode_status;

   FUNCTION_START(DVBMH_PlayAudio);

   prev_pid = STB_DPGetAudioPID(path);
   prev_codec = STB_DPGetAudioCodec(path);
   DBGPRINT("prev_PID=%x audio_PID=%x; prev_CDC=%x audio_CDC=%x",prev_pid,audio_pid,prev_codec,codec)

   if (audio_pid != prev_pid || codec != prev_codec || ad_pid != STB_DPGetADPID(path))
   {
      decode_status = STB_DPGetAudioStatus(path);
      if ((decode_status != DECODE_STOPPING) || (decode_status != DECODE_STOPPED))
      {
         STB_DPStopAudioDecoding(path);
      }

      STB_DPSetAudioCodec(path, codec);
      if (ad_pid != INVALID_PID)
      {
         STB_DPSetADPID(path,ad_pid);
         STB_DPSetADEnabled(path, TRUE);
      }
      STB_DPSetAudioPID(path, audio_pid);
   }

   decode_status = STB_DPGetAudioStatus(path);
   if ((decode_status != DECODE_STARTING) && (decode_status != DECODE_RUNNING) && (audio_pid != 0))
   {
      STB_DPStartAudioDecoding(path);
   }

   FUNCTION_FINISH(DVBMH_PlayAudio);
}

/**
 * @brief   MHEG5 engine gives control of Audio to external application.
 *          Set decoding and presentation of Audio according to the default
 *          user selection.
 * @return  MHERR_OK - Success, MHERR_OTHER - failure
 */
E_MhegErr DVB_MhegAudioPlayDefault(void)
{
   E_ACTL_DECODE_CHANGE change;
   E_MhegErr result;
   void *srv_ptr;
   U8BIT path;

   FUNCTION_START(DVB_MhegAudioPlayDefault);

   ACTL_SetVolumeAdjustment(0);
   ACTL_SetMhegAVControl(FALSE);

   path = STB_DPGetLivePath();
   if (path == INVALID_RES_ID)
   {
      result = MHERR_OTHER;
   }
   else
   {
      srv_ptr = STB_DPGetTunedService(path);
      if (srv_ptr == NULL)
      {
         result = MHERR_OTHER;
      }
      else
      {
         result = MHERR_OK;
         /* Ensure PCR PID is setup */
         STB_DPSetPCRPID(path, ADB_GetServicePCRPid(srv_ptr));
         change = ACTL_SetupAudioDecoding(path, srv_ptr);
         if (change & ACTL_ADESC_CHANGE)
         {
            STB_DPStopADDecoding(path);
         }
         if (change & ACTL_AUDIO_CHANGE)
         {
            STB_DPStopAudioDecoding(path);
         }
         STB_DPStartAudioDecoding(path);
         STB_DPStartADDecoding(path);
      }
   }

   FUNCTION_FINISH(DVB_MhegAudioPlayDefault);

   return result;
}

/**
 * @brief   MHEG5 engine takes control of Audio - and informs that Audio must be
 *          switched to the specified Audio stream. This setting takes effect
 *          regardless of whether or not an audio stream is currently playing.
 *          If the locator references a service component that is not currently
 *          being broadcast, the audio should be muted until the referenced
 *          component is being broadcast.
 *
 * @return  MHERR_OK - Success, MHERR_OTHER - failure
 */
E_MhegErr DVB_MhegAudioPlayStream(S_DvbComponent *pDvbComponent)
{
   E_MhegErr result = MHERR_OTHER;
   void *srv_ptr;
   void **stream_list;
   U16BIT num, i;
   U8BIT path, num_tags, j;

   FUNCTION_START(DVB_MhegAudioPlayStream);

   DBGPRINT("serv_id=%d ctag=%d",pDvbComponent->service_id,pDvbComponent->ComponentTag)
   ACTL_SetMhegAVControl(TRUE);

   path = STB_DPGetLivePath();
   if (path != INVALID_RES_ID)
   {
      num = pDvbComponent->transport_stream_id;
      if (num == 0)
      {
         num = ADB_GetTransportTid(ADB_GetTunedTransport(path));
         if ( num == 0 )
         {
            DBGPRINT("Failed to get transport ID for %p", ADB_GetTunedTransport(path))
            num = ADB_INVALID_DVB_ID;
         }
      }
      srv_ptr = ADB_FindServiceByIds(pDvbComponent->original_network_id, num, pDvbComponent->service_id);
      if (srv_ptr == NULL)
      {
         DBGPRINT("Failed to find service (%x.%x.%x)",pDvbComponent->original_network_id, num, pDvbComponent->service_id)
      }
      else
      {
         S16BIT ctag = pDvbComponent->ComponentTag;
         /* Ensure PCR PID is setup */
         STB_DPSetPCRPID(path, ADB_GetServicePCRPid(srv_ptr));
         if (ctag == -1)
         {
            E_ACTL_DECODE_CHANGE change = ACTL_SetupAudioDecoding(path, srv_ptr);
            if (change & ACTL_ADESC_CHANGE)
            {
               STB_DPStopADDecoding(path);
            }
            if (change & ACTL_AUDIO_CHANGE)
            {
               STB_DPStopAudioDecoding(path);
            }
            STB_DPStartAudioDecoding(path);
            STB_DPStartADDecoding(path);
            result = MHERR_OK;
         }
         else
         {
            ADB_GetStreamList(srv_ptr, ADB_AUDIO_LIST_STREAM|ADB_DATA_LIST_STREAM, &stream_list, &num);
            if (stream_list == NULL)
            {
               DBGPRINT("Failed to get AUDIO stream list")
            }
            else
            {
               DBGPRINT("service LCN=%u type=%x streams num %u",
                  ADB_GetServiceLcn(srv_ptr),ADB_GetServiceType(srv_ptr),num)
               for (i = 0; i != num; i++)
               {
                  num_tags = ADB_GetStreamNumTags(stream_list[i]);
                  DBGPRINT("tags num %u",num_tags)
                  for (j = 0; j != num_tags; j++)
                  {
                     DBGPRINT("stream: tag %u",ADB_GetStreamTag(stream_list[i],j))
                     if (ctag == ADB_GetStreamTag(stream_list[i], j))
                     {
                        result = MHERR_OK;
                        DVBMH_PlayAudio(path, ADB_GetStreamPID(stream_list[i]),
                           EvalAudioCodec(srv_ptr,ADB_GetStreamType(stream_list[i])), INVALID_PID);
                        break;
                     }
                  }
                  if (result == MHERR_OK)
                     break;
               }
               ADB_ReleaseStreamList(stream_list, num);
            }
         }
      }
   }

   FUNCTION_FINISH(DVB_MhegAudioPlayStream);

   return result;
}

/**
 * @brief   MHEG5 engine takes control of Audio - and specifies that
 *          decoding and presentation of any Audio stream must be stopped
 * @return  MHERR_OK - Success, MHERR_OTHER - failure
 */
E_MhegErr DVB_MhegAudioStopStream(void)
{
   E_MhegErr result;
   U8BIT path;
   FUNCTION_START(DVB_MhegAudioStopStream);
   ACTL_SetMhegAVControl(TRUE);
   path = STB_DPGetLivePath();
   if (path == INVALID_RES_ID)
   {
      result = MHERR_OTHER;
   }
   else
   {
      result = MHERR_OK;
      STB_DPStopAudioDecoding(path);
   }
   FUNCTION_FINISH(DVB_MhegAudioStopStream);
   return result;
}

/**
 * @brief   Instruct the controlling application that the provided audio sound clip
 *          should be played. Playback from an audio stream and playback of an audio
 *          clip are mutually exclusive.
 *          If an audio clip is still playing due to a previous call to
 *          DVB_MhegPlayAudioClip(), the previous playback should be silently terminated
 *          without a call to DVB_MhegNotifyAudioPlayStopped.
 *          The audio data passed MUST be copied before use, as the caller
 *          retains exclusive ownership. The audio data can be discarded when
 *          either the playback of the last loop finishes, DVB_MhegPlayAudioClip() is
 *          invoked with a new audio clip or DVB_MhegAudioStopClip() is invoked.
 *          Each block of audio data is formatted as either:
 *            1. MPEG2 audio elementary stream, or
 *            2. HE-AAC audio elementary stream, or
 *            3. E-AC3 audio elementary stream
 *          The audio clip should be played a number of times, as indicated by
 *          the repeatCount parameter. Once playback of all loops has completed,
 *          the DVB_MhegNotifyAudioPlayStopped() function should be called.
 *          Playback can be aborted by a call to DVB_MhegAudioStopClip(). In this case,
 *          the DVB_MhegNotifyAudioPlayStopped() function will not be invoked.
 *          This MUST be a non-blocking function.
 * @param   pAudioData Audio data to play.
 * @param   dataLength Length of the audio data.
 * @param   repeatCount Number of times to repeat play of audio data.
 * @param   encoding encoding - see E_AudioEncoding
 * @return  - MHERR_OK on success
 *          - MHERR_OTHER controlling application specific error.
 */
E_MhegErr DVB_MhegAudioPlayClip( U8BIT *pAudioData,
   U32BIT dataLength,
   U32BIT repeatCount,
   E_AudioEncoding encoding )
{
   E_MhegErr result;
   U8BIT decoder;
   FUNCTION_START(DVB_MhegAudioPlayClip);
   if (encoding != AUDIO_ENC_MPEG2)
   {
      result = MHERR_UNSUPPORTED_EXTENSION;
   }
   else
   {
      decoder = STB_DPGetPathAudioDecoder(STB_DPGetLivePath());
      if (decoder == INVALID_RES_ID)
      {
         result = MHERR_OTHER;
      }
      else
      {
         if (STB_AVLoadAudioSample(decoder, pAudioData, dataLength) != HW_OK)
         {
            result = MHERR_OTHER;
         }
         else
         {
            if (STB_AVPlayAudioSample(decoder, repeatCount) != HW_OK)
            {
               result = MHERR_OTHER;
            }
            else
            {
               STB_ERSendEvent(FALSE, FALSE, EV_CLASS_DECODE, EV_TYPE_SAMPLE_STARTED, NULL, 0);
               result = MHERR_OK;
            }
         }
      }
   }
   FUNCTION_FINISH(DVB_MhegAudioPlayClip);
   return result;
}

/**
 * @brief   Instruct the controlling application that playback of the previously
 *          supplied audio clip should be stopped.
 *          This should not generate a DVB_EVENT_SAMPLE_STOPPED event
 *          If no audio clip currently playing, the call should be silently ignored.
 * @return  - MHERR_OK on success
 *          - MHERR_OTHER controlling application specific error.
 */
E_MhegErr DVB_MhegAudioStopClip(void)
{
   E_MhegErr result;
   U8BIT decoder;
   FUNCTION_START(DVB_MhegAudioStopClip);
   decoder = STB_DPGetPathAudioDecoder(STB_DPGetLivePath());
   if (decoder == INVALID_RES_ID)
   {
      result = MHERR_OTHER;
   }
   else
   {
      result = MHERR_OK;
      STB_AVStopAudioSample(decoder);
   }
   FUNCTION_FINISH(DVB_MhegAudioStopClip);
   return result;
}

/**
 * @brief   Adjust sound level for any external application sourced audio. A
 *          setting of 0 dB means leave the audio volume unchanged. A setting of
 *          greater than 0 dB may be approximated as 0 dB. A setting of less than
 *          0 dB may be approximated as a mute. This affects playback of audio from
 *          stream and audio clips.
 * @param   volumeAdjust New setting of the amount by which to adjust volume in dB
 * @return  - MHERR_OK on success
 *          - MHERR_BAD_PARAMETER invalid parameter
 *          - MHERR_OTHER controlling application specific error.
 */
E_MhegErr DVB_MhegAudioSetVolume(S32BIT volumeAdjust)
{
   FUNCTION_START(DVB_MhegAudioSetVolume);
   if (volumeAdjust < -100)
   {
      volumeAdjust = -100;
   }
   else if (volumeAdjust > 100)
   {
      volumeAdjust = 100;
   }
   ACTL_SetVolumeAdjustment((S8BIT)volumeAdjust);
   FUNCTION_FINISH(DVB_MhegAudioSetVolume);
   return MHERR_OK;
}

/**
 * @brief   Instruct the controlling application that playback of the previously
 *          supplied audio clip should be paused or restarted.
 *          When there is no audio clip currently playing, it should be silently
 *          ignored.
 *          This MUST be a non-blocking function.
 * @param   speed The new speed for audio. As speed can only be 0 or 1.
 * @return  - MHERR_OK on success
 *          - MHERR_BAD_PARAMETER invalid parameter
 *          - MHERR_OTHER controlling application specific error.
 */
E_MhegErr DVB_MhegAudioSetSpeed(E_AudioSpeed speed)
{
   E_MhegErr result;
   U8BIT decoder;
   FUNCTION_START(DVB_MhegAudioSetSpeed);
   decoder = STB_DPGetPathAudioDecoder(STB_DPGetLivePath());
   if (decoder == INVALID_RES_ID)
   {
      result = MHERR_OTHER;
   }
   else
   {
      switch (speed)
      {
         case AUDIO_SPEED_PAUSE:
            if (STB_AVPauseAudioSample(decoder) != HW_OK)
            {
               result = MHERR_OTHER;
            }
            else
            {
               result = MHERR_OK;
            }
            break;

         case AUDIO_SPEED_PLAY:
            if (STB_AVResumeAudioSample(decoder) != HW_OK)
            {
               result = MHERR_OTHER;
            }
            else
            {
               result = MHERR_OK;
            }
            break;
         default:
            result = MHERR_BAD_PARAMETER;
      }
   }
   FUNCTION_FINISH(DVB_MhegAudioSetSpeed);
   return result;
}

/**
 * @brief   Get list of preferred audio languages. If there is no preference
 *          then zero is returned. This only writes into array up to 'max' items
 *          This MUST be a non-blocking function, returning results immediately.
 * @param   langs Array of ISO 639 language codes to be returned
 * @param   max maximum number of preferred languages required by caller
 * @return  U32BIT - number of actual language codes returned
 */
U8BIT DVB_MhegPrefAudioLangs(U32BIT *langs,U8BIT max)
{
   U32BIT country_code;
   U8BIT *ids;
   U8BIT count = 0;
   FUNCTION_START(DVB_MhegPrefAudioLangs);
   if (max != 0)
   {
      country_code = ACFG_GetCountry();
      ids = ACFG_GetDbLangId(country_code,ACFG_GetPrimaryAudioLangId());
      if (ids != NULL)
      {
         *langs = ACFG_ConvertLangIdToCode(ids[0]);
         count++;
      }
      if (count != max)
      {
         ids = ACFG_GetDbLangId(ACFG_GetCountry(),ACFG_GetSecondaryAudioLangId());
         if (ids != NULL)
         {
            langs[count] = ACFG_ConvertLangIdToCode(ids[0]);
            count++;
         }
         if (count != max)
         {
            ids = ACFG_GetDbLangId(country_code,ACFG_GetDefaultSecondaryLangId());
            if (ids != NULL)
            {
               langs[count] = ACFG_ConvertLangIdToCode(ids[0]);
               count++;
            }
         }
      }
   }
   FUNCTION_FINISH(DVB_MhegPrefAudioLangs);
   return count;
}

/**
 * @brief   This function is used by the MHEG-5 engine to determine whether audio
 *          description is currently enabled. This may be used to allow a content
 *          provider to make available streams with and without audio description
 *          to avoid streaming unused data.
 *          This MUST be a non-blocking function.
 * @return  TRUE if the current viewer preferences is for the receiver to use
 *          the audio description soundtrack
 */
BOOLEAN DVB_MhegAudioDescriptionPref(void)
{
   S_ACB_UI_INFO info;
   BOOLEAN result = FALSE;
   FUNCTION_START(DVB_MhegAudioDescriptionPref);
   info.type = ACB_GET_AD_PREF;
   if(ACB_GetUIInformation(&info) == TRUE)
   {
     result = info.u.ad.pref;
   }
   FUNCTION_FINISH(DVB_MhegAudioDescriptionPref);
   return result;
}

