/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2014 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief      DVB Interface for MHEG5 engine - Audio
 * @file       dvbmh_pvr.c
 * @date       Janauary 2015
 * @author     Adam Sturtridge
 */

/*---includes for this file--------------------------------------------------*/

/* compiler library header files */
#include <stdio.h>

/* third party header files */

#include <techtype.h>
#include <dbgfuncs.h>


#include "dvb_pvr.h"

/**
 * @brief   The function adds an event to the PVRs list of scheduled events
 *          to record. The type of CRID can be any of a single programme or series.
 *          The type is defined in crid_type to aid the PVR in finding the required CID.
 *          Where the CRID is a series CRID the booking relates to all programme
 *          events that are part of the series.
 *          The PVR is required to validate the CRID and check that resources are
 *          available for the booking. This may involve searching for multiple instances
 *          of an event until one is found that does not clash with a previous booking.
 *          Where no such instance is found the PVR may choose to indicate the conflict
 *          to the viewer, giving them the option to cancel one or more of the
 *          bookings.
 *          Each 'booking' in 'details' has a logical 'OR' between them. So receiver
 *          should choose one booking. The PVR must be able to record all 'locations'
 *          for the booking to be successful.
 *          The platform implementation of this funtion should only return MHERR_CONFLICT
 *          when it does not ask the user, via the native GUI, for input regarding conflicts.
 *          If it does ask for user input, then it should return MHEG5_RESULT_PENDING,
 *          and subsequently call MHG_NotifyBookingResult()
 * @param   crid Content Reference Identifier string
 * @param   details Pointer to structure with description of Content Reference
 * @return  MHERR_OK                   -  Success,
 *          MHERR_CONFLICT         -  conflict with a previous booking
 *          MHEG5_IGNOR_REQUEST        -  booking cancelled by user
 *          MHEG5_INSUFFICIENT_SPACE   -  booking failed due to insufficient space
 *          MHERR_OVERBOOKED       -  booking failed due to too many bookings
 *          MHERR_OTHER            -  booking failed for other reason
 *          MHEG5_RESULT_PENDING       -  result pending, and will mean a subsequent
 *          call to MHG_NotifyBookingResult()
 */
E_MhegErr DVB_MhegPvrMakeBooking( S_STRING crid, S_CRID_DETAILS *details )
{
   FUNCTION_START(DVB_MhegPvrMakeBooking);
   USE_UNWANTED_PARAM(crid);
   USE_UNWANTED_PARAM(details);
   FUNCTION_FINISH(DVB_MhegPvrMakeBooking);
   return MHERR_OK;
}

/**
 * @brief   Removes an event from the PVR schedule
 * @param   crid Content Reference Identifier string
 * @return  MHERR_OK                   - Success, booking removed
 *          MHERR_FILE_NOT_FOUND   - Booking not found
 *          MHERR_OTHER            - removal failed for other reason
 */
E_MhegErr DVB_MhegPvrCancelBooking( S_STRING crid, U16BIT crid_type )
{
   FUNCTION_START(DVB_MhegPvrCancelBooking);
   USE_UNWANTED_PARAM(crid);
   USE_UNWANTED_PARAM(crid_type);
   FUNCTION_FINISH(DVB_MhegPvrCancelBooking);
   return MHERR_OK;
}

/**
 * @brief   Returns an array of CRIDs (and CRID types for NZ profile) that are
 *          being monitored by the PVR.
 * @param   number Number of Content References in array
 * @return  Pointer to array of S_CRID_REC,
 *          or NULL if there are no bookings
 */
S_CRID_REC* DVB_MhegPvrListBookings( U16BIT *number )
{
   FUNCTION_START(DVB_MhegPvrListBookings);
   USE_UNWANTED_PARAM(number);
   FUNCTION_FINISH(DVB_MhegPvrListBookings);
   return NULL;
}

/**
 * @brief   MHEG indicates that it is finished using the array returned by
 *          DVB_MhegPvrListBookings().
 *          For example, platforms that malloc memory for this, may wish to free
 *          the memory here.
 * @param   crid_array Array of Content Reference Identifier's
 * @return  void
 */
void DVB_MhegPvrReleaseList( S_CRID_REC *crid_array )
{
   FUNCTION_START(DVB_MhegPvrReleaseList);
   USE_UNWANTED_PARAM(crid_array);
   FUNCTION_FINISH(DVB_MhegPvrReleaseList);
}

/**
 * @brief   Required by Australia profile only.
 *          Returns the type, name, description and a single booking section only
 *          containing location details required to resolve the booking.
 * @param   crid Content Reference Identifier string
 * @return  Pointer to S_CRID_DETAILS, if success
 *          NULL, if CRID not found
 */
S_CRID_DETAILS* DVB_MhegPvrGetDetails( S_STRING crid )
{
   FUNCTION_START(DVB_MhegPvrGetDetails);
   USE_UNWANTED_PARAM(crid);
   FUNCTION_FINISH(DVB_MhegPvrGetDetails);
   return NULL;
}

/**
 * @brief   Required by Australia profile only.
 *          MHEG indicates that it is finished using the CRID details returned by
 *          DVB_MhegPvrGetDetails(). For platforms that malloc memory for this, it is a
 *          chance to free the memory.
 * @param   details Pointer of Content Reference details structure
 * @return  void
 */
void DVB_MhegPvrReleaseDetails( S_CRID_DETAILS *details )
{
   FUNCTION_START(DVB_MhegPvrReleaseDetails);
   USE_UNWANTED_PARAM(details);
   FUNCTION_FINISH(DVB_MhegPvrReleaseDetails);
}

