/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2014 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief      DVB Interface for MHEG5 engine - Audio
 * @file       misc.c
 * @date       Janauary 2014
 * @author     Adam Sturtridge
 */

/*---includes for this file--------------------------------------------------*/

/* compiler library header files */
#include <stdio.h>

/* third party header files */

#include <techtype.h>
#include <dbgfuncs.h>

#include "stbgc.h"
#include "stberc.h"
#include "app.h"
#include "ap_cfg.h"
#include "ap_uiinfo.h"

#include "dvb_misc.h"
#include "dvb_ci.h"


/*---constant definitions for this file--------------------------------------*/

/*---local typedef structs for this file-------------------------------------*/

/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local)                 */

/*---local function prototypes for this file---------------------------------*/
/*   (internal functions declared static to make them local)                 */

/*---local function definitions----------------------------------------------*/

/*!**************************************************************************
 * @fn       ExtractDate
 * @brief    Extract date (year, month, day) from Modified Julian Date.
 * @param    U16BIT      code                 Modified Julian Date
 *           U32BIT *    year                 Year
 *           U32BIT *    month                Month
 *           U32BIT *    day                  Day of the month
 * @return   None.
 * @warning  Julian date is 16 bits (wraps around on 22 April 2038).
 * @bug      None.
 ****************************************************************************/
static void ExtractDate(U16BIT code, U32BIT *year, U32BIT *month, U32BIT *day)
{
   FUNCTION_START(ExtractDate);

   if (day != NULL && month != NULL && year != NULL)
   {
      S32BIT p, q, r, s, t, u, v;

      p = code + 2468570;
      q = 4 * p / 146097;
      r = p - (146097 * q + 3) / 4;
      s = 4000 * (r + 1) / 1461001;
      t = r - 1461 * s / 4 + 31;
      u = 80 * t / 2447;
      v = u / 11;

      *year = 100 * (q - 49) + s + v;
      *month = u + 2 - 12 * v;
      *day = t - 2447 * u / 80;
   }

   FUNCTION_FINISH(ExtractDate);
}

/*---global function definitions---------------------------------------------*/

/**
 * @brief   This function requests exclusive access to a MPEG decoder and video
 *          plane to display I-frames. The MPEG decoder shall be available when no
 *          other application environment is active.
 * @param   request MHEG5_DECODER_REQUEST if access to the MPEG decoder is
 *          requested
 *          MHEG5_DECODER_RELEASE if access to the MPEG decoder is
 *          released
 * @param   pResult TRUE if access to the MPEG decoder is granted
 *          FALSE if access to the MPEG decoder is not granted,
 *          or if request is MHEG5_DECODER_RELEASE
 * @return  The following return values are allowed from this function:
 *          MHERR_OK                - Success.
 *          MHERR_BAD_PARAMETER - Invalid parameter.
 *          MHERR_OTHER         - Controlling application specific error.
 */
E_MhegErr DVB_MhegRequestMPEGDecoder(E_DecoderRequest request, BOOLEAN *pResult)
{
   USE_UNWANTED_PARAM(request);
   *pResult = TRUE;
   return MHERR_OK;
}


/**
 * @brief   Toggle between displaying MHEG graphics and subtitles on platforms that
 *          do not support them both symultaneously, while kepping the MHEG
 *          application running. Platforms that can display both symultaneously may
 *          ignore calls to this function.
 * @param   suppress TRUE: Suppress the MHEG Graphics and enable the subtitles.
 *          FALSE: Enable the MHEG Graphics and suppress subtitles.
 * @return  The following return values are allowed from this function:
 *          MHERR_OK            - Success.
 *          MHERR_BAD_PARAMETER - Invalid parameter.
 *          MHERR_OTHER         - Controlling application specific error.
 */
E_MhegErr DVB_MhegSuppressMHEGGraphics(BOOLEAN suppress)
{
   /*nothing required on platforms that can support MHEG and subtitles simultaneously*/
   USE_UNWANTED_PARAM(suppress);
   return MHERR_OK;
}

/**
 * @brief   This function is relevant to PVR products that support
 *          NativeApplicationExtension for broadcast-triggered native applications,
 *          or "Promotional Linking". See sections 8.5.11 and  13.6.4 of reference [1].
 *
 *          The state can be changed by MHEG application at any time (eg. while
 *          promotional link icon is being displayed).
 *          The PVR product must remember this state. It can only be changed by another
 *          call to this function from Voyager.
 *
 *          If the external application is displaying any promotional linking and the
 *          state is set to FALSE, then this must be removed from the screen.
 *
 *          Also, the state indicates whether MHEG group 3 keys (CANCEL, RED, GREEN,
 *          YELLOW, BLUE, TEXT) may be intercepted by the external application for
 *          promotional linking. Only when the state is set to TRUE, can group 3 keys
 *          be used for broadcast-triggered native application.
 *          If the key is not consumed for this purpose, then it must be presented
 *          to VGR_NotifyKeyPress(), as normal.
 *
 * @param   isEnabled TRUE  - Promotional linking is allowed
 *                    FALSE - disable promotional linking.
 *
 */
void DVB_MhegPromotionalLinkControl( BOOLEAN isEnabled )
{
   FUNCTION_START(DVB_MhegPromotionalLinkControl);
   USE_UNWANTED_PARAM(isEnabled);

   STB_ERSendEvent(FALSE, FALSE, EV_CLASS_MHEG, EV_TYPE_MHEG_PROMO_LINK_CHANGE, NULL, 0);

   FUNCTION_FINISH(DVB_MhegPromotionalLinkControl);
}

/**
 * @brief   Provide the current local time and date, normally from the system
 *          real time clock, with any local time conversions (if necessary). The
 *          returned time should take into account local timezone and daylight
 *          saving settings.
 *
 *          This MUST be a non-blocking function, returning results immediately.
 *
 * @para    pDateAndTime  Pointer to return current local time and date
 */
E_MhegErr DVB_MhegGetLocalTime(S_DateTime *pDateAndTime)
{
   E_MhegErr result;
   U16BIT code;
   U8BIT hour;
   U8BIT min;
   U8BIT secs;

   FUNCTION_START(DVB_MhegGetLocalTime);
   if (pDateAndTime != NULL)
   {
      /* Get GMT time */
      STB_GCGetGMTDateTime(&code, &hour, &min, &secs);

   #ifdef DEBUG_PRINTING_ENABLED
      ExtractDate(code, &pDateAndTime->year, &pDateAndTime->month, &pDateAndTime->day);
      DBGPRINT(" GMT  day=%d-%d-%d time=%d:%d:%d", pDateAndTime->day, pDateAndTime->month, pDateAndTime->year, hour, min, secs);
   #endif

      /* Convert to local time */
      STB_GCConvertDateTime(code, hour, min, secs, &code, &hour, &min, &secs, CONV_LOCAL);

      pDateAndTime->hour = hour;
      pDateAndTime->minute = min;
      pDateAndTime->second = secs;

      /* Extract day, month and year */
      ExtractDate(code, &pDateAndTime->year, &pDateAndTime->month, &pDateAndTime->day);

      DBGPRINT("LOCAL day=%d-%d-%d time=%d:%d:%d", pDateAndTime->day, pDateAndTime->month, pDateAndTime->year, hour, min, secs);
   }

   result = MHERR_OK;

   FUNCTION_FINISH(DVB_MhegGetLocalTime);

   return result;
}

/**
 * @brief   Return the language of the native UI (menu etc.) as a list of language codes
 *          in 32 bit integer form. This may include alternative language codes which refer
 *          to same language (e.g. 'wel' and 'cym' which give 0x)
 * @param   language pointer to array of language codes
 * @param   max max number of language codes
 * @return  The number of language code 32 bit values
 */
U8BIT DVB_MhegOsdLanguageCodes(U32BIT *langcodes, U8BIT max)
{
   S_ACB_UI_INFO info;
   U8BIT *lang_ids;
   U8BIT count = 0;

   FUNCTION_START(DVB_MhegOsdLanguageCodes);
   info.type = ACB_GET_UI_LANG_PREF;
   if(ACB_GetUIInformation(&info) == TRUE)
   {
      lang_ids = ACFG_GetDbLangId(ACFG_GetCountry(),info.u.ui_lang_id);
      if (lang_ids != NULL)
      {
         for (; count != max; count++, langcodes++)
         {
            *langcodes = ACFG_ConvertLangIdToCode(lang_ids[count]);
            DBGPRINT("langcode=%x",*langcodes)
            if (*langcodes == 0)
            {
               break;
            }
         }
      }
   }
   FUNCTION_FINISH(DVB_MhegOsdLanguageCodes);
   return count;
}
