/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2014 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief      DVB Interface for MHEG5 engine - Tuner
 * @file       service.c
 * @date       Janauary 2014
 * @author     Adam Sturtridge
 */
//#define DEBUG_PRINTING_ENABLED

/*---includes for this file--------------------------------------------------*/

/* compiler library header files */
#include <stdio.h>
#include <string.h>

/* third party header files */

#include <techtype.h>
#include <dbgfuncs.h>
#include "stberc.h"
#include "ap_dbacc.h"
#include "ap_tmr.h"
#include "app.h"
#include "ap_cfg.h"
#include "ap_uiinfo.h"
#include "ap_si.h"
#if defined(COMMON_INTERFACE)
#include "ap_ci.h"
#endif

#define INCLUDE_SI_EXTENSION
#include "dvb_service.h"
#include "dvb_native.h"
#include "mheg5_keypress.h"
#include "stbheap.h"
#include "stbuni.h"

/*---constant definitions for this file--------------------------------------*/

#define HEX(x) (((x) > 9) ? 'A' + (x) - 10 : '0' + (x))

/* Should probably be ((x) >= 1 && (x) <= 1023), but at least check doesn't overflow U16BIT */
#define CHECK_LCN(x) ((x) < 65536)

/* Difference in days between MJD starting on 1858-11-17 to Unix 1970-01-01 */
#define ADJUST_MJD_TO_1970 40587

/* Number of seconds in a day */
#define SECONDS_PER_MINUTE 60
#define SECONDS_PER_HOUR (60 * SECONDS_PER_MINUTE)
#define SECONDS_PER_DAY (24 * SECONDS_PER_HOUR)

/*---local typedef structs for this file-------------------------------------*/

/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local)                 */

/*---local function prototypes for this file---------------------------------*/
/*   (internal functions declared static to make them local)                 */

#if defined(COMMON_INTERFACE)
static F_NotifyDvbTuning notify_tuning_func = NULL;
#endif

/*---local function definitions----------------------------------------------*/

/*---global function definitions---------------------------------------------*/


/**
 * @brief   Retrieve demux resource reference
 * @return  DMXREF demux resource reference
 */
DMXREF DVB_MhegGetDemuxPath(void)
{
   U8BIT path;
   DMXREF dmxref;
   FUNCTION_START(DVB_MhegGetDemuxPath);
   path = STB_DPGetLivePath();
   if (path == INVALID_RES_ID)
   {
      dmxref = INVALID_DMXREF;
   }
   else
   {
      dmxref = STB_DPGetPathDemux(path);
   }
   FUNCTION_FINISH(DVB_MhegGetDemuxPath);
   return dmxref;
}

/**
 * @brief   Retrieve the "service index" of the current tuned service.
 * @return  - MHERR_OK on success
 *          - MHERR_BAD_PARAMETER invalid parameter
 *          - MHERR_OTHER controlling application specific error.
 */
E_MhegErr DVB_MhegGetTunedService(S32BIT *serviceIndex)
{
   E_MhegErr result;
   void *srv_ptr;
   U8BIT path;
   FUNCTION_START(DVB_MhegGetTunedService);
   path = STB_DPGetLivePath();
   if (path == INVALID_RES_ID)
   {
      result = MHERR_OTHER;
   }
   else
   {
      srv_ptr = STB_DPGetTunedService(path);
      if (srv_ptr == NULL)
      {
         result = MHERR_OTHER;
      }
      else
      {
         *serviceIndex = (S32BIT)ADB_GetServiceLcn(srv_ptr);
         result = MHERR_OK;
      }
   }
   FUNCTION_FINISH(DVB_MhegGetTunedService);
   return result;
}

/**
 * @brief   Get a DVB implementation dependant service index - an integer greater
 *          or equal to 0. This may be the Logical Channel Number.
 *          It is used to reference a service, based on the DVB URL (S_DvbLocation).
 *          Service availability is determined by the description of currently
 *          running services provided by the SI in the currently selected TS.
 *          This MUST be a non-blocking function, returning results immediately.
 * @param   pDvbLocator pointer to Digital Video Broadcaster (DVB) locator information.
 * @param   pServiceIndex pointer to service index. Set to -1 if service not found
 * @return  - MHERR_OK on success
 *          - MHERR_BAD_PARAMETER invalid parameter
 *          - MHERR_OTHER controlling application specific error.
 */
E_MhegErr DVB_MhegDvbLocatorToIndex(S_DvbLocator *pDvbLocator, S32BIT *pServiceIndex)
{
   E_MhegErr result;
   U16BIT tid;
   void *srv_ptr;

   FUNCTION_START(DVB_MhegDvbLocatorToIndex);

   if (pDvbLocator == NULL || pServiceIndex == NULL)
   {
      DBGPRINT("ERROR: bad param")
      result = MHERR_BAD_PARAMETER;
   }
   else
   {
      tid = pDvbLocator->transport_stream_id;
      if (tid == 0)
      {
         tid = ADB_INVALID_DVB_ID;
      }
      srv_ptr = ADB_FindServiceByIds(pDvbLocator->original_network_id, tid, pDvbLocator->service_id);
      if (srv_ptr == NULL)
      {
         DBGPRINT("ERROR: Failed to find service 0x%04x/0x%04x/0x%04x",
            pDvbLocator->original_network_id, tid, pDvbLocator->service_id)
         result = MHERR_OTHER;
      }
      else
      {
         *pServiceIndex = (S32BIT)ADB_GetServiceLcn(srv_ptr);
         result = MHERR_OK;
      }
   }
   FUNCTION_FINISH(DVB_MhegDvbLocatorToIndex);
   return result;
}

/**
 * @brief   Convert "service index" into DVB locator information. This being
 *          original network ID, transport stream ID, and service ID
 * @param   serviceIndex service index
 * @param   location pointer for returning DVB location
 * @return  - MHERR_OK on success
 *          - MHERR_BAD_PARAMETER invalid parameter
 *          - MHERR_OTHER controlling application specific error.
 */
E_MhegErr DVB_MhegIndexToDvbLocator( S32BIT serviceIndex, S_DvbLocator *locator )
{
   void *srv_ptr;
   E_MhegErr result = MHERR_BAD_PARAMETER;
   FUNCTION_START(DVB_MhegLcnToDvbLocator);

   if (CHECK_LCN(serviceIndex))
   {
      srv_ptr = ADB_FindServiceByLcn(ADB_SERVICE_LIST_ALL, (U16BIT)serviceIndex, TRUE);
      if (srv_ptr != NULL)
      {
         ADB_GetServiceIds(srv_ptr, &(locator->original_network_id),
            &(locator->transport_stream_id), &(locator->service_id));
         result = MHERR_OK;
      }
   }

   FUNCTION_FINISH(DVB_MhegLcnToDvbLocator);
   return result;
}

/**
 * @brief   Convert Service index into full DVB location information. This being
 *          original network ID, transport stream ID, service ID, and network ID
 * @param   serviceIndex service index
 * @param   location pointer for returning DVB location
 * @return  - MHERR_OK on success
 *          - MHERR_BAD_PARAMETER invalid parameter
 *          - MHERR_OTHER controlling application specific error.
 */
E_MhegErr DVB_MhegIndexToDvbLocation( S32BIT serviceIndex, S_DvbLocation *location )
{
   void *srv_ptr;
   E_MhegErr result = MHERR_BAD_PARAMETER;
   FUNCTION_START(DVB_MhegLcnToDvbLocation);

   if (CHECK_LCN(serviceIndex))
   {
      srv_ptr = ADB_FindServiceByLcn(ADB_SERVICE_LIST_ALL, (U16BIT)serviceIndex, TRUE);
      if (srv_ptr != NULL)
      {
         ADB_GetServiceIds(srv_ptr, &(location->original_network_id),
            &(location->transport_stream_id), &(location->service_id));
         location->network_id = ADB_GetNetworkId(ADB_GetTransportNetworkPtr(ADB_GetServiceTransportPtr(srv_ptr)));
         result = MHERR_OK;
      }
   }

   FUNCTION_FINISH(DVB_MhegLcnToDvbLocation);
   return result;
}

/**
 * @brief   Convert Logical channel number into DVB locator information.
 * @param   lcn Logical channel number
 * @param   locator pointer for returning DVB locator
 * @return  - MHERR_OK on success
 *          - MHERR_BAD_PARAMETER invalid parameter
 *          - MHERR_OTHER controlling application specific error.
 */
E_MhegErr DVB_MhegLcnToDvbLocator( U32BIT lcn, S_DvbLocator *locator)
{
   E_MhegErr result = MHERR_BAD_PARAMETER;
   void *srv_ptr;
   FUNCTION_START(DVB_MhegLcnToDvbLocator);

   if (CHECK_LCN(lcn))
   {
      srv_ptr = ADB_FindServiceByLcn(ADB_SERVICE_LIST_ALL, (U16BIT)lcn, TRUE);
      if (srv_ptr != NULL)
      {
         result = MHERR_OK;
         ADB_GetServiceIds(srv_ptr, &(locator->original_network_id),
            &(locator->transport_stream_id), &(locator->service_id));
      }
   }

   FUNCTION_FINISH(DVB_MhegLcnToDvbLocator);
   return result;
}

/**
 * @brief   Tunes to the specified service. This MUST be a non-blocking function
 *          If the tuner fails to tune to the service it should attempt to return
 *          to the previously tuned service. If tuning to previously tuned service
 *          fails, the receiver shall allow the user to choose another service.
 *          When tuning to a new service, MHEG5 must first be stopped.
 *          Once a service has been tuned, then MHEG5 may be restarted.
 * @param   serviceIndex service index to which the tuner should attempt to tune
 * @param   tuneQuietly
 * @return  - MHERR_OK on success
 *          - MHERR_BAD_PARAMETER invalid parameter
 *          - MHERR_OTHER controlling application specific error.
 */
E_MhegErr DVB_MhegTuneIndex(S32BIT serviceIndex)
{
   E_MhegErr result = MHERR_BAD_PARAMETER;
   void *srv_ptr;
   FUNCTION_START(DVB_MhegTuneIndex);

   if (CHECK_LCN(serviceIndex))
   {
      srv_ptr = ADB_FindServiceByLcn(ADB_SERVICE_LIST_ALL, (U16BIT)serviceIndex, TRUE);
      if (srv_ptr != NULL)
      {
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_MHEG, EV_TYPE_MHEG_TUNE_INDEX, &srv_ptr, sizeof(void *));
         result = MHERR_OK;
      }
   }

   FUNCTION_FINISH(DVB_MhegTuneIndex);
   return result;
}

/**
 * @brief   Retrieves information about the indicated service.
 *          (See ETSI MHEG-5 Broadcast Profile, ES 202 184)
 *          Provided MHERR_OK is returned, the engine will call DVB_MhegReleaseServiceDetails
 *          to release allocated data on the same task and immediately after the engine has
 *          copied the data.
 * @param   serviceIndex Receiver dependent identifier for the required service
 * @param   details Pointer to S_ServiceDetails structure to place required
 *          service information (this will NOT be NULL)
 * @return  MHERR_OK on success
 *          MHERR_BAD_PARAMETER if serviceIndex is invalid
 *          MHERR_OTHER otherwise.
 */
E_MhegErr DVB_MhegGetServiceDetails( S32BIT serviceIndex, S_ServiceDetails *details )
{
   E_MhegErr result = MHERR_BAD_PARAMETER;
   void *srv_ptr;
   U8BIT *dvbServiceName;
   U8BIT *dvbServiceProvider;

   FUNCTION_START(DVB_MhegGetServiceDetails);

   memset(details, 0, sizeof(*details));

   if (CHECK_LCN(serviceIndex))
   {
      srv_ptr = ADB_FindServiceByLcn(ADB_SERVICE_LIST_ALL, serviceIndex, TRUE);
      if (srv_ptr != NULL)
      {
         details->serviceIndex = serviceIndex;

         dvbServiceName = ADB_GetServiceFullName(srv_ptr, TRUE);
         if (dvbServiceName)
         {
            details->serviceName.zptr = dvbServiceName;

            /* Get the length of the (UTF8) octetstring */
            details->serviceName.zlen = STB_GetNumBytesInString(details->serviceName.zptr) - 1;

            if (details->serviceName.zlen <= 1)
            {
               STB_ReleaseUnicodeString( details->serviceName.zptr );
               details->serviceName.zptr = NULL;
               details->serviceName.zlen = 0;
            }
            else
            {
               /* Skip the Unicode start character, and adjust length */
               details->serviceName.zptr++;
               details->serviceName.zlen--;
            }
         }

         dvbServiceProvider = ADB_GetServiceProviderName(srv_ptr);
         if (dvbServiceProvider)
         {
            details->serviceProvider.zptr = dvbServiceProvider;

            /* Get the length of the (UTF8) octetstring */
            details->serviceProvider.zlen = STB_GetNumBytesInString(details->serviceProvider.zptr) - 1;

            if (details->serviceProvider.zlen <= 1)
            {
               STB_ReleaseUnicodeString( details->serviceProvider.zptr );
               details->serviceProvider.zptr = NULL;
               details->serviceProvider.zlen = 0;
            }
            else
            {
               /* Skip the Unicode start character, and adjust length */
               details->serviceProvider.zptr++;
               details->serviceProvider.zlen--;
            }
         }

         details->serviceType = ADB_GetServiceType(srv_ptr);
         details->serviceLcn = serviceIndex;

         result = MHERR_OK;
      }
   }

   FUNCTION_FINISH(DVB_MhegGetServiceDetails);

   return result;
}

/**
 * @brief   Release data allocated by DVB_MhegGetServiceDetails (e.g. String data).
 *          Always called immediately after the engine has copied data.
 *          DVB stack may have other data management (using dvb_user_data)
 * @param   details Pointer to S_ServiceDetails
 * @return  void
 */
void DVB_MhegReleaseServiceDetails( S_ServiceDetails *details )
{
   FUNCTION_START(DVB_MhegReleaseServiceDetails);

   if (details->serviceName.zptr != NULL)
   {
      /* Adjust pointer back to original position */
      details->serviceName.zptr--;
      details->serviceName.zlen++;

      STB_ReleaseUnicodeString(details->serviceName.zptr);

      details->serviceName.zptr = NULL;
      details->serviceName.zlen = 0;
   }

   if (details->serviceProvider.zptr != NULL)
   {
      /* Adjust pointer back to original position */
      details->serviceProvider.zptr--;
      details->serviceProvider.zlen++;

      STB_ReleaseUnicodeString(details->serviceProvider.zptr);

      details->serviceProvider.zptr = NULL;
      details->serviceProvider.zlen = 0;
   }

   FUNCTION_FINISH(DVB_MhegReleaseServiceDetails);
}

/**
 * @brief   Return the number of services in the service list.
 *          This MUST be a non-blocking function, returning results immediately.
 * @param   number The number of services in the service list. The
 *          returned number shall only include visible services,
 *          i.e. services with the visible_service_flag in the
 *          service_availability_descriptor set to 1.
 * @return  The following return values are allowed from this function
 *          MHERR_OK             - Success
 *          MHERR_BAD_PARAMETER  - Invalid parameter.
 *          MHERR_OTHER          - Controlling application specific
 *          error.
 */
E_MhegErr DVB_MhegGetNumberOfServices(S32BIT *number)
{
   FUNCTION_START(DVB_MhegGetNumberOfServices);

   *number = (S32BIT) ADB_GetNumServicesInList(ADB_SERVICE_LIST_ALL, FALSE);

   FUNCTION_FINISH(DVB_MhegGetNumberOfServices);

   return MHERR_OK;
}

/**
 * @brief   Return the index of the next service in the service list.
 *          The input of this function is the index of the service from which
 *          to determine the next service index.
 *          The returned value is the index of the next visible service in the
 *          service list. The receiver shall skip services with the
 *          visible_service_flag in the service_availability_descriptor set to 0.
 *          If the input value of index is -1, the index of the first visible
 *          service in the service list is returned.
 *          If the input value of index is the last service in the list then -1
 *          is returned.
 *          This MUST be a non-blocking function, returning results immediately.
 * @param   serviceIndex Input: current service index
 *          Output: next service index
 * @param   type Service type - TV, Radio, or any
 * @return  The following return values are allowed from this function
 *          MHERR_OK             - Success
 *          MHERR_BAD_PARAMETER  - Invalid parameter.
 *          MHERR_OTHER          - Controlling application specific
 *          error.
 */
E_MhegErr DVB_MhegNextServiceIndex(S32BIT *serviceIndex, E_ServiceType type)
{
   void *srv_ptr;
   void *next = NULL;
   ADB_SERVICE_LIST_TYPE adb_serv_type;

   FUNCTION_START(DVB_MhegNextServiceIndex);

   switch (type)
   {
      default:
      case SRV_TYP_ALL:     adb_serv_type = ADB_SERVICE_LIST_ALL; break;
      case SRV_TYP_TV_DATA: adb_serv_type = ADB_SERVICE_LIST_TV_DATA; break;
      case SRV_TYP_RADIO:   adb_serv_type = ADB_SERVICE_LIST_RADIO; break;
   }

   if (CHECK_LCN(*serviceIndex))
   {
      srv_ptr = ADB_FindServiceByLcn(adb_serv_type, (U16BIT) *serviceIndex, TRUE);
      next = ADB_GetNextServiceInList(adb_serv_type, srv_ptr);
   }
   else
   {
      if (*serviceIndex == -1)
      {
         next = ADB_GetNextServiceInList(adb_serv_type, NULL);
      }
   }

   /* Skip hidden services */
   while (next != NULL && ADB_GetServiceHiddenFlag(next))
   {
      next = ADB_GetNextServiceInList(adb_serv_type, next);
   }

   if (next != NULL)
   {
      *serviceIndex = (S32BIT)ADB_GetServiceLcn(next);
   }
   else
   {
      *serviceIndex = -1;
   }

   FUNCTION_FINISH(DVB_MhegNextServiceIndex);

   return MHERR_OK;
}

/**
 * @brief   Return the index of the previous service in the service list.
 *          The input of this function is the index of the service from which
 *          to determine the previous service index.
 *          The returned value is the index of the previous visible service in the
 *          service list. The receiver shall skip services with the
 *          visible_service_flag in the service_availability_descriptor set to 0.
 *          If the input value of index is -1, the index of the last visible
 *          service in the service list is returned.
 *          If the input value of index is the first service in the list then -1
 *          is returned.
 *          This MUST be a non-blocking function, returning results immediately.
 * @param   serviceIndex Input: current service index
 *          Output: next service index
 * @param   type Service type - TV, Radio, or any
 * @return  The following return values are allowed from this function
 *          MHERR_OK             - Success
 *          MHERR_BAD_PARAMETER  - Invalid parameter.
 *          MHERR_OTHER          - Controlling application specific
 *          error.
 */
E_MhegErr DVB_MhegPrevServiceIndex(S32BIT *serviceIndex, E_ServiceType type)
{
   void *srv_ptr;
   void *prev = NULL;
   ADB_SERVICE_LIST_TYPE adb_serv_type;

   FUNCTION_START(DVB_MhegPrevServiceIndex);

   switch (type)
   {
      default:
      case SRV_TYP_ALL:     adb_serv_type = ADB_SERVICE_LIST_ALL; break;
      case SRV_TYP_TV_DATA: adb_serv_type = ADB_SERVICE_LIST_TV_DATA; break;
      case SRV_TYP_RADIO:   adb_serv_type = ADB_SERVICE_LIST_RADIO; break;
   }

   if (CHECK_LCN(*serviceIndex))
   {
      srv_ptr = ADB_FindServiceByLcn(adb_serv_type, (U16BIT) *serviceIndex, TRUE);
      prev = ADB_GetPrevServiceInList(adb_serv_type, srv_ptr);
   }
   else
   {
      if (*serviceIndex == -1)
      {
         prev = ADB_GetPrevServiceInList(adb_serv_type, NULL);
      }
   }

   /* Skip hidden services */
   while (prev != NULL && ADB_GetServiceHiddenFlag(prev))
   {
      prev = ADB_GetPrevServiceInList(adb_serv_type, prev);
   }

   if (prev != NULL)
   {
      *serviceIndex = (S32BIT)ADB_GetServiceLcn(prev);
   }
   else
   {
      *serviceIndex = -1;
   }

   FUNCTION_FINISH(DVB_MhegPrevServiceIndex);

   return MHERR_OK;
}

/**
 * @brief   Return the running status of the specified service.
 *          This MUST be a non-blocking function, returning results immediately.
 * @param   serviceIndex The index of the service to return the running status.
 * @param   runningStatus The running status of the service as indicated by the
 *          running_status field in the SDT.
 * @return  The following return values are allowed from this function
 *          MHERR_OK             - Success
 *          MHERR_BAD_PARAMETER  - Invalid parameter.
 *          MHERR_OTHER          - Controlling application specific
 *          error.
 */
E_MhegErr DVB_MhegGetServiceRunningStatus(S32BIT serviceIndex, U8BIT *runningStatus)
{
   E_MhegErr result = MHERR_OTHER;
   void *srv_ptr;

   FUNCTION_START(DVB_MhegGetServiceRunningStatus);

   if (CHECK_LCN(serviceIndex))
   {
      srv_ptr = ADB_FindServiceByLcn(ADB_SERVICE_LIST_ALL, serviceIndex, TRUE);
      if (srv_ptr != NULL)
      {
         *runningStatus = ADB_GetServiceRunningStatus(srv_ptr);
         result = MHERR_OK;
      }
   }

   FUNCTION_FINISH(DVB_MhegGetServiceRunningStatus);

   return result;
}

/**
 * @brief   Return the channel number assigned by the receiver to the specified
 *          service.
 *          This MUST be a non-blocking function, returning results immediately.
 * @param   serviceIndex The index of the service to return the channel number.
 * @param   channelNum The channel number assigned to the service by the
 *          receiver.
 * @return  The following return values are allowed from this function
 *          MHERR_OK             - Success
 *          MHERR_BAD_PARAMETER  - Invalid parameter.
 *          MHERR_OTHER          - Controlling application specific
 *          error.
 */
E_MhegErr DVB_MhegGetChannelNumber(S32BIT serviceIndex, S32BIT *channelNum)
{
   E_MhegErr result = MHERR_OTHER;
   void *srv_ptr;

   FUNCTION_START(DVB_MhegGetChannelNumber);

   if (CHECK_LCN(serviceIndex))
   {
      srv_ptr = ADB_FindServiceByLcn(ADB_SERVICE_LIST_ALL, (U16BIT) serviceIndex, TRUE);
      if (srv_ptr != NULL)
      {
         *channelNum = serviceIndex;
         result = MHERR_OK;
      }
   }

   FUNCTION_FINISH(DVB_MhegGetChannelNumber);

   return result;
}

/**
 * @brief Create a single S32BIT number that encapsulates the serviceIndex and eventId
 * @param serviceIndex value to be encapsaulted in the S32BIT number
 * @param eventId value to be encapsaulted in the S32BIT number
 * @return Combination of the serviceIndex and eventId
 *
 * @note Value -1 should be impossible to return, as it is used to signal an unavailable event.
 */
static S32BIT MakeMHEGEventId(U16BIT serviceIndex, U16BIT eventId)
{
   S32BIT mhegEventId;

   FUNCTION_START(MakeMHEGEventId);

   mhegEventId = (S32BIT) ((((U32BIT) serviceIndex) << 16) | ((U32BIT) eventId));

   /* Assume serviceIndex is "too small" to form 0xffffffff, i.e. -1 */

   FUNCTION_FINISH(MakeMHEGEventId);

   return mhegEventId;
}

/**
 * @brief Recover the serviceIndex and eventId from a single S32BIT number as derived using MakeMHEGEventId()
 * @param mhegEventId a value encapulating both a serviceIndex and an eventId (or -1 if invalid)
 * @param serviceIndex pointer to recover the value from mhegEventId
 * @param eventId pointer to recover the value from mhegEventId
 *
 * @return TRUE if serviceIndex and eventId could be successfully recovered, FALSE otherwise.
 */
static BOOLEAN GetValuesFromMHEGEventId(S32BIT mhegEventId, U16BIT *serviceIndex, U16BIT *eventId)
{
   BOOLEAN success = FALSE;

   FUNCTION_START(GetValuesFromMHEGEventId);

   if (mhegEventId != -1)
   {
      *serviceIndex = (U16BIT) (((U32BIT) mhegEventId) >> 16);
      *eventId = (U16BIT) (((U32BIT) mhegEventId) & 0x0000ffff);

      success = TRUE;
   }

   FUNCTION_FINISH(GetValuesFromMHEGEventId);

   return success;
}

/**
 * @brief Recover the service and event structures from given serviceIndex and eventId values.
 * @param mhegEventId a value encapulating both a serviceIndex and an eventId (or -1 if invalid)
 * @param pservice pointer to recover the service pointer based on serviceIndex derived from mhegEventId
 * @param pevent pointer to recover the event pointer based on eventId derived from mhegEventId
 *
 * @return TRUE if the service and event could be successfully recovered, FALSE otherwise.
 */
static BOOLEAN GetServiceEventFromMHEGEventId(U16BIT serviceIndex, S32BIT mhegEventId, void **pservice, void **pevent)
{
   BOOLEAN success = FALSE;
   U16BIT eventServiceIndex;
   U16BIT eventId;
   void *srv_ptr;
   void *event_ptr;

   if (GetValuesFromMHEGEventId(mhegEventId, &eventServiceIndex, &eventId) == TRUE)
   {
      /* reinstate if make stronger if(CHECK_LCN(serviceIndex)) */
      {
         if (eventServiceIndex == serviceIndex)
         {
            srv_ptr = ADB_FindServiceByLcn(ADB_SERVICE_LIST_ALL, (U16BIT) eventServiceIndex, TRUE);
            if (srv_ptr != NULL)
            {
               event_ptr = ADB_GetEvent(srv_ptr, eventId);
               if (event_ptr != NULL)
               {
                  *pservice = srv_ptr;
                  *pevent = event_ptr;
                  success = TRUE;
               }
            }
         }
      }
   }

   return success;
}

/**
 * @brief Recover the present or following event for a specific serviceIndex.
 * @param serviceIndex service index
 * @param porf FALSE if recover present EIT event, TRUE if recover following EIT event.
 *
 * @return Non-NULL event if it could be successfully recovered, NULL otherwise.
 */
static void* GetPorfEvent( S32BIT serviceIndex, BOOLEAN porf )
{
   void *srv_ptr;
   void *found = NULL;

   FUNCTION_START(GetPorfEvent);

   if (CHECK_LCN(serviceIndex))
   {
      srv_ptr = ADB_FindServiceByLcn(ADB_SERVICE_LIST_ALL, serviceIndex, TRUE);
      if (srv_ptr != NULL)
      {
         if (porf == FALSE)
         {
            ADB_GetNowNextEvents(srv_ptr, &found, NULL);
         }
         else
         {
            ADB_GetNowNextEvents(srv_ptr, NULL, &found);
         }
      }
   }

   FUNCTION_FINISH(GetPorfEvent);

   return found;
}

/**
 * @brief Initialize values of an S_EventDetails structure, given an event.
 * @param event information from which to initialize details
 * @param details pointer to the S_EventDetails structure to be initialized
 *
 * @return  The following return values are allowed from this function
 *          MHERR_OK             - Success
 *          MHERR_BAD_PARAMETER  - Invalid parameter.
 *          MHERR_OTHER          - Controlling application specific
 *          error.
 */
static E_MhegErr InitializeEventDetails(void *event_ptr, S_EventDetails *details)
{
   E_MhegErr result = MHERR_OTHER;
   U8BIT length;
   U8BIT *contentData;
   U8BIT *dvbEventName;
   U8BIT *dvbShortDescription;
   int i;

   FUNCTION_START(InitializeEventDetails);

   memset(details, 0, sizeof(*details));

   if (event_ptr != NULL)
   {
      dvbEventName = ADB_GetEventName(event_ptr);
      if (dvbEventName)
      {
         details->eventName.zptr = dvbEventName;

         /* Get the length of the (UTF8) octetstring */
         details->eventName.zlen = STB_GetNumBytesInString(details->eventName.zptr) - 1;

         if (details->eventName.zlen <= 1)
         {
            STB_ReleaseUnicodeString( details->eventName.zptr );
            details->eventName.zptr = NULL;
            details->eventName.zlen = 0;
         }
         else
         {
            /* Skip the Unicode start character, and adjust length */
            details->eventName.zptr++;
            details->eventName.zlen--;
         }
      }

      dvbShortDescription = ADB_GetEventDescription(event_ptr);
      if (dvbShortDescription)
      {
         details->shortDescription.zptr = dvbShortDescription;

         /* Get the length of the (UTF8) octetstring */
         details->shortDescription.zlen = STB_GetNumBytesInString(details->shortDescription.zptr) - 1;

         if (details->shortDescription.zlen <= 1)
         {
            STB_ReleaseUnicodeString( details->shortDescription.zptr );
            details->shortDescription.zptr = NULL;
            details->shortDescription.zlen = 0;
         }
         else
         {
            /* Skip the Unicode start character, and adjust length */
            details->shortDescription.zptr++;
            details->shortDescription.zlen--;
         }
      }

      details->parentalRating = (U32BIT) ADB_GetEventParentalAge(event_ptr);

      /* This is already a suitable MJD, no need to modify */
      details->startDate = ADB_GetEventStartDate(event_ptr);

      details->startTime.hours = ADB_GetEventStartHour(event_ptr);
      details->startTime.minutes = ADB_GetEventStartMin(event_ptr);
      details->startTime.seconds = ADB_GetEventStartSecs(event_ptr);

      details->duration.hours = ADB_GetEventDurationHour(event_ptr);
      details->duration.minutes = ADB_GetEventDurationMin(event_ptr);
      details->duration.seconds = ADB_GetEventDurationSecs(event_ptr);

      contentData = ADB_GetEventContentData(event_ptr, &length);
      if (contentData != NULL && length != 0)
      {
         details->category.zptr = STB_AppGetMemory(length / 2);
         if (details->category.zptr != NULL)
         {
            details->category.zlen = length / 2;

            /* Ignore user nibbles in content data */
            for (i = 0; i < (int) details->category.zlen; i++)
            {
               details->category.zptr[i] = contentData[i * 2];
            }
         }
      }

      details->freeNotCA = ADB_GetEventFreeToAir(event_ptr);

      result = MHERR_OK;
   }

   FUNCTION_FINISH(InitializeEventDetails);

   return result;
}

/**
 * @brief   Release data allocated by DVB_MhegGetEventDetails (e.g. String data).
 *          Always called immediately after the engine has copied data.
 *          DVB stack may have other data management (using dvb_user_data)
 * @param   details Pointer to S_EventDetails
 * @return  void
 */
void DVB_MhegReleaseEventDetails( S_EventDetails *details )
{
   FUNCTION_START(DVB_MhegReleaseEventDetails);

   if (details->eventName.zptr != NULL)
   {
      /* Adjust pointer back to original position */
      details->eventName.zptr--;
      details->eventName.zlen++;

      STB_ReleaseUnicodeString( details->eventName.zptr );

      details->eventName.zptr = NULL;
      details->eventName.zlen = 0;
   }

   if (details->shortDescription.zptr != NULL)
   {
      /* Adjust pointer back to original position */
      details->shortDescription.zptr--;
      details->shortDescription.zlen++;

      STB_ReleaseUnicodeString( details->shortDescription.zptr );

      details->shortDescription.zptr = NULL;
      details->shortDescription.zlen = 0;
   }

   if (details->category.zptr != NULL)
   {
      STB_AppFreeMemory(details->category.zptr);

      details->category.zptr = NULL;
      details->category.zlen = 0;
   }

   FUNCTION_FINISH(DVB_MhegReleaseEventDetails);
}

/**
 * @brief   Retrieves information about the present or following event on a service.
 *          (See ETSI MHEG-5 Broadcast Profile, ES 202 184)
 *          Provided this returns MHERR_OK, the engine will call DVB_MhegReleaseEventDetails
 *          to release allocated data on the same task and immediately after the engine has
 *          copied the data.
 * @param   serviceIndex Receiver dependent identifier for the required service
 * @param   porf Indicates whether the present or following event should
 *          be retrieved, 0 = present, 1 = following
 * @param   details Pointer to S_EventDetails structure to place required
 *          event information (this will NOT be NULL)
 * @return  MHERR_OK on success
 *          MHERR_BAD_PARAMETER if serviceIndex is invalid
 *          MHERR_OTHER otherwise.
 */
E_MhegErr DVB_MhegGetEventDetails( S32BIT serviceIndex, BOOLEAN porf, S_EventDetails *details )
{
   E_MhegErr result;
   void *event_ptr;

   FUNCTION_START(DVB_MhegGetEventDetails);

   event_ptr = GetPorfEvent( serviceIndex, porf );
   result = InitializeEventDetails(event_ptr, details);

   if (event_ptr != NULL)
   {
      ADB_ReleaseEventData(event_ptr);
   }

   FUNCTION_FINISH(DVB_MhegGetEventDetails);

   return result;
}

/**
 * @brief   Retrieves event id for the present or following event on a service.
 * @param   serviceIndex Receiver dependent identifier for the required service
 *          as returned by DVB_MhegDvbLocatorToIndex
 * @param   porf Indicates whether the present or following event id
 *          should be retrieved, 0 = present, 1 = following
 * @param   eventId Pointer to event id
 * @return  MHERR_OK on success
 *          MHERR_BAD_PARAMETER if serviceIndex is invalid
 *          MHERR_OBJECT_NOT_FOUND if reciever cannot access event in EIT
 */
E_MhegErr DVB_MhegGetEventId( S32BIT serviceIndex, BOOLEAN porf, S32BIT *mhegEventId )
{
   E_MhegErr result = MHERR_OBJECT_NOT_FOUND;
   void *event_ptr;

   FUNCTION_START(DVB_MhegGetEventId);

   *mhegEventId = 0;

   event_ptr = GetPorfEvent( serviceIndex, porf );
   if (event_ptr != NULL)
   {
      *mhegEventId = MakeMHEGEventId(serviceIndex, ADB_GetEventId(event_ptr));
      ADB_ReleaseEventData(event_ptr);

      result = MHERR_OK;
   }

   FUNCTION_FINISH(DVB_MhegGetEventId);

   return result;
}

/**
 * @brief   Set the position of the Event Iterator to the specified service,
 *          date and time.
 *          This function returns the unique event identifier for the event in the
 *          EIT schedule that the Event Iterator has been set to.
 *          When set to an event signalled by the current EITp/f, the Resident
 *          Programs shall return the id of the event in the EITs and not EITp/f.
 *          If there is no EIT information cached for the given time and date then
 *          the receiver shall return the event id of the first event in the cache
 *          after the specified date and time. If there are no such events then
 *          the receiver shall return -1.
 *          This MUST be a non-blocking function, returning results immediately.
 * @param   serviceIndex The index of the service on which to set the Event
 *          Iterator.
 * @param   startDate The start date at which to set the Event Iterator.
 *          This is encoded as the number of days since midnight
 *          on November 17, 1858 (Modified Julian Date).
 * @param   startTime The start time at which to set the Event Iterator.
 *          This is encoded as the number of seconds since midnight.
 * @param   eventId The unique event identifier for the event in the EIT
 *          schedule that the Event Iterator has been set to.
 * @return  The following return values are allowed from this function
 *          MHERR_OK             - Success
 *          MHERR_BAD_PARAMETER  - Invalid parameter.
 *          MHERR_OTHER          - Controlling application specific
 *          error.
 */
E_MhegErr DVB_MhegSetEventIterator(S32BIT serviceIndex, S32BIT startDate, S32BIT startTime, S32BIT *mhegEventId)
{
   E_MhegErr result = MHERR_OTHER;
   void *srv_ptr;
   void *event_ptr;
   U32DHMS datetime;
   U32BIT timestamp;

   FUNCTION_START(DVB_MhegSetEventIterator);

   *mhegEventId = -1;

   while (startTime < 0)
   {
      if (startDate < 1)
      {
         return MHERR_BAD_PARAMETER;
      }

      startDate--;
      startTime += SECONDS_PER_DAY;
   }
   while (startTime >= SECONDS_PER_DAY)
   {
      startDate++;
      startTime -= SECONDS_PER_DAY;
   }

   if (startDate < ADJUST_MJD_TO_1970)
   {
      return MHERR_BAD_PARAMETER;
   }

   /* Adjust to Unix 1970-01-01 from 1858-11-17 */
   startDate -= ADJUST_MJD_TO_1970;

   timestamp = (U32BIT) (startDate * SECONDS_PER_DAY);
   timestamp += (U32BIT) startTime;

   datetime = STB_GCConvertTimestamp(timestamp);

   if (CHECK_LCN(serviceIndex))
   {
      srv_ptr = ADB_FindServiceByLcn(ADB_SERVICE_LIST_ALL, (U16BIT) serviceIndex, TRUE);
      if (srv_ptr != NULL)
      {
         event_ptr = ADB_FindEventFromTime(datetime, srv_ptr);
         if (event_ptr != NULL)
         {
            *mhegEventId = MakeMHEGEventId(serviceIndex, ADB_GetEventId(event_ptr));
            ADB_ReleaseEventData(event_ptr);

            result = MHERR_OK;
         }
         else
         {
            DBGPRINT("DVB_MhegSetEventIterator event_ptr == NULL");
         }
      }
      else
      {
         DBGPRINT("DVB_MhegSetEventIterator srv_ptr == NULL");
      }
   }
   else
   {
      DBGPRINT("DVB_MhegSetEventIterator serviceIndex is out of bounds");
   }

   FUNCTION_FINISH(DVB_MhegSetEventIterator);

   return result;
}

/**
 * @brief   Increment the Event Iterator to the event chronologically following
 *          the event to which it is currently set and returns the event id.
 *          This function returns the unique event identifier for the event to
 *          which the Event Iterator is set.
 *          If there is no event chronologically following the current event in
 *          the EIT cache then the receiver shall return the event id of the first
 *          event in the cache occurring after the current event. If there are no
 *          such events then the receiver shall return -1.
 *          If the Event Iterator has gone stale, i.e. if the event to which the
 *          Event Iterator is currently set is no longer in the EIT cache, the
 *          receiver shall return -1.
 *          This MUST be a non-blocking function, returning results immediately.
 * @param   eventId The unique event identifier for the event to
 *          which the Event Iterator is set.
 * @return  The following return values are allowed from this function
 *          MHERR_OK             - Success
 *          MHERR_BAD_PARAMETER  - Invalid parameter.
 *          MHERR_OTHER          - Controlling application specific
 *          error.
 */
E_MhegErr DVB_MhegIncrementEventIterator(S32BIT *mhegEventId)
{
   E_MhegErr result = MHERR_OTHER;
   BOOLEAN valid;
   U16BIT eventServiceIndex;
   U16BIT eventId;
   void *srv_ptr;
   void *event_ptr;
   void *next;

   FUNCTION_START(DVB_MhegIncrementEventIterator);

   valid = GetValuesFromMHEGEventId(*mhegEventId, &eventServiceIndex, &eventId);
   *mhegEventId = -1;

   if (valid == TRUE)
   {
      /* reinstate if make stronger if(CHECK_LCN(serviceIndex)) */
      {
         srv_ptr = ADB_FindServiceByLcn(ADB_SERVICE_LIST_ALL, (U16BIT) eventServiceIndex, TRUE);
         if (srv_ptr != NULL)
         {
            event_ptr = ADB_GetEvent(srv_ptr, eventId);
            if (event_ptr != NULL)
            {
               next = ADB_LaterEvent(srv_ptr, ADB_GetEventEndDateTime(event_ptr));
               ADB_ReleaseEventData(event_ptr);

               if (next != NULL)
               {
                  *mhegEventId = MakeMHEGEventId(eventServiceIndex, ADB_GetEventId(next));
                  ADB_ReleaseEventData(next);

                  DBGPRINT("DVB_MhegIncrementEventIterator *mhegEventId = 0x%08x", *mhegEventId);

                  result = MHERR_OK;
               }
               else
               {
                  DBGPRINT("DVB_MhegIncrementEventIterator next == NULL");
               }
            }
            else
            {
               DBGPRINT("DVB_MhegIncrementEventIterator event_ptr == NULL");
            }
         }
         else
         {
            DBGPRINT("DVB_MhegIncrementEventIterator srv_ptr == NULL");
         }
      }
   }
   else
   {
      DBGPRINT("DVB_MhegIncrementEventIterator valid != TRUE");
   }

   FUNCTION_FINISH(DVB_MhegIncrementEventIterator);

   return result;
}

/**
 * @brief   Returns event information for a specified service index and event id.
 *          This MUST be a non-blocking function, returning results immediately.
 * @param   serviceIndex A receiver specific identifier for the service
 *          signalling the event.
 * @param   mhegEventId The unique identifier for the event to return
 *          information.
 * @param   status TRUE if the receiver can access the event in the EIT
 *          cache, FALSE otherwise.
 * @param   details The scheduled event details.
 * @return  The following return values are allowed from this function
 *          MHERR_OK             - Success
 *          MHERR_BAD_PARAMETER  - Invalid parameter.
 *          MHERR_OTHER          - Controlling application specific
 *          error.
 */
E_MhegErr DVB_MhegGetScheduledEventDetails(S32BIT serviceIndex, S32BIT mhegEventId, BOOLEAN *status, S_EventDetails *details)
{
   E_MhegErr result = MHERR_OTHER;
   void *srv_ptr;
   void *event_ptr;

   FUNCTION_START(DVB_MhegGetScheduledEventDetails);

   *status = FALSE;

   if (GetServiceEventFromMHEGEventId(serviceIndex, mhegEventId, &srv_ptr, &event_ptr) == TRUE)
   {
      result = InitializeEventDetails(event_ptr, details);
      ADB_ReleaseEventData(event_ptr);
     *status = TRUE;
   }

   FUNCTION_FINISH(DVB_MhegGetScheduledEventDetails);

   return result;
}

/**
 * @brief Initialize values of an S_TIMER_INFO structure, given a serviceIndex and an
 *        mhegEventId (created via MakeMHEGEventId()).
 * @param   serviceIndex A receiver specific integer identifying a service.
 *          This value shall be consistent with the values
 *          returned by DVB_MhegDvbLocatorToIndex.
 * @param   mhegEventId The unique identifier for the event to initialize a timer
 * @param info pointer to the S_TIMER_INFO structure to be initialized
 *
 * @return  TRUE if the S_TIMER_INFO structure could be initialized successfully, FALSE otherwise.
 */
static BOOLEAN InitializeTimerInfo(S32BIT serviceIndex, S32BIT mhegEventId, S_TIMER_INFO *info)
{
   void *srv_ptr;
   void *event_ptr;
   BOOLEAN success = FALSE;

   FUNCTION_START(InitializeTimerInfo);

   if (GetServiceEventFromMHEGEventId(serviceIndex, mhegEventId, &srv_ptr, &event_ptr) == TRUE)
   {
      info->start_time = ADB_GetEventStartDateTime(event_ptr);
      ADB_ReleaseEventData(event_ptr);

      info->type = TIMER_TYPE_ALARM;
      strncpy((char *) info->name, "MHEG reminder", TMR_MAX_NAME_LENGTH);
      info->frequency = TIMER_FREQ_ONCE;
      info->u.alarm.change_service = FALSE;
      ADB_GetServiceIds(srv_ptr, &info->u.alarm.orig_net_id, &info->u.alarm.transport_id, &info->u.alarm.service_id);
      info->u.alarm.ramp_volume = FALSE;
      success = TRUE;
   }

   FUNCTION_FINISH(InitializeTimerInfo);

   return success;
}

/**
 * @brief Search for a timerhandle corresponding to a particular S_TIMER_INFO structure's values
 * @param info pointer to the S_TIMER_INFO structure from which the timerhandle should be recovered
 * @param timerhandle pointer to return the recovered timer handle
 *
 * @return TRUE if a timerhandle corresponding to info could be found, FALSE otherwise.
 */
static BOOLEAN FindMHEGTimerHandle(S_TIMER_INFO *info, U32BIT *timerhandle)
{
   BOOLEAN success = FALSE;
   S_TIMER_INFO test;
   U32BIT *timer_list;
   U16BIT list_size;
   int i;

   FUNCTION_START(findMHEGTimeHandle);

   if (ATMR_GetTimerList(&timer_list, &list_size, TIMER_TYPE_ALARM, TRUE) == TRUE)
   {
      for (i = 0; i < list_size; i++)
      {
         if (ATMR_GetTimerInfo(timer_list[i], &test) != TRUE)
            continue;

         if (memcmp(info->name, test.name, TMR_MAX_NAME_LENGTH) != 0)
            continue;

         if (info->start_time != test.start_time)
            continue;

         if (info->u.alarm.orig_net_id != test.u.alarm.orig_net_id ||
             info->u.alarm.transport_id != test.u.alarm.transport_id ||
             info->u.alarm.service_id != test.u.alarm.service_id)
            continue;

         *timerhandle = timer_list[i];
         success = TRUE;
         break;
      }

      ATMR_ReleaseTimerList(timer_list, list_size);
   }

   FUNCTION_FINISH(findMHEGTimeHandle);

   return success;
}

/**
 * @brief   Cancel any native reminder set for the given event on the given service.
 *          This MUST be a non-blocking function, returning results immediately.
 * @param   serviceIndex A receiver specific identifier for the service.
 * @param   eventId The unique event identifier on the given serviceIndex
 *          of the event to cancel a reminder for.
 * @return  The following return values are allowed from this function
 *          MHERR_OK                 - Success
 *          MHERR_BAD_PARAMETER  - Invalid parameter.
 *          MHERR_OTHER          - Controlling application specific
 *          error.
 */
E_MhegErr DVB_MhegCancelNativeReminder(S32BIT serviceIndex, S32BIT mhegEventId)
{
   E_MhegErr result = MHERR_OTHER;
   S_TIMER_INFO info;
   U32BIT timerhandle;

   FUNCTION_START(DVB_MhegCancelNativeReminder);

   if (InitializeTimerInfo(serviceIndex, mhegEventId, &info) == TRUE)
   {
      if (FindMHEGTimerHandle(&info, &timerhandle) == TRUE)
      {
         if (ATMR_DeleteTimer(timerhandle) == TRUE)
         {
            result = MHERR_OK;
         }
      }
   }

   FUNCTION_FINISH(DVB_MhegCancelNativeReminder);

   return result;
}

/**
 * @brief   Return whether a native reminder has been set for the given event
 *          on the given service.
 *          This MUST be a non-blocking function, returning results immediately.
 * @param   serviceIndex A receiver specific identifier for the service.
 * @param   eventId The unique event identifier on the given serviceIndex
 *          of the event to return the reminder setting for.
 * @param   set TRUE if a reminder has been set for the given eventId
 *          and service, FALSE otherwise.
 * @return  The following return values are allowed from this function
 *          MHERR_OK                 - Success
 *          MHERR_BAD_PARAMETER  - Invalid parameter.
 *          MHERR_OTHER          - Controlling application specific error.
 */
E_MhegErr DVB_MhegGetNativeReminder(S32BIT serviceIndex, S32BIT mhegEventId, BOOLEAN *set)
{
   S_TIMER_INFO info;
   U32BIT timerhandle;

   FUNCTION_START(DVB_MhegGetNativeReminder);

   if (InitializeTimerInfo(serviceIndex, mhegEventId, &info) == TRUE)
   {
      *set = FindMHEGTimerHandle(&info, &timerhandle);
   }

   FUNCTION_FINISH(DVB_MhegGetNativeReminder);

   return MHERR_OK;
}

/**
 * @brief   Set a native reminder for the given event on the given service.
 *          This MUST be a non-blocking function, returning results immediately.
 * @param   serviceIndex A receiver specific identifier for the service.
 * @param   eventId The unique event identifier on the given serviceIndex
 *          of the event to set a reminder for.
 * @param   result TRUE if the receiver could access the event in the EIT
 *          cache, FALSE otherwise.
 * @return  The following return values are allowed from this function
 *          MHERR_OK                 - Success
 *          MHERR_BAD_PARAMETER  - Invalid parameter.
 *          MHERR_OTHER          - Controlling application specific error.
 */
E_MhegErr DVB_MhegSetNativeReminder(S32BIT serviceIndex, S32BIT mhegEventId, BOOLEAN *success)
{
   S_TIMER_INFO info;
   U32BIT timerhandle;

   FUNCTION_START(DVB_MhegSetNativeReminder);

   *success = FALSE;
   if (InitializeTimerInfo(serviceIndex, mhegEventId, &info) == TRUE)
   {
      if (FindMHEGTimerHandle(&info, &timerhandle) == FALSE)
      {
         if (ATMR_AddTimer(&info) != INVALID_TIMER_HANDLE)
         {
            *success = TRUE;
         }
      }
   }

   FUNCTION_FINISH(DVB_MhegSetNativeReminder);

   return MHERR_OK;
}

/**
 * @brief   Used to determine if an ADB_EVENT_COMPONENT_INFO indicates an HD video component.
 * @param   info an ADB_EVENT_COMPONENT_INFO descibing a component to be tested.
 * @return  TRUE if the ADB_EVENT_COMPONENT_INFO indicates an HD video component, FALSE otherwise.
 */
static BOOLEAN IsHdVideo(ADB_EVENT_COMPONENT_INFO *info)
{
   BOOLEAN result = FALSE;

   if (info->stream_content == 0x01)
   {
      switch (info->component_type)
      {
         case 0x09: /* MPEG-2 high definition video, 4:3 aspect ratio, 25 Hz */
         case 0x0A: /* MPEG-2 high definition video, 16:9 aspect ratio with pan vectors, 25 Hz */
         case 0x0B: /* MPEG-2 high definition video, 16:9 aspect ratio without pan vectors, 25 Hz */
         case 0x0C: /* MPEG-2 high definition video, > 16:9 aspect ratio, 25 Hz */
         case 0x0D: /* MPEG-2 high definition video, 4:3 aspect ratio, 30 Hz */
         case 0x0E: /* MPEG-2 high definition video, 16:9 aspect ratio with pan vectors, 30 Hz */
         case 0x0F: /* MPEG-2 high definition video, 16:9 aspect ratio without pan vectors, 30 Hz */
         case 0x10: /* MPEG-2 high definition video, > 16:9 aspect ratio, 30 Hz */
            result = TRUE;
      }
   }

   if (info->stream_content == 0x03)
   {
      switch (info->component_type)
      {
         case 0x40: /* video up-sampled from standard definition source material */
            result = TRUE;
      }
   }

   if (info->stream_content == 0x05)
   {
      switch (info->component_type)
      {
         case 0x0B: /* H.264/AVC high definition video, 16:9 aspect ratio, 25 Hz */
         case 0x0C: /* H.264/AVC high definition video, > 16:9 aspect ratio, 25 Hz */
         case 0x0F: /* H.264/AVC high definition video, 16:9 aspect ratio, 30 Hz */
         case 0x10: /* H.264/AVC high definition video, > 16:9 aspect ratio, 30 Hz */
         case 0x80: /* H.264/AVC plano-stereoscopic frame compatible high definition video, 16:9 aspect ratio, 25 Hz, Side-by-Side */
         case 0x81: /* H.264/AVC plano-stereoscopic frame compatible high definition video, 16:9 aspect ratio, 25 Hz, Top-and-Bottom */
         case 0x82: /* H.264/AVC plano-stereoscopic frame compatible high definition video, 16:9 aspect ratio, 30 Hz, Side-by-Side */
         case 0x83: /* H.264/AVC stereoscopic frame compatible high definition video, 16:9 aspect ratio, 30 Hz, Top-and-Bottom */
            result = TRUE;
      }
   }

   return result;
}

/**
 * @brief   Used to determine if an ADB_EVENT_COMPONENT_INFO indicates a main Audio component.
 * @param   info an ADB_EVENT_COMPONENT_INFO descibing a component to be tested.
 * @return  TRUE if the ADB_EVENT_COMPONENT_INFO indicates a main Audio component, FALSE otherwise.
 */
static BOOLEAN IsAudio(ADB_EVENT_COMPONENT_INFO *info)
{
   BOOLEAN result = FALSE;

   if (info->stream_content == 0x02)
   {
      switch (info->component_type)
      {
         case 0x01: /* MPEG-1 Layer 2 audio, single mono channel */
         case 0x02: /* MPEG-1 Layer 2 audio, dual mono channel */
         case 0x03: /* MPEG-1 Layer 2 audio, stereo (2 channel) */
         case 0x04: /* MPEG-1 Layer 2 audio, multi-lingual, multi-channel */
         case 0x05: /* MPEG-1 Layer 2 audio, surround sound */
         case 0x41: /* MPEG-1 Layer 2 audio for the hard of hearing */
         case 0x42: /* receiver-mix supplementary audio as per annex E of TS 101 154 */
            result = TRUE;
      }
   }

   if (info->stream_content == 0x04)
   {
      switch (info->component_type)
      {
         result = TRUE;
      }
   }

   if (info->stream_content == 0x06)
   {
      switch (info->component_type)
      {
         case 0x01: /* HE-AAC audio, single mono channel */
         case 0x03: /* HE-AAC audio, stereo */
         case 0x05: /* HE-AAC audio, surround sound */
         case 0x41: /* HE-AAC audio for the hard of hearing */
         case 0x42: /* HE-AAC receiver-mix supplementary audio as per annex E of TS 101 154 */
         case 0x43: /* HE-AAC v2 audio, stereo */
         case 0x45: /* HE-AAC v2 audio for the hard of hearing */
         case 0x46: /* HE-AAC v2 receiver-mix supplementary audio as per annex E of TS 101 154 */
            result = TRUE;
      }
   }

   if (info->stream_content == 0x07)
   {
      if (info->component_type < 0x80)
      {
         /* reserved for DTS and DTS-HD audio modes */
         result = TRUE;
      }
   }

   return result;
}

/**
 * @brief   Used to determine if an ADB_EVENT_COMPONENT_INFO indicates a Dolby Audio component.
 * @param   info an ADB_EVENT_COMPONENT_INFO descibing a component to be tested.
 * @return  TRUE if the ADB_EVENT_COMPONENT_INFO indicates a Dolby Audio component, FALSE otherwise.
 */
static BOOLEAN IsDolbyAudio(ADB_EVENT_COMPONENT_INFO *info)
{
   BOOLEAN result = FALSE;

   if (info->stream_content == 0x04)
   {
      result = TRUE;
   }

   if (info->stream_content == 0x07)
   {
      if (info->component_type < 0x80)
      {
         /* reserved for DTS and DTS-HD audio modes */
         result = TRUE;
      }
   }

   return result;
}

/**
 * @brief   Used to determine if an ADB_EVENT_COMPONENT_INFO indicates an audio description component.
 * @param   info an ADB_EVENT_COMPONENT_INFO descibing a component to be tested.
 * @return  TRUE if the ADB_EVENT_COMPONENT_INFO indicates an audio description component, FALSE otherwise.
 */
static BOOLEAN IsAudioDescription(ADB_EVENT_COMPONENT_INFO *info)
{
   BOOLEAN result = FALSE;

   if (info->stream_content == 0x02)
   {
      switch (info->component_type)
      {
         case 0x40: /* MPEG-1 Layer 2 audio description for the visually impaired */
         case 0x47: /* MPEG-1 Layer 2 audio, receiver-mix audio description */
         case 0x48: /* MPEG-1 Layer 2 audio, broadcast-mix audio description */
            result = TRUE;
      }
   }

   if (info->stream_content == 0x06)
   {
      switch (info->component_type)
      {
         case 0x40: /* HE-AAC audio description for the visually impaired */
         case 0x44: /* HE-AAC v2 audio description for the visually impaired */
         case 0x47: /* HE-AAC receiver-mix audio description for the visually impaired */
         case 0x48: /* HE-AAC broadcast-mix audio description for the visually impaired */
         case 0x49: /* HE-AAC v2 receiver-mix audio description for the visually impaired */
         case 0x4A: /* HE-AAC v2 broadcast-mix audio description for the visually impaired */
            result = TRUE;
      }
   }

   return result;
}

/**
 * @brief   Used to determine if an ADB_EVENT_COMPONENT_INFO indicates a subtitle component.
 * @param   info an ADB_EVENT_COMPONENT_INFO descibing a component to be tested.
 * @return  TRUE if the ADB_EVENT_COMPONENT_INFO indicates a subtitle component, FALSE otherwise.
 */
static BOOLEAN IsSubtitle(ADB_EVENT_COMPONENT_INFO *info)
{
   BOOLEAN result = FALSE;

   if (info->stream_content == 0x03)
   {
      switch (info->component_type)
      {
         case 0x01: /* EBU Teletext subtitles */
         case 0x10: /* DVB subtitles (normal) with no monitor aspect ratio criticality */
         case 0x11: /* DVB subtitles (normal) for display on 4:3 aspect ratio monitor */
         case 0x12: /* DVB subtitles (normal) for display on 16:9 aspect ratio monitor */
         case 0x13: /* DVB subtitles (normal) for display on 2.21:1 aspect ratio monitor */
         case 0x14: /* DVB subtitles (normal) for display on a high definition monitor */
         case 0x15: /* DVB subtitles (normal) with plano-stereoscopic disparity for display on a high definition monitor */
         case 0x20: /* DVB subtitles (for the hard of hearing) with no monitor aspect ratio criticality */
         case 0x21: /* DVB subtitles (for the hard of hearing) for display on 4:3 aspect ratio monitor */
         case 0x22: /* DVB subtitles (for the hard of hearing) for display on 16:9 aspect ratio monitor */
         case 0x23: /* DVB subtitles (for the hard of hearing) for display on 2.21:1 aspect ratio monitor */
         case 0x24: /* DVB subtitles (for the hard of hearing) for display on a high definition monitor */
         case 0x25: /* DVB subtitles (for the hard of hearing) with plano-stereoscopic disparity for display on a high definition monitor */
            result = TRUE;
      }
   }

   return result;
}

/**
 * @brief   Return information about the component streams for a specified
 *          service index and event id. The information returned is extracted from
 *          Component Descriptors in the EITs as defined in ETSI EN 300468 V1.12.1
 *          clause 6.2.8.
 *          This MUST be a non-blocking function, returning results immediately.
 * @param   serviceIndex A receiver specific identifier for the service
 *          signalling the event.
 * @param   eventId The unique identifier for the event to return
 *          information.
 * @param   result TRUE if the receiver can access the event in the EIT
 *          cache, FALSE otherwise.
 * @param   key A constant identifying the feature for which
 *          availability of a component is to be returned.
 * @param   value The availability of a component for the feature given
 *          by key.
 * @return  The following return values are allowed from this function
 *          MHERR_OK             - Success
 *          MHERR_BAD_PARAMETER  - Invalid parameter.
 *          MHERR_OTHER          - Controlling application specific
 *          error.
 */
E_MhegErr DVB_MhegGetComponentInfo(S32BIT serviceIndex, S32BIT porfIndex, BOOLEAN *success, E_ComponentKey key, BOOLEAN *value)
{
   void *event_ptr;
   ADB_EVENT_COMPONENT_INFO *componentList;
   U8BIT noOfComponents;
   U8BIT i;
   U8BIT *firstLanguage;

   FUNCTION_START(DVB_MhegGetComponentInfo);
   USE_UNWANTED_PARAM(value);

   *success = FALSE;
   if (porfIndex == 0 || porfIndex == 1)
   {
      event_ptr = GetPorfEvent( serviceIndex, (porfIndex == 1) ? TRUE : FALSE );
      if(event_ptr != NULL)
      {
         noOfComponents = ADB_GetEventComponentList(event_ptr, &componentList);
         ADB_ReleaseEventData(event_ptr);

         firstLanguage = NULL;
         for(i = 0; i != noOfComponents;i++)
         {
            ADB_EVENT_COMPONENT_INFO *info = &(componentList[i]);

            if (key == MHEG5_COMPONENT_SUBTITLES && IsSubtitle(info) == TRUE)
            {
               *success = TRUE;
               break;
            }
            if (key == MHEG5_COMPONENT_AUDIO_DESC && IsAudioDescription(info) == TRUE)
            {
               *success = TRUE;
               break;
            }
            if (key == MHEG5_COMPONENT_DOLBY && IsDolbyAudio(info) == TRUE)
            {
               *success = TRUE;
               break;
            }
            if (key == MHEG5_COMPONENT_HD && IsHdVideo(info) == TRUE)
            {
               *success = TRUE;
               break;
            }
            if (key == MHEG5_COMPONENT_ALT_LANG && IsAudio(info) == TRUE)
            {
               if (firstLanguage != NULL && memcmp(firstLanguage, info->language_code, 3) != 0)
               {
                  *success = TRUE;
                  break;
               }

               firstLanguage = info->language_code;
            }
         }

         if(noOfComponents != 0)
         {
            STB_AppFreeMemory(componentList);
         }
      }
   }

   FUNCTION_FINISH(DVB_MhegGetComponentInfo);

   return MHERR_OTHER;
}

/**
 * @brief   Return the Component Descriptor data for a specified service index and
 *          event id. The data returned is extracted from Component Descriptors in
 *          the EITs as defined in ETSI EN 300468 V1.12.1 clause 6.2.8.
 *          This MUST be a non-blocking function, returning results immediately.
 * @param   serviceIndex A receiver specific identifier for the service
 *          signalling the event.
 * @param   eventId The unique identifier for the event to return
 *          information.
 * @param   result TRUE if the receiver can access the event in the EIT
 *          cache, FALSE otherwise.
 * @param   strCont A space separated, null-terminated list of 1 char hex
 *          values ascii representing the values of
 *          stream_content for each component of the event (up to
 *          256 bytes including null-terminator).
 * @param   compType A space separated, null-terminated list of 2 char hex
 *          values ascii representing the values of content_type
 *          for each component of the event (up to 256 bytes
 *          including null-terminator).
 * @param   lang A space separated, null-terminated list of 3 char ascii
 *          strings representing the values of language for each
 *          component of the event (up to 256 bytes including
 *          null-terminator).
 * @return  The following return values are allowed from this function
 *          MHERR_OK             - Success
 *          MHERR_BAD_PARAMETER  - Invalid parameter.
 *          MHERR_OTHER          - Controlling application specific
 *          error.
 */
E_MhegErr DVB_MhegGetComponentData(S32BIT serviceIndex, S32BIT porfIndex, BOOLEAN *success, U8BIT *strCont, U8BIT *compType, U8BIT *lang)
{
   void *event_ptr;
   ADB_EVENT_COMPONENT_INFO *componentList;
   U8BIT noOfComponents;
   char stream[2];
   char component[3];
   char language[4];
   U8BIT i;

   FUNCTION_START(DVB_MhegGetComponentData);

   *success = FALSE;
   if (porfIndex == 0 || porfIndex == 1)
   {
      event_ptr = GetPorfEvent( serviceIndex, (porfIndex == 1) ? TRUE : FALSE );
      if (event_ptr != NULL)
      {
         noOfComponents = ADB_GetEventComponentList(event_ptr, &componentList);
         ADB_ReleaseEventData(event_ptr);

         for (i = 0; i != noOfComponents && i < (256 / 4); i++)
         {
            ADB_EVENT_COMPONENT_INFO *info = &(componentList[i]);

            stream[0] = HEX((info->stream_content & 0x0F));
            stream[1] = '\0';

            component[0] = HEX(((info->component_type & 0xF0) >> 4));
            component[1] = HEX((info->component_type & 0x0F));
            component[2] = '\0';

            language[0] = info->language_code[0];
            language[1] = info->language_code[1];
            language[2] = info->language_code[2];
            language[3] = '\0';

            if (i == 0)
            {
               strncpy((char *) strCont, stream, 2);
               strncpy((char *) compType, component, 3);
               strncpy((char *) lang, language, 4);
            }
            else
            {
               strncat((char *) strCont, " ", 2);
               strncat((char *) compType, " ", 2);
               strncat((char *) lang, " ", 2);

               strncat((char *) strCont, stream, 2);
               strncat((char *) compType, component, 3);
               strncat((char *) lang, language, 4);
            }
         }
      }
   }

   FUNCTION_FINISH(DVB_MhegGetComponentData);

   return MHERR_OTHER;
}

/**
 * @brief   A general purpose routine that returns a space separated set of languages for components
 *          (of a present/following event on a given service) matching a filter condition
 * @return  - MHERR_OK on success
 *          - MHERR_BAD_PARAMETER invalid parameter
 *          - MHERR_OTHER controlling application specific error.
 */
static E_MhegErr GetLangs(S32BIT serviceIndex, S32BIT porfIndex, U8BIT *langs,
   BOOLEAN (*filter)(ADB_EVENT_COMPONENT_INFO *info))
{
   void *event_ptr;
   ADB_EVENT_COMPONENT_INFO *componentList;
   U8BIT noOfComponents;
   char isolang[4];
   U8BIT i;

   FUNCTION_START(GetLangs);

   strncpy((char *) langs, "", 1);

   if (porfIndex == 0 || porfIndex == 1)
   {
      event_ptr = GetPorfEvent( serviceIndex, (porfIndex == 1) ? TRUE : FALSE );
      if (event_ptr != NULL)
      {
         noOfComponents = ADB_GetEventComponentList(event_ptr, &componentList);
         ADB_ReleaseEventData(event_ptr);

         for (i = 0; i != noOfComponents; i++)
         {
            ADB_EVENT_COMPONENT_INFO *info = &(componentList[i]);

            if (filter(info) == FALSE)
            {
               continue;
            }

            /* get a NULL-terminated string */
            isolang[0] = info->language_code[0];
            isolang[1] = info->language_code[1];
            isolang[2] = info->language_code[2];
            isolang[3] = '\0';

            if (strstr((char *) langs, isolang) != NULL)
            {
               /* language already found */
               continue;
            }

            if (strlen((char *) langs) != 0)
            {
               strncat((char *) langs, " ", 2);
            }

            strncat((char *) langs, isolang, 4);
         }

         if (noOfComponents != 0)
         {
            STB_AppFreeMemory(componentList);
         }
      }
   }

   FUNCTION_FINISH(GetLangs);

   return MHERR_OK;
}

/**
 * @brief   Return a list of available audio description languages for the
 *          specified event.
 *          The language list is extracted from the component descriptors of
 *          the present or following event (EITpf). It is returned as a
 *          null-terminated sequence of 3-letter codes separated by a space
 *          character.
 *          The buffer provided is allocated by the MHEG-5 engine and is
 *          large enough to store up to 64 languages (256 bytes including the
 *          null-terminator).
 *          This MUST be a non-blocking function, returning results immediately.
 * @param   serviceIndex A receiver specific integer identifying a service.
 *          This value shall be consistent with the values
 *          returned by DVB_MhegDvbLocatorToIndex.
 * @param   eventIndex 0 for the current event, 1 for the following event
 * @param   adLangs A space separated list of 3 character language codes.
 *          Each code corresponds to an audio description stream
 *          that is available to the event as signalled by the
 *          EITpf.
 * @return  The following return values are allowed from this function
 *          MHERR_OK                 - Success
 *          MHERR_BAD_PARAMETER  - Invalid parameter.
 *          MHERR_OTHER          - Controlling application specific
 *          error.
 */
E_MhegErr DVB_MhegGetADLangs(S32BIT serviceIndex, S32BIT porfIndex, U8BIT *adLangs)
{
   E_MhegErr result;

   FUNCTION_START(DVB_MhegGetADLangs);

   result = GetLangs(serviceIndex, porfIndex, adLangs, IsAudioDescription);

   FUNCTION_FINISH(DVB_MhegGetADLangs);

   return result;
}

/**
 * @brief   Return a list of available audio languages for the specified event.
 *          The language list is extracted from the component descriptors of
 *          the present or following event (EITpf). It is returned as a
 *          null-terminated sequence of 3-letter codes separated by a space
 *          character.
 *          The buffer provided is allocated by the MHEG-5 engine and is
 *          large enough to store up to 64 languages (256 bytes including the
 *          null-terminator).
 *          This MUST be a non-blocking function, returning results immediately.
 * @param   serviceIndex A receiver specific integer identifying a service.
 *          This value shall be consistent with the values
 *          returned by DVB_MhegDvbLocatorToIndex.
 * @param   eventIndex 0 for the current event, 1 for the following event
 * @param   audioLangs A space separated list of 3 character language codes.
 *          Each code corresponds to an audio stream that is
 *          available to the event as signalled by the EITpf.
 * @return  The following return values are allowed from this function
 *          MHERR_OK                 - Success
 *          MHERR_BAD_PARAMETER  - Invalid parameter.
 *          MHERR_OTHER          - Controlling application specific
 *          error.
 */
E_MhegErr DVB_MhegGetAudioLangs(S32BIT serviceIndex, S32BIT porfIndex, U8BIT *audioLangs)
{
   E_MhegErr result;

   FUNCTION_START(DVB_MhegGetAudioLangs);

   result = GetLangs(serviceIndex, porfIndex, audioLangs, IsAudio);

   FUNCTION_FINISH(DVB_MhegGetAudioLangs);

   return result;
}

/**
 * @brief   Return a list of available subtitles languages for the specified event.
 *          The language list is extracted from the component descriptors of
 *          the present or following event (EITpf). It is returned as a
 *          null-terminated sequence of 3-letter codes separated by a space
 *          character.
 *          The buffer provided is allocated by the MHEG-5 engine and is
 *          large enough to store up to 64 languages (256 bytes including the
 *          null-terminator).
 *          This MUST be a non-blocking function, returning results immediately.
 * @param   serviceIndex A receiver specific integer identifying a service.
 *          This value shall be consistent with the values
 *          returned by DVB_MhegDvbLocatorToIndex.
 * @param   eventIndex 0 for the current event, 1 for the following event
 * @param   subsLangs A space separated list of 3 character language codes.
 *          Each code corresponds to a subtitles stream that is
 *          available to the event as signalled by the EITpf.
 * @return  The following return values are allowed from this function
 *          MHERR_OK                 - Success
 *          MHERR_BAD_PARAMETER  - Invalid parameter.
 *          MHERR_OTHER          - Controlling application specific
 *          error.
 */
E_MhegErr DVB_MhegGetSubtitleLangs(S32BIT serviceIndex, S32BIT porfIndex, U8BIT *subsLangs)
{
   E_MhegErr result;

   FUNCTION_START(DVB_MhegGetSubtitleLangs);

   result = GetLangs(serviceIndex, porfIndex, subsLangs, IsSubtitle);

   FUNCTION_FINISH(DVB_MhegGetSubtitleLangs);

   return result;
}

/**
 * @brief   Return the banner transparency as a percentage 0-100, where 100 is
 *          fully transparent.
 *          This setting is related to the information banner on the native UI
 *          of the receiver.
 * @param   transparency  Pointer to return banner transparency as a percentage
 */
E_MhegErr DVB_MhegGetBannerTransparency(S32BIT *transparency)
{
   S_ACB_UI_INFO info;
   BOOLEAN result = MHERR_OTHER;

   FUNCTION_START(DVB_MhegGetBannerTransparency);

   info.type = ACB_GET_UI_BANNER_TRANSPARENCY;
   if (ACB_GetUIInformation(&info) == TRUE)
   {
      *transparency = info.u.banner_transparency;
      result = MHERR_OK;
   }

   FUNCTION_FINISH(DVB_MhegGetBannerTransparency);

   return result;
}

/**
 * @brief   Return the banner timeout period in seconds.
 *          This setting is related to the information banner on the native UI
 *          of the receiver.
 * @param   timeout  Pointer to return banner timeout in seconds
 */
E_MhegErr DVB_MhegGetBannerTimeout(S32BIT *timeout)
{
   S_ACB_UI_INFO info;
   BOOLEAN result = MHERR_OTHER;

   FUNCTION_START(DVB_MhegGetBannerTimeout);

   info.type = ACB_GET_UI_BANNER_TIMEOUT;
   if (ACB_GetUIInformation(&info) == TRUE)
   {
      *timeout = info.u.banner_timeout;
      result = MHERR_OK;
   }

   FUNCTION_FINISH(DVB_MhegGetBannerTimeout);

   return result;
}

/**
 * @brief   Return the language of the native UI (menu etc.) as a three character
 *          code as defined by ISO 639-2.
 *          This is a non-blocking function.
 * @param   timeout OSD language
 * @return  The following return values are allowed from this function
 *          MHERR_OK                - Success.
 *          MHERR_BAD_PARAMETER - Invalid parameter.
 *          MHERR_OTHER         - Controlling application specific error.
 */
E_MhegErr DVB_MhegGetOSDLanguage(U8BIT *language)
{
   S_ACB_UI_INFO info;
   BOOLEAN result = MHERR_OTHER;
   U8BIT *osd_language;
   U8BIT *lang_ids;

   FUNCTION_START(DVB_MhegGetOSDLanguage);

   info.type = ACB_GET_UI_LANG_PREF;
   if (ACB_GetUIInformation(&info) == TRUE)
   {
      lang_ids = ACFG_GetDbLangId(ACFG_GetCountry(),info.u.ui_lang_id);
      if (lang_ids != NULL)
      {
         osd_language = STB_GCGetLangCodeString(ACFG_ConvertLangIdToCode(lang_ids[0]));
         language[0] = osd_language[0];
         language[1] = osd_language[1];
         language[2] = osd_language[2];
         result = MHERR_OK;
      }
   }

   FUNCTION_FINISH(DVB_MhegGetOSDLanguage);

   return result;
}

/**
 * @brief   Advise DVB stack that PMT may be required soon in DVB_MhegVideoPlayStream() or
 *          DVB_MhegAudioPlayStream(). This gives DVB stack opportunity to prioritise
 *          aquiring of PMT for the specified service. This service will not be the one
 *          currently tuned to.
 * @param   pDvbLocator pointer to Digital Video Broadcaster (DVB) locator information.
 * @return  void
 */
void DVB_MhegServiceAddPmtListen(S_DvbLocator dvbLocator)
{
   FUNCTION_START(DVB_MhegServiceAddPmtListen);

   ASI_AddServiceToPmtList(dvbLocator.service_id);

   FUNCTION_FINISH(DVB_MhegServiceAddPmtListen);
}

/**
 * @brief   Advise DVB stack to remove prioritising of PMT for the service that was
 *          set up by DVB_MhegServiceAddPmtListen().
 * @param   pDvbLocator pointer to Digital Video Broadcaster (DVB) locator information.
 * @return  void
 */
void DVB_MhegServiceRemovePmtListen(S_DvbLocator dvbLocator)
{
   FUNCTION_START(DVB_MhegServiceRemovePmtListen);

   ASI_RemoveServiceFromPmtList(dvbLocator.service_id);

   FUNCTION_FINISH(DVB_MhegServiceRemovePmtListen);
}

#if defined(COMMON_INTERFACE)
/**
 * @brief   Notify that DVB is about to tune to service or transport stream.
 *          This function would be called as a result of STB_CITuneBroadcastRequest(),
 *          STB_CITuneLCNRequest(), STB_CITuneTripletRequest(), or STB_CITuneIPRequest()
 *          but before tuning occurs.
 * @param   onet_id original network id
 * @param   trans_id transport id
 * @param   serv_id service id
 * @param   tune flags Presentation engine tuning flags (Quiet tune / Keep app running)
 * @return  n/a
 */
static void NotifyStartTuning(U32BIT module, U16BIT onet_id, U16BIT trans_id, U16BIT serv_id, E_CIP_TUNE_FLAGS flags)
{
   void *srv_ptr;

   FUNCTION_START(NotifyStartTuning);

   if (notify_tuning_func)
   {
      if (trans_id == 0)
      {
         trans_id = ADB_INVALID_DVB_ID;
      }
      srv_ptr = ADB_FindServiceByIds(onet_id, trans_id, serv_id);
      if (srv_ptr != NULL)
      {
         notify_tuning_func((S32BIT)ADB_GetServiceLcn(srv_ptr), (E_TuningFlags)flags);
      }
   }

   FUNCTION_FINISH(NotifyStartTuning);
}

/**
 * @brief   Register callback function to notify start of tuning process due to CI
 *          tune request
 * @param   cb_func Callback function - NULL disables the call.
 */
void DVB_MhegRegisterDvbTuning( F_NotifyDvbTuning cb_func )
{
   FUNCTION_START(DVB_MhegRegisterDvbTuning);

   notify_tuning_func = cb_func;
   ACI_RegisterStartTuneNotifyCallback((cb_func != NULL)? NotifyStartTuning : NULL);

   FUNCTION_FINISH(DVB_MhegRegisterDvbTuning);
}
#endif /*COMMON_INTERFACE*/


/**
 * @brief   Get the host service mode from external application. This is
 *          the current service type choosen by the user.
 * @return  E_ServiceType - TV (including data) services, or Radio services
 *          or All Services.
 */
E_ServiceType DVB_MhegGetServiceMode(void)
{
   S_ACB_UI_INFO info;
   E_ServiceType serv_type = SRV_TYP_ALL;

   FUNCTION_START(DVB_MhegGetOSDLanguage);

   info.type = ACB_GET_UI_SERVICE_MODE;
   if (ACB_GetUIInformation(&info) == TRUE)
   {
      serv_type = (E_ServiceType)info.u.serv_type;
   }

   FUNCTION_FINISH(DVB_MhegGetOSDLanguage);

   return serv_type;
}



