/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2014 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief      DVB Interface for MHEG5 engine - Audio
 * @file       ics.c
 * @date       Janauary 2014
 * @author     Adam Sturtridge
 */
//#define DEBUG_PRINTING_ENABLED

/*---includes for this file--------------------------------------------------*/

/* compiler library header files */
#include <stdio.h>

/* third party header files */

#include <techtype.h>
#include <dbgfuncs.h>

#include "stberc.h"
#include "stbdpc.h"
#include "ap_dbacc.h"
#include "ap_cntrl.h"
#include "app_nvm.h"
#include "stbhwdmx.h"
#include "dvbmh_int.h"
#include "stb_memory.h"

#include "dvb_ics.h"
#include "dvb_audio.h"
#include "dvb_video.h"


/*---constant definitions for this file--------------------------------------*/

#define LANG_CODE_UND 0x756e64

/*---local typedef structs for this file-------------------------------------*/

/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local)                 */

static U8BIT streaming_path = INVALID_RES_ID;
static U8BIT monitor_si_path = INVALID_RES_ID;
static void *tuned_service_ptr = NULL;

static S_ICSLangInfo *g_lang_info = NULL;
static U16BIT g_lang_total = 0;

/*---local function prototypes for this file---------------------------------*/
/*   (internal functions declared static to make them local)                 */

/*---local function definitions----------------------------------------------*/

/**
 * @brief   Determines subtitle PID and whether it is DVB or teletext, for the given service
 * @param   serv_ptr service
 * @param   subt_info pointer to ADB_SUBT_INFO for extra info on subtitles
 * @return  PID to collect subtitles data, zero if none available
 */
static U16BIT GetSubtitlePid(void *serv_ptr, ADB_SUBT_INFO *subt_info)
{
   S_ICSLangInfo *lang_info;
   U32BIT langs[4];
   U32BIT lang;
   U16BIT count, pid = 0;
   U8BIT scnt, i, reqd_type;

#ifdef DEBUG_PRINTING_ENABLED
   DBGPRINT("serv_ptr=%p lang_total=%u",serv_ptr,g_lang_total);
#else
   USE_UNWANTED_PARAM(serv_ptr);
#endif

   if (g_lang_total && g_lang_info)
   {
      if (ASPECT_RATIO_4_3 == (E_STB_AV_ASPECT_RATIO)APP_NvmRead(ASPECT_RATIO_NVM))
      {
         if (SUBTITLE_NORMAL == (E_SUBTITLE_TYPE)APP_NvmRead(SUBTITLE_TYPE_NVM))
         {
            reqd_type = ADB_SUBTITLE_TYPE_DVB_4_3;
         }
         else
         {
            reqd_type = ADB_SUBTITLE_TYPE_DVB_HARD_HEARING_4_3;
         }
      }
      else
      {
         if (SUBTITLE_NORMAL == (E_SUBTITLE_TYPE)APP_NvmRead(SUBTITLE_TYPE_NVM))
         {
            reqd_type = ADB_SUBTITLE_TYPE_DVB_16_9;
         }
         else
         {
            reqd_type = ADB_SUBTITLE_TYPE_DVB_HARD_HEARING_16_9;
         }
      }
      scnt = DVB_MhegPrefSubtitleLangs(langs,3);
      langs[scnt] = LANG_CODE_UND;
      scnt++;
      for (i = 0; i != scnt; i++)
      {
         lang = langs[i];
         for (lang_info = g_lang_info, count = 0; count != g_lang_total; count++, lang_info++)
         {
            if (lang_info->lang_code == lang && lang_info->s_type == reqd_type)
            {
               pid = lang_info->pid;
               subt_info->u.subt.cpage = lang_info->u.page.composition;
               subt_info->u.subt.apage = lang_info->u.page.ancillary;
               break;
            }
         }
         if (pid != 0) break;
         for (lang_info = g_lang_info, count = 0; count != g_lang_total; count++, lang_info++)
         {
            if (lang_info->lang_code == lang &&
               (lang_info->s_type&0xF0) == (reqd_type&0xF0))
            {
               pid = lang_info->pid;
               subt_info->u.subt.cpage = lang_info->u.page.composition;
               subt_info->u.subt.apage = lang_info->u.page.ancillary;
               break;
            }
         }
         if (pid != 0) break;
      }
      subt_info->is_dvb_subt = TRUE;
      DBGPRINT("PID=%u cpage=%u apage=%u",pid, subt_info->u.subt.cpage, subt_info->u.subt.apage);
   }
   return pid;
}

/*---global function definitions---------------------------------------------*/


/**
 * @brief   This function tells the external application that an IC delivered
 *          stream is to be played using DVB_MhegHandleStreamData.
 *          The function provides the pids to be used in decoding the stream.
 *          Where a component is not active, the PID value will be zero.
 *          The function also provides optional encryption keys for audio and
 *          video components.
 *          The controlling application may return immediately with status
 *          MHERR_OK, indicating that streaming can start. If this cannot be
 *          done synchronously, the controlling application may return
 *          MHERR_PENDING, and when ready it can send DVB_EVENT_STREAMING_READY
 * @param   pids Pids for audio, video, subtitle and PCR for IP service
 * @param   audio_keys Information about audio encryption keys (or NULL)
 * @param   video_keys Information about video encryption keys (or NULL)
 * @param   video_termination if video active, termination attribute to be applied
 *          to the last frame.
 * @return  The following return values are allowed from this function:
 *          MHERR_OK            - Success
 *          MHERR_PENDING       - Start is pending
 *          MHERR_OTHER         - Controlling application specific error
 */
E_MhegErr DVB_MhegICStreamSetup(S_StreamPids pids, S_ICSPidKeys *audio_keys,
   S_ICSPidKeys *video_keys, E_VideoTermination video_termination,
   E_ICSAudioCodec audio_codec, E_ICSVideoCodec video_codec)
{
   E_MhegErr result = MHERR_OTHER;
   U8BIT live_path, dmx_path;

   FUNCTION_START(DVB_MhegICStreamSetup);
   USE_UNWANTED_PARAM(video_termination);

   if (streaming_path == INVALID_RES_ID)
   {
      /* Release the decoders from the live path so that playback can start.
       * This means the live path will continue to monitor SI data */
      live_path = STB_DPGetLivePath();
      if (live_path != INVALID_RES_ID)
      {
         tuned_service_ptr = ADB_GetTunedService(live_path);
         ACTL_TuneOff(live_path);
         STB_DPReleasePath(live_path, RES_OWNER_NONE);
      }
      else
      {
         tuned_service_ptr = NULL;
      }
      streaming_path = STB_DPAcquirePlaybackPath(tuned_service_ptr);
   }
   if (streaming_path != INVALID_RES_ID)
   {
      result = MHERR_OK;
      if (pids.audio_pid == 0 && pids.video_pid == 0)
      {
         STB_DPSetPCRPID(streaming_path, 0);
      }
      else
      {
         STB_DPSetPCRPID(streaming_path, pids.pcr_pid);
      }
      if (pids.audio_pid != 0)
      {
         switch (audio_codec)
         {
            case ICS_AC_AAC_ADTS:
               DVBMH_PlayAudio(streaming_path, pids.audio_pid, AUDIO_CODEC_AAC_ADTS, INVALID_PID);
               break;
            case ICS_AC_HE_AAC_LATM:
               DVBMH_PlayAudio(streaming_path, pids.audio_pid, AUDIO_CODEC_HEAAC, INVALID_PID);
               break;
            case ICS_AC_EAC3:
               DVBMH_PlayAudio(streaming_path, pids.audio_pid, AUDIO_CODEC_EAC3, INVALID_PID);
               break;
            default:
               break;
         }
         if (audio_keys != NULL)
         {
            dmx_path = STB_DPGetPathDemux(streaming_path);
            STB_DMXGetDescramblerKey(dmx_path, DESC_TRACK_AUDIO);
            STB_DMXSetKeyUsage(dmx_path, DESC_TRACK_AUDIO, KEY_USAGE_TRANSPORT);
            STB_DMXSetDescramblerType(dmx_path, DESC_TRACK_AUDIO, DESC_TYPE_AES_SCTE_52);
            STB_DMXSetDescramblerKeyData(dmx_path, DESC_TRACK_AUDIO, KEY_PARITY_EVEN, audio_keys->even);
            STB_DMXSetDescramblerKeyData(dmx_path, DESC_TRACK_AUDIO, KEY_PARITY_ODD, audio_keys->odd);
         }
      }
      if (pids.video_pid != 0)
      {
         switch (video_codec)
         {
            case ICS_VC_H264:
               DVBMH_PlayVideo(streaming_path,pids.video_pid,VIDEO_CODEC_H264);
               break;
            default:
               break;
         }
         if (video_keys != NULL)
         {
            dmx_path = STB_DPGetPathDemux(streaming_path);
            STB_DMXGetDescramblerKey(dmx_path, DESC_TRACK_VIDEO);
            STB_DMXSetKeyUsage(dmx_path, DESC_TRACK_VIDEO, KEY_USAGE_TRANSPORT);
            STB_DMXSetDescramblerType(dmx_path, DESC_TRACK_VIDEO, DESC_TYPE_AES_SCTE_52);
            STB_DMXSetDescramblerKeyData(dmx_path, DESC_TRACK_VIDEO, KEY_PARITY_EVEN, video_keys->even);
            STB_DMXSetDescramblerKeyData(dmx_path, DESC_TRACK_VIDEO, KEY_PARITY_ODD, video_keys->odd);
         }
      }
      g_lang_info = pids.info_ptr;
      g_lang_total = pids.info_num;
      ACTL_SetMhegSubtitlePidFunc(GetSubtitlePid);
      if (g_lang_info && DVBMH_ShowSubtitles() && !ACTL_AreSubtitlesStarted() && DVB_MhegGetSubtitlePref())
      {
         ACTL_StartSubtitles();
      }
      if (tuned_service_ptr != NULL)
      {
         /* Acquire a decode path to monitor the SI data while playback is in progress */
         monitor_si_path = ACTL_AcquirePathForService(tuned_service_ptr, FALSE, FALSE, NULL);
         if (monitor_si_path != INVALID_RES_ID)
         {
            ACTL_TuneToService(monitor_si_path, NULL, tuned_service_ptr, TRUE, FALSE);
         }
      }
   }

   FUNCTION_FINISH(DVB_MhegICStreamSetup);

   return result;
}

/**
 * @brief  This function tells the external application that the IC delivered
 *         stream is stopped. Presentation of all stream components (audio,
 *         video and/or subtitles) should cease.
 *         This function may be called while DVB_MhegHandleStreamData is in progress.
 *         In this case DVB_MhegHandleStreamData should return indicating that the
 *         entire block has been processed.
 * @return The following return values are allowed from this function
 *         MHERR_OK            - Success
 *         MHERR_OTHER         - Controlling application specific error
 */
E_MhegErr DVB_MhegICStreamRelease(void)
{
   E_MhegErr result = MHERR_OK;
   U8BIT path;

   FUNCTION_START(DVB_MhegICStreamRelease);

   if (streaming_path != INVALID_RES_ID)
   {
      ACTL_StopSubtitles();
      ACTL_SetMhegSubtitlePidFunc(NULL);
      if (g_lang_info)
      {
         STB_MemFree( g_lang_info );
         g_lang_info = NULL;
         g_lang_total= 0;
      }
      STB_AVBlankVideo(STB_DPGetPathVideoDecoder(streaming_path), TRUE);
      STB_DPStopVideoDecoding(streaming_path);
      STB_DPStopAudioDecoding(streaming_path);
      STB_DPSetAudioPID(streaming_path, 0);
      STB_DPSetVideoPID(streaming_path, 0);
      STB_DPSetPCRPID(streaming_path, 0);
      /* Free all the descramblers (if they haven't been allocated it shouldn't be a problem */
      STB_DMXFreeDescramblerKey(STB_DPGetPathDemux((streaming_path)), DESC_TRACK_AUDIO);
      STB_DMXFreeDescramblerKey(STB_DPGetPathDemux((streaming_path)), DESC_TRACK_VIDEO);
      STB_DPReleasePath(streaming_path, RES_OWNER_NONE);
      streaming_path = INVALID_RES_ID;
   }

   if (monitor_si_path != INVALID_RES_ID)
   {
      ACTL_TuneOff(monitor_si_path);
      STB_DPReleasePath(monitor_si_path, RES_OWNER_NONE);
      monitor_si_path = INVALID_RES_ID;
   }

   if (tuned_service_ptr != NULL)
   {
      if (!ACTL_CanServiceBeViewed(tuned_service_ptr))
      {
         DBGPRINT("Saved service %p cannot be viewed", tuned_service_ptr)
         /* All tuners are being used and the service being watched when playback was started
          * can no longer be tuned to, so need to tune to one that can */
         tuned_service_ptr = NULL;

         for (path = 0; (path < STB_DPGetNumPaths()) && (tuned_service_ptr == NULL); path++)
         {
            tuned_service_ptr = ADB_GetTunedService(path);
         }

         DBGPRINT("Tuned service is %p", tuned_service_ptr)
      }

      if (tuned_service_ptr != NULL)
      {
         ACTL_AcquirePathForService(tuned_service_ptr, TRUE, FALSE, NULL);
         tuned_service_ptr = NULL;
      }
   }

   FUNCTION_FINISH(DVB_MhegICStreamRelease);

   return result;
}

/**
 * @brief  Notify video has started, and unblank video if IC Stream is valid
 */
void DVBMH_NotifyVideoStarted(void)
{
   FUNCTION_START(DVBMH_NotifyVideoStarted);

   if (streaming_path != INVALID_RES_ID)
   {
      STB_AVBlankVideo(STB_DPGetPathVideoDecoder(streaming_path), FALSE);
   }

   FUNCTION_FINISH(DVBMH_NotifyVideoStarted);
}

/**
 * @brief  Handle transport stream data. The data is part of a single-program
 *         transport stream, containing audio, video and/or subtitles (in one
 *         or more languages). The video is H.264 SD video and the audio is
 *         HE-AAC audio. The stream contains PAT and PMT (other SI data can
 *         be ignored).
 *         This function can block until the data has been processed / buffered.
 *         It may also return when part of the block has been processed. This
 *         would cause a subsequent call with the rest of the block.
 *         When the last block is presented (last=TRUE), the function should
 *         indicate that the block has been completely processed only when this
 *         is a true reflection of the presentation status.
 * @param  data Stream data (whole transport stream packets)
 * @param  len Length of data in bytes (multiple of 188)
 * @param  last Whether data block contains the end of the stream
 * @return Number of bytes handled. This must be a multiple of 188.
 */
U32BIT DVB_MhegICStreamHandleData(U8BIT *data, U32BIT len, BOOLEAN last)
{
   U32BIT result;

   FUNCTION_START(DVB_MhegICStreamHandleData);
   USE_UNWANTED_PARAM(last);

   if (streaming_path == INVALID_RES_ID)
   {
      result = 0;
   }
   else
   {
      STB_DMXWriteDemux(STB_DPGetPathDemux(streaming_path), data, len);
      result = len;
   }

   FUNCTION_FINISH(DVB_MhegICStreamHandleData);

   return result;
}

/******************************************************************************
 * @brief  Pause presentation of audio and/or video components of the currently
 *         playing IC delivered stream. Following this call, the video (if any)
 *         should be frozen and audio muted. Unused data delivered by
 *         DVB_MhegHandleStreamData may be returned to the MHEG-5 engine (using
 *         the return value).
 * @return The following return values are allowed from this function
 *         MHERR_OK            - Success
 *         MHERR_OTHER         - Controlling application specific error
 ******************************************************************************/
E_MhegErr DVB_MhegICStreamPause(void)
{
   E_MhegErr result;
   U8BIT decoder;

   FUNCTION_START(DVB_MhegICStreamPause);

   if (streaming_path == INVALID_RES_ID)
   {
      result = MHERR_OTHER;
   }
   else
   {
      decoder = STB_DPGetPathVideoDecoder(streaming_path);
      if (decoder == INVALID_RES_ID)
      {
         result = MHERR_OTHER;
      }
      else
      {
         STB_AVPauseVideoDecoding(decoder);
         result = MHERR_OK;
      }
   }

   FUNCTION_FINISH(DVB_MhegICStreamPause);

   return result;
}

/******************************************************************************
 * @brief  Resume presentation of audio and/or video components of the currently
 *         playing IC delivered stream. This function may be called by the
 *         MHEG-5 engine after DVB_MhegPauseICS (alternatively, DVB_MhegStopICS
 *         may be called). Following this call, available data is once again
 *         delivered through DVB_MhegHandleStreamData.
 * @return The following return values are allowed from this function (see
 *         note [a]):
 *         MHERR_OK            - Success
 *         MHERR_OTHER         - Controlling application specific error
 ******************************************************************************/
E_MhegErr DVB_MhegICStreamResume(void)
{
   E_MhegErr result;
   U8BIT decoder;

   FUNCTION_START(DVB_MhegICStreamResume);

   if (streaming_path == INVALID_RES_ID)
   {
      result = MHERR_OTHER;
   }
   else
   {
      decoder = STB_DPGetPathVideoDecoder(streaming_path);
      if (decoder == INVALID_RES_ID)
      {
         result = MHERR_OTHER;
      }
      else
      {
         STB_AVResumeVideoDecoding(decoder);
         result = MHERR_OK;
      }
   }

   FUNCTION_FINISH(DVB_MhegICStreamResume);

   return result;
}

/******************************************************************************
 * @brief  Set volume for audio component of the currently playing IC stream
 *         This function is equivalent to DVB_MhegAudioSetVolume.
 * @param  volumeAdjust New setting of the amount by which to adjust volume in dB
 * @return The following return values are allowed from this function
 *         MHERR_OK            - Success
 *         MHERR_BAD_PARAMETER - Invalid parameter
 *         MHERR_OTHER         - Controlling application specific error
 ******************************************************************************/
E_MhegErr DVB_MhegICStreamAudioSetVolume(S32BIT volumeAdjust)
{
   FUNCTION_START(DVB_MhegICStreamAudioSetVolume);
   FUNCTION_FINISH(DVB_MhegICStreamAudioSetVolume);
   return DVB_MhegAudioSetVolume(volumeAdjust);
}

/******************************************************************************
 * @brief  This function asks the external application to obtain permission from
 *         the user to present IP-delivered content. The external application may
 *         use the "restriction" string to tell the user why access to the
 *         content is restricted. How permission is obtained is receiver specific.
 * @param  restriction A string that can be presented to the user to indicate
 *         why verification is required
 * @param  show Indicates whether the prompt should be shown or hidden
 *         (TRUE=show, FALSE=hide/remove)
 * @return The following return values are allowed from this function
 *         MHERR_OK            - Success
 *         MHERR_BAD_PARAMETER - Invalid parameter
 *         MHERR_OTHER         - Controlling application specific error
 ******************************************************************************/
E_MhegErr DVB_MhegPromptForGuidance(U8BIT *restriction, BOOLEAN show)
{
   E_MhegErr result;

   FUNCTION_START(DVB_MhegPromptForGuidance);

   if (restriction == NULL)
   {
      result = MHERR_BAD_PARAMETER;
   }
   else
   {
      if (!show)
      {
         restriction = NULL;
      }
      DBGPRINT("PIN request %s",(show)?"show":"hide")
      STB_ERSendEvent(FALSE, FALSE, EV_CLASS_MHEG, EV_TYPE_MHEG_PIN_REQUEST, &restriction, sizeof(U8BIT*));
      result = MHERR_OK;
   }

   FUNCTION_FINISH(DVB_MhegPromptForGuidance);

   return result;
}

/******************************************************************************
 * @brief  Returns the PIN support in the receiver.
 *         This MUST be a non-blocking function, returning results immediately.
 * @param  status MHEG5_ICS_PIN_SUPPORT_NONE - PIN is not supported
 *         MHEG5_ICS_PIN_SUPPORT_DISABLED - PIN is supported and disabled
 *         MHEG5_ICS_PIN_SUPPORT_ENABLED - PIN is supported and enabled
 * @return The following return values are allowed from this function
 *         MHERR_OK            - Success
 *         MHERR_BAD_PARAMETER - Invalid parameter
 *         MHERR_OTHER         - Controlling application specific error
 ******************************************************************************/
E_MhegErr DVB_MhegGetPINSupport(E_ICSPinSupport *support)
{
   FUNCTION_START(DVB_MhegGetPINSupport);

   if (((E_PARENTAL_LOCK)APP_NvmRead(PARENTAL_LOCK_NVM) == PARENTAL_LOCK_ON))
   {
      *support = MHEG5_ICS_PIN_SUPPORT_ENABLED;
   }
   else
   {
      *support = MHEG5_ICS_PIN_SUPPORT_DISABLED;
   }

   FUNCTION_FINISH(DVB_MhegGetPINSupport);

   return MHERR_OK;
}

#ifdef INCLUDE_FREESAT
/******************************************************************************
 * @brief  Tells whether an IP connection is allowed on the current service.
 *         This information is extracted from the free_satellite_servicegroup
 *         descriptor, which contains the return_channel_access_flag. If this
 *         flag is set to 1 in any of the groups of which the current service
 *         is a member, then the connection is allowed. Otherwise, it is not
 *         allowed.
 *         This MUST be a non-blocking function, returning results immediately.
 * @param  pAllowed Whether IP connection is allowed for current service
 * @return The following return values are allowed from this function (see
 *         note [a]):
 *         MHERR_OK            - Success
 *         MHERR_BAD_PARAMETER - Invalid parameter
 *         MHERR_OTHER         - Controlling application specific error
 ******************************************************************************/
E_MhegErr DVB_MhegIsConnectionAllowed(BOOLEAN *pAllowed)
{
   E_MhegErr result;

   FUNCTION_START(DVB_Mheg);

   result = MHERR_OK;

   FUNCTION_FINISH(DVB_Mheg);

   return result;
}

/******************************************************************************
 * @brief  Returns whether IP streamed content is currently protected by a PIN.
 *         This MUST be a non-blocking function, returning results immediately.
 * @param  status MHEG5_ICS_PIN_REQ_NEVER - no PIN protection
 *         MHEG5_ICS_PIN_REQ_GUIDANCE_ONLY - PIN required for "Guidance" only
 *         MHEG5_ICS_PIN_REQ_ALWAYS - PIN required for all streamed content
 * @return The following return values are allowed from this function
 *         MHERR_OK                - Success
 *         MHERR_BAD_PARAMETER - Invalid parameter
 *         MHERR_OTHER         - Controlling application specific error
 ******************************************************************************/
E_MhegErr DVB_MhegGetPINRequirement(E_ICSPinReq *status)
{
   E_MhegErr result;

   FUNCTION_START(DVB_Mheg);

   result = MHERR_OK;

   FUNCTION_FINISH(DVB_Mheg);

   return result;
}

/******************************************************************************
 * @brief  Determine whether the PIN number entered is valid.
 *         This MUST be a non-blocking function, returning results immediately.
 * @param  pin The PIN entered by the viewer (null terminated)
 * @param  status MHEG5_ICS_PIN_STATUS_NOT_REQUIRED - PIN not required
 *         MHEG5_ICS_PIN_STATUS_OK           - PIN ok
 *         MHEG5_ICS_PIN_STATUS_INVALID      - PIN invalid
 * @return The following return values are allowed from this function
 *         MHERR_OK            - Success
 *         MHERR_BAD_PARAMETER - Invalid parameter
 *         MHERR_OTHER         - Controlling application specific error
 ******************************************************************************/
E_MhegErr DVB_MhegValidatePIN(U8BIT *pin, E_ICSPinStatus *status)
{
   E_MhegErr result;

   FUNCTION_START(DVB_Mheg);

   result = MHERR_OK;

   FUNCTION_FINISH(DVB_Mheg);

   return result;
}

/******************************************************************************
 * @brief  Return the MAC address of the receiver. The MAC address is a 6-byte
 *         value (usually written as xx:xx:xx:xx:xx:xx using hexadecimal
 *         digits).
 *         This MUST be a non-blocking function, returning results immediately.
 * @param  address The MAC address (6 bytes)
 * @return The following return values are allowed from this function
 *         MHERR_OK            - Success
 *         MHERR_BAD_PARAMETER - Invalid parameter
 *         MHERR_OTHER         - Controlling application specific error
 ******************************************************************************/
E_MhegErr DVB_MhegGetMACAddress(U8BIT address[6])
{
   E_MhegErr result;

   FUNCTION_START(DVB_Mheg);

   result = MHERR_OK;

   FUNCTION_FINISH(DVB_Mheg);

   return result;
}

/******************************************************************************
 * @brief  Return the private key of the receiver and its version.
 *         The private key is an ASN.1 encoded 1024-bit RSA key in DER format.
 *         The version is an integer.
 *         The key should be copied into the buffer (the buffer is cleared
 *         after the key is used). If the buffer is too small for the key,
 *         nothing should be copied. The length of the key in bytes is
 *         returned in any case.
 *         This MUST be a non-blocking function, returning results immediately.
 * @param  buffer Buffer for the key
 * @param  bufsize Buffer size in bytes
 * @param  keylen Length of key in bytes
 * @param  version Key version
 * @return The following return values are allowed from this function
 *         MHERR_OK            - Success
 *         MHERR_BAD_PARAMETER - Invalid parameter
 *         MHERR_OTHER         - Controlling application specific error
 ******************************************************************************/
E_MhegErr DVB_MhegGetPrivateKey(U8BIT *buffer, U16BIT bufsize,
   U16BIT *keylen, U32BIT *version)
{
   E_MhegErr result;

   FUNCTION_START(DVB_Mheg);

   result = MHERR_OK;

   FUNCTION_FINISH(DVB_Mheg);

   return result;
}

#endif /* INCLUDE_FREESAT */
