/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2014 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief      DVB Interface for MHEG5 engine - Tuner
 * @file       video.c
 * @date       February 2014
 * @author     Adam Sturtridge
 */
/*#define DEBUG_PRINTING_ENABLED*/

/*---includes for this file--------------------------------------------------*/

/* compiler library header files */
#include <stdio.h>

/* third party header files */

#include <techtype.h>
#include <dbgfuncs.h>

#include "stberc.h"
#include "stbhwav.h"
#include "stbdpc.h"
#include "stbvtc.h"

#include "ap_dbacc.h"
#include "ap_cntrl.h"
#include "app.h"
#include "ap_cfg.h"
#include "ap_uiinfo.h"
#include "dvbmh_int.h"

#include "dvb_video.h"

/*---constant definitions for this file--------------------------------------*/

/*---local typedef structs for this file-------------------------------------*/

/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local)                 */
static E_DvbSubtitleShowState subt_show_state = MHEG5_DVB_SUBTITLE_SHOW;

/*---local function prototypes for this file---------------------------------*/
/*   (internal functions declared static to make them local)                 */

/*---local function definitions----------------------------------------------*/

static E_STB_DP_VIDEO_CODEC EvalVideoCodec(void *service, ADB_STREAM_TYPE stream_type)
{
   E_STB_DP_VIDEO_CODEC codec = VIDEO_CODEC_AUTO;
   DBGPRINT("stream_type=%x ServiceType=%x",stream_type,ADB_GetServiceType(service))
   switch (stream_type)
   {
      case ADB_H264_VIDEO_STREAM:
      {
         codec = VIDEO_CODEC_H264;
         break;
      }
      case ADB_VIDEO_STREAM:
      {
         codec = VIDEO_CODEC_MPEG2;
         break;
      }
      case ADB_H265_VIDEO_STREAM:
      {
         codec = VIDEO_CODEC_H265;
         break;
      }
      default:
      case ADB_DATA_STREAM:
      {
         switch (ADB_GetServiceType(service))
         {
            default:
            {
               DBGPRINT("ServiceType=%x",ADB_GetServiceType(service))
               /* Fall tho */
            }
            case ADB_SERVICE_TYPE_TV:
            case ADB_SERVICE_TYPE_MPEG2_HD:
            {
               codec = VIDEO_CODEC_MPEG2;
               break;
            }
            case ADB_SERVICE_TYPE_AVC_SD_TV:
            case ADB_SERVICE_TYPE_HD_TV:
            {
               codec = VIDEO_CODEC_H264;
               break;
            }
            case ADB_SERVICE_TYPE_UHD_TV:
            {
               codec = VIDEO_CODEC_H265;
               break;
            }
         }
         break;
      }
   }
   return codec;
}

/*---global function definitions---------------------------------------------*/

void DVBMH_PlayVideo(U8BIT path, U16BIT video_pid, E_STB_DP_VIDEO_CODEC codec)
{
   U16BIT prev_pid;
   FUNCTION_START(DVBMH_PlayVideo);

   /* Difference from SetupForDecoding(): MHEG doesn't specify that H265 is supported */
   if (codec == VIDEO_CODEC_H265)
   {
      codec = VIDEO_CODEC_AUTO;
   }

   prev_pid = STB_DPGetVideoPID(path);
   if (video_pid != prev_pid || codec != STB_DPGetVideoCodec(path))
   {
      if (STB_DPGetVideoStatus(path) == DECODE_RUNNING)
      {
         STB_DPStopVideoDecoding(path);
      }

      STB_DPSetVideoCodec(path, codec);
      STB_DPSetVideoPID(path, video_pid);
   }
   if (STB_DPGetVideoStatus(path) != DECODE_RUNNING && video_pid != 0)
   {
      STB_DPStartVideoDecoding(path);

      if (ACTL_HasDecodingStarted(path))
      {
         STB_AVBlankVideo(STB_DPGetPathVideoDecoder(path), FALSE);
      }
   }
   FUNCTION_FINISH(DVBMH_PlayVideo);
}

/**
 * @brief   MHEG5 engine gives control of Video to external application.
 *          Set decoding and presentation of Video according to the default
 *          user selection.
 * @return  MHERR_OK - Success, MHERR_OTHER - failure
 */
E_MhegErr DVB_MhegVideoPlayDefault(void)
{
   E_MhegErr result;
   void *srv_ptr;
   U8BIT path;
   FUNCTION_START(DVB_MhegVideoPlayDefault);
   ACTL_SetMhegAVControl(FALSE);
   STB_VTSetMhegEnable(FALSE);
   path = STB_DPGetLivePath();
   if (path == INVALID_RES_ID)
   {
      result = MHERR_OTHER;
   }
   else
   {
      srv_ptr = STB_DPGetTunedService(path);
      if (srv_ptr == NULL)
      {
         result = MHERR_OTHER;
      }
      else
      {
         result = MHERR_OK;
         /* Ensure PCR PID is setup */
         STB_DPSetPCRPID(path, ADB_GetServicePCRPid(srv_ptr));
         DVBMH_PlayVideo( path, ADB_GetServiceVideoPid(srv_ptr),
                     EvalVideoCodec(srv_ptr,ADB_GetServiceVideoType(srv_ptr)) );
         if ((subt_show_state == MHEG5_DVB_SUBTITLE_SHOW) && !ACTL_AreSubtitlesStarted() &&
            DVB_MhegGetSubtitlePref())
         {
            ACTL_StartSubtitles();
         }
      }
   }
   FUNCTION_FINISH(DVB_MhegVideoPlayDefault);
   return result;
}

/**
 * @brief   MHEG5 engine takes control of Video - and informs that Video must be
 *          switched to the specified Video stream. This setting takes effect
 *          regardless of whether or not an video stream is currently playing.
 *          If the locator references a service component that is not currently
 *          being broadcast, the audio should be muted until the referenced
 *          component is being broadcast.
 * @return  - MHERR_OK on success
 *          - MHERR_BAD_PARAMETER invalid parameter
 *          - MHERR_OTHER controlling application specific error.
 */
E_MhegErr DVB_MhegVideoPlayStream(S_DvbComponent *pDvbComponent)
{
   E_MhegErr result = MHERR_OTHER;
   void *srv_ptr;
   void **stream_list;
   U16BIT num, i;
   U8BIT path, num_tags, j;
   FUNCTION_START(DVB_MhegVideoPlayStream);
   ACTL_SetMhegAVControl(TRUE);
   STB_VTSetMhegEnable(TRUE);
   path = STB_DPGetLivePath();
   if (path != INVALID_RES_ID)
   {
      num = pDvbComponent->transport_stream_id;
      DBGPRINT("service (%x.%x.%x)",pDvbComponent->original_network_id, num, pDvbComponent->service_id)
      if (num == 0)
      {
         num = ADB_GetTransportTid(ADB_GetTunedTransport(path));
         if ( num == 0 )
         {
            DBGPRINT("Failed to get transport ID for %p", ADB_GetTunedTransport(path))
            num = ADB_INVALID_DVB_ID;
         }
      }
      srv_ptr = ADB_FindServiceByIds(pDvbComponent->original_network_id, num, pDvbComponent->service_id);
      if (srv_ptr == NULL)
      {
         DBGPRINT("Failed to find service (%x.%x.%x)",pDvbComponent->original_network_id, num, pDvbComponent->service_id)
      }
      else
      {
         S16BIT ctag = pDvbComponent->ComponentTag;
         /* Ensure PCR PID is setup */
         STB_DPSetPCRPID(path, ADB_GetServicePCRPid(srv_ptr));
         if (ctag == -1)
         {
            DVBMH_PlayVideo( path,
               ADB_GetServiceVideoPid(srv_ptr),
               EvalVideoCodec(srv_ptr,ADB_GetServiceVideoType(srv_ptr)) );
            if ((subt_show_state == MHEG5_DVB_SUBTITLE_SHOW) && !ACTL_AreSubtitlesStarted() &&
               DVB_MhegGetSubtitlePref())
            {
               ACTL_StartSubtitles();
            }
            result = MHERR_OK;
         }
         else
         {
            ADB_GetStreamList(srv_ptr, ADB_VIDEO_LIST_STREAM|ADB_DATA_LIST_STREAM, &stream_list, &num);
            if (stream_list == NULL)
            {
               DBGPRINT("Failed to get stream list")
            }
            else
            {
               DBGPRINT("streams num %u",num)
               for (i = 0; i != num; i++)
               {
                  num_tags = ADB_GetStreamNumTags(stream_list[i]);
                  DBGPRINT("tags num %u",num_tags)
                  for (j = 0; j != num_tags; j++)
                  {
                     DBGPRINT("stream: tag %u",ADB_GetStreamTag(stream_list[i],j))
                     if (ctag == ADB_GetStreamTag(stream_list[i], j))
                     {
                        result = MHERR_OK;
                        DVBMH_PlayVideo( path, ADB_GetStreamPID(stream_list[i]),
                                   EvalVideoCodec(srv_ptr,ADB_GetStreamType(stream_list[i])) );
                        break;
                     }
                  }
                  if (result == MHERR_OK)
                     break;
               }
               ADB_ReleaseStreamList(stream_list, num);
            }
         }
      }
   }
   FUNCTION_FINISH(DVB_MhegVideoPlayStream);
   return result;
}

/**
 * @brief   MHEG5 engine takes control of Video - and specifies that
 *          decoding and presentation of any Video stream must be stopped
 * @return  MHERR_OK - Success, MHERR_OTHER - failure
 */
E_MhegErr DVB_MhegVideoStopStream(void)
{
   E_MhegErr result;
   U8BIT path;
   FUNCTION_START(DVB_MhegVideoStopStream);
   ACTL_SetMhegAVControl(TRUE);
   STB_VTSetMhegEnable(TRUE);
   path = STB_DPGetLivePath();
   if (path == INVALID_RES_ID)
   {
      result = MHERR_OTHER;
   }
   else
   {
      result = MHERR_OK;
      if (STB_DPGetVideoPID(path) != 0)
      {
         ACTL_StopSubtitles();
         STB_AVBlankVideo(STB_DPGetPathVideoDecoder(path), TRUE);
         STB_DPStopVideoDecoding(path);
      }
   }
   FUNCTION_FINISH(DVB_MhegVideoStopStream);
   return result;
}

/**
 * @brief   Instruct the controlling application to decode supplied video
 *          data. The I-Frame is displayed in the video decoder display buffer. Its
 *          scaling and position is affected by calls to DVB_MhegVideoPosition().
 *          Decode and display of I-frames is mutually exclusive to the decoding of
 *          video streams.
 *          The display of I-frames is subject to decoder format conversion (DFC)
 *          and display signaling as specified in section 6.5.4.6 of reference [1].
 *          The DFC and signaling applied is dependant on the Scene or Image aspect
 *          ratio, the TV aspect ratio and the I-Frame scaling mode (as derived from
 *          the content hook).
 *          The I-Frame data passed MUST be copied if it is required after this
 *          function returns, as the caller retains exclusive ownership.
 *          All other video/iframe functions (eg. DVB_MhegVideoIFrameStop) should apply
 *          to the current Iframe (regardless of the encoding specified)
 * @param pVideoData Pointer to video data block
 * @param dataLength Length of video data block.
 * @param encoding   Video encoding used in iframe
 * @return  - MHERR_OK on success
 *          - MHERR_BAD_PARAMETER invalid parameter
 *          - MHERR_OTHER controlling application specific error.
 */
E_MhegErr DVB_MhegVideoPlayIFrame( U8BIT *pVideoData,
   U32BIT dataLength,
   E_VideoEncoding encoding )
{
   E_MhegErr result = MHERR_OTHER;
   U8BIT decoder;
   FUNCTION_START(DVB_MhegVideoPlayIFrame);
   decoder = STB_DPGetPathVideoDecoder(STB_DPGetLivePath());
   if (decoder != INVALID_RES_ID)
   {
      switch (encoding)
      {
         case VIDEO_ENC_MPEG2:
            /* Check I-Frame header: Sequence start code 0x000001B3 Big-Endian */
            if (dataLength >= 4 && pVideoData[0] == 0x00 && pVideoData[1] == 0x00 &&
                pVideoData[2] == 0x01 && pVideoData[3] == 0xB3)
            {
               if (STB_AVSetIFrameCodec(decoder, AV_VIDEO_CODEC_MPEG2))
               {
                  STB_AVLoadIFrame(decoder, pVideoData, dataLength);
                  STB_AVShowIFrame(decoder);
                  result = MHERR_OK;
               }
            }
            break;

         case VIDEO_ENC_H_264:
            /* TODO: do more complete "Check of H264 I-Frame header" */
            if (dataLength >= 4 && pVideoData[0] == 0x00 && pVideoData[1] == 0x00)
            {
               if (STB_AVSetIFrameCodec(decoder, AV_VIDEO_CODEC_H264))
               {
                  STB_AVLoadIFrame(decoder, pVideoData, dataLength);
                  STB_AVShowIFrame(decoder);
                  result = MHERR_OK;
               }
            }
            break;

         default:
            result = MHERR_BAD_PARAMETER;
      }
   }
   FUNCTION_FINISH(DVB_MhegVideoPlayIFrame);
   return result;
}

/**
 * @brief   Instruct the controlling application to stop displaying the previously
 *          supplied video data in DVB_MhegVideoPlayIFrame.
 * @return  - MHERR_OK on success
 *          - MHERR_OTHER controlling application specific error.
 */
E_MhegErr DVB_MhegVideoStopIFrame(void)
{
   E_MhegErr result = MHERR_OTHER;
   U8BIT decoder;
   FUNCTION_START(DVB_MhegVideoStopIFrame);
   decoder = STB_DPGetPathVideoDecoder(STB_DPGetLivePath());
   if (decoder != INVALID_RES_ID)
   {
      result = MHERR_OK;
      STB_AVHideIFrame(decoder);
   }
   FUNCTION_FINISH(DVB_MhegVideoStopIFrame);
   return result;
}

/**
 * @brief   Switch between frozen or unfrozen video display (reacting on either
 *          the current or subsequent frame), and remain in force until set again.
 *          This setting takes effect regardless of whether a video stream is
 *          currently playing.
 *          This MUST be a non-blocking function.
 * @param   displayState  VGR_VIDEO_FREEZE or VGR_VIDEO_UNFREEZE
 * @return  - MHERR_OK on success
 *          - MHERR_OTHER controlling application specific error.
 */
E_MhegErr DVB_MhegVideoSetUpdating(E_VideoState displayState)
{
   E_MhegErr result;
   U8BIT decoder;
   FUNCTION_START(DVB_MhegVideoSetUpdating);
   decoder = STB_DPGetPathVideoDecoder(STB_DPGetLivePath());
   switch (displayState)
   {
      case VIDEO_FREEZE:
         DBGPRINT("STB_AVPauseVideoDecoding(%d)", decoder);
         STB_AVPauseVideoDecoding(decoder);
         ACTL_PauseSubtitles();
         result = MHERR_OK;
         break;

      case VIDEO_UNFREEZE:
         STB_AVResumeVideoDecoding(decoder);
         ACTL_ResumeSubtitles();
         result = MHERR_OK;
         break;

      default:
         result = MHERR_BAD_PARAMETER;
   }
   FUNCTION_FINISH(DVB_MhegVideoSetUpdating);
   return result;
}

/**
 * @brief   Show or hide DVB subtitles. This operation remains in force until set
 *          again by another call to this function.
 *          This setting takes effect regardless of whether there is a subtitle
 *          stream currently playing. If a subtitle stream starts playing whilst in
 *          the hidden state, it will not have a visible effect until the show state
 *          is selected.
 * @param   showState MHEG5_DVB_SUBTITLE_SHOW if subtitles should be shown
 *                    MHEG5_DVB_SUBTITLE_HIDE if subtitles should be hidden
 * @return  - MHERR_OK on success
 *          - MHERR_OTHER controlling application specific error.
 */
E_MhegErr DVB_MhegSubtitleSetVisibility(E_DvbSubtitleShowState showState)
{
   FUNCTION_START(DVB_MhegSubtitleSetVisibility);
   DBGPRINT("showState %d", showState);

   subt_show_state = showState;
   if (showState == MHEG5_DVB_SUBTITLE_SHOW)
   {
      STB_ERSendEvent(FALSE, FALSE, EV_CLASS_MHEG, EV_TYPE_START_SUBTITLES, NULL, 0);
   }
   else
   {
      STB_ERSendEvent(FALSE, FALSE, EV_CLASS_MHEG, EV_TYPE_STOP_SUBTITLES, NULL, 0);
   }

   FUNCTION_FINISH(DVB_MhegSubtitleSetVisibility);
   return MHERR_OK;
}

BOOLEAN DVBMH_ShowSubtitles(void)
{
   return (subt_show_state == MHEG5_DVB_SUBTITLE_SHOW)? TRUE : FALSE;
}


/**
 * @brief   This function is used by the MHEG-5 engine to determine whether
 *          subtitles are currently enabled. This may be used to allow a content
 *          provider to make available streams with and without subtitles to
 *          avoid streaming unused data.
 * @return   BOOLEAN; TRUE if the user turned the subtitles on,
 *                    FALSE otherwise.
 */
BOOLEAN DVB_MhegGetSubtitlePref(void)
{
   S_ACB_UI_INFO info;
   BOOLEAN result = FALSE;
   FUNCTION_START(DVB_MhegGetSubtitlePref);
   info.type = ACB_GET_SUBTITLE_PREF;
   if (ACB_GetUIInformation(&info) == TRUE)
   {
      result = info.u.subtitle.pref;
   }
   FUNCTION_FINISH(DVB_MhegGetSubtitlePref);
   return result;
}

/**
 * @brief   Get list of preferred subtitle languages. If there is no preference
 *          then zero is returned. This only writes into array up to 'max' items
 *          This MUST be a non-blocking function, returning results immediately.
 * @param   langs Array of ISO 639 language codes to be returned
 * @param   max maximum number of preferred languages required by caller
 * @return  number of actual language code preferences
 */
U8BIT DVB_MhegPrefSubtitleLangs(U32BIT *langs,U8BIT max)
{
   U32BIT country_code;
   U8BIT *ids;
   U8BIT count = 0;
   FUNCTION_START(DVB_MhegPrefSubtitleLangs);
   if (count != max)
   {
      country_code = ACFG_GetCountry();
      ids = ACFG_GetDbLangId(country_code,ACFG_GetPrimaryTextLangId());
      if (ids != NULL)
      {
         *langs = ACFG_ConvertLangIdToCode(ids[0]);
         count++;
      }
      if (count != max)
      {
         ids = ACFG_GetDbLangId(country_code,ACFG_GetSecondaryTextLangId());
         if (ids != NULL)
         {
            langs[count] = ACFG_ConvertLangIdToCode(ids[0]);
            count++;
         }
         if (count != max)
         {
            ids = ACFG_GetDbLangId(country_code,ACFG_GetDefaultSecondaryLangId());
            if (ids != NULL)
            {
               langs[count] = ACFG_ConvertLangIdToCode(ids[0]);
               count++;
            }
         }
      }
   }
   FUNCTION_FINISH(DVB_MhegPrefSubtitleLangs);
   return count;
}

/*!**************************************************************************
 * @brief   Set Mheg scaling resolution for video
 * @param   width - width resolution
 * @param   height - height resolution
 * @return  None
 ****************************************************************************/
void DVB_MhegSetScalingResolution(U16BIT width, U16BIT height)
{
   STB_VTSetMhegScalingResolution(width, height);
}

/*!**************************************************************************
 * @brief   Set Mheg video scaling
 * @param   scaling - scaling transformation (offset, size)
 * @return  None
 ****************************************************************************/
void DVB_MhegSetVideoScaling(S_RECTANGLE *scaling)
{
   STB_VTSetMhegVideoScaling(scaling);
}

/*!**************************************************************************
 * @brief   Set display aspect ratio
 * @param   aspect - display aspect ratio
 * @return  None
 ****************************************************************************/
void DVB_MhegSetAspectRatio(E_ASPECT_RATIO aspect)
{
   STB_VTSetMhegAspectRatio(aspect);
}

/*!**************************************************************************
 * @brief   Get display aspect ratio
 * @return  display aspect ratio
 ****************************************************************************/
E_ASPECT_RATIO DVB_MhegGetDisplayAspectRatio(void)
{
   return STB_VTGetDisplayAspectRatio();
}

/*!**************************************************************************
 * @brief    Return the current decoder format conversion
 * @param    context - transformation calculator context
 * @return   Decoder format conversion
 ****************************************************************************/
E_FORMAT_CONVERSION DVB_MhegGetDecoderFormatConversion(void)
{
   return STB_VTGetDecoderFormatConversion();
}

void DVB_MhegSetVideoAlignment(E_VIDEO_ASPECT_MODE mode)
{
   STB_VTSetMhegVideoAlignment(mode);
}

