/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2013 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Glue layer between DVB and conditional access systems
 * @file    ca_glue.c
 * @date    18/09/2013
 * @author  Steve Ford
 */

#define CA_DEBUG

/*---Includes for this file---------------------------------------------------*/
/* compiler library header files */

/* Third party header files */

/* Ocean Blue Software header files */
#include <techtype.h>
#include <dbgfuncs.h>
#include <stbhwc.h>
#include <stbhwcas.h>
#include "ca_glue.h"
#include "stbsitab.h"

/*---constant definitions for this file----------------------------------------*/

#ifdef CA_DEBUG
#define CA_DBG(X)    STB_SPDebugWrite X
#else
#define CA_DBG(X)
#endif

/*---local typedef structs for this file---------------------------------------*/

/*---local (static) variable declarations for this file------------------------*/
/*   (internal variables declared static to make them local) */

/*---local function prototypes for this file-----------------------------------*/
/*   (internal functions declared static to make them local) */


/*---global function definitions-----------------------------------------------*/

/*!**************************************************************************
 * @brief   Called once on system startup to allow initialisation of the CA systems
 * @return  TRUE if initialisation is successful, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_CAInitialise(void)
{
   FUNCTION_START(STB_CAInitialise);

   CA_DBG(("%s", __FUNCTION__));

   FUNCTION_FINISH(STB_CAInitialise);

   return(TRUE);
}

/*!**************************************************************************
 * @brief   This function is used by the resource manager to acquire a CA descrambler
 *          that's able to descramble a service that uses one of the CA systems
 *          defined by the array of CA system IDs (ca_ids). If a descrambler is
 *          available then a handle should be returned in 'handle' which will be
 *          used in all future calls related to this descrambler.
 *          If the CA software needs to set the demux descrambling keys, or create
 *          any filters to monitor SI data, the given demux handle should be used.
 * @param   demux - demux to be used if a descrambler is acquired
 * @param   serv_id - ID of the service the descrambler is being acquired for
 * @param   ca_ids - array of CA system IDs for the service
 * @param   num_ca_ids - number of CA system IDs in the array
 * @param   handle - pointer to return a handle to identify the acquired CA descrambler
 * @return  TRUE if a descrambler is acquired, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_CAAcquireDescrambler(U8BIT demux, U16BIT serv_id, U16BIT *ca_ids, U16BIT num_ca_ids,
   U32BIT *handle)
{
   FUNCTION_START(STB_CAAcquireDescrambler);

   CA_DBG(("%s(demux=%u, serv_id=%u, ca_ids=%p, num_ca_ids=%u)", __FUNCTION__, demux, serv_id,
      ca_ids, num_ca_ids));

   USE_UNWANTED_PARAM(demux);
   USE_UNWANTED_PARAM(serv_id);
   USE_UNWANTED_PARAM(ca_ids);
   USE_UNWANTED_PARAM(num_ca_ids);
   USE_UNWANTED_PARAM(handle);
   
   *handle = (serv_id << 16) + demux;

   FUNCTION_FINISH(STB_CAAcquireDescrambler);

   return(TRUE);
}

/*!**************************************************************************
 * @brief   Will be called when a CA descrambler is no longer required.
 * @param   handle - CA descrambler handle being released
 * @return  TRUE if the descrambler is released, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_CAReleaseDescrambler(U32BIT handle)
{
   FUNCTION_START(STB_CAReleaseDescrambler);

   CA_DBG(("%s(0x%lx)", __FUNCTION__, handle));

   USE_UNWANTED_PARAM(handle);

   FUNCTION_FINISH(STB_CAReleaseDescrambler);

   return(TRUE);
}

/*!**************************************************************************
 * @brief   This function will be called when decoding of a service is about to start
 *          and there's an associated descrambler.
 * @param   handle - CA descrambler handle
 ****************************************************************************/
void STB_CADescrambleServiceStart(U32BIT handle)
{
   FUNCTION_START(STB_CADescrambleServiceStart);

   CA_DBG(("%s(0x%lx)", __FUNCTION__, handle));

   USE_UNWANTED_PARAM(handle);
   
   STB_CASStartDescrambler(handle);

   FUNCTION_FINISH(STB_CADescrambleServiceStart);
}

/*!**************************************************************************
 * @brief   This function will be called when decoding of a service is stopped
 * @param   handle - CA descrambler handle
 ****************************************************************************/
void STB_CADescrambleServiceStop(U32BIT handle)
{
   FUNCTION_START(STB_CADescrambleServiceStop);

   CA_DBG(("%s(0x%lx)", __FUNCTION__, handle));

   USE_UNWANTED_PARAM(handle);
   
   STB_CASStopDescrambler(handle);

   FUNCTION_FINISH(STB_CADescrambleServiceStop);
}

/*!**************************************************************************
 * @brief   When there's an update to the PMT for a service, the updated PMT
 *          will be reported to the CA system using this function.
 * @param   handle - CA descrambler handle
 * @param   pmt_data - raw PMT section data
 * @param   data_len - number of bytes in the PMT
 ****************************************************************************/
void STB_CAReportPMT(U32BIT handle, U8BIT *pmt_data, U16BIT data_len)
{
   FUNCTION_START(STB_CAReportPMT);

   CA_DBG(("%s(handle=0x%lx, pmt_data=%p, data_len=%u)", __FUNCTION__, handle, pmt_data, data_len));

   USE_UNWANTED_PARAM(handle);
   USE_UNWANTED_PARAM(pmt_data);
   USE_UNWANTED_PARAM(data_len);

   FUNCTION_FINISH(STB_CAReportPMT);
}

/*!**************************************************************************
 * @brief   When there's an update to the CAT for a service, the updated CAT
 *          will be reported to the CA system using this function. The data is
 *          provided a section at a time, rather than as a complete table.
 * @param   handle - CA descrambler handle
 * @param   cat_data - raw CAT section data
 * @param   data_len - number of bytes in the CAT section
 ****************************************************************************/
void STB_CAReportCAT(U32BIT handle, U8BIT *cat_data, U16BIT data_len)
{
   FUNCTION_START(STB_CAReportCAT);

   CA_DBG(("%s(handle=0x%lx, cat_data=%p, data_len=%u)", __FUNCTION__, handle, cat_data, data_len));

   USE_UNWANTED_PARAM(handle);
   USE_UNWANTED_PARAM(cat_data);
   USE_UNWANTED_PARAM(data_len);

   FUNCTION_FINISH(STB_CAReportCAT);
}

/*!**************************************************************************
 * @brief   When there's an update to the CAT for a service, the updated CAT
 *          will be reported to the CA system using this function.
 * @param   path - decode path
 * @param   handle - CA descrambler handle
 * @param   ca_system_id - CA system ID
 * @param   pid - the EMM PID
 ****************************************************************************/
void STB_CAReportEmmPid(U8BIT path, U32BIT handle, U16BIT ca_system_id, U16BIT pid)
{
   FUNCTION_START(STB_CAReportEmmPid);

   CA_DBG(("%s(handle=0x%lx, ca_system_id=0x%04Xp, pid=%u)", __FUNCTION__, handle, ca_system_id, pid));
   STB_CASSetEmmPid(path, handle, ca_system_id, pid);

   FUNCTION_FINISH(STB_CAReportEmmPid);
}

/*!**************************************************************************
 * @brief   When there's an update to the BAT, the updated BAT will be reported
 *          to the CA system using this function. The data is provided a section
 *          at a time, rather than as a complete table.
 * @param   handle - CA descrambler handle
 * @param   bat_data - raw BAT section data
 * @param   data_len - number of bytes in the BAT section
 ****************************************************************************/
void STB_CAReportBAT(U32BIT handle, U8BIT *bat_data, U16BIT data_len)
{
   FUNCTION_START(STB_CAReportBAT);

   CA_DBG(("%s(handle=0x%lx, bat_data=%p, data_len=%u)", __FUNCTION__, handle, bat_data, data_len));

   USE_UNWANTED_PARAM(handle);
   USE_UNWANTED_PARAM(bat_data);
   USE_UNWANTED_PARAM(data_len);

   FUNCTION_FINISH(STB_CAReportBAT);
}

/*!**************************************************************************
 * @brief   When there's an update to the NIT, the updated NIT will be reported
 *          to the CA system using this function. The data is provided a section
 *          at a time, rather than as a complete table.
 * @param   handle - CA descrambler handle
 * @param   nit_data - raw NIT section data
 * @param   data_len - number of bytes in the NIT section
 ****************************************************************************/
void STB_CAReportNIT(U32BIT handle, U8BIT *nit_data, U16BIT data_len)
{
   FUNCTION_START(STB_CAReportNIT);

   CA_DBG(("%s(handle=0x%lx, nit_data=%p, data_len=%u)", __FUNCTION__, handle, nit_data, data_len));

   USE_UNWANTED_PARAM(handle);
   USE_UNWANTED_PARAM(nit_data);
   USE_UNWANTED_PARAM(data_len);

   FUNCTION_FINISH(STB_CAReportNIT);
}

/*!**************************************************************************
 * @brief   Notifies the CA system of a change in the video decoding state
 * @param   handle - CA descrambler handle
 * @param   decode_status - decoding status
 ****************************************************************************/
void STB_CADecodeVideoStatus(U32BIT handle, E_CA_DECODE_STATUS decode_status)
{
   FUNCTION_START(STB_CADecodeVideoStatus);

   CA_DBG(("%s(handle=0x%lx, status=%u)", __FUNCTION__, handle, decode_status));

   USE_UNWANTED_PARAM(handle);
   USE_UNWANTED_PARAM(decode_status);

   FUNCTION_FINISH(STB_CADecodeVideoStatus);
}

/*!**************************************************************************
 * @brief   Notifies the CA system of a change in the audio decoding state
 * @param   handle - CA descrambler handle
 * @param   decode_status - decoding status
 ****************************************************************************/
void STB_CADecodeAudioStatus(U32BIT handle, E_CA_DECODE_STATUS decode_status)
{
   FUNCTION_START(STB_CADecodeAudioStatus);

   CA_DBG(("%s(handle=0x%lx, status=%u)", __FUNCTION__, handle, decode_status));

   USE_UNWANTED_PARAM(handle);
   USE_UNWANTED_PARAM(decode_status);

   FUNCTION_FINISH(STB_CADecodeAudioStatus);
}

/*!**************************************************************************
 * @brief   Notifies the CA system of a change in the AD decoding state
 * @param   handle - CA descrambler handle
 * @param   decode_status - decoding status
 ****************************************************************************/
void STB_CADecodeADStatus(U32BIT handle, E_CA_DECODE_STATUS decode_status)
{
   FUNCTION_START(STB_CADecodeADStatus);

   CA_DBG(("%s(handle=0x%lx, status=%u)", __FUNCTION__, handle, decode_status));

   USE_UNWANTED_PARAM(handle);
   USE_UNWANTED_PARAM(decode_status);

   FUNCTION_FINISH(STB_CADecodeADStatus);
}

/*!**************************************************************************
 * @brief   This function will be called when there's a change to the running status
 *          of a service being descrambled as indicated by the running_status field
 *          in the SDT.
 * @param   handle - CA descrambler handle
 * @param   status - running status as defined in the SDT
 ****************************************************************************/
void STB_CANotifyRunningStatus(U32BIT handle, U8BIT status)
{
   FUNCTION_START(STB_CANotifyRunningStatus);

   CA_DBG(("%s(handle=0x%lx, status=%u)", __FUNCTION__, handle, status));

   USE_UNWANTED_PARAM(handle);
   USE_UNWANTED_PARAM(status);

   FUNCTION_FINISH(STB_CANotifyRunningStatus);
}

/*!**************************************************************************
 * @brief   This function specifies whether a CA descrambler is required to playback
 *          a recording with one of the given CA system IDs.
 * @param   ca_ids - array of CA system IDs
 * @param   num_ca_ids - number of CA system IDs in the array
 * @return  TRUE if a CA descrambler is required, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_CADescramblerRequiredForPlayback(U16BIT *ca_ids, U16BIT num_ca_ids)
{
   FUNCTION_START(STB_CADescramblerRequiredForPlayback);

   CA_DBG(("%s(ca_ids=%p, num_ca_ids=%u)", __FUNCTION__, ca_ids, num_ca_ids));

   USE_UNWANTED_PARAM(ca_ids);
   USE_UNWANTED_PARAM(num_ca_ids);

   FUNCTION_FINISH(STB_CADescramblerRequiredForPlayback);

   return(FALSE);
}

/*!**************************************************************************
 * @brief   This function specifies whether a CA descrambler is required to record
 *          a service with one of the given CA system IDs.
 * @param   ca_ids - array of CA system IDs
 * @param   num_ca_ids - number of CA system IDs in the array
 * @return  TRUE if a CA descrambler is required, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_CADescramblerRequiredForRecording(U16BIT *ca_ids, U16BIT num_ca_ids)
{
   FUNCTION_START(STB_CADescramblerRequiredForRecording);

   CA_DBG(("%s(ca_ids=%p, num_ca_ids=%u)", __FUNCTION__, ca_ids, num_ca_ids));

   USE_UNWANTED_PARAM(ca_ids);
   USE_UNWANTED_PARAM(num_ca_ids);

   FUNCTION_FINISH(STB_CADescramblerRequiredForRecording);

   return(FALSE);
}

/*!**************************************************************************
 * @brief   This function is called to get an array of PIDs that need to be recorded
 *          for the CA system required for the given PMT. The array must be allocated
 *          by this function, which also returns the number of items in the array.
 * @param   pmt_data - raw PMT section data
 * @param   pid_array - pointer to an array allocated by this function on return,
 *                      containing the PIDs to be recorded
 * @return  the number of PIDs in the returned array
 ****************************************************************************/
U16BIT STB_CAGetRecordingPids(U8BIT *pmt_data, U16BIT **pid_array)
{
   U16BIT num_pids;

   FUNCTION_START(STB_CAGetRecordingPids);

   num_pids = 0;

   USE_UNWANTED_PARAM(pmt_data);
   USE_UNWANTED_PARAM(pid_array);

   CA_DBG(("%s(pmt_data=%p, pid_array=%u): %u", __FUNCTION__, pmt_data, pid_array, num_pids));

   FUNCTION_FINISH(STB_CAGetRecordingPids);

   return(num_pids);
}

/*!**************************************************************************
 * @brief   Called to free the array of PIDs allocated by STB_CAGetRecordingPids.
 * @param   pid_array - array of PIDs to be freed
 * @param   num_pids - number of PIDs in the array
 ****************************************************************************/
void STB_CAReleaseRecordingPids(U16BIT *pid_array, U16BIT num_pids)
{
   FUNCTION_START(STB_CAReleaseRecordingPids);

   CA_DBG(("%s(pid_array=%p, num_pids=%u)", __FUNCTION__, pid_array, num_pids));

   USE_UNWANTED_PARAM(pid_array);
   USE_UNWANTED_PARAM(num_pids);

   FUNCTION_FINISH(STB_CAReleaseRecordingPids);
}

/*!**************************************************************************
 * @brief   This function is called when a recording starts and when it stops
 * @param   handle - CA descrambler handle
 * @param   status - TRUE when a recording starts, FALSE when it stops
 ****************************************************************************/
void STB_CANotifyRecordingStatus(U32BIT handle, BOOLEAN status)
{
   FUNCTION_START(STB_CANotifyRecordingStatus);

   CA_DBG(("%s(handle=0x%lx, status=%u)", __FUNCTION__, handle, status));

   USE_UNWANTED_PARAM(handle);
   USE_UNWANTED_PARAM(status);

   FUNCTION_FINISH(STB_CANotifyRecordingStatus);
}

/******************************************************************************
** End of file
******************************************************************************/
