/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2010 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   JPEG image functions
 * @file    image_jpeg.c
 * @date    1 December 2010
 * @author  Steve Ford
 */

/*#define ENABLE_DEBUG*/

/*---includes for this file--------------------------------------------------*/

/* compiler library header files */
#include <stdio.h>      /* Required for JPEG library */
#include <string.h>
#include <setjmp.h>

/* third party header files */
#include "jpeglib.h"
#include "jerror.h"

/* DVBCore header files */
#include "techtype.h"
#include "dbgfuncs.h"

#include "stbheap.h"

#ifdef OSD_ST_MODE
/* Macro to convert 8-bit alpha values to values supported in "32" bit mode for stapi */
#define ST_ALPHA(x)        (((x) * 128 + 127) / 255)
#endif

#define INPUT_BUF_SIZE     4096  /* choose an efficiently fread'able size */


/*---constant definitions for this file--------------------------------------*/
#ifdef ENABLE_DEBUG
#define DBG(x)    STB_SPDebugWrite x
#else
#define DBG(x)
#endif


/*---local typedef structs for this file-------------------------------------*/
typedef struct
{
   U8BIT *data;
   U32BIT posn;
   U32BIT left;
} S_PSEUDO_FILE;

struct my_error_mgr
{
   struct jpeg_error_mgr pub;
   jmp_buf setjmp_buffer;
};


typedef struct
{
   struct jpeg_source_mgr pub; /* public fields */

   S_PSEUDO_FILE *infile;  /* source stream */
   JOCTET *buffer;         /* start of buffer */
   BOOLEAN start_of_file;  /* have we gotten any data yet? */
} my_source_mgr;


/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local) */
typedef struct my_error_mgr *my_error_ptr;
typedef my_source_mgr *my_src_ptr;


/*---local function prototypes for this file---------------------------------*/
/*   (internal functions declared static to make them local) */
static BOOLEAN JPGGetImageSize(U8BIT *image_data, U32BIT data_size, U16BIT *pixel_width,
   U16BIT *pixel_height, U32BIT *row_bytes);
static BOOLEAN JPGConvertImage(U8BIT *image_data, U32BIT data_size, U8BIT *output_data);

static void my_error_exit( j_common_ptr cinfo );
static void jobs_stdio_src(j_decompress_ptr cinfo, S_PSEUDO_FILE *infile);
static void init_source(j_decompress_ptr cinfo);
static boolean fill_input_buffer(j_decompress_ptr cinfo);
static void skip_input_data(j_decompress_ptr cinfo, long num_bytes);
static void term_source(j_decompress_ptr cinfo);


/*---global function definitions---------------------------------------------*/

/*!**************************************************************************
 * @brief   Converts the given JPEG image data to a bitmap image that can be displayed on-screen,
 *          but no scaling is applied.
 * @param   image_data - PNG image data
 * @param   image_data_size - number of PNG image data bytes
 * @param   output_data - address of pointer to buffer allocated for output data
 * @param   output_data_size - pointer to number of bytes in output bitmap
 * @param   pixel_width - pointer to returned width in pixels
 * @param   pixel_height - pointer to returned height in pixels
 * @return  TRUE if image data is a valid PNG and conversion succeeds, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_IMGConvertJPEG(U8BIT *image_data, U32BIT image_data_size, U8BIT **output_data,
   U32BIT *output_data_size, U16BIT *pixel_width, U16BIT *pixel_height)
{
   BOOLEAN retval;
   U32BIT rowbytes;

   FUNCTION_START(STB_IMGConvertJPEG);

   retval = FALSE;
   *output_data = NULL;
   *output_data_size = 0;

   if ((image_data != NULL) && (image_data_size > 0))
   {
      if (JPGGetImageSize(image_data, image_data_size, pixel_width, pixel_height, &rowbytes))
      {
         DBG(("STB_IMGConvertJPEG: pixel_width=%u, pixel_height=%u, rowbytes=%lu",
              *pixel_width, *pixel_height, rowbytes));

         /* Allocate memory for the decompressed image */
         *output_data_size = rowbytes * *pixel_height;

         *output_data = (U8BIT *)STB_GetMemory(*output_data_size);
         if (*output_data != NULL)
         {
            retval = JPGConvertImage(image_data, image_data_size, *output_data);
            if (!retval)
            {
               DBG(("STB_IMGConvertJPEG: Failed to decompress image"));

               STB_FreeMemory(*output_data);
               *output_data = NULL;
               *output_data_size = 0;
            }
         }
      }
   }

   FUNCTION_FINISH(STB_IMGConvertJPEG);

   return(retval);
}

/*---local function definitions---------------------------------------------*/

/*!**************************************************************************
 * @brief   Returns the size in pixels and the number of bytes required for a single row.
 * @param   image_data - JPEG image data to be decoded
 * @param   data_size - number of bytes of image data
 * @param   pixel_width - returned width in pixels
 * @param   pixel_height - returned height in pixels
 * @param   row_bytes - returned bytes per line
 * @return  TRUE if size data is returned, FALSE otherwise
 ****************************************************************************/
static BOOLEAN JPGGetImageSize(U8BIT *image_data, U32BIT data_size, U16BIT *pixel_width,
   U16BIT *pixel_height, U32BIT *row_bytes)
{
   BOOLEAN retval;
   struct jpeg_decompress_struct cinfo;
   struct my_error_mgr jerr;
   S_PSEUDO_FILE pseudo_file;

   FUNCTION_START(JPGGetImageSize);

   retval = FALSE;

   if ((image_data != NULL) && (data_size > 0))
   {
      cinfo.err = jpeg_std_error(&jerr.pub);
      jerr.pub.error_exit = my_error_exit;

      if (setjmp(jerr.setjmp_buffer))
      {
         /* If we get here, the JPEG code has signaled an error.
          * We need to clean up the JPEG object, close the input file, and return */
         jpeg_destroy_decompress(&cinfo);
      }
      else
      {
         pseudo_file.data = image_data;
         pseudo_file.posn = 0;
         pseudo_file.left = data_size;

         jpeg_create_decompress(&cinfo);
         jobs_stdio_src( &cinfo, &pseudo_file );

         jpeg_read_header(&cinfo, TRUE);

         *pixel_width = cinfo.output_width;
         *pixel_height = cinfo.output_height;

#ifdef OSD_16_BIT
         *row_bytes = cinfo.output_width * sizeof(U16BIT);
#else /* OSD_32_BIT */
         *row_bytes = cinfo.output_width * sizeof(U32BIT);
#endif

         jpeg_destroy_decompress(&cinfo);
      }
   }

   FUNCTION_FINISH(JPGGetImageSize);

   return(retval);
}

/*!**************************************************************************
 * @brief   Decompresses the JPEG image and converts it to image data the correct format for the OSD
 * @param   image_data - JPEG image data to be decoded
 * @param   data_size - number of bytes of image data
 * @param   output_data - preallocated buffer to take the output data
 * @return  TRUE if the image is decompressed successfully, FALSE otherwise
 ****************************************************************************/
static BOOLEAN JPGConvertImage(U8BIT *image_data, U32BIT data_size, U8BIT *output_data)
{
   BOOLEAN retval;
   struct jpeg_decompress_struct cinfo;
   struct my_error_mgr jerr;
   S_PSEUDO_FILE pseudo_file;
   U8BIT *row_ptr;
   U16BIT pixel;
   U8BIT red, green, blue;
   U8BIT *src;
#ifdef OSD_16_BIT
   U16BIT *dest_line;
   U16BIT *dest;
#else /* OSD_32_BIT */
   U8BIT alpha;
   U32BIT *dest_line;
   U32BIT *dest;
#endif

   FUNCTION_START(JPGConvertImage);

   retval = FALSE;

   if ((image_data != NULL) && (data_size > 0))
   {
      cinfo.err = jpeg_std_error(&jerr.pub);
      jerr.pub.error_exit = my_error_exit;

      if (setjmp(jerr.setjmp_buffer))
      {
         /* If we get here, the JPEG code has signaled an error.
          * We need to clean up the JPEG object, close the input file, and return */
         jpeg_destroy_decompress(&cinfo);
      }
      else
      {
         pseudo_file.data = image_data;
         pseudo_file.posn = 0;
         pseudo_file.left = data_size;

         jpeg_create_decompress(&cinfo);
         jobs_stdio_src( &cinfo, &pseudo_file );

         jpeg_read_header(&cinfo, TRUE);

         /* Allocate memory to decompress a single line into */
         row_ptr = STB_GetMemory(cinfo.output_width * cinfo.output_components);
         if (row_ptr != NULL)
         {
            jpeg_start_decompress(&cinfo);

#ifdef OSD_16_BIT
            dest_line = (U16BIT *)output_data;
#else /* OSD_32_BIT */
            dest_line = (U32BIT *)output_data;
#endif

            /* Decompress the image a line at a time, converting each one to the required output format */
            while (cinfo.output_scanline < cinfo.output_height)
            {
               if (jpeg_read_scanlines(&cinfo, &row_ptr, 1) == 1)
               {
                  if (cinfo.output_components == 3)
                  {
                     /* RGB image */
                     for (pixel = cinfo.output_width, dest = dest_line, src = row_ptr; pixel > 0;
                          dest++, src += 3, pixel--)
                     {
                        red = *src;
                        green = *(src + 1);
                        blue = *(src + 2);
#ifdef OSD_16_BIT
                        *dest = 0x0f;
                        *dest = (*dest << 4) | (red >> 4);
                        *dest = (*dest << 4) | (green >> 4);
                        *dest = (*dest << 4) | (blue >> 4);
#else /* OSD_32_BIT */
#ifdef OSD_ST_MODE
                        alpha = ST_ALPHA(0xff);
#else
                        alpha = 0xff;
#endif
                        *dest = (alpha << 24) | (red << 16) | (green << 8) | blue;
#endif
                     }
                  }
               }
            }

            jpeg_finish_decompress(&cinfo);

            STB_FreeMemory(row_ptr);
         }

         jpeg_destroy_decompress(&cinfo);
      }
   }

   FUNCTION_FINISH(JPGConvertImage);

   return(retval);
}

static void my_error_exit( j_common_ptr cinfo )
{
   /* cinfo->err really points to a my_error_mgr struct, so coerce pointer */
   my_error_ptr myerr = (my_error_ptr) cinfo->err;

   /* Always display the message. */
   /* We could postpone this until after returning, if we chose. */
   (*cinfo->err->output_message)(cinfo);

   /* Return control to the setjmp point */
   longjmp(myerr->setjmp_buffer, 1);
}

static void jobs_stdio_src(j_decompress_ptr cinfo, S_PSEUDO_FILE *infile)
{
   my_src_ptr src;

   /* The source object and input buffer are made permanent so that a series
    * of JPEG images can be read from the same file by calling jpeg_stdio_src
    * only before the first one.  (If we discarded the buffer at the end of
    * one image, we'd likely lose the start of the next one.)
    * This makes it unsafe to use this manager and a different source
    * manager serially with the same JPEG object.  Caveat programmer.
    */
   if (cinfo->src == NULL)
   {
      /* first time for this JPEG object? */
      cinfo->src = (struct jpeg_source_mgr *)(*cinfo->mem->alloc_small)((j_common_ptr)cinfo,
            JPOOL_PERMANENT, sizeof(my_source_mgr));

      src = (my_src_ptr) cinfo->src;

      src->buffer = (JOCTET *)(*cinfo->mem->alloc_small)((j_common_ptr) cinfo,
            JPOOL_PERMANENT, INPUT_BUF_SIZE * sizeof(JOCTET));
   }

   src = (my_src_ptr) cinfo->src;
   src->pub.init_source = init_source;
   src->pub.fill_input_buffer = fill_input_buffer;
   src->pub.skip_input_data = skip_input_data;
   src->pub.resync_to_restart = jpeg_resync_to_restart; /* use default method */
   src->pub.term_source = term_source;
   src->infile = infile;
   src->pub.bytes_in_buffer = 0; /* forces fill_input_buffer on first read */
   src->pub.next_input_byte = NULL; /* until buffer loaded */
}

static void init_source(j_decompress_ptr cinfo)
{
   my_src_ptr src = (my_src_ptr) cinfo->src;

   /* We reset the empty-input-file flag for each image,
    * but we don't clear the input buffer.
    * This is correct behavior for reading a series of images from one source.
    */
   src->start_of_file = TRUE;
}

/*
 * Fill the input buffer --- called whenever buffer is emptied.
 *
 * In typical applications, this should read fresh data into the buffer
 * (ignoring the current state of next_input_byte & bytes_in_buffer),
 * reset the pointer & count to the start of the buffer, and return TRUE
 * indicating that the buffer has been reloaded.  It is not necessary to
 * fill the buffer entirely, only to obtain at least one more byte.
 *
 * There is no such thing as an EOF return.  If the end of the file has been
 * reached, the routine has a choice of ERREXIT() or inserting fake data into
 * the buffer.  In most cases, generating a warning message and inserting a
 * fake EOI marker is the best course of action --- this will allow the
 * decompressor to output however much of the image is there.  However,
 * the resulting error message is misleading if the real problem is an empty
 * input file, so we handle that case specially.
 *
 * In applications that need to be able to suspend compression due to input
 * not being available yet, a FALSE return indicates that no more data can be
 * obtained right now, but more may be forthcoming later.  In this situation,
 * the decompressor will return to its caller (with an indication of the
 * number of scanlines it has read, if any).  The application should resume
 * decompression after it has loaded more data into the input buffer.  Note
 * that there are substantial restrictions on the use of suspension --- see
 * the documentation.
 *
 * When suspending, the decompressor will back up to a convenient restart point
 * (typically the start of the current MCU). next_input_byte & bytes_in_buffer
 * indicate where the restart point will be if the current call returns FALSE.
 * Data beyond this point must be rescanned after resumption, so move it to
 * the front of the buffer rather than discarding it.
 */

static boolean fill_input_buffer(j_decompress_ptr cinfo)
{
   my_src_ptr src = (my_src_ptr) cinfo->src;
   S_PSEUDO_FILE *pseudo_file = src->infile;
   U32BIT nbytes = INPUT_BUF_SIZE;

   if (pseudo_file->left < INPUT_BUF_SIZE)
   {
      nbytes = pseudo_file->left;
      if (nbytes == 0)
      {
         if (src->start_of_file)
         {
            /* Treat empty input file as fatal error */
            ERREXIT(cinfo, JERR_INPUT_EMPTY);
         }
         WARNMS(cinfo, JWRN_JPEG_EOF);
         /* Insert a fake EOI marker */
         src->buffer[0] = (JOCTET) 0xFF;
         src->buffer[1] = (JOCTET) JPEG_EOI;
         nbytes = 2;
      }
   }

   memcpy( src->buffer, pseudo_file->data + pseudo_file->posn, nbytes );
   pseudo_file->posn += nbytes;
   pseudo_file->left -= nbytes;

   src->pub.next_input_byte = src->buffer;
   src->pub.bytes_in_buffer = nbytes;
   src->start_of_file = FALSE;

   return TRUE;
}

/*
 * Skip data --- used to skip over a potentially large amount of
 * uninteresting data (such as an APPn marker).
 *
 * Writers of suspendable-input applications must note that skip_input_data
 * is not granted the right to give a suspension return.  If the skip extends
 * beyond the data currently in the buffer, the buffer can be marked empty so
 * that the next read will cause a fill_input_buffer call that can suspend.
 * Arranging for additional bytes to be discarded before reloading the input
 * buffer is the application writer's problem.
 */

static void skip_input_data(j_decompress_ptr cinfo, long num_bytes)
{
   my_src_ptr src = (my_src_ptr) cinfo->src;

   /* Just a dumb implementation for now.  Could use fseek() except
    * it doesn't work on pipes.  Not clear that being smart is worth
    * any trouble anyway --- large skips are infrequent.
    */
   if (num_bytes > 0)
   {
      while (num_bytes > (long) src->pub.bytes_in_buffer)
      {
         num_bytes -= (long) src->pub.bytes_in_buffer;
         (void) fill_input_buffer(cinfo);
         /* note we assume that fill_input_buffer will never return FALSE,
          * so suspension need not be handled.
          */
      }
      src->pub.next_input_byte += (size_t) num_bytes;
      src->pub.bytes_in_buffer -= (size_t) num_bytes;
   }
}

/*
 * An additional method that can be provided by data source modules is the
 * resync_to_restart method for error recovery in the presence of RST markers.
 * For the moment, this source module just uses the default resync method
 * provided by the JPEG library.  That method assumes that no backtracking
 * is possible.
 */


/*
 * Terminate source --- called by jpeg_finish_decompress
 * after all data has been read.  Often a no-op.
 *
 * NB: *not* called by jpeg_abort or jpeg_destroy; surrounding
 * application must deal with any cleanup that should happen even
 * for error exit.
 */

static void term_source(j_decompress_ptr cinfo)
{
   /* no work necessary here */
   USE_UNWANTED_PARAM(cinfo);
}

/****************************************************************************
 * End of file
 *****************************************************************************/
