/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2010 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   PNG image functions
 * @file    image_png.c
 * @date    22 November 2010
 * @author  Steve Ford
 */

/*#define ENABLE_DEBUG*/

/*---includes for this file--------------------------------------------------*/

/* compiler library header files */
#include <string.h>

/* third party header files */
#include <png.h>

/* DVBCore header files */
#include "techtype.h"
#include "dbgfuncs.h"

#include "stbheap.h"


/*---constant definitions for this file--------------------------------------*/
#ifdef ENABLE_DEBUG
#define DBG(x)    STB_SPDebugWrite x
#else
#define DBG(x)
#endif

#define PNG_HEADER_SIZE    8

#define ST_ALPHA(x)        (((x) * 128 + 127) / 255)


/*---local typedef structs for this file-------------------------------------*/
typedef struct
{
   U8BIT *data;
   U32BIT length;
   U32BIT offset;
} S_PNG_BUFFER;

typedef struct
{
   png_struct *png_ptr;
   png_info *info_ptr;
   png_info *end_info;
   S_PNG_BUFFER buffer;
} S_PNG_READER;


/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local) */


/*---local function prototypes for this file---------------------------------*/
/*   (internal functions declared static to make them local) */
static BOOLEAN ReadPNGHeader(U8BIT *image_data, U32BIT num_bytes, U16BIT *pixel_width,
   U16BIT *pixel_height, U32BIT *bytes_per_row, BOOLEAN keep_open, S_PNG_READER *reader);

static png_voidp PNG_Malloc(png_structp png_ptr, png_size_t size);
static void PNG_Free(png_structp png_ptr, png_voidp addr);
static void PNG_ReadData(png_structp png_ptr, png_bytep data, png_size_t length);


/*---global function definitions---------------------------------------------*/

/*!**************************************************************************
 * @brief   Converts the given PNG image data to a bitmap image that can be displayed on-screen
 *          with the given bit depth.
 * @param   image_data - PNG image data
 * @param   image_data_size - number of PNG image data bytes
 * @param   output_data - address of pointer to buffer allocated for output data
 * @param   output_data_size - pointer to number of bytes in output bitmap
 * @param   pixel_width - pointer to returned width in pixels
 * @param   pixel_height - pointer to returned height in pixels
 * @return  TRUE if image data is a valid PNG and conversion succeeds, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_IMGConvertPNG(U8BIT *image_data, U32BIT image_data_size, U8BIT **output_data,
   U32BIT *output_data_size, U16BIT *pixel_width, U16BIT *pixel_height)
{
   BOOLEAN retval;
   U32BIT rowbytes;
   S_PNG_READER reader;
   U8BIT **row_pointers;
   U8BIT *source_data;
   U8BIT *src;
   U32BIT row, pixel;
   U32BIT bytes_per_line;
#ifdef OSD_16_BIT
   U16BIT *dest_line;
   U16BIT *dest;
#else
   U32BIT *dest_line;
   U32BIT *dest;
#endif
   U32BIT channels;
   U8BIT alpha, red, green, blue;

   FUNCTION_START(STB_IMGConvertPNG);

   retval = FALSE;
   *output_data = NULL;
   *output_data_size = 0;

   if ((image_data != NULL) && (image_data_size >= PNG_HEADER_SIZE))
   {
      if (ReadPNGHeader(image_data, image_data_size, pixel_width, pixel_height, &rowbytes, TRUE, &reader))
      {
         /* Allocate memory for the output image */
#ifdef OSD_16_BIT
         bytes_per_line = *pixel_width * 2;
#else /* OSD_32_BIT */
         bytes_per_line = *pixel_width * 4;
#endif
         *output_data_size = *pixel_height * bytes_per_line;

         *output_data = (U8BIT *)STB_GetMemory(*output_data_size);
         if (*output_data != NULL)
         {
            memset(*output_data, 0, *output_data_size);

            /* Allocate memory to read the entire image */
            source_data = (U8BIT *)STB_GetMemory(rowbytes * *pixel_height);
            if (source_data != NULL)
            {
               /* Need pointers to start of each row for reading */
               row_pointers = (U8BIT **)STB_GetMemory(*pixel_height * sizeof(U8BIT *));
               if (row_pointers != NULL)
               {
                  for (row = 0; row < *pixel_height; row++)
                  {
                     row_pointers[row] = source_data + (row * rowbytes);
                  }

                  png_read_image(reader.png_ptr, row_pointers);

                  channels = png_get_channels(reader.png_ptr, reader.info_ptr);

                  /* Now process the image to convert it into an image compatible with the OSD */
#ifdef OSD_16_BIT
                  dest_line = (U16BIT *)*output_data;
#else /* OSD_32_BIT */
                  dest_line = (U32BIT *)*output_data;
#endif

                  for (row = 0; row < *pixel_height; dest_line += *pixel_width, row++)
                  {
                     src = row_pointers[row];

                     if (channels == 3)
                     {
                        /* RGB image */
                        for (pixel = *pixel_width, dest = dest_line; pixel > 0;
                             dest++, src += 3, pixel--)
                        {
                           red = *src;
                           green = *(src + 1);
                           blue = *(src + 2);
#ifdef OSD_16_BIT
                           *dest = 0x0f;
                           *dest = (*dest << 4) | (red >> 4);
                           *dest = (*dest << 4) | (green >> 4);
                           *dest = (*dest << 4) | (blue >> 4);
#else /* OSD_32_BIT */
#ifdef OSD_ST_MODE
                           alpha = ST_ALPHA(0xff);
#else
                           alpha = 0xff;
#endif
                           *dest = (alpha << 24) | (red << 16) | (green << 8) | blue;
#endif
                        }
                     }
                     else
                     {
                        /* ARGB image */
                        for (pixel = *pixel_width, dest = dest_line; pixel > 0;
                             dest++, src += 4, pixel--)
                        {
                           alpha = *src;
                           red = *(src + 1);
                           green = *(src + 2);
                           blue = *(src + 3);
#ifdef OSD_16_BIT
                           *dest = (alpha >> 4);
                           *dest = (*dest << 4) | (red >> 4);
                           *dest = (*dest << 4) | (green >> 4);
                           *dest = (*dest << 4) | (blue >> 4);
#else /* OSD_32_BIT */
#ifdef OSD_ST_MODE
                           alpha = ST_ALPHA(alpha);
#endif
                           *dest = (alpha << 24) | (red << 16) | (green << 8) | blue;
#endif
                        }
                     }
                  }

                  STB_FreeMemory(row_pointers);

                  retval = TRUE;
               }

               STB_FreeMemory(source_data);
            }
            else
            {
               STB_FreeMemory(*output_data);
               *output_data = NULL;
            }
         }

         png_destroy_read_struct(&reader.png_ptr, &reader.info_ptr, &reader.end_info);
      }
   }

   FUNCTION_FINISH(STB_IMGConvertPNG);

   return(retval);
}

/*!**************************************************************************
 * @brief   Common function to read PNG header and return image info
 * @param   image_data - image data containing header
 * @param   num_bytes - number of header bytes supplied
 * @param   pixel_width - pointer to return image width in pixels
 * @param   pixel_height - pointer to return image height in pixels
 * @param   bytes_per_row - pointer to return num bytes per row
 * @param   keep_open - TRUE to keep the PNG open
 * @param   reader - used to return reading related info if PNG is kept open
 * @return  TRUE if reading succeeds, FALSE otherwise
 ****************************************************************************/
static BOOLEAN ReadPNGHeader(U8BIT *image_data, U32BIT num_bytes, U16BIT *pixel_width,
   U16BIT *pixel_height, U32BIT *bytes_per_row, BOOLEAN keep_open, S_PNG_READER *reader)
{
   BOOLEAN retval;
   U32BIT num_header_bytes;
   png_struct *png_ptr;
   png_infop info_ptr;
   png_infop end_info;
   S_PNG_BUFFER buffer;
#if (PNG_LIBPNG_VER != 10202) || defined (ENABLE_DEBUG)
   U32BIT bitdepth;
#endif
#ifdef ENABLE_DEBUG
   U32BIT resx, resy, res_unit;
#endif

   FUNCTION_START(ReadPNGHeader);

   retval = FALSE;

   if ((image_data != NULL) && (num_bytes > 0))
   {
      if (num_bytes < PNG_HEADER_SIZE)
      {
         num_header_bytes = num_bytes;
      }
      else
      {
         num_header_bytes = PNG_HEADER_SIZE;
      }

      /* Check that the PNG header indicates a valid PNG image */
      if (png_sig_cmp(image_data, 0, num_header_bytes) == 0)
      {
         png_ptr = png_create_read_struct_2(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL, NULL,
               PNG_Malloc, PNG_Free);
         if (png_ptr != NULL)
         {
            info_ptr = png_create_info_struct(png_ptr);
            if (info_ptr != NULL)
            {
               end_info = png_create_info_struct(png_ptr);
               if (end_info != NULL)
               {
                  /* Initialise the structure used for reading data */
                  if (reader != NULL)
                  {
                     reader->buffer.data = image_data;
                     reader->buffer.length = num_bytes;
                     reader->buffer.offset = 0;

                     /* Set the function that the PNG lib should use to read data */
                     png_set_read_fn(png_ptr, &reader->buffer, PNG_ReadData);
                  }
                  else
                  {
                     buffer.data = image_data;
                     buffer.length = num_bytes;
                     buffer.offset = 0;

                     /* Set the function that the PNG lib should use to read data */
                     png_set_read_fn(png_ptr, &buffer, PNG_ReadData);
                  }

                  /* Read the PNG header info to get details on the image */
                  png_read_info(png_ptr, info_ptr);

#ifdef ENABLE_DEBUG
                  DBG(("Original PNG image:"));
                  DBG(("  width=%d", png_get_image_width(png_ptr, info_ptr)));
                  DBG(("  height=%d", png_get_image_height(png_ptr, info_ptr)));
                  DBG(("  bitdepth=%d", png_get_bit_depth(png_ptr, info_ptr)));
                  DBG(("  colortype=%d", png_get_color_type(png_ptr, info_ptr)));
                  DBG(("  rowbytes=%d", png_get_rowbytes(png_ptr, info_ptr)));
                  DBG(("  channels=%d", png_get_channels(png_ptr, info_ptr)));
                  DBG(("  compression=%d", png_get_compression_type(png_ptr, info_ptr)));
                  if (png_get_pHYs(png_ptr, info_ptr, &resx, &resy, &res_unit) != 0)
                  {
                     DBG(("  resx=%lu, resy=%lu, unit=%lu", resx, resy, res_unit));
                  }
#endif

                  /* Set option to read 16-bit data as RGB RGB rather than RRGGBB */
                  png_set_strip_16(png_ptr);

                  /* Set the input transformations that are to be applied depending on the type
                   * of the image. The data is transformed so that it always reads as ARGB on input. */
#if (PNG_LIBPNG_VER != 10202) || defined (ENABLE_DEBUG)
                  bitdepth = png_get_bit_depth(png_ptr, info_ptr);
#endif

                  switch (png_get_color_type(png_ptr, info_ptr))
                  {
                     case PNG_COLOR_TYPE_GRAY:
#if (PNG_LIBPNG_VER != 10202)
                        if (bitdepth < 8)
                        {
                           png_set_expand_gray_1_2_4_to_8(png_ptr);
                        }
#endif
                        png_set_gray_to_rgb(png_ptr);
                        break;

                     case PNG_COLOR_TYPE_PALETTE:
                        png_set_palette_to_rgb(png_ptr);
                        if (png_get_valid(png_ptr, info_ptr, PNG_INFO_tRNS))
                        {
                           png_set_tRNS_to_alpha(png_ptr);
                        }
                        break;

                     case PNG_COLOR_TYPE_RGB:
                        break;

                     case PNG_COLOR_TYPE_RGB_ALPHA:
                        png_set_swap_alpha(png_ptr);
                        break;

                     default:
                        DBG(("PNG: unhandled PNG colour type %lu", png_get_color_type(png_ptr, info_ptr)));
                        break;
                  }

                  /* Now re-read the image info to take into account the input transforms */
                  png_read_update_info(png_ptr, info_ptr);

#ifdef ENABLE_DEBUG
                  DBG(("Output PNG image:"));
                  DBG(("  width=%d", png_get_image_width(png_ptr, info_ptr)));
                  DBG(("  height=%d", png_get_image_height(png_ptr, info_ptr)));
                  DBG(("  bitdepth=%d", png_get_bit_depth(png_ptr, info_ptr)));
                  DBG(("  colortype=%d", png_get_color_type(png_ptr, info_ptr)));
                  DBG(("  rowbytes=%d", png_get_rowbytes(png_ptr, info_ptr)));
                  DBG(("  channels=%d", png_get_channels(png_ptr, info_ptr)));
                  DBG(("  compression=%d", png_get_compression_type(png_ptr, info_ptr)));
                  if (png_get_pHYs(png_ptr, info_ptr, &resx, &resy, &res_unit) != 0)
                  {
                     DBG(("  resx=%lu, resy=%lu, unit=%lu", resx, resy, res_unit));
                  }
#endif

                  *pixel_width = (U16BIT)png_get_image_width(png_ptr, info_ptr);
                  *pixel_height = (U16BIT)png_get_image_height(png_ptr, info_ptr);
                  *bytes_per_row = png_get_rowbytes(png_ptr, info_ptr);

                  if (keep_open && (reader != NULL))
                  {
                     /* Pass the PNG reader info back to the calling function so it can be reused */
                     reader->png_ptr = png_ptr;
                     reader->info_ptr = info_ptr;
                     reader->end_info = end_info;
                  }
                  else
                  {
                     png_destroy_read_struct(&png_ptr, &info_ptr, &end_info);
                  }

                  retval = TRUE;
               }
               else
               {
                  png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
               }
            }
            else
            {
               png_destroy_read_struct(&png_ptr, NULL, NULL);
            }
         }
      }
   }

   FUNCTION_FINISH(ReadPNGHeader);

   return(retval);
}

/*!**************************************************************************
 * @brief   Function used by PNG library to allocate memory
 * @param   png_ptr - PNG structure
 * @param   size - number of bytes to be allocated
 * @return  address of allocated memory
 ****************************************************************************/
static png_voidp PNG_Malloc(png_structp png_ptr, png_size_t size)
{
   png_voidp addr;

   FUNCTION_START(PNG_Malloc);
   USE_UNWANTED_PARAM(png_ptr);

   addr = STB_GetMemory(size);

   FUNCTION_FINISH(PNG_Malloc);

   return(addr);
}

/*!**************************************************************************
 * @brief   Function used by PNG library to free memory
 * @param   png_ptr - PNG structure
 * @param   addr - address of memory to be freed
 ****************************************************************************/
static void PNG_Free(png_structp png_ptr, png_voidp addr)
{
   FUNCTION_START(PNG_Free);
   USE_UNWANTED_PARAM(png_ptr);

   STB_FreeMemory(addr);

   FUNCTION_FINISH(PNG_Free);
}

/*!**************************************************************************
 * @brief   Function used by PNG library to read data
 * @param   png_ptr - PNG structure
 * @param   data - address to which data is to be read
 * @param   length - number of bytes to read
 ****************************************************************************/
static void PNG_ReadData(png_structp png_ptr, png_bytep data, png_size_t length)
{
   S_PNG_BUFFER *buffer = (S_PNG_BUFFER *)png_get_io_ptr(png_ptr);

   FUNCTION_START(PNG_ReadData);

   if (length > buffer->length - buffer->offset)
   {
      length = buffer->length - buffer->offset;
   }

   if (length > 0)
   {
      memcpy(data, buffer->data + buffer->offset, length);
      buffer->offset += length;
   }

   FUNCTION_FINISH(PNG_ReadData);
}

