/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2009 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   API interfacing the midware with loader library
 * @file    stbota.c
 * @date    01/10/2009
 * @author  Sergio Panseri
 */

/*#define OTA_DEBUG*/

/*---includes for this file--------------------------------------------------*/
/* compiler library header files */
#include <stdio.h>

/* third party header files */

/* DVBCore header files */
#include <techtype.h>
#include <dbgfuncs.h>

#include "stbhwc.h"
#include "stbhwos.h"
#include "stbhwupg.h"

#include "stbdpc.h"
#include "stbheap.h"
#include "stberc.h"
#include "stbota.h"
//#include "stbver.h"
#include "stbsitab.h"

#include "dsm_control.h"
#include "dsm_client.h"

/*---constant definitions for this file--------------------------------------*/

#ifdef OTA_DEBUG
#define OTA_DBG(X)   STB_SPDebugWrite X
#else
#define OTA_DBG(X)
#endif

#define OTA_NVM_TASK_STACK       1024
#define OTA_NVM_TASK_PRIORITY    6
#define DSMCC_TASK_PRIORITY      6
#define MAX_NVM_MSGS             30

/*---local typedef structs for this file-------------------------------------*/

typedef enum
{
   MSG_INIT,
   MSG_START,
   MSG_RECEIVE,
   MSG_SAVE,
   MSG_STOP,
   MSG_DONE,
   MSG_ERROR,
   MSG_ABORT,
   MSG_QUIT
} E_MSG_TYPE;

typedef struct s_nvm_msg
{
   E_MSG_TYPE type;
   U32BIT moduleRef;
   U32BIT offset;
   U32BIT size;
   U8BIT *data;
} S_NVM_MSG;

typedef struct s_upg_mod
{
   U8BIT *name;
   U8BIT mtyp;
} S_UPG_MOD;

/*---local (static) variable declarations for this file----------------------*/

static H_DsmControl dsmcc_control = NULL;
static H_DsmCarousel dsmcc_carousel = NULL;
static void *wait_sem = NULL;
static void *dsm_mutex = NULL;
static BOOLEAN continue_with_download = FALSE;
static void* ota_nvm_queue = NULL;
static U32BIT total_image_size;
static U32BIT saved_image_size;
static U8BIT ota_download_percent = 0;
static S_UPG_MOD ota_upg_mod;
static F_SSU_VERSION_CALLBACK ssu_version_cb = NULL;

/*---local function prototypes for this file---------------------------------*/
static void* MemAlloc(U32BIT memSize);
static void MemFree(void* ptr);
static BOOLEAN ParseSsuSelectorBytes( U8BIT *selPtr, U8BIT selLen );
static void SsuStatusEvent( H_DsmCarousel carouselRef, E_UCLoadStatus status, U_StatusRef sr, U32BIT data );
static BOOLEAN SsuWanted( U32BIT grp_id, U32BIT grp_size, S_SsuModelVersion *smv );
static H_UsrRef SsuStartModule( E_ModuleType mtyp, U8BIT *name );
static BOOLEAN SsuWantModuleData( H_UsrRef usrRef, U32BIT moduleRef,
   U32BIT offset, U32BIT size, U32BIT crc );
static void SsuSaveModuleData( H_UsrRef usrRef, U32BIT moduleRef, U32BIT offset, U32BIT size, U8BIT *data );
static H_DsmControl DsmccOpen(void);
static U32BIT GetOui(void);
static void NvmSaveTask(void *param);


/*---global function definitions---------------------------------------------*/

/**
 * @brief   Starts the over-the-air download process
 * @param   onet_id original network ID of the transport containing the DSM-CC carousel
 * @param   tran_id transport ID of the transport containing the DSM-CC carousel
 * @param   serv_id service ID of the DSM-CC carousel, or DVB_INVALID_ID if the PID is given
 * @param   download_pid PID containing the download, or DVB_INVALID_ID if the service ID is given
 * @param   version_cb callback function that will be called when version info becomes available.
 *                     If the callback is NULL then the update will always be accepted
 * @return  TRUE if the update process is started, FALSE otherwise
 */
BOOLEAN STB_OTAStartLoader(U8BIT path, U16BIT onet_id, U16BIT tran_id, U16BIT serv_id,
   U16BIT download_pid, F_SSU_VERSION_CALLBACK version_cb)
{
   BOOLEAN retval;

   FUNCTION_START(STB_OTAStartLoader);
   ASSERT(path != INVALID_RES_ID)

   retval = TRUE;

   if (dsmcc_control == NULL)
   {
      retval = FALSE;

      dsmcc_control = DsmccOpen();
      if (dsmcc_control != NULL)
      {
         if (wait_sem == NULL)
         {
            wait_sem = STB_OSCreateSemaphore();
         }

         /* This semaphore is used to prevent the update process from continuing once the
          * download has completed */
         STB_OSSemaphoreWait(wait_sem);

         if (dsm_mutex == NULL)
         {
            dsm_mutex = STB_OSCreateMutex();
         }

         if (ota_nvm_queue == NULL)
         {
            ota_nvm_queue = STB_OSCreateQueue(sizeof(S_NVM_MSG), MAX_NVM_MSGS);

            STB_OSCreateTask(NvmSaveTask, NULL, OTA_NVM_TASK_STACK, OTA_NVM_TASK_PRIORITY,
               (U8BIT *)"OTA Nvm Task");
         }

         retval = TRUE;
      }
      else
      {
         OTA_DBG(("%s: Failed to create a DSMCC controller to start the download", __FUNCTION__));
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_OTA_SW_UPGRADE, EV_TYPE_OTA_SW_UPGRADE_ERROR, NULL, 0);
      }
   }

   if (retval)
   {
      total_image_size = 0;
      saved_image_size = 0;
      ota_download_percent = 0;
      ssu_version_cb = version_cb;

      STB_OSMutexLock(dsm_mutex);

      DSMCC_SetTunedTransportInfo(dsmcc_control, onet_id, tran_id, STB_DPGetPathDemux(path));

      if (download_pid == DVB_INVALID_ID)
      {
         OTA_DBG(("%s: Start DSMCC on service ID %u", __FUNCTION__, serv_id));
         retval = DSMCC_StartSsu(dsmcc_control, serv_id, GetOui());
      }
      else
      {
         OTA_DBG(("%s: Start DSMCC on PID %u", __FUNCTION__, download_pid));
         retval = DSMCC_StartSsuWithPid(dsmcc_control, download_pid, GetOui());
      }

      STB_OSMutexUnlock(dsm_mutex);

      if (!retval)
      {
         OTA_DBG(("%s: Failed to start DSMCC", __FUNCTION__));
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_OTA_SW_UPGRADE, EV_TYPE_OTA_SW_UPGRADE_ERROR, NULL, 0);
      }
   }

   FUNCTION_FINISH(STB_OTAStartLoader);

   return(retval);
}

/**
 * @brief   Stops the download, if still in progress, but should also be called when the
 *          update process has completed (unless the update results in a system reboot).
 */
void STB_OTAStopLoader(void)
{
   FUNCTION_START(STB_OTAStopLoader);

   if (dsmcc_control != NULL)
   {
      /* Set the flag to prevent the download from continuing */
      continue_with_download = FALSE;
      STB_OSSemaphoreSignal(wait_sem);

      STB_OSMutexLock(dsm_mutex);

      if (dsmcc_control != NULL)
      {
         OTA_DBG(("%s: Stopping DSMCC", __FUNCTION__));
         DSMCC_Stop(dsmcc_control, RST_MODE_FORCE);
         DSMCC_Close(dsmcc_control);
         dsmcc_control = NULL;
      }

      STB_OSMutexUnlock(dsm_mutex);
   }

   FUNCTION_FINISH(STB_OTAStopLoader);
}

/**
 * @brief   Returns the SSU download progress as a percentage
 * @return  Download progress as a percentage
 */
U8BIT STB_OTAGetProgress(void)
{
#if 0
   U32BIT percent;
#ifdef OTA_DEBUG
   static U32BIT last_pc = 0xFFFFFFFF;
#endif

   FUNCTION_START(STB_OTAGetProgress);

   STB_OSMutexLock(dsm_mutex);

   if ((dsmcc_control == NULL) || (total_image_size == 0))
   {
      percent = 0;
   }
   else
   {
      /* U32BIT isn't always big enough to handle the calculation for percentage, so change to KB */
      percent = ((saved_image_size / 1024) * 100) / (total_image_size / 1024);
#ifdef OTA_DEBUG
      if (percent != last_pc)
      {
         last_pc = percent;
         OTA_DBG(("%s: saved=%lu total=%lu percent=%lu", __FUNCTION__, saved_image_size,
            total_image_size, percent));
      }
#endif
   }

   STB_OSMutexUnlock(dsm_mutex);

   FUNCTION_FINISH(STB_OTAGetProgress);

   return (U8BIT)percent;
#else
   return ota_download_percent;
#endif
}

/**
 * @brief   This function should be called to unblock the download process.
 *          This can be used to in the case where an update may not be mandatory to allow
 *          a user to reject an update.
 * @param   status TRUE if the download should continue, FALSE otherwise
 */
void STB_OTAContinueDownload(BOOLEAN status)
{
   FUNCTION_START(STB_OTAContinueDownload);

   STB_OSMutexLock(dsm_mutex);

   if (dsmcc_control != NULL)
   {
      /* Set the flag indicating whether the download should continue and then allow it to proceed */
      continue_with_download = status;

      STB_OSSemaphoreSignal(wait_sem);
   }

   STB_OSMutexUnlock(dsm_mutex);

   FUNCTION_FINISH(STB_OTAContinueDownload);
}

/*---local function definitions----------------------------------------------*/

/**
 * @brief   Allocates the specified number of bytes
 * @param   size Required size of new memory block.
 * @return  Pointer to memory location.
 */
static void* MemAlloc(U32BIT memSize)
{
   void *data;

   FUNCTION_START(MemAlloc);

   data = STB_AppGetMemory(memSize);

   FUNCTION_FINISH(MemAlloc);

   return data;
}

/**
 * @brief   Releases previously allocated memory.
 * @param   ptr Pointer to the memory block to be freed
 */
static void MemFree(void* ptr)
{
   FUNCTION_START(MemFree);

   STB_AppFreeMemory(ptr);

   FUNCTION_FINISH(MemFree);
}

static BOOLEAN ParseSsuSelectorBytes( U8BIT *selPtr, U8BIT selLen )
{
   FUNCTION_START(ParseSsuSelectorBytes);
   USE_UNWANTED_PARAM(selPtr);
   USE_UNWANTED_PARAM(selLen);
   FUNCTION_FINISH(ParseSsuSelectorBytes);
   return TRUE;
}

/**
 * @brief   SSU callback to inform client of status
 * @param   hCarousel Handle to the DSMCC Update Carousel
 * @param   status Current load status
 * @param   id Meaning depends on status value:
 *             SSU_NOT_AVAILABLE - id is OUI
 *             SSU_LOAD_STARTING - id is Group or download ID
 *             SSU_LOAD_COMPLETE - id is Group or download ID
 *             SSU_LOAD_ABORTED
 * @return  void
 */
static void SsuStatusEvent( H_DsmCarousel carouselRef, E_UCLoadStatus status, U_StatusRef sr, U32BIT data )
{
   S_NVM_MSG msg;

   FUNCTION_START(SsuStatusEvent);
   USE_UNWANTED_PARAM(sr);

   switch (status)
   {
      case SSU_NOT_AVAILABLE:
      {
         OTA_DBG(("%s: SSU_NOT_AVAILABLE", __FUNCTION__));
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_OTA_SW_UPGRADE, EV_TYPE_OTA_SW_UPGRADE_NOTFOUND, NULL, 0);
         dsmcc_carousel = carouselRef;
         break;
      }
      case SSU_CRSL_READY:
      {
         OTA_DBG(("%s: SSU_CRSL_READY", __FUNCTION__));
         dsmcc_carousel = carouselRef;
         /* 'data' contains size of all module data */
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_OTA_SW_UPGRADE, EV_TYPE_OTA_SW_UPGRADE_DOWNLOADING, NULL, 0);
         total_image_size = data;
         break;
      }
      case SSU_LOAD_STARTING:
      {
         OTA_DBG(("%s: SSU_LOAD_STARTING", __FUNCTION__));
         msg.type = MSG_START;
         /* 'data' contains size of data for this module type */
         STB_OSWriteQueue(ota_nvm_queue, (void *)&msg, sizeof(S_NVM_MSG), TIMEOUT_NEVER);
         break;
      }
      case SSU_LOAD_RECEIVING:
      {
         OTA_DBG(("%s: SSU_LOAD_RECEIVING", __FUNCTION__));
         /* 'data' contains the download progress of received module data */
         ota_download_percent = data;
         //dsmcc_carousel = carouselRef;
         U32BIT percent = data;
         msg.type = MSG_RECEIVE;
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_OTA_SW_UPGRADE, EV_TYPE_OTA_SW_UPGRADE_DOWNLOADING, &percent, sizeof(U32BIT));
         break;
      }
      case SSU_LOAD_COMPLETE:
      {
         OTA_DBG(("%s: SSU_LOAD_COMPLETE", __FUNCTION__));
         msg.type = MSG_STOP;
         STB_OSWriteQueue(ota_nvm_queue, (void *)&msg, sizeof(S_NVM_MSG), TIMEOUT_NEVER);
         break;
      }
      case SSU_LOAD_ABORTED:
      {
         OTA_DBG(("%s: SSU_LOAD_ABORTED", __FUNCTION__));
         msg.type = MSG_ERROR;
         STB_OSWriteQueue(ota_nvm_queue, (void *)&msg, sizeof(S_NVM_MSG), TIMEOUT_NEVER);
         break;
      }
      case SSU_CRSL_DONE:
      {
         OTA_DBG(("%s: SSU_CRSL_DONE, saved_image_size=%lu, total_image_size=%lu", __FUNCTION__,
            saved_image_size, total_image_size));
         if (saved_image_size != total_image_size)
         {
            msg.type = MSG_ERROR;
         }
         else
         {
            msg.type = MSG_DONE;
         }
         STB_OSWriteQueue(ota_nvm_queue, (void *)&msg, sizeof(S_NVM_MSG), TIMEOUT_NEVER);
         break;
      }
      case SSU_CRSL_ABORT:
      {
         OTA_DBG(("%s: SSU_CRSL_ABORT", __FUNCTION__));
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_OTA_SW_UPGRADE, EV_TYPE_OTA_SW_UPGRADE_NOTFOUND, NULL, 0);
         msg.type = MSG_ABORT;
         saved_image_size = total_image_size;
         STB_OSWriteQueue(ota_nvm_queue, (void *)&msg, sizeof(S_NVM_MSG), TIMEOUT_NEVER);
         break;
      }
      case SSU_CRSL_UNLOAD:
      {
         OTA_DBG(("%s: SSU_CRSL_UNLOAD", __FUNCTION__));
         break;
      }
      default:
      {
         OTA_DBG(("%s: Invalid status %d", __FUNCTION__, status));
         break;
      }
   }

   FUNCTION_FINISH(SsuStatusEvent);
}

/**
 * @brief   SSU callback to ask client whether these download details match the product
 *          is a valid sofware upgrade.
 * @param   grp_id Group ID (Bits: 1-15 are Download ID)
 * @param   grp_size Total size of group (i.e. size of image data)
 * @param   swv Model and version details for this group
 * @return  BOOLEAN - TRUE if this is required
 */
static BOOLEAN SsuWanted( U32BIT grp_id, U32BIT grp_size, S_SsuModelVersion *smv )
{
   BOOLEAN result = FALSE;
   BOOLEAN valid_download = TRUE;

   FUNCTION_START(SsuWanted);

   if (ssu_version_cb != NULL)
   {
      /* Call provided callback to see if the update should proceed */
      valid_download = (*ssu_version_cb)(grp_id, grp_size, smv->hw_model, smv->hw_version,
         smv->sw_model, smv->sw_version);
   }

   if (valid_download)
   {
      /* Update is available and its valid, so inform the app that an update has been found */
      STB_ERSendEvent(FALSE, FALSE, EV_CLASS_OTA_SW_UPGRADE, EV_TYPE_OTA_SW_UPGRADE_FOUND, NULL, 0);

      /* The update can continue but may be up to the user, so block here until told to continue */
      STB_OSSemaphoreWait(wait_sem);

      result = continue_with_download;
      total_image_size = grp_size;
      saved_image_size = 0;
      OTA_DBG(("%s: continue=%u, total=%lu", __FUNCTION__, result, total_image_size));
   }

   FUNCTION_FINISH(SsuWanted);

   return result;
}

/**
 * @brief   SSU callback to ask client whether if client wants to download data connected
 *          to this module. On returning TRUE, this is always followed by calls to
 *          F_SsuWantModuleData() and F_SsuSaveModuleData().
 *          Another call to this function will only occur after SSU_LOAD_COMPLETE event
 * @param   mtyp Type of module download
 * @param   name Name of module download
 * @return  H_UsrRef - NULL if module data is not required
 */
static H_UsrRef SsuStartModule( E_ModuleType mtyp, U8BIT *name )
{
   H_UsrRef retval = NULL;

   FUNCTION_START(SsuStartModule);

   if (STB_UPGStart((U8BIT)mtyp, name ))
   {
      ota_upg_mod.mtyp = (U8BIT)mtyp;
      ota_upg_mod.name = name;
      retval = &ota_upg_mod;
   }
#ifdef OTA_DEBUG
   else
   {
      OTA_DBG(("STB_UPGStart failed"));
   }
#endif

   FUNCTION_FINISH(SsuStartModule);

   return retval;
}

/**
 * @brief   SSU callback function supplied by client which DSM-CC calls to check
 *          whether client requires this module. It is expected that the client
 *          may keep previously downloaded modules in NVM, and therefore avoid
 *          repeating downloads if power cycle interrupted a previously started update.
 * @param   usrRef Reference returned by F_SsuStartModule()
 * @param   moduleRef Reference to module data
 * @param   offset Offset for the module data
 * @param   size Size of this module data
 * @param   crc CRC for of this module data
 * @return  BOOLEAN - TRUE if module is required
 */
static BOOLEAN SsuWantModuleData( H_UsrRef usrRef, U32BIT moduleRef,
   U32BIT offset, U32BIT size, U32BIT crc )
{
   FUNCTION_START(SsuWantModuleData);
   USE_UNWANTED_PARAM(usrRef);
   USE_UNWANTED_PARAM(moduleRef);
   USE_UNWANTED_PARAM(offset);
   USE_UNWANTED_PARAM(size);
   USE_UNWANTED_PARAM(crc);
#ifdef TODO_CHECK_NVM
   /* TODO: Check whether NVM has this module using        */
#endif
   FUNCTION_FINISH(SsuWantModuleData);
   return TRUE;
}

/**
 * @brief   SSU callback function supplied by client - called when download of a
 *          module has completed. The data will remain in memory after returning
 *          from this funcrtion and until the client calls DSMCC_SsuUnloadModule
 * @param   usrRef Reference returned by F_SsuStartModule()
 * @param   moduleRef Reference to module data
 * @param   offset Offset for the module data
 * @param   size Size of the module data
 * @param   data Pointer to the data for this module data
 * @return  void
 */
static void SsuSaveModuleData( H_UsrRef usrRef, U32BIT moduleRef, U32BIT offset, U32BIT size, U8BIT *data )
{
   S_NVM_MSG msg;

   FUNCTION_START(SsuSaveModuleData);

   USE_UNWANTED_PARAM(usrRef);

   msg.type = MSG_SAVE;
   msg.moduleRef = moduleRef;
   msg.offset = offset;
   msg.size = size;
   msg.data = data;

   OTA_DBG(("%s: saved=%lu total=%lu", __FUNCTION__, saved_image_size, total_image_size));
   STB_OSWriteQueue(ota_nvm_queue, (void *)&msg, sizeof(S_NVM_MSG), TIMEOUT_NEVER);

   saved_image_size += size;

   FUNCTION_FINISH(SsuSaveModuleData);
}

static H_DsmControl DsmccOpen(void)
{
   S_DsmccConfig config;

   FUNCTION_START(DsmccOpen);

   config.taskPriority = DSMCC_TASK_PRIORITY;
   config.sectionBuffPoolSize = 180;
   config.sectionBuffCacheSize = 16;
   config.memAlloc = MemAlloc;
   config.memFree = MemFree;
   config.parsePmtInit = NULL;
   config.parseDataBroadcastId = NULL;
   config.parseSsuSelectorBytes = ParseSsuSelectorBytes;
   config.parsePmtDone = NULL;
   config.ssuFuncs.status = SsuStatusEvent;
   config.ssuFuncs.wanted = SsuWanted;
   config.ssuFuncs.startModule = SsuStartModule;
   config.ssuFuncs.wantModuleData = SsuWantModuleData;
   config.ssuFuncs.saveModuleData = SsuSaveModuleData;
   config.notifyAitInfo = NULL;
   config.controlFlags = GET_PMT_DIRECT_FROM_HW_FLAG;

   FUNCTION_FINISH(DsmccOpen);

   return DSMCC_Open( &config );
}

static U32BIT GetOui(void)
{
   U32BIT oui;
   U8BIT *oui_str = STB_HWGetOUI();

   FUNCTION_START(GetOui);

   if ((oui_str = STB_HWGetOUI()) != NULL)
   {
      oui = *oui_str++;
      oui = (oui << 8) | *oui_str++;
      oui = (oui << 8) | *oui_str;
   }
   else
   {
      oui = 0;
   }

   FUNCTION_FINISH(GetOui);

   return oui;
}

static void NvmSaveTask(void *param)
{
   S_NVM_MSG msg;

   FUNCTION_START(NvmSaveTask);
   USE_UNWANTED_PARAM(param);

   msg.type = MSG_INIT;

   while (msg.type != MSG_QUIT)
   {
      if (STB_OSReadQueue(ota_nvm_queue, (void *)&msg, sizeof(S_NVM_MSG), TIMEOUT_NEVER))
      {
         switch (msg.type)
         {
            case MSG_START:
            {
               break;
            }
            case MSG_SAVE:
            {
               /* save data */
               OTA_DBG(("%s: SAVING moduleRef=%x offset=%d size=%d", __FUNCTION__, msg.moduleRef,
                  msg.offset, msg.size));
               if (!STB_UPGWrite(ota_upg_mod.mtyp, msg.offset, msg.size, msg.data))
               {
                  OTA_DBG(("%s: Failed to write OTA at offset %d bytes=%d", __FUNCTION__,
                     msg.offset, msg.size));
                  //TODO: do something - re-try?
               }
               else
               {
                  STB_OSMutexLock(dsm_mutex);
                  if (dsmcc_control != NULL)
                  {
                     DSMCC_SsuUnloadModule(dsmcc_control, dsmcc_carousel, msg.moduleRef);
                  }
                  STB_OSMutexUnlock(dsm_mutex);
               }
               break;
            }
            case MSG_STOP:
            {
               STB_UPGFinish( ota_upg_mod.mtyp, TRUE );
               STB_ERSendEvent(FALSE, FALSE, EV_CLASS_OTA_SW_UPGRADE, EV_TYPE_OTA_SW_UPGRADE_COMPLETED, NULL, 0);
               break;
            }
            case MSG_ERROR:
            {
               STB_UPGFinish( ota_upg_mod.mtyp, FALSE );
               STB_ERSendEvent(FALSE, FALSE, EV_CLASS_OTA_SW_UPGRADE, EV_TYPE_OTA_SW_UPGRADE_ERROR, NULL, 0);
               break;
            }
            case MSG_DONE:
            {
               STB_OSMutexLock(dsm_mutex);
               if (dsmcc_control != NULL)
               {
                  DSMCC_Stop(dsmcc_control,RST_MODE_FORCE);
               }
               STB_OSMutexUnlock(dsm_mutex);
               break;
            }
            case MSG_QUIT:
            default:
            {
               STB_OSMutexLock(dsm_mutex);
               if (dsmcc_control != NULL)
               {
                  DSMCC_Stop(dsmcc_control,RST_MODE_FORCE);
               }
               STB_OSMutexUnlock(dsm_mutex);
               break;
            }
         }
      }
   }

   FUNCTION_FINISH(NvmSaveTask);
}

/****************************************************************************
** End of file
*****************************************************************************/


