/*******************************************************************************
 * Copyright  2016 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief    CI Content Control
 * @file     stbcicc.h
 * @date     28 November 2016
 * @author   Adam Sturtridge
 */

#ifndef _STBCICC_H
#define _STBCICC_H

#include "techtype.h"

/*---Constant and macro definitions for public use-----------------------------*/

#define CIP_CICAM_ID_LEN            8
#define CIP_URI_LEN                 8
#define CIP_MAX_PIN_LENGTH          8
#define CIP_PIN_PRIVATE_DATA_SIZE   15

/*---Enumerations for public use-----------------------------------------------*/

/*---Global type defs for public use-------------------------------------------*/

/*---Global Function prototypes for public use---------------------------------*/

/**
 * @brief    CI content control support initialise
 */
void STB_CiCcInitialise(void);

/**
 * @brief   Handle CAM removal from slot
 * @param   slot_id zero-based CI slot identifier (0, 1, ...)
 */
void STB_CiCcRemove(U8BIT slot_id);

/**
 * @brief    Notify module insertion
 * @param    slot_id - slot ID for module
 */
void STB_CiCcNotifyModuleInsert(U8BIT slot_id);

/**
 * @brief    Notify module removal
 * @param    slot_id - slot ID for module
 */
void STB_CiCcNotifyModuleRemove(U8BIT slot_id);

/**
 * @brief   Return whether CI slot is ready
 * @param   slot_id - slot ID (0, 1, ...)
 * @return  BOOLEAN - TRUE if ready
 */
BOOLEAN STB_CiCcIsSlotReady(U8BIT slot_id);

/**
 * @brief   To implement the first part of the diagram in "Figure 10.2: Shunning Operation" of
 *          ci-plus_specification_v1.3.2.pdf, this function passes the status of the SDT acquisition
 *          to the ci+ glue.
 * @param   sdt_acquisition_status TRUE if the SDT acquisition is completed (either successfully if
 *          the table has been received, or unsuccessfully if the SDT timeout has expired), FALSE if
 *          the DVB engine is waiting for an SDT.
 */
void STB_CiCcSetSDTAcquisitionStatus(BOOLEAN complete);

/**
 * @brief   Tell whether the service is allowed. This function checks whether
 *          the CI Protection Descriptor allows the service to be routed
 *          through the CI module.
 * @param   slot_id slot ID (0, 1, ...)
 * @param   ci_prot_desc CI Protection Descriptor for the service
 * @return  BOOLEAN - TRUE if allowed
 */
BOOLEAN STB_CiCcIsServiceAllowed(U8BIT slot_id, U8BIT *ci_prot_desc);

/**
 * @brief   Tell whether authenticated CI+ module is in the slot
 * @param   slot_id slot ID (0, 1, ...)
 * @return  BOOLEAN - TRUE if authenticated
 */
BOOLEAN STB_CiCcAuthenticated(U8BIT slot_id);

/**
 * @brief   Checks CAMs in all slots to find the one with the given CAM id
 * @param   cicam_id - CAM id to look for
 * @return  slot id if found, INVALID_RES_ID if not found
 */
U8BIT STB_CiCcFindSlotForCicamId(U8BIT cicam_id[CIP_CICAM_ID_LEN]);

/**
 * @brief    Return the current URI for the given service
 * @param    slot_id                  - slot ID
 * @param    service_id               - service ID
 * @param    uri                      - the URI for the service
 * @return   URI for the service if available or "empty" URI otherwise
 */
void STB_CiCcGetUsageRulesInfo(U8BIT slot_id, U16BIT service_id,
   U8BIT uri[CIP_URI_LEN]);

/**
 * @brief   Sets the default URI values into the given array. The URI requires HDCP
 * @param   raw_uri - returns with the default URI settings
 */
void STB_CiCcGetDefaultUsageRulesInfo(U8BIT raw_uri[CIP_URI_LEN]);

/**
 * @brief    Return the retention limit given a packed URI
 * @param    uri                      - Usage Rules Information
 * @return   Retention limit in minutes (0 = no limit)
 */
U32BIT STB_CiCcGetRetentionLimit(U8BIT uri[CIP_URI_LEN]);

/**
 * @brief    Apply the given Usage Rules Information
 * @param    uri                      - Usage Rules Information
 */
void STB_CiCcApplyUsageRulesInfo(U8BIT *uri);

/**
 * @brief    Apply the given Usage Rules Information for playback
 * @param    uri                      - Usage Rules Information
 */
void STB_CiCcApplyUsageRulesInfoForPlayback(U8BIT *uri);

/**
 * @brief    Tell whether the given service requires HDCP
 * @param    service_id - Service ID
 */
BOOLEAN STB_CiCcIsHDCPRequired(U16BIT service_id);

/**
 * @brief   Sends a CICAM licence to a module during playback, which will result
 *          in a modified licence being notified through STB_CINotifyCicamLicence
 * @param   slot_id - slot
 * @param   program_number - program number being played back
 * @param   licence - CICAM licence
 * @param   licence_len - licence length in bytes
 * @return  TRUE if operation succeeded, FALSE otherwise
 */
BOOLEAN STB_CiCcSendPlaybackLicence(U8BIT slot_id, U16BIT program_number, U8BIT *licence,
   U16BIT licence_len);

/**
 * @brief   Returns the last licence received from the CAM when recording
 * @param   slot_id - slot
 * @param   licence_status - status value associated with the licence
 * @param   licence_len - pointer to return the licence length in bytes
 * @param   raw_uri - returns with the packed URI data associated with the licence
 * @return  pointer to the licence data, mustn't be freed. NULL if no licence
 */
U8BIT *STB_CiCcGetRecordingLicence(U8BIT slot_id, U8BIT * licence_status, U16BIT * licence_len,
                               U8BIT raw_uri[CIP_URI_LEN]);

/**
 * @brief   Returns the last licence received from the CAM during playback
 * @param   slot_id - slot
 * @param   licence_status - status value associated with the licence
 * @param   licence_len - pointer to return the licence length in bytes
 * @param   raw_uri - returns with the packed URI data associated with the licence
 * @return  pointer to the licence data, mustn't be freed. NULL if no licence
 */
U8BIT *STB_CiCcGetPlaybackLicence(U8BIT slot_id, U8BIT * licence_status, U16BIT * licence_len,
                              U8BIT raw_uri[CIP_URI_LEN]);

/**
 * @brief   Called by the host to check whether a CAM pin is valid.
 *          An STB_EVENT_CI_PIN_STATUS event will be sent to notify the host
 *          of the validity, or otherwise, of the pin.
 * @param   slot_id - slot
 * @param   pin_data - ASCII encoded pin data, null terminated
 * @return  TRUE if the pin was sent successfully, FALSE otherwise
 */
BOOLEAN STB_CiCcSendPin(U8BIT slot_id, U8BIT *pin_data);

/**
 * @brief   Returns the information to be stored with a pin event when recording
 * @param   slot_id - slot being used for the recording
 * @param   status - pin status code
 * @param   age_rating - returns the age rating supplied with the pin event
 * @param   private_data - returns a pointer to the private data to be saved - don't free
 * @param   date_code - MJD UTC date code when the pin is to be applied
 * @param   hour - UTC hour when the pin is to be applied
 * @param   min - UTC minute when the pin is to be applied
 * @param   sec - UTC second when the pin is to be applied
 * @return  TRUE if the info is returned, FALSE otherwise
 */
BOOLEAN STB_CiCcGetRecordingPinInfo(U8BIT slot_id, U8BIT *status, U8BIT *age_rating,
   U8BIT **private_data, U16BIT *date_code, U8BIT *hour, U8BIT *min, U8BIT *sec);

/**
 * @brief   Sends a pin event to the CAM during playback
 * @param   slot_id - slot
 * @param   age_rating - rating as provided by the pin event during recording
 * @param   private_data - pin private data as provided by the pin event during recording
 * @return  TRUE on success, FALSE otherwise
 */
BOOLEAN STB_CiCcSendPinPlayback(U8BIT slot_id, U8BIT age_rating, U8BIT *private_data);

/**
 * @brief   Sets the record operating mode for the given slot id
 * @param   slot_id - slot to be, or being, used for recording
 * @param   mode - operating mode of the host
 */
void STB_CiCcSetRecordOperatingMode(U8BIT slot_id, U32BIT mode, U16BIT service_id);

/**
 * @brief   Called by the app when a recording is to be started on a CA protected service
 * @param   slot_id - slot being used for the recording
 * @param   program_number - service ID
 * @param   pin_string - pin as a null terminated ascii string
 * @return  TRUE on success, but this doesn't mean the recording can start, FALSE otherwise
 */
BOOLEAN STB_CiCcSendRecordStart(U8BIT slot_id, U16BIT program_number, U8BIT *pin_string);

/**
 * @brief   Called by the app when a recording is stopped or completes
 * @param   slot_id - slot to be, or being, used for recording
 * @return  TRUE on success, FALSE otherwise
 */
BOOLEAN STB_CiCcSendRecordStop(U8BIT slot_id);

#endif /* _STBCICC_H */
