/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Header file - macros and function prototypes for public use
 *
 * @file    stbdpc.h
 * @date    26/09/2000
 */

// pre-processor mechanism so multiple inclusions don't cause compilation error

#ifndef _STBDPC_H

#define _STBDPC_H

#include "techtype.h"
#include "stbhwtun.h"


//---Constant and macro definitions for public use-----------------------------

// Maximum DISEQC_1_2 dish position & invalid position index.
// Invalid dish pos is zero, to help reissue the pos command when coming out of
// standby or leave the positioning screen without saving.
// NB. issuing the command move to index zero will center the dish (due south).
//     and the function in stbdpc.c should stop the valid command being sent
//     by move command.
#define MAX_DISEQC_1_2_POSITION     60
#define INVALID_POSITION_INDEX      0x00

#define MAX_UNICABLE_BANKS 8

#ifndef INVALID_RES_ID
#define INVALID_RES_ID     ((U8BIT)0xFF)     /* ID used to represent an invalid resource */
#endif

#define INVALID_CA_DESCRAMBLER_HANDLE  ((U32BIT)0x0000)

//---Enumerations for public use-----------------------------------------------

typedef enum e_stb_dp_tune_status
{
   TUNE_WAITING = 0,
   TUNE_NO_LOCK = 1,
   TUNE_LOCKED = 2,
   TUNE_STOPPED = 3
}
E_STB_DP_TUNE_STATUS;

typedef enum e_stb_dp_decode_status
{
   DECODE_STOPPING = 0,
   DECODE_STARTING = 1,
   DECODE_STOPPED = 2,
   DECODE_RUNNING = 3,
   DECODE_LOCKED = 4
}
E_STB_DP_DECODE_STATUS;

typedef enum e_stb_dp_demux_source
{
   DEMUX_SOURCE_TUNER = 0,
   DEMUX_SOURCE_FILE = 1,
   DEMUX_SOURCE_LINEIN = 2
}
E_STB_DP_DEMUX_SOURCE;

typedef enum e_stb_dp_decode_source
{
   DECODE_SOURCE_DEMUX = 0,
   DECODE_SOURCE_FILE = 1,
   DECODE_SOURCE_LINEIN = 2
}
E_STB_DP_DECODE_SOURCE;

typedef enum e_stb_dp_signal_type
{
   SIGNAL_NONE = 0,
   SIGNAL_QPSK = 1,      // digital satellite
   SIGNAL_COFDM = 2,     // digital terrestrial
   SIGNAL_QAM = 4,       // digital cable
   SIGNAL_ANALOG = 8
}
E_STB_DP_SIGNAL_TYPE;

typedef enum e_stb_dp_lnb_power
{
   LNB_POWER_OFF = 0,
   LNB_POWER_ON = 1,
   LNB_POWER_AUTO = 2
}
E_STB_DP_LNB_POWER;

typedef enum e_stb_dp_lnb_type
{
   LNB_TYPE_SINGLE = 0,
   LNB_TYPE_UNIVERSAL = 1,
   LNB_TYPE_UNICABLE = 2,
   LNB_TYPE_USER_DEFINED = 3,
   LNB_TYPE_MAX
}
E_STB_DP_LNB_TYPE;

typedef enum e_stb_dp_diseqc_cswitch
{
   DISEQC_CSWITCH_OFF = 0,
   DISEQC_CSWITCH_A = 1,
   DISEQC_CSWITCH_B = 2,
   DISEQC_CSWITCH_C = 3,
   DISEQC_CSWITCH_D = 4
}
E_STB_DP_DISEQC_CSWITCH;

typedef enum e_stb_dp_diseqc_tone
{
   DISEQC_TONE_OFF = 0,
   DISEQC_TONE_A = 1,
   DISEQC_TONE_B = 2
}
E_STB_DP_DISEQC_TONE;

typedef enum e_stb_dp_polarity
{
   POLARITY_HORIZONTAL = 0,
   POLARITY_VERTICAL = 1,
   POLARITY_LEFT = 2,
   POLARITY_RIGHT = 3
}
E_STB_DP_POLARITY;

typedef enum e_stb_dp_fec
{
   FEC_AUTOMATIC = 0,
   FEC_1_2 = 1,
   FEC_2_3 = 2,
   FEC_3_4 = 3,
   FEC_5_6 = 4,
   FEC_7_8 = 5,
   // Extra FEC modes for DVB-S2
   FEC_1_4 = 6,
   FEC_1_3 = 7,
   FEC_2_5 = 8,
   FEC_8_9 = 9,
   FEC_9_10 = 10,   
   FEC_3_5 = 11,
   FEC_4_5 = 12
}
E_STB_DP_FEC;

typedef enum e_stb_dp_fec_type
{
   FEC_TYPE_AUTO,
   FEC_TYPE_DVBS1,
   FEC_TYPE_DVBS2
}
E_STB_DP_FEC_TYPE;

typedef enum e_stb_dp_modulation
{
   MOD_AUTO,
   MOD_QPSK,   /* quartenary phase shift key */
   MOD_8PSK,   /* octenary phase shift key */
   MOD_16QAM   /* Not valid for DVB-S2 */
}
E_STB_DP_MODULATION;

typedef enum e_stb_dp_tmode
{
   MODE_COFDM_2K,
   MODE_COFDM_8K,
   MODE_COFDM_4K,
   MODE_COFDM_1K,
   MODE_COFDM_16K,
   MODE_COFDM_32K,
   MODE_COFDM_UNDEFINED
}
E_STB_DP_TMODE;

typedef enum e_stb_dp_tbwidth
{
   TBWIDTH_8MHZ,
   TBWIDTH_7MHZ,
   TBWIDTH_6MHZ,
   TBWIDTH_5MHZ,
   TBWIDTH_10MHZ,
   TBWIDTH_UNDEFINED
}
E_STB_DP_TBWIDTH;

typedef enum  e_stb_dp_cmode
{
   MODE_QAM_AUTO,
   MODE_QAM_4,
   MODE_QAM_8,
   MODE_QAM_16,
   MODE_QAM_32,
   MODE_QAM_64,
   MODE_QAM_128,
   MODE_QAM_256
}
E_STB_DP_CMODE;

typedef enum e_stb_dp_tune_terr_type
{
   TERR_TYPE_UNKNOWN,
   TERR_TYPE_DVBT,
   TERR_TYPE_DVBT2,
} E_STB_DP_TTYPE;

typedef enum e_stb_dp_audio_mode
{
   AUDIO_STEREO = 0,
   AUDIO_LEFT = 1,
   AUDIO_RIGHT = 2,
   AUDIO_MONO = 3,
   AUDIO_MULTICHANNEL = 4,
   AUDIO_UNDEF = 5
}
E_STB_DP_AUDIO_MODE;

typedef enum e_stb_ota_sw_upgrade_search_mode
{
   OTA_SEARCH_OFF = 0,
   OTA_SEARCH_AUTO = 1,
   OTA_SEARCH_MANUAL = 2
} E_STB_OTA_SW_UPGRADE_SEARCH_MODE;

typedef enum e_stb_dp_analogue_video_type
{
   ANLG_VIDEO_PAL_I = 0,
   ANLG_VIDEO_PAL_B = 1,
   ANLG_VIDEO_PAL_G = 2,
   ANLG_VIDEO_PAL_D = 3,
   ANLG_VIDEO_PAL_K = 4,
   ANLG_VIDEO_PAL_L = 5,
   ANLG_VIDEO_PAL_LDASH = 6
} E_STB_DP_ANALOG_VIDEO_TYPE;

typedef enum e_stb_dp_video_codec
{
   VIDEO_CODEC_AUTO = 0,
   VIDEO_CODEC_MPEG1 = 1,
   VIDEO_CODEC_MPEG2 = 2,
   VIDEO_CODEC_H264 = 3,
   VIDEO_CODEC_H265 = 4
} E_STB_DP_VIDEO_CODEC;

typedef enum e_stb_dp_audio_codec
{
   AUDIO_CODEC_AUTO = 0,
   AUDIO_CODEC_MP2 = 1,
   AUDIO_CODEC_MP3 = 2,
   AUDIO_CODEC_AC3 = 3,
   AUDIO_CODEC_EAC3 = 4,
   AUDIO_CODEC_AAC = 5,
   AUDIO_CODEC_HEAAC = 6,
   AUDIO_CODEC_AAC_ADTS = 7
} E_STB_DP_AUDIO_CODEC;

typedef enum e_stb_dp_ad_audio
{
   AD_AUDIO_OFF = 0,
   AD_AUDIO_ON = 1,
   AD_AUDIO_PAUSED = 2,
   AD_AUDIO_PLAYING = 3,
   AD_AUDIO_WAITING = 4
} E_STB_DP_AD_AUDIO;


typedef enum
{
   RES_OWNER_NONE,
   RES_OWNER_DVB,
   RES_OWNER_CIPLUS
} E_STB_DP_RES_OWNER;

typedef enum
{
   DP_PRIORITY_LOW,
   DP_PRIORITY_HIGH
} E_STB_DP_PRIORITY;

typedef struct
{
   E_STB_DP_POLARITY polarity;
   U16BIT min_freq;
   U16BIT max_freq;
   U16BIT local_oscillator_frequency;
   E_STB_TUNE_LNB_VOLTAGE lnb_voltage;
   BOOLEAN tone_22k;
} S_STB_DP_LNB_BAND;


//---Global type defs for public use-------------------------------------------

//---Global Function prototypes for public use---------------------------------

void STB_DPInitialise(void);

U8BIT STB_DPGetNumPaths(void);

U8BIT STB_DPAcquireTunerPath(E_STB_DP_SIGNAL_TYPE tuner_type, void *service, void *transport,
   E_STB_DP_RES_OWNER owner, E_STB_DP_PRIORITY priority, BOOLEAN with_decoders, BOOLEAN for_recording);

U8BIT STB_DPAcquirePlaybackPath(void *service);

void STB_DPReleaseDecoders(U8BIT path);

BOOLEAN STB_DPReleasePath(U8BIT path, E_STB_DP_RES_OWNER owner);
void STB_DPReleaseAllPaths(void);

U8BIT STB_DPAcquireCISlotForPath(U8BIT path, U8BIT *pmt_data, U8BIT *ci_protection_desc);
BOOLEAN STB_DPUseCISlotWithPath(U8BIT path, U8BIT slot_id);
void STB_DPReleaseCISlotFromPath(U8BIT path);
U8BIT STB_DPGetPathCISlot(U8BIT path);
U8BIT STB_DPIsCISlotInUse(U8BIT start_path, U8BIT slot_id, U8BIT ignore_path);

BOOLEAN STB_DPAcquireCADescramblerForPath(U8BIT path, U8BIT *pmt_data, U32BIT *ca_handle);
void STB_DPReleaseCADescramblerFromPath(U8BIT path);
BOOLEAN STB_DPGetPathCADescrambler(U8BIT path, U32BIT *handle);

U8BIT STB_DPGetLivePath(void);
BOOLEAN STB_DPIsLivePath(U8BIT path);
BOOLEAN STB_DPIsRecordingPath(U8BIT path);
BOOLEAN STB_DPIsDecodingPath(U8BIT path);
U8BIT STB_DPGetPlaybackPath(void);
BOOLEAN STB_DPIsMonitoringPath(U8BIT path);
U8BIT STB_DPGetMonitoringPath(void);

U8BIT STB_DPPathForTuner(U8BIT start_path, U8BIT tuner_num);
U8BIT STB_DPPathForAudioDecoder(U8BIT decoder_num);
U8BIT STB_DPPathForADDecoder(U8BIT decoder_num);
U8BIT STB_DPPathForVideoDecoder(U8BIT decoder_num);

void STB_DPSetTunedTransport(U8BIT path, void *transport);
void* STB_DPGetTunedTransport(U8BIT path);
void STB_DPSetTunedService(U8BIT path, void *service);
void* STB_DPGetTunedService(U8BIT path);

U8BIT STB_DPGetPathForService(void *service);

/**
 * @brief   Checks whether there's a tuner available to tune to the given service or transport
 * @param   service service to be tuned to, can be NULL
 * @param   transport transport to be tuned to
 * @return  TRUE if there's a tuner available
 */
BOOLEAN STB_DPCanTuneTo(E_STB_DP_SIGNAL_TYPE tuner_type, void *service, void *transport);

U8BIT STB_DPGetPathTuner(U8BIT path);
E_STB_DP_SIGNAL_TYPE STB_DPGetPathTunerType(U8BIT path);

/**
 * @brief   Sets the 'disabled' state of all tuners to FALSE so they can all be used
 *          by the resource manager
 */
void STB_DPEnableAllTuners(void);

/**
 * @brief   Disables/enables the tuner associated with this decode path
 * @param   path decode path
 * @param   disabled TRUE if the tuner is to be disabled, FALSE otherwise
 */
void STB_DPSetTunerDisabled(U8BIT path, BOOLEAN disabled);

/**
 * @brief   Returns the number of enabled tuners (i.e. that haven't been disabled)
 * @return  number of enabled tuners
 */
U8BIT STB_DPGetNumEnabledTuners(void);

/**
 * @brief   Returns the 'enabled' status for the given tuner number
 * @param   tuner_num number of the tuner to be checked
 * @return  TRUE if the tuner number is valid and the tuner is enabled, FALSE otherwise
 */
BOOLEAN STB_DPIsTunerEnabled(U8BIT tuner_num);

U8BIT STB_DPGetPathDemux(U8BIT path);
U8BIT STB_DPGetPathAudioDecoder(U8BIT path);
U8BIT STB_DPGetPathVideoDecoder(U8BIT path);

U8BIT STB_DPGetMHEGPath(void);

void STB_DPSetOwner(U8BIT path, E_STB_DP_RES_OWNER owner);
BOOLEAN STB_DPSetOwnerData(U8BIT path, void *data, U32BIT data_size);
BOOLEAN STB_DPIsOwnedBy(U8BIT path, E_STB_DP_RES_OWNER owner);
void* STB_DPGetOwnerData(U8BIT path, U32BIT *data_size);

void STB_DPSetDecodeSource(U8BIT path, E_STB_DP_DECODE_SOURCE source, U32BIT param);
void STB_DPGetDecodeSource(U8BIT path, E_STB_DP_DECODE_SOURCE *source, U32BIT *param);

BOOLEAN STB_DPStartRecording(U8BIT path, U32BIT param);
void STB_DPStopRecording(U8BIT path);
BOOLEAN STB_DPIsRecording(U8BIT path, U32BIT *handle);

void STB_DPStartTune(U8BIT path);
void STB_DPStartScan(U8BIT path);
void STB_DPStopTune(U8BIT path);
void STB_DPTuneOff(U8BIT path);

void STB_DPStartVideoDecoding(U8BIT path);
void STB_DPStartADDecoding(U8BIT path);
void STB_DPStartAudioDecoding(U8BIT path);
void STB_DPStartDecoding(U8BIT path);
void STB_DPStopVideoDecoding(U8BIT path);
void STB_DPStopADDecoding(U8BIT path);
void STB_DPStopAudioDecoding(U8BIT path);
void STB_DPStopDecoding(U8BIT path);

void STB_DPStartSI(U8BIT path);
void STB_DPStopSI(U8BIT path);

void STB_DPRequestSIExtendedEvent(U8BIT path, U32BIT start_date, U32BIT start_hour, U32BIT start_min);
U32BIT STB_DPGetSIRequestParam1(U8BIT path);
U32BIT STB_DPGetSIRequestParam2(U8BIT path);
U32BIT STB_DPGetSIRequestParam3(U8BIT path);

void STB_DPSetTuneStatus(U8BIT path, E_STB_DP_TUNE_STATUS state);
E_STB_DP_TUNE_STATUS STB_DPGetTuneStatus(U8BIT path);
void STB_DPSetADEnabled(U8BIT path, BOOLEAN state);
void STB_DPSetADAudio(U8BIT path, E_STB_DP_AD_AUDIO state);
E_STB_DP_AD_AUDIO STB_DPGetADAudio(U8BIT path);
void STB_DPSetADStatus(U8BIT path, E_STB_DP_DECODE_STATUS state);
E_STB_DP_DECODE_STATUS STB_DPGetADStatus(U8BIT path);

void STB_DPSetAudioStatus(U8BIT path, E_STB_DP_DECODE_STATUS state);
E_STB_DP_DECODE_STATUS STB_DPGetAudioStatus(U8BIT path);

void STB_DPSetVideoStatus(U8BIT path, E_STB_DP_DECODE_STATUS decoder);
E_STB_DP_DECODE_STATUS STB_DPGetVideoStatus(U8BIT path);

void STB_DPSetSignalType(U8BIT path, E_STB_DP_SIGNAL_TYPE sigtype);
E_STB_DP_SIGNAL_TYPE STB_DPGetSignalType(U8BIT path);

void STB_DPSetTuneRelock(U8BIT path, BOOLEAN state);
BOOLEAN STB_DPGetTuneRelock(U8BIT path);

void STB_DPSetLockEnable(U8BIT path, BOOLEAN state);
BOOLEAN STB_DPGetLockEnable(U8BIT path);

void STB_DPSetSearchMode(U8BIT path, BOOLEAN state);
BOOLEAN STB_DPGetSearchMode(U8BIT path);

void STB_DPSetTVSearch(U8BIT path, BOOLEAN state);
BOOLEAN STB_DPGetTVSearch(U8BIT path);

void STB_DPSetRadioSearch(U8BIT path, BOOLEAN state);
BOOLEAN STB_DPGetRadioSearch(U8BIT path);

void STB_DPSetFTASearch(U8BIT path, BOOLEAN state);
BOOLEAN STB_DPGetFTASearch(U8BIT path);

void STB_DPSetScramSearch(U8BIT path, BOOLEAN state);
BOOLEAN STB_DPGetScramSearch(U8BIT path);

void STB_DPSetNetworkSearch(U8BIT path, BOOLEAN state);
BOOLEAN STB_DPGetNetworkSearch(U8BIT path);

void STB_DPSetLNBPower(U8BIT path, E_STB_DP_LNB_POWER state);
E_STB_DP_LNB_POWER STB_DPGetLNBPower(U8BIT path);

void STB_DPSetLNBType(U8BIT path, E_STB_DP_LNB_TYPE type);
E_STB_DP_LNB_TYPE STB_DPGetLNBType(U8BIT path);

/**
 * @brief   Stores the pointer to the array of band definitions for the user defined LNB
 * @param   path ID of the decode path to use
 * @param   number_of_bands Number of bands in the array
 * @param   band_definitions Array containing the definition of the behaviour of user defined LNB
 *          for for each band. A band is defined as the combination of a frequency range and a
 *          polarity. It is responsibility of the caller to ensure that the pointer remains valid
 *          until this function is called passing a different one.
 */
void STB_DPSetUserDefinedLNBBands(U8BIT path, U8BIT number_of_bands,
   S_STB_DP_LNB_BAND *band_definitions);

void STB_DPSetLNB22k(U8BIT path, BOOLEAN state);
BOOLEAN STB_DPGetLNB22k(U8BIT path);

void STB_DPSetLNB12v(U8BIT path, BOOLEAN state);
BOOLEAN STB_DPGetLNB12v(U8BIT path);

void STB_DPSetPulsePosition(U8BIT path, BOOLEAN state);
BOOLEAN STB_DPGetPulsePosition(U8BIT path);

void STB_DPSetDISEQCPosition(U8BIT path, BOOLEAN state);
BOOLEAN STB_DPGetDISEQCPosition(U8BIT path);

void STB_DPSetDISEQCCSwitch(U8BIT path, E_STB_DP_DISEQC_CSWITCH state);
E_STB_DP_DISEQC_CSWITCH STB_DPGetDISEQCCSwitch(U8BIT path);

void STB_DPSetDISEQCUSwitch(U8BIT path, U8BIT state);
U8BIT STB_DPGetDISEQCUSwitch(U8BIT path);

void STB_DPSetDISEQCTone(U8BIT path, E_STB_DP_DISEQC_TONE state);
E_STB_DP_DISEQC_TONE STB_DPGetDISEQCTone(U8BIT path);

void STB_DPSetDISEQCSMATV(U8BIT path, BOOLEAN state);
BOOLEAN STB_DPGetDISEQCSMATV(U8BIT path);

void STB_DPSetDISEQCRepeats(U8BIT path, U8BIT count);
U8BIT STB_DPGetDISEQCRepeats(U8BIT path);

U8BIT STB_DPGetUnicableParams(U8BIT path, U32BIT unicable_if[MAX_UNICABLE_BANKS], U32BIT * lnb_lo_freq, U32BIT * lnb_hi_freq);
void STB_DPSetUnicablePositionB(U8BIT path, BOOLEAN position_b);
BOOLEAN STB_DPGetUnicablePositionB(U8BIT path);
void STB_DPSetUnicableChannel(U8BIT path, U8BIT chan);
U8BIT STB_DPGetUnicableChannel(U8BIT path);
void STB_DPSetUnicableFrequency(U8BIT path, U32BIT freq);
U32BIT STB_DPGetUnicableFrequency(U8BIT path);

void STB_DPSetDishLimitE(U8BIT path);
void STB_DPSetDishLimitW(U8BIT path);
void STB_DPEnableDishLimits(U8BIT path, U16BIT ecount, U16BIT wcount);
void STB_DPDisableDishLimits(U8BIT path);

void STB_DPStartDishMoveE(U8BIT path, U16BIT count);
void STB_DPStartDishMoveW(U8BIT path, U16BIT count);
void STB_DPStopDishMove(U8BIT path);
void STB_DPCentreDishMove(U8BIT path);
void STB_DPStoreDishPosition(U8BIT path, U16BIT count);

void STB_DPSetDishPosition(U8BIT path, U16BIT count);
U16BIT STB_DPGetDishPosition(U8BIT path);
U16BIT STB_DPGetDishRequest(U8BIT path);

void STB_DPSetSkewPosition(U8BIT path, U16BIT count);
U16BIT STB_DPGetSkewPosition(U8BIT path);

void STB_DPSetFrequency(U8BIT path, U32BIT freq);
U32BIT STB_DPGetFrequency(U8BIT path);

void STB_DPSetPolarity(U8BIT path, E_STB_DP_POLARITY pol);
E_STB_DP_POLARITY STB_DPGetPolarity(U8BIT path);

void STB_DPSetSymbolRate(U8BIT path, U16BIT sym);
U16BIT STB_DPGetSymbolRate(U8BIT path);

void STB_DPSetFEC(U8BIT path, E_STB_DP_FEC fec);
E_STB_DP_FEC STB_DPGetFEC(U8BIT path);

void STB_DPSetDVBS2(U8BIT path, BOOLEAN dvb_s2);
BOOLEAN STB_DPGetDVBS2(U8BIT path);

void STB_DPSetModulation(U8BIT path, E_STB_DP_MODULATION modulation);
E_STB_DP_MODULATION STB_DPGetModulation(U8BIT path);

void STB_DPSetTerrMode(U8BIT path, E_STB_DP_TMODE mode);
E_STB_DP_TMODE STB_DPGetTerrMode(U8BIT path);

void STB_DPSetTerrBandwidth(U8BIT path, E_STB_DP_TBWIDTH bwidth);
E_STB_DP_TBWIDTH STB_DPGetTerrBandwidth(U8BIT path);

void STB_DPSetTerrFreqOff(U8BIT path, S8BIT offset);
S8BIT STB_DPGetTerrFreqOff(U8BIT path);

void STB_DPSetTerrType(U8BIT path, E_STB_DP_TTYPE type);
E_STB_DP_TTYPE STB_DPGetTerrType(U8BIT path);

void STB_DPSetTerrPLP(U8BIT path, U8BIT plp_id);
U8BIT STB_DPGetTerrPLP(U8BIT path);

void STB_DPSetCableMode(U8BIT path, E_STB_DP_CMODE mode);
E_STB_DP_CMODE STB_DPGetCableMode(U8BIT path);

void STB_DPSetAnalogVideoType(U8BIT path, E_STB_DP_ANALOG_VIDEO_TYPE vtype);
E_STB_DP_ANALOG_VIDEO_TYPE STB_DPGetAnalogVideoType(U8BIT path);

void STB_DPSetAnalogFreqOff(U8BIT path, S8BIT offset);
S8BIT STB_DPGetAnalogFreqOff(U8BIT path);

/**
 * @brief   Sets an array of frequencies that can be tried when tuning to a transport if the tuner
 *          fails to lock using the primary frequency
 * @param   path decode path that will be used to tune
 * @param   num_freqs number of additional frequencies in the array
 * @param   frequencies array of additional frequencies; this array will be copied
 */
void STB_DPSetAdditionalFrequencies(U8BIT path, U8BIT num_freqs, U32BIT *frequencies);

/**
 * @brief   Checks to see if there's an additional frequency available on the given decode path
 *          that hasn't yet been tried and sets it as the next frequency for tuning
 * @param   path decode path being used for tuning
 * @return  TRUE if a new frequency is set for tuning, FALSE otherwise
 */
BOOLEAN STB_DPTryAdditionalFrequency(U8BIT path);

void STB_DPSetPCRPID(U8BIT path, U16BIT pid);
U16BIT STB_DPGetPCRPID(U8BIT path);

void STB_DPSetVideoPID(U8BIT path, U16BIT pid);
U16BIT STB_DPGetVideoPID(U8BIT path);

void STB_DPSetAudioPID(U8BIT path, U16BIT pid);
U16BIT STB_DPGetAudioPID(U8BIT path);

void STB_DPSetADPID(U8BIT path, U16BIT pid);
U16BIT STB_DPGetADPID(U8BIT path);
void STB_DPSetTextPID(U8BIT path, U16BIT pid);
U16BIT STB_DPGetTextPID(U8BIT path);

void STB_DPSetDataPID(U8BIT path, U16BIT pid);
U16BIT STB_DPGetDataPID(U8BIT path);

void STB_DPSetDecodePIDs(U8BIT path, U16BIT pcr_pid, U16BIT video_pid, U16BIT audio_pid,
   U16BIT text_pid, U16BIT data_pid
   , U16BIT AD_pid
   );

void STB_DPSetEcmPIDs(U8BIT path, U16BIT ecm_pid, U16BIT video_ecm_pid, U16BIT audio_ecm_pid,
   U16BIT text_ecm_pid, U16BIT data_ecm_pid, U16BIT AD_ecm_pid);

void STB_DPSetADMode(U8BIT path, E_STB_DP_AUDIO_MODE mode);
E_STB_DP_AUDIO_MODE STB_DPGetADMode(U8BIT path);

void STB_DPSetAudioMode(U8BIT path, E_STB_DP_AUDIO_MODE mode);
E_STB_DP_AUDIO_MODE STB_DPGetAudioMode(U8BIT path);

void STB_DPSetLockMode(U8BIT path, BOOLEAN mode);
BOOLEAN STB_DPGetLockMode(U8BIT path);

void STB_DPSetOTASearchMode(U8BIT path, E_STB_OTA_SW_UPGRADE_SEARCH_MODE mode);
E_STB_OTA_SW_UPGRADE_SEARCH_MODE STB_DPGetOTASearchMode(U8BIT path);
BOOLEAN STB_DPOTASearchEnabled(U8BIT path);

BOOLEAN STB_DPSetVideoCodec(U8BIT path, E_STB_DP_VIDEO_CODEC codec);
E_STB_DP_VIDEO_CODEC STB_DPGetVideoCodec(U8BIT path);
BOOLEAN STB_DPSetAudioCodec(U8BIT path, E_STB_DP_AUDIO_CODEC codec);
E_STB_DP_AUDIO_CODEC STB_DPGetAudioCodec(U8BIT path);
BOOLEAN STB_DPSetADCodec(U8BIT path, E_STB_DP_AUDIO_CODEC codec);
E_STB_DP_AUDIO_CODEC STB_DPGetADCodec(U8BIT path);

U8BIT STB_DPGetPathSecondaryVideoDecoder(U8BIT path);

U8BIT STB_DPGetPathOfDemux(U8BIT demux_num);
U8BIT STB_DPGetPathOfVideoDecoder(U8BIT decoder_num);
U8BIT STB_DPGetPathOfAudioDecoder(U8BIT decoder_num);
U8BIT STB_DPGetPathOfDescrambler(U32BIT handle);
void STB_DPSwitchLivePath(U8BIT fromPath, U8BIT toPath);

#endif //  _STBDPC_H

//*****************************************************************************
// End of file
//*****************************************************************************

