/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Header file - EBU Teletext driver
 *
 * @file    stbebutt.h
 * @date    04/02/2004
 * @author  Ocean Blue
 */

// pre-processor mechanism so multiple inclusions don't cause compilation error

#ifndef _STBEBUTT_H
#define _STBEBUTT_H

#include "techtype.h"

//---Constant and macro definitions for public use-----------------------------

//---Enumerations for public use-----------------------------------------------

typedef enum e_ebutt_event
{
   // These are the four FastText shortcuts, usually represented by red, green,
   // yellow and blue keys on the handset.
   EBUTT_EVENT_QUICK_NAVIGATE_1,
   EBUTT_EVENT_QUICK_NAVIGATE_2,
   EBUTT_EVENT_QUICK_NAVIGATE_3,
   EBUTT_EVENT_QUICK_NAVIGATE_4,
   // The ten numeric keys used to input page indexes.
   EBUTT_EVENT_0,
   EBUTT_EVENT_1,
   EBUTT_EVENT_2,
   EBUTT_EVENT_3,
   EBUTT_EVENT_4,
   EBUTT_EVENT_5,
   EBUTT_EVENT_6,
   EBUTT_EVENT_7,
   EBUTT_EVENT_8,
   EBUTT_EVENT_9,
   // This is the home key, which returns to the nominated index page for this
   //   service.
   EBUTT_EVENT_INDEXPAGE,
   // These are used to quickly increment/decrement the page index.
   EBUTT_EVENT_NEXTPAGE,
   EBUTT_EVENT_PREVIOUSPAGE,
   // These are used to navigate the sub-pages when in 'hold' mode.
   EBUTT_EVENT_NEXTSUBPAGE,
   EBUTT_EVENT_PREVIOUSSUBPAGE,
   // These are used to traverse the page history (if caching requested).
   EBUTT_EVENT_BACKPAGE,
   EBUTT_EVENT_FORWARDPAGE,
   // This is used to toggle hold on the current page.
   EBUTT_EVENT_HOLD,
   // Reveal hidden page content (as defined in EBU specification)
   EBUTT_EVENT_REVEAL,
   // This key toggles 'clear' mode (page hidden until updated)
   EBUTT_EVENT_CLEAR,
   // Used to toggle transparent background ('video mix' mode)
   EBUTT_EVENT_MIX_VIDEO,
   // Used to toggle double height top / double-height bottom / normal height display.
   EBUTT_EVENT_DOUBLE_HEIGHT,
   // Functional enhancement may offer finer scrolling of double-height display.
   EBUTT_EVENT_DOUBLE_SCROLL_UP,
   EBUTT_EVENT_DOUBLE_SCROLL_DOWN,
   // Used to initiate/cancel 'timer' mode (clear and re-display page at set time)
   EBUTT_EVENT_TIMER
}
E_EBUTT_EVENT;


typedef enum e_ebutt_caching_method
{
   // No caching is performed - only presently displayed page and optionally next requested
   // page information is held in the system.
   EBUTT_CACHING_METHOD_NONE = 0,

   // An nominal number of pages in the carousel numbering sequence that are close to the currently
   // displayed page are cached.
   // The minimal amount of memory allocation is required for this simple caching.
   EBUTT_CACHING_METHOD_PREVIOUS_NEXT,

   // As for EBUTT_CACHING_METHOD_PREVIOUS_NEXT, but also...
   // Previously visited pages are cached, up to a nominal limit.
   // The allows for easy travelral of a chain of recently visited pages.
   // Some memory allocation is required for this caching.
   EBUTT_CACHING_METHOD_HISTORY,

   // As for EBUTT_CACHING_METHOD_HISTORY, but also...
   // The FLOF (or FastText) page navigation links are cached for the
   // currently displayed page.
   // More complex memory allocation is required for this caching.
   EBUTT_CACHING_METHOD_NAVIGATION,

   // As for EBUTT_CACHING_METHOD_NAVIGATION, but also...
   // The pages defined in a tree of navigation links to the currently
   // displayed page are cached.
   // Complex memory allocation is required for this caching.
   EBUTT_CACHING_METHOD_NAVIGATION_TREE,

   // All TeleText carousel content is cached - offers optimal performance.
   // Significant amounts of memory allocation are required for this caching.
   EBUTT_CACHING_METHOD_ALL
}
E_EBUTT_CACHING_METHOD;

typedef enum e_ebutt_character_set_designation
{
   EBUTT_CHARACTER_SET_DESIGNATION_LATIN_DEFAULT = 0,
   EBUTT_CHARACTER_SET_DESIGNATION_LATIN_POLISH = 1,
   EBUTT_CHARACTER_SET_DESIGNATION_LATIN_TURKISH = 2,
   EBUTT_CHARACTER_SET_DESIGNATION_LATIN_SERBIAN_RUMANIAN = 3,
   EBUTT_CHARACTER_SET_DESIGNATION_CYRILLIC = 4,
   EBUTT_CHARACTER_SET_DESIGNATION_GREEK_TURKISH = 6,
   EBUTT_CHARACTER_SET_DESIGNATION_LATIN_ARABIC = 8,
   EBUTT_CHARACTER_SET_DESIGNATION_HEBREW_ARABIC = 10
}
E_EBUTT_CHARACTER_SET_DESIGNATION;

//---Global type defs for public use-------------------------------------------

typedef struct s_ebutt_font
{
   U8BIT character_width;
   U8BIT character_height;

   U32BIT seperator_row_mask;
   U32BIT seperator_column_mask;

   U8BIT *font_table_set_ptr[13];           // each points to 96 * character_width * character_height values
                                            // whereby each value is a pixel representation.

   U8BIT *font_table_option_subset_ptr;     // each points to 169 * character_width * character_height values
                                            // whereby each value is a pixel representation.
}
S_EBUTT_FONT;

/* Pointer to "midware held" ebutt font data */
extern const S_EBUTT_FONT *const ebutt_font_ptr;

//---Global Function prototypes for public use---------------------------------

/**
 *

 *
 * @brief   This must be called to invoke the EBU TeleText driver module before all other
 *                 functionality can be accessed. This initiates the relevant resources used by the
 *                 driver module.
 *                 As a consequence of calling this function, no TeleText processing will occur until
 *                 other functions are called.
 *                 It is envisaged that this function may be called within the general initialisation
 *                 of the STB layer.
 *

 *
 * @return   TRUE if initialisation was successful, FALSE otherwise.
 *
 */
BOOLEAN STB_EBUTT_Initialise(void);

/**
 *

 *
 * @brief   This is the accompanying function to STB_EBUTT_Initialise( ), and is called to
 *                 shut-down the EBU TeleText driver module and release the relevant resources used
 *                 by the driver module.
 *                 This may not be used in some implementations.
 *

 *

 *
 */
void STB_EBUTT_Kill(U8BIT path);

/**
 *

 *
 * @brief   These functions are called to control whether received TeleText data is processed.
 *                 Using this functionality ensures that the process of collating page information can
 *                 be halted when processing resources are a consideration.
 *                 It is envisaged that some implementations may wish only to collate TeleText content
 *                 only when the display is being shown,
 *                 whilst other designs may which to cache significant page content as a background
 *                 process.
 *                 Abstracting this functionality from the initialisation routines gives the option
 *                 to support both solution types.
 *
 * @param   BOOLEAN reset_cache - clears all pages from the cache if set to TRUE, saving
 *                                       memory resources.
 *

 *
 */
void STB_EBUTT_Start(U8BIT path, U16BIT text_pid, U8BIT magazine, U8BIT page);

void STB_EBUTT_Stop(U8BIT path, BOOLEAN reset_cache);

/*!**************************************************************************
 * @brief   Allows teletext PES data packets to be injected by an external module,
 *          which will be decoded and displayed.
 * @param   data_ptr - pointer to first whole PES packet data
 * @param   data_length - number of bytes of data provided
 * @return  TRUE if the data is valid EBU teletext PES data, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_EBUTT_InjectData(U8BIT *data_ptr, U32BIT data_length);

/**
 *

 *
 * @brief   Called to initiate continuous display of TeleText page content.
 *                 This will result in either the index page defined in the service data stream
 *                 being selected, or the previously displayed page (prior to a call to
 *                 STB_EBUTT_Hide( ) being made) from being reinstated.
 *                 This function can be called irrespective of the use of the STB_EBUTT_Start( ) /
 *                 STB_EBUTT_Stop( ) functions, but of course may result in no TeleText data being
 *                 displayed if none already exists within the cache.
 *                 The return value indicates whether initialisation of the display mechanism has been
 *                 successful or not.
 *
 * @param   E_STB_EBUTT_CHARACTER_SET_DESIGNATION character_set_designation -
 *                      used to specifiy the default character set designation for a level 1.0
 *                      Teletext broadcast. This is useful to configure the implementation to work
 *                      correctly when used with regionalised broadcasts.
 * @param BOOLEAN navigate_to_index_page - set it TRUE if the index page defined within the
 *                                                  Teletext service is used, or the last visited page
 *                                                  is re-displayed.
 *
 * @return   TRUE if initialisation of diaply mechanism was successful, FALSE otherwise.
 *
 */
BOOLEAN STB_EBUTT_Show(E_EBUTT_CHARACTER_SET_DESIGNATION character_set_designation,
   BOOLEAN navigate_to_index_page, BOOLEAN show_header);

/**
 *

 *
 * @brief   Called to halt the continuous display of TeleText page content, and to clear the
 *                 relevant area of the application display.
 *

 *

 *
 */
void STB_EBUTT_Hide(void);

/**
 *

 *
 * @brief   Called to define the strategy used to determine what page content is to be
 * displayed form the TeleText carousel content.
 *
 * @param   E_EBUTT_CACHING_METHOD ebutt_caching_method -
 *

 *
 */
void STB_EBUTT_SetCacheMethod(E_EBUTT_CACHING_METHOD ebutt_caching_method);

/**
 *

 *
 * @brief   Called whenever a TeleText-specific event being invoked. This is used to pass
 *                 relevant user input to the driver, and the enumerate type supplied is defined
 *                 independently of any IR handset / keypad.
 *
 * @param   E_EBUTT_EVENT event_type -
 *

 *
 */
void STB_EBUTT_NotifyEvent(E_EBUTT_EVENT event_type);

/**
 *

 *
 * @brief   Called whenever the application permits a TV/Radio service change during a TeleText
 *                 display session. This is provided because an update of the display to new page
 *                 content is required.
 *

 *

 *
 */
void STB_EBUTT_NotifyServiceChange(void);

/**
 *

 *
 * @brief   Called to adjust the display brightness (colour intensity) of the Teletext pages.
 *
 * @param   U8BIT gun_intensity - a RGB gun intensity, ranging from 255 (brightest) to
 *                                       127 (darkest)
 *

 *
 */
void STB_EBUTT_SetDisplayBrightness(U8BIT gun_intensity);

/**
 *

 *
 * @brief   Called to adjust the display brightness (colour intensity) of the Teletext pages.
 *

 *

 *
 */
void STB_EBUTT_IncreaseDisplayBrightness(void);

void STB_EBUTT_DecreaseDisplayBrightness(void);

/**
 *

 *
 * @brief   Called to adjust the display anti-aliasing level of the Teletext pages.
 *                 This is used to control 'flicker' on TV displays.
 *
 * @param   U8BIT antialias_level - a factor between zero (no anti-alaising) to 8 (full
 *                                         anti-aliasing
 *

 *
 */
void STB_EBUTT_SetDisplayAntiAliasing(U8BIT antialias_level);

/**
 *

 *
 * @brief   Called to adjust the display anti-aliasing level of the Teletext pages.
 *

 *

 *
 */
void STB_EBUTT_IncreaseDisplayAntiAliasing(void);
void STB_EBUTT_DecreaseDisplayAntiAliasing(void);

/**
 *

 *
 * @brief   Called to adjust the display video-mix transparency level of the Teletext pages.
 *
 * @param   U8BIT transparency_level - a factor between zero (no transparency) to 255 (full
 *                                         transparency)
 *

 *
 */
void STB_EBUTT_SetDisplayMixTransparency(U8BIT transparency_level);

/**
 *

 *
 * @brief   Called to adjust the display video-mix transparency level of the Teletext pages.
 *

 *

 *
 */
void STB_EBUTT_IncreaseDisplayMixTransparency(void);
void STB_EBUTT_DecreaseDisplayMixTransparency(void);

/**
 *

 *
 * @brief   Called to ascertain whether the present page is in 'hold' mode or not.
 *                 Thus function is useful when allocating handet keys to events on the basis of
 *                 the current functional state of the Teletext driver.
 *

 *
 * @return   TRUE if the displayed page is held, FALSE otherwise.
 *
 */
BOOLEAN STB_EBUTT_IsDisplayHeld(void);

/**
 *

 *
 * @brief   Called to ascertain whether the present page is in 'double height' mode or not.
 *                 Thus function is useful when allocating handet keys to events on the basis of
 *                 the current functional state of the Teletext driver.
 *

 *
 * @return   TRUE if the displayed page is double height, FALSE otherwise.
 *
 */
BOOLEAN STB_EBUTT_IsDisplayDoubleHeight(void);

#endif //  _STBEBUTT_H

//*****************************************************************************
// End of file
//*****************************************************************************
