/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Header file - Function prototypes for Event Reporting
 *
 * @file    stberc.h
 * @date    15/02/2001
 */

// pre-processor mechanism so multiple inclusions don't cause compilation error

#ifndef _STBERC_H

#define _STBERC_H

#include "techtype.h"

//---Constant and macro definitions for public use-----------------------------

// Event classses
// This is an 8-bit number arranges as follows:
//
//     bit    bit
//      7      0
//      xiiiiiii
//
// where:
//  x        is set only if the class of event originates from hardware operated by the user
//           such peripherals have a meaningful ability to be auto-repeated.
//  iiiiiii  is the unique class number, offering up a range from (0x01 to 0x7f inclusive)

#define EV_CLASS_MASK      0x007f
#define EV_CLASS_CAN_REPEAT_FLAG 0x0080
#define EV_CLASS_IS_REPEAT 0x0100

#define HARDWARE_EVENT(id) (id & EV_CLASS_MASK)
#define USER_EVENT(id)     ((id & EV_CLASS_MASK) | EV_CLASS_CAN_REPEAT_FLAG)

/* Event codes used within the DVB stack are 32-bit values derived from the
 * event's class and event type. */
#define EVENT_CODE(class, type)   (((class ) << 16) | (type))

/* Macros to extract each part of the info stored in an event code */
#define EVENT_CLASS(code)        ((code >> 16) & 0xff)
#define EVENT_TYPE(code)         (code & 0xffff)


#define EV_CLASS_HSET                        USER_EVENT(0x01)
#define EV_CLASS_KEYP                        USER_EVENT(0x02)
#define EV_CLASS_TUNE                        HARDWARE_EVENT(0x03)
#define EV_CLASS_DECODE                      HARDWARE_EVENT(0x04)
#define EV_CLASS_SEARCH                      HARDWARE_EVENT(0x05)
#define EV_CLASS_LNB                         HARDWARE_EVENT(0x06)
#define EV_CLASS_DISH                        HARDWARE_EVENT(0x07)
#define EV_CLASS_SKEW                        HARDWARE_EVENT(0x08)
#define EV_CLASS_SCART                       HARDWARE_EVENT(0x09)
#define EV_CLASS_UI                          HARDWARE_EVENT(0x0a)
#define EV_CLASS_OTA_SW_UPGRADE              HARDWARE_EVENT(0x0b)

#define EV_CLASS_MHEG                        HARDWARE_EVENT(0xc)
#define EV_CLASS_MHEG_TUNE_REQUEST           HARDWARE_EVENT(0xd)
#define EV_CLASS_PVR                         HARDWARE_EVENT(0xe)
#define EV_CLASS_TIMER                       HARDWARE_EVENT(0xf)
#define EV_CLASS_APPLICATION                 HARDWARE_EVENT(0x10)
#define EV_CLASS_DVD                         HARDWARE_EVENT(0x12)
#define EV_CLASS_MHEG_DISPLAY_INFO           HARDWARE_EVENT(0x13)

#define EV_CLASS_CI                          HARDWARE_EVENT(0x14)
#define EV_CLASS_DISK                        HARDWARE_EVENT(0x15)
#define EV_CLASS_HDMI                        HARDWARE_EVENT(0x16)
#define EV_CLASS_MHEG_STREAMING_REQUEST      HARDWARE_EVENT(0x17)
#define EV_CLASS_MHEG_ACTION_REQUEST         HARDWARE_EVENT(0x18)
#define EV_CLASS_CA                          HARDWARE_EVENT(0x19)

#define EV_CLASS_PRIVATE                     HARDWARE_EVENT(0xFF)

// Event types
#define EV_TYPE_FAIL                         0x0000
#define EV_TYPE_SUCCESS                      0x0001
#define EV_TYPE_FORCE                        0x0002
#define EV_TYPE_LOCKED                       0x0003
#define EV_TYPE_NOTLOCKED                    0x0004
#define EV_TYPE_STARTED                      0x0005
#define EV_TYPE_STOPPED                      0x0006
#define EV_TYPE_AUDIO_STARTED                0x0007
#define EV_TYPE_AUDIO_STOPPED                0x0008
#define EV_TYPE_VIDEO_STARTED                0x0009
#define EV_TYPE_VIDEO_STOPPED                0x000a
#define EV_TYPE_PAUSED                       0x000b
#define EV_TYPE_16_9                         0x000c
#define EV_TYPE_4_3                          0x000d
#define EV_TYPE_SIGNAL_DATA_BAD              0x000e
#define EV_TYPE_SIGNAL_DATA_OK               0x000f
#define EV_TYPE_DISCONNECTED                 0x0010
#define EV_TYPE_SAMPLE_STARTED               0x0011
#define EV_TYPE_SAMPLE_STOPPED               0x0012
#define EV_TYPE_AD_STARTED                   0x0013
#define EV_TYPE_AD_STOPPED                   0x0014
#define EV_TYPE_CONNECTED                    0x0015
#define EV_TYPE_AUDIO_UNDERFLOW              0x0016
#define EV_TYPE_VIDEO_UNDERFLOW              0x0017
#define EV_TYPE_UPDATE                       0x0018
#define EV_TYPE_FORCED_SERVICE_CHANGE        0x0019
#define EV_TYPE_USER_INTERACTION_TIMEOUT     0x0020

// Over the air s/w upgrade related events
#define EV_TYPE_OTA_SW_UPGRADE_FOUND         0x3000
#define EV_TYPE_OTA_SW_UPGRADE_NOTFOUND      0x3001

#define EV_TYPE_OTA_SW_UPGRADE_DOWNLOADING   0x3002
#define EV_TYPE_OTA_SW_UPGRADE_ERROR         0x3003
#define EV_TYPE_OTA_SW_UPGRADE_BURNING       0x3004
#define EV_TYPE_OTA_SW_UPGRADE_COMPLETED     0x3005

/* Mheg event types */
#define EV_TYPE_START_SUBTITLES              0x4001
#define EV_TYPE_STOP_SUBTITLES               0x4002
#define EV_TYPE_MHEG_TUNE_INDEX              0x4003
#define EV_TYPE_MHEG_TUNE_DSD                0x4004
#define EV_TYPE_MHEG_TUNE_TRIPLET            0x4005

// special terrestrial search event type
#define EV_TYPE_TERR_NETWORK_CHANGED         0x5000

// PVR related events
#define EV_TYPE_PVR_REC_START                0x6000
#define EV_TYPE_PVR_REC_STOP                 0x6001
#define EV_TYPE_PVR_PLAY_START               0x6002
#define EV_TYPE_PVR_PLAY_STOP                0x6003
#define EV_TYPE_PVR_PLAY_BOF                 0x6004
#define EV_TYPE_PVR_PLAY_EOF                 0x6005

//DVD related events
#define EV_TYPE_DVD_DISK_INSERTED            0x7000
#define EV_TYPE_DVD_DISK_REMOVED             0x7001

// MHEG event types
#define EV_TYPE_MHEG_TUNE_NORMALLY           0x8000
#define EV_TYPE_MHEG_TUNE_QUIETLY            0x8001
#define EV_TYPE_MHEG_PROMO_LINK_CHANGE       0x8002

/* CI event types */
#define EV_TYPE_CI_INSERT                    0x9001
#define EV_TYPE_CI_REMOVE                    0x9002
#define EV_TYPE_CI_CAM_UPGRADE_PROGRESS      0x9003
#define EV_TYPE_CI_CAM_UPGRADE_COMPLETE      0x9004
#define EV_TYPE_CI_CAM_UPGRADE_FAILED        0x9005
#define EV_TYPE_CI_SLOT_STATUS_UPDATED       0x9006
#define EV_TYPE_CI_URI_UPDATED               0x9007
#define EV_TYPE_CI_KEYS_UPDATED              0x9008
#define EV_TYPE_CI_OPEN                      0x9009
#define EV_TYPE_CI_CLOSE                     0x900A
#define EV_TYPE_CI_TUNE                      0x900B
#define EV_TYPE_CI_RELEASE_REPLY             0x900C
#define EV_TYPE_CI_REQUEST_OPERATOR_SEARCH   0x900D
#define EV_TYPE_CI_OPERATOR_SEARCH_FINISHED  0x900E
#define EV_TYPE_CI_RECORD_START              0x900F
#define EV_TYPE_CI_RECORD_START_FAILED       0x9010
#define EV_TYPE_CI_RECORD_LICENCE_UPDATED    0x9011
#define EV_TYPE_CI_PLAYBACK_LICENCE_UPDATED  0x9012
#define EV_TYPE_CI_PLAYBACK_LICENCE_STATUS   0x9013
#define EV_TYPE_CI_PIN_STATUS                0x9014
#define EV_TYPE_CI_RECORD_PIN                0x9015
#define EV_TYPE_CI_PLAYBACK_BLANK_VIDEO      0x9016
#define EV_TYPE_CI_PLAYBACK_PIN_STATUS       0x9017
#define EV_TYPE_CI_APP_INFO                  0x9018
#define EV_TYPE_CI_DIAGNOSTIC_SCREEN_SHOW    0x9019
#define EV_TYPE_CI_DIAGNOSTIC_SCREEN_HIDE    0x9020

/* Disk related events */
#define EV_TYPE_DISK_CONNECTED               0xA000
#define EV_TYPE_DISK_REMOVED                 0xA001
#define EV_TYPE_DISK_FULL                    0xA002

/* MHEG streaming event types */
#define EV_TYPE_MHEG_STREAMING_SETUP         0xB001
#define EV_TYPE_MHEG_STREAMING_STOP          0xB002
#define EV_TYPE_MHEG_STREAMING_PAUSE         0xB003
#define EV_TYPE_MHEG_PIN_REQUEST             0xB004

/* MHEG action requests */
#define EV_TYPE_MHEG_REQUEST_VIDEO           0xC001
#define EV_TYPE_MHEG_REQUEST_AUDIO           0xC002
#define EV_TYPE_MHEG_REQUEST_RESTORE_VIDEO   0xc003
#define EV_TYPE_MHEG_REQUEST_RESTORE_AUDIO   0xC004

/* Tuner events */
#define STB_EVENT_TUNE_LOCKED                   EVENT_CODE(EV_CLASS_TUNE, EV_TYPE_LOCKED)
#define STB_EVENT_TUNE_NOTLOCKED                EVENT_CODE(EV_CLASS_TUNE, EV_TYPE_NOTLOCKED)
#define STB_EVENT_TUNE_STOPPED                  EVENT_CODE(EV_CLASS_TUNE, EV_TYPE_STOPPED)
#define STB_EVENT_TUNE_SIGNAL_DATA_BAD          EVENT_CODE(EV_CLASS_TUNE, EV_TYPE_SIGNAL_DATA_BAD)
#define STB_EVENT_TUNE_SIGNAL_DATA_OK           EVENT_CODE(EV_CLASS_TUNE, EV_TYPE_SIGNAL_DATA_OK)

/* SI engine events */
#define STB_EVENT_SEARCH_FAIL                   EVENT_CODE(EV_CLASS_SEARCH, EV_TYPE_FAIL)
#define STB_EVENT_SEARCH_SUCCESS                EVENT_CODE(EV_CLASS_SEARCH, EV_TYPE_SUCCESS)

/* MPEG decoder events */
#define STB_EVENT_AD_DECODE_STARTED             EVENT_CODE(EV_CLASS_DECODE, EV_TYPE_AD_STARTED)
#define STB_EVENT_AD_DECODE_STOPPED             EVENT_CODE(EV_CLASS_DECODE, EV_TYPE_AD_STOPPED)
#define STB_EVENT_AUDIO_DECODE_STARTED          EVENT_CODE(EV_CLASS_DECODE, EV_TYPE_AUDIO_STARTED)
#define STB_EVENT_AUDIO_DECODE_STOPPED          EVENT_CODE(EV_CLASS_DECODE, EV_TYPE_AUDIO_STOPPED)
#define STB_EVENT_AUDIO_DECODE_UNDERFLOW        EVENT_CODE(EV_CLASS_DECODE, EV_TYPE_AUDIO_UNDERFLOW)
#define STB_EVENT_VIDEO_DECODE_STARTED          EVENT_CODE(EV_CLASS_DECODE, EV_TYPE_VIDEO_STARTED)
#define STB_EVENT_VIDEO_DECODE_STOPPED          EVENT_CODE(EV_CLASS_DECODE, EV_TYPE_VIDEO_STOPPED)
#define STB_EVENT_VIDEO_DECODE_UNDERFLOW        EVENT_CODE(EV_CLASS_DECODE, EV_TYPE_VIDEO_UNDERFLOW)
#define STB_EVENT_SAMPLE_DECODE_STARTED         EVENT_CODE(EV_CLASS_DECODE, EV_TYPE_SAMPLE_STARTED)
#define STB_EVENT_SAMPLE_DECODE_STOPPED         EVENT_CODE(EV_CLASS_DECODE, EV_TYPE_SAMPLE_STOPPED)
#define STB_EVENT_DECODE_LOCKED                 EVENT_CODE(EV_CLASS_DECODE, EV_TYPE_LOCKED)

#define STB_EVENT_FORCED_SERVICE_CHANGE         EVENT_CODE(EV_CLASS_APPLICATION, EV_TYPE_FORCED_SERVICE_CHANGE)

/* Timer events */
#define STB_EVENT_TIMER_NOTIFY                  EVENT_CODE(EV_CLASS_TIMER, EV_TYPE_FAIL)
#define STB_EVENT_TIMER_EXPIRE                  EVENT_CODE(EV_CLASS_TIMER, EV_TYPE_SUCCESS)
#define STB_EVENT_USER_INTERACTION_TIMEOUT      EVENT_CODE(EV_CLASS_TIMER, EV_TYPE_USER_INTERACTION_TIMEOUT)

/* HDMI events */
#define STB_EVENT_HDMI_CONNECTED                EVENT_CODE(EV_CLASS_HDMI, EV_TYPE_CONNECTED)
#define STB_EVENT_HDMI_DISCONNECTED             EVENT_CODE(EV_CLASS_HDMI, EV_TYPE_DISCONNECTED)

/* Over the air software upgrade search events */
#define STB_EVENT_OTA_SW_UPGRADE_FOUND          EVENT_CODE(EV_CLASS_OTA_SW_UPGRADE, EV_TYPE_OTA_SW_UPGRADE_FOUND)
#define STB_EVENT_OTA_SW_UPGRADE_NOTFOUND       EVENT_CODE(EV_CLASS_OTA_SW_UPGRADE, EV_TYPE_OTA_SW_UPGRADE_NOTFOUND)
#define STB_EVENT_OTA_SW_UPGRADE_DOWNLOADING    EVENT_CODE(EV_CLASS_OTA_SW_UPGRADE, EV_TYPE_OTA_SW_UPGRADE_DOWNLOADING)
#define STB_EVENT_OTA_SW_UPGRADE_BURNING        EVENT_CODE(EV_CLASS_OTA_SW_UPGRADE, EV_TYPE_OTA_SW_UPGRADE_BURNING)
#define STB_EVENT_OTA_SW_UPGRADE_RESET_REQD     EVENT_CODE(EV_CLASS_OTA_SW_UPGRADE, EV_TYPE_OTA_SW_UPGRADE_RESET_REQD)
#define STB_EVENT_OTA_SW_UPGRADE_ERROR          EVENT_CODE(EV_CLASS_OTA_SW_UPGRADE, EV_TYPE_OTA_SW_UPGRADE_ERROR)
#define STB_EVENT_OTA_SW_UPGRADE_COMPLETED      EVENT_CODE(EV_CLASS_OTA_SW_UPGRADE, EV_TYPE_OTA_SW_UPGRADE_COMPLETED)

/* CI events */
#define STB_EVENT_CI_OPEN_MODULE                EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_OPEN)
#define STB_EVENT_CI_CLOSE_MODULE               EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_CLOSE)
#define STB_EVENT_CI_REMOVE                     EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_REMOVE)
#define STB_EVENT_CI_INSERT                     EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_INSERT)
#define STB_EVENT_CI_CAM_UPGRADE_PROGRESS       EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_CAM_UPGRADE_PROGRESS)
#define STB_EVENT_CI_CAM_UPGRADE_FAILED         EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_CAM_UPGRADE_FAILED)
#define STB_EVENT_CI_CAM_UPGRADE_COMPLETE       EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_CAM_UPGRADE_COMPLETE)
#define STB_EVENT_CI_SLOT_STATUS_CHANGED        EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_SLOT_STATUS_UPDATED)
#define STB_EVENT_CI_URI_CHANGED                EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_URI_UPDATED)
#define STB_EVENT_CI_KEYS_CHANGED               EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_KEYS_UPDATED)
#define STB_EVENT_CI_TUNE_REQUEST               EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_TUNE)
#define STB_EVENT_CI_RELEASE_REPLY              EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_RELEASE_REPLY)
#define STB_EVENT_CI_REQUEST_OPERATOR_SEARCH    EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_REQUEST_OPERATOR_SEARCH)
#define STB_EVENT_CI_OPERATOR_SEARCH_FINISHED   EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_OPERATOR_SEARCH_FINISHED)
#define STB_EVENT_CI_RECORD_START               EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_RECORD_START)
#define STB_EVENT_CI_RECORD_START_FAILED        EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_RECORD_START_FAILED)
#define STB_EVENT_CI_RECORD_LICENCE_CHANGED     EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_RECORD_LICENCE_UPDATED)
#define STB_EVENT_CI_PLAYBACK_LICENCE_CHANGED   EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_PLAYBACK_LICENCE_UPDATED)
#define STB_EVENT_CI_PLAYBACK_LICENCE_STATUS    EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_PLAYBACK_LICENCE_STATUS)
#define STB_EVENT_CI_PIN_STATUS                 EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_PIN_STATUS)
#define STB_EVENT_CI_RECORD_PIN                 EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_RECORD_PIN)
#define STB_EVENT_CI_PLAYBACK_BLANK_VIDEO       EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_PLAYBACK_BLANK_VIDEO)
#define STB_EVENT_CI_PLAYBACK_PIN_STATUS        EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_PLAYBACK_PIN_STATUS)
#define STB_EVENT_CI_APP_INFO                   EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_APP_INFO)
#define STB_EVENT_CI_DIAGNOSTIC_SCREEN_SHOW     EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_DIAGNOSTIC_SCREEN_SHOW)
#define STB_EVENT_CI_DIAGNOSTIC_SCREEN_HIDE     EVENT_CODE(EV_CLASS_CI, EV_TYPE_CI_DIAGNOSTIC_SCREEN_HIDE)

/* PVR events */
#define STB_EVENT_PVR_REC_START                 EVENT_CODE(EV_CLASS_PVR, EV_TYPE_PVR_REC_START)
#define STB_EVENT_PVR_REC_STOP                  EVENT_CODE(EV_CLASS_PVR, EV_TYPE_PVR_REC_STOP)
#define STB_EVENT_PVR_PLAY_START                EVENT_CODE(EV_CLASS_PVR, EV_TYPE_PVR_PLAY_START)
#define STB_EVENT_PVR_PLAY_STOP                 EVENT_CODE(EV_CLASS_PVR, EV_TYPE_PVR_PLAY_STOP)
#define STB_EVENT_PVR_PLAY_BOF                  EVENT_CODE(EV_CLASS_PVR, EV_TYPE_PVR_PLAY_BOF)
#define STB_EVENT_PVR_PLAY_EOF                  EVENT_CODE(EV_CLASS_PVR, EV_TYPE_PVR_PLAY_EOF)

/* Disk events */
#define STB_EVENT_DISK_CONNECTED                EVENT_CODE(EV_CLASS_DISK, EV_TYPE_DISK_CONNECTED)
#define STB_EVENT_DISK_REMOVED                  EVENT_CODE(EV_CLASS_DISK, EV_TYPE_DISK_REMOVED)
#define STB_EVENT_DISK_FULL                     EVENT_CODE(EV_CLASS_DISK, EV_TYPE_DISK_FULL)

/* UI events */
#define UI_EVENT_UPDATE                         EVENT_CODE(EV_CLASS_UI, EV_TYPE_UPDATE)

/* MHEG events */
#define STB_EVENT_MHEG_PIN_REQUEST              EVENT_CODE(EV_CLASS_MHEG, EV_TYPE_MHEG_PIN_REQUEST)
#define STB_EVENT_MHEG_START_SUBTITLES          EVENT_CODE(EV_CLASS_MHEG, EV_TYPE_START_SUBTITLES)
#define STB_EVENT_MHEG_STOP_SUBTITLES           EVENT_CODE(EV_CLASS_MHEG, EV_TYPE_STOP_SUBTITLES)
#define STB_EVENT_MHEG_LIFE_CYCLE               EVENT_CODE(EV_CLASS_MHEG, EV_TYPE_MHEG_LIFE_CYCLE)
#define STB_EVENT_MHEG_TUNE_INDEX               EVENT_CODE(EV_CLASS_MHEG, EV_TYPE_MHEG_TUNE_INDEX)
#define STB_EVENT_MHEG_TUNE_DSD                 EVENT_CODE(EV_CLASS_MHEG, EV_TYPE_MHEG_TUNE_DSD)
#define STB_EVENT_MHEG_TUNE_TRIPLET             EVENT_CODE(EV_CLASS_MHEG, EV_TYPE_MHEG_TUNE_TRIPLET)

#define STB_EVENT_REPEAT                        (EV_CLASS_IS_REPEAT << 16)

//---Enumerations for public use-----------------------------------------------

//---Global type defs for public use-------------------------------------------

//---Global Function prototypes for public use---------------------------------

void STB_ERInitialise(void);
void STB_ERRegisterHandler(BOOLEAN (*func)(BOOLEAN latched, BOOLEAN repeat, U16BIT event_class, U16BIT event_type, void *data, U32BIT data_size));
void STB_ERNotifyEvent(U8BIT event_class, U16BIT event_type);
void STB_ERSendEvent(BOOLEAN latched, BOOLEAN repeat, U16BIT path_class, U16BIT type, void *data, U32BIT data_size);

/**
 * @brief   Sets the time, in seconds, after which the STB_EVENT_USER_INTERACTION_TIMEOUT event
 *          will be sent to indicate that there hasn't been any user interaction for the set
 *          period of time.
 * @param   timeout time in seconds after which an event will be sent
 */
void STB_ERSetUserInteractionTimeout(U32BIT timeout);

#endif //  _STBERC_H

//*****************************************************************************
// End of file
//*****************************************************************************

