/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Header file - macros and function prototypes for public use
 *
 * @file    stbgc.h
 * @date    13/10/2000
 */
#ifndef _STBGC_H

#define _STBGC_H

#include "techtype.h"

//---Constant and macro definitions for public use-----------------------------

/* Macros for DHMS formatted date/time - a 32 bit value where components of time
 * are held and extracted with bit-shifting. */
#define D_DSHFT                     17
#define D_HSHFT                     12
#define D_MSHFT                     6

/* DHMS Creation macros */
#if defined(DEBUG_ASSERT) || defined(DEBUG_ASSERT_STANDARD)
 #define DHMS_CREATE(dt, hr, mn, sc)   STB_GCCreateDebugDHMS(dt, hr, mn, sc)
#else
 #define DHMS_CREATE(dy, hr, mn, sc)   ((U32DHMS)dy << D_DSHFT | (U32DHMS)hr << D_HSHFT | (U32DHMS)mn << D_MSHFT | (sc))
#endif

/* DHMS extraction macros */
#define DHMS_DAYS(dhms)              (dhms >> D_DSHFT)
#define DHMS_DATE32(dhms)           ((dhms >> D_DSHFT) | 0x8000)
#define DHMS_HOUR32(dhms)           ((dhms >> D_HSHFT) & 0x1f)
#define DHMS_MINS32(dhms)           ((dhms >> D_MSHFT) & 0x3f)
#define DHMS_SECS32(dhms)           (dhms & 0x3f)
#define DHMS_DATE(dhms)             (U16BIT)DHMS_DATE32(dhms)
#define DHMS_HOUR(dhms)             (U8BIT) DHMS_HOUR32(dhms)
#define DHMS_MINS(dhms)             (U8BIT) DHMS_MINS32(dhms)
#define DHMS_SECS(dhms)             (U8BIT) DHMS_SECS32(dhms)

//---Enumerations for public use-----------------------------------------------

typedef enum
{
   WEEKDAY_MONDAY = 1,
   WEEKDAY_TUESDAY = 2,
   WEEKDAY_WEDNESDAY = 3,
   WEEKDAY_THURSDAY = 4,
   WEEKDAY_FRIDAY = 5,
   WEEKDAY_SATURDAY = 6,
   WEEKDAY_SUNDAY = 7
} E_STB_GC_WEEKDAY;

typedef enum
{
   CALC_ADD = 0,
   CALC_SUB = 1
} E_STB_GC_CALCTYPE;

typedef enum
{
   CONV_LOCAL = 0,
   CONV_GMT = 1
} E_STB_GC_CONVTYPE;

/* CQ99999:PRM:06/02/28 - CHANGED COMP_EQUAL to COMP_MATCH multiple Definition */
typedef enum
{
   COMP_MATCH = 0,
   COMP_1GT2 = 1,
   COMP_1GE2 = 2,
   COMP_1LT2 = 3,
   COMP_1LE2 = 4,
   COMP_2GT1 = 5,
   COMP_2GE1 = 6,
   COMP_2LT1 = 7,
   COMP_2LE1 = 8
} E_STB_GC_COMPTYPE;

typedef enum
{
   TIME_12H = 0,
   TIME_24H = 1
} E_STB_GC_TIMETYPE;

typedef enum
{
   DATE_DMY = 0,
   DATE_YMD = 1
} E_STB_GC_DATETYPE;

typedef enum
{
   CLOCK_HMS = 0,
   CLOCK_SMH = 1,
   CLOCK_HM = 2,
   CLOCK_MH = 3
} E_STB_GC_CLOCKTYPE;

//---Global type defs for public use-------------------------------------------

typedef U32BIT U32DHMS;

//---Global Function prototypes for public use---------------------------------

void STB_GCInitialise(void);

void STB_GCSetSearchLangCode(U32BIT lang);
U32BIT STB_GCGetSearchLangCode(void);
U8BIT* STB_GCGetLangCodeString(U32BIT lang);

void STB_GCSetCIStandby(BOOLEAN standby);

U8BIT* STB_GCGetFullSerialString(void);
U8BIT* STB_GCGetVersionNumberString(void);

void STB_GCSetAudioSignal(BOOLEAN state);
BOOLEAN STB_GCGetAudioSignal(void);

void STB_GCSetLocalTimeOffset(U8BIT ohour, U8BIT omin, BOOLEAN neg);
void STB_GCGetLocalTimeOffset(U8BIT *ohour, U8BIT *omin, BOOLEAN *neg);
void STB_GCSetLocalTimeChange(U16BIT code, U8BIT hour, U8BIT min, U8BIT secs,
   U8BIT ohour1, U8BIT omin1, U8BIT ohour2, U8BIT omin2, BOOLEAN neg);
void STB_GCGetLocalTimeChange(U16BIT code, U8BIT hour, U8BIT min, U8BIT secs,
   U8BIT *ohour, U8BIT *omin, BOOLEAN *neg);

void STB_GCSetGMTTime(U8BIT hour, U8BIT min, U8BIT secs);
U8BIT STB_GCGetGMTHour(void);
U8BIT STB_GCGetGMTMin(void);
U8BIT STB_GCGetGMTSecs(void);

void STB_GCSetGMTDate(U16BIT code);
U16BIT STB_GCGetGMTDate(void);
E_STB_GC_WEEKDAY STB_GCGetGMTWeekDay(void);
U8BIT STB_GCGetGMTDay(void);
U8BIT STB_GCGetGMTMonth(void);
U16BIT STB_GCGetGMTYear(void);

void STB_GCGetGMTDateTime(U16BIT *code, U8BIT *hour, U8BIT *min, U8BIT *secs);
BOOLEAN STB_GCIsFutureDateTime(U16BIT code, U8BIT hour, U8BIT min, U8BIT secs);

E_STB_GC_WEEKDAY STB_GCGetDateWeekDay(U16BIT code);
BOOLEAN STB_GCIsDateDayWeek(U16BIT code);
BOOLEAN STB_GCIsDateDayWeekend(U16BIT code);

void STB_GCCalculateDateTime(U16BIT code, U8BIT hour, U8BIT min, U8BIT secs,
   U8BIT ohour, U8BIT omin, U8BIT osecs,
   U16BIT *rcode, U8BIT *rhour, U8BIT *rmin, U8BIT *rsecs,
   E_STB_GC_CALCTYPE calc);
void STB_GCConvertDateTime(U16BIT code, U8BIT hour, U8BIT min, U8BIT secs,
   U16BIT *rcode, U8BIT *rhour, U8BIT *rmin, U8BIT *rsecs,
   E_STB_GC_CONVTYPE conv);
BOOLEAN STB_GCCompareDateTime(U16BIT code1, U8BIT hour1, U8BIT min1, U8BIT secs1,
   U16BIT code2, U8BIT hour2, U8BIT min2, U8BIT secs2,
   E_STB_GC_COMPTYPE comp);
S32BIT STB_GCDateTimeDiff(U16BIT code1, U8BIT hour1, U8BIT min1, U8BIT secs1,
   U16BIT code2, U8BIT hour2, U8BIT min2, U8BIT secs2);

U8BIT* STB_GCGetTimeString(U16BIT code, U8BIT hour, U8BIT min, E_STB_GC_TIMETYPE format);
U8BIT* STB_GCGetDateString(U16BIT code, U8BIT hour, U8BIT min, E_STB_GC_DATETYPE format);
U8BIT* STB_GCGetClockString(U8BIT hour, U8BIT min, U8BIT secs, E_STB_GC_CLOCKTYPE format);

void STB_GCGetDateInfo(U16BIT code, U8BIT hour, U8BIT min, U8BIT *day, U8BIT *wday,
   U8BIT *month, U16BIT *year);

void STB_GCGetMJDDateInfo(U16BIT code, U8BIT *day, U8BIT *wday, U8BIT *month, U16BIT *year);

void STB_GCGetLocalDateTime(U16BIT *year, U8BIT *month, U8BIT *day, U8BIT *hour, U8BIT *mins);

/*!**************************************************************************
 * @brief   Sets whether the date/time are taken from the broadcast or the system.
 *          The default is to use the broadcast for date/time.
 * @param   state - FALSE to use date/time from the system
 ****************************************************************************/
void STB_GCUseBroadcastTime(BOOLEAN state);

/*
 * Functions that create/use DHMS formatted date/time
 * Note: 'Compare' functionality is simply done with standard integer compare operators
 */
U32DHMS STB_GCCreateDebugDHMS( U32BIT date, U32BIT hour, U32BIT mins, U32BIT secs );
U32DHMS STB_GCCreateDHMS( U16BIT date, U8BIT hour, U8BIT mins, U8BIT secs );

/**
 * @brief   Creates a DHMS value consisting of hours, minutes and seconds from a number of seconds
 * @param   num_seconds number of seconds to be converted
 * @return  converted value in hours, minutes and seconds as a U32DHMS value
 */
U32DHMS STB_GCCreateDHMSFromSeconds(U32BIT num_seconds);

/**
 * @brief   Calculates the date/time when the period is added/subtracted to/from dhms
 * @param   dhms base date/time
 * @param   period days/hours/mins/secs to be added or subtracted
 * @param   calc calculation to be performed
 *               CALC_ADD adds period to dhms
 *               CALC_SUB subtracts period from dhms
 * @return resulting date/time following the calculation
 */
U32DHMS STB_GCCalculateDHMS(U32DHMS dhms, U32DHMS period, E_STB_GC_CALCTYPE calc);

/**
 * @brief   Converts the given date/time to local or GMT
 * @param   dhms date/time to be converted
 * @param   conv conversion to be performed
 *               CONV_LOCAL = convert to local
 *               CONV_GMT = convert to GMT
 * @return  converted date/time
 */
U32DHMS STB_GCConvertDHMS(U32DHMS dhms, E_STB_GC_CONVTYPE conv);

U32DHMS STB_GCNowDHMSGmt(void);
U32DHMS STB_GCNowDHMSLocal(void);
U8BIT* STB_GCGetDateStringDHMS(U32DHMS dhms, E_STB_GC_DATETYPE format);
U8BIT* STB_GCGetTimeStringDHMS(U32DHMS dhms, E_STB_GC_TIMETYPE format);

/**
 * @brief   Converts a timestamp expressed in number of seconds since midnight (UTC) 1 January 1970
 * @param   timestamp number of seconds since midnight (UTC) 1 January 1970
 * @return  time expressed as U32DHMS
 */
U32DHMS STB_GCConvertTimestamp(U32BIT timestamp);

/**
 * @brief   Returns the number of seconds from midnight (UTC) 1 January 1970 to the specified
 *          U32DHMS time
 * @param   time time in U32DHMS format
 */
U32BIT STB_GCConvertToTimestamp(U32DHMS time);

#endif //  _STBGC_H

