/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Header file - Function prototypes for linked lists
 *
 * @file    stbllist.h
 * @date    06/09/2000
 */
#ifndef _STBLLIST_H

#define _STBLLIST_H

#include "techtype.h"

//---Constant and macro definitions for public use-----------------------------

#define CREATE_LINK_LIST_HEADER(list) static LINK_LIST_HEADER list = {NULL, &list, &list}

//---Enumerations for public use-----------------------------------------------

//---Global type defs for public use-------------------------------------------

typedef struct
{
   void *next;
   void *prev;
} LINK_LIST_PTR_BLK;


typedef struct
{
   void *null_next;  // this MUST be in the same position as the next field in LINK_LIST_PTR_BLK. It
                     // is used to identify the header from an real block
   void *first;
   void *last;
} LINK_LIST_HEADER;

//---Global Function prototypes for public use---------------------------------
// Initialise the header variables of the linked list
// Note: Header space defined (NOT static) but not initlaised.
void STB_LLInitialiseHeader(LINK_LIST_HEADER *hdr);

void STB_LLAddBlockToEnd(LINK_LIST_HEADER *hdr, LINK_LIST_PTR_BLK *new_blk);
void STB_LLAddBlockToStart(LINK_LIST_HEADER *hdr, LINK_LIST_PTR_BLK *new_blk);
void STB_LLAddBlockBefore(LINK_LIST_PTR_BLK *blk, LINK_LIST_PTR_BLK *new_blk);
void STB_LLAddBlockAfter(LINK_LIST_PTR_BLK *blk, LINK_LIST_PTR_BLK *new_blk);

void STB_LLRemoveBlock(LINK_LIST_PTR_BLK *blk);

LINK_LIST_PTR_BLK* STB_LLGetNextBlock(LINK_LIST_PTR_BLK *blk);
LINK_LIST_PTR_BLK* STB_LLGetPrevBlock(LINK_LIST_PTR_BLK *blk);
LINK_LIST_PTR_BLK* STB_LLGetFirstBlock(LINK_LIST_HEADER *hdr);
LINK_LIST_PTR_BLK* STB_LLGetLastBlock(LINK_LIST_HEADER *hdr);
LINK_LIST_PTR_BLK* STB_LLGetBlock(LINK_LIST_HEADER *hdr, U16BIT num);
U16BIT STB_LLGetNumBlocks(LINK_LIST_HEADER *hdr);

BOOLEAN STB_LLCheckBlockInList(LINK_LIST_HEADER *hdr, LINK_LIST_PTR_BLK *blk);

/**

 * @brief   Sorts the blocks of a link list object in ascending or descending order
 *                ----------------------------------------------------------------------
 *                NOTE: The order in which the blocks will be sorted is determined solely
 *                      by the callback compare function
 *                NOTE: callback function must Not alter contents of linked list
 *                NOTE: STB_LLSort is NOT re-entrant -- ie. It cannot be called recursively
 *                ----------------------------------------------------------------------
 * @param   ll_hdr   - pointer to the HEADER of the linked list to be sorted
 *                cmp_func - pointer to the compare function
 *                           ------------------------------------------------
 *                           S16BIT (*cmp_func)(LINK_LIST_PTR_BLK** blk_1, LINK_LIST_PTR_BLK** blk_2)
 *                           ------------------------------------------------
 *                           The callback function must take two LINK_LIST_PTR_BLK pointer to pointer params
 *                           and must return an S16BIT value of:-
 *                              equal to  0 -- if both compare blocks are identical
 *                              less than 0 -- if block_1 has a lower ordinal position than block_2
 *                           greater than 0 -- if block_1 has a higher ordinal position than block_2
 *                           ----------------------------------------------------------------------
 *                           NOTE: It is the order of the ordinal positions that determines whether
 *                                 the linked list is sorted in ascending of descending order
 *                           ----------------------------------------------------------------------
 * @return   TRUE  if successful, FALSE if failed
 */
BOOLEAN STB_LLSort(LINK_LIST_HEADER * ll_hdr, S16BIT (*cmp_func)(LINK_LIST_PTR_BLK **, LINK_LIST_PTR_BLK **));

#endif //  _STBLLIST_H

//*****************************************************************************
// End of file
//*****************************************************************************
