/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Header file - macros and function prototypes for public use
 *
 * @file    stbpvr.h
 * @date    07/02/2003
 */

// pre-processor mechanism so multiple inclusions don't cause compilation error

#ifndef _STBPVR_H
#define _STBPVR_H

#include "techtype.h"
#include "stbpvrpr.h"
#include "stbhwdmx.h"

//---Constant and macro definitions for public use-----------------------------

#define STB_PVR_NAME_LEN            255
#define STB_PVR_BASENAME_LEN        9
#define STB_PVR_SERVICE_LEN         32
#define STB_PVR_ADDITIONAL_INFO_LEN 255

#define STB_PVR_NORMAL_PLAY_SPEED   100
#define STB_PVR_INVALID_HANDLE      0

#define STB_PVR_MAX_CRID_LEN        65

#define STB_PVR_MAX_ADD_INFO_LEN    255

//---Enumerations for public use-----------------------------------------------

typedef enum
{
   FORMAT_START = 0,
   FORMAT_PROGRESS = 1,
   FORMAT_END = 2
} E_STB_PVR_FORMATMODE;

typedef enum
{
   REPAIR_ENQUIRE = 0,
   REPAIR_START = 1,
   REPAIR_PROGRESS = 2,
   REPAIR_END = 3
} E_STB_PVR_REPAIRMODE;

typedef enum
{
   RECORDING_STATUS_STARTED,
   RECORDING_STATUS_FAILED,
   RECORDING_STATUS_COMPLETED
} E_PVR_RECORDING_STATUS;

//---Global type defs for public use-------------------------------------------

//---Global Function prototypes for public use---------------------------------

BOOLEAN STB_PVRInitialise(void);
void STB_PVRSetStandbyState(BOOLEAN state);
BOOLEAN STB_PVRIsInitialised(void);

/**
 * @brief   Sets the disk ID to be used as the default disk for PVR recordings.
 *          If this is set to INVALID_DISK_ID then the first mounted disk will be used.
 * @param   disk_id ID of the disk to use as the default
 */
void STB_PVRSetDefaultDisk(U16BIT disk_id);

/**
 * @brief   Returns the set default disk, or finds the first mounted (usable) disk
 *          if a default hasn't been set
 * @return  ID of the disk, or INVALID_DISK_ID if none found
 */
U16BIT STB_PVRGetDefaultDisk(void);

BOOLEAN STB_PVRCanDiskBeUsed(U16BIT disk_id);

BOOLEAN STB_PVRFormat(U16BIT disk_id, E_STB_PVR_FORMATMODE mode, U8BIT *prog);
BOOLEAN STB_PVRRepair(U16BIT disk_id, E_STB_PVR_REPAIRMODE mode, U8BIT *prog);

void STB_PVRUpdateRecordings(BOOLEAN force_load);

BOOLEAN STB_PVRIsValidHandle(U32BIT handle);

BOOLEAN STB_PVRCreateRecording(U16BIT disk_id, U8BIT *name, U32BIT *handle);
BOOLEAN STB_PVRDestroyRecording(U32BIT handle);

void STB_PVRSaveRecording(U32BIT handle);

U16BIT STB_PVRGetRecordingHandles(U32BIT **handle_array);
void STB_PVRReleaseRecordingHandles(U32BIT *handle_array);

BOOLEAN STB_PVRFindRecordingFromCrid(U8BIT *prog_crid, U32BIT *handle);
BOOLEAN STB_PVRFindNextSplitRecording(U32BIT curr_handle, U32BIT *next_handle);

U32BIT STB_PVRGetHandleForRecordingIndex(U8BIT rec_index);
U8BIT STB_PVRGetPathForRecordingIndex(U8BIT rec_index);

void STB_PVRRecordingSetName(U32BIT handle, U8BIT *name);
U8BIT* STB_PVRRecordingGetName(U32BIT handle);
BOOLEAN STB_PVRRecordingGetDateTime(U32BIT handle, U16BIT *date, U8BIT *hours, U8BIT *mins, U8BIT *secs);
BOOLEAN STB_PVRRecordingGetLength(U32BIT handle, U8BIT *length_hours, U8BIT *length_mins, U8BIT *length_secs, U32BIT *rec_size_kb);
U16BIT STB_PVRRecordingGetDiskId(U32BIT handle);
void STB_PVRRecordingSetSeries(U32BIT handle);
void STB_PVRRecordingSetRecommendation(U32BIT handle);
BOOLEAN STB_PVRRecordingGetRecommendation(U32BIT handle);
BOOLEAN STB_PVRRecordingGetSeries(U32BIT handle);
void STB_PVRRecordingSetOtherCrid(U32BIT handle, U8BIT *crid);
BOOLEAN STB_PVRRecordingGetOtherCrid(U32BIT handle, U8BIT *crid, U16BIT name_len);

/**
 * @brief   Sets the additional info string for a recording
 * @param   handle recording handle
 * @param   additional_info additional info to be set
 */
void STB_PVRRecordingSetAdditionalInfo(U32BIT handle, U8BIT *additional_info);

/**
 * @brief   Gets the additional info string pointer for a recording
 * @param   handle - recording handle
 * @return  Pointer to the additional info
 */
U8BIT* STB_PVRRecordingGetAdditionalInfo(U32BIT handle);

/**
 * @brief   Sets the parental rating age for the specified recording
 * @param   handle recording handle
 * @param   parental_rating parental rating age
 */
void STB_PVRRecordingSetParentalRatingAge(U32BIT handle, U32BIT parental_rating);

/**
 * @brief   Returns the parental rating age for the specified recording as set by
 *          STB_PVRRecordingSetParentalRatingAge
 * @param   handle recording handle
 * @return  Parental rating age
 */
U32BIT STB_PVRRecordingGetParentalRating(U32BIT handle);

/**
 * @brief   Sets the start padding value for the specified recording
 * @param   handle recording handle
 * @param   start_padding start padding value
 */
void STB_PVRRecordingSetStartPadding(U32BIT handle, S32BIT start_padding);

/**
 * @brief   Gets the start padding value for the specified recording as set by
 *          STB_PVRRecordingSetStartPadding
 * @param   handle recording handle
 * @return  Start padding value
 */
S32BIT STB_PVRRecordingGetStartPadding(U32BIT handle);

/**
 * @brief   Sets the end padding value for the specified recording
 * @param   handle recording handle
 * @param   start_padding end padding value
 */
void STB_PVRRecordingSetEndPadding(U32BIT handle, S32BIT end_padding);

/**
 * @brief   Gets the end padding value for the specified recording as set by
 *          STB_PVRRecordingSetEndPadding
 * @param   handle recording handle
 * @return  End padding value
 */
S32BIT STB_PVRRecordingGetEndPadding(U32BIT handle);

void STB_PVRRecordingSetServiceName(U32BIT handle, U8BIT *service_name);
BOOLEAN STB_PVRRecordingGetServiceName(U32BIT handle, U8BIT *service_name, U16BIT name_len);
void STB_PVRRecordingSetDescription(U32BIT handle, U8BIT *description);
BOOLEAN STB_PVRRecordingGetDescription(U32BIT handle, U8BIT *description, U16BIT desc_len);
U16BIT STB_PVRRecordingGetDescriptionLen(U32BIT handle);
void STB_PVRRecordingSetExtendedDescription(U32BIT handle, U8BIT *description);
BOOLEAN STB_PVRRecordingGetExtendedDescription(U32BIT handle, U8BIT *description, U16BIT desc_len);
U16BIT STB_PVRRecordingGetExtendedDescriptionLen(U32BIT handle);
void STB_PVRRecordingSetCrid(U32BIT handle, U8BIT *crid);
BOOLEAN STB_PVRRecordingGetCrid(U32BIT handle, U8BIT *crid, U16BIT name_len);
BOOLEAN STB_PVRRecordingSetLocked(U32BIT handle, BOOLEAN state);
BOOLEAN STB_PVRRecordingGetLocked(U32BIT handle);
BOOLEAN STB_PVRRecordingSetSelected(U32BIT handle, BOOLEAN state);
BOOLEAN STB_PVRRecordingGetSelected(U32BIT handle);
BOOLEAN STB_PVRRecordingIsEncrypted(U32BIT handle);
BOOLEAN STB_PVRRecordingSetParentalLock(U32BIT handle, BOOLEAN state);
BOOLEAN STB_PVRRecordingGetParentalLock(U32BIT handle);
BOOLEAN STB_PVRRecordingSetGuidance(U32BIT handle, U8BIT *text);
BOOLEAN STB_PVRRecordingHasGuidance(U32BIT handle);
BOOLEAN STB_PVRRecordingGetGuidance(U32BIT handle, U8BIT *text, U16BIT text_len);
U16BIT STB_PVRRecordingGetGuidanceLen(U32BIT handle);

/**
 * @brief   Set the status to be saved with the recording
 * @param   handle recording handle
 * @param   status status of the recording
 * @return  TRUE if the status is successfully saved with the given recording, FALSE otherwise
 */
BOOLEAN STB_PVRRecordingSetStatus(U32BIT handle, E_PVR_RECORDING_STATUS status);

/**
 * @brief   Returns the status saved with the given recording
 * @param   handle recording handle
 * @param   status pointer to recording status if read successfully
 * @return  TRUE if status is returned, FALSE otherwise
 */
BOOLEAN STB_PVRRecordingGetStatus(U32BIT handle, E_PVR_RECORDING_STATUS *status);

BOOLEAN STB_PVRRecordingSetCicamId(U32BIT handle, U8BIT *cicam_id, U8BIT id_len);
BOOLEAN STB_PVRRecordingGetCicamId(U32BIT handle, U8BIT *cicam_id, U8BIT id_len);
BOOLEAN STB_PVRRecordingAddURI(U32BIT handle, U8BIT *uri);
BOOLEAN STB_PVRRecordingUpdateURI(U32BIT handle, U8BIT *uri);
BOOLEAN STB_PVRRecordingAddLicence(U32BIT handle, U8BIT *licence, U16BIT licence_len);
BOOLEAN STB_PVRRecordingUpdateLicence(U32BIT handle, U8BIT *licence, U16BIT licence_len);
BOOLEAN STB_PVRRecordingAddPin(U32BIT handle, U8BIT age_rating, U8BIT *private_data,
   U16BIT date_code, U8BIT hour, U8BIT min, U8BIT secs);

BOOLEAN STB_PVRIsBeingRecorded(U32BIT handle);

U32BIT STB_PVRGetTimeOfAllRecordings(U16BIT disk_id);
U32BIT STB_PVRGetSizeOfAllRecordings(U16BIT disk_id);

BOOLEAN STB_PVRToggleBookmark(U8BIT path);
BOOLEAN STB_PVRGotoNextBookmark(U8BIT path);

/**
 * @brief   Allocates and returns an array containing the handles of the bookmarks for
 *          the recording currently being played.
 * @param   path Playback decode path
 * @param   bookmarks Pointer to array of returned bookmark handles
 * @return  The number of handles in the allocated array. If this is 0 then array will be NULL.
 */
U16BIT STB_PVRGetBookmarks(U8BIT path, void ***bookmarks);

/**
 * @brief   Allocates and returns an array containing the handles of the bookmarks for
 *          the recording specified by the given handle
 * @param   handle Recording handle
 * @param   bookmarks Pointer to array of returned bookmark handles
 * @return  The number of handles in the allocated array. If this is 0 then array will be NULL.
 */
U16BIT STB_PVRGetBookmarksForRecording(U32BIT handle, void ***bookmarks);

/**
 * @brief   Frees a previously allocated array of bookmark handles
 * @param   bookmarks Array of bookmark handles to be freed
 * @param   num Number of handles in the array.
 */
void STB_PVRReleaseBookmarks(void **bookmarks, U16BIT num);

void STB_PVRPlaybackNotifyTime(U8BIT path);

BOOLEAN STB_PVRStartPlaying(U8BIT path, U32BIT handle, BOOLEAN resume);
BOOLEAN STB_PVRIsPlaying(U8BIT path, U32BIT *handle);

/**
 * @brief   Saves the bookmark holding the playback position. This bookmark is used to resume a
 *          playback when STB_PVRStartPlaying is called with resume = TRUE. This function should
 *          be called while decoders are still running, to be sure the position in the playback
 *          is accurate.
 * @param   U8BIT path Playback path
 */
void STB_PVRSavePlayPosition(U8BIT path);

/**
 * @brief   Returns the bookmarked play position for the given recording
 * @param   handle handle of recording
 * @return  play position in seconds; will be 0 if the recording hasn't been played yet
 */
U32BIT STB_PVRGetPlayPosition(U32BIT handle);

void STB_PVRStopPlaying(U8BIT path);
BOOLEAN STB_PVRRecordingGetTriplet(U32BIT handle, U16BIT *serv_id, U16BIT *ts_id, U16BIT *orig_net_id);

void STB_PVRStartPlayRunning(U8BIT decoder);
void STB_PVRStartPlayPaused(U8BIT decoder);
void STB_PVRStartPlaySync(U8BIT decoder);
BOOLEAN STB_PVRIsPlayAudio(U8BIT decoder);

BOOLEAN STB_PVRIsPlayVideo(U32BIT handle);

BOOLEAN STB_PVRSetRecordingPids(U8BIT path, U16BIT num_pids, S_PVR_PID_INFO *pid_array);
BOOLEAN STB_PVRUpdateRecordingPids(U8BIT path, U16BIT num_pids, S_PVR_PID_INFO *pid_array);

BOOLEAN STB_PVRSetRecoringTriplet(U8BIT path, U16BIT serv_id, U16BIT ts_id, U16BIT orig_net_id);

BOOLEAN STB_PVRStartRecording(U8BIT path, U32BIT handle);
BOOLEAN STB_PVRPauseRecording(U8BIT path);
BOOLEAN STB_PVRResumeRecording(U8BIT path);
void STB_PVRStopRecording(U8BIT path);
BOOLEAN STB_PVRIsRecording(U8BIT path, U32BIT *handle);

void STB_PVRStartRecordRunning(U8BIT path);
void STB_PVRStartRecordPaused(U8BIT path, U32BIT timeshift_seconds);
BOOLEAN STB_PVRIsRecordVideo(U8BIT path);
BOOLEAN STB_PVRIsRecordAudio(U8BIT path);

void STB_PVREncryptRecording(U8BIT path, BOOLEAN state);

#ifdef COMMON_INTERFACE
BOOLEAN STB_PVRSetDescramblerKey(U8BIT path, E_STB_DMX_DESC_TYPE desc_type,
   E_STB_DMX_DESC_KEY_PARITY parity, U8BIT *key, U8BIT *iv);
#endif

void STB_PVRPlayNormal(U8BIT decoder);
BOOLEAN STB_PVRIsPlayNormal(U8BIT decoder);

void STB_PVRPlayPause(U8BIT decoder);
BOOLEAN STB_PVRIsPlayPause(U8BIT decoder);

void STB_PVRPlayForward(U8BIT decoder);
BOOLEAN STB_PVRIsPlayForward(U8BIT decoder);

void STB_PVRPlayReverse(U8BIT decoder);
BOOLEAN STB_PVRIsPlayReverse(U8BIT decoder);

void STB_PVRPlayFrameInc(U8BIT decoder);
void STB_PVRPlayFrameDec(U8BIT decoder);

void STB_PVRPlayMedium(U8BIT decoder);
BOOLEAN STB_PVRIsPlayMedium(U8BIT decoder);

void STB_PVRPlaySlower(U8BIT decoder, BOOLEAN include_slow_speeds);
BOOLEAN STB_PVRIsPlaySlowest(U8BIT decoder);

void STB_PVRPlayFaster(U8BIT decoder, BOOLEAN include_slow_speeds);
BOOLEAN STB_PVRIsPlayFastest(U8BIT decoder);

S16BIT STB_PVRGetMinPlaySpeed(U8BIT path);
S16BIT STB_PVRGetMaxPlaySpeed(U8BIT path);

void STB_PVRDiskUsed(U16BIT disk_id, U8BIT *hours, U8BIT *mins);
void STB_PVRDiskFree(U16BIT disk_id, U8BIT *hours, U8BIT *mins);
U32BIT STB_PVRDiskSize(U16BIT disk_id);

/**
 * @brief   Creates a bookmark associated with the a recording.
 * @param   handle Recording handle
 * @param   time Time in seconds since the beginning of the recording
 * @param   name NULL terminated string representing the bookmark name, maximum length
 *          STB_PVR_NAME_LEN. If name is NULL, a string is formed to represent the bookmark time in
 *          the format hh:mm:ss
 * @return  TRUE if the bookmark was successfully created, FALSE otherwise.
 */
BOOLEAN STB_PVRCreateBookmark(U32BIT handle, U32BIT time, U8BIT *name);

/**
 * @brief   Deletes a bookmark associated with the a recording.
 * @param   handle Recording handle
 * @param   time Time in seconds since the beginning of the recording
 * @param   name NULL terminated string representing the bookmark name, maximum length
 *          STB_PVR_NAME_LEN. If name is NULL, a string is formed to represent the bookmark time in
 *          the format hh:mm:ss
 * @return  TRUE if the bookmark was successfully deleted, FALSE otherwise.
 */
BOOLEAN STB_PVRDeleteBookmark(U32BIT handle, U32BIT time, U8BIT *name);

/**
 * @brief   Returns the time associated with a bookmark
 * @param   bookmark_handle Bookmark handle
 * @return  Bookmark time
 */
U32BIT STB_PVRGetBookmarkTime(void *bookmark_handle);

/**
 * @brief   Allocates and returns the name associated with a bookmark
 * @param   bookmark_handle Bookmark handle
 * @return  Pointer to the name string. This pointer must freed with STB_FreeMemory
 */
U8BIT *STB_PVRGetBookmarkName(void *bookmark_handle);

#endif //  _STBPVR_H

//*****************************************************************************
// End of file
//*****************************************************************************

