/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Header file - macros and function prototypes for public use
 *
 * @file    stbsiflt.h
 * @date    06/03/2003
 */

// pre-processor mechanism so multiple inclusions don't cause compilation error

#ifndef _STBSIFLT_H
#define _STBSIFLT_H

#include "techtype.h"

//---Constant and macro definitions for public use-----------------------------

// macros for use in calls to STB_SIRequestTable
#define SI_BUFFER_1K                1024
#define SI_BUFFER_2K                2048
#define SI_BUFFER_4K                4096
#define SI_BUFFER_8K                8192
#define SI_BUFFER_16K               16384
#define SI_BUFFER_32K               32768
#define SI_BUFFER_64K               65535
#define SI_XTID_MATCH_DONT_CARE     0x0000
#define SI_XTID_MASK_DONT_CARE      0x0000



//---Enumerations for public use-----------------------------------------------

// SI events passed to the application event handler
typedef enum
{
   STOP_SI,
   START_SI_SEARCHING,
   START_SI_UPDATING_NEW_TRANSPORT,
   START_SI_UPDATING_SAME_TRANSPORT,
   SI_TIMEOUT
} E_APP_SI_EVENT_TYPE;

// enumerations used to define filter requests
typedef enum
{
   ONE_SHOT_REQUEST,
   CONTINUOUS_REQUEST
} E_SI_REQUEST_TYPE;

typedef enum
{
   SINGLE_SECT,      // single section tables have short header up to section length only e.g. TOT
   MULTI_SECT        // multi section table headers include version num, section num and last section e.g. PAT
} E_SI_TABLE_FORMAT_TYPE;



//------------------------
// Table record structures
typedef struct si_section_record
{
   struct si_section_record *next;
   U8BIT sect_num;
   U16BIT data_len;
   U8BIT data_start;
} SI_SECTION_RECORD;

typedef struct si_table_record
{
   U8BIT path;
   U8BIT tid;
   U16BIT xtid;
   U16BIT pid;
   U8BIT version;
   U16BIT num_sect;
   SI_SECTION_RECORD *section_list;
} SI_TABLE_RECORD;

typedef void (*F_AppSiEventHandler)(U8BIT, E_APP_SI_EVENT_TYPE);

//---Global type defs for public use-------------------------------------------

//---Global Function prototypes for public use---------------------------------

void STB_SIReportCurrentPmt(U16BIT service_id, SI_TABLE_RECORD *table_rec, BOOLEAN new_serv,
   BOOLEAN new_pmt_version);

/**
 * @brief   Registers a function to be called to handle SI events.
 *          The currently registered event handler is returned, which allows a module to
 *          override the existing function and reset it
 * @param   func_ptr function being registered to handle SI events
 * @return  previously registered function
 */
F_AppSiEventHandler STB_SIRegisterAppSiEventHandler(F_AppSiEventHandler func_ptr);

void *STB_SIRequestTable(U8BIT path, U16BIT pid, U8BIT tid_match, U8BIT tid_mask,
   E_SI_TABLE_FORMAT_TYPE format, U16BIT expected_tables,
   U16BIT xtid_match, U16BIT xtid_mask, E_SI_REQUEST_TYPE req_type,
   U16BIT buff_size, BOOLEAN crc, BOOLEAN low_priority,
   void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param);

void STB_SIModifyTableRequest(void *filter_handle, U8BIT tid_match, U8BIT tid_mask,
   U16BIT xtid_match, U16BIT xtid_mask, U16BIT expected_tables);

void STB_SIRestartTableRequest(void *filter_handle);

void STB_SICancelTableRequest(void *filter_ptr);

void STB_SIReleaseTableRecord(SI_TABLE_RECORD *table_rec);

void STB_SISearchComplete(U8BIT path, BOOLEAN success, void *event_data, U32BIT data_size);

/**
 * @brief   Reports the CAT has been received so it can be passed on to other modules
 * @param   table_rec - table record containing the CAT section data
 * @return  TRUE if the CAT is passed to the CA module, FALSE otherwise
 */
BOOLEAN STB_SIReportCat(SI_TABLE_RECORD *table_rec);

/**
 * @brief   Reports the BAT has been received so it can be passed on to other modules
 * @param   table_rec - table record containing the BAT section data
 * @return  TRUE if the BAT is passed to the CA module, FALSE otherwise
 */
BOOLEAN STB_SIReportBat(SI_TABLE_RECORD *table_rec);

/**
 * @brief   Reports the NIT has been received so it can be passed on to other modules
 * @param   table_rec - table record containing the NIT section data
 * @return  TRUE if the NIT is passed to the CA module, FALSE otherwise
 */
BOOLEAN STB_SIReportNit(SI_TABLE_RECORD *table_rec);

#endif //  _STBSIFLT_H

//*****************************************************************************
// End of file
//*****************************************************************************

