/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Header file - macros and function prototypes for public use
 *
 * @file    stbsitab.h
 * @date    06/03/2003
 */

// pre-processor mechanism so multiple inclusions don't cause compilation error

#ifndef _STBSITAB_H
#define _STBSITAB_H

#include "techtype.h"
#include "stbsiflt.h"
#include "stbdpc.h"

//---Constant and macro definitions for public use-----------------------------

/* CRID related defines */
#define CRID_LOCATION_MASK          0x03
#define CRID_TYPE_MASK              0xfc
#define CRID_TYPE_SHIFT             2

#define CRID_LOCATION_0             0x00
#define CRID_LOCATION_1             0x01

#define DVB_INVALID_ID              0x1fff
#define DVB_INVALID_CAROUSEL_ID     0xffffffff


//---Enumerations for public use-----------------------------------------------

typedef enum
{
   USER_DEF_DESCRIP_NOT_USED = 0,
   USER_DEF_DESCRIP_LOGICAL_CHAN_NUM = 1,
   USER_DEF_DESCRIP_PREF_NAME_LIST = 2,
   USER_DEF_DESCRIP_PREF_NAME_ID = 3,
   USER_DEF_DESCRIP_SDT_CODE = 4,
   USER_DEF_DESCRIP_SERIES
} STB_SI_USER_DEF_DESCRIP_FUNCTION;

typedef enum
{
   EIT_NOW_NEXT_ACT = 0,
   EIT_NOW_NEXT_OTHER = 1,
   EIT_NOW_NEXT_ALL = 2,
   EIT_PF_PLUS = 3
} E_SI_EIT_TABLE_REQ;

typedef enum
{
   EIT_SCHED_ACT = 0,
   EIT_SCHED_OTHER = 1,
   EIT_SCHED_ACT_4DAY = 2,
   EIT_SCHED_OTHER_4DAY = 3,
   EIT_SCHED_ALL_4DAY = 4,
   EIT_SCHED_ALL_8DAY = 5,
   EIT_SCHED_ALL = 6
} E_SI_SCHED_TABLE_REQ;

typedef enum
{
   CTYPE_DVB_SUBTITLE = 0x310,
   CTYPE_DVB_SUBTITLE_4_3 = 0x311,
   CTYPE_DVB_SUBTITLE_16_9 = 0x312,
   CTYPE_DVB_SUBTITLE_221_1 = 0x313,
   CTYPE_DVB_SUBTITLE_HD = 0x314,

   CTYPE_DVB_SUBTITLE_HH = 0x320,
   CTYPE_DVB_SUBTITLE_HH_4_3 = 0x321,
   CTYPE_DVB_SUBTITLE_HH_16_9 = 0x322,
   CTYPE_DVB_SUBTITLE_HH_221_1 = 0x323,
   CTYPE_DVB_SUBTITLE_HH_HD = 0x324,
} E_SI_COMPONENT_TYPE_SUBTITLE;

typedef enum /* see ETSI 300468 */
{
   /* MPEG1 Audio Description types */
   CTYPE_MPEG1_AD_VISUAL = 0x240,
   CTYPE_MPEG1_AD_HEARING = 0x241,
   CTYPE_MPEG1_AD_RCVR_MIX = 0x247,
   CTYPE_MPEG1_AD_BCAST_MIX = 0x248,

   /* AC3, Full Service Audio description types */
   CTYPE_AC3_FS_VISUAL = 0x450,            /* to 0x457 */
   CTYPE_AC3_FS_HEARING = 0x458,           /* to 0x45f */
   CTYPE_AC3_FS_DIALOGUE = 0x420,          /* to 0x427 */
   CTYPE_AC3_FS_COMMENTARY = 0x468,
   CTYPE_AC3_FS_EMERGENCY = 0x470,

   /* E-AC3, Full Service Audio description types */
   CTYPE_EAC3_FS_VISUAL = 0x4d0,           /* to 0x4d7 */
   CTYPE_EAC3_FS_HEARING = 0x4d8,          /* to 0x4df */
   CTYPE_EAC3_FS_DIALOGUE = 0x4a0,         /* to 0x4a7 */
   CTYPE_EAC3_FS_COMMENTARY = 0x4e8,
   CTYPE_EAC3_FS_EMERGENCY = 0x4f0,

   /* HE-AAC Audio Description types */
   CTYPE_HEAAC_AD_VISUAL = 0x640,
   CTYPE_HEAAC_AD_HEARING = 0x641,
   CTYPE_HEAAC_AD_RCVR_MIX = 0x647,
   CTYPE_HEAAC_AD_BCAST_MIX = 0x648,

   /* HE-AAC v2 Audio Description types */
   CTYPE_HEAACV2_AD_VISUAL = 0x644,
   CTYPE_HEAACV2_AD_HEARING = 0x645,
   CTYPE_HEAACV2_AD_RCVR_MIX = 0x649,
   CTYPE_HEAACV2_AD_BCAST_MIX = 0x64a,

   /* Supplementary Audio */
   CTYPE_MPEG1_SA_RCVR_MIX = 0x242,
   CTYPE_HEAAC_SA_RCVR_MIX = 0x642,
   CTYPE_HEAACV2_SA_RCVR_MIX = 0x646
} E_SI_COMPONENT_TYPE_AUDIO_DESC;

//---Global type defs for public use-------------------------------------------

// common structures used in following tables
typedef struct si_string
{
   U8BIT nbytes;
   U8BIT *str_ptr;
} SI_STRING_DESC;

typedef struct si_linkage_desc
{
   struct si_linkage_desc *next;
   U32BIT private_data_code;
   U16BIT orig_net_id;
   U16BIT tran_id;
   U16BIT serv_id;
   U8BIT link_type;
   U8BIT data_length;
   U8BIT data;
} SI_LINKAGE_DESC_ENTRY;

typedef struct si_guidance_desc
{
   U8BIT guidance_type;
   BOOLEAN guidance_mode;
   U8BIT num_langs;
   U32BIT *lang_codes;
   SI_STRING_DESC **strings;
} SI_GUIDANCE_DESC;

typedef struct si_fta_content_desc
{
   BOOLEAN do_not_scramble;
   U8BIT access_over_internet;
   BOOLEAN do_not_apply_revocation;
} SI_FTA_CONTENT_DESC;

typedef struct si_target_region
{
   struct si_target_region *next;
   U8BIT region_depth;
   U8BIT primary_region_code;
   U8BIT secondary_region_code;
   U8BIT tertiary_region_code;
} SI_TARGET_REGION;

typedef struct si_target_region_desc
{
   struct si_target_region_desc *next;
   U32BIT country_code;
   SI_TARGET_REGION *region_list;
} SI_TARGET_REGION_DESC;

typedef struct si_serv_avail_desc
{
   struct si_serv_avail_desc *next;
   U16BIT num_of_cell_ids;
   U16BIT *cell_ids;
   BOOLEAN availability_flag;
} SI_SERV_AVAIL_DESC;

typedef enum
{
   ICON_TRANS_LOCAL = 0x0,
   ICON_TRANS_URL = 0x1,
   ICON_TRANS_RES1 = 0x2,
   ICON_TRANS_RES2 = 0x3
} E_ICON_TRANSPORT_MODE;

typedef enum
{
   ICON_COORDS_576 = 0x0,
   ICON_COORDS_720 = 0x1,
   ICON_COORDS_1080 = 0x2
} E_ICON_COORD_SYSTEM;

typedef struct si_image_icon_desc
{
   U8BIT desc_num;
   U8BIT last_desc_num;
   U8BIT icon_id;
   E_ICON_TRANSPORT_MODE transport_mode;
   BOOLEAN position_defined;
   E_ICON_COORD_SYSTEM coord_system;
   U16BIT x_pos;
   U16BIT y_pos;
   U8BIT *icon_type;
   U32BIT data_len;
   U8BIT *icon_data;
} SI_IMAGE_ICON_DESC;

typedef struct si_uri_linkage_desc
{
   U32BIT private_data_code;
   U8BIT uri_linkage_type;
   SI_STRING_DESC *uri;
   U16BIT min_polling_interval;
   U8BIT private_data_length;
   U8BIT *private_data;
   struct si_uri_linkage_desc *next;
} SI_URI_LINKAGE_DESC;

typedef struct si_sdt_code_desc
{
   U8BIT data_length;
   U8BIT *data;
} SI_SDT_CODE_DESC;

/* CNS series (linking) descriptor */
typedef struct si_series_desc
{
    SI_STRING_DESC *episode_name;    /*!< episode name */
    U16BIT episode_number;           /*!< episode number */
    U8BIT content_status;            /*!< content status, 0=normal, 1=premier, 2=finale */
    BOOLEAN last_episode;            /*!< indicates the last episode */
} SI_SERIES_DESC;


//--------------------------------------------------
// PAT table structures
typedef struct si_pat_service_entry
{
   struct si_pat_service_entry *next;
   U16BIT serv_id;
   U16BIT pmt_pid;
} SI_PAT_SERVICE_ENTRY;

typedef struct si_pat_table
{
   U8BIT version;
   U16BIT tran_id;
   U16BIT num_services;
   SI_PAT_SERVICE_ENTRY *service_list;
   SI_PAT_SERVICE_ENTRY *last_service_entry;
} SI_PAT_TABLE;

//--------------------------------------------------
// PMT table structures
typedef struct si_dvb_subt_desc
{
   U32BIT lang_code;
   U8BIT type;
   U16BIT composition_page;
   U16BIT ancillary_page;
} SI_SUBTITLE_DESC;

typedef struct si_teletext_desc
{
   U32BIT lang_code;
   U8BIT type;
   U8BIT magazine;
   U8BIT page;
} SI_TELETEXT_DESC;

typedef struct si_ca_desc
{
   U16BIT ca_id;
   U16BIT ca_pid;
} SI_CA_DESC;

typedef struct si_iso_lang_desc
{
   U32BIT lang_code;
   U8BIT audio_type;
} SI_ISO_LANG_DESC;

/* Structure used when parsing AC-3 and E-AC3 descriptors */
typedef struct si_ac3_desc
{
   U8BIT dtag;
   BOOLEAN component_type_flag;
   U8BIT component_type;
   BOOLEAN bsid_flag;
   U8BIT bsid;
   BOOLEAN mainid_flag;
   U8BIT mainid;
   BOOLEAN asvc_flag;
   U8BIT asvc;
   BOOLEAN mixinfoexists;
   BOOLEAN substream1_flag;
   U8BIT substream1;
   BOOLEAN substream2_flag;
   U8BIT substream2;
   BOOLEAN substream3_flag;
   U8BIT substream3;
} SI_AC3_DESC;

typedef struct si_ad_desc
{
   U32BIT lang_code;
   BOOLEAN mix_type;
   U8BIT edit_class;
} SI_AD_DESC;

typedef struct si_aac_desc
{
   U8BIT profile_level;
   BOOLEAN type_present;
   U8BIT aac_type;
} SI_AAC_DESC;

typedef struct si_service_move_desc
{
   U16BIT onet_id;
   U16BIT ts_id;
   U16BIT service_id;
} SI_SERVICE_MOVE_DESC;

typedef struct si_app_sig_desc
{
   U16BIT app_type;
   U8BIT ait_version;
} SI_APP_SIG_DESC;

typedef enum
{
   SI_STREAM_TYPE_VIDEO1 = 0x01,
   SI_STREAM_TYPE_VIDEO2 = 0x02,
   SI_STREAM_TYPE_AUDIO1 = 0x03,
   SI_STREAM_TYPE_AUDIO2 = 0x04,
   SI_STREAM_TYPE_PRIVATE = 0x05,
   SI_STREAM_TYPE_PES_PKT = 0x06,
   SI_STREAM_TYPE_MHEG = 0x07,
   SI_STREAM_TYPE_DATA_A = 0x0a,
   SI_STREAM_TYPE_DATA_B = 0x0b,
   SI_STREAM_TYPE_DATA_C = 0x0c,
   SI_STREAM_TYPE_DATA_D = 0x0d,
   SI_STREAM_TYPE_AUX = 0x0e,
   SI_STREAM_TYPE_AAC = 0x0f,
   SI_STREAM_TYPE_HEAAC = 0x11,
   SI_STREAM_TYPE_H264 = 0x1b,
   SI_STREAM_TYPE_H265 = 0x24,
   SI_STREAM_TYPE_AC3 = 0x81
} SI_STREAM_TYPE;

typedef struct si_freesat_tunnelled_data_desc
{
   U8BIT data_type;
   U8BIT component_tag;
} SI_FREESAT_TUNNELLED_DATA_DESC;

typedef struct si_pmt_stream_entry
{
   struct si_pmt_stream_entry *next;
   SI_STREAM_TYPE type;
   U16BIT pid;
   U8BIT *tag_array_ptr;
   U8BIT num_tag_entries;
   U16BIT num_iso_lang_entries;
   SI_ISO_LANG_DESC *iso_lang_desc_array;
   U16BIT num_ca_entries;
   SI_CA_DESC *ca_desc_array;
   U16BIT num_subtitle_entries;
   SI_SUBTITLE_DESC *subtitle_desc_array;
   U16BIT num_teletext_entries;
   SI_TELETEXT_DESC *teletext_desc_array;
   SI_SERVICE_MOVE_DESC *service_move;
   U16BIT num_app_sig_entries;
   SI_APP_SIG_DESC *app_sig_desc_array;
   BOOLEAN has_ait;
   BOOLEAN has_rct;
   U8BIT num_tunnelled_entries;
   SI_FREESAT_TUNNELLED_DATA_DESC *tunnelled_desc_array;
   U32BIT carousel_id;

   SI_AC3_DESC *ac3_descriptor;
   SI_AD_DESC *audio_desc;
   SI_AAC_DESC *aac_descriptor;
} SI_PMT_STREAM_ENTRY;

typedef struct si_pmt_table
{
   U8BIT version;
   U16BIT serv_id;
   U16BIT pcr_pid;
   U16BIT num_ca_entries;
   SI_CA_DESC *ca_desc_array;
   U8BIT num_tunnelled_entries;
   SI_FREESAT_TUNNELLED_DATA_DESC *tunnelled_desc_array;
   U8BIT content_protection_level;
   U16BIT num_streams;
   SI_PMT_STREAM_ENTRY *stream_list;
   SI_PMT_STREAM_ENTRY *last_stream_entry;
} SI_PMT_TABLE;

//--------------------------------------------------
// NIT table structures
typedef struct si_multiling_net_name_desc
{
   U32BIT lang_code;
   SI_STRING_DESC *name_str;
} SI_MULTILING_NET_NAME_DESC;

typedef struct si_serv_list_desc
{
   U16BIT serv_id;
   U8BIT serv_type;
} SI_SERV_LIST_DESC;

typedef struct uk_dtt_lcn_desc
{
   U16BIT serv_id;
   U16BIT serv_lcn;
   BOOLEAN visible;
} SI_LCN_DESC;

typedef struct
{
   U16BIT serv_id;
   U16BIT serv_lcn;
   BOOLEAN visible;
} SI_NORDIG_SERV_LCN;

typedef struct
{
   U8BIT chan_list_id;
   SI_STRING_DESC *chan_list_name;
   U32BIT country_code;
   U8BIT num_services;
   SI_NORDIG_SERV_LCN *serv_array;
} SI_NORDIG_LCN_DESC;

typedef struct si_terr_del_sys_desc
{
   U32BIT freq_hz;
   E_STB_DP_TBWIDTH bwidth;
   E_STB_DP_TMODE mode;
} SI_TERR_DEL_SYS_DESC;

typedef struct si_t2_del_sys_cell
{
   U16BIT cell_id;
   U8BIT num_freqs;
   U32BIT freq_hz[6];
} SI_T2_DEL_SYS_CELL;

typedef struct si_t2_del_sys_desc
{
   U8BIT plp_id;
   U16BIT t2_system_id;
   E_STB_DP_TBWIDTH bwidth;
   E_STB_DP_TMODE mode;
   U8BIT num_cells;
   SI_T2_DEL_SYS_CELL *cell;
} SI_T2_DEL_SYS_DESC;

typedef struct
{
   BOOLEAN is_t2;
   union
   {
      SI_TERR_DEL_SYS_DESC t1;
      SI_T2_DEL_SYS_DESC t2;
   } u;
} SI_TERR_DEL_SYS;

typedef struct si_cable_del_sys_desc
{
   U32BIT freq_hz;
   U8BIT fec_outer;
   U8BIT fec_inner;
   U8BIT modulation;
   U32BIT symbol_rate;
} SI_CABLE_DEL_SYS_DESC;

typedef struct si_sat_del_sys_desc
{
   U32BIT freq_hz;
   U16BIT position;
   U16BIT sym_rate;
   E_STB_DP_POLARITY polarity;
   BOOLEAN east_west;
   E_STB_DP_FEC fec_code;
   BOOLEAN dvb_s2;
   E_STB_DP_MODULATION modulation;
} SI_SAT_DEL_SYS_DESC;

typedef enum si_delivery_sys_desc_type
{
   SI_DEL_SYS_DESC_TYPE_NONE,
   SI_DEL_SYS_DESC_TYPE_TERR,
   SI_DEL_SYS_DESC_TYPE_SAT,
   SI_DEL_SYS_DESC_TYPE_CABLE
} SI_DELIVERY_SYS_DESC_TYPE;

typedef union si_delivery_sys_desc
{
   SI_SAT_DEL_SYS_DESC sat;
   SI_CABLE_DEL_SYS_DESC cable;
   SI_TERR_DEL_SYS terr;
} SI_DELIVERY_SYS_DESC;

typedef struct si_serv_attribute_desc
{
   U16BIT serv_id;
   BOOLEAN service_selectable;
   BOOLEAN service_visible;
} SI_SERV_ATTRIBUTE_DESC;

typedef struct si_nit_region_name
{
   U8BIT region_depth;
   SI_STRING_DESC *region_name;
   U8BIT primary_region_code;
   U8BIT secondary_region_code;
   U16BIT tertiary_region_code;
} SI_NIT_REGION_NAME;

typedef struct si_nit_target_region_name_desc
{
   struct si_nit_target_region_name_desc *next;
   U32BIT country_code;
   U32BIT lang_code;
   U8BIT num_names;
   SI_NIT_REGION_NAME *name_array;
} SI_NIT_TARGET_REGION_NAME_DESC;

typedef struct si_ciplus_service
{
   struct si_ciplus_service *next;
   U16BIT id;
   U8BIT type;
   BOOLEAN visible;
   BOOLEAN selectable;
   U16BIT lcn;
   SI_STRING_DESC *provider_str;
   SI_STRING_DESC *name_str;
} SI_CIPLUS_SERVICE;

typedef enum
{
   FREQ_LIST_CODING_TYPE_SATELLITE = 1,
   FREQ_LIST_CODING_TYPE_CABLE = 2,
   FREQ_LIST_CODING_TYPE_TERRESTRIAL = 3
} SI_FREQ_LIST_CODING_TYPE;

typedef struct si_nit_frequency_list_desc
{
   struct si_nit_frequency_list_desc *next;
   U8BIT coding_type;
   U8BIT num_frequencies;
   U32BIT *frequency_array;
} SI_NIT_FREQUENCY_LIST_DESC;

typedef struct si_nit_transport_entry
{
   struct si_nit_transport_entry *next;
   U16BIT tran_id;
   U16BIT orig_net_id;
   SI_DELIVERY_SYS_DESC_TYPE del_sys_desc_type;
   SI_DELIVERY_SYS_DESC *del_sys_desc;
   SI_NIT_FREQUENCY_LIST_DESC *freq_list;
   U16BIT num_serv_list_entries;
   SI_SERV_LIST_DESC *serv_list_desc_array;
   U16BIT num_lcn_entries;
   SI_LCN_DESC *lcn_desc_array;
   SI_STRING_DESC *def_authority;
   U16BIT num_serv_attr_entries;
   SI_SERV_ATTRIBUTE_DESC *serv_attr_array;
   U16BIT num_nordig_lcn_entries;
   SI_NORDIG_LCN_DESC *nordig_lcn_desc_array;
   U16BIT num_hd_lcn_entries;
   SI_LCN_DESC *hd_lcn_desc_array;
   SI_FTA_CONTENT_DESC *fta_content_desc;
   SI_TARGET_REGION_DESC *target_region_list;
   U16BIT num_ciplus_services;
   SI_CIPLUS_SERVICE *ciplus_service_list;
   SI_CIPLUS_SERVICE *last_ciplus_service;
} SI_NIT_TRANSPORT_ENTRY;

typedef enum
{
   CATEGORY_ALL_RECEIVERS = 0,
   CATEGORY_T2_RECEIVERS = 1
} SI_NIT_RECEIVER_CATEGORY;

typedef enum
{
   NET_CHANGE_MESSAGE_ONLY = 0x0,
   NET_CHANGE_MINOR_DEFAULT = 0x1,
   NET_CHANGE_MULTIPLEX_REMOVE = 0x2,
   NET_CHANGE_SERVICE_CHANGE = 0x3,
   NET_CHANGE_MINOR_RESERVED_1 = 0x4,
   NET_CHANGE_MINOR_RESERVED_2 = 0x5,
   NET_CHANGE_MINOR_RESERVED_3 = 0x6,
   NET_CHANGE_MINOR_RESERVED_4 = 0x7,
   NET_CHANGE_MAJOR_DEFAULT = 0x8,
   NET_CHANGE_MULTIPLEX_FREQ_CHANGE = 0x9,
   NET_CHANGE_MULTIPLEX_COVERAGE_CHANGE = 0xA,
   NET_CHANGE_MULTIPLEX_ADDED = 0xB,
   NET_CHANGE_MAJOR_RESERVED_1 = 0xC,
   NET_CHANGE_MAJOR_RESERVED_2 = 0xD,
   NET_CHANGE_MAJOR_RESERVED_3 = 0xE,
   NET_CHANGE_MAJOR_RESERVED_4 = 0xF
} SI_NIT_NETWORK_CHANGE_TYPE;

typedef struct si_nit_change_entry
{
   U8BIT change_id;
   U8BIT version;
   U16BIT start_date;
   U8BIT start_hours;
   U8BIT start_mins;
   U8BIT start_secs;
   U8BIT dur_hours;
   U8BIT dur_mins;
   U8BIT dur_secs;
   SI_NIT_RECEIVER_CATEGORY receiver_category;
   SI_NIT_NETWORK_CHANGE_TYPE change_type;
   U8BIT message_id;
   BOOLEAN invariant_ts_present;
   U16BIT invariant_ts_tsid;
   U16BIT invariant_ts_onid;
} SI_NIT_CHANGE_ENTRY;

typedef struct si_nit_change_notify_desc
{
   U16BIT cell_id;
   U8BIT num_changes;
   SI_NIT_CHANGE_ENTRY *change_array;
} SI_NIT_CHANGE_NOTIFY_DESC;

typedef struct si_nit_message_entry
{
   U8BIT message_id;
   U32BIT lang_code;
   SI_STRING_DESC *message;
} SI_NIT_MESSAGE_ENTRY;

typedef struct si_freesat_linkage_desc
{
   U16BIT onet_id;
   U16BIT trans_id;
   U16BIT serv_id;
   U8BIT num_data_types;
   U8BIT *data_types;
   struct si_freesat_linkage_desc *next;
} SI_FREESAT_LINKAGE_DESC;

typedef struct si_freesat_prefix_desc
{
   U8BIT prefix_index;
   SI_STRING_DESC *uri_prefix;
   struct si_freesat_prefix_desc *next;
} SI_FREESAT_PREFIX_DESC;

typedef struct si_ciplus_virtual_channel
{
   U16BIT lcn;
   U8BIT event_info_len;
   U8BIT app_domain_len;
   SI_STRING_DESC *provider_str;
   SI_STRING_DESC *name_str;
   U8BIT *event_info;
   U8BIT *app_domain_id;
} SI_CIPLUS_VIRTUAL_CHANNEL;

typedef struct si_nit_table
{
   U8BIT version;
   U16BIT net_id;
   SI_STRING_DESC *name_str;
   U16BIT num_multiling_net_names;
   SI_MULTILING_NET_NAME_DESC *multiling_net_name_desc_array;
   U16BIT num_linkage_entries;
   SI_LINKAGE_DESC_ENTRY *linkage_desc_list;
   SI_LINKAGE_DESC_ENTRY *last_linkage_entry;
   SI_STRING_DESC *def_authority;
   U16BIT num_transports;
   SI_NIT_TRANSPORT_ENTRY *transport_list;
   SI_NIT_TRANSPORT_ENTRY *last_transport_entry;
   U16BIT num_change_notifies;
   SI_NIT_CHANGE_NOTIFY_DESC *change_notify_array;
   U16BIT num_messages;
   SI_NIT_MESSAGE_ENTRY *message_array;
   SI_FTA_CONTENT_DESC *fta_content_desc;
   SI_NIT_TARGET_REGION_NAME_DESC *target_region_name_list;
   SI_TARGET_REGION_DESC *target_region_list;
   SI_FREESAT_LINKAGE_DESC *freesat_linkage_desc;
   SI_FREESAT_PREFIX_DESC *freesat_prefix_list;
   SI_URI_LINKAGE_DESC *uri_linkage_list;
   SI_CIPLUS_VIRTUAL_CHANNEL *ciplus_virtual_channel;
} SI_NIT_TABLE;

//--------------------------------------------------
// SDT table structures
typedef struct si_multiling_serv_name_desc
{
   U32BIT lang_code;
   SI_STRING_DESC *name_str;
   SI_STRING_DESC *provider_str;
} SI_MULTILING_SERV_NAME_DESC;

typedef struct
{
   U32BIT lang_code;
   SI_STRING_DESC *name_str;
} SI_MULTILING_SHORT_NAME_DESC;

typedef struct si_preferred_name_desc
{
   U32BIT lang_code;
   U8BIT name_id;
   SI_STRING_DESC *name_str;
} SI_PREFERRED_NAME_DESC;

typedef enum
{
   RUN_STATE_UNDEFINED = 0,
   RUN_STATE_NOT_RUNNING,
   RUN_STATE_STARTS_SOON,
   RUN_STATE_PAUSING,
   RUN_STATE_RUNNING,
   RUN_STATE_OFF_AIR
} E_RUNNING_STATE;

typedef struct si_sdt_service_entry
{
   struct si_sdt_service_entry *next;
   U16BIT serv_id;
   BOOLEAN eit_sched_avail;
   BOOLEAN eit_now_next_avail;
   U8BIT running_status;
   BOOLEAN all_streams_free;
   U8BIT serv_type;
   SI_STRING_DESC *name_str;
   SI_STRING_DESC *provider_str;
   U8BIT num_ca_id_entries;
   U16BIT *ca_id_desc_array;
   U16BIT num_multiling_names;
   SI_MULTILING_SERV_NAME_DESC *multiling_name_desc_array;
   U16BIT num_preferred_names;
   SI_PREFERRED_NAME_DESC *preferred_name_desc_array;
   U16BIT num_linkage_entries;
   SI_LINKAGE_DESC_ENTRY *linkage_desc_list;
   SI_LINKAGE_DESC_ENTRY *last_linkage_entry;
   SI_STRING_DESC *def_authority;
   SI_STRING_DESC *short_name_str;
   U16BIT num_multiling_short_names;
   SI_MULTILING_SHORT_NAME_DESC *multiling_short_name_array;
   SI_GUIDANCE_DESC *guidance;
   SI_FTA_CONTENT_DESC *fta_content_desc;
   SI_TARGET_REGION_DESC *target_region_list;
   SI_SERV_AVAIL_DESC *serv_avail_desc_list;
   U8BIT *ci_protection_desc;
   SI_FREESAT_PREFIX_DESC *freesat_prefix_list;
   SI_URI_LINKAGE_DESC *uri_linkage_list;
   SI_SDT_CODE_DESC *sdt_code_desc;
} SI_SDT_SERVICE_ENTRY;

typedef struct si_sdt_table
{
   U8BIT version;
   U16BIT tran_id;
   U16BIT orig_net_id;
   U16BIT num_services;
   SI_SDT_SERVICE_ENTRY *service_list;
   SI_SDT_SERVICE_ENTRY *last_service_entry;
} SI_SDT_TABLE;


//--------------------------------------------------
// BAT table structures

typedef struct si_freesat_lcn
{
   BOOLEAN numeric_selection;
   BOOLEAN visible_flag;
   BOOLEAN user_cust;
   U16BIT lcn;
   U16BIT region_id;
   struct si_freesat_lcn *next;
} SI_FREESAT_LCN;

typedef struct si_bat_freesat_region_lcn_entry
{
   U16BIT tran_id;
   U16BIT orig_net_id;
   U16BIT service_id;
   U16BIT freesat_id;
   struct si_bat_freesat_region_lcn_entry *next;
   SI_FREESAT_LCN *freesat_lcn_list;
} SI_BAT_FREESAT_REGION_LCN_ENTRY;

typedef struct si_bat_transport_entry
{
   struct si_bat_transport_entry *next;
   U16BIT tran_id;
   U16BIT orig_net_id;
   SI_SERV_LIST_DESC *serv_list_desc_array;
   U16BIT num_serv_list_entries;
   U16BIT num_lcn_entries;
   SI_LCN_DESC *lcn_desc_array;
   SI_BAT_FREESAT_REGION_LCN_ENTRY *lcn_region_list;
   SI_STRING_DESC *def_authority;
   SI_FTA_CONTENT_DESC *fta_content_desc;
   U16BIT *int_rest_serv_array; /* Interactive restrictive services */
   U8BIT num_int_rest_serv;
} SI_BAT_TRANSPORT_ENTRY;

typedef struct si_multilang_group_name_desc
{
   U32BIT lang_code;
   SI_STRING_DESC *name_str;
} SI_MULTILANG_GROUP_NAME_DESC;

typedef struct si_bat_freesat_group_name_entry
{
   U8BIT group_type;
   U16BIT group_id;
   SI_MULTILANG_GROUP_NAME_DESC *string_array;
   U8BIT num_group_names;
   struct si_bat_freesat_group_name_entry *next_group;
} SI_BAT_FREESAT_GROUP_NAME_ENTRY;

typedef struct si_bat_freesat_serv_group_entry
{
   U16BIT group_id;
   BOOLEAN non_destructive_flag;
   BOOLEAN return_channel_access_flag;
   BOOLEAN g2_extension_flag;
   U8BIT g2_flags;
   U8BIT num_services;
   U16BIT *freesat_id;
   struct si_bat_freesat_serv_group_entry *next_group;
} SI_BAT_FREESAT_SERV_GROUP_ENTRY;

typedef struct si_bat_freesat_iactive_storage_desc
{
   U8BIT *data;
   U8BIT length;
   struct si_bat_freesat_iactive_storage_desc *next_desc;
} SI_BAT_FREESAT_IACTIVE_STORAGE_DESC;

typedef struct si_bat_freesat_region
{
   U16BIT region_id;
   U32BIT lang_code;
   SI_STRING_DESC *region_name;
   struct si_bat_freesat_region *next;
} SI_BAT_FREESAT_REGION;

typedef struct si_freesat_info_location
{
   U16BIT orig_net_id;
   U16BIT transport_id;
   U16BIT service_id;
   struct si_freesat_info_location *next;
} SI_BAT_FREESAT_INFO_LOCATION;

typedef struct si_bat_table
{
   U8BIT version;
   U16BIT bouquet_id;
   SI_STRING_DESC *bouquet_name;
   SI_STRING_DESC *def_authority;
   SI_BAT_FREESAT_REGION *region_list;
   U16BIT num_linkage_entries;
   SI_LINKAGE_DESC_ENTRY *linkage_desc_list;
   SI_LINKAGE_DESC_ENTRY *last_linkage_entry;
   U16BIT num_transports;
   SI_BAT_TRANSPORT_ENTRY *transport_list;
   SI_BAT_TRANSPORT_ENTRY *last_transport_entry;
   SI_BAT_FREESAT_GROUP_NAME_ENTRY *group_name_list;
   U16BIT num_serv_groups;
   SI_BAT_FREESAT_SERV_GROUP_ENTRY *serv_group_array;
   SI_FTA_CONTENT_DESC *fta_content_desc;
   SI_BAT_FREESAT_IACTIVE_STORAGE_DESC *iactive_storage_desc_list;
   SI_BAT_FREESAT_INFO_LOCATION *info_location_list;
   SI_FREESAT_PREFIX_DESC *freesat_prefix_list;
   SI_URI_LINKAGE_DESC *uri_linkage_list;
} SI_BAT_TABLE;


//--------------------------------------------------
// EIT table structures

typedef struct si_component_desc
{
   U8BIT tag;
   U8BIT content;
   U8BIT type;
   U32BIT lang_code;
   SI_STRING_DESC *desc_str;
} SI_COMPONENT_DESC;

typedef struct si_multiling_component_desc
{
   U8BIT tag;
   U32BIT lang_code;
   SI_STRING_DESC *desc_str;
} SI_MULTILING_COMPONENT_DESC;

typedef struct si_content_desc
{
   U8BIT level_1;
   U8BIT level_2;
   U8BIT user_1;
   U8BIT user_2;
} SI_CONTENT_DESC;

typedef struct si_parental_rating_desc
{
   U32BIT country_code;
   U8BIT rating;
} SI_PARENTAL_RATING_DESC;

typedef struct si_short_event_desc
{
   U32BIT lang_code;
   SI_STRING_DESC *name_str;
   SI_STRING_DESC *desc_str;
} SI_SHORT_EVENT_DESC;

typedef struct si_extended_event_desc
{
   U8BIT desc_number;
   U8BIT last_desc_number;
   U32BIT lang_code;
   U8BIT num_items;
   SI_STRING_DESC **item_desc_array;
   SI_STRING_DESC **item_text_array;
   SI_STRING_DESC *text_str;
} SI_EXTENDED_EVENT_DESC;

typedef struct si_crid_desc
{
   struct si_crid_desc *next;
   U8BIT type;
   SI_STRING_DESC *crid_str;
} SI_CRID_DESC;

typedef struct si_eit_event_entry
{
   struct si_eit_event_entry *next;
   U8BIT sect_num;
   U16BIT event_id;
   U16BIT start_date;      // mjd format
   U8BIT start_hrs;
   U8BIT start_mins;
   U8BIT start_secs;
   U8BIT duration_hrs;
   U8BIT duration_mins;
   U8BIT duration_secs;
   U8BIT running_status;
   BOOLEAN all_streams_free;
   U8BIT preferred_name_id;
   U8BIT num_ca_id_entries;
   U16BIT *ca_id_desc_array;
   U8BIT num_component_entries;
   SI_COMPONENT_DESC *component_desc_array;
   U8BIT num_content_entries;
   SI_CONTENT_DESC *content_desc_array;
   U8BIT num_multiling_component_entries;
   SI_MULTILING_COMPONENT_DESC *multiling_component_desc_array;
   U8BIT num_parental_rating_entries;
   SI_PARENTAL_RATING_DESC *parental_rating_desc_array;
   U8BIT num_short_event_entries;
   SI_SHORT_EVENT_DESC *short_event_desc_array;
   U8BIT num_extended_event_entries;
   SI_EXTENDED_EVENT_DESC *extended_event_desc_array;
   U8BIT num_crids;
   SI_CRID_DESC *crid_list;
   SI_CRID_DESC *last_crid_entry;
   SI_GUIDANCE_DESC *guidance;
   SI_FTA_CONTENT_DESC *fta_content_desc;
   SI_SERIES_DESC *series_desc;
} SI_EIT_EVENT_ENTRY;

typedef struct si_eit_table
{
   U8BIT version;
   U8BIT table_id;
   U16BIT serv_id;
   U16BIT tran_id;
   U16BIT orig_net_id;
   U8BIT last_table_id;
   U16BIT num_events;
   SI_EIT_EVENT_ENTRY *event_list;
   SI_EIT_EVENT_ENTRY *last_event_entry;
} SI_EIT_TABLE;


//--------------------------------------------------
// Time and date table structures
typedef struct si_lto_desc
{
   U32BIT country_code;
   U8BIT region;
   BOOLEAN offset_negative;
   U8BIT offset_hrs;
   U8BIT offset_mins;
   U16BIT change_date;        // mjd format
   U8BIT change_hrs;
   U8BIT change_mins;
   U8BIT change_secs;
   U8BIT next_offset_hrs;
   U8BIT next_offset_mins;
} SI_LTO_DESC;

typedef struct si_time_table
{
   U16BIT date;               // mjd format
   U8BIT hrs;
   U8BIT mins;
   U8BIT secs;
   U16BIT num_lto_entries;
   SI_LTO_DESC *lto_desc_array;
} SI_TIME_TABLE;


//--------------------------------------------------
// Related content table (RCT) structures
typedef enum
{
   RCT_LINK_TYPE_URI = 0x0,
   RCT_LINK_TYPE_BINARY = 0x1,
   RCT_LINK_TYPE_URI_BINARY = 0x2,
   RCT_LINK_TYPE_DESCRIPTOR = 0x3
} E_RCT_LINK_TYPE;

typedef enum
{
   RCT_HOW_RELATED_TVA_2004 = 0x00,
   RCT_HOW_RELATED_TVA_2005 = 0x01,
   RCT_HOW_RELATED_TVA_2007 = 0x02
} E_RCT_HOW_RELATED;

typedef enum
{
   RCT_TERMID_IS_TRAILER_OF = 0x0002,
   RCT_TERMID_IS_GROUP_TRAILER_OF = 0x0005
} E_RCT_TERM_ID;

typedef struct si_rct_promo_text
{
   U32BIT lang_code;
   SI_STRING_DESC *string;
} SI_RCT_PROMO_TEXT;

typedef struct si_rct_link_info
{
   E_RCT_LINK_TYPE link_type;
   E_RCT_HOW_RELATED how_related;
   U16BIT term_id;
   U8BIT group_id;
   U8BIT precedence;
   U8BIT *uri_string;
   U8BIT num_items;
   SI_RCT_PROMO_TEXT *promo_text_array;
   BOOLEAN can_use_default_icon;
   U8BIT icon_id;
   SI_SHORT_EVENT_DESC *event_desc;
   U8BIT num_icons;
   SI_IMAGE_ICON_DESC *icon_array;
} SI_RCT_LINK_INFO;

typedef struct si_rct_subtable_data
{
   U16BIT year_offset;
   U8BIT link_count;
   SI_RCT_LINK_INFO *link_array;
   U8BIT num_icons;
   SI_IMAGE_ICON_DESC *icon_array;
   struct si_rct_subtable_data *next;
} SI_RCT_SUBTABLE_DATA;

typedef struct si_rct_subtable
{
   U8BIT version;
   U16BIT service_id;
   SI_RCT_SUBTABLE_DATA *data;
   struct si_rct_subtable *next;
} SI_RCT_SUBTABLE;

typedef struct si_rct_table
{
   SI_RCT_SUBTABLE *subtables;
} SI_RCT_TABLE;

//--------------------------------------------------
// SSU update notification table structures
typedef struct si_ssu_location_desc
{
   U16BIT data_broadcast_id;
   U16BIT association_tag;
   U8BIT private_data_length;
   U8BIT *private_data_byte;
} SI_SSU_LOCATION_DESC;

typedef struct si_unt_table
{
   U8BIT version;
   U8BIT action_type;
   U32BIT oui;
   U8BIT processing_order;
   //
   SI_SSU_LOCATION_DESC *ssu_location_desc;
   //
   SI_DELIVERY_SYS_DESC_TYPE del_sys_desc_type;
   SI_DELIVERY_SYS_DESC *del_sys_desc;
} SI_UNT_TABLE;

//--------------------------------------------------
// CAT conditional access table structures
typedef struct si_cat_table
{
   U8BIT version;
   U16BIT num_ca_entries;
   SI_CA_DESC *ca_desc_array;
} SI_CAT_TABLE;

//---Global Function prototypes for public use---------------------------------

/**
 * @brief   Sets the country specific private data specifier code that will be used when parsing
 *          SI tables
 * @param   code country specific private data code
 */
void STB_SISetCountryPrivateDataSpecifier(U32BIT code);

/**
 * @brief   Enables or disables use of the Freesat private data specifier when parsing SI tables
 * @param   mode TRUE to enable, FALSE to disable
 */
void STB_SISetFreesatPrivateDataSpecifierMode(BOOLEAN mode);

/**
 * @brief   Enables or disables use of the CI+ private data specifier when parsing SI tables
 * @param   mode TRUE to enable, FALSE to disable
 */
void STB_SISetCiplusPrivateDataSpecifierMode(BOOLEAN mode);

/**
 * @brief   Enables or disables use of the EACEM private data specifier when parsing SI tables
 * @param   mode TRUE to enable, FALSE to disable
 */
void STB_SISetEacemPrivateDataSpecifierMode(BOOLEAN mode);

/**
 * @brief   Enables or disables use of the New Zealand satellite private data specifier when parsing SI tables
 * @param   mode TRUE to enable, FALSE to disable
 */
void STB_SISetNZSatPrivateDataSpecifierMode(BOOLEAN mode);

/**
 * @brief   Enables or disables use of the Nordig private data specifier when parsing SI tables
 * @param   mode TRUE to enable, FALSE to disable
 */
void STB_SISetNordigPrivateDataSpecifierMode(BOOLEAN mode);

void STB_SISetUserDefinedDescriptorFunction(U8BIT dtag, STB_SI_USER_DEF_DESCRIP_FUNCTION func);
void STB_SIClearUserDefinedDescriptorFunctions(void);

void *STB_SIRequestPat(U8BIT path, E_SI_REQUEST_TYPE req_type,
                       void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param);

void *STB_SIRequestPmt(U8BIT path, E_SI_REQUEST_TYPE req_type, U16BIT pmt_pid,
                       U16BIT sid_match, U16BIT sid_mask, U16BIT table_count,
                       void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param);
void STB_SIModifyPmtRequest(void *fhandle, U16BIT sid_match, U16BIT sid_mask, U16BIT table_count);

void *STB_SIRequestNit(U8BIT path, E_SI_REQUEST_TYPE req_type,
                       void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param);

/**
 * @brief   Create an SI filter for an NIT table
 * @param   path demux path to be used
 * @param   pid PID the table will be available on - this allows the standard PID to be overridden
 * @param   actual TRUE if NITactual is to be retrieved, FALSE for NITother
 * @param   req_type ONE_SHOT_REQUEST or CONTINUOUS_REQUEST
 * @param   callback function to be called when table is received
 * @param   ret_param value that will be passed to the callback function
 * @return  filter handle, NULL if filter isn't created
 */
void *STB_SIRequestNitFromPid(U8BIT path, U16BIT pid, BOOLEAN actual, E_SI_REQUEST_TYPE req_type,
                              void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param);

/**
 * @brief   Generates request for NITactual and NITother for the given network ID
 * @param   path the decode path to use
 * @param   network_id network ID being requested
 * @param   req_type ONE_SHOT or CONTINUOUS request
 * @param   callback callback function to receive the parsed nit table
 * @param   ret_param parameter to be returned to callback function
 * @return  filter handle or NULL if filter not successfully setup
 */
void* STB_SIRequestNitWithId(U8BIT path, U16BIT network_id, E_SI_REQUEST_TYPE req_type,
   void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param);

void *STB_SIRequestSdt(U8BIT path, E_SI_REQUEST_TYPE req_type,
                       BOOLEAN inc_sdt_actual, BOOLEAN inc_sdt_other,
                       U16BIT tran_id_match, U16BIT tran_id_mask, U16BIT table_count,
                       void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param);

void *STB_SIRequestSdtFromPid(U8BIT path, U16BIT pid, E_SI_REQUEST_TYPE req_type,
                              BOOLEAN inc_sdt_actual, BOOLEAN inc_sdt_other,
                              U16BIT tran_id_match, U16BIT tran_id_mask, U16BIT table_count,
                              void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param);

void STB_SIModifySdtRequest(void *fhandle, BOOLEAN inc_sdt_actual, BOOLEAN inc_sdt_other,
   U16BIT tran_id_match, U16BIT tran_id_mask, U16BIT table_count);

void *STB_SIRequestBat(U8BIT path, E_SI_REQUEST_TYPE req_type, U16BIT bouquet_id_match,
                       U16BIT bouquet_id_mask, U16BIT table_count, void (*callback)(void *, U32BIT, SI_TABLE_RECORD *),
                       U32BIT ret_param);

void *STB_SIRequestBatFromPid(U8BIT path, U16BIT pid, E_SI_REQUEST_TYPE req_type,
                              U16BIT tran_id_match, U16BIT tran_id_mask, U16BIT table_count,
                              void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param);

void *STB_SIRequestEit(U8BIT path, E_SI_REQUEST_TYPE req_type, E_SI_EIT_TABLE_REQ reqd_eit_tables,
                       U16BIT sid_match, U16BIT sid_mask, U16BIT table_count,
                       void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param);

void *STB_SIRequestEitFromPid(U8BIT path, U16BIT pid, E_SI_REQUEST_TYPE req_type, E_SI_EIT_TABLE_REQ reqd_eit_tables,
                              U16BIT sid_match, U16BIT sid_mask, U16BIT table_count,
                              void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param);

void STB_SIModifyEitRequest(void *fhandle, E_SI_EIT_TABLE_REQ reqd_eit_tables,
   U16BIT sid_match, U16BIT sid_mask, U16BIT table_count);

void *STB_SIRequestSched(U8BIT path, E_SI_REQUEST_TYPE req_type, E_SI_SCHED_TABLE_REQ reqd_eit_tables,
                         U16BIT sid_match, U16BIT sid_mask, U16BIT table_count,
                         void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param);

void *STB_SIRequestSchedFromPid(U8BIT path, U16BIT pid, E_SI_REQUEST_TYPE req_type, E_SI_SCHED_TABLE_REQ reqd_eit_tables,
                                U16BIT sid_match, U16BIT sid_mask, U16BIT table_count,
                                void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param);

void *STB_SIRequestTdt(U8BIT path, E_SI_REQUEST_TYPE req_type,
                       void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param);

void *STB_SIRequestTdtFromPid(U8BIT path, U16BIT pid, E_SI_REQUEST_TYPE req_type,
                              void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param);

void *STB_SIRequestTot(U8BIT path, E_SI_REQUEST_TYPE req_type,
                       void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param);

void *STB_SIRequestTotFromPid(U8BIT path, U16BIT pid, E_SI_REQUEST_TYPE req_type,
                              void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param);

void *STB_SIRequestCat(U8BIT path, E_SI_REQUEST_TYPE req_type,
                       void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param);

void *STB_SIRequestRct(U8BIT path, E_SI_REQUEST_TYPE req_type, U16BIT rct_pid,
                       void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param);

void* STB_SIRequestUnt(U8BIT path, E_SI_REQUEST_TYPE req_type,
                      void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param);

void* STB_SIRequestUntFromPid(U8BIT path, U16BIT pid, E_SI_REQUEST_TYPE req_type,
                             void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param );

SI_PAT_TABLE* STB_SIParsePatTable(SI_TABLE_RECORD *table_rec);
SI_PMT_TABLE* STB_SIParsePmtTable(SI_TABLE_RECORD *table_rec);
SI_NIT_TABLE* STB_SIParseNitTable(SI_TABLE_RECORD *table_rec);
SI_SDT_TABLE* STB_SIParseSdtTable(SI_TABLE_RECORD *table_rec);
SI_EIT_TABLE* STB_SIParseEitTable(SI_TABLE_RECORD *table_rec);
SI_TIME_TABLE* STB_SIParseTimeTable(SI_TABLE_RECORD *table_rec);
SI_RCT_TABLE* STB_SIParseRctTable(SI_TABLE_RECORD *table_rec);
SI_BAT_TABLE* STB_SIParseBatTable(SI_TABLE_RECORD *table_rec);
SI_UNT_TABLE* STB_SIParseUntTable(SI_TABLE_RECORD *table_rec);
SI_CAT_TABLE* STB_SIParseCatTable(SI_TABLE_RECORD *table_rec);

U8BIT* STB_SIReadString(U8BIT nbytes, U8BIT *dptr, SI_STRING_DESC **str_ptr);

BOOLEAN STB_SIParseDelSysDesc(U8BIT *data, SI_DELIVERY_SYS_DESC_TYPE *type, SI_DELIVERY_SYS_DESC **desc);
BOOLEAN STB_SIParseServiceDescriptor(U8BIT *data, U8BIT *type, SI_STRING_DESC **provider,
   SI_STRING_DESC **name);
BOOLEAN STB_SIParseShortEventDescriptor(U8BIT *data, SI_SHORT_EVENT_DESC **event_desc);

/*!**************************************************************************
 * @brief   Parses the given PMT to produce an array of the CA system IDs required
 *          by the service or streams on the service. The array of IDs will be
 *          allocated by this function and should be freed using STB_SIReleaseCaIdDescArray.
 * @param   pmt_data - raw PMT section data
 * @param   pmt_ca_ids - pointer to an array that will be allocated containing
 *                       the CA IDs found in the PMT
 * @return  Number of CA IDs found in the PMT and returned in the array,
 *          0 if none are found.
 ****************************************************************************/
U16BIT STB_SIGetPmtCaIdDescArray(U8BIT *pmt_data, U16BIT **pmt_ca_ids);

/**
 * @brief   Generates request for AIT on given PID
 * @param   path The demux path to use
 * @param   req_type ONE_SHOT or CONTINUOUS request
 * @param   ait_pid Pid for the AIT
 * @param   callback Callback function to receive the parsed pmt table
 * @param   ret_param Parameter to be returned to callback function
 * @return  Filter handle, NULL if filter not successfully setup
 */
void *STB_SIRequestAit(U8BIT path, E_SI_REQUEST_TYPE req_type, U16BIT ait_pid,
                       void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param);

void STB_SIReleasePatTable(SI_PAT_TABLE *pat_table);
void STB_SIReleasePmtTable(SI_PMT_TABLE *pmt_table);
void STB_SIReleaseNitTable(SI_NIT_TABLE *nit_table);
void STB_SIReleaseSdtTable(SI_SDT_TABLE *sdt_table);
void STB_SIReleaseEitTable(SI_EIT_TABLE *eit_table);
void STB_SIReleaseTimeTable(SI_TIME_TABLE *time_table);
void STB_SIReleaseRctTable(SI_RCT_TABLE *rct_table);
void STB_SIReleaseBatTable(SI_BAT_TABLE *bat_table);
void STB_SIReleaseUntTable(SI_UNT_TABLE *unt_table);
void STB_SIReleaseCatTable(SI_CAT_TABLE *cat_table);

// the following functions can be used to release individual table entry structures from the parsed
// tables. They are mainly intended for releasing entries which have been transferred
// out of the parsed tables by the application SI handler and are therefore not released when the
// table is released.
void STB_SIReleasePatStreamEntry(SI_PAT_SERVICE_ENTRY *entry_ptr);
void STB_SIReleasePmtStreamEntry(SI_PMT_STREAM_ENTRY *entry_ptr);
void STB_SIReleaseNitTransportEntry(SI_NIT_TRANSPORT_ENTRY *entry_ptr);
void STB_SIReleaseSdtServiceEntry(SI_SDT_SERVICE_ENTRY *entry_ptr);
void STB_SIReleaseEitEventEntry(SI_EIT_EVENT_ENTRY *entry_ptr);
void STB_SIReleaseRctSubtable(SI_RCT_SUBTABLE *sub_ptr);
void STB_SIReleaseRctSubtableData(SI_RCT_SUBTABLE_DATA *data_ptr);

/* TODO These can probably become local functions */
void STB_SIReleaseBatTransportEntry(SI_BAT_TRANSPORT_ENTRY *entry_ptr);
void STB_SIReleaseBatLcnEntry(SI_BAT_FREESAT_REGION_LCN_ENTRY *entry_ptr);
// the following functions can be used to release individual descriptor array elements from the
// parsed tables. They are mainly intended for releasing descriptors which have been transferred
// out of the parsed tables by the application SI handler (by setting the table element pointing to
// the descriptor array to NULL), and are therefore not released when the table is released.
void STB_SIReleaseDelSysDesc(SI_DELIVERY_SYS_DESC *desc, SI_DELIVERY_SYS_DESC_TYPE type);
void STB_SIReleaseStringDesc(SI_STRING_DESC *desc);
void STB_SIReleaseCaDescArray(SI_CA_DESC *desc_array, U16BIT num_entries);
void STB_SIReleaseCaIdDescArray(U16BIT *desc_array, U8BIT num_entries);
void STB_SIReleaseComponentDescArray(SI_COMPONENT_DESC *desc_array, U8BIT num_entries);
void STB_SIReleaseContentDescArray(SI_CONTENT_DESC *desc_array, U8BIT num_entries);
void STB_SIReleaseFreqListDescArray(SI_NIT_FREQUENCY_LIST_DESC *freq_list);
void STB_SIReleaseGuidanceDesc(SI_GUIDANCE_DESC *desc_ptr);
void STB_SIReleaseImageIconDescArray(SI_IMAGE_ICON_DESC *icon_array, U8BIT num_icons);
void STB_SIReleaseIsoLangDescArray(SI_ISO_LANG_DESC *desc_array, U16BIT num_entries);
void STB_SIReleaseLinkageDescList(SI_LINKAGE_DESC_ENTRY *desc_list, U16BIT num_entries);
void STB_SIReleaseLtoDescArray(SI_LTO_DESC *desc_array, U16BIT num_entries);
void STB_SIReleaseMultilingComponentDescArray(SI_MULTILING_COMPONENT_DESC *desc_array, U8BIT num_entries);
void STB_SIReleaseMultilingNetNameDescArray(SI_MULTILING_NET_NAME_DESC *desc_array, U16BIT num_entries);
void STB_SIReleaseMultilingServNameDescArray(SI_MULTILING_SERV_NAME_DESC *desc_array, U16BIT num_entries);
void STB_SIReleaseMultilingShortNameArray(SI_MULTILING_SHORT_NAME_DESC *desc_array, U16BIT num_entries);
void STB_SIReleaseParentalRatingDescArray(SI_PARENTAL_RATING_DESC *desc_array, U8BIT num_entries);
void STB_SIReleaseServListDescArray(SI_SERV_LIST_DESC *desc_array, U16BIT num_entries);
void STB_SIReleaseShortEventDescArray(SI_SHORT_EVENT_DESC *desc_array, U8BIT num_entries);
void STB_SIReleaseExtendedEventDescArray(SI_EXTENDED_EVENT_DESC *desc_array, U8BIT num_entries);
void STB_SIReleaseSsuLocationDesc(SI_SSU_LOCATION_DESC *desc_ptr);
void STB_SIReleaseSubtitleDescArray(SI_SUBTITLE_DESC *desc_array, U16BIT num_entries);
void STB_SIReleaseTeletextDescArray(SI_TELETEXT_DESC *desc_array, U16BIT num_entries);
void STB_SIReleaseLcnDescArray(SI_LCN_DESC *desc_array, U16BIT num_entries);
void STB_SIReleaseNordigLcn2DescArray(SI_NORDIG_LCN_DESC *desc_array, U16BIT num_entries);
void STB_SIReleasePrefNameDescArray(SI_PREFERRED_NAME_DESC *desc_array, U16BIT num_entries);
void STB_SIReleaseCRIDList(SI_CRID_DESC *crid_list);
void STB_SIReleaseTargetRegionNameList(SI_NIT_TARGET_REGION_NAME_DESC *desc_list);
void STB_SIReleaseTargetRegionList(SI_TARGET_REGION_DESC *desc_list);
void STB_SIReleaseRctLinkInfo(SI_RCT_LINK_INFO *link_info);
void STB_SIReleaseFreesatLinkageDesc(SI_FREESAT_LINKAGE_DESC *desc);
void STB_SIReleaseFreesatPrefixList(SI_FREESAT_PREFIX_DESC *list);
void STB_SIReleaseAvailabilityDescriptorList(SI_SERV_AVAIL_DESC *desc_list);
void STB_SIReleaseCIPlusServiceList(SI_CIPLUS_SERVICE *service_list);
void STB_SIReleaseURILinkageList(SI_URI_LINKAGE_DESC *list);
void STB_SIReleaseSdtCodeDesc(SI_SDT_CODE_DESC *desc);
void STB_SIReleaseSeriesDesc(SI_SERIES_DESC *desc);

#endif //  _STBSITAB_H

//*****************************************************************************
// End of file
//*****************************************************************************

