/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Header for STB unicode string handling routines
 *
 * @file    stbuni.h
 * @date    31/05/2001
 */

#ifndef _STBUNI_H

#define _STBUNI_H

#include "techtype.h"


//---constant definitions---------------------------------------------------------------------------

/*
 * Convert the number of Unicode characters to the number of bytes for the Unicode string.
 * I.e. (2*<NumberUnicodeChars>) + 1 byte header + 2 byte null terminator.
 */
#define STB_UTF16_LEN_TO_BYTES_IN_STRING(x)     (((x) *2) + 3)
#define INVALID_UNICODE_CHAR                    0xFFFF      // 0xFFFF is an invalid unicode char value

//---typedefs-structs-enumerations------------------------------------------------------------------


//---function prototypes----------------------------------------------------------------------------
U8BIT* STB_SetUnicodeStringChar(U8BIT *string_ptr, U16BIT char_id, U16BIT code);

U8BIT* STB_DeleteUnicodeStringChar(U8BIT *string_ptr, U16BIT char_id);

U32BIT STB_GetUnicodeStringChar(U8BIT *string_ptr, U16BIT char_id);

U8BIT* STB_ConcatUnicodeStrings(U8BIT *string1_ptr, U8BIT *string2_ptr);

U8BIT* STB_UnicodeStringTokenise(U8BIT *str, U8BIT **save_ptr);

U8BIT* STB_UnicodeStrStr(U8BIT *str1, U8BIT *str2, BOOLEAN ignore_case);

S8BIT STB_CompareUnicodeStrings(U8BIT *string1_ptr, U8BIT *string2_ptr, BOOLEAN exact_match, BOOLEAN ignore_case);

U8BIT* STB_ConvertStringToUnicode(U8BIT *string, BOOLEAN *reverse_dir, U16BIT *nchar,
   BOOLEAN strip_DVB_cntrl_char, U32BIT lang_code);
U8BIT* STB_ConvertStringToUTF8(U8BIT *string, U16BIT *nchar, BOOLEAN strip_DVB_cntrl_char,
   U32BIT lang_code);

void STB_ReleaseUnicodeString(U8BIT *string);

U8BIT* STB_ConvertUTF16toUTF8( U8BIT *src, U32BIT *outlen );

BOOLEAN STB_IsUnicodeString(U8BIT *string_ptr);

BOOLEAN STB_IsNormalString(U8BIT *string_ptr);

U32BIT STB_GetNumBytesInString(U8BIT *string_ptr);

BOOLEAN STB_IsUnicodeStringReversed(U8BIT *string_ptr);

BOOLEAN STB_IsStringEmpty(U8BIT *string_ptr);

/**
 * @brief   Sets default ascii table to be used, if not overridden by the table index at the start
 *          of a string
 * @param   table - index of the table to be used (0 to 15), as defined in ETSI 300 468,
 *                  Annex A, table A.3
 */
void STB_SetDefaultAsciiTable(U8BIT table);

S8BIT STB_CompareStringsIgnoreCase(U8BIT *string1_ptr, U8BIT *string2_ptr);

U8BIT* STB_FormatUnicodeString( BOOLEAN strip_DVB_cntrl_char, BOOLEAN *reverse_dir, const U8BIT *const format_ptr, ... );

U32BIT STB_UnicodeStringLen(U8BIT *string_ptr);

U8BIT* STB_UnicodeInsertString(U8BIT *src_str, U16BIT insert_pos, U8BIT *insert_str, BOOLEAN replace_char);

/**
 * @brief   Strips the DVB control characters from a string that's already in UTF-8 or UTF-16
 *          format. The control chars that are stripped are DVB emphasis on/off and DVB CR/LF.
 *          The input string isn't changed and the returned string must be freed by calling
 *          STB_ReleaseUnicodeString.
 * @param   string_ptr UTF-8 or UTF-16 string from which the control chars are to be stripped
 * @return  new string in the same format as the input string
 */
U8BIT* STB_UnicodeStripControlChars(U8BIT *string_ptr);

#endif //  _STBUNI_H

//**************************************************************************************************
// End of File
//**************************************************************************************************
