/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2012 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Header file - macros and function prototypes for public use
 *
 * @file    stbvtc.h
 * @date    27/02/2012
 * @author  Ocean Blue
 */

// pre-processor mechanism so multiple inclusions don't cause compilation error

#ifndef _STBVTC_H
#define _STBVTC_H

#include "techtype.h"
#include "osdtype.h"
#include "vtctype.h"

//---Constant and macro definitions for public use-----------------------------

//---Enumerations for public use-----------------------------------------------

typedef void (*F_NOTIFY_VIDEO_AR)(E_ASPECT_RATIO ar);

//---Global type defs for public use-------------------------------------------

//---Global Function prototypes for public use---------------------------------

/*!**************************************************************************
 * @brief   Initialise the VTC module
 * @param   None
 * @return  None
 ****************************************************************************/
void STB_VTInitialise(void);

/*!**************************************************************************
 * @brief   DVBCore application output window for entire video
 * @param   output - output video rectangle
 * @return  None
 ****************************************************************************/
void STB_VTSetVideoOutput(S_RECTANGLE *output);

/*!**************************************************************************
 * @brief   Get display aspect ratio
 * @param   aspect_ratio - display aspect ratio
 * @return  None
 ****************************************************************************/
E_ASPECT_RATIO STB_VTGetDisplayAspectRatio(void);

/*!**************************************************************************
 * @brief   Set display aspect ratio
 * @param   aspect_ratio - display aspect ratio
 * @return  None
 ****************************************************************************/
void STB_VTSetDisplayAspectRatio(E_ASPECT_RATIO aspect_ratio);

/*!**************************************************************************
 * @brief   Turn on or off MHEG5 scaling calculation
 * @param   enable TRUE - turn on MHEG5 calculations
 * @return  None
 ****************************************************************************/
void STB_VTSetMhegEnable(BOOLEAN enable);

/*!**************************************************************************
 * @brief   Set MHEG5 scaling resolution for video
 * @param   width - width resolution
 * @param   height - height resolution
 * @return  None
 ****************************************************************************/
void STB_VTSetMhegScalingResolution(U16BIT width, U16BIT height);

/*!**************************************************************************
 * @brief   Set video scaling by MHEG5
 * @param   scaling - scaling transformation (offset, size)
 * @return  None
 ****************************************************************************/
void STB_VTSetMhegVideoScaling(S_RECTANGLE *scaling);

/*!**************************************************************************
 * @brief    Set scene aspect ratio (MHEG-5 specific)
 * @param    aspect_ratio - scene aspect ratio
 * @return  None
 ****************************************************************************/
void STB_VTSetMhegAspectRatio(E_ASPECT_RATIO aspect_ratio);

/*!**************************************************************************
 * @brief   Set MHEG5 widescreen alignment
 * @param   mode - widescreen alignment mode
 * @return  None
 ****************************************************************************/
void STB_VTSetMhegVideoAlignment(E_VIDEO_ASPECT_MODE mode);

/*!**************************************************************************
 * @brief   Turn on or off HBBTV scaling calculation
 * @param   enable TRUE - turn on HbbTV calculations
 * @return  None
 ****************************************************************************/
void STB_VTSetHbbtvEnable(BOOLEAN enable);

/*!**************************************************************************
 * @brief   Set video scaling by HBBTV
 * @param   rect output window rectangle
 * @return  None
 ****************************************************************************/
void STB_VTSetHbbtvVideoWindow(S_RECTANGLE *rect);

/*!**************************************************************************
 * @brief   Set video alignment preference
 * @param   pref - video alignment
 * @return  None
 ****************************************************************************/
void STB_VTSetVideoAlignmentPref(E_VIDEO_ASPECT_MODE pref);

/**
 * @brief   Sets the callback used to control the format conversion when the aspect mode set by
 *          STB_VTSetVideoAlignmentPref is ASPECT_MODE_CUSTOM
 * @param   callback Function used by the VTC module to determine what format conversion needs to
 *          be applied.
 */
void STB_VTCSetCustomModeCallback(F_VT_CUSTOM_MODE_CALLBACK callback);

/*!**************************************************************************
 * @brief   Set video preferences change callback
 *
 * The callback function is called when video transformation is changed as a
 * result of a user preference change (only).
 *
 * @param   callback - callback for notification
 * @param   user_data - user data for the callback
 * @return  None
 ****************************************************************************/
void STB_VTSetVideoPrefChangedCallback(void (*callback)(void *),
                                       void *user_data);

/*!**************************************************************************
 * @brief   Set video preferences change callback
 *
 * The callback function is called when video transformation is changed as a
 * result of a user preference change (only).
 *
 * @param   callback - callback for notification
 * @param   user_data - user data for the callback
 * @return  None
 ****************************************************************************/
void STB_VTSetVideoRatioCallback(F_NOTIFY_VIDEO_AR ar);

/*!**************************************************************************
 * @brief   Return the current decoder format conversion
 * @return  The current format conversion
 ****************************************************************************/
E_FORMAT_CONVERSION STB_VTGetDecoderFormatConversion(void);

/*!**************************************************************************
 * @brief   Return the current video resolution
 * @param   width - video width
 * @param   height - video height
 * @return  None
 ****************************************************************************/
void STB_VTGetVideoResolution(U16BIT *width, U16BIT *height);

/*!**************************************************************************
 * @brief   Return the current video aspect ratio
 * @return  E_ASPECT_RATIO aspect ratio
 ****************************************************************************/
E_ASPECT_RATIO STB_VTGetVideoAspectRatio(void);

#endif //  _STBVTC_H

//*****************************************************************************
// End of file
//*****************************************************************************

