/*******************************************************************************
 * Copyright  2016 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2009 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief    CI Content Control - descrambler keys
 * @file     stbcikeys.c
 * @date     28 November 2016
 * @author   Adam Sturtridge
 */
//#define DEBUG_PRINTING_ENABLED

/*---includes for this file--------------------------------------------------*/

/* compiler library header files */
#include <string.h>

/* third party header files */

/* DVBCore header files */
#include "techtype.h"
#include "dbgfuncs.h"
#include "stbhwos.h"
#include "stbci.h"
#include "stberc.h"
#include "stbdpc.h"
#include "stbpvr.h"
#include "stbheap.h"

#include "stbcica.h"
#include "stbci_int.h"

/*---constant definitions for this file--------------------------------------*/

//#define DEBUG_CA
#ifdef DEBUG_CA
 #define DBG_CA(x, ...) STB_SPDebugWrite( "%s:%d " x, __FUNCTION__, __LINE__, ##__VA_ARGS__);
#else
 #define DBG_CA(x, ...)
#endif

/*---local typedef enums for this file-------------------------------------*/

/*---local typedef structs for this file-------------------------------------*/

typedef struct s_ca_info
{
   struct s_ca_info *next;
   U32BIT module;
   BOOLEAN disabled;
   U8BIT slot_id;
   U8BIT num_ca_ids;
   U8BIT last_pmt_version;
   U16BIT last_pmt_program;
} S_CA_INFO;

/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local) */

static S_CA_INFO *cica_slots = NULL;
static void *cica_mutex;

static S_CA_INFO *FindSlot(U8BIT slot_id);
static BOOLEAN UpdateCASystemInfo(S_CA_INFO *slot_info, U8BIT *pmt);

/*---global function definitions---------------------------------------------*/

/**
 * @brief    CI conditional access support initialise
 */
void STB_CiCaInitialise(void)
{
   cica_mutex = STB_OSCreateMutex();
}

/**
 * @brief   Disable CA for slot, as CAM is being removed
 * @param   slot_id zero-based CI slot identifier (0, 1, ...)
 */
void STB_CiCaDisable(U8BIT slot_id)
{
   S_CA_INFO *slot_info;

   FUNCTION_START(STB_CiKeysDisable);

   DBG_CA("(slot=%u)", slot_id)

   slot_info = FindSlot(slot_id);
   if (slot_info != NULL)
   {
      slot_info->disabled = TRUE;
   }
   FUNCTION_FINISH(STB_CiKeysDisable);
}

/**
 * @brief   Process CAM removal from slot for CA support
 * @param   slot_id zero-based CI slot identifier (0, 1, ...)
 */
void STB_CiCaRemove(U8BIT slot_id)
{
   S_CA_INFO *slot;
   S_CA_INFO **pslot;

   FUNCTION_START(STB_CiCaRemove);
   DBG_CA("(slot=%u)", slot_id)
   pslot = &cica_slots;
   STB_OSMutexLock(cica_mutex);
   slot = cica_slots;
   while (slot != NULL)
   {
      if (slot->slot_id == slot_id)
      {
         *pslot = slot->next;
         STB_FreeMemory(slot);
         break;
      }
      pslot = &(slot->next);
      slot = slot->next;
   }
   STB_OSMutexUnlock(cica_mutex);

   FUNCTION_FINISH(STB_CiCaRemove);
}

/**
 * @brief    Notify CA system IDs
 * @param    module - module ID
 * @param    ca_ids - array of CA system IDs
 * @param    num_ca_ids - number of CA system IDs in the array
 */
void STB_CINotifyCaSystems(U32BIT module, U16BIT *ca_ids, U8BIT num_ca_ids)
{
   S_CA_INFO *slot;
   U8BIT slot_id;

   FUNCTION_START(STB_CINotifyCaSystems);

   DBG_CA("mod=%x slot=%u", module, STB_GetCIConditionalAccessSlotId(module))

   slot_id = STB_GetCIConditionalAccessSlotId(module);
   slot = FindSlot(slot_id);
   if (slot != NULL && num_ca_ids > slot->num_ca_ids)
   {
      STB_CiCaRemove(slot_id);
      slot = NULL;
   }
   if (slot == NULL)
   {
      slot = STB_GetMemory(sizeof(S_CA_INFO)+ (num_ca_ids * sizeof(U16BIT)));
      if (slot != NULL)
      {
         slot->slot_id = slot_id;
         slot->disabled = FALSE;
         STB_OSMutexLock(cica_mutex);
         slot->next = cica_slots;
         cica_slots = slot;
         STB_OSMutexUnlock(cica_mutex);
      }
   }
   if (slot != NULL)
   {
      slot->module = module;
      slot->num_ca_ids = num_ca_ids;
      memcpy(&slot[1], ca_ids, num_ca_ids * sizeof(U16BIT));

      /* Force resending of PMT */
      slot->last_pmt_version = 0xFF;
      slot->last_pmt_program = 0xFFFF;

      STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_SLOT_STATUS_UPDATED, &slot_id, sizeof slot_id);
   }

   FUNCTION_FINISH(STB_CINotifyCaSystems);
}

void STB_CINotifyPmtReply(U32BIT module, U16BIT program_number, U8BIT version_number,
   U8BIT current_next_indicator, E_STB_CI_DESC ca_enable)
{
   FUNCTION_START(STB_CINotifyPmtReply);
   USE_UNWANTED_PARAM(module);
   USE_UNWANTED_PARAM(program_number);
   USE_UNWANTED_PARAM(version_number);
   USE_UNWANTED_PARAM(current_next_indicator);
   USE_UNWANTED_PARAM(ca_enable);
   FUNCTION_FINISH(STB_CINotifyPmtReply);
}

/**
 * @brief    Report PMT to CI stack
 * @param    slot_id - slot ID (0, 1, ...)
 * @param    pmt - PMT for the service (section)
 */
void STB_CiCaReportPmt(U8BIT slot_id, U8BIT *pmt)
{
   S_CA_INFO *slot_info;
   U16BIT program_number;
   U8BIT version_number;

   FUNCTION_START(STB_CiCaReportPmt);

   DBG_CA("(slot=%u)", slot_id)
   slot_info = FindSlot(slot_id);
   if (slot_info != NULL )
   {
      if (slot_info->num_ca_ids != 0)
      {
         /* CAM is ready and intereseted, report PMT if necessary */
         program_number = pmt[3] << 8 | pmt[4];
         version_number = (pmt[5] >> 1) & 0x1f;

         if ((slot_info->last_pmt_program != program_number) ||
             (slot_info->last_pmt_version != version_number))
         {
            slot_info->last_pmt_program = program_number;
            slot_info->last_pmt_version = version_number;

            DBG_CA("mod=%x p_num=%x v_num=%x", slot_info->module, program_number, version_number)
            STB_CIUpdatePmt(slot_info->module, pmt, TRUE, STB_CI_PMT_CMD_OK_DESCRAMBLE);
         }
      }
   }

   FUNCTION_FINISH(STB_CiCaReportPmt);
}

/**
 * @brief   Tell whether CA descriptors were found in PMT and that CA system is
 *          supported by CI slot.
 * @param   slot_id slot ID (0, 1, ...)
 * @param   pmt_data PMT data for the service
 * @return  BOOLEAN - TRUE if allowed
 */
BOOLEAN STB_CiCaSystemSupported(U8BIT slot_id, U8BIT *pmt_data)
{
   S_CA_INFO *slot_info;
   BOOLEAN supported;

   FUNCTION_START(STB_CiCaSystemSupported);

   DBG_CA("(slot=%u)", slot_id)
   slot_info = FindSlot(slot_id);
   if (slot_info != NULL )
   {
      supported = UpdateCASystemInfo(slot_info, pmt_data);
   }
   else
   {
      supported = FALSE;
   }

   FUNCTION_FINISH(STB_CiCaSystemSupported);

   return supported;
}

/*---local function definitions----------------------------------------------*/

static S_CA_INFO *FindSlot(U8BIT slot_id)
{
   S_CA_INFO *slot_info;
   STB_OSMutexLock(cica_mutex);
   slot_info = cica_slots;
   while (slot_info != NULL)
   {
      if (slot_info->slot_id == slot_id)
      {
         if (slot_info->disabled)
         {
            slot_info = NULL;
         }
         break;
      }
      slot_info = slot_info->next;
   }
   STB_OSMutexUnlock(cica_mutex);
   return slot_info;
}

/**
 * @brief    Tell whether the CA system is supported by the module
 *
 * This function checks whether one of the CA systems mentioned in the
 * PMT is supported by the CAM. If there are no CA descriptors in the
 * PMT, it is assumed that the service is not scrambled so the CAM can
 * support it.
 *
 * @param    slot_info - Slot state information
 * @param    pmt - The PMT for the service
 * @return   TRUE if the module supports the CA system, FALSE otherwise
 */
static BOOLEAN UpdateCASystemInfo(S_CA_INFO *slot_info, U8BIT *pmt)
{
   U16BIT ca_system_id;
   U16BIT section_length;
   U16BIT program_info_length;
   U16BIT es_info_length;
   U16BIT es_offset;
   U16BIT offset;
   U16BIT i;
   U16BIT *ca_ids;
   U8BIT desc_tag;
   U8BIT desc_len;
   BOOLEAN ca_found;
   BOOLEAN supported;

   FUNCTION_START(UpdateCASystemInfo);

   supported = FALSE;
   ca_found = FALSE;

   if (pmt != NULL && pmt[0] == 0x02)
   {
      /* Assuming that the PMT is valid */
      section_length = (pmt[1] << 8 | pmt[2]) & 0xfff;

      /* Program info */
      program_info_length = (pmt[10] << 8 | pmt[11]) & 0xfff;
      offset = 12;
      while (!supported && offset < 12 + program_info_length)
      {
         desc_tag = pmt[offset];
         desc_len = pmt[offset + 1];
         if (desc_tag == 0x09)
         {
            /* CA_descriptor() */
            ca_system_id = pmt[offset + 2] << 8 | pmt[offset + 3];
            ca_found = TRUE;
            ca_ids = (U16BIT*)(slot_info + 1);
            DBG_CA("Found CA system %04x in program info", ca_system_id)
            for (i = 0; i != slot_info->num_ca_ids; i++)
            {
               if (ca_ids[i] == ca_system_id)
               {
                  DBG_CA("CA system is supported")
                  supported = TRUE;
                  break;
               }
            }
         }
         offset += 2 + desc_len;
      }

      /* ES info */
      while (!supported && offset < section_length - 1)
      {
         offset += 3;
         es_info_length = (pmt[offset] << 8 | pmt[offset + 1]) & 0xfff;
         offset += 2;
         es_offset = offset;
         while (!supported && offset < es_offset + es_info_length)
         {
            desc_tag = pmt[offset];
            desc_len = pmt[offset + 1];
            if (desc_tag == 0x09)
            {
               /* CA_descriptor() */
               ca_system_id = pmt[offset + 2] << 8 | pmt[offset + 3];
               ca_found = TRUE;
               ca_ids = (U16BIT*)(slot_info + 1);
               DBG_CA("Found CA system %04x in ES info", ca_system_id)
               for (i = 0; i != slot_info->num_ca_ids; i++)
               {
                  if (ca_ids[i] == ca_system_id)
                  {
                     DBG_CA("CA system is supported")
                     supported = TRUE;
                     break;
                  }
               }
            }
            offset += 2 + desc_len;
         }
      }
   }

   if (ca_found)
   {
      DBG_CA("CA is used")
      if (supported)
      {
         DBG_CA("CA system is support")
      }
   }

   FUNCTION_FINISH(UpdateCASystemInfo);

   return supported;
}
