/*******************************************************************************
 * Copyright  2016 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief    CI Content Control
 * @file     stbcicc.c
 * @date     28 November 2016
 * @author   Adam Sturtridge
 */
//#define DEBUG_PRINTING_ENABLED

/*---includes for this file--------------------------------------------------*/

/* compiler library header files */

/* third party header files */
#include <string.h>

/* DVBCore header files */
#include "techtype.h"
#include "dbgfuncs.h"
#include "stbhwos.h"
#include "stbhwci.h"
#include "stbci.h"
#include "stberc.h"
#include "stbhwav.h"
#include "stbheap.h"
#include "stbdpc.h"

#include "stbcicc.h"
#include "stbci_int.h"

/*---constant definitions for this file--------------------------------------*/

#define MAX_CICAM_BRANDS      126

//#define DEBUG_CC
#ifdef DEBUG_CC
   #define DBG_CC(x, ...) STB_SPDebugWrite( "%s:%d " x, __FUNCTION__, __LINE__, ##__VA_ARGS__);
#else
   #define DBG_CC(x, ...)
#endif

//#define DEBUG_REC
#ifdef DEBUG_REC
   #define DBG_REC(x, ...) STB_SPDebugWrite( x, ##__VA_ARGS__ );
#else
   #define DBG_REC(x, ...)
#endif

/*---local typedef enums for this file-------------------------------------*/

typedef enum
{
   CI_AUTH_NONE = 0,
   CI_AUTH_IN_PROGRESS,
   CI_AUTH_SUCCESS,
   CI_AUTH_FAIL
} E_CI_AUTH_STATUS;

/*---local typedef structs for this file-------------------------------------*/

typedef struct ci_protection
{
   BOOLEAN sdt_acquisition_complete;
   BOOLEAN free_ci_mode_flag;
   BOOLEAN match_brand_flag;
   U8BIT number_of_entries;
   U16BIT entries[MAX_CICAM_BRANDS];
} S_CI_PROTECTION;

typedef struct s_slot_state
{
   struct s_slot_state *next;

   /* Authenticated CI+ module */
   E_CI_AUTH_STATUS auth_status;

   U16BIT program_number;

   /* CI+ CAM brand ID */
   U16BIT cicam_brand_id;

   /* Slot_id (0, 1, 2, ...) */
   U8BIT slot_id;

   /* URI information */
   S_STB_CI_URI uri;

   BOOLEAN ready;
   BOOLEAN disabled;

   /* Recording */
   BOOLEAN recording;
   U16BIT record_service_id;
   BOOLEAN op_mode_set;
   E_STB_CI_OPERATING_MODE operating_mode;
   S_STB_CI_URI recording_uri;
   U8BIT recording_licence_status;
   U8BIT *recording_licence;
   U16BIT recording_licence_len;
   U8BIT record_age_rating;
   U8BIT record_private_data[CIP_PIN_PRIVATE_DATA_SIZE];
   S_STB_CI_DATE pin_event_date;
   S_STB_CI_TIME pin_event_time;
   U8BIT record_pin_status;

   BOOLEAN pin_caps_requested;
   BOOLEAN pin_caps_received;
   E_STB_CI_PIN_CAPS pin_caps;
   S_STB_CI_DATE pin_date;
   S_STB_CI_TIME pin_time;
   U8BIT age_rating;

   /* Playback */
   BOOLEAN play_pin_sent;
   U16BIT play_service_id;
   S_STB_CI_URI playback_uri;
   U8BIT playback_licence_status;
   U8BIT *playback_licence;
   U16BIT playback_licence_len;
} S_SLOT_STATE;

/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local) */

static S_CI_PROTECTION ci_protection_info = {FALSE, FALSE, FALSE, 0, {0}};
static S_SLOT_STATE *cicc_slots = NULL;
static void *cicc_mutex;

static S_SLOT_STATE *FindSlot(U8BIT slot_id);
static BOOLEAN IsServiceAllowed(S_SLOT_STATE *slot_state,
   U8BIT *ci_protection_descriptor);
static void ParseCiProtectionDescriptor(U8BIT *ci_protection_descriptor);
static void ApplyURIv1(S_STB_CI_URI *uri, BOOLEAN playback);
static void ApplyURIv2(S_STB_CI_URI *uri, BOOLEAN playback);
static void ApplyInvalidURI(void);
static void PackUri(S_STB_CI_URI *uri, U8BIT buffer[CIP_URI_LEN]);
static void UnpackUri(U8BIT buffer[CIP_URI_LEN], S_STB_CI_URI *uri);

/*---global function definitions---------------------------------------------*/

/**
 * @brief    CI content control support initialise
 */
void STB_CiCcInitialise(void)
{
   cicc_mutex = STB_OSCreateMutex();
   STB_CiKeysInitialise();
   STB_CiCaInitialise();
}

/**
 * @brief    Notify module insertion
 * @param    slot_id - slot ID for module
 */
void STB_CiCcNotifyModuleInsert(U8BIT slot_id)
{
   S_SLOT_STATE *slot_info;

   FUNCTION_START(STB_CiCcNotifyModuleInsert);

   DBGPRINT("slot_id=%d", slot_id)

   slot_info = FindSlot(slot_id);
   if (slot_info != NULL)
   {
      DBGPRINT("CAM already inserted to slot %d! Inconsistent state!", slot_id);
   }
   else
   {
      slot_info = STB_GetMemory(sizeof(S_SLOT_STATE));
      if (slot_info != NULL)
      {
         slot_info->ready = FALSE;
         slot_info->disabled = FALSE;
         slot_info->slot_id = slot_id;
         slot_info->auth_status = CI_AUTH_NONE;
         slot_info->cicam_brand_id = 0x0000;
         slot_info->program_number = 0xFFFF;
         slot_info->play_service_id = 0xFFFF;
         slot_info->record_service_id = 0xFFFF;

         slot_info->recording = FALSE;
         slot_info->op_mode_set = FALSE;
         slot_info->operating_mode = STB_CI_MODE_UNATTENDED_RECORDING;
         slot_info->recording_licence = NULL;
         slot_info->recording_licence_len = 0;
         slot_info->pin_caps_requested = FALSE;
         slot_info->pin_caps_received = FALSE;
         slot_info->pin_caps = STB_CI_PIN_CAPS_NONE;
         slot_info->play_pin_sent = FALSE;
         slot_info->playback_licence = NULL;
         slot_info->playback_licence_len = 0;

         STB_OSMutexLock(cicc_mutex);
         slot_info->next = cicc_slots;
         cicc_slots = slot_info;
         STB_OSMutexUnlock(cicc_mutex);

         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_INSERT, &slot_id, sizeof(slot_id));
      }
   }

   FUNCTION_FINISH(STB_CiCcNotifyModuleInsert);
}

/**
 * @brief    Notify module insertion
 * @param    slot_id - slot ID for module
 * @param    ci_plus - TRUE if the module claims to be a CI+ module
 * @param    mask - a bit mask indicating which features are supported
 */
void STB_CINotifyModuleReady(U8BIT slot_id, BOOLEAN ci_plus, U32BIT mask)
{
   S_SLOT_STATE *slot;

   FUNCTION_START(STB_CINotifyModuleReady);

   DBGPRINT("slot_id=%d", slot_id)

   slot = FindSlot(slot_id);
   if (slot != NULL)
   {
      slot->ready = TRUE;
      STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_SLOT_STATUS_UPDATED, &slot_id, sizeof slot_id);
   }

   FUNCTION_FINISH(STB_CINotifyModuleReady);
}

/**
 * @brief    Notify module removal
 * @param    slot_id - slot ID for module
 */
void STB_CiCcNotifyModuleRemove(U8BIT slot_id)
{
   S_SLOT_STATE *slot;

   FUNCTION_START(STB_CiCcNotifyModuleRemove);

   DBGPRINT("slot_id=%d", slot_id)

   STB_CiCaDisable(slot_id);
   STB_CiKeysDisable(slot_id);

   slot = FindSlot(slot_id);
   if (slot != NULL)
   {
      slot->disabled = TRUE;
   }

   STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_REMOVE, &slot_id, sizeof(slot_id));

   FUNCTION_FINISH(STB_CiCcNotifyModuleRemove);
}

/**
 * @brief   Handle CAM removal from slot
 * @param   slot_id zero-based CI slot identifier (0, 1, ...)
 */
void STB_CiCcRemove(U8BIT slot_id)
{
   S_SLOT_STATE *slot;
   S_SLOT_STATE **pslot;

   FUNCTION_START(STB_CiCcRemove);

   pslot = &cicc_slots;
   STB_OSMutexLock(cicc_mutex);
   slot = cicc_slots;
   while (slot != NULL)
   {
      if (slot->slot_id == slot_id)
      {
         *pslot = slot->next;
         if (slot->recording_licence != NULL)
         {
            STB_FreeMemory(slot->recording_licence);
            slot->recording_licence = NULL;
         }
         if (slot->playback_licence != NULL)
         {
            STB_FreeMemory(slot->playback_licence);
            slot->playback_licence = NULL;
         }
         STB_FreeMemory(slot);
         break;
      }
      pslot = &(slot->next);
      slot = slot->next;
   }
   STB_OSMutexUnlock(cicc_mutex);

   STB_CiKeysRemove(slot_id);
   STB_CiCaRemove(slot_id);

   FUNCTION_FINISH(STB_CiCcRemove);
}

/**
 * @brief   Return whether CI slot is ready
 * @param   slot_id - slot ID (0, 1, ...)
 * @return  BOOLEAN - TRUE if ready
 */
BOOLEAN STB_CiCcIsSlotReady(U8BIT slot_id)
{
   S_SLOT_STATE *slot;
   BOOLEAN ready;

   FUNCTION_START(STB_CiCcIsSlotReady);

   slot = FindSlot(slot_id);
   if (slot != NULL)
   {
      ready = slot->ready;
   }
   else
   {
      ready = FALSE;
   }

   FUNCTION_FINISH(STB_CiCcIsSlotReady);

   return ready;
}

/**
 * @brief   Tell whether the service is allowed. This function checks whether
 *          the CI Protection Descriptor allows the service to be routed
 *          through the CI module.
 * @param   slot_id slot ID (0, 1, ...)
 * @param   ci_prot_desc CI Protection Descriptor for the service
 * @return  BOOLEAN - TRUE if allowed
 */
BOOLEAN STB_CiCcIsServiceAllowed(U8BIT slot_id, U8BIT *ci_prot_desc)
{
   S_SLOT_STATE *slot;
   BOOLEAN allowed;

   FUNCTION_START(STB_CiCcIsServiceAllowed);

   allowed = FALSE;

   slot = FindSlot(slot_id);
   if (slot != NULL && slot->ready)
   {
      allowed = IsServiceAllowed(slot, ci_prot_desc);
   }

   FUNCTION_FINISH(STB_CiCcIsServiceAllowed);

   return allowed;
}

/**
 * @brief   Tell whether the CI slot has been authenticated
 * @param   slot_id slot ID (0, 1, ...)
 * @return  BOOLEAN - TRUE if authenticated
 */
BOOLEAN STB_CiCcAuthenticated(U8BIT slot_id)
{
   S_SLOT_STATE *slot;
   BOOLEAN auth;

   FUNCTION_START(STB_CiCcAuthenticated);

   auth = FALSE;

   slot = FindSlot(slot_id);
   if (slot != NULL && slot->ready)
   {
      auth = (slot->auth_status == CI_AUTH_SUCCESS)? TRUE : FALSE;
   }

   FUNCTION_FINISH(STB_CiCcAuthenticated);

   return auth;
}

/**
 * @brief   Checks CAMs in all slots to find the one with the given CAM id
 * @param   cicam_id - CAM id to look for
 * @return  slot id if found, INVALID_RES_ID if not found
 */
U8BIT STB_CiCcFindSlotForCicamId(U8BIT cicam_id[CIP_CICAM_ID_LEN])
{
   U8BIT slot_id, num_slots;
   U8BIT id[CIP_CICAM_ID_LEN];

   FUNCTION_START(FindSlotForCicamId);
   num_slots = STB_CIGetSlotCount();
   for (slot_id = 0; slot_id != num_slots; slot_id++)
   {
      if (STB_CIGetCicamId(slot_id, id))
      {
         if (memcmp(id, cicam_id, CIP_CICAM_ID_LEN) == 0)
         {
            /* Slot with same CAM id found */
            break;
         }
      }
   }
   if (slot_id == num_slots)
   {
      slot_id = INVALID_RES_ID;
   }
   FUNCTION_FINISH(FindSlotForCicamId);

   return(slot_id);
}

/**
 * @brief   To implement the first part of the diagram in Figure 10.2: Shunning Operation of
 *          ci-plus_specification_v1.3.2.pdf, this function passes the status of the SDT acquisition
 *          to the ci+ glue.
 * @param   sdt_acquisition_status TRUE if the SDT acquisition is completed (either successfully if
 *          the table has been received, or unsuccessfully if the SDT timeout has expired), FALSE if
 *          the DVB engine is waiting for an SDT.
 */
void STB_CiCcSetSDTAcquisitionStatus(BOOLEAN complete)
{
   FUNCTION_START(STB_CiCcSetSDTAcquisitionStatus);
   DBGPRINT("%s", complete ? "TRUE" : "FALSE")
   ci_protection_info.sdt_acquisition_complete = complete;
   FUNCTION_FINISH(STB_CiCcSetSDTAcquisitionStatus);
}

/**
 * @brief    Return the current URI for the given service
 * @param    slot_id - slot ID
 * @param    service_id - service ID
 * @param    uri - the URI for the service
 * @return   URI for the service if available or "empty" URI otherwise
 */
void STB_CiCcGetUsageRulesInfo(U8BIT slot_id, U16BIT service_id,
   U8BIT raw_uri[CIP_URI_LEN])
{
   S_SLOT_STATE *slot_state;

   FUNCTION_START(STB_CiCcGetUsageRulesInfo);

   memset(raw_uri, 0, CIP_URI_LEN);

   slot_state = FindSlot(slot_id);
   if ((slot_state != NULL) && (slot_state->program_number == service_id))
   {
      DBGPRINT("Found URI for service_id = %d", service_id)
      PackUri(&slot_state->uri, raw_uri);
   }
   else
   {
      DBGPRINT("cannot find URI")
      STB_CiCcGetDefaultUsageRulesInfo(raw_uri);
   }

   FUNCTION_FINISH(STB_CiCcGetUsageRulesInfo);
}

/**
 * @brief   Sets the default URI values into the given array. The URI requires HDCP
 * @param   raw_uri - returns with the default URI settings
 */
void STB_CiCcGetDefaultUsageRulesInfo(U8BIT raw_uri[CIP_URI_LEN])
{
   FUNCTION_START(STB_CiCcGetDefaultUsageRulesInfo);

   memset(raw_uri, 0x00, CIP_URI_LEN);

   /* Set default URI with emi bits set so that HDCP is required */
   raw_uri[0] = 0x01;
   raw_uri[1] = 0x30;

   FUNCTION_FINISH(STB_CiCcGetDefaultUsageRulesInfo);
}

/**
 * @brief    Return the retention limit given a packed URI
 * @param    uri - Usage Rules Information
 * @return   Retention limit in minutes (0 = no limit)
 */
U32BIT STB_CiCcGetRetentionLimit(U8BIT raw_uri[CIP_URI_LEN])
{
   S_STB_CI_URI uri;
   U32BIT retention_limit;

   FUNCTION_START(STB_CiCcGetRetentionLimit);

   /* No limit unless explicitly limited */
   retention_limit = 0;

   UnpackUri(raw_uri, &uri);
   if ((uri.protocol_version == 0x01) || (uri.protocol_version == 0x02))
   {
      if (uri.rl_copy_control_info == 0x0)
      {
         /* 90 minutes */
         retention_limit = 90;
      }
      else if (uri.rl_copy_control_info == 0x1)
      {
         /* 6 hours */
         retention_limit = 6 * 60;
      }
      else if (uri.rl_copy_control_info == 0x2)
      {
         /* 12 hours */
         retention_limit = 12 * 60;
      }
      else
      {
         if (uri.protocol_version == 0x01)
         {
            /* 1-61 multiples of 24 hours */
            retention_limit = (uri.rl_copy_control_info - 2) * 24 * 60;
         }
         else if (uri.rl_copy_control_info != 0xff)
         {
            /* 1-252 multiples of 24 hours */
            retention_limit = (uri.rl_copy_control_info - 2) * 24 * 60;
         }
      }
   }

   FUNCTION_FINISH(STB_CiCcGetRetentionLimit);

   return retention_limit;
}

/**
 * @brief    Apply the given Usage Rules Information
 * @param    uri - Usage Rules Information
 */
void STB_CiCcApplyUsageRulesInfo(U8BIT raw_uri[CIP_URI_LEN])
{
   S_STB_CI_URI uri;

   FUNCTION_START(STB_CiCcApplyUsageRulesInfo);

   UnpackUri(raw_uri, &uri);

   switch (uri.protocol_version)
   {
      case 0x1:
         ApplyURIv1(&uri, FALSE);
         break;

      case 0x02:
         ApplyURIv2(&uri, FALSE);
         break;

      default:
         DBGPRINT("Invalid URI version")
         ApplyInvalidURI();
         break;
   }

   FUNCTION_FINISH(STB_CiCcApplyUsageRulesInfo);
}

/**
 * @brief    Apply the given Usage Rules Information for playback
 * @param    uri - Usage Rules Information
 */
void STB_CiCcApplyUsageRulesInfoForPlayback(U8BIT *raw_uri)
{
   S_STB_CI_URI uri;

   FUNCTION_START(STB_CiCcApplyUsageRulesInfoForPlayback);

   UnpackUri(raw_uri, &uri);

   switch (uri.protocol_version)
   {
      case 0x1:
         ApplyURIv1(&uri, TRUE);
         break;

      case 0x02:
         ApplyURIv2(&uri, TRUE);
         break;

      default:
         DBGPRINT("Invalid URI version")
         ApplyInvalidURI();
         break;
   }

   FUNCTION_FINISH(STB_CiCcApplyUsageRulesInfoForPlayback);
}

/**
 * @brief    Tell whether the given service requires HDCP
 * @param    service_id - Service ID
 */
BOOLEAN STB_CiCcIsHDCPRequired(U16BIT service_id)
{
   S_SLOT_STATE *slot_info;
   BOOLEAN required;

   FUNCTION_START(STB_CiCcIsHDCPRequired);

   required = FALSE;

   slot_info = cicc_slots;
   while (slot_info != NULL)
   {
      if (slot_info->program_number == service_id)
      {
         /* If program_number is set, the URI has been notified.
          * Any valid URI requires HDCP, so the answer is TRUE.
          */
         required = TRUE;
         break;
      }
      slot_info = slot_info->next;
   }

   FUNCTION_FINISH(STB_CiCcIsHDCPRequired);

   return required;
}

/*****************************************************************************
 *
 *  CONTENT CONTROL FUNCTIONS
 *
 ****************************************************************************/

/**
 * @brief    Notify content control progress
 * @param    module - module ID
 * @param    state - content control state
 * @param    cicam_brand_id - CICAM brand identifier
 */
void STB_CINotifyProgress(U8BIT slot_id, E_STB_CI_CC_STATE state, U16BIT cicam_brand_id)
{
   S_SLOT_STATE *slot_state;
   U8BIT path, tuner_path;

   FUNCTION_START(STB_CINotifyProgress);

   DBG_CC("State: %s, CICAM brand ID: %04x",
           state == STB_CI_CC_START ? "STB_CI_CC_START" :
           state == STB_CI_CC_VERIFIED ? "STB_CI_CC_VERIFIED" :
           state == STB_CI_CC_CERT_EXPIRED ? "STB_CI_CC_CERT_EXPIRED" :
           state == STB_CI_CC_PROTOCOL_ERROR ? "STB_CI_CC_PROTOCOL_ERROR" :
           state == STB_CI_CC_AUTH_ERROR ? "STB_CI_CC_AUTH_ERROR" :
           "UNKNOWN", cicam_brand_id)

   slot_state = FindSlot(slot_id);
   if (slot_state != NULL)
   {
      if (STB_CIGetContentControlVersion(slot_id) >= 2)
      {
         if (!slot_state->pin_caps_requested)
         {
            DBG_CC("%s(%u): Requesting PIN capabilities", __FUNCTION__, slot_id)

            slot_state->pin_caps_received = FALSE;

            if (STB_CIRequestPinCaps(slot_id))
            {
               slot_state->pin_caps_requested = TRUE;
            }
#ifdef DEBUG_CC
            else
            {
               DBG_CC("%s(%u): Failed to request PIN capabilities", __FUNCTION__, slot_id)
            }
#endif
         }
      }

      switch (state)
      {
         case STB_CI_CC_START:
         {
            slot_state->auth_status = CI_AUTH_IN_PROGRESS;

            /* The CICAM needs the TS to have the correct time to evaluate the certificates. At the
               end of the certification the DVB stack will re-evaluate the necessity to have the TS
               routed through the CICAM */
            if ((path = STB_DPGetLivePath()) != INVALID_RES_ID)
            {
               if ((tuner_path = STB_DPGetPathTuner(path)) != INVALID_RES_ID)
               {
                  STB_CIRouteTS(tuner_path, slot_state->slot_id, TRUE);
               }
            }
            break;
         }
         case STB_CI_CC_AUTH_ERROR:
         case STB_CI_CC_STATUS_AUTHENTICATION_FAILED:
         case STB_CI_CC_PROTOCOL_ERROR:
         {
            slot_state->auth_status = CI_AUTH_FAIL;
            break;
         }
         case STB_CI_CC_VERIFIED:
         {
            slot_state->auth_status = CI_AUTH_SUCCESS;
            slot_state->cicam_brand_id = cicam_brand_id;
            break;
         }
         default:
         {
            break;
         }
      }
   }

   STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_SLOT_STATUS_UPDATED, &slot_id, sizeof slot_id);

   FUNCTION_FINISH(STB_CINotifyProgress);
}


/**
 * @brief    Handle URI (usage rules information)
 * @param    slot_id - slot ID
 * @param    program_number - program number
 * @param    uri - the URI
 */
void STB_CINotifyURI(U8BIT slot_id, U16BIT program_number, S_STB_CI_URI *uri)
{
   S_SLOT_STATE *slot_state;

   FUNCTION_START(STB_CINotifyURI);

   slot_state = FindSlot(slot_id);
   if (slot_state != NULL)
   {
      DBGPRINT("Storing URI for program_number = %d", program_number)
      DBGPRINT("URI = {pv=%d, aps=%d, emi=%d, ict=%d, rct=%d, dot=%d, rl=%d",
           uri->protocol_version,
           uri->aps_copy_control_info,
           uri->emi_copy_control_info,
           uri->ict_copy_control_info,
           uri->rct_copy_control_info,
           uri->dot_copy_control_info,
           uri->rl_copy_control_info)
      slot_state->program_number = program_number;
      slot_state->uri = *uri;

      STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_URI_UPDATED, &slot_id, sizeof(slot_id));
   }

   FUNCTION_FINISH(STB_CINotifyURI);
}

/**
 * @brief   This function is called by the CI+ stack to send an SRM
 *          (System Renewability Message) to the host. SRM data files
 *          perform the function of blacklist for HDCP.
 * @param   slot_id zero-based CI slot identifier (0, 1, ...)
 * @param   srm_type the type of data in data field, for example DHCP or DTCP
 * @param   data SRM data
 * @param   len size of SRM data in bytes
 * @return  STB_CI_SRM_OK
 *          STB_CI_SRM_NO_CC_SUPPORT
 *          STB_CI_SRM_HOST_BUSY
 *          STB_CI_SRM_NOT_REQUIRED
 */

U8BIT STB_CINotifySRM(U8BIT slot_id, E_STB_SRM_DATA_TYPE srm_type, U8BIT *data, U16BIT len)
{
   U8BIT rc;
   E_STB_AV_SRM_REPLY srm_reply;

   FUNCTION_START(STB_CINotifySRM);
   USE_UNWANTED_PARAM(slot_id);

   if (srm_type == STB_CI_SRM_HDCP)
   {
      srm_reply = STB_AVApplySRM(0, data, len);
      switch (srm_reply)
      {
         case SRM_OK:
            rc = STB_CI_SRM_OK;
            break;
         case SRM_BUSY:
            rc = STB_CI_SRM_HOST_BUSY;
            break;
         case SRM_NOT_REQUIRED:
            rc = STB_CI_SRM_NOT_REQUIRED;
            break;
         default:
            /* Unknown response, use another */
            DBGPRINT("Unexpected reply 0x%x", srm_reply)
            rc = STB_CI_SRM_HOST_BUSY;
      }
   }
   else /*srm_type == STB_CI_SRM_DTCP*/
   {
      DBGPRINT("DTCP not supported")
      rc = STB_CI_SRM_NO_CC_SUPPORT;
   }

   FUNCTION_FINISH(STB_CINotifySRM);

   return rc;
}

/*****************************************************************************
 *
 * Function Name: STB_CINotifyPinCaps
 *
 * Description:   This function is called by the CI+ stack to send information
 *                about PIN code capabilities.
 *
 * Parameters:    slot_id - zero-based CI slot identifier (0, 1, ...)
 *                capability - PIN code capability
 *                pin_date - date of last PIN update
 *                pin_time - time of last PIN update
 *                rating - current rating for CICAM to request PIN
 *
 * Returns:       Nothing
 *
 ****************************************************************************/
void STB_CINotifyPinCaps(U8BIT slot_id, E_STB_CI_PIN_CAPS capability,
   S_STB_CI_DATE *pin_date, S_STB_CI_TIME *pin_time,
   U8BIT rating)
{
   S_SLOT_STATE *slot_state;

   FUNCTION_START(STB_CINotifyPinCaps);

   DBG_CC("(slot=%u, caps=%u, rating=%u)", slot_id, capability, rating)

   if ((slot_state = FindSlot(slot_id)) != NULL)
   {
      if (slot_state->pin_caps_requested)
      {
         slot_state->age_rating = rating;
         slot_state->pin_caps = capability;
         slot_state->pin_caps_received = TRUE;
         slot_state->pin_date = *pin_date;
         slot_state->pin_time = *pin_time;
      }
#ifdef DEBUG_CC
      else
      {
         DBG_CC("%s(%u): PIN capabilities weren't requested!", __FUNCTION__, slot_id)
      }
#endif
   }

   FUNCTION_FINISH(STB_CINotifyPinCaps);
}

/**
 * @brief   Called by the host to check whether a CAM pin is valid.
 *          An STB_EVENT_CI_PIN_STATUS event will be sent to notify the host
 *          of the validity, or otherwise, of the pin.
 * @param   slot_id - slot
 * @param   pin_data - ASCII encoded pin data, null terminated
 * @return  TRUE if the pin was sent successfully, FALSE otherwise
 */
BOOLEAN STB_CiCcSendPin(U8BIT slot_id, U8BIT *pin_data)
{
   BOOLEAN retval;
   S_SLOT_STATE *slot_state;

   FUNCTION_START(STB_CiCcSendPin);

   retval = FALSE;

   DBG_CC("(slot=%u, pin=\"%s\")", slot_id, pin_data)

   if (((slot_state = FindSlot(slot_id)) != NULL) &&
       slot_state->pin_caps_received && (slot_state->pin_caps != STB_CI_PIN_CAPS_NONE))
   {
      slot_state->play_pin_sent = FALSE;
      retval = STB_CISendPinCmd(slot_id, pin_data);
   }

   FUNCTION_FINISH(STB_CiCcSendPin);

   return(retval);
}

/*****************************************************************************
 *
 * Function Name: STB_CINotifyPinReply
 *
 * Description:   This function is called by the CI+ stack to send a PIN
 *                status to the host. This function is called in response
 *                to the host calling STB_CISendPinCmd or STB_CISendPinPlayback.
 *
 * Parameters:    slot_id - zero-based CI slot identifier (0, 1, ...)
 *                status - PIN code status
 *
 * Returns:       Nothing
 *
 ****************************************************************************/
void STB_CINotifyPinReply(U8BIT slot_id, E_STB_CI_PIN_STATUS status)
{
   S_SLOT_STATE *slot_state;

   FUNCTION_START(STB_CINotifyPinReply);

   DBG_CC("(slot=%u, status=%d)", slot_id, status)

   if ((slot_state = FindSlot(slot_id)) != NULL)
   {
      /* Notify the host of the PIN result */
      if (slot_state->play_pin_sent)
      {
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_PLAYBACK_PIN_STATUS,
            &status, sizeof(E_STB_CI_PIN_STATUS));
      }
      else
      {
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_PIN_STATUS,
            &status, sizeof(E_STB_CI_PIN_STATUS));
      }
   }

   FUNCTION_FINISH(STB_CINotifyPinReply);
}

/*****************************************************************************
 *
 * Function Name: STB_CINotifyPinEvent
 *
 * Description:   This function is called by the CI+ stack to send a PIN
 *                event to the host. This function is called when PIN
 *                requirements change and to acknowledge the PIN code.
 *
 * Parameters:    slot_id - zero-based CI slot identifier (0, 1, ...)
 *                program_number - program number
 *                status - PIN code status
 *                rating - age rating of new program (3+rating=age)
 *                event_date - date of event
 *                event_time - time of event
 *                private_data - event private data (15 bytes)
 *
 * Returns:       Nothing
 *
 ****************************************************************************/
void STB_CINotifyPinEvent(U8BIT slot_id, U16BIT program_number,
   E_STB_CI_PIN_STATUS status, U8BIT rating,
   S_STB_CI_DATE *event_date, S_STB_CI_TIME *event_time,
   U8BIT private_data[CIP_PIN_PRIVATE_DATA_SIZE])
{
   S_SLOT_STATE *slot_state;

   FUNCTION_START(STB_CINotifyPinEvent);

   DBG_CC("(slot=%u, program=%u, status=%u, date=%u/%u/%u, time=%u:%02u:%02u, rating=%u)",
           slot_id, program_number, status, event_date->day, event_date->month,
           event_date->year, event_time->hour, event_time->minute, event_time->second, rating)

   if ((slot_state = FindSlot(slot_id)) != NULL)
   {
      /* Only handle pin events for the service being recorded */
      if (slot_state->recording && (program_number == slot_state->record_service_id))
      {
         /* Save the rating and private data so they can be requested for saving with the recording
          * and send an event to notify the host */
         slot_state->record_pin_status = status;
         slot_state->record_age_rating = rating;
         slot_state->pin_event_date = *event_date;
         slot_state->pin_event_time = *event_time;
         memcpy(slot_state->record_private_data, private_data, CIP_PIN_PRIVATE_DATA_SIZE);

         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_RECORD_PIN, &slot_id, sizeof(slot_id));
      }
   }

   FUNCTION_FINISH(STB_CINotifyPinEvent);
}

/**
 * @brief   Returns the information to be stored with a pin event when recording
 * @param   slot_id - slot being used for the recording
 * @param   status - pin status code
 * @param   age_rating - returns the age rating supplied with the pin event
 * @param   private_data - returns a pointer to the private data to be saved - don't free
 * @param   date_code - MJD UTC date code when the pin is to be applied
 * @param   hour - UTC hour when the pin is to be applied
 * @param   min - UTC minute when the pin is to be applied
 * @param   sec - UTC second when the pin is to be applied
 * @return  TRUE if the info is returned, FALSE otherwise
 */
BOOLEAN STB_CiCcGetRecordingPinInfo(U8BIT slot_id, U8BIT *status, U8BIT *age_rating,
   U8BIT **private_data, U16BIT *date_code, U8BIT *hour, U8BIT *min, U8BIT *sec)
{
   BOOLEAN retval;
   S_SLOT_STATE *slot_state;

   FUNCTION_START(STB_CiCcGetRecordingPinInfo);

   retval = FALSE;

   if ((slot_state = FindSlot(slot_id)) != NULL)
   {
      if (slot_state->recording)
      {
         *status = slot_state->record_pin_status;
         *age_rating = slot_state->record_age_rating;
         *private_data = slot_state->record_private_data;
         *date_code = slot_state->pin_event_date.mjd;
         *hour = slot_state->pin_event_time.hour;
         *min = slot_state->pin_event_time.minute;
         *sec = slot_state->pin_event_time.second;
         retval = TRUE;
      }
   }

   FUNCTION_FINISH(STB_CiCcGetRecordingPinInfo);

   return(retval);
}

/**
 * @brief   Sends a pin event to the CAM during playback
 * @param   slot_id - slot
 * @param   age_rating - rating as provided by the pin event during recording
 * @param   private_data - pin private data as provided by the pin event during recording
 * @return  TRUE on success, FALSE otherwise
 */
BOOLEAN STB_CiCcSendPinPlayback(U8BIT slot_id, U8BIT age_rating, U8BIT *private_data)
{
   BOOLEAN retval;
   S_SLOT_STATE *slot_state;

   FUNCTION_START(STB_CiCcSendPinPlayback);

   retval = FALSE;

   DBG_REC("%s(slot=%u, age=%u)", __FUNCTION__, slot_id, age_rating)

   if ((slot_state = FindSlot(slot_id)) != NULL)
   {
      slot_state->play_pin_sent = TRUE;
      retval = STB_CISendPinPlayback(slot_id, age_rating, private_data);
   }

   FUNCTION_FINISH(STB_CiCcSendPinPlayback);

   return(retval);
}

/**
 * @brief   Sets the record operating mode for the given slot id
 * @param   slot_id - slot to be used for the recording
 * @param   mode - mode in which the host is in
 * @param   service_id -
 */
void STB_CiCcSetRecordOperatingMode(U8BIT slot_id, U32BIT mode, U16BIT service_id)
{//E_STB_CI_OPERATING_MODE
   S_SLOT_STATE *slot_state;

   FUNCTION_START(STB_CiCcSetRecordOperatingMode);

   DBG_REC("%s(slot=%u, mode=%s, service_id=%u)", __FUNCTION__, slot_id,
            ((mode == STB_CI_MODE_WATCH_AND_BUFFER) ? "WATCH_AND_BUFFER" :
             (mode == STB_CI_MODE_TIMESHIFT) ? "TIMESHIFT" :
             (mode == STB_CI_MODE_UNATTENDED_RECORDING) ? "UNATTENDED" : "UNKNOWN!"), service_id)

   if (mode <= STB_CI_MODE_UNATTENDED_RECORDING)
   {
      if ((slot_state = FindSlot(slot_id)) != NULL)
      {
         /* Only need to do this for CI+ v1.3 and above */
         if ((STB_CIGetContentControlVersion(slot_id) >= 2) &&
             (!slot_state->op_mode_set || (mode != slot_state->operating_mode)))
         {
            DBG_REC("%s: Current mode %s", __FUNCTION__,
                     ((slot_state->operating_mode == STB_CI_MODE_WATCH_AND_BUFFER) ? "WATCH_AND_BUFFER" :
                      (slot_state->operating_mode == STB_CI_MODE_TIMESHIFT) ? "TIMESHIFT" :
                      (slot_state->operating_mode == STB_CI_MODE_UNATTENDED_RECORDING) ? "UNATTENDED" : "UNKNOWN!"))

            /* Inform the CAM about the change of mode */
            if (slot_state->recording == TRUE)
            {
               slot_state->op_mode_set = STB_CIChangeOperatingMode(slot_id, mode, service_id);
            }
         }

         slot_state->operating_mode = mode;
      }
   }

   FUNCTION_FINISH(STB_CiCcSetRecordOperatingMode);
}

/**
 * @brief   Called by the app when a recording is to be started on a CA protected service
 * @param   slot_id - slot being used for the recording
 * @param   program_number - service ID
 * @param   pin_string - pin as a null terminated ascii string
 * @return  TRUE on success, but this doesn't mean the recording can start, FALSE otherwise
 */
BOOLEAN STB_CiCcSendRecordStart(U8BIT slot_id, U16BIT program_number, U8BIT *pin_string)
{
   S_SLOT_STATE *slot_state;
   BOOLEAN retval;
   FUNCTION_START(STB_CiCcSendRecordStart);

   retval = FALSE;

   DBG_REC("%s(slot=%u, program=%u, pin=\"%s\")", __FUNCTION__, slot_id, program_number, pin_string)

   if ((slot_state = FindSlot(slot_id)) != NULL)
   {
      slot_state->recording = TRUE;
      slot_state->record_service_id = program_number;

      /* Only need to do this for CI+ v1.3 and above */
      if ((STB_CIGetContentControlVersion(slot_id) >= 2) &&
          slot_state->pin_caps_received && (slot_state->pin_caps != STB_CI_PIN_CAPS_NONE))
      {
         DBG_REC("%s(%u): Asking CAM to start recording in %s mode using pin \"%s\"",
                  __FUNCTION__, slot_id,
                  ((slot_state->operating_mode == STB_CI_MODE_WATCH_AND_BUFFER) ? "WATCH_AND_BUFFER" :
                   (slot_state->operating_mode == STB_CI_MODE_TIMESHIFT) ? "TIMESHIFT" :
                   (slot_state->operating_mode == STB_CI_MODE_UNATTENDED_RECORDING) ? "UNATTENDED" : "UNKNOWN"),
                  pin_string)

         retval = STB_CISendRecordStart(slot_id, slot_state->operating_mode, program_number, pin_string);
         if (!retval)
         {
            DBG_REC("%s(%u): Failed to send record start to CAM", __FUNCTION__, slot_id)
            slot_state->recording = FALSE;
            slot_state->record_service_id = 0xFFFF;
         }
      }
      else
      {
         /* The recording can be started */
         DBG_REC("%s(%u): No need to ask CAM, recording can start", __FUNCTION__, slot_id)

         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_RECORD_START, &slot_id, sizeof(slot_id));
         retval = TRUE;
      }
   }

   FUNCTION_FINISH(STB_CiCcSendRecordStart);

   return(retval);
}

/**
 *
 * Function Name: STB_CINotifyRecordStartStatus
 *
 * Description:   This function is called by the CI+ stack to send the record
 *                start status to the host. This function should be called in
 *                response to STB_CISendRecordStart.
 *
 * Parameters:    slot_id - zero-based CI slot identifier (0, 1, ...)
 *                status - record start status
 *
 * Returns:       Nothing
 *
 */
void STB_CINotifyRecordStartStatus(U8BIT slot_id, U8BIT status)
{
   FUNCTION_START(STB_CINotifyRecordStartStatus);

   DBG_REC("%s(slot=%u, status=%u)", __FUNCTION__, slot_id, status)

   if (status == STB_CI_CC_STATUS_OK)
   {
      /* The recording can be started */
      STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_RECORD_START, &slot_id, sizeof(slot_id));
   }
   else
   {
      /* Recording shouldn't be started */
      STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_RECORD_START_FAILED, &slot_id, sizeof(slot_id));
   }

   FUNCTION_FINISH(STB_CINotifyRecordStartStatus);
}

/**
 *
 * Function Name: STB_CINotifyRecordingLicence
 *
 * Description:   This function is called by the CI+ stack to send the licence
 *                status, the URI and the optional licence data to the host.
 *
 *                This function may be called during a recording (after
 *                a call to STB_CISendRecordStart and before a call to
 *                STB_CISendRecordStop).
 *
 *                The licence must be stored on the host and sent during
 *                playback to the module using CIP_SendCicamLicence.
 *
 * Parameters:    slot_id - zero-based CI slot identifier (0, 1, ...)
 *                program_number - the program number
 *                licence_status - licence status
 *                uri - usage rules structure for the recording
 *                licence - the CICAM licence (or NULL if not required)
 *                licence_len - licence length in bytes
 *
 * Returns:       Nothing
 *
 */
void STB_CINotifyRecordingLicense(U8BIT slot_id, U16BIT program_number,
   U8BIT licence_status, S_STB_CI_URI *uri,
   U8BIT *licence, U16BIT licence_len)
{
   S_SLOT_STATE *slot_state;

   FUNCTION_START(STB_CINotifyRecordingLicense);

   USE_UNWANTED_PARAM(licence_status);

   DBG_REC("%s(slot=%u, program=%u, licence_status=%u, uri=%p, licence=%p, licence_len=%u)",
            __FUNCTION__, slot_id, program_number, licence_status, uri, licence, licence_len)

   if ((slot_state = FindSlot(slot_id)) != NULL)
   {
      if (slot_state->recording && (slot_state->record_service_id == program_number))
      {
         /* Save the URI so it can be requested by the host */
         slot_state->recording_uri = *uri;

         if ((slot_state->recording_licence != NULL) &&
             (slot_state->recording_licence_len != licence_len))
         {
            /* Licence length has changed so free the current one */
            STB_FreeMemory(slot_state->recording_licence);
            slot_state->recording_licence = NULL;
            slot_state->recording_licence_len = 0;
         }

         /* Save the licence and inform the host of the licence update */
         if (slot_state->recording_licence == NULL)
         {
            if ((slot_state->recording_licence = STB_GetMemory(licence_len)) != NULL)
            {
               slot_state->recording_licence_len = licence_len;
            }
         }

         if (slot_state->recording_licence != NULL)
         {
            slot_state->recording_licence_status = licence_status;

            memcpy(slot_state->recording_licence, licence, licence_len);

            /* Inform the host of the licence update */
            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_RECORD_LICENCE_UPDATED, &slot_id,
               sizeof(slot_id));
         }
      }
#ifdef DEBUG_REC
      else
      {
         if (!slot_state->recording)
         {
            DBG_REC("%s(%u): Slot isn't being used for recording", __FUNCTION__, slot_id)
         }
         else
         {
            DBG_REC("%s(%u): Slot is recording program %u", __FUNCTION__, slot_id, slot_state->record_service_id)
         }
      }
#endif
   }

   FUNCTION_FINISH(STB_CINotifyRecordingLicense);
}

/**
 * @brief   Returns the last licence received from the CAM when recording
 * @param   slot_id - slot
 * @param   licence_status - status value associated with the licence
 * @param   licence_len - pointer to return the licence length in bytes
 * @param   raw_uri - returns with the packed URI data associated with the licence
 * @return  pointer to the licence data, mustn't be freed. NULL if no licence
 */
U8BIT* STB_CiCcGetRecordingLicence(U8BIT slot_id, U8BIT *licence_status, U16BIT *licence_len,
   U8BIT raw_uri[CIP_URI_LEN])
{
   U8BIT *licence;
   S_SLOT_STATE *slot_state;

   FUNCTION_START(STB_CiCcGetRecordingLicence);

   licence = NULL;

   if ((slot_state = FindSlot(slot_id)) != NULL)
   {
      licence = slot_state->recording_licence;
      *licence_status = slot_state->recording_licence_status;
      *licence_len = slot_state->recording_licence_len;
      PackUri(&slot_state->recording_uri, raw_uri);
   }

   FUNCTION_FINISH(STB_CiCcGetRecordingLicence);

   return(licence);
}

/**
 * @brief   Sends a CICAM licence to a module during playback, which will result
 *          in a modified licence being notified through STB_CINotifyCicamLicence
 * @param   slot_id - slot
 * @param   program_number - program number being played back
 * @param   licence - CICAM licence
 * @param   licence_len - licence length in bytes
 * @return  TRUE if operation succeeded, FALSE otherwise
 */
BOOLEAN STB_CiCcSendPlaybackLicence(U8BIT slot_id, U16BIT program_number, U8BIT *licence,
   U16BIT licence_len)
{
   BOOLEAN retval;
   S_SLOT_STATE *slot_state;

   FUNCTION_START(STB_CiCcSendPlaybackLicence);

   DBG_REC("%s(slot=%u, program=%u, licence=%p, licence_len=%u)", __FUNCTION__, slot_id,
            program_number, licence, licence_len)

   retval = FALSE;

   if ((slot_state = FindSlot(slot_id)) != NULL)
   {
      slot_state->play_service_id = program_number;
      retval = STB_CISendPlaybackLicense(slot_id, program_number, licence, licence_len);
      if (!retval)
      {
         slot_state->play_service_id = 0xFFFF;
         DBG_REC("%s(%u): Failed to send playback licence", __FUNCTION__, slot_id)
      }
   }

   FUNCTION_FINISH(STB_CiCcSendPlaybackLicence);

   return(retval);
}

/**
 *
 * Function Name: STB_CINotifyPlaybackLicense
 *
 * Description:   This function is called by the CI+ stack to send the licence
 *                status, the URI and the optional licence data to the host.
 *
 *                The function is called in response to a playback start
 *                message from the host (sent using STB_CISendPlaybackLicence).
 *
 *                The licence and URI replace the ones that were stored
 *                with the recording. If the licence was updated, the new
 *                licence will have the same size as the original licence.
 *
 * Parameters:    slot_id - zero-based CI slot identifier (0, 1, ...)
 *                program_number - the program number
 *                licence_status - licence status
 *                uri - usage rules structure for the recording
 *                licence - the CICAM licence (or NULL if not required)
 *                licence_len - licence length in bytes
 *
 * Returns:       Nothing
 *
 */
void STB_CINotifyPlaybackLicense(U8BIT slot_id, U16BIT program_number,
   U8BIT licence_status, S_STB_CI_URI *uri,
   U8BIT *licence, U16BIT licence_len)
{
   S_SLOT_STATE *slot_state;

   FUNCTION_START(STB_CINotifyPlaybackLicense);

   USE_UNWANTED_PARAM(licence_status);

   DBG_REC("%s(slot=%u, program=%u, licence_status=%u, uri=%p, licence=%p, licence_len=%u)",
            __FUNCTION__, slot_id, program_number, licence_status, uri, licence, licence_len)

   if ((slot_state = FindSlot(slot_id)) != NULL)
   {
      if (slot_state->play_service_id == program_number)
      {
         /* Save the URI so it can be requested by the host */
         slot_state->playback_uri = *uri;

         if ((slot_state->playback_licence != NULL) &&
             (slot_state->playback_licence_len != licence_len))
         {
            /* Licence length has changed so free the current one */
            STB_FreeMemory(slot_state->playback_licence);
            slot_state->playback_licence = NULL;
            slot_state->playback_licence_len = 0;
         }

         /* Save the licence and inform the host of the licence update */
         if (slot_state->playback_licence == NULL)
         {
            if ((slot_state->playback_licence = STB_GetMemory(licence_len)) != NULL)
            {
               slot_state->playback_licence_len = licence_len;
            }
         }

         if (slot_state->playback_licence != NULL)
         {
            slot_state->playback_licence_status = licence_status;

            memcpy(slot_state->playback_licence, licence, licence_len);

            /* Inform the host of the licence update */
            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_PLAYBACK_LICENCE_UPDATED, &slot_id,
               sizeof(slot_id));
         }
      }
      else
      {
         DBG_REC("%s(%u): Slot is playing program %u", __FUNCTION__, slot_id, slot_state->play_service_id)
      }
   }

   FUNCTION_FINISH(STB_CINotifyPlaybackLicense);
}

/**
 * @brief   Returns the last licence received from the CAM during playback
 * @param   slot_id - slot
 * @param   licence_status - status value associated with the licence
 * @param   licence_len - pointer to return the licence length in bytes
 * @param   raw_uri - returns with the packed URI data associated with the licence
 * @return  pointer to the licence data, mustn't be freed. NULL if no licence
 */
U8BIT* STB_CiCcGetPlaybackLicence(U8BIT slot_id, U8BIT *licence_status, U16BIT *licence_len,
   U8BIT raw_uri[CIP_URI_LEN])
{
   U8BIT *licence;
   S_SLOT_STATE *slot_state;

   FUNCTION_START(STB_CiCcGetPlaybackLicence);

   licence = NULL;

   if ((slot_state = FindSlot(slot_id)) != NULL)
   {
      licence = slot_state->playback_licence;
      *licence_status = slot_state->playback_licence_status;
      *licence_len = slot_state->playback_licence_len;
      PackUri(&slot_state->playback_uri, raw_uri);
   }

   FUNCTION_FINISH(STB_CiCcGetPlaybackLicence);

   return(licence);
}

/**
 *
 * Function Name: STB_CINotifyLicenseStatus
 *
 * Description:   This function is called by the CI+ stack to send the licence
 *                status and the remaining number of plays that the user is
 *                entitled to.
 *
 *                The function should be called in response to a call to
 *                CIP_RequestLicenceCheck.
 *
 * Parameters:    slot_id - zero-based CI slot identifier (0, 1, ...)
 *                licence_status - the licence status
 *                play_count - number of plays remaining
 *
 * Returns:       Nothing
 *
 */
void STB_CINotifyLicenseStatus(U8BIT slot_id, U8BIT licence_status, U8BIT play_count)
{
   FUNCTION_START(STB_CINotifyLicenseStatus);

   DBG_REC("%s(slot=%u, licence_status=%u, play_count=%u)", __FUNCTION__, slot_id, licence_status,
            play_count)

   USE_UNWANTED_PARAM(slot_id);
   USE_UNWANTED_PARAM(licence_status);
   USE_UNWANTED_PARAM(play_count);

   FUNCTION_FINISH(STB_CINotifyLicenseStatus);
}

/**
 *
 * Function Name: STB_CINotifyModeChangeStatus
 *
 * Description:   This function is called by the CI+ stack to send the mode
 *                change status to the host. This function should be called in
 *                response to CIP_ChangeOperatingMode.
 *
 * Parameters:    slot_id - zero-based CI slot identifier (0, 1, ...)
 *                status - mode change status
 *
 * Returns:       Nothing
 *
 */
void STB_CINotifyModeChangeStatus(U8BIT slot_id, U8BIT status)
{
   FUNCTION_START(STB_CINotifyModeChangeStatus);

   DBG_REC("%s(slot=%u, status=%u)", __FUNCTION__, slot_id, status)

   USE_UNWANTED_PARAM(slot_id);
   USE_UNWANTED_PARAM(status);

   FUNCTION_FINISH(STB_CINotifyModeChangeStatus);
}

/**
 * @brief   Called by the app when a recording is stopped or completes
 * @param   slot_id - slot being used for the recording
 * @return  TRUE on success, FALSE otherwise
 */
BOOLEAN STB_CiCcSendRecordStop(U8BIT slot_id)
{
   BOOLEAN retval;
   S_SLOT_STATE *slot_state;

   FUNCTION_START(STB_CiCcSendRecordStop);

   retval = FALSE;

   DBG_REC("%s(slot=%u)", __FUNCTION__, slot_id)

   if ((slot_state = FindSlot(slot_id)) != NULL)
   {
      if (slot_state->recording)
      {
         if ((STB_CIGetContentControlVersion(slot_id) >= 2) &&
             slot_state->pin_caps_received && (slot_state->pin_caps != STB_CI_PIN_CAPS_NONE))
         {
            DBG_REC("%s(%u): Sending record stop to CAM", __FUNCTION__, slot_id)

            /* Inform the CAM that the recording has stopped */
            retval = STB_CISendRecordStop(slot_id, slot_state->record_service_id);
#ifdef DEBUG_REC
            if (!retval)
            {
               DBG_REC("%s(%u): Failed to send record stop to CAM", __FUNCTION__, slot_id)
            }
#endif
         }
         else
         {
            DBG_REC("%s(%u): No need to inform CAM", __FUNCTION__, slot_id)
            retval = TRUE;
         }

         slot_state->recording = FALSE;
      }
   }

   FUNCTION_FINISH(STB_CiCcSendRecordStop);

   return(retval);
}

/**
 *
 * Function Name: STB_CINotifyRecordStopStatus
 *
 * Description:   This function is called by the CI+ stack to send the record
 *                stop status to the host. This function should be called in
 *                response to STB_CISendRecordStop.
 *
 * Parameters:    slot_id - zero-based CI slot identifier (0, 1, ...)
 *                status - record stop status
 *
 * Returns:       Nothing
 *
 */
void STB_CINotifyRecordStopStatus(U8BIT slot_id, U8BIT status)
{
   FUNCTION_START(STB_CINotifyRecordStopStatus);

   DBG_REC("%s(slot=%u, status=%u)", __FUNCTION__, slot_id, status)

   USE_UNWANTED_PARAM(slot_id);
   USE_UNWANTED_PARAM(status);

   FUNCTION_FINISH(STB_CINotifyRecordStopStatus);
}


/*---local function definitions----------------------------------------------*/

static S_SLOT_STATE *FindSlot(U8BIT slot_id)
{
   S_SLOT_STATE *slot_info;
   STB_OSMutexLock(cicc_mutex);
   slot_info = cicc_slots;
   while (slot_info != NULL)
   {
      if (slot_info->slot_id == slot_id)
      {
         if (slot_info->disabled)
         {
            slot_info = NULL;
         }
         break;
      }
      slot_info = slot_info->next;
   }
   STB_OSMutexUnlock(cicc_mutex);
   return slot_info;
}

/**
 * @brief    Tell whether the service is allowed
 *
 * This function checks whether the ci_protection descriptor allows the
 * service to be routed through the module.
 *
 * @param    slot_state - Slot state information
 * @param    ci_protection_descriptor - Raw CI protection descriptor
 * @return   TRUE if the service is allowed, FALSE otherwise
 */
static BOOLEAN IsServiceAllowed(S_SLOT_STATE *slot_state,
   U8BIT *ci_protection_descriptor)
{
   BOOLEAN allowed;
   U8BIT i;

   FUNCTION_START(IsServiceAllowed);

   allowed = FALSE;

   if (ci_protection_info.sdt_acquisition_complete)
   {
      ParseCiProtectionDescriptor(ci_protection_descriptor);
      if (ci_protection_info.free_ci_mode_flag == FALSE)
      {
         /* No restriction */
         DBGPRINT("free_ci_mode_flag == FALSE: allowed = TRUE")
         allowed = TRUE;
      }
      else if (slot_state->auth_status == CI_AUTH_SUCCESS)
      {
         if (ci_protection_info.match_brand_flag == FALSE)
         {
            /* CI+ only, but no brand ID matching */
            DBGPRINT("CI+ only, no brand matching, authenticated: allowed = TRUE")
            allowed = TRUE;
         }
         else
         {
            for (i = 0; i < ci_protection_info.number_of_entries; i++)
            {
               if (slot_state->cicam_brand_id == ci_protection_info.entries[i])
               {
                  /* Found CICAM brand match */
                  DBGPRINT("CI+ only, brand found: allowed = TRUE")
                  allowed = TRUE;
                  break;
               }
            }
         }
      }
   }

   if (!allowed)
   {
      DBGPRINT("allowed = FALSE")
   }

   FUNCTION_FINISH(IsServiceAllowed);

   return allowed;
}

/**
 * @brief    Parse ci_protection_descriptor
 * @param    ci_protection_descriptor - ci_protection_descriptor
 * @return   Nothing
 */
static void ParseCiProtectionDescriptor(U8BIT *ci_protection_descriptor)
{
   U8BIT *d;
   U8BIT tag;
   U8BIT len;
   U8BIT number_of_entries;
   U16BIT offset;
   U16BIT *entry;

   FUNCTION_START(ParseCiProtectionDescriptor);

   if (ci_protection_descriptor == NULL)
   {
      /* No descriptor = no restriction */
      DBGPRINT("ci_protection_descriptor == NULL: no restriction")
      ci_protection_info.free_ci_mode_flag = FALSE;
      ci_protection_info.match_brand_flag = FALSE;
      ci_protection_info.number_of_entries = 0;
   }
   else
   {
      /* We assume that the descriptor is valid */
      d = ci_protection_descriptor;
      tag = d[0];
      len = d[1];

      DBGPRINT("ci_protection_descriptor: tag=%x, len=%d", tag, len)

      ci_protection_info.free_ci_mode_flag = TRUE;
      ci_protection_info.match_brand_flag = TRUE;
      ci_protection_info.number_of_entries = 0;

      if (tag == 0xCE && len >= 1)
      {
         DBGPRINT("d2=%x", d[2])
         if (((d[2] >> 7) & 0x1) == 0x0)
         {
            ci_protection_info.free_ci_mode_flag = FALSE;
         }
         else
         {
            ci_protection_info.free_ci_mode_flag = TRUE;
         }

         if (((d[2] >> 6) & 0x1) == 0x0)
         {
            ci_protection_info.match_brand_flag = FALSE;
         }
         else
         {
            ci_protection_info.match_brand_flag = TRUE;
            ci_protection_info.number_of_entries = 0;

            /* Service requires CI+ protection */
            if (len >= 2)
            {
               /* Parse valid brands (as much as possible) */
               number_of_entries = d[3];
               DBGPRINT("number_of_entries = %d", number_of_entries)
               entry = ci_protection_info.entries;
               offset = 2;
               while (ci_protection_info.number_of_entries < number_of_entries &&
                      offset + 1 < len)
               {
                  *entry = d[2 + offset] << 8 | d[2 + offset + 1];
                  DBGPRINT("entry: %04x", *entry)
                  ci_protection_info.number_of_entries++;
                  entry++;
                  offset += 2;
               }
            }
         }
      }
   }

   FUNCTION_FINISH(ParseCiProtectionDescriptor);
}

/**
 * @brief    Apply URI version 1
 * @param    uri - usage rules information
 * @param    playback - whether this is a playback URI
 * @return   Nothing
 */
static void ApplyURIv1(S_STB_CI_URI *uri, BOOLEAN playback)
{
   U8BIT aps, emi, rct, ict;
   S_STB_AV_COPY_PROTECTION cp;

   memset(&cp, 0, sizeof cp);

   /* If URI is NULL, copy protection is not required */
   if (uri != NULL)
   {
      aps = uri->aps_copy_control_info;
      emi = uri->emi_copy_control_info;
      rct = uri->rct_copy_control_info;
      ict = uri->ict_copy_control_info;

      /* Macrovision:
       * If rct=1 and/or emi=01,10,11, the value should be aps */
      cp.macrovision_set = TRUE;
      cp.macrovision = ((rct || emi) * 3) & aps;

      /* APS = Analog Protection System
       * If emi=01,10,11, the value should be aps */
      cp.aps_set = TRUE;
      cp.aps = ((emi != 0) * 3) & aps;

      /* CGMS-A = Copy Generation Management System - Analog */
      cp.cgms_a_set = TRUE;
      cp.cgms_a = emi;

      /* Note 6 in ILA */
      if ((rct == 0) && (emi == 0x2))
      {
         if (playback)
         {
            /* Macrovision is active, CGMS-A is "copy no more" */
            cp.cgms_a = 0x1;
         }
         else
         {
            /* Macrovision is inctive, CGMS-A is "copy once" */
            cp.macrovision = 0;
         }
      }

      /* Image constraint */
      cp.ict_set = TRUE;
      cp.ict = ict;

      /* HDCP */
      cp.hdcp_set = TRUE;
      cp.hdcp = TRUE;

      /* SCMS */
      cp.scms_set = TRUE;
      if (emi == 0)
      {
         cp.scms = 0x2;
      }
      else
      {
         cp.scms = emi & 0x1;
      }

      /* DOT */
      cp.dot_set = FALSE;
   }

   STB_AVSetCopyProtection(&cp);
}

/**
 * @brief    Apply URI version 2
 * @param    uri - usage rules information
 * @param    playback - whether this is a playback URI
 * @return   Nothing
 */
static void ApplyURIv2(S_STB_CI_URI *uri, BOOLEAN playback)
{
   U8BIT aps, emi, rct, ict;
   S_STB_AV_COPY_PROTECTION cp;

   memset(&cp, 0, sizeof cp);

   /* If URI is NULL, copy protection is not required */
   if (uri != NULL)
   {
      aps = uri->aps_copy_control_info;
      emi = uri->emi_copy_control_info;
      rct = uri->rct_copy_control_info;
      ict = uri->ict_copy_control_info;

      /* Macrovision:
       * If rct=1 and/or emi=01,10,11, the value should be aps */
      cp.macrovision_set = TRUE;
      cp.macrovision = ((rct || emi) * 3) & aps;

      /* APS = Analog Protection System
       * If emi=01,10,11, the value should be aps */
      cp.aps_set = TRUE;
      cp.aps = ((emi != 0) * 3) & aps;

      /* CGMS-A = Copy Generation Management System - Analog */
      cp.cgms_a_set = TRUE;
      cp.cgms_a = emi;

      /* Note 6 in ILA */
      if ((rct == 0) && (emi == 0x2))
      {
         if (playback)
         {
            /* Macrovision is active, CGMS-A is "copy no more" */
            cp.cgms_a = 0x1;
         }
         else
         {
            /* Macrovision is inctive, CGMS-A is "copy once" */
            cp.macrovision = 0;
         }
      }

      /* Image constraint */
      cp.ict_set = TRUE;
      cp.ict = ict;

      /* HDCP */
      cp.hdcp_set = TRUE;
      cp.hdcp = TRUE;

      /* SCMS */
      cp.scms_set = TRUE;
      if (emi == 0)
      {
         cp.scms = 0x2;
      }
      else
      {
         cp.scms = emi & 0x1;
      }

      /* DOT */
      cp.dot_set = TRUE;
      cp.dot = uri->dot_copy_control_info;
   }

   STB_AVSetCopyProtection(&cp);
}

/**
 * @brief    Apply invalid URI
 * @return   Nothing
 */
static void ApplyInvalidURI(void)
{
   S_STB_AV_COPY_PROTECTION cp;

   memset(&cp, 0, sizeof cp);
   STB_AVSetCopyProtection(&cp);
}

/**
 * @brief    Pack URI into 64-bit representation
 * @param    uri - URI to pack
 * @param    buffer - buffer to hold the packed URI
 * @return   Nothing
 */
static void PackUri(S_STB_CI_URI *uri, U8BIT buffer[CIP_URI_LEN])
{
   FUNCTION_START(PackUri);

   memset(buffer, 0x00, CIP_URI_LEN);

   if (uri != NULL)
   {
      if (uri->protocol_version == 0x01)
      {
         /* protocol_version */
         buffer[0] = 0x01;

         buffer[1] |= (uri->aps_copy_control_info & 0x3) << 6;
         buffer[1] |= (uri->emi_copy_control_info & 0x3) << 4;
         buffer[1] |= (uri->ict_copy_control_info & 0x1) << 3;
         buffer[1] |= (uri->rct_copy_control_info & 0x1) << 2;
         buffer[1] |= 0x3;

         buffer[2] |= 0xc0 << 6;
         buffer[2] |= (uri->rl_copy_control_info & 0x3f);

         /* reserved */
         buffer[3] = buffer[4] = buffer[5] = buffer[6] = buffer[7] = 0xff;
      }
      else if (uri->protocol_version == 0x02)
      {
         /* protocol_version */
         buffer[0] = 0x02;

         buffer[1] |= (uri->aps_copy_control_info & 0x3) << 6;
         buffer[1] |= (uri->emi_copy_control_info & 0x3) << 4;
         buffer[1] |= (uri->ict_copy_control_info & 0x1) << 3;

         if ((uri->emi_copy_control_info & 0x03) == 0x00)
         {
            buffer[1] |= (uri->rct_copy_control_info & 0x1) << 2;
         }

         /* reserved bit */
         buffer[1] |= 0x2;

         if ((uri->emi_copy_control_info & 0x03) == 0x03)
         {
            buffer[1] |= (uri->dot_copy_control_info & 0x01);
            buffer[2] = uri->rl_copy_control_info & 0xfe;
         }

         /* reserved */
         buffer[3] = buffer[4] = buffer[5] = buffer[6] = buffer[7] = 0xff;
      }
   }

   FUNCTION_FINISH(PackUri);
}

/**
 * @brief    Unpack URI from 64-bit representation
 * @param    buffer - buffer to unpack
 * @param    uri - unpacked URI
 * @return   Nothing
 */
static void UnpackUri(U8BIT buffer[CIP_URI_LEN], S_STB_CI_URI *uri)
{
   FUNCTION_START(UnpackUri);

   if (buffer[0] == 0x01)
   {
      /* Assume URI v1 */
      uri->protocol_version = buffer[0];

      uri->aps_copy_control_info = (buffer[1] >> 6) & 0x3;
      uri->emi_copy_control_info = (buffer[1] >> 4) & 0x3;
      uri->ict_copy_control_info = (buffer[1] >> 3) & 0x1;
      uri->rct_copy_control_info = (buffer[1] >> 2) & 0x1;

      uri->rl_copy_control_info = buffer[2] & 0x3f;
   }
   else if (buffer[0] == 0x02)
   {
      /* URI v2 */
      uri->protocol_version = buffer[0];

      uri->aps_copy_control_info = (buffer[1] >> 6) & 0x3;
      uri->emi_copy_control_info = (buffer[1] >> 4) & 0x3;
      uri->ict_copy_control_info = (buffer[1] >> 3) & 0x1;
      uri->rct_copy_control_info = (buffer[1] >> 2) & 0x1;
      uri->dot_copy_control_info = buffer[1] & 0x1;

      uri->rl_copy_control_info = buffer[2] & 0xfe;
   }
   else
   {
      /* Invalid URI */
      uri->protocol_version = 0x00;
   }

   FUNCTION_FINISH(UnpackUri);
}

