/*******************************************************************************
 * Copyright  2016 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2009 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief    CI Content Control - descrambler keys
 * @file     stbcikeys.c
 * @date     28 November 2016
 * @author   Adam Sturtridge
 */
//#define DEBUG_PRINTING_ENABLED

/*---includes for this file--------------------------------------------------*/

/* compiler library header files */
#include <string.h>

/* third party header files */

/* DVBCore header files */
#include "techtype.h"
#include "dbgfuncs.h"
#include "stbhwos.h"
#include "stbci.h"
#include "stberc.h"
#include "stbdpc.h"
#include "stbpvr.h"
#include "stbheap.h"
#ifdef CI_TA_HANDLES_KEYS
#include "stbhwci.h"
#endif

#include "stbcikeys.h"
#include "stbci_int.h"

/*---constant definitions for this file--------------------------------------*/

#define EVEN_KEY_REGISTER     0
#define ODD_KEY_REGISTER      1

/*---local typedef enums for this file-------------------------------------*/

/*---local typedef structs for this file-------------------------------------*/

typedef struct key_info
{
   BOOLEAN valid;
   BOOLEAN *applied;
   U8BIT key[16];
   U8BIT civ[16];
} S_KEY_INFO;

typedef struct s_cikeys_info
{
   struct s_cikeys_info *next;
   U8BIT slot_id;
   U8BIT cipher;
   BOOLEAN disabled;
   S_KEY_INFO keys[2];
} S_CIKEYS_INFO;

#ifdef CI_TA_HANDLES_KEYS
typedef struct s_cikey_avail
{
   BOOLEAN notify;
   U32BIT obj_ref;
} S_CIKEY_AVAIL;
#endif

/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local) */

static BOOLEAN *desc_allocated = NULL;
static S_CIKEYS_INFO *cikeys_slots = NULL;
static void *cikeys_mutex;
#ifdef CI_TA_HANDLES_KEYS
static S_CIKEY_AVAIL *cikeys_avail = NULL;
#endif

static S_CIKEYS_INFO *FindSlot(U8BIT slot_id);
static void AllocateDescramblers(U8BIT path, U8BIT cipher);
static BOOLEAN SetDescramblerKey(U8BIT path, E_STB_DMX_DESC_KEY_PARITY parity,
   U8BIT cipher, S_KEY_INFO *key_info);


/*---global function definitions---------------------------------------------*/

/**
 * @brief    CI descrambler keys support initialise
 */
void STB_CiKeysInitialise(void)
{
   cikeys_mutex = STB_OSCreateMutex();
}

/**
 * @brief   Disable CC keys for slot, as CAM has been removed
 * @param   slot_id zero-based CI slot identifier (0, 1, ...)
 */
void STB_CiKeysDisable(U8BIT slot_id)
{
   S_CIKEYS_INFO *slot_info;

   FUNCTION_START(STB_CiKeysDisable);

   DBGPRINT("(slot=%u)", slot_id)

   slot_info = FindSlot(slot_id);
   if (slot_info != NULL)
   {
      slot_info->disabled = TRUE;
   }
   FUNCTION_FINISH(STB_CiKeysDisable);
}

/**
 * @brief   Notify module removal
 * @param   slot_id zero-based CI slot identifier (0, 1, ...)
 */
void STB_CiKeysRemove(U8BIT slot_id)
{
   S_CIKEYS_INFO *slot;
   S_CIKEYS_INFO **pslot;

   FUNCTION_START(STB_CiKeysRemove);
   DBGPRINT("(slot=%u)", slot_id)
   pslot = &cikeys_slots;
   STB_OSMutexLock(cikeys_mutex);
   slot = cikeys_slots;
   while (slot != NULL)
   {
      if (slot->slot_id == slot_id)
      {
         *pslot = slot->next;
         STB_FreeMemory(slot);
         break;
      }
      pslot = &(slot->next);
      slot = slot->next;
   }
   STB_OSMutexUnlock(cikeys_mutex);

   FUNCTION_FINISH(STB_CiKeysRemove);
}

/**
 * @brief   Handle content control keys
 * @param   slot_id zero-based CI slot identifier (0, 1, ...)
 * @param   cipher cipher to be used
 * @param   key_register key register to be used
 * @param   key AES/DES key (according to cipher)
 * @param   civ initialisation vector for cipher
 */
void STB_CINotifyCCKey(U8BIT slot_id, U8BIT cipher, U8BIT key_register,
   U8BIT *key, U8BIT *civ)
{
   S_CIKEYS_INFO *slot_info;
   U8BIT path, num_paths;

   FUNCTION_START(STB_CINotifyCCKey);

   DBGPRINT("slot_id = %u, cipher = %s, key_register = %u",
           slot_id,
           cipher == STB_CI_CIPHER_DES ? "STB_CI_CIPHER_DES" :
           cipher == STB_CI_CIPHER_AES ? "STB_CI_CIPHER_AES" :
           "UNKNOWN",
           key_register)

   slot_info = FindSlot(slot_id);
   num_paths = STB_DPGetNumPaths();
   if (slot_info == NULL)
   {
      slot_info = STB_GetMemory(sizeof(S_CIKEYS_INFO)+ (2 * num_paths * sizeof(BOOLEAN)));
      if (slot_info != NULL)
      {
         slot_info->disabled = FALSE;
         slot_info->keys[EVEN_KEY_REGISTER].applied = (BOOLEAN*)(slot_info + 1);
         memset(slot_info->keys[EVEN_KEY_REGISTER].applied, 0, 2 * num_paths * sizeof(BOOLEAN));
         slot_info->keys[ODD_KEY_REGISTER].applied = slot_info->keys[EVEN_KEY_REGISTER].applied + num_paths;
         slot_info->keys[EVEN_KEY_REGISTER].valid = FALSE;
         slot_info->keys[ODD_KEY_REGISTER].valid = FALSE;
         slot_info->slot_id = slot_id;
         DBGPRINT("Add slot info; id=%u", slot_id)

         STB_OSMutexLock(cikeys_mutex);
         slot_info->next = cikeys_slots;
         cikeys_slots = slot_info;
         STB_OSMutexUnlock(cikeys_mutex);

         if (desc_allocated == NULL)
         {
            desc_allocated = STB_GetMemory(num_paths * sizeof(BOOLEAN));
            if (desc_allocated != NULL)
            {
               for (path = 0; path != num_paths; path++)
               {
                  desc_allocated[path] = FALSE;
               }
            }
         }
      }
   }
   if (slot_info != NULL)
   {
      /* Store CC keys */
      if ((key_register == EVEN_KEY_REGISTER) ||
          (key_register == ODD_KEY_REGISTER))
      {
         slot_info->cipher = cipher;
         if (cipher == STB_CI_CIPHER_AES)
         {
            memcpy(slot_info->keys[key_register].key, key, 16);
            memcpy(slot_info->keys[key_register].civ, civ, 16);
         }
         else
         {
            memcpy(slot_info->keys[key_register].key, key, 8);
         }
         slot_info->keys[key_register].valid = TRUE;

         for (path = 0; path != num_paths; path++)
         {
            slot_info->keys[key_register].applied[path] = FALSE;
         }

         DBGPRINT("Send CI KEYS changed/updated slot=%u", slot_id)
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_KEYS_UPDATED, &slot_id, sizeof(U8BIT));
      }
   }

   FUNCTION_FINISH(STB_CINotifyCCKey);
}

#ifdef CI_TA_HANDLES_KEYS
/**
 * @brief   This function is called by the CI+ stack to signal availability
 *          of the content control keys required for decrypting the
 *          audio/video content encrypted by the module.
 *          The object reference object in secure NVM that contains cipher used (DES/AES),
 *          the key parity, as well as the key material and initialisation
 *          vector (if cipher is AES-CBC).
 * @param   slot_id zero-based CI slot identifier (0, 1, ...)
 * @param   obj_ref Object Reference for CCK
 */
void STB_CINotifyCckAvailable(U8BIT slot_id, U32BIT obj_ref)
{
   U8BIT num_slots;
   FUNCTION_START(STB_CINotifyCckAvailable);
   DBGPRINT("slot=%u oref=%x", slot_id, obj_ref)
   num_slots = STB_CIGetSlotCount();
   if (slot_id > num_slots)
   {
      DBGPRINT("Error: bad slot_id %u", slot_id)
   }
   else
   {
      if (cikeys_avail == NULL)
      {
         cikeys_avail = (S_CIKEY_AVAIL *)STB_GetMemory(num_slots * sizeof(S_CIKEY_AVAIL));
         if (cikeys_avail != NULL)
         {
            memset(cikeys_avail, 0, num_slots * sizeof(S_CIKEY_AVAIL));
         }
      }
      if (cikeys_avail != NULL)
      {
         cikeys_avail[slot_id].obj_ref = obj_ref;
         cikeys_avail[slot_id].notify = TRUE;

         DBGPRINT("Send CI KEYS changed/updated slot=%u", slot_id)
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_KEYS_UPDATED, &slot_id, sizeof(U8BIT));
      }
   }
   FUNCTION_FINISH(STB_CINotifyCckAvailable);
}

#endif /*CI_TA_HANDLES_KEYS*/

/**
 * @brief   Apply stored CC keys (if any) on given path
 * @param   path Path to apply keys on
 * @param   slot_id zero-based CI slot identifier (0, 1, ...)
 */
void STB_CiKeysApply(U8BIT path, U8BIT slot_id)
{
   S_CIKEYS_INFO *slot_info;

   FUNCTION_START(STB_CiKeysApply);

   DBGPRINT("(path=%u,slot=%u)", path, slot_id)

   slot_info = FindSlot(slot_id);
   if (slot_info != NULL)
   {
      if (((slot_info->keys[EVEN_KEY_REGISTER].valid) &&
           (!slot_info->keys[EVEN_KEY_REGISTER].applied[path])) ||
          ((slot_info->keys[ODD_KEY_REGISTER].valid) &&
           (!slot_info->keys[ODD_KEY_REGISTER].applied[path])))
      {
         if (STB_DPIsDecodingPath(path))
         {
            if ((desc_allocated != NULL) && (!desc_allocated[path]))
            {
               /* Allocate descramblers (if required) */
               DBGPRINT("Allocating descramblers")
               AllocateDescramblers(path, slot_info->cipher);
            }

            if ((desc_allocated != NULL) && (desc_allocated[path]))
            {
               if ((slot_info->keys[EVEN_KEY_REGISTER].valid) &&
                   (!slot_info->keys[EVEN_KEY_REGISTER].applied[path]))
               {
                  DBGPRINT("Setting descrambler even keys")
                  if (SetDescramblerKey(path, KEY_PARITY_EVEN, slot_info->cipher,
                         &slot_info->keys[EVEN_KEY_REGISTER]))
                  {
                     slot_info->keys[EVEN_KEY_REGISTER].applied[path] = TRUE;
                  }
               }
               if ((slot_info->keys[ODD_KEY_REGISTER].valid) &&
                   (!slot_info->keys[ODD_KEY_REGISTER].applied[path]))
               {
                  DBGPRINT("Setting descrambler odd keys")
                  if (SetDescramblerKey(path, KEY_PARITY_ODD, slot_info->cipher,
                         &slot_info->keys[ODD_KEY_REGISTER]))
                  {
                     slot_info->keys[ODD_KEY_REGISTER].applied[path] = TRUE;
                  }
               }
            }
         }

         if (STB_DPIsRecordingPath(path))
         {
            /* Need to let the PVR code know the keys so they can be used when recording */
            if ((slot_info->keys[EVEN_KEY_REGISTER].valid) &&
                (!slot_info->keys[EVEN_KEY_REGISTER].applied[path]))
            {
               DBGPRINT("(%u) Saving descrambler even keys", path)

               STB_PVRSetDescramblerKey(path, slot_info->cipher, KEY_PARITY_EVEN,
                  slot_info->keys[EVEN_KEY_REGISTER].key, slot_info->keys[EVEN_KEY_REGISTER].civ);
               slot_info->keys[EVEN_KEY_REGISTER].applied[path] = TRUE;
            }

            if ((slot_info->keys[ODD_KEY_REGISTER].valid) &&
                (!slot_info->keys[ODD_KEY_REGISTER].applied[path]))
            {
               DBGPRINT("(%u): Saving descrambler odd keys", path)

               STB_PVRSetDescramblerKey(path, slot_info->cipher, KEY_PARITY_ODD,
                  slot_info->keys[ODD_KEY_REGISTER].key, slot_info->keys[ODD_KEY_REGISTER].civ);
               slot_info->keys[ODD_KEY_REGISTER].applied[path] = TRUE;
            }
         }
      }
   }
#ifdef CI_TA_HANDLES_KEYS
   else if (cikeys_avail != NULL && cikeys_avail[slot_id].notify)
   {
      if (STB_DPIsDecodingPath(path) || STB_DPIsRecordingPath(path))
      {
         /* Tell CI+ Trusted App to inform demux of Content Control keys */
         STB_CISendCckToDemuxTa(slot_id, cikeys_avail[slot_id].obj_ref, path);
      }
      cikeys_avail[slot_id].notify = FALSE;
   }
#endif

   FUNCTION_FINISH(STB_CiKeysApply);
}

/**
 * @brief   Clear CC keys (if any) from given path
 * @param   path Path to clear keys from
 * @param   slot_id zero-based CI slot identifier (0, 1, ...)
 */
void STB_CiKeysClear(U8BIT path, U8BIT slot_id)
{
   U8BIT demux;
   S_CIKEYS_INFO *slot_info;

   FUNCTION_START(STB_CiKeysClear);

   DBGPRINT("(path=%u)", path)

   demux = STB_DPGetPathDemux(path);
   DBGPRINT("(%u): demux=%u", path, demux)
   if (demux != INVALID_RES_ID)
   {
      if ((desc_allocated != NULL) && (desc_allocated[path]))
      {
         DBGPRINT("Freeing descramblers")
         STB_DMXFreeDescramblerKey(demux, DESC_TRACK_AUDIO);
         STB_DMXFreeDescramblerKey(demux, DESC_TRACK_VIDEO);
         STB_DMXFreeDescramblerKey(demux, DESC_TRACK_TEXT);
         desc_allocated[path] = FALSE;
      }

      slot_info = FindSlot(slot_id);
      if (slot_info != NULL)
      {
         slot_info->keys[EVEN_KEY_REGISTER].applied[path] = FALSE;
         slot_info->keys[ODD_KEY_REGISTER].applied[path] = FALSE;
      }
   }

   FUNCTION_FINISH(STB_CiKeysClear);
}

/*---local function definitions----------------------------------------------*/

static S_CIKEYS_INFO *FindSlot(U8BIT slot_id)
{
   S_CIKEYS_INFO *slot_info;
   STB_OSMutexLock(cikeys_mutex);
   slot_info = cikeys_slots;
   while (slot_info != NULL)
   {
      if (slot_info->slot_id == slot_id)
      {
         if (slot_info->disabled)
         {
            slot_info = NULL;
         }
         break;
      }
      slot_info = slot_info->next;
   }
   STB_OSMutexUnlock(cikeys_mutex);
   return slot_info;
}

/**
 * @brief    Allocate descramblers for path
 * @param    path - decode path
 * @param    cipher - cipher type to use
 * @note     The function updates desc_allocated if successful
 */
static void AllocateDescramblers(U8BIT path, U8BIT cipher)
{
   BOOLEAN result;
   E_STB_DMX_DESC_TYPE type;
   U8BIT demux;

   FUNCTION_START(AllocateDescramblers);

   demux = STB_DPGetPathDemux(path);
   if (desc_allocated != NULL && demux != INVALID_RES_ID)
   {
      switch (cipher)
      {
         case STB_CI_CIPHER_DES:
            type = DESC_TYPE_DES;
            break;
         case STB_CI_CIPHER_AES:
            type = DESC_TYPE_AES;
            break;
      }

      result = STB_DMXGetDescramblerKey(demux, DESC_TRACK_AUDIO);
      if (result)
      {
         STB_DMXSetKeyUsage(demux, DESC_TRACK_AUDIO, KEY_USAGE_TRANSPORT);
         STB_DMXSetDescramblerType(demux, DESC_TRACK_AUDIO, type);

         result = STB_DMXGetDescramblerKey(demux, DESC_TRACK_VIDEO);
         if (result)
         {
            STB_DMXSetKeyUsage(demux, DESC_TRACK_VIDEO, KEY_USAGE_TRANSPORT);
            STB_DMXSetDescramblerType(demux, DESC_TRACK_VIDEO, type);

            result = STB_DMXGetDescramblerKey(demux, DESC_TRACK_TEXT);
            if (result)
            {
               STB_DMXSetKeyUsage(demux, DESC_TRACK_TEXT, KEY_USAGE_TRANSPORT);
               STB_DMXSetDescramblerType(demux, DESC_TRACK_TEXT, type);

               DBGPRINT("Descramblers allocated successfully")
               desc_allocated[path] = TRUE;
            }
            else
            {
               STB_DMXFreeDescramblerKey(demux, DESC_TRACK_VIDEO);
               STB_DMXFreeDescramblerKey(demux, DESC_TRACK_AUDIO);
            }
         }
         else
         {
            STB_DMXFreeDescramblerKey(demux, DESC_TRACK_AUDIO);
         }
      }

      if (!desc_allocated[path])
      {
         DBGPRINT("Could not allocate descramblers")
      }
   }

   FUNCTION_FINISH(AllocateDescramblers);
}

/**
 * @brief    Set key for descramblers
 * @param    path - decode path
 * @param    cipher - cipher type to use
 * @note     The function updates desc_allocated if successful
 */
static BOOLEAN SetDescramblerKey(U8BIT path, E_STB_DMX_DESC_KEY_PARITY parity,
   U8BIT cipher, S_KEY_INFO *key_info)
{
   U8BIT *key_data;
   U8BIT key_buffer[32];
   BOOLEAN result;
   U8BIT demux;

   FUNCTION_START(SetDescramblerKey);

   result = FALSE;

   demux = STB_DPGetPathDemux(path);
   if (key_info->valid && demux != INVALID_RES_ID)
   {
      switch (cipher)
      {
         case STB_CI_CIPHER_DES:
            key_data = key_info->key;
            break;
         case STB_CI_CIPHER_AES:
            memcpy(key_buffer, key_info->key, 16);
            memcpy(key_buffer + 16, key_info->civ, 16);
            key_data = key_buffer;
            break;
      }

      result = STB_DMXSetDescramblerKeyData(demux, DESC_TRACK_AUDIO, parity, key_data);
      DBGPRINT("STB_DMXSetDescramblerKeyData(%d, DESC_TRACK_AUDIO) returned %d", demux, result)
      result = STB_DMXSetDescramblerKeyData(demux, DESC_TRACK_VIDEO, parity, key_data);
      DBGPRINT("STB_DMXSetDescramblerKeyData(%d, DESC_TRACK_VIDEO) returned %d", demux, result)
      result = STB_DMXSetDescramblerKeyData(demux, DESC_TRACK_TEXT, parity, key_data);
      DBGPRINT("STB_DMXSetDescramblerKeyData(%d, DESC_TRACK_TEXT) returned %d", demux, result)
   }

   FUNCTION_FINISH(SetDescramblerKey);
   return(result);
}
