/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Decode Path control functions
 *
 * @file    stbdpc.c
 * @date    26/09/2000
 */

/* #define DEBUG_PRINTING_ENABLED */

// gives direct COM port access
#define STB_DEBUG
/*#define STB_RES_DEBUG*/
#define STB_TUNE_DEBUG

//---includes for this file----------------------------------------------------
// compiler library header files
#include <string.h>
#include <stdlib.h>

// third party header files

// Ocean Blue Software header files

#include <techtype.h>
#include <dbgfuncs.h>

#include "stbhwav.h"
#include "stbdpc.h"
#include "stbheap.h"
#include "stbsic.h"
#include "stberc.h"
#include "stbpvr.h"
#include "stbpvrpr.h"
#include "stbhwav.h"
#include "stbhwtun.h"
#include "stbhwdmx.h"
#include "stbhwos.h"
#include "stbdpc.h"
#include "stbresmgr.h"

#ifdef COMMON_INTERFACE
#include "stbhwci.h"
#endif

#include "stbsitab.h"
#include "ca_glue.h"


//---constant definitions for this file----------------------------------------
#ifdef STB_DP_PRINT_REQUIRED
   #define STB_DP_PRINT(x) DEBUG_PRINTX_CONDITIONAL(DEBUG_STB_DP) x
#else
   #ifdef STB_DEBUG
      #define STB_DP_PRINT(x) STB_SPDebugWrite x
   #else
      #define STB_DP_PRINT(x)
   #endif
#endif

#ifdef STB_RES_DEBUG
   #define STB_RES_PRINT(x) STB_SPDebugWrite x
#else
   #define STB_RES_PRINT(x)
#endif

#ifdef STB_TUNE_DEBUG
   #define STB_TUNE_PRINT(x) STB_SPDebugWrite x
#else
   #define STB_TUNE_PRINT(x)
#endif

#define IS_LIVE_PATH(path)       (path_status[path].in_use && \
                                 (path_status[path].tuner_no != INVALID_RES_ID) && \
                                 (path_status[path].audio_decoder_no != INVALID_RES_ID) && \
                                 (path_status[path].video_decoder_no != INVALID_RES_ID))

#define IS_RECORDING_PATH(path)  (path_status[path].in_use && \
                                 (path_status[path].tuner_no != INVALID_RES_ID) && \
                                 (path_status[path].is_recording))

#define IS_PLAYBACK_PATH(path)   (path_status[path].in_use && \
                                 (path_status[path].tuner_no == INVALID_RES_ID) && \
                                 (path_status[path].audio_decoder_no != INVALID_RES_ID) && \
                                 (path_status[path].video_decoder_no != INVALID_RES_ID))
                                 
#define IS_MONITORING_PATH(path) (path_status[path].in_use && \
                                 (path_status[path].tuner_no != INVALID_RES_ID) && \
                                 (path_status[path].audio_decoder_no == INVALID_RES_ID) && \
                                 (path_status[path].video_decoder_no == INVALID_RES_ID) && \
                                 (path_status[path].is_recording == FALSE))

#define ACTION_DISH        0x01
#define ACTION_SKEW        0x02
#define ACTION_TUNE        0x04
#define ACTION_SI_NEW      0x08

#define ACTION_VIDEO       0x10
#define ACTION_AUDIO       0x20
#define ACTION_AD          0x40
#define ACTION_DECODE_MASK 0xf0

/* AD Status */
#define AD_ENABLED         0x01
#define AD_PID_PRESENT     0x02
#define AD_PAUSED          0x04

#define TUNER_MIN_SRATE    2000
#define TUNER_MAX_SRATE    45000

#define UNICABLE_COMMAND_SIZE 5
#define UNICABLE_SIGNAL_THRESHOLD 50

#define DISEQC_MESSAGE_DELAY  25
#define DISEQC_REPEAT_DELAY   100

//---local typedef structs for this file---------------------------------------

typedef struct
{
   BOOLEAN in_use;               /* Flag indicating whether a path is being used */
   E_STB_DP_PRIORITY priority;   /* Priority used when acquiring the path */
   void *service;                /* Service that the path is tuned to */

   U8BIT tuner_no;               /* Tuner being used by this path */
   U8BIT demux_no;               /* Demux being used by this path */
   U8BIT video_decoder_no;       /* Video decoder being used by this path */
   U8BIT secondary_video_decoder_no;    /* Secondary video decoder being used by this path */
   U8BIT audio_decoder_no;       /* Audio decoder being used by this path */
   U8BIT ad_decoder_no;          /* AD decoder being used by this path */
   U8BIT slot_id;                /* id of CI slot assigned to the path */
   BOOLEAN ca_acquired;          /* Is CA handle valid? */
   U32BIT ca_handle;             /* Handle of CA assigned to the path */
   BOOLEAN lock_enable;          /* TRUE if the path is parentally locked */
   BOOLEAN lock_mode;
   BOOLEAN search_mode;
   BOOLEAN is_recording;
   E_STB_DP_RES_OWNER owner;     /* Owner of the decode path */
   void *owner_data;             /* Data held for the owner of the path */
   U32BIT owner_data_size;
} PATH_STATUS;

typedef struct
{
   U8BIT number_of_bands;
   S_STB_DP_LNB_BAND *band_list;
} S_LNB_DEFINITION;

typedef struct
{
   E_STB_DP_TUNE_STATUS tune_stat;
   U8BIT actions;
   E_STB_DP_SIGNAL_TYPE signal_type;
   BOOLEAN auto_relock;
   BOOLEAN tv_search;
   BOOLEAN rad_search;
   BOOLEAN fta_search;
   BOOLEAN scram_search;
   BOOLEAN net_search;
   U32BIT si_rparam1;
   U32BIT si_rparam2;
   U32BIT si_rparam3;
   E_STB_OTA_SW_UPGRADE_SEARCH_MODE ota_search_status;
   E_STB_DP_LNB_POWER lnb_power;
   E_STB_DP_LNB_TYPE lnb_type;
   S_LNB_DEFINITION lnb_definition[LNB_TYPE_MAX];
   BOOLEAN lnb_22k;
   BOOLEAN lnb_12v;
   BOOLEAN pulse_position;
   BOOLEAN diseqc_position;
   E_STB_DP_DISEQC_CSWITCH diseqc_cswitch;
   U8BIT diseqc_uswitch;
   E_STB_DP_DISEQC_TONE diseqc_tone;
   BOOLEAN diseqc_smatv;
   U8BIT diseqc_repeats;
   U16BIT dish_pos;
   U16BIT skew_pos;
   U32BIT frequency;
   E_STB_DP_POLARITY polarity;
   U16BIT sym_rate;
   E_STB_DP_FEC fec;
   BOOLEAN dvb_s2;
   E_STB_DP_MODULATION modulation;
   E_STB_DP_TMODE terr_mode;
   E_STB_DP_TBWIDTH terr_bwidth;
   E_STB_DP_TTYPE terr_type;
   U8BIT plp_id;
   S8BIT freq_offset;               /* used for both digital terrestrial and analog frequency offset */
   E_STB_DP_CMODE cable_mode;
   E_STB_DP_ANALOG_VIDEO_TYPE anlg_vtype;
   U32BIT unicable_if;
   U8BIT unicable_chan;
   BOOLEAN unicable_position_b;
   U32BIT diseqc_msg_time;
   U8BIT num_additional_frequencies;
   U32BIT *additional_frequencies;
   E_STB_DP_TBWIDTH cable_bwidth;
} TUNER_STATUS;

typedef struct
{
   U16BIT pcr_pid;
   U16BIT video_pid;
   U16BIT audio_pid;
   U16BIT AD_pid;
   U16BIT text_pid;
   U16BIT data_pid;

   E_STB_DP_VIDEO_CODEC video_codec;
   E_STB_DP_AUDIO_CODEC audio_codec;
   E_STB_DP_AUDIO_CODEC ad_codec;

   U8BIT actions;

   E_STB_DP_DEMUX_SOURCE source_type;
   U32BIT source_param;

   U16BIT ecm_pid;
   U16BIT video_ecm_pid;
   U16BIT audio_ecm_pid;
   U16BIT AD_ecm_pid;
   U16BIT text_ecm_pid;
   U16BIT data_ecm_pid;
} DEMUX_STATUS;

typedef struct
{
   E_STB_DP_DECODE_STATUS audio_stat;

   U8BIT actions;

   E_STB_DP_DECODE_SOURCE source_type;
   U32BIT source_param;

   E_STB_DP_AUDIO_MODE audio_mode;

   E_STB_DP_AUDIO_CODEC codec;

   U8BIT source_dmx;
} AUDIO_DECODE_STATUS;

typedef struct
{
   E_STB_DP_DECODE_STATUS audio_stat;
   E_STB_DP_AD_AUDIO ad_audio;
   U8BIT ad_status;
   U8BIT actions;

   E_STB_DP_DECODE_SOURCE source_type;
   U32BIT source_param;
   E_STB_DP_AUDIO_MODE audio_mode;
   E_STB_DP_AUDIO_CODEC codec;

   U8BIT source_dmx;
} AD_DECODE_STATUS;

typedef struct
{
   E_STB_DP_DECODE_STATUS video_stat;
   U8BIT actions;
   E_STB_DP_DECODE_SOURCE source_type;
   U32BIT source_param;

   E_STB_DP_VIDEO_CODEC codec;

   U8BIT source_dmx;
} VIDEO_DECODE_STATUS;

//---local (static) variable declarations for this file------------------------
//   (internal variables declared static to make them local)

static U8BIT num_paths;
static PATH_STATUS *path_status;

static U8BIT num_tuners;
static TUNER_STATUS *tuner_status;

static U8BIT num_demuxes;
static DEMUX_STATUS *demux_status;

static U8BIT num_audio_decoders;
static AUDIO_DECODE_STATUS *audio_decoder_status;
static AD_DECODE_STATUS *ad_decoder_status;
static U8BIT num_video_decoders;
static VIDEO_DECODE_STATUS *video_decoder_status;

static void *dp_protect;
static BOOLEAN lnb_currently_powered;
static BOOLEAN lnb_awake;

static S_STB_DP_LNB_BAND lnb_single[2] = {
   /* polarity, min_freq, max_freq, local_oscillator_frequency, lnb_power, tone_22k */
   {POLARITY_HORIZONTAL, 0, 0, 5150, LNB_VOLTAGE_18V, FALSE},
   {POLARITY_VERTICAL, 0, 0, 5150, LNB_VOLTAGE_14V, FALSE}
};
static S_STB_DP_LNB_BAND lnb_universal[4] = {
   /* polarity, min_freq, max_freq, local_oscillator_frequency, lnb_power, tone_22k */
   {POLARITY_HORIZONTAL, 0, 11750, 9750, LNB_VOLTAGE_18V, FALSE},
   {POLARITY_HORIZONTAL, 11750, 0, 10600, LNB_VOLTAGE_18V, TRUE},
   {POLARITY_VERTICAL, 0, 11750, 9750, LNB_VOLTAGE_14V, FALSE},
   {POLARITY_VERTICAL, 11750, 0, 10600, LNB_VOLTAGE_14V, TRUE}
};
static S_STB_DP_LNB_BAND lnb_unicable[4] = {
   /* polarity, min_freq, max_freq, local_oscillator_frequency, lnb_power, tone_22k */
   {POLARITY_HORIZONTAL, 0, 11750, 9750, LNB_VOLTAGE_14V, FALSE},
   {POLARITY_HORIZONTAL, 11750, 0, 10600, LNB_VOLTAGE_14V, FALSE},
   {POLARITY_VERTICAL, 0, 11750, 9750, LNB_VOLTAGE_14V, FALSE},
   {POLARITY_VERTICAL, 11750, 0, 10600, LNB_VOLTAGE_14V, FALSE}
};


//---local function prototypes for this file-----------------------------------
//   (internal functions declared static to make them local)

static BOOLEAN AcquireResources(U8BIT path, U16BIT demux_caps, BOOLEAN with_decoders);
static BOOLEAN StartTune(U8BIT path);
static U8BIT GetADEnabled(U8BIT path);
static void UnicableSendCommand(U8BIT path, U8BIT cmd[UNICABLE_COMMAND_SIZE]);
static void SendDISEQCMessage(U8BIT tuner, U8BIT *msg_data, U8BIT msg_size);
static S_STB_DP_LNB_BAND *FindLNBBand(U8BIT path);


//---global function definitions-----------------------------------------------

/*!**************************************************************************
 * @brief   Initialises decode path control and sets up the structures for
 *          restricting access to resources
 ****************************************************************************/
void STB_DPInitialise(void)
{
   E_STB_DMX_DEMUX_SOURCE dmx_source;
   U8BIT dmx_param;
   U8BIT i;

   FUNCTION_START(STB_DPInitialise);

   num_tuners = STB_HWGetTunerPaths();
   num_demuxes = STB_HWGetDemuxPaths();
   num_audio_decoders = STB_HWGetAudioDecodePaths();
   num_video_decoders = STB_HWGetVideoDecodePaths();

   num_paths = num_demuxes;

   STB_RES_PRINT(("STB_DPInitialise(tuners = %d, demuxes = %d, audio/video decoders = %d/%d)",
                  num_tuners, num_demuxes, num_audio_decoders, num_video_decoders));
   STB_RES_PRINT(("STB_DPInitialise: Max number of paths = %d", num_paths));

   tuner_status = (TUNER_STATUS *)STB_GetMemory((U32BIT)(sizeof(TUNER_STATUS) * num_tuners));
   if (tuner_status != NULL)
   {
      memset(tuner_status, 0, num_tuners * sizeof(TUNER_STATUS));

      for (i = 0; i < num_tuners; i++)
      {
         tuner_status[i].tune_stat = TUNE_NO_LOCK;
         tuner_status[i].actions = 0;
         tuner_status[i].signal_type = SIGNAL_NONE;
         tuner_status[i].auto_relock = TRUE;
         tuner_status[i].tv_search = TRUE;
         tuner_status[i].rad_search = TRUE;
         tuner_status[i].fta_search = TRUE;
         tuner_status[i].scram_search = TRUE;
         tuner_status[i].net_search = TRUE;
         tuner_status[i].si_rparam1 = 0;
         tuner_status[i].si_rparam2 = 0;
         tuner_status[i].si_rparam3 = 0;
         tuner_status[i].ota_search_status = OTA_SEARCH_OFF;
         tuner_status[i].lnb_power = LNB_POWER_AUTO;
         tuner_status[i].lnb_type = LNB_TYPE_UNIVERSAL;

         memset(tuner_status[i].lnb_definition, 0, LNB_TYPE_MAX * sizeof(S_LNB_DEFINITION));

         tuner_status[i].lnb_definition[LNB_TYPE_SINGLE].band_list = lnb_single;
         tuner_status[i].lnb_definition[LNB_TYPE_SINGLE].number_of_bands =
            sizeof(lnb_single) / sizeof(S_STB_DP_LNB_BAND);

         tuner_status[i].lnb_definition[LNB_TYPE_UNIVERSAL].band_list = lnb_universal;
         tuner_status[i].lnb_definition[LNB_TYPE_UNIVERSAL].number_of_bands =
            sizeof(lnb_universal) / sizeof(S_STB_DP_LNB_BAND);

         tuner_status[i].lnb_definition[LNB_TYPE_UNICABLE].band_list = lnb_unicable;
         tuner_status[i].lnb_definition[LNB_TYPE_UNICABLE].number_of_bands =
            sizeof(lnb_unicable) / sizeof(S_STB_DP_LNB_BAND);

         tuner_status[i].lnb_22k = FALSE;
         tuner_status[i].lnb_12v = FALSE;
         tuner_status[i].pulse_position = FALSE;
         tuner_status[i].diseqc_position = FALSE;
         tuner_status[i].diseqc_cswitch = DISEQC_CSWITCH_OFF;
         tuner_status[i].diseqc_uswitch = 0;
         tuner_status[i].diseqc_tone = DISEQC_TONE_OFF;
         tuner_status[i].diseqc_smatv = FALSE;
         tuner_status[i].diseqc_repeats = 0;
         tuner_status[i].dish_pos = 0;
         tuner_status[i].skew_pos = 0;
         tuner_status[i].frequency = 0;
         tuner_status[i].polarity = POLARITY_HORIZONTAL;
         tuner_status[i].sym_rate = 0;
         tuner_status[i].fec = FEC_AUTOMATIC;
         tuner_status[i].dvb_s2 = FALSE;
         tuner_status[i].modulation = MOD_QPSK;
         tuner_status[i].terr_mode = MODE_COFDM_2K;
         tuner_status[i].terr_bwidth = TBWIDTH_8MHZ;
         tuner_status[i].terr_type = TERR_TYPE_UNKNOWN;
         tuner_status[i].plp_id = 0;
         tuner_status[i].freq_offset = 0;
         tuner_status[i].cable_mode = MODE_QAM_AUTO;
         tuner_status[i].diseqc_msg_time = 0;
      }
   }

   demux_status = (DEMUX_STATUS *)STB_GetMemory((U32BIT)(sizeof(DEMUX_STATUS) * num_demuxes));
   if (demux_status != NULL)
   {
      for (i = 0; i < num_demuxes; i++)
      {
         demux_status[i].pcr_pid = 0;
         demux_status[i].video_pid = 0;
         demux_status[i].audio_pid = 0;
         demux_status[i].AD_pid = 0;
         demux_status[i].text_pid = 0;
         demux_status[i].data_pid = 0;

         demux_status[i].actions = 0;

         STB_DMXGetDemuxSource(i, &dmx_source, &dmx_param);

         switch (dmx_source)
         {
            case DMX_TUNER:
               demux_status[i].source_type = DEMUX_SOURCE_TUNER;
               break;

            case DMX_MEMORY:
               demux_status[i].source_type = DEMUX_SOURCE_FILE;
               break;

            default:
               demux_status[i].source_type = DEMUX_SOURCE_LINEIN;
               break;
         }

         demux_status[i].source_param = (U32BIT)dmx_param;
      }
   }

   ad_decoder_status = (AD_DECODE_STATUS *)STB_GetMemory((U32BIT)(sizeof(AD_DECODE_STATUS) * num_audio_decoders));
   if (ad_decoder_status != NULL)
   {
      for (i = 0; i < num_audio_decoders; i++)
      {
         ad_decoder_status[i].audio_stat = DECODE_STOPPED;
         ad_decoder_status[i].ad_audio = AD_AUDIO_OFF;
         ad_decoder_status[i].ad_status = 0;
         ad_decoder_status[i].actions = 0;
         ad_decoder_status[i].source_type = DECODE_SOURCE_DEMUX;
         ad_decoder_status[i].source_param = i;
         ad_decoder_status[i].audio_mode = AUDIO_MONO;
         ad_decoder_status[i].codec = AUDIO_CODEC_AUTO;
      }
   }

   audio_decoder_status = (AUDIO_DECODE_STATUS *)STB_GetMemory((U32BIT)(sizeof(AUDIO_DECODE_STATUS) * num_audio_decoders));
   if (audio_decoder_status != NULL)
   {
      for (i = 0; i < num_audio_decoders; i++)
      {
         audio_decoder_status[i].audio_stat = DECODE_STOPPED;
         audio_decoder_status[i].actions = 0;
         audio_decoder_status[i].source_type = DECODE_SOURCE_DEMUX;
         audio_decoder_status[i].source_param = i;

         audio_decoder_status[i].audio_mode = AUDIO_STEREO;
         audio_decoder_status[i].codec = AUDIO_CODEC_AUTO;
      }
   }

   video_decoder_status = (VIDEO_DECODE_STATUS *)STB_GetMemory((U32BIT)(sizeof(VIDEO_DECODE_STATUS) * num_video_decoders));
   if (video_decoder_status != NULL)
   {
      for (i = 0; i < num_video_decoders; i++)
      {
         video_decoder_status[i].video_stat = DECODE_STOPPED;
         video_decoder_status[i].actions = 0;
         video_decoder_status[i].source_type = DECODE_SOURCE_DEMUX;
         video_decoder_status[i].source_param = i;
         video_decoder_status[i].codec = VIDEO_CODEC_AUTO;
      }
   }

   path_status = (PATH_STATUS *)STB_GetMemory((U32BIT)(sizeof(PATH_STATUS) * num_paths));
   if (path_status != NULL)
   {
      for (i = 0; i < num_paths; i++)
      {
         memset(&path_status[i], 0, sizeof(PATH_STATUS));

         path_status[i].tuner_no = INVALID_RES_ID;
         path_status[i].demux_no = INVALID_RES_ID;
         path_status[i].video_decoder_no = INVALID_RES_ID;
         path_status[i].secondary_video_decoder_no = INVALID_RES_ID;
         path_status[i].audio_decoder_no = INVALID_RES_ID;
         path_status[i].ad_decoder_no = INVALID_RES_ID;
         path_status[i].slot_id = INVALID_RES_ID;
         path_status[i].ca_acquired = FALSE;
         path_status[i].ca_handle = INVALID_CA_DESCRAMBLER_HANDLE;
         path_status[i].lock_enable = FALSE;
         path_status[i].lock_mode = FALSE;
         path_status[i].search_mode = FALSE;
         path_status[i].is_recording = FALSE;
         path_status[i].owner = RES_OWNER_NONE;
         path_status[i].owner_data = NULL;
         path_status[i].owner_data_size = 0;
      }
   }

   lnb_awake = FALSE;
   lnb_currently_powered = FALSE;

   dp_protect = STB_OSCreateSemaphore();

   FUNCTION_FINISH(STB_DPInitialise);
}

/*!**************************************************************************
 * @brief   Returns the maximum number of decode paths
 * @param   None
 * @return  Number of decode paths
 ****************************************************************************/
U8BIT STB_DPGetNumPaths(void)
{
   FUNCTION_START(STB_DPGetNumPaths);
   FUNCTION_FINISH(STB_DPGetNumPaths);
   return(num_paths);
}

/*!**************************************************************************
 * @brief   Acquires a decode path and all the required resources (tuner, demux, audio
 *          and video decoders). If no tuners are available for live
 *          viewing then a tuner currently being used for recording will be acquired.
 *          It's assumed that there's only one live path in existence and if there's an
 *          attempt to acquire a live path but one exists with a different tuner type then
 *          the acquire will fail, so paths must be released before a new one is acquired.
 * @param   tuner_type - type of tuner to be acquired
 * @param   service - the service that the path will be tuned to, can be NULL
 * @param   transport - the transport that the path will be tuned to, can be NULL
 * @param   owner - module that's acquiring ownership of the path
 * @param   priority - used to control acquisition of a tuner; high priority may result in
 *                     a tuner that's already in use by a lower priority path being acquired,
 *                     but low priority will never reuse a tuner if the tuner isn't already
 *                     tuned to the correct transport.
 * @param   with_decoders - TRUE if decoding resources are to be acquired
 * @param   for_recording - TRUE if the path will be used for recording
 * @return  ID of acquired decode path, or INVALID_RES_ID if none are available
 ****************************************************************************/
U8BIT STB_DPAcquireTunerPath(E_STB_DP_SIGNAL_TYPE tuner_type, void *service, void *transport,
   E_STB_DP_RES_OWNER owner, E_STB_DP_PRIORITY priority, BOOLEAN with_decoders, BOOLEAN for_recording)
{
   U8BIT path, i, j;
   U8BIT tuner_num;
   BOOLEAN high_priority;
   BOOLEAN tuner_taken;
   U16BIT dmx_caps;

   FUNCTION_START(STB_DPAcquireTunerPath);

   STB_RES_PRINT(("STB_DPAcquireTunerPath(type=%u, serv=%p, trans=%p, owner=%u, priority=%u, decoders=%u, recording=%u)",
      tuner_type, service, transport, owner, priority, with_decoders, for_recording));

   path = INVALID_RES_ID;

   /* Find a free path */
   for (i = 0; (i < num_paths) && (path == INVALID_RES_ID); i++)
   {
      if (!path_status[i].in_use)
      {
         if (priority == DP_PRIORITY_HIGH)
         {
            high_priority = TRUE;
         }
         else
         {
            high_priority = FALSE;
         }

         /* Acquire a tuner */
         tuner_num = STB_RESAcquireTuner(tuner_type, NULL /* acquire a new tuner */, owner, high_priority, &tuner_taken);
         if (tuner_num != INVALID_RES_ID)
         {
            path_status[i].tuner_no = tuner_num;

            if (for_recording)
            {
               dmx_caps = DMX_CAPS_RECORDING;
            }
            else if (with_decoders)
            {
               dmx_caps = DMX_CAPS_LIVE;
            }
            else
            {
               /* Need a demux that can monitor SI data */
               dmx_caps = DMX_CAPS_MONITOR_SI;
            }

            if (AcquireResources(i, dmx_caps, with_decoders))
            {
               if (tuner_taken || (STB_RESGetTunedTransport(tuner_num) != transport))
               {
                  /* Tuner needs to be retuned so set the tuner status to not locked */
                  tuner_status[tuner_num].tune_stat = TUNE_NO_LOCK;
                  STB_RESSetTunedTransport(tuner_num, transport);

                  if (tuner_taken)
                  {
                     /* The tuner being used by another path is about to be retuned.
                      * Find all paths using this tuner so they can be updated */
                     for (j = 0; j < num_paths; j++)
                     {
                        if ((j != i) && path_status[j].in_use &&
                           (path_status[j].tuner_no == tuner_num) && (path_status[j].service != service))
                        {
                           /* This path is using the tuner that's just been taken, so the service
                            * it's tuned to will change */
                           path_status[j].service = service;

                           STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_TYPE_FORCED_SERVICE_CHANGE,
                              &j, sizeof(U8BIT));
                        }
                     }
                  }
               }

               tuner_status[tuner_num].signal_type = tuner_type;

               path_status[i].in_use = TRUE;
               path_status[i].priority = priority;
               path_status[i].service = service;
               path_status[i].owner = owner;
               path_status[i].is_recording = for_recording;
               path = i;
            }
            else
            {
               /* Release the tuner */
               STB_RESReleaseTuner(tuner_num, high_priority, owner);
               path_status[i].tuner_no = INVALID_RES_ID;
            }
         }
      }
   }

#ifdef STB_RES_DEBUG
   if (path != INVALID_RES_ID)
   {
      STB_RES_PRINT(("STB_DPAcquireTunerPath: Acquired path %u, tuner=%u, demux=%u, video=%u, audio=%u",
                     path, path_status[path].tuner_no, path_status[path].demux_no,
                     path_status[path].video_decoder_no, path_status[path].audio_decoder_no));
   }
   else
   {
      STB_RES_PRINT(("STB_DPAcquireTunerPath: Failed to acquired tuner path"));
   }
#endif

   FUNCTION_FINISH(STB_DPAcquireTunerPath);

   return(path);
}

/*!**************************************************************************
 * @brief   Acquires a decode path and all the required resources (demux, audio and
 *          video decoders) for playing back a recording.
 * @param   void *service - service pointer associated to the playback
 * @return  ID of the acquired playback path, or INVALID_RES_ID if none are available
 ****************************************************************************/
U8BIT STB_DPAcquirePlaybackPath(void *service)
{
   U8BIT path, i;
   U32BIT param;

   FUNCTION_START(STB_DPAcquirePlaybackPath);

   path = INVALID_RES_ID;

   /* See if there's a playback path already. A playback path is identified as having audio and
    * video decoders, but no tuner */
   for (i = 0; (i < num_paths) && (path == INVALID_RES_ID); i++)
   {
      if (IS_PLAYBACK_PATH(i))
      {
         path = i;
      }
   }

   if (path == INVALID_RES_ID)
   {
      /* Look for a path that hasn't been acquired */
      for (i = 0; (i < num_paths) && (path == INVALID_RES_ID); i++)
      {
         if (!path_status[i].in_use)
         {
            /* Found an unused path, now acquire the resources needed for playback */
            path_status[i].demux_no = STB_RESAcquireDemux(INVALID_RES_ID, DMX_CAPS_PLAYBACK);
            if (path_status[i].demux_no != INVALID_RES_ID)
            {
               path_status[i].video_decoder_no = STB_RESAcquireVideoDecoder();
               if (path_status[i].video_decoder_no != INVALID_RES_ID)
               {
                  param = (INVALID_RES_ID << 8) | path_status[i].demux_no;
                  STB_RES_PRINT(("STB_DPAcquirePlaybackPath: path=%u, setting video source=0x%x param=0x%x", i, AV_DEMUX, param));
                  STB_AVSetVideoSource(path_status[i].video_decoder_no, AV_DEMUX, param);

                  path_status[i].audio_decoder_no = STB_RESAcquireAudioDecoder();
                  if (path_status[i].audio_decoder_no != INVALID_RES_ID)
                  {
                     /* Acquired all necessary resources */
                     STB_RES_PRINT(("STB_DPAcquirePlaybackPath: path=%u, setting audio source=0x%x param=0x%x", i, AV_DEMUX, param));

                     STB_AVSetAudioSource(path_status[i].audio_decoder_no, AV_DEMUX, param);

                     path_status[i].ad_decoder_no = STB_RESAcquireADDecoder();
                     if (path_status[i].ad_decoder_no != INVALID_RES_ID)
                     {
                        STB_AVSetADSource(path_status[i].ad_decoder_no, AV_DEMUX, param);
                     }

                     path_status[i].in_use = TRUE;
                     path_status[i].service = service;
                     path_status[i].owner = RES_OWNER_NONE;
                     path = i;

                     /* Set the demux to get its data from memory */
                     STB_DMXSetDemuxSource(path_status[i].demux_no, DMX_MEMORY, 0);
                  }
                  else
                  {
                     /* Not all resources are available so release acquired resources */
                     STB_RESReleaseVideoDecoder(path_status[i].video_decoder_no);
                     path_status[i].video_decoder_no = INVALID_RES_ID;

                     /* Set the demux to release get its data from a tuner */
                     STB_DMXSetDemuxSource(path_status[i].demux_no, DMX_INVALID, INVALID_RES_ID);
                     STB_RESReleaseDemux(path_status[i].demux_no);
                     path_status[i].demux_no = INVALID_RES_ID;
                  }
               }
               else
               {
                  /* Set the demux to release get its data from a tuner */
                  STB_DMXSetDemuxSource(path_status[i].demux_no, DMX_INVALID, INVALID_RES_ID);
                  /* Not all resources are available so release acquired resources */
                  STB_RESReleaseDemux(path_status[i].demux_no);
                  path_status[i].demux_no = INVALID_RES_ID;
               }
            }
         }
      }
   }

#ifdef STB_RES_DEBUG
   if (path != INVALID_RES_ID)
   {
      STB_RES_PRINT(("STB_DPAcquirePlaybackPath: Acquired playback path %u, demux=%u, video=%u, audio=%u",
                     path, path_status[path].demux_no, path_status[path].video_decoder_no,
                     path_status[path].audio_decoder_no));
   }
   else
   {
      STB_RES_PRINT(("STB_DPAcquirePlaybackPath: Acquired playback path %u", path));
   }
#endif

   FUNCTION_FINISH(STB_DPAcquirePlaybackPath);

   return(path);
}

/*!**************************************************************************
 * @brief   Releases the decode path and all resources no longer needed.
 *          The path won't be released if the path is owned by a module and the
 *          owner argument doesn't match the path's owner
 * @param   path - decode path
 * @param   owner - module releasing the path
 * @return  TRUE if the path is released, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_DPReleasePath(U8BIT path, E_STB_DP_RES_OWNER owner)
{
   BOOLEAN retval;
   BOOLEAN high_priority;

   FUNCTION_START(STB_DPReleasePath);

   ASSERT(path < num_paths);

   retval = FALSE;

   if (path < num_paths)
   {
      BOOLEAN preload = STB_DPIsMonitoringPath(path);
      STB_RES_PRINT(("STB_DPReleasePath(path=%u, owner=%u): in_use=%u, owner=%u",
                     path, owner, path_status[path].in_use, path_status[path].owner));

      if (path_status[path].in_use &&
         ((path_status[path].owner == RES_OWNER_NONE) || (path_status[path].owner == owner)))
      {
         path_status[path].in_use = FALSE;

         /* This path isn't being used anymore so all resources can be released */
         STB_DPReleaseCISlotFromPath(path);

         if (path_status[path].tuner_no != INVALID_RES_ID)
         {
            if (path_status[path].priority == DP_PRIORITY_HIGH)
            {
               high_priority = TRUE;
            }
            else
            {
               high_priority = FALSE;
            }

            STB_RESReleaseTuner(path_status[path].tuner_no, high_priority, owner);
            path_status[path].tuner_no = INVALID_RES_ID;
         }

         path_status[path].service = NULL;

         /* Set the demux to release get its data from a tuner */
         if (TRUE == preload)
            STB_DMXSetDemuxSource((path_status[path].demux_no | (1 << 7)), DMX_INVALID, INVALID_RES_ID);         
         else
            STB_DMXSetDemuxSource(path_status[path].demux_no, DMX_INVALID, INVALID_RES_ID);

         STB_RESReleaseDemux(path_status[path].demux_no);
         path_status[path].demux_no = INVALID_RES_ID;

         if (path_status[path].video_decoder_no != INVALID_RES_ID)
         {
            STB_RESReleaseVideoDecoder(path_status[path].video_decoder_no);
            path_status[path].video_decoder_no = INVALID_RES_ID;
         }

         if (path_status[path].secondary_video_decoder_no != INVALID_RES_ID)
         {
            STB_RESReleaseVideoDecoder(path_status[path].secondary_video_decoder_no);
            path_status[path].secondary_video_decoder_no = INVALID_RES_ID;
         }

         if (path_status[path].audio_decoder_no != INVALID_RES_ID)
         {
            STB_RESReleaseAudioDecoder(path_status[path].audio_decoder_no);
            path_status[path].audio_decoder_no = INVALID_RES_ID;
         }

         if (path_status[path].ad_decoder_no != INVALID_RES_ID)
         {
            STB_RESReleaseADDecoder(path_status[path].ad_decoder_no);
            path_status[path].ad_decoder_no = INVALID_RES_ID;
         }

         if (path_status[path].ca_acquired)
         {
            path_status[path].ca_acquired = !STB_CAReleaseDescrambler(path_status[path].ca_handle);
         }
#ifdef STB_RES_DEBUG
         else
         {
            STB_RES_PRINT(("STB_DPReleasePath(path=%u): no CA slot on this path", path));
         }
#endif

         path_status[path].owner = RES_OWNER_NONE;

         if (path_status[path].owner_data != NULL)
         {
            STB_FreeMemory(path_status[path].owner_data);
            path_status[path].owner_data = NULL;
            path_status[path].owner_data_size = 0;
         }

         retval = TRUE;
      }
#ifdef STB_RES_DEBUG
      else
      {
         STB_RES_PRINT(("STB_DPReleasePath(path=%u): Failed to release path", path));
      }
#endif
   }

   FUNCTION_FINISH(STB_DPReleasePath);

   return(retval);
}

/*!**************************************************************************
 * @brief   Releases all decoders from the given path
 ****************************************************************************/
void STB_DPReleaseDecoders(U8BIT path)
{
   FUNCTION_START(STB_DPReleaseDecoders);

   ASSERT(path < num_paths);

   STB_RES_PRINT(("STB_DPReleaseDecoders(%u)", path));

   if (path < num_paths)
   {
      if (path_status[path].video_decoder_no != INVALID_RES_ID)
      {
         STB_RESReleaseVideoDecoder(path_status[path].video_decoder_no);
         path_status[path].video_decoder_no = INVALID_RES_ID;
      }

      if (path_status[path].secondary_video_decoder_no != INVALID_RES_ID)
      {
         STB_RESReleaseVideoDecoder(path_status[path].secondary_video_decoder_no);
         path_status[path].secondary_video_decoder_no = INVALID_RES_ID;
      }

      if (path_status[path].audio_decoder_no != INVALID_RES_ID)
      {
         STB_RESReleaseAudioDecoder(path_status[path].audio_decoder_no);
         path_status[path].audio_decoder_no = INVALID_RES_ID;
      }

      if (path_status[path].ad_decoder_no != INVALID_RES_ID)
      {
         STB_RESReleaseADDecoder(path_status[path].ad_decoder_no);
         path_status[path].ad_decoder_no = INVALID_RES_ID;
      }
   }

   FUNCTION_FINISH(STB_DPReleaseDecoders);
}

/*!**************************************************************************
 * @brief   Releases all decode paths and all the resources they are using
 * @return  None
 ****************************************************************************/
void STB_DPReleaseAllPaths(void)
{
   U8BIT path, num_paths;

   FUNCTION_START(STB_DPReleaseAllPaths);

   STB_RES_PRINT(("STB_DPReleaseAllPaths()"));

   num_paths = STB_DPGetNumPaths();

   for (path = 0; path < num_paths; path++)
   {
      STB_DPReleasePath(path, RES_OWNER_NONE);
   }

   FUNCTION_FINISH(STB_DPReleaseAllPaths);
}

/*!**************************************************************************
 * @brief   Acquire a CI slot and assign it to the given path
 * @param   path - decode path wanting the CI slot resource
 * @return  slot id acquired, INVALID_RES_ID if none acquired
 ****************************************************************************/
U8BIT STB_DPAcquireCISlotForPath(U8BIT path, U8BIT *pmt_data, U8BIT *ci_protection_desc)
{
   U8BIT retval;

   FUNCTION_START(STB_DPAcquireCISlotForPath);

#ifndef COMMON_INTERFACE
   USE_UNWANTED_PARAM(path);
   USE_UNWANTED_PARAM(pmt_data);
   USE_UNWANTED_PARAM(ci_protection_desc);
#endif

   ASSERT(path < num_paths);

   retval = INVALID_RES_ID;

#ifdef COMMON_INTERFACE
   if (path < num_paths)
   {
      if (path_status[path].slot_id == INVALID_RES_ID)
      {
         retval = STB_RESAcquireCISlot(path_status[path].service,
            pmt_data, ci_protection_desc);
         if (retval != INVALID_RES_ID)
         {
            /* Route the TS from the path's tuner through the CI slot if the tuner is valid
             * and the slot wasn't already being used */
            if ((path_status[path].tuner_no != INVALID_RES_ID) &&
                (STB_RESGetCISlotUsageCount(retval) == 1))
            {
               STB_CIRouteTS(path_status[path].tuner_no, retval, TRUE);
            }
            path_status[path].slot_id = retval;
         }
      }
      else
      {
         retval = path_status[path].slot_id;
         STB_RES_PRINT(("%s(%u): Path already has CI slot %u", __FUNCTION__, path,
                        path_status[path].slot_id));
      }
   }
#endif

   FUNCTION_FINISH(STB_DPAcquireCISlotForPath);

   return(retval);
}

/*!**************************************************************************
 * @brief   Use the given CI slot with the given path
 * @param   path - decode path
 * @param   slot_id - CI slot id
 * @return  TRUE if the slot can be used, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_DPUseCISlotWithPath(U8BIT path, U8BIT slot_id)
{
   BOOLEAN retval;

   FUNCTION_START(STB_DPUseCISlotWithPath);

#ifndef COMMON_INTERFACE
   USE_UNWANTED_PARAM(path);
   USE_UNWANTED_PARAM(slot_id);
#endif

   retval = FALSE;

   STB_RES_PRINT(("%s(path=%u, slot=%u)", __FUNCTION__, path, slot_id));

#ifdef COMMON_INTERFACE
   if (STB_RESUseCISlot(slot_id))
   {
      path_status[path].slot_id = slot_id;

      /* Route the TS from the path's tuner through the CI slot if the tuner is valid
       * and the slot wasn't already being used */
      if (path_status[path].tuner_no != INVALID_RES_ID)
      {
         STB_CIRouteTS(path_status[path].tuner_no, slot_id, TRUE);
      }

      retval = TRUE;
   }
#endif

   FUNCTION_FINISH(STB_DPUseCISlotWithPath);

   return(retval);
}

/*!**************************************************************************
 * @brief   Release the CI slot associated with the given path
 * @param   path - decode path using the CI slot
 ****************************************************************************/
void STB_DPReleaseCISlotFromPath(U8BIT path)
{
   FUNCTION_START(STB_DPReleaseCISlotFromPath);

   ASSERT(path < num_paths);

   STB_RES_PRINT(("STB_DPReleaseCISlotFromPath(%u): slot_id=%u", path, path_status[path].slot_id));

   if ((path < num_paths) && (path_status[path].slot_id != INVALID_RES_ID))
   {
      STB_RESReleaseCISlot(path_status[path].slot_id);

#ifdef COMMON_INTERFACE
      if ((path_status[path].tuner_no != INVALID_RES_ID) &&
          (STB_RESGetCISlotUsageCount(path_status[path].slot_id) == 0))
      {
         /* Stop routing the TS from this tuner through the CI slot */
         STB_CIRouteTS(path_status[path].tuner_no, path_status[path].slot_id, FALSE);
      }
#endif

      path_status[path].slot_id = INVALID_RES_ID;
   }

   FUNCTION_FINISH(STB_DPReleaseCISlotFromPath);
}

/*!**************************************************************************
 * @brief   Returns the CI slot id associated with the given path
 * @return  id of assigned CI slot or INVALID_RES_ID if not assigned
 ****************************************************************************/
U8BIT STB_DPGetPathCISlot(U8BIT path)
{
   U8BIT slot_id;

   FUNCTION_START(STB_DPGetPathCISlot);

   slot_id = INVALID_RES_ID;

   if (path < num_paths)
   {
      slot_id = path_status[path].slot_id;
   }

   FUNCTION_FINISH(STB_DPGetPathCISlot);

   return(slot_id);
}

/*!**************************************************************************
 * @brief   Checks whether a given CI slot is in use
 * @param   start_path - path id to start the search with. Use INVALID_RES_ID to start from first path
 * @param   slot_id - CI slot to be checked
 * @param   ignore_path - decode path to be ignored (i.e. don't check if current
 *                        path is using a slot). This can be INVALID_RES_ID to
 *                        check all paths
 * @return  decode path using the slot or INVALID_RES_ID if not in use
 ****************************************************************************/
U8BIT STB_DPIsCISlotInUse(U8BIT start_path, U8BIT slot_id, U8BIT ignore_path)
{
   U8BIT retval;
   U8BIT path;

   FUNCTION_START(STB_DPIsCISlotInUse);

   retval = INVALID_RES_ID;

   if (start_path == INVALID_RES_ID)
   {
      start_path = 0;
   }

   for (path = start_path; (path < num_paths) && (retval == INVALID_RES_ID); path++)
   {
      if (path_status[path].slot_id == slot_id)
      {
         if ((ignore_path == INVALID_RES_ID) || (path != ignore_path))
         {
            retval = path;
         }
      }
   }

   FUNCTION_FINISH(STB_DPIsCISlotInUse);

   return(retval);
}

/*!**************************************************************************
 * @brief   Acquire a CA descrambler and assign it to the given path
 * @param   path - decode path wanting the CI slot resource
 * @param   pmt_data - pmt section data for the present service on the path
 * @param   ca_handle - pointer in which the CA descrambler handle is returned
 * @return  TRUE if a CA descrambler is already associated with the given path,
 *          or one is successfully acquired, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_DPAcquireCADescramblerForPath(U8BIT path, U8BIT *pmt_data, U32BIT *ca_handle)
{
   BOOLEAN retval;
   U16BIT num_pmt_ca_ids;
   U16BIT *pmt_ca_ids;
   U8BIT demux_path;
   U16BIT serv_id;

   FUNCTION_START(STB_DPAcquireCADescramblerForPath);

   ASSERT(path < num_paths);

   if (path < num_paths)
   {
      retval = path_status[path].ca_acquired;
      if ((pmt_data != NULL) && !path_status[path].ca_acquired)
      {
         demux_path = STB_DPGetPathDemux(path);
         if (demux_path != INVALID_RES_ID)
         {
            /* Service ID is the xtid in the PMT */
            serv_id = (pmt_data[3] << 8) + pmt_data[4];

            num_pmt_ca_ids = STB_SIGetPmtCaIdDescArray(pmt_data, &pmt_ca_ids);

            /* Always attempt to acquire a CA descrambler even if there aren't any CA IDs
             * in the PMT, because some CA systems want full control over free services too */
            path_status[path].ca_acquired = STB_CAAcquireDescrambler(demux_path, serv_id,
               pmt_ca_ids, num_pmt_ca_ids, &path_status[path].ca_handle);
            if (path_status[path].ca_acquired)
            {
               retval = TRUE;
               *ca_handle = path_status[path].ca_handle;
               STB_RES_PRINT(("%s(%u): Acquired CA descrambler 0x%lx", __FUNCTION__, path,
                              path_status[path].ca_handle));
            }
#ifdef STB_RES_DEBUG
            else
            {
               STB_RES_PRINT(("%s(%u): No CA descrambler acquired", __FUNCTION__, path));
            }
#endif
            STB_SIReleaseCaIdDescArray(pmt_ca_ids, num_pmt_ca_ids);
         }
#ifdef STB_RES_DEBUG
         else
         {
            STB_RES_PRINT(("%s(%u), invalid demux path", __FUNCTION__, path));
         }
#endif
      }
#ifdef STB_RES_DEBUG
      else
      {
         STB_RES_PRINT(("%s(%u): Path already has a CA descrambler", __FUNCTION__, path));
      }
#endif
   }
   else
   {
      retval = FALSE;
      STB_RES_PRINT(("%s(%u): Invalid path", __FUNCTION__, path));
   }

   FUNCTION_FINISH(STB_DPAcquireCADescramblerForPath);

   return(retval);
}

/*!**************************************************************************
 * @brief   Release the CA descrambler associated with the given path
 * @param   path - decode path from which the CA descrambler is to be released
 ****************************************************************************/
void STB_DPReleaseCADescramblerFromPath(U8BIT path)
{
   FUNCTION_START(STB_DPReleaseCADescramblerFromPath);

   ASSERT(path < num_paths);

   STB_RES_PRINT(("%s(path=%u): acquired=%u, handle=0x%lx", __FUNCTION__, path,
                  path_status[path].ca_acquired, path_status[path].ca_handle));

   if ((path < num_paths) && path_status[path].ca_acquired)
   {
      path_status[path].ca_acquired = !STB_CAReleaseDescrambler(path_status[path].ca_handle);
   }

   FUNCTION_FINISH(STB_DPReleaseCADescramblerFromPath);
}

/*!**************************************************************************
 * @brief   Get the handle of the CA descrambler associated with the given path
 * @param   path - decode path
 * @param   handle - pointer in which the CA descrambler handle will be returned
 * @return  TRUE if the path has a CA descrambler associated with it and the handle
 *          is returned, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_DPGetPathCADescrambler(U8BIT path, U32BIT *handle)
{
   BOOLEAN retval;

   FUNCTION_START(STB_DPGetPathCADescrambler);

   retval = FALSE;
   *handle = INVALID_CA_DESCRAMBLER_HANDLE;

   if (path < num_paths)
   {
      retval = path_status[path].ca_acquired;
      if (retval)
      {
         *handle = path_status[path].ca_handle;
      }
   }

   FUNCTION_FINISH(STB_DPGetPathCADescrambler);

   return(retval);
}

/*!**************************************************************************
 * @brief   Returns the ID of the decode path being used for live viewing
 * @return  ID of live path or INVALID_RES_ID if none found
 ****************************************************************************/
U8BIT STB_DPGetLivePath(void)
{
   U8BIT path, i;

   FUNCTION_START(STB_DPGetLivePath);

   path = INVALID_RES_ID;

   for (i = 0; (i < num_paths) && (path == INVALID_RES_ID); i++)
   {
      if (IS_LIVE_PATH(i))
      {
         path = i;
      }
   }

   FUNCTION_FINISH(STB_DPGetLivePath);

   return(path);
}

/*!**************************************************************************
 * @brief   Is the given decode path being used for live viewing
 * @param   path - ID of decode path
 * @return  TRUE if the path is being used for live viewing
 ****************************************************************************/
BOOLEAN STB_DPIsLivePath(U8BIT path)
{
   BOOLEAN is_live = FALSE;

   FUNCTION_START(STB_DPIsLivePath);

   ASSERT(path < num_paths);

   if (path < num_paths)
   {
      is_live = IS_LIVE_PATH(path);
   }

   FUNCTION_FINISH(STB_DPIsLivePath);

   return(is_live);
}

/*!**************************************************************************
 * @brief   Is the given decode path being used for recording
 * @param   path - ID of decode path
 * @return  TRUE if the path is being used for recording
 ****************************************************************************/
BOOLEAN STB_DPIsRecordingPath(U8BIT path)
{
   BOOLEAN is_recording = FALSE;

   FUNCTION_START(STB_DPIsRecordingPath);

   ASSERT(path < num_paths);

   if (path < num_paths)
   {
      if (IS_RECORDING_PATH(path))
      {
         is_recording = TRUE;
      }
   }

   FUNCTION_FINISH(STB_DPIsRecordingPath);

   return(is_recording);
}

/*!**************************************************************************
 * @brief   Is the given decode path being used for decoding
 * @param   path - ID of decode path
 * @return  TRUE if the path has an audio or video decoder assigned to it
 ****************************************************************************/
BOOLEAN STB_DPIsDecodingPath(U8BIT path)
{
   BOOLEAN is_decoding = FALSE;

   FUNCTION_START(STB_DPIsDecodingPath);

   ASSERT(path < num_paths);

   if (path < num_paths)
   {
      if ((path_status[path].video_decoder_no != INVALID_RES_ID) ||
         (path_status[path].audio_decoder_no != INVALID_RES_ID))
      {
         is_decoding = TRUE;
      }
   }

   FUNCTION_FINISH(STB_DPIsDecodingPath);

   return(is_decoding);
}

/*!**************************************************************************
 * @brief   Returns the ID of the decode path being used for playback
 * @param   None
 * @return  ID of playback path or INVALID_RES_ID if none found
 ****************************************************************************/
U8BIT STB_DPGetPlaybackPath(void)
{
   U8BIT path, i;

   FUNCTION_START(STB_DPGetPlaybackPath);

   path = INVALID_RES_ID;

   for (i = 0; (i < num_paths) && (path == INVALID_RES_ID); i++)
   {
      if (IS_PLAYBACK_PATH(i))
      {
         path = i;
      }
   }

   FUNCTION_FINISH(STB_DPGetPlaybackPath);

   return(path);
}

BOOLEAN STB_DPIsMonitoringPath(U8BIT path)
{
   BOOLEAN is_monitoring = FALSE;

   FUNCTION_START(STB_DPIsMonitoringPath);

   ASSERT(path < num_paths);

   if (path < num_paths)
   {
      if (IS_MONITORING_PATH(path))
      {
         is_monitoring = TRUE;
      }
   }

   FUNCTION_FINISH(STB_DPIsMonitoringPath);

   return(is_monitoring);
}

/*!**************************************************************************
 * @brief   Returns the ID of the decode path being used for monitoring
 * @return  ID of live path or INVALID_RES_ID if none found
 ****************************************************************************/
U8BIT STB_DPGetMonitoringPath(void)
{
   U8BIT path, i;

   FUNCTION_START(STB_DPGetMonitoringPath);

   path = INVALID_RES_ID;

   for (i = 0; (i < num_paths) && (path == INVALID_RES_ID); i++)
   {
      if (IS_MONITORING_PATH(i))
      {
         path = i;
      }
   }

   FUNCTION_FINISH(STB_DPGetMonitoringPath);

   return(path);
}

/*!**************************************************************************
 * @brief   Returns the decode path that is using the given tuner ID
 * @param   start_path - path to start searching from, use INVALID_RES_ID to start from 0
 * @param   tuner_num - ID of tuner
 * @return  ID of the decode path or INVALID_RES_ID if none found
 ****************************************************************************/
U8BIT STB_DPPathForTuner(U8BIT start_path, U8BIT tuner_num)
{
   U8BIT path = INVALID_RES_ID;
   U8BIT i;

   FUNCTION_START(STB_DPPathForTuner);

   ASSERT(tuner_num < num_tuners);

   if (start_path == INVALID_RES_ID)
   {
      i = 0;
   }
   else
   {
      i = start_path + 1;
   }

   for (; (i < num_paths) && (path == INVALID_RES_ID); i++)
   {
      if (path_status[i].tuner_no == tuner_num)
      {
         path = i;
      }
   }

   FUNCTION_FINISH(STB_DPPathForTuner);

   return(path);
}

/*!**************************************************************************
 * @brief   Returns the decode path that is using the given audio decoder
 * @param   decoder_num - ID of the audio decoder
 * @return  ID of the decode path or INVALID_RES_ID if none found
 ****************************************************************************/
U8BIT STB_DPPathForAudioDecoder(U8BIT decoder_num)
{
   U8BIT path = INVALID_RES_ID;
   U8BIT i;

   FUNCTION_START(STB_DPPathForAudioDecoder);

   ASSERT(decoder_num < num_audio_decoders);

   for (i = 0; (i < num_paths) && (path == INVALID_RES_ID); i++)
   {
      if (path_status[i].audio_decoder_no == decoder_num)
      {
         path = i;
      }
   }

   FUNCTION_FINISH(STB_DPPathForAudioDecoder);

   return(path);
}

/*!**************************************************************************
 * @brief   Returns the decode path that is using the given AD decoder
 * @param   decoder_num - ID of the AD decoder
 * @return  ID of the decode path or INVALID_RES_ID if none found
 ****************************************************************************/
U8BIT STB_DPPathForADDecoder(U8BIT decoder_num)
{
   U8BIT path = INVALID_RES_ID;
   U8BIT i;

   FUNCTION_START(STB_DPPathForADDecoder);

   ASSERT(decoder_num < num_audio_decoders);

   for (i = 0; (i < num_paths) && (path == INVALID_RES_ID); i++)
   {
      if (path_status[i].ad_decoder_no == decoder_num)
      {
         path = i;
      }
   }

   FUNCTION_FINISH(STB_DPPathForADDecoder);

   return(path);
}

/*!**************************************************************************
 * @brief   Returns the decode path that is using the given video decoder
 * @param   decoder_num - ID of the video decoder
 * @return  ID of the decode path or INVALID_RES_ID if none found
 ****************************************************************************/
U8BIT STB_DPPathForVideoDecoder(U8BIT decoder_num)
{
   U8BIT path = INVALID_RES_ID;
   U8BIT i;

   FUNCTION_START(STB_DPPathForVideoDecoder);

   ASSERT(decoder_num < num_video_decoders);

   for (i = 0; (i < num_paths) && (path == INVALID_RES_ID); i++)
   {
      if (path_status[i].video_decoder_no == decoder_num)
      {
         path = i;
      }
   }

   FUNCTION_FINISH(STB_DPPathForVideoDecoder);

   return(path);
}

/*!**************************************************************************
 * @brief   Saves the given transport with the specified decode path
 * @param   path - ID of decode path
 * @param   transport - pointer to transport to be saved
 * @return  None
 ****************************************************************************/
void STB_DPSetTunedTransport(U8BIT path, void *transport)
{
   FUNCTION_START(STB_DPSetTunedTransport);

   ASSERT(path < num_paths);

   if ((path < num_paths) && (path_status[path].tuner_no != INVALID_RES_ID))
   {
      STB_RESSetTunedTransport(path_status[path].tuner_no, transport);
   }

   FUNCTION_FINISH(STB_DPSetTunedTransport);
}

/*!**************************************************************************
 * @brief   Returns the transport saved with the given decode path
 * @param   path - ID of decode path
 * @return  Pointer to saved transport
 ****************************************************************************/
void* STB_DPGetTunedTransport(U8BIT path)
{
   void *transport = NULL;

   FUNCTION_START(STB_DPGetTunedTransport);

   ASSERT(path < num_paths);

   if ((path < num_paths) && (path_status[path].tuner_no != INVALID_RES_ID))
   {
      if (tuner_status[path_status[path].tuner_no].tune_stat == TUNE_LOCKED)
      {
         transport = STB_RESGetTunedTransport(path_status[path].tuner_no);
      }
   }

   FUNCTION_FINISH(STB_DPGetTunedTransport);

   return(transport);
}

/*!**************************************************************************
 * @brief   Saves the given service with the specified decode path
 * @param   path - ID of decode path
 * @param   service - pointer to service to be saved
 ****************************************************************************/
void STB_DPSetTunedService(U8BIT path, void *service)
{
   FUNCTION_START(STB_DPSetTunedService);

   ASSERT(path < num_paths);

   if ((path < num_paths) && path_status[path].in_use)
   {
      path_status[path].service = service;
   }

   FUNCTION_FINISH(STB_DPSetTunedService);
}

/*!**************************************************************************
 * @brief   Returns the service saved with the given decode path
 * @param   path - ID of decode path
 * @return  Pointer to saved service
 ****************************************************************************/
void* STB_DPGetTunedService(U8BIT path)
{
   void *service = NULL;

   FUNCTION_START(STB_DPGetTunedService);

   ASSERT(path < num_paths);

   if ((path < num_paths) && path_status[path].in_use)
   {
      service = path_status[path].service;
   }

   FUNCTION_FINISH(STB_DPGetTunedService);

   return(service);
}

/*!**************************************************************************
 * @brief   Returns the tuner ID acquired by the given decode path
 * @param   path - ID of decode path
 * @return  ID of tuner used by the decode path
 ****************************************************************************/
U8BIT STB_DPGetPathTuner(U8BIT path)
{
   U8BIT tuner_num = INVALID_RES_ID;

   FUNCTION_START(STB_DPGetPathTuner);

   ASSERT(path < num_paths);

   if (path < num_paths)
   {
      tuner_num = path_status[path].tuner_no;
   }

   FUNCTION_FINISH(STB_DPGetPathTuner);

   return(tuner_num);
}

/*!**************************************************************************
 * @brief   Returns the tuner type for the given path
 * @param   path - ID of decode path
 * @return  tuner type
 ****************************************************************************/
E_STB_DP_SIGNAL_TYPE STB_DPGetPathTunerType(U8BIT path)
{
   E_STB_DP_SIGNAL_TYPE type;

   FUNCTION_START(STB_DPGetPathTunerType);

   ASSERT(path < num_paths);

   type = SIGNAL_NONE;

   if (path < num_paths)
   {
      type = STB_RESGetTunerType(path_status[path].tuner_no);
   }

   FUNCTION_FINISH(STB_DPGetPathTunerType);

   return(type);
}

/**
 * @brief   Sets the 'disabled' state of all tuners to FALSE so they can all be used
 *          by the resource manager
 */
void STB_DPEnableAllTuners(void)
{
   U8BIT id;

   FUNCTION_START(STB_DPEnableAllTuners);

   for (id = 0; id < num_tuners; id++)
   {
      STB_RESSetTunerDisabled(id, FALSE);
   }

   FUNCTION_FINISH(STB_DPEnableAllTuners);
}

/**
 * @brief   Disables/enables the tuner associated with this decode path
 * @param   path decode path
 * @param   disabled TRUE if the tuner is to be disabled, FALSE otherwise
 */
void STB_DPSetTunerDisabled(U8BIT path, BOOLEAN disabled)
{
   FUNCTION_START(STB_DPSetTunerDisabled);

   if ((path < num_paths) && (path_status[path].tuner_no != INVALID_RES_ID))
   {
      STB_RESSetTunerDisabled(path_status[path].tuner_no, disabled);
   }

   FUNCTION_FINISH(STB_DPSetTunerDisabled);
}

/**
 * @brief   Returns the number of enabled tuners (i.e. that haven't been disabled)
 * @return  number of enabled tuners
 */
U8BIT STB_DPGetNumEnabledTuners(void)
{
   FUNCTION_START(STB_DPGetNumEnabledTuners);
   FUNCTION_FINISH(STB_DPGetNumEnabledTuners);
   return(STB_RESNumEnabledTuners());
}

/**
 * @brief   Returns the 'enabled' status for the given tuner number
 * @param   tuner_num number of the tuner to be checked
 * @return  TRUE if the tuner number is valid and the tuner is enabled, FALSE otherwise
 */
BOOLEAN STB_DPIsTunerEnabled(U8BIT tuner_num)
{
   FUNCTION_START(STB_DPIsTunerEnabled);
   FUNCTION_FINISH(STB_DPIsTunerEnabled);
   return(!STB_RESIsTunerDisabled(tuner_num));
}

/*!**************************************************************************
 * @brief   Returns the demux path ID acquired by the given decode path
 * @param   path - ID of decode path
 * @return  ID of demux path used by the decode path
 ****************************************************************************/
U8BIT STB_DPGetPathDemux(U8BIT path)
{
   U8BIT demux_num = INVALID_RES_ID;

   FUNCTION_START(STB_DPGetPathDemux);

   ASSERT(path < num_paths);

   if (path < num_paths)
   {
      demux_num = path_status[path].demux_no;
   }

   FUNCTION_FINISH(STB_DPGetPathDemux);

   return(demux_num);
}

/*!**************************************************************************
 * @brief   Returns the audio decoder ID acquired by the given decode path
 * @param   path - ID of decode path
 * @return  ID of audio decoder used by the decode path
 ****************************************************************************/
U8BIT STB_DPGetPathAudioDecoder(U8BIT path)
{
   U8BIT decoder_num = INVALID_RES_ID;

   FUNCTION_START(STB_DPGetPathAudioDecoder);

   ASSERT(path < num_paths);

   if (path < num_paths)
   {
      decoder_num = path_status[path].audio_decoder_no;
   }

   FUNCTION_FINISH(STB_DPGetPathAudioDecoder);

   return(decoder_num);
}

/*!**************************************************************************
 * @brief   Returns the video decoder ID acquired by the given decode path
 * @param   path - ID of decode path
 * @return  ID of video decoder used by the decode path
 ****************************************************************************/
U8BIT STB_DPGetPathVideoDecoder(U8BIT path)
{
   U8BIT decoder_num = INVALID_RES_ID;

   FUNCTION_START(STB_DPGetPathVideoDecoder);

   ASSERT(path < num_paths);

   if (path < num_paths)
   {
      decoder_num = path_status[path].video_decoder_no;
   }

   FUNCTION_FINISH(STB_DPGetPathVideoDecoder);

   return(decoder_num);
}

/*!**************************************************************************
 * @brief   Returns the secondary video decoder ID acquired by the given
 *          decode path
 * @param   path - ID of decode path
 * @return  ID of secondary video decoder used by the decode path
 ****************************************************************************/
U8BIT STB_DPGetPathSecondaryVideoDecoder(U8BIT path)
{
   U8BIT decoder_num = INVALID_RES_ID;

   FUNCTION_START(STB_DPGetPathSecondaryVideoDecoder);

   ASSERT(path < num_paths);

   if (path < num_paths)
   {
      decoder_num = path_status[path].secondary_video_decoder_no;
   }

   FUNCTION_FINISH(STB_DPGetPathSecondaryVideoDecoder);

   return(decoder_num);
}

/*!**************************************************************************
 * @brief   Checks whether any of the paths are tuned to the given service
 * @param   service - service to look for
 * @return  The path of the given service, or INVALID_RES_ID if it isn't found
 ****************************************************************************/
U8BIT STB_DPGetPathForService(void *service)
{
   U8BIT path;
   U8BIT tuned_path = INVALID_RES_ID;

   FUNCTION_START(STB_DPGetPathForService);

   for (path = 0; (path < num_paths) && (tuned_path == INVALID_RES_ID); path++)
   {
      if (path_status[path].in_use && (path_status[path].service == service))
      {
         tuned_path = path;
      }
   }

   FUNCTION_FINISH(STB_DPGetPathForService);

   return(tuned_path);
}

/**
 * @brief   Checks whether there's a tuner available to tune to the given service or transport
 * @param   service service to be tuned to, can be NULL
 * @param   transport transport to be tuned to
 * @return  TRUE if there's a tuner available
 */
BOOLEAN STB_DPCanTuneTo(E_STB_DP_SIGNAL_TYPE tuner_type, void *service, void *transport)
{
   BOOLEAN can_tune;
   U8BIT i;

   FUNCTION_START(STB_DPCanTuneTo);

   can_tune = FALSE;

   if (service != NULL)
   {
      /* Check if there's a path already tuned to this service */
      for (i = 0; (i < num_paths) && !can_tune; i++)
      {
         if (path_status[i].in_use && (path_status[i].service == service))
         {
            can_tune = TRUE;
         }
      }
   }

   if (!can_tune)
   {
      /* See if there's a tuner tuned to the transport */
      can_tune = STB_RESCanTuneToTransport(tuner_type, transport);
   }

   FUNCTION_FINISH(STB_DPCanTuneTo);

   return(can_tune);
}

/**
 *

 *
 * @brief   Returns the path for Cabot MHEG software to use
 *

 *
 * @return   U8BIT - path to search for MHEG applications
 *
 */
U8BIT STB_DPGetMHEGPath(void)
{
   FUNCTION_START(STB_DPGetMHEGPath);
   FUNCTION_FINISH(STB_DPGetMHEGPath);
   return(STB_DPGetLivePath());
}

/*!**************************************************************************
 * @brief   Sets the owner of the given path
 * @param   path - decode path
 * @param   owner - owner module
 ****************************************************************************/
void STB_DPSetOwner(U8BIT path, E_STB_DP_RES_OWNER owner)
{
   FUNCTION_START(STB_DPSetOwner);

   if (path < num_paths)
   {
      path_status[path].owner = owner;

      if (path_status[path].tuner_no != INVALID_RES_ID)
      {
         STB_RESSetTunerOwner(path_status[path].tuner_no, owner);
      }
   }

   FUNCTION_FINISH(STB_DPSetOwner);
}

/*!**************************************************************************
 * @brief   Saves a copy of the given data with the path. If data is passed as NULL
 *          and owner data already exists, then it will be freed. if data isn't NULL
 *          and owner data exists then the new data will replace the existing data.
 * @param   path - decode path
 * @param   data - data to be saved with the path, can be NULL
 * @param   data_size - size of data, 0 if data is NULL
 * @return  TRUE if data is successfully saved, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_DPSetOwnerData(U8BIT path, void *data, U32BIT data_size)
{
   BOOLEAN retval;

   FUNCTION_START(STB_DPSetOwnerData);

   retval = FALSE;

   if (path < num_paths)
   {
      if (data != NULL)
      {
         if (path_status[path].owner_data != NULL)
         {
            /* Free existing data */
            STB_FreeMemory(path_status[path].owner_data);

            path_status[path].owner_data = NULL;
            path_status[path].owner_data_size = 0;
         }

         /* Copy the data */
         if ((path_status[path].owner_data = STB_GetMemory(data_size)) != NULL)
         {
            memcpy(path_status[path].owner_data, data, data_size);
            path_status[path].owner_data_size = data_size;
            retval = TRUE;
         }
      }
      else
      {
         if (path_status[path].owner_data != NULL)
         {
            /* Free existing data */
            STB_FreeMemory(path_status[path].owner_data);

            path_status[path].owner_data = NULL;
            path_status[path].owner_data_size = 0;
         }

         retval = TRUE;
      }
   }

   FUNCTION_FINISH(STB_DPSetOwnerData);

   return(retval);
}

/*!**************************************************************************
 * @brief   Checks whether the path is owned by the given owner
 * @param   path - decode path
 * @param   owner - is path owned by this module
 * @return  TRUE if owned by the given module, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_DPIsOwnedBy(U8BIT path, E_STB_DP_RES_OWNER owner)
{
   BOOLEAN retval;

   FUNCTION_START(STB_DPIsOwnedBy);

   retval = FALSE;

   if (path < num_paths)
   {
      if (path_status[path].owner == owner)
      {
         retval = TRUE;
      }
   }

   FUNCTION_FINISH(STB_DPIsOwnedBy);

   return(retval);
}

/*!**************************************************************************
 * @brief   Returns the owner data saved with the path. This data should not be freed.
 * @param   path - decode path
 * @param   data_size - size of the returned data, 0 if return is NULL
 * @return  Pointer to the owner data, which can be NULL.
 ****************************************************************************/
void* STB_DPGetOwnerData(U8BIT path, U32BIT *data_size)
{
   void *data;

   FUNCTION_START(STB_DPGetTunerOwnerData);

   if (path < num_paths)
   {
      data = path_status[path].owner_data;
      *data_size = path_status[path].owner_data_size;
   }
   else
   {
      data = NULL;
      *data_size = 0;
   }

   FUNCTION_FINISH(STB_DPGetTunerOwnerData);

   return(data);
}

/**
 * @brief   Sets source for specified decoder.
 * @param   U8BIT path - the ID of the path to use
 * @param   E_STB_DP_DECODE_SOURCE source - the source type to use
 * @param   U32BIT param - paramter of the source type
 */
void STB_DPSetDecodeSource(U8BIT path, E_STB_DP_DECODE_SOURCE source, U32BIT param)
{
   U8BIT audio_num, video_num;
   U8BIT ad_num;
   U8BIT demux_num;
   E_STB_AV_DECODE_SOURCE av_source;

   FUNCTION_START(STB_DPSetDecodeSource);

   ASSERT(path < num_paths);

   STB_DP_PRINT(("STB_DPSetDecodeSource(%d): %d 0x%x", path, source, param));

   ad_num = path_status[path].ad_decoder_no;
   audio_num = path_status[path].audio_decoder_no;
   video_num = path_status[path].video_decoder_no;
   demux_num = path_status[path].demux_no;

   STB_DP_PRINT(("STB_DPSetDecodeSource(%d): audio = %d, AD = %d, video = %d", path, audio_num, ad_num, video_num));

   if ((audio_num != INVALID_RES_ID) && (video_num != INVALID_RES_ID))
   {
      if ((source != audio_decoder_status[audio_num].source_type) ||
          (param != audio_decoder_status[audio_num].source_param) ||
          (source != video_decoder_status[video_num].source_type) ||
          (param != video_decoder_status[video_num].source_param) ||
          (source != ad_decoder_status[ad_num].source_type) ||
          (param != ad_decoder_status[ad_num].source_param) ||
          (demux_num != ad_decoder_status[ad_num].source_dmx) ||
          (demux_num != audio_decoder_status[audio_num].source_dmx) ||
          (demux_num != video_decoder_status[video_num].source_dmx))
      {
         ad_decoder_status[ad_num].source_type = source;
         ad_decoder_status[ad_num].source_param = param;
         ad_decoder_status[ad_num].actions |= ACTION_AD;

         audio_decoder_status[audio_num].source_type = source;
         audio_decoder_status[audio_num].source_param = param;
         audio_decoder_status[audio_num].actions |= ACTION_AUDIO;

         video_decoder_status[video_num].source_type = source;
         video_decoder_status[video_num].source_param = param;
         video_decoder_status[video_num].actions |= ACTION_VIDEO;

         // select correct av source
         switch (source)
         {
            case DECODE_SOURCE_DEMUX:  {av_source = AV_DEMUX;  break; }
            case DECODE_SOURCE_FILE:   {av_source = AV_MEMORY; break; }
            default:                   {av_source = AV_DEMUX; break; }
         }
         STB_AVSetADSource(ad_num, av_source, demux_num);
         STB_AVSetAudioSource(audio_num, av_source, demux_num);
         STB_AVSetVideoSource(video_num, av_source, demux_num);
      }
   }

   FUNCTION_FINISH(STB_DPSetDecodeSource);
}

/**
 * @brief   Gets source for specified decoder.
 * @param   U8BIT decoder - the ID of the decode path to use
 * @param   E_STB_DP_DECODE_SOURCE* source - returns the source type
 * @param   U32BIT* param - returns the parameter of the source type
 */
void STB_DPGetDecodeSource(U8BIT path, E_STB_DP_DECODE_SOURCE *source, U32BIT *param)
{
   U8BIT decode_num;

   FUNCTION_START(STB_DPGetDecodeSource);

   ASSERT(path < num_paths);

   decode_num = path_status[path].video_decoder_no;

   if (decode_num != INVALID_RES_ID)
   {
      *source = video_decoder_status[decode_num].source_type;
      *param = video_decoder_status[decode_num].source_param;
   }

   FUNCTION_FINISH(STB_DPGetDecodeSource);
}

/**
 * @brief   Requests start of recording on specified path.
 * @param   U8BIT path - the ID of the decode path to use
 * @param   U32BIT param - file handle for recording
 * @return   BOOLEAN - indicates whether recording has started successfully or not
 */
BOOLEAN STB_DPStartRecording(U8BIT path, U32BIT param)
{
   U32BIT temp_val;
   U8BIT demux_num;
   BOOLEAN retval = FALSE;

   FUNCTION_START(STB_DPStartRecording);

   ASSERT(path < num_paths);

   STB_DP_PRINT(("STB_DPStartRecording(%d)", path));

   if (STB_PVRIsInitialised())
   {
      /* Stop old PVR recording? */
      if (STB_PVRIsRecording(path, &temp_val) == TRUE)
      {
         STB_PVRStopRecording(path);
      }

      demux_num = path_status[path].demux_no;

      ASSERT(demux_num != INVALID_RES_ID);

      if ((demux_status[demux_num].actions & ACTION_DECODE_MASK) != 0)
      {
         STB_DP_PRINT(("STB_DPStartRecording(%u): Updating audio/video/pcr/AD PIDs (%u/%u/%u/%u) for demux %u", path,
                       demux_status[demux_num].audio_pid, demux_status[demux_num].video_pid,
                       demux_status[demux_num].pcr_pid, demux_status[demux_num].AD_pid, demux_num));

         /* Change of decode params - so update PIDs before recording is started */
         demux_status[demux_num].actions &= ~ACTION_DECODE_MASK;

         STB_DMXChangeDecodePIDs(demux_num, demux_status[demux_num].pcr_pid,
            demux_status[demux_num].video_pid, demux_status[demux_num].audio_pid,
            demux_status[demux_num].text_pid, demux_status[demux_num].data_pid,
            demux_status[demux_num].AD_pid);

         STB_DMXChangeEcmPIDs(
            demux_num,
            demux_status[demux_num].ecm_pid,
            demux_status[demux_num].video_ecm_pid,
            demux_status[demux_num].audio_ecm_pid,
            demux_status[demux_num].text_ecm_pid,
            demux_status[demux_num].data_ecm_pid,
            demux_status[demux_num].AD_ecm_pid
         );
      }

      /* Start new PVR recording */
      retval = STB_PVRStartRecording(path, param);
   }

   FUNCTION_FINISH(STB_DPStartRecording);

   return(retval);
}

/**
 * @brief   Requests stop of recording on specified path.
 * @param   U8BIT path - the ID of the decode path to use
 */
void STB_DPStopRecording(U8BIT path)
{
   U32BIT temp_val;

   FUNCTION_START(STB_DPStopRecording);

   ASSERT(path < num_paths);

   STB_DP_PRINT(("STB_DPStopRecording(%d)", path));

   // stop current PVR recording
   if (STB_PVRIsRecording(path, &temp_val) == TRUE)
   {
      STB_PVRStopRecording(path);
   }

   FUNCTION_FINISH(STB_DPStopRecording);
}

/**
 * @brief   Returns status of recording on specified path.
 * @param   U8BIT path - the ID of the decode path to use
 * @param   U32BIT* handle - returns handle in use
 * @return   BOOLEAN - TRUE if recording is in progress.
 */
BOOLEAN STB_DPIsRecording(U8BIT path, U32BIT *handle)
{
   BOOLEAN ret_val;

   FUNCTION_START(STB_DPIsRecording);

   ret_val = STB_PVRIsRecording(path, handle);

   FUNCTION_FINISH(STB_DPIsRecording);

   return(ret_val);
}

/**
 * @brief   Requests start of tuning process.
 * @param   U8BIT path - the ID of the decode path to use
 */
void STB_DPStartTune(U8BIT path)
{
   E_STB_DP_TUNE_STATUS status;
   U8BIT tuner_num;

   FUNCTION_START(STB_DPStartTune);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;
   status = STB_DPGetTuneStatus(path);
   STB_TUNE_PRINT(("STB_DPStartTune(%d): Tuner %d, status=%d", path, tuner_num, status));

   if ((tuner_status[tuner_num].actions & ACTION_TUNE) != 0)
   {
      // change of tuning params - so stop SI and force re-tune
      tuner_status[tuner_num].actions &= ~ACTION_TUNE;
      STB_DPStopSI(path);

      status = TUNE_NO_LOCK;
      tuner_status[tuner_num].actions |= ACTION_SI_NEW;
   }

   switch (status)
   {
      case TUNE_WAITING:
      case TUNE_NO_LOCK:
      case TUNE_STOPPED:
      {
         STB_TUNE_PRINT(("STB_DPStartTune(%d): re-tune", path));

         // setup dish / lnb / DiSEqC and request tune
         if (StartTune(path) == FALSE)
         {
            STB_TUNE_PRINT(("STB_DPStartTune(%d): Tune failed", path));
            /* If StartTune fails then it won't have attempted to start tuning so
             * send tune event from here */
            STB_DPSetTuneStatus(path, TUNE_NO_LOCK);
            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_TUNE, EV_TYPE_NOTLOCKED, &tuner_num, sizeof(U8BIT));
         }
         break;
      }

      case TUNE_LOCKED:
      {
         STB_TUNE_PRINT(("STB_DPStartTune(%d): re-tune not required", path));
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_TUNE, EV_TYPE_LOCKED, &tuner_num, sizeof(U8BIT));
         break;
      }

      default:
      {
         STB_TUNE_PRINT(("STB_DPStartTune(%d): no action", path));
         break;
      }
   }

   FUNCTION_FINISH(STB_DPStartTune);
}

/**
 * @brief   Requests start of tuning signal scan process.
 * @param   U8BIT path - the ID of the decode path to use
 */
void STB_DPStartScan(U8BIT path)
{
   FUNCTION_START(STB_DPStartScan);

   USE_UNWANTED_PARAM(path);

   ASSERT(path < num_paths);

   FUNCTION_FINISH(STB_DPStartScan);
}

/**
 * @brief   Stops of tuning signal scan and unlocks.
 * @param   U8BIT path - the ID of the decode path to use
 */
void STB_DPStopTune(U8BIT path)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPStopTune);

   ASSERT(path < num_paths);

   switch (STB_DPGetTuneStatus(path))
   {
      case TUNE_WAITING:
      case TUNE_NO_LOCK:
      case TUNE_LOCKED:
      {
         tuner_num = path_status[path].tuner_no;

         STB_TUNE_PRINT(("STB_DPStopTune(%d): stop tuner %d", path, tuner_num));

         STB_DPStopSI(path);

         /* Set path status */
         STB_DPSetTuneStatus(path, TUNE_STOPPED);

         /* Only stop the tuner if it isn't being used by another path */
         if (STB_RESTunerUsageCount(tuner_num) < 2)
         {
            STB_TuneStopTuner(tuner_num);
            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_TUNE, EV_TYPE_STOPPED, &tuner_num, sizeof(U8BIT));
         }
         break;
      }

      default:
      {
         STB_TUNE_PRINT(("STB_DPStopTune(%d): no action", path));
         break;
      }
   }

   FUNCTION_FINISH(STB_DPStopTune);
}

/**
 * @brief   Requests stop of tuning and powers down tuner.
 * @param   U8BIT path - the ID of the decode path to use
 */
void STB_DPTuneOff(U8BIT path)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPTuneOff);

   ASSERT(path < num_paths);

   switch (STB_DPGetTuneStatus(path))
   {
      case TUNE_WAITING:
      case TUNE_NO_LOCK:
      case TUNE_LOCKED:
         tuner_num = path_status[path].tuner_no;
         if (tuner_num != INVALID_RES_ID)
         {
            STB_TUNE_PRINT(("STB_DPTuneOff(%d): Tuner %d off", path, tuner_num));

            STB_DPStopSI(path);

            /* Set path status */
            STB_DPSetTuneStatus(path, TUNE_STOPPED);

            /* Only stop the tuner if it isn't being used by another path */
            if (STB_RESTunerUsageCount(tuner_num) < 2)
            {
               if (TRUE == STB_DPIsMonitoringPath(path)) {
                  if (TRUE == STB_DPGetSearchMode(path))
                     STB_TuneStopTuner(tuner_num);
                  else
                     STB_TuneStopTuner(tuner_num | (1 << 7));
               }
               else
                  STB_TuneStopTuner(tuner_num);

               if (tuner_status[tuner_num].signal_type == SIGNAL_QPSK)
               {
                  if (tuner_status[tuner_num].lnb_power == LNB_POWER_AUTO)
                  {
                     STB_TuneSetLNBVoltage(tuner_num, LNB_VOLTAGE_OFF);
                     STB_TuneSet22kState(tuner_num, FALSE);
                  }
               }

               /* Send an event to indicate the tuner has been stopped */
               STB_ERSendEvent(FALSE, FALSE, EV_CLASS_TUNE, EV_TYPE_STOPPED, &tuner_num, sizeof(U8BIT));
            }
         }
         break;

      default:
         STB_TUNE_PRINT(("STB_DPTuneOff(%d): status=%u, no action", path, STB_DPGetTuneStatus(path)));
         break;
   }

   // On any applying of power will now cause a delay of 1s
   // ONLY if a DiSEqC 1.2 motor is fitted.
   lnb_currently_powered = FALSE;

   FUNCTION_FINISH(STB_DPTuneOff);
}

/**
 * @brief   Requests start of channel decoding process.
 * @param   U8BIT path - the ID of the decode path to use
 */
void STB_DPStartVideoDecoding(U8BIT path)
{
   BOOLEAN lock = FALSE;
   E_STB_DP_DECODE_STATUS status;
   U8BIT decoder_num;
   U8BIT demux_num;

   FUNCTION_START(STB_DPStartVideoDecoding);

   ASSERT(path < num_paths);

   status = STB_DPGetVideoStatus(path);

   decoder_num = path_status[path].video_decoder_no;
   demux_num = path_status[path].demux_no;

   if ((demux_status[demux_num].actions & ACTION_VIDEO) != 0)
   {
      /* Change of demux params so force re-start */
      demux_status[demux_num].actions &= ~ACTION_VIDEO;
      status = DECODE_STOPPED;

      /* Only use PIDs if source is from demux */
      if (video_decoder_status[decoder_num].source_type == DECODE_SOURCE_DEMUX)
      {
         STB_DP_PRINT(("STB_DPStartVideoDecoding(%d): setting pids (video=%u, audio=%u, ad=%u)", path,
                       demux_status[demux_num].video_pid, demux_status[demux_num].audio_pid, demux_status[demux_num].AD_pid));
         /* Update PIDs */
         STB_DMXChangeDecodePIDs(demux_num, demux_status[demux_num].pcr_pid,
            demux_status[demux_num].video_pid, demux_status[demux_num].audio_pid,
            demux_status[demux_num].text_pid, demux_status[demux_num].data_pid,
            demux_status[demux_num].AD_pid);

         STB_DMXChangeEcmPIDs(
            demux_num,
            demux_status[demux_num].ecm_pid,
            demux_status[demux_num].video_ecm_pid,
            demux_status[demux_num].audio_ecm_pid,
            demux_status[demux_num].text_ecm_pid,
            demux_status[demux_num].data_ecm_pid,
            demux_status[demux_num].AD_ecm_pid
         );
      }
   }

   if ((video_decoder_status[decoder_num].actions & ACTION_VIDEO) != 0)
   {
      /* Change of video decoder params so force re-start */
      video_decoder_status[decoder_num].actions &= ~ACTION_VIDEO;
      status = DECODE_STOPPED;
   }

   lock = (path_status[path].lock_mode && path_status[path].lock_enable);

   if (lock == TRUE)
   {
      STB_DP_PRINT(("STB_DPStartVideoDecoding(%d): locked", path));
      STB_DPSetVideoStatus(path, DECODE_LOCKED);
      /*STB_AVBlankVideo(decoder_num, TRUE);*/
      STB_AVStopVideoDecoding(decoder_num);
      STB_ERSendEvent(FALSE, FALSE, EV_CLASS_DECODE, EV_TYPE_LOCKED, &decoder_num, sizeof(U8BIT));
   }
   else
   {
      switch (status)
      {
         case DECODE_STOPPING:
         case DECODE_STOPPED:
         case DECODE_LOCKED:
         {
            // request start decode
            STB_DP_PRINT(("STB_DPStartVideoDecoding(%d): starting video decoder %d", path, decoder_num));
            STB_DPSetVideoStatus(path, DECODE_STARTING);
            STB_AVStartVideoDecoding(decoder_num);
            break;
         }

         case DECODE_RUNNING:
         {
            STB_DP_PRINT(("STB_DPStartVideoDecoding(%d): re-start not required", path));
            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_DECODE, EV_TYPE_VIDEO_STARTED, &decoder_num, sizeof(U8BIT));
            break;
         }

         default:
         {
            STB_DP_PRINT(("STB_DPStartVideoDecoding(%d): no action", path));
            break;
         }
      }
   }

   FUNCTION_FINISH(STB_DPStartVideoDecoding);
}

/**
 * @brief   Requests start of channel decoding process.
 * @param   U8BIT decoder - the ID of the decode path to use
 */
void STB_DPStartADDecoding(U8BIT path)
{
   E_STB_AV_AUDIO_MODE audio_mode;
   BOOLEAN lock = FALSE;
   E_STB_DP_DECODE_STATUS status;
   U8BIT decoder_num;
   U8BIT demux_num;
   U8BIT ad_status;

   FUNCTION_START(STB_DPStartADDecoding);

   ASSERT(path < num_paths);

   status = STB_DPGetADStatus(path);
   ad_status = GetADEnabled(path);
   decoder_num = path_status[path].ad_decoder_no;
   demux_num = path_status[path].demux_no;
   ASSERT(demux_num < num_demuxes);

   STB_DP_PRINT(("STB_DPStartADDecoding %s, %s\n", (ad_status & AD_ENABLED) ? "On" : "Off",
                 (ad_status & AD_PID_PRESENT) ? "PID" : "No PID"));

   if ((demux_status[demux_num].actions & ACTION_AD) != 0)
   {
      /* Change of demux params so force re-start */
      demux_status[demux_num].actions &= ~ACTION_AD;
      status = DECODE_STOPPED;
      if (ad_decoder_status[decoder_num].source_type == DECODE_SOURCE_DEMUX)
      {
         STB_DMXChangeDecodePIDs(demux_num, demux_status[demux_num].pcr_pid,
            demux_status[demux_num].video_pid, demux_status[demux_num].audio_pid,
            demux_status[demux_num].text_pid, demux_status[demux_num].data_pid,
            demux_status[demux_num].AD_pid);

         STB_DMXChangeEcmPIDs(
            demux_num,
            demux_status[demux_num].ecm_pid,
            demux_status[demux_num].video_ecm_pid,
            demux_status[demux_num].audio_ecm_pid,
            demux_status[demux_num].text_ecm_pid,
            demux_status[demux_num].data_ecm_pid,
            demux_status[demux_num].AD_ecm_pid
         );
      }
   }

   if ((ad_status & AD_ENABLED) && (ad_status & AD_PID_PRESENT))
   {
      if ((ad_decoder_status[decoder_num].actions & ACTION_AD) != 0)
      {
         // change of decode params - so force re-start
         ad_decoder_status[decoder_num].actions &= ~ACTION_AD;
         status = DECODE_STOPPED;
      }

      /* Note that AD is mono but in the future it may be stereo, also broadcasters may play on one channel */
      switch (ad_decoder_status[decoder_num].audio_mode)
      {
         case AUDIO_MONO:     {audio_mode = AV_AUDIO_MONO;     break; }
         case AUDIO_STEREO:   {audio_mode = AV_AUDIO_STEREO;   break; }
         case AUDIO_LEFT:     {audio_mode = AV_AUDIO_LEFT;     break; }
         case AUDIO_RIGHT:    {audio_mode = AV_AUDIO_RIGHT;    break; }
         case AUDIO_MULTICHANNEL: {audio_mode = AV_AUDIO_MULTICHANNEL; break; }
         default:             {audio_mode = AV_AUDIO_MONO;     break; }
      }

      STB_AVChangeADMode(decoder_num, audio_mode);

      lock = (path_status[path].lock_mode && path_status[path].lock_enable);
      if (lock == TRUE)
      {
         STB_DP_PRINT(("STB_DPStartADDecoding(%d): locked", path));
         STB_DPSetADStatus(path, DECODE_LOCKED);
         STB_AVStopADDecoding(decoder_num);
         STB_DPSetADAudio(path, AD_AUDIO_OFF);
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_DECODE, EV_TYPE_LOCKED, &decoder_num, sizeof(U8BIT));
      }
      else
      {
         switch (status)
         {
            case DECODE_STOPPING:
            case DECODE_STOPPED:
            case DECODE_LOCKED:
            {
               // request start decode
               STB_DP_PRINT(("STB_DPStartADDecoding(%d): start", path));
               STB_DPSetADStatus(path, DECODE_STARTING);
               ad_decoder_status[decoder_num].ad_status |= AD_ENABLED;
               STB_AVStartADDecoding(decoder_num);
               STB_DPSetADAudio(path, AD_AUDIO_PLAYING);
               break;
            }
            case DECODE_RUNNING:
            {
               STB_DP_PRINT(("STB_DPStartADDecoding(%d): re-start not required", path));
               ad_decoder_status[decoder_num].ad_audio = AD_AUDIO_PLAYING;
               STB_ERSendEvent(FALSE, FALSE, EV_CLASS_DECODE, EV_TYPE_AD_STARTED, &decoder_num, sizeof(U8BIT));
               break;
            }
            default:
            {
               STB_DP_PRINT(("STB_DPStartADDecoding(%d): no action", path));
               break;
            }
         }
      }
   }

   FUNCTION_FINISH(STB_DPStartADDecoding);
}

/**
 * @brief   Requests start of channel decoding process.
 * @param   U8BIT decoder - the ID of the decode path to use
 */
void STB_DPStartAudioDecoding(U8BIT path)
{
   E_STB_AV_AUDIO_MODE audio_mode;
   BOOLEAN lock = FALSE;
   E_STB_DP_DECODE_STATUS status;
   U8BIT decoder_num;
   U8BIT demux_num;

   FUNCTION_START(STB_DPStartAudioDecoding);

   ASSERT(path < num_paths);

   status = STB_DPGetAudioStatus(path);

   decoder_num = path_status[path].audio_decoder_no;
   demux_num = path_status[path].demux_no;

   ASSERT(demux_num < num_demuxes);

   if ((demux_status[demux_num].actions & ACTION_AUDIO) != 0)
   {
      /* Change of demux params so force re-start */
      demux_status[demux_num].actions &= ~ACTION_AUDIO;
      status = DECODE_STOPPED;

      if (audio_decoder_status[decoder_num].source_type == DECODE_SOURCE_DEMUX)
      {
         STB_DMXChangeDecodePIDs(demux_num, demux_status[demux_num].pcr_pid,
            demux_status[demux_num].video_pid, demux_status[demux_num].audio_pid,
            demux_status[demux_num].text_pid, demux_status[demux_num].data_pid,
            demux_status[demux_num].AD_pid);

         STB_DMXChangeEcmPIDs(
            demux_num,
            demux_status[demux_num].ecm_pid,
            demux_status[demux_num].video_ecm_pid,
            demux_status[demux_num].audio_ecm_pid,
            demux_status[demux_num].text_ecm_pid,
            demux_status[demux_num].data_ecm_pid,
            demux_status[demux_num].AD_ecm_pid
         );
      }
   }

   if ((audio_decoder_status[decoder_num].actions & ACTION_AUDIO) != 0)
   {
      // change of decode params - so force re-start
      audio_decoder_status[decoder_num].actions &= ~ACTION_AUDIO;
      status = DECODE_STOPPED;
   }

   // select correct audio mode
   switch (audio_decoder_status[decoder_num].audio_mode)
   {
      case AUDIO_MONO:     {audio_mode = AV_AUDIO_MONO;    break; }
      case AUDIO_STEREO:   {audio_mode = AV_AUDIO_STEREO;  break; }
      case AUDIO_LEFT:     {audio_mode = AV_AUDIO_LEFT;    break; }
      case AUDIO_RIGHT:    {audio_mode = AV_AUDIO_RIGHT;   break; }
      case AUDIO_MULTICHANNEL: {audio_mode = AV_AUDIO_MULTICHANNEL; break; }
      default:             {audio_mode = AV_AUDIO_STEREO;  break; }
   }

   STB_AVChangeAudioMode(decoder_num, audio_mode);

   lock = (path_status[path].lock_mode && path_status[path].lock_enable);

   if (lock == TRUE)
   {
      STB_DP_PRINT(("STB_DPStartAudioDecoding(%d): locked", path));
      STB_DPSetAudioStatus(path, DECODE_LOCKED);
      STB_AVStopADDecoding(decoder_num);
      STB_AVStopAudioDecoding(decoder_num);
      STB_ERSendEvent(FALSE, FALSE, EV_CLASS_DECODE, EV_TYPE_LOCKED, &decoder_num, sizeof(U8BIT));
   }
   else
   {
      switch (status)
      {
         case DECODE_STOPPING:
         case DECODE_STOPPED:
         case DECODE_LOCKED:
         {
            // request start decode
            STB_DP_PRINT(("STB_DPStartAudioDecoding(%d): start", path));
            STB_DPSetAudioStatus(path, DECODE_STARTING);
            STB_AVStartAudioDecoding(decoder_num);
            break;
         }

         case DECODE_RUNNING:
         {
            STB_DP_PRINT(("STB_DPStartAudioDecoding(%d): re-start not required", path));
            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_DECODE, EV_TYPE_AUDIO_STARTED, &decoder_num, sizeof(U8BIT));
            break;
         }

         default:
         {
            STB_DP_PRINT(("STB_DPStartAudioDecoding(%d): no action", path));
            break;
         }
      }
   }

   FUNCTION_FINISH(STB_DPStartAudioDecoding);
}

/**
 * @brief   Requests start of channel decoding process.
 * @param   U8BIT decoder - the ID of the decode path to use
 */
void STB_DPStartDecoding(U8BIT path)
{
   U8BIT decoder_num;
   BOOLEAN preload = (path & (1<<7)) ? TRUE : FALSE;

   FUNCTION_START(STB_DPStartDecoding);

   if (preload) {
      U8BIT decode_path = path & 0x7F;
      // preloading bit is set for the fast-channel-change
      U8BIT decoder_num = path_status[decode_path].video_decoder_no;
      if (decoder_num < num_video_decoders) {
         STB_AVStartDecoding(decoder_num);
      }
      return;
   }

   ASSERT(path < num_paths);

   STB_DP_PRINT(("%s(%u)", __FUNCTION__, path));

   //FIXME: always start descrambling
   //if (path_status[path].ca_acquired)
   {
      if (FALSE == preload)
         STB_CADescrambleServiceStart(path_status[path].ca_handle);
   }

   STB_DPStartVideoDecoding(path);
   STB_DPStartAudioDecoding(path);

   if ((decoder_num = path_status[path].ad_decoder_no) != INVALID_RES_ID)
   {
      if ((ad_decoder_status[decoder_num].ad_status & AD_ENABLED) != 0)
      {
         STB_DPStartADDecoding(path);
      }
   }

   FUNCTION_FINISH(STB_DPStartDecoding);
}

/**
 * @brief   Requests stop of channel decoding (blanks screen).
 * @param   U8BIT decoder - the ID of the decode path to use
 */
void STB_DPStopVideoDecoding(U8BIT path)
{
   U8BIT demux_num, decoder_num;

   FUNCTION_START(STB_DPStopVideoDecoding);

   ASSERT(path < num_paths);

   decoder_num = path_status[path].video_decoder_no;
   demux_num = path_status[path].demux_no;

   ASSERT(demux_num < num_demuxes);

   switch (STB_DPGetVideoStatus(path))
   {
      case DECODE_RUNNING:
      {
         // request stop decode
         STB_DP_PRINT(("STB_DPStopVideoDecoding(%d): stop (was running)", path));
         STB_DPSetVideoStatus(path, DECODE_STOPPING);
         if ((demux_num != INVALID_RES_ID) && (decoder_num != INVALID_RES_ID))
         {
            STB_AVStopVideoDecoding(decoder_num);
         }
         break;
      }

      case DECODE_STARTING:
      {
         STB_DP_PRINT(("STB_DPStopVideoDecoding(%d): stop (was starting)", path));
         STB_DPSetVideoStatus(path, DECODE_STOPPED);
         if ((demux_num != INVALID_RES_ID) && (decoder_num != INVALID_RES_ID))
         {
            STB_AVStopVideoDecoding(decoder_num);
         }
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_DECODE, EV_TYPE_VIDEO_STOPPED, &decoder_num, sizeof(U8BIT));
         break;
      }

      case DECODE_STOPPED:
      case DECODE_LOCKED:
      {
         STB_DP_PRINT(("STB_DPStopVideoDecoding(%d): already stopped", path));
         STB_DPSetVideoStatus(path, DECODE_STOPPED);
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_DECODE, EV_TYPE_VIDEO_STOPPED, &decoder_num, sizeof(U8BIT));
         break;
      }

      default:
      {
         STB_DP_PRINT(("STB_DPStopVideoDecoding(%d): no action", path));
         break;
      }
   }

   FUNCTION_FINISH(STB_DPStopVideoDecoding);
}

/**
 * @brief   Requests stop of channel decoding.
 * @param   U8BIT decoder - the ID of the decode path to use
 */
void STB_DPStopADDecoding(U8BIT path)
{
   U8BIT demux_num, decoder_num;
   FUNCTION_START(STB_DPStopADDecoding);

   STB_DP_PRINT(("STB_DPStopADDecoding"));
   ASSERT(path < num_paths);

   decoder_num = path_status[path].ad_decoder_no;
   demux_num = path_status[path].demux_no;

   ASSERT(decoder_num < num_audio_decoders);
   ASSERT(demux_num < num_demuxes);

   switch (STB_DPGetADStatus(path))
   {
      case DECODE_RUNNING:
      {
         // request stop decode
         STB_DP_PRINT(("STB_DPStopADDecoding(%d): stop (was running)", path));
         STB_DPSetADStatus(path, DECODE_STOPPING);
         if ((demux_num != INVALID_RES_ID) && (decoder_num != INVALID_RES_ID))
         {
            STB_AVStopADDecoding(decoder_num);
            STB_DPSetADAudio(path, AD_AUDIO_OFF);
         }
         break;
      }

      case DECODE_STARTING:
      {
         STB_DP_PRINT(("STB_DPStopADDecoding(%d): stop (was starting)", path));
         STB_DPSetADStatus(path, DECODE_STOPPED);
         if ((demux_num != INVALID_RES_ID) && (decoder_num != INVALID_RES_ID))
         {
            STB_AVStopADDecoding(decoder_num);
            STB_DPSetADAudio(path, AD_AUDIO_OFF);
         }
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_DECODE, EV_TYPE_AD_STOPPED, &decoder_num, sizeof(U8BIT));
         break;
      }

      case DECODE_STOPPED:
      case DECODE_LOCKED:
      {
         STB_DP_PRINT(("STB_DPStopADDecoding(%d): already stopped", path));
         STB_DPSetADStatus(path, DECODE_STOPPED);
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_DECODE, EV_TYPE_AD_STOPPED, &decoder_num, sizeof(U8BIT));
         break;
      }

      default:
      {
         STB_DP_PRINT(("STB_DPStopADDecoding(%d): no action", path));
         break;
      }
   }

   FUNCTION_FINISH(STB_DPStopADDecoding);
}

/**
 * @brief   Requests stop of channel decoding.
 * @param   U8BIT decoder - the ID of the decode path to use
 */
void STB_DPStopAudioDecoding(U8BIT path)
{
   U8BIT demux_num, decoder_num;

   FUNCTION_START(STB_DPStopAudioDecoding);

   ASSERT(path < num_paths);

   decoder_num = path_status[path].audio_decoder_no;
   demux_num = path_status[path].demux_no;

   ASSERT(decoder_num < num_audio_decoders);
   ASSERT(demux_num < num_demuxes);

   switch (STB_DPGetAudioStatus(path))
   {
      case DECODE_RUNNING:
      {
         // request stop decode
         STB_DP_PRINT(("STB_DPStopAudioDecoding(%d): stop (was running)", path));
         STB_DPSetAudioStatus(path, DECODE_STOPPING);
         if ((demux_num != INVALID_RES_ID) && (decoder_num != INVALID_RES_ID))
         {
            STB_AVStopADDecoding(decoder_num);
            STB_AVStopAudioDecoding(decoder_num);
         }
         break;
      }

      case DECODE_STARTING:
      {
         STB_DP_PRINT(("STB_DPStopAudioDecoding(%d): stop (was starting)", path));
         STB_DPSetAudioStatus(path, DECODE_STOPPED);
         if ((demux_num != INVALID_RES_ID) && (decoder_num != INVALID_RES_ID))
         {
            STB_AVStopAudioDecoding(decoder_num);
         }
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_DECODE, EV_TYPE_AUDIO_STOPPED, &decoder_num, sizeof(U8BIT));
         break;
      }

      case DECODE_STOPPED:
      case DECODE_LOCKED:
      {
         STB_DP_PRINT(("STB_DPStopAudioDecoding(%d): already stopped", path));
         STB_DPSetAudioStatus(path, DECODE_STOPPED);
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_DECODE, EV_TYPE_AUDIO_STOPPED, &decoder_num, sizeof(U8BIT));
         break;
      }

      default:
      {
         STB_DP_PRINT(("STB_DPStopAudioDecoding(%d): no action", path));
         break;
      }
   }

   FUNCTION_FINISH(STB_DPStopAudioDecoding);
}

/**
 * @brief   Requests stop of channel decoding (blanks screen).
 * @param   U8BIT decoder - the ID of the decode path to use
 */
void STB_DPStopDecoding(U8BIT path)
{
   FUNCTION_START(STB_DPStopDecoding);

   ASSERT(path < num_paths);

   STB_DP_PRINT(("%s(%u)", __FUNCTION__, path));

   STB_DPStopADDecoding(path);
   STB_DPStopAudioDecoding(path);
   STB_DPStopVideoDecoding(path);

   if (path_status[path].ca_acquired)
   {
      STB_CADescrambleServiceStop(path_status[path].ca_handle);
   }

   FUNCTION_FINISH(STB_DPStopDecoding);
}

/**
 * @brief   Requests start of SI engine, mode is determined by flags in general control.
 * @param   U8BIT path - the ID of the decode path to use
 */
void STB_DPStartSI(U8BIT path)
{
   U8BIT tuner_num;
   E_STB_SI_STATUS status;

   FUNCTION_START(STB_DPStartSI);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   if (tuner_num != INVALID_RES_ID)
   {
#ifdef STB_DEBUG
      if (tuner_status[tuner_num].signal_type == SIGNAL_COFDM)
      {
         STB_DP_PRINT(("Signal Type = COFDM..."));
      }
      else if (tuner_status[tuner_num].signal_type == SIGNAL_QPSK)
      {
         STB_DP_PRINT(("Signal Type = QPSK..."));
      }
      else if (tuner_status[tuner_num].signal_type == SIGNAL_QAM)
      {
         STB_DP_PRINT(("Signal Type = QAM..."));
      }
      else
      {
         STB_DP_PRINT(("Signal type = UNRECOGNISED!"));
      }
#endif

      status = STB_SITerrGetStatus(path);

      if ((tuner_status[tuner_num].actions & ACTION_SI_NEW) != 0)
      {
         // change of transport - so force SI re-start
         tuner_status[tuner_num].actions &= ~ACTION_SI_NEW;
         status = SI_STOPPED;
      }

      // start SI task
      if (path_status[path].search_mode == TRUE)
      {
         switch (status)
         {
            case SI_STOPPED:
            {
               STB_DP_PRINT(("STB_DPStartSI(%d): tuner %d - search invoked", path, tuner_num));
               STB_SITerrSendEvent(path, SI_EVENT_SEARCH);
               break;
            }

            default:
            {
               STB_DP_PRINT(("STB_DPStartSI(%d): tuner %d - no action", path, tuner_num));
               break;
            }
         }
      }
      else
      {
         switch (status)
         {
            case SI_UPDATING:
            {
               STB_DP_PRINT(("STB_DPStartSI(%d): tuner %d - service change", path, tuner_num));
               STB_SITerrSendEvent(path, SI_EVENT_SERVICE_CHANGE);
               break;
            }

            case SI_STOPPED:
            {
               STB_DP_PRINT(("STB_DPStartSI(%d): tuner %d - start update", path, tuner_num));
               STB_SITerrSendEvent(path, SI_EVENT_UPDATE);
               break;
            }

            default:
            {
               STB_DP_PRINT(("STB_DPStartSI(%d): tuner %d - no action", path, tuner_num));
               break;
            }
         }
      }
   }
   else
   {
      /* Assume we want to get SI data from the playback path (no tuner associated) */
      status = STB_SITerrGetStatus(path);
      if (path_status[path].search_mode == TRUE)
      {
         switch (status)
         {
            case SI_STOPPED:
            {
               STB_DP_PRINT(("STB_DPStartSI(%d): playback - search", path));
               STB_SITerrSendEvent(path, SI_EVENT_SEARCH);
               break;
            }

            default:
            {
               STB_DP_PRINT(("STB_DPStartSI(%d): playback - no action", path));
               break;
            }
         }
      }
      else
      {
         switch (status)
         {
            case SI_UPDATING:
            {
               STB_DP_PRINT(("STB_DPStartSI(%d): playback - service change", path));
               STB_SITerrSendEvent(path, SI_EVENT_SERVICE_CHANGE);
               break;
            }

            case SI_STOPPED:
            {
               STB_DP_PRINT(("STB_DPStartSI(%d): playback - start update", path));
               STB_SITerrSendEvent(path, SI_EVENT_UPDATE);
               break;
            }

            default:
            {
               STB_DP_PRINT(("STB_DPStartSI(%d): playback - no action", path));
               break;
            }
         }
      }
   }

   FUNCTION_FINISH(STB_DPStartSI);
}

/**
 * @brief   Requests stop of SI engine.
 * @param   U8BIT path - the ID of the decode path to use
 */
void STB_DPStopSI(U8BIT path)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPStopSI);

   ASSERT(path < num_paths);

   tuner_num = STB_DPGetPathTuner(path);
   STB_DP_PRINT(("STB_DPStopSI(path=%d), tuner_num=%d", path, tuner_num));

   if (tuner_num != INVALID_RES_ID)
   {
#ifdef STB_DEBUG
      if (tuner_status[tuner_num].signal_type == SIGNAL_COFDM)
      {
         STB_DP_PRINT(("Signal Type = COFDM..."));
      }
      else if (tuner_status[tuner_num].signal_type == SIGNAL_QPSK)
      {
         STB_DP_PRINT(("Signal Type = QPSK..."));
      }
      else if (tuner_status[tuner_num].signal_type == SIGNAL_QAM)
      {
         STB_DP_PRINT(("Signal Type = QAM..."));
      }
      else
      {
         STB_DP_PRINT(("Signal Type = UNRECOGNISED!"));
      }
#endif

      // stop SI task
      switch (STB_SITerrGetStatus(path))
      {
         case SI_SEARCHING:
         case SI_UPDATING:
         {
            STB_DP_PRINT(("STB_DPStopSI(%d): tuner %d - stop invoked", path, tuner_num));
            STB_SITerrSendEvent(path, SI_EVENT_STOP);
            // wait for SI task to have stopped
            while (STB_SITerrGetStatus(path) != SI_STOPPED)
            {
               STB_OSTaskDelay(10);
            }
            STB_DP_PRINT(("STB_DPStopSI(%d): tuner %d - stopped", path, tuner_num));
            break;
         }

         default:
         {
            STB_DP_PRINT(("STB_DPStopSI(%d)       : tuner %d - no action", path, tuner_num));
            break;
         }
      }
   }
   else
   {
      /* Assume we want to get SI data from the playback path (no tuner associated) */
      switch (STB_SITerrGetStatus(path))
      {
         case SI_SEARCHING:
         case SI_UPDATING:
         {
            STB_DP_PRINT(("STB_DPStopSI(%d): terrestrial - stop", path));
            STB_SITerrSendEvent(path, SI_EVENT_STOP);
            // wait for SI task to have stopped
            while (STB_SITerrGetStatus(path) != SI_STOPPED)
            {
               STB_OSTaskDelay(10);
            }
            STB_DP_PRINT(("STB_DPStopSI(%d): Stopped", path));
            break;
         }

         default:
         {
            STB_DP_PRINT(("STB_DPStopSI(%d): status = 0x%x terrestrial - no action", path, STB_SITerrGetStatus(path)));
            break;
         }
      }
   }

   FUNCTION_FINISH(STB_DPStopSI);
}

/**
 * @brief   Requests SI engine to get extended event data for specified event.
 * @param   U8BIT path - the ID of the decode path to use
 * @param   U32BIT start_date - starting date
 * @param   U32BIT start_hour - and time
 * @param   U32BIT start_min - for requested event
 */
void STB_DPRequestSIExtendedEvent(U8BIT path, U32BIT start_date, U32BIT start_hour, U32BIT start_min)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPRequestSIExtendedEvent);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   tuner_status[tuner_num].si_rparam1 = start_date;
   tuner_status[tuner_num].si_rparam2 = start_hour;
   tuner_status[tuner_num].si_rparam3 = start_min;

#ifdef STB_DEBUG
   if (tuner_status[tuner_num].signal_type == SIGNAL_COFDM)
   {
      STB_DP_PRINT(("Signal Type = COFDM..."));
   }
   else if (tuner_status[tuner_num].signal_type == SIGNAL_QPSK)
   {
      STB_DP_PRINT(("Signal Type = QPSK..."));
   }
   else if (tuner_status[tuner_num].signal_type == SIGNAL_QAM)
   {
      STB_DP_PRINT(("Signal Type = QAM..."));
   }
   else
   {
      STB_DP_PRINT(("Signal Type = UNRECOGNISED!"));
   }
#endif

   // send request to terrestrial SI task
   switch (STB_SITerrGetStatus(path))
   {
      case SI_UPDATING:
      {
         STB_DP_PRINT(("STB_DPRequestSIExtendedEvent(%d): tuner %d - request", path, tuner_num));
         STB_SITerrSendEvent(path, SI_EVENT_EXTENDED_EVENT);
         break;
      }

      default:
      {
         STB_DP_PRINT(("STB_DPRequestSIExtendedEvent(%d): tuner %d - no action", path, tuner_num));
         break;
      }
   }

   FUNCTION_FINISH(STB_DPRequestSIExtendedEvent);
}

/**
 * @brief   Returns parameters for previous SI request.
 * @param   U8BIT path - the ID of the decode path to use
 * @return   U32BIT - first parameter
 */
U32BIT STB_DPGetSIRequestParam1(U8BIT path)
{
   U8BIT tuner_num;
   U32BIT ret_val;

   FUNCTION_START(STB_DPGetSIRequestParam1);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].si_rparam1;

   FUNCTION_FINISH(STB_DPGetSIRequestParam1);

   return(ret_val);
}

/**
 * @brief   Returns parameters for previous SI request.
 * @param   U8BIT path - the ID of the decode path to use
 * @return   U32BIT - second parameter
 */
U32BIT STB_DPGetSIRequestParam2(U8BIT path)
{
   U8BIT tuner_num;
   U32BIT ret_val;

   FUNCTION_START(STB_DPGetSIRequestParam2);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].si_rparam2;

   FUNCTION_FINISH(STB_DPGetSIRequestParam2);

   return(ret_val);
}

/**
 * @brief   Returns parameters for previous SI request.
 * @param   U8BIT path - the ID of the decode path to use
 * @return   U32BIT - third parameter
 */
U32BIT STB_DPGetSIRequestParam3(U8BIT path)
{
   U8BIT tuner_num;
   U32BIT ret_val;

   FUNCTION_START(STB_DPGetSIRequestParam3);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].si_rparam3;

   FUNCTION_FINISH(STB_DPGetSIRequestParam3);

   return(ret_val);
}

/**
 * @brief   Writes the tuning status into decode path store.
 * @param   U8BIT path - the ID of the decode path to use
 * @param   E_STB_DP_TUNE_STATUS state - new status
 */
void STB_DPSetTuneStatus(U8BIT path, E_STB_DP_TUNE_STATUS state)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetTuneStatus);

   ASSERT(path < num_paths);

   STB_OSSemaphoreWait(dp_protect);

   tuner_num = path_status[path].tuner_no;

   STB_TUNE_PRINT(("STB_DPSetTuneStatus(%u, %u): tuner=%u", path, state, tuner_num));

   ASSERT(tuner_num < num_tuners);

   tuner_status[tuner_num].tune_stat = state;

   STB_OSSemaphoreSignal(dp_protect);

   FUNCTION_FINISH(STB_DPSetTuneStatus);
}

/**
 * @brief   Reads the tuning status from decode path store.
 * @param   U8BIT path - the ID of the decode path to use
 * @return   E_STB_DP_TUNE_STATUS - tune status value.
 */
E_STB_DP_TUNE_STATUS STB_DPGetTuneStatus(U8BIT path)
{
   U8BIT tuner_num;
   E_STB_DP_TUNE_STATUS ret_val;

   FUNCTION_START(STB_DPGetTuneStatus);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_OSSemaphoreWait(dp_protect);

   ret_val = tuner_status[tuner_num].tune_stat;

   //STB_TUNE_PRINT(("STB_DPGetTuneStatus(%u): tuner=%u, state=%u", path, tuner_num, ret_val));

   STB_OSSemaphoreSignal(dp_protect);

   FUNCTION_FINISH(STB_DPGetTuneStatus);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes the AD Audio into decode path store. To be set to ON or OFF. The exact ON state is depends if AD
//                      PID is present or not.
 *
 * @param   U8BIT decoder - the ID of the decode path to use
 * @param   BOOLEAN - Note this should only be True for ON, False for OFF
 *

 *
 */
void STB_DPSetADEnabled(U8BIT path, BOOLEAN state)
{
   U8BIT decoder_num;

   FUNCTION_START(STB_DPSetADEnabled);

   STB_DP_PRINT(("STB_DPSetADEnabled"));
   ASSERT(path < num_paths);

   decoder_num = path_status[path].ad_decoder_no;

   ASSERT(decoder_num < num_audio_decoders);

   STB_DP_PRINT(("STB_DPSetADEnabled(%d): decoder=%d, audio state=%d", path, decoder_num, state));

   STB_OSSemaphoreWait(dp_protect);

   if (state == FALSE)
   {
      ad_decoder_status[decoder_num].ad_status &= ~AD_ENABLED;
   }
   else
   {
      ad_decoder_status[decoder_num].ad_status |= AD_ENABLED;
   }

   STB_OSSemaphoreSignal(dp_protect);

   FUNCTION_FINISH(STB_DPSetADEnabled);
}

/**
 *

 *
 * @brief   Writes the AD Audio into decode path store. To be set to ON or OFF. The exact ON state is depends if AD
 *                      PID is present or not.
 *
 * @param   U8BIT decoder - the ID of the decode path to use
 * @param   BOOLEAN - Note this should only be True for ON, False for OFF
 *

 *
 */
static U8BIT GetADEnabled(U8BIT path)
{
   U8BIT decoder_num;
   U8BIT ad_status;

   FUNCTION_START(GetADEnabled);

   ASSERT(path < num_paths);

   decoder_num = path_status[path].ad_decoder_no;

   ASSERT(decoder_num < num_audio_decoders);

   STB_OSSemaphoreWait(dp_protect);

   ad_status = ad_decoder_status[decoder_num].ad_status;

   STB_OSSemaphoreSignal(dp_protect);

   STB_DP_PRINT(("GetADEnabled(%d): decoder=%d, ad status=%d", path, decoder_num, ad_status));

   FUNCTION_FINISH(GetADEnabled);

   return(ad_status);
}

/**
 *

 *
 * @brief   Reads the AD audio status from decode path store.
 *
 * @param   U8BIT decoder - the ID of the decode path to use
 *
 * @return   E_STB_DP_AD_AUDIO - audio status value.
 *
 */
void STB_DPSetADAudio(U8BIT path, E_STB_DP_AD_AUDIO ad_audio)
{
   U8BIT decoder_num;

   FUNCTION_START(STB_DPSetADAudio);

   ASSERT(path < num_paths);

   decoder_num = path_status[path].ad_decoder_no;

   ASSERT(decoder_num < num_audio_decoders);

   STB_OSSemaphoreWait(dp_protect);

   ad_decoder_status[decoder_num].ad_audio = ad_audio;

   STB_OSSemaphoreSignal(dp_protect);

   STB_DP_PRINT(("STB_DPSetADAudio = %d\n", ad_audio));
   FUNCTION_FINISH(STB_DPSetADAudio);

   return;
}

/**
 *

 *
 * @brief   Reads the AD audio status from decode path store.
 *
 * @param   U8BIT decoder - the ID of the decode path to use
 *
 * @return   E_STB_DP_AD_AUDIO - audio status value.
 *
 */
E_STB_DP_AD_AUDIO STB_DPGetADAudio(U8BIT path)
{
   U8BIT decoder_num;
   E_STB_DP_AD_AUDIO ret_val;

   FUNCTION_START(STB_DPGetADAudio);

   ASSERT(path < num_paths);

   decoder_num = path_status[path].ad_decoder_no;

   ASSERT(decoder_num < num_audio_decoders);

   STB_OSSemaphoreWait(dp_protect);

   ret_val = ad_decoder_status[decoder_num].ad_audio;

   STB_OSSemaphoreSignal(dp_protect);

   STB_DP_PRINT(("STB_DPGetADAudio = %d\n", ret_val));
   FUNCTION_FINISH(STB_DPGetADAudio);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes the AD status into decode path store.
 *
 * @param   U8BIT decoder - the ID of the decode path to use
 * @param   E_STB_DP_DECODE_STATUS state - new status
 *

 *
 */
void STB_DPSetADStatus(U8BIT path, E_STB_DP_DECODE_STATUS state)
{
   U8BIT decoder_num;

   FUNCTION_START(STB_DPSetADStatus);

   ASSERT(path < num_paths);

   decoder_num = path_status[path].ad_decoder_no;

   ASSERT(decoder_num < num_audio_decoders);

   STB_DP_PRINT(("STB_DPSetADStatus(%d): decoder=%d, state=%d", path, decoder_num, state));

   STB_OSSemaphoreWait(dp_protect);

   if ((ad_decoder_status[decoder_num].audio_stat != state) &&
       path_status[path].ca_acquired)
   {
      /* Inform the CA system of the change in the AD decoding state */
      switch (state)
      {
         case DECODE_STARTING:
            STB_CADecodeADStatus(path_status[path].ca_handle, CA_DECODE_STATUS_STARTING);
            break;

         case DECODE_RUNNING:
            STB_CADecodeADStatus(path_status[path].ca_handle, CA_DECODE_STATUS_STARTED);
            break;

         case DECODE_STOPPED:
            STB_CADecodeADStatus(path_status[path].ca_handle, CA_DECODE_STATUS_STOPPED);
            break;

         default:
            break;
      }
   }

   ad_decoder_status[decoder_num].audio_stat = state;

   STB_OSSemaphoreSignal(dp_protect);

   FUNCTION_FINISH(STB_DPSetADStatus);
}

/**
 *

 *
 * @brief   Reads the AD status from decode path store.
 *
 * @param   U8BIT decoder - the ID of the decode path to use
 *
 * @return   E_STB_DP_DECODE_STATUS - audio status value.
 *
 */
E_STB_DP_DECODE_STATUS STB_DPGetADStatus(U8BIT path)
{
   U8BIT decoder_num;
   E_STB_DP_DECODE_STATUS ret_val;

   FUNCTION_START(STB_DPGetADStatus);

   ASSERT(path < num_paths);

   decoder_num = path_status[path].ad_decoder_no;

   ASSERT(decoder_num < num_audio_decoders);

   STB_OSSemaphoreWait(dp_protect);

   ret_val = ad_decoder_status[decoder_num].audio_stat;

   STB_OSSemaphoreSignal(dp_protect);

   FUNCTION_FINISH(STB_DPGetADStatus);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes the audio status into decode path store.
 *
 * @param   U8BIT decoder - the ID of the decode path to use
 * @param   E_STB_DP_DECODE_STATUS state - new status
 *

 *
 */
void STB_DPSetAudioStatus(U8BIT path, E_STB_DP_DECODE_STATUS state)
{
   U8BIT decoder_num;

   FUNCTION_START(STB_DPSetAudioStatus);

   ASSERT(path < num_paths);

   decoder_num = path_status[path].audio_decoder_no;

   ASSERT(decoder_num < num_audio_decoders);

   STB_DP_PRINT(("STB_DPSetAudioStatus(%d): decoder=%d, state=%d", path, decoder_num, state));

   STB_OSSemaphoreWait(dp_protect);

   if ((audio_decoder_status[decoder_num].audio_stat != state) &&
       path_status[path].ca_acquired)
   {
      /* Inform the CA system of the change in the audio decoding state */
      switch (state)
      {
         case DECODE_STARTING:
            STB_CADecodeAudioStatus(path_status[path].ca_handle, CA_DECODE_STATUS_STARTING);
            break;

         case DECODE_RUNNING:
            STB_CADecodeAudioStatus(path_status[path].ca_handle, CA_DECODE_STATUS_STARTED);
            break;

         case DECODE_STOPPED:
            STB_CADecodeAudioStatus(path_status[path].ca_handle, CA_DECODE_STATUS_STOPPED);
            break;

         default:
            break;
      }
   }

   audio_decoder_status[decoder_num].audio_stat = state;

   STB_OSSemaphoreSignal(dp_protect);

   FUNCTION_FINISH(STB_DPSetAudioStatus);
}

/**
 *

 *
 * @brief   Reads the audio status from decode path store.
 *
 * @param   U8BIT decoder - the ID of the decode path to use
 *
 * @return   E_STB_DP_DECODE_STATUS - audio status value.
 *
 */
E_STB_DP_DECODE_STATUS STB_DPGetAudioStatus(U8BIT path)
{
   U8BIT decoder_num;
   E_STB_DP_DECODE_STATUS ret_val;

   FUNCTION_START(STB_DPGetAudioStatus);

   ASSERT(path < num_paths);

   decoder_num = path_status[path].audio_decoder_no;

   ASSERT(decoder_num < num_audio_decoders);

   STB_OSSemaphoreWait(dp_protect);

   ret_val = audio_decoder_status[decoder_num].audio_stat;

   STB_OSSemaphoreSignal(dp_protect);

   FUNCTION_FINISH(STB_DPGetAudioStatus);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes the video status into decode path store.
 *
 * @param   U8BIT decoder - the ID of the decode path to use
 * @param   E_STB_DP_DECODE_STATUS state - new status
 *

 *
 */
void STB_DPSetVideoStatus(U8BIT path, E_STB_DP_DECODE_STATUS state)
{
   U8BIT decoder_num;

   FUNCTION_START(STB_DPSetVideoStatus);

   ASSERT(path < num_paths);

   decoder_num = path_status[path].video_decoder_no;

   ASSERT(decoder_num < num_video_decoders);

   STB_DP_PRINT(("STB_DPSetVideoStatus(%d): decoder=%d, state=%d", path, decoder_num, state));

   STB_OSSemaphoreWait(dp_protect);

   if ((video_decoder_status[decoder_num].video_stat != state) &&
       path_status[path].ca_acquired)
   {
      /* Inform the CA system of the change in the video decoding state */
      switch (state)
      {
         case DECODE_STARTING:
            STB_CADecodeVideoStatus(path_status[path].ca_handle, CA_DECODE_STATUS_STARTING);
            break;

         case DECODE_RUNNING:
            STB_CADecodeVideoStatus(path_status[path].ca_handle, CA_DECODE_STATUS_STARTED);
            break;

         case DECODE_STOPPED:
            STB_CADecodeVideoStatus(path_status[path].ca_handle, CA_DECODE_STATUS_STOPPED);
            break;

         default:
            break;
      }
   }

   video_decoder_status[decoder_num].video_stat = state;

   STB_OSSemaphoreSignal(dp_protect);

   FUNCTION_FINISH(STB_DPSetVideoStatus);
}

/**
 *

 *
 * @brief   Reads the video status from decode path store.
 *
 * @param   U8BIT decoder - the ID of the decode path to use
 *
 * @return   E_STB_DP_DECODE_STATUS - video status value.
 *
 */
E_STB_DP_DECODE_STATUS STB_DPGetVideoStatus(U8BIT path)
{
   U8BIT decoder_num;
   E_STB_DP_DECODE_STATUS ret_val;

   FUNCTION_START(STB_DPGetVideoStatus);

   ASSERT(path < num_paths);

   decoder_num = path_status[path].video_decoder_no;

   ASSERT(decoder_num < num_video_decoders);

   STB_OSSemaphoreWait(dp_protect);

   ret_val = video_decoder_status[decoder_num].video_stat;

   STB_OSSemaphoreSignal(dp_protect);

   FUNCTION_FINISH(STB_DPGetVideoStatus);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes signal type value into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   E_STB_DP_SIGNAL_TYPE sigtype - the signal type to use
 *

 *
 */
void STB_DPSetSignalType(U8BIT path, E_STB_DP_SIGNAL_TYPE sigtype)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetSignalType);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetSignalType(%d): sigtype %d", path, sigtype));
   if (sigtype != tuner_status[tuner_num].signal_type)
   {
      tuner_status[tuner_num].signal_type = sigtype;
      tuner_status[tuner_num].actions |= ACTION_TUNE;
   }

   FUNCTION_FINISH(STB_DPSetSignalType);
}

/**
 *

 *
 * @brief   Reads the signal type value from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   E_STB_DP_SIGNAL_TYPE - signal type.
 *
 */
E_STB_DP_SIGNAL_TYPE STB_DPGetSignalType(U8BIT path)
{
   U8BIT tuner_num;
   E_STB_DP_SIGNAL_TYPE ret_val;

   FUNCTION_START(STB_DPGetSignalType);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].signal_type;

   FUNCTION_FINISH(STB_DPGetSignalType);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes tuner auto relock flag into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   BOOLEAN state - TRUE to set, else FALSE.
 *

 *
 */
void STB_DPSetTuneRelock(U8BIT path, BOOLEAN state)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetTuneRelock);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetTuneRelock(%d): tuner %d state %d", path, tuner_num, state));
   tuner_status[tuner_num].auto_relock = state;

   STB_TuneAutoRelock(tuner_num, state);

   FUNCTION_FINISH(STB_DPSetTuneRelock);
}

/**
 *

 *
 * @brief   Reads the tuner auto relock flag state from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   BOOLEAN - TRUE if set, else FALSE.
 *
 */
BOOLEAN STB_DPGetTuneRelock(U8BIT path)
{
   U8BIT tuner_num;
   BOOLEAN ret_val;

   FUNCTION_START(STB_DPGetTuneRelock);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].auto_relock;

   FUNCTION_FINISH(STB_DPGetTuneRelock);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes lock enable flag into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   BOOLEAN state - TRUE to set, else FALSE.
 *

 *
 */
void STB_DPSetLockEnable(U8BIT path, BOOLEAN state)
{
   FUNCTION_START(STB_DPSetLockEnable);

   ASSERT(path < num_paths);

   STB_DP_PRINT(("STB_DPSetLockEnable(%d): state %d", path, state));
   path_status[path].lock_enable = state;

   FUNCTION_FINISH(STB_DPSetLockEnable);
}

/**
 *

 *
 * @brief   Reads the lock enable flag state from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   BOOLEAN - TRUE if set, else FALSE.
 *
 */
BOOLEAN STB_DPGetLockEnable(U8BIT path)
{
   BOOLEAN ret_val;

   FUNCTION_START(STB_DPGetLockEnable);

   ASSERT(path < num_paths);

   ret_val = path_status[path].lock_enable;

   FUNCTION_FINISH(STB_DPGetLockEnable);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes SI search mode flag into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   BOOLEAN state - TRUE to set, else FALSE.
 *

 *
 */
void STB_DPSetSearchMode(U8BIT path, BOOLEAN state)
{
   FUNCTION_START(STB_DPSetSearchMode);

   ASSERT(path < num_paths);

   STB_DP_PRINT(("STB_DPSetSearchMode(%d): state %d", path, state));
   path_status[path].search_mode = state;

   FUNCTION_FINISH(STB_DPSetSearchMode);
}

/**
 *

 *
 * @brief   Reads the SI search mode flag state from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   BOOLEAN - TRUE if set, else FALSE.
 *
 */
BOOLEAN STB_DPGetSearchMode(U8BIT path)
{
   BOOLEAN ret_val;

   FUNCTION_START(STB_DPGetSearchMode);

   ASSERT(path < num_paths);

   ret_val = path_status[path].search_mode;

   FUNCTION_FINISH(STB_DPGetSearchMode);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes SI TV chan search flag into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   BOOLEAN state - TRUE to set, else FALSE.
 *

 *
 */
void STB_DPSetTVSearch(U8BIT path, BOOLEAN state)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetTVSearch);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_DP_PRINT(("STB_DPSetTVSearch(%d): tuner %d state %d", path, tuner_num, state));
   tuner_status[tuner_num].tv_search = state;

   FUNCTION_FINISH(STB_DPSetTVSearch);
}

/**
 *

 *
 * @brief   Reads the SI TV chan search flag state from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   BOOLEAN - TRUE if set, else FALSE.
 *
 */
BOOLEAN STB_DPGetTVSearch(U8BIT path)
{
   U8BIT tuner_num;
   BOOLEAN ret_val;

   FUNCTION_START(STB_DPGetTVSearch);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].tv_search;

   FUNCTION_FINISH(STB_DPGetTVSearch);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes SI radio chan search flag into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   BOOLEAN state - TRUE to set, else FALSE.
 *

 *
 */
void STB_DPSetRadioSearch(U8BIT path, BOOLEAN state)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetRadioSearch);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_DP_PRINT(("STB_DPSetRadioSearch(%d): tuner %d state %d", path, tuner_num, state));
   tuner_status[tuner_num].rad_search = state;

   FUNCTION_FINISH(STB_DPSetRadioSearch);
}

/**
 *

 *
 * @brief   Reads the SI radio chan search flag state from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   BOOLEAN - TRUE if set, else FALSE.
 *
 */
BOOLEAN STB_DPGetRadioSearch(U8BIT path)
{
   U8BIT tuner_num;
   BOOLEAN ret_val;

   FUNCTION_START(STB_DPGetRadioSearch);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].rad_search;

   FUNCTION_FINISH(STB_DPGetRadioSearch);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes SI FTA chan search flag into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   BOOLEAN state - TRUE to set, else FALSE.
 *

 *
 */
void STB_DPSetFTASearch(U8BIT path, BOOLEAN state)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetFTASearch);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_DP_PRINT(("STB_DPSetFTASearch(%d): tuner %d state %d", path, tuner_num, state));
   tuner_status[tuner_num].fta_search = state;

   FUNCTION_FINISH(STB_DPSetFTASearch);
}

/**
 *

 *
 * @brief   Reads the SI FTA chan search flag state from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   BOOLEAN - TRUE if set, else FALSE.
 *
 */
BOOLEAN STB_DPGetFTASearch(U8BIT path)
{
   U8BIT tuner_num;
   BOOLEAN ret_val;

   FUNCTION_START(STB_DPGetFTASearch);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].fta_search;

   FUNCTION_FINISH(STB_DPGetFTASearch);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes SI scram chan search flag into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   BOOLEAN state - TRUE to set, else FALSE.
 *

 *
 */
void STB_DPSetScramSearch(U8BIT path, BOOLEAN state)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetScramSearch);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_DP_PRINT(("STB_DPSetScramSearch(%d): tuner %d state %d", path, tuner_num, state));
   tuner_status[tuner_num].scram_search = state;

   FUNCTION_FINISH(STB_DPSetScramSearch);
}

/**
 *

 *
 * @brief   Reads the SI scram chan search flag state from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   BOOLEAN - TRUE if set, else FALSE.
 *
 */
BOOLEAN STB_DPGetScramSearch(U8BIT path)
{
   U8BIT tuner_num;
   BOOLEAN ret_val;

   FUNCTION_START(STB_DPGetScramSearch);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].scram_search;

   FUNCTION_FINISH(STB_DPGetScramSearch);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes SI network search flag into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   BOOLEAN state - TRUE to set, else FALSE.
 *

 *
 */
void STB_DPSetNetworkSearch(U8BIT path, BOOLEAN state)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetNetworkSearch);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_DP_PRINT(("STB_DPSetNetworkSearch(%d): tuner %d state %d", path, tuner_num, state));
   tuner_status[tuner_num].net_search = state;

   FUNCTION_FINISH(STB_DPSetNetworkSearch);
}

/**
 *

 *
 * @brief   Reads the SI network search flag state from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   BOOLEAN - TRUE if set, else FALSE.
 *
 */
BOOLEAN STB_DPGetNetworkSearch(U8BIT path)
{
   U8BIT tuner_num;
   BOOLEAN ret_val;

   FUNCTION_START(STB_DPGetNetworkSearch);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].net_search;

   FUNCTION_FINISH(STB_DPGetNetworkSearch);

   return(ret_val);
}

/**
 *

 *
 * @brief   Sets the 'Over The Air' Software Upgrade Search Mode for a given path.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   E_STB_OTA_SW_UPGRADE_SEARCH_MODE = [OTA_SEARCH_OFF |
 *                                                     OTA_SEARCH_AUTO|
 *                                                     OTA_SEARCH_MANUAL]
 *

 *
 */
void STB_DPSetOTASearchMode(U8BIT path, E_STB_OTA_SW_UPGRADE_SEARCH_MODE mode)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetOTASearchMode);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_DP_PRINT(("STB_DPSetOTASearchMode(%d): tuner %d mode %d", path, tuner_num, mode));
   tuner_status[tuner_num].ota_search_status = mode;

   FUNCTION_FINISH(STB_DPSetOTASearchMode);
}

/**
 *

 *
 * @brief   Returns the current 'Over The Air' Software Upgrade Search Mode for the given path.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   E_STB_OTA_SW_UPGRADE_SEARCH_MODE = [OTA_SEARCH_OFF |
 *                                                     OTA_SEARCH_AUTO|
 *                                                     OTA_SEARCH_MANUAL]
 *
 */
E_STB_OTA_SW_UPGRADE_SEARCH_MODE STB_DPGetOTASearchMode(U8BIT path)
{
   U8BIT tuner_num;
   E_STB_OTA_SW_UPGRADE_SEARCH_MODE ret_val;

   FUNCTION_START(STB_DPGetOTASearchMode);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].ota_search_status;

   FUNCTION_FINISH(STB_DPGetOTASearchMode);

   return(ret_val);
}

/**
 *

 *
 * @brief   Returns TRUE if the current 'Over The Air' Software Upgrade Search Mode is manual
 *                 or automatic, FALSE otherwise.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   TRUE or FALSE
 *
 */
BOOLEAN STB_DPOTASearchEnabled(U8BIT path)
{
   U8BIT tuner_num;
   BOOLEAN ret_val;

   FUNCTION_START(STB_DPOTASearchEnabled);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = (tuner_status[tuner_num].ota_search_status != OTA_SEARCH_OFF);

   FUNCTION_FINISH(STB_DPOTASearchEnabled);

   return(ret_val);
}

/**
 *

 *
 * @brief   Sets LNB power on or off and writes status into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   E_STB_DP_LNB_POWER state - LNB power mode
 *

 *
 */
void STB_DPSetLNBPower(U8BIT path, E_STB_DP_LNB_POWER state)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetLNBPower);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetLNBPower(%d)  : tuner %d state %d", path, tuner_num, state));
   if (state != tuner_status[tuner_num].lnb_power)
   {
      tuner_status[tuner_num].lnb_power = state;
      tuner_status[tuner_num].actions |= ACTION_TUNE;
   }

   FUNCTION_FINISH(STB_DPSetLNBPower);
}

/**
 *

 *
 * @brief   Reads the LNB power status from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   E_STB_DP_LNB_POWER - LNB power mode.
 *
 */
E_STB_DP_LNB_POWER STB_DPGetLNBPower(U8BIT path)
{
   U8BIT tuner_num;
   E_STB_DP_LNB_POWER ret_val;

   FUNCTION_START(STB_DPGetLNBPower);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].lnb_power;

   FUNCTION_FINISH(STB_DPGetLNBPower);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes LNB type value into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   E_STB_DP_LNB_TYPE type - the LNB type
 *

 *
 */
void STB_DPSetLNBType(U8BIT path, E_STB_DP_LNB_TYPE type)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetLNBType);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetLNBType(%d)   : tuner %d type %d", path, tuner_num, type));
   if (type != tuner_status[tuner_num].lnb_type)
   {
      tuner_status[tuner_num].lnb_type = type;
      tuner_status[tuner_num].actions |= ACTION_TUNE;
   }

   FUNCTION_FINISH(STB_DPSetLNBType);
}

/**
 *

 *
 * @brief   Reads the LNB type value from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   E_STB_DP_LNB_TYPE - LNB type.
 *
 */
E_STB_DP_LNB_TYPE STB_DPGetLNBType(U8BIT path)
{
   U8BIT tuner_num;
   E_STB_DP_LNB_TYPE ret_val;

   FUNCTION_START(STB_DPGetLNBType);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].lnb_type;

   FUNCTION_FINISH(STB_DPGetLNBType);

   return(ret_val);
}

/**
 * @brief   Stores the pointer to the array of band definitions for the user defined LNB
 * @param   path ID of the decode path to use
 * @param   number_of_bands Number of bands in the array
 * @param   band_definitions Array containing the definition of the behaviour of user defined LNB
 *          for for each band. A band is defined as the combination of a frequency range and a
 *          polarity. It is responsibility of the caller to ensure that the pointer remains valid
 *          until this function is called passing a different one.
 */
void STB_DPSetUserDefinedLNBBands(U8BIT path, U8BIT number_of_bands, S_STB_DP_LNB_BAND *band_definitions)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetUserDefinedLNBBands);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   tuner_status[tuner_num].lnb_definition[LNB_TYPE_USER_DEFINED].number_of_bands = number_of_bands;
   tuner_status[tuner_num].lnb_definition[LNB_TYPE_USER_DEFINED].band_list = band_definitions;

   FUNCTION_FINISH(STB_DPSetUserDefinedLNBBands);
}

/**
 *

 *
 * @brief   Sets LNB 22kHz on or off and writes status into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   BOOLEAN state - TRUE to turn on, else FALSE.
 *

 *
 */
void STB_DPSetLNB22k(U8BIT path, BOOLEAN state)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetLNB22k);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetLNB22k(%d)    : tuner %d state %d", path, tuner_num, state));
   if (state != tuner_status[tuner_num].lnb_22k)
   {
      tuner_status[tuner_num].lnb_22k = state;

      // check if we need to turn on/off 22KHz
      if (tuner_status[tuner_num].lnb_power != LNB_POWER_OFF)
      {
         // only switch tone if single LNB
         if (tuner_status[tuner_num].lnb_type == LNB_TYPE_SINGLE)
         {
            tuner_status[tuner_num].actions |= ACTION_TUNE;
         }
      }
   }

   FUNCTION_FINISH(STB_DPSetLNB22k);
}

/**
 *

 *
 * @brief   Reads the status of LNB 22kHz from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   BOOLEAN - TRUE if 22k tone is on, else FALSE.
 *
 */
BOOLEAN STB_DPGetLNB22k(U8BIT path)
{
   U8BIT tuner_num;
   BOOLEAN ret_val;

   FUNCTION_START(STB_DPGetLNB22k);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].lnb_22k;

   FUNCTION_FINISH(STB_DPGetLNB22k);

   return(ret_val);
}

/**
 *

 *
 * @brief   Sets LNB 12v on or off and writes status into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   BOOLEAN state - TRUE to turn on, else FALSE.
 *

 *
 */
void STB_DPSetLNB12v(U8BIT path, BOOLEAN state)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetLNB12v);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetLNB12v(%d)    : tuner %d state %d", path, tuner_num, state));
   if (state != tuner_status[tuner_num].lnb_12v)
   {
      tuner_status[tuner_num].lnb_12v = state;
      tuner_status[tuner_num].actions |= ACTION_TUNE;
   }

   FUNCTION_FINISH(STB_DPSetLNB12v);
}

/**
 *

 *
 * @brief   Reads the status of LNB 12v from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   BOOLEAN - TRUE if 12v tone is on, else FALSE.
 *
 */
BOOLEAN STB_DPGetLNB12v(U8BIT path)
{
   U8BIT tuner_num;
   BOOLEAN ret_val;

   FUNCTION_START(STB_DPGetLNB12v);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].lnb_12v;

   FUNCTION_FINISH(STB_DPGetLNB12v);

   return(ret_val);
}

/**
 *

 *
 * @brief   Enables/disables pulse positioning and writes status into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   BOOLEAN state - TRUE to turn on, else FALSE.
 *

 *
 */
void STB_DPSetPulsePosition(U8BIT path, BOOLEAN state)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetPulsePosition);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetPulsePosition(%d): state %d", path, state));
   tuner_status[tuner_num].pulse_position = state;

   FUNCTION_FINISH(STB_DPSetPulsePosition);
}

/**
 *

 *
 * @brief   Reads the status of pulse positioning from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   BOOLEAN - TRUE if pulse positioning is on, else FALSE.
 *
 */
BOOLEAN STB_DPGetPulsePosition(U8BIT path)
{
   U8BIT tuner_num;
   BOOLEAN ret_val;

   FUNCTION_START(STB_DPGetPulsePosition);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].pulse_position;

   FUNCTION_FINISH(STB_DPGetPulsePosition);

   return(ret_val);
}

/**
 *

 *
 * @brief   Enables/disables DiSEqC positioning and writes status into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   BOOLEAN state - TRUE to turn on, else FALSE.
 *

 *
 */
void STB_DPSetDISEQCPosition(U8BIT path, BOOLEAN state)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetDISEQCPosition);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetDISEQCPosition(%d): state %d", path, state));
   tuner_status[tuner_num].diseqc_position = state;

   FUNCTION_FINISH(STB_DPSetDISEQCPosition);
}

/**
 *

 *
 * @brief   Reads the status of DiSEqC positioning from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   BOOLEAN - TRUE if DiSEqC positioning is on, else FALSE.
 *
 */
BOOLEAN STB_DPGetDISEQCPosition(U8BIT path)
{
   U8BIT tuner_num;
   BOOLEAN ret_val;

   FUNCTION_START(STB_DPGetDISEQCPosition);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].diseqc_position;

   FUNCTION_FINISH(STB_DPGetDISEQCPosition);

   return(ret_val);
}

/**
 *

 *
 * @brief   Sets DiSEqC committed switch and writes status into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   E_STB_DP_DISEQC_CSWITCH state - switch state.
 *

 *
 */
void STB_DPSetDISEQCCSwitch(U8BIT path, E_STB_DP_DISEQC_CSWITCH state)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetDISEQCCSwitch);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetDISEQCCSwitch(%d): state %d", path, state));
   if (state != tuner_status[tuner_num].diseqc_cswitch)
   {
      tuner_status[tuner_num].diseqc_cswitch = state;
      tuner_status[tuner_num].actions |= ACTION_TUNE;
   }

   FUNCTION_FINISH(STB_DPSetDISEQCCSwitch);
}

/**
 *

 *
 * @brief   Reads the status of DiSEqC committed switch from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   E_STB_DP_DISEQC_CSWITCH - switch state.
 *
 */
E_STB_DP_DISEQC_CSWITCH STB_DPGetDISEQCCSwitch(U8BIT path)
{
   U8BIT tuner_num;
   E_STB_DP_DISEQC_CSWITCH ret_val;

   FUNCTION_START(STB_DPGetDISEQCCSwitch);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].diseqc_cswitch;

   FUNCTION_FINISH(STB_DPGetDISEQCCSwitch);

   return(ret_val);
}

/**
 *

 *
 * @brief   Sets DiSEqC uncommitted switch and writes status into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   U8BIT state - switch state.
 *

 *
 */
void STB_DPSetDISEQCUSwitch(U8BIT path, U8BIT state)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetDISEQCUSwitch);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetDISEQCUSwitch(%d): state %d", path, state));
   if (state != tuner_status[tuner_num].diseqc_uswitch)
   {
      tuner_status[tuner_num].diseqc_uswitch = state;
      tuner_status[tuner_num].actions |= ACTION_TUNE;
   }

   FUNCTION_FINISH(STB_DPSetDISEQCUSwitch);
}

/**
 *

 *
 * @brief   Reads the status of DiSEqC uncommitted switch from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   U8BIT - switch state.
 *
 */
U8BIT STB_DPGetDISEQCUSwitch(U8BIT path)
{
   U8BIT tuner_num;
   U8BIT ret_val;

   FUNCTION_START(STB_DPGetDISEQCUSwitch);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].diseqc_uswitch;

   FUNCTION_FINISH(STB_DPGetDISEQCUSwitch);

   return(ret_val);
}

/**
 *

 *
 * @brief   Sets DiSEqC tone and writes status into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   E_STB_DP_DISEQC_TONE state - tone state.
 *

 *
 */
void STB_DPSetDISEQCTone(U8BIT path, E_STB_DP_DISEQC_TONE state)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetDISEQCTone);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetDISEQCTone(%d): state %d", path, state));
   if (state != tuner_status[tuner_num].diseqc_tone)
   {
      tuner_status[tuner_num].diseqc_tone = state;
      tuner_status[tuner_num].actions |= ACTION_TUNE;
   }

   FUNCTION_FINISH(STB_DPSetDISEQCTone);
}

/**
 *

 *
 * @brief   Reads the status of DiSEqC tone from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   E_STB_DP_DISEQC_TONE - tone state.
 *
 */
E_STB_DP_DISEQC_TONE STB_DPGetDISEQCTone(U8BIT path)
{
   U8BIT tuner_num;
   E_STB_DP_DISEQC_TONE ret_val;

   FUNCTION_START(STB_DPGetDISEQCTone);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].diseqc_tone;

   FUNCTION_FINISH(STB_DPGetDISEQCTone);

   return(ret_val);
}

/**
 *

 *
 * @brief   Enables/disables DiSEqC SMATV and writes status into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   BOOLEAN state - TRUE to turn on, else FALSE.
 *

 *
 */
void STB_DPSetDISEQCSMATV(U8BIT path, BOOLEAN state)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetDISEQCSMATV);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetDISEQCSMATV(%d): state %d", path, state));
   if (state != tuner_status[tuner_num].diseqc_smatv)
   {
      tuner_status[tuner_num].diseqc_smatv = state;
      tuner_status[tuner_num].actions |= ACTION_TUNE;
   }

   FUNCTION_FINISH(STB_DPSetDISEQCSMATV);
}

/**
 *

 *
 * @brief   Reads the status of DiSEqC SMATV from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   BOOLEAN - TRUE if on, else FALSE.
 *
 */
BOOLEAN STB_DPGetDISEQCSMATV(U8BIT path)
{
   U8BIT tuner_num;
   BOOLEAN ret_val;

   FUNCTION_START(STB_DPGetDISEQCSMATV);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].diseqc_smatv;

   FUNCTION_FINISH(STB_DPGetDISEQCSMATV);

   return(ret_val);
}

/**
 *

 *
 * @brief   Sets DiSEqC repeats and writes value into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   U8BIT count - the value of DiSEqC repeats
 *

 *
 */
void STB_DPSetDISEQCRepeats(U8BIT path, U8BIT count)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetDISEQCRepeats);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetDISEQCRepeats(%u, %u): current count=%u", path, count,
                   tuner_status[tuner_num].diseqc_repeats));

   if (count != tuner_status[tuner_num].diseqc_repeats)
   {
      tuner_status[tuner_num].diseqc_repeats = count;
      tuner_status[tuner_num].actions |= ACTION_TUNE;
   }

   FUNCTION_FINISH(STB_DPSetDISEQCRepeats);
}

/**
 *

 *
 * @brief   Reads the value of DiSEqC repeats from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   U8BIT - value of DiSEqC repeats.
 *
 */
U8BIT STB_DPGetDISEQCRepeats(U8BIT path)
{
   U8BIT tuner_num;
   BOOLEAN ret_val;

   FUNCTION_START(STB_DPGetDISEQCRepeats);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].diseqc_repeats;

   FUNCTION_FINISH(STB_DPGetDISEQCRepeats);

   return(ret_val);
}

/**
 *

 *
 * @brief   Interrogates Unicable LNB for its user and local oscillator frequencies
 *
 * @param   U8BIT path
 * @param   U32BIT unicable_if - caller's array to hold up to 8 bank IF frequencies
 * @param   U32BIT* lnb_lo_freq - caller's variable to hold returned low LO frequency
 * @param   U32BIT* lnb_hi_freq - caller's variable to hold returned high LO frequency
 *
 * @return   U8BIT - number of user banks successfully found
 *
 */
U8BIT STB_DPGetUnicableParams(U8BIT path, U32BIT unicable_if[MAX_UNICABLE_BANKS], U32BIT *lnb_lo_freq, U32BIT *lnb_hi_freq)
{
   U8BIT tuner;
   U8BIT cmd_signals_on[5] = {0xe0, 0x10, 0x5b, 0x00, 0x00};
   U8BIT cmd_lofreq[5] = {0xe0, 0x10, 0x5b, 0x02, 0x00};
   U8BIT stren = 0;
   U32BIT f;
   U8BIT bank_n = 0;
   U8BIT lut;
   // Unicable LO frequencies look-up table
   U32BIT lofreq_lut[0x0c] = {9750000, 10000000, 10600000, 10750000, 11000000, 11250000, 11475000, 20250000, 5150000, 1585000, 13850000};

   tuner = path_status[path].tuner_no;
   *lnb_lo_freq = 0;
   *lnb_hi_freq = 0;

   STB_TuneSetLNBVoltage(tuner, LNB_VOLTAGE_14V);
   STB_OSTaskDelay(1000);
   UnicableSendCommand(tuner, cmd_signals_on);

   // First find what frequencies the banks are located at
   for (f = 950000; f < 2150000; f += 4000)
   {
      stren = STB_TuneSatGetCarrierStrength(tuner, f);
      if (stren > UNICABLE_SIGNAL_THRESHOLD)
      {
         unicable_if[bank_n] = f;
         bank_n++;
      }
   }

   // If banks were found, get the LO frequencies for this LNB
   // assume LO frequencies are same for all banks
   if (bank_n > 0)
   {
      // Now query the device's local oscillator frequencies
      for (lut = 0x02; lut < 0x0c; lut++)
      {
         cmd_lofreq[4] = lut;
         UnicableSendCommand(tuner, cmd_lofreq);
         // "No" answer is flagged by carrier 20MHz above bank freq
         stren = STB_TuneSatGetCarrierStrength(tuner, unicable_if[bank_n] + 20000);
         if (stren < UNICABLE_SIGNAL_THRESHOLD)
         {
            // "Yes" answer is flagged by carrier at bank freq
            stren = STB_TuneSatGetCarrierStrength(tuner, unicable_if[bank_n]);
            if (stren > UNICABLE_SIGNAL_THRESHOLD)
            {
               // Populate the lo/hi osc frequencies (in ascending order)
               if (*lnb_lo_freq == 0)
               {
                  *lnb_lo_freq = lofreq_lut[lut];
               }
               else if (*lnb_hi_freq == 0)
               {
                  *lnb_hi_freq = lofreq_lut[lut];
               }
            }
         }
      }
   }

   // Return number of Unicable banks found (0 if error or none found)
   return bank_n;
}

/**
 *

 *
 * @brief   Sets whether to use the alternate satellite position (Position B)
 *
 * @param   U8BIT path
 * @param   BOOLEAN position_b - TRUE if position B to be used
 *

 *
 */
void STB_DPSetUnicablePositionB(U8BIT path, BOOLEAN position_b)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetUnicablePositionB);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   tuner_status[tuner_num].unicable_position_b = position_b;

   FUNCTION_FINISH(STB_DPSetUnicablePositionB);
}

/**
 *

 *
 * @brief   Gets whether Satellite Position B is currently set
 *
 * @param   U8BIT path
 *
 * @return   TRUE if position B used, FALSE otherwise
 *
 */
BOOLEAN STB_DPGetUnicablePositionB(U8BIT path)
{
   U8BIT tuner_num;
   U8BIT ret_val;

   FUNCTION_START(STB_DPGetUnicablePositionB);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].unicable_position_b;

   FUNCTION_FINISH(STB_DPGetUnicablePositionB);

   return(ret_val);
}

/**
 *

 *
 * @brief   Sets the current Unicable Channel (User Band).
 *
 * @param   U8BIT path
 * @param   U8BIT chan - The Unicable Channel to use
 *

 *
 */
void STB_DPSetUnicableChannel(U8BIT path, U8BIT chan)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetUnicableChannel);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   tuner_status[tuner_num].unicable_chan = chan;

   FUNCTION_FINISH(STB_DPSetUnicableChannel);
}

/**
 *

 *
 * @brief   Gets the currently set Unicable Channel (User Band).
 *
 * @param   U8BIT path
 *
 * @return   The Unicable channel
 *
 */
U8BIT STB_DPGetUnicableChannel(U8BIT path)
{
   U8BIT tuner_num;
   U8BIT ret_val;

   FUNCTION_START(STB_DPGetUnicableChannel);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].unicable_chan;

   FUNCTION_FINISH(STB_DPGetUnicableChannel);

   return(ret_val);
}

/**
 *

 *
 * @brief   Sets the Unicable (IF) frequency.
 *
 * @param   U8BIT path
 * @param   U32BIT freq - the IF in KHz
 *

 *
 */
void STB_DPSetUnicableFrequency(U8BIT path, U32BIT freq)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetUnicableFrequency);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   tuner_status[tuner_num].unicable_if = freq;

   FUNCTION_FINISH(STB_DPSetUnicableFrequency);
}

/**
 *

 *
 * @brief   Gets the currently set Unicable (IF) frequency.
 *
 * @param   U8BIT path
 *
 * @return   The frequency in KHz
 *
 */
U32BIT STB_DPGetUnicableFrequency(U8BIT path)
{
   U8BIT tuner_num;
   U32BIT ret_val;

   FUNCTION_START(STB_DPGetUnicableFrequency);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].unicable_if;

   FUNCTION_FINISH(STB_DPGetUnicableFrequency);

   return(ret_val);
}

/**
 *

 *
 * @brief   Sets current sat dish position as East limit.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *

 *
 */
void STB_DPSetDishLimitE(U8BIT path)
{
   U8BIT tuner_num;
   U8BIT dmsg_data[4];
   U16BIT pos;

   FUNCTION_START(STB_DPSetDishLimitE);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetDishLimitE(%d)", path));

   if (tuner_status[tuner_num].pulse_position == TRUE)
   {
      pos = STB_TuneGetPulsePosition(tuner_num);
      STB_TuneSetPulseLimitEast(tuner_num, pos);
   }

   if (tuner_status[tuner_num].diseqc_position == TRUE)
   {
      dmsg_data[0] = 0xe0;
      dmsg_data[1] = 0x31;
      dmsg_data[2] = 0x66;
      SendDISEQCMessage(tuner_num, dmsg_data, 3);
   }

   FUNCTION_FINISH(STB_DPSetDishLimitE);
}

/**
 *

 *
 * @brief   Sets current sat dish position as West limit.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *

 *
 */
void STB_DPSetDishLimitW(U8BIT path)
{
   U8BIT tuner_num;
   U8BIT dmsg_data[4];
   U16BIT pos;

   FUNCTION_START(STB_DPSetDishLimitW);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetDishLimitW(%d)", path));

   if (tuner_status[tuner_num].pulse_position == TRUE)
   {
      pos = STB_TuneGetPulsePosition(tuner_num);
      STB_TuneSetPulseLimitWest(tuner_num, pos);
   }

   if (tuner_status[tuner_num].diseqc_position == TRUE)
   {
      dmsg_data[0] = 0xe0;
      dmsg_data[1] = 0x31;
      dmsg_data[2] = 0x67;
      SendDISEQCMessage(tuner_num, dmsg_data, 3);
   }

   FUNCTION_FINISH(STB_DPSetDishLimitW);
}

/**
 *

 *
 * @brief   Enables sat dish limits.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   U8BIT ecount - east limit count (ignored for DiSEqC)
 * @param   U8BIT wcount - west limit count (ignored for DiSEqC)
 *

 *
 */
void STB_DPEnableDishLimits(U8BIT path, U16BIT ecount, U16BIT wcount)
{
   U8BIT tuner_num;
   U8BIT dmsg_data[4];

   FUNCTION_START(STB_DPEnableDishLimits);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPEnableDishLimits(%d)", path));

   if (tuner_status[tuner_num].pulse_position == TRUE)
   {
      STB_TuneSetPulseLimitEast(tuner_num, ecount);
      STB_TuneSetPulseLimitWest(tuner_num, wcount);
   }

   if (tuner_status[tuner_num].diseqc_position == TRUE)
   {
      dmsg_data[0] = 0xe0;
      dmsg_data[1] = 0x31;
      dmsg_data[2] = 0x6a;
      dmsg_data[3] = 0x00;
      SendDISEQCMessage(tuner_num, dmsg_data, 4);
   }

   FUNCTION_FINISH(STB_DPEnableDishLimits);
}

/**
 *

 *
 * @brief   Disables sat dish limits.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *

 *
 */
void STB_DPDisableDishLimits(U8BIT path)
{
   U8BIT tuner_num;
   U8BIT dmsg_data[4];

   FUNCTION_START(STB_DPDisableDishLimits);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPDisableDishLimits(%d)", path));

   if (tuner_status[tuner_num].pulse_position == TRUE)
   {
      STB_TuneSetPulseLimitEast(tuner_num, 1);
      STB_TuneSetPulseLimitWest(tuner_num, 9999);
   }

   if (tuner_status[tuner_num].diseqc_position == TRUE)
   {
      dmsg_data[0] = 0xe0;
      dmsg_data[1] = 0x31;
      dmsg_data[2] = 0x63;
      SendDISEQCMessage(tuner_num, dmsg_data, 3);
   }

   FUNCTION_FINISH(STB_DPDisableDishLimits);
}

/**
 *

 *
 * @brief   Starts sat dish moving east.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   U16BIT count - step count (0 = continuous)
 *

 *
 */
void STB_DPStartDishMoveE(U8BIT path, U16BIT count)
{
   U8BIT tuner_num;
   U8BIT dmsg_data[4];
   U16BIT pos;

   FUNCTION_START(STB_DPStartDishMoveE);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPStartDishMoveE(%d): count %d", path, count));

   tuner_status[tuner_num].actions |= (ACTION_TUNE | ACTION_DISH);

   if (tuner_status[tuner_num].pulse_position == TRUE)
   {
      pos = STB_TuneGetPulsePosition(tuner_num);
      if ((pos > count) && (count != 0))
      {
         pos -= count;
      }
      else
      {
         pos = 1;
      }

      STB_TuneChangePulsePosition(tuner_num, pos);
   }

   if (tuner_status[tuner_num].diseqc_position == TRUE)
   {
      if ((U8BIT)count < 0x80)
      {
         dmsg_data[0] = 0xe0;
         dmsg_data[1] = 0x31;
         dmsg_data[2] = 0x68;
         if (count == 0)
         {
            dmsg_data[3] = 0x00;
         }
         else
         {
            dmsg_data[3] = (0xff - (U8BIT)count + 1);
         }

         SendDISEQCMessage(tuner_num, dmsg_data, 4);
      }
   }

   FUNCTION_FINISH(STB_DPStartDishMoveE);
}

/**
 *

 *
 * @brief   Starts sat dish moving west.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   U16BIT count - step count (0 = continuous)
 *

 *
 */
void STB_DPStartDishMoveW(U8BIT path, U16BIT count)
{
   U8BIT tuner_num;
   U8BIT dmsg_data[4];
   U16BIT pos;

   FUNCTION_START(STB_DPStartDishMoveW);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPStartDishMoveW(%d): count %d", path, count));

   tuner_status[tuner_num].actions |= (ACTION_TUNE | ACTION_DISH);

   if (tuner_status[tuner_num].pulse_position == TRUE)
   {
      pos = STB_TuneGetPulsePosition(tuner_num);
      if ((pos < (9999 - count)) && (count != 0))
      {
         pos += count;
      }
      else
      {
         pos = 9999;
      }
      STB_TuneChangePulsePosition(tuner_num, pos);
   }

   if (tuner_status[tuner_num].diseqc_position == TRUE)
   {
      if ((U8BIT)count < 0x80)
      {
         dmsg_data[0] = 0xe0;
         dmsg_data[1] = 0x31;
         dmsg_data[2] = 0x69;
         if (count == 0)
         {
            dmsg_data[3] = 0x00;
         }
         else
         {
            dmsg_data[3] = (0xff - (U8BIT)count + 1);
         }
         SendDISEQCMessage(tuner_num, dmsg_data, 4);
      }
   }

   FUNCTION_FINISH(STB_DPStartDishMoveW);
}

/**
 *

 *
 * @brief   Stops sat dish moving.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *

 *
 */
void STB_DPStopDishMove(U8BIT path)
{
   U8BIT tuner_num;
   U8BIT dmsg_data[4];
   U16BIT pos;

   FUNCTION_START(STB_DPStopDishMove);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPStopDishMove(%d)", path));

   if (tuner_status[tuner_num].pulse_position == TRUE)
   {
      pos = 0xffff;
      STB_TuneChangePulsePosition(tuner_num, pos);
   }

   if (tuner_status[tuner_num].diseqc_position == TRUE)
   {
      dmsg_data[0] = 0xe0;
      dmsg_data[1] = 0x31;
      dmsg_data[2] = 0x60;
      SendDISEQCMessage(tuner_num, dmsg_data, 3);
   }

   FUNCTION_FINISH(STB_DPStopDishMove);
}

/**
 *

 *
 * @brief   Moves dish to centre.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *

 *
 */
void STB_DPCentreDishMove(U8BIT path)
{
   U8BIT tuner_num;
   U8BIT dmsg_data[4];

   FUNCTION_START(STB_DPCentreDishMove);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPCentreDishMove(%d)", path));

   tuner_status[tuner_num].actions |= (ACTION_TUNE | ACTION_DISH);

   if (tuner_status[tuner_num].diseqc_position == TRUE)
   {
      dmsg_data[0] = 0xe0;
      dmsg_data[1] = 0x31;
      dmsg_data[2] = 0x6b;
      dmsg_data[3] = 0x00;
      SendDISEQCMessage(tuner_num, dmsg_data, 4);
   }

   FUNCTION_FINISH(STB_DPCentreDishMove);
}

/**
 *

 *
 * @brief   Stores sat dish position as given DiSEqC index number, or pulse count.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   U16BIT count - the index or pulse count to store
 *

 *
 */
void STB_DPStoreDishPosition(U8BIT path, U16BIT count)
{
   U8BIT tuner_num;
   U8BIT dmsg_data[4];

   FUNCTION_START(STB_DPStoreDishPosition);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPStoreDishPosition(%d): count %d", path, count));

   if (tuner_status[tuner_num].pulse_position == TRUE)
   {
      if (count == 0)
      {
         count = STB_TuneGetPulsePosition(tuner_num);
      }
      STB_TuneAtPulsePosition(tuner_num, count);
   }

   if (tuner_status[tuner_num].diseqc_position == TRUE)
   {
      if (count != 0)
      {
         dmsg_data[0] = 0xe0;
         dmsg_data[1] = 0x31;
         dmsg_data[2] = 0x6a;
         dmsg_data[3] = (U8BIT)count;
         SendDISEQCMessage(tuner_num, dmsg_data, 4);
      }
   }

   tuner_status[tuner_num].dish_pos = count;

   FUNCTION_FINISH(STB_DPStoreDishPosition);
}

/**
 *

 *
 * @brief   Writes sat dish position value into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   U16BIT count - the value of dish position count
 *

 *
 */
void STB_DPSetDishPosition(U8BIT path, U16BIT count)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetDishPosition);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetDishPosition(p%d): count %d", path, count));
   if (count != tuner_status[tuner_num].dish_pos)
   {
      // Zero is NOT a valid position to store a satellite. But allowed to be entered into the
      // decode path so that the move to position command is force to be sent for the current
      // service; hence allowed to set but is not flagged as to be actioned.
      tuner_status[tuner_num].dish_pos = count;
      if (count != 0)
      {
         tuner_status[tuner_num].actions |= (ACTION_TUNE | ACTION_DISH);
      }
   }

   FUNCTION_FINISH(STB_DPSetDishPosition);
}

/**
 *

 *
 * @brief   Reads the actual sat dish position value.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   U16BIT - value of dish position count.
 *
 */
U16BIT STB_DPGetDishPosition(U8BIT path)
{
   U8BIT tuner_num;
   U16BIT ret_val;

   FUNCTION_START(STB_DPGetDishPosition);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].dish_pos;

   // return current position for pulse
   if (tuner_status[tuner_num].pulse_position == TRUE)
   {
      ret_val = STB_TuneGetPulsePosition(tuner_num);
   }

   FUNCTION_FINISH(STB_DPGetDishPosition);

   return(ret_val);
}

/**
 *

 *
 * @brief   Reads the sat dish position value from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   U16BIT - value of dish position count.
 *
 */
U16BIT STB_DPGetDishRequest(U8BIT path)
{
   U8BIT tuner_num;
   U16BIT ret_val;

   FUNCTION_START(STB_DPGetDishRequest);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].dish_pos;

   FUNCTION_FINISH(STB_DPGetDishRequest);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes sat skew position value into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   U16BIT count - the value of skew position count
 *

 *
 */
void STB_DPSetSkewPosition(U8BIT path, U16BIT count)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetSkewPosition);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetSkewPosition(p%d,t%d): count %d", path, tuner_num, count));
   if (count != tuner_status[tuner_num].skew_pos)
   {
      tuner_status[tuner_num].skew_pos = count;
      tuner_status[tuner_num].actions |= (ACTION_TUNE | ACTION_SKEW);
   }

   FUNCTION_FINISH(STB_DPSetSkewPosition);
}

/**
 *

 *
 * @brief   Reads the sat skew position value from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   U16BIT - value of skew position count.
 *
 */
U16BIT STB_DPGetSkewPosition(U8BIT path)
{
   U8BIT tuner_num;
   U16BIT ret_val;

   FUNCTION_START(STB_DPGetSkewPosition);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].skew_pos;

   FUNCTION_FINISH(STB_DPGetSkewPosition);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes frequency value into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   U32BIT freq - the value of frequency
 *

 *
 */
void STB_DPSetFrequency(U8BIT path, U32BIT freq)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetFrequency);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetFrequency(%d) : tuner %d freq %d", path, tuner_num, freq));
   if (freq != tuner_status[tuner_num].frequency)
   {
      tuner_status[tuner_num].frequency = freq;
      tuner_status[tuner_num].actions |= ACTION_TUNE;
   }

   FUNCTION_FINISH(STB_DPSetFrequency);
}

/**
 *

 *
 * @brief   Reads the frequency value from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   U32BIT - value of frequency.
 *
 */
U32BIT STB_DPGetFrequency(U8BIT path)
{
   U8BIT tuner_num;
   U32BIT ret_val;

   FUNCTION_START(STB_DPGetFrequency);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].frequency;

   FUNCTION_FINISH(STB_DPGetFrequency);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes polarity value into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   E_STB_DP_POLARITY pol - the value of polarity
 *

 *
 */
void STB_DPSetPolarity(U8BIT path, E_STB_DP_POLARITY pol)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetPolarity);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetPolarity(%d)  : tuner %d pol %d", path, tuner_num, pol));
   if (pol != tuner_status[tuner_num].polarity)
   {
      tuner_status[tuner_num].polarity = pol;
      tuner_status[tuner_num].actions |= ACTION_TUNE;
   }

   FUNCTION_FINISH(STB_DPSetPolarity);
}

/**
 *

 *
 * @brief   Reads the polarity value from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   E_STB_DP_POLARITY - value of polarity.
 *
 */
E_STB_DP_POLARITY STB_DPGetPolarity(U8BIT path)
{
   U8BIT tuner_num;
   E_STB_DP_POLARITY ret_val;

   FUNCTION_START(STB_DPGetPolarity);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].polarity;

   FUNCTION_FINISH(STB_DPGetPolarity);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes symbol rate value into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   U16BIT sym - the value of symbol rate
 *

 *
 */
void STB_DPSetSymbolRate(U8BIT path, U16BIT sym)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetSymbolRate);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetSymbolRate(%d): tuner %d sym %d", path, tuner_num, sym));
   if (sym != tuner_status[tuner_num].sym_rate)
   {
      tuner_status[tuner_num].sym_rate = sym;
      tuner_status[tuner_num].actions |= ACTION_TUNE;
   }

   FUNCTION_FINISH(STB_DPSetSymbolRate);
}

/**
 *

 *
 * @brief   Reads the symbol rate value from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   U16BIT - value of symbol rate.
 *
 */
U16BIT STB_DPGetSymbolRate(U8BIT path)
{
   U8BIT tuner_num;
   U16BIT ret_val;

   FUNCTION_START(STB_DPGetSymbolRate);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].sym_rate;

   FUNCTION_FINISH(STB_DPGetSymbolRate);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes FEC value into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   E_STB_DP_POLARITY fec - the value of FEC
 *

 *
 */
void STB_DPSetFEC(U8BIT path, E_STB_DP_FEC fec)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetFEC);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetFEC(%d)       : tuner %d fec %d", path, tuner_num, fec));
   if (fec != tuner_status[tuner_num].fec)
   {
      tuner_status[tuner_num].fec = fec;
      tuner_status[tuner_num].actions |= ACTION_TUNE;
   }

   FUNCTION_FINISH(STB_DPSetFEC);
}

/**
 *

 *
 * @brief   Reads the FEC value from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   E_STB_DP_FEC - value of FEC.
 *
 */
E_STB_DP_FEC STB_DPGetFEC(U8BIT path)
{
   U8BIT tuner_num;
   E_STB_DP_FEC ret_val;

   FUNCTION_START(STB_DPGetFEC);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].fec;

   FUNCTION_FINISH(STB_DPGetFEC);

   return(ret_val);
}

/*!**************************************************************************
 * @brief   Set whether the satellite should be tuned to DVB-S/S2 for the tuner in the given decode path
 * @param   path - decode path
 * @param   dvb_s2 - TRUE for DVB-S2
 ****************************************************************************/
void STB_DPSetDVBS2(U8BIT path, BOOLEAN dvb_s2)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetDVBS2);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetDVBS2(%u): tuner %u DVB-S2 %d", path, tuner_num, dvb_s2));

   if (dvb_s2 != tuner_status[tuner_num].dvb_s2)
   {
      tuner_status[tuner_num].dvb_s2 = dvb_s2;
      tuner_status[tuner_num].actions |= ACTION_TUNE;
   }

   FUNCTION_FINISH(STB_DPSetDVBS2);
}

/*!**************************************************************************
 * @brief   Returns whether the sat tuner is tuned to DVB-S or DVB-S2
 * @param   path - decode path
 * @return  DVB-S/DVB-S2 for the decode path
 ****************************************************************************/
BOOLEAN STB_DPGetDVBS2(U8BIT path)
{
   U8BIT tuner_num;
   BOOLEAN retval;

   FUNCTION_START(STB_DPGetDVBS2);

   tuner_num = path_status[path].tuner_no;
   ASSERT(tuner_num < num_tuners);
   retval = tuner_status[tuner_num].dvb_s2;

   FUNCTION_FINISH(STB_DPGetDVBS2);

   return(retval);
}

/*!**************************************************************************
 * @brief   Set the satellite modulation for the tuner in the given decode path
 * @param   path - decode path
 * @param   modulation - satellite modulation type
 ****************************************************************************/
void STB_DPSetModulation(U8BIT path, E_STB_DP_MODULATION modulation)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetModulation);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetModulation(%u): tuner %u modulation %d", path, tuner_num, modulation));

   if (modulation != tuner_status[tuner_num].modulation)
   {
      tuner_status[tuner_num].modulation = modulation;
      tuner_status[tuner_num].actions |= ACTION_TUNE;
   }

   FUNCTION_FINISH(STB_DPSetModulation);
}

/*!**************************************************************************
 * @brief   Returns the satellite modulation type for the give decode path
 * @param   path - decode path
 * @return  Satellite modulation type for the decode path
 ****************************************************************************/
E_STB_DP_MODULATION STB_DPGetModulation(U8BIT path)
{
   U8BIT tuner_num;
   E_STB_DP_MODULATION retval;

   FUNCTION_START(STB_DPGetModulation);

   tuner_num = path_status[path].tuner_no;
   ASSERT(tuner_num < num_tuners);
   retval = tuner_status[tuner_num].modulation;

   FUNCTION_FINISH(STB_DPGetModulation);

   return(retval);
}

/**
 *

 *
 * @brief   Writes terrestrial tuning mode value into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   E_STB_DP_TMODE mode - the value of mode
 *

 *
 */
void STB_DPSetTerrMode(U8BIT path, E_STB_DP_TMODE mode)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetTerrMode);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetTerrMode(%d): mode %d", path, mode));
   if (mode != tuner_status[tuner_num].terr_mode)
   {
      tuner_status[tuner_num].terr_mode = mode;
      tuner_status[tuner_num].actions |= ACTION_TUNE;
   }

   FUNCTION_FINISH(STB_DPSetTerrMode);
}

/**
 *

 *
 * @brief   Reads the terrestrial tuning mode value from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   E_STB_DP_TMODE - value of mode.
 *
 */
E_STB_DP_TMODE STB_DPGetTerrMode(U8BIT path)
{
   U8BIT tuner_num;
   E_STB_DP_TMODE ret_val;

   FUNCTION_START(STB_DPGetTerrMode);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].terr_mode;

   FUNCTION_FINISH(STB_DPGetTerrMode);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes terrestrial bandwidth value into decode path store.
 *
 * @param   path   - the ID of the decode path to use
 * @param   bwidth - the value of bandwidth
 *

 *
 */
void STB_DPSetTerrBandwidth(U8BIT path, E_STB_DP_TBWIDTH bwidth)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetTerrBandwidth);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetTerrBandwidth(%d): bwidth %d", path, bwidth));
   if (bwidth != tuner_status[tuner_num].terr_bwidth)
   {
      tuner_status[tuner_num].terr_bwidth = bwidth;
      tuner_status[tuner_num].actions |= ACTION_TUNE;
   }

   FUNCTION_FINISH(STB_DPSetTerrBandwidth);
}

/**
 *

 *
 * @brief   Reads the terrestrial frequency offset value from decode path store.
 *
 * @param   path - the ID of the decode path to use
 *
 * @return   bandwidth
 *
 */
E_STB_DP_TBWIDTH STB_DPGetTerrBandwidth(U8BIT path)
{
   U8BIT tuner_num;
   E_STB_DP_TBWIDTH ret_val;

   FUNCTION_START(STB_DPGetTerrBandwidth);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].terr_bwidth;

   FUNCTION_FINISH(STB_DPGetTerrBandwidth);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes terrestrial frequency offset value into decode path store.
 *
 * @param   path   - the ID of the decode path to use
 * @param   offset - the value of offset
 *

 *
 */
void STB_DPSetTerrFreqOff(U8BIT path, S8BIT offset)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetTerrFreqOff);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetTerrFreqOff(%d): offset %d", path, offset));
   if (offset != tuner_status[tuner_num].freq_offset)
   {
      tuner_status[tuner_num].freq_offset = offset;
      tuner_status[tuner_num].actions |= ACTION_TUNE;
   }

   FUNCTION_FINISH(STB_DPSetTerrFreqOff);
}

/**
 *

 *
 * @brief   Reads the terrestrial frequency offset value from decode path store.
 *
 * @param   path - the ID of the decode path to use
 *
 * @return   value of offset.
 *
 */
S8BIT STB_DPGetTerrFreqOff(U8BIT path)
{
   U8BIT tuner_num;
   S8BIT ret_val;

   FUNCTION_START(STB_DPGetTerrFreqOff);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].freq_offset;

   FUNCTION_FINISH(STB_DPGetTerrFreqOff);

   return(ret_val);
}

/**
 *

 *
 * @brief   Sets terrestrial tuning type into decode path store.
 *
 * @param   path - the ID of the decode path to use
 * @param   type - tune type
 *

 *
 */
void STB_DPSetTerrType(U8BIT path, E_STB_DP_TTYPE type)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetTerrType);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetTerrType(%d): DVB-%s", path, ((type == TERR_TYPE_DVBT) ? "T" :
                                                           ((type == TERR_TYPE_DVBT2) ? "T2" : "unknown"))));

   if (type != tuner_status[tuner_num].terr_type)
   {
      tuner_status[tuner_num].terr_type = type;
      tuner_status[tuner_num].actions |= ACTION_TUNE;
   }

   FUNCTION_FINISH(STB_DPSetTerrType);
}

/**
 *

 *
 * @brief   Reads the terrestrial tuner type from decode path store.
 *
 * @param   path - the ID of the decode path to use
 *
 * @return   terrestrial type
 *
 */
E_STB_DP_TTYPE STB_DPGetTerrType(U8BIT path)
{
   U8BIT tuner_num;
   E_STB_DP_TTYPE ret_val;

   FUNCTION_START(STB_DPGetTerrType);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].terr_type;

   FUNCTION_FINISH(STB_DPGetTerrType);

   return(ret_val);
}

/**
 *

 *
 * @brief   Sets terrestrial T2 PLP id into decode path store.
 *
 * @param   path - the ID of the decode path to use
 * @param   plp_id -
 *

 *
 */
void STB_DPSetTerrPLP(U8BIT path, U8BIT plp_id)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetTerrPLP);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetTerrPLP(%d, %u)", path, plp_id));

   if (plp_id != tuner_status[tuner_num].plp_id)
   {
      tuner_status[tuner_num].plp_id = plp_id;
      tuner_status[tuner_num].actions |= ACTION_TUNE;
   }

   FUNCTION_FINISH(STB_DPSetTerrPLP);
}

/**
 *

 *
 * @brief   Reads the terrestrial T2 PLP id from decode path store.
 *
 * @param   path - the ID of the decode path to use
 *
 * @return   PLP id
 *
 */
U8BIT STB_DPGetTerrPLP(U8BIT path)
{
   U8BIT tuner_num;
   U8BIT ret_val;

   FUNCTION_START(STB_DPGetTerrPLP);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].plp_id;

   STB_TUNE_PRINT(("STB_DPGetTerrPLP(%d): %u", path, tuner_status[tuner_num].plp_id));

   FUNCTION_FINISH(STB_DPGetTerrPLP);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes cable tuning mode value into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   E_STB_DP_CMODE mode - the value of mode
 *

 *
 */
void STB_DPSetCableMode(U8BIT path, E_STB_DP_CMODE mode)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetCableMode);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetCableMode(%d): mode %d", path, mode));
   if (mode != tuner_status[tuner_num].cable_mode)
   {
      tuner_status[tuner_num].cable_mode = mode;
      tuner_status[tuner_num].actions |= ACTION_TUNE;
   }

   FUNCTION_FINISH(STB_DPSetCableMode);
}

/**
 *

 *
 * @brief   Reads the cable tuning mode value from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   E_STB_DP_CMODE - value of mode.
 *
 */
E_STB_DP_CMODE STB_DPGetCableMode(U8BIT path)
{
   U8BIT tuner_num;
   E_STB_DP_CMODE ret_val;

   FUNCTION_START(STB_DPGetCableMode);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].cable_mode;

   FUNCTION_FINISH(STB_DPGetCableMode);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes analogue video type into decode path store.
 *
 * @param   path  - the ID of the decode path to use
 * @param   vtype - the value of video type
 *

 *
 */
void STB_DPSetAnalogVideoType(U8BIT path, E_STB_DP_ANALOG_VIDEO_TYPE vtype)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetAnalogVideoType);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetAnalogVideoType(%d): type %d", path, vtype));
   if (vtype != tuner_status[tuner_num].anlg_vtype)
   {
      tuner_status[tuner_num].anlg_vtype = vtype;
      tuner_status[tuner_num].actions |= ACTION_TUNE;
   }

   FUNCTION_FINISH(STB_DPSetAnalogVideoType);
}

/**
 *

 *
 * @brief   Reads the analogue video type value from decode path store.
 *
 * @param   path - the ID of the decode path to use
 *
 * @return   value of video type.
 *
 */
E_STB_DP_ANALOG_VIDEO_TYPE STB_DPGetAnalogVideoType(U8BIT path)
{
   U8BIT tuner_num;
   E_STB_DP_ANALOG_VIDEO_TYPE ret_val;

   FUNCTION_START(STB_DPGetAnalogVideoType);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].anlg_vtype;

   FUNCTION_FINISH(STB_DPGetAnalogVideoType);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes analogue frequency offset value into decode path store.
 *
 * @param   path   - the ID of the decode path to use
 * @param   offset - the value of offset
 *

 *
 */
void STB_DPSetAnalogFreqOff(U8BIT path, S8BIT offset)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetAnalogFreqOff);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   STB_TUNE_PRINT(("STB_DPSetAnalogFreqOff(%d): offset %d", path, offset));
   if (offset != tuner_status[tuner_num].freq_offset)
   {
      tuner_status[tuner_num].freq_offset = offset;
      tuner_status[tuner_num].actions |= ACTION_TUNE;
   }

   FUNCTION_FINISH(STB_DPSetAnalogFreqOff);
}

/**
 *

 *
 * @brief   Reads the analogue frequency offset value from decode path store.
 *
 * @param   path - the ID of the decode path to use
 *
 * @return   value of offset.
 *
 */
S8BIT STB_DPGetAnalogFreqOff(U8BIT path)
{
   U8BIT tuner_num;
   S8BIT ret_val;

   FUNCTION_START(STB_DPGetAnalogFreqOff);

   ASSERT(path < num_paths);

   tuner_num = path_status[path].tuner_no;

   ASSERT(tuner_num < num_tuners);

   ret_val = tuner_status[tuner_num].freq_offset;

   FUNCTION_FINISH(STB_DPGetAnalogFreqOff);

   return(ret_val);
}

/**
 * @brief   Sets an array of frequencies that can be tried when tuning to a transport if the tuner
 *          fails to lock using the primary frequency
 * @param   path decode path that will be used to tune
 * @param   num_freqs number of additional frequencies in the array
 * @param   frequencies array of additional frequencies; this array will be copied
 */
void STB_DPSetAdditionalFrequencies(U8BIT path, U8BIT num_freqs, U32BIT *frequencies)
{
   U8BIT tuner_num;

   FUNCTION_START(STB_DPSetAdditionalFrequencies);

   if (path < num_paths)
   {
      tuner_num = path_status[path].tuner_no;
      if (tuner_num < num_tuners)
      {
         if (tuner_status[tuner_num].additional_frequencies != NULL)
         {
            STB_FreeMemory(tuner_status[tuner_num].additional_frequencies);

            tuner_status[tuner_num].additional_frequencies = NULL;
            tuner_status[tuner_num].num_additional_frequencies = 0;
         }

         if ((num_freqs != 0) && (frequencies != NULL))
         {
            tuner_status[tuner_num].additional_frequencies = STB_GetMemory(num_freqs * sizeof(U32BIT));
            if (tuner_status[tuner_num].additional_frequencies != NULL)
            {
               memcpy(tuner_status[tuner_num].additional_frequencies, frequencies, num_freqs * sizeof(U32BIT));
               tuner_status[tuner_num].num_additional_frequencies = num_freqs;
            }
         }
      }
   }

   FUNCTION_FINISH(STB_DPSetAdditionalFrequencies);
}

/**
 * @brief   Checks to see if there's an additional frequency available on the given decode path
 *          that hasn't yet been tried and sets it as the next frequency for tuning
 * @param   tuner_num tuner to be checked for an additional frequency
 * @return  TRUE if a new frequency is set for tuning, FALSE otherwise
 */
BOOLEAN STB_DPTryAdditionalFrequency(U8BIT tuner_num)
{
   BOOLEAN retval;

   FUNCTION_START(STB_DPTryAdditionalFrequency);

   retval = FALSE;

   if (tuner_num < num_tuners)
   {
      if ((tuner_status[tuner_num].additional_frequencies != NULL) &&
         (tuner_status[tuner_num].num_additional_frequencies != 0))
      {
         /* Set the next frequency to be tried.
          * We work backwards through the array so the count will be zero when
          * all frequencies have been used */
         tuner_status[tuner_num].num_additional_frequencies--;
         tuner_status[tuner_num].frequency = tuner_status[tuner_num].additional_frequencies[tuner_status[tuner_num].num_additional_frequencies];
         tuner_status[tuner_num].actions |= ACTION_TUNE;
         retval = TRUE;
      }
   }

   FUNCTION_FINISH(STB_DPTryAdditionalFrequency);

   return(retval);
}

/**
 *

 *
 * @brief   Writes PCR PID value into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   U16BIT pid - the value of PCR PID
 *

 *
 */
void STB_DPSetPCRPID(U8BIT path, U16BIT pid)
{
   U8BIT demux_num;

   FUNCTION_START(STB_DPSetPCRPID);

   ASSERT(path < num_paths);

   demux_num = path_status[path].demux_no;

   ASSERT(demux_num < num_demuxes);

   STB_DP_PRINT(("STB_DPSetPCRPID(%d): pid %x", path, pid));

   if (pid != demux_status[demux_num].pcr_pid)
   {
      demux_status[demux_num].pcr_pid = pid;
      demux_status[demux_num].actions |= (ACTION_VIDEO | ACTION_AUDIO);
   }

   FUNCTION_FINISH(STB_DPSetPCRPID);
}

/**
 *

 *
 * @brief   Reads the PCR PID value from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   U16BIT - value of PCR PID.
 *
 */
U16BIT STB_DPGetPCRPID(U8BIT path)
{
   U8BIT demux_num;
   U16BIT ret_val;

   FUNCTION_START(STB_DPGetPCRPID);

   ASSERT(path < num_paths);

   demux_num = path_status[path].demux_no;

   ASSERT(demux_num < num_demuxes);

   ret_val = demux_status[demux_num].pcr_pid;

   FUNCTION_FINISH(STB_DPGetPCRPID);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes video PID value into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   U16BIT pid - the value of video PID
 *

 *
 */
void STB_DPSetVideoPID(U8BIT path, U16BIT pid)
{
   U8BIT demux_num;

   FUNCTION_START(STB_DPSetVideoPID);

   ASSERT(path < num_paths);

   demux_num = path_status[path].demux_no;

   ASSERT(demux_num < num_demuxes);

   STB_DP_PRINT(("STB_DPSetVideoPID(%d): pid %u", path, pid));

   if (pid != demux_status[demux_num].video_pid)
   {
      demux_status[demux_num].video_pid = pid;
      demux_status[demux_num].actions |= ACTION_VIDEO;
   }

   FUNCTION_FINISH(STB_DPSetVideoPID);
}

/**
 *

 *
 * @brief   Reads the video PID value from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   U16BIT - value of video PID.
 *
 */
U16BIT STB_DPGetVideoPID(U8BIT path)
{
   U8BIT demux_num;
   U16BIT ret_val;

   FUNCTION_START(STB_DPGetVideoPID);

   ASSERT(path < num_paths);

   demux_num = path_status[path].demux_no;

   ASSERT(demux_num < num_demuxes);

   ret_val = demux_status[demux_num].video_pid;

   FUNCTION_FINISH(STB_DPGetVideoPID);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes audio PID value into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   U16BIT pid - the value of audio PID
 *

 *
 */
void STB_DPSetAudioPID(U8BIT path, U16BIT pid)
{
   U8BIT demux_num;

   FUNCTION_START(STB_DPSetAudioPID);

   ASSERT(path < num_paths);

   demux_num = path_status[path].demux_no;

   ASSERT(demux_num < num_demuxes);

   STB_DP_PRINT(("STB_DPSetAudioPID(%d): pid %u", path, pid));

   if (pid != demux_status[demux_num].audio_pid)
   {
      demux_status[demux_num].audio_pid = pid;
      demux_status[demux_num].actions |= ACTION_AUDIO;
   }

   FUNCTION_FINISH(STB_DPSetAudioPID);
}

/**
 *

 *
 * @brief   Reads the audio PID value from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   U16BIT - value of audio PID.
 *
 */
U16BIT STB_DPGetAudioPID(U8BIT path)
{
   U8BIT demux_num;
   U16BIT ret_val;

   FUNCTION_START(STB_DPGetAudioPID);

   ASSERT(path < num_paths);

   demux_num = path_status[path].demux_no;

   ASSERT(demux_num < num_demuxes);

   ret_val = demux_status[demux_num].audio_pid;

   FUNCTION_FINISH(STB_DPGetAudioPID);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes AD PID value into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   U16BIT pid - the value of AD PID
 *

 *
 */
void STB_DPSetADPID(U8BIT path, U16BIT pid)
{
   U8BIT demux_num;
   U8BIT decoder_num;

   FUNCTION_START(STB_DPSetADPID);

   ASSERT(path < num_paths);

   demux_num = path_status[path].demux_no;
   decoder_num = path_status[path].ad_decoder_no;

   ASSERT(demux_num < num_demuxes);

   STB_DP_PRINT(("STB_DPSetADPID(%d): pid %u", path, pid));

   if (pid == 0)
   {
      ad_decoder_status[decoder_num].ad_status &= ~AD_PID_PRESENT;
   }
   else
   {
      ad_decoder_status[decoder_num].ad_status |= AD_PID_PRESENT;
   }

   if (pid != demux_status[demux_num].AD_pid)
   {
      demux_status[demux_num].AD_pid = pid;
      demux_status[demux_num].actions |= ACTION_AD;
   }

   FUNCTION_FINISH(STB_DPSetADPID);
}

/**
 *

 *
 * @brief   Reads the audio PID value from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   U16BIT - value of audio PID.
 *
 */
U16BIT STB_DPGetADPID(U8BIT path)
{
   U8BIT demux_num;
   U16BIT ret_val;

   FUNCTION_START(STB_DPGetADPID);

   ASSERT(path < num_paths);

   demux_num = path_status[path].demux_no;

   ASSERT(demux_num < num_demuxes);

   ret_val = demux_status[demux_num].AD_pid;

   FUNCTION_FINISH(STB_DPGetADPID);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes teletext PID value into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   U16BIT pid - the value of teletext PID
 *

 *
 */
void STB_DPSetTextPID(U8BIT path, U16BIT pid)
{
   U8BIT demux_num;

   FUNCTION_START(STB_DPSetTextPID);

   ASSERT(path < num_paths);

   if ((demux_num = path_status[path].demux_no) != INVALID_RES_ID)
   {
      STB_DP_PRINT(("STB_DPSetTextPID(%d): pid %u", path, pid));

      if (pid != demux_status[demux_num].text_pid)
      {
         demux_status[demux_num].text_pid = pid;

         STB_DMXChangeTextPID(demux_num, pid);
         /*      demux_status[demux_num].actions |= (ACTION_VIDEO | ACTION_AUDIO);*/
      }
   }

   FUNCTION_FINISH(STB_DPSetTextPID);
}

/**
 *

 *
 * @brief   Reads the teletext PID value from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   U16BIT - value of teletext PID.
 *
 */
U16BIT STB_DPGetTextPID(U8BIT path)
{
   U8BIT demux_num;
   U16BIT ret_val;

   FUNCTION_START(STB_DPGetTextPID);

   ASSERT(path < num_paths);

   demux_num = path_status[path].demux_no;

   ASSERT(demux_num < num_demuxes);

   ret_val = demux_status[demux_num].text_pid;

   FUNCTION_FINISH(STB_DPGetTextPID);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes data PID value into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   U16BIT pid - the value of data PID
 *

 *
 */
void STB_DPSetDataPID(U8BIT path, U16BIT pid)
{
   U8BIT demux_num;

   FUNCTION_START(STB_DPSetDataPID);

   ASSERT(path < num_paths);

   demux_num = path_status[path].demux_no;

   ASSERT(demux_num < num_demuxes);

   STB_DP_PRINT(("STB_DPSetDataPID(%d): pid %u", path, pid));

   if (pid != demux_status[demux_num].data_pid)
   {
      demux_status[demux_num].data_pid = pid;
      demux_status[demux_num].actions |= (ACTION_VIDEO | ACTION_AUDIO);
   }

   FUNCTION_FINISH(STB_DPSetDataPID);
}

/**
 *

 *
 * @brief   Reads the data PID value from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   U16BIT - value of data PID.
 *
 */
U16BIT STB_DPGetDataPID(U8BIT path)
{
   U8BIT demux_num;
   U16BIT ret_val;

   FUNCTION_START(STB_DPGetDataPID);

   ASSERT(path < num_paths);

   demux_num = path_status[path].demux_no;

   ASSERT(demux_num < num_demuxes);

   ret_val = demux_status[demux_num].data_pid;

   FUNCTION_FINISH(STB_DPGetDataPID);

   return(ret_val);
}

/*!**************************************************************************
 * @brief
 * @param
 * @return
 ****************************************************************************/
void STB_DPSetDecodePIDs(U8BIT path, U16BIT pcr_pid, U16BIT video_pid, U16BIT audio_pid,
   U16BIT text_pid, U16BIT data_pid, U16BIT AD_pid)
{
   U8BIT demux_num;

   FUNCTION_START(STB_DPSetDecodePIDs);

   ASSERT(path < num_paths);

   demux_num = path_status[path].demux_no;

   ASSERT(demux_num < num_demuxes);

   STB_DP_PRINT(("STB_DPSetDecodePIDs(%u): PIDs = %u/%u/%u/%u/%u/%u", path, pcr_pid, video_pid,
                 audio_pid, text_pid, data_pid, AD_pid));

   demux_status[demux_num].pcr_pid = pcr_pid;
   demux_status[demux_num].video_pid = video_pid;
   demux_status[demux_num].audio_pid = audio_pid;
   demux_status[demux_num].AD_pid = AD_pid;
   demux_status[demux_num].text_pid = text_pid;
   demux_status[demux_num].data_pid = data_pid;

   if (TRUE == STB_DPIsMonitoringPath(path)) {
      // set preloading bit
      demux_num |= (1 << 7);
   }

   STB_DMXChangeEcmPIDs(
      demux_num,
      demux_status[demux_num].ecm_pid,
      demux_status[demux_num].video_ecm_pid,
      demux_status[demux_num].audio_ecm_pid,
      demux_status[demux_num].text_ecm_pid,
      demux_status[demux_num].data_ecm_pid,
      demux_status[demux_num].AD_ecm_pid
   );
   STB_DMXChangeDecodePIDs(demux_num, pcr_pid, video_pid, audio_pid, text_pid, data_pid, AD_pid);

   FUNCTION_FINISH(STB_DPSetDecodePIDs);
}

/*!**************************************************************************
 * @brief
 * @param
 * @return
 ****************************************************************************/
void STB_DPSetEcmPIDs(U8BIT path, U16BIT ecm_pid, U16BIT video_ecm_pid, U16BIT audio_ecm_pid,
   U16BIT text_ecm_pid, U16BIT data_ecm_pid, U16BIT AD_ecm_pid)
{
   U8BIT demux_num;

   FUNCTION_START(STB_DPSetEcmPIDs);

   ASSERT(path < num_paths);

   demux_num = path_status[path].demux_no;

   ASSERT(demux_num < num_demuxes);

   STB_DP_PRINT(("STB_DPSetEcmPIDs(%u): PIDs = %u,%u/%u/%u/%u/%u", path, ecm_pid, video_ecm_pid,
                 audio_ecm_pid, text_ecm_pid, data_ecm_pid, AD_ecm_pid));

   demux_status[demux_num].ecm_pid = video_ecm_pid;
   demux_status[demux_num].video_ecm_pid = video_ecm_pid;
   demux_status[demux_num].audio_ecm_pid = audio_ecm_pid;
   demux_status[demux_num].AD_ecm_pid = AD_ecm_pid;
   demux_status[demux_num].text_ecm_pid = text_ecm_pid;
   demux_status[demux_num].data_ecm_pid = data_ecm_pid;

   if (TRUE == STB_DPIsMonitoringPath(path)) {
      // set preloading bit
      demux_num |= (1 << 7);
   }
   STB_DMXChangeEcmPIDs(demux_num, ecm_pid, video_ecm_pid, audio_ecm_pid, text_ecm_pid, data_ecm_pid, AD_ecm_pid);

   FUNCTION_FINISH(STB_DPSetEcmPIDs);
}

/**
 *

 *
 * @brief   Writes AD mode value into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   E_STB_DP_AUDIO_MODE mode - audio mode
 *

 *
 */
void STB_DPSetADMode(U8BIT path, E_STB_DP_AUDIO_MODE mode)
{
   U8BIT decoder_num;

   FUNCTION_START(STB_DPSetADMode);

   ASSERT(path < num_paths);

   decoder_num = path_status[path].ad_decoder_no;

   ASSERT(decoder_num < num_audio_decoders);

   STB_DP_PRINT(("STB_DPSetADMode(%d): mode %d", path, mode));

   if (mode != ad_decoder_status[decoder_num].audio_mode)
   {
      ad_decoder_status[decoder_num].audio_mode = mode;
   }

   FUNCTION_FINISH(STB_DPSetADMode);
}

/**
 *

 *
 * @brief   Reads the AD mode value from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   E_STB_DP_AUDIO_MODE - audio mode.
 *
 */
E_STB_DP_AUDIO_MODE STB_DPGetADMode(U8BIT path)
{
   U8BIT decoder_num;
   E_STB_DP_AUDIO_MODE ret_val;

   FUNCTION_START(STB_DPGetADMode);

   ASSERT(path < num_paths);

   decoder_num = path_status[path].ad_decoder_no;

   ASSERT(decoder_num < num_audio_decoders);

   ret_val = ad_decoder_status[decoder_num].audio_mode;

   FUNCTION_FINISH(STB_DPGetADMode);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes audio mode value into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   E_STB_DP_AUDIO_MODE mode - audio mode
 *

 *
 */
void STB_DPSetAudioMode(U8BIT path, E_STB_DP_AUDIO_MODE mode)
{
   U8BIT decoder_num;

   FUNCTION_START(STB_DPSetAudioMode);

   ASSERT(path < num_paths);

   decoder_num = path_status[path].audio_decoder_no;

   ASSERT(decoder_num < num_audio_decoders);

   STB_DP_PRINT(("STB_DPSetAudioMode(%d): mode %d", path, mode));

   if (mode != audio_decoder_status[decoder_num].audio_mode)
   {
      audio_decoder_status[decoder_num].audio_mode = mode;
   }

   FUNCTION_FINISH(STB_DPSetAudioMode);
}

/**
 *

 *
 * @brief   Reads the audio mode value from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   E_STB_DP_AUDIO_MODE - audio mode.
 *
 */
E_STB_DP_AUDIO_MODE STB_DPGetAudioMode(U8BIT path)
{
   U8BIT decoder_num;
   E_STB_DP_AUDIO_MODE ret_val;

   FUNCTION_START(STB_DPGetAudioMode);

   ASSERT(path < num_paths);

   decoder_num = path_status[path].audio_decoder_no;

   ASSERT(decoder_num < num_audio_decoders);

   ret_val = audio_decoder_status[decoder_num].audio_mode;

   FUNCTION_FINISH(STB_DPGetAudioMode);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes lock mode value into decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   BOOLEAN mode - lock mode
 *

 *
 */
void STB_DPSetLockMode(U8BIT path, BOOLEAN mode)
{
   FUNCTION_START(STB_DPSetLockMode);

   ASSERT(path < num_paths);

   STB_DP_PRINT(("STB_DPSetLockMode(%d): mode %d", path, mode));

   if (mode != path_status[path].lock_mode)
   {
      path_status[path].lock_mode = mode;
   }

   FUNCTION_FINISH(STB_DPSetLockMode);
}

/**
 *

 *
 * @brief   Reads the lock mode value from decode path store.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   BOOLEAN - lock mode.
 *
 */
BOOLEAN STB_DPGetLockMode(U8BIT path)
{
   BOOLEAN ret_val;

   FUNCTION_START(STB_DPGetLockMode);

   ASSERT(path < num_paths);

   ret_val = path_status[path].lock_mode;

   FUNCTION_FINISH(STB_DPGetLockMode);

   return(ret_val);
}

/**
 *

 *
 * @brief   Sets the type of CODEC to be used for Video decoding
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   E_STB_DP_VIDEO_CODEC - the codec to be used
 *
 * @return   BOOLEAN - success
 *
 */
BOOLEAN STB_DPSetVideoCodec(U8BIT path, E_STB_DP_VIDEO_CODEC codec)
{
   U8BIT demux_num;
   U8BIT decoder_num;
   E_STB_AV_VIDEO_CODEC av_codec;
   BOOLEAN ret_val;

   FUNCTION_START(STB_DPSetVideoCodec);

   ASSERT(path < num_paths);

   demux_num = path_status[path].demux_no;
   decoder_num = path_status[path].video_decoder_no;

   ASSERT(demux_num < num_demuxes);

   demux_status[demux_num].video_codec = codec;
   ret_val = TRUE;

   if (decoder_num < num_video_decoders)
   {
      switch (codec)
      {
         case VIDEO_CODEC_MPEG1:
            av_codec = AV_VIDEO_CODEC_MPEG1;
            break;

         case VIDEO_CODEC_MPEG2:
            av_codec = AV_VIDEO_CODEC_MPEG2;
            break;

         case VIDEO_CODEC_H264:
            av_codec = AV_VIDEO_CODEC_H264;
            break;

         case VIDEO_CODEC_H265:
            av_codec = AV_VIDEO_CODEC_H265;
            break;

         default:
            av_codec = AV_VIDEO_CODEC_AUTO;
            break;
      }

      ret_val = STB_AVSetVideoCodec(decoder_num, av_codec);
      if (ret_val)
      {
         video_decoder_status[decoder_num].codec = codec;
      }
   }

   FUNCTION_FINISH(STB_DPSetVideoCodec);

   return ret_val;
}

/**
 *

 *
 * @brief   Returns the type of CODEC to be used for Video decoding
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   E_STB_DP_VIDEO_CODEC - the codec to be used
 *
 */
E_STB_DP_VIDEO_CODEC STB_DPGetVideoCodec(U8BIT path)
{
   U8BIT demux_num;
   U8BIT decoder_num;
   E_STB_DP_VIDEO_CODEC ret_val;

   FUNCTION_START(STB_DPGetVideoCodec);

   ASSERT(path < num_paths);

   decoder_num = path_status[path].video_decoder_no;

   if (decoder_num < num_video_decoders)
   {
      ret_val = video_decoder_status[decoder_num].codec;
   }
   else
   {
      demux_num = path_status[path].demux_no;
      ASSERT(demux_num < num_demuxes);
      ret_val = demux_status[demux_num].video_codec;
   }

   FUNCTION_FINISH(STB_DPGetVideoCodec);

   return(ret_val);
}

/**
 *

 *
 * @brief   Sets the type of CODEC to be used for AD decoding
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   E_STB_DP_AUDIO_CODEC - the codec to be used
 *
 * @return   BOOLEAN - success
 *
 */
BOOLEAN STB_DPSetADCodec(U8BIT path, E_STB_DP_AUDIO_CODEC codec)
{
   U8BIT demux_num;
   U8BIT decoder_num;
   E_STB_AV_AUDIO_CODEC av_codec;
   BOOLEAN ret_val;

   FUNCTION_START(STB_DPSetADCodec);

   ASSERT(path < num_paths);

   demux_num = path_status[path].demux_no;
   decoder_num = path_status[path].ad_decoder_no;

   ASSERT(demux_num < num_demuxes);

   demux_status[demux_num].ad_codec = codec;
   ret_val = TRUE;

   if (decoder_num < num_audio_decoders)
   {
      switch (codec)
      {
         case AUDIO_CODEC_MP2:
            av_codec = AV_AUDIO_CODEC_MP2;
            break;

         case AUDIO_CODEC_AC3:
            av_codec = AV_AUDIO_CODEC_AC3;
            break;

         case AUDIO_CODEC_EAC3:
            av_codec = AV_AUDIO_CODEC_EAC3;
            break;

         case AUDIO_CODEC_AAC:
            av_codec = AV_AUDIO_CODEC_AAC;
            break;

         case AUDIO_CODEC_HEAAC:
            av_codec = AV_AUDIO_CODEC_HEAAC;
            break;

         default:
            av_codec = AV_AUDIO_CODEC_MP2;
            break;
      }

      ret_val = STB_AVSetADCodec(decoder_num, av_codec);
      if (ret_val)
      {
         ad_decoder_status[decoder_num].codec = codec;
      }
   }

   FUNCTION_FINISH(STB_DPSetADCodec);

   return ret_val;
}

/**
 *

 *
 * @brief   Returns the type of CODEC to be used for AD decoding
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   E_STB_DP_AUDIO_CODEC - the codec to be used
 *
 */
E_STB_DP_AUDIO_CODEC STB_DPGetADCodec(U8BIT path)
{
   U8BIT demux_num;
   E_STB_DP_AUDIO_CODEC ret_val;

   FUNCTION_START(STB_DPGetADCodec);

   ASSERT(path < num_paths);

   demux_num = path_status[path].demux_no;
   ASSERT(demux_num < num_demuxes);
   ret_val = demux_status[demux_num].ad_codec;

   FUNCTION_FINISH(STB_DPGetADCodec);

   return(ret_val);
}

/**
 *

 *
 * @brief   Sets the type of CODEC to be used for Audio decoding
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   E_STB_DP_AUDIO_CODEC - the codec to be used
 *
 * @return   BOOLEAN - success
 *
 */
BOOLEAN STB_DPSetAudioCodec(U8BIT path, E_STB_DP_AUDIO_CODEC codec)
{
   U8BIT demux_num;
   U8BIT decoder_num;
   E_STB_AV_AUDIO_CODEC av_codec;
   BOOLEAN ret_val;

   FUNCTION_START(STB_DPSetAudioCodec);

   ASSERT(path < num_paths);

   demux_num = path_status[path].demux_no;
   decoder_num = path_status[path].audio_decoder_no;

   ASSERT(demux_num < num_demuxes);

   demux_status[demux_num].audio_codec = codec;
   ret_val = TRUE;

   if (decoder_num < num_video_decoders)
   {
      switch (codec)
      {
         case AUDIO_CODEC_MP2:
            av_codec = AV_AUDIO_CODEC_MP2;
            break;

         case AUDIO_CODEC_MP3:
            av_codec = AV_AUDIO_CODEC_MP3;
            break;

         case AUDIO_CODEC_AC3:
            av_codec = AV_AUDIO_CODEC_AC3;
            break;

         case AUDIO_CODEC_EAC3:
            av_codec = AV_AUDIO_CODEC_EAC3;
            break;

         case AUDIO_CODEC_AAC:
            av_codec = AV_AUDIO_CODEC_AAC;
            break;

         case AUDIO_CODEC_HEAAC:
            av_codec = AV_AUDIO_CODEC_HEAAC;
            break;

         case AUDIO_CODEC_AAC_ADTS:
            av_codec = AV_AUDIO_CODEC_AAC_ADTS;
            break;

         default:
            av_codec = AV_AUDIO_CODEC_AUTO;
            break;
      }

      ret_val = STB_AVSetAudioCodec(decoder_num, av_codec);
      if (ret_val)
      {
         audio_decoder_status[decoder_num].codec = codec;
      }
   }

   FUNCTION_FINISH(STB_DPSetAudioCodec);

   return ret_val;
}

/**
 *

 *
 * @brief   Returns the type of CODEC to be used for Audio decoding
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   E_STB_DP_AUDIO_CODEC - the codec to be used
 *
 */
E_STB_DP_AUDIO_CODEC STB_DPGetAudioCodec(U8BIT path)
{
   U8BIT demux_num;
   U8BIT decoder_num;
   E_STB_DP_AUDIO_CODEC ret_val;

   FUNCTION_START(STB_DPGetAudioCodec);

   ASSERT(path < num_paths);

   decoder_num = path_status[path].audio_decoder_no;

   if (decoder_num < num_audio_decoders)
   {
      ret_val = audio_decoder_status[decoder_num].codec;
   }
   else
   {
      demux_num = path_status[path].demux_no;
      ASSERT(demux_num < num_demuxes);
      ret_val = demux_status[demux_num].audio_codec;
   }

   FUNCTION_FINISH(STB_DPGetAudioCodec);

   return(ret_val);
}

//---local function definitions------------------------------------------------

/*!**************************************************************************
 * @brief   Acquires all the resources needed for live viewing except the tuner
 * @param   path - The ID of the decode path being acquired for live viewing
 * @param   tuner_num - ID of an already acquired tuner that will be assigned to the path
 * @param   with_decoders - TRUE if decoding resources are to be acquired
 * @return  TRUE if the resources are acquired successfully
 ****************************************************************************/
static BOOLEAN AcquireResources(U8BIT path, U16BIT demux_caps, BOOLEAN with_decoders)
{
   U32BIT param;
   BOOLEAN result = FALSE;

   FUNCTION_START(AcquireResources);

   if (path != INVALID_RES_ID)
   {
      /* Acquire any resources that aren't currently available to this path */
      path_status[path].demux_no = STB_RESAcquireDemux(INVALID_RES_ID, demux_caps);
      if (path_status[path].demux_no != INVALID_RES_ID)
      {
         if (with_decoders)
         {
            path_status[path].video_decoder_no = STB_RESAcquireVideoDecoder();
            if (path_status[path].video_decoder_no != INVALID_RES_ID)
            {
               path_status[path].audio_decoder_no = STB_RESAcquireAudioDecoder();
               if (path_status[path].audio_decoder_no != INVALID_RES_ID)
               {
                  /* Acquired all necessary resources */
                  result = TRUE;

                  /* Set the demux to get its data from the tuner */
                  STB_DMXSetDemuxSource(path_status[path].demux_no, DMX_TUNER, path_status[path].tuner_no);

                  /* Set the source of the A/V decoders */
                  param = ((U32BIT)path_status[path].tuner_no << 8) | path_status[path].demux_no;
                  STB_AVSetAudioSource(path_status[path].audio_decoder_no, AV_DEMUX, param);
                  STB_AVSetVideoSource(path_status[path].video_decoder_no, AV_DEMUX, param);

                  /* Failing to acquire an AD decoder isn't seen as a failure */
                  path_status[path].ad_decoder_no = STB_RESAcquireADDecoder();
                  if (path_status[path].ad_decoder_no != INVALID_RES_ID)
                  {
                     STB_AVSetADSource(path_status[path].ad_decoder_no, AV_DEMUX, param);
                  }

                  /* Try to acquire a secondary video decoder */
                  path_status[path].secondary_video_decoder_no = STB_RESAcquireVideoDecoder();
                  if (path_status[path].secondary_video_decoder_no != INVALID_RES_ID)
                  {
                     STB_AVSetVideoSource(path_status[path].secondary_video_decoder_no, AV_MEMORY, 0);
                  }
               }
               else
               {
                  /* Not all resources are available so release acquired resources */
                  STB_RESReleaseVideoDecoder(path_status[path].video_decoder_no);
                  path_status[path].video_decoder_no = INVALID_RES_ID;

                  /* Set the demux to release get its data from a tuner */
                  STB_DMXSetDemuxSource(path_status[path].demux_no, DMX_INVALID, INVALID_RES_ID);
                  STB_RESReleaseDemux(path_status[path].demux_no);
                  path_status[path].demux_no = INVALID_RES_ID;
               }
            }
            else
            {
               /* Set the demux to release get its data from a tuner */
               STB_DMXSetDemuxSource(path_status[path].demux_no, DMX_INVALID, INVALID_RES_ID);
               /* Not all resources are available so release acquired resources */
               STB_RESReleaseDemux(path_status[path].demux_no);
               path_status[path].demux_no = INVALID_RES_ID;
            }
         }
         else
         {
            /* No decoding required */
            path_status[path].video_decoder_no = INVALID_RES_ID;
            path_status[path].secondary_video_decoder_no = INVALID_RES_ID;
            path_status[path].audio_decoder_no = INVALID_RES_ID;
            path_status[path].ad_decoder_no = INVALID_RES_ID;

            /* Acquired all necessary resources */
            result = TRUE;

            /* Set the demux to get its data from a tuner */
            if (DMX_CAPS_MONITOR_SI == demux_caps) {
               if (INVALID_RES_ID != STB_DPGetLivePath())
                  STB_DMXSetDemuxSource((path_status[path].demux_no | (1 << 7)), DMX_TUNER, path_status[path].tuner_no);
               else
                  STB_DMXSetDemuxSource(path_status[path].demux_no, DMX_TUNER, path_status[path].tuner_no);
            }    
            else
               STB_DMXSetDemuxSource(path_status[path].demux_no, DMX_TUNER, path_status[path].tuner_no);
         }
      }
   }

   FUNCTION_FINISH(AcquireResources);

   return(result);
}

/*!**************************************************************************
 * @brief   Sends a DiSEqC message, ensuring that at least 25ms has lapsed since
 *          the previous message was sent.
 * @param   tuner - tuner path
 * @param   msg_data - DiSEqC message data to be sent
 * @param   msg_size - number of bytes in the message
 ****************************************************************************/
static void SendDISEQCMessage(U8BIT tuner, U8BIT *msg_data, U8BIT msg_size)
{
   U32BIT delay_time;
   U32BIT repeat_time;
   U8BIT i;
   U8BIT dmsg_data[3];

   FUNCTION_START(SendDISEQCMessage);

   if (!lnb_awake)
   {
      /* This is the first DiSEqC message being sent so make sure everything
       * is ready to receive messages by sending the Awake message first */
      repeat_time = 0;

      for (i = 0; i < (tuner_status[tuner].diseqc_repeats + 1); i++)
      {
         if (i == 0)
         {
            dmsg_data[0] = 0xe0;
         }
         else
         {
            dmsg_data[0] = 0xe1;
         }

         dmsg_data[1] = 0x10;
         dmsg_data[2] = 0x31;

         /* Make sure we wait at least 100ms before sending any repeated commands */
         if (i != 0)
         {
            if ((delay_time = STB_OSGetClockDiff(repeat_time)) < DISEQC_REPEAT_DELAY)
            {
               STB_OSTaskDelay(DISEQC_REPEAT_DELAY - delay_time);
            }
         }

         STB_TuneSendDISEQCMessage(tuner, dmsg_data, 3);

         tuner_status[tuner].diseqc_msg_time = STB_OSGetClockMilliseconds();
         repeat_time = STB_OSGetClockMilliseconds();
      }

      lnb_awake = TRUE;
   }

   /* The time of the last diseqc message will be 0 if nothing has been sent yet */
   if (tuner_status[tuner].diseqc_msg_time != 0)
   {
      /* There must be at least 25ms from the end of the last DiSEqC message to this one being sent */
      if ((delay_time = STB_OSGetClockDiff(tuner_status[tuner].diseqc_msg_time)) < DISEQC_MESSAGE_DELAY)
      {
         STB_OSTaskDelay(DISEQC_MESSAGE_DELAY - delay_time);
      }
   }

   STB_TuneSendDISEQCMessage(tuner, msg_data, msg_size);

   tuner_status[tuner].diseqc_msg_time = STB_OSGetClockMilliseconds();

   FUNCTION_FINISH(SendDISEQCMessage);
}

/**
 *

 *
 * @brief   Does LNB/dish setup and sends any relevant DiSEqC messages then starts tuner
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   TRUE if ok, FALSE if some parameters invalid so cannot tune
 *
 */
static BOOLEAN StartTune(U8BIT path)
{
   U8BIT tuner;

   E_STB_TUNE_SYSTEM_TYPE system_type;
   // SAT
   E_STB_TUNE_MODULATION tnr_mod;
   E_STB_TUNE_FEC tnr_fec;
   E_STB_TUNE_LNB_VOLTAGE lnb_volts;
   S_STB_DP_LNB_BAND *band;

   // Terrestrial
   E_STB_TUNE_TMODE tnr_tmode;
   E_STB_TUNE_TBWIDTH tnr_tbwidth;
   E_STB_TUNE_ANALOG_VIDEO_TYPE tnr_avtype;

   E_STB_TUNE_CMODE cmode;

   U8BIT dmsg_data[6];
   U8BIT i;
   S32BIT tnr_freq;
   S32BIT tnr_min_freq;
   S32BIT tnr_max_freq;
   S32BIT sig_freq;
   S32BIT lo_freq;
   U8BIT data_1;
   U8BIT data_2;
   U16BIT t_word;
   U8BIT unicablecmd[UNICABLE_COMMAND_SIZE];
   BOOLEAN high_band;
   BOOLEAN tone_22khz_on;
   U32BIT repeat_time;
   U32BIT delay_time;
   BOOLEAN retval = FALSE;

   FUNCTION_START(StartTune);

   tuner = path_status[path].tuner_no;

   switch (tuner_status[tuner].signal_type)
   {
      case SIGNAL_QPSK:
      {
         sig_freq = tuner_status[tuner].frequency * 1000;

         // work out lnb band to be used (and set 22kHz tone at the same time)
         tnr_min_freq = (S32BIT)STB_TuneGetMinTunerFreqKHz(tuner);
         tnr_max_freq = (S32BIT)STB_TuneGetMaxTunerFreqKHz(tuner);

         band = FindLNBBand(tuner);
         if (band != NULL)
         {
            lo_freq = band->local_oscillator_frequency * 1000; // convert to kHz
         }
         else
         {
            lo_freq = 0;
         }

         DBGPRINT("QPSK f(%d,%d,%d) sr=%d", tnr_min_freq, sig_freq, tnr_max_freq, tuner_status[tuner].sym_rate);

         tone_22khz_on = FALSE;
         high_band = FALSE;

         // now calculate tuner frequency from required signal frequency and lnb lo freq
         switch (tuner_status[tuner].lnb_type)
         {
            case LNB_TYPE_SINGLE:
            {
               high_band = FALSE;
               tone_22khz_on = tuner_status[tuner].lnb_22k;

               /* Calculate the IF frequency to tune to.
                * Tuning to C band satellites can result in negative frequencies,
                * so check for this and use the absolute value */
               if ((tnr_freq = sig_freq - lo_freq) < 0)
               {
                  tnr_freq = -tnr_freq;
               }
               break;
            }

            case LNB_TYPE_UNICABLE:
            {
               /* Only two bands for this type, it's high band if the mininum frequency is
                  not 0 */
               if ((band != NULL) && (band->min_freq > 0))
               {
                  high_band = TRUE;
               }
               else
               {
                  high_band = FALSE;
               }

               // Fixed IF frequency
               // Actual sig_freq will be tuned by LNB via DiSEqC command
               tnr_freq = tuner_status[tuner].unicable_if;
               break;
            }

            case LNB_TYPE_USER_DEFINED:
            {
               if (band != NULL)
               {
                  tone_22khz_on = band->tone_22k;
                  high_band = FALSE;
               }

               // Calculate the IF frequency to tune to
               tnr_freq = sig_freq - lo_freq;
               break;
            }

            case LNB_TYPE_UNIVERSAL:
            default:
            {
               /* Only two bands for this type, it's high band if the mininum frequency is
                  not 0 */
               if (band != NULL)
               {
                  tone_22khz_on = band->tone_22k;
                  if (band->min_freq > 0)
                  {
                     high_band = TRUE;
                  }
                  else
                  {
                     high_band = FALSE;
                  }
               }

               // Calculate the IF frequency to tune to
               tnr_freq = sig_freq - lo_freq;
               break;
            }
         }

         if ((tnr_min_freq <= tnr_freq) && (tnr_freq <= tnr_max_freq) &&
             (tuner_status[tuner].sym_rate >= TUNER_MIN_SRATE) &&
             (tuner_status[tuner].sym_rate <= TUNER_MAX_SRATE))
         {
            // work out required lnb voltage
            switch (tuner_status[tuner].lnb_power)
            {
               case LNB_POWER_ON:
               case LNB_POWER_AUTO:
               {
                  if (band != NULL)
                  {
                     lnb_volts = band->lnb_voltage;
                  }
                  else
                  {
                     lnb_volts = LNB_VOLTAGE_OFF;
                  }
                  break;
               }

               case LNB_POWER_OFF:
               default:
               {
                  lnb_volts = LNB_VOLTAGE_OFF;
                  break;
               }
            }

            // Now ready to start controlling front end...
            // first set lnb voltage to required level and set 12v switch state
            STB_TuneSetLNBVoltage(tuner, lnb_volts);
            STB_TuneSet12VSwitch(tuner, tuner_status[tuner].lnb_12v);
            STB_TuneSetLOFrequency(tuner, lo_freq);

            if ((tuner_status[tuner].diseqc_position == TRUE) &&
                (lnb_currently_powered == FALSE) &&
                (lnb_volts != LNB_VOLTAGE_OFF))
            {
               // On applying power to LNB, if there is a DiSEqC 1.2 motor fitted then wait
               // after applying power to the LNB. This allows enough time for the DiSEqC
               // IC to power up to receive the message.
               STB_OSTaskDelay(1000);

               lnb_currently_powered = TRUE;
            }

            // if lnb voltage is not off, send any diseqc messages
            if (lnb_volts != LNB_VOLTAGE_OFF)
            {
               repeat_time = 0;

               for (i = 0; i < (tuner_status[tuner].diseqc_repeats + 1); i++)
               {
                  // send DiSEqC committed switch messages
                  if (i == 0)
                  {
                     dmsg_data[0] = 0xe0;
                  }
                  else
                  {
                     dmsg_data[0] = 0xe1;
                  }

                  dmsg_data[1] = 0x10;
                  dmsg_data[2] = 0x38;
                  dmsg_data[3] = 0xf0;

                  if (tuner_status[tuner].polarity == POLARITY_HORIZONTAL)
                  {
                     dmsg_data[3] |= 0x02;
                  }
                  if (high_band == TRUE)
                  {
                     dmsg_data[3] |= 0x01;
                  }

                  if (i != 0)
                  {
                     /* Make sure we wait at least 100ms before sending any repeated commands */
                     if ((delay_time = STB_OSGetClockDiff(repeat_time)) < DISEQC_REPEAT_DELAY)
                     {
                        STB_OSTaskDelay(DISEQC_REPEAT_DELAY - delay_time);
                     }
                  }

                  switch (tuner_status[tuner].diseqc_cswitch)
                  {
                     case DISEQC_CSWITCH_A:
                     {
                        dmsg_data[3] |= 0x00;
                        SendDISEQCMessage(tuner, dmsg_data, 4);
                        repeat_time = STB_OSGetClockMilliseconds();
                        break;
                     }
                     case DISEQC_CSWITCH_B:
                     {
                        dmsg_data[3] |= 0x04;
                        SendDISEQCMessage(tuner, dmsg_data, 4);
                        repeat_time = STB_OSGetClockMilliseconds();
                        break;
                     }
                     case DISEQC_CSWITCH_C:
                     {
                        dmsg_data[3] |= 0x08;
                        SendDISEQCMessage(tuner, dmsg_data, 4);
                        repeat_time = STB_OSGetClockMilliseconds();
                        break;
                     }
                     case DISEQC_CSWITCH_D:
                     {
                        dmsg_data[3] |= 0x0c;
                        SendDISEQCMessage(tuner, dmsg_data, 4);
                        repeat_time = STB_OSGetClockMilliseconds();
                        break;
                     }
                     default:
                     {
                        break;
                     }
                  }

                  // send DiSEqC uncommitted switch messages
                  if (tuner_status[tuner].diseqc_uswitch != 0)
                  {
                     dmsg_data[1] = 0x10;
                     dmsg_data[2] = 0x39;
                     dmsg_data[3] = 0xf0;
                     dmsg_data[3] |= (tuner_status[tuner].diseqc_uswitch - 1);
                     SendDISEQCMessage(tuner, dmsg_data, 4);
                  }
               }

               // check if SMATV is enabled
               if (tuner_status[tuner].diseqc_smatv == TRUE)
               {
                  // send DiSEqC committed switch message if not already sent
                  if (tuner_status[tuner].diseqc_cswitch == DISEQC_CSWITCH_OFF)
                  {
                     dmsg_data[0] = 0xe0;
                     dmsg_data[1] = 0x10;
                     dmsg_data[2] = 0x38;
                     dmsg_data[3] = 0xf0;
                     if (tuner_status[tuner].polarity == POLARITY_HORIZONTAL)
                     {
                        dmsg_data[3] |= 0x02;
                     }
                     if (high_band == TRUE)
                     {
                        dmsg_data[3] |= 0x01;
                     }
                     SendDISEQCMessage(tuner, dmsg_data, 4);
                  }

                  // send DiSEqC SMATV messages
                  sig_freq = tuner_status[tuner].frequency;
                  dmsg_data[0] = 0xe0;
                  dmsg_data[1] = 0x10;
                  dmsg_data[2] = 0x58;
                  dmsg_data[3] = (U8BIT)((sig_freq / 10000) << 4);
                  sig_freq = (sig_freq % 10000);
                  dmsg_data[3] |= (U8BIT)(sig_freq / 1000);
                  sig_freq = (sig_freq % 1000);
                  dmsg_data[4] = (U8BIT)((sig_freq / 100) << 4);
                  sig_freq = (sig_freq % 100);
                  dmsg_data[4] |= (U8BIT)(sig_freq / 10);
                  sig_freq = (sig_freq % 10);
                  dmsg_data[5] = (U8BIT)((sig_freq / 1) << 4);
                  SendDISEQCMessage(tuner, dmsg_data, 6);
               }

               if (tuner_status[tuner].diseqc_tone != DISEQC_TONE_OFF)
               {
                  // send DiSEqC tone burst
                  switch (tuner_status[tuner].diseqc_tone)
                  {
                     case DISEQC_TONE_A:
                        dmsg_data[0] = 0x00;
                        SendDISEQCMessage(tuner, dmsg_data, 1);
                        break;

                     case DISEQC_TONE_B:
                        dmsg_data[0] = 0xff;
                        SendDISEQCMessage(tuner, dmsg_data, 1);
                        break;

                     default:
                        break;
                  }
               }
            }

            // check if skew position has changed
            if ((tuner_status[tuner].actions & ACTION_SKEW) != 0)
            {
               tuner_status[tuner].actions &= ~ACTION_SKEW;
               STB_TuneChangeSkewPosition(tuner, tuner_status[tuner].skew_pos);
            }

            // check if dish position has changed
            if ((tuner_status[tuner].actions & ACTION_DISH) != 0)
            {
               tuner_status[tuner].actions &= ~ACTION_DISH;
               if (tuner_status[tuner].pulse_position == TRUE)
               {
                  STB_TuneChangePulsePosition(tuner, tuner_status[tuner].dish_pos);
               }
               if ((tuner_status[tuner].diseqc_position == TRUE) && (lnb_volts != LNB_VOLTAGE_OFF))
               {
                  dmsg_data[0] = 0xe0;
                  dmsg_data[1] = 0x31;
                  dmsg_data[2] = 0x6b;
                  dmsg_data[3] = (U8BIT)(tuner_status[tuner].dish_pos & 0x00ff);

                  // If zero has been placed in the decode path, it should not be sent as not a
                  // valid position; but is a reference position and has a function of its own to
                  // centre dish.
                  if (dmsg_data[3] != 0x00)
                  {
                     SendDISEQCMessage(tuner, dmsg_data, 4);
                  }
               }
            }

            /* Set modulation */
            switch (tuner_status[tuner].modulation)
            {
               case MOD_QPSK:
                  tnr_mod = TUNE_MOD_QPSK;
                  break;

               case MOD_8PSK:
                  tnr_mod = TUNE_MOD_8PSK;
                  break;

               case MOD_16QAM:
                  tnr_mod = TUNE_MOD_16QAM;
                  break;

               case MOD_AUTO:
               default:
                  tnr_mod = TUNE_MOD_AUTO;
                  break;
            }

            STB_TuneSetModulation(tuner, tnr_mod);

            // set FEC param
            switch (tuner_status[tuner].fec)
            {
               case FEC_AUTOMATIC: {tnr_fec = TUNE_FEC_AUTOMATIC;  break; }
               case FEC_1_2: {tnr_fec = TUNE_FEC_1_2;  break; }
               case FEC_2_3: {tnr_fec = TUNE_FEC_2_3;  break; }
               case FEC_3_4: {tnr_fec = TUNE_FEC_3_4;  break; }
               case FEC_5_6: {tnr_fec = TUNE_FEC_5_6;  break; }
               case FEC_7_8: {tnr_fec = TUNE_FEC_7_8;  break; }
               case FEC_1_4: {tnr_fec = TUNE_FEC_1_4;  break; }
               case FEC_1_3: {tnr_fec = TUNE_FEC_1_3;  break; }
               case FEC_2_5: {tnr_fec = TUNE_FEC_2_5;  break; }
               case FEC_8_9: {tnr_fec = TUNE_FEC_8_9;  break; }
               case FEC_9_10: {tnr_fec = TUNE_FEC_9_10;  break; }
               case FEC_3_5: {tnr_fec = TUNE_FEC_3_5;  break; }
               case FEC_4_5: {tnr_fec = TUNE_FEC_4_5;  break; }
               default: {tnr_fec = TUNE_FEC_AUTOMATIC;  break; }
            }

            // Set system type (S/S2)
            if (tuner_status[tuner].dvb_s2 == TRUE)
            {
               system_type = TUNE_SYSTEM_TYPE_DVBS2;
            }
            else
            {
               system_type = TUNE_SYSTEM_TYPE_DVBS;
            }
            STB_TuneSetSystemType(tuner, system_type);

            STB_TUNE_PRINT(("STB_DP:StartTune(%d,%d): %d %s fec %d (%s_BAND, LO %d, LNB_%s, %s)",
                            path, tuner, tnr_freq / 1000,
                            (tuner_status[tuner].polarity == POLARITY_VERTICAL) ? "V" : "H",
                            tuner_status[tuner].fec,
                            high_band == TRUE ? "HIGH" : "LOW",
                            lo_freq / 1000,
                            (tone_22khz_on == TRUE) ? "22khz" : "OFF",
                            (lnb_volts == LNB_VOLTAGE_18V) ? "18V" : ((lnb_volts == LNB_VOLTAGE_14V) ? "14V" : "OFF")));

            if (tuner_status[tuner].lnb_type == LNB_TYPE_UNICABLE)
            {
               // User Bank 0 - 7 on bits 7-5
               data_1 = (tuner_status[path].unicable_chan & 0x07) << 5;

               // Signal Bank 0 - 7 on bits 4-2
               if (tuner_status[path].unicable_position_b == TRUE)
               {
                  data_1 |= (1 << 4);
               }
               if (tuner_status[tuner].polarity == POLARITY_HORIZONTAL)
               {
                  data_1 |= (1 << 3);
               }
               if (high_band == TRUE)
               {
                  data_1 |= (1 << 2);
               }

               // Tuning word
               t_word = (U16BIT)(((abs((sig_freq / 1000) - (lo_freq / 1000)) + (tnr_freq / 1000)) / 4) - 350);
               data_1 |= (((U8BIT) (t_word >> 8)) & 0x03);
               data_2 = (U8BIT) (t_word & 0x00ff);

               // Send DiSEqC code E0 10 5A channel_byte_1 channel_byte_2
               unicablecmd[0] = 0xE0; // FRAMING
               unicablecmd[1] = 0x10; // ADDRESS
               unicablecmd[2] = 0x5A; // NORMAL COMMAND
               unicablecmd[3] = data_1;
               unicablecmd[4] = data_2;
               UnicableSendCommand(tuner, unicablecmd);
            }
            else
            {
               // Now set the 22kHz tone state
               STB_TuneSet22kState(tuner, tone_22khz_on);
            }

            STB_DPSetTuneStatus(path, TUNE_WAITING);
            STB_TuneStartTuner(tuner, tnr_freq, tuner_status[tuner].sym_rate * 1000, tnr_fec, 0, 0, 0, 0, 0);
            retval = TRUE;
         }
         break;
      }

      case SIGNAL_COFDM:
      {
         // get required signal freq and check it is valid. For cofdm frequency is stored in Hz.
         tnr_freq = tuner_status[tuner].frequency;
         tnr_min_freq = (S32BIT)STB_TuneGetMinTunerFreqKHz(tuner) * 1000;
         tnr_max_freq = (S32BIT)STB_TuneGetMaxTunerFreqKHz(tuner) * 1000;
         if ((tnr_min_freq <= tnr_freq) && (tnr_freq <= tnr_max_freq))
         {
            // set tmode param
            switch (tuner_status[tuner].terr_mode)
            {
               case MODE_COFDM_2K: {tnr_tmode = TUNE_MODE_COFDM_2K;  break; }
               case MODE_COFDM_8K: {tnr_tmode = TUNE_MODE_COFDM_8K;  break; }
               default: {tnr_tmode = TUNE_MODE_COFDM_UNDEFINED;  break; }
            }
            // set tbwidth param
            switch (tuner_status[tuner].terr_bwidth)
            {
               case TBWIDTH_8MHZ: {tnr_tbwidth = TUNE_TBWIDTH_8MHZ;  break; }
               case TBWIDTH_7MHZ: {tnr_tbwidth = TUNE_TBWIDTH_7MHZ;  break; }
               case TBWIDTH_6MHZ: {tnr_tbwidth = TUNE_TBWIDTH_6MHZ;  break; }
               default: {tnr_tbwidth = TUNE_TBWIDTH_8MHZ;  break; }
            }

            // start the tuner tuning - for COFDM only freq, tfoff, tmode and tbwidth are used
            STB_DPSetTuneStatus(path, TUNE_WAITING);
            switch (tuner_status[tuner].terr_type)
            {
               case TERR_TYPE_DVBT: {system_type = TUNE_SYSTEM_TYPE_DVBT; break; }
               case TERR_TYPE_DVBT2: {system_type = TUNE_SYSTEM_TYPE_DVBT2; break; }
               default: {system_type = TUNE_SYSTEM_TYPE_DVBT; break; }
            }
            STB_TuneSetSystemType(tuner, system_type);
            STB_TuneSetPLP(tuner, tuner_status[tuner].plp_id);
            STB_TuneStartTuner(tuner, tnr_freq, 0, 0, tuner_status[tuner].freq_offset, tnr_tmode,
               tnr_tbwidth, TUNE_MODE_QAM_UNDEFINED, 0);
            retval = TRUE;
         }
         break;
      }

      case SIGNAL_QAM:
      {
         tnr_freq = tuner_status[tuner].frequency;
         tnr_min_freq = (S32BIT)STB_TuneGetMinTunerFreqKHz(tuner) * 1000;
         tnr_max_freq = (S32BIT)STB_TuneGetMaxTunerFreqKHz(tuner) * 1000;
         if ((tnr_min_freq <= tnr_freq) && (tnr_freq <= tnr_max_freq))
         {
            switch (tuner_status[tuner].cable_mode)
            {
               case MODE_QAM_4:
                  cmode = TUNE_MODE_QAM_4;
                  break;

               case MODE_QAM_8:
                  cmode = TUNE_MODE_QAM_8;
                  break;

               case MODE_QAM_16:
                  cmode = TUNE_MODE_QAM_16;
                  break;

               case MODE_QAM_32:
                  cmode = TUNE_MODE_QAM_32;
                  break;

               case MODE_QAM_64:
                  cmode = TUNE_MODE_QAM_64;
                  break;

               case MODE_QAM_128:
                  cmode = TUNE_MODE_QAM_128;
                  break;

               case MODE_QAM_256:
                  cmode = TUNE_MODE_QAM_256;
                  break;

               default:
                  cmode = TUNE_MODE_QAM_UNDEFINED;
                  break;
            }

            switch (tuner_status[tuner].fec)
            {
               case FEC_AUTOMATIC:
                  tnr_fec = TUNE_FEC_AUTOMATIC;
                  break;
               case FEC_1_2:
                  tnr_fec = TUNE_FEC_1_2;
                  break;
               case FEC_2_3:
                  tnr_fec = TUNE_FEC_2_3;
                  break;
               case FEC_3_4:
                  tnr_fec = TUNE_FEC_3_4;
                  break;
               case FEC_5_6:
                  tnr_fec = TUNE_FEC_5_6;
                  break;
               case FEC_7_8:
                  tnr_fec = TUNE_FEC_7_8;
                  break;
               default:
                  tnr_fec = TUNE_FEC_AUTOMATIC;
                  break;
            }

            // start the tuner tuning - for COFDM only freq, tfoff, tmode and tbwidth are used
            STB_DPSetTuneStatus(path, TUNE_WAITING);
            STB_TuneStartTuner(tuner, tnr_freq, tuner_status[tuner].sym_rate * 1000, tnr_fec, 0, 0,
               TUNE_TBWIDTH_6MHZ, cmode, 0);
            retval = TRUE;
         }
         break;
      }

      case SIGNAL_ANALOG:
      {
         // get required signal freq and check it is valid. For analogue frequency is stored in Hz.
         tnr_freq = tuner_status[tuner].frequency;
         tnr_min_freq = (S32BIT)STB_TuneGetMinTunerFreqKHz(tuner) * 1000;
         tnr_max_freq = (S32BIT)STB_TuneGetMaxTunerFreqKHz(tuner) * 1000;
         if ((tnr_min_freq <= tnr_freq) && (tnr_freq <= tnr_max_freq))
         {
            // set analog video type param
            switch (tuner_status[tuner].anlg_vtype)
            {
               case ANLG_VIDEO_PAL_I:     {tnr_avtype = TUNE_ANLG_VIDEO_PAL_I; break; }
               case ANLG_VIDEO_PAL_B:     {tnr_avtype = TUNE_ANLG_VIDEO_PAL_B; break; }
               case ANLG_VIDEO_PAL_G:     {tnr_avtype = TUNE_ANLG_VIDEO_PAL_G; break; }
               case ANLG_VIDEO_PAL_D:     {tnr_avtype = TUNE_ANLG_VIDEO_PAL_D; break; }
               case ANLG_VIDEO_PAL_K:     {tnr_avtype = TUNE_ANLG_VIDEO_PAL_K; break; }
               case ANLG_VIDEO_PAL_L:     {tnr_avtype = TUNE_ANLG_VIDEO_PAL_L; break; }
               case ANLG_VIDEO_PAL_LDASH: {tnr_avtype = TUNE_ANLG_VIDEO_PAL_LDASH; break; }
               default:                   {tnr_avtype = TUNE_ANLG_VIDEO_PAL_I; break; }
            }

            // start the tuner tuning - for analog only freq, and video type are used
            STB_DPSetTuneStatus(path, TUNE_WAITING);
            STB_TuneStartTuner(tuner, tnr_freq, 0, 0, tuner_status[tuner].freq_offset, 0, 0, 0,
               tnr_avtype);
            retval = TRUE;
         }
         break;
      }

      default:
      {
         STB_TUNE_PRINT(("Unhandled SIGNAL_TYPE %d", tuner_status[tuner].signal_type));
         break;
      }
   }

   FUNCTION_FINISH(StartTune);

   return(retval);
}

/*!**************************************************************************
 * @brief   Sends a Unicable command (modulated onto 18V DC level)
 * @param   path - The tuner path to use
 * @param   cmd - The unicable command bytes to send
 ****************************************************************************/
static void UnicableSendCommand(U8BIT path, U8BIT cmd[UNICABLE_COMMAND_SIZE])
{
   STB_TuneSetLNBVoltage(path, LNB_VOLTAGE_18V);
   SendDISEQCMessage(path, cmd, UNICABLE_COMMAND_SIZE);
   STB_TuneSetLNBVoltage(path, LNB_VOLTAGE_14V);
}

static S_STB_DP_LNB_BAND *FindLNBBand(U8BIT tuner)
{
   S_STB_DP_LNB_BAND *band;
   U8BIT n, i;

   FUNCTION_START(FindLNBBand);

   if (tuner_status[tuner].lnb_definition[tuner_status[tuner].lnb_type].band_list != NULL)
   {
      n = tuner_status[tuner].lnb_definition[tuner_status[tuner].lnb_type].number_of_bands;
      band = tuner_status[tuner].lnb_definition[tuner_status[tuner].lnb_type].band_list;
      i = 0;
      while ((band != NULL) && (i < n))
      {
         if ((band->polarity == tuner_status[tuner].polarity) &&
            (tuner_status[tuner].frequency > band->min_freq) &&
            ((band->max_freq == 0) || (tuner_status[tuner].frequency <= band->max_freq)))
         {
            break;
         }
         band++;
         i++;
         if (i >= n)
         {
            band = NULL;
         }
      }
   }
   else
   {
      band = NULL;
   }

   FUNCTION_FINISH(FindLNBBand);

   return band;
}

U8BIT STB_DPGetPathOfDemux(U8BIT demux_num)
{
   U8BIT path = INVALID_RES_ID;

   ASSERT(demux_num < num_demuxes);

   for (int i=0; i < num_paths; i++) {
      if (demux_num == path_status[i].demux_no) {
         path = i;
         break;
      }
   }

   return path;
}

U8BIT STB_DPGetPathOfVideoDecoder(U8BIT decoder_num)
{
   U8BIT path = INVALID_RES_ID;

   ASSERT(decoder_num < num_video_decoders);

   for (int i=0; i < num_paths; i++) {
      if (decoder_num == path_status[i].video_decoder_no) {
         path = i;
         break;
      }
   }

   return path;
}

U8BIT STB_DPGetPathOfAudioDecoder(U8BIT decoder_num)
{
   U8BIT path = INVALID_RES_ID;

   ASSERT(decoder_num < num_audio_decoders);

   for (int i=0; i < num_paths; i++) {
      if (decoder_num == path_status[i].audio_decoder_no) {
         path = i;
         break;
      }
   }

   return path;
}

U8BIT STB_DPGetPathOfDescrambler(U32BIT handle)
{
   //return STB_DPGetPathOfDemux((U8BIT)(handle & 0xFF));
   return (U8BIT)(handle & 0xFF);
}

void STB_DPSwitchLivePath(U8BIT fromPath, U8BIT toPath)
{
   PATH_STATUS status = path_status[toPath];

   //path_status[toPath].demux_no = path_status[fromPath].demux_no;
   //path_status[toPath].tuner_no = path_status[fromPath].tuner_no;
   path_status[toPath].video_decoder_no = path_status[fromPath].video_decoder_no;
   path_status[toPath].secondary_video_decoder_no = path_status[fromPath].secondary_video_decoder_no;
   path_status[toPath].audio_decoder_no = path_status[fromPath].audio_decoder_no;
   path_status[toPath].ad_decoder_no = path_status[fromPath].ad_decoder_no;

   //path_status[fromPath].demux_no = status.demux_no;
   //path_status[fromPath].tuner_no = status.tuner_no;
   path_status[fromPath].video_decoder_no = status.video_decoder_no;
   path_status[fromPath].secondary_video_decoder_no = status.secondary_video_decoder_no;
   path_status[fromPath].audio_decoder_no = status.audio_decoder_no;
   path_status[fromPath].ad_decoder_no = status.ad_decoder_no;

   STB_AVSwitchLivePath(path_status[toPath].video_decoder_no, path_status[toPath].demux_no);
   STB_SDKSwitchLivePath(fromPath, toPath);
}

//*****************************************************************************
// End of file
//*****************************************************************************

