/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Header file - Function prototypes for DVB subtitles
 *
 * @file    stbds.h
 * @date    25/09/2003
 */

// pre-processor mechanism so multiple inclusions don't cause compilation error

#ifndef _STBDS_H
#define _STBDS_H

#include "techtype.h"

//---Constant and macro definitions for public use-----------------------------

// Page states
#define NORMAL_CASE        0x00
#define ACQUISITION_POINT  0x01
#define MODE_CHANGE        0x02
#define RES_PAGE_STATE     0x03

// Memory allocation trace routines for subtitle debug only.
//#define DS_HEAP_USAGE
#ifdef DS_HEAP_USAGE
   #define MAX_ARRAY_INDEX  255
#endif

//---Enumerations for public use-----------------------------------------------

//---Global type defs for public use-------------------------------------------

// bits for dvb segment info holding
typedef struct object
{
   struct object *next;

   // ETS 300 743 7.2.4
   U16BIT object_id;
   U8BIT object_version_number;
   U8BIT object_coding_method;
   U8BIT non_modifying_colour_flag;

   U8BIT *map_table_2_to_4_bit;
   U8BIT *map_table_2_to_8_bit;
   U8BIT *map_table_4_to_8_bit;

   U8BIT *character_code;
   U16BIT num_chars;

   // These are varibles for physical control/settings
   // Here they where obtained from top_pixel_data_sub_block & bottom_pixel_data_sub_block
   U8BIT *bitmap;
   U16BIT width;
   U16BIT height;
} S_OBJECT;

typedef struct clut
{
   struct clut *next;

   // ETS 300 743 7.2.3
   U8BIT clut_id;
   U8BIT clut_version_number;

   U32BIT *clut_2_bit;
   U32BIT *clut_4_bit;
   U32BIT *clut_8_bit;

   // These are varibles for physical control/settings
   BOOLEAN default_2_bit;
   BOOLEAN default_4_bit;
   BOOLEAN default_8_bit;
} S_CLUT;

typedef struct region_object
{
   struct region_object *next;

   // ETS 300 743 7.2.2 - RCS object list
   U16BIT object_id;
   U8BIT object_type;
   U8BIT object_provider_flag;
   U16BIT object_horizontal_position;
   U16BIT object_vertical_position;
   U8BIT object_foreground_pixel_code;
   U8BIT object_background_pixel_code;
} S_REGION_OBJECT;

typedef struct epoch_region
{
   struct epoch_region *next;

   // ETS 300 743 7.2.2 - from the RCS
   U8BIT region_id;
   U8BIT region_version_number;
   U8BIT region_fill_flag;
   U16BIT region_width;
   U16BIT region_height;
   U8BIT region_level_of_compatibilty;
   U8BIT region_colour_depth;
   U8BIT region_clut_id;
   U8BIT region_8_bit_pixel_code;
   U8BIT region_4_bit_pixel_code;
   U8BIT region_2_bit_pixel_code;
   S_REGION_OBJECT *region_object_list;
} S_EPOCH_REGION;

typedef struct region
{
   struct region *next;
//   U8BIT  region_version_number;

   // ETS 300 743 7.2.1 - PCS region list
   U8BIT region_id;
   U16BIT region_horizontal_address;
   U16BIT region_vertical_address;
} S_REGION;

typedef struct page_composition
{
   // ETS 300 743 7.2.1 - from the PCS
   U8BIT page_time_out;
   U8BIT page_version_number;
   U8BIT page_state;
   S_REGION *region_list;

   // These are varibles for physical control/settings
   U8BIT pts[5];
   U32BIT presentation_time_ms;
   U32BIT timeout_start;
   BOOLEAN display_set_shown;
   BOOLEAN display_set_removed;

   /* These variables are used for HD subtitles */
   BOOLEAN dds_present;
   U8BIT dds_version;
   U16BIT display_width;
   U16BIT display_height;
   BOOLEAN display_window;
   U16BIT window_x;
   U16BIT window_y;
   U16BIT window_width;
   U16BIT window_height;
} S_PAGE_COMPOSITION;

typedef struct display_set
{
   S_PAGE_COMPOSITION *page_display_buffer;
   S_PAGE_COMPOSITION *page_composition_buffer;
   S_EPOCH_REGION *region_list;
   S_OBJECT *object_list;
   S_CLUT *clut_list;
} S_DISPLAY_SET;

//---Global Function prototypes for public use---------------------------------

// From the file stbdsfn.c
BOOLEAN STB_DSInitialiseDVBSubtitlesProcessing(void);
S_CLUT* STB_DSGetClut(S_CLUT *clut_list, U16BIT clut_id);
BOOLEAN STB_DSSegmentDDS(U8BIT *data, U16BIT pes_len, U16BIT processed_bytes);
BOOLEAN STB_DSSegmentPCS(U8BIT *data, U16BIT pes_len, U16BIT processed_bytes, BOOLEAN force_acquisition);
BOOLEAN STB_DSSegmentRCS(U8BIT *data, U16BIT pes_len, U16BIT processed_bytes);
BOOLEAN STB_DSSegmentCDS(U8BIT *data, U16BIT pes_len, U16BIT processed_bytes);
BOOLEAN STB_DSSegmentODS(U8BIT *data, U16BIT pes_len, U16BIT processed_bytes);
BOOLEAN STB_DSSegmentEDS(U8BIT *data, U16BIT pes_len, U16BIT processed_bytes);
BOOLEAN STB_DSGetPesPts(U8BIT *data, U8BIT *pts);
BOOLEAN STB_DSSetDisplaySetPts(U8BIT path, U8BIT *pts);
S_DISPLAY_SET* STB_DSGetDetails(void);
void STB_DSClearDisplaySetStruct(void);
void STB_DSClearCompositionPageDetails(void);


// From the file stbdsdis.c
void STB_DSCheckDisplaySetTimeout(S_DISPLAY_SET *subtitle_display_set, BOOLEAN timeout_override);
void STB_DSDisplay(U8BIT path, S_DISPLAY_SET *subtitle_display_set);
void STB_DSRegisterCharRenderFunction(void (*DSCreateBitmap)(U8BIT *bitmap, U16BIT width, U16BIT height, U8BIT *char_array,
      U8BIT *tycrcb_palette, U8BIT fgnd_col, U8BIT bkgnd_col));
void STB_DSShow(void);
void STB_DSHide(void);
void STB_DSResetPhysicalDisplayRegions(void);
void STB_DSResetPhysicalCompositionRegions(void);
void STB_DSTerminateDisplayCycle(void);
void STB_DSCreateCompositionRegion(S_EPOCH_REGION *region, BOOLEAN page_reset);
void STB_DSRenderBitmapToRegion(S_EPOCH_REGION *region_list, S_OBJECT *object, U8BIT *scan_line,
   U16BIT y, U16BIT w, U16BIT h);
void STB_DSFillRegion(U16BIT region_id, U8BIT fillcode);
U32BIT STB_DSNumPixelOperations(S_DISPLAY_SET *subtitle_display_set);
void STB_DSFillEmptyRegions(S_EPOCH_REGION *epoch_region_list, S_REGION *region_list);

// stbdsc.c
void STB_SUBInitialise(void);
void* STB_DSGetQueue(void);
BOOLEAN STB_DSGetNextPesPts(U8BIT *next_pts);

#endif //  _STBDS_H

//*****************************************************************************
// End of file
//*****************************************************************************

