/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   DVB subtitle control tasks
 *
 * @file    stbdsc.c
 * @date    12/09/2003
 */

//---includes for this file----------------------------------------------------
// compiler library header files

#include <string.h>
#include <stdio.h>

// third party header files

// Ocean Blue Software header files

#include <techtype.h>
#include <dbgfuncs.h>

#include "stbhwos.h"
#include "stbhwosd.h"
#include "stbhwdmx.h"
#include "stbheap.h"
#include "stbds.h"
#include "stbdsapi.h"
#include "stbdpc.h"
#include "stbpes.h"

//---constant definitions for this file----------------------------------------

#define DVB_SUBT_TASK_STACK_SIZE       1024 * 5
#define DVB_SUBT_TASK_PRIORITY         9

// Debug message macros

#ifdef STB_CONTROL_TASK_PRINT_REQUIRED
   #ifdef STB_CONTROL_TASK_CHANNEL_DEBUG_REQUIRED
      #define STB_CONTROL_TASK_PRINT(x) DEBUG_PRINTX_CONDITIONAL(DEBUG_STB_OSD) x
   #else
      #define STB_CONTROL_TASK_PRINT(x) STB_SPDebugWrite x
   #endif
#else
   #define STB_CONTROL_TASK_PRINT(x)
#endif

//**************************************************************************************************
// DVB subtitle segment macros
//**************************************************************************************************

#define PCS       0x10  // Page Composition Segment
#define RCS       0x11  // Region Composition Segment
#define CDS       0x12  // CLUT Definition Segment
#define ODS       0x13  // Object Data Segment
#define DDS       0x14  // Display Definition Segment

// values 0x40 - 0x7f reserved for future use

// values 0x80 - 0xef reserved for future use
#define EDS       0x80  // End of Display set Segment - DTG Subtitling:
                        // 28 August 1998: Version: 3.0 Clause 5.7.1

// ETS 300 743 7.1
#define END_OF_PES_DATA_FIELD_MARKER  0xff

// all other values reserved for future use

//**************************************************************************************************
// end DVB subtitle segment macros
//**************************************************************************************************

#define SYNC_BYTE 0x0f

// For use with the memcmp function, ease of reading code only
#define MATCH              0x00

// pes_hdr_data_len byte location withinthe PES packet structure
#define PES_HDR_DATA_LEN_BYTE 0x08

// Represents the max value of a 16bit number and the prefix code, stream id and the length bytes.
#define MAX_PES_SIZE           (0xffff + 0x06)

#ifdef DS_HEAP_USAGE
   #define STB_GetMemory(x)  STB_DSGetMemory(x)
   #define STB_FreeMemory(x) STB_DSFreeMemory(x)
#endif

//---local typedef structs for this file---------------------------------------

typedef struct stream
{
   BOOLEAN started;
   BOOLEAN stopped;
   BOOLEAN enabled;

   U8BIT decode_path;
   U16BIT subtitle_pid;
   U16BIT composition_page_id;
   U16BIT ancillary_page_id;
} S_STREAM;

//---local (static) variable declarations for this file------------------------
//   (internal variables declared static to make them local)

static BOOLEAN initialised = FALSE;

//Assuming that we are best able to work out whether we should use the next acquisition point
//as a mode change point (unless mode change occurs first).
//Used to tell lower level that we suspect they need to synchronise at next opportunity.
static BOOLEAN force_acquisition = TRUE;

static S_STREAM subtitle_status;
static void *dvb_subtitle_queue;
static void *dvb_subtitle_ptr;
static void *dvb_decoder_sem;
static void *dvb_display_sem;
static void *set_stream_sem;

static U8BIT last_pts[5] = {0xff, 0xff, 0xff, 0xff, 0xff};
static U8BIT curr_pts[5] = {0xff, 0xff, 0xff, 0xff, 0xff};

static U8BIT *next_pes;

static U32BIT pes_collection_callback_handle;

#ifdef DS_HEAP_USAGE
static U32BIT malloc_array[2][MAX_ARRAY_INDEX];
static U32BIT hi_water_allocations;
static S16BIT num_allocations;
static U32BIT hi_water_mem_alloc;
static U32BIT mem_allocation;
static U8BIT i;
static BOOLEAN found;
static BOOLEAN done_one_malloc_report;
#endif

//---local function prototypes for this file-----------------------------------
//   (internal functions declared static to make them local)

static void DVBSubtitlePesCallback(U32BIT handle, U8BIT data_identifier, void *buffer_ptr, U32BIT buffer_size);
static void DelayIfRequired(void);
static void FlushQueue(void);
static BOOLEAN ParseSegment(U8BIT path, U8BIT *data, U16BIT *processed_bytes, U16BIT pes_length);
static void DVBSubtTask(void *unwanted_p);

//---local function definitions------------------------------------------------

/**
 *

 *
 * @brief   If a stop is required enters a tight loop sleep for a second and check if
 *                 starting required.
 *
 * @param   handle          - unique handle to the PES callback request.
 * @param   data_identifier - ID byte, first byte in PES_data_field.
 * @param   buffer_ptr      - complete PES packet.
 * @param   buffer_size     - number of bytes at buffer_ptr, in packet.
 *

 *
 */
static void DVBSubtitlePesCallback(U32BIT handle, U8BIT data_identifier, void *buffer_ptr, U32BIT buffer_size)
{
   void *buffer;
   BOOLEAN success;

   FUNCTION_START(DVBSubtitlePesCallback);

   USE_UNWANTED_PARAM(handle);
   USE_UNWANTED_PARAM(data_identifier);

   buffer = STB_GetMemory(buffer_size);
   if (buffer != NULL)
   {
      memcpy(buffer, buffer_ptr, buffer_size);

      success = STB_OSWriteQueue(dvb_subtitle_queue, (void *)&buffer, sizeof(U8BIT *), TIMEOUT_NOW);
      if (!success)
      {
         DBGPRINT("DVBSubtitlePesCallback: Failed to add pes buffer to subtitle queue");
         STB_FreeMemory(buffer);
      }
   }

   FUNCTION_FINISH(DVBSubtitlePesCallback);
}

/**
 *

 *
 * @brief   Called from the PES packet parsing loop, because if the packet is large [tens of
 *                 thousand of bytes] this is to allow the circular buffer and PES assembly tasks
 *                 to run.
 *

 *

 *
 */
static void DelayIfRequired(void)
{
   static U32BIT timestamp;
   U32BIT diff;


   FUNCTION_START(DelayIfRequired);

   diff = (STB_OSGetClockDiff(timestamp));

   if (diff > 250)
   {
      STB_OSTaskDelay(10);
      timestamp = STB_OSGetClockMilliseconds();
   }

   FUNCTION_FINISH(DelayIfRequired);
}

/**
 *

 *
 * @brief   Flushes the queues that feed the DVB subtitles task, and clears the subtitle
 *                 structure.
 *

 *

 *
 */
static void FlushQueue(void)
{
   U8BIT *pes_ptr;
   BOOLEAN read_pes_from_queue;


   FUNCTION_START(FlushQueue);
   DBGPRINT("FlushQueue");

   if (next_pes != NULL)
   {
      STB_FreeMemory(next_pes);
      next_pes = NULL;
   }

   do
   {
      read_pes_from_queue = STB_OSReadQueue(dvb_subtitle_queue, &pes_ptr, sizeof(U8BIT *), TIMEOUT_NOW);
      if (read_pes_from_queue == TRUE)
      {
         STB_FreeMemory(pes_ptr);
      }
   }
   while (read_pes_from_queue);

   FUNCTION_FINISH(FlushQueue);
}

/**
 *

 *
 * @brief   data pointers is at the start of segment, this function determines what type of
 *                 of segment it is then checks if it is of the correct page ID.  The length of the
 *                 segment is added to processed_bytes whether or not it was for this page.
 *
 * @param   data            - pointer to the beginning of a segment.
 * @param   processed_bytes - running total that the segment length is added.
 *

 *
 */
static BOOLEAN ParseSegment(U8BIT path, U8BIT *data, U16BIT *processed_bytes, U16BIT pes_length)
{
   U16BIT segment_length;
   U16BIT page_id;
   U8BIT segment_type;
   BOOLEAN parsed_ok = FALSE;
   BOOLEAN ancillary_pg_present;

   // Stores whether we're currently happy with Page composition state
   static BOOLEAN page_okay = FALSE;
   static BOOLEAN comp_cds_allowed;
   static BOOLEAN comp_ods_allowed;
   // Have we processed subtitle data, which we haven't yet displayed?
   static BOOLEAN undisplayed_data = FALSE;


   FUNCTION_START(ParseSegment);

   // sync_byte = data[0];
   segment_type = data[1];

   page_id = (data[2] << 8) + data[3];

   // EN 300 468 version 1.3.1 clause 6.2.30 Subtitling descriptor
   // The values in the ancillary_page_id and the composition_page_id fields
   // shall be the same if no ancillary page is provided.

   // NB. the second clause where pages do not match and the ancillary page is zero
   // is for Channel 4, but strictly speaking this is wrong.
   if ((subtitle_status.composition_page_id == subtitle_status.ancillary_page_id) ||
       (subtitle_status.ancillary_page_id == 0))
   {
      ancillary_pg_present = FALSE;
   }
   else
   {
      ancillary_pg_present = TRUE;
   }

   segment_length = (data[4] << 8) + data[5] + 6;

   /* Check length segment len is sensible */
   if ((*processed_bytes + segment_length) >= pes_length)
   {
      STB_CONTROL_TASK_PRINT(("ERR LENGTH wrong! peslen=%d,seglen=%d, prc=%d",
                              pes_length, segment_length, *processed_bytes));

      segment_length = pes_length - *processed_bytes;
   }
   /* Check we're not about to lose sync with segment data / pes data */
   else if ((data[segment_length] == SYNC_BYTE) ||
            (data[segment_length] == END_OF_PES_DATA_FIELD_MARKER))
   {
      switch (segment_type)
      {
         case DDS:
         {
            if (page_id == subtitle_status.composition_page_id)
            {
               STB_CONTROL_TASK_PRINT(("\nDDS"));

               STB_OSSemaphoreWait(dvb_display_sem);
               parsed_ok = STB_DSSegmentDDS(data, pes_length, *processed_bytes);
               STB_OSSemaphoreSignal(dvb_display_sem);

#ifdef STB_CONTROL_TASK_PRINT_REQUIRED
               if (!parsed_ok)
               {
                  STB_CONTROL_TASK_PRINT(("DDS parse error"));
               }
#endif
            }
#ifdef STB_CONTROL_TASK_PRINT_REQUIRED
            else
            {
               STB_CONTROL_TASK_PRINT(("\nDDS - not composing page %d", page_id));
            }
#endif
            break;
         }

         case PCS:
         {
            STB_CONTROL_TASK_PRINT(("\nPCS - mode %d - %s",
                                    ((data[7] & 0x0c) >> 2),
                                    (force_acquisition ? "FORCE MODE CHANGE" : "")));

            // Check that PCS describes the page that we wish to compose
            if (page_id == subtitle_status.composition_page_id)
            {
               // Remember PTS
               memcpy((void *)last_pts, (void *)curr_pts, (size_t)5);

               /* If there's undisplayed data, then no EDS segment was found.
                * Any existing data should be discarded, which should happen when the next
                * acquisition point is detected */
               if (undisplayed_data)
               {
                  undisplayed_data = FALSE;
               }

               // Until we parse PCS succesfully we need to inhibit any further processing
               page_okay = FALSE;

               STB_CONTROL_TASK_PRINT(("PCS composing page %d", page_id));

               //Process the PCS segment.
               //force_acquisition == TRUE signals that lower level should treat the first acquisition
               //point as a mode change (unless a mode_change occurs first)
               STB_OSSemaphoreWait(dvb_display_sem);
               parsed_ok = STB_DSSegmentPCS(data, pes_length, *processed_bytes, force_acquisition);
               STB_OSSemaphoreSignal(dvb_display_sem);

               if (!parsed_ok)
               {
                  force_acquisition = TRUE;
                  STB_CONTROL_TASK_PRINT(("PCS parse error in STB_DSSegmentPCS!!"));
               }
               else
               {
                  //Lower level remembers this signal, we don't send again, until we think we need
                  //to re-acquire again
                  force_acquisition = FALSE;

                  // Allow processing of following segments within page
                  page_okay = TRUE;
               }

               if (parsed_ok)
               {
                  STB_DSSetDisplaySetPts(path, curr_pts);
                  // Remember that we have started defining a page which has yet to be displayed
                  undisplayed_data = TRUE;
                  comp_cds_allowed = TRUE;
                  comp_ods_allowed = TRUE;
               }
            }
#ifdef STB_CONTROL_TASK_PRINT_REQUIRED
            else
            {
               STB_CONTROL_TASK_PRINT(("PCS - not composing page %d", page_id));
            }
#endif
            break;
         }

         case RCS:
         {
#ifdef STB_CONTROL_TASK_PRINT_REQUIRED
            if (page_okay)
            {
               STB_CONTROL_TASK_PRINT(("+RCS: page=%d", page_id));
            }
            else
            {
               STB_CONTROL_TASK_PRINT(("-rcs: page=%d", page_id));
            }
#endif

            if (page_okay &&
                ((page_id == subtitle_status.composition_page_id) ||
                 ((page_id == subtitle_status.ancillary_page_id) && ancillary_pg_present)))
            {
               STB_OSSemaphoreWait(dvb_display_sem);
               parsed_ok = STB_DSSegmentRCS(data, pes_length, *processed_bytes);
               STB_OSSemaphoreSignal(dvb_display_sem);

               if (!parsed_ok)
               {
                  STB_CONTROL_TASK_PRINT(("RCS parse error"));
               }
            }
#ifdef STB_CONTROL_TASK_PRINT_REQUIRED
            else if (page_okay)
            {
               STB_CONTROL_TASK_PRINT(("RCS - not composing page %d", page_id));
            }
#endif
            break;
         }

         case CDS:
         {
#ifdef STB_CONTROL_TASK_PRINT_REQUIRED
            if (page_okay && comp_cds_allowed)
            {
               STB_CONTROL_TASK_PRINT(("+CDS: page=%d", page_id));
            }
            else
            {
               STB_CONTROL_TASK_PRINT(("-cds: page=%d", page_id));
            }
#endif

            if (page_okay &&
                ((page_id == subtitle_status.composition_page_id) ||
                 (page_id == subtitle_status.ancillary_page_id)))
            {
               if ((page_id == subtitle_status.ancillary_page_id) && (ancillary_pg_present))
               {
                  comp_cds_allowed = FALSE;
               }
               if ((page_id == subtitle_status.composition_page_id) && (comp_cds_allowed))
               {
                  parsed_ok = STB_DSSegmentCDS(data, pes_length, *processed_bytes);
                  if (!parsed_ok)
                  {
                     STB_CONTROL_TASK_PRINT(("CDS parse error"));
                  }
               }
            }
#ifdef STB_CONTROL_TASK_PRINT_REQUIRED
            else if (page_okay)
            {
               STB_CONTROL_TASK_PRINT(("CDS - not composing page %d", page_id));
            }
#endif

            break;
         }

         case ODS:
         {
#ifdef STB_CONTROL_TASK_PRINT_REQUIRED
            if (page_okay && comp_ods_allowed)
            {
               STB_CONTROL_TASK_PRINT(("+ODS: page=%d", page_id));
            }
            else
            {
               STB_CONTROL_TASK_PRINT(("-ods: page=%d", page_id));
            }
#endif

            if (page_okay &&
                ((page_id == subtitle_status.composition_page_id) ||
                 (page_id == subtitle_status.ancillary_page_id)))
            {
               if ((page_id == subtitle_status.ancillary_page_id) && (ancillary_pg_present))
               {
                  comp_ods_allowed = FALSE;
                  comp_cds_allowed = FALSE;
               }
               if ((page_id == subtitle_status.composition_page_id) && (comp_ods_allowed))
               {
                  STB_OSSemaphoreWait(dvb_display_sem);
                  parsed_ok = STB_DSSegmentODS(data, pes_length, *processed_bytes);
                  STB_OSSemaphoreSignal(dvb_display_sem);
                  if (!parsed_ok)
                  {
                     STB_CONTROL_TASK_PRINT(("ODS parse error"));
                  }
               }
            }
#ifdef STB_CONTROL_TASK_PRINT_REQUIRED
            else if (page_okay)
            {
               STB_CONTROL_TASK_PRINT(("ODS - not composing page %d", page_id));
            }
#endif

            break;
         }

         case EDS:
         {
#ifdef STB_CONTROL_TASK_PRINT_REQUIRED
            if (page_okay)
            {
               STB_CONTROL_TASK_PRINT(("+EDS: page=%d", page_id));
            }
            else
            {
               STB_CONTROL_TASK_PRINT(("-eds: page=%d", page_id));
            }
#endif

            if (page_okay &&
                ((page_id == subtitle_status.composition_page_id) ||
                 ((page_id == subtitle_status.ancillary_page_id) && ancillary_pg_present)))
            {
               parsed_ok = STB_DSSegmentEDS(data, pes_length, *processed_bytes);

               if (parsed_ok)
               {
                  STB_OSSemaphoreWait(dvb_display_sem);
                  STB_DSDisplay(path, STB_DSGetDetails());
                  STB_OSSemaphoreSignal(dvb_display_sem);
                  undisplayed_data = FALSE;
               }
               else
               {
                  STB_CONTROL_TASK_PRINT(("EDS parse error"));
               }
            }
#ifdef STB_CONTROL_TASK_PRINT_REQUIRED
            else if (page_okay)
            {
               STB_CONTROL_TASK_PRINT(("EDS - not composing page %d", page_id));
            }
#endif

            break;
         }

         default:
         {
            STB_CONTROL_TASK_PRINT(("Unknown segment code 0x%02x", segment_type));
            break;
         }
      }
   }
   else
   {
      STB_CONTROL_TASK_PRINT(("LOST sync with segments; PESlen=%d, prc_posn=%d,  seglen=%d, dseg=%x",
                              pes_length, *processed_bytes, segment_length, data[segment_length]));
   }

   (*processed_bytes) += segment_length;

   FUNCTION_FINISH(ParseSegment);

   return(parsed_ok);
}

/**
 *

 *
 * @brief   The DVB subtitle task; receives a PES packet parses all segments in it, obtains
 *                 PTS info, and times display of display sets.
 *

 *

 *
 */
static void DVBSubtTask(void *unwanted_p)
{
   U8BIT *pes_ptr;
   U8BIT *data;

   U16BIT data_id_stream_id;
   U16BIT pes_length;
   U16BIT processed_bytes;

   U8BIT pts_dts_flags;
   U8BIT pes_hdr_data_len;
   U8BIT segment_data;

   BOOLEAN data_alignment_indicator;
   BOOLEAN pts_present;
   BOOLEAN read_pes_from_queue;
   BOOLEAN terminate;

   FUNCTION_START(DVBSubtTask);

   USE_UNWANTED_PARAM(unwanted_p);

   subtitle_status.decode_path = INVALID_RES_ID;
   subtitle_status.subtitle_pid = 0;
   subtitle_status.composition_page_id = 0;
   subtitle_status.ancillary_page_id = 0;

   subtitle_status.stopped = FALSE;
   subtitle_status.started = FALSE;
   subtitle_status.enabled = FALSE;

   pes_ptr = NULL;
   data = NULL;

   data_id_stream_id = 0;
   pes_length = 0;
   processed_bytes = 0;

   pts_dts_flags = 0;
   pes_hdr_data_len = 0;
   segment_data = 0;

   data_alignment_indicator = FALSE;
   pts_present = FALSE;
   read_pes_from_queue = FALSE;

   while (1)
   {
      read_pes_from_queue = STB_OSReadQueue(dvb_subtitle_queue, &pes_ptr, sizeof(U8BIT *), 250);

      if (subtitle_status.decode_path != INVALID_RES_ID)
      {
         STB_OSSemaphoreWait(dvb_display_sem);
         STB_DSCheckDisplaySetTimeout(STB_DSGetDetails(), FALSE);
         STB_OSSemaphoreSignal(dvb_display_sem);
      }

      if (read_pes_from_queue)
      {
         // Got PES check packet_start_code_prefix 24bit value 0x000001
         if ((pes_ptr[0] == 0x00) && (pes_ptr[1] == 0x00) && (pes_ptr[2] == 0x01))
         {
            // Check stream_id
            if (pes_ptr[3] == 0xbd)
            {
               // private_stream_1
               // This value is from this point forward, the size of the entire PES packet
               // is (pes_length + 6); as the bytes before and the length bytes are not included.
               pes_length = (pes_ptr[4] << 8) + pes_ptr[5] + 6;

               data_alignment_indicator = (pes_ptr[6] >> 2) & 0x01;

               pts_dts_flags = (pes_ptr[7] & 0xc0) >> 6;
               pts_present = ((pts_dts_flags & 0x2) >> 1);

               // PES_header_data_length - An 8bit field specifying the total number of bytes
               // occupied by the optional fields and any stuffing bytes in this PES packet header.
               pes_hdr_data_len = pes_ptr[PES_HDR_DATA_LEN_BYTE];

               if (pts_present)
               {
                  // PTS_DTS_flags when set either pts present[10] or pts & dts present[11]
                  // neither present[00], and no pts and a dts[01] not allowed; as dts is
                  // listed after pts, the pts is in the same location.
                  if (STB_DSGetPesPts(&pes_ptr[9], curr_pts) == FALSE)
                  {
                     // One or more of the 3 marker_bits' are not set correctly.
                     memset(curr_pts, 0x00, 5);
                     STB_CONTROL_TASK_PRINT(("PTS error"));
                  }
               }
               else
               {
                  // No PTS; but this value has the effect of displaying immediately.
                  memset(curr_pts, 0x00, 5);
                  STB_CONTROL_TASK_PRINT(("No PTS"));
               }

               // This should place data ptr at the beginning of the PES_packet_data_byte area
               segment_data = PES_HDR_DATA_LEN_BYTE + pes_hdr_data_len + 1;
               data = &pes_ptr[segment_data];

               data_id_stream_id = (data[0] << 8) + data[1];
               data += 2;
               processed_bytes = segment_data + 2;

               if (subtitle_status.decode_path != INVALID_RES_ID)
               {
                  // ETS 300 743 5.1.2 - In summary, all of the segments of a single display set
                  // shall be carried in one (or more) PES packets that have the same PTS value.
                  if (memcmp(last_pts, curr_pts, 5) != MATCH)
                  {
                     STB_OSSemaphoreWait(dvb_display_sem);
                     STB_DSCheckDisplaySetTimeout(STB_DSGetDetails(), FALSE);
                     STB_OSSemaphoreSignal(dvb_display_sem);
                  }
               }

               if ((data_id_stream_id == 0x2000) &&
                   (data_alignment_indicator))
               {
                  terminate = FALSE;

                  while (processed_bytes < pes_length)
                  {
                     switch (data[0])
                     {
                        case SYNC_BYTE:
                        {
                           ParseSegment(subtitle_status.decode_path, data,
                              &processed_bytes, pes_length);
                           break;
                        }

                        case END_OF_PES_DATA_FIELD_MARKER:
                        {
                           processed_bytes++;
                           break;
                        }

                        default:
                        {
                           processed_bytes++;
                           STB_CONTROL_TASK_PRINT(("Not a sync byte, PES len=%u, processed %u bytes, value=0x%02x",
                                                   pes_length, processed_bytes, data[0]));
                           terminate = TRUE;
                           break;
                        }
                     }

                     if (terminate)
                     {
                        break;
                     }

                     // Move data to next segment
                     data = &pes_ptr[processed_bytes];
                     DelayIfRequired();
                  }
               }
            }
         }
      }

      if (pes_ptr != NULL)
      {
         STB_FreeMemory(pes_ptr);
         pes_ptr = NULL;
      }
   } //loop

   /* Execution never gets here! */
#if 0
   STB_UnregisterPesCollectionCallback(pes_collection_callback_handle);
   STB_SUBStop();
#endif

   FUNCTION_FINISH(DVBSubtTask);
}

//---global function definitions-----------------------------------------------

#ifdef DS_HEAP_USAGE
/**
 *

 *
 * @brief   Traces memory allocations and frees, for debug.
 *
 * @param   bytes - malloc request.
 *
 * @return   Pointer to the malloced memory.
 *
 */
void* STB_DSGetMemory(U32BIT bytes)
{
   void *temp;
   BOOLEAN hi_water;

   temp = STB_GetMemory(bytes);
   if (temp != NULL)
   {
      num_allocations++;
      mem_allocation += bytes;
      found = FALSE;
      hi_water = FALSE;

      for (i = 0; i < MAX_ARRAY_INDEX; i++)
      {
         if (malloc_array[0][i] == 0)
         {
            found = TRUE;
            break;
         }
      }
      if (found)
      {
         malloc_array[0][i] = temp;
         malloc_array[1][i] = bytes;
      }
      else
      {
         STB_SPDebugWrite("Malloc trace array exceeded");
      }

      if (num_allocations > hi_water_allocations)
      {
         hi_water_allocations = num_allocations;
         hi_water = TRUE;
      }
      if (mem_allocation > hi_water_mem_alloc)
      {
         hi_water_mem_alloc = mem_allocation;
         hi_water = TRUE;
      }

      if (hi_water)
      {
         STB_SPDebugWrite("Malloc: HiNum %02d; Hisize %02d",
            hi_water_allocations, hi_water_mem_alloc);
      }
      if ((done_one_malloc_report) && (num_allocations > 2))
      {
         done_one_malloc_report = FALSE;
      }
   }
   return(temp);
}

/**
 *

 *
 * @brief   Traces memory allocations and frees, for debug.
 *
 * @param   adde - pointer to memory to be freed.
 *

 *
 */
void STB_DSFreeMemory(void *addr)
{
   if (addr != NULL)
   {
      found = FALSE;
      STB_FreeMemory(addr);
      num_allocations--;

      for (i = 0; i < MAX_ARRAY_INDEX; i++)
      {
         if (malloc_array[0][i] == (U32BIT)addr)
         {
            found = TRUE;
            break;
         }
      }

      if (found)
      {
         mem_allocation -= malloc_array[1][i];
         malloc_array[0][i] = 0;
         malloc_array[1][i] = 0;
      }
      else
      {
         STB_SPDebugWrite("NO record of malloc!!!");
      }

      if ((num_allocations == 1) && (!done_one_malloc_report))
      {
         STB_SPDebugWrite("One malloc, scratch pad");
         done_one_malloc_report = TRUE;
      }
      if (num_allocations < 0)
      {
         STB_SPDebugWrite("Minus Frees!");
      }
   }
}

#endif

/**
 *

 *
 * @brief   Initialises subtitling control.
 *

 *

 *
 */
void STB_SUBInitialise(void)
{
   FUNCTION_START(STB_SUBInitialise);

   if (!initialised)
   {
      subtitle_status.started = FALSE;
      subtitle_status.stopped = TRUE;
      subtitle_status.enabled = FALSE;

      subtitle_status.decode_path = INVALID_RES_ID;
      subtitle_status.subtitle_pid = 0;
      subtitle_status.composition_page_id = 0;
      subtitle_status.ancillary_page_id = 0;

      dvb_subtitle_queue = (void *)STB_OSCreateQueue(sizeof(U8BIT *), 50);

      dvb_decoder_sem = STB_OSCreateSemaphore();
      dvb_display_sem = STB_OSCreateSemaphore();
      set_stream_sem = STB_OSCreateSemaphore();

      STB_OSSemaphoreWait(dvb_decoder_sem);

      STB_DSInitialiseDVBSubtitlesProcessing();
      dvb_subtitle_ptr = STB_OSCreateTask(DVBSubtTask, NULL, DVB_SUBT_TASK_STACK_SIZE,
            DVB_SUBT_TASK_PRIORITY, (U8BIT *)"DStask");

      if ((dvb_subtitle_ptr != NULL) && (dvb_decoder_sem != NULL) &&
          (dvb_display_sem != NULL) && (set_stream_sem != NULL))
      {
         STB_CONTROL_TASK_PRINT(("\nDVB subtitle task, bitmaps rendered directly into region, double buffered"));

         pes_collection_callback_handle = STB_RegisterPesCollectionCallback(DVBSubtitlePesCallback, 0x20, 0x20);

         STB_OSDEnable(FALSE);
         initialised = TRUE;
      }
   }

   FUNCTION_FINISH(STB_SUBInitialise);
}

/**
 *

 *
 * @brief   Enables or disables subtitling display.
 *
 * @param   BOOLEAN enable - TRUE to enable display
 *

 *
 */
void STB_SUBEnable(U8BIT path, BOOLEAN enable)
{
   FUNCTION_START(STB_SUBEnable);
   USE_UNWANTED_PARAM(path);

   if (initialised)
   {
      DBGPRINT("STB_SUBEnable(%d): enable=%u", path, enable);

      // Show or Hide the subtitling display
      if (enable == TRUE)
      {
         STB_DSShow();
      }
      else
      {
         STB_OSSemaphoreWait(set_stream_sem);
         STB_DSTerminateDisplayCycle();
         STB_OSSemaphoreSignal(set_stream_sem);

         STB_DSHide();
         STB_OSSemaphoreWait(dvb_display_sem);
         STB_DSCheckDisplaySetTimeout(STB_DSGetDetails(), TRUE);
         STB_OSSemaphoreSignal(dvb_display_sem);
      }

      subtitle_status.enabled = enable;
   }
   else
   {
      DBGPRINT("STB_SUBEnable called before initialisation -- IGNORING ");
   }

   FUNCTION_FINISH(STB_SUBEnable);
}

/**
 *

 *
 * @brief   Starts subtitling display.
 *
 * @param   U16BIT pid             - The PID of the subtitle data
 * @param   U16BIT composition_id  - The ID of the composition page
 * @param   U16BIT ancillary_id    - The ID of the ancillary page
 *

 *
 */
void STB_SUBStart(U8BIT path, U16BIT pid, U16BIT composition_id, U16BIT ancillary_id)
{
   FUNCTION_START(STB_SUBStart);

   if (initialised)
   {
      DBGPRINT("STB_SUBStart(%d): PID %d comp pg %d anc pg %d", path, pid, composition_id, ancillary_id);

      STB_SUBSetStream(path, pid, composition_id, ancillary_id);
   }

   FUNCTION_FINISH(STB_SUBStart);
}

/**
 *

 *
 * @brief   Stops subtitling display.
 *

 *

 *
 */
void STB_SUBStop(U8BIT path)
{
   FUNCTION_START(STB_SUBStop);

   if (initialised)
   {
      DBGPRINT(">>>>>>>>>>>>>>>>>>STB_SUBStop(%d)", path);
      STB_SUBSetStream(path, 0, 0, 0);

      STB_OSSemaphoreWait(dvb_display_sem);
      subtitle_status.decode_path = INVALID_RES_ID;
      STB_OSSemaphoreSignal(dvb_display_sem);
      DBGPRINT("<<<<<<<<<<<<<<<STB_SUBStop()");
   }

   FUNCTION_FINISH(STB_SUBStop);
}

/**
 *

 *
 * @brief   Sets the stream IDs to be decoded
 *
 *                 EN 300 468 version 1.3.1 clause 6.2.30 Subtitling descriptor
 *                 The values in the ancillary_page_id and the composition_page_id fields
 *                 shall be the same if no ancillary page is provided.
 *
 * @param   U16BIT pid             - The PID of the subtitle data
 * @param   U16BIT composition_id  - The ID of the composition page
 * @param   U16BIT ancillary_id    - The ID of the ancillary page
 *

 *
 */
void STB_SUBSetStream(U8BIT path, U16BIT pid, U16BIT composition_id, U16BIT ancillary_id)
{
   FUNCTION_START(STB_SUBSetStream);

   if (initialised)
   {
      DBGPRINT("STB_SUBSetStream(%d): %d %d %d", path, pid, composition_id, ancillary_id);

      STB_OSSemaphoreWait(set_stream_sem);
      STB_DSTerminateDisplayCycle();
      if (subtitle_status.started == TRUE)
      {
         STB_OSSemaphoreWait(dvb_decoder_sem);
         subtitle_status.started = FALSE;
      }
      STB_OSSemaphoreSignal(set_stream_sem);

      /* STB_SUBEnable can wait for a timeout on the next display set, but calling
       * STB_DSTerminateDisplayCycle will make it drop out of the wait loop, so
       * this must be called before STB_SUBEnable */
      STB_SUBEnable(path, FALSE);

      //Make sure that if we see an Acquisition Point before the next Mode Change, we treat it
      //as a Mode Change
      force_acquisition = TRUE;
      DBGPRINT("STB_SUBSetStream 2");
      STB_OSSemaphoreWait(dvb_display_sem);

      STB_DSResetPhysicalDisplayRegions();
      STB_DSResetPhysicalCompositionRegions();

      if ((pid != STB_DPGetTextPID(path)) || (pid != subtitle_status.subtitle_pid))
      {
         DBGPRINT("STB_SUBSetStream 3a");
         STB_DSClearDisplaySetStruct();

         // Flush PES collection task & queue
         STB_ChangePesCollectionPID(path, pid);
//         STB_FlushPesCollectionTask();
         FlushQueue();
      }
      else
      {
         DBGPRINT("STB_SUBSetStream 3b");
         if ((composition_id != subtitle_status.composition_page_id) ||
             (ancillary_id != subtitle_status.ancillary_page_id))
         {
            STB_DSClearDisplaySetStruct();
         }
      }

      STB_OSSemaphoreSignal(dvb_display_sem);

      // Set the new stream parameters
      subtitle_status.decode_path = path;
      subtitle_status.subtitle_pid = pid;
      subtitle_status.composition_page_id = composition_id;
      subtitle_status.ancillary_page_id = ancillary_id;

      // Start the subtitling decoder
      subtitle_status.started = TRUE;
      STB_OSSemaphoreSignal(dvb_decoder_sem);

      // Do not show yet.
      STB_SUBEnable(path, FALSE);
   }
   else
   {
      DBGPRINT("STB_SUBSetStream called before initialisation -- IGNORING");
   }

   FUNCTION_FINISH(STB_SUBSetStream);
}

/**
 *

 *
 * @brief   returns the current status information
 *
 * @param   
 *

 *
 */
void STB_SUBStatus(BOOLEAN *started, BOOLEAN *shown)
{
   FUNCTION_START(STB_SUBStatus);

   *started = subtitle_status.started;
   *shown = subtitle_status.enabled;

   FUNCTION_FINISH(STB_SUBStatus);
}

/**
 *

 *
 * @brief   returns the current status information
 *
 * @param   started - returns TRUE if subtitles are decoding, FALSE otherwise
 * @param pid     - returns the pid in use. Not valid if running is FALSE
 * @param comp_id - returns the comp page id. Not valid if running is FALSE
 * @param anc_id  - returns the ancillary page id. Not valid if running is FALSE
 * @param enabled - returns TRUE if subtitle display is enabled, FALSE otherwise
 *

 *
 */
void STB_SUBReadSettings(BOOLEAN *started, U16BIT *pid, U16BIT *comp_id, U16BIT *anc_id, BOOLEAN *enabled)
{
   FUNCTION_START(STB_SUBReadSettings);

   *started = subtitle_status.started;
   *pid = subtitle_status.subtitle_pid;
   *comp_id = subtitle_status.composition_page_id;
   *anc_id = subtitle_status.ancillary_page_id;
   *enabled = subtitle_status.enabled;

   FUNCTION_FINISH(STB_SUBReadSettings);
}

/**
 *

 *
 * @brief   Get the PTS of the next PES packet, does not process next PES, but places on the
 *                next PES pointer.
 *
 * @param   
 *
 * @return   TRUE or FALSE for success or failure.
 *
 */
BOOLEAN STB_DSGetNextPesPts(U8BIT *next_pts)
{
   U8BIT *pes_ptr;
   BOOLEAN read_pes_from_queue;

   FUNCTION_START(STB_DSGetNextPesPts);

   next_pts[0] = 0;
   next_pts[1] = 0;
   next_pts[2] = 0;
   next_pts[3] = 0;
   next_pts[4] = 0;

   pes_ptr = NULL;

   if (next_pes == NULL)
   {
      read_pes_from_queue = STB_OSReadQueue(dvb_subtitle_queue, &pes_ptr, sizeof(U8BIT *), 0);
      next_pes = pes_ptr;
   }
   else
   {
      pes_ptr = next_pes;
      read_pes_from_queue = TRUE;
   }

   if (next_pes != NULL)
   {
      if (STB_DSGetPesPts(&pes_ptr[9], next_pts) == FALSE)
      {
         next_pts[0] = 0;
         next_pts[1] = 0;
         next_pts[2] = 0;
         next_pts[3] = 0;
         next_pts[4] = 0;
      }
   }

   FUNCTION_FINISH(STB_DSGetNextPesPts);

   return(read_pes_from_queue);
}

//*****************************************************************************
// End of file
//*****************************************************************************

