/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   DVB subtitle display functions
 *
 * @file    stbdsdis.c
 * @date    25/09/2003
 */
/* #define STB_ERROR_PRINT_REQUIRED*/
/* #define TIMINGS */
/* #define STB_TIMING_PRINT_REQUIRED */
/* #define STB_DISPLAY_PRINT_REQUIRED */
/* #define STB_PIXEL_PRINT_REQUIRED */

//---includes for this file----------------------------------------------------
// compiler library header files

#include <string.h>
#include <stdlib.h>

// third party header files

// Ocean Blue Software header files

#include <techtype.h>
#include <dbgfuncs.h>

#include "stbhwos.h"
#include "stbhwosd.h"
#include "stbheap.h"
#include "stbds.h"
#include "stbhwosd.h"
#include "stbhwav.h"
#include "stbdpc.h"

//---constant definitions for this file----------------------------------------

// System Time Clock frequency - 90kHz
#define STC_FREQ                 90000
#define QUARTER_SEC_STC_FREQ     (STC_FREQ / 4)
// But for the PTS's, of the 5 bytes the 4MSB only are used.
#define QUARTER_SEC_STC_FREQ_MSB (QUARTER_SEC_STC_FREQ >> 8)

#ifdef STB_DEBUG_PRINT_REQUIRED
   #ifdef STB_DEBUG_CHANNEL_DEBUG_REQUIRED
      #define STB_DBG_PRINT(x) DEBUG_PRINTX_CONDITIONAL(DEBUG_STB_OSD) x
   #else
      #define STB_DBG_PRINT(x) STB_SPDebugWrite x
   #endif
#else
   #define STB_DBG_PRINT(x)
#endif

#ifdef STB_ERROR_PRINT_REQUIRED
   #ifdef STB_ERROR_CHANNEL_DEBUG_REQUIRED
      #define STB_ERROR_PRINT(x) DEBUG_PRINTX_CONDITIONAL(DEBUG_STB_OSD) x
   #else
      #define STB_ERROR_PRINT(x) STB_SPDebugWrite x
   #endif
#else
   #define STB_ERROR_PRINT(x)
#endif

#ifdef STB_TIMING_PRINT_REQUIRED
   #ifdef STB_TIMING_CHANNEL_DEBUG_REQUIRED
      #define STB_TIMING_PRINT(x) DEBUG_PRINTX_CONDITIONAL(DEBUG_STB_OSD) x
   #else
      #define STB_TIMING_PRINT(x) STB_SPDebugWrite x
   #endif
#else
   #define STB_TIMING_PRINT(x)
#endif

#ifdef STB_DISPLAY_PRINT_REQUIRED
   #ifdef STB_DISPLAY_MOVE_CHANNEL_DEBUG_REQUIRED
      #define STB_DISPLAY_PRINT(x) DEBUG_PRINTX_CONDITIONAL(DEBUG_STB_OSD) x
   #else
      #define STB_DISPLAY_PRINT(x) STB_SPDebugWrite x
   #endif
#else
   #define STB_DISPLAY_PRINT(x)
#endif

#ifdef STB_PIXEL_PRINT_REQUIRED
   #ifdef STB_PIXEL_MOVE_CHANNEL_DEBUG_REQUIRED
      #define STB_PIXEL_PRINT(x) DEBUG_PRINTX_CONDITIONAL(DEBUG_STB_OSD) x
   #else
      #define STB_PIXEL_PRINT(x) STB_SPDebugWrite x
   #endif
#else
   #define STB_PIXEL_PRINT(x)
#endif

#ifdef DS_HEAP_USAGE
   #define STB_GetMemory(x)  STB_DSGetMemory(x)
   #define STB_FreeMemory(x) STB_DSFreeMemory(x)
#endif

//---local typedef structs for this file---------------------------------------

typedef struct physical_region
{
   struct physical_region *next;
   void *handle;
   U8BIT region_id;
   U8BIT region_version_number;

   // These are varibles for physical control/settings
   U16BIT current_x;
   U16BIT current_y;
   U16BIT region_width;
   U16BIT region_height;
   U16BIT region_colour_depth;

   BOOLEAN visible;
} S_PHYSICAL_REGION;


//---local (static) variable declarations for this file------------------------
//   (internal variables declared static to make them local)

static BOOLEAN terminate_subtitles;
static BOOLEAN show_subtitles;

static BOOLEAN time_diff_set = FALSE;
static U32BIT time_diff_value;

static S_PHYSICAL_REGION *physical_display_region_list;
static S_PHYSICAL_REGION *physical_composition_region_list;

// Callback function for the stb_ui library to use font fusion to render a string of characters
// into a provided area of memory.
static void (*CreateBitmapCallback)(U8BIT *bitmap, U16BIT width, U16BIT height, U8BIT *char_array,
   U8BIT *tycrcb_palette, U8BIT fgnd_col, U8BIT bkgnd_col);

// Display set timimg variables
#ifdef TIMINGS
static U32BIT shown_time;
static U32BIT diff_ms;
static S32BIT delay;
static BOOLEAN do_timing;
#endif

//---local function prototypes for this file-----------------------------------
//   (internal functions declared static to make them local)

static BOOLEAN DetectPcrDiscontinuity(U8BIT path);
static BOOLEAN WaitToDisplay(U8BIT path, S_PAGE_COMPOSITION *pg_disp_buff, S_PAGE_COMPOSITION *pg_comp_buff);
static void HideAllShownRegions(S_REGION *region_list);
static void SwapDisplayAndCompositionBuffers(S_PAGE_COMPOSITION **buff_1,
   S_PAGE_COMPOSITION **buff_2);
static void CheckToHideCopiedRegion(S_PHYSICAL_REGION *physical_region, S_REGION *region_list);
static void CompleteCompositionList(S_DISPLAY_SET *subtitle_display_set);
static void MoveRegions(S_REGION *region_list);
static void UpdateDisplaySet(U8BIT path, S_DISPLAY_SET *subtitle_display_set);
static void ColourDepthAdjustTo2Bits(U8BIT **region_adjusted_bitmap, U8BIT *bitmap,
   U16BIT width, U16BIT height);
static void ColourDepthAdjustTo4Bits(U8BIT **region_adjusted_bitmap, U8BIT *bitmap,
   U16BIT width, U16BIT height);
static void LoadPalettesIntoRegions(S_EPOCH_REGION *region_list, S_CLUT *clut_list);
static void ShowDisplaySet(S_REGION *region_list);
static void remove_entry_from_list(S_PHYSICAL_REGION *entry, S_PHYSICAL_REGION **list);
#ifdef STB_DISPLAY_PRINT_REQUIRED
void dump_physical_regions(const char *label);
#endif

//---local function definitions------------------------------------------------

/**
 *

 *
 * @brief   Detect a relatively large PCR change over a timed period in ms.
 *

 *
 * @return   TRUE - Discontinuity detected.
 *
 */
static BOOLEAN DetectPcrDiscontinuity(U8BIT path)
{
   static U32BIT last_time;
   static U8BIT last_stc[5];

   U32BIT stc_timestamp;
   U32BIT diff_stc_ms;
   U32BIT stc_err;
   U32BIT projected_stc;
   U32BIT new_time;
   U32BIT diff_last_time;
   U8BIT new_stc[5];

   BOOLEAN retval;


   FUNCTION_START(DetectPcrDiscontinuity);

   diff_last_time = STB_OSGetClockDiff(last_time);

   // System Time Clock frequency - 90kHz
   diff_stc_ms = ((diff_last_time * 90) >> 8);

   new_time = STB_OSGetClockMilliseconds();
   STB_AVGetSTC(STB_DPGetPathVideoDecoder(path), new_stc);

   stc_timestamp = ((last_stc[0] << 24) + (last_stc[1] << 16) + (last_stc[2] << 8) + last_stc[3]);
   projected_stc = stc_timestamp + diff_stc_ms;
   stc_timestamp = ((new_stc[0] << 24) + (new_stc[1] << 16) + (new_stc[2] << 8) + new_stc[3]);
   stc_err = abs(projected_stc - stc_timestamp);

   /* Allow STC to be out by 1 minute before signalling a discontinuity */
   if (stc_err > ((STC_FREQ >> 8) * 60))
   {
      STB_TIMING_PRINT(("DetectPcrDiscontinuity"));
      retval = TRUE;
   }
   else
   {
      retval = FALSE;
   }

   memcpy((void *)last_stc, (void *)new_stc, (size_t)5);
   last_time = new_time;

   FUNCTION_FINISH(DetectPcrDiscontinuity);

   return(retval);
}

/**
 *

 *
 * @brief   Display the next display set.  If the display time is not yet met then the
 *                 function blocks until the display time is reached.  During this period it
 *                 checks the timeout of the current display set, and removes if/when necessary.
 *
 * @param   pg_disp_buff - the currently displaying display set PCS info.
 * @param   pg_comp_buff - the currently compiling display set PCS info.
 *

 *
 */
static BOOLEAN WaitToDisplay(U8BIT path, S_PAGE_COMPOSITION *pg_disp_buff, S_PAGE_COMPOSITION *pg_comp_buff)
{
   U32BIT time_ms;
   U32BIT pg_timestamp;
   U32BIT stc_timestamp;
   U32BIT difference_seconds;
   U32BIT waitdelay;
   U8BIT stc[5];
   BOOLEAN display;

   FUNCTION_START(WaitToDisplay);

   ASSERT(pg_disp_buff != NULL);
   ASSERT(pg_comp_buff != NULL);

   display = TRUE;

   // PTS values are used discounting the least signifiant byte, leaving 4 bytes using
   // 25 bits not the full 33 bits.
   pg_timestamp = ((pg_comp_buff->pts[0] << 24) +
                   (pg_comp_buff->pts[1] << 16) +
                   (pg_comp_buff->pts[2] << 8) +
                   (pg_comp_buff->pts[3]));

   STB_AVGetSTC(STB_DPGetPathVideoDecoder(path), stc);
   stc_timestamp = ((stc[0] << 24) + (stc[1] << 16) + (stc[2] << 8) + stc[3]);

   if (!time_diff_set && (stc_timestamp != 0) && (pg_timestamp != 0))
   {
      if ((stc[0] == 0) && (pg_comp_buff->pts[0] == 1))
      {
         /* Top bit of STC isn't present so make sure PTS doesn't have it set either */
         pg_timestamp &= 0x00ffffff;
      }
      else if ((stc[0] == 1) && (pg_comp_buff->pts[0] == 0))
      {
         /* Top bit of STC is present so make sure PTS hae it set too */
         pg_timestamp |= 0x01000000;
      }

      time_diff_value = 0;

      if (pg_timestamp > stc_timestamp)
      {
         if ((pg_timestamp - stc_timestamp) > 0x10000)
         {
            time_diff_value = (pg_timestamp & 0xffffff00) - (stc_timestamp & 0xffffff00);
         }
      }
      else
      {
         if ((stc_timestamp - pg_timestamp) > 0x10000)
         {
            time_diff_value = (stc_timestamp & 0xffffff00) - (pg_timestamp & 0xffffff00);
         }
      }

      time_diff_set = TRUE;
   }

   if (time_diff_set)
   {
      if (stc[0] == 0)
      {
         /* Top bit of STC isn't present so make sure PTS doesn't have it set either */
         pg_timestamp &= 0x00ffffff;
      }

      if (pg_timestamp > stc_timestamp)
      {
         if ((pg_timestamp - stc_timestamp) > 0x10000)
         {
            pg_timestamp -= time_diff_value;
         }
      }
      else
      {
         if ((stc_timestamp - pg_timestamp) > 0x10000)
         {
            pg_timestamp += time_diff_value;
         }
      }
   }

   while (1)
   {
      time_ms = STB_OSGetClockMilliseconds();

      if (terminate_subtitles)
      {
         terminate_subtitles = FALSE;
         break;
      }

      if (DetectPcrDiscontinuity(path))
      {
         break;
      }

      // Check timeout of the displayed display set, in seconds. Rounds up to the next second.
      difference_seconds = (STB_OSGetClockDiff(pg_disp_buff->timeout_start) / 1000) + 1;

      if ((pg_disp_buff->display_set_shown) && (!pg_disp_buff->display_set_removed))
      {
         if (pg_disp_buff->page_time_out <= difference_seconds)
         {
            // Clear display set
            STB_TIMING_PRINT(("$ WaitToDisplay @ %d, expired clear OSD", __LINE__));
            HideAllShownRegions(pg_disp_buff->region_list);
            STB_OSDUpdateRegions();
            pg_disp_buff->display_set_removed = TRUE;
         #ifdef TIMINGS
            diff_ms = STB_OSGetClockDiff(shown_time);
            do_timing = FALSE;
         #endif
         }
         else
         {
            STB_TIMING_PRINT(("$ WaitToDisplay @ %d, %d > %d",
                              __LINE__, pg_disp_buff->page_time_out, difference_seconds));
         }
      }
      else
      {
         STB_TIMING_PRINT(("$ WaitToDisplay @ %d, !shown || removed", __LINE__));
      }

      STB_AVGetSTC(STB_DPGetPathVideoDecoder(path), stc);
      stc_timestamp = ((stc[0] << 24) + (stc[1] << 16) + (stc[2] << 8) + stc[3]);

#if 0
      STB_TIMING_PRINT(("Time=%lu, pres_time=%lu, PTS=0x%lx, STC=0x%lx, PTS-STC=%ld\n", time_ms,
         pg_comp_buff->presentation_time_ms, pg_timestamp, stc_timestamp,
         (S32BIT)(pg_timestamp - stc_timestamp)));
#endif

      // As page PTS is some time in the future, if the current AV PTS is greater
      // than or equal to; then its time to display page
      STB_TIMING_PRINT(("%d >? %d ~%d  ||  %d >?= %d ~%d",
                        stc_timestamp, pg_timestamp, (stc_timestamp >= pg_timestamp),
                        time_ms, pg_comp_buff->presentation_time_ms,
                        (time_ms >= pg_comp_buff->presentation_time_ms)));

      if (stc_timestamp == 0)
      {
         display = FALSE;
         break;
      }

      if ((stc_timestamp >= pg_timestamp) || (time_ms >= pg_comp_buff->presentation_time_ms))
      {
         STB_TIMING_PRINT(("$ WaitToDisplay, PTS now"));
         break;
      }

      /* Make sure the delay isn't too long */
      if ((waitdelay = pg_comp_buff->presentation_time_ms - time_ms) > 250)
      {
         waitdelay = 250;
      }

      STB_OSTaskDelay(waitdelay);
   }

   FUNCTION_FINISH(WaitToDisplay);

   return(display);
}

/**
 *

 *
 * @brief   Hides all currently shown subtitle regions.  For each region in the list passed
 *                 find the physical structure and hide.
 *
 * @param   region_list - list of displayed regions.
 *

 *
 */
static void HideAllShownRegions(S_REGION *region_list)
{
   S_PHYSICAL_REGION *physical_region;
   S_REGION *region;

   FUNCTION_START(HideAllShownRegions);

   if ((region_list != NULL) && (physical_display_region_list != NULL))
   {
      region = region_list;
      while (region != NULL)
      {
         physical_region = physical_display_region_list;
         while (physical_region != NULL)
         {
            ASSERT(region != NULL);

            if ((physical_region->region_id == region->region_id) && physical_region->visible)
            {
               STB_DISPLAY_PRINT(("%s: Hide region pr%d, %x", __FUNCTION__, physical_region->region_id,
                                  physical_region->handle));

               STB_OSDHideRegion(physical_region->handle);
               physical_region->visible = FALSE;
               break;
            }
            physical_region = physical_region->next;
         }
         region = region->next;
      }
   }

   FUNCTION_FINISH(HideAllShownRegions);
}

/**
 *

 *
 * @brief   Swaps the pointers for the params & the physical buffers.
 *
 * @param   buff_1 - page composition structure
 * @param   buff_2 - page composition structure
 *

 *
 */
static void SwapDisplayAndCompositionBuffers(S_PAGE_COMPOSITION **buff_1, S_PAGE_COMPOSITION **buff_2)
{
   S_PAGE_COMPOSITION *tmp_pg;
   S_PHYSICAL_REGION *temp;

   FUNCTION_START(SwapDisplayAndCompositionBuffers);

   ASSERT(*buff_1 != NULL);
   ASSERT(*buff_2 != NULL);

   //STB_DISPLAY_PRINT(("Display & comp buffers swapped"));

   if ((*buff_1)->dds_present && ((*buff_1)->dds_version != (*buff_2)->dds_version))
   {
      (*buff_2)->dds_present = (*buff_1)->dds_present;
      (*buff_2)->dds_version = (*buff_1)->dds_version;
      (*buff_2)->display_width = (*buff_1)->display_width;
      (*buff_2)->display_height = (*buff_1)->display_height;
      (*buff_2)->display_window = (*buff_1)->display_window;
      if ((*buff_1)->display_window)
      {
         (*buff_2)->window_x = (*buff_1)->window_x;
         (*buff_2)->window_y = (*buff_1)->window_y;
         (*buff_2)->window_width = (*buff_1)->window_width;
         (*buff_2)->window_height = (*buff_1)->window_height;
      }
   }

   tmp_pg = *buff_1;
   *buff_1 = *buff_2;
   *buff_2 = tmp_pg;

   // Now swap the physical buffers to match
   temp = physical_display_region_list;
   physical_display_region_list = physical_composition_region_list;
   physical_composition_region_list = temp;

   FUNCTION_FINISH(SwapDisplayAndCompositionBuffers);
}

/**
 *

 *
 * @brief   Called when a region is being moved from the current display list to the
 *                 soon to be shown composition list.  We need the region for the epoch, but
 *                 it is currently visible - should it be hidden?
 *

 *

 *
 */
static void CheckToHideCopiedRegion(S_PHYSICAL_REGION *physical_region, S_REGION *region_list)
{
   S_REGION *region = region_list;

   while ((region != NULL) && (region->region_id != physical_region->region_id))
   {
      region = region->next;
   }
   //either we found a match, or we ran out of regions

   if (region == NULL)
   {
      //We didn't find a matching region in the PCS list => we should not be showning this region
      STB_DISPLAY_PRINT(("%s: Hide region pr%d, %x", __FUNCTION__, physical_region->region_id,
                         physical_region->handle));

      STB_OSDHideRegion(physical_region->handle);
      physical_region->visible = FALSE;
   }
}

/**
 *

 *
 * @brief   For double buffering, this function moves any regions appearing in the epoch
 *                 list, which do not already exist in the composition list, but do exist in the
 *                 display list, from the display list into the composition list
 *

 *

 *
 */
static void CompleteCompositionList(S_DISPLAY_SET *subtitle_display_set)
{
   S_PHYSICAL_REGION *display_region;
   S_PHYSICAL_REGION *prev_display_region;
   S_PHYSICAL_REGION *composition_region;
   S_EPOCH_REGION *legal_region;

   FUNCTION_START(CompleteCompositionList);

   // dump_physical_regions("PRE-MOVE");

   // Iterate over EPOCH's master region list
   legal_region = subtitle_display_set->region_list;

   while (legal_region != NULL)
   {
      composition_region = physical_composition_region_list;

      while (composition_region != NULL)
      {
         if (legal_region->region_id == composition_region->region_id)
         {
            break;
         }
         composition_region = composition_region->next;
      }

      if (composition_region == NULL)
      {
         // We don't currently have this region in the compose list - see if we can grab it
         // from the displayed list
         display_region = physical_display_region_list;
         prev_display_region = NULL;

         while (display_region != NULL)
         {
            // Remember who pointed to the display region
            if (legal_region->region_id == display_region->region_id)
            {
               // Move it across!
               if (prev_display_region == NULL)
               {
                  // display region is first in display list, amend list
                  physical_display_region_list = display_region->next;
               }
               else
               {
                  prev_display_region->next = display_region->next;
               }
               display_region->next = physical_composition_region_list;
               physical_composition_region_list = display_region;

               // We may want to retain the region, but should it be visible?
               if (display_region->visible)
               {
                  CheckToHideCopiedRegion(display_region,
                     subtitle_display_set->page_composition_buffer->region_list);
               }
               break;
            }
            prev_display_region = display_region;
            display_region = display_region->next;
         }
      }
      // else - Use new region

      legal_region = legal_region->next;
   }

   // dump_physical_regions("POST-MOVE");

   FUNCTION_FINISH(CompleteCompositionList);
}

/**
 *

 *
 * @brief   For the provided list each region is positioned at the desired location.
 *
 * @param   region_list - list a required regions
 *

 *
 */
static void MoveRegions(S_REGION *region_list)
{
   S_REGION *region;
   S_PHYSICAL_REGION *physical_composition_region;

   U16BIT region_id;


   FUNCTION_START(MoveRegions);

   if (region_list != NULL)
   {
      if (physical_composition_region_list != NULL)
      {
         // For each region listed in the PCS...
         region = region_list;
         while (region != NULL)
         {
            region_id = region->region_id;

            // Find PCS regions' physical composition region from list...
            physical_composition_region = physical_composition_region_list;
            while (physical_composition_region != NULL)
            {
               if (physical_composition_region->region_id == region_id)
               {
                  if ((physical_composition_region->current_x != region->region_horizontal_address) ||
                      (physical_composition_region->current_y != region->region_vertical_address))
                  {
                     STB_OSDMoveRegion(physical_composition_region->handle,
                        region->region_horizontal_address,
                        region->region_vertical_address);
                     physical_composition_region->current_x = region->region_horizontal_address;
                     physical_composition_region->current_y = region->region_vertical_address;
                     STB_DISPLAY_PRINT(("Move region %d (%x)",
                                        region->region_id, physical_composition_region->handle));
                  }
                  break;
               }
               physical_composition_region = physical_composition_region->next;
            }
            region = region->next;
         } //while(region != NULL)
      }
      else if (physical_display_region_list != NULL)
      {
         // For each region listed in the PCS...
         region = region_list;
         while (region != NULL)
         {
            region_id = region->region_id;

            // Find PCS regions' physical composition region from list...
            physical_composition_region = physical_display_region_list;
            while (physical_composition_region != NULL)
            {
               if (physical_composition_region->region_id == region_id)
               {
                  if ((physical_composition_region->current_x != region->region_horizontal_address) ||
                      (physical_composition_region->current_y != region->region_vertical_address))
                  {
                     STB_OSDMoveRegion(physical_composition_region->handle,
                        region->region_horizontal_address,
                        region->region_vertical_address);
                     physical_composition_region->current_x = region->region_horizontal_address;
                     physical_composition_region->current_y = region->region_vertical_address;
                     STB_DISPLAY_PRINT(("Move region %d (%x)",
                                        region->region_id, physical_composition_region->handle));
                  }
                  break;
               }
               physical_composition_region = physical_composition_region->next;
            }
            region = region->next;
         } //while(region != NULL)
      }
   }

   FUNCTION_FINISH(MoveRegions);
}

/**
 *

 *
 * @brief   Updates the display set, whether displaying the next or removing the previous.
 *                 Replaces the displayed display set with the composed display set.
 *
 * @param   subtitle_display_set - both composition and display buffers.
 *

 *
 */
static void UpdateDisplaySet(U8BIT path, S_DISPLAY_SET *subtitle_display_set)
{
   FUNCTION_START(UpdateDisplaySet);

   ASSERT(subtitle_display_set != NULL);

   USE_UNWANTED_PARAM(path);

// dump_physical_regions("Pre update display");

#ifdef TIMINGS
   if (do_timing)
   {
      diff_ms = STB_OSGetClockDiff(shown_time);
      do_timing = FALSE;
   }

   if (subtitle_display_set->page_display_buffer->region_list != NULL)
   {
      STB_TIMING_PRINT(("# pg.ver %02d shown for %04dms    %04dms late    ms_clk 0x%08x",
                        subtitle_display_set->page_display_buffer->page_version_number,
                        diff_ms, delay, STB_OSGetClockMilliseconds()));
   }
   else
   {
      STB_TIMING_PRINT(("  pg.ver %02d shown for %04dms    %04dms late    ms_clk 0x%08x",
                        subtitle_display_set->page_display_buffer->page_version_number,
                        diff_ms, delay, STB_OSGetClockMilliseconds()));
   }

   do_timing = TRUE;
#endif

   //If this is to be treated as a normal page...
   //Note dsfn.c code may force an acquisition point to be a mode change (by overriding page_state)
   if (subtitle_display_set->page_composition_buffer->page_state != MODE_CHANGE)
   {
      // Composition list contains all regions which we have built for this page
      // We need to grab older regions for this Page's epoch list from the now
      // undisplayed displayed list
      CompleteCompositionList(subtitle_display_set);
   }
   // else - we have been told to ignore history

   // Hide all regions remaining in display list - if they were needed and legal they'd already
   // have been moved into the composition list
   HideAllShownRegions(subtitle_display_set->page_display_buffer->region_list);
   subtitle_display_set->page_display_buffer->display_set_removed = TRUE;

   SwapDisplayAndCompositionBuffers(&subtitle_display_set->page_composition_buffer,
      &subtitle_display_set->page_display_buffer);

   ShowDisplaySet(subtitle_display_set->page_display_buffer->region_list);

   subtitle_display_set->page_display_buffer->display_set_shown = TRUE;
   subtitle_display_set->page_display_buffer->timeout_start = STB_OSGetClockMilliseconds();

   // All regions remaining in the old display list (now the composition list) can be freed
   STB_DSResetPhysicalCompositionRegions();

   STB_OSDUpdateRegions();

#ifdef TIMINGS
   {
      U8BIT stc[5];
      U32BIT stc_timestamp;
      U32BIT pg_timestamp;

      pg_timestamp = ((subtitle_display_set->page_display_buffer->pts[0] << 24) +
                      (subtitle_display_set->page_display_buffer->pts[1] << 16) +
                      (subtitle_display_set->page_display_buffer->pts[2] << 8) +
                      (subtitle_display_set->page_display_buffer->pts[3]));

      shown_time = STB_OSGetClockMilliseconds();
      STB_AVGetSTC(STB_DPGetPathVideoDecoder(path), stc);
      stc_timestamp = ((stc[0] << 24) + (stc[1] << 16) + (stc[2] << 8) + stc[3]);
      delay = ((((S32BIT)stc_timestamp - (S32BIT)pg_timestamp) << 8) / 90);
      do_timing = TRUE;
   }
#endif
// dump_physical_regions("Post update display");

   FUNCTION_FINISH(UpdateDisplaySet);
}

/**
 *

 *
 * @brief   Creates a bitmap in 2bits from 8bit image past, into the memory area the calling
 *                 function provides.
 *
 * @param   region_adjusted_bitmap - memory for reduced depth image to be placed.
 * @param   bitmap                 - the original 8bit bitmap.
 * @param   width                  - the width in pixels of the image
 * @param   height                 - the height in pixels of the image
 *

 *
 */
static void ColourDepthAdjustTo2Bits(U8BIT **region_adjusted_bitmap, U8BIT *bitmap,
   U16BIT width, U16BIT height)
{
   U32BIT size;
   U32BIT i;
   U32BIT j;
   U8BIT tmp_byte;

   U8BIT *p2;
   U8BIT *p8;


   FUNCTION_START(ColourDepthAdjustTo2Bits);

   ASSERT(*region_adjusted_bitmap != NULL);
   ASSERT(bitmap != NULL);

   // ETS 300 743 9.2 8 to 2-bit reduction
   // Let the input value be represented by an 8-bit field, the individual
   // bits of which are called bi1, bi2, bi3, bi4, bi5, bi6, bi7 and bi8
   // where bi1 is received first and bi8 is received last. Let the output
   // value be represented by a 2-bit field bo1, bo2.
   // The relation between output and input bits is:
   // bo1 = bi1 : bo2 = bi2 | bi3 | bi4
   p2 = *region_adjusted_bitmap;
   p8 = bitmap;

   if (*region_adjusted_bitmap != NULL)
   {
      if (width % 4 == 0)
      {
         STB_PIXEL_PRINT(("To2 No padding (%dx%d) @ %d", width, height, __LINE__));
         // No padding
         size = height * width;
         for (i = 0; i < size; i += 4)
         {
            static U8BIT lastcolin = 0xff;

            if (lastcolin != *p8)
            {
               STB_PIXEL_PRINT(("    in %02x -cda2-> %02x",
                                *p8, (((*p8 & 0x80) >> 6) | ((*p8 & 0x70) != 0))));
               lastcolin = *p8;
            }

            tmp_byte = (((*p8 & 0x80) >> 6) | ((*p8 & 0x70) != 0)) << 6;
            p8++;
            tmp_byte |= (((*p8 & 0x80) >> 6) | ((*p8 & 0x70) != 0)) << 4;
            p8++;
            tmp_byte |= (((*p8 & 0x80) >> 6) | ((*p8 & 0x70) != 0)) << 2;
            p8++;
            tmp_byte |= (((*p8 & 0x80) >> 6) | ((*p8 & 0x70) != 0));
            p8++;
            *p2 = tmp_byte;
            p2++;
         }
      }
      else
      {
         STB_PIXEL_PRINT(("To2 Padding (%dx%d) @ %d", width, height, __LINE__));
         for (i = 0; i < height; i++)
         {
            for (j = 0; j <= width; j += 4)
            {
               if ((j + 3) <= width)
               {
                  tmp_byte = (((*p8 & 0x80) >> 6) | ((*p8 & 0x70) != 0)) << 6;
                  p8++;
                  tmp_byte |= (((*p8 & 0x80) >> 6) | ((*p8 & 0x70) != 0)) << 4;
                  p8++;
                  tmp_byte |= (((*p8 & 0x80) >> 6) | ((*p8 & 0x70) != 0)) << 2;
                  p8++;
                  tmp_byte |= (((*p8 & 0x80) >> 6) | ((*p8 & 0x70) != 0));
                  p8++;
               }
               else
               {
                  tmp_byte = (((*p8 & 0x80) >> 6) | ((*p8 & 0x70) != 0)) << 6;
                  p8++;

                  if ((j + 1) < width)
                  {
                     tmp_byte |= (((*p8 & 0x80) >> 6) | ((*p8 & 0x70) != 0)) << 4;
                     p8++;

                     if ((j + 2) < width)
                     {
                        tmp_byte |= (((*p8 & 0x80) >> 6) | ((*p8 & 0x70) != 0)) << 2;
                        p8++;
                     }
                  }
               }
               *p2 = tmp_byte;
               p2++;
            }
         }
      }
   }

   FUNCTION_FINISH(ColourDepthAdjustTo2Bits);
}

/**
 *

 *
 * @brief   Creates a bitmap in 4bits from 8bit image past, into the memory area the calling
 *                 function provides.
 *
 * @param   region_adjusted_bitmap - memory for reduced depth image to be placed.
 * @param   bitmap                 - the original 8bit bitmap.
 * @param   width                  - the width in pixels of the image
 * @param   height                 - the height in pixels of the image
 *

 *
 */
static void ColourDepthAdjustTo4Bits(U8BIT **region_adjusted_bitmap, U8BIT *bitmap,
   U16BIT width, U16BIT height)
{
   U32BIT size;
   U32BIT i;
   U32BIT j;
   U8BIT tmp_byte;

   U8BIT *p4;
   U8BIT *p8;


   FUNCTION_START(ColourDepthAdjustTo4Bits);

   ASSERT(*region_adjusted_bitmap != NULL);
   ASSERT(bitmap != NULL);

   // ETS 300 743 9.3 8 to 4-bit reduction
   // Let the input value be represented by a 8-bit field, the individual bits of which are called
   // bi1, bi2, bi3, bi4, bi5, bi6, bi7 and bi8 where bi1 is received first and bi8 is received
   // last. Let the output value be represented by a 4-bit field bo1 to bo4. The relation between
   // output and input bits is: bo1 = bi1 : bo2 = bi2 : bo3 = bi3 : bo4 = bi4
   if (*region_adjusted_bitmap != NULL)
   {
      p4 = *region_adjusted_bitmap;
      p8 = bitmap;

      // Reduce 8 bit to 4 bit bitmap
      if (width % 2 == 0)
      {
         STB_PIXEL_PRINT(("To4 No Padding (%dx%d) @ %d", width, height, __LINE__));
         // No padding
         size = height * width;
         for (i = 0; i < size; i += 2)
         {
            STB_PIXEL_PRINT(("    in %02x + %02x -cda4-> %02x @ %x",
                             *p8, *(p8 + 1), ((*p8 & 0xf0) | (((*(p8 + 1) & 0xf0)) >> 4)), p4));

            tmp_byte = (*p8 & 0xf0);
            p8++;
            tmp_byte |= ((*p8 & 0xf0) >> 4);
            p8++;
            *p4 = tmp_byte;
            p4++;
         }
      }
      else
      {
         STB_PIXEL_PRINT(("To4 Padding (%dx%d) @ %d", width, height, __LINE__));
         for (i = 0; i < height; i++)
         {
            for (j = 0; j <= width; j += 2)
            {
               STB_PIXEL_PRINT(("    in %02x + %02x -cda4-> %02x @ %x",
                                *p8, *(p8 + 1), ((*p8 & 0xf0) | (((*(p8 + 1) & 0xf0)) >> 4)), p4));

               if ((j + 1) < width)
               {
                  tmp_byte = (*p8 & 0xf0);
                  p8++;
                  tmp_byte |= ((*p8 & 0xf0) >> 4);
                  p8++;
               }
               else
               {
                  tmp_byte = (*p8 & 0xf0);
                  p8++;
               }
               *p4 = tmp_byte;
               p4++;
            }
         }
      }
   }

   FUNCTION_FINISH(ColourDepthAdjustTo4Bits);
}

/**
 *

 *
 * @brief   Loads the CLUT into regions
 *
 * @param   region_list - list of regions to load CLUTs for.
 * @param   clut_list   - list of CLUTs available.
 *

 *
 */
static void LoadPalettesIntoRegions(S_EPOCH_REGION *region_list, S_CLUT *clut_list)
{
   S_PHYSICAL_REGION *physical_region;
   S_EPOCH_REGION *region;
   S_CLUT *clut;


   FUNCTION_START(LoadPalettesIntoRegions);

   physical_region = physical_composition_region_list;
   while (physical_region != NULL)
   {
      region = region_list;

      while (region)
      {
         ASSERT(region_list != NULL);

         if (physical_region->region_id == region->region_id)
         {
            clut = STB_DSGetClut(clut_list, region->region_clut_id);

            ASSERT(clut_list != NULL);

            switch (physical_region->region_colour_depth)
            {
               case 2:
               {
                  STB_OSDSetYCrCbPalette(physical_region->handle, clut->clut_2_bit);
                  break;
               }

               case 4:
               {
                  STB_OSDSetYCrCbPalette(physical_region->handle, clut->clut_4_bit);
                  break;
               }

               case 8:
               {
                  STB_OSDSetYCrCbPalette(physical_region->handle, clut->clut_8_bit);
                  break;
               }
            }
            break;
         }
         region = region->next;
      }
      physical_region = physical_region->next;
   }

   FUNCTION_FINISH(LoadPalettesIntoRegions);
}

/**
 *

 *
 * @brief   Shows the OSD regions for the region list passed.
 *
 * @param   region_list - list of regions to show.
 *

 *
 */
static void ShowDisplaySet(S_REGION *region_list)
{
   S_REGION *region;
   S_PHYSICAL_REGION *physical_display_region;
   U16BIT region_id;

   FUNCTION_START(ShowDisplaySet);

   if ((region_list != NULL) && (physical_display_region_list != NULL))
   {
      // For each region listed in the PCS...
      region = region_list;
      while (region != NULL)
      {
         region_id = region->region_id;

         // Find PCS regions' physical region from list...
         physical_display_region = physical_display_region_list;
         while (physical_display_region != NULL)
         {
            if (physical_display_region->region_id == region_id)
            {
               if (show_subtitles && !physical_display_region->visible)
               {
                  STB_OSDShowRegion(physical_display_region->handle);
                  STB_DISPLAY_PRINT(("   Show region %d, %x", physical_display_region->region_id,
                                     physical_display_region->handle));
                  physical_display_region->visible = TRUE;
               }
               break;
            }
            physical_display_region = physical_display_region->next;
         }
         region = region->next;
      } //while(region != NULL)
   }

   FUNCTION_FINISH(ShowDisplaySet);
}

//---global function definitions-----------------------------------------------

/**
 *

 *
 * @brief   It checks the timeout of the current display set, and removes if/when necessary.
 *
 * @param   subtitle_display_set - subtitle display structure
 *

 *
 */
void STB_DSCheckDisplaySetTimeout(S_DISPLAY_SET *subtitle_display_set, BOOLEAN timeout_override)
{
   U32BIT difference_seconds;

   FUNCTION_START(STB_DSCheckDisplaySetTimeout);

   ASSERT(subtitle_display_set != NULL);

   if ((subtitle_display_set->page_display_buffer->region_list != NULL) &&
       (subtitle_display_set->page_display_buffer->display_set_shown) &&
       (!subtitle_display_set->page_display_buffer->display_set_removed))
   {
      difference_seconds = (STB_OSGetClockDiff(subtitle_display_set->page_display_buffer->timeout_start) / 1000) + 1;

      if ((subtitle_display_set->page_display_buffer->page_time_out <= difference_seconds) || timeout_override)
      {
         // Clear display set
         STB_TIMING_PRINT(("  CheckDisplay, cleared OSD"));
         HideAllShownRegions(subtitle_display_set->page_display_buffer->region_list);
         subtitle_display_set->page_display_buffer->display_set_removed = TRUE;
         STB_OSDUpdateRegions();
      }
   }

   FUNCTION_FINISH(STB_DSCheckDisplaySetTimeout);
}

/**
 *

 *
 * @brief   The subtitle display function. Provides timing of display and clean up when
 *                 necessary.
 *
 * @param   subtitle_display_set - the complete subtitle display buffers, vis & composition
 *

 *
 */
void STB_DSDisplay(U8BIT path, S_DISPLAY_SET *subtitle_display_set )
{
   FUNCTION_START(STB_DSDisplay);

   ASSERT(subtitle_display_set != NULL);

   if (!subtitle_display_set->page_composition_buffer->display_set_shown)
   {
      /* Set display size so that subtitle regions can be scaled correctly */
      STB_OSDSetRegionDisplaySize(subtitle_display_set->page_composition_buffer->display_width + 1,
         subtitle_display_set->page_composition_buffer->display_height + 1);

      LoadPalettesIntoRegions(subtitle_display_set->region_list, subtitle_display_set->clut_list);

      MoveRegions(subtitle_display_set->page_composition_buffer->region_list);

      if (WaitToDisplay(path, subtitle_display_set->page_display_buffer,
             subtitle_display_set->page_composition_buffer))
      {
         UpdateDisplaySet(path, subtitle_display_set);
      }
   }

   FUNCTION_FINISH(STB_DSDisplay);
}

/**
 *

 *
 * @brief   Registers a STBUI function to use whatever font tool is present [if any] for the
 *                 creation of a bitmap of characters.
 *
 * @param   DSCreateBitmap - function pointer to the callback function.
 *

 *
 */
void STB_DSRegisterCharRenderFunction(void (*DSCreateBitmap)(U8BIT *bitmap, U16BIT width, U16BIT height, U8BIT *char_array,
      U8BIT *tycrcb_palette, U8BIT fgnd_col, U8BIT bkgnd_col))
{
   FUNCTION_START(STB_DSRegisterCharRenderFunction);

   ASSERT(DSCreateBitmap != NULL);

   CreateBitmapCallback = DSCreateBitmap;

   FUNCTION_FINISH(STB_DSRegisterCharRenderFunction);
}

/**
 *

 *
 * @brief   Allows subtitles to display what it has decoded.
 *

 *

 *
 */
void STB_DSShow(void)
{
   FUNCTION_START(STB_DSShow);

   show_subtitles = TRUE;
   time_diff_set = FALSE;
   STB_OSDDisableUIRegion();

   FUNCTION_FINISH(STB_DSShow);
}

/**
 *

 *
 * @brief   Prevents subtitles to display what it has decoded.
 *

 *

 *
 */
void STB_DSHide(void)
{
   FUNCTION_START(STB_DSHide);

   show_subtitles = FALSE;
   time_diff_set = FALSE;
   STB_OSDEnableUIRegion();

   FUNCTION_FINISH(STB_DSHide);
}

/**
 *

 *
 * @brief   Deletes all regions and clears serice_aquired flag.
 *

 *

 *
 */
void STB_DSResetPhysicalDisplayRegions(void)
{
   S_PHYSICAL_REGION *physical_region;
   S_PHYSICAL_REGION *temp;

   FUNCTION_START(STB_DSResetPhysicalDisplayRegions);

   physical_region = physical_display_region_list;
   while (physical_region != NULL)
   {
      STB_DISPLAY_PRINT(("    STB_DSResetPhysicalDisplayRegions pr%d %x", physical_region->region_id,
                         physical_region->handle));
      STB_OSDDestroyRegion(physical_region->handle);
      temp = physical_region->next;
      STB_FreeMemory(physical_region);
      physical_region = temp;
   }
   physical_display_region_list = NULL;

   FUNCTION_FINISH(STB_DSResetPhysicalDisplayRegions);
}

/**
 *

 *
 * @brief   Deletes all regions and clears serice_aquired flag.
 *

 *

 *
 */
void STB_DSResetPhysicalCompositionRegions(void)
{
   S_PHYSICAL_REGION *physical_region;
   S_PHYSICAL_REGION *temp;

   FUNCTION_START(STB_DSResetPhysicalCompositionRegions);

   physical_region = physical_composition_region_list;
   while (physical_region != NULL)
   {
      STB_DISPLAY_PRINT(("    STB_DSResetPhysicalCompositionRegions pr%d %x", physical_region->region_id,
                         physical_region->handle));
      STB_OSDDestroyRegion(physical_region->handle);
      temp = physical_region->next;
      STB_FreeMemory(physical_region);
      physical_region = temp;
   }
   physical_composition_region_list = NULL;

   FUNCTION_FINISH(STB_DSResetPhysicalCompositionRegions);
}

/**
 *

 *
 * @brief   Sets flag to exit display wait while loop.
 *

 *

 *
 */
void STB_DSTerminateDisplayCycle(void)
{
   FUNCTION_START(STB_DSTerminateDisplayCycle);

   terminate_subtitles = TRUE;

   FUNCTION_FINISH(STB_DSTerminateDisplayCycle);
}

/**
 *

 *
 * @brief   By end of call ensures that the specified region has a suitable physical region
 *                 waiting in the physical composition list
 *
 * @param   region     - the epoch region definition.
 *                 page_reset - true => disregard old content (mode_change / needed acquisition)
 *

 *
 */
void STB_DSCreateCompositionRegion(S_EPOCH_REGION *region, BOOLEAN page_reset)
{
   //The physical region to be used for the specified epoch region
   S_PHYSICAL_REGION *physical_region = NULL;
   //Pointer to existing, older, region if one exists
   S_PHYSICAL_REGION *reference_region = NULL;


   FUNCTION_START(STB_DSCreateCompositionRegion);

   ASSERT(region != NULL);

   if (page_reset)
   {
      //Not using anything old!
      physical_region = NULL;
   }
   else
   {
      //Look in what's currently being displayed
      physical_region = physical_display_region_list;
   }

   while (physical_region != NULL)
   {
      if (physical_region->region_id == region->region_id)
      {
         reference_region = physical_region;
         break;
      }
      physical_region = physical_region->next;
   }

   if ((physical_region != NULL) &&
       ((physical_region->region_width != region->region_width) ||
        (physical_region->region_height != region->region_height) ||
        (physical_region->region_colour_depth != region->region_colour_depth)))
   {
      //We found an old region that has incompatible parameters
      //This is a sign that we've missed a mode change / acquisition point that we needed
      //Or the stream didn't supply one...
      STB_ERROR_PRINT(("!! STB_DSCreateCompositionRegion - SIZE MISMATCH %s @ %d !!",
                       __FILE__, __LINE__));
      //Make a new one!
      physical_region = NULL;
      //Don't copy from the old one
      reference_region = NULL;
   }

   if ((physical_region == NULL) ||
       (physical_region->region_version_number != region->region_version_number))
   {
      // We're going to need a new region, either because no previous one existed,
      // OR we're going to need a new region due to a new version
      // OR old (illegal) one was the wrong size

      physical_region = (S_PHYSICAL_REGION *)STB_GetMemory(sizeof(S_PHYSICAL_REGION));

      if (physical_region != NULL)
      {
         physical_region->handle = STB_OSDCreateRegion(region->region_width,
               region->region_height,
               region->region_colour_depth);

         if (physical_region->handle != 0)
         {
            physical_region->next = physical_composition_region_list;
            physical_composition_region_list = physical_region;

            physical_region->region_id = region->region_id;
            physical_region->region_version_number = region->region_version_number;
            physical_region->region_width = region->region_width;
            physical_region->region_height = region->region_height;
            physical_region->region_colour_depth = region->region_colour_depth;

            physical_region->current_x = 0;
            physical_region->current_y = 0;

            physical_region->visible = FALSE;

            if (region->region_fill_flag)
            {
               //We're using a fresh region, which we need to fill
               STB_DISPLAY_PRINT(("    STB_DSCreateCompositionRegion pr%d (%dx%d)  NEW %x - FILL",
                                  region->region_id, region->region_width, region->region_height,
                                  physical_region->handle));
               switch (region->region_colour_depth)
               {
                  case 2:
                     STB_DSFillRegion(region->region_id, region->region_2_bit_pixel_code);
                     break;
                  case 4:
                     STB_DSFillRegion(region->region_id, region->region_4_bit_pixel_code);
                     break;
                  case 8:
                     STB_DSFillRegion(region->region_id, region->region_8_bit_pixel_code);
                     break;
               }
               // Don't fill again
               region->region_fill_flag = FALSE;
            }
            //Now did we have an original region (whose data we need to copy over)?
            else if (reference_region != NULL)
            {
               STB_DISPLAY_PRINT(("    STB_DSCreateCompositionRegion r%d (%dx%d)  Copy %x <- %x",
                                  region->region_id, region->region_width, region->region_height,
                                  physical_region->handle, reference_region->handle));
               STB_OSDRegionToRegionCopy(physical_region->handle, reference_region->handle);
            }
            else
            {
               STB_DISPLAY_PRINT(("    STB_DSCreateCompositionRegion pr%d (%dx%d)  NEW %x - no fill",
                                  region->region_id, region->region_width, region->region_height,
                                  physical_region->handle));
            }
            //else - We didn't need to fill the new region
         }
         else
         {
            // Couldn't get OSD resource
            STB_FreeMemory(physical_region);
            physical_region = NULL;
            STB_ERROR_PRINT(("!! STB_DSCreateCompositionRegion - STB_OSDCreateRegion failed %s @ %d !!",
                             __FILE__, __LINE__));
         }
      }
      else
      {
         // Couldn't get OSD resource
         STB_ERROR_PRINT(("!! STB_DSCreateCompositionRegion - STB_GetMemory failed %s @ %d !!",
                          __FILE__, __LINE__));
      }
   }
   else
   {
      physical_region->region_version_number = region->region_version_number;

      STB_DISPLAY_PRINT(("    STB_DSCreateCompositionRegion pr%d (%dx%d)  Reuse",
                         region->region_id, region->region_width, region->region_height));

      //Move region over to the composition list
      remove_entry_from_list(physical_region, &physical_display_region_list);
      physical_region->next = physical_composition_region_list;
      physical_composition_region_list = physical_region;
   }

   FUNCTION_FINISH(STB_DSCreateCompositionRegion);
}

/**
 *

 *
 * @brief   For the passed region list [PCS specified] get region handle, load the region
 *                 with the correct CLUT, move regions if needed, and for each region render
 *                 objects into.
 *
 * @param   region_list - pass region list to search and region bitmap into.
 * @param   object      - the new object needing rendering.
 * @param   bitmap      - the bitmap to render.
 * @param   y           - the y coordinate in region.
 * @param   w           - width of bitmap.
 * @param   h           - height of bitmap.
 *

 *
 */
void STB_DSRenderBitmapToRegion(S_EPOCH_REGION *region_list, S_OBJECT *object, U8BIT *bitmap,
   U16BIT y, U16BIT w, U16BIT h)
{
   S_PHYSICAL_REGION *physical_composition_region;
   S_EPOCH_REGION *region;
   S_REGION_OBJECT *region_object;
   U32BIT size;
   U8BIT *region_adjusted_bitmap;

   FUNCTION_START(STB_DSRenderBitmapToRegion);

   ASSERT(object != NULL);
   ASSERT(bitmap != NULL);

   region_adjusted_bitmap = NULL;

   region = region_list;
   while (region)
   {
      region_object = region->region_object_list;
      while (region_object)
      {
         if (region_object->object_id == object->object_id)
         {
            physical_composition_region = physical_composition_region_list;
            while (physical_composition_region != NULL)
            {
               if (physical_composition_region->region_id == region->region_id)
               {
                  switch (physical_composition_region->region_colour_depth)
                  {
                     case 2:
                     {
                        size = (h * (w + 4)) / 4;
                        region_adjusted_bitmap = STB_GetMemory(size);

                        ColourDepthAdjustTo2Bits(&region_adjusted_bitmap, bitmap, w, h);
                        break;
                     }

                     case 4:
                     {
                        size = (h * (w + 2)) / 2;
                        region_adjusted_bitmap = STB_GetMemory(size);

                        ColourDepthAdjustTo4Bits(&region_adjusted_bitmap, bitmap, w, h);
                        break;
                     }

                     case 8:
                     {
                        region_adjusted_bitmap = bitmap;
                        break;
                     }
                  }

                  if ((region_object->object_horizontal_position < physical_composition_region->region_width) &&
                     (region_object->object_vertical_position + y < physical_composition_region->region_height))
                  {
                     if ((region_object->object_horizontal_position + w) > physical_composition_region->region_width)
                     {
                        STB_DISPLAY_PRINT(("    o%d Truncate w - %d + %d > %d ==> %d",
                           object->object_id,
                           region_object->object_horizontal_position,
                           w,
                           physical_composition_region->region_width,
                           physical_composition_region->region_width - region_object->object_horizontal_position));

                        w = physical_composition_region->region_width - region_object->object_horizontal_position;
                     }

                     if ((region_object->object_vertical_position + h) > physical_composition_region->region_height)
                     {
                        STB_DISPLAY_PRINT(("    o%d Truncate h - %d + %d > %d ==> %d",
                           object->object_id,
                           region_object->object_vertical_position,
                           h,
                           physical_composition_region->region_height,
                           physical_composition_region->region_height - region_object->object_vertical_position));

                        h = physical_composition_region->region_height - region_object->object_vertical_position;
                     }

                     if ((y < physical_composition_region->region_height) &&
                         ((region_object->object_vertical_position + y) < physical_composition_region->region_height))
                     {
                        if ((region_object->object_vertical_position + y + h) >
                            physical_composition_region->region_height)
                        {
                           h = (region_object->object_vertical_position + y + h) -
                              physical_composition_region->region_height;
                        }

                        if (h > 0)
                        {
                           physical_composition_region->region_version_number = region->region_version_number;

                           STB_OSDDrawBitmapInRegion(physical_composition_region->handle,
                              region_object->object_horizontal_position,
                              region_object->object_vertical_position + y,
                              w,
                              h,
                              region_adjusted_bitmap,
                              object->non_modifying_colour_flag);
                        }
                     }
                     else
                     {
                        STB_DISPLAY_PRINT(("    o%d Out of region's height", object->object_id));
                     }
                  }
                  else
                  {
                     STB_DISPLAY_PRINT(("    o%d Outside of region", object->object_id));
                  }

                  if ((physical_composition_region->region_colour_depth == 2) ||
                      (physical_composition_region->region_colour_depth == 4))
                  {
                     STB_FreeMemory(region_adjusted_bitmap);
                  }
                  break;
               }
               physical_composition_region = physical_composition_region->next;
            }
         }
         region_object = region_object->next;
      }
      region = region->next;
   }

   FUNCTION_FINISH(STB_DSRenderBitmapToRegion);
}

/**
 *

 *
 * @brief   Flood fills the identified region, with the specified colour index.
 *
 * @param   region_id - the regions id.
 * @param   fillcode  - the colour index in the CLUT to fill region with.
 *

 *
 */
void STB_DSFillRegion(U16BIT region_id, U8BIT fillcode)
{
   S_PHYSICAL_REGION *physical_composition_region;


   FUNCTION_START(STB_DSFillRegion);

   physical_composition_region = physical_composition_region_list;
   while (physical_composition_region != NULL)
   {
      if (physical_composition_region->region_id == region_id)
      {
         STB_DISPLAY_PRINT(("    STB_DSFillRegion r%d (%x) with %d",
                            region_id, physical_composition_region->handle, fillcode));
         STB_OSDFillRegion(physical_composition_region->handle, fillcode);
         break;
      }
      physical_composition_region = physical_composition_region->next;
   }

   FUNCTION_FINISH(STB_DSFillRegion);
}

/**
 *

 *
 * @brief   Calculates the number of individual pixels in volved in the next display set.
 *
 * @param   The complete display set structure.
 *
 * @return   Count of pixels
 *
 */
U32BIT STB_DSNumPixelOperations(S_DISPLAY_SET *subtitle_display_set)
{
   S_EPOCH_REGION *temp;
   S_REGION *tmp;
   U32BIT pixel_operations;


   FUNCTION_START(STB_DSNumPixelOperations);

   pixel_operations = 0;

   // removal of existing display set
   tmp = subtitle_display_set->page_display_buffer->region_list;
   while (tmp != NULL)
   {
      temp = subtitle_display_set->region_list;
      while (temp != NULL)
      {
         if (tmp->region_id == temp->region_id)
         {
            pixel_operations += temp->region_width * temp->region_height * temp->region_colour_depth;
            break;
         }
         temp = temp->next;
      }
      tmp = tmp->next;
   }

   // displaying of new display set
   tmp = subtitle_display_set->page_composition_buffer->region_list;
   while (tmp != NULL)
   {
      temp = subtitle_display_set->region_list;
      while (temp != NULL)
      {
         if (tmp->region_id == temp->region_id)
         {
            pixel_operations += temp->region_width * temp->region_height * temp->region_colour_depth;
            break;
         }
         temp = temp->next;
      }
      tmp = tmp->next;
   }

   FUNCTION_FINISH(STB_DSNumPixelOperations);

   return(pixel_operations);
}

static void remove_entry_from_list(S_PHYSICAL_REGION *entry, S_PHYSICAL_REGION **list)
{
   S_PHYSICAL_REGION *itor;

   if (*list == entry)
   {
      *list = entry->next;
   }
   else
   {
      itor = *list;

      while ((itor != NULL) && (itor->next != entry))
      {
         itor = itor->next;
      }

      if (itor != NULL)
      {
         itor->next = entry->next;
      }
      else
      {
         STB_ERROR_PRINT(("Couldn't find entry %d in list %s @ %d",
                          entry->region_id, __FILE__, __LINE__));
      }
   }
}

#ifdef STB_DISPLAY_PRINT_REQUIRED
void dump_physical_regions(const char *label)
{
   S_PHYSICAL_REGION *region;

   FUNCTION_START(dump_physical_regions);

   STB_DISPLAY_PRINT(("%s - Dump OSD resources\n  Display", label));
   region = physical_display_region_list;
   while (region != NULL)
   {
      STB_DISPLAY_PRINT(("    r%d (v%d) (w%d h%d d%d) @ (%d %d) %c",
                         (int) region->region_id,
                         (int) region->region_version_number,
                         (int) region->region_width,
                         (int) region->region_height,
                         (int) region->region_colour_depth,
                         (int) region->current_x,
                         (int) region->current_y,
                         (region->visible ? 'V' : '.')));
      region = region->next;
   }
   STB_DISPLAY_PRINT(("    <END>\n  Composition"));
   region = physical_composition_region_list;
   while (region != NULL)
   {
      STB_DISPLAY_PRINT(("    r%d (v%d) (w%d h%d d%d) @ (%d %d) %c",
                         (int) region->region_id,
                         (int) region->region_version_number,
                         (int) region->region_width,
                         (int) region->region_height,
                         (int) region->region_colour_depth,
                         (int) region->current_x,
                         (int) region->current_y,
                         (region->visible ? 'v' : '.')));
      region = region->next;
   }
   STB_DISPLAY_PRINT(("    <END>"));

   FUNCTION_FINISH(dump_physical_regions);
}

#endif

//*****************************************************************************
// End of file
//*****************************************************************************


