/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   DVB subtitle control processing functions
 *
 * @file    stbdsfn.c
 * @date    12/09/2003
 */

//---includes for this file----------------------------------------------------
// compiler library header files

#include <string.h>

// third party header files

// Ocean Blue Software header files

#include <techtype.h>
#include <dbgfuncs.h>

#include "stbhwos.h"
#include "stbhwav.h"
#include "stbhwosd.h"

#include "stbheap.h"
#include "stbds.h"
#include "stbdpc.h"


//---constant definitions for this file----------------------------------------

/* #define STB_SEGMENT_PRINT_REQUIRED */
/* #define STB_SUMMARY_PRINT_REQUIRED */
/* #define STB_SUB_DRIVER_PRINT_REQUIRED */
/* #define STB_REGION_PRINT_REQUIRED */
/* #define STB_ODS_PRINT_REQUIRED */
/* #define STB_OBJECT_PRINT_REQUIRED */
/* #define STB_PIXMAP_PRINT_REQUIRED */
/* #define STB_CDS_PRINT_REQUIRED */
/* #define STB_CLUT_PRINT_REQUIRED */
/* #define STB_END_PRINT_REQUIRED */

//Initialisation checking
#ifdef STB_SUB_DRIVER_PRINT_REQUIRED
   #ifdef STB_SUB_DRIVER_CHANNEL_DEBUG_REQUIRED
      #define STB_SUB_DRIVER_PRINT(x) DEBUG_PRINTX_CONDITIONAL(DEBUG_STB_OSD) x
   #else
      #define STB_SUB_DRIVER_PRINT(x) STB_SPDebugWrite x
   #endif
#else
   #define STB_SUB_DRIVER_PRINT(x)
#endif

//Top level information - brief description of PCS, RCS, ODS etc
//Intended to establish context for more detailed dump of individual segment types
#ifdef STB_SEGMENT_PRINT_REQUIRED
   #ifdef STB_SUB_DRIVER_CHANNEL_DEBUG_REQUIRED
      #define STB_SEGMENT_PRINT(x) DEBUG_PRINTX_CONDITIONAL(DEBUG_STB_OSD) x
   #else
      #define STB_SEGMENT_PRINT(x) STB_SPDebugWrite x
   #endif
#else
   #define STB_SEGMENT_PRINT(x)
#endif

//Gives summary of state on reception of EDS
#ifdef STB_SUMMARY_PRINT_REQUIRED
   #ifdef STB_SUB_DRIVER_CHANNEL_DEBUG_REQUIRED
      #define STB_SUMMARY_PRINT(x) DEBUG_PRINTX_CONDITIONAL(DEBUG_STB_OSD) x
   #else
      #define STB_SUMMARY_PRINT(x) STB_SPDebugWrite x
   #endif
#else
   #define STB_SUMMARY_PRINT(x)
#endif

//Check footprint
#ifdef STB_REGION_PRINT_REQUIRED
   #ifdef STB_REGION_CHANNEL_DEBUG_REQUIRED
      #define STB_REGION_PRINT(x) DEBUG_PRINTX_CONDITIONAL(DEBUG_STB_OSD) x
   #else
      #define STB_REGION_PRINT(x) STB_SPDebugWrite x
   #endif
#else
   #define STB_REGION_PRINT(x)
#endif

#ifdef STB_CDS_PRINT_REQUIRED
   #ifdef STB_CLUT_CHANNEL_DEBUG_REQUIRED
      #define STB_CDS_PRINT(x) DEBUG_PRINTX_CONDITIONAL(DEBUG_STB_OSD) x
   #else
      #define STB_CDS_PRINT(x) STB_SPDebugWrite x
   #endif
#else
   #define STB_CDS_PRINT(x)
#endif

#ifdef STB_CLUT_PRINT_REQUIRED
   #ifdef STB_CLUT_CHANNEL_DEBUG_REQUIRED
      #define STB_CLUT_PRINT(x) DEBUG_PRINTX_CONDITIONAL(DEBUG_STB_OSD) x
   #else
      #define STB_CLUT_PRINT(x) STB_SPDebugWrite x
   #endif
#else
   #define STB_CLUT_PRINT(x)
#endif

#ifdef STB_ODS_PRINT_REQUIRED
   #ifdef STB_ODS_CHANNEL_DEBUG_REQUIRED
      #define STB_ODS_PRINT(x) DEBUG_PRINTX_CONDITIONAL(DEBUG_STB_OSD) x
   #else
      #define STB_ODS_PRINT(x) STB_SPDebugWrite x
   #endif
#else
   #define STB_ODS_PRINT(x)
#endif

#ifdef STB_OBJECT_PRINT_REQUIRED
   #ifdef STB_OBJECT_CHANNEL_DEBUG_REQUIRED
      #define STB_OBJECT_PRINT(x) DEBUG_PRINTX_CONDITIONAL(DEBUG_STB_OSD) x
   #else
      #define STB_OBJECT_PRINT(x) STB_SPDebugWrite x
   #endif
#else
   #define STB_OBJECT_PRINT(x)
#endif

#ifdef STB_PIXMAP_PRINT_REQUIRED
   #ifdef STB_PIXMAP_CHANNEL_DEBUG_REQUIRED
      #define STB_PIXMAP_PRINT(x) DEBUG_PRINTX_CONDITIONAL(DEBUG_STB_OSD) x
   #else
      #define STB_PIXMAP_PRINT(x) STB_SPDebugWrite x
   #endif
#else
   #define STB_PIXMAP_PRINT(x)
#endif

#ifdef STB_END_PRINT_REQUIRED
   #ifdef STB_END_CHANNEL_DEBUG_REQUIRED
      #define STB_END_PRINT(x) DEBUG_PRINTX_CONDITIONAL(DEBUG_STB_OSD) x
   #else
      #define STB_END_PRINT(x) STB_SPDebugWrite x
   #endif
#else
   #define STB_END_PRINT(x)
#endif


#define TEMP_BUFFER_SIZE   (1024 * 2)
#define SCAN_LINE_WIDTH    1950

#define MAX_REGION_WIDTH   1920
#define MAX_REGION_HEIGHT  1080

#define FIELD_SCAN_LINES   (MAX_REGION_HEIGHT / 2) // this is a half value because on interlacing

#ifdef DS_HEAP_USAGE
   #define STB_GetMemory(x)  STB_DSGetMemory(x)
   #define STB_FreeMemory(x) STB_DSFreeMemory(x)
#endif

//---local typedef structs for this file---------------------------------------

// pixel subdata block intermediate struct
typedef struct pixel_sub_data_block
{
   U16BIT num_lines;
   U16BIT max_line_size;
   U16BIT line_size[FIELD_SCAN_LINES];
   U8BIT *data_line[FIELD_SCAN_LINES];
} S_PIXEL_SUB_DATA_BLOCK;


//---local (static) variable declarations for this file------------------------
//   (internal variables declared static to make them local)

static S_DISPLAY_SET subtitle_display_set = {NULL, NULL, NULL, NULL, NULL};
static S_PAGE_COMPOSITION page_1;
static S_PAGE_COMPOSITION page_2;
static U8BIT *temp_buffer;


// 32-bit CLUT where the 32bit value is in format of [TYCrCb] as 8bit values
// 0xff000000 is trgb transparent

// 2 Bit default CLUT
static U32BIT default_2_bit_clut[4] =
{
   0xff108080, 0x00478080, 0x00108080, 0x002b8080
};

// 4 Bit default CLUT
static U32BIT default_4_bit_clut[16] =
{
   0xff108080, 0x00209c76, 0x0030686d, 0x00418463, 0x00167b9c, 0x00279792, 0x00376389, 0x00478080,
   0x00108080, 0x00188e7b, 0x00207476, 0x00288271, 0x00137d8e, 0x001b8b89, 0x00237184, 0x002b8080
};

// 8 Bit default CLUT
static U32BIT default_8_bit_clut[256] =
{
   0xff108080, 0xbf209c76, 0xbf30686d, 0xbf418463, 0xbf167b9c, 0xbf279792, 0xbf376389, 0xbf478080,
   0x7f108080, 0x7f15897c, 0x7f1a7879, 0x7f208176, 0x7f127e89, 0x7f178786, 0x7f1c7683, 0x7f228080,
   0x001b9379, 0x00209c76, 0x00258b73, 0x002b9470, 0x001d9182, 0x00229a7f, 0x0028897c, 0x002d9379,
   0x7f1b9379, 0x7f209c76, 0x7f258b73, 0x7f2b9470, 0x7f1d9182, 0x7f229a7f, 0x7f28897c, 0x7f2d9379,
   0x00257073, 0x002b7970, 0x0030686d, 0x00367169, 0x00276e7c, 0x002d7779, 0x00326676, 0x00387073,
   0x7f257073, 0x7f2b7970, 0x7f30686d, 0x7f367169, 0x7f276e7c, 0x7f2d7779, 0x7f326676, 0x7f387073,
   0x0030836c, 0x00368c69, 0x003b7b66, 0x00418463, 0x00338176, 0x00388a73, 0x003d7970, 0x0043836c,
   0x7f30836c, 0x7f368c69, 0x7f3b7b66, 0x7f418463, 0x7f338176, 0x7f388a73, 0x7f3d7970, 0x7f43836c,
   0x00147c93, 0x0019868f, 0x001f758c, 0x00247e89, 0x00167b9c, 0x001b8499, 0x00217396, 0x00267c93,
   0x7f147c93, 0x7f19868f, 0x7f1f758c, 0x7f247e89, 0x7f167b9c, 0x7f1b8499, 0x7f217396, 0x7f267c93,
   0x001f8f8c, 0x00249989, 0x002a8886, 0x002f9183, 0x00218e96, 0x00269792, 0x002c868f, 0x00318f8c,
   0x7f1f8f8c, 0x7f249989, 0x7f2a8886, 0x7f2f9183, 0x7f218e96, 0x7f269792, 0x7f2c868f, 0x7f318f8c,
   0x002a6c86, 0x002f7683, 0x00346580, 0x003a6e7d, 0x002c6b8f, 0x0031748c, 0x00366389, 0x003c6c86,
   0x7f2a6c86, 0x7f2f7683, 0x7f346580, 0x7f3a6e7d, 0x7f2c6b8f, 0x7f31748c, 0x7f366389, 0x7f3c6c86,
   0x00358080, 0x003a897c, 0x00407879, 0x00458176, 0x00377e89, 0x003c8786, 0x00427683, 0x00478080,
   0x7f358080, 0x7f3a897c, 0x7f407879, 0x7f458176, 0x7f377e89, 0x7f3c8786, 0x7f427683, 0x7f478080,
   0x00108080, 0x0012847e, 0x00157c7c, 0x0018807b, 0x00117f84, 0x00138383, 0x00167b81, 0x00198080,
   0x00108080, 0x0012847e, 0x00157c7c, 0x0018807b, 0x00117f84, 0x00138383, 0x00167b81, 0x00198080,
   0x0015897c, 0x00188e7b, 0x001a8579, 0x001d8a78, 0x00168881, 0x00198d7f, 0x001b847e, 0x001e897c,
   0x0015897c, 0x00188e7b, 0x001a8579, 0x001d8a78, 0x00168881, 0x00198d7f, 0x001b847e, 0x001e897c,
   0x001a7879, 0x001d7c78, 0x00207476, 0x00227875, 0x001b777e, 0x001e7c7c, 0x0021737b, 0x00237879,
   0x001a7879, 0x001d7c78, 0x00207476, 0x00227875, 0x001b777e, 0x001e7c7c, 0x0021737b, 0x00237879,
   0x00208176, 0x00238675, 0x00257d73, 0x00288271, 0x0021807b, 0x00248579, 0x00267c78, 0x00298176,
   0x00208176, 0x00238675, 0x00257d73, 0x00288271, 0x0021807b, 0x00248579, 0x00267c78, 0x00298176,
   0x00127e89, 0x00148387, 0x00177a86, 0x001a7f84, 0x00137d8e, 0x0015828c, 0x0018798a, 0x001b7e89,
   0x00127e89, 0x00148387, 0x00177a86, 0x001a7f84, 0x00137d8e, 0x0015828c, 0x0018798a, 0x001b7e89,
   0x00178786, 0x001a8c84, 0x001c8383, 0x001f8881, 0x0018878a, 0x001b8b89, 0x001e8387, 0x00208786,
   0x00178786, 0x001a8c84, 0x001c8383, 0x001f8881, 0x0018878a, 0x001b8b89, 0x001e8387, 0x00208786,
   0x001c7683, 0x001f7b81, 0x00227280, 0x0025777e, 0x001d7587, 0x00207a86, 0x00237184, 0x00267683,
   0x001c7683, 0x001f7b81, 0x00227280, 0x0025777e, 0x001d7587, 0x00207a86, 0x00237184, 0x00267683,
   0x00228080, 0x0025847e, 0x00277c7c, 0x002a807b, 0x00237f84, 0x00268383, 0x00287b81, 0x002b8080,
   0x00228080, 0x0025847e, 0x00277c7c, 0x002a807b, 0x00237f84, 0x00268383, 0x00287b81, 0x002b8080
};

static U8BIT default_2_bit_4_bit_map[4] =
{
   0x00, 0x07, 0x08, 0x0f
};

static U8BIT default_2_bit_8_bit_map[4] =
{
   0x00, 0x77, 0x88, 0xff
};

static U8BIT default_4_bit_8_bit_map[16] =
{
   0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff
};

#define TOP_FIELD_ONLY        1
#define BOTTOM_FIELD_ONLY     2
#define BOTH_FIELDS           0

//---local function prototypes for this file-----------------------------------
//   (internal functions declared static to make them local)

static BOOLEAN ProcessObjectCodingMethod(U8BIT *data, S_OBJECT *object, U16BIT coding_method, U16BIT seg_len);
static BOOLEAN ProcessPixelDataSubBlock(U8BIT *data, S_OBJECT *object, U16BIT field_data_block_length, U8BIT field, S_PIXEL_SUB_DATA_BLOCK *decoded_data_block);
static BOOLEAN Process2BitPixelCodeString(U16BIT *run_length, U8BIT *colour,
   U8BIT *my2_u8p, U16BIT *my2_u16, U8BIT *my2_u8);
static BOOLEAN Process4BitPixelCodeString(U16BIT *run_length, U8BIT *colour,
   U8BIT *my4_u8p, U16BIT *my4_u16, U8BIT *my4_u8);
static BOOLEAN Process8BitPixelCodeString(U16BIT *run_length, U8BIT *colour,
   U8BIT *data, U16BIT *byte_count);
static BOOLEAN Process2To4BitMapTable(U8BIT *data, U16BIT *processed_bits, U8BIT *decoded_data);
static BOOLEAN Process2To8BitMapTable(U8BIT *data, U16BIT *processed_bits, U8BIT *decoded_data);
static BOOLEAN Process4To8BitMapTable(U8BIT *data, U16BIT *processed_bits, U8BIT *decoded_data);
static BOOLEAN MakeClutEntry(S_CLUT *clut, U16BIT clut_id, U8BIT clut_entry_id, U8BIT clut_bit_entry_flags, U32BIT palette_entry);
#ifdef STB_REGION_PRINT_REQUIRED
void CheckRegionFootPrint(S_EPOCH_REGION *region, U16BIT region_width, U16BIT region_height,
   U16BIT region_level_of_compatability, U16BIT region_depth, U16BIT clut_id);
#endif
static void DestroyObject(S_OBJECT **obj, BOOLEAN del_obj);
static void DeleteObjectList(S_OBJECT **obj);
static void DeleteClutList(S_CLUT **clut);
static void DeleteEpochRegionList(S_EPOCH_REGION **region_list);
static void DeletePageCompositionRegions(S_REGION **comp_pg);
static void STB_DSDumpPageState(const char *label);
//PDH
extern void dump_physical_regions(const char *label);

static void   Get2Bit_Initialise(const U8BIT *new_root, U8BIT **my2_u8p, U16BIT *my2_u16, U8BIT *my2_u8);
static void   Get2Bit_SetAlignment(U16BIT *my2_u16, U8BIT *my2_u8);
static U16BIT Get2Bit_Offset(U16BIT my2_u16);
static U8BIT  Get2Bit_Get(U8BIT *my2_u8p, U16BIT *my2_u16, U8BIT *my2_u8);

static void   Get4Bit_Initialise(const U8BIT *new_root, U8BIT **my4_u8p, U16BIT *my4_u16, U8BIT *my4_u8);
static void   Get4Bit_SetAlignment(U16BIT *my4_u16, U8BIT *my4_u8);
static U16BIT Get4Bit_Offset(U16BIT my4_u16);
static U8BIT  Get4Bit_Get(U8BIT *my4_u8p, U16BIT *my4_u16, U8BIT *my4_u8);

//---local function definitions------------------------------------------------

static void Get2Bit_Initialise(const U8BIT *new_root, U8BIT **my_root, U16BIT *my_offset, U8BIT *my_counter)
{
   *my_root = (U8BIT *) new_root;
   *my_offset = 0;
   *my_counter = 0;
}

static void Get2Bit_SetAlignment(U16BIT *my_offset, U8BIT *my_counter)
{
   // If not byte aligned, move to start of next byte
   if (*my_counter != 0)
   {
      *my_counter = 0;
      *my_offset += 1;
   }
   // else - nothing to do
}

//A bit overkill, but trying to make the point - how the 2bit code works ISN'T CLIENT'S BUSINESS!!
//OO approach - these are my internal variables, you shouldn't be aware of them, let alone
//using them!
static U16BIT Get2Bit_Offset(U16BIT my_offset)
{
   return my_offset;
}

static U8BIT Get2Bit_Get(U8BIT *my_root, U16BIT *my_offset, U8BIT *my_counter)
{
   U8BIT retval = 0xff;

   switch (*my_counter)
   {
      case 0:
         retval = (my_root[*my_offset]) >> 6;
         break;
      case 1:
         retval = (((my_root[*my_offset]) >> 4) & 0x03);
         break;
      case 2:
         retval = (((my_root[*my_offset]) >> 2) & 0x03);
         break;
      case 3:
         retval = ((my_root[*my_offset]) & 0x03);
         *my_offset += 1;
         break;
      default:
         // Hell no!  somebody's being playing with my state, or not initialised me
         *my_offset = 0xffff;
         break;
   }

   *my_counter = (*my_counter + 1) % 4;

   return retval;
}

static void Get4Bit_Initialise(const U8BIT *new_root, U8BIT **my_root, U16BIT *my_offset, U8BIT *my_counter)
{
   *my_root = (U8BIT *) new_root;
   *my_offset = 0;
   *my_counter = 0;
}

static void Get4Bit_SetAlignment(U16BIT *my_offset, U8BIT *my_counter)
{
   // If not byte aligned, move to start of next byte
   if (*my_counter != 0)
   {
      *my_counter = 0;
      *my_offset += 1;
   }
   // else - nothing to do
}

//A bit overkill, but trying to make the point - how the 2bit code works ISN'T CLIENT'S BUSINESS!!
//OO approach - these are my internal variables, you shouldn't be aware of them, let alone
//using them!
static U16BIT Get4Bit_Offset(U16BIT my_offset)
{
   return my_offset;
}

static U8BIT Get4Bit_Get(U8BIT *my_root, U16BIT *my_offset, U8BIT *my_counter)
{
   U8BIT retval = 0xff;

   switch (*my_counter)
   {
      case 0:
         retval = (my_root[*my_offset]) >> 4;
//   STB_OBJECT_PRINT(("     G<4> @%x --> %x", my_root+*my_offset, retval));

         break;
      case 1:
         retval = ((my_root[*my_offset]) & 0x0f);
//   STB_OBJECT_PRINT(("     g<4> @%x --> %x", my_root+*my_offset, retval));

         *my_offset += 1;
         break;
      default:
         // Hell no!  somebody's being playing with my state, or not initialised me
         *my_offset = 0xffff;
         break;
   }

   *my_counter = (*my_counter + 1) % 2;

   return retval;
}

/**
 *

 *
 * @brief   Processing of data from ODS, data in a ODS can only be of one type with
 *                 subsequent iterations all being of this type.
 *
 * @param   data          - data of segment points to the sync_byte
 * @param   object        - the object that is being decoded
 * @param   coding_method - the type of encoding [bitmap or string of characters]
 * @param   seg_len       - the length of the segment
 *
 * @return   TRUE  - Successful processing of the passed data and object.
 *                 FALSE - Unsuccessful malloc failed somewhere, poss incomplete object structure.
 *
 */
static BOOLEAN ProcessObjectCodingMethod(U8BIT *data, S_OBJECT *object, U16BIT coding_method, U16BIT seg_len)
{
   S_PIXEL_SUB_DATA_BLOCK top_field;
   S_PIXEL_SUB_DATA_BLOCK bottom_field;
   U16BIT top_field_data_block_length;
   U16BIT bottom_field_data_block_length;
   BOOLEAN retval;

   FUNCTION_START(ProcessObjectCodingMethod);

   ASSERT(data != NULL);
   ASSERT(object != NULL);

   retval = TRUE;

   switch (coding_method)
   {
      case 0:
      {
         // coding of pixels
         memset((void *)&top_field, 0x00, sizeof(S_PIXEL_SUB_DATA_BLOCK));
         memset((void *)&bottom_field, 0x00, sizeof(S_PIXEL_SUB_DATA_BLOCK));

         top_field_data_block_length = (data[0] << 8) + data[1];
         bottom_field_data_block_length = (data[2] << 8) + data[3];

         if ((top_field_data_block_length + bottom_field_data_block_length) < seg_len)
         {
            // position pointer to beginning of top field data
            data += 4;

            if (object->bitmap != NULL)
            {
               STB_FreeMemory(object->bitmap);
               object->bitmap = NULL;
            }

            if (bottom_field_data_block_length == 0)
            {
               // no data for bottom field ETS 300 743 - 7.2.4
               // top field data is valid for the bottom field
               STB_OBJECT_PRINT(("bottom_field_data_block not present"));

               retval = ProcessPixelDataSubBlock(&data[0], object, top_field_data_block_length, BOTH_FIELDS, &top_field);
            }
            else
            {
               retval = ProcessPixelDataSubBlock(&data[0], object, top_field_data_block_length, TOP_FIELD_ONLY, &top_field);
               retval &= ProcessPixelDataSubBlock(&data[top_field_data_block_length], object, bottom_field_data_block_length, BOTTOM_FIELD_ONLY, &bottom_field);
            }
         }
         break;
      }

      case 1:
      {
         // coding as a string of characters
         object->num_chars = data[9];
         object->character_code = (U8BIT *)STB_GetMemory((object->num_chars * 2));
         if (object->character_code != NULL)
         {
            STB_OBJECT_PRINT(("object_coding_method string of %d characters", object->num_chars));
            memcpy((void *)object->character_code, (void *)&data[10], (size_t)(object->num_chars * 2));
         }
         else
         {
            STB_OBJECT_PRINT(("object_coding_method char malloc fail"));
            object->num_chars = 0;
            retval = FALSE;
         }
         break;
      }

      case 2:
      case 3:
      default:
      {
         STB_OBJECT_PRINT(("object_coding_method INVALID"));
         retval = FALSE;
         break;
      }
   }

   FUNCTION_FINISH(ProcessObjectCodingMethod);

   return(retval);
}

/**
 *

 *
 * @brief   Decodes field_data_block_length as a pixel_data_sub_block as specified in
 *                 ETS 300 743, 7.2.4.1
 *                 NB. also
 *                 DTG - Digital Terrestrial Television Requirements for Interoperability
 *                 Subtitling: 28 August 1998: Version: 3.0 Clause 5.4 Corrigenda to ETS 300 743
 *
 * @param   data - data of segment points to the sync_byte
 * @param   object -
 * @param   field_data_block_length - the number of bytes that makeup this block
 * @param   decoded_data_block - where all decoded data is placed prior to creating bitmap
 *
 * @return   TRUE - Successful decode; FALSE - Failed in a pixel_data_sub_block
 *
 */
static BOOLEAN ProcessPixelDataSubBlock(U8BIT *data,
   S_OBJECT *object,
   U16BIT field_data_block_length,
   U8BIT field,
   S_PIXEL_SUB_DATA_BLOCK *decoded_data_block)
{
   U8BIT *decoded_data;

   U16BIT processed_length;
   U16BIT decode_data_size;
   U16BIT run_length;
   U16BIT scanline;
   U8BIT data_type;
   U8BIT colour;
   BOOLEAN end_of_string;
   BOOLEAN terminate = FALSE;
   BOOLEAN retval;

   //2 bit processing private state
   U8BIT *my2_u8p;
   U16BIT my2_u16;
   U8BIT my2_u8;

   //4 bit processing private state
   U8BIT *my4_u8p;
   U16BIT my4_u16;
   U8BIT my4_u8;

   static BOOLEAN error_flag;
   static BOOLEAN error_line;

   FUNCTION_START(ProcessPixelDataSubBlock);

   ASSERT(data != NULL);
   ASSERT(object != NULL);
   ASSERT(decoded_data_block != NULL);

   USE_UNWANTED_PARAM(decoded_data_block);

   STB_OBJECT_PRINT(("  Pixel-data_sub-block %d bytes", field_data_block_length));

   colour = 0;
   run_length = 0;

   retval = FALSE;

   decode_data_size = 0;
   processed_length = 0;
   scanline = 0;

   decoded_data = temp_buffer;

   end_of_string = FALSE;

   switch (field)
   {
      case TOP_FIELD_ONLY:
      {
         scanline = 0;
         error_flag = FALSE;
         error_line = 0;
         break;
      }

      case BOTTOM_FIELD_ONLY:
      {
         scanline = 1;
         break;
      }

      case BOTH_FIELDS:
      {
         scanline = 0;
         error_flag = FALSE;
         error_line = 0;
         break;
      }

      default:
      {
         error_flag = FALSE;
         error_line = 0;
         terminate = TRUE;
         break;
      }
   }

   while (!terminate && (processed_length < field_data_block_length))
   {
      data_type = data[processed_length];

      STB_OBJECT_PRINT(("   data_type %02x @ %x / %d", data_type, data + processed_length, processed_length));

      //Move pointer to first byte after data_type field
      processed_length++;

      switch (data_type)
      {
         case 0x0f:
         {
            // This accounts for the unnused bytes sync_byte, segment_byte, segement_length, data_type & page_id
            processed_length += 6;
            break;
         }

         case 0x10:
         {
            //Initialise the 2 bit getter code to start at first byte of bufefr
            Get2Bit_Initialise((const U8BIT *) (data + processed_length), &my2_u8p, &my2_u16, &my2_u8);

            do
            {
               end_of_string = Process2BitPixelCodeString(&run_length, &colour, my2_u8p, &my2_u16, &my2_u8 );

               if (((decode_data_size + run_length) > SCAN_LINE_WIDTH) ||
                   (processed_length + Get2Bit_Offset(my2_u16) > field_data_block_length))
               {
                  break;
               }
               else
               {
                  if (run_length != 0)
                  {
                     static U8BIT lastcolin = 0xff;

                     if ((lastcolin != colour) && (run_length > 12))
                     {
//                        STB_PIXMAP_PRINT(("    10>  in %02x -2to8-> %02x",
//                                          colour, object->map_table_2_to_8_bit[colour]));
                        lastcolin = colour;
                     }

                     if (object->map_table_2_to_8_bit != default_2_bit_8_bit_map)
                     {
                        /* Convert directly from 2 to 8-bit */
                        colour = object->map_table_2_to_8_bit[colour];
                     }
                     else
                     {
                        /* Convert to 8-bit using the 2-to-4 bit and 4-to-8 colour map because
                         * there may have been a change to the default mapping with the data */
                        colour = object->map_table_2_to_4_bit[colour];
                        colour = object->map_table_4_to_8_bit[colour];
                     }

                     memset((void *)&decoded_data[decode_data_size], colour, run_length);
                     decode_data_size += run_length;
                  }
               }
            }
            while (!end_of_string);

            //Recover byte alignment
            Get2Bit_SetAlignment(&my2_u16, &my2_u8);

            //Update how many bytes have been processed
            processed_length += Get2Bit_Offset(my2_u16);
            break;
         }

         case 0x11:
         {
            //Initialise the 4 bit getter code to start at first byte of bufefr
            Get4Bit_Initialise((const U8BIT *) (data + processed_length), &my4_u8p, &my4_u16, &my4_u8);

            do
            {
               end_of_string = Process4BitPixelCodeString(&run_length, &colour, my4_u8p, &my4_u16, &my4_u8 );

               if (((decode_data_size + run_length) > SCAN_LINE_WIDTH) ||
                   (processed_length + Get4Bit_Offset(my4_u16) > field_data_block_length))
               {
                  if ((decode_data_size + run_length) > SCAN_LINE_WIDTH)
                  {
                     STB_OBJECT_PRINT(("   !! Decoded + run length > scan line width  %d + %d > %d",
                                       decode_data_size, run_length, SCAN_LINE_WIDTH));
                  }
                  if (processed_length + Get4Bit_Offset(my4_u16) > field_data_block_length)
                  {
                     STB_OBJECT_PRINT(("   !! processed_length + Get4Bit_Offset(my4_u16) > field_data_block_length  %d + %d > %d",
                                       processed_length, Get4Bit_Offset(my4_u16), field_data_block_length));
                  }

                  break;
               }
               else
               {
                  if (run_length != 0)
                  {
                     static U8BIT lastcolin = 0xff;

                     if ((lastcolin != colour) && (run_length > 12))
                     {
//                        STB_PIXMAP_PRINT(("    11>  in %02x -4to8-> %02x",
//                                          colour, object->map_table_4_to_8_bit[colour]));
                        lastcolin = colour;
                     }

                     // Convert to 8bit colour
                     colour = object->map_table_4_to_8_bit[colour];
                     memset((void *)&decoded_data[decode_data_size], colour, run_length);
                     decode_data_size += run_length;
                  }
               }
            }
            while (!end_of_string);

            //Recover byte alignment
            Get4Bit_SetAlignment(&my4_u16, &my4_u8);

            //Update how many bytes have been processed
            processed_length += Get4Bit_Offset(my4_u16);
            break;
         }

         case 0x12:
         {
            // 8 bit/pixel code string group
            do
            {
               end_of_string = Process8BitPixelCodeString(&run_length, &colour, data, &processed_length);

               if (((decode_data_size + run_length) > SCAN_LINE_WIDTH) || (processed_length > field_data_block_length))
               {
                  break;
               }
               else
               {
                  if (run_length != 0)
                  {
                     static U8BIT lastcolin = 0xff;

                     if ((lastcolin != colour) && (run_length > 12))
                     {
//                        STB_PIXMAP_PRINT(("    12>  in %02x", colour));
                        lastcolin = colour;
                     }
                     memset((void *)&decoded_data[decode_data_size], colour, run_length);
                     decode_data_size += run_length;
                  }
               }
            }
            while (!end_of_string);
            break;
         }

         case 0x20:
         {
            // 2 to 4 bit map table data
            Process2To4BitMapTable(data, &processed_length, decoded_data);
            if (object->map_table_2_to_4_bit != (U8BIT *)default_2_bit_4_bit_map)
            {
               STB_FreeMemory(object->map_table_2_to_4_bit);
            }
            object->map_table_2_to_4_bit = (U8BIT *)STB_GetMemory(4);
            if (object->map_table_2_to_4_bit != NULL)
            {
               memcpy((void *)object->map_table_2_to_4_bit, (void *)decoded_data, (size_t)4);

               STB_OBJECT_PRINT(("    20> 2To4Map %02x %02x %02x %02x",
                                 decoded_data[0], decoded_data[1], decoded_data[2], decoded_data[3]));
            }
            else
            {
               object->map_table_2_to_4_bit = default_2_bit_4_bit_map;
            }
            break;
         }

         case 0x21:
         {
            // 2 to 8 bit map table data
            Process2To8BitMapTable(data, &processed_length, decoded_data);

            if (object->map_table_2_to_8_bit != (U8BIT *)default_2_bit_8_bit_map)
            {
               STB_FreeMemory(object->map_table_2_to_8_bit);
            }
            object->map_table_2_to_8_bit = (U8BIT *)STB_GetMemory(4);
            if (object->map_table_2_to_8_bit != NULL)
            {
               memcpy((void *)object->map_table_2_to_8_bit, (void *)decoded_data, (size_t)4);

               STB_OBJECT_PRINT(("    21> 2To8Map %02x %02x %02x %02x",
                                 decoded_data[0], decoded_data[1], decoded_data[2], decoded_data[3]));
            }
            else
            {
               object->map_table_2_to_8_bit = default_2_bit_8_bit_map;
            }
            break;
         }

         case 0x22:
         {
            // 4 to 8 bit map table data
            Process4To8BitMapTable(data, &processed_length, decoded_data);

            if (object->map_table_4_to_8_bit != (U8BIT *)default_4_bit_8_bit_map)
            {
               STB_FreeMemory(object->map_table_4_to_8_bit);
            }
            object->map_table_4_to_8_bit = (U8BIT *)STB_GetMemory(16);
            if (object->map_table_4_to_8_bit != NULL)
            {
               memcpy((void *)object->map_table_4_to_8_bit, (void *)decoded_data, (size_t)16);
               STB_OBJECT_PRINT(("    22> 4To8Map %02x %02x %02x %02x  %02x %02x %02x %02x  %02x %02x %02x %02x  %02x %02x %02x %02x",
                                 decoded_data[ 0], decoded_data[ 1], decoded_data[ 2], decoded_data[ 3],
                                 decoded_data[ 4], decoded_data[ 5], decoded_data[ 6], decoded_data[ 7],
                                 decoded_data[ 8], decoded_data[ 9], decoded_data[10], decoded_data[11],
                                 decoded_data[12], decoded_data[13], decoded_data[14], decoded_data[15]));
            }
            else
            {
               object->map_table_4_to_8_bit = default_4_bit_8_bit_map;
            }
            break;
         }

         case 0xf0:
         {
            STB_OBJECT_PRINT(("    End of Line after %d pixels", decode_data_size));
            if (decode_data_size != 0)
            {
               STB_DSRenderBitmapToRegion(subtitle_display_set.region_list, object, decoded_data,
                  scanline, decode_data_size, 1);
            }

            if (field == BOTTOM_FIELD_ONLY)
            {
               // error correction bit
               if (error_flag)
               {
                  if ((scanline + 1) >= error_line)
                  {
                     scanline++;
                     if (decode_data_size != 0)
                     {
                        STB_DSRenderBitmapToRegion(subtitle_display_set.region_list, object, decoded_data,
                           scanline, decode_data_size, 1);
                     }
                     scanline++;
                  }
                  else
                  {
                     scanline += 2;
                  }
               }
               else
               {
                  scanline += 2;
               }
            }
            else
            {
               scanline++;
               if (decode_data_size != 0)
               {
                  STB_DSRenderBitmapToRegion(subtitle_display_set.region_list, object, decoded_data,
                     scanline, decode_data_size, 1);
               }
               scanline++;
            }
            decode_data_size = 0;
            decoded_data = temp_buffer;
            break;
         }

         default:
         {
            STB_OBJECT_PRINT(("Unknown data_type 0x%02x", data_type));
            terminate = TRUE;

            if (field == TOP_FIELD_ONLY)
            {
               error_flag = TRUE;
               error_line = scanline;
               STB_OBJECT_PRINT(("ERR Detected TOP %d", scanline));
            }
            else
            {
               STB_OBJECT_PRINT(("ERR Detected BOT %d", scanline));
            }
            break;
         }
      }

      if (terminate)
      {
         break;
      }
   }

   if (terminate)
   {
      STB_OBJECT_PRINT(("PIXPARSE TERMINATE @ %d", __LINE__));
      if (object->map_table_2_to_4_bit != (U8BIT *)default_2_bit_4_bit_map)
      {
         STB_FreeMemory(object->map_table_2_to_4_bit);
         object->map_table_2_to_4_bit = (U8BIT *)default_2_bit_4_bit_map;
      }

      if (object->map_table_2_to_8_bit != (U8BIT *)default_2_bit_8_bit_map)
      {
         STB_FreeMemory(object->map_table_2_to_8_bit);
         object->map_table_2_to_8_bit = (U8BIT *)default_2_bit_8_bit_map;
      }

      if (object->map_table_4_to_8_bit != (U8BIT *)default_4_bit_8_bit_map)
      {
         STB_FreeMemory(object->map_table_4_to_8_bit);
         object->map_table_4_to_8_bit = (U8BIT *)default_4_bit_8_bit_map;
      }
   }

   if (!terminate && (processed_length == field_data_block_length))
   {
      retval = TRUE;
   }

   FUNCTION_FINISH(ProcessPixelDataSubBlock);

   return(retval);
}

/**
 *

 *
 * @brief   Decodes a 2 bit pixel code string as specified in ETS 300 743, 7.2.4.2
 *                 Output is as 8bit colour codes
 *
 * @param   data            - data of segment; points to the data_type of the pixel_data_sub_block
 * @param   byte_count      - the number of bytes processed for this structure
 * @param   run_length      - the number of bytes in this string
 * @param   colour          - colour of this string
 *
 * @return   TRUE or FALSE whether or not the end of line has been found.
 *
 */
static BOOLEAN Process2BitPixelCodeString(U16BIT *run_length, U8BIT *colour,
   U8BIT *my2_u8p, U16BIT *my2_u16, U8BIT *my2_u8)
{
   U8BIT bit_pattern;
   BOOLEAN end_of_2_bit_pixel_code_string = FALSE;


   FUNCTION_START(Process2BitPixelCodeString);

   ASSERT(run_length != NULL);
   ASSERT(colour != NULL);

   bit_pattern = Get2Bit_Get(my2_u8p, my2_u16, my2_u8);

   if (bit_pattern != 0x00)
   {
      // bit_pattern is a 2 bit pixel code
      *run_length = 1;
      *colour = bit_pattern;
   }
   else
   {
      // 2_bit_zero
      bit_pattern = Get2Bit_Get(my2_u8p, my2_u16, my2_u8);
      if (bit_pattern == 0x01)
      {
         // 1 pixel set to entry 00
         *run_length = 1;
         *colour = 0x00;
      }
      else
      {
         if (bit_pattern >= 0x2)
         {
            // run_length_3_10
            *run_length = ((bit_pattern & 1) << 2) + Get2Bit_Get(my2_u8p, my2_u16, my2_u8) + 3;
            *colour = Get2Bit_Get(my2_u8p, my2_u16, my2_u8);
         }
         else
         {
            if (bit_pattern == 0)
            {
               // This is switch 3
               bit_pattern = Get2Bit_Get(my2_u8p, my2_u16, my2_u8);
               if (bit_pattern == 0x1)
               {
                  // 2 pixel set to entry 00
                  *run_length = 2;
                  *colour = 0;
               }
               else
               {
                  if (bit_pattern == 0x2)
                  {
                     // run_length_12_27
                     *run_length = (Get2Bit_Get(my2_u8p, my2_u16, my2_u8) << 2) +
                        Get2Bit_Get(my2_u8p, my2_u16, my2_u8) + 12;
                     *colour = Get2Bit_Get(my2_u8p, my2_u16, my2_u8);
                  }
                  else
                  {
                     if (bit_pattern == 0x3)
                     {
                        // run_length_29_284
                        *run_length = (Get2Bit_Get(my2_u8p, my2_u16, my2_u8) << 6) +
                           (Get2Bit_Get(my2_u8p, my2_u16, my2_u8) << 4) +
                           (Get2Bit_Get(my2_u8p, my2_u16, my2_u8) << 2) +
                           Get2Bit_Get(my2_u8p, my2_u16, my2_u8) + 29;
                        *colour = Get2Bit_Get(my2_u8p, my2_u16, my2_u8);
                     }
                     else
                     {
                        // end of 2 bit pixel_code_string
                        *run_length = 0;
                        *colour = 0;
                        end_of_2_bit_pixel_code_string = TRUE;
                     }
                  }
               }
            }
         }
      }
   }

   if (!end_of_2_bit_pixel_code_string)
   {
      STB_PIXMAP_PRINT(("    <2> col %d x %3d", *colour, *run_length));
   }
   else
   {
      STB_PIXMAP_PRINT(("    <2> End of String"));
   }

   FUNCTION_FINISH(Process2BitPixelCodeString);

   return(end_of_2_bit_pixel_code_string);
}

/**
 *

 *
 * @brief   Decodes a 4 bit pixel code string as specified in ETS 300 743, 7.2.4.2
 *                 Output is as 8bit colour codes
 *
 * @param   data            - data of segment; points to the data_type of the pixel_data_sub_block
 * @param   byte_count      - the number of bytes processed for this structure
 * @param   run_length      - the number of bytes in this string
 * @param   colour          - colour of this string
 *
 * @return   TRUE or FALSE whether or not the end of line has been found.
 *
 */
static BOOLEAN Process4BitPixelCodeString(U16BIT *run_length, U8BIT *colour,
   U8BIT *my4_u8p, U16BIT *my4_u16, U8BIT *my4_u8)
{
   U8BIT bit_pattern;
   BOOLEAN end_of_4_bit_pixel_code_string = FALSE;


   FUNCTION_START(Process4BitPixelCodeString);

   ASSERT(run_length != NULL);
   ASSERT(colour != NULL);

   bit_pattern = (U8BIT)Get4Bit_Get(my4_u8p, my4_u16, my4_u8);

   if (bit_pattern > 0x00)
   {
      // 4_bit_pixel_code
      *run_length = 1;
      *colour = bit_pattern;
   }
   else
   {
      bit_pattern = (U8BIT)Get4Bit_Get(my4_u8p, my4_u16, my4_u8);

      if (bit_pattern == 0x00)
      {
         // end_of_string_signal
         *run_length = 0x00;
         *colour = 0;
         end_of_4_bit_pixel_code_string = TRUE;
      }
      else
      {
         if (bit_pattern <= 0x07)
         {
            // run_length_3_9
            *run_length = (bit_pattern + 2);
            *colour = 0;
         }
         else
         {
            if (bit_pattern <= 0x0b)
            {
               // run_length_4_7
               *run_length = ((bit_pattern & 0x03) + 4);
               *colour = (U8BIT)Get4Bit_Get(my4_u8p, my4_u16, my4_u8);
            }
            else
            {
               if (bit_pattern <= 0x0d)
               {
                  *run_length = ((bit_pattern & 0x03) + 1);
                  *colour = 0;
               }
               else
               {
                  if (bit_pattern == 0x0e)
                  {
                     // run_length_9_24
                     *run_length = ((U8BIT)Get4Bit_Get(my4_u8p, my4_u16, my4_u8) + 9);
                     *colour = (U8BIT)Get4Bit_Get(my4_u8p, my4_u16, my4_u8);
                  }
                  else
                  {
                     if (bit_pattern == 0x0f)
                     {
                        // run_length_25_280
                        *run_length = (((U8BIT)Get4Bit_Get(my4_u8p, my4_u16, my4_u8) << 4) & 0xf0);
                        *run_length += (U8BIT)Get4Bit_Get(my4_u8p, my4_u16, my4_u8);
                        *run_length += 25;
                        *colour = (U8BIT)Get4Bit_Get(my4_u8p, my4_u16, my4_u8);
                     }
                  }
               }
            }
         }
      }
   }

   if (!end_of_4_bit_pixel_code_string)
   {
      STB_PIXMAP_PRINT(("    <4> col %2d x %3d", *colour, *run_length));
   }
   else
   {
      STB_PIXMAP_PRINT(("    <4> End of String"));
   }

   FUNCTION_FINISH(Process4BitPixelCodeString);

   return(end_of_4_bit_pixel_code_string);
}

/**
 *

 *
 * @brief   Decodes a 8 bit pixel code string as specified in ETS 300 743, 7.2.4.2
 *
 * @param   data            - data of segment; points to the data_type of the pixel_data_sub_block
 * @param   byte_count      - the number of bytes processed for this structure
 * @param   run_length      - the number of bytes in this string
 * @param   colour          - colour of this string
 *
 * @return   TRUE or FALSE whether or not the end of line has been found.
 *
 */
static BOOLEAN Process8BitPixelCodeString(U16BIT *run_length, U8BIT *colour, U8BIT *data, U16BIT *byte_count)
{
   U8BIT bit_pattern;
   BOOLEAN end_of_8_bit_pixel_code_string;


   FUNCTION_START(Process8BitPixelCodeString);

   ASSERT(data != NULL);
   ASSERT(byte_count != NULL);
   ASSERT(run_length != NULL);
   ASSERT(colour != NULL);

   end_of_8_bit_pixel_code_string = FALSE;

   bit_pattern = data[(*byte_count)];
   *byte_count += 1;
   if (bit_pattern != 0x00)
   {
      // bit_pattern is an 8 bit pixel code
      *run_length = 1;
      *colour = bit_pattern;
   }
   else
   {
      // 8 bit zero
      bit_pattern = data[(*byte_count)];
      *byte_count += 1;
      if (bit_pattern == 0x00)
      {
         // end_of_string_signal
         *run_length = 0x00;
         *colour = 0;
         end_of_8_bit_pixel_code_string = TRUE;
      }
      else
      {
         if (bit_pattern <= 0x7f)
         {
            // run_length_1_127
            // n pixel set to entry 00
            *run_length = bit_pattern;
            *colour = 0;
         }
         else
         {
            // run_length_3_127
            *run_length = (bit_pattern & 0x7f);
            *colour = data[(*byte_count)];
            *byte_count += 1;
         }
      }
   }

   if (!end_of_8_bit_pixel_code_string)
   {
      STB_PIXMAP_PRINT(("    <8> col %d x %3d", *colour, *run_length));
   }
   else
   {
      STB_PIXMAP_PRINT(("    <8> End of String"));
   }

   FUNCTION_FINISH(Process8BitPixelCodeString);

   return(end_of_8_bit_pixel_code_string);
}

/**
 *

 *
 * @brief   Processes data declared as a 2 to 4 bit map table into the 2 to 4 bit map table
 *
 * @param   data            - data of segment
 * @param   processed_bytes - the number of bytes processed for this structure
 * @param   decoded_data    - pointer to decoded data
 *
 * @return   TRUE or FALSE whether or not the end of line has been found.
 *
 */
static BOOLEAN Process2To4BitMapTable(U8BIT *data, U16BIT *processed_bytes, U8BIT *decoded_data)
{
   FUNCTION_START(Process2To4BitMapTable);

   ASSERT(data != NULL);
   ASSERT(processed_bytes != NULL);
   ASSERT(decoded_data != NULL);

   // 2_to_4_bit_map_table_data
   decoded_data[0] = (data[*processed_bytes] & 0xf0) >> 4;
   decoded_data[1] = (data[*processed_bytes] & 0x0f);
   decoded_data[2] = (data[(*processed_bytes) + 1] & 0xf0) >> 4;
   decoded_data[3] = (data[(*processed_bytes) + 1] & 0x0f);
   *processed_bytes += 2;

   FUNCTION_FINISH(Process2To4BitMapTable);

   return(TRUE);
}

/**
 *

 *
 * @brief   Processes data declared as a 2 to 8 bit map table into the 2 to 4 bit map table
 *                 as specified in ETS 300 743, 7.2.4.1
 *
 *
 * @param   data            - data of segment
 * @param   processed_bytes - the number of bytes processed for this structure
 * @param   decoded_data    - pointer to decoded data
 *
 * @return   TRUE or FALSE whether or not the end of line has been found.
 *
 */
static BOOLEAN Process2To8BitMapTable(U8BIT *data, U16BIT *processed_bytes, U8BIT *decoded_data)
{
   FUNCTION_START(Process2To8BitMapTable);

   ASSERT(data != NULL);
   ASSERT(processed_bytes != NULL);
   ASSERT(decoded_data != NULL);

   // 2_to_8_bit_map_table_data
   decoded_data[0] = data[*processed_bytes];
   decoded_data[1] = data[(*processed_bytes) + 1];
   decoded_data[2] = data[(*processed_bytes) + 2];
   decoded_data[3] = data[(*processed_bytes) + 3];
   *processed_bytes += 4;

   FUNCTION_FINISH(Process2To8BitMapTable);

   return(TRUE);
}

/**
 *

 *
 * @brief   Processes data declared as a 4 to 8 bit map table into the 2 to 4 bit map table
 *                 as specified in ETS 300 743, 7.2.4.1
 *
 * @param   data            - data of segment
 * @param   processed_bytes - the number of bytes processed for this structure
 * @param   decoded_data    - pointer to decoded data
 *
 * @return   TRUE or FALSE whether or not the end of line has been found.
 *
 */
static BOOLEAN Process4To8BitMapTable(U8BIT *data, U16BIT *processed_bytes, U8BIT *decoded_data)
{
   FUNCTION_START(Process4To8BitMapTable);

   ASSERT(data != NULL);
   ASSERT(processed_bytes != NULL);
   ASSERT(decoded_data != NULL);

   memcpy(decoded_data, data + *processed_bytes, 16);

   *processed_bytes += 16;

   FUNCTION_FINISH(Process4To8BitMapTable);

   return(TRUE);
}

/**
 *

 *
 * @brief   Enters a single value into a single given entry in a given CLUT.
 *
 * @param   clut                 - pointer to a S_CLUT structure
 * @param   clut_id              - the index of the CLUT to be modified
 * @param   clut_entry_id        - the index of the entry in the CLUT to be modified
 * @param   clut_bit_entry_flags - which CLUT 2, 4 or 8 bit
 * @param   palette_entry        - the value of the entry
 *
 * @return   TRUE for success; FALSE malloc fail or clut_bit_entry_flags referencing more than
 *                 a single CLUT in a 'family' of CLUTs.
 *
 */
static BOOLEAN MakeClutEntry(S_CLUT *clut, U16BIT clut_id, U8BIT clut_entry_id, U8BIT clut_bit_entry_flags, U32BIT palette_entry)
{
   BOOLEAN retval;


   FUNCTION_START(MakeClutEntry);

   ASSERT(clut != NULL);

   USE_UNWANTED_PARAM(clut_id);

   retval = TRUE;

   if (clut_bit_entry_flags & 0x4)
   {
      if (clut_entry_id < 4)
      {
         if (clut->default_2_bit)
         {
            clut->clut_2_bit = (U32BIT *)STB_GetMemory(4 * 4);
            if (clut->clut_2_bit == NULL)
            {
               clut->clut_2_bit = default_2_bit_clut;
               retval = FALSE;
            }
            else
            {
               memcpy((void *)clut->clut_2_bit, (void *)default_2_bit_clut, (size_t)(4 * 4));
               clut->default_2_bit = FALSE;
               clut->clut_2_bit[clut_entry_id] = palette_entry;
            }
         }
         else
         {
            clut->clut_2_bit[clut_entry_id] = palette_entry;
         }
      }
   }

   if (clut_bit_entry_flags & 0x2)
   {
      if (clut_entry_id < 16)
      {
         if (clut->default_4_bit)
         {
            clut->clut_4_bit = (U32BIT *)STB_GetMemory(16 * 4);
            if (clut->clut_4_bit == NULL)
            {
               clut->clut_4_bit = default_4_bit_clut;
               retval = FALSE;
            }
            else
            {
               memcpy((void *)clut->clut_4_bit, (void *)default_4_bit_clut, (size_t)(16 * 4));
               clut->default_4_bit = FALSE;
               clut->clut_4_bit[clut_entry_id] = palette_entry;
            }
         }
         else
         {
            clut->clut_4_bit[clut_entry_id] = palette_entry;
         }
      }
   }

   if (clut_bit_entry_flags & 0x1)
   {
      if (clut->default_8_bit)
      {
         clut->clut_8_bit = (U32BIT *)STB_GetMemory(256 * 4);
         if (clut->clut_8_bit == NULL)
         {
            clut->clut_8_bit = default_8_bit_clut;
            retval = FALSE;
         }
         else
         {
            memcpy((void *)clut->clut_8_bit, (void *)default_8_bit_clut, (size_t)(256 * 4));
            clut->default_8_bit = FALSE;
            clut->clut_8_bit[clut_entry_id] = palette_entry;
         }
      }
      else
      {
         clut->clut_8_bit[clut_entry_id] = palette_entry;
      }
   }

   FUNCTION_FINISH(MakeClutEntry);

   return(retval);
}

#ifdef STB_REGION_PRINT_REQUIRED
/**
 *

 *
 * @brief   Debug function that checks that the 5 values [region_width, region_height,
 *                 region_level_of_compatability, region_depth & clut_id] which cannot change,
 *                 are indeed the same values. ETS 300 743 5.1.4
 *
 * @param   region                        - region being checked from previous RCS definition
 * @param   region_width                  - the value from this RCS
 * @param   region_height,                - the value from this RCS
 * @param   region_level_of_compatability - the value from this RCS
 * @param   region_depth                  - the value from this RCS
 * @param   clut_id                       - the value from this RCS
 *

 *
 */
void CheckRegionFootPrint(S_EPOCH_REGION *region, U16BIT region_width, U16BIT region_height,
   U16BIT region_level_of_compatability, U16BIT region_depth, U16BIT clut_id)
{
   FUNCTION_START(CheckRegionFootPrint);

   // The following fields cannot change once set ETS 300 743 5.1.4
   if (region->region_width != region_width)
   {
      STB_REGION_PRINT(("Attempt to redefine width"));
   }
   if (region->region_height != region_height)
   {
      STB_REGION_PRINT(("Attempt to redefine height"));
   }
   if (region->region_level_of_compatibilty != region_level_of_compatability)
   {
      STB_REGION_PRINT(("Attempt to redefine compatibilty"));
   }
   if (region->region_colour_depth != region_depth)
   {
      STB_REGION_PRINT(("Attempt to redefine depth"));
   }
   if (region->region_clut_id != clut_id)
   {
      STB_REGION_PRINT(("Attempt to redefine clut id"));
   }

   FUNCTION_FINISH(CheckRegionFootPrint);
}

#endif

/**
 *

 *
 * @brief   Destroys all the passed object's allocated memory, if the flag is set the
 *                 object itself is also freed.
 *
 * @param   obj     - the object
 * @param   del_obj - whether the object is freed
 *

 *
 */
static void DestroyObject(S_OBJECT **obj, BOOLEAN del_obj)
{
   FUNCTION_START(DestroyObject);

   ASSERT(*obj != NULL);

   if ((*obj)->bitmap)
   {
      STB_FreeMemory((*obj)->bitmap);
   }
   if ((*obj)->character_code)
   {
      STB_FreeMemory((*obj)->character_code);
   }
   if ((*obj)->map_table_2_to_4_bit != (U8BIT *)default_2_bit_4_bit_map)
   {
      STB_FreeMemory((*obj)->map_table_2_to_4_bit);
   }
   if ((*obj)->map_table_2_to_8_bit != (U8BIT *)default_2_bit_8_bit_map)
   {
      STB_FreeMemory((*obj)->map_table_2_to_8_bit);
   }
   if ((*obj)->map_table_4_to_8_bit != (U8BIT *)default_4_bit_8_bit_map)
   {
      STB_FreeMemory((*obj)->map_table_4_to_8_bit);
   }

   if (del_obj)
   {
      STB_FreeMemory(*obj);
      *obj = NULL;
   }
   else
   {
      memset((void *)*obj, 0x00, sizeof(S_OBJECT));
      (*obj)->map_table_2_to_4_bit = (U8BIT *)default_2_bit_4_bit_map;
      (*obj)->map_table_2_to_8_bit = (U8BIT *)default_2_bit_8_bit_map;
      (*obj)->map_table_4_to_8_bit = (U8BIT *)default_4_bit_8_bit_map;
   }

   FUNCTION_FINISH(DestroyObject);
}

/**
 *

 *
 * @brief   Delete a list of objects from the object passed.
 *
 * @param   obj - the point at which a list is to be deleted.
 *

 *
 */
static void DeleteObjectList(S_OBJECT **obj)
{
   S_OBJECT *ptr_1;
   S_OBJECT *ptr_2;


   FUNCTION_START(DeleteObjectList);

   ptr_1 = *obj;

   while (ptr_1 != NULL)
   {
      ptr_2 = ptr_1->next;
      DestroyObject(&ptr_1, TRUE);
      ptr_1 = ptr_2;
   }

   *obj = NULL;

   FUNCTION_FINISH(DeleteObjectList);
}

/**
 *

 *
 * @brief   Delete a list of CLUTs from the CLUT passed.
 *
 * @param   clut - pointer to a S_CLUT structure
 *

 *
 */
static void DeleteClutList(S_CLUT **clut)
{
   S_CLUT *ptr_1;
   S_CLUT *ptr_2;


   FUNCTION_START(DeleteClutList);

   ptr_1 = *clut;

   while (ptr_1 != NULL)
   {
      if (!ptr_1->default_2_bit)
      {
         STB_FreeMemory(ptr_1->clut_2_bit);
      }

      if (!ptr_1->default_4_bit)
      {
         STB_FreeMemory(ptr_1->clut_4_bit);
      }

      if (!ptr_1->default_8_bit)
      {
         STB_FreeMemory(ptr_1->clut_8_bit);
      }

      ptr_2 = ptr_1->next;
      STB_FreeMemory(ptr_1);
      ptr_1 = ptr_2;
   }

   *clut = NULL;

   FUNCTION_FINISH(DeleteClutList);
}

/**
 *

 *
 * @brief   Deletes the contents of the epoch region list from the region_ptr passed.
 *
 * @param   region_ptr - the epoch region list beginning to delete from.
 *

 *
 */
static void DeleteEpochRegionList(S_EPOCH_REGION **region_list)
{
   S_EPOCH_REGION *ptr_r1;
   S_EPOCH_REGION *ptr_r2;
   S_REGION_OBJECT *ptr_o1;
   S_REGION_OBJECT *ptr_o2;


   FUNCTION_START(DeleteEpochRegionList);

   ptr_r1 = *region_list;

   while (ptr_r1 != NULL)
   {
      ptr_o1 = ptr_r1->region_object_list;
      while (ptr_o1 != NULL)
      {
         // Delete all S_REGION_OBJECT's from each S_EPOCH_REGION
         ptr_o2 = ptr_o1->next;
         STB_FreeMemory(ptr_o1);
         ptr_o1 = ptr_o2;
      }

      // Delete S_EPOCH_REGION
      ptr_r2 = ptr_r1->next;
      STB_FreeMemory(ptr_r1);
      ptr_r1 = ptr_r2;
   }

   *region_list = NULL;

   FUNCTION_FINISH(DeleteEpochRegionList);
}

/**
 *

 *
 * @brief   Deletes the contents of the page composition region list from the comp_pg
 * @param   pointer passed
 *
 * @param   comp_pg - the composition page's region list to be deleted from pointer.
 *

 *
 */
static void DeletePageCompositionRegions(S_REGION **comp_pg)
{
   S_REGION *ptr_1;
   S_REGION *ptr_2;


   FUNCTION_START(DeletePageCompositionRegions);

   ptr_1 = *comp_pg;

   while (ptr_1 != NULL)
   {
      ptr_2 = ptr_1->next;
      STB_FreeMemory(ptr_1);
      ptr_1 = ptr_2;
   }

   *comp_pg = NULL;

   FUNCTION_FINISH(DeletePageCompositionRegions);
}

//---global function definitions-----------------------------------------------

/**
 *

 *
 * @brief   Initilises DVB subtitles, allocates a temporary buffer for use as a scratchpad
 *                 and zero's the displayset structure and assigns the pages.
 *

 *
 * @return   BOOLEAN - True/False for success/failure.
 *
 */
BOOLEAN STB_DSInitialiseDVBSubtitlesProcessing(void)
{
   FUNCTION_START(STB_DSInitialiseDVBSubtitlesProcessing);

   temp_buffer = (U8BIT *)STB_GetMemory(TEMP_BUFFER_SIZE);
   if (temp_buffer == NULL)
   {
      STB_SUB_DRIVER_PRINT(("Initialation failure!!!"));
   }
   else
   {
      STB_SUB_DRIVER_PRINT(("Initialised OK"));

      memset((void *)&subtitle_display_set, 0x00, sizeof(S_DISPLAY_SET));
      memset((void *)&page_1, 0x00, sizeof(S_PAGE_COMPOSITION));
      memset((void *)&page_2, 0x00, sizeof(S_PAGE_COMPOSITION));

      /* Initialise the DDS version to an invalid value to ensure the first DDS segment found is used */
      page_1.dds_present = FALSE;
      page_1.dds_version = 0xff;
      page_1.display_width = 719;
      page_1.display_height = 575;
      page_1.display_window = FALSE;
      page_1.page_version_number = 0xff;

      page_2.dds_present = FALSE;
      page_2.dds_version = 0xff;
      page_2.display_width = 719;
      page_2.display_height = 575;
      page_2.display_window = FALSE;
      page_2.page_version_number = 0xff;

      subtitle_display_set.page_display_buffer = &page_1;
      subtitle_display_set.page_composition_buffer = &page_2;
   }

   FUNCTION_FINISH(STB_DSInitialiseDVBSubtitlesProcessing);

   return(FALSE);
}

/**
 *

 *
 * @brief   Finds or malloc's a new CLUT and assigns the family pointers to the default data.
 *                 version_id is set to 0xff where the valid values are in the range 0 .. 0xf[4bits]
 *
 * @param   clut_list - pointer to a S_CLUT structure
 * @param   clut_id   - the index of the CLUT to be modified
 *
 * @return   S_CLUT pointer.
 *
 */
S_CLUT* STB_DSGetClut(S_CLUT *clut_list, U16BIT clut_id)
{
   S_CLUT *clut;

   FUNCTION_START(STB_DSGetClut);

   clut = clut_list;

   // search clut list to find clut ID
   while (clut != NULL)
   {
      if (clut->clut_id == clut_id)
      {
         break;
      }
      clut = clut->next;
   }

   if (clut == NULL)
   {
      clut = (S_CLUT *)STB_GetMemory(sizeof(S_CLUT));
      if (clut != NULL)
      {
         STB_CDS_PRINT(("  +c%d", clut_id));

         clut->clut_id = clut_id;
         clut->next = NULL;
         clut->clut_2_bit = default_2_bit_clut;
         clut->clut_4_bit = default_4_bit_clut;
         clut->clut_8_bit = default_8_bit_clut;
         clut->default_2_bit = TRUE;
         clut->default_4_bit = TRUE;
         clut->default_8_bit = TRUE;
         clut->clut_version_number = 0xff;

         // Insert at head of link list of cluts
         clut->next = subtitle_display_set.clut_list;
         subtitle_display_set.clut_list = clut;
      }
      else
      {
         STB_CDS_PRINT(("CLUT %d malloc fail", clut_id));
      }
   }
   // else - found an existing CLUT

   FUNCTION_FINISH(STB_DSGetClut);

   return(clut);
}

/**
 *

 *
 * @brief   Processes the Display Definition segment in accordance with ETSI document
 *                 ETS 300 743.
 *
 * @param   data - pointer to the beginning of the segment, the sync_byte.
 *
 * @return   BOOLEAN - True/False for success/failure.
 *
 */
BOOLEAN STB_DSSegmentDDS(U8BIT *data, U16BIT pes_len, U16BIT processed_bytes)
{
   U16BIT segment_length;
   U16BIT processed_length;
   U8BIT dds_version;
   BOOLEAN display_window;
   U16BIT display_width, display_height;
   BOOLEAN retval;

   FUNCTION_START(STB_DSSegmentDDS);

   ASSERT(data != NULL);

   // data[0] - sync_byte
   // data[1] - segment_type
   // data[2] - page_id MS8Bits
   // data[3] - page_id LS8Bits

   segment_length = (data[4] << 8) + data[5];

   STB_SEGMENT_PRINT(("DDS: segment_length=%u", segment_length));

   if ((processed_bytes + segment_length + 5) > pes_len)
   {
      STB_SEGMENT_PRINT(("\n\nABORT DDS processing - pes too short"));
      return(FALSE);
   }

   dds_version = (data[6] & 0xf0) >> 4;

   display_window = ((data[6] & 0x08) != 0);
   display_width = (data[7] << 8) + data[8];
   display_height = (data[9] << 8) + data[10];

   processed_length = 10;

   STB_SEGMENT_PRINT(("DDS: display %ux%u", display_width, display_height));

   if (display_window)
   {
      processed_length += 8;
   }

   if (subtitle_display_set.page_composition_buffer != NULL)
   {
      subtitle_display_set.page_composition_buffer->dds_present = TRUE;

      if (dds_version != subtitle_display_set.page_composition_buffer->dds_version)
      {
         subtitle_display_set.page_composition_buffer->dds_version = dds_version;

         subtitle_display_set.page_composition_buffer->display_width = display_width;
         subtitle_display_set.page_composition_buffer->display_height = display_height;
         subtitle_display_set.page_composition_buffer->display_window = display_window;

         if (display_window)
         {
            subtitle_display_set.page_composition_buffer->window_x = (data[11] << 8) + data[12];
            subtitle_display_set.page_composition_buffer->window_width = ((data[13] << 8) + data[14]) -
               subtitle_display_set.page_composition_buffer->window_x + 1;
            subtitle_display_set.page_composition_buffer->window_y = (data[15] << 8) + data[16];
            subtitle_display_set.page_composition_buffer->window_height = ((data[17] << 8) + data[18]) -
               subtitle_display_set.page_composition_buffer->window_y + 1;

            STB_SEGMENT_PRINT(("DDS: window %u,%u : %ux%u",
                               subtitle_display_set.page_composition_buffer->window_x,
                               subtitle_display_set.page_composition_buffer->window_y,
                               subtitle_display_set.page_composition_buffer->window_width,
                               subtitle_display_set.page_composition_buffer->window_height));
         }
      }
   }

   // Return OK only if we processed the number of bytes we were given
   retval = (processed_length == (segment_length + 5));

   FUNCTION_FINISH(STB_DSSegmentDDS);

   return(retval);
}

/**
 *

 *
 * @brief   Processes the Page Composition segment in accordance with ETSI document
 *                 ETS 300 743.
 *
 * @param   data - pointer to the beginning of the segment, the sync_byte.
 *                    force_acquisition - signal to treat next acquisition point as a mode change,
 *                    unless a mode change occurs first
 *
 * @return   BOOLEAN - True/False for success/failure.
 *
 */
BOOLEAN STB_DSSegmentPCS(U8BIT *data, U16BIT pes_len, U16BIT processed_bytes, BOOLEAN external_force_acquisition)
{
   S_REGION *region;
   U16BIT segment_length;
   U16BIT processed_length;

   U8BIT page_time_out;
   U8BIT page_version_number;
   U8BIT page_state;
   BOOLEAN retval = FALSE;

   static BOOLEAN force_acquisition = TRUE;

   FUNCTION_START(STB_DSSegmentPCS);

   ASSERT(data != NULL);

   // data[0] - sync_byte
   // data[1] - segment_type
   // data[2] - page_id MS8Bits
   // data[3] - page_id LS8Bits

   //force_update if we wanted to last time OR we've been told to this time
   force_acquisition |= external_force_acquisition;
   // Remember if we've been told to force at any point - in case of errors,
   //  we can reset force_acquisition
   external_force_acquisition = force_acquisition;

   segment_length = (data[4] << 8) + data[5];

   if ((processed_bytes + segment_length + 6) > pes_len)
   {
      STB_SEGMENT_PRINT(("\n\nABORT PCS processing - pes too short"));

      return(FALSE);
   }

   page_time_out = data[6];
   page_version_number = (data[7] & 0xf0) >> 4;
   page_state = (data[7] & 0x0c) >> 2;

   processed_length = 2;

   // page_state '11' reserved for future use.
   if ((subtitle_display_set.page_composition_buffer != NULL) && (page_state != RES_PAGE_STATE))
   {
      /* Check whether a DDS has been seen */
      if (!subtitle_display_set.page_composition_buffer->dds_present)
      {
         /* Set default display settings */
         subtitle_display_set.page_composition_buffer->display_width = 719;
         subtitle_display_set.page_composition_buffer->display_height = 575;
         subtitle_display_set.page_composition_buffer->display_window = FALSE;
      }

      // Check PCS describes a new version of Page
      if ((subtitle_display_set.page_display_buffer->page_version_number != page_version_number) || force_acquisition)
      {
         STB_SEGMENT_PRINT(("\n\nPCS <state %d> Ver v%d ( Comp v%d / Disp v%d )  timeout %d",
                            page_state, page_version_number,
                            subtitle_display_set.page_composition_buffer->page_version_number,
                            subtitle_display_set.page_display_buffer->page_version_number,
                            page_time_out));

         // Delete the existing page composition buffer of region lists
         STB_DSClearCompositionPageDetails();

         if ((page_state == MODE_CHANGE) ||
             ((page_state == ACQUISITION_POINT) && force_acquisition))
         {
            page_state = MODE_CHANGE;
            //Okay we're forcing!
            force_acquisition = FALSE;
         }

         switch (page_state)
         {
            case NORMAL_CASE:
            {
               STB_SEGMENT_PRINT(("pcs NORMAL_CASE"));
               // '00' normal case The page composition segment is followed by an
               // incomplete region set.
               break;
            }

            case ACQUISITION_POINT:
            {
               // NB we could have started with an acquisition point, but have been forced to
               // treat this as a mode change
               STB_SEGMENT_PRINT(("pcs ACQUISITION_POINT - treating as mode change"));
               page_state = MODE_CHANGE;
               // '01' acquisition point The page composition segment is followed
               // by a complete region set describing the current memory plan.
               break;
            }

            case MODE_CHANGE:
            {
               if (external_force_acquisition)
               {
                  STB_SEGMENT_PRINT(("pcs FORCED Reset display region list + Obj + CLUT ++ EPOCH"));
               }
               else
               {
                  STB_SEGMENT_PRINT(("pcs Reset display region list + Obj + CLUT ++ EPOCH"));
               }
               // '10' mode change The page composition segment is followed by regions
               // describing a new memory plan.

               // Forget outdated list of Regions / Objects / Cluts from previous Epoch
               DeleteEpochRegionList(&subtitle_display_set.region_list);
               DeleteObjectList(&subtitle_display_set.object_list);
               DeleteClutList(&subtitle_display_set.clut_list);
               // Clear out any OSD resources that we accumulated in previous Epoch
               STB_DSResetPhysicalCompositionRegions();
               break;
            }
         }

         // Make the composition buffer ours
         subtitle_display_set.page_composition_buffer->page_time_out = page_time_out;
         subtitle_display_set.page_composition_buffer->page_version_number = page_version_number;
         subtitle_display_set.page_composition_buffer->page_state = page_state;

         // ETS 300 743 7.2.1
         // page_time_out: The period, expressed in seconds, after which the page is no longer
         // valid and consequently shall be erased from the screen, should it not have been
         // redefined before that. The time-out period starts at the first reception of the
         // page_composition_segment. If the same segment with the same version number is received
         // again the time-out counter shall not be reloaded. The purpose of the time-out period is
         // to avoid that a page remains on the screen "for ever" if the IRD happens to have missed
         // the page's redefinition or deletion. The time-out period does not need to be counted
         // very accurately by the IRD: a reaction inaccuracy of -0/+5 seconds is good enough.
         subtitle_display_set.page_composition_buffer->timeout_start = STB_OSGetClockMilliseconds();

         // Move data pointer to 1st region
         data += 8;

         // Build list of regions to be displayed in this page
         while (processed_length < segment_length)
         {
            region = (S_REGION *)STB_GetMemory(sizeof(S_REGION));

            if (region != NULL)
            {
               region->region_id = data[0];
               region->region_horizontal_address = (data[2] << 8) + data[3];
               region->region_vertical_address = (data[4] << 8) + data[5];

               if (subtitle_display_set.page_composition_buffer->display_window)
               {
                  region->region_horizontal_address += subtitle_display_set.page_composition_buffer->window_x;
                  region->region_vertical_address += subtitle_display_set.page_composition_buffer->window_y;

                  if (region->region_horizontal_address > subtitle_display_set.page_composition_buffer->display_width)
                  {
                     region->region_horizontal_address = subtitle_display_set.page_composition_buffer->display_width;
                  }

                  if (region->region_vertical_address > subtitle_display_set.page_composition_buffer->display_height)
                  {
                     region->region_vertical_address = subtitle_display_set.page_composition_buffer->display_height;
                  }
               }
               else
               {
                  if (region->region_horizontal_address > subtitle_display_set.page_composition_buffer->display_width)
                  {
                     region->region_horizontal_address = subtitle_display_set.page_composition_buffer->display_width;
                  }

                  if (region->region_vertical_address > subtitle_display_set.page_composition_buffer->display_height)
                  {
                     region->region_vertical_address = subtitle_display_set.page_composition_buffer->display_height;
                  }
               }

               // Insert at head of link list of composition regions
               region->next = subtitle_display_set.page_composition_buffer->region_list;
               subtitle_display_set.page_composition_buffer->region_list = region;

               STB_SEGMENT_PRINT(("  +r%d @ (%3d, %3d)",
                                  region->region_id,
                                  region->region_horizontal_address, region->region_vertical_address));
            }

            // Move data pointer to next region
            data += 6;
            processed_length += 6;
         }

         retval = TRUE;
      }
      else
      {
         STB_SEGMENT_PRINT(("\n\nPCS <state %d> Ver v%d ( Comp v%d / Disp v%d ) -- Unchanged version - IGNORING (force_acquisition=%s)",
                            page_state, page_version_number,
                            subtitle_display_set.page_composition_buffer->page_version_number,
                            subtitle_display_set.page_display_buffer->page_version_number,
                            force_acquisition ? "TRUE" : "FALSE"));

         subtitle_display_set.page_composition_buffer->page_state = page_state;
         retval = TRUE;
      }

      // Error if we already had an error OR we didn't use all of the expected bytes
      retval &= (processed_length == segment_length);
   }
   else
   {
      // We have no composition buffer OR received page's state is invalid
      STB_SEGMENT_PRINT(("\n\nPCS -- Not processed, illegal state %s@%d", __FILE__, __LINE__));
      retval = FALSE;
   }

   FUNCTION_FINISH(STB_DSSegmentPCS);

   return(retval);
}

/**
 *

 *
 * @brief   Processes the Region Composition segment in accordance with ETSI document
 *                 ETS 300 743.
 *
 * @param   data - pointer to the beginning of the segment, the sync_byte.
 *
 * @return   BOOLEAN - True/False for success/failure.
 *
 */
BOOLEAN STB_DSSegmentRCS(U8BIT *data, U16BIT pes_len, U16BIT processed_bytes)
{
   S_REGION_OBJECT *object;
   S_REGION_OBJECT *existing_region_object;
   S_EPOCH_REGION *region;
   U16BIT segment_length;
   U16BIT processed_length;
   U16BIT region_id;
   U8BIT region_version_number;
   U16BIT object_id;
   U16BIT object_type;

   BOOLEAN retval;

   FUNCTION_START(STB_DSSegmentRCS);

   ASSERT(data != NULL);

   // data[0] - sync_byte
   // data[1] - segment_type
   // data[2] - page_id MS8Bits
   // data[3] - page_id LS8Bits

   segment_length = (data[4] << 8) + data[5];

   if ((processed_bytes + segment_length + 6) > pes_len)
   {
      STB_SEGMENT_PRINT(("\n\nABORT RCS processing - pes too short"));

      return(FALSE);
   }

   region_id = data[6];
   region_version_number = (data[7] & 0xf0) >> 4;

   processed_length = 10;

   // search EPOCH's master region list to find region ID
   region = subtitle_display_set.region_list;

   while (region != NULL)
   {
      if (region->region_id == region_id)
      {
         break;
      }
      region = region->next;
   }

   // NB if we are at Epoch change then we should NEVER find an existing region
   if (region != NULL)
   {
      // Region with this ID was mentioned at last mode_change / Epoch
      if (region->region_version_number != region_version_number)
      {
         // Existing region in epoch's region list, NEW version => reset the object list
         S_REGION_OBJECT *object_1;
         S_REGION_OBJECT *object_2;

         STB_SEGMENT_PRINT(("RCS Region r%d (existing version %d) new ver %d  c%d  %s 8_%d 4_%d 2_%d",
                            region_id, region->region_version_number, region_version_number,
                            data[13],
                            ((data[7] & 0x08) >> 3) ? "FILL" : "no fill",
                            data[14], (data[15] & 0xf0) >> 4, (data[15] & 0x0c) >> 2));

         object_1 = region->region_object_list;
         while (object_1 != NULL)
         {
            // Delete all S_REGION_OBJECT's from each S_EPOCH_REGION
            object_2 = object_1->next;
            STB_FreeMemory(object_1);
            object_1 = object_2;
         }
         region->region_object_list = NULL;
      }
      else
      {
         // Existing region in epoch's region list, NOT NEW version
         STB_SEGMENT_PRINT(("RCS Region r%d version number %d unchanged -- IGNORING",
                            region_id, region->region_version_number, region_version_number));
         // No version change => ignore this RCS.  Set region = NULL so no further processing
         region = NULL;
      }
   }
   // else - no existing Region found, can only create one if we are in MODE_CHANGE state
   // (which might have been caused by Acquisition Point being forced to mode change - see PCS)
   else if (subtitle_display_set.page_composition_buffer->page_state == MODE_CHANGE)
   {
      // New region - unknown in master region list
      // But we have a suitable page_state => we should create a new region
      region = (S_EPOCH_REGION *)STB_GetMemory(sizeof(S_EPOCH_REGION));

      if (region != NULL)
      {
         // Initialise ourselves
         region->region_id = region_id;
         region->region_version_number = region_version_number;
         region->region_object_list = NULL;

         // The following fields cannot change once set ETS 300 743 5.1.4
         region->region_width = (data[8] << 8) + data[9];

         region->region_height = (data[10] << 8) + data[11];
         region->region_level_of_compatibilty = (data[12] & 0xe0) >> 5;
         region->region_level_of_compatibilty = 1 << region->region_level_of_compatibilty;
         region->region_clut_id = data[13];
         region->region_colour_depth = (data[12] & 0x1c) >> 2;
         region->region_colour_depth = 1 << region->region_colour_depth;

         // These parts of the region can change
         region->region_fill_flag = (data[7] & 0x08) >> 3;
         region->region_8_bit_pixel_code = data[14];
         region->region_4_bit_pixel_code = (data[15] & 0xf0) >> 4;
         region->region_2_bit_pixel_code = (data[15] & 0x0c) >> 2;

         STB_SEGMENT_PRINT(("RCS r%d NEW ver %d  (%dx%d(x%d)) compat %x  c%d  %s 8_%d 4_%d 2_%d",
                            region->region_id,
                            region->region_version_number,
                            region->region_width,
                            region->region_height,
                            region->region_colour_depth,
                            region->region_level_of_compatibilty,
                            data[13],
                            ((data[7] & 0x08) >> 3) ? "FILL" : "no fill",
                            data[14], (data[15] & 0xf0) >> 4, (data[15] & 0x0c) >> 2));

         // Insert at head of master link list of regions
         region->next = subtitle_display_set.region_list;
         subtitle_display_set.region_list = region;

         // Get CLUT for defined ID (may get default CLUT if no matching CLUT found)
         STB_DSGetClut(subtitle_display_set.clut_list, region->region_clut_id);

         #ifdef STB_REGION_PRINT_REQUIRED
         if (subtitle_display_set.page_composition_buffer->page_state != MODE_CHANGE)
         {
            CheckRegionFootPrint(region,
               region->region_width,
               region->region_height,
               region->region_level_of_compatibilty,
               region->region_colour_depth,
               region->region_clut_id);
         }
         #endif
      }
   }
   else
   {
      // page_state means that we shouldn't be creating regions !!
      STB_SEGMENT_PRINT(("RCS Region r%d (No pre-existing vn) new ver %d - CANNOT CREATE - page state %d",
                         region_id, region_version_number,
                         subtitle_display_set.page_composition_buffer->page_state));
      // region is still NULL
      region = NULL;
   }

   // if non-null Region now points to entry in master list of regions (EPOCH's list)
   //   all internal objects have been deleted
   // if NULL then we failed to malloc memory for a new region OR weren't allowed to create
   if (region != NULL)
   {
      region->region_version_number = region_version_number;

      // These parts of the region can change
      region->region_fill_flag = (data[7] & 0x08) >> 3;
      region->region_8_bit_pixel_code = data[14];
      region->region_4_bit_pixel_code = (data[15] & 0xf0) >> 4;
      region->region_2_bit_pixel_code = (data[15] & 0x0c) >> 2;

      // Create a new composition region, and attaches to the physical composition list.
      STB_DSCreateCompositionRegion(region,
         subtitle_display_set.page_composition_buffer->page_state);

      data += 16;

      // Add the object references listed in the RCS
      while (processed_length < segment_length)
      {
         object_id = (data[0] << 8) + data[1];
         object_type = (data[2] & 0xc0) >> 6;

         existing_region_object = region->region_object_list;
         while (existing_region_object != NULL)
         {
            if (existing_region_object->object_id == object_id)
            {
               object = existing_region_object;
               break;
            }
            existing_region_object = existing_region_object->next;
         }

         if (existing_region_object == NULL)
         {
            // Region needs to add new object
            object = (S_REGION_OBJECT *)STB_GetMemory(sizeof(S_REGION_OBJECT));
         }

         if (object != NULL)
         {
            object->object_id = object_id;
            object->object_type = (data[2] & 0xc0) >> 6;
            object->object_provider_flag = (data[2] & 0x30) >> 4;
            object->object_horizontal_position = ((data[2] & 0x0f) << 8) + data[3];
            object->object_vertical_position = ((data[4] & 0x0f) << 8) + data[5];

            if ((object->object_type == 0x01) || (object->object_type == 0x02))
            {
               object->object_foreground_pixel_code = data[6];
               object->object_background_pixel_code = data[7];
            }

            if (existing_region_object == NULL)
            {
               object->next = region->region_object_list;
               region->region_object_list = object;
               STB_SEGMENT_PRINT(("  +o%d @(%d,%d)",
                                  object->object_id,
                                  object->object_horizontal_position,
                                  object->object_vertical_position));
            }
            else
            {
               STB_SEGMENT_PRINT(("  ~o%d @(%d,%d)",
                                  object->object_id,
                                  object->object_horizontal_position,
                                  object->object_vertical_position));
            }
         }
         else
         {
            // Object malloc failed, subsequent mallocs are likley to fail for this segment.
            // Will cause parse error <= processed length will be wrong
            break;
         }

         if ((object_type == 0x01) || (object_type == 0x02))
         {
            // Move data pointer to next object
            data += 8;
            processed_length += 8;
         }
         else
         {
            // Move data pointer to next object
            data += 6;
            processed_length += 6;
         }
      }
   }

   // Return OK only if we processed the number of bytes we were given
   retval = (processed_length == segment_length);

   FUNCTION_FINISH(STB_DSSegmentRCS);

   return(retval);
}

/**
 *

 *
 * @brief   Processes the CLUT Definition segment in accordance with ETSI document
 *                 ETS 300 743.
 *
 * @param   data - pointer to the beginning of the segment, the sync_byte.
 *
 * @return   BOOLEAN - True/False for success/failure.
 *
 */
BOOLEAN STB_DSSegmentCDS(U8BIT *data, U16BIT pes_len, U16BIT processed_bytes)
{
   S_CLUT *clut;
   U32BIT palette_entry;
   U16BIT segment_length;
   U16BIT processed_length;
   U16BIT clut_id;
   U8BIT clut_version_number;
   U8BIT display_version_number;
   U16BIT clut_entries = 0;

   U8BIT clut_entry_id;
   U8BIT clut_2_4_8_bit_entry_flags;
   U8BIT full_range_flag;

   U8BIT y;
   U8BIT cr;
   U8BIT cb;
   U8BIT t;

   BOOLEAN retval;


   FUNCTION_START(STB_DSSegmentCDS);

   ASSERT(data != NULL);

   // data[0] - sync_byte
   // data[1] - segment_type
   // data[2] - page_id MS8Bits
   // data[3] - page_id LS8Bits

   segment_length = (data[4] << 8) + data[5];

   if ((processed_bytes + segment_length + 6) > pes_len)
   {
      return(FALSE);
   }

   clut_id = data[6];

   clut_version_number = (data[7] & 0xf0) >> 4;
   data += 8;

   processed_length = 2;

   clut = STB_DSGetClut(subtitle_display_set.clut_list, clut_id);

   display_version_number = clut->clut_version_number;

   if (display_version_number == clut_version_number)
   {
      // ETS 300 743 8.1 Scope of Identifiers
      // 'All identifiers (region_id, CLUT_id, object_id) are unique within a
      // display built from a composition page and an ancillary page.'

      // DTG - Digital Terrestrial Television Requirements for Interoperability
      // Subtitling: 28 August 1998: Version: 3.0 Clause 5.5.5.3
      // 'Decoder behaviour is not defined if segments of the same type and id
      // value are repeated within a single display set.'
      STB_SEGMENT_PRINT(("CDS c%d version number %d unchanged - ignoring",
                         clut_id, clut_version_number));

      /* Don't need to process the segment.
       * Set the processed length to avoid an error being shown */
      clut = NULL;
      processed_length = segment_length;
   }
   //else - new version.  Now parse entries.

   if (clut != NULL)
   {
      STB_CLUT_PRINT(("CDS c%d v%d", clut_id, clut_version_number));
      while (processed_length < segment_length)
      {
         clut_entry_id = data[0];
         clut_2_4_8_bit_entry_flags = (data[1] & 0xe0) >> 5;
         full_range_flag = data[1] & 0x01;

         if (full_range_flag)
         {
            y = data[2];
            cr = data[3];
            cb = data[4];
            t = data[5];

            processed_length += 6;
            data += 6;
         }
         else
         {
            /* Reduced range defining the most significant bits of each value */
            y = (data[2] & 0xfc);
            cr = ((data[2] & 0x03) << 6) + ((data[3] & 0xc0) >> 2);
            cb = (data[3] & 0x3c) << 2;
            t = (data[3] & 0x03) << 6;

            processed_length += 4;
            data += 4;
         }

         // ETS 300 743 7.2.3
         // Y_value: The Y output value of the CLUT for this entry. A value of zero in the Y_value
         // field signals full transparency. In that case the values in the Cr_value, Cb_value and
         // T_value fields are irrelevant and shall be set to zero.
         // Full transparency is acquired through a value of zero in the Y_value field.
         if (y == 0x00)
         {
            palette_entry = 0xff000000;
         }
         else
         {
            palette_entry = (t << 24) + (y << 16) + (cr << 8) + cb;
         }

         MakeClutEntry(clut, clut_id, clut_entry_id, clut_2_4_8_bit_entry_flags, palette_entry);
         STB_CLUT_PRINT(("  %3d> p%08x f%02x  ", clut_entry_id, palette_entry, clut_2_4_8_bit_entry_flags));
         ++clut_entries;
      }

      STB_SEGMENT_PRINT(("CDS c%d v%d -- %d entries", clut_id, clut_version_number, clut_entries));
      clut->clut_version_number = clut_version_number;
   }

   if (processed_length == segment_length)
   {
      retval = TRUE;
   }
   else
   {
      retval = FALSE;
   }

   FUNCTION_FINISH(STB_DSSegmentCDS);

   return(retval);
}

/**
 *

 *
 * @brief   Processes the Object Data segment in accordance with ETSI document
 *                 ETS 300 743.
 *
 * @param   data - pointer to the beginning of the segment, the sync_byte.
 *
 * @return   BOOLEAN - True/False for success/failure.
 *
 */
BOOLEAN STB_DSSegmentODS(U8BIT *data, U16BIT pes_len, U16BIT processed_bytes)
{
   S_OBJECT *object;
   U16BIT segment_length;
   U16BIT object_id;
   U8BIT object_version_number;
   U8BIT object_coding_method;
   U8BIT non_modifying_colour_flag;
   BOOLEAN retval;

   FUNCTION_START(STB_DSSegmentODS);

   ASSERT(data != NULL);

   retval = FALSE;

   // data[0] - sync_byte
   // data[1] - segment_type
   // data[2] - page_id MS8Bits
   // data[3] - page_id LS8Bits

   segment_length = (data[4] << 8) + data[5];

   if ((processed_bytes + segment_length + 6) > pes_len)
   {
      return(FALSE);
   }

   object_id = (data[6] << 8) + data[7];

   object_version_number = (data[8] & 0xf0) >> 4;
   object_coding_method = (data[8] & 0x0c) >> 2;
   non_modifying_colour_flag = (data[8] & 0x02) >> 1;

   object = subtitle_display_set.object_list;

   // search object list to find object ID
   while (object != NULL)
   {
      if (object->object_id == object_id)
      {
         break;
      }
      object = object->next;
   }

   if (object != NULL)
   {
      if (object->object_version_number != object_version_number)
      {
         // new version of existing object
         STB_SEGMENT_PRINT(("ODS o%d (existing version v%d) new ver v%d",
                            object_id,
                            object->object_version_number,
                            object_version_number));
      }
      else
      {
         STB_SEGMENT_PRINT(("ODS o%d version number v%d unchanged -- IGNORING",
                            object_id, object_version_number));
         /* ETS 300 743 8.1 Scope of Identifiers
          * 'All identifiers (region_id, CLUT_id, object_id) are unique within a
          * display built from a composition page and an ancillary page.' */

         /* DTG - Digital Terrestrial Television Requirements for Interoperability
          * Subtitling: 28 August 1998: Version: 3.0 Clause 5.5.5.3
          * 'Decoder behaviour is not defined if segments of the same type and id
          * value are repeated within a single display set.' */

         /* The above statement defined the original behaviour with the object being set
          * to NULL which meant a parsing error was detected and no subtitles were shown.
          * However, it's desirable for subtitles to be shown if at all possible, so this
          * is no longer treated as an error. */
         /* object = NULL; */
      }
   }
   else
   {
      // New object
      object = (S_OBJECT *)STB_GetMemory(sizeof(S_OBJECT));
      if (object != NULL)
      {
         STB_SEGMENT_PRINT(("ODS o%d ver v%d", object_id, object_version_number));

         memset((void *)object, 0x00, sizeof(S_OBJECT));

         object->next = subtitle_display_set.object_list;
         subtitle_display_set.object_list = object;
      }
      else
      {
         STB_SEGMENT_PRINT(("ODS o%d ver v%d malloc fail", object_id, object_version_number));
      }
   }

   if (object != NULL)
   {
      object->object_id = object_id;
      object->object_version_number = object_version_number;
      object->object_coding_method = object_coding_method;
      object->non_modifying_colour_flag = non_modifying_colour_flag;
      object->map_table_2_to_4_bit = default_2_bit_4_bit_map;
      object->map_table_2_to_8_bit = default_2_bit_8_bit_map;
      object->map_table_4_to_8_bit = default_4_bit_8_bit_map;

      if (ProcessObjectCodingMethod(data + 9, object, object_coding_method, segment_length))
      {
         // processed_length == segment_length
         retval = TRUE;
      }
      else
      {
         // processed_length != segment_length
         retval = FALSE;
      }
   }

   FUNCTION_FINISH(STB_DSSegmentODS);

   return(retval);
}

/**
 *

 *
 * @brief   DTG - Digital Terrestrial Television Requirements for Interoperability
 *                 Subtitling: 28 August 1998: Version: 3.0 Clause 5.7.1
 *                 End of data segment, receit of this means the page is complete.
 *
 * @param   data - pointer to the beginning of the segment, the sync_byte.
 *
 * @return   TRUE - success, FALSE for failure.
 *
 */
BOOLEAN STB_DSSegmentEDS(U8BIT *data, U16BIT pes_len, U16BIT processed_bytes)
{
   U16BIT segment_length;
   BOOLEAN retval;


   FUNCTION_START(STB_DSSegmentEDS);

   ASSERT(data != NULL);

   STB_SEGMENT_PRINT(("EDS - End of Page"));

   // data[0] - sync_byte
   // data[1] - segment_type
   // data[2] - page_id MS8Bits
   // data[3] - page_id LS8Bits

   segment_length = (data[4] << 8) + data[5];

   if ((processed_bytes + segment_length + 6) > pes_len)
   {
      return(FALSE);
   }

   if (segment_length == 0)
   {
      retval = TRUE;
   }
   else
   {
      retval = FALSE;
   }

   STB_DSDumpPageState("post-EDS");

   FUNCTION_FINISH(STB_DSSegmentEDS);

   return(retval);
}

/**
 *

 *
 * @brief   Extracts the PTS from the PES packet header.
 *
 * @param   data - pointer to the 1st byte of the PTS/DTS bytes
 * @param   pts  - 5 byte array for the PTS info.
 *
 * @return   TRUE  - if all 3 marker_bits' are set correctly.
 *                 FALSE - if any of the 3 marker_bits' are not set.
 *
 */
BOOLEAN STB_DSGetPesPts(U8BIT *data, U8BIT *pts)
{
   BOOLEAN retval;


   FUNCTION_START(STB_DSGetPesPts);

   pts[0] = (data[0] & 0x08) >> 3;
   pts[1] = ((data[0] & 0x06) << 5) + ((data[1] & 0xfc) >> 2);
   pts[2] = ((data[1] & 0x03) << 6) + ((data[2] & 0xfc) >> 2);
   pts[3] = ((data[2] & 0x02) << 6) + ((data[3] & 0xfe) >> 1);
   pts[4] = ((data[3] & 0x01) << 7) + ((data[4] & 0xfe) >> 1);

   retval = (data[0] & 0x01) & (data[2] & 0x01) & (data[4] & 0x01);

   FUNCTION_FINISH(STB_DSGetPesPts);

   return(retval);
}

/**
 *

 *
 * @brief   Copies the passed PTS into the display set structure.
 *
 * @param   pts - 5 byte array containing the PTS.
 *
 * @return   TRUE
 *
 */
BOOLEAN STB_DSSetDisplaySetPts(U8BIT path, U8BIT *pts)
{
   U8BIT stc[5];
   U32BIT time_ms;
   U32BIT stc_timestamp;
   U32BIT pts_timestamp;
   U32BIT stc_delay;
   U32BIT ms_delay;


   FUNCTION_START(STB_DSSetDisplaySetPts);

   ASSERT(pts != NULL);

   time_ms = STB_OSGetClockMilliseconds();
   STB_AVGetSTC(STB_DPGetPathVideoDecoder(path), stc);

   stc_timestamp = ((stc[0] << 24) + (stc[1] << 16) + (stc[2] << 8) + stc[3]);
   pts_timestamp = ((pts[0] << 24) + (pts[1] << 16) + (pts[2] << 8) + pts[3]);

   memcpy((void *)subtitle_display_set.page_composition_buffer->pts, (void *)pts, (size_t)5);

   if (pts_timestamp > stc_timestamp)
   {
      stc_timestamp = ((stc[1] << 24) + (stc[2] << 16) + (stc[3] << 8) + stc[4]);
      pts_timestamp = ((pts[1] << 24) + (pts[2] << 16) + (pts[3] << 8) + pts[4]);

      stc_delay = pts_timestamp - stc_timestamp;
      ms_delay = stc_delay / 90;

      subtitle_display_set.page_composition_buffer->presentation_time_ms = time_ms + ms_delay;
   }
   else
   {
      subtitle_display_set.page_composition_buffer->presentation_time_ms = time_ms;
   }

   FUNCTION_FINISH(STB_DSSetDisplaySetPts);

   return(TRUE);
}

/**
 *

 *
 * @brief   Gets a pointer to the display set structure.
 *

 *
 * @return   Returns pointer to display set structure.
 *
 */
S_DISPLAY_SET* STB_DSGetDetails(void)
{
   FUNCTION_START(STB_DSGetDetails);
   FUNCTION_FINISH(STB_DSGetDetails);
   return(&subtitle_display_set);
}

/**
 *

 *
 * @brief   Clears all information from the display set structure.
 *

 *

 *
 */
void STB_DSClearDisplaySetStruct(void)
{
   FUNCTION_START(STB_DSClearDisplaySetStruct);

   DeletePageCompositionRegions(&subtitle_display_set.page_composition_buffer->region_list);
   DeletePageCompositionRegions(&subtitle_display_set.page_display_buffer->region_list);
   DeleteEpochRegionList(&subtitle_display_set.region_list);
   DeleteObjectList(&subtitle_display_set.object_list);
   DeleteClutList(&subtitle_display_set.clut_list);

   memset((void *)&subtitle_display_set, 0x00, sizeof(S_DISPLAY_SET));
   memset((void *)&page_1, 0x00, sizeof(S_PAGE_COMPOSITION));
   memset((void *)&page_2, 0x00, sizeof(S_PAGE_COMPOSITION));

   /* Initialise the DDS version to an invalid value to ensure the first DDS segment found is used */
   page_1.dds_present = FALSE;
   page_1.dds_version = 0xff;
   page_1.display_width = 719;
   page_1.display_height = 575;
   page_1.display_window = FALSE;
   page_1.page_version_number = 0xff;

   page_2.dds_present = FALSE;
   page_2.dds_version = 0xff;
   page_2.display_width = 719;
   page_2.display_height = 575;
   page_2.display_window = FALSE;
   page_2.page_version_number = 0xff;

   subtitle_display_set.page_display_buffer = &page_1;
   subtitle_display_set.page_composition_buffer = &page_2;

   FUNCTION_FINISH(STB_DSClearDisplaySetStruct);
}

/**
 *

 *
 * @brief   Clears all information from the display set composition page.
 *

 *

 *
 */
void STB_DSClearCompositionPageDetails(void)
{
   FUNCTION_START(STB_DSClearCompositionPageDetails);

   DeletePageCompositionRegions(&subtitle_display_set.page_composition_buffer->region_list);

   subtitle_display_set.page_composition_buffer->pts[0] = 0x00;
   subtitle_display_set.page_composition_buffer->pts[1] = 0x00;
   subtitle_display_set.page_composition_buffer->pts[2] = 0x00;
   subtitle_display_set.page_composition_buffer->pts[3] = 0x00;
   subtitle_display_set.page_composition_buffer->pts[4] = 0x00;

   subtitle_display_set.page_composition_buffer->display_set_shown = FALSE;
   subtitle_display_set.page_composition_buffer->display_set_removed = FALSE;
   subtitle_display_set.page_composition_buffer->dds_present = FALSE;

   FUNCTION_FINISH(STB_DSClearCompositionPageDetails);
}

static void STB_DSDumpPageState(const char *label)
{
   S_EPOCH_REGION *regione;
   S_REGION *region;
#ifdef STB_CLUT_PRINT_REQUIRED
   S_CLUT *clut;
#endif
#ifdef STB_OBJECT_PRINT_REQUIRED
   S_OBJECT *object;
   S_REGION_OBJECT *region_object;
#endif

#ifdef STB_SUMMARY_PRINT_REQUIRED
   STB_SUMMARY_PRINT(("\nPage state dump \"%s\"\n  EPOCH Reg", label));
#else
   USE_UNWANTED_PARAM(label);
#endif

   // Dump EPOCH's master region list
   {
      regione = subtitle_display_set.region_list;

      while (regione != NULL)
      {
         STB_SUMMARY_PRINT(("    r%d (v%d) (w%d, h%d, d%d) c%d",
                            regione->region_id,
                            regione->region_version_number,
                            regione->region_width,
                            regione->region_height,
                            regione->region_colour_depth,
                            regione->region_clut_id));
#ifdef STB_OBJECT_PRINT_REQUIRED
         region_object = regione->region_object_list;
         while (region_object != NULL)
         {
            STB_SUMMARY_PRINT(("      o%3d @(%d,%d)",
                               region_object->object_id,
                               region_object->object_horizontal_position,
                               region_object->object_vertical_position));
            region_object = region_object->next;
         }
#endif
         regione = regione->next;
      }
   }
   STB_SUMMARY_PRINT(("    <END>\n  Composition"));
   // Dump PCS and its region list
   {
      region = subtitle_display_set.page_composition_buffer->region_list;

      while (region != NULL)
      {
         STB_SUMMARY_PRINT(("    r%d @ (%dx%d)",
                            region->region_id,
                            region->region_horizontal_address,
                            region->region_vertical_address));
         region = region->next;
      }
   }
   STB_SUMMARY_PRINT(("    <END>\n  Display"));
   // Dump PCS and its region list
   {
      region = subtitle_display_set.page_display_buffer->region_list;

      while (region != NULL)
      {
         STB_SUMMARY_PRINT(("    r%d @ (%dx%d)",
                            region->region_id,
                            region->region_horizontal_address,
                            region->region_vertical_address));

         region = region->next;
      }
   }
#ifdef STB_OBJECT_PRINT_REQUIRED
   // Can cause a lot of print outs in snake subtitles
   STB_SUMMARY_PRINT(("    <END>\n  Objects"));
   {
      object = subtitle_display_set.object_list;

      while (object != NULL)
      {
         STB_SUMMARY_PRINT(("    o%3d v%2d  (%dx%d)  coding %d %s",
                            object->object_id,
                            object->object_version_number,
                            object->width, object->height,
                            object->object_coding_method,
                            (object->non_modifying_colour_flag) ? "nonmod" : "MODIFY"));
         object = object->next;
      }
   }
#endif
#ifdef STB_CLUT_PRINT_REQUIRED
   STB_SUMMARY_PRINT(("    <END>\n  Clut"));
   {
      clut = subtitle_display_set.clut_list;

      while (clut != NULL)
      {
         STB_SUMMARY_PRINT(("    c%d v%3d  2-%s  4-%s  8-%s",
                            clut->clut_id,
                            clut->clut_version_number,
                            (clut->default_2_bit) ? "default" : "CHANGED",
                            (clut->default_4_bit) ? "default" : "CHANGED",
                            (clut->default_8_bit) ? "default" : "CHANGED"));
         clut = clut->next;
      }
   }
#endif
   STB_SUMMARY_PRINT(("    <END>"));
}

//*****************************************************************************
// End of file
//*****************************************************************************
