/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   EBU Teletext driver
 *
 * @file    stbebutt.c
 * @date    04/02/2004
 * @author  Ocean Blue
 */

//---includes for this file----------------------------------------------------
// compiler library header files
/*#define EBU_DEBUG*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

// third party header files

// Ocean Blue Software header files

#include <techtype.h>
#include <dbgfuncs.h>

#define _STBEBUTT_C

#include <stbhwmem.h>
#include "stbebutt.h"
#include "stbhwos.h"
#include "stbheap.h"
#include "stbpes.h"
#include "stbhwosd.h"
#include "stbdpc.h"
#include "stbhwav.h"

//---constant definitions for this file----------------------------------------

#define COLLATION_QUEUE_SIZE     400

#define SD_WIDTH                 720
#define SD_HEIGHT                576

#ifdef EBU_DEBUG
#define EBU_DBG(x, ...)      STB_SPDebugWrite( "%s:%d(%d)" x, __FUNCTION__,__LINE__,STB_OSGetClockMilliseconds(), ##__VA_ARGS__)
#define EBU_PRINT(x, ...)  STB_SPDebugNoCnWrite(x, ##__VA_ARGS__)
#define EBU_TPRINT(x, ...)
#else
#define EBU_DBG(x, ...)
#define EBU_PRINT(x, ...)
#define EBU_TPRINT(x, ...)
#endif

#if 0
#ifdef FUNCTION_START
#undef FUNCTION_START
#endif
#define FUNCTION_START(x)  STB_SPDebugWrite(">> %s\n", # x)

#ifdef FUNCTION_FINISH
#undef FUNCTION_FINISH
#endif
#define FUNCTION_FINISH(x)  STB_SPDebugWrite("<< %s\n", # x)
#endif

// Used within PESCollectionCallback() to extract individual Teletext packets from a PES data block.
#define PES_DATA_FIELD_WIDTH        46

// Hamming decoding constants
#define HAMMING_8_4_DATA_MASK       0x000f
#define HAMMING_CORRECTION_MASK     0x03f0
#define HAMMING_ERROR_MASK          0xfc00

#define HAMMING_CORRECTION_COUNT(a) ((a & HAMMING_CORRECTION_MASK) >> 4)
#define HAMMING_ERROR_COUNT(a)      ((a & HAMMING_ERROR_MASK) >> 10)

#define COLLATION_TASK_PRIORITY         10
#define COLLATION_TASK_STACK_SIZE       4096

#define DISPLAY_TASK_PRIORITY         10
#define DISPLAY_TASK_STACK_SIZE       4096

// These flags are used to mark pages in the cache according to the relevamt requirement to keep
// the data - this is based apon the caching requests made by the driver.
// These flags are stored in the upper unused bits of S_CACHE_PAGE->page_number
#define CACHE_PAGE_NUMBER_MASK         0x0fff
#define CACHE_PAGE_FLOF_REQUESTED      0x8000
#define CACHE_PAGE_HISTORY_REQUESTED   0x4000
#define CACHE_PAGE_PREVNEXT_REQUESTED  0x2000
#define CACHE_PAGE_LOCKED              0x1000 // set when displayed or requested!

// This is used when (for instance) no sub-code is specified when making a display request
#define PAGE_SUB_CODE_UNDEFINED 0xffff

// Cache constants used to limit data capacity.
#define MAX_HISTORY_CACHE_REQUESTS 32

// This defines the maximum number of page FLOF editorial links that are analysed - this figure
// can be no higher than 6, but stting it to 4 will result in no localised index definitions
// being used.
#define MAX_EDITORIAL_LINKS 6

// This constant defines how deeply the editorial links from the currently displayed page are
// analysed to determine which pages should be cached.
// The minimum sensible value is 2, and values over 5 are not recommended if memory constaints are
// a consideration!
#define MAX_EDITORIAL_LINK_TIERS 5

// These constants determine how many directly adjacent pages before and after the currently
// displayed page are cached.
#define NUM_NEXT_CACHE_REQUESTS     16
#define NUM_PREVIOUS_CACHE_REQUESTS 8

#define INVALID_PREVNEXT_CACHE_PAGE_NUMBER 0x1000

#define MAX_PAGE_HISTORY 32

// The palette used by our system consists of a bank of 256 TRGB values.

// The first value in the bank is set to transparent.
// (This is used as background color for areas of unused teletext plane)
#define PALETTE_BACKGROUND_TRANSPARENT   0

// This background allocation is fixed as back colur - this differs from PALETTE_BACKGROUND_BLACK
// in that it is unaffected by 'video-mix' display mode.
// This is essentailly used to enforce a non-transparent display of page header content under
// certain conditions when the rest of the page has a transparent backdrop.
#define PALETTE_BACKGROUND_BLACK_FIXED   1

// These are used for background colour references, and will be made transparent when
// video 'mix' mode toggling.
#define PALETTE_BACKGROUND_BLACK         2
#define PALETTE_BACKGROUND_RED           3
#define PALETTE_BACKGROUND_GREEN         4
#define PALETTE_BACKGROUND_YELLOW        5
#define PALETTE_BACKGROUND_BLUE          6
#define PALETTE_BACKGROUND_MAGENTA       7
#define PALETTE_BACKGROUND_CYAN          8
#define PALETTE_BACKGROUND_WHITE         9

// These are used for foreground colour references, and are UNAFFECTED by
// video 'mix' mode toggling.
#define PALETTE_FOREGROUND_BLACK        10
#define PALETTE_FOREGROUND_RED          11
#define PALETTE_FOREGROUND_GREEN        12
#define PALETTE_FOREGROUND_YELLOW       13
#define PALETTE_FOREGROUND_BLUE         14
#define PALETTE_FOREGROUND_MAGENTA      15
#define PALETTE_FOREGROUND_CYAN         16
#define PALETTE_FOREGROUND_WHITE        17

// This definition is for clarity - when displaying page content outside of Subtitle and newsflash
// text boxes, the display is simply painted for transparency.
#define PALETTE_FOREGROUND_UNBOXED       0

// This constant defines the LUT offset of the first dynamically defined palette entry.
// These entries, known as 'palette effects' are reserved for foreground background colour pairings
// involved in palette related effects such as flashing and concealed content. This method permits
// dynamic page content to be maintained with no display updates, merely bay altering the relevant
// palette entries.
#define PALETTE_EFFECT_ORIGIN           18

#define NUM_PALETTE_EFFECTS             (256 - PALETTE_EFFECT_ORIGIN)

// The remaning values are dynamically adjusted to give flash and conceal functionality.
// The palette colours used by these LUT entries are defined in an array of page palette
// requirements as defined below
#define PALETTE_EFFECT_FLASH     0x01
#define PALETTE_EFFECT_CONCEAL   0x02
#define PALETTE_EFFECT_UNDEFINED 0    // Used as demarkation of first unused attribute array entry.

// Indexes used to reference specific font tables within the S_EBUTT_FONT->font_table_set_ptr array
// These are also used for the S_TELETEXT_CHARACTER->info mask
#define FONT_INDEX_LATIN_G0_SET                                  0
#define FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET                    1
#define FONT_INDEX_CYRILLIC_G0_SET_OPTION_1_SERBIAN_CROATIAN     2
#define FONT_INDEX_CYRILLIC_G0_SET_OPTION_2_RUSSIAN_BULGARIAN    3
#define FONT_INDEX_CYRILLIC_G0_SET_OPTION_3_UKRANIAN             4
#define FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET                 5
#define FONT_INDEX_GREEK_G0_SET                                  6
#define FONT_INDEX_GREEK_G2_SUPPLIMENTARY_SET                    7
#define FONT_INDEX_ARABIC_G0_SET                                 8
#define FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET                   9
#define FONT_INDEX_HEBREW_G0_SET                                 10
#define FONT_INDEX_G1_BLOCK_MOSAICS_SET                          11
#define FONT_INDEX_G3_SMOOTH_MOSAICS_LINE_DRAWING_SET            12

#define FONT_INDEX_UNDEFINED                                     13

// These are only used for the S_TELETEXT_CHARACTER->imfo mask
#define FONT_INDEX_NATIONAL_OPTION_SUBSET                        14
#define FONT_INDEX_NO_BITMAP                                     15

// Indexes used to reference National Option Subsets for the Latin font.
// These are effectively offsets to the tables within the S_EBUTT_FONT->font_table_set_ptr array
// These are also used for the S_TELETEXT_CHARACTER->info mask to store the font required for each
// character to be rendered.
#define NATIONALITY_INDEX_CZECH_SLOVAK          0
#define NATIONALITY_INDEX_ENGLISH               1
#define NATIONALITY_INDEX_ESTONIAN              2
#define NATIONALITY_INDEX_FRENCH                3
#define NATIONALITY_INDEX_GERMAN                4
#define NATIONALITY_INDEX_ITALIAN               5
#define NATIONALITY_INDEX_LETTISH_LITHUANIAN    6
#define NATIONALITY_INDEX_POLISH                7
#define NATIONALITY_INDEX_PORTUGUESE_SPANISH    8
#define NATIONALITY_INDEX_RUMANIAN              9
#define NATIONALITY_INDEX_SERB_CROAT_SLOVENIAN  10
#define NATIONALITY_INDEX_SWEDISH_FINNISH       11
#define NATIONALITY_INDEX_TURKISH               12

#define NATIONALITY_INDEX_UNDEFINED             13

// This mask maps the bits in S_TELETEXT_CHARACTER-> info onto the FONT_INDEX_???? contants
#define TELETEXT_CHARACTER_FONT_INDEX_MASK    0x0f00

// This is the absolute index of the character within the given font table.
// To acquire a pointer to the bitmap, this will have to be scaled up to a data offset.
#define TELETEXT_CHARACTER_BITMAP_INDEX_MASK  0x00ff

// These two bits are used in the S_TELETEXT_CHARACTER-> info
#define TELETEXT_CHARACTER_DOUBLE_HEIGHT      0x8000
#define TELETEXT_CHARACTER_SEPERATED_MOSAIC   0x4000

// This nominal constant defines the maximum zoom factor permitted by the system.
// At present, there is no functionality that allows the zoom factor to be altered - there is only
// the standard double-height support.
// NB - this is not related to the scalars passed to STB_EBUTT_DefineDisplayProperties()
#define MAX_DISPLAY_ZOOM_SCALAR 2

#define MAX_COLUMN            40
#define MAX_ROWS              25
#define display_x_scalar      1
#define display_y_scalar      2

#define SUBTITLE_TIMEOUT      5000 /*5 secs*/

// This is the essential enumerate used for the state machine used for the user input / page
// selection mechanism encapsulated in S_PAGE_CONTROL_INFO.
typedef enum e_page_control_state
{
   PAGE_CONTROL_STATE_IDLE,   // Initial state (when no page has been selected) and state when
                              // requested page has been dislayed.

   PAGE_CONTROL_STATE_INPUT,  // The user is in process of inputting a three-digit page number.

   PAGE_CONTROL_STATE_HOLD    // The currently displayed page is being held (no sub-page updates)
}
E_PAGE_CONTROL_STATE;

// This enumerate is used to define the user requested page requirement and are used in
// S_PAGE_REQUEST_INFO->type
typedef enum e_request_type
{
   REQUEST_TYPE_UNDEFINED = 0,
   REQUEST_TYPE_EXPLICIT,
   REQUEST_TYPE_INDEX,
   REQUEST_TYPE_NEXT_PREV_AVAILABLE,
   REQUEST_TYPE_NEXT_PREV_VISITED,
   REQUEST_TYPE_NEXT_PREV_SUB,
   REQUEST_TYPE_EDITORIAL_LINK,

   REQUEST_TYPE_CANCEL
}
E_REQUEST_TYPE;

// These constants are used to define the contents of the three-digit page display within the
// S_PAGE_DISPLAY_INFO.
// This enumerate is used as it is independant of any ASCII or font-related representation.
typedef enum E_PAGE_INDEX_DIGIT
{
   PAGE_INDEX_DIGIT_0,
   PAGE_INDEX_DIGIT_1,
   PAGE_INDEX_DIGIT_2,
   PAGE_INDEX_DIGIT_3,
   PAGE_INDEX_DIGIT_4,
   PAGE_INDEX_DIGIT_5,
   PAGE_INDEX_DIGIT_6,
   PAGE_INDEX_DIGIT_7,
   PAGE_INDEX_DIGIT_8,
   PAGE_INDEX_DIGIT_9,
   PAGE_INDEX_DIGIT_A,
   PAGE_INDEX_DIGIT_B,
   PAGE_INDEX_DIGIT_C,
   PAGE_INDEX_DIGIT_D,
   PAGE_INDEX_DIGIT_E,
   PAGE_INDEX_DIGIT_F,
   PAGE_INDEX_DIGIT_UNDEFINED
}
E_PAGE_INDEX_DIGIT;

// These constants are used to define the state of the double-height display mechanism within the
// S_PAGE_DISPLAY_INFO.
typedef enum e_page_scale_status
{
   PAGE_SCALE_STATUS_NORMAL = 0,   // Normal page scale - the whole page should be visible.

   PAGE_SCALE_STATUS_ZOOM_TOP,     // Zoom activated - should show top portion of page.
                                   // This is also used when scrolling the zoomed display.

   PAGE_SCALE_STATUS_ZOOM_BOTTOM   // Only used if zoom is x2 - allows double height toggling.
}
E_PAGE_SCALE_STATUS;

// This is used to define which type of page reference is to be extracted when using
// GetPageReference() to polulate the S_PAGE_REFERENCE_DATA structure.
typedef enum e_page_reference_type
{
   PAGE_REFERENCE_TYPE_HEADER,  // Get all the poage header info, including control bits.

   PAGE_REFERENCE_TYPE_EDITORIAL_LINK, // Get a FLOF shortcut link for a given page.

   PAGE_REFERENCE_TYPE_GLOBAL_INDEX // Get a global index page reference from magazine-related data.
}
E_PAGE_REFERENCE_TYPE;

//---local typedef structs for this file---------------------------------------

// This structure is used to encapsulate the core varibales used in the user-input functionality.
// A single instance of this structure is defined within this module, and the data is then
// analysed by the caching mechanism and the display update task.
typedef struct s_page_control_info
{
   // The variables here are accessed by the control interface, caching, and page display threads,
   // but there is no requirement for a semaphore mechanism at present.

   // The core flag defining the user page number input state-machine.
   E_PAGE_CONTROL_STATE state;

   // These are used to store the page number as it is input.
   U8BIT page_digit[3];
   U16BIT page_digit_offset;

   // These three flags define user-selected functionality that is analysed by the page display
   // task.
   BOOLEAN reveal_required;

   BOOLEAN video_mix_required;

   BOOLEAN clear_required;
}
S_PAGE_CONTROL_INFO;

// This structure is used to encapsulate data that is used to specify each page request.
// A single instance of this structure is defined within this module, and is used by all page
// request functions (principally RequestPage() ) to record such page selections.
typedef struct s_page_request_info
{
   // These are variables configured by page request functionality, and then analysed by the
   // caching mechanism.

   // The nature of the page request.
   E_REQUEST_TYPE type;

   S16BIT param;

   // These are set only when the type is REQUEST_TYPE_EXPLICIT.
   U16BIT page_number;
   U16BIT page_sub_code;

   // Set to TRUE if the parameters above require anaysis.
   BOOLEAN pending;

   // This is the semaphore used to ptotect multi-threaded access to the variables above.
   void *semaphore;

   // The request is converted by the caching mechanism into a specific page number/sub-code.
   U16BIT target_page_number;
   U16BIT target_page_sub_code;
}
S_PAGE_REQUEST_INFO;

// Used to define which Teletext packets are relevant to this driver
// implemetation.
typedef struct s_packet_validation
{
   BOOLEAN is_required;

   // Only relevant to packets X/26, X/28, and X/29 at present.
   BOOLEAN has_designation_code;

   // Designation codes are 4-bit vales (0 - 15).
   U8BIT lowest_designation_code;
   U8BIT highest_designation_code;
}
S_PACKET_VALIDATION;

// This structure is used to define any specific page information - this can be
// magazine-wide or by single page config.
typedef struct s_specific_data
{
   U8BIT s_data[2][MAX_COLUMN];
   BOOLEAN is_defined[2];
}
S_SPECIFIC_DATA;

// This is the core structure used to store collated Teletext page content.
// The same containment is used in the magazines used during the packet collation of each page, and
// the storage of cached pages within the driver's heap.
// It is this structure's content which is the principle source of data for the parsing of pages
// by the display task.
typedef struct s_magazine_page
{
   // This is an X/0 packet, which is a page header
   U8BIT header[MAX_COLUMN];

   // This is the X/27/0 packet, which is editorial page linking data (for FLOF navigation).
   U8BIT editorial_page_linking_data[MAX_COLUMN];

   // This flag stops empty magazine content from being cached!
   // may also apply if magazine is reset (TV radio service changed, etc.)
   BOOLEAN is_header_defined;

   BOOLEAN is_editorial_page_linking_data_defined;

   U16BIT basic_enhancement_data_defined_mask;

   U32BIT display_data_defined_mask;

   // These are X/26/0 to X/26/15 packets, which are enhanced data.
   U8BIT basic_enhancement_data[16][MAX_COLUMN];

   // These are X/1 to X/25 packets, which contain all the information that is subsequently
   // parsed to give a TeleText page display.
   U8BIT display_data[MAX_ROWS][MAX_COLUMN];

   // This is either:
   // the X/28/0 Format 1 packet, which is page specific data.
   // or
   // the X/28/1 packet, which is page specific data.
   S_SPECIFIC_DATA specific_data;
}
S_MAGAZINE_PAGE;

// These are the containers used for caching page content within this driver.
// For any given page, there are one or more sub-pages which actually contain the displayed data.
// All the cached pages from a specific_data magazine are held in this link-list, and can be refreshed
// or deleted as the caching mechanism judges.
typedef struct s_cache_subpage
{
   // This differentiates sub-pages on the same page broadcast
   U16BIT sub_code;

   // This is used to store the image of the collated data within the relevant magazine.
   S_MAGAZINE_PAGE collated_data;

   struct s_cache_subpage *next_ptr;
}
S_CACHE_SUBPAGE;

typedef struct s_cache_page
{
   // This has a theoretical range of 0x100 to 0x8ff
   // The top four bits are reserved as internal flags.
   U16BIT page_number;

   // This is a link-list of all the subpages.
   S_CACHE_SUBPAGE *subpage_ptr;

   // This is just a pointer to the latest subpage to be added/updated.
   S_CACHE_SUBPAGE *latest_subpage_ptr;

   struct s_cache_page *next_ptr;
}
S_CACHE_PAGE;

// This structure is used to define an array of magazine-related info
typedef struct s_magazine_info
{
   S_MAGAZINE_PAGE m_page;

   // This is either:
   // the M/29/0 packet, which is magazine specific data.
   // or
   // the M/29/1 packet, which is magazine specific data.
   S_SPECIFIC_DATA m_specific_data;

   // A 256-bit flag array set as each page is received from collation.
   // This is set regardless of whether the page is cached.
   // Array is cleared when TV/Radio service changes.
   U8BIT m_cache_valid_page_array[32];

   // header for a link-list of cached page content (as described in S_CACHE_PAGE).
   S_CACHE_PAGE *m_cache_page_ptr;

   void *m_mutex;
}
S_MAGAZINE_INFO;

// This is used to define which type of page reference is to be extracted when using
// GetPageReference() to polulate the S_PAGE_REFERENCE_DATA structure.
typedef struct s_page_reference_data
{
   U16BIT page_number;
   U16BIT page_sub_code;
   U16BIT page_offset;

   U16BIT magazine_number;

   BOOLEAN erase_page;
   BOOLEAN newsflash;
   BOOLEAN subtitle;
   BOOLEAN suppress_header;
   BOOLEAN update_indicator;
   BOOLEAN interrupted_sequence;
   BOOLEAN inhibit_display;
   BOOLEAN magazine_serial;
   U8BIT national_option_character_subset;
}
S_PAGE_REFERENCE_DATA;

// This structure is used to encapsulate a link-list of references to recently visited teletext
// pages. As each page is vitsited, it is moved to the head of the link-list.
// Cached page content is marked as being referenced in this way by the setting of
// CACHE_PAGE_HISTORY_REQUESTED within the upper unused bits of S_CACHE_PAGE->page_number.
//
// NB - This link-list is only used if the caching requirements defined to a call to
//      STB_EBUTT_SetCacheMethod() is EBUTT_CACHING_METHOD_HISTORY, EBUTT_CACHING_METHOD_NAVIGATION,
//      EBUTT_CACHING_METHOD_NAVIGATION_TREE, or EBUTT_CACHING_METHOD_ALL.
typedef struct s_history_cache_request
{
   // Thus has a theoretical range of 0x100 to 0x8ff
   U16BIT page_number;

   struct s_history_cache_request *next_ptr;
}
S_HISTORY_CACHE_REQUEST;

// These structures qre used to encapsulate a link-tree of references to FLOF shorcut references to
// other pages for the currently displayed teletext pages.
// This tree is dynamically modified as new collated page data is passed to the cache, and the
// depth of this tree is restricted by the MAX_EDITORIAL_LINKS constant.
// Cached page content is marked as being referenced in this way by the setting of
// CACHE_PAGE_FLOF_REQUESTED within the upper unused bits of S_CACHE_PAGE->page_number.
//
// NB - This link-list is only used if the caching requirements defined to a call to
//      STB_EBUTT_SetCacheMethod() is EBUTT_CACHING_METHOD_NAVIGATION,
//      EBUTT_CACHING_METHOD_NAVIGATION_TREE, or EBUTT_CACHING_METHOD_ALL.
typedef struct s_editorial_link_cache_request_sub_page
{
   // Thus has a theoretical range of 0x0000 to 0x3f7f
   U16BIT sub_code;


   BOOLEAN is_parsed;

   U16BIT link_page_number[MAX_EDITORIAL_LINKS];
   U16BIT link_sub_code[MAX_EDITORIAL_LINKS];
   void *link[MAX_EDITORIAL_LINKS];  // Should be cast to S_EDITORIAL_LINK_CACHE_REQUEST*

   struct s_editorial_link_cache_request_sub_page *next_ptr;
}
S_EDITORIAL_LINK_CACHE_REQUEST_SUB_PAGE;

typedef struct s_editorial_link_cache_request
{
   // Thus has a theoretical range of 0x100 to 0x8ff
   U16BIT page_number;

   U16BIT tier;

   S_EDITORIAL_LINK_CACHE_REQUEST_SUB_PAGE *sub_page_list;
}
S_EDITORIAL_LINK_CACHE_REQUEST;

// This structure is used to encapsulate all data relevant to each display character location
// on the Teletext page display.
// An array of these structures is located within the S_PAGE_DISPLAY_INFO, and it is into this area
// that the collated page content from the cache is parsed into specific font bitmap and
// palette LUT references, ready to be subsequently rendered.
typedef struct s_teletext_character
{
   // This is used to ascertain which font bitmap table is being accessed, and which bitmap is used
   // within the font table.
   U16BIT info;
   // Here is the same information for an optional supplimentary diacritic mark.
   U16BIT diacritic_info;

   // These are indexes into the palette LUT.
   // The content of the LUT can be dynamically altered to achieve effects such as flashing/reveal
   // characters.
   // NB - if this value is set to PALETTE_FOREGROUND_UNBOXED then the character is outside of a
   //      boxed area on a newsflash/subtitle screen, and should be shown painted transparent.
   //      (Unless of course the upwardly adjacent character has info set with
   //       TELETEXT_CHARACTER_DOUBLE_HEIGHT, in which case nothing is to be done).

   U8BIT foreground_index;
   U8BIT background_index;
}
S_TELETEXT_CHARACTER;

// Palette attributes represent the additional font requirements for a page display, which include
// the dynamically altered content used to achieve effects such as flashing/reveal characters.
// The number of these palette references that are required is dependant on the complexity of the
// page content, and also on the size of the unused area of the palette LUT not bing used by the
// standard (fixed) foreground and background colour definitions.
typedef struct s_palette_attribute
{
   U8BIT fore_colour;
   U8BIT back_colour;
   U8BIT effect_mask;
   BOOLEAN fore_colour_used;
}
S_PALETTE_ATTRIBUTE;

// Palette attributes represent the additional font requirements used to achieve anti-aliasing on a
// page display.
// The number of these palette references that are required is dependant on the complexity of the
// page content, and also on the size of the unused area of the palette LUT not bing used by the
// standard (fixed) foreground and background colour definitions.
typedef struct s_palette_alias
{
   U8BIT fore_colour;
   U8BIT back_colour;
}
S_PALETTE_ALIAS;

// This structure is used to encapsulate the core data content of a parsed Teletext page.
// A single instance of this structure is defined within this module, and the data that is
// passed from by the caching mechanism is rendered into this and the display update task.
typedef struct s_page_display_info
{
   U16BIT page_number;
   U16BIT page_sub_code;

   S_MAGAZINE_PAGE page_source;

   // This is either:
   // the M/29/0 packet, which is magazine specific data.
   // or
   // the M/29/1 packet, which is magazine specific data.
   S_SPECIFIC_DATA mgzn_data;

   BOOLEAN page_clock_valid;

   BOOLEAN page_updated;

   // This is the semaphore used to ptotect multi-threaded access to the variables above.
   void *page_semaphore;

   E_PAGE_INDEX_DIGIT page_index_digit[3];
   BOOLEAN page_index_updated;
   BOOLEAN page_index_held;

   // This is the semaphore used to ptotect multi-threaded access to the three variables above.
   void *page_index_semaphore;

   U16BIT carousel_page_number;
   U8BIT carousel_page_header[MAX_COLUMN];
   BOOLEAN carousel_page_updated;

   // This is the semaphore used to ptotect multi-threaded access to the three variables above.
   void *carousel_page_semaphore;

   U8BIT time_filler_header_data[32];
   BOOLEAN clock_updated;

   // This is the semaphore used to ptotect multi-threaded access to the two variables above.
   void *time_filler_header_data_semaphore;

   E_PAGE_SCALE_STATUS page_scale_status;
   U16BIT page_scale_start_row;  // defaults to 0
   U16BIT page_scale_num_rows; // defaults to 25
   U16BIT page_scale_zoom_scalar; // defaults to 1
   BOOLEAN page_scale_params_defined;
   BOOLEAN page_rescaled;

   // This is the semaphore used to ptotect multi-threaded access to the six variables above.
   void *page_scale_semaphore;

   // All varibales above are set by the cache collation task as new page data is acquired,
   // and accessed by the display update task.
   // The consequent parsing results in the population of the data content below. No semaphore
   // mechanism is required for access to this parsed data, as it is the display update thread that
   // solely manipulates this data in various looped processes.

   BOOLEAN subtitle;
   BOOLEAN pts_valid;
   U8BIT page_pts[5];
   S32BIT time_offset;

   // Flag used to denote that page content has been parsed, and that a display update is required.
   BOOLEAN render_required;

   // These variables are used to prompt for a clearing of the page display
   BOOLEAN clear_screen;

   U8BIT screen_colour;

   // This is the two-dimensioanl array representing each page character in the 25 rows of 40
   // characters that represent a full Teletext page.
   S_TELETEXT_CHARACTER character_map[MAX_ROWS][MAX_COLUMN];
   // These two row-related flag arrays are used to denote when a display row has valid content,
   // and when it needs to be re-rendered.
   BOOLEAN valid_row[MAX_ROWS];
   BOOLEAN render_row[MAX_ROWS];

   // Flag used to denote that the palette LUT needs to be recompiled to meet the requirements of
   // page display.
   // NB - this also occurs whenever characters are flashing and/or concelaed text is toggled.
   BOOLEAN palette_update_required;

   // These flags are used to control the 'video mix' mode whereby the background of the Teletext
   // display can be toggled to transparency, thus allowing viewing of the current TV service with
   // the page text superimposed.
   BOOLEAN video_mix_set;
   BOOLEAN video_mix_required;

   // These flags are used to control the status of the flashing content of the Teletext display.
   // This process is automatically undertaken by the display update task (if required).
   BOOLEAN flash_set;
   BOOLEAN flash_required;

   // These flags are used to control the status of the concealed content of the Teletext display.
   // This process is toggled at the operator's request.
   BOOLEAN reveal_required;
   BOOLEAN reveal_set;
   BOOLEAN reveal_available;

   // These flags are used to control the status of the 'page clear' functionaliity of the Teletext
   // display.
   // This process is toggled at the operator's request.
   BOOLEAN clear_required;
   BOOLEAN clear_set;

   // These to arrays are populated with the dynamic palette requirements of the teletext page
   // encapsulated in this structure.
   S_PALETTE_ATTRIBUTE palette_attribute_array[NUM_PALETTE_EFFECTS];
   S_PALETTE_ALIAS palette_alias_array[NUM_PALETTE_EFFECTS / 2];

   // This array is used to house the resultant TRGB palette LUT.
   U32BIT palette_array[256];
}
S_PAGE_DISPLAY_INFO;

typedef struct s_character_set_mapping
{
   U8BIT G0_index;
   U8BIT G2_index;
   U8BIT national_option_subset_index;
}
S_CHARACTER_SET_MAPPING;

typedef struct s_diacritic_equivalent
{
   U8BIT font_index;
   U16BIT character_offset;
   U8BIT diacritic_offset;

   U8BIT alternative_font_index;
   U16BIT alternative_character_offset;
}
S_DIACRITIC_EQUIVALENT;

typedef struct s_diacritic_substitution
{
   U8BIT font_index;
   U16BIT character_offset;

   U16BIT alternative_character_offset;
}
S_DIACRITIC_SUBSTITUTION;

typedef struct s_collation_queue_item
{
   BOOLEAN pts_present;
   U8BIT pts[5];
   U8BIT *data;
} S_COLLATION_QUEUE_ITEM;


//---local (static) variable declarations for this file------------------------
//   (internal variables declared static to make them local)

static const S_PACKET_VALIDATION packet_validation_array[32] =
{
   // X/O page header
   { TRUE, FALSE, 0, 0 },
   // X/1 to X/25 basic page display content
   { TRUE, FALSE, 0, 0 }, { TRUE, FALSE, 0, 0 }, { TRUE, FALSE, 0, 0 },
   { TRUE, FALSE, 0, 0 }, { TRUE, FALSE, 0, 0 }, { TRUE, FALSE, 0, 0 },
   { TRUE, FALSE, 0, 0 }, { TRUE, FALSE, 0, 0 }, { TRUE, FALSE, 0, 0 },
   { TRUE, FALSE, 0, 0 }, { TRUE, FALSE, 0, 0 }, { TRUE, FALSE, 0, 0 },
   { TRUE, FALSE, 0, 0 }, { TRUE, FALSE, 0, 0 }, { TRUE, FALSE, 0, 0 },
   { TRUE, FALSE, 0, 0 }, { TRUE, FALSE, 0, 0 }, { TRUE, FALSE, 0, 0 },
   { TRUE, FALSE, 0, 0 }, { TRUE, FALSE, 0, 0 }, { TRUE, FALSE, 0, 0 },
   { TRUE, FALSE, 0, 0 }, { TRUE, FALSE, 0, 0 }, { TRUE, FALSE, 0, 0 },
   { TRUE, FALSE, 0, 0 },
   // X/26/0-14 character ammendements
   { TRUE, TRUE, 0, 14 },
   // X/27/0 editorial page linking - FLOF navigation
   { TRUE, TRUE, 0, 0 },
   // X/28/0 format 0 page specific data
   // X/28/1 page specific data
   { TRUE, TRUE, 0, 1 },
   // X/29/1 character set designation
   { TRUE, TRUE, 1, 1 },
   // 8/30/0-3 broadcast service data packets
   { TRUE, TRUE, 0, 3 },
   // X/31
   { FALSE, TRUE, 0, 31 }
};

// This table is used to decode Hamming 8/4 encoded bytes.
// The bottom 4 bits are the (corrected) data bits, so none of the lengthy algoritm
// needs to be performed on the encoded data.
// If there is a single bit error, then this is reflected in bit 4 being set.
// Similarily, an uncorrectable error is reflected in bit 10 being set.
// NB - this look-up table was derived by logging the result of passing all
//      8-bit values through a conventional algorithm. Simple really.

// Use HAMMING_8_4_CORRECTION_MASK and HAMMING_8_4_ERROR_MASK in conjunctiona with this table.

static const U16BIT hamming_8_4_table[256] =
{
   0x0011, 0x040f, 0x040f, 0x0018, 0x040f, 0x001c, 0x0014, 0x040f,
   0x040f, 0x0018, 0x0018, 0x0008, 0x0016, 0x040f, 0x040f, 0x0018,
   0x040f, 0x001a, 0x0012, 0x040f, 0x0016, 0x040f, 0x040f, 0x001f,
   0x0016, 0x040f, 0x040f, 0x0018, 0x0006, 0x0016, 0x0016, 0x040f,
   0x040f, 0x001a, 0x0014, 0x040f, 0x0014, 0x040f, 0x0004, 0x0014,
   0x0010, 0x040f, 0x040f, 0x0018, 0x040f, 0x001d, 0x0014, 0x040f,
   0x001a, 0x000a, 0x040f, 0x001a, 0x040f, 0x001a, 0x0014, 0x040f,
   0x040f, 0x001a, 0x0013, 0x040f, 0x0016, 0x040f, 0x040f, 0x001e,
   0x0001, 0x0011, 0x0011, 0x040f, 0x0011, 0x040f, 0x040f, 0x001f,
   0x0011, 0x040f, 0x040f, 0x0018, 0x040f, 0x001d, 0x0015, 0x040f,
   0x0011, 0x040f, 0x040f, 0x001f, 0x040f, 0x001f, 0x001f, 0x000f,
   0x040f, 0x001b, 0x0013, 0x040f, 0x0016, 0x040f, 0x040f, 0x001f,
   0x0011, 0x040f, 0x040f, 0x0019, 0x040f, 0x001d, 0x0014, 0x040f,
   0x040f, 0x001d, 0x0013, 0x040f, 0x001d, 0x000d, 0x040f, 0x001d,
   0x040f, 0x001a, 0x0013, 0x040f, 0x0017, 0x040f, 0x040f, 0x001f,
   0x0013, 0x040f, 0x0003, 0x0013, 0x040f, 0x001d, 0x0013, 0x040f,
   0x040f, 0x001c, 0x0012, 0x040f, 0x001c, 0x000c, 0x040f, 0x001c,
   0x0010, 0x040f, 0x040f, 0x0018, 0x040f, 0x001c, 0x0015, 0x040f,
   0x0012, 0x040f, 0x0002, 0x0012, 0x040f, 0x001c, 0x0012, 0x040f,
   0x040f, 0x001b, 0x0012, 0x040f, 0x0016, 0x040f, 0x040f, 0x001e,
   0x0010, 0x040f, 0x040f, 0x0019, 0x040f, 0x001c, 0x0014, 0x040f,
   0x0000, 0x0010, 0x0010, 0x040f, 0x0010, 0x040f, 0x040f, 0x001e,
   0x040f, 0x001a, 0x0012, 0x040f, 0x0017, 0x040f, 0x040f, 0x001e,
   0x0010, 0x040f, 0x040f, 0x001e, 0x040f, 0x001e, 0x001e, 0x000e,
   0x0011, 0x040f, 0x040f, 0x0019, 0x040f, 0x001c, 0x0015, 0x040f,
   0x040f, 0x001b, 0x0015, 0x040f, 0x0015, 0x040f, 0x0005, 0x0015,
   0x040f, 0x001b, 0x0012, 0x040f, 0x0017, 0x040f, 0x040f, 0x001f,
   0x001b, 0x000b, 0x040f, 0x001b, 0x040f, 0x001b, 0x0015, 0x040f,
   0x040f, 0x0019, 0x0019, 0x0009, 0x0017, 0x040f, 0x040f, 0x0019,
   0x0010, 0x040f, 0x040f, 0x0019, 0x040f, 0x001d, 0x0015, 0x040f,
   0x0017, 0x040f, 0x040f, 0x0019, 0x0007, 0x0017, 0x0017, 0x040f,
   0x040f, 0x001b, 0x0013, 0x040f, 0x0017, 0x040f, 0x040f, 0x001e
};

// This table generates the parity checks used for Hamming 24/18 decoding.
// There are three banks if values - one for each of the tests A,B,C
// The three triplet values are used to each reference a value here,
// which are XOR-ed into a single byte value. This value is an index used to
// access the two following tables.
// NB - the first back is also used for single-byte parity checking.

static const U8BIT hamming_24_18_parity_table[3][256] =
{
   {
      0, 40, 39, 15, 38, 14, 1, 41, 37, 13, 2, 42, 3, 43, 36, 12,
      36, 12, 3, 43, 2, 42, 37, 13, 1, 41, 38, 14, 39, 15, 0, 40,
      35, 11, 4, 44, 5, 45, 34, 10, 6, 46, 33, 9, 32, 8, 7, 47,
      7, 47, 32, 8, 33, 9, 6, 46, 34, 10, 5, 45, 4, 44, 35, 11,
      34, 10, 5, 45, 4, 44, 35, 11, 7, 47, 32, 8, 33, 9, 6, 46,
      6, 46, 33, 9, 32, 8, 7, 47, 35, 11, 4, 44, 5, 45, 34, 10,
      1, 41, 38, 14, 39, 15, 0, 40, 36, 12, 3, 43, 2, 42, 37, 13,
      37, 13, 2, 42, 3, 43, 36, 12, 0, 40, 39, 15, 38, 14, 1, 41,
      33, 9, 6, 46, 7, 47, 32, 8, 4, 44, 35, 11, 34, 10, 5, 45,
      5, 45, 34, 10, 35, 11, 4, 44, 32, 8, 7, 47, 6, 46, 33, 9,
      2, 42, 37, 13, 36, 12, 3, 43, 39, 15, 0, 40, 1, 41, 38, 14,
      38, 14, 1, 41, 0, 40, 39, 15, 3, 43, 36, 12, 37, 13, 2, 42,
      3, 43, 36, 12, 37, 13, 2, 42, 38, 14, 1, 41, 0, 40, 39, 15,
      39, 15, 0, 40, 1, 41, 38, 14, 2, 42, 37, 13, 36, 12, 3, 43,
      32, 8, 7, 47, 6, 46, 33, 9, 5, 45, 34, 10, 35, 11, 4, 44,
      4, 44, 35, 11, 34, 10, 5, 45, 33, 9, 6, 46, 7, 47, 32, 8
   },
   {
      0, 48, 47, 31, 46, 30, 1, 49, 45, 29, 2, 50, 3, 51, 44, 28,
      44, 28, 3, 51, 2, 50, 45, 29, 1, 49, 46, 30, 47, 31, 0, 48,
      43, 27, 4, 52, 5, 53, 42, 26, 6, 54, 41, 25, 40, 24, 7, 55,
      7, 55, 40, 24, 41, 25, 6, 54, 42, 26, 5, 53, 4, 52, 43, 27,
      42, 26, 5, 53, 4, 52, 43, 27, 7, 55, 40, 24, 41, 25, 6, 54,
      6, 54, 41, 25, 40, 24, 7, 55, 43, 27, 4, 52, 5, 53, 42, 26,
      1, 49, 46, 30, 47, 31, 0, 48, 44, 28, 3, 51, 2, 50, 45, 29,
      45, 29, 2, 50, 3, 51, 44, 28, 0, 48, 47, 31, 46, 30, 1, 49,
      41, 25, 6, 54, 7, 55, 40, 24, 4, 52, 43, 27, 42, 26, 5, 53,
      5, 53, 42, 26, 43, 27, 4, 52, 40, 24, 7, 55, 6, 54, 41, 25,
      2, 50, 45, 29, 44, 28, 3, 51, 47, 31, 0, 48, 1, 49, 46, 30,
      46, 30, 1, 49, 0, 48, 47, 31, 3, 51, 44, 28, 45, 29, 2, 50,
      3, 51, 44, 28, 45, 29, 2, 50, 46, 30, 1, 49, 0, 48, 47, 31,
      47, 31, 0, 48, 1, 49, 46, 30, 2, 50, 45, 29, 44, 28, 3, 51,
      40, 24, 7, 55, 6, 54, 41, 25, 5, 53, 42, 26, 43, 27, 4, 52,
      4, 52, 43, 27, 42, 26, 5, 53, 41, 25, 6, 54, 7, 55, 40, 24
   },
   {
      63, 31, 8, 40, 9, 41, 62, 30, 10, 42, 61, 29, 60, 28, 11, 43,
      11, 43, 60, 28, 61, 29, 10, 42, 62, 30, 9, 41, 8, 40, 63, 31,
      12, 44, 59, 27, 58, 26, 13, 45, 57, 25, 14, 46, 15, 47, 56, 24,
      56, 24, 15, 47, 14, 46, 57, 25, 13, 45, 58, 26, 59, 27, 12, 44,
      13, 45, 58, 26, 59, 27, 12, 44, 56, 24, 15, 47, 14, 46, 57, 25,
      57, 25, 14, 46, 15, 47, 56, 24, 12, 44, 59, 27, 58, 26, 13, 45,
      62, 30, 9, 41, 8, 40, 63, 31, 11, 43, 60, 28, 61, 29, 10, 42,
      10, 42, 61, 29, 60, 28, 11, 43, 63, 31, 8, 40, 9, 41, 62, 30,
      14, 46, 57, 25, 56, 24, 15, 47, 59, 27, 12, 44, 13, 45, 58, 26,
      58, 26, 13, 45, 12, 44, 59, 27, 15, 47, 56, 24, 57, 25, 14, 46,
      61, 29, 10, 42, 11, 43, 60, 28, 8, 40, 63, 31, 62, 30, 9, 41,
      9, 41, 62, 30, 63, 31, 8, 40, 60, 28, 11, 43, 10, 42, 61, 29,
      60, 28, 11, 43, 10, 42, 61, 29, 9, 41, 62, 30, 63, 31, 8, 40,
      8, 40, 63, 31, 62, 30, 9, 41, 61, 29, 10, 42, 11, 43, 60, 28,
      15, 47, 56, 24, 57, 25, 14, 46, 58, 26, 13, 45, 12, 44, 59, 27,
      59, 27, 12, 44, 13, 45, 58, 26, 14, 46, 57, 25, 56, 24, 15, 47
   }
};

// This table simplifies the process of extracting the lower 4 bits
// of the LSB from a Hamming 24/18 encoded triplet.

static const U8BIT hamming_24_18_first_byte_table[256] =
{
   0, 0, 8, 8, 4, 4, 12, 12, 2, 2, 10, 10, 6, 6, 14, 14,
   0, 0, 8, 8, 4, 4, 12, 12, 2, 2, 10, 10, 6, 6, 14, 14,
   1, 1, 9, 9, 5, 5, 13, 13, 3, 3, 11, 11, 7, 7, 15, 15,
   1, 1, 9, 9, 5, 5, 13, 13, 3, 3, 11, 11, 7, 7, 15, 15,
   0, 0, 8, 8, 4, 4, 12, 12, 2, 2, 10, 10, 6, 6, 14, 14,
   0, 0, 8, 8, 4, 4, 12, 12, 2, 2, 10, 10, 6, 6, 14, 14,
   1, 1, 9, 9, 5, 5, 13, 13, 3, 3, 11, 11, 7, 7, 15, 15,
   1, 1, 9, 9, 5, 5, 13, 13, 3, 3, 11, 11, 7, 7, 15, 15,
   0, 0, 8, 8, 4, 4, 12, 12, 2, 2, 10, 10, 6, 6, 14, 14,
   0, 0, 8, 8, 4, 4, 12, 12, 2, 2, 10, 10, 6, 6, 14, 14,
   1, 1, 9, 9, 5, 5, 13, 13, 3, 3, 11, 11, 7, 7, 15, 15,
   1, 1, 9, 9, 5, 5, 13, 13, 3, 3, 11, 11, 7, 7, 15, 15,
   0, 0, 8, 8, 4, 4, 12, 12, 2, 2, 10, 10, 6, 6, 14, 14,
   0, 0, 8, 8, 4, 4, 12, 12, 2, 2, 10, 10, 6, 6, 14, 14,
   1, 1, 9, 9, 5, 5, 13, 13, 3, 3, 11, 11, 7, 7, 15, 15,
   1, 1, 9, 9, 5, 5, 13, 13, 3, 3, 11, 11, 7, 7, 15, 15
};

// This table simplifies the process of extracting the middle and upper 7 bit pairs
// of the middle and MSB from a Hamming 24/18 encoded triplet.
static const U8BIT hamming_24_18_other_bytes_table[256] =
{
   0x00, 0x00, 0x40, 0x40, 0x20, 0x20, 0x60, 0x60, 0x10, 0x10, 0x50, 0x50, 0x30, 0x30, 0x70, 0x70,
   0x08, 0x08, 0x48, 0x48, 0x28, 0x28, 0x68, 0x68, 0x18, 0x18, 0x58, 0x58, 0x38, 0x38, 0x78, 0x78,
   0x04, 0x04, 0x44, 0x44, 0x24, 0x24, 0x64, 0x64, 0x14, 0x14, 0x54, 0x54, 0x34, 0x34, 0x74, 0x74,
   0x0c, 0x0c, 0x4c, 0x4c, 0x2c, 0x2c, 0x6c, 0x6c, 0x1c, 0x1c, 0x5c, 0x5c, 0x3c, 0x3c, 0x7c, 0x7c,
   0x02, 0x02, 0x42, 0x42, 0x22, 0x22, 0x62, 0x62, 0x12, 0x12, 0x52, 0x52, 0x32, 0x32, 0x72, 0x72,
   0x0a, 0x0a, 0x4a, 0x4a, 0x2a, 0x2a, 0x6a, 0x6a, 0x1a, 0x1a, 0x5a, 0x5a, 0x3a, 0x3a, 0x7a, 0x7a,
   0x06, 0x06, 0x46, 0x46, 0x26, 0x26, 0x66, 0x66, 0x16, 0x16, 0x56, 0x56, 0x36, 0x36, 0x76, 0x76,
   0x0e, 0x0e, 0x4e, 0x4e, 0x2e, 0x2e, 0x6e, 0x6e, 0x1e, 0x1e, 0x5e, 0x5e, 0x3e, 0x3e, 0x7e, 0x7e,
   0x01, 0x01, 0x41, 0x41, 0x21, 0x21, 0x61, 0x61, 0x11, 0x11, 0x51, 0x51, 0x31, 0x31, 0x71, 0x71,
   0x09, 0x09, 0x49, 0x49, 0x29, 0x29, 0x69, 0x69, 0x19, 0x19, 0x59, 0x59, 0x39, 0x39, 0x79, 0x79,
   0x05, 0x05, 0x45, 0x45, 0x25, 0x25, 0x65, 0x65, 0x15, 0x15, 0x55, 0x55, 0x35, 0x35, 0x75, 0x75,
   0x0d, 0x0d, 0x4d, 0x4d, 0x2d, 0x2d, 0x6d, 0x6d, 0x1d, 0x1d, 0x5d, 0x5d, 0x3d, 0x3d, 0x7d, 0x7d,
   0x03, 0x03, 0x43, 0x43, 0x23, 0x23, 0x63, 0x63, 0x13, 0x13, 0x53, 0x53, 0x33, 0x33, 0x73, 0x73,
   0x0b, 0x0b, 0x4b, 0x4b, 0x2b, 0x2b, 0x6b, 0x6b, 0x1b, 0x1b, 0x5b, 0x5b, 0x3b, 0x3b, 0x7b, 0x7b,
   0x07, 0x07, 0x47, 0x47, 0x27, 0x27, 0x67, 0x67, 0x17, 0x17, 0x57, 0x57, 0x37, 0x37, 0x77, 0x77,
   0x0f, 0x0f, 0x4f, 0x4f, 0x2f, 0x2f, 0x6f, 0x6f, 0x1f, 0x1f, 0x5f, 0x5f, 0x3f, 0x3f, 0x7f, 0x7f
};

// This table is the mapping from parity checks made by from the result
// of analysis of hamming_24_18_parity_table, to the error incurred.
// If there is a single bit error, then this is reflected in bit 4 being set.
// Similarily, an uncorrectable error is reflected in bit 10 being set.

// Use HAMMING_8_4_CORRECTION_MASK and HAMMING_8_4_ERROR_MASK in conjunctiona with this table.

static const U16BIT hamming_24_18_error_table[64] =
{
   0x0000, 0x0400, 0x0400, 0x0400, 0x0400, 0x0400, 0x0400, 0x0400,
   0x0400, 0x0400, 0x0400, 0x0400, 0x0400, 0x0400, 0x0400, 0x0400,
   0x0400, 0x0400, 0x0400, 0x0400, 0x0400, 0x0400, 0x0400, 0x0400,
   0x0400, 0x0400, 0x0400, 0x0400, 0x0400, 0x0400, 0x0400, 0x0400,
   0x0010, 0x0010, 0x0010, 0x0010, 0x0010, 0x0010, 0x0010, 0x0010,
   0x0010, 0x0010, 0x0010, 0x0010, 0x0010, 0x0010, 0x0010, 0x0010,
   0x0010, 0x0010, 0x0010, 0x0010, 0x0010, 0x0010, 0x0010, 0x0010,
   0x0400, 0x0400, 0x0400, 0x0400, 0x0400, 0x0400, 0x0400, 0x0400
};

// This table is the mapping from parity checks made by from the result
// of analysis of hamming_24_18_parity_table, to the relevant faulty bit
// in the decoded 18 bit word.

static const U32BIT hamming_24_18_correction_table[64] =
{
   0x00000L, 0x00000L, 0x00000L, 0x00000L, 0x00000L, 0x00000L, 0x00000L, 0x00000L,
   0x00000L, 0x00000L, 0x00000L, 0x00000L, 0x00000L, 0x00000L, 0x00000L, 0x00000L,
   0x00000L, 0x00000L, 0x00000L, 0x00000L, 0x00000L, 0x00000L, 0x00000L, 0x00000L,
   0x00000L, 0x00000L, 0x00000L, 0x00000L, 0x00000L, 0x00000L, 0x00000L, 0x00000L,
   0x00000L, 0x00000L, 0x00000L, 0x00001L, 0x00000L, 0x00002L, 0x00004L, 0x00008L,
   0x00000L, 0x00010L, 0x00020L, 0x00040L, 0x00080L, 0x00100L, 0x00200L, 0x00400L,
   0x00000L, 0x00800L, 0x01000L, 0x02000L, 0x04000L, 0x08000L, 0x10000L, 0x20000L,
   0x00000L, 0x00000L, 0x00000L, 0x00000L, 0x00000L, 0x00000L, 0x00000L, 0x00000L
};

// This table is used to convert the character set deignations supplied in teletext packets (as
// defined in ETSI EN 300 706 V1.2.1, Table 32) into the font table references used within
// the S_EBUTT_FONT data structure.

static const S_CHARACTER_SET_MAPPING character_set_mapping_array[256] =
{
   // Triplet_1 value is 0000
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_ENGLISH },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_GERMAN  },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_SWEDISH_FINNISH },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_ITALIAN },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_FRENCH },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_PORTUGUESE_SPANISH },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_CZECH_SLOVAK },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_UNDEFINED },
   // Triplet_1 value is 0001
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_POLISH },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_GERMAN  },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_SWEDISH_FINNISH },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_ITALIAN },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_FRENCH },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_CZECH_SLOVAK },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_UNDEFINED },
   // Triplet_1 value is 0010
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_ENGLISH },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_GERMAN  },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_SWEDISH_FINNISH },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_ITALIAN },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_FRENCH },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_PORTUGUESE_SPANISH },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_TURKISH },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_UNDEFINED },
   // Triplet_1 value is 0011
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_UNDEFINED  },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_SERB_CROAT_SLOVENIAN },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_RUMANIAN },
   // Triplet_1 value is 0100
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_1_SERBIAN_CROATIAN, FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_GERMAN  },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_ESTONIAN },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_LETTISH_LITHUANIAN },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_2_RUSSIAN_BULGARIAN, FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_3_UKRANIAN, FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_CZECH_SLOVAK },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   // Triplet_1 value is 0101 - reserved
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   // Triplet_1 value is 0110
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED  },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_TURKISH },
   { FONT_INDEX_GREEK_G0_SET, FONT_INDEX_GREEK_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_UNDEFINED },
   // Triplet_1 value is 0111 - reserved
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   // Triplet_1 value is 1000
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_ENGLISH },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED  },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_LATIN_G0_SET, FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_FRENCH },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_ARABIC_G0_SET, FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_UNDEFINED },
   // Triplet_1 value is 1001 - reserved
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   // Triplet_1 value is 1010
   { FONT_INDEX_UNDEFINED, FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED  },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_HEBREW_G0_SET, FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_ARABIC_G0_SET, FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, NATIONALITY_INDEX_UNDEFINED },
   // Triplet_1 value is 1011 - reserved
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   // Triplet_1 value is 1100 - reserved
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   // Triplet_1 value is 1101 - reserved
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   // Triplet_1 value is 1110 - reserved
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   // Triplet_1 value is 1111 - reserved
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED },
   { FONT_INDEX_UNDEFINED, FONT_INDEX_UNDEFINED, NATIONALITY_INDEX_UNDEFINED }
};

// This table defines specific character and diacritic pairings that have a more suitable equivalent
// elsewhere within the font tables.
// This is an enhancement specifically designed for this driver, which ensures that the optimal
// height character is used to ensure clarity for the diacritic mark.
// The substitutions are only required for upper diacritics.

static const S_DIACRITIC_EQUIVALENT diacritic_equivalent_array[] =
{
   // C with accent acute (')
   { FONT_INDEX_LATIN_G0_SET, 0x23, 2, FONT_INDEX_NATIONAL_OPTION_SUBSET, 49 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_1_SERBIAN_CROATIAN, 0x33, 2, FONT_INDEX_NATIONAL_OPTION_SUBSET, 49 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_2_RUSSIAN_BULGARIAN, 0x33, 2, FONT_INDEX_NATIONAL_OPTION_SUBSET, 49 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_3_UKRANIAN, 0x33, 2, FONT_INDEX_NATIONAL_OPTION_SUBSET, 49 },
   { FONT_INDEX_GREEK_G2_SUPPLIMENTARY_SET, 0x40, 2, FONT_INDEX_NATIONAL_OPTION_SUBSET, 49 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x23, 2, FONT_INDEX_NATIONAL_OPTION_SUBSET, 49 },
   { FONT_INDEX_HEBREW_G0_SET, 0x23, 2, FONT_INDEX_NATIONAL_OPTION_SUBSET, 49 },

   // E with accent acute (')
   { FONT_INDEX_LATIN_G0_SET, 0x25, 2, FONT_INDEX_NATIONAL_OPTION_SUBSET, 37 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_1_SERBIAN_CROATIAN, 0x25, 2, FONT_INDEX_NATIONAL_OPTION_SUBSET, 37 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_2_RUSSIAN_BULGARIAN, 0x25, 2, FONT_INDEX_NATIONAL_OPTION_SUBSET, 37 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_3_UKRANIAN, 0x25, 2, FONT_INDEX_NATIONAL_OPTION_SUBSET, 37 },
   { FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, 0x41, 2, FONT_INDEX_NATIONAL_OPTION_SUBSET, 37 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x25, 2, FONT_INDEX_NATIONAL_OPTION_SUBSET, 37 },
   { FONT_INDEX_HEBREW_G0_SET, 0x25, 2, FONT_INDEX_NATIONAL_OPTION_SUBSET, 37 },

   // K with accent acute (')
   { FONT_INDEX_LATIN_G0_SET, 0x2b, 2, FONT_INDEX_CYRILLIC_G0_SET_OPTION_1_SERBIAN_CROATIAN, 0x31 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_2_RUSSIAN_BULGARIAN, 0x2b, 2, FONT_INDEX_CYRILLIC_G0_SET_OPTION_1_SERBIAN_CROATIAN, 0x31 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_3_UKRANIAN, 0x2b, 2, FONT_INDEX_CYRILLIC_G0_SET_OPTION_1_SERBIAN_CROATIAN, 0x31 },
   { FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, 0x46, 2, FONT_INDEX_CYRILLIC_G0_SET_OPTION_1_SERBIAN_CROATIAN, 0x31 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x2b, 2, FONT_INDEX_CYRILLIC_G0_SET_OPTION_1_SERBIAN_CROATIAN, 0x31 },
   { FONT_INDEX_HEBREW_G0_SET, 0x2b, 2, FONT_INDEX_CYRILLIC_G0_SET_OPTION_1_SERBIAN_CROATIAN, 0x31 },

   // S with accent acute (')
   { FONT_INDEX_LATIN_G0_SET, 0x33, 2, FONT_INDEX_NATIONAL_OPTION_SUBSET, 59 },
   { FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, 0x4b, 2, FONT_INDEX_NATIONAL_OPTION_SUBSET, 59 },
   { FONT_INDEX_GREEK_G2_SUPPLIMENTARY_SET, 0x48, 2, FONT_INDEX_NATIONAL_OPTION_SUBSET, 59 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x33, 2, FONT_INDEX_NATIONAL_OPTION_SUBSET, 59 },
   { FONT_INDEX_HEBREW_G0_SET, 0x33, 2, FONT_INDEX_NATIONAL_OPTION_SUBSET, 59 },

   // A with circumflex (^)
   { FONT_INDEX_LATIN_G0_SET, 0x21, 3, FONT_INDEX_NATIONAL_OPTION_SUBSET, 48 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_1_SERBIAN_CROATIAN, 0x21, 3, FONT_INDEX_NATIONAL_OPTION_SUBSET, 48 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_2_RUSSIAN_BULGARIAN, 0x21, 3, FONT_INDEX_NATIONAL_OPTION_SUBSET, 48 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_3_UKRANIAN, 0x21, 3, FONT_INDEX_NATIONAL_OPTION_SUBSET, 48 },
   { FONT_INDEX_GREEK_G0_SET, 0x21, 3, FONT_INDEX_NATIONAL_OPTION_SUBSET, 48 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x21, 3, FONT_INDEX_NATIONAL_OPTION_SUBSET, 48 },
   { FONT_INDEX_HEBREW_G0_SET, 0x21, 3, FONT_INDEX_NATIONAL_OPTION_SUBSET, 48 },

   // I with circumflex (^)
   { FONT_INDEX_LATIN_G0_SET, 0x29, 3, FONT_INDEX_NATIONAL_OPTION_SUBSET, 87 },
   { FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, 0x44, 3, FONT_INDEX_NATIONAL_OPTION_SUBSET, 87 },
   { FONT_INDEX_GREEK_G0_SET, 0x29, 3, FONT_INDEX_NATIONAL_OPTION_SUBSET, 87 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x29, 3, FONT_INDEX_NATIONAL_OPTION_SUBSET, 87 },
   { FONT_INDEX_HEBREW_G0_SET, 0x29, 3, FONT_INDEX_NATIONAL_OPTION_SUBSET, 87 },

   // O with tilde (~)
   { FONT_INDEX_LATIN_G0_SET, 0x2f, 4, FONT_INDEX_NATIONAL_OPTION_SUBSET, 93 },
   { FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, 0x48, 4, FONT_INDEX_NATIONAL_OPTION_SUBSET, 93 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x2e, 4, FONT_INDEX_NATIONAL_OPTION_SUBSET, 93 },
   { FONT_INDEX_HEBREW_G0_SET, 0x2e, 4, FONT_INDEX_NATIONAL_OPTION_SUBSET, 93 },

   // I with upper dot
   { FONT_INDEX_LATIN_G0_SET, 0x29, 7, FONT_INDEX_NATIONAL_OPTION_SUBSET, 38 },
   { FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, 0x44, 7, FONT_INDEX_NATIONAL_OPTION_SUBSET, 38 },
   { FONT_INDEX_GREEK_G0_SET, 0x29, 7, FONT_INDEX_NATIONAL_OPTION_SUBSET, 38 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x29, 7, FONT_INDEX_NATIONAL_OPTION_SUBSET, 38 },
   { FONT_INDEX_HEBREW_G0_SET, 0x29, 7, FONT_INDEX_NATIONAL_OPTION_SUBSET, 38 },

   // A with umlaut (two dots above)
   { FONT_INDEX_LATIN_G0_SET, 0x21, 8, FONT_INDEX_NATIONAL_OPTION_SUBSET, 41 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_1_SERBIAN_CROATIAN, 0x21, 8, FONT_INDEX_NATIONAL_OPTION_SUBSET, 41 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_2_RUSSIAN_BULGARIAN, 0x21, 8, FONT_INDEX_NATIONAL_OPTION_SUBSET, 41 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_3_UKRANIAN, 0x21, 8, FONT_INDEX_NATIONAL_OPTION_SUBSET, 41 },
   { FONT_INDEX_GREEK_G0_SET, 0x21, 8, FONT_INDEX_NATIONAL_OPTION_SUBSET, 41 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x21, 8, FONT_INDEX_NATIONAL_OPTION_SUBSET, 41 },
   { FONT_INDEX_HEBREW_G0_SET, 0x21, 8, FONT_INDEX_NATIONAL_OPTION_SUBSET, 41 },

   // E with umlaut (two dots above)
   { FONT_INDEX_LATIN_G0_SET, 0x25, 8, FONT_INDEX_NATIONAL_OPTION_SUBSET, 23 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_1_SERBIAN_CROATIAN, 0x25, 8, FONT_INDEX_NATIONAL_OPTION_SUBSET, 23 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_2_RUSSIAN_BULGARIAN, 0x25, 8, FONT_INDEX_NATIONAL_OPTION_SUBSET, 23 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_3_UKRANIAN, 0x25, 8, FONT_INDEX_NATIONAL_OPTION_SUBSET, 23 },
   { FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, 0x41, 8, FONT_INDEX_NATIONAL_OPTION_SUBSET, 23 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x25, 8, FONT_INDEX_NATIONAL_OPTION_SUBSET, 23 },
   { FONT_INDEX_HEBREW_G0_SET, 0x25, 8, FONT_INDEX_NATIONAL_OPTION_SUBSET, 23 },

   // I with umlaut (two dots above)
   { FONT_INDEX_LATIN_G0_SET, 0x29, 8, FONT_INDEX_CYRILLIC_G0_SET_OPTION_3_UKRANIAN, 0x3f },
   { FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, 0x44, 8, FONT_INDEX_CYRILLIC_G0_SET_OPTION_3_UKRANIAN, 0x3f },
   { FONT_INDEX_GREEK_G0_SET, 0x29, 8, FONT_INDEX_CYRILLIC_G0_SET_OPTION_3_UKRANIAN, 0x3f },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x29, 8, FONT_INDEX_CYRILLIC_G0_SET_OPTION_3_UKRANIAN, 0x3f },
   { FONT_INDEX_HEBREW_G0_SET, 0x29, 8, FONT_INDEX_CYRILLIC_G0_SET_OPTION_3_UKRANIAN, 0x3f },

   // O with umlaut (two dots above)
   {  FONT_INDEX_LATIN_G0_SET, 0x2e, 8, FONT_INDEX_NATIONAL_OPTION_SUBSET, 54 },
   {  FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, 0x48, 8, FONT_INDEX_NATIONAL_OPTION_SUBSET, 54 },
   {  FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x2e, 8, FONT_INDEX_NATIONAL_OPTION_SUBSET, 54 },
   {  FONT_INDEX_HEBREW_G0_SET, 0x2e, 8, FONT_INDEX_NATIONAL_OPTION_SUBSET, 54 },

   // U with umlaut (two dots above)
   { FONT_INDEX_LATIN_G0_SET, 0x35, 8, FONT_INDEX_NATIONAL_OPTION_SUBSET, 80 },
   { FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, 0x4c, 8, FONT_INDEX_NATIONAL_OPTION_SUBSET, 80 },
   { FONT_INDEX_GREEK_G2_SUPPLIMENTARY_SET, 0x49, 8, FONT_INDEX_NATIONAL_OPTION_SUBSET, 80 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x35, 8, FONT_INDEX_NATIONAL_OPTION_SUBSET, 80 },
   { FONT_INDEX_HEBREW_G0_SET, 0x35, 8, FONT_INDEX_NATIONAL_OPTION_SUBSET, 80 },

   // A with bolle (small circle above)
   { FONT_INDEX_LATIN_G0_SET, 0x21, 10, FONT_INDEX_NATIONAL_OPTION_SUBSET, 76 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_1_SERBIAN_CROATIAN, 0x21, 10, FONT_INDEX_NATIONAL_OPTION_SUBSET, 76 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_2_RUSSIAN_BULGARIAN, 0x21, 10, FONT_INDEX_NATIONAL_OPTION_SUBSET, 76 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_3_UKRANIAN, 0x21, 10, FONT_INDEX_NATIONAL_OPTION_SUBSET, 76 },
   { FONT_INDEX_GREEK_G0_SET, 0x21, 10, FONT_INDEX_NATIONAL_OPTION_SUBSET, 76 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x21, 10, FONT_INDEX_NATIONAL_OPTION_SUBSET, 76 },
   { FONT_INDEX_HEBREW_G0_SET, 0x21, 10, FONT_INDEX_NATIONAL_OPTION_SUBSET, 76 },

   // A with hacek (small v above)
   { FONT_INDEX_LATIN_G0_SET, 0x21, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 74 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_1_SERBIAN_CROATIAN, 0x21, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 74 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_2_RUSSIAN_BULGARIAN, 0x21, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 74 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_3_UKRANIAN, 0x21, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 74 },
   { FONT_INDEX_GREEK_G0_SET, 0x21, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 74 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x21, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 74 },
   { FONT_INDEX_HEBREW_G0_SET, 0x21, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 74 },

   // C with hacek (small v above)
   { FONT_INDEX_LATIN_G0_SET, 0x23, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 36 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_1_SERBIAN_CROATIAN, 0x33, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 36 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_2_RUSSIAN_BULGARIAN, 0x33, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 36 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_3_UKRANIAN, 0x33, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 36 },
   { FONT_INDEX_GREEK_G2_SUPPLIMENTARY_SET, 0x40, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 36 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x23, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 36 },
   { FONT_INDEX_HEBREW_G0_SET, 0x23, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 36 },

   // S with hacek (small v above)
   { FONT_INDEX_LATIN_G0_SET, 0x33, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 32 },
   { FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, 0x4b, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 32 },
   { FONT_INDEX_GREEK_G2_SUPPLIMENTARY_SET, 0x48, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 32 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x33, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 32 },
   { FONT_INDEX_HEBREW_G0_SET, 0x33, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 32 },

   // G with hacek (small v above)
   { FONT_INDEX_LATIN_G0_SET, 0x27, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 103 },
   { FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, 0x43, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 103 },
   { FONT_INDEX_GREEK_G2_SUPPLIMENTARY_SET, 0x43, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 103 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x27, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 103 },
   { FONT_INDEX_HEBREW_G0_SET, 0x27, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 103 },

   // Z with hacek (small v above)
   { FONT_INDEX_LATIN_G0_SET, 0x3a, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 67 },
   { FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, 0x4f, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 67 },
   { FONT_INDEX_GREEK_G2_SUPPLIMENTARY_SET, 0x4d, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 67 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x3a, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 67 },
   { FONT_INDEX_HEBREW_G0_SET, 0x3a, 15, FONT_INDEX_NATIONAL_OPTION_SUBSET, 67 },

   { FONT_INDEX_UNDEFINED, 0x00, 0, FONT_INDEX_UNDEFINED, 0  }
};

// This table defines sppecific character and diacritic pairings that have a more custom defined
// equivalent within the unused characters 0x40 to 0x5f within the G1 block mosaics font table.
// This is an enhancement specifically designed for this driver, which ensures that the optimal
// height character is used to ensure clarity for the diacritic mark.
// The substitutions are only required for upper diacritics.

static const S_DIACRITIC_SUBSTITUTION diacritic_substitution_array[] =
{
   // A
   { FONT_INDEX_LATIN_G0_SET, 0x21, 0 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_1_SERBIAN_CROATIAN, 0x21, 0 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_2_RUSSIAN_BULGARIAN, 0x21, 0 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_3_UKRANIAN, 0x21, 0 },
   { FONT_INDEX_GREEK_G0_SET, 0x21, 0 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x21, 0 },
   { FONT_INDEX_HEBREW_G0_SET, 0x21, 0 },

   // E
   { FONT_INDEX_LATIN_G0_SET, 0x25, 1 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_1_SERBIAN_CROATIAN, 0x25, 1 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_2_RUSSIAN_BULGARIAN, 0x25, 1 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_3_UKRANIAN, 0x25, 1 },
   { FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, 0x41, 1 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x25, 1 },
   { FONT_INDEX_HEBREW_G0_SET, 0x25, 1 },

   // I
   { FONT_INDEX_LATIN_G0_SET, 0x29, 2 },
   { FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, 0x44, 2 },
   { FONT_INDEX_GREEK_G0_SET, 0x29, 2 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x29, 2 },
   { FONT_INDEX_HEBREW_G0_SET, 0x29, 2 },

   // O
   { FONT_INDEX_LATIN_G0_SET, 0x2f, 3 },
   { FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, 0x48, 3 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x2e, 3 },
   { FONT_INDEX_HEBREW_G0_SET, 0x2e, 3 },

   // U
   { FONT_INDEX_LATIN_G0_SET, 0x35, 4 },
   { FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, 0x4c, 4 },
   { FONT_INDEX_GREEK_G2_SUPPLIMENTARY_SET, 0x49, 4 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x35, 4 },
   { FONT_INDEX_HEBREW_G0_SET, 0x35, 4 },

   // C
   { FONT_INDEX_LATIN_G0_SET, 0x23, 5 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_1_SERBIAN_CROATIAN, 0x33, 5 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_2_RUSSIAN_BULGARIAN, 0x33, 5 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_3_UKRANIAN, 0x33, 5 },
   { FONT_INDEX_GREEK_G2_SUPPLIMENTARY_SET, 0x40, 5 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x23, 5 },
   { FONT_INDEX_HEBREW_G0_SET, 0x23, 5 },

   // S
   { FONT_INDEX_LATIN_G0_SET, 0x33, 6 },
   { FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, 0x4b, 6 },
   { FONT_INDEX_GREEK_G2_SUPPLIMENTARY_SET, 0x48, 6 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x33, 6 },
   { FONT_INDEX_HEBREW_G0_SET, 0x33, 6 },

   // K
   { FONT_INDEX_LATIN_G0_SET, 0x2b, 7 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_2_RUSSIAN_BULGARIAN, 0x2b, 7 },
   { FONT_INDEX_CYRILLIC_G0_SET_OPTION_3_UKRANIAN, 0x2b, 7 },
   { FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, 0x46, 7 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x2b, 7 },
   { FONT_INDEX_HEBREW_G0_SET, 0x2b, 7 },

   // Y
   { FONT_INDEX_LATIN_G0_SET, 0x39, 8 },
   { FONT_INDEX_GREEK_G2_SUPPLIMENTARY_SET, 0x4c, 8 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x39, 8 },
   { FONT_INDEX_HEBREW_G0_SET, 0x39, 8 },

   // Z
   { FONT_INDEX_LATIN_G0_SET, 0x3a, 9 },
   { FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, 0x4f, 9 },
   { FONT_INDEX_GREEK_G2_SUPPLIMENTARY_SET, 0x4d, 9 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x3a, 9 },
   { FONT_INDEX_HEBREW_G0_SET, 0x3a, 9 },

   // G
   { FONT_INDEX_LATIN_G0_SET, 0x27, 10 },
   { FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, 0x43, 10 },
   { FONT_INDEX_GREEK_G2_SUPPLIMENTARY_SET, 0x43, 10 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x27, 10 },
   { FONT_INDEX_HEBREW_G0_SET, 0x27, 10 },

   // T
   { FONT_INDEX_LATIN_G0_SET, 0x34, 11 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x34, 11 },
   { FONT_INDEX_HEBREW_G0_SET, 0x34, 11 },

   // N
   { FONT_INDEX_LATIN_G0_SET, 0x2e, 12 },
   { FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, 0x48, 12 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x2e, 12 },
   { FONT_INDEX_HEBREW_G0_SET, 0x2e, 12 },

   // i
   { FONT_INDEX_LATIN_G0_SET, 0x49, 13 },
   { FONT_INDEX_CYRILLIC_G2_SUPPLIMENTARY_SET, 0x54, 13 },
   { FONT_INDEX_ARABIC_G2_SUPPLIMENTARY_SET, 0x49, 13 },

   { FONT_INDEX_UNDEFINED, 0x00, 0 }
};

static const U16BIT cache_removal_priority_mask_array[8] =
{
   0,
   CACHE_PAGE_HISTORY_REQUESTED,
   CACHE_PAGE_PREVNEXT_REQUESTED,
   CACHE_PAGE_HISTORY_REQUESTED | CACHE_PAGE_PREVNEXT_REQUESTED,
   CACHE_PAGE_FLOF_REQUESTED,
   CACHE_PAGE_HISTORY_REQUESTED | CACHE_PAGE_FLOF_REQUESTED,
   CACHE_PAGE_PREVNEXT_REQUESTED | CACHE_PAGE_FLOF_REQUESTED,
   CACHE_PAGE_HISTORY_REQUESTED | CACHE_PAGE_PREVNEXT_REQUESTED | CACHE_PAGE_FLOF_REQUESTED
};

static const U8BIT option_subset_array[96] =
{
   0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 5, 6, 7, 8,
   9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 11, 12, 13, 0
};

static const U8BIT subset_offset_array[14] =
{
   0, 0, 13, 26, 39, 52, 65, 78, 91, 104, 117, 130, 143, 156
};


static U32BIT pes_collection_callback_handle;

static S_PAGE_REQUEST_INFO page_request_info;

static BOOLEAN is_initialised;
static BOOLEAN is_shown;
static BOOLEAN is_visible;

static BOOLEAN stream_processing_enabled;

static S_MAGAZINE_INFO magazine_array[8];

static S_HISTORY_CACHE_REQUEST *history_cache_request_list;
static BOOLEAN history_cache_reset_requested;

static S_EDITORIAL_LINK_CACHE_REQUEST *editorial_link_request_root;
static S_EDITORIAL_LINK_CACHE_REQUEST *temp_editorial_link_request_root;
static BOOLEAN editorial_link_cache_reset_requested;

static U16BIT next_cache_request_array[NUM_NEXT_CACHE_REQUESTS];
static U16BIT previous_cache_request_array[NUM_PREVIOUS_CACHE_REQUESTS];
static BOOLEAN previousnext_cache_reset_requested;
static U16BIT next_prev_page_number;

static E_EBUTT_CACHING_METHOD caching_method;

static S_PAGE_CONTROL_INFO page_control_info;

static S_PAGE_DISPLAY_INFO page_display_info;
static S_PAGE_DISPLAY_INFO page_display_copy;

static U8BIT broadcast_service_data[2][MAX_COLUMN];
static BOOLEAN is_broadcast_service_data_defined[2];

static BOOLEAN broadcast_index_page_invocation_required;
static BOOLEAN show_header_row;

static BOOLEAN stop_teletext;
static U8BIT teletext_path;

static U8BIT default_character_set_mapping;

static void *display_task_ptr;
static void *display_task_semaphore;

static BOOLEAN palette_reset_required;

static U16BIT display_margin_top;
static U16BIT display_margin_left;

static U8BIT *display_char_buffer_origin;
static U8BIT *display_char_buffer;
static U16BIT display_char_buffer_size;

static U8BIT current_gun_intensity = 223; // 87% brightness
static U8BIT current_antialias_level = 8; // Full anti-aliasing
static U8BIT current_transparency_level = 255; // Full mixed-mode transparency

static BOOLEAN diacritic_substitutions_avaialable;

static U16BIT page_history_array[MAX_PAGE_HISTORY];
static U16BIT page_history_request_page_number;
static U16BIT page_history_lwm;
static U16BIT page_history_hwm;
static U16BIT page_history_mwm;

static U32BIT subtitle_timeout;

static void *ttxt_osd_region;

static void *collation_start;
static void *collation_stopped;
static BOOLEAN stop_collation;
static BOOLEAN collation_started;
static void *collation_queue;
static U16BIT queue_count;
static U16BIT max_queue_count;
static void *page_free_sem;

//---local function prototypes------------------------------------------------

static void* GetMemory(U32BIT bytes);

// Data decoding funtions

static BOOLEAN GetHamming8Byte(void *byte, U8BIT *value);
static BOOLEAN GetHamming8Multiple(void *ptr, U8BIT num_bytes, U32BIT *value);
static BOOLEAN GetHamming8DoubleWord(void *dword, U16BIT *value);
static BOOLEAN GetHammimg24Value(void *triplet_ptr, U32BIT *value);

static BOOLEAN GetParity7Byte(U8BIT byte, U8BIT *value);

// General packet data service routines

static BOOLEAN GetPageReference(U8BIT *link_ptr, S_PAGE_REFERENCE_DATA *page_reference_data_ptr,
   E_PAGE_REFERENCE_TYPE reference_type, U16BIT base_magazine_number);

static BOOLEAN IsPacketQueueingRequired(void);

static BOOLEAN GetPacketAddress(U8BIT *packet_number, U8BIT *magazine_number,
   U8BIT *teletext_packet_ptr );

static U16BIT GetMagazineIndexFromPageNumber(U16BIT page_number);

static BOOLEAN IsRollingSubCode(U16BIT page_sub_code);

static void GetBasicPageDisplayInfo(S_PAGE_DISPLAY_INFO *ptr, U8BIT *default_setting,
   U8BIT *second_setting, BOOLEAN *boxing_required,
   BOOLEAN *header_required, BOOLEAN *body_required,
   BOOLEAN *default_row_background_colour_defined,
   U8BIT *default_row_background_colour);

static void GetCharacterPaletteIndexes(S_PAGE_DISPLAY_INFO *display_info_ptr,
   U8BIT *foreground_index, U8BIT *background_index,
   U8BIT foreground_colour, U8BIT background_colour,
   BOOLEAN flashing, BOOLEAN conceal,
   BOOLEAN boxing_required, BOOLEAN boxed,
   BOOLEAN video_mix_overridden);

// PES data collection

static void PESCollectionCallback(U32BIT handle, U8BIT data_identifier, void *data_ptr, U32BIT data_length);
static BOOLEAN AddPESDataToQueue(void *data_ptr, U32BIT data_length);

static BOOLEAN InitialiseCollation(void);

BOOLEAN PerformPESCollation(U8BIT *packet_ptr, BOOLEAN pts_valid, U8BIT *pts);

// Page request funtions

static void RequestPage(U16BIT page_number, U16BIT page_sub_code);
static void RequestIndexPage(void);
static void RequestNextPrevious(E_REQUEST_TYPE request_type, BOOLEAN increment);
static void RequestNextAvailablePage(void);
static void RequestPreviousAvailablePage(void);
static void RequestNextVisitedPage(void);
static void RequestPreviousVisitedPage(void);
static void RequestEditorialLinkPage(U16BIT index);
static void RequestNextSubPage(void);
static void RequestPreviousSubPage(void);

// Editorial link heirarchy analysis functions

static U16BIT GetMaximumEditorialLinkTiers(void);

static BOOLEAN FindEditorialLinkRecursive(S_EDITORIAL_LINK_CACHE_REQUEST *base_ptr,
   S_EDITORIAL_LINK_CACHE_REQUEST **ptr,
   S_EDITORIAL_LINK_CACHE_REQUEST **parent_ptr,
   U16BIT page_number);

static BOOLEAN FindEditorialLinkPageRequest(S_EDITORIAL_LINK_CACHE_REQUEST *base_ptr,
   S_EDITORIAL_LINK_CACHE_REQUEST **ptr,
   S_EDITORIAL_LINK_CACHE_REQUEST **parent_ptr,
   U16BIT page_number);

static BOOLEAN GetEditorialLink(S_EDITORIAL_LINK_CACHE_REQUEST *base_ptr,
   S_EDITORIAL_LINK_CACHE_REQUEST **ptr, U16BIT page_number);

static BOOLEAN RemoveEditorialLinkPageRequest(S_EDITORIAL_LINK_CACHE_REQUEST **base_ptr,
   S_EDITORIAL_LINK_CACHE_REQUEST *ptr);

static BOOLEAN GetSubPageEditorialLinks(S_CACHE_SUBPAGE *sub_page_ptr,
   U16BIT *page_number_link_array,
   U16BIT *page_sub_code_link_array,
   U16BIT max_num_links, U16BIT magazine_number);

static BOOLEAN CreateEditorialLinkPageRequest(S_EDITORIAL_LINK_CACHE_REQUEST **ptr,
   U16BIT page_number, U16BIT sub_code);

static void ReTierEditorialLinkPageRequests(S_EDITORIAL_LINK_CACHE_REQUEST *ptr, U16BIT tier);

static void KillEditorialLinkPageRequests(S_EDITORIAL_LINK_CACHE_REQUEST **base_ptr);

static void UpdateEditorialLinkPageRequestRecursive(S_EDITORIAL_LINK_CACHE_REQUEST *ptr,
   U16BIT tier);

static void UpdateEditorialLinkPageRequest(U16BIT page_number, U16BIT sub_code);

static void FlagEditorialLinkPageRequests(S_EDITORIAL_LINK_CACHE_REQUEST *ptr);

static void DeterminePageEditorialLinks(S_EDITORIAL_LINK_CACHE_REQUEST *ptr,
   S_CACHE_SUBPAGE *subpage_ptr);

// Visited Page history analysis functions

static BOOLEAN SearchRequestArray(U16BIT *array_ptr, U16BIT array_size, BOOLEAN find_highest,
   U16BIT **return_array_ptr);
static BOOLEAN SearchPreviousRequestArray(BOOLEAN find_highest, U16BIT **return_array_ptr);
static BOOLEAN SearchNextRequestArray(BOOLEAN find_highest, U16BIT **return_array_ptr);

static BOOLEAN IsInPreviousRequestArray(U16BIT page_number);
static BOOLEAN IsInNextRequestArray(U16BIT page_number);

// Event-related page request functions

static BOOLEAN GetGlobalIndexPage(U16BIT *page_number, U16BIT *page_sub_code);
static BOOLEAN GetIndexPage(U16BIT *page_number, U16BIT *page_sub_code);
static BOOLEAN GetAvailablePage(U16BIT *page_number, S16BIT page_increment);
static BOOLEAN GetVisitedPage(U16BIT *page_number, U16BIT *page_sub_code, S16BIT page_increment);
static BOOLEAN GetEditorialLinkPage(U16BIT *page_number, U16BIT *page_sub_code, U16BIT link_index);
static BOOLEAN GetSubPage(U16BIT *new_page_sub_code, U16BIT current_page_number,
   U16BIT current_page_sub_code, S16BIT page_increment);

// Page request handling functionality (polled from collation task)

static void CheckPageRequest(BOOLEAN pts_valid, U8BIT *pts);

// Page request chache tidyup functions

static BOOLEAN RemoveCachePage(U16BIT magazine_number, U16BIT mask);
static BOOLEAN RemoveCacheContent(U16BIT magazine_number, BOOLEAN exclude_magazine);
static BOOLEAN RemoveUnrequestedCacheContent(U16BIT magazine_number, BOOLEAN exclude_magazine);
static BOOLEAN RemoveUnrequestedCachePage(U16BIT magazine_number);

#if 0
static void RemoveAllCacheContent(void);
#endif

// Page display update invocation function

static void NotifyDisplayUpdate(S_MAGAZINE_PAGE *collated_page_ptr, U16BIT magazine_number,
   U16BIT page_number, U16BIT page_sub_code, BOOLEAN update_clock, BOOLEAN pts_valid, U8BIT *pts);

// Page request cache checking function

static void CheckPageCaching(S_MAGAZINE_PAGE *collated_page_ptr,
   S_PAGE_REFERENCE_DATA *reference_data_ptr, BOOLEAN pts_valid, U8BIT *pts);

static void CheckTimeFillingPacket(S_MAGAZINE_PAGE *collated_page_ptr, U16BIT page_number);

static void CheckHeaderClockContent(S_MAGAZINE_PAGE *collated_page_ptr);

// Page request history functions

static void NotifyRequestToPageHistory(U16BIT page_number);
static U16BIT MoveHistoryMarker(U16BIT marker, BOOLEAN advance);
static void NotifyDisplayToPageHistory(U16BIT page_number);
static BOOLEAN GetPreviousPageFromHistory(U16BIT *page_number);
static BOOLEAN GetNextPageFromHistory(U16BIT *page_number);

static void NotifyPageCollated(S_MAGAZINE_PAGE *collated_page_ptr, U8BIT magazine_number,
   BOOLEAN pts_valid, U8BIT *pts);

static BOOLEAN InitialiseDisplayTask(void);
static void KillDisplayTask(void);

static BOOLEAN IsNationalOptionSubsetIndex(U8BIT character_index, U8BIT *subset_offset);

static void GetTripletFields(U32BIT triplet_value, U8BIT *address, U8BIT *mode, U8BIT *data);

static void DecodeTeletextPageBody(S_PAGE_DISPLAY_INFO *display_info_ptr, BOOLEAN enforce_header_display);
static void DecodeTeletextPageNumber(S_PAGE_DISPLAY_INFO *display_info_ptr, BOOLEAN video_mix_overridden);
static void DecodeTeletextCarouselHeader(S_PAGE_DISPLAY_INFO *display_info_ptr);
static void DecodeTeletextClock(S_PAGE_DISPLAY_INFO *display_info_ptr, BOOLEAN video_mix_overridden);

static void GetAliasedColourIndexes(S_TELETEXT_CHARACTER *character_ptr, U8BIT *bitmap_ptr,
   U8BIT *aliased_foreground_index, U8BIT *aliased_background_index,
   S_PAGE_DISPLAY_INFO *display_info_ptr);

static void RenderCharacter(U16BIT x, U16BIT y, U16BIT x_scalar, U16BIT y_scalar,
   S_TELETEXT_CHARACTER *character_ptr, S_PAGE_DISPLAY_INFO *display_info_ptr);

static U32BIT GetRGBSetting(BOOLEAN tranparency_required, U8BIT transparency_level,
   BOOLEAN red_required, BOOLEAN green_required, BOOLEAN blue_required,
   U8BIT gun_intensity);

static U32BIT GetRGBRamping(U32BIT fore_colour, U32BIT back_colour, U32BIT foreground_sixteenths);

static void RenderDisplay(S_PAGE_DISPLAY_INFO *display_info_ptr);

static void PerformDisplayUpdate(void);
static void DisplayUpdateTask(void *unwanted_ptr);
static void PESCollationTask(void *param);

//---local function definitions------------------------------------------------

/**
 *

 *
 * @brief   Used to allocate memory from the system heap.
 *                 This function acts memerly as a wrapper for the STB_GetMemory() function, but
 *                 also enaures that the memory resources used by the Teletext caching mechanism
 *                 do not consume all of the system heap.
 *
 * @param   U32BIT bytes - the number of bytes required.
 *
 * @return   void* - a pointer to the heap allocation, or NULL if no memory available.
 *
 */
static void* GetMemory(U32BIT bytes)
{
   // Only allocate heap memory if more than 10% of the total resources are currently available.
   // As other STB layer modules consume the remaining memory, this system will ensure that this
   // driver will 'retreat' in terms of it's total memory consumption if required.
   if (STB_MEMSysRAMUsed() >= 90)
   {
      return(NULL);
   }
   else
   {
      return(STB_GetMemory(bytes));
   }
}

/**
 *

 *
 * @brief   Decodes a hamming 8/4 encoded byte, returning the actual or corrected value.
 *
 * @param   void* byte - a pointer to the 8-bit hamming encoded value to be analysed.
 * @param   U8BIT* value - the decoded value will be returned here by pointer reference.
 *                                Only the botton 4 bits of theis byte constitute data,
 *                                NB - this pointer is optional (pass NULL just to test data)
 *
 * @return   BOOLEAN - TRUE if decoding was successful, FALSE if too many bits error were
 *                           encountered.
 *
 */
static BOOLEAN GetHamming8Byte(void *byte, U8BIT *value)
{
   BOOLEAN retval;

   FUNCTION_START(GetHamming8Byte);

   retval = TRUE;

   if (value != NULL)
   {
      *value = hamming_8_4_table[*(U8BIT *)byte] & HAMMING_8_4_DATA_MASK;
   }

   if (hamming_8_4_table[*(U8BIT *)byte] & HAMMING_ERROR_MASK)
   {
      retval = FALSE;
   }

   FUNCTION_FINISH(GetHamming8Byte);

   return(retval);
}

/**
 *

 *
 * @brief   Decodes a block of hamming 8/4 encoded bytes, returning the actual or corrected
 *                 value.
 *
 * @param   void* ptr - the first byte of the hamming encoded value to be analysed.
 * @param   U8BIT num_bytes - the number of bytes to decode (1 to 8)
 * @param   U32BIT* value - the decoded value will be returned here by pointer reference.
 *                                 NB - this pointer is optional (pass NULL just to test data)
 *
 * @return   BOOLEAN - TRUE if decoding was successful, FALSE if too many bit errors were
 *                           encountered.
 *
 */
static BOOLEAN GetHamming8Multiple(void *ptr, U8BIT num_bytes, U32BIT *value)
{
   U8BIT byte_index, byte_value;
   BOOLEAN retval;
   U8BIT *byte_ptr;

   FUNCTION_START(GetHamming8Multiple);

   retval = FALSE;

   if ((num_bytes > 0) &&
       (num_bytes < 9))
   {
      retval = TRUE;

      if (value != NULL)
      {
         *value = 0;
      }

      byte_ptr = (U8BIT *)ptr;
      byte_ptr += num_bytes - 1;

      for (byte_index = 0; byte_index < num_bytes; byte_index++)
      {
         if (!GetHamming8Byte(byte_ptr, &byte_value))
         {
            retval = FALSE;
            break;
         }

         if (value != NULL)
         {
            *value = (*value << 4L) | byte_value;
         }

         byte_ptr--;
      }
   }

   FUNCTION_FINISH(GetHamming8Multiple);

   return(retval);
}

/**
 *

 *
 * @brief   Decodes four of hamming 8/4 encoded bytes, returning the actual or corrected
 *                 value.
 *
 * @param   void* dword - a pointer to the 32-bit (four byte) hamming encoded value to be
 *                               analysed.
 * @param   U16BIT* value - the decoded value will be returned here by pointer reference.
 *                                 NB - this pointer is optional (pass NULL just to test data)
 *
 * @return   BOOLEAN - TRUE if decoding was successful, FALSE if too many bit errors were
 *                           encountered.
 *
 */
static BOOLEAN GetHamming8DoubleWord(void *dword, U16BIT *value)
{
   U32BIT long_value;
   BOOLEAN retval;

   FUNCTION_START(GetHamming8DoubleWord);

   retval = GetHamming8Multiple(dword, 4, &long_value);

   if (value != NULL)
   {
      *value = (U16BIT)long_value;
   }

   FUNCTION_FINISH(GetHamming8DoubleWord);

   return(retval);
}

/**
 *

 *
 * @brief   Decodes a triplet hamming 24/18 encoded bytes, returning the actual or corrected
 *                 value.
 *
 * @param   void* triplet_ptr - a pointer to the array of three bytes to be decoded.
 * @param   U32BIT* value - the decoded value will be returned here by pointer reference.
 *                                 Only the botton 18 bits of theis double-word constitute data,
 *                                 NB - this pointer is optional (pass NULL just to test data)
 *
 * @return   BOOLEAN - TRUE if decoding was successful, FALSE if too many bit errors were
 *                           encountered.
 *
 */
static BOOLEAN GetHammimg24Value(void *triplet_ptr, U32BIT *value)
{
   U8BIT parity_index;
   BOOLEAN retval;
   U8BIT *byte_ptr;

   FUNCTION_START(GetHammimg24Value);

   retval = TRUE;

   byte_ptr = (U8BIT *)triplet_ptr;

   parity_index = hamming_24_18_parity_table[0][*byte_ptr] ^
      hamming_24_18_parity_table[1][*(byte_ptr + 1)] ^
      hamming_24_18_parity_table[2][*(byte_ptr + 2)];

   if (value != NULL)
   {
      *value = (U32BIT)hamming_24_18_first_byte_table[*byte_ptr];
      *value += (U32BIT)(hamming_24_18_other_bytes_table[*(byte_ptr + 1)] & 0x7f) << 4L;
      *value += (U32BIT)(hamming_24_18_other_bytes_table[*(byte_ptr + 2)] & 0x7f) << 11L;

      *value ^= hamming_24_18_correction_table[parity_index];
   }

   if (hamming_24_18_error_table[parity_index] & HAMMING_ERROR_MASK)
   {
      retval = FALSE;
   }

   FUNCTION_FINISH(GetHammimg24Value);

   return(retval);
}

/**
 *

 *
 * @brief   Decodes a odd parity encoded byte, returning the actual value.
 *
 * @param   U8BIT byte - the byte to be decoded.
 * @param   U8BIT* value - the decoded 7-bit value will be returned here by pointer reference.
 *                                 NB - this pointer is optional (pass NULL just to test data)
 *
 * @return   BOOLEAN - TRUE if decoding was successful, FALSE if a parity error was
 *                           encountered.
 *
 */
static BOOLEAN GetParity7Byte(U8BIT byte, U8BIT *value)
{
   BOOLEAN retval;

   FUNCTION_START(GetParity7Byte);

   retval = FALSE;

   // Use one of the Hamming decoding LUTS to check if the supplied value passes
   // odd parity decoding tests.
   if (hamming_24_18_parity_table[0][byte] & 32)
   {
      if (value != NULL)
      {
         // This LUT effectively extracts the seven data bits and reversed their order.
         *value = hamming_24_18_other_bytes_table[byte];
      }

      retval = TRUE;
   }

   FUNCTION_FINISH(GetParity7Byte);

   return(retval);
}

/**
 *

 *
 * @brief   Decodes an 8-byte page reference block from a Teletext packet.
 *                 Depending on the source of the link, the data has to be decoded in varying ways.
 *
 * @param   U8BIT* link_ptr - a pointer to the page reference data block.
 * @param   S_PAGE_REFERENCE_DATA* page_reference_data_ptr - the structure in which all
 *                                                                  data is returned.
 * @param   E_PAGE_REFERENCE_TYPE reference_type - the nature of the page reference suppied.
 *
 * @param   U16BIT base_magazine_number - the magazine number from which the supplied data
 *                                               originates - this is essential to determining
 *                                               page numbering.
 *
 * @return   BOOLEAN - TRUE if decoding was successful, FALSE if too many bit errors were
 *                           encountered.
 *
 */
static BOOLEAN GetPageReference(U8BIT *link_ptr, S_PAGE_REFERENCE_DATA *page_reference_data_ptr,
   E_PAGE_REFERENCE_TYPE reference_type, U16BIT base_magazine_number)
{
   U8BIT page_number_units, page_number_tens, control_byte;
   BOOLEAN retval;

   static U8BIT reversed_tribit[8] = { 0, 4, 2, 6, 1, 5, 3, 7 };

   FUNCTION_START(GetPageReference);

   retval = FALSE;

   // Attempt to decode the components of the Teletext Page number from the pointer to the editorial
   // link data block.
   if (GetHamming8Byte(link_ptr, &page_number_units) == TRUE)
   {
      if (GetHamming8Byte(link_ptr + 1, &page_number_tens) == TRUE)
      {
         // Attempt to decode the Teletext Page sub-code
         if (GetHamming8DoubleWord(link_ptr + 2, &page_reference_data_ptr->page_sub_code) == TRUE)
         {
            // Mask out any control bits from the sub-code
            page_reference_data_ptr->page_sub_code &= 0x3f7f;

            // Calculate the page offset (within it's magazine -> ?xx)
            page_reference_data_ptr->page_offset = (U16BIT)((page_number_tens << 4) +
                                                            page_number_units);

            // Now perform functionality depending on the source of the editorial link...
            switch (reference_type)
            {
               // This is an X/0 packet, which is a page header
               // Hence we can determine the control flags (as defined in ETSI EN 300 706 V1.2.1,
               // section 9.3.1.3 - Control Bits)
               case PAGE_REFERENCE_TYPE_HEADER:
               {
                  // We analyse several bytes within the data block, each of which yeilds control
                  // bit statuses. In each case we will clear each flag by default, in case there
                  // is a problem parsing the Hamming encoded bytes.

                  page_reference_data_ptr->erase_page = FALSE;

                  if (GetHamming8Byte(link_ptr + 3, &control_byte) == TRUE)
                  {
                     if (control_byte & 0x08)
                     {
                        page_reference_data_ptr->erase_page = TRUE;
                     }
                  }

                  page_reference_data_ptr->newsflash = FALSE;
                  page_reference_data_ptr->subtitle = FALSE;

                  if (GetHamming8Byte(link_ptr + 5, &control_byte) == TRUE)
                  {
                     if (control_byte & 0x04)
                     {
                        page_reference_data_ptr->newsflash = TRUE;
                     }
                     if (control_byte & 0x08)
                     {
                        page_reference_data_ptr->subtitle = TRUE;
                     }
                  }

                  page_reference_data_ptr->suppress_header = FALSE;
                  page_reference_data_ptr->update_indicator = FALSE;
                  page_reference_data_ptr->interrupted_sequence = FALSE;
                  page_reference_data_ptr->inhibit_display = FALSE;

                  if (GetHamming8Byte(link_ptr + 6, &control_byte) == TRUE)
                  {
                     if (control_byte & 0x01)
                     {
                        page_reference_data_ptr->suppress_header = TRUE;
                     }
                     if (control_byte & 0x02)
                     {
                        page_reference_data_ptr->update_indicator = TRUE;
                     }
                     if (control_byte & 0x04)
                     {
                        page_reference_data_ptr->interrupted_sequence = TRUE;
                     }
                     if (control_byte & 0x08)
                     {
                        page_reference_data_ptr->inhibit_display = TRUE;
                     }
                  }

                  page_reference_data_ptr->magazine_serial = FALSE;
                  page_reference_data_ptr->national_option_character_subset = 0;

                  if (GetHamming8Byte(link_ptr + 7, &control_byte) == TRUE)
                  {
                     if (control_byte & 0x01)
                     {
                        page_reference_data_ptr->magazine_serial = TRUE;
                     }

                     page_reference_data_ptr->national_option_character_subset =
                        reversed_tribit[(control_byte & 0x0e) >> 1];
                  }

                  break;
               }

               // This is the X/27/0 packet, which is editorial page linking data (as defined in
               // ETSI EN 300 706 V1.2.1, section 9.6.1 - Packets X/27/0 to X/27/3 for Editoral
               // Linking)
               case PAGE_REFERENCE_TYPE_EDITORIAL_LINK:
               {
                  // We analyse two bytes within the data block, so that the magazine number of
                  // the page link can de determined from the source magazine number.
                  if (GetHamming8Byte(link_ptr + 3, &control_byte) == TRUE)
                  {
                     if (control_byte & 0x08)
                     {
                        base_magazine_number ^= 0x01;
                     }
                  }

                  if (GetHamming8Byte(link_ptr + 5, &control_byte) == TRUE)
                  {
                     if (control_byte & 0x04)
                     {
                        base_magazine_number ^= 0x02;
                     }
                     if (control_byte & 0x08)
                     {
                        base_magazine_number ^= 0x04;
                     }
                  }
                  break;
               }

               // This is the X/28/0 Format 1 packet, which yeilds a service global index page
               // (as defined in ETSI EN 300 706 V1.2.1, section 9.4.2 - Packets X/28/0 Format 1)
               case PAGE_REFERENCE_TYPE_GLOBAL_INDEX:
               {
                  // We analyse two bytes within the data block, so that the magazine number of
                  // the page link can de determined - this is irrespective of the source magazine
                  // number.
                  base_magazine_number = 0;

                  if (GetHamming8Byte(link_ptr + 3, &control_byte) == TRUE)
                  {
                     if (control_byte & 0x08)
                     {
                        base_magazine_number |= 0x01;
                     }
                  }

                  if (GetHamming8Byte(link_ptr + 5, &control_byte) == TRUE)
                  {
                     if (control_byte & 0x04)
                     {
                        base_magazine_number |= 0x02;
                     }
                     if (control_byte & 0x08)
                     {
                        base_magazine_number |= 0x04;
                     }
                  }
                  break;
               }
            }

            // Now we work out the page number of the reference.
            if (base_magazine_number == 0)
            {
               // Magazine 0 represents pages 0x800 to 0x8ff (0x899 is highest selectable)
               page_reference_data_ptr->page_number =
                  2048 + page_reference_data_ptr->page_offset;                    // 0x800 + ??
            }
            else
            {
               // Magazine 1 represents pages 0x100 to 0x1ff (0x199 is highest selectable)
               // This pattern applies for magazines 2 to 7 as well.
               page_reference_data_ptr->page_number =
                  ((U16BIT)base_magazine_number << 8) + page_reference_data_ptr->page_offset;
            }

            page_reference_data_ptr->magazine_number = base_magazine_number;

            retval = TRUE;

            // Now we check the validity of the page reference.
            switch (reference_type)
            {
               case PAGE_REFERENCE_TYPE_EDITORIAL_LINK:
               {
                  if (page_reference_data_ptr->page_offset == 0x00ff)
                  {
                     retval = FALSE;
                  }
                  else if (page_reference_data_ptr->page_sub_code == 0x3f7f)
                  {
                     page_reference_data_ptr->page_sub_code = PAGE_SUB_CODE_UNDEFINED;
                  }
                  break;
               }

               case PAGE_REFERENCE_TYPE_GLOBAL_INDEX:
               {
                  if (page_reference_data_ptr->page_offset == 0x00ff)
                  {
                     page_reference_data_ptr->page_number = 0x0100;
                     page_reference_data_ptr->page_sub_code = PAGE_SUB_CODE_UNDEFINED;
                  }
                  else if (page_reference_data_ptr->page_sub_code == 0x3f7f)
                  {
                     page_reference_data_ptr->page_sub_code = PAGE_SUB_CODE_UNDEFINED;
                  }
                  break;
               }

               default:
                  break;
            }
         }
      }
   }

   FUNCTION_FINISH(GetPageReference);

   return(retval);
}

/**
 *

 *
 * @brief   Called from the PES data collection callback to determine of Teletext packets
 *                 are to be placed on the local queue - these will be processed by an internal
 *                 task.
 *

 *
 * @return   BOOLEAN
 *
 */
static BOOLEAN IsPacketQueueingRequired(void)
{
   BOOLEAN retval;

   FUNCTION_START(IsPacketQueueingRequired);

   retval = stream_processing_enabled;

   FUNCTION_FINISH(IsPacketQueueingRequired);

   return(retval);
}

/**
 *

 *
 * @brief   Called to extract and decode a teletext packet's number and magazine index.
 *
 * @param   U8BIT* packet_number - a pointer into which the packet number is returned.
 *                                        This is a value in the range 0 to 31.
 *                                        NB - this pointer can be set to NULL.
 * @param   U8BIT* magazine_number - a pointer into which the magazine index is returned.
 *                                        This is a value in the range 0 to 7.
 *                                        NB - this pointer can be set to NULL.
 * @param   void* teletext_packet_ptr - a void pointer to a 40-byte packet which represents
 *                                             each data packet (as defined in ETSI EN 300 706
 *                                             V1.2.1, section 9 - Coding of data packets), minus
 *                                             the three leading bytes representing clock run-in and
 *                                             framing  code.
 *                                             Much of this data content is Hamming encoded.
 *                                             This is equivalent to this structure:
 *
 *                                              typedef struct
 *                                              {
 *                                                U8BIT magazine_number;
 *                                                U8BIT packet_number;
 *                                                U8BIT data[40];
 *                                              }
 *
 *
 * @return   BOOLEAN - TRUE if packets required
 *
 */
static BOOLEAN GetPacketAddress(U8BIT *packet_number, U8BIT *magazine_number,
   U8BIT *teletext_packet_ptr )
{
   const S_PACKET_VALIDATION *validation_ptr;
   BOOLEAN retval;
   U8BIT designation_code;

   FUNCTION_START(GetPacketAddress);

   retval = FALSE;

   if (GetHamming8Byte(teletext_packet_ptr + 1, packet_number))
   {
      if (GetHamming8Byte(teletext_packet_ptr, magazine_number))
      {
         // The LSBit of the five-bit packet number is held within the magazine index number.
         *packet_number <<= 1;

         if (*magazine_number & 0x08)
         {
            *packet_number += 1;
         }

         *magazine_number &= 0x07;

         if (*packet_number < 32)
         {
            validation_ptr = packet_validation_array + *packet_number;
            if (validation_ptr->is_required == TRUE)
            {
               if (validation_ptr->has_designation_code == TRUE)
               {
                  // If the packet supports a 'designation code' then this will be the first
                  // data byte.
                  if (GetHamming8Byte(teletext_packet_ptr + 2, &designation_code))
                  {
                     // Check the designation code is within the range required for this packet type.
                     if ((designation_code >= validation_ptr->lowest_designation_code) &&
                         (designation_code <= validation_ptr->highest_designation_code))
                     {
                        retval = TRUE;
                     }
                  }
               }
               else
               {
                  retval = TRUE;
               }
            }
         }
         DBGPRINT("time=%d pkt_num=%d mg_num=%d rtn=%d", STB_OSGetClockMilliseconds(), *packet_number, *magazine_number, retval);
      }
   }

   FUNCTION_FINISH(GetPacketAddress);

   return(retval);
}

/**
 *

 *
 * @brief   Used to determine the magazine number from which a page number would be derived.
 *
 * @param   U16BIT page_number - the page number to be analysed.
 *
 * @return   The calculated magazine number - a value expected to be in the range 0 to 7.
 *
 */
static U16BIT GetMagazineIndexFromPageNumber(U16BIT page_number)
{
   U16BIT retval;

   FUNCTION_START(GetMagazineIndexFromPageNumber);

   page_number >>= 8;
   if (page_number == 8)
   {
      retval = 0;
   }
   else
   {
      retval = page_number;
   }

   FUNCTION_FINISH(GetMagazineIndexFromPageNumber);

   return(retval);
}

/**
 *

 *
 * @brief   Used to determine whether a Teletext sub-page is of a 'rolling' group of pages,
 *                 which are used to offer periodically updated page content.
 *                 Such sub-pages have the value of 0x0001 to 0x0010, 0x0020 to 0x0029, .. ,
 *                 ending with 0x0070 to 0x0079.
 *                 This presumably means that there can only be 80 such sub-pages for a
 *                 given Teletext page.
 *
 * @param   U16BIT page_sub_code - the sub_page value to be tested.
 *
 * @return   BOOLEAN - Returns TRUE if sub-code is of a rolling sub-page, FALSE otherwise.
 *
 */
static BOOLEAN IsRollingSubCode(U16BIT page_sub_code)
{
   U16BIT sub_code_tens, sub_code_units;
   BOOLEAN retval;

   FUNCTION_START(IsRollingSubCode);

   retval = FALSE;

   if (page_sub_code != PAGE_SUB_CODE_UNDEFINED)
   {
      if (!(page_sub_code & 0xff00))
      {
         sub_code_tens = (page_sub_code & 0x00f0) >> 4;
         if (sub_code_tens <= 7)
         {
            sub_code_units = page_sub_code & 0x000f;
            if (sub_code_units <= 9)
            {
               if ((sub_code_tens != 0) ||
                   (sub_code_units != 0))
               {
                  retval = TRUE;
               }
            }
         }
      }
   }

   FUNCTION_FINISH(IsRollingSubCode);

   return(retval);
}

/**
 *

 *
 * @brief   Used to determine some basic display requirements for the given Teletext page.
 *                 This includes the default and secondary G0 and G2 character set mappings, and
 *                 whether the page is of the newsflash/subtitle variety, requiring wholesale
 *                 transparency around pertinent text content.
 *
 * @param   S_PAGE_DISPLAY_INFO* ptr - the Teletext page to be analysed.
 * @param   U8BIT* default_setting - a pointer into which an index into the
 *                                          character_set_mapping_array[] table is returned
 *                                          representing the default set.
 * @param   U8BIT* second_setting - as above, but for the secondary setting
 * @param   BOOLEAN *boxing_required - a pointer into which is reported whether text boxing
 *                                            is required for this page.
 * @param   BOOLEAN* header_required - returns TRUE if the page header (row 1) is to be
 *                                            displayed, FALSE otherwise.
 * @param   BOOLEAN* body_required - returns TRUE if the main page (rows 2 to 25) is to be
 *                                            displayed, FALSE otherwise.
 * @param   BOOLEAN* default_row_background_colour_defined - TRUE if a default row background
 *                                                                  colour is specified.
 * @param   U8BIT* default_row_background_colour - returns a default row background colour
 *                                                        (if applicable).
 *

 *
 */
static void GetBasicPageDisplayInfo(S_PAGE_DISPLAY_INFO *ptr, U8BIT *default_setting,
   U8BIT *second_setting, BOOLEAN *boxing_required,
   BOOLEAN *header_required, BOOLEAN *body_required,
   BOOLEAN *default_row_background_colour_defined,
   U8BIT *default_row_background_colour)
{
   U16BIT magazine_number;
   U32BIT triplet_value;
   BOOLEAN default_mapping_defined, second_mapping_defined;
   S_PAGE_REFERENCE_DATA page_reference_data;

   FUNCTION_START(GetBasicPageDisplayInfo);

   // Start with default character set designations
   if (default_setting != NULL)
   {
      *default_setting = default_character_set_mapping << 3;
   }
   if (second_setting != NULL)
   {
      *second_setting = default_character_set_mapping << 3;
   }

   if (boxing_required != NULL)
   {
      *boxing_required = FALSE;
   }
   if (header_required != NULL)
   {
      *header_required = show_header_row;
   }
   if (body_required != NULL)
   {
      *body_required = TRUE;
   }

   default_mapping_defined = FALSE;
   second_mapping_defined = FALSE;

   if (default_row_background_colour_defined != NULL)
   {
      *default_row_background_colour_defined = FALSE;
   }

   // Attempt to get default and secondary GO and G2 sets and national option sub-set designation
   // from X/28/0
   if (ptr->page_source.specific_data.is_defined[0] == TRUE)
   {
      if (GetHammimg24Value(&ptr->page_source.specific_data.s_data[0][1], &triplet_value) == TRUE)
      {
         if (default_setting != NULL)
         {
            *default_setting = (U8BIT)((triplet_value & 0x003f80L) >> 7L);
         }
         default_mapping_defined = TRUE;

         if (second_setting != NULL)
         {
            *second_setting = (U8BIT)((triplet_value & 0x03c000L) >> 14L);
         }

         if (GetHammimg24Value(&ptr->page_source.specific_data.s_data[0][4], &triplet_value) == TRUE)
         {
            if (second_setting != NULL)
            {
               *second_setting |= (U8BIT)((triplet_value & 0x000007L) << 4L);
            }
            second_mapping_defined = TRUE;
         }
      }

      if (GetHammimg24Value(&ptr->page_source.specific_data.s_data[0][37], &triplet_value) == TRUE)
      {
         if (triplet_value & 0x004000L)
         {
            if (default_row_background_colour_defined != NULL)
            {
               if (default_row_background_colour != NULL)
               {
                  *default_row_background_colour = (U8BIT)((triplet_value & 0x003e00L) >> 9L);
               }
               *default_row_background_colour_defined = TRUE;
            }
         }
      }
   }

   if (ptr->page_source.specific_data.is_defined[1] == TRUE)
   {
      if (default_mapping_defined == FALSE)
      {
         // Attempt to get default and secondary GO and G2 sets and national option sub-set
         // designation from X/28/1
         if (GetHammimg24Value(&ptr->page_source.specific_data.s_data[1][1], &triplet_value) == TRUE)
         {
            if (default_setting != NULL)
            {
               *default_setting = (U8BIT)((triplet_value & 0x003f80L) >> 7L);
            }
            default_mapping_defined = TRUE;

            if (second_setting != NULL)
            {
               *second_setting = (U8BIT)((triplet_value & 0x03c000L) >> 14L);
            }

            if (GetHammimg24Value(&ptr->page_source.specific_data.s_data[1][4], &triplet_value) == TRUE)
            {
               if (second_setting != NULL)
               {
                  *second_setting |= (U8BIT)((triplet_value & 0x000007L) << 4L);
               }
               second_mapping_defined = TRUE;
            }
         }
      }
      else if (second_mapping_defined == FALSE)
      {
         if (GetHammimg24Value(&ptr->page_source.specific_data.s_data[1][1], &triplet_value) == TRUE)
         {
            if (second_setting != NULL)
            {
               *second_setting = (U8BIT)((triplet_value & 0x03c000L) >> 14L);
            }

            if (GetHammimg24Value(&ptr->page_source.specific_data.s_data[1][4], &triplet_value) == TRUE)
            {
               if (second_setting != NULL)
               {
                  *second_setting |= (U8BIT)((triplet_value & 0x000007L) << 4L);
               }
               second_mapping_defined = TRUE;
            }
         }
      }

      if (default_row_background_colour_defined != NULL)
      {
         if (*default_row_background_colour_defined == FALSE)
         {
            if (GetHammimg24Value(&ptr->page_source.specific_data.s_data[1][37], &triplet_value) == TRUE)
            {
               if (triplet_value & 0x004000L)
               {
                  if (default_row_background_colour != NULL)
                  {
                     *default_row_background_colour = (U8BIT)((triplet_value & 0x003e00L) >> 9L);
                  }
                  *default_row_background_colour_defined = TRUE;
               }
            }
         }
      }
   }

   if (ptr->mgzn_data.is_defined[0] == TRUE)
   {
      if (default_mapping_defined == FALSE)
      {
         // Attempt to get default and secondary GO and G2 sets and national option sub-set
         // designation from M/29/0
         if (GetHammimg24Value(&ptr->mgzn_data.s_data[0][1], &triplet_value) == TRUE)
         {
            if (default_setting != NULL)
            {
               *default_setting = (U8BIT)((triplet_value & 0x003f80L) >> 7L);
            }
            default_mapping_defined = TRUE;

            if (second_setting != NULL)
            {
               *second_setting = (U8BIT)((triplet_value & 0x03c000L) >> 14L);
            }

            if (GetHammimg24Value(&ptr->mgzn_data.s_data[0][4], &triplet_value) == TRUE)
            {
               if (second_setting != NULL)
               {
                  *second_setting |= (U8BIT)((triplet_value & 0x000007L) << 4L);
               }
               second_mapping_defined = TRUE;
            }
         }
      }
      else if (second_mapping_defined == FALSE)
      {
         if (GetHammimg24Value(&ptr->mgzn_data.s_data[0][1], &triplet_value) == TRUE)
         {
            if (second_setting != NULL)
            {
               *second_setting = (U8BIT)((triplet_value & 0x03c000L) >> 14L);
            }

            if (GetHammimg24Value(&ptr->mgzn_data.s_data[1][4], &triplet_value) == TRUE)
            {
               if (second_setting != NULL)
               {
                  *second_setting |= (U8BIT)((triplet_value & 0x000007L) << 4L);
               }
               second_mapping_defined = TRUE;
            }
         }
      }

      if (default_row_background_colour_defined != NULL)
      {
         if (*default_row_background_colour_defined == FALSE)
         {
            if (GetHammimg24Value(&ptr->mgzn_data.s_data[0][37], &triplet_value) == TRUE)
            {
               if (triplet_value & 0x004000L)
               {
                  if (default_row_background_colour != NULL)
                  {
                     *default_row_background_colour = (U8BIT)((triplet_value & 0x003e00L) >> 9L);
                  }
                  *default_row_background_colour_defined = TRUE;
               }
            }
         }
      }
   }

   if (ptr->mgzn_data.is_defined[1] == TRUE)
   {
      if (default_mapping_defined == FALSE)
      {
         // Attempt to get default and secondary GO and G2 sets and national option sub-set
         // designation from M/29/1
         if (GetHammimg24Value(&ptr->mgzn_data.s_data[1][1], &triplet_value) == TRUE)
         {
            if (default_setting != NULL)
            {
               *default_setting = (U8BIT)((triplet_value & 0x003f80L) >> 7L);
            }
            default_mapping_defined = TRUE;

            if (second_setting != NULL)
            {
               *second_setting = (U8BIT)((triplet_value & 0x03c000L) >> 14L);
            }

            if (GetHammimg24Value(&ptr->mgzn_data.s_data[1][4], &triplet_value) == TRUE)
            {
               if (second_setting != NULL)
               {
                  *second_setting |= (U8BIT)((triplet_value & 0x000007L) << 4L);
               }
               second_mapping_defined = TRUE;
            }
         }
      }
      else if (second_mapping_defined == FALSE)
      {
         if (GetHammimg24Value(&ptr->mgzn_data.s_data[1][1], &triplet_value) == TRUE)
         {
            if (second_setting != NULL)
            {
               *second_setting = (U8BIT)((triplet_value & 0x03c000L) >> 14L);
            }

            if (GetHammimg24Value(&ptr->mgzn_data.s_data[1][4], &triplet_value) == TRUE)
            {
               if (second_setting != NULL)
               {
                  *second_setting |= (U8BIT)((triplet_value & 0x000007L) << 4L);
               }
               second_mapping_defined = TRUE;
            }
         }
      }

      if (default_row_background_colour_defined != NULL)
      {
         if (*default_row_background_colour_defined == FALSE)
         {
            if (GetHammimg24Value(&ptr->mgzn_data.s_data[1][37], &triplet_value) == TRUE)
            {
               if (triplet_value & 0x004000L)
               {
                  if (default_row_background_colour != NULL)
                  {
                     *default_row_background_colour = (U8BIT)((triplet_value & 0x003e00L) >> 9L);
                  }
                  *default_row_background_colour_defined = TRUE;
               }
            }
         }
      }
   }

   // Finally override the default G0 set national option sub-set designation from setting in X/0
   // header.
   if (ptr->page_source.is_header_defined == TRUE)
   {
      magazine_number = GetMagazineIndexFromPageNumber(page_display_info.page_number);

      if (GetPageReference(ptr->page_source.header, &page_reference_data,
             PAGE_REFERENCE_TYPE_HEADER, magazine_number) == TRUE)
      {
         if (default_setting != NULL)
         {
            *default_setting &= 0xf8;
            *default_setting |= page_reference_data.national_option_character_subset;
         }
      }

      ptr->subtitle = page_reference_data.subtitle;

      if ((page_reference_data.newsflash == TRUE) ||
          (page_reference_data.subtitle == TRUE))
      {
         if (boxing_required != NULL)
         {
            *boxing_required = TRUE;
         }
      }

      if (page_reference_data.suppress_header == TRUE)
      {
         if (header_required != NULL)
         {
            *header_required = FALSE;
         }
      }

      if (page_reference_data.inhibit_display == TRUE)
      {
         if (body_required != NULL)
         {
            *body_required = FALSE;
         }
      }
   }

   FUNCTION_FINISH(GetBasicPageDisplayInfo);
}

/**
 *

 *
 * @brief   Used to determine the the foreground pallete indexes to be used for a specific_data
 *                 atttrubute collection derived from teletext page content.
 *                 This routine primarily exists to collate any dynamic palette requirements (such
 *                 as those for flashing and concealed text), and also to map the absolute colour
 *                 specifications from page data into specific_data palette allocations.
 *
 * @param   U8BIT* foreground_index, U8BIT* background_index -
 *                   the variables into which palette LUT references are returned.
 * @param   U8BIT foreground_colour, U8BIT background_colour -
 *                   the colour attribute values as defined in the telextext data (0 to 7)
 * @param   BOOLEAN flashing, BOOLEAN conceal -
 *                   the display attributes curently required for the teletext display.
 * @param   BOOLEAN boxing_required -
 * @param   BOOLEAN boxed -
 *                   the subtitle/newsflash related attributes curently required for the teletext
 *                   display.
 * @param   BOOLEAN video_mix_overridden - used to suppress video mix effect when displaying
 *                                                the page header.
 *

 *
 */
static void GetCharacterPaletteIndexes(S_PAGE_DISPLAY_INFO *display_info_ptr,
   U8BIT *foreground_index, U8BIT *background_index,
   U8BIT foreground_colour, U8BIT background_colour,
   BOOLEAN flashing, BOOLEAN conceal,
   BOOLEAN boxing_required, BOOLEAN boxed,
   BOOLEAN video_mix_overridden)
{
   U8BIT fore_colour, back_colour, effect_mask;
   U16BIT index;

   FUNCTION_START(GetCharacterPaletteIndexes);

   if ((boxing_required == TRUE) &&
       (boxed == FALSE))
   {
      // If we are displaying a newsflash/subtitle screen and the current text is not boxed, then
      // mark the foreground index to denote that nothing is to be displayed.
      *foreground_index = PALETTE_FOREGROUND_UNBOXED;
   }
   else
   {
      if (foreground_colour < 8)
      {
         // Remap the absolute Teletext colour reference to a foreground reservation in the palette.
         fore_colour = foreground_colour + PALETTE_FOREGROUND_BLACK;
      }
      else
      {
         // The colour reference is invalid - default to white
         fore_colour = PALETTE_FOREGROUND_WHITE;
      }

      if (background_colour < 8)
      {
         // Remap the absolute Teletext colour reference to a background reservation in the palette.
         back_colour = background_colour + PALETTE_BACKGROUND_BLACK;
      }
      else
      {
         // The colour reference is invalid - default to black
         back_colour = PALETTE_BACKGROUND_BLACK;
      }

      // determine which display attributes result in a 'dynamic' palette requirement.
      effect_mask = PALETTE_EFFECT_UNDEFINED;

      if (flashing == TRUE)
      {
         effect_mask = PALETTE_EFFECT_FLASH;
      }
      if (conceal == TRUE)
      {
         effect_mask |= PALETTE_EFFECT_CONCEAL;
      }

      // If there is a need for a reserved palette entry...
      if (effect_mask != PALETTE_EFFECT_UNDEFINED)
      {
         for (index = 0; index < NUM_PALETTE_EFFECTS; index++)
         {
            if (display_info_ptr->palette_attribute_array[index].effect_mask ==
                PALETTE_EFFECT_UNDEFINED)
            {
               // If we have reached the end of the currently allocated reservations, the add a new
               // one.
               display_info_ptr->palette_attribute_array[index].fore_colour = fore_colour;
               display_info_ptr->palette_attribute_array[index].back_colour = back_colour;
               display_info_ptr->palette_attribute_array[index].effect_mask = effect_mask;

               if (index < (NUM_PALETTE_EFFECTS - 1))
               {
                  display_info_ptr->palette_attribute_array[index + 1].effect_mask =
                     PALETTE_EFFECT_UNDEFINED;
               }

               fore_colour = index + PALETTE_EFFECT_ORIGIN;
               break;
            }


            // If a suitable allocation already exists, then use it.
            if ((display_info_ptr->palette_attribute_array[index].fore_colour == fore_colour) &&
                (display_info_ptr->palette_attribute_array[index].back_colour == back_colour) &&
                (display_info_ptr->palette_attribute_array[index].effect_mask == effect_mask))
            {
               fore_colour = index + PALETTE_EFFECT_ORIGIN;
               break;
            }
         }
      }

      *foreground_index = fore_colour;

      if ((display_info_ptr->video_mix_set == TRUE) &&
          (video_mix_overridden == TRUE))
      {
         if (back_colour == PALETTE_BACKGROUND_BLACK)
         {
            *background_index = PALETTE_BACKGROUND_BLACK_FIXED;
         }
      }
      else
      {
         *background_index = back_colour;
      }
   }

   FUNCTION_FINISH(GetCharacterPaletteIndexes);
}

static BOOLEAN GetPesPts(U8BIT *data, U8BIT *pts)
{
   BOOLEAN retval;

   FUNCTION_START(GetPesPts);

   pts[0] = (data[0] & 0x08) >> 3;
   pts[1] = ((data[0] & 0x06) << 5) + ((data[1] & 0xfc) >> 2);
   pts[2] = ((data[1] & 0x03) << 6) + ((data[2] & 0xfc) >> 2);
   pts[3] = ((data[2] & 0x02) << 6) + ((data[3] & 0xfe) >> 1);
   pts[4] = ((data[3] & 0x01) << 7) + ((data[4] & 0xfe) >> 1);

   retval = (data[0] & 0x01) & (data[2] & 0x01) & (data[4] & 0x01);

   FUNCTION_FINISH(GetPesPts);

   return(retval);
}

/**
 *

 *
 * @brief   Callback passed by call to STB_RegisterPesCollectionCallback(), which is the
 *                 portal by which all Teletext data is sent to this driver mechanism.
 *                 This function is repeatedly called by other thread(s) for each set of data
 *                 packet(s).
 *
 * @param   U32BIT handle - the handle of the callback registration - NOT USED
 * @param   U8BIT data_identifier - the identifier of the PES data - NOT USED
 * @param   void* data_ptr - a pointer to a data block containing one (or more) whole
 *                                  Teletext packets.
 * @param   U32BIT data_length - the length of the data block (in bytes)
 *

 *
 */
static void PESCollectionCallback(U32BIT handle, U8BIT data_identifier, void *data_ptr, U32BIT data_length)
{
   FUNCTION_START(PESCollectionCallback);
   USE_UNWANTED_PARAM(handle);
   USE_UNWANTED_PARAM(data_identifier);

   AddPESDataToQueue(data_ptr, data_length);

   FUNCTION_FINISH(PESCollectionCallback);
}

static BOOLEAN AddPESDataToQueue(void *data_ptr, U32BIT data_length)
{
   BOOLEAN retval;
   U16BIT data_remaining;

   // The PES packet header (as defined in BS ISO/IEC 13818-1, Table 2-17 - PES packet) precedes
   // one or more Teletext packets in PES data fields
   //
   // Thus is equivalent to this structure:
   //
   //    typedef struct
   //    {
   //       U8BIT start_code_prefix[3];   // This should have values of 0x00, 0x00, 0x01
   //       U8BIT stream_id;              // Thus should be 0xbd, which is 'private_stream_1'.
   //
   //       U8BIT packet_length[2];       // This is a big endian value, hence the byte pair will
   //                                     // need conversion.
   //                                     // NB - this must be non-zero!
   //
   //       U8BIT padding[2];             // Two irrelevant bytes.
   //
   //       U8BIT header_length;          // Offset applied to <header_end_marker> that gives the
   //                                     // location of the 'PES data field' block.
   //       U8BIT header_end_marker;
   //    }
   U8BIT *pes_packet_header_ptr;

   // Within the PES packet is the PES data block which in turn contains 'PES data fields' (as
   // defined in ETSI EN 300 706 V1.2.1, section 4.3 - Syntax of PES data field, Table 1: Syntax
   // for PES data field) is found all the Teletext packet(s).
   //
   // Thus is equivalent to these structures:
   //
   //    typedef struct
   //    {
   //       U8BIT data_unit_id;     // This value is irrelevant.
   //
   //       U8BIT data_unit_length; // This MUST be 0x2c (44 bytes) as it signifies the total
   //                               // length of data represented by the three subsequent elements.
   //                               // This should be 1 + 1 + 42  = 44 bytes!
   //
   //       // These definitions below map to the content of the 'data_field' of the 'PES data field'
   //
   //       U8BIT field_byte;
   //       // These values should be validated against specified constants as part of the packet
   //       // validation process.
   //       U8BIT framing_code;
   //
   //       S_TELETEXT_PACKET teletext_packet;
   //    }
   //    S_PES_DATA_FIELD;
   //
   //    typedef struct s_pes_data
   //    {
   //       U8BIT data_indentifier; // This should be in the renage 0x10 to 0x1f.
   //                               // This is valid range for PES content relating to EBU Teletext
   //                               // data.
   //
   //       S_PES_DATA_FIELD data_field;
   //    }
   U8BIT *data_identifier_ptr;
   U8BIT *pes_data_field_ptr;
   U8BIT pts_dts_flags, pts_present;
   U8BIT pts[5];
   S_COLLATION_QUEUE_ITEM queue_item;

   // These are equivalent to <start_code_prefix> and <stream_id> of the PES packet header
   // The valid value of 0xbd denotes this as 'private_stream_1'
   static U8BIT pes_packet_header_validator[4] = { 0x00, 0x00, 0x01, 0xbd };

   FUNCTION_START(AddPESDataToQueue);

   USE_UNWANTED_PARAM(data_length);

   retval = FALSE;

   if (IsPacketQueueingRequired() == TRUE)
   {
      pes_packet_header_ptr = (U8BIT *)data_ptr;

      // Validate the <start_code_prefix> (3 byte signature) and the <stream_id>
      if (!memcmp(pes_packet_header_ptr, pes_packet_header_validator, sizeof(pes_packet_header_validator)))
      {
         // Get pointer to start of PES data content within PES packet - after header.
         data_identifier_ptr = pes_packet_header_ptr + *(pes_packet_header_ptr + 8) + 9;

         // Check that data identifier denotes that this PES data is 'EBU data'
         if ((*data_identifier_ptr >= 0x10) &&
             (*data_identifier_ptr <= 0x1f))
         {
            /* Extract the PTS from the PES header if it's present */
            pts_dts_flags = (pes_packet_header_ptr[7] & 0xc0) >> 6;
            pts_present = ((pts_dts_flags & 0x2) >> 1);

            if (pts_present)
            {
               if (!GetPesPts(&pes_packet_header_ptr[9], pts))
               {
                  // One or more of the 3 marker_bits are not set correctly.
                  memset(pts, 0x00, 5);
               }
            }

            // get the packet length, and then subtract the header length.
            // This results in the amount of data which constitutes a sequence of
            // PES data fields.
            data_remaining = (*(pes_packet_header_ptr + 4) << 8) + *(pes_packet_header_ptr + 5) - 4;
            data_remaining -= *(pes_packet_header_ptr + 8);

            // Get a pointer to the first PES data field.
            pes_data_field_ptr = data_identifier_ptr + 1;

            // Each PES data field has four header bytes, and then 42 bytes of actual Teletext
            // packet data.
            while (data_remaining >= PES_DATA_FIELD_WIDTH)
            {
               retval = TRUE;

               // The <data_unit_length> should always be 0x2c, which is 44 bytes.
               // The <framing_code> should always be 0xe4.
               if ((*(pes_data_field_ptr + 1) == 0x2c) &&
                   (*(pes_data_field_ptr + 3) == 0xe4))
               {
                  /* Allocate memory for this data packet and write it to the queue */
                  if ((queue_item.data = STB_GetMemory(PES_DATA_FIELD_WIDTH - 4)) != NULL)
                  {
                     memcpy(queue_item.data, pes_data_field_ptr + 4, PES_DATA_FIELD_WIDTH - 4);

                     queue_item.pts_present = pts_present;
                     if (pts_present)
                     {
                        memcpy(queue_item.pts, pts, 5);
                     }

                     if (STB_OSWriteQueue(collation_queue, (void *)&queue_item, sizeof(queue_item), TIMEOUT_NOW))
                     {
                        queue_count++;
                        if (queue_count > max_queue_count)
                        {
                           max_queue_count = queue_count;
                           EBU_DBG("%s: max_queue_count=%u\n", __FUNCTION__, max_queue_count);
                        }
                     }
                     else
                     {
                        EBU_DBG("%s: Failed to write to collation queue\n", __FUNCTION__);
                        STB_FreeMemory(queue_item.data);
                     }
                  }
               }

               // Advance to the next PES data field (if possible)
               pes_data_field_ptr += PES_DATA_FIELD_WIDTH;
               data_remaining -= PES_DATA_FIELD_WIDTH;
            }
         }
      }
   }

   FUNCTION_FINISH(AddPESDataToQueue);

   return(retval);
}

static void PESCollationTask(void *param)
{
   S_COLLATION_QUEUE_ITEM queue_item;

   USE_UNWANTED_PARAM(param);

   while (1)
   {
      STB_OSSemaphoreWait(collation_start);
      collation_started = TRUE;
      while (!stop_collation)
      {
         if (STB_OSReadQueue(collation_queue, &queue_item, sizeof(queue_item), TIMEOUT_NEVER))
         {
            queue_count--;

            if (queue_item.data != NULL)
            {
               /* Process the teletext packet, which consists of two leading bytes for the packet and
                * magazine identification, and then 40 bytes of packet data */
               PerformPESCollation(queue_item.data, queue_item.pts_present, queue_item.pts);

               STB_FreeMemory(queue_item.data);
            }
         }
      }

      STB_OSSemaphoreSignal(collation_stopped);
      collation_started = FALSE;
   }
}

/**
 *

 *
 * @brief   Called from STB_EBUTT_Initialise( ) to initilaise the semaphore and task thread
 *                 used to collate teletext packets from the internal queue into Teletext pages,
 *                 which asre then passed on to the cache mechanism.
 *

 *
 * @return   BOOLEAN - TRUE if initialised successfully, FALSE otherwise.
 *
 */
static BOOLEAN InitialiseCollation(void)
{
   U16BIT i;
   BOOLEAN retval;

   FUNCTION_START(InitialiseCollation);

   retval = FALSE;

   // There is currently no way to kill a semaphore, so initialisation is a single-shot affair!

   if (page_request_info.semaphore == NULL)
   {
      page_request_info.semaphore = STB_OSCreateSemaphore();
   }

   if (page_display_info.page_semaphore == NULL)
   {
      page_display_info.page_semaphore = STB_OSCreateSemaphore();
   }
   if (page_display_info.page_index_semaphore == NULL)
   {
      page_display_info.page_index_semaphore = STB_OSCreateSemaphore();
   }
   if (page_display_info.carousel_page_semaphore == NULL)
   {
      page_display_info.carousel_page_semaphore = STB_OSCreateSemaphore();
   }
   if (page_display_info.time_filler_header_data_semaphore == NULL)
   {
      page_display_info.time_filler_header_data_semaphore = STB_OSCreateSemaphore();
   }
   if (page_display_info.page_scale_semaphore == NULL)
   {
      page_display_info.page_scale_semaphore = STB_OSCreateSemaphore();
   }

   is_broadcast_service_data_defined[0] = FALSE;
   is_broadcast_service_data_defined[1] = FALSE;

   memset(magazine_array, 0, sizeof(S_MAGAZINE_INFO) * 8 );

   // If we have a semaphore (either from just above or a previos call to this function then
   // continue with the initialisation of the corresponding task.
   if ((page_request_info.semaphore != NULL) &&
       (page_display_info.page_semaphore != NULL) &&
       (page_display_info.page_index_semaphore != NULL) &&
       (page_display_info.carousel_page_semaphore != NULL) &&
       (page_display_info.time_filler_header_data_semaphore != NULL) &&
       (page_display_info.page_scale_semaphore != NULL))
   {
      retval = TRUE;
      for (i = 0; i != 8; i++)
      {
         magazine_array[i].m_mutex = STB_OSCreateMutex();
         if (magazine_array[i].m_mutex == NULL)
         {
            retval = FALSE;
            DBGPRINT("STB_OSCreateMutex failed");
            break;
         }
      }
   }

   FUNCTION_FINISH(InitialiseCollation);

   return(retval);
}

/**
 *

 *
 * @brief   This process is called to service the queue populated with Teletext packets by
 *                 the PES collectiom callback.
 *                 Here the process of analysing each data packet, and grouping these packets into
 *                 whole page data blocks is perfomred,
 *

 *
 * @return   TRUE if a PES packet was avaiable on the colleation queue (and has subsequently
 *                 been processed, FALSE if no data was avaiable.
 *
 */
BOOLEAN PerformPESCollation(U8BIT *data_ptr, BOOLEAN pts_valid, U8BIT *pts)
{
   U8BIT packet_number, magazine_number, control_byte;
   U8BIT designation_code, magazine_index, format_number;
   U32BIT triplet_value;
   BOOLEAN retval, update_required;

   FUNCTION_START(PerformPESCollation);

   retval = FALSE;

   CheckPageRequest(pts_valid, pts);

   if (GetPacketAddress(&packet_number, &magazine_number, data_ptr))
   {
      data_ptr += 2;

      // On the basis of the packet number, determine how the data content should be handled.
      if (packet_number == 0)
      {
         // This is an X/0 packet, which is a page header
         // (as defined in ETSI EN 300 706 V1.2.1, section 9.3.1 - Page Header)
         //
         // Thus is equivalent to this structure:
         //
         //    typedef struct
         //    {
         //       // Byte-pair used with the magazine index to genereate a page index.
         //       // (data is Hamming 8/4 protected, data in bits 2,4,6,8)
         //       U8BIT units;
         //       U8BIT tens;
         //
         //       // Page sub-code, and some control bits.
         //       // (data is Hamming 8/4 protected, sub-code in 13 of sixteen bits 2,4,6,...,30,32)
         //       // NB - control bits are 'Erase Page', 'Newsflash', and 'Subtitle'.
         //       union
         //       {
         //          U32BIT sub_code;
         //          U8BIT control_bytes_a[4];  // only control_bytes_a[1] and control_bytes_a[3]
         //                                     // have control bit content.
         //       };
         //
         //       // The remaining control bits.
         //       // (data is Hamming 8/4 protected, data in bits 2,4,6,8)
         //       // NB - control bits are 'Suppress Header', 'Update Indicator',
         //       //      'Interrupted Sequence', 'Inhibit Display', 'Magazine Serial', and
         //       //      'Nat. Option Char. Subset'.
         //       U8BIT control_bytes_b[2];
         //
         //       // Character codes for top line after edit index and page index (8 characters)
         //       // (for each byte, odd partiy coded in MSbit.)
         //       U8BIT data[32];
         //    }
         //    S_PAGE_HEADER;

         // To get to the X/0 header within the Teletext packet, we must skip over the page and
         // magazine number byte pair.

         // Before copying the data into the magazine, the existing content is checked and passed on
         // to the cache mechanism for further processing - this is achieved by calling
         // NotifyPageCollated( ).

         // Get hold of the last control byte
         if (GetHamming8Byte(data_ptr + 7, &control_byte) == TRUE)
         {
            if (control_byte & 0x01)
            {
               // 'Magazine Serial' mode is enabled.
               // When each page header is received, the last fully collated page is passed onto the
               // cache - irrespective of it's magazine.

               // Search all the magazines for a previously transmitted page...
               for (magazine_index = 0; magazine_index != 8; magazine_index++)
               {
                  // ..if found, pass onto the cacheing interface.
                  if (magazine_array[magazine_index].m_page.is_header_defined == TRUE)
                  {
                     NotifyPageCollated(&magazine_array[magazine_index].m_page, magazine_index,
                        pts_valid, pts);

                     magazine_array[magazine_index].m_page.is_header_defined = FALSE;
                     break;
                  }
               }
            }
            else
            {
               // 'Magazine Serial' mode is disabled.
               // When each page header is received, the last page in the SAME magazine is passed
               // onto the cache.

               // Pass page onto cache - don't bother clearing the is_header_defined flag,
               // as we are about to set it anyway!
               if (magazine_array[magazine_number].m_page.is_header_defined == TRUE)
               {
                  NotifyPageCollated(&magazine_array[magazine_number].m_page, magazine_number,
                     pts_valid, pts);
               }
            }

            // Copy header packet into allotted space in relevant magazine.
            memcpy(magazine_array[magazine_number].m_page.header, data_ptr, MAX_COLUMN);

            magazine_array[magazine_number].m_page.is_header_defined = TRUE;

            if (magazine_array[magazine_number].m_page.display_data_defined_mask)
            {
               DBGPRINT("reseting mask %#x", magazine_array[magazine_number].m_page.display_data_defined_mask );
            }
            magazine_array[magazine_number].m_page.display_data_defined_mask = 0;

            magazine_array[magazine_number].m_page.basic_enhancement_data_defined_mask = 0;

            magazine_array[magazine_number].m_page.is_editorial_page_linking_data_defined = FALSE;

            magazine_array[magazine_number].m_page.specific_data.is_defined[0] = FALSE;
            magazine_array[magazine_number].m_page.specific_data.is_defined[1] = FALSE;
         }
      }
      else if ((packet_number >= 1) &&
               (packet_number <= MAX_ROWS))
      {
         // These are X/1 to X/25 packets, which contain all the information that is subsequently
         // parsed to give a TeleText page display.
         // Control codes within the header packet determine the exact nature of some of these
         // packets, and the packet content may be inherited by subsequent pages of the 'Erase page'
         // control code is not set.
         // (as defined in ETSI EN 300 706 V1.2.1, section 9.3.2 - Packets X/1 to X/25)
         //
         // Thus is equivalent to this structure:
         //
         //    char data[40];

         // To get to the X/25 data within the Teletext packet, we must skip over the page and
         // magazine number byte pair.

         // Copy the data packet into allotted space in relevant magazine.
         memcpy(magazine_array[magazine_number].m_page.display_data[packet_number], data_ptr, MAX_COLUMN);

         // Set the relevant bit in the definition notification mask.
         magazine_array[magazine_number].m_page.display_data_defined_mask |= 1 << packet_number;
      }
      else if (packet_number == 26)
      {
         // These are X/26/0 to X/26/15 packets, which are enhanced data.
         // (as defined in ETSI EN 300 706 V1.2.1, section 9.3.2 - Packets X/1 to X/25)
         //
         // Thus is equivalent to this structure:
         //
         //    typedef struct
         //    {
         //       // This is the 'Designation Code' used as index for enhancement packet array.
         //       // (data is Hamming 8/4 protected, data in bits 2,4,6,8)
         //       U8BIT designation_code;
         //
         //       // 13 triplets, each describing different attributes.
         //       // (data is Hamming 24/18 protected)
         //       U8BIT data[39];
         //    }
         //    S_PAGE_BASIC_ENHANCEMENT_DATA;

         // To get to the X/26 enhancement data within the Teletext packet, we must skip over the
         // page and magazine number byte pair.

         // The first byte is the designation code, which yelids a 4-bit value.
         if (GetHamming8Byte(data_ptr, &designation_code) == TRUE)
         {
            // Copy the enhanced data packet into allotted space in relevant magazine.
            memcpy(magazine_array[magazine_number].m_page.basic_enhancement_data[designation_code],
               data_ptr, MAX_COLUMN);

            if (designation_code)
            {
               magazine_array[magazine_number].m_page.basic_enhancement_data_defined_mask |=
                  1 << designation_code;
            }
            else
            {
               magazine_array[magazine_number].m_page.basic_enhancement_data_defined_mask |= 1;
            }
         }
      }
      else if (packet_number == 27)
      {
         // This is the X/27/0 packet, which is editorial page linking data (for FLOF navigations).
         // (as defined in ETSI EN 300 706 V1.2.1, section 9.6.1 - Packets X/27/0 to X/27/3 for
         // Editoral Linking)
         //
         // Thus is equivalent to this structure:
         //
         //    typedef struct
         //    {
         //       // This is the 'Designation Code' used as index for enhancement packet array.
         //       // (data is Hamming 8/4 protected, data in bits 2,4,6,8)
         //       U8BIT designation_code;
         //
         //       // First four links are FastText navigations, sixth is local index page.
         //       S_EDITORIAL_LINK editorial_link[6];
         //
         //       // Bit 4 determines if packet Y/24 is displayed - this has coloured text for four
         //       // links.
         //       U8BIT link_contol_byte;
         //
         //       U16BIT crc;
         //    }
         //    S_PAGE_EDITORIAL_LINKING_DATA;
         //
         //    typedef struct
         //    {
         //       // Byte-pair used with the magazine index to genereate a page index.
         //       // (data is Hamming 8/4 protected, data in bits 2,4,6,8)
         //       U8BIT units;
         //       U8BIT tens;
         //
         //       // Page sub-code, and some control bits.
         //       // (data is Hamming 8/4 protected, sub-code in 13 of sixteen bits 2,4,6,...,30,32)
         //       // NB - control bits are 'Erase Page', 'Newsflash', and 'Subtitle'
         //       union
         //       {
         //          U32BIT sub_code;
         //          U8BIT magazine_bytes_a[4];  // only magazine_bytes_a[1] and magazine_bytes_a[3]
         //                                      // have content.
         //       };
         //    }
         //    S_EDITORIAL_LINK;


         // To get to the X/27 editorial link data within the Teletext packet, we must skip over the
         // page and magazine number byte pair.

         // The first byte is the designation code, which yelids a 4-bit value.
         if (GetHamming8Byte(data_ptr, &designation_code) == TRUE)
         {
            // We are only interested in X/27/0
            if (designation_code == 0)
            {
               // Copy the editorial page linking data packet into allotted space in relevant
               // magazine.
               memcpy(magazine_array[magazine_number].m_page.editorial_page_linking_data,
                  data_ptr, MAX_COLUMN);

               magazine_array[magazine_number].m_page.is_editorial_page_linking_data_defined = TRUE;
            }
         }
      }
      else if (packet_number == 28)
      {
         // This is either:
         // the X/28/0 Format 1 packet, which is page specific data.
         // (as defined in ETSI EN 300 706 V1.2.1, section 9.4.2 - Packets X/28/0 Format 1)
         // or
         // the X/28/1 packet, which is page specific data.
         // (as defined in ETSI EN 300 706 V1.2.1, section 9.4.4 - Packets X/28/1)
         //
         //    typedef struct
         //    {
         //       (data is Hamming 8/4 protected, data in bits 2,4,6,8)
         //       U8BIT designation_code;
         //
         //       // 13 triplets, each describing different attributes.
         //       // (data is Hamming 24/18 protected)
         //       U8BIT data[39];
         //    }
         //    S_PAGE_BASIC_ENHANCEMENT_DATA;


         // To get to the X/28 page specific data within the Teletext packet, we must skip over the
         // page and magazine number byte pair.

         // The first byte is the designation code, which yelids a 4-bit value.
         if (GetHamming8Byte(data_ptr, &designation_code) == TRUE)
         {
            // We are only interested in X/28/0
            if (designation_code == 0)
            {
               if (GetHammimg24Value(data_ptr + 1, &triplet_value) == TRUE)
               {
                  // We are only interested in the page function of type 'Basic Level 1 Teletext
                  // page (LOP)'
                  if ((triplet_value & 0x0fL) == 0)
                  {
                     // Copy the page specific data packet into allotted space in relevant magazine.
                     memcpy(magazine_array[magazine_number].m_page.specific_data.s_data[0],
                        data_ptr, MAX_COLUMN);

                     magazine_array[magazine_number].m_page.specific_data.is_defined[0] = TRUE;
                  }
               }
            }
            // We are only interested in X/28/1
            else if (designation_code == 1)
            {
               // Copy the page specific_data data packet into allotted space in relevant magazine.
               memcpy(magazine_array[magazine_number].m_page.specific_data.s_data[1], data_ptr, MAX_COLUMN);

               magazine_array[magazine_number].m_page.specific_data.is_defined[1] = TRUE;
            }
         }
      }
      else if (packet_number == 29)
      {
         // This is either:
         // the M/29/0 packet, which is magazine specific_data data.
         // [ this has the same format as the X/28/0 Format 1 packet, which is page specific_data data.]
         // (as defined in ETSI EN 300 706 V1.2.1, section 9.4.2 - Packets X/28/0 Format 1)
         // or
         // the M/29/1 packet, which is magazine specific_data data.
         // [ this has the same format as the X/28/1 packet, which is page specific_data data. ]
         // (as defined in ETSI EN 300 706 V1.2.1, section 9.4.4 - Packets X/28/1)
         //
         //    typedef struct
         //    {
         //       (data is Hamming 8/4 protected, data in bits 2,4,6,8)
         //       U8BIT designation_code;
         //
         //       // 13 triplets, each describing different attributes.
         //       // (data is Hamming 24/18 protected)
         //       U8BIT data[39];
         //    }
         //    S_PAGE_BASIC_ENHANCEMENT_DATA;


         // To get to the M/29 magazine specific_data data within the Teletext packet, we must skip over
         // the page and magazine number byte pair.

         // The first byte is the designation code, which yelids a 4-bit value.
         if (GetHamming8Byte(data_ptr, &designation_code) == TRUE)
         {
            // We are only interested in M/29/0 and M/29/1
            if ((designation_code == 0) ||
                (designation_code == 1))
            {
               if (GetHammimg24Value(data_ptr + 1, &triplet_value) == TRUE)
               {
                  // We are only interested in the page function of type 'Basic Level 1 Teletext
                  // page (LOP)'
                  if ((triplet_value & 0x0fL) == 0)
                  {
                     update_required = FALSE;

                     if (magazine_array[magazine_number].m_specific_data.is_defined[designation_code] == TRUE)
                     {
                        if (memcmp(magazine_array[magazine_number].m_specific_data.s_data[designation_code],
                               data_ptr, MAX_COLUMN))
                        {
                           update_required = TRUE;
                        }
                     }
                     else
                     {
                        update_required = TRUE;
                     }

                     if (update_required == TRUE)
                     {
                        // Copy the page specific_data data packet into allotted space in relevant magazine.
                        memcpy(magazine_array[magazine_number].m_specific_data.s_data[designation_code],
                           data_ptr, MAX_COLUMN);

                        magazine_array[magazine_number].m_specific_data.is_defined[designation_code] = TRUE;

                        if (page_display_info.page_number)
                        {
                           if (GetMagazineIndexFromPageNumber(page_display_info.page_number) ==
                               magazine_number)
                           {
                              /* Wait until page_display_info is available to be used */
                              STB_OSSemaphoreWait(page_free_sem);

                              STB_OSSemaphoreWait(page_display_info.page_semaphore);

                              memcpy(page_display_info.mgzn_data.s_data[designation_code],
                                 data_ptr, MAX_COLUMN);

                              page_display_info.mgzn_data.is_defined[designation_code] = TRUE;

                              page_display_info.pts_valid = pts_valid;
                              if (pts_valid)
                              {
                                 memcpy(page_display_info.page_pts, pts, 5);
                                 EBU_DBG("PageUpdate PTS: 0x%02x%02x%02x%02x%02x", pts[0], pts[1],
                                          pts[2], pts[3], pts[4]);
                              }

                              page_display_info.page_updated = TRUE;

                              STB_OSSemaphoreSignal(page_display_info.page_semaphore);
                           }
                        }
                     }
                  }
               }
            }
         }
      }
      else if ((packet_number == 30) &&
               (magazine_number == 0))
      {
         // This is either:
         // the 8/30 Format 1 and 2 packets, which is page broadcast service data.
         // (as defined in ETSI EN 300 706 V1.2.1, section 9.8 - Broadcast Service Data Packets)
         //
         //    typedef struct s_page_broadcast_service_sub_data_format_1
         //    {
         //       U16BIT network_identification_code;
         //
         //       U8BIT time_offset_code;
         //       U8BIT mjd_time[3]; // Modified Julian Date triplet
         //       U8BIT utc_time[3]; // Universal Time Co-ordinated triplet
         //
         //       U8BIT reserved[4];
         //    }
         //    S_PAGE_BROADCAST_SERVICE_SUB_DATA_FORMAT_1;
         //
         //    typedef struct s_page_broadcast_service_sub_data_format_2
         //    {
         //       U8BIT program_identification_data[13];
         //    }
         //    S_PAGE_BROADCAST_SERVICE_SUB_DATA_FORMAT_2;
         //
         //
         //    typedef struct s_page_broadcast_service_data
         //    {
         //       // This is the 'Designation Code' used as index for enhancement packet array.
         //       // (data is Hamming 8/4 protected, data in bits 2,4,6,8)
         //       U8BIT designation_code;
         //
         //       // Byte-pair used with the magazine index to genereate a page index.
         //       // (data is Hamming 8/4 protected, data in bits 2,4,6,8)
         //       U8BIT units;
         //       U8BIT tens;
         //
         //       // Page sub-code, and some control bits.
         //       // (data is Hamming 8/4 protected, sub-code in 13 of sixteen bits 2,4,6,...,30,32)
         //       // NB - control bits are 'Erase Page', 'Newsflash', and 'Subtitle'
         //       union
         //       {
         //          U32BIT sub_code;
         //          U8BIT magazine_bytes[4];  // only magazine_bytes[1] and magazine_bytes[3] have
         //                                   // magazine index content
         //       };
         //
         //       union
         //       {
         //          S_PAGE_BROADCAST_SERVICE_SUB_DATA_FORMAT_1 data_format_1;
         //          S_PAGE_BROADCAST_SERVICE_SUB_DATA_FORMAT_2 data_format_2;
         //       };
         //
         //       U8BIT status_display[20];
         //    }
         //    S_PAGE_BROADCAST_SERVICE_DATA;

         // To get to the 8/30 page specific_data data within the Teletext packet, we must skip over the
         // page and magazine number byte pair.

         // The first byte is the designation code, which yelids a 4-bit value.
         if (GetHamming8Byte(data_ptr, &designation_code) == TRUE)
         {
            // We are only interested in Format 1 and 2 packets, which are represented by the
            // designation code pairs 0,1 and 2,3 repectively.
            if (designation_code < 4)
            {
               format_number = designation_code >> 1;

               memcpy(broadcast_service_data[format_number], data_ptr, MAX_COLUMN);
               is_broadcast_service_data_defined[format_number] = TRUE;

               if (broadcast_index_page_invocation_required == TRUE)
               {
                  broadcast_index_page_invocation_required = FALSE;
                  RequestIndexPage();
               }
            }
         }
      }
      retval = TRUE;
   }

   FUNCTION_FINISH(PerformPESCollation);

   return(retval);
}

/**
 *

 *
 * @brief   Used to notify the cache mechanism that the user has requested a specific_data
 *                 TeleText page. This is primarily called as a consequence of numerical input of
 *                 the page number, hence the page number passed is a three-digit hexadecimal value.
 *                 As a consequence of a call to this function:
 *                 1) the caching task will determine if the requested page is already cached, and
 *                    if not so, then records the request. When the relevant page is collated from
 *                    PES data supplied in the service stream, it is then cached.
 *                 2) if a delay ensues whilst awating page collation, the display update task will
 *                    be kept busy updating the page number as the carousel loops.
 *                 2) when the page is available in the cache, the display update task is informed
 *                    by means of a call to  NotifyPageDisplayUpdate( ).
 *                This is, of course, dependant on the page number requested being validated first.
 *
 * @param   U16BIT page_number - the three-digit binary-coded decimal index of the Teletext
 *                                      page requested to be displayed.
 * @param   U16BIT page_sub_code - the four-digit binary-coded decimal index of the Teletext
 *                                        sub-page requested to be displayed. In no specific_data page is
 *                                        required, then PAGE_SUB_CODE_UNDEFINED should be used.
 *

 *
 */
static void RequestPage(U16BIT page_number, U16BIT page_sub_code)
{
   BOOLEAN confirm_request;

   FUNCTION_START(RequestPage);

   confirm_request = FALSE;

   if (page_request_info.type != REQUEST_TYPE_EXPLICIT)
   {
      confirm_request = TRUE;
   }

   page_number &= CACHE_PAGE_NUMBER_MASK;

   // Only trigger a requested page notification if the requested page is different from the last
   // one.
   if (page_number != page_request_info.page_number)
   {
      confirm_request = TRUE;
   }

   if (page_sub_code == PAGE_SUB_CODE_UNDEFINED)
   {
      if (page_request_info.page_sub_code != PAGE_SUB_CODE_UNDEFINED)
      {
         confirm_request = TRUE;
      }
   }
   else if ((page_sub_code & 0x3f7f) != page_request_info.page_sub_code)
   {
      page_sub_code &= 0x3f7f;
      confirm_request = TRUE;
   }

   if (confirm_request == TRUE)
   {
      STB_OSSemaphoreWait(page_request_info.semaphore);

      page_request_info.type = REQUEST_TYPE_EXPLICIT;

      page_request_info.page_number = page_number;
      page_request_info.page_sub_code = page_sub_code;

      page_request_info.pending = TRUE;

      STB_OSSemaphoreSignal(page_request_info.semaphore);
   }

   FUNCTION_FINISH(RequestPage);
}

/**
 *

 *
 * @brief   Used to notify the cache mechanism that the user has requested the index
 *                 TeleText page. This may prove to be the global index page (usually 0x100), but
 *                 it can also be a page-related index reference instead.
 *

 *

 *
 */
static void RequestIndexPage(void)
{
   FUNCTION_START(RequestIndexPage);

   if (page_request_info.type != REQUEST_TYPE_INDEX)
   {
      STB_OSSemaphoreWait(page_request_info.semaphore);

      page_request_info.type = REQUEST_TYPE_INDEX;

      page_request_info.pending = TRUE;

      STB_OSSemaphoreSignal(page_request_info.semaphore);
   }

   FUNCTION_FINISH(RequestIndexPage);
}

/**
 *

 *
 * @brief   Macro function used by RequestNextAvailablePage(), RequestPreviousAvailablePage(),
 *                 RequestNextVisitedPage(), RequestPreviousVisitedPage(), RequestNextSubPage(),
 *                 and RequestPreviousSubPage().
 *
 * @param   E_REQUEST_TYPE request_type - can be one of the following values:
 *                                               REQUEST_TYPE_NEXT_PREV_AVAILABLE
 *                                               REQUEST_TYPE_NEXT_PREV_VISITED
 *                                               REQUEST_TYPE_NEXT_PREV_SUB
 *
 * @param   BOOLEAN increment - TRUE if the next page is required, FALSE for the previous.
 *

 *
 */
static void RequestNextPrevious(E_REQUEST_TYPE request_type, BOOLEAN increment)
{
   FUNCTION_START(RequestNextPrevious);

   if (page_request_info.type != request_type)
   {
      // Page request type has changed

      STB_OSSemaphoreWait(page_request_info.semaphore);

      page_request_info.type = request_type;

      // Set the iteration parameter to +1 or -1
      if (increment == TRUE)
      {
         page_request_info.param = 1;
      }
      else
      {
         page_request_info.param = -1;
      }

      page_request_info.pending = TRUE;

      STB_OSSemaphoreSignal(page_request_info.semaphore);
   }
   else
   {
      // Page request type has been repeated

      STB_OSSemaphoreWait(page_request_info.semaphore);

      if (page_request_info.pending == TRUE)
      {
         // If the previous similar request(s) have not been met then increase the iteration
         // parameter accordingly.
         if (increment == TRUE)
         {
            page_request_info.param++;
         }
         else
         {
            page_request_info.param--;
         }
      }
      else
      {
         // If the previous similar request(s) have been met then set the iteration parameter
         // to +1 or -1
         if (increment == TRUE)
         {
            page_request_info.param = 1;
         }
         else
         {
            page_request_info.param = -1;
         }

         page_request_info.pending = TRUE;
      }

      STB_OSSemaphoreSignal(page_request_info.semaphore);
   }

   FUNCTION_FINISH(RequestNextPrevious);
}

/**
 *

 *
 * @brief   Used to notify the cache mechanism that the user has requested the next
 *                 TeleText page to the currently displayed page. This is not always the numerically
 *                 adjacent page number - the caching mechanism is aware of the entire range of
 *                 pages being broadcast on a service carousel (once the carousel has all been
 *                 collated at least once).
 *

 *

 *
 */
static void RequestNextAvailablePage(void)
{
   FUNCTION_START(RequestNextAvailablePage);

   RequestNextPrevious(REQUEST_TYPE_NEXT_PREV_AVAILABLE, TRUE);

   FUNCTION_FINISH(RequestNextAvailablePage);
}

/**
 *

 *
 * @brief   Used to notify the cache mechanism that the user has requested the previous
 *                 TeleText page to the currently displayed page. This is not always the numerically
 *                 adjacent page number - the caching mechanism is aware of the entire range of
 *                 pages being broadcast on a service carousel (once the carousel has all been
 *                 collated at least once).
 *

 *

 *
 */
static void RequestPreviousAvailablePage(void)
{
   FUNCTION_START(RequestPreviousAvailablePage);

   RequestNextPrevious(REQUEST_TYPE_NEXT_PREV_AVAILABLE, FALSE);

   FUNCTION_FINISH(RequestPreviousAvailablePage);
}

/**
 *

 *
 * @brief   Used to notify the cache mechanism that the user has requested the next page
 *                 within the visited page history stack. The caching mechanism records a rangle of
 *                 visited pages, and after call(s) to RequestPreviousVisitedPage() this function
 *                 can be called to navigate back to the most recent page in the history.
 *

 *

 *
 */
static void RequestNextVisitedPage(void)
{
   FUNCTION_START(RequestNextVisitedPage);

   RequestNextPrevious(REQUEST_TYPE_NEXT_PREV_VISITED, TRUE);

   FUNCTION_FINISH(RequestNextVisitedPage);
}

/**
 *

 *
 * @brief   Used to notify the cache mechanism that the user has requested the previous page
 *                 within the visited page history stack. The caching mechanism records a rangle of
 *                 visited pages, and this function can be called to navigate to the previously
 *                 displayed pages.
 *

 *

 *
 */
static void RequestPreviousVisitedPage(void)
{
   FUNCTION_START(RequestPreviousVisitedPage);

   RequestNextPrevious(REQUEST_TYPE_NEXT_PREV_VISITED, FALSE);

   FUNCTION_FINISH(RequestPreviousVisitedPage);
}

/**
 *

 *
 * @brief   Used to notify the cache mechanism that the user has requested the an editorial
 *                 linked page (FLOF navigation element).
 *
 * @param   U16BIT index - a link index, in the range of 0 to (MAX_EDITORIAL_LINKS-1)
 *

 *
 */
static void RequestEditorialLinkPage(U16BIT index)
{
   FUNCTION_START(RequestEditorialLinkPage);

   if (index < MAX_EDITORIAL_LINKS)
   {
      if (page_request_info.type != REQUEST_TYPE_EDITORIAL_LINK)
      {
         STB_OSSemaphoreWait(page_request_info.semaphore);

         page_request_info.type = REQUEST_TYPE_EDITORIAL_LINK;

         page_request_info.param = index;

         page_request_info.pending = TRUE;

         STB_OSSemaphoreSignal(page_request_info.semaphore);
      }
      else
      {
         STB_OSSemaphoreWait(page_request_info.semaphore);

         if (page_request_info.param != index)
         {
            page_request_info.param = index;

            page_request_info.pending = TRUE;
         }

         STB_OSSemaphoreSignal(page_request_info.semaphore);
      }
   }

   FUNCTION_FINISH(RequestEditorialLinkPage);
}

/**
 *

 *
 * @brief   Used to notify the cache mechanism that the user has requested the next sub-page
 *                 for the presently displayed Teletext page. The caching mechanism will ignore this
 *                 request if the displayed page is not held.
 *                 NB - this function is usually called in conjunction with driver functionality
 *                      associated with processing the EBUTT_EVENT_NEXTSUBPAGE event sent to
 *                      the STB_EBUTT_NotifyEvent( ) function from the STB application.
 *

 *

 *
 */
static void RequestNextSubPage(void)
{
   FUNCTION_START(RequestNextSubPage);

   RequestNextPrevious(REQUEST_TYPE_NEXT_PREV_SUB, TRUE);

   FUNCTION_FINISH(RequestNextSubPage);
}

/**
 *

 *
 * @brief   Used to notify the cache mechanism that the user has requested the previous
 *                 sub-page for the presently displayed Teletext page. The caching mechanism will
 *                 ignore this request if the displayed page is not held.
 *                 NB - this function is usually called in conjunction with driver functionality
 *                      associated with processing the EBUTT_EVENT_PREVIOUSSUBPAGE event sent to
 *                      the STB_EBUTT_NotifyEvent( ) function from the STB application.
 *

 *

 *
 */
static void RequestPreviousSubPage(void)
{
   FUNCTION_START(RequestPreviousSubPage);

   RequestNextPrevious(REQUEST_TYPE_NEXT_PREV_SUB, FALSE);

   FUNCTION_FINISH(RequestPreviousSubPage);
}

/**
 *

 *
 * @brief   Returns the maximum depthe that the driver will cache FLOF (editorial links)
 *                 for the current page. This determines how deep the link tree can become, and
 *                 hence how much memory resource is used.
 *                 As the depth is increased, the potential memory resources used will increase
 *                 exponentially, so care in adjusting this figure is recommended.
 *

 *
 * @return   U16BIT - the number of teirs of editorial links what will be cached in memory.
 *
 */
static U16BIT GetMaximumEditorialLinkTiers(void)
{
   U16BIT retval;

   FUNCTION_START(GetMaximumEditorialLinkTiers);

   if (caching_method == EBUTT_CACHING_METHOD_NAVIGATION)
   {
      // For this caching method, we only cache the editorially-linked pages of the currently
      // displayed page.
      retval = 1;
   }
   else
   {
      // For the EBUTT_CACHING_METHOD_NAVIGATION_TREE and EBUTT_CACHING_METHOD_ALL caching method,
      // we cache the up to a nominally defined depth of pages from of the currently displayed page.
      retval = MAX_EDITORIAL_LINK_TIERS;
   }

   FUNCTION_FINISH(GetMaximumEditorialLinkTiers);

   return(retval);
}

/**
 *

 *
 *
 * @brief   Finds an editorial link cache request item of the corresponding page number.
 *                 This item contains the FLOF links for the Teletext page.
 *                 NB - This is a dedicated recursive sub-funtion of FindEditorialLinkPageRequest.
 *
 * @param   S_EDITORIAL_LINK_CACHE_REQUEST* base_ptr - the base item for the search.
 *                                                            Usually this is the root of an item
 *                                                            tree.
 * @param   S_EDITORIAL_LINK_CACHE_REQUEST** ptr - a double-pointer in which a matching
 *                                                        reference is returned.
 *                                                        This pointer is optional,and can be set to
 *                                                        NULL.
 * @param   S_EDITORIAL_LINK_CACHE_REQUEST** ptr - a double-pointer in which the parent of a
 *                                                        matching reference is returned.
 *                                                        This pointer is optional,and can be set to
 *NULL.
 * @param   U16BIT page_number - the page number to be searched for.
 *
 *
 * @return   TRUE if matching item found (and optionally passed back), FALSE otherwise.
 *
 */
static BOOLEAN FindEditorialLinkRecursive(S_EDITORIAL_LINK_CACHE_REQUEST *base_ptr,
   S_EDITORIAL_LINK_CACHE_REQUEST **ptr,
   S_EDITORIAL_LINK_CACHE_REQUEST **parent_ptr,
   U16BIT page_number)
{
   U16BIT link_index;
   BOOLEAN retval;
   S_EDITORIAL_LINK_CACHE_REQUEST *found_ptr;
   S_EDITORIAL_LINK_CACHE_REQUEST_SUB_PAGE *sub_page_ptr;

   FUNCTION_START(FindEditorialLinkRecursive);

   retval = FALSE;

   // Only perform search if the root item is defined!
   if (base_ptr != NULL)
   {
      // If the root item has a matching page number...
      if (base_ptr->page_number == page_number)
      {
         // ...then optionally return it's reference
         if (ptr != NULL)
         {
            *ptr = base_ptr;
         }
         retval = TRUE;
      }
      else
      {
         // We must now traverse the page's editorial link data looking for child page references.
         sub_page_ptr = base_ptr->sub_page_list;
         // For each sub-page...
         while (sub_page_ptr != NULL)
         {
            // ...if there are any editorial links...
            if (sub_page_ptr->is_parsed == TRUE)
            {
               // ... then for each link...
               for (link_index = 0; link_index < MAX_EDITORIAL_LINKS; link_index++)
               {
                  // ... find the page item with the matching page number in the entire request tree.
                  // At this point, the link's sub-code is irrelevant.
                  // NB - this is a recursive call to this fuction!
                  if (FindEditorialLinkRecursive(
                         (S_EDITORIAL_LINK_CACHE_REQUEST *)sub_page_ptr->link[link_index],
                         &found_ptr, parent_ptr, page_number) == TRUE)
                  {
                     // If we have found a page item with a matching page number then optionally
                     // return it's reference (and it's parent as well)
                     if (ptr != NULL)
                     {
                        *ptr = found_ptr;
                     }
                     if (parent_ptr != NULL)
                     {
                        if (*parent_ptr == NULL)
                        {
                           *parent_ptr = base_ptr;
                        }
                     }

                     // Changer eturn value to denote a succesful search.
                     retval = TRUE;
                     break;
                  }
               }

               // If match found, then stop looking at other links of this sub-page.
               if (retval == TRUE)
               {
                  break;
               }
            }

            // If we are here then no page item exists in the item tree which matches the requuied
            // page number, so we must procees with checks of editorial links of other sub-pages.
            sub_page_ptr = sub_page_ptr->next_ptr;
         }
      }
   }

   FUNCTION_FINISH(FindEditorialLinkRecursive);

   return(retval);
}

/**
 *

 *
 *
 * @brief   Finds an editorial link cache request item of the corresponding page number.
 *                 This item contains the FLOF links for the Teletext page.
 *
 * @param   S_EDITORIAL_LINK_CACHE_REQUEST* base_ptr - the base item for the search.
 *                                                            Usually this is the root of an item
 *                                                            tree.
 * @param   S_EDITORIAL_LINK_CACHE_REQUEST** ptr - a double-pointer in which a matching
 *                                                        reference is returned.
 *                                                        This pointer is optional,and can be set to
 *                                                        NULL.
 * @param   S_EDITORIAL_LINK_CACHE_REQUEST** ptr - a double-pointer in which the parent of a
 *                                                        matching reference is returned.
 *                                                        This pointer is optional,and can be set to
 *                                                        NULL.
 * @param   U16BIT page_number - the page number to be searched for.
 *
 *
 * @return   TRUE if matching item found (and optionally passed back), FALSE otherwise.
 *
 */
static BOOLEAN FindEditorialLinkPageRequest(S_EDITORIAL_LINK_CACHE_REQUEST *base_ptr,
   S_EDITORIAL_LINK_CACHE_REQUEST **ptr,
   S_EDITORIAL_LINK_CACHE_REQUEST **parent_ptr,
   U16BIT page_number)
{
   BOOLEAN retval;

   FUNCTION_START(FindEditorialLinkPageRequest);

   retval = FALSE;

   // Only perform search if the root item is defined!
   if (base_ptr != NULL)
   {
      if (parent_ptr != NULL)
      {
         *parent_ptr = NULL;
      }

      // If the root item has a matching page number...
      if (base_ptr->page_number == page_number)
      {
         // ...then optionally return it's reference
         if (ptr != NULL)
         {
            *ptr = base_ptr;
         }
         retval = TRUE;
      }
      else
      {
         // Perform a recursive serch for a page request with a matching page number down the entire
         // request tree.
         if (FindEditorialLinkRecursive(base_ptr, ptr, parent_ptr, page_number) == TRUE)
         {
            if (parent_ptr != NULL)
            {
               if (*parent_ptr == NULL)
               {
                  *parent_ptr = base_ptr;
               }
            }

            retval = TRUE;
         }
      }
   }

   FUNCTION_FINISH(FindEditorialLinkPageRequest);

   return(retval);
}

/**
 *

 *
 * @brief   Maintains the editorial link cache request item tree by either fudning a tree
 *                 item matching the passed page number, or creating a new cache request item if the
 *                 passed page number is relevant.
 *                 The returned item contains the FLOF links for the Teletext page.
 *                 This function is called directly from the caching function CheckPageCaching().
 *
 * @param   S_EDITORIAL_LINK_CACHE_REQUEST* base_ptr - the base item for the search.
 *                                                            Usually this is the root of an item
 *                                                            tree.
 * @param   S_EDITORIAL_LINK_CACHE_REQUEST** ptr - a double-pointer in which a matching
 *                                                        reference is returned.
 *                                                        This pointer is optional,and can be set to
 *                                                        NULL.
 * @param   U16BIT page_number - the page number to be searched for (or added).
 *
 *
 * @return   TRUE if matching item found or correctly added (and optionally passed back),
 *                 FALSE otherwise.
 *
 */
static BOOLEAN GetEditorialLink(S_EDITORIAL_LINK_CACHE_REQUEST *base_ptr,
   S_EDITORIAL_LINK_CACHE_REQUEST **ptr, U16BIT page_number)
{
   U16BIT magazine_number, link_index;
   BOOLEAN retval;
   S_EDITORIAL_LINK_CACHE_REQUEST *found_ptr;
   S_EDITORIAL_LINK_CACHE_REQUEST_SUB_PAGE *sub_page_ptr;

   FUNCTION_START(GetEditorialLink);

   retval = FALSE;

   // Only perform search if the root item is defined!
   if (base_ptr != NULL)
   {
      // If the root item has a matching page number...
      if (base_ptr->page_number == page_number)
      {
         // ...then optionally return it's reference
         if (ptr != NULL)
         {
            *ptr = base_ptr;
         }
         retval = TRUE;
      }
      else
      {
         magazine_number = GetMagazineIndexFromPageNumber(page_number);

         // We must now traverse the page's editorial link data looking for child page references.
         sub_page_ptr = base_ptr->sub_page_list;
         // For each sub-page...
         while (sub_page_ptr != NULL)
         {
            // ...if there are any editorial links...
            if (sub_page_ptr->is_parsed == TRUE)
            {
               // ... then for each link...
               for (link_index = 0; link_index < MAX_EDITORIAL_LINKS; link_index++)
               {
                  // ... find the page item with the matching page number in the entire request tree.
                  // At this point, the link's sub-code is irrelevant.
                  // NB - this is a recursive call to this fuction!
                  if (GetEditorialLink((S_EDITORIAL_LINK_CACHE_REQUEST *)sub_page_ptr->link[link_index],
                         &found_ptr, page_number) == TRUE)
                  {
                     // If we have found a page item with a matching page number then optionally
                     // return it's reference
                     if (ptr != NULL)
                     {
                        *ptr = found_ptr;
                     }

                     // Change return value to denote a succesful search.
                     retval = TRUE;
                     break;
                  }

                  if (ptr != NULL)
                  {
                     // If no match was found in the tree below, then if any of the page links
                     // explicitly reference this page number, then a new editorial cache request
                     // entity will be added to the request tree.
                     if ((sub_page_ptr->link_page_number[link_index] == page_number) &&
                         (sub_page_ptr->link[link_index] == NULL))
                     {
                        if (base_ptr->tier <= GetMaximumEditorialLinkTiers())
                        {
                           *ptr = (S_EDITORIAL_LINK_CACHE_REQUEST *)GetMemory(
                                 sizeof(S_EDITORIAL_LINK_CACHE_REQUEST));
                           while (*ptr == NULL)
                           {
                              if (RemoveUnrequestedCacheContent(magazine_number, FALSE) == FALSE)
                              {
                                 break;
                              }

                              *ptr = (S_EDITORIAL_LINK_CACHE_REQUEST *)GetMemory(
                                    sizeof(S_EDITORIAL_LINK_CACHE_REQUEST));
                           }

                           if (*ptr != NULL)
                           {
                              (*ptr)->page_number = sub_page_ptr->link_page_number[link_index];
                              (*ptr)->tier = base_ptr->tier + 1;

                              (*ptr)->sub_page_list = NULL;

                              sub_page_ptr->link[link_index] = (void *)*ptr;

                              retval = TRUE;
                              break;
                           }
                        }
                     }
                  }
               }

               // If match found, then stop looking at other links of this sub-page.
               if (retval == TRUE)
               {
                  break;
               }
            }

            // If we are here then no page item exists in the item tree which matches the requuied
            // page number, so we must procees with checks of editorial links of other sub-pages.
            sub_page_ptr = sub_page_ptr->next_ptr;
         }
      }
   }

   FUNCTION_FINISH(GetEditorialLink);

   return(retval);
}

/**
 *

 *
 * @brief   Finds an editorial link cache request item of the corresponding page number
 *                 and removes it, clearing the pointer reference supplied.
 *                 This funtion does NOT destroy any cache request content or free memory - it
 *                 merely extracts the reference to the supplied cache request from the tree
 *                 supplied.
 *
 * @param   S_EDITORIAL_LINK_CACHE_REQUEST** base_ptr - the base item for the search.
 *                                                            Usually this is the root of an item
 *                                                            tree.
 * @param   S_EDITORIAL_LINK_CACHE_REQUEST* ptr - the page item to be searched for.
 *
 *
 * @return   TRUE if matching item found (and removed from the tree), FALSE otherwise.
 *
 */
static BOOLEAN RemoveEditorialLinkPageRequest(S_EDITORIAL_LINK_CACHE_REQUEST **base_ptr,
   S_EDITORIAL_LINK_CACHE_REQUEST *ptr)
{
   U16BIT link_index;
   BOOLEAN retval;
   S_EDITORIAL_LINK_CACHE_REQUEST_SUB_PAGE *sub_page_ptr;

   FUNCTION_START(RemoveEditorialLinkPageRequest);

   retval = FALSE;

   // Only perform search if the root item is defined!
   if (*base_ptr != NULL)
   {
      // If the root item is the matching page item...
      if (*base_ptr == ptr)
      {
         // Then simply remove it from the tree by resetting the item tree.
         *base_ptr = NULL;
         retval = TRUE;
      }
      else
      {
         // We must now traverse the page's editorial link data looking for child page references.
         sub_page_ptr = (*base_ptr)->sub_page_list;
         // For each sub-page...
         while (sub_page_ptr != NULL)
         {
            // ...if there are any editorial links...
            if (sub_page_ptr->is_parsed == TRUE)
            {
               // ... then for each link...
               for (link_index = 0; link_index < MAX_EDITORIAL_LINKS; link_index++)
               {
                  // ... that is defined...
                  if (sub_page_ptr->link[link_index] != NULL)
                  {
                     // ... find the supplied page item in the entire request tree.
                     // At this point, the link's sub-code is irrelevant.
                     // NB - this is a recursive call to this fuction!
                     if (RemoveEditorialLinkPageRequest(
                            (S_EDITORIAL_LINK_CACHE_REQUEST **)&sub_page_ptr->link[link_index],
                            ptr) == TRUE)
                     {
                        // Change return value to denote a succesful search and removal.
                        retval = TRUE;
                        break;
                     }
                  }
               }

               // If match found, then stop looking at other links of this sub-page.
               if (retval == TRUE)
               {
                  break;
               }
            }

            // If we are here then no page item exists in the item tree which matches the requuied
            // page item, so we must procees with checks of editorial links of other sub-pages.
            sub_page_ptr = sub_page_ptr->next_ptr;
         }
      }
   }

   FUNCTION_FINISH(RemoveEditorialLinkPageRequest);

   return(retval);
}

/**
 *

 *
 * @brief   Populates arrays of page numbers and page sub-codes representing editorial links
 *                 for a given sub-page.
 *
 * @param   S_CACHE_SUBPAGE* sub_page_ptr - the cached sub-page whose teletext data will be
 *                                                 anlaysed to determine it's editorial links.
 * @param   U16BIT* page_number_link_array - a pointer to an array into which is returned
 *                                                  a block of editorial link page numbers.
 * @param   U16BIT* page_sub_code_link_array - a pointer to an array into which is returned
 *                                                  a block of editorial link page sub-codes.
 * @param   U16BIT max_num_links - the maximum number of U16BIT elements in the two arrays.
 * @param   U16BIT magazine_number - the magazine number of the parent cached page to which
 *                                          this sub-page belongs.
 *
 * @return   TRUE if any editorial links were correctly decoded, FALSE otherwise.
 *
 */
static BOOLEAN GetSubPageEditorialLinks(S_CACHE_SUBPAGE *sub_page_ptr,
   U16BIT *page_number_link_array,
   U16BIT *page_sub_code_link_array,
   U16BIT max_num_links, U16BIT magazine_number)
{
   U16BIT link_index;
   BOOLEAN retval;
   S_PAGE_REFERENCE_DATA page_reference_data;
   U8BIT *link_ptr;

   FUNCTION_START(GetSubPageEditorialLinks);

   retval = FALSE;

   // Check of collated teletext data of the sub-page actually contains editorial link content.
   if (sub_page_ptr->collated_data.is_editorial_page_linking_data_defined == TRUE)
   {
      // Sets the link data block pointer up to just after the designation codee byte.
      link_ptr = &sub_page_ptr->collated_data.editorial_page_linking_data[1];

      // For each editorial page link required in the arrays...
      for (link_index = 0; link_index < max_num_links; link_index++)
      {
         // ...if we can decode the link data block...
         if (GetPageReference(link_ptr, &page_reference_data, PAGE_REFERENCE_TYPE_EDITORIAL_LINK,
                magazine_number) == TRUE)
         {
            // ...then populate the array items.
            *page_number_link_array = page_reference_data.page_number;
            *page_sub_code_link_array = page_reference_data.page_sub_code;

            retval = TRUE;
         }
         else
         {
            // ...otherwise mark the array item as undefined.
            *page_number_link_array = 0;
            *page_sub_code_link_array = 0;
         }

         // Advance our array pointers.
         page_number_link_array++;
         page_sub_code_link_array++;
         // Move the link data block pointer to the next editorial link
         link_ptr += 6;
      }
   }

   FUNCTION_FINISH(GetSubPageEditorialLinks);

   return(retval);
}

/**
 *

 *

 *
 * @param   S_EDITORIAL_LINK_CACHE_REQUEST** ptr - a pinter in which the newly created item
 *                                                        will be returned.
 * @param   U16BIT page_number -
 * @param   U16BIT sub_code - the page number / sub-code to be searched for.
 *
 *
 * @return   TRUE if matching item is found (or created), FALSE otherwise.
 *
 */
static BOOLEAN CreateEditorialLinkPageRequest(S_EDITORIAL_LINK_CACHE_REQUEST **ptr,
   U16BIT page_number, U16BIT sub_code)
{
   U16BIT magazine_number;
   BOOLEAN retval;
   BOOLEAN is_found;
   S_CACHE_PAGE *page_ptr;
   S_CACHE_SUBPAGE *subpage_ptr;
   S_EDITORIAL_LINK_CACHE_REQUEST_SUB_PAGE *request_sub_page_ptr;

   FUNCTION_START(CreateEditorialLinkPageRequest);

   retval = FALSE;

   magazine_number = GetMagazineIndexFromPageNumber(page_number);

   *ptr = (S_EDITORIAL_LINK_CACHE_REQUEST *)GetMemory(sizeof(S_EDITORIAL_LINK_CACHE_REQUEST));
   while (*ptr == NULL)
   {
      if (RemoveUnrequestedCacheContent(magazine_number, FALSE) == FALSE)
      {
         break;
      }

      *ptr = (S_EDITORIAL_LINK_CACHE_REQUEST *)GetMemory(sizeof(S_EDITORIAL_LINK_CACHE_REQUEST));
   }

   if (*ptr != NULL)
   {
      (*ptr)->page_number = page_number;
      (*ptr)->tier = 0;
      (*ptr)->sub_page_list = NULL;

      //... and locate mathcing page in cache.
      STB_OSMutexLock( magazine_array[magazine_number].m_mutex );
      page_ptr = magazine_array[magazine_number].m_cache_page_ptr;
      while (page_ptr != NULL)
      {
         if ((page_ptr->page_number & CACHE_PAGE_NUMBER_MASK) == page_number)
         {
            subpage_ptr = page_ptr->subpage_ptr;
            is_found = FALSE;

            while (subpage_ptr != NULL)
            {
               if (subpage_ptr->sub_code == sub_code)
               {
                  is_found = TRUE;
               }

               request_sub_page_ptr = (S_EDITORIAL_LINK_CACHE_REQUEST_SUB_PAGE *)GetMemory(
                     sizeof(S_EDITORIAL_LINK_CACHE_REQUEST_SUB_PAGE));
               while (request_sub_page_ptr == NULL)
               {
                  if (RemoveUnrequestedCacheContent(magazine_number, TRUE) == FALSE)
                  {
                     break;
                  }

                  request_sub_page_ptr = (S_EDITORIAL_LINK_CACHE_REQUEST_SUB_PAGE *)GetMemory(
                        sizeof(S_EDITORIAL_LINK_CACHE_REQUEST_SUB_PAGE));
               }

               if (request_sub_page_ptr != NULL)
               {
                  if (GetSubPageEditorialLinks(subpage_ptr, request_sub_page_ptr->link_page_number,
                         request_sub_page_ptr->link_sub_code,
                         MAX_EDITORIAL_LINKS, magazine_number) == TRUE)
                  {
                     request_sub_page_ptr->sub_code = subpage_ptr->sub_code;
                     memset(request_sub_page_ptr->link, 0, sizeof(void *) * MAX_EDITORIAL_LINKS);
                     request_sub_page_ptr->is_parsed = TRUE;
                     request_sub_page_ptr->next_ptr = (*ptr)->sub_page_list;
                     (*ptr)->sub_page_list = request_sub_page_ptr;
                  }
                  else
                  {
                     STB_FreeMemory(request_sub_page_ptr);
                  }
               }

               subpage_ptr = subpage_ptr->next_ptr;
            }

            if ((is_found == FALSE) &&
                (sub_code != PAGE_SUB_CODE_UNDEFINED))
            {
               request_sub_page_ptr = (S_EDITORIAL_LINK_CACHE_REQUEST_SUB_PAGE *)GetMemory(
                     sizeof(S_EDITORIAL_LINK_CACHE_REQUEST_SUB_PAGE));
               while (request_sub_page_ptr == NULL)
               {
                  if (RemoveUnrequestedCacheContent(magazine_number, TRUE) == FALSE)
                  {
                     break;
                  }

                  request_sub_page_ptr = (S_EDITORIAL_LINK_CACHE_REQUEST_SUB_PAGE *)GetMemory(
                        sizeof(S_EDITORIAL_LINK_CACHE_REQUEST_SUB_PAGE));
               }

               if (request_sub_page_ptr != NULL)
               {
                  request_sub_page_ptr->sub_code = sub_code;
                  request_sub_page_ptr->is_parsed = FALSE;
                  request_sub_page_ptr->next_ptr = (*ptr)->sub_page_list;
                  (*ptr)->sub_page_list = request_sub_page_ptr;
               }
            }

            break;
         }

         page_ptr = page_ptr->next_ptr;
      }
      STB_OSMutexUnlock( magazine_array[magazine_number].m_mutex );


      retval = TRUE;
   }

   FUNCTION_FINISH(CreateEditorialLinkPageRequest);

   return(retval);
}

/**
 *

 *

 *
 * @param   S_EDITORIAL_LINK_CACHE_REQUEST* ptr -
 * @param   U16BIT tier -
 *

 *
 */
static void ReTierEditorialLinkPageRequests(S_EDITORIAL_LINK_CACHE_REQUEST *ptr, U16BIT tier)
{
   U16BIT link_index;
   S_EDITORIAL_LINK_CACHE_REQUEST_SUB_PAGE *sub_page_ptr;

   FUNCTION_START(ReTierEditorialLinkPageRequests);

   if (ptr != NULL)
   {
      tier++;
      ptr->tier = tier;

      sub_page_ptr = ptr->sub_page_list;
      while (sub_page_ptr != NULL)
      {
         if (sub_page_ptr->is_parsed == TRUE)
         {
            for (link_index = 0; link_index < MAX_EDITORIAL_LINKS; link_index++)
            {
               if (sub_page_ptr->link[link_index] != NULL)
               {
                  ReTierEditorialLinkPageRequests((S_EDITORIAL_LINK_CACHE_REQUEST *)sub_page_ptr->link[link_index], tier);
               }
            }
         }

         sub_page_ptr = sub_page_ptr->next_ptr;
      }
   }

   FUNCTION_FINISH(ReTierEditorialLinkPageRequests);
}

/**
 *

 *

 *
 * @param   S_EDITORIAL_LINK_CACHE_REQUEST** base_ptr - the base item for the deletion
 *                                                             Usually this is the root of an item
 *                                                             tree.
 *

 *
 */
static void KillEditorialLinkPageRequests(S_EDITORIAL_LINK_CACHE_REQUEST **base_ptr)
{
   U16BIT link_index;
   S_EDITORIAL_LINK_CACHE_REQUEST_SUB_PAGE *sub_page_ptr;
   S_EDITORIAL_LINK_CACHE_REQUEST_SUB_PAGE *next_sub_page_ptr;

   FUNCTION_START(KillEditorialLinkPageRequests);

   if (base_ptr != NULL)
   {
      if (*base_ptr != NULL)
      {
         sub_page_ptr = (*base_ptr)->sub_page_list;
         while (sub_page_ptr != NULL)
         {
            if (sub_page_ptr->is_parsed == TRUE)
            {
               for (link_index = 0; link_index < MAX_EDITORIAL_LINKS; link_index++)
               {
                  KillEditorialLinkPageRequests((S_EDITORIAL_LINK_CACHE_REQUEST **)&sub_page_ptr->link[link_index]);
               }
            }

            next_sub_page_ptr = sub_page_ptr->next_ptr;

            STB_FreeMemory(sub_page_ptr);

            sub_page_ptr = next_sub_page_ptr;
         }

         STB_FreeMemory(*base_ptr);

         *base_ptr = NULL;
      }
   }

   FUNCTION_FINISH(KillEditorialLinkPageRequests);
}

/**
 *

 *

 *
 * @param   S_EDITORIAL_LINK_CACHE_REQUEST* ptr -
 * @param   U16BIT tier -
 *

 *
 */
static void UpdateEditorialLinkPageRequestRecursive(S_EDITORIAL_LINK_CACHE_REQUEST *ptr,
   U16BIT tier)
{
   U16BIT link_index;
   S_EDITORIAL_LINK_CACHE_REQUEST_SUB_PAGE *sub_page_ptr;
   S_EDITORIAL_LINK_CACHE_REQUEST *other_ptr;
   S_EDITORIAL_LINK_CACHE_REQUEST *parent_ptr;

   FUNCTION_START(UpdateEditorialLinkPageRequestRecursive);

   tier++;
   ptr->tier = tier;

   RemoveEditorialLinkPageRequest(&editorial_link_request_root, ptr);

   sub_page_ptr = ptr->sub_page_list;
   while (sub_page_ptr != NULL)
   {
      if (sub_page_ptr->is_parsed == TRUE)
      {
         for (link_index = 0; link_index < MAX_EDITORIAL_LINKS; link_index++)
         {
            if (sub_page_ptr->link[link_index] == NULL)
            {
               // Attempt to set this link pointer with equivalent editorial link request in
               // original list.
               // NB - the relevant link pointer will be updated if the editorial link is found.
               if (FindEditorialLinkPageRequest(editorial_link_request_root,
                      (S_EDITORIAL_LINK_CACHE_REQUEST **)&sub_page_ptr->link[link_index],
                      NULL, sub_page_ptr->link_page_number[link_index]) == TRUE)
               {
                  // As we are inserting an editorial link in the new tree, we will have to adjust
                  // it's tier value (which also goes for all it's child links)
                  ReTierEditorialLinkPageRequests(
                     (S_EDITORIAL_LINK_CACHE_REQUEST *)sub_page_ptr->link[link_index],
                     tier);

                  // If found, then call function recursively.
                  // NB - The first thing that will happen is the removal of the newly copied link
                  //      from the original list!
                  UpdateEditorialLinkPageRequestRecursive(
                     (S_EDITORIAL_LINK_CACHE_REQUEST *)sub_page_ptr->link[link_index],
                     tier);
               }
               // Attempt to find the equivalent editorial link request in our new list.
               else if (FindEditorialLinkPageRequest(temp_editorial_link_request_root, &other_ptr,
                           &parent_ptr,
                           sub_page_ptr->link_page_number[link_index]) == TRUE)
               {
                  // If found then check if it is placed at a higher tier postion.
                  if (parent_ptr != NULL)
                  {
                     if (parent_ptr->tier > tier)
                     {
                        // It's at a higher tier position, so move it to this new location, and
                        // recursively adjust it's child links.
                        RemoveEditorialLinkPageRequest(&parent_ptr, other_ptr);

                        // As we are inserting it in a new postion, we will have to adjust it's tier
                        // value (which also goes for all it's child links)
                        ReTierEditorialLinkPageRequests(other_ptr, tier);

                        sub_page_ptr->link[link_index] = other_ptr;

                        UpdateEditorialLinkPageRequestRecursive(
                           (S_EDITORIAL_LINK_CACHE_REQUEST *)sub_page_ptr->link[link_index],
                           tier);
                     }
                  }
                  else
                  {
                     sub_page_ptr->link[link_index] = NULL;
                  }
               }
               else
               {
                  sub_page_ptr->link[link_index] = NULL;
               }
            }
            else
            {
               UpdateEditorialLinkPageRequestRecursive(
                  (S_EDITORIAL_LINK_CACHE_REQUEST *)sub_page_ptr->link[link_index],
                  tier);
            }
         }
      }

      sub_page_ptr = sub_page_ptr->next_ptr;
   }

   FUNCTION_FINISH(UpdateEditorialLinkPageRequestRecursive);
}

/**
 *

 *

 *
 * @param   U16BIT page_number -
 * @param   U16BIT sub_code -
 *

 *
 */
static void UpdateEditorialLinkPageRequest(U16BIT page_number, U16BIT sub_code)
{
   S_EDITORIAL_LINK_CACHE_REQUEST *ptr;

   FUNCTION_START(UpdateEditorialLinkPageRequest);

   // Attempt to find cache of editorial page links associated witht the supplied page number.
   if (FindEditorialLinkPageRequest(editorial_link_request_root, &ptr, NULL, page_number) == TRUE)
   {
      // If found, and already at root of tree, then nothin needs to be done.
      if (ptr == editorial_link_request_root)
      {
         ptr = NULL;
      }
   }
   else
   {
      // If not found, then create new request record for this page number
      if (CreateEditorialLinkPageRequest(&ptr, page_number, sub_code) == FALSE)
      {
         ptr = NULL;
      }
   }

   // If we have a non-root cache request for the supplied page number...
   if (ptr != NULL)
   {
      // Then set as the root of our temporary request tree
      temp_editorial_link_request_root = ptr;

      // and recursivley populate this temporary tree with items from the first that
      // are required.
      UpdateEditorialLinkPageRequestRecursive(ptr, 0);

      // Now kill remianing items in original tree
      KillEditorialLinkPageRequests(&editorial_link_request_root);

      // And replace the original tree with the newly collated temporary tree
      editorial_link_request_root = temp_editorial_link_request_root;
      temp_editorial_link_request_root = NULL;
   }

   FUNCTION_FINISH(UpdateEditorialLinkPageRequest);
}

static void SetPageFlag( U16BIT page_num, U16BIT flag )
{
   S_CACHE_PAGE *page_ptr;
   S_MAGAZINE_INFO *mgzn_ptr;

   mgzn_ptr = magazine_array + GetMagazineIndexFromPageNumber(page_num);
   STB_OSMutexLock( mgzn_ptr->m_mutex );
   page_ptr = mgzn_ptr->m_cache_page_ptr;
   while (page_ptr != NULL)
   {
      if ((page_ptr->page_number & CACHE_PAGE_NUMBER_MASK) == page_num)
      {
         page_ptr->page_number |= flag;
         break;
      }
      page_ptr = page_ptr->next_ptr;
   }
   STB_OSMutexUnlock( mgzn_ptr->m_mutex );
}

static void UnsetPageFlag( U16BIT page_num, U16BIT flag )
{
   S_CACHE_PAGE *page_ptr;
   S_MAGAZINE_INFO *mgzn_ptr;

   mgzn_ptr = magazine_array + GetMagazineIndexFromPageNumber(page_num);
   STB_OSMutexLock( mgzn_ptr->m_mutex );
   page_ptr = mgzn_ptr->m_cache_page_ptr;
   while (page_ptr != NULL)
   {
      if ((page_ptr->page_number & CACHE_PAGE_NUMBER_MASK) == page_num)
      {
         page_ptr->page_number &= ~flag;
         break;
      }
      page_ptr = page_ptr->next_ptr;
   }
   STB_OSMutexUnlock( mgzn_ptr->m_mutex );
}

/**
 *

 *

 *
 * @param   S_EDITORIAL_LINK_CACHE_REQUEST* ptr -
 *

 *
 */
static void FlagEditorialLinkPageRequests(S_EDITORIAL_LINK_CACHE_REQUEST *ptr)
{
   U16BIT link_index;
   S_EDITORIAL_LINK_CACHE_REQUEST_SUB_PAGE *sub_page_ptr;

   FUNCTION_START(FlagEditorialLinkPageRequests);

   if (ptr != NULL)
   {
      SetPageFlag( ptr->page_number, CACHE_PAGE_FLOF_REQUESTED );

      sub_page_ptr = ptr->sub_page_list;
      while (sub_page_ptr != NULL)
      {
         if (sub_page_ptr->is_parsed == TRUE)
         {
            for (link_index = 0; link_index < MAX_EDITORIAL_LINKS; link_index++)
            {
               FlagEditorialLinkPageRequests((S_EDITORIAL_LINK_CACHE_REQUEST *)sub_page_ptr->link[link_index]);
            }
         }

         sub_page_ptr = sub_page_ptr->next_ptr;
      }
   }

   FUNCTION_FINISH(FlagEditorialLinkPageRequests);
}

/**
 *

 *

 *
 * @param   S_EDITORIAL_LINK_CACHE_REQUEST* ptr -
 * @param   S_CACHE_SUBPAGE* subpage_ptr -
 *

 *
 */
static void DeterminePageEditorialLinks(S_EDITORIAL_LINK_CACHE_REQUEST *ptr,
   S_CACHE_SUBPAGE *subpage_ptr)
{
   U16BIT magazine_number, link_index;
   BOOLEAN is_found;
   S_EDITORIAL_LINK_CACHE_REQUEST_SUB_PAGE *request_sub_page_ptr;
   S_CACHE_PAGE *page_ptr;
   S_EDITORIAL_LINK_CACHE_REQUEST *new_ptr;

   FUNCTION_START(DeterminePageEditorialLinks);

   if ((ptr != NULL) &&
       (ptr->tier <= GetMaximumEditorialLinkTiers()))
   {
      request_sub_page_ptr = ptr->sub_page_list;
      while (request_sub_page_ptr != NULL)
      {
         if (request_sub_page_ptr->sub_code == subpage_ptr->sub_code)
         {
            break;
         }
         request_sub_page_ptr = request_sub_page_ptr->next_ptr;
      }

      magazine_number = GetMagazineIndexFromPageNumber(ptr->page_number);

      if (request_sub_page_ptr == NULL)
      {
         request_sub_page_ptr = (S_EDITORIAL_LINK_CACHE_REQUEST_SUB_PAGE *)GetMemory(
               sizeof(S_EDITORIAL_LINK_CACHE_REQUEST_SUB_PAGE));
         while (request_sub_page_ptr == NULL)
         {
            if (RemoveUnrequestedCacheContent(magazine_number, FALSE) == FALSE)
            {
               break;
            }

            request_sub_page_ptr = (S_EDITORIAL_LINK_CACHE_REQUEST_SUB_PAGE *)GetMemory(
                  sizeof(S_EDITORIAL_LINK_CACHE_REQUEST_SUB_PAGE));
         }

         if (request_sub_page_ptr != NULL)
         {
            request_sub_page_ptr->sub_code = subpage_ptr->sub_code;
            request_sub_page_ptr->is_parsed = FALSE;
            request_sub_page_ptr->next_ptr = ptr->sub_page_list;
            ptr->sub_page_list = request_sub_page_ptr;
         }
      }

      if (request_sub_page_ptr != NULL)
      {
         if (request_sub_page_ptr->is_parsed == FALSE)
         {
            if (GetSubPageEditorialLinks(subpage_ptr, request_sub_page_ptr->link_page_number,
                   request_sub_page_ptr->link_sub_code, MAX_EDITORIAL_LINKS,
                   magazine_number) == TRUE)
            {
               memset(request_sub_page_ptr->link, 0, sizeof(void *) * MAX_EDITORIAL_LINKS);
               request_sub_page_ptr->is_parsed = TRUE;

               for (link_index = 0; link_index < MAX_EDITORIAL_LINKS; link_index++)
               {
                  if (request_sub_page_ptr->link_page_number[link_index] != 0)
                  {
                     // Find page in cache

                     is_found = FALSE;

                     // ...determine the relevant magazine
                     magazine_number = GetMagazineIndexFromPageNumber(request_sub_page_ptr->link_page_number[link_index]);

                     STB_OSMutexLock( magazine_array[magazine_number].m_mutex );
                     //... and locate mathcing page in cache.
                     page_ptr = magazine_array[magazine_number].m_cache_page_ptr;
                     while (page_ptr != NULL)
                     {
                        if ((page_ptr->page_number & CACHE_PAGE_NUMBER_MASK) ==
                            request_sub_page_ptr->link_page_number[link_index])
                        {
                           subpage_ptr = page_ptr->subpage_ptr;

                           while (subpage_ptr != NULL)
                           {
                              if ((subpage_ptr->sub_code == request_sub_page_ptr->link_sub_code[link_index]) ||
                                  (request_sub_page_ptr->link_sub_code[link_index] == PAGE_SUB_CODE_UNDEFINED))
                              {
                                 is_found = TRUE;
                                 break;
                              }

                              subpage_ptr = subpage_ptr->next_ptr;
                           }

                           if (is_found == TRUE)
                           {
                              break;
                           }
                        }

                        page_ptr = page_ptr->next_ptr;
                     }
                     STB_OSMutexUnlock( magazine_array[magazine_number].m_mutex );

                     if (is_found == TRUE)
                     {
                        if (FindEditorialLinkPageRequest(editorial_link_request_root, NULL, NULL,
                               request_sub_page_ptr->link_page_number[link_index]) == FALSE)
                        {
                           // No page request in editorial link tree
                           new_ptr = (S_EDITORIAL_LINK_CACHE_REQUEST *)GetMemory(
                                 sizeof(S_EDITORIAL_LINK_CACHE_REQUEST));
                           while (new_ptr == NULL)
                           {
                              if (RemoveUnrequestedCacheContent(magazine_number, FALSE) == FALSE)
                              {
                                 break;
                              }

                              new_ptr = (S_EDITORIAL_LINK_CACHE_REQUEST *)GetMemory(
                                    sizeof(S_EDITORIAL_LINK_CACHE_REQUEST));
                           }

                           if (new_ptr != NULL)
                           {
                              new_ptr->page_number =
                                 request_sub_page_ptr->link_page_number[link_index];
                              new_ptr->tier = ptr->tier + 1;

                              new_ptr->sub_page_list = NULL;

                              request_sub_page_ptr->link[link_index] = (void *)new_ptr;

                              DeterminePageEditorialLinks(new_ptr, subpage_ptr);
                           }
                        }
                     }
                  }
               }
            }
         }
      }
   }

   FUNCTION_FINISH(DeterminePageEditorialLinks);
}

/**
 *

 *

 *
 * @param   U16BIT* array_ptr -
 * @param   U16BIT array_size -
 * @param   BOOLEAN find_highest -
 * @param   U16BIT** return_array_ptr -
 *
 * @return   
 *
 */
static BOOLEAN SearchRequestArray(U16BIT *array_ptr, U16BIT array_size, BOOLEAN find_highest,
   U16BIT **return_array_ptr)
{
   U16BIT index, found_page_number;
   BOOLEAN retval;

   FUNCTION_START(SearchRequestArray);

   retval = FALSE;

   if (find_highest)
   {
      found_page_number = 0;
   }
   else
   {
      found_page_number = INVALID_PREVNEXT_CACHE_PAGE_NUMBER;
   }

   for (index = 0; index < array_size; index++)
   {
      if (find_highest)
      {
         if (array_ptr[index] > found_page_number)
         {
            found_page_number = array_ptr[index];
            *return_array_ptr = &array_ptr[index];
            retval = TRUE;
         }
      }
      else
      {
         if (array_ptr[index] < found_page_number)
         {
            found_page_number = array_ptr[index];
            *return_array_ptr = &array_ptr[index];
            retval = TRUE;
         }
      }
   }

   if (retval == FALSE)
   {
      *return_array_ptr = &array_ptr[0];
   }

   FUNCTION_FINISH(SearchRequestArray);

   return(retval);
}

/**
 *

 *

 *
 * @param   BOOLEAN find_highest -
 * @param   U16BIT** return_array_ptr -
 *
 * @return   
 *
 */
static BOOLEAN SearchPreviousRequestArray(BOOLEAN find_highest, U16BIT **return_array_ptr)
{
   BOOLEAN retval;

   FUNCTION_START(SearchPreviousRequestArray);

   retval = SearchRequestArray(previous_cache_request_array, NUM_PREVIOUS_CACHE_REQUESTS, find_highest, return_array_ptr);

   FUNCTION_FINISH(SearchPreviousRequestArray);

   return(retval);
}

/**
 *

 *

 *
 * @param   BOOLEAN find_highest -
 * @param   U16BIT** return_array_ptr -
 *
 * @return   
 *
 */
static BOOLEAN SearchNextRequestArray(BOOLEAN find_highest, U16BIT **return_array_ptr)
{
   U16BIT index;
   BOOLEAN retval;

   FUNCTION_START(SearchNextRequestArray);

   if (next_cache_request_array[0] == 0)
   {
      for (index = 0; index < NUM_NEXT_CACHE_REQUESTS; index++)
      {
         next_cache_request_array[index] = INVALID_PREVNEXT_CACHE_PAGE_NUMBER;
      }
   }

   retval = SearchRequestArray(next_cache_request_array, NUM_NEXT_CACHE_REQUESTS, find_highest,
         return_array_ptr);

   FUNCTION_FINISH(SearchNextRequestArray);

   return(retval);
}

/**
 *

 *

 *
 * @param   U16BIT page_number -
 *
 * @return   
 *
 */
static BOOLEAN IsInPreviousRequestArray(U16BIT page_number)
{
   U16BIT index;
   BOOLEAN retval;

   FUNCTION_START(IsInPreviousRequestArray);

   retval = FALSE;

   for (index = 0; index < NUM_PREVIOUS_CACHE_REQUESTS; index++)
   {
      if (previous_cache_request_array[index] == page_number)
      {
         retval = TRUE;
         break;
      }
   }

   FUNCTION_FINISH(IsInPreviousRequestArray);

   return(retval);
}

/**
 *

 *

 *
 * @param   U16BIT page_number -
 *
 * @return   
 *
 */
static BOOLEAN IsInNextRequestArray(U16BIT page_number)
{
   U16BIT index;
   BOOLEAN retval;

   FUNCTION_START(IsInNextRequestArray);

   retval = FALSE;

   if (next_cache_request_array[0] == 0)
   {
      for (index = 0; index < NUM_NEXT_CACHE_REQUESTS; index++)
      {
         next_cache_request_array[index] = INVALID_PREVNEXT_CACHE_PAGE_NUMBER;
      }
   }

   for (index = 0; index < NUM_NEXT_CACHE_REQUESTS; index++)
   {
      if (next_cache_request_array[index] == page_number)
      {
         retval = TRUE;
         break;
      }
   }

   FUNCTION_FINISH(IsInNextRequestArray);

   return(retval);
}

/**
 *

 *

 *
 * @param   U16BIT* page_number -
 * @param   U16BIT* page_sub_code -
 *
 * @return   
 *
 */
static BOOLEAN GetGlobalIndexPage(U16BIT *page_number, U16BIT *page_sub_code)
{
   BOOLEAN retval;
   S_PAGE_REFERENCE_DATA page_reference_data;

   FUNCTION_START(GetGlobalIndexPage);

   retval = FALSE;

   if (is_broadcast_service_data_defined[0] == TRUE)
   {
      if (GetPageReference(&broadcast_service_data[0][1], &page_reference_data,
             PAGE_REFERENCE_TYPE_GLOBAL_INDEX, 0) == TRUE)
      {
         retval = TRUE;
      }
   }

   if (retval == FALSE)
   {
      if (is_broadcast_service_data_defined[1] == TRUE)
      {
         if (GetPageReference(&broadcast_service_data[1][1], &page_reference_data,
                PAGE_REFERENCE_TYPE_GLOBAL_INDEX, 0) == TRUE)
         {
            retval = TRUE;
         }
      }
   }

   if (retval == TRUE)
   {
      *page_number = page_reference_data.page_number;
      *page_sub_code = page_reference_data.page_sub_code;
   }

   FUNCTION_FINISH(GetGlobalIndexPage);

   return(retval);
}

/**
 *

 *

 *
 * @param   U16BIT* page_number -
 * @param   U16BIT* page_sub_code -
 *
 * @return   
 *
 */
static BOOLEAN GetIndexPage(U16BIT *page_number, U16BIT *page_sub_code)
{
   U16BIT magazine_number;
   BOOLEAN retval;
   S_PAGE_REFERENCE_DATA page_reference_data;

   FUNCTION_START(GetIndexPage);

   retval = FALSE;

   // Check to see if there is a sixth editorial link defined within the currently displayed page.
   // This will be in the X/27/0 packet.
   if (page_display_info.page_number)
   {
      if (page_display_info.page_source.is_editorial_page_linking_data_defined == TRUE)
      {
         magazine_number = GetMagazineIndexFromPageNumber(page_display_info.page_number);

         if (GetPageReference(&page_display_info.page_source.editorial_page_linking_data[31],
                &page_reference_data,
                PAGE_REFERENCE_TYPE_EDITORIAL_LINK, magazine_number) == TRUE)
         {
            *page_number = page_reference_data.page_number;
            *page_sub_code = page_reference_data.page_sub_code;

            retval = TRUE;
         }
      }
   }

   if (retval == FALSE)
   {
      retval = GetGlobalIndexPage(page_number, page_sub_code);
   }

   FUNCTION_FINISH(GetIndexPage);

   return(retval);
}

/**
 *

 *

 *
 * @param   U16BIT* page_number -
 * @param   S16BIT page_increment -
 *
 * @return   
 *
 */
static BOOLEAN GetAvailablePage(U16BIT *page_number, S16BIT page_increment)
{
   U8BIT flag;
   U16BIT page_tens, page_units, magazine_number, num_page_tests, page_offset;
   BOOLEAN retval;

   FUNCTION_START(GetAvailablePage);

   retval = FALSE;

   if (page_number != NULL)
   {
      magazine_number = GetMagazineIndexFromPageNumber(*page_number);

      page_tens = (*page_number & 0x00f0) >> 4;
      page_units = *page_number & 0x000f;

      num_page_tests = 0;

      while ((page_increment != 0) &&
             (num_page_tests < 800))
      {
         if (page_increment > 0)
         {
            if (page_units == 9)
            {
               page_units = 0;

               if (page_tens == 9)
               {
                  page_tens = 0;

                  if (magazine_number == 7)
                  {
                     magazine_number = 0;
                  }
                  else
                  {
                     magazine_number++;
                  }
               }
               else
               {
                  page_tens++;
               }
            }
            else
            {
               page_units++;
            }
         }
         else
         {
            if (page_units == 0)
            {
               page_units = 9;

               if (page_tens == 0)
               {
                  page_tens = 9;

                  if (magazine_number == 0)
                  {
                     magazine_number = 7;
                  }
                  else
                  {
                     magazine_number--;
                  }
               }
               else
               {
                  page_tens--;
               }
            }
            else
            {
               page_units--;
            }
         }

         page_offset = (page_tens << 4) + page_units;

         flag = magazine_array[magazine_number].m_cache_valid_page_array[page_offset >> 3] & 1 <<
            (page_offset & 0x07);

         if (flag)
         {
            if (page_increment > 0)
            {
               page_increment--;
            }
            else
            {
               page_increment++;
            }

            if (page_increment == 0)
            {
               if (magazine_number == 0)
               {
                  *page_number = 0x800 + page_offset;
               }
               else
               {
                  *page_number = (magazine_number << 8) + page_offset;
               }

               retval = TRUE;
               break;
            }
         }

         num_page_tests++;
      }
   }

   FUNCTION_FINISH(GetAvailablePage);

   return(retval);
}

/**
 *

 *

 *
 * @param   U16BIT* page_number -
 * @param   U16BIT* page_sub_code -
 * @param   S16BIT page_increment -
 *
 * @return   
 *
 */
static BOOLEAN GetVisitedPage(U16BIT *page_number, U16BIT *page_sub_code, S16BIT page_increment)
{
   BOOLEAN retval;

   FUNCTION_START(GetVisitedPage);

   USE_UNWANTED_PARAM(page_sub_code);

   retval = FALSE;

   while (page_increment != 0)
   {
      if (page_increment > 0)
      {
         if (GetNextPageFromHistory(page_number) == FALSE)
         {
            break;
         }
         page_increment--;
      }
      else
      {
         if (GetPreviousPageFromHistory(page_number) == FALSE)
         {
            break;
         }
         page_increment++;
      }

      retval = TRUE;
   }

   FUNCTION_FINISH(GetVisitedPage);

   return(retval);
}

/**
 *

 *

 *
 * @param   U16BIT* page_number -
 * @param   U16BIT* page_sub_code -
 * @param   U16BIT link_index -
 *
 * @return   
 *
 */
static BOOLEAN GetEditorialLinkPage(U16BIT *page_number, U16BIT *page_sub_code, U16BIT link_index)
{
   U16BIT magazine_number;
   BOOLEAN retval;
   S_PAGE_REFERENCE_DATA page_reference_data;
   U8BIT *link_ptr;

   FUNCTION_START(GetEditorialLinkPage);

   retval = FALSE;

   // Check of collated teletext data of the sub-page actually contains editorial link content.
   if (page_display_info.page_source.is_editorial_page_linking_data_defined == TRUE)
   {
      // Sets the link data block pointer up to just after the designation codee byte.
      link_ptr = &page_display_info.page_source.editorial_page_linking_data[1];

      link_ptr += link_index * 6;

      magazine_number = GetMagazineIndexFromPageNumber(page_display_info.page_number);

      // ...if we can decode the link data block...
      if (GetPageReference(link_ptr, &page_reference_data, PAGE_REFERENCE_TYPE_EDITORIAL_LINK,
             magazine_number) == TRUE)
      {
         // ...then populate the array items.
         *page_number = page_reference_data.page_number;
         *page_sub_code = page_reference_data.page_sub_code;

         retval = TRUE;
      }
   }

   FUNCTION_FINISH(GetEditorialLinkPage);

   return(retval);
}

/**
 *

 *

 *
 * @param   U16BIT* new_page_sub_code -
 * @param   U16BIT current_page_number -
 * @param   U16BIT current_page_sub_code -
 * @param   S16BIT page_increment -
 *
 * @return   
 *
 */
static BOOLEAN GetSubPage(U16BIT *new_page_sub_code, U16BIT current_page_number,
   U16BIT current_page_sub_code, S16BIT page_increment)
{
   BOOLEAN retval;
   U16BIT magazine_number, num_sub_pages, ref_index1, ref_index2;
   S_CACHE_PAGE *page_ptr;
   S_CACHE_SUBPAGE *subpage_ptr;
   S_CACHE_SUBPAGE *temp_sub_page_ref;
   S_CACHE_SUBPAGE **sub_page_ref_array;

   FUNCTION_START(GetSubPage);

   retval = FALSE;

   if (IsRollingSubCode(current_page_sub_code) == TRUE)
   {
      magazine_number = GetMagazineIndexFromPageNumber(current_page_number);

      STB_OSMutexLock( magazine_array[magazine_number].m_mutex );

      page_ptr = magazine_array[magazine_number].m_cache_page_ptr;

      while (page_ptr != NULL)
      {
         if ((page_ptr->page_number & CACHE_PAGE_NUMBER_MASK) == current_page_number)
         {
            num_sub_pages = 0;
            subpage_ptr = page_ptr->subpage_ptr;
            while (subpage_ptr != NULL)
            {
               if (IsRollingSubCode(subpage_ptr->sub_code) == TRUE)
               {
                  num_sub_pages++;
               }
               subpage_ptr = subpage_ptr->next_ptr;
            }

            if (num_sub_pages < 2)
            {
               break;
            }

            sub_page_ref_array = (S_CACHE_SUBPAGE **)GetMemory(num_sub_pages *
                  sizeof(S_CACHE_SUBPAGE *));
            if (sub_page_ref_array == NULL)
            {
               break;
            }

            ref_index1 = 0;
            subpage_ptr = page_ptr->subpage_ptr;
            while (subpage_ptr != NULL)
            {
               if (IsRollingSubCode(subpage_ptr->sub_code) == TRUE)
               {
                  sub_page_ref_array[ref_index1] = subpage_ptr;
                  ref_index1++;
               }
               subpage_ptr = subpage_ptr->next_ptr;
            }

            for (ref_index1 = 0; ref_index1 < (num_sub_pages - 1); ref_index1++)
            {
               for (ref_index2 = ref_index1 + 1; ref_index2 < num_sub_pages; ref_index2++)
               {
                  if (sub_page_ref_array[ref_index1]->sub_code >
                      sub_page_ref_array[ref_index2]->sub_code)
                  {
                     temp_sub_page_ref = sub_page_ref_array[ref_index1];
                     sub_page_ref_array[ref_index1] = sub_page_ref_array[ref_index2];
                     sub_page_ref_array[ref_index2] = temp_sub_page_ref;
                  }
               }
            }

            for (ref_index1 = 0; ref_index1 < num_sub_pages; ref_index1++)
            {
               if (sub_page_ref_array[ref_index1]->sub_code == current_page_sub_code)
               {
                  if (page_increment > 0)
                  {
                     ref_index1 += (U16BIT)page_increment;
                  }
                  else
                  {
                     ref_index1 += num_sub_pages - ((U16BIT)-page_increment % num_sub_pages);
                  }

                  ref_index1 %= num_sub_pages;

                  *new_page_sub_code = sub_page_ref_array[ref_index1]->sub_code;
                  retval = TRUE;
                  break;
               }
            }

            STB_FreeMemory(sub_page_ref_array);

            // If we have been anable to allocate memory for a temporary ordered buffer, then this
            // contingency code is used to return an unsorted sub-code reference.
            if (retval == FALSE)
            {
               // Firstly, find current sub-code reference in the page's link-list.
               subpage_ptr = page_ptr->subpage_ptr;
               while (subpage_ptr != NULL)
               {
                  if (subpage_ptr->sub_code == current_page_sub_code)
                  {
                     break;
                  }
                  subpage_ptr = subpage_ptr->next_ptr;
               }

               if (subpage_ptr != NULL)
               {
                  // If found, then look for next compatible sub-code.
                  subpage_ptr = subpage_ptr->next_ptr;

                  while (subpage_ptr != NULL)
                  {
                     if (IsRollingSubCode(subpage_ptr->sub_code) == TRUE)
                     {
                        *new_page_sub_code = subpage_ptr->sub_code;
                        retval = TRUE;
                        break;
                     }
                     subpage_ptr = subpage_ptr->next_ptr;
                  }

                  if (subpage_ptr == NULL)
                  {
                     // If not found, then we have reached the end of the link-list, so continue
                     // search from start.
                     subpage_ptr = page_ptr->subpage_ptr;

                     while (subpage_ptr != NULL)
                     {
                        if (subpage_ptr->sub_code == current_page_sub_code)
                        {
                           // We have done a full traversal of the link-list, so quit.
                           break;
                        }

                        if (IsRollingSubCode(subpage_ptr->sub_code) == TRUE)
                        {
                           *new_page_sub_code = subpage_ptr->sub_code;
                           retval = TRUE;
                           break;
                        }
                        subpage_ptr = subpage_ptr->next_ptr;
                     }
                  }
               }
            }

            break;
         }

         page_ptr = page_ptr->next_ptr;
      }
      STB_OSMutexUnlock( magazine_array[magazine_number].m_mutex );
   }

   FUNCTION_FINISH(GetSubPage);

   return(retval);
}

/**
 *

 *

 *

 *

 *
 */
static void DriverReset(void)
{
   U16BIT magazine_number;
   S_CACHE_PAGE *page_ptr;
   S_CACHE_PAGE *next_page_ptr;
   S_CACHE_SUBPAGE *subpage_ptr;
   S_CACHE_SUBPAGE *next_subpage_ptr;

   FUNCTION_START(DriverReset);

   DBGPRINT(" ****** CLEARING DATA ****** ");

   for (magazine_number = 0; magazine_number != 8; magazine_number++)
   {
      STB_OSMutexLock( magazine_array[magazine_number].m_mutex );

      page_ptr = magazine_array[magazine_number].m_cache_page_ptr;

      while (page_ptr != NULL)
      {
         subpage_ptr = page_ptr->subpage_ptr;
         while (subpage_ptr != NULL)
         {
            next_subpage_ptr = subpage_ptr->next_ptr;

            STB_FreeMemory(subpage_ptr);

            subpage_ptr = next_subpage_ptr;
         }

         next_page_ptr = page_ptr->next_ptr;

         STB_FreeMemory(page_ptr);

         page_ptr = next_page_ptr;
      }

      memset(magazine_array[magazine_number].m_cache_valid_page_array, 0, 32);

      magazine_array[magazine_number].m_cache_page_ptr = NULL;

      magazine_array[magazine_number].m_specific_data.is_defined[0] = FALSE;
      magazine_array[magazine_number].m_specific_data.is_defined[1] = FALSE;

      memset( &magazine_array[magazine_number].m_page, 0x00, sizeof(S_MAGAZINE_PAGE));

      STB_OSMutexUnlock( magazine_array[magazine_number].m_mutex );
   }

   is_broadcast_service_data_defined[0] = FALSE;
   is_broadcast_service_data_defined[1] = FALSE;

   page_history_lwm = 0;
   page_history_hwm = 0;
   page_history_mwm = 0;
   page_history_request_page_number = 0xffff;

   history_cache_reset_requested = TRUE;
   editorial_link_cache_reset_requested = TRUE;

   page_control_info.state = PAGE_CONTROL_STATE_IDLE;
   page_control_info.reveal_required = FALSE;

   page_control_info.clear_required = FALSE;

   page_display_info.page_number = 0;
   page_display_info.page_sub_code = 0;
   page_display_info.subtitle = FALSE;
   page_display_info.pts_valid = FALSE;
   page_display_info.time_offset = 0;

   /* Reset the character map and page source header otherwise the index row (0) from a previous channel may be displayed */
   memset( page_display_info.character_map, 0x00, sizeof(page_display_info.character_map));
   memset( &page_display_info.page_source, 0x00, sizeof(S_MAGAZINE_PAGE));

   /* display copy is also cleared to prevent obsolete data being displayed */
   page_display_copy.page_number = 0;
   page_display_copy.page_sub_code = 0;
   page_display_copy.subtitle = FALSE;
   page_display_copy.pts_valid = FALSE;
   page_display_copy.time_offset = 0;

   memset( page_display_copy.character_map, 0x00, sizeof(page_display_copy.character_map));
   memset( &page_display_copy.page_source, 0x00, sizeof(S_MAGAZINE_PAGE));

   subtitle_timeout = 0;

   FUNCTION_FINISH(DriverReset);
}

/**
 *

 *

 *

 *

 *
 */
static void CheckPageRequest(BOOLEAN pts_valid, U8BIT *pts)
{
   S_CACHE_PAGE *page_ptr;
   S_CACHE_SUBPAGE *subpage_ptr;
   S_HISTORY_CACHE_REQUEST *history_ptr;
   S_HISTORY_CACHE_REQUEST *prev_history_ptr;
   U16BIT magazine_index, page_number, page_sub_code, item_count;
   BOOLEAN update_cache_requests;
   U8BIT index;

   FUNCTION_START(CheckPageRequest);

   update_cache_requests = FALSE;

   STB_OSSemaphoreWait(page_request_info.semaphore);

   if (page_request_info.pending == TRUE)
   {
      page_request_info.pending = FALSE;

      switch (page_request_info.type)
      {
         case REQUEST_TYPE_EXPLICIT:
         {
            page_request_info.target_page_number = page_request_info.page_number;
            page_request_info.target_page_sub_code = page_request_info.page_sub_code;
            update_cache_requests = TRUE;

            NotifyRequestToPageHistory(page_request_info.target_page_number);
            break;
         }
         case REQUEST_TYPE_INDEX:
         {
            if (GetIndexPage(&page_request_info.target_page_number,
                   &page_request_info.target_page_sub_code) == TRUE)
            {
               update_cache_requests = TRUE;

               NotifyRequestToPageHistory(page_request_info.target_page_number);
            }
            break;
         }
         case REQUEST_TYPE_NEXT_PREV_AVAILABLE:
         {
            page_request_info.target_page_sub_code = PAGE_SUB_CODE_UNDEFINED;

            if (page_request_info.target_page_number == 0)
            {
               // No page requested prior to this request, so next/prev availibility will have to
               // be w.r.t the displayed page.

               if (!page_display_info.page_number)
               {
                  // If there is no diplsayed page, then we cannot do anything more.
                  break;
               }

               page_request_info.target_page_number = page_display_info.page_number;
            }

            if (GetAvailablePage(&page_request_info.target_page_number,
                   page_request_info.param) == TRUE)
            {
               NotifyRequestToPageHistory(page_request_info.target_page_number);

               update_cache_requests = TRUE;
            }

            break;
         }
         case REQUEST_TYPE_NEXT_PREV_VISITED:
         {
            if (GetVisitedPage(&page_request_info.target_page_number,
                   &page_request_info.target_page_sub_code,
                   page_request_info.param) == TRUE)
            {
               update_cache_requests = TRUE;
            }
            break;
         }
         case REQUEST_TYPE_EDITORIAL_LINK:
         {
            if (page_display_info.page_number)
            {
               if (GetEditorialLinkPage(&page_request_info.target_page_number,
                      &page_request_info.target_page_sub_code,
                      page_request_info.param) == TRUE)
               {
                  NotifyRequestToPageHistory(page_request_info.target_page_number);

                  update_cache_requests = TRUE;
               }
            }
            break;
         }
         case REQUEST_TYPE_NEXT_PREV_SUB:
         {
            if (page_display_info.page_number)
            {
               if (GetSubPage(&page_request_info.target_page_sub_code, page_display_info.page_number,
                      page_display_info.page_sub_code,
                      page_request_info.param) == TRUE)
               {
                  page_request_info.target_page_number = page_display_info.page_number;
                  update_cache_requests = TRUE;
               }
            }
            break;
         }
         case REQUEST_TYPE_CANCEL:
         {
            page_request_info.target_page_number = page_display_info.page_number;
            page_request_info.target_page_sub_code = page_display_info.page_sub_code;

            NotifyRequestToPageHistory(0xffff);
            break;
         }
         default:
         {
            page_request_info.pending = FALSE;
            break;
         }
      }
   }

   STB_OSSemaphoreSignal(page_request_info.semaphore);

   if (previousnext_cache_reset_requested == TRUE)
   {
      for (index = 0; index < NUM_PREVIOUS_CACHE_REQUESTS; index++)
      {
         previous_cache_request_array[index] = 0;
      }
      for (index = 0; index < NUM_NEXT_CACHE_REQUESTS; index++)
      {
         next_cache_request_array[index] = INVALID_PREVNEXT_CACHE_PAGE_NUMBER;
      }

      previousnext_cache_reset_requested = FALSE;
   }

   if (history_cache_reset_requested == TRUE)
   {
      while (history_cache_request_list != NULL)
      {
         history_ptr = history_cache_request_list->next_ptr;

         STB_FreeMemory(history_cache_request_list);

         history_cache_request_list = history_ptr;
      }

      update_cache_requests = TRUE;

      history_cache_reset_requested = FALSE;
   }

   if (editorial_link_cache_reset_requested == TRUE)
   {
      KillEditorialLinkPageRequests(&editorial_link_request_root);

      update_cache_requests = TRUE;

      editorial_link_cache_reset_requested = FALSE;
   }

   if (update_cache_requests == TRUE)
   {
      page_number = page_request_info.target_page_number;
      page_sub_code = page_request_info.target_page_sub_code;

      STB_OSSemaphoreWait(page_display_info.page_index_semaphore);

      page_display_info.page_index_digit[0] = ((page_number & 0x0f00) >> 8) + PAGE_INDEX_DIGIT_0;
      page_display_info.page_index_digit[1] = ((page_number & 0x00f0) >> 4) + PAGE_INDEX_DIGIT_0;
      page_display_info.page_index_digit[2] = (page_number & 0x000f) + PAGE_INDEX_DIGIT_0;

      page_display_info.page_index_updated = TRUE;

      STB_OSSemaphoreSignal(page_display_info.page_index_semaphore);

      // Determine the relevant magazine
      magazine_index = GetMagazineIndexFromPageNumber(page_number);

      STB_OSMutexLock( magazine_array[magazine_index].m_mutex );

      //... and locate matching page in cache.
      page_ptr = magazine_array[magazine_index].m_cache_page_ptr;
      while (page_ptr != NULL)
      {
         if ((page_ptr->page_number & CACHE_PAGE_NUMBER_MASK) == page_number)
         {
            if (page_sub_code == PAGE_SUB_CODE_UNDEFINED)
            {
               if (page_ptr->latest_subpage_ptr != NULL)
               {
                  NotifyDisplayUpdate(&page_ptr->latest_subpage_ptr->collated_data, magazine_index,
                     page_number, page_ptr->latest_subpage_ptr->sub_code, FALSE, pts_valid, pts);
                  break;
               }
            }
            else
            {
               subpage_ptr = page_ptr->subpage_ptr;

               while (subpage_ptr != NULL)
               {
                  if (subpage_ptr->sub_code == page_sub_code)
                  {
                     NotifyDisplayUpdate(&subpage_ptr->collated_data, magazine_index, page_number,
                        page_sub_code, FALSE, pts_valid, pts);
                     break;
                  }

                  subpage_ptr = subpage_ptr->next_ptr;
               }

               if (subpage_ptr != NULL)
               {
                  break;
               }
            }
         }

         page_ptr = page_ptr->next_ptr;
      }
      STB_OSMutexUnlock( magazine_array[magazine_index].m_mutex );

      //----------------------------------------------------------------------------------
      // Clearing flags
      //----------------------------------------------------------------------------------

      // Go through all pages in cache clearing each page's relevant flags.
      for (magazine_index = 0; magazine_index != 8; magazine_index++)
      {
         STB_OSMutexLock( magazine_array[magazine_index].m_mutex );
         page_ptr = magazine_array[magazine_index].m_cache_page_ptr;

         while (page_ptr != NULL)
         {
            page_ptr->page_number &= ~(CACHE_PAGE_FLOF_REQUESTED |
                                       CACHE_PAGE_HISTORY_REQUESTED |
                                       CACHE_PAGE_LOCKED);

            page_ptr = page_ptr->next_ptr;
         }
         STB_OSMutexUnlock( magazine_array[magazine_index].m_mutex );
      }

      //----------------------------------------------------------------------------------
      // Request history link-list maintainance
      //----------------------------------------------------------------------------------

      // Search for a match for this page number in the requested page history.
      // This is done to advance an existing record to the top of the link-list, which
      // represents the most recently made requests.
      history_ptr = history_cache_request_list;
      prev_history_ptr = NULL;
      item_count = 0;

      while (history_ptr != NULL)
      {
         item_count++;

         // If we find the page number somewhere in the history link-list...
         if (history_ptr->page_number == page_number)
         {
            // then assuming it's not the top-most item...
            if (history_ptr != history_cache_request_list)
            {
               // we remove it from it's present position within the link-list...
               prev_history_ptr->next_ptr = history_ptr->next_ptr;
               break;
            }
         }

         // No match found yet, so advance to next link-list item.
         prev_history_ptr = history_ptr;
         history_ptr = history_ptr->next_ptr;
      }

      switch (caching_method)
      {
         case EBUTT_CACHING_METHOD_HISTORY:
         case EBUTT_CACHING_METHOD_NAVIGATION:
         case EBUTT_CACHING_METHOD_NAVIGATION_TREE:
         case EBUTT_CACHING_METHOD_ALL:
         {
            // If the requested page is not found in the link-list (i.e. has not been requested in
            // recent history)
            // Then we must attempt to either add a new record to the link-list, or relocate the
            // tail (oldest) record at the top of the link-list, and modify this one.
            if (history_ptr == NULL)
            {
               if (item_count < MAX_HISTORY_CACHE_REQUESTS)
               {
                  history_ptr = (S_HISTORY_CACHE_REQUEST *)GetMemory(sizeof(S_HISTORY_CACHE_REQUEST));
                  while (history_ptr == NULL)
                  {
                     // Determine the relevant magazine
                     magazine_index = GetMagazineIndexFromPageNumber(page_number);

                     if (RemoveUnrequestedCacheContent(magazine_index, FALSE) == FALSE)
                     {
                        break;
                     }

                     history_ptr = (S_HISTORY_CACHE_REQUEST *)GetMemory(sizeof(S_HISTORY_CACHE_REQUEST));
                  }
               }

               // If either allcation of a new item failed, or was blocked because the link-list
               // already has too many records ...
               if (history_ptr == NULL)
               {
                  if (history_cache_request_list != NULL)
                  {
                     if (history_cache_request_list->next_ptr == NULL)
                     {
                        // If there is a single record in the history, then just modify this one.
                        history_cache_request_list->page_number = page_number;
                     }
                     else
                     {
                        // Find the last record in the link-list...
                        history_ptr = history_cache_request_list;
                        prev_history_ptr = NULL;
                        while (history_ptr->next_ptr != NULL)
                        {
                           prev_history_ptr = history_ptr;
                           history_ptr = history_ptr->next_ptr;
                        }

                        if (prev_history_ptr != NULL)
                        {
                           // we remove it from it's end position within the link-list
                           prev_history_ptr->next_ptr = NULL;
                        }
                     }
                  }
               }
            }
            break;
         }

         default:
            break;
      }

      // If we have a valid pointer here, then we must populate it with the latest
      // page request, and add it to the top of the link-list.
      if ((history_ptr != NULL) &&
          (history_ptr != history_cache_request_list))
      {
         history_ptr->page_number = page_number;
         history_ptr->next_ptr = history_cache_request_list;
         history_cache_request_list = history_ptr;
      }

      // Finally, go through each item in the page request link-list, and if the relevant
      // record is found within the page case, then set the CACHE_PAGE_HISTORY_REQUESTED flag to
      // donote this.
      history_ptr = history_cache_request_list;
      // For each page request...
      while (history_ptr != NULL)
      {
         SetPageFlag( history_ptr->page_number, CACHE_PAGE_HISTORY_REQUESTED );

         history_ptr = history_ptr->next_ptr;
      }

      //----------------------------------------------------------------------------------
      // Request FLOF tree maintainance
      //----------------------------------------------------------------------------------

      switch (caching_method)
      {
         case EBUTT_CACHING_METHOD_NAVIGATION:
         case EBUTT_CACHING_METHOD_NAVIGATION_TREE:
         case EBUTT_CACHING_METHOD_ALL:
         {
            // Look for editorial link request item representing this page and reposition at top
            // of request tree if necessary. In doing this, all editorial link items within the
            // tree are repositioned w.r.t the new root item if possible.
            UpdateEditorialLinkPageRequest(page_number, page_sub_code);

            // Then go through all items left in tree, and mark corresponding existing cahced pages
            // accordingly.
            FlagEditorialLinkPageRequests(editorial_link_request_root);

            break;
         }

         default:
            break;
      }

      SetPageFlag( page_number, CACHE_PAGE_LOCKED );

      if (page_display_info.page_number)
      {
         SetPageFlag( page_display_info.page_number, CACHE_PAGE_LOCKED );
      }
   }

   FUNCTION_FINISH(CheckPageRequest);
}

/**
 *

 *

 *
 * @param   U16BIT magazine_number -
 * @param   U16BIT mask -
 *
 * @return   
 *
 */
static BOOLEAN RemoveCachePage(U16BIT magazine_number, U16BIT mask)
{
   BOOLEAN retval;
   S_CACHE_PAGE *page_ptr;
   S_CACHE_PAGE *last_page_ptr;
   S_CACHE_PAGE *search_page_ptr;
   S_CACHE_PAGE *search_last_page_ptr;
   S_CACHE_SUBPAGE *subpage_ptr;
   S_CACHE_SUBPAGE *temp_subpage_ptr;

   FUNCTION_START(RemoveCachePage);

   retval = FALSE;

   // Search for last page in cache link-list with no caching flags set.
   search_page_ptr = magazine_array[magazine_number].m_cache_page_ptr;
   search_last_page_ptr = NULL;
   page_ptr = NULL;
   last_page_ptr = NULL;

   while (search_page_ptr != NULL)
   {
      if (!(search_page_ptr->page_number & ~(CACHE_PAGE_NUMBER_MASK | mask)))
      {
         page_ptr = search_page_ptr;
         last_page_ptr = search_last_page_ptr;
         break;
      }
      search_last_page_ptr = search_page_ptr;
      search_page_ptr = search_page_ptr->next_ptr;
   }

   if (page_ptr != NULL)
   {
      // Firstly free all the subpages.
      subpage_ptr = page_ptr->subpage_ptr;
      while (subpage_ptr != NULL)
      {
         temp_subpage_ptr = subpage_ptr->next_ptr;

         STB_FreeMemory(subpage_ptr);

         subpage_ptr = temp_subpage_ptr;
      }

      if (last_page_ptr == NULL)
      {
         magazine_array[magazine_number].m_cache_page_ptr = page_ptr->next_ptr;
      }
      else
      {
         last_page_ptr->next_ptr = page_ptr->next_ptr;
      }

      // Now we can free the removed cache page.
      STB_FreeMemory(page_ptr);

      retval = TRUE;
   }

   FUNCTION_FINISH(RemoveCachePage);

   return(retval);
}

/**
 *

 *

 *
 * @param   U16BIT magazine_number -
 *
 * @return   
 *
 */
static BOOLEAN RemoveUnrequestedCachePage(U16BIT magazine_number)
{
   BOOLEAN retval;
   S_CACHE_PAGE *page_ptr;
   S_CACHE_PAGE *last_page_ptr;
   S_CACHE_PAGE *search_page_ptr;
   S_CACHE_PAGE *search_last_page_ptr;
   S_CACHE_SUBPAGE *subpage_ptr;
   S_CACHE_SUBPAGE *temp_subpage_ptr;

   FUNCTION_START(RemoveUnrequestedCachePage);

   retval = FALSE;

   STB_OSMutexLock( magazine_array[magazine_number].m_mutex );

   // Search for last page in cache link-list with no caching flags set.
   search_page_ptr = magazine_array[magazine_number].m_cache_page_ptr;
   search_last_page_ptr = NULL;
   page_ptr = NULL;
   last_page_ptr = NULL;

   while (search_page_ptr != NULL)
   {
      if (!(search_page_ptr->page_number & ~CACHE_PAGE_NUMBER_MASK))
      {
         page_ptr = search_page_ptr;
         last_page_ptr = search_last_page_ptr;
         break;
      }
      search_last_page_ptr = search_page_ptr;
      search_page_ptr = search_page_ptr->next_ptr;
   }

   if (page_ptr != NULL)
   {
      // Firstly free all the subpages.
      subpage_ptr = page_ptr->subpage_ptr;
      while (subpage_ptr != NULL)
      {
         temp_subpage_ptr = subpage_ptr->next_ptr;

         STB_FreeMemory(subpage_ptr);

         subpage_ptr = temp_subpage_ptr;
      }

      if (last_page_ptr == NULL)
      {
         magazine_array[magazine_number].m_cache_page_ptr = page_ptr->next_ptr;
      }
      else
      {
         last_page_ptr->next_ptr = page_ptr->next_ptr;
      }

      // Now we can free the removed cache page.
      STB_FreeMemory(page_ptr);

      retval = TRUE;
   }

   STB_OSMutexUnlock( magazine_array[magazine_number].m_mutex );

   FUNCTION_FINISH(RemoveUnrequestedCachePage);

   return(retval);
}

/**
 *

 *

 *
 * @param   U16BIT magazine_number -
 * @param   BOOLEAN exclude_magazine -
 *
 * @return   
 *
 */
static BOOLEAN RemoveCacheContent(U16BIT magazine_number, BOOLEAN exclude_magazine)
{
   U16BIT mask_index, offset, mgzn_ndx;
   BOOLEAN retval;

   FUNCTION_START(RemoveCacheContent);

   retval = FALSE;

   for (mask_index = 0; (mask_index != 8) && (retval == FALSE); mask_index++)
   {
      if (exclude_magazine == FALSE)
      {
         retval = RemoveCachePage(magazine_number, cache_removal_priority_mask_array[mask_index]);
      }

      for (offset = 1; retval == FALSE && offset != 8; offset++)
      {
         mgzn_ndx = (magazine_number + offset) % 8;
         STB_OSMutexLock( magazine_array[mgzn_ndx].m_mutex );
         retval = RemoveCachePage(mgzn_ndx, cache_removal_priority_mask_array[mask_index]);
         STB_OSMutexUnlock( magazine_array[mgzn_ndx].m_mutex );
      }
   }

   FUNCTION_FINISH(RemoveCacheContent);

   return(retval);
}

/**
 *

 *

 *
 * @param   U16BIT magazine_number -
 * @param   BOOLEAN exclude_magazine -
 *
 * @return   
 *
 */
static BOOLEAN RemoveUnrequestedCacheContent(U16BIT magazine_number, BOOLEAN exclude_magazine)
{
   U16BIT magazine_offset;
   BOOLEAN retval;

   FUNCTION_START(RemoveUnrequestedCacheContent);

   retval = FALSE;

   if (exclude_magazine == FALSE)
   {
      retval = RemoveUnrequestedCachePage(magazine_number);
   }

   for (magazine_offset = 1; retval == FALSE && magazine_offset != 8; magazine_offset++)
   {
      retval = RemoveUnrequestedCachePage((magazine_number + magazine_offset) % 8);
   }

   FUNCTION_FINISH(RemoveUnrequestedCacheContent);

   return(retval);
}

#if 0
/**
 *

 *

 *

 *

 *
 */
static void RemoveAllCacheContent(void)
{
   U16BIT magazine_number;
   S_CACHE_PAGE *page_ptr;
   S_CACHE_PAGE *next_page_ptr;
   S_CACHE_SUBPAGE *subpage_ptr;
   S_CACHE_SUBPAGE *next_subpage_ptr;

   FUNCTION_START(RemoveAllCacheContent);

   for (magazine_number = 0; magazine_number != 8; magazine_number++)
   {
      STB_OSMutexLock( magazine_array[magazine_number].m_mutex );

      page_ptr = magazine_array[magazine_number].m_cache_page_ptr;

      while (page_ptr != NULL)
      {
         subpage_ptr = page_ptr->subpage_ptr;
         while (subpage_ptr != NULL)
         {
            next_subpage_ptr = subpage_ptr->next_ptr;

            STB_FreeMemory(subpage_ptr);

            subpage_ptr = next_subpage_ptr;
         }

         next_page_ptr = page_ptr->next_ptr;

         STB_FreeMemory(page_ptr);

         page_ptr = next_page_ptr;
      }

      memset(magazine_array[magazine_number].m_cache_valid_page_array, 0, sizeof(U8BIT) * 32);
      magazine_array[magazine_number].m_cache_page_ptr = NULL;

      STB_OSMutexUnlock( magazine_array[magazine_number].m_mutex );
   }

   FUNCTION_FINISH(RemoveAllCacheContent);
}

#endif

/**
 *

 *

 *
 * @param   S_MAGAZINE_PAGE* collated_page_ptr -
 * @param   U16BIT magazine_number -
 * @param   U16BIT page_number -
 * @param   U16BIT page_sub_code -
 * @param   BOOLEAN update_clock -
 *

 *
 */
static void NotifyDisplayUpdate(S_MAGAZINE_PAGE *collated_page_ptr, U16BIT magazine_number,
   U16BIT page_number, U16BIT page_sub_code, BOOLEAN update_clock, BOOLEAN pts_valid, U8BIT *pts)
{
   U16BIT index;

   FUNCTION_START(NotifyDisplayUpdate);

   // If we are about to display a page, then there is no need to continue search for the
   // broadcaster-defined index page. We may have collated page 100 (which we ask for by default,
   // or the operator may have manually chosen a page themseleves.
   broadcast_index_page_invocation_required = FALSE;

   NotifyDisplayToPageHistory(page_number);

   DBGPRINT("%03x:%04x\n", page_number, page_sub_code);

   /* Wait until page_display_info is available to be used */
   STB_OSSemaphoreWait(page_free_sem);

   STB_OSSemaphoreWait(page_display_info.page_semaphore);

   memcpy(&page_display_info.page_source, collated_page_ptr, sizeof(S_MAGAZINE_PAGE));

   if (page_display_info.page_number)
   {
      if (GetMagazineIndexFromPageNumber(page_display_info.page_number) != magazine_number)
      {
         STB_OSMutexLock( magazine_array[magazine_number].m_mutex );

         for (index = 0; index != 2; index++)
         {
            if (magazine_array[magazine_number].m_specific_data.is_defined[index] == TRUE)
            {
               memcpy(page_display_info.mgzn_data.s_data[index],
                  magazine_array[magazine_number].m_specific_data.s_data[index], MAX_COLUMN);

               page_display_info.mgzn_data.is_defined[index] = TRUE;
            }
         }

         STB_OSMutexUnlock( magazine_array[magazine_number].m_mutex );
      }
   }

   if ((page_display_info.page_number != page_number) ||
       (page_display_info.page_sub_code != page_sub_code))
   {
      page_display_info.reveal_set = FALSE;
   }

   page_display_info.clear_set = FALSE;
   page_control_info.clear_required = FALSE;
   page_display_info.clear_required = FALSE;

   page_display_info.page_number = page_number;
   page_display_info.page_sub_code = page_sub_code;

   page_display_info.page_clock_valid = update_clock;

   page_display_info.pts_valid = pts_valid;
   if (pts_valid)
   {
      memcpy(page_display_info.page_pts, pts, 5);
   }

   page_display_info.page_updated = TRUE;

   STB_OSSemaphoreSignal(page_display_info.page_semaphore);

   if (!show_header_row)
   {
      subtitle_timeout = STB_OSGetClockMilliseconds();
      DBGPRINT("set Subtitle Timeout @ %lu", subtitle_timeout);
   }

   FUNCTION_FINISH(NotifyDisplayUpdate);
}

/**
 *

 *

 *
 * @param   S_MAGAZINE_PAGE* collated_page_ptr -
 * @param   S_PAGE_REFERENCE_DATA* reference_data_ptr -
 *

 *
 */
static void CheckPageCaching(S_MAGAZINE_PAGE *collated_page_ptr, S_PAGE_REFERENCE_DATA *reference_data_ptr,
   BOOLEAN pts_valid, U8BIT *pts)
{
   U16BIT flag, index, page_sub_code, magazine_index, num_unrequested_pages;
   BOOLEAN caching_required;
   BOOLEAN caching_successful;
   U16BIT *next_request_array_ptr;
   U16BIT *previous_request_array_ptr;
   S_CACHE_PAGE *page_ptr;
   S_CACHE_SUBPAGE *subpage_ptr;
   S_EDITORIAL_LINK_CACHE_REQUEST *editorial_link_ptr;
   S_HISTORY_CACHE_REQUEST *history_ptr;

   FUNCTION_START(CheckPageCaching);

   flag = 0;
   caching_required = FALSE;
   editorial_link_ptr = NULL;

   if ((caching_method == EBUTT_CACHING_METHOD_NAVIGATION) ||
       (caching_method == EBUTT_CACHING_METHOD_NAVIGATION_TREE) ||
       (caching_method == EBUTT_CACHING_METHOD_ALL))
   {
      if (GetEditorialLink(editorial_link_request_root, &editorial_link_ptr,
             reference_data_ptr->page_number) == TRUE)
      {
         flag = CACHE_PAGE_FLOF_REQUESTED;
         caching_required = TRUE;
      }
   }

   if ((caching_method == EBUTT_CACHING_METHOD_HISTORY) ||
       (caching_method == EBUTT_CACHING_METHOD_NAVIGATION) ||
       (caching_method == EBUTT_CACHING_METHOD_NAVIGATION_TREE) ||
       (caching_method == EBUTT_CACHING_METHOD_ALL))
   {
      history_ptr = history_cache_request_list;
      // For each page request...
      while (history_ptr != NULL)
      {
         if (history_ptr->page_number == reference_data_ptr->page_number)
         {
            flag |= CACHE_PAGE_HISTORY_REQUESTED;
            caching_required = TRUE;
            break;
         }

         history_ptr = history_ptr->next_ptr;
      }
   }

   for (index = 0; index < NUM_PREVIOUS_CACHE_REQUESTS; index++)
   {
      if (previous_cache_request_array[index] == reference_data_ptr->page_number)
      {
         flag |= CACHE_PAGE_PREVNEXT_REQUESTED;
         caching_required = TRUE;
         break;
      }
   }

   if (page_display_info.page_number)
   {
      if (page_display_info.page_number != next_prev_page_number)
      {
         do
         {
            // Find lowest page number in request next array
            SearchNextRequestArray(FALSE, &next_request_array_ptr);

            // If it's greater than the displayed page then no modification to the array is
            // required, as all the array records are still greater than the displayed page.
            if (page_display_info.page_number < *next_request_array_ptr)
            {
               // exit loop
               break;
            }

            // If the displayed page is in the array, then remove it.
            if (page_display_info.page_number == *next_request_array_ptr)
            {
               // First see if the relevant record exists in the page cache.
               // If found, then remove relevant flag,
               UnsetPageFlag( *next_request_array_ptr, CACHE_PAGE_PREVNEXT_REQUESTED );

               // Clear the item in the next array
               *next_request_array_ptr = INVALID_PREVNEXT_CACHE_PAGE_NUMBER;
            }
            else
            {
               // If we are here then the array record is lower than the request page number.
               // In this case we will transfer the array record to the previous request array.

               // Find lowest element
               SearchPreviousRequestArray(FALSE, &previous_request_array_ptr);

               if (*previous_request_array_ptr != 0)
               {
                  // First see if the relevant record exists in the page cache.
                  // If found, then remove relevant flag,
                  UnsetPageFlag( *previous_request_array_ptr, CACHE_PAGE_PREVNEXT_REQUESTED );
               }

               // Transfer the page request from the next array to the previous array
               *previous_request_array_ptr = *next_request_array_ptr;
               // Clear the item in the next array
               *next_request_array_ptr = INVALID_PREVNEXT_CACHE_PAGE_NUMBER;
            }
         }
         while (1);

         do
         {
            // Find highest page number in request previous array
            SearchPreviousRequestArray(TRUE, &previous_request_array_ptr);

            // If it's less than the displayed page then no modification to the array is
            // required, as all the array records are still less than the displayed page.
            if (page_display_info.page_number > *previous_request_array_ptr)
            {
               break;
            }

            // If the displayed page is in the array, then remove it.
            if (page_display_info.page_number == *previous_request_array_ptr)
            {
               // First see if the relevant record exists in the page cache.
               // If found, then remove relevant flag,
               UnsetPageFlag( *previous_request_array_ptr, CACHE_PAGE_PREVNEXT_REQUESTED );

               // Clear the item in the previous array
               *previous_request_array_ptr = 0;
            }
            else
            {
               // If we are here then the array record is higher than the request page number.
               // In this case we will transfer the array record to the next request array.

               // Find highest element
               SearchNextRequestArray(TRUE, &next_request_array_ptr);

               if (*next_request_array_ptr != INVALID_PREVNEXT_CACHE_PAGE_NUMBER)
               {
                  // First see if the relevant record exists in the page cache.
                  // If found, then remove relevant flag,
                  UnsetPageFlag( *next_request_array_ptr, CACHE_PAGE_PREVNEXT_REQUESTED );
               }

               // Transfer the page request from the previous array to the next array
               *next_request_array_ptr = *previous_request_array_ptr;
               // Clear the item in the previous array
               *previous_request_array_ptr = 0;
            }
         }
         while (1);

         next_prev_page_number = page_display_info.page_number;
      }

      if (reference_data_ptr->page_number > page_display_info.page_number)
      {
         // Find highest page number in request next array
         SearchNextRequestArray(TRUE, &next_request_array_ptr);

         if (reference_data_ptr->page_number < *next_request_array_ptr)
         {
            if (IsInNextRequestArray(reference_data_ptr->page_number) == FALSE)
            {
               if (*next_request_array_ptr != INVALID_PREVNEXT_CACHE_PAGE_NUMBER)
               {
                  UnsetPageFlag( *next_request_array_ptr, CACHE_PAGE_PREVNEXT_REQUESTED );
               }

               *next_request_array_ptr = reference_data_ptr->page_number;

               SetPageFlag( *next_request_array_ptr, CACHE_PAGE_PREVNEXT_REQUESTED );
            }
         }
      }
      else if (reference_data_ptr->page_number < page_display_info.page_number)
      {
         // Find lowest page number in request previous array
         SearchPreviousRequestArray(FALSE, &previous_request_array_ptr);

         if (reference_data_ptr->page_number > *previous_request_array_ptr)
         {
            if (IsInPreviousRequestArray(reference_data_ptr->page_number) == FALSE)
            {
               if (*previous_request_array_ptr != 0)
               {
                  UnsetPageFlag( *previous_request_array_ptr, CACHE_PAGE_PREVNEXT_REQUESTED );
               }

               *previous_request_array_ptr = reference_data_ptr->page_number;

               SetPageFlag( *previous_request_array_ptr, CACHE_PAGE_PREVNEXT_REQUESTED );
            }
         }
      }
   }

   for (index = 0; index < NUM_NEXT_CACHE_REQUESTS; index++)
   {
      if (next_cache_request_array[index] == reference_data_ptr->page_number)
      {
         flag |= CACHE_PAGE_PREVNEXT_REQUESTED;
         caching_required = TRUE;
         break;
      }
   }

   if (caching_method == EBUTT_CACHING_METHOD_ALL)
   {
      caching_required = TRUE;
   }

   // As soon as a requested page arrives, update/add it's content to the cache and immediately lock
   // it. We will subsequently invoke a display update.
   // Similarily, this code will execute when the page arrives on the carousel repeatedly thereafter.
   if (page_request_info.target_page_number == reference_data_ptr->page_number)
   {
      flag |= CACHE_PAGE_LOCKED;
      caching_required = TRUE;
   }

   // This is done just in case we are in a transitional period when the currently displayed page
   // is updated whilst we are waiting for a newly requested page.
   // In this circumstance we will not invoke a display update, but merely lock the page in the
   // cache.
   if (page_display_info.page_number == reference_data_ptr->page_number)
   {
      flag |= CACHE_PAGE_LOCKED;
   }

   if (caching_required == TRUE)
   {
      // ...determine the relevant magazine
      magazine_index = GetMagazineIndexFromPageNumber(reference_data_ptr->page_number);

      caching_successful = FALSE;

      if (GetHamming8DoubleWord(collated_page_ptr->header + 2, &page_sub_code) == TRUE)
      {
         page_sub_code &= 0x3f7f;

         STB_OSMutexLock( magazine_array[magazine_index].m_mutex );

         //... and locate mathcing page in cache.
         page_ptr = magazine_array[magazine_index].m_cache_page_ptr;
         while (page_ptr != NULL)
         {
            // If the page already exists in the cache...
            if ((page_ptr->page_number & CACHE_PAGE_NUMBER_MASK) == reference_data_ptr->page_number)
            {
               // update it's flags and sub-page content.
               page_ptr->page_number &= CACHE_PAGE_NUMBER_MASK;
               page_ptr->page_number |= flag;

               subpage_ptr = page_ptr->subpage_ptr;

               while (subpage_ptr != NULL)
               {
                  if (subpage_ptr->sub_code == page_sub_code)
                  {
                     if (reference_data_ptr->erase_page == TRUE)
                     {
                        memcpy(&subpage_ptr->collated_data, collated_page_ptr,
                           sizeof(S_MAGAZINE_PAGE));
                     }
                     else
                     {
                        memcpy(subpage_ptr->collated_data.header, collated_page_ptr->header, MAX_COLUMN);
                        subpage_ptr->collated_data.is_header_defined = TRUE;

                        for (index = 1; index != MAX_ROWS; index++)
                        {
                           if (collated_page_ptr->display_data_defined_mask & (1 << index))
                           {
                              memcpy(subpage_ptr->collated_data.display_data[index],
                                 collated_page_ptr->display_data[index], MAX_COLUMN);
                              subpage_ptr->collated_data.display_data_defined_mask |= 1 << index;
                           }
                        }

                        for (index = 0; index != 16; index++)
                        {
                           if (collated_page_ptr->basic_enhancement_data_defined_mask & (1 << index))
                           {
                              memcpy(subpage_ptr->collated_data.basic_enhancement_data[index],
                                 collated_page_ptr->basic_enhancement_data[index], MAX_COLUMN);
                              subpage_ptr->collated_data.basic_enhancement_data_defined_mask |=
                                 1 << index;
                           }
                        }

                        if (collated_page_ptr->is_editorial_page_linking_data_defined == TRUE)
                        {
                           memcpy(subpage_ptr->collated_data.editorial_page_linking_data,
                              collated_page_ptr->editorial_page_linking_data, MAX_COLUMN);
                           subpage_ptr->collated_data.is_editorial_page_linking_data_defined = TRUE;
                        }

                        for (index = 0; index != 2; index++)
                        {
                           if (collated_page_ptr->specific_data.is_defined[index] == TRUE)
                           {
                              memcpy(subpage_ptr->collated_data.specific_data.s_data[index],
                                 collated_page_ptr->specific_data.s_data[index], MAX_COLUMN);
                              subpage_ptr->collated_data.specific_data.is_defined[index] = TRUE;
                           }
                        }
                     }

                     page_ptr->latest_subpage_ptr = subpage_ptr;

                     DeterminePageEditorialLinks(editorial_link_ptr, subpage_ptr);

                     caching_successful = TRUE;
                     break;
                  }

                  subpage_ptr = subpage_ptr->next_ptr;
               }

               if (subpage_ptr == NULL)
               {
                  subpage_ptr = (S_CACHE_SUBPAGE *)GetMemory(sizeof(S_CACHE_SUBPAGE));
                  while (subpage_ptr == NULL)
                  {
                     // Remove old cache page(s) if no memory available!
                     if (RemoveCacheContent(magazine_index, TRUE) == FALSE)
                     {
                        break;
                     }
                     subpage_ptr = (S_CACHE_SUBPAGE *)GetMemory(sizeof(S_CACHE_SUBPAGE));
                  }

                  if (subpage_ptr != NULL)
                  {
                     subpage_ptr->sub_code = page_sub_code;
                     memcpy(&subpage_ptr->collated_data, collated_page_ptr,
                        sizeof(S_MAGAZINE_PAGE));
                     subpage_ptr->next_ptr = page_ptr->subpage_ptr;

                     page_ptr->subpage_ptr = subpage_ptr;
                     page_ptr->latest_subpage_ptr = subpage_ptr;

                     DeterminePageEditorialLinks(editorial_link_ptr, subpage_ptr);

                     caching_successful = TRUE;
                  }
               }

               break;
            }

            page_ptr = page_ptr->next_ptr;
         }

         if (page_ptr == NULL)
         {
            // Add new page AND subpage here....
            // NB - remove old page if no memory available!
            page_ptr = (S_CACHE_PAGE *)GetMemory(sizeof(S_CACHE_PAGE));
            while (page_ptr == NULL)
            {
               // Remove old cache page(s) if no memory available!
               if (RemoveCacheContent(magazine_index, FALSE) == FALSE)
               {
                  break;
               }
               page_ptr = (S_CACHE_PAGE *)GetMemory(sizeof(S_CACHE_PAGE));
            }

            if (page_ptr != NULL)
            {
               page_ptr->page_number = reference_data_ptr->page_number | flag;

               subpage_ptr = (S_CACHE_SUBPAGE *)GetMemory(sizeof(S_CACHE_SUBPAGE));
               while (subpage_ptr == NULL)
               {
                  // Remove old cache page(s) if no memory available!
                  if (RemoveCacheContent(magazine_index, TRUE) == FALSE)
                  {
                     break;
                  }
                  subpage_ptr = (S_CACHE_SUBPAGE *)GetMemory(sizeof(S_CACHE_SUBPAGE));
               }

               if (subpage_ptr != NULL)
               {
                  subpage_ptr->sub_code = page_sub_code;
                  memcpy(&subpage_ptr->collated_data, collated_page_ptr, sizeof(S_MAGAZINE_PAGE));
                  subpage_ptr->next_ptr = NULL;

                  page_ptr->subpage_ptr = subpage_ptr;
                  page_ptr->latest_subpage_ptr = subpage_ptr;

                  page_ptr->next_ptr = magazine_array[magazine_index].m_cache_page_ptr;

                  magazine_array[magazine_index].m_cache_page_ptr = page_ptr;

                  caching_successful = TRUE;
               }
               else
               {
                  STB_FreeMemory(page_ptr);
               }
            }
         }

         STB_OSMutexUnlock( magazine_array[magazine_index].m_mutex );
      }

      if (caching_successful == TRUE)
      {
         if (page_display_info.page_number == reference_data_ptr->page_number)
         {
            // We have just updated cache for currently displayed page.

            if (page_control_info.state == PAGE_CONTROL_STATE_HOLD)
            {
               // When page is held, we only prompt for a display update when the currently
               // displayed sub-page is refreshed in the cache.
               if (page_display_info.page_sub_code == page_sub_code)
               {
                  NotifyDisplayUpdate(collated_page_ptr, magazine_index,
                     reference_data_ptr->page_number, page_sub_code, TRUE, pts_valid, pts);
               }
            }
            else
            {
               if (page_control_info.state != PAGE_CONTROL_STATE_INPUT)
               {
                  if (page_request_info.target_page_number == reference_data_ptr->page_number)
                  {
                     // When display is not held, then we always prompt for a display update
                     // as each sub-page is refreshed in the cache.
                     if (page_display_info.page_sub_code != page_sub_code)
                     {
                        if ((IsRollingSubCode(page_display_info.page_sub_code) == TRUE) &&
                            (collated_page_ptr->display_data_defined_mask != 0))
                        {
                           NotifyDisplayUpdate(collated_page_ptr, magazine_index,
                              reference_data_ptr->page_number, page_sub_code, TRUE, pts_valid, pts);
                        }
                     }
                     else
                     {
                        if (collated_page_ptr->display_data_defined_mask != 0)
                        {
                           NotifyDisplayUpdate(collated_page_ptr, magazine_index,
                              reference_data_ptr->page_number, page_sub_code, TRUE, pts_valid, pts);
                        }
                     }
                  }
               }
            }
         }
         else
         {
            // We have just updated cache for page that is not currently displayed.

            if (page_request_info.target_page_number == reference_data_ptr->page_number)
            {
               // This page is curently requested as the display page!
               NotifyDisplayUpdate(collated_page_ptr, magazine_index,
                  reference_data_ptr->page_number, page_sub_code, TRUE, pts_valid, pts);
            }
         }
      }

      //Lastly, check that not too many unrequested pages are cached for each magazine.
      if (caching_method != EBUTT_CACHING_METHOD_ALL)
      {
         for (magazine_index = 0; magazine_index != 8; magazine_index++)
         {
            STB_OSMutexLock( magazine_array[magazine_index].m_mutex );
            page_ptr = magazine_array[magazine_index].m_cache_page_ptr;
            num_unrequested_pages = 0;

            while (page_ptr != NULL)
            {
               if (!(page_ptr->page_number & ~CACHE_PAGE_NUMBER_MASK))
               {
                  num_unrequested_pages++;
               }

               page_ptr = page_ptr->next_ptr;
            }
            STB_OSMutexUnlock( magazine_array[magazine_index].m_mutex );

            while (num_unrequested_pages > 10)
            {
               if (RemoveUnrequestedCachePage(magazine_index) == FALSE)
               {
                  break;
               }

               num_unrequested_pages--;
            }
         }
      }
   }

   FUNCTION_FINISH(CheckPageCaching);
}

/**
 *

 *

 *
 * @param   S_MAGAZINE_PAGE* collated_page_ptr -
 * @param   U16BIT page_number -
 *

 *
 */
static void CheckTimeFillingPacket(S_MAGAZINE_PAGE *collated_page_ptr, U16BIT page_number)
{
   FUNCTION_START(CheckTimeFillingPacket);

   // Page is 'Time Filling Header' page, which is used to update the display of real-time clock.

   if ((page_display_info.page_number & 0x0f00) == (page_number & 0x0f00))
   {
      STB_OSSemaphoreWait(page_display_info.time_filler_header_data_semaphore);

      memcpy(page_display_info.time_filler_header_data, &collated_page_ptr->header[8], 32);

      page_display_info.clock_updated = TRUE;

      STB_OSSemaphoreSignal(page_display_info.time_filler_header_data_semaphore);
   }

   FUNCTION_FINISH(CheckTimeFillingPacket);
}

/**
 *

 *

 *
 * @param   S_MAGAZINE_PAGE* collated_page_ptr -
 *

 *
 */
static void CheckHeaderClockContent(S_MAGAZINE_PAGE *collated_page_ptr)
{
   FUNCTION_START(CheckHeaderClockContent);

   if (memcmp(&page_display_info.time_filler_header_data[24], &collated_page_ptr->header[32], 8))
   {
      STB_OSSemaphoreWait(page_display_info.time_filler_header_data_semaphore);

      memcpy(page_display_info.time_filler_header_data, &collated_page_ptr->header[8], 32);

      page_display_info.clock_updated = TRUE;

      STB_OSSemaphoreSignal(page_display_info.time_filler_header_data_semaphore);
   }

   FUNCTION_FINISH(CheckHeaderClockContent);
}

/**
 *

 *
 * @brief   Called from the page request mechanism to declare that page number is to be
//                 displayed. The page number will not actually be added to the history until it is
//                 displayed - this stops automatic display updates of pages (for instance when
//                 fresh sub-pages arrive) from interfering with the history buffer.
 *
 * @param   U16BIT page_number - the page to be potentially added.
 *

 *
 */
static void NotifyRequestToPageHistory(U16BIT page_number)
{
   FUNCTION_START(NotifyRequestToPageHistory);

   page_history_request_page_number = page_number;

   FUNCTION_FINISH(NotifyRequestToPageHistory);
}

/**
 *

 *

 *
 * @param   U16BIT marker -
 * @param   BOOLEAN advance -
 *
 * @return   
 *
 */
static U16BIT MoveHistoryMarker(U16BIT marker, BOOLEAN advance)
{
   FUNCTION_START(MoveHistoryMarker);

   if (advance == TRUE)
   {
      marker++;
      if (marker >= MAX_PAGE_HISTORY)
      {
         marker = 0;
      }
   }
   else
   {
      if (marker == 0)
      {
         marker = MAX_PAGE_HISTORY - 1;
      }
      else
      {
         marker--;
      }
   }

   FUNCTION_FINISH(MoveHistoryMarker);

   return(marker);
}

/**
 *

 *
 * @brief   Called from the display update notification mechanism to ensure that page numbers
 *                 are added to the page history buffer.
 *                 NB - pages are only added if specifically requested by the user - this is denoted
 *                      by a prior call to NotifyRequestToPageHistory()
 *
 * @param   U16BIT page_number - the page to be potentially added.
 *
 *

 *
 */
static void NotifyDisplayToPageHistory(U16BIT page_number)
{
   FUNCTION_START(NotifyDisplayToPageHistory);

   // NB - if prior calls have been made to GetPreviousPageFromHistory() then we will reset the
   //      high-water-mark to a new position, potentially reducing the page history.

   if (page_number == page_history_request_page_number)
   {
      // If the page number being displayed is as a result of an explicit request, then we add
      // the page number to the history.
      page_history_array[page_history_hwm] = page_number;

      // Advance the high-water-mark
      page_history_hwm = MoveHistoryMarker(page_history_hwm, TRUE);

      // If we have collided with the low-water-mark, then force it's advancement, thus losing
      // oldest record in history.
      if (page_history_hwm == page_history_lwm)
      {
         page_history_lwm = MoveHistoryMarker(page_history_lwm, TRUE);
      }
      // Now adjsut the middle water-mark.
      page_history_mwm = page_history_hwm;

      // Reset the requested page record.
      page_history_request_page_number = 0xffff;
   }

   FUNCTION_FINISH(NotifyDisplayToPageHistory);
}

/**
 *

 *

 *
 * @param   U16BIT* page_number -
 *
 * @return   
 *
 */
static BOOLEAN GetPreviousPageFromHistory(U16BIT *page_number)
{
   U16BIT temp_mwm;
   BOOLEAN retval;

   FUNCTION_START(GetPreviousPageFromHistory);

   retval = FALSE;

   if (page_history_hwm != page_history_lwm)
   {
      temp_mwm = MoveHistoryMarker(page_history_mwm, FALSE);

      if (temp_mwm != page_history_lwm)
      {
         temp_mwm = MoveHistoryMarker(temp_mwm, FALSE);

         *page_number = page_history_array[temp_mwm];

         page_history_mwm = MoveHistoryMarker(temp_mwm, TRUE);

         retval = TRUE;
      }
   }

   FUNCTION_FINISH(GetPreviousPageFromHistory);

   return(retval);
}

/**
 *

 *

 *
 * @param   U16BIT* page_number -
 *
 * @return   
 *
 */
static BOOLEAN GetNextPageFromHistory(U16BIT *page_number)
{
   BOOLEAN retval;

   FUNCTION_START(GetNextPageFromHistory);

   retval = FALSE;

   if (page_history_mwm != page_history_hwm)
   {
      *page_number = page_history_array[page_history_mwm];

      page_history_mwm = MoveHistoryMarker(page_history_mwm, TRUE);

      retval = TRUE;
   }

   FUNCTION_FINISH(GetNextPageFromHistory);

   return(retval);
}

/**
 *

 *
 * @brief   Called when all the data has been collated for a TeleText page.
 *                   The page will be assessed to determine if an existing cache record requires
 *                   updating, or if a corresponding page request has been made. In either which
 *                   case a new page record is added to the cache.
 *                   If there is no requirement to cache the passed page, the function returns and
 *                   the data is ignored.
 *                   In the cases whereby the data of the presently displayed page is updated in the
 *                   cache, the display update task will be informed of this by the return value as
 *                   it continuously polls the function GetParsedPage( ) - a consequence of the data
 *                   being available is that the display update task will then parse the relevant
 *                   page content, and render it to the user interface.
 *
 * @param   S_MAGAZINE_PAGE* collated_page_ptr - all the packets for the complete Teletext
 *                                                      page.
 * @param   U8BIT magazine_number - the magzine index of the collated page (0 to 7)
 *

 *
 */
static void NotifyPageCollated(S_MAGAZINE_PAGE *collated_page_ptr, U8BIT magazine_number,
   BOOLEAN pts_valid, U8BIT *pts)
{
   BOOLEAN update_carousel_display;
   S_PAGE_REFERENCE_DATA page_reference_data;

   FUNCTION_START(NotifyPageCollated);

   if (GetPageReference(collated_page_ptr->header, &page_reference_data, PAGE_REFERENCE_TYPE_HEADER,
          magazine_number) == TRUE)
   {
      if (page_reference_data.page_offset == 0xff)
      {
         if (page_reference_data.page_sub_code <= 0x3f7e)
         {
            CheckTimeFillingPacket(collated_page_ptr, page_reference_data.page_number);
         }
      }
      else
      {
#ifdef EBU_DEBUG
         if ((page_display_info.page_number >> 8) == magazine_number)
         {
            EBU_DBG("PageCollated: #=%03x:%04x, display_data_defined=0x%08x",
                     page_reference_data.page_number, page_reference_data.page_sub_code,
                     collated_page_ptr->display_data_defined_mask);
         }
#endif

         CheckHeaderClockContent(collated_page_ptr);

         // Mark the relevant bit in the byte array used to represent each of the 256 pages
         // within this magazine.
         magazine_array[magazine_number].m_cache_valid_page_array[page_reference_data.page_offset >> 3] |=
            1 << (page_reference_data.page_offset & 0x07);

         // Check if the 'Interrupted Sequnce' mode is cleared.
         // If so, then the displayed page index record for this magazine is updated.
         if (page_reference_data.interrupted_sequence == FALSE)
         {
            if (page_control_info.state != PAGE_CONTROL_STATE_HOLD)
            {
               update_carousel_display = FALSE;

               if (page_request_info.target_page_number)
               {
                  if (page_request_info.target_page_sub_code == PAGE_SUB_CODE_UNDEFINED)
                  {
                     if (page_request_info.target_page_number != page_display_info.page_number)
                     {
                        if (page_reference_data.magazine_serial == TRUE)
                        {
                           update_carousel_display = TRUE;
                        }
                        else
                        {
                           if (GetMagazineIndexFromPageNumber(page_reference_data.page_number) ==
                               GetMagazineIndexFromPageNumber(page_request_info.target_page_number))
                           {
                              update_carousel_display = TRUE;
                           }
                        }
                     }
                  }
                  else
                  {
                     if ((page_request_info.target_page_number != page_display_info.page_number) ||
                         (page_request_info.target_page_sub_code != page_display_info.page_sub_code))
                     {
                        if (page_reference_data.magazine_serial == TRUE)
                        {
                           update_carousel_display = TRUE;
                        }
                        else
                        {
                           if (GetMagazineIndexFromPageNumber(page_reference_data.page_number) ==
                               GetMagazineIndexFromPageNumber(page_request_info.target_page_number))
                           {
                              update_carousel_display = TRUE;
                           }
                        }
                     }
                  }
               }
               else
               {
                  update_carousel_display = TRUE;
               }

               if (update_carousel_display == TRUE && show_header_row == TRUE)
               {
                  STB_OSSemaphoreWait(page_display_info.carousel_page_semaphore);

                  page_display_info.carousel_page_number = page_reference_data.page_number;

                  if (collated_page_ptr->is_header_defined == TRUE)
                  {
                     memcpy(page_display_info.carousel_page_header, collated_page_ptr->header, MAX_COLUMN);
                  }
                  page_display_info.carousel_page_updated = TRUE;

                  STB_OSSemaphoreSignal(page_display_info.carousel_page_semaphore);
               }
            }
         }

         CheckPageCaching(collated_page_ptr, &page_reference_data, pts_valid, pts);
      }
   }

   FUNCTION_FINISH(NotifyPageCollated);
}

/**
 *

 *
 * @brief   Called from STB_EBUTT_Initialise( ) to initilaise the semaphore and task thread
 *                 used to ??????.
 *

 *
 * @return   BOOLEAN - TRUE if initialised successfully, FALSE otherwise.
 *
 */
static BOOLEAN InitialiseDisplayTask(void)
{
   BOOLEAN retval;

   FUNCTION_START(InitialiseDisplayTask);

   retval = FALSE;

   if (display_task_semaphore == NULL)
   {
      display_task_semaphore = STB_OSCreateSemaphore();
   }
   if (display_task_semaphore != NULL)
   {
      STB_OSSemaphoreWait(display_task_semaphore);

      display_task_ptr = STB_OSCreateTask(DisplayUpdateTask, NULL, DISPLAY_TASK_STACK_SIZE,
            DISPLAY_TASK_PRIORITY, (U8BIT *)"EBUTT_DISPLAY");
      if (display_task_ptr)
      {
         retval = TRUE;
      }
   }

   FUNCTION_FINISH(InitialiseDisplayTask);

   return(retval);
}

/**
 *

 *
 * @brief   Called from STB_EBUTT_Kill( ) to free resources for the Teletext page display
 *                 update task.
 *

 *

 *
 */
static void KillDisplayTask(void)
{
   FUNCTION_START(KillDisplayTask);

   if (display_task_ptr != NULL)
   {
      STB_OSSemaphoreWait(display_task_semaphore);

      STB_OSDestroyTask(display_task_ptr);
      display_task_ptr = NULL;
   }

   FUNCTION_FINISH(KillDisplayTask);
}

/**
 *

 *

 *
 * @param   U8BIT character_index -
 * @param   U8BIT* subset_offset -
 *
 * @return   
 *
 */
static BOOLEAN IsNationalOptionSubsetIndex(U8BIT character_index, U8BIT *subset_offset)
{
   U8BIT option_subset;
   BOOLEAN retval;

   FUNCTION_START(IsNationalOptionSubsetIndex);

   retval = FALSE;

   if ((character_index >= 0x20) &&
       (character_index <= 0x7f))
   {
      option_subset = option_subset_array[character_index - 0x20];

      if (option_subset)
      {
         *subset_offset = subset_offset_array[option_subset];

         retval = TRUE;
      }
   }

   FUNCTION_FINISH(IsNationalOptionSubsetIndex);

   return(retval);
}

/**
 *

 *

 *
 * @param   U32BIT triplet_value -
 * @param   U8BIT* address -
 * @param   U8BIT* mode -
 * @param   U8BIT* data -
 *

 *
 */
static void GetTripletFields(U32BIT triplet_value, U8BIT *address, U8BIT *mode, U8BIT *data)
{
   FUNCTION_START(GetTripletFields);

   *address = (U8BIT) (triplet_value & 0x00003fL);
   *mode = (U8BIT)((triplet_value & 0x0007c0L) >> 6L);
   *data = (U8BIT)((triplet_value & 0x03f800L) >> 11L);

   FUNCTION_FINISH(GetTripletFields);
}

/**
 *

 *
 * @brief   Parses the packet data relating to the page display.
 *                 The resulting charactyer map gives absolute offsets to the required bitmaps
 *                 within the specified font, and palette LUT definitions.
 *
 * @param   S_PAGE_DISPLAY_INFO* display_info_ptr - the collated page to be decoded.
 * @param   BOOLEAN enforce_header_display -
 *

 *
 */
static void DecodeTeletextPageBody(S_PAGE_DISPLAY_INFO *display_info_ptr, BOOLEAN enforce_header_display)
{
   U8BIT foreground_colour, background_colour, foreground_index, background_index;
   U8BIT default_mapping, second_mapping, hold_mosaics_char;
   U8BIT subset_offset, current_source_char, previous_source_char;
   U8BIT default_row_background_colour;
   U8BIT triplet_address, triplet_mode, triplet_data;
   U16BIT row, column, last_column;
   U16BIT packet_index, triplet_index, font_index, character_offset, diacritic_offset, equivalence_index;
   U32BIT triplet_value;
   U8BIT *source_data_ptr;
   U8BIT *triplet_ptr;
   BOOLEAN double_height, flashing, concealed, boxed, boxing_required;
   BOOLEAN mosaic_enabled, contiguous_mosaic_enabled, hold_mosaic_enabled;
   BOOLEAN header_required, body_required, valid_row;
   BOOLEAN default_row_background_colour_defined, double_height_in_previous_line, terminated;
   const S_CHARACTER_SET_MAPPING *default_mapping_ptr;
   const S_CHARACTER_SET_MAPPING *second_mapping_ptr;
   const S_CHARACTER_SET_MAPPING *current_mapping_ptr;
   S_TELETEXT_CHARACTER *destination_data_ptr;

   static const U8BIT diacritic_substitution_offset[16] = { 0, 0x30, 0x30, 0x20, 0x20, 0x20, 0x30, 0x20,
                                                            0x20, 0x30, 0, 0x20, 0, 0, 0, 0x30 };

   FUNCTION_START(DecodeTeletextPageBody);

   current_mapping_ptr = NULL;

   // Get the default and second character font set mappings for this page.
   GetBasicPageDisplayInfo(display_info_ptr, &default_mapping, &second_mapping, &boxing_required,
      &header_required, &body_required,
      &default_row_background_colour_defined,
      &default_row_background_colour);

   default_mapping_ptr = &character_set_mapping_array[default_mapping];
   if (second_mapping == 0xff)
   {
      second_mapping_ptr = NULL;
   }
   else
   {
      second_mapping_ptr = &character_set_mapping_array[second_mapping];
   }

   display_info_ptr->render_required = TRUE;

   if ((boxing_required || !show_header_row) && (page_control_info.state == PAGE_CONTROL_STATE_IDLE))
   {
      display_info_ptr->screen_colour = PALETTE_BACKGROUND_TRANSPARENT;
   }
   else
   {
      display_info_ptr->screen_colour = PALETTE_BACKGROUND_BLACK;
   }

   display_info_ptr->clear_screen = TRUE;

   display_info_ptr->palette_update_required = TRUE;

   double_height_in_previous_line = FALSE;

   display_info_ptr->palette_attribute_array[0].effect_mask = PALETTE_EFFECT_UNDEFINED;
   display_info_ptr->palette_alias_array[0].fore_colour = PALETTE_BACKGROUND_TRANSPARENT;

   for (row = 0; row <= 24; row++)
   {
      valid_row = FALSE;

      if (row == 0)
      {
         if (display_info_ptr->page_source.is_header_defined == TRUE)
         {
            if ((header_required == TRUE) ||
                (enforce_header_display == TRUE))
            {
               valid_row = TRUE;
            }
         }
      }
      else
      {
         if ((double_height_in_previous_line == FALSE) &&
             (body_required == TRUE) &&
             (display_info_ptr->page_source.display_data_defined_mask & (1 << row)))
         {
            valid_row = TRUE;
         }
      }

      if (valid_row == TRUE)
      {
         display_info_ptr->valid_row[row] = TRUE;
         display_info_ptr->render_row[row] = TRUE;

         // We are displaying a new row of characters, so set all the status flags to their
         // default settings.
         current_mapping_ptr = default_mapping_ptr;

         if (row == 0)
         {
            if (boxing_required == TRUE)
            {
               for (column = 0; column != 8; column++)
               {
                  display_info_ptr->character_map[0][column].foreground_index = PALETTE_FOREGROUND_UNBOXED;
               }
            }
            else
            {
               for (column = 0; column != 8; column++)
               {
                  display_info_ptr->character_map[0][column].background_index = PALETTE_BACKGROUND_BLACK;
               }
            }

            column = 8;
            if (display_info_ptr->page_clock_valid == TRUE)
            {
               last_column = 39;
            }
            else
            {
               last_column = 31;
            }
         }
         else
         {
            column = 0;
            last_column = 39;
         }
         foreground_colour = 7;
         if (default_row_background_colour_defined == TRUE)
         {
            background_colour = default_row_background_colour;
         }
         else
         {
            background_colour = 0;
         }

         double_height = FALSE;
         flashing = FALSE;
         concealed = FALSE;
         boxed = FALSE;
         mosaic_enabled = FALSE;
         contiguous_mosaic_enabled = TRUE;
         hold_mosaic_enabled = FALSE;
         hold_mosaics_char = 0; // white-space
         previous_source_char = 0;

         GetCharacterPaletteIndexes(display_info_ptr,
            &foreground_index, &background_index,
            foreground_colour, background_colour,
            flashing, concealed,
            boxing_required, boxed, FALSE);

         if (row == 0)
         {
            source_data_ptr = &display_info_ptr->page_source.header[8];
            destination_data_ptr = &display_info_ptr->character_map[row][8];
         }
         else
         {
            source_data_ptr = display_info_ptr->page_source.display_data[row];
            destination_data_ptr = display_info_ptr->character_map[row];
         }


         while (column <= last_column)
         {
            // Default setting for the caracter map entry.
            if (mosaic_enabled == TRUE)
            {
               if (hold_mosaic_enabled == TRUE)
               {
                  destination_data_ptr->info = (FONT_INDEX_G1_BLOCK_MOSAICS_SET << 8) +
                     hold_mosaics_char;
               }
               else
               {
                  destination_data_ptr->info = FONT_INDEX_G1_BLOCK_MOSAICS_SET << 8;
               }

               if (contiguous_mosaic_enabled == FALSE)
               {
                  destination_data_ptr->info |= TELETEXT_CHARACTER_SEPERATED_MOSAIC;
               }
            }
            else
            {
               if (current_mapping_ptr->G0_index == FONT_INDEX_UNDEFINED)
               {
                  destination_data_ptr->info = FONT_INDEX_LATIN_G0_SET << 8;
               }
               else
               {
                  destination_data_ptr->info = current_mapping_ptr->G0_index << 8;
               }
            }

            if (double_height == TRUE)
            {
               destination_data_ptr->info |= TELETEXT_CHARACTER_DOUBLE_HEIGHT;
            }

            destination_data_ptr->diacritic_info = 0;

            destination_data_ptr->foreground_index = foreground_index;
            destination_data_ptr->background_index = background_index;

            if (GetParity7Byte(*source_data_ptr, &current_source_char) == TRUE)
            {
               switch (current_source_char & 0x70)
               {
                  case 0x00: // Standard spacing attributes
                  {
                     switch (current_source_char & 0x0f)
                     {
                        case 0x08: // Flash ("set after")
                        {
                           flashing = TRUE;
                           GetCharacterPaletteIndexes(display_info_ptr,
                              &foreground_index, &background_index,
                              foreground_colour, background_colour,
                              flashing, concealed,
                              boxing_required, boxed, FALSE);

                           display_info_ptr->flash_required = TRUE;
                           break;
                        }
                        case 0x09: // Steady ("set at")
                        {
                           flashing = FALSE;
                           GetCharacterPaletteIndexes(display_info_ptr,
                              &foreground_index, &background_index,
                              foreground_colour, background_colour,
                              flashing, concealed,
                              boxing_required, boxed, FALSE);

                           destination_data_ptr->foreground_index = foreground_index;
                           break;
                        }
                        case 0x0a: // End Box ("set after")
                        {
                           if ((boxing_required == TRUE) &&
                               (boxed == TRUE))
                           {
                              boxed = FALSE;
                              GetCharacterPaletteIndexes(display_info_ptr,
                                 &foreground_index, &background_index,
                                 foreground_colour, background_colour,
                                 flashing, concealed,
                                 boxing_required, boxed, FALSE);
                           }
                           break;
                        }
                        case 0x0b: // Start Box ("set after")
                        {
                           if ((boxing_required == TRUE) &&
                               (boxed == FALSE))
                           {
                              if (previous_source_char == 0x0b)
                              {
                                 boxed = TRUE;
                                 GetCharacterPaletteIndexes(display_info_ptr,
                                    &foreground_index, &background_index,
                                    foreground_colour, background_colour,
                                    flashing, concealed,
                                    boxing_required, boxed, FALSE);
                              }
                           }
                           break;
                        }
                        case 0x0c: // Normal Size ("set at")
                        {
                           if (double_height == TRUE)
                           {
                              double_height = FALSE;
                              destination_data_ptr->info &= ~TELETEXT_CHARACTER_DOUBLE_HEIGHT;

                              hold_mosaics_char = 0; // white-space
                           }
                           break;
                        }
                        case 0x0d: // Double Height ("set after")
                        {
                           if (double_height == FALSE)
                           {
                              if ((row > 0) && (row < 24))
                              {
                                 double_height = TRUE;
                                 double_height_in_previous_line = TRUE;
                              }
                              hold_mosaics_char = 0; // white-space
                           }
                           break;
                        }
                        case 0x0e: // Double Width ("set after")
                        case 0x0f: // Double Size ("set after")
                        {
                           // Ignored by this decoder!
                           break;
                        }
                        default: // 0x00 to 0x07 - Alpha Black to Alpha White ("set after")
                        {
                           foreground_colour = current_source_char & 0x07;
                           concealed = FALSE;
                           mosaic_enabled = FALSE;
                           hold_mosaic_enabled = FALSE;
                           hold_mosaics_char = 0; // white-space

                           GetCharacterPaletteIndexes(display_info_ptr,
                              &foreground_index, &background_index,
                              foreground_colour, background_colour,
                              flashing, concealed,
                              boxing_required, boxed, FALSE);
                           break;
                        }
                     }
                     break;
                  }
                  case 0x10:
                  {
                     switch (current_source_char & 0x0f)
                     {
                        case 0x08: // Conceal ("set at")
                        {
                           concealed = TRUE;
                           GetCharacterPaletteIndexes(display_info_ptr,
                              &foreground_index, &background_index,
                              foreground_colour, background_colour,
                              flashing, concealed,
                              boxing_required, boxed, FALSE);

                           destination_data_ptr->foreground_index = foreground_index;

                           display_info_ptr->reveal_available = TRUE;
                           break;
                        }
                        case 0x09: // Contiguous Mosiac Graphics ("set at")
                        {
                           contiguous_mosaic_enabled = TRUE;

                           destination_data_ptr->info &= ~TELETEXT_CHARACTER_SEPERATED_MOSAIC;
                           break;
                        }
                        case 0x0a: // Seperated Mosiac Graphics ("set at")
                        {
                           contiguous_mosaic_enabled = FALSE;

                           destination_data_ptr->info |= TELETEXT_CHARACTER_SEPERATED_MOSAIC;
                           break;
                        }
                        case 0x0b: // ASCII 0x1b - ESC charecter used to toggle between default and
                                   //              second G0 sets.
                        {
                           if (second_mapping_ptr != NULL)
                           {
                              if (current_mapping_ptr == default_mapping_ptr)
                              {
                                 current_mapping_ptr = second_mapping_ptr;
                              }
                              else
                              {
                                 current_mapping_ptr = default_mapping_ptr;
                              }
                           }
                           break;
                        }
                        case 0x0c: // Black Background ("set at")
                        {
                           if (default_row_background_colour_defined == TRUE)
                           {
                              background_colour = default_row_background_colour;
                           }
                           else
                           {
                              background_colour = 0;
                           }

                           GetCharacterPaletteIndexes(display_info_ptr,
                              &foreground_index, &background_index,
                              foreground_colour, background_colour,
                              flashing, concealed,
                              boxing_required, boxed, FALSE);

                           // NB - the forcground colour is also set here in case a new flashing or
                           //      conceal attrubute has been genereated.
                           destination_data_ptr->foreground_index = foreground_index;
                           destination_data_ptr->background_index = background_index;
                           break;
                        }
                        case 0x0d: // New Background ("set at")
                        {
                           background_colour = foreground_colour;

                           GetCharacterPaletteIndexes(display_info_ptr,
                              &foreground_index, &background_index,
                              foreground_colour, background_colour,
                              flashing, concealed,
                              boxing_required, boxed, FALSE);

                           // NB - the forcground colour is also set here in case a new flashing or
                           //      conceal attrubute has been genereated.
                           destination_data_ptr->foreground_index = foreground_index;
                           destination_data_ptr->background_index = background_index;
                           break;
                        }
                        case 0x0e: // Hold Mosaics ("set at")
                        {
                           if (hold_mosaic_enabled == FALSE)
                           {
                              destination_data_ptr->info &= ~(TELETEXT_CHARACTER_FONT_INDEX_MASK |
                                                              TELETEXT_CHARACTER_BITMAP_INDEX_MASK);
                              destination_data_ptr->info |= (FONT_INDEX_G1_BLOCK_MOSAICS_SET << 8) +
                                 hold_mosaics_char;

                              hold_mosaic_enabled = TRUE;
                           }
                           break;
                        }
                        case 0x0f: // Release Mosaics ("set after")
                        {
                           if (mosaic_enabled == TRUE)
                           {
                              hold_mosaic_enabled = FALSE;
                           }
                           break;
                        }
                        default: // 0x0 to 0x07 - Mosaic Black to Mosaic White ("set after")
                        {
                           foreground_colour = current_source_char & 0x07;
                           concealed = FALSE;
                           mosaic_enabled = TRUE;

                           GetCharacterPaletteIndexes(display_info_ptr,
                              &foreground_index, &background_index,
                              foreground_colour, background_colour,
                              flashing, concealed,
                              boxing_required, boxed, FALSE);
                           break;
                        }
                     }
                     break;
                  }
                  default: // 0x20 to 0x7f - Font character
                  {
                     destination_data_ptr->info &= ~TELETEXT_CHARACTER_BITMAP_INDEX_MASK;
                     destination_data_ptr->info |= current_source_char - 0x20;

                     if (mosaic_enabled == TRUE)
                     {
                        if (((current_source_char & 0x70) == 0x40) ||
                            ((current_source_char & 0x70) == 0x50))
                        {
                           if ((IsNationalOptionSubsetIndex(current_source_char, &subset_offset) == TRUE) &&
                               (current_mapping_ptr->national_option_subset_index != NATIONALITY_INDEX_UNDEFINED))
                           {
                              destination_data_ptr->info &= ~(TELETEXT_CHARACTER_FONT_INDEX_MASK |
                                                              TELETEXT_CHARACTER_BITMAP_INDEX_MASK |
                                                              TELETEXT_CHARACTER_SEPERATED_MOSAIC);

                              destination_data_ptr->info |= FONT_INDEX_NATIONAL_OPTION_SUBSET << 8;

                              destination_data_ptr->info |= subset_offset +
                                 current_mapping_ptr->national_option_subset_index;
                           }
                           else
                           {
                              destination_data_ptr->info &= ~(TELETEXT_CHARACTER_FONT_INDEX_MASK |
                                                              TELETEXT_CHARACTER_SEPERATED_MOSAIC);

                              if (current_mapping_ptr->G0_index == FONT_INDEX_UNDEFINED)
                              {
                                 destination_data_ptr->info |= FONT_INDEX_LATIN_G0_SET << 8;
                              }
                              else
                              {
                                 destination_data_ptr->info |= current_mapping_ptr->G0_index << 8;
                              }
                           }
                        }
                        else
                        {
                           hold_mosaics_char = current_source_char - 0x20;
                        }
                     }
                     else
                     {
                        if ((current_mapping_ptr->G0_index == FONT_INDEX_LATIN_G0_SET) ||
                            (current_mapping_ptr->G0_index == FONT_INDEX_UNDEFINED))
                        {
                           if ((IsNationalOptionSubsetIndex(current_source_char, &subset_offset) == TRUE) &&
                               (current_mapping_ptr->national_option_subset_index != NATIONALITY_INDEX_UNDEFINED))
                           {
                              destination_data_ptr->info &= ~(TELETEXT_CHARACTER_FONT_INDEX_MASK |
                                                              TELETEXT_CHARACTER_BITMAP_INDEX_MASK);

                              destination_data_ptr->info |= FONT_INDEX_NATIONAL_OPTION_SUBSET << 8;

                              destination_data_ptr->info |= subset_offset +
                                 current_mapping_ptr->national_option_subset_index;
                           }
                        }
                     }

                     break;
                  }
               }
            }

            previous_source_char = current_source_char;
            source_data_ptr++;
            destination_data_ptr++;
            column++;
         }

         if (row == 0)
         {
            if (boxing_required == FALSE)
            {
               for (column = last_column + 1; column != MAX_COLUMN; column++)
               {
                  display_info_ptr->character_map[0][column].foreground_index =
                     display_info_ptr->character_map[0][31].foreground_index;
                  display_info_ptr->character_map[0][column].background_index =
                     display_info_ptr->character_map[0][31].background_index;
               }
            }
         }
      }
      else
      {
         if (double_height_in_previous_line == TRUE)
         {
            for (column = 0; column != MAX_COLUMN; column++)
            {
               display_info_ptr->character_map[row][column].info = 0;
               display_info_ptr->character_map[row][column].foreground_index =
                  display_info_ptr->character_map[row - 1][column].foreground_index;
               display_info_ptr->character_map[row][column].background_index =
                  display_info_ptr->character_map[row - 1][column].background_index;
            }

            display_info_ptr->valid_row[row] = TRUE;
            display_info_ptr->render_row[row] = TRUE;

            double_height_in_previous_line = FALSE;
         }
         else
         {
            DBGPRINT("Clear row %d  [%d,%d]", row, display_info_ptr->valid_row[row], display_info_ptr->render_row[row]);
            // No data for this row, so clear it.
            display_info_ptr->valid_row[row] = FALSE;
         }
      }
   }

   row = 0;
   column = 0;
   terminated = FALSE;

   if (display_info_ptr->page_source.basic_enhancement_data_defined_mask)
   {
      for (packet_index = 0; (packet_index != 16) && (terminated == FALSE); packet_index++)
      {
         if (display_info_ptr->page_source.basic_enhancement_data_defined_mask & (1 << packet_index))
         {
            triplet_ptr = &display_info_ptr->page_source.basic_enhancement_data[packet_index][1];

            for (triplet_index = 0; (triplet_index != 13) && (terminated == FALSE); triplet_index++)
            {
               if (GetHammimg24Value((void *)triplet_ptr, &triplet_value) == TRUE)
               {
                  GetTripletFields(triplet_value, &triplet_address, &triplet_mode, &triplet_data);

                  if (triplet_address >= MAX_COLUMN)
                  {
                     // Row address group
                     switch (triplet_mode)
                     {
                        case 4: // Set Active Position
                        {
                           if (triplet_address == MAX_COLUMN)
                           {
                              row = 24;
                           }
                           else
                           {
                              row = triplet_address - MAX_COLUMN;
                           }
                           if (triplet_data < MAX_COLUMN)
                           {
                              column = triplet_data;
                           }
                           break;
                        }
                        case 7: // Address Display Row 0
                        {
                           row = 0;
                           column = 0;
                           break;
                        }
                        case 31: // Termination Marker
                        {
                           terminated = TRUE;
                           break;
                        }
                     }
                  }
                  else
                  {
                     // Column address group
                     switch (triplet_mode)
                     {
                        case 1: // Block Mosaic Character from the G1 set
                        {
                           column = triplet_address;

                           if ((row != 0) ||
                               (column >= 8))
                           {
                              if (triplet_data >= 0x20)
                              {
                                 if (((triplet_data & 0x70) == 0x40) ||
                                     ((triplet_data & 0x70) == 0x50))
                                 {
                                    // Uses character from gurrent G0 set, so nothing to do here!
                                 }
                                 else
                                 {
                                    destination_data_ptr = &display_info_ptr->character_map[row][column];

                                    destination_data_ptr->info &= ~(TELETEXT_CHARACTER_FONT_INDEX_MASK |
                                                                    TELETEXT_CHARACTER_BITMAP_INDEX_MASK);

                                    destination_data_ptr->info |= FONT_INDEX_G1_BLOCK_MOSAICS_SET << 8;
                                    destination_data_ptr->info |= triplet_data - 0x20;
                                 }
                              }
                           }
                           break;
                        }
                        case 2: // Line Drawing or Smoothed Mosaic Character from the G3 set
                        {
                           column = triplet_address;

                           if ((row != 0) ||
                               (column >= 8))
                           {
                              if (triplet_data >= 0x20)
                              {
                                 if ((triplet_data != 0x6e) &&
                                     (triplet_data != 0x6f) &&
                                     (triplet_data != 0x7e) &&
                                     (triplet_data != 0x7f))
                                 {
                                    destination_data_ptr = &display_info_ptr->character_map[row][column];

                                    destination_data_ptr->info &= ~(TELETEXT_CHARACTER_FONT_INDEX_MASK |
                                                                    TELETEXT_CHARACTER_BITMAP_INDEX_MASK);

                                    destination_data_ptr->info |= FONT_INDEX_G3_SMOOTH_MOSAICS_LINE_DRAWING_SET << 8;
                                    destination_data_ptr->info |= triplet_data - 0x20;
                                 }
                              }
                           }
                           break;
                        }
                        case 9: // Character from the current G0 Set
                        {
                           column = triplet_address;

                           if (triplet_data >= 0x20)
                           {
                              destination_data_ptr = &display_info_ptr->character_map[row][column];

                              destination_data_ptr->info &= ~TELETEXT_CHARACTER_BITMAP_INDEX_MASK;

                              destination_data_ptr->info |= triplet_data - 0x20;
                           }
                           break;
                        }
                        case 15: // Character from the G2 Supplimentary Set
                        {
                           column = triplet_address;

                           if (triplet_data >= 0x20)
                           {
                              destination_data_ptr = &display_info_ptr->character_map[row][column];

                              destination_data_ptr->info &= ~(TELETEXT_CHARACTER_FONT_INDEX_MASK |
                                                              TELETEXT_CHARACTER_BITMAP_INDEX_MASK);

                              if (current_mapping_ptr->G2_index == FONT_INDEX_UNDEFINED)
                              {
                                 destination_data_ptr->info |= FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET << 8;
                              }
                              else
                              {
                                 destination_data_ptr->info |= current_mapping_ptr->G2_index << 8;
                              }
                              destination_data_ptr->info |= triplet_data - 0x20;
                           }
                           break;
                        }
                        default:
                        {
                           if (triplet_mode >= 16)
                           {
                              column = triplet_address;

                              destination_data_ptr = &display_info_ptr->character_map[row][column];

                              destination_data_ptr->info &= ~TELETEXT_CHARACTER_BITMAP_INDEX_MASK;

                              if (triplet_mode == 16)
                              {
                                 if (triplet_data == 0x2a)
                                 {
                                    font_index = (destination_data_ptr->info & TELETEXT_CHARACTER_FONT_INDEX_MASK) >> 8;

                                    if ((font_index == FONT_INDEX_LATIN_G0_SET) ||
                                        (font_index == FONT_INDEX_CYRILLIC_G0_SET_OPTION_1_SERBIAN_CROATIAN) ||
                                        (font_index == FONT_INDEX_CYRILLIC_G0_SET_OPTION_2_RUSSIAN_BULGARIAN) ||
                                        (font_index == FONT_INDEX_CYRILLIC_G0_SET_OPTION_3_UKRANIAN) ||
                                        (font_index == FONT_INDEX_GREEK_G0_SET) ||
                                        (font_index == FONT_INDEX_ARABIC_G0_SET) ||
                                        (font_index == FONT_INDEX_HEBREW_G0_SET))
                                    {
                                       destination_data_ptr->info &= ~TELETEXT_CHARACTER_FONT_INDEX_MASK;

                                       destination_data_ptr->info |= FONT_INDEX_LATIN_G0_SET << 8;

                                       destination_data_ptr->info |= 0x20; // '*' changed to '@'
                                    }
                                    else
                                    {
                                       destination_data_ptr->info |= 0x0a;
                                    }

                                    destination_data_ptr->diacritic_info = 0;
                                 }
                                 else if (triplet_data >= 0x20)
                                 {
                                    destination_data_ptr->info |= triplet_data - 0x20;

                                    destination_data_ptr->diacritic_info = 0;
                                 }
                              }
                              else
                              {
                                 if (triplet_data >= 0x20)
                                 {
                                    font_index = (destination_data_ptr->info & TELETEXT_CHARACTER_FONT_INDEX_MASK) >> 8;
                                    character_offset = triplet_data - 0x20;
                                    diacritic_offset = triplet_mode - 16;

                                    for (equivalence_index = 0; diacritic_equivalent_array[equivalence_index].font_index != FONT_INDEX_UNDEFINED; equivalence_index++)
                                    {
                                       if ((diacritic_equivalent_array[equivalence_index].font_index == font_index) &&
                                           (diacritic_equivalent_array[equivalence_index].character_offset == character_offset) &&
                                           (diacritic_equivalent_array[equivalence_index].diacritic_offset == diacritic_offset))
                                       {
                                          destination_data_ptr->info &= ~TELETEXT_CHARACTER_FONT_INDEX_MASK;

                                          destination_data_ptr->info |= diacritic_equivalent_array[equivalence_index].alternative_font_index << 8;

                                          destination_data_ptr->info |= diacritic_equivalent_array[equivalence_index].alternative_character_offset;

                                          destination_data_ptr->diacritic_info = 0;
                                          break;
                                       }
                                    }

                                    if (diacritic_equivalent_array[equivalence_index].font_index == FONT_INDEX_UNDEFINED)
                                    {
                                       if ((diacritic_substitutions_avaialable == TRUE) &&
                                           (diacritic_substitution_offset[diacritic_offset] != 0))
                                       {
                                          for (equivalence_index = 0; diacritic_substitution_array[equivalence_index].font_index != FONT_INDEX_UNDEFINED; equivalence_index++)
                                          {
                                             if ((diacritic_substitution_array[equivalence_index].font_index == font_index) &&
                                                 (diacritic_substitution_array[equivalence_index].character_offset == character_offset))
                                             {
                                                destination_data_ptr->info &= ~TELETEXT_CHARACTER_FONT_INDEX_MASK;

                                                destination_data_ptr->info |= FONT_INDEX_G1_BLOCK_MOSAICS_SET << 8;

                                                destination_data_ptr->info |= diacritic_substitution_offset[diacritic_offset] +
                                                   diacritic_substitution_array[equivalence_index].alternative_character_offset;

                                                if (current_mapping_ptr->G2_index == FONT_INDEX_UNDEFINED)
                                                {
                                                   destination_data_ptr->diacritic_info = FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET << 8;
                                                }
                                                else
                                                {
                                                   destination_data_ptr->diacritic_info = current_mapping_ptr->G2_index << 8;
                                                }
                                                destination_data_ptr->diacritic_info |= 0x20 + diacritic_offset;
                                                break;
                                             }
                                          }
                                       }
                                    }

                                    if (diacritic_substitution_array[equivalence_index].font_index == FONT_INDEX_UNDEFINED)
                                    {
                                       destination_data_ptr->info |= character_offset;

                                       if (current_mapping_ptr->G2_index == FONT_INDEX_UNDEFINED)
                                       {
                                          destination_data_ptr->diacritic_info = FONT_INDEX_LATIN_G2_SUPPLIMENTARY_SET << 8;
                                       }
                                       else
                                       {
                                          destination_data_ptr->diacritic_info = current_mapping_ptr->G2_index << 8;
                                       }
                                       destination_data_ptr->diacritic_info |= 0x20 + diacritic_offset;
                                    }
                                 }
                              }
                           }
                           break;
                        }
                     }
                  }
               }

               triplet_ptr += 3;
            }
         }
      }
   }

   FUNCTION_FINISH(DecodeTeletextPageBody);
}

/**
 *

 *

 *
 * @param   S_PAGE_DISPLAY_INFO* display_info_ptr - the collated page to be decoded.
 * @param   BOOLEAN video_mix_overridden -
 *

 *
 */
static void DecodeTeletextPageNumber(S_PAGE_DISPLAY_INFO *display_info_ptr, BOOLEAN video_mix_overridden)
{
   U16BIT column;
   BOOLEAN default_row_background_colour_defined;
   U8BIT default_row_background_colour;
   BOOLEAN set_background;

   static U8BIT index_digit_mapping[17] =
   {
      0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
      0X21, 0X22, 0X23, 0X24, 0X25, 0X26,
      0x0d
   };

   FUNCTION_START(DecodeTeletextPageNumber);

   GetBasicPageDisplayInfo(display_info_ptr, NULL, NULL, NULL, NULL, NULL,
      &default_row_background_colour_defined,
      &default_row_background_colour);

   display_info_ptr->render_required = TRUE;

   if (display_info_ptr->page_index_held == TRUE)
   {
      for (column = 0; column != 8; column++)
      {
         display_info_ptr->character_map[0][column].info = FONT_INDEX_LATIN_G0_SET << 8;
         display_info_ptr->character_map[0][column].foreground_index = PALETTE_FOREGROUND_RED;
         if (default_row_background_colour_defined == TRUE)
         {
            display_info_ptr->character_map[0][column].background_index = default_row_background_colour;
         }
         else
         {
            display_info_ptr->character_map[0][column].background_index = PALETTE_BACKGROUND_BLACK;
         }

         display_info_ptr->character_map[0][2].info |= 0x28;
         display_info_ptr->character_map[0][3].info |= 0x2f;
         display_info_ptr->character_map[0][4].info |= 0x2c;
         display_info_ptr->character_map[0][5].info |= 0x24;

         display_info_ptr->palette_update_required = TRUE;
      }
   }
   else
   {
      for (column = 0; column != 8; column++)
      {
         display_info_ptr->character_map[0][column].info = FONT_INDEX_LATIN_G0_SET << 8;
         display_info_ptr->character_map[0][column].foreground_index = PALETTE_FOREGROUND_WHITE;
         if ((display_info_ptr->video_mix_set == TRUE) &&
             (video_mix_overridden == TRUE))
         {
            display_info_ptr->character_map[0][column].background_index = PALETTE_BACKGROUND_BLACK_FIXED;
         }
         else
         {
            if (default_row_background_colour_defined == TRUE)
            {
               display_info_ptr->character_map[0][column].background_index = default_row_background_colour;
            }
            else
            {
               display_info_ptr->character_map[0][column].background_index = PALETTE_BACKGROUND_BLACK;
            }
         }
      }

      for (column = 0; column != 3; column++)
      {
         display_info_ptr->character_map[0][column + 3].info |= index_digit_mapping[display_info_ptr->page_index_digit[column] - PAGE_INDEX_DIGIT_0];
      }
   }

   /*
    * If the header is blank, except for the index number and this is the first time it is being
    * displayed the header will be incorrectly transparent and showing the video. Check to see
    * if there is any data in the display page or it's copy. If there isn't set the background
    * colour to be the index's background colour (so if black, it's black. If transparent it's
    * transparent etc.)
    */
   set_background = TRUE;
   for (column = 8; column != MAX_COLUMN; column++)
   {
      if ((display_info_ptr->character_map[0][column].info != 0x00) &&
          (page_display_info.character_map[0][column].info != 0x00))
      {
         set_background = FALSE;
      }
   }
   if (set_background != FALSE)
   {
      for (column = 8; column != MAX_COLUMN; column++)
      {
         display_info_ptr->character_map[0][column].foreground_index = display_info_ptr->character_map[0][0].foreground_index;
         display_info_ptr->character_map[0][column].background_index = display_info_ptr->character_map[0][0].background_index;
      }
   }

   display_info_ptr->valid_row[0] = TRUE;
   display_info_ptr->render_row[0] = TRUE;

   FUNCTION_FINISH(DecodeTeletextPageNumber);
}

/**
 *

 *

 *
 * @param   S_PAGE_DISPLAY_INFO* display_info_ptr - the collated page to be decoded.
 *

 *
 */
static void DecodeTeletextCarouselHeader(S_PAGE_DISPLAY_INFO *display_info_ptr)
{
   U8BIT foreground_colour, background_colour, foreground_index, background_index;
   U8BIT default_mapping, second_mapping, hold_mosaics_char;
   U8BIT subset_offset, current_source_char;
   U8BIT default_row_background_colour;
   U16BIT column;
   U8BIT *source_data_ptr;
   BOOLEAN flashing, concealed;
   BOOLEAN mosaic_enabled, contiguous_mosaic_enabled, hold_mosaic_enabled;
   BOOLEAN default_row_background_colour_defined;
   const S_CHARACTER_SET_MAPPING *default_mapping_ptr;
   const S_CHARACTER_SET_MAPPING *second_mapping_ptr;
   const S_CHARACTER_SET_MAPPING *current_mapping_ptr;
   S_TELETEXT_CHARACTER *destination_data_ptr;

   FUNCTION_START(DecodeTeletextCarouselHeader);

   // Get the default and second character font set mappings for this page.
   GetBasicPageDisplayInfo(display_info_ptr, &default_mapping, &second_mapping, NULL,
      NULL, NULL,
      &default_row_background_colour_defined,
      &default_row_background_colour);

   default_mapping_ptr = &character_set_mapping_array[default_mapping];
   if (second_mapping == 0xff)
   {
      second_mapping_ptr = NULL;
   }
   else
   {
      second_mapping_ptr = &character_set_mapping_array[second_mapping];
   }

   display_info_ptr->render_required = TRUE;
   display_info_ptr->palette_update_required = TRUE;

   display_info_ptr->valid_row[0] = TRUE;
   display_info_ptr->render_row[0] = TRUE;

   // We are displaying a new row of characters, so set all the status flags to their
   // default settings.
   current_mapping_ptr = default_mapping_ptr;

   foreground_colour = 2;
   if (default_row_background_colour_defined == TRUE)
   {
      background_colour = default_row_background_colour;
   }
   else
   {
      background_colour = 0;
   }

   flashing = FALSE;
   concealed = FALSE;
   mosaic_enabled = FALSE;
   contiguous_mosaic_enabled = TRUE;
   hold_mosaic_enabled = FALSE;
   hold_mosaics_char = 0; // white-space

   GetCharacterPaletteIndexes(display_info_ptr,
      &foreground_index, &background_index,
      foreground_colour, background_colour,
      flashing, concealed,
      FALSE, FALSE, TRUE);

   source_data_ptr = &display_info_ptr->carousel_page_header[8];
   destination_data_ptr = &display_info_ptr->character_map[0][8];

   for (column = 8; column != MAX_COLUMN; column++)
   {
      // Default setting for the caracter map entry.
      if (mosaic_enabled == TRUE)
      {
         if (hold_mosaic_enabled == TRUE)
         {
            destination_data_ptr->info = (FONT_INDEX_G1_BLOCK_MOSAICS_SET << 8) +
               hold_mosaics_char;
         }
         else
         {
            destination_data_ptr->info = FONT_INDEX_G1_BLOCK_MOSAICS_SET << 8;
         }

         if (contiguous_mosaic_enabled == FALSE)
         {
            destination_data_ptr->info |= TELETEXT_CHARACTER_SEPERATED_MOSAIC;
         }
      }
      else
      {
         if (current_mapping_ptr->G0_index == FONT_INDEX_UNDEFINED)
         {
            destination_data_ptr->info = FONT_INDEX_LATIN_G0_SET << 8;
         }
         else
         {
            destination_data_ptr->info = current_mapping_ptr->G0_index << 8;
         }
      }

      destination_data_ptr->foreground_index = foreground_index;
      destination_data_ptr->background_index = background_index;

      if (GetParity7Byte(*source_data_ptr, &current_source_char) == TRUE)
      {
         switch (current_source_char & 0x70)
         {
            case 0x00: // Standard spacing attributes
            {
               switch (current_source_char & 0x0f)
               {
                  case 0x08: // Flash ("set after")
                  {
                     flashing = TRUE;
                     GetCharacterPaletteIndexes(display_info_ptr,
                        &foreground_index, &background_index,
                        foreground_colour, background_colour,
                        flashing, concealed,
                        FALSE, FALSE, TRUE);
                     break;
                  }
                  case 0x09: // Steady ("set at")
                  {
                     flashing = FALSE;
                     GetCharacterPaletteIndexes(display_info_ptr,
                        &foreground_index, &background_index,
                        foreground_colour, background_colour,
                        flashing, concealed,
                        FALSE, FALSE, TRUE);

                     destination_data_ptr->foreground_index = foreground_index;
                     break;
                  }
                  case 0x0a: // End Box ("set after")
                  {
                     break;
                  }
                  case 0x0b: // Start Box ("set after")
                  {
                     break;
                  }
                  case 0x0c: // Normal Size ("set at")
                  {
                     hold_mosaics_char = 0; // white-space
                     break;
                  }
                  case 0x0d: // Double Height ("set after")
                  {
                     hold_mosaics_char = 0; // white-space
                     break;
                  }
                  case 0x0e: // Double Width ("set after")
                  case 0x0f: // Double Size ("set after")
                  {
                     // Ignored by this decoder!
                     break;
                  }
                  default: // 0x00 to 0x07 - Alpha Black to Alpha White ("set after")
                  {
                     foreground_colour = current_source_char & 0x07;
                     concealed = FALSE;
                     mosaic_enabled = FALSE;
                     hold_mosaic_enabled = FALSE;
                     hold_mosaics_char = 0; // white-space

                     GetCharacterPaletteIndexes(display_info_ptr,
                        &foreground_index, &background_index,
                        foreground_colour, background_colour,
                        flashing, concealed,
                        FALSE, FALSE, TRUE);
                     break;
                  }
               }
               break;
            }
            case 0x10:
            {
               switch (current_source_char & 0x0f)
               {
                  case 0x08: // Conceal ("set at")
                  {
                     concealed = TRUE;
                     GetCharacterPaletteIndexes(display_info_ptr,
                        &foreground_index, &background_index,
                        foreground_colour, background_colour,
                        flashing, concealed,
                        FALSE, FALSE, TRUE);

                     destination_data_ptr->foreground_index = foreground_index;
                     break;
                  }
                  case 0x09: // Contiguous Mosiac Graphics ("set at")
                  {
                     contiguous_mosaic_enabled = TRUE;

                     destination_data_ptr->info &= ~TELETEXT_CHARACTER_SEPERATED_MOSAIC;
                     break;
                  }
                  case 0x0a: // Seperated Mosiac Graphics ("set at")
                  {
                     contiguous_mosaic_enabled = FALSE;

                     destination_data_ptr->info |= TELETEXT_CHARACTER_SEPERATED_MOSAIC;
                     break;
                  }
                  case 0x0b: // ASCII 0x1b - ESC charecter used to toggle between default and
                             //              second G0 sets.
                  {
                     if (second_mapping_ptr != NULL)
                     {
                        if (current_mapping_ptr == default_mapping_ptr)
                        {
                           current_mapping_ptr = second_mapping_ptr;
                        }
                        else
                        {
                           current_mapping_ptr = default_mapping_ptr;
                        }
                     }
                     break;
                  }
                  case 0x0c: // Black Background ("set at")
                  {
                     if (default_row_background_colour_defined == TRUE)
                     {
                        background_colour = default_row_background_colour;
                     }
                     else
                     {
                        background_colour = 0;
                     }

                     GetCharacterPaletteIndexes(display_info_ptr,
                        &foreground_index, &background_index,
                        foreground_colour, background_colour,
                        flashing, concealed,
                        FALSE, FALSE, TRUE);

                     // NB - the forcground colour is also set here in case a new flashing or
                     //      conceal attrubute has been genereated.
                     destination_data_ptr->foreground_index = foreground_index;
                     destination_data_ptr->background_index = background_index;
                     break;
                  }
                  case 0x0d: // New Background ("set at")
                  {
                     background_colour = foreground_colour;

                     GetCharacterPaletteIndexes(display_info_ptr,
                        &foreground_index, &background_index,
                        foreground_colour, background_colour,
                        flashing, concealed,
                        FALSE, FALSE, TRUE);

                     // NB - the forcground colour is also set here in case a new flashing or
                     //      conceal attrubute has been genereated.
                     destination_data_ptr->foreground_index = foreground_index;
                     destination_data_ptr->background_index = background_index;
                     break;
                  }
                  case 0x0e: // Hold Mosaics ("set at")
                  {
                     if (mosaic_enabled == FALSE)
                     {
                        if (hold_mosaic_enabled == FALSE)
                        {
                           destination_data_ptr->info &= ~(TELETEXT_CHARACTER_FONT_INDEX_MASK |
                                                           TELETEXT_CHARACTER_BITMAP_INDEX_MASK);
                           destination_data_ptr->info |= (FONT_INDEX_G1_BLOCK_MOSAICS_SET << 8) +
                              hold_mosaics_char;

                           hold_mosaic_enabled = TRUE;
                        }
                     }
                     break;
                  }
                  case 0x0f: // Release Mosaics ("set after")
                  {
                     if (mosaic_enabled == TRUE)
                     {
                        hold_mosaic_enabled = FALSE;
                     }
                     break;
                  }
                  default: // 0x0 to 0x07 - Mosaic Black to Mosaic White ("set after")
                  {
                     foreground_colour = current_source_char & 0x07;
                     concealed = FALSE;
                     mosaic_enabled = TRUE;

                     GetCharacterPaletteIndexes(display_info_ptr,
                        &foreground_index, &background_index,
                        foreground_colour, background_colour,
                        flashing, concealed,
                        FALSE, FALSE, TRUE);
                     break;
                  }
               }
               break;
            }
            default: // 0x20 to 0x7f - Font character
            {
               destination_data_ptr->info &= ~TELETEXT_CHARACTER_BITMAP_INDEX_MASK;
               destination_data_ptr->info |= current_source_char - 0x20;

               if (mosaic_enabled == TRUE)
               {
                  if (((current_source_char & 0x70) == 0x40) ||
                      ((current_source_char & 0x70) == 0x50))
                  {
                     if ((IsNationalOptionSubsetIndex(current_source_char, &subset_offset) == TRUE) &&
                         (current_mapping_ptr->national_option_subset_index != NATIONALITY_INDEX_UNDEFINED))
                     {
                        destination_data_ptr->info &= ~(TELETEXT_CHARACTER_FONT_INDEX_MASK |
                                                        TELETEXT_CHARACTER_BITMAP_INDEX_MASK |
                                                        TELETEXT_CHARACTER_SEPERATED_MOSAIC);

                        destination_data_ptr->info |= FONT_INDEX_NATIONAL_OPTION_SUBSET << 8;

                        destination_data_ptr->info |= subset_offset +
                           current_mapping_ptr->national_option_subset_index;
                     }
                     else
                     {
                        destination_data_ptr->info &= ~(TELETEXT_CHARACTER_FONT_INDEX_MASK |
                                                        TELETEXT_CHARACTER_SEPERATED_MOSAIC);

                        if (current_mapping_ptr->G0_index == FONT_INDEX_UNDEFINED)
                        {
                           destination_data_ptr->info |= FONT_INDEX_LATIN_G0_SET << 8;
                        }
                        else
                        {
                           destination_data_ptr->info |= current_mapping_ptr->G0_index << 8;
                        }
                     }
                  }
                  else
                  {
                     hold_mosaics_char = current_source_char - 0x20;
                  }
               }
               else
               {
                  if ((current_mapping_ptr->G0_index == FONT_INDEX_LATIN_G0_SET) ||
                      (current_mapping_ptr->G0_index == FONT_INDEX_UNDEFINED))
                  {
                     if ((IsNationalOptionSubsetIndex(current_source_char, &subset_offset) == TRUE) &&
                         (current_mapping_ptr->national_option_subset_index != NATIONALITY_INDEX_UNDEFINED))
                     {
                        destination_data_ptr->info &= ~(TELETEXT_CHARACTER_FONT_INDEX_MASK |
                                                        TELETEXT_CHARACTER_BITMAP_INDEX_MASK);

                        destination_data_ptr->info |= FONT_INDEX_NATIONAL_OPTION_SUBSET << 8;

                        destination_data_ptr->info |= subset_offset +
                           current_mapping_ptr->national_option_subset_index;
                     }
                  }
               }
               break;
            }
         }
      }

      source_data_ptr++;
      destination_data_ptr++;
   }

   if (display_info_ptr->character_map[0][8].background_index == PALETTE_BACKGROUND_BLACK_FIXED)
   {
      for (column = 0; column != 8; column++)
      {
         display_info_ptr->character_map[0][column].background_index = PALETTE_BACKGROUND_BLACK_FIXED;
      }
      for (column = 32; column != MAX_COLUMN; column++)
      {
         display_info_ptr->character_map[0][column].background_index = PALETTE_BACKGROUND_BLACK_FIXED;
      }
   }

   FUNCTION_FINISH(DecodeTeletextCarouselHeader);
}

/**
 *

 *

 *
 * @param   S_PAGE_DISPLAY_INFO* display_info_ptr - the collated page to be decoded.
 * @param   BOOLEAN video_mix_overridden -
 *

 *
 */
static void DecodeTeletextClock(S_PAGE_DISPLAY_INFO *display_info_ptr, BOOLEAN video_mix_overridden)
{
   U8BIT foreground_colour, background_colour, foreground_index, background_index;
   U8BIT default_mapping, second_mapping, hold_mosaics_char;
   U8BIT subset_offset, current_source_char;
   U8BIT default_row_background_colour;
   U16BIT column;
   U8BIT *source_data_ptr;
   BOOLEAN flashing, concealed, header_required;
   BOOLEAN mosaic_enabled, contiguous_mosaic_enabled, hold_mosaic_enabled;
   BOOLEAN default_row_background_colour_defined;
   const S_CHARACTER_SET_MAPPING *default_mapping_ptr;
   const S_CHARACTER_SET_MAPPING *second_mapping_ptr;
   const S_CHARACTER_SET_MAPPING *current_mapping_ptr;
   S_TELETEXT_CHARACTER *destination_data_ptr;

   FUNCTION_START(DecodeTeletextClock);

   // Get the default and second character font set mappings for this page.
   GetBasicPageDisplayInfo(display_info_ptr, &default_mapping, &second_mapping, NULL,
      &header_required, NULL,
      &default_row_background_colour_defined,
      &default_row_background_colour);

   if (header_required == TRUE)
   {
      default_mapping_ptr = &character_set_mapping_array[default_mapping];
      if (second_mapping == 0xff)
      {
         second_mapping_ptr = NULL;
      }
      else
      {
         second_mapping_ptr = &character_set_mapping_array[second_mapping];
      }

      display_info_ptr->render_required = TRUE;
      display_info_ptr->palette_update_required = TRUE;

      display_info_ptr->valid_row[0] = TRUE;
      display_info_ptr->render_row[0] = TRUE;

      // We are displaying a new row of characters, so set all the status flags to their
      // default settings.
      current_mapping_ptr = default_mapping_ptr;

      foreground_colour = 2;
      background_colour = 0;

      flashing = FALSE;
      concealed = FALSE;
      mosaic_enabled = FALSE;
      contiguous_mosaic_enabled = TRUE;
      hold_mosaic_enabled = FALSE;
      hold_mosaics_char = 0; // white-space

      GetCharacterPaletteIndexes(display_info_ptr,
         &foreground_index, &background_index,
         foreground_colour, background_colour,
         flashing, concealed,
         FALSE, FALSE, video_mix_overridden);

      source_data_ptr = &display_info_ptr->time_filler_header_data[24];
      destination_data_ptr = &display_info_ptr->character_map[0][32];

      for (column = 32; column != MAX_COLUMN; column++)
      {
         // Default setting for the caracter map entry.
         if (mosaic_enabled == TRUE)
         {
            if (hold_mosaic_enabled == TRUE)
            {
               destination_data_ptr->info = (FONT_INDEX_G1_BLOCK_MOSAICS_SET << 8) +
                  hold_mosaics_char;
            }
            else
            {
               destination_data_ptr->info = FONT_INDEX_G1_BLOCK_MOSAICS_SET << 8;
            }

            if (contiguous_mosaic_enabled == FALSE)
            {
               destination_data_ptr->info |= TELETEXT_CHARACTER_SEPERATED_MOSAIC;
            }
         }
         else
         {
            if (current_mapping_ptr->G0_index == FONT_INDEX_UNDEFINED)
            {
               destination_data_ptr->info = FONT_INDEX_LATIN_G0_SET << 8;
            }
            else
            {
               destination_data_ptr->info = current_mapping_ptr->G0_index << 8;
            }
         }

         if (GetParity7Byte(*source_data_ptr, &current_source_char) == TRUE)
         {
            switch (current_source_char & 0x70)
            {
               case 0x00: // Standard spacing attributes
               {
                  switch (current_source_char & 0x0f)
                  {
                     case 0x08: // Flash ("set after")
                     {
                        flashing = TRUE;
                        GetCharacterPaletteIndexes(display_info_ptr,
                           &foreground_index, &background_index,
                           foreground_colour, background_colour,
                           flashing, concealed,
                           FALSE, FALSE, video_mix_overridden);
                        break;
                     }
                     case 0x09: // Steady ("set at")
                     {
                        flashing = FALSE;
                        GetCharacterPaletteIndexes(display_info_ptr,
                           &foreground_index, &background_index,
                           foreground_colour, background_colour,
                           flashing, concealed,
                           FALSE, FALSE, video_mix_overridden);

                        destination_data_ptr->foreground_index = foreground_index;
                        break;
                     }
                     case 0x0a: // End Box ("set after")
                     {
                        break;
                     }
                     case 0x0b: // Start Box ("set after")
                     {
                        break;
                     }
                     case 0x0c: // Normal Size ("set at")
                     {
                        break;
                     }
                     case 0x0d: // Double Height ("set after")
                     {
                        break;
                     }
                     case 0x0e: // Double Width ("set after")
                     case 0x0f: // Double Size ("set after")
                     {
                        // Ignored by this decoder!
                        break;
                     }
                     default: // 0x00 to 0x07 - Alpha Black to Alpha White ("set after")
                     {
                        //                     foreground_colour = current_source_char & 0x07;
                        //                     concealed = FALSE;
                        //                     mosaic_enabled = FALSE;
                        //                     hold_mosaic_enabled = FALSE;
                        //
                        //                     GetCharacterPaletteIndexes(display_info_ptr,
                        //                                                &foreground_index, &background_index,
                        //                                                foreground_colour, background_colour,
                        //                                                flashing, concealed,
                        //                                                FALSE, FALSE, video_mix_overridden);
                        break;
                     }
                  }
                  break;
               }
               case 0x10:
               {
                  switch (current_source_char & 0x0f)
                  {
                     case 0x08: // Conceal ("set at")
                     {
                        concealed = TRUE;
                        GetCharacterPaletteIndexes(display_info_ptr,
                           &foreground_index, &background_index,
                           foreground_colour, background_colour,
                           flashing, concealed,
                           FALSE, FALSE, video_mix_overridden);

                        destination_data_ptr->foreground_index = foreground_index;
                        break;
                     }
                     case 0x09: // Contiguous Mosiac Graphics ("set at")
                     {
                        contiguous_mosaic_enabled = TRUE;

                        destination_data_ptr->info &= ~TELETEXT_CHARACTER_SEPERATED_MOSAIC;
                        break;
                     }
                     case 0x0a: // Seperated Mosiac Graphics ("set at")
                     {
                        contiguous_mosaic_enabled = FALSE;

                        destination_data_ptr->info |= TELETEXT_CHARACTER_SEPERATED_MOSAIC;
                        break;
                     }
                     case 0x0b: // ASCII 0x1b - ESC charecter used to toggle between default and
                                //              second G0 sets.
                     {
                        if (second_mapping_ptr != NULL)
                        {
                           if (current_mapping_ptr == default_mapping_ptr)
                           {
                              current_mapping_ptr = second_mapping_ptr;
                           }
                           else
                           {
                              current_mapping_ptr = default_mapping_ptr;
                           }
                        }
                        break;
                     }
                     case 0x0c: // Black Background ("set at")
                     {
                        //                     if(default_row_background_colour_defined == TRUE)
                        //                     {
                        //                        background_colour = default_row_background_colour;
                        //                     }
                        //                     else
                        //                     {
                        //                        background_colour = 0;
                        //                     }
                        //
                        //                     GetCharacterPaletteIndexes(display_info_ptr,
                        //                                                &foreground_index, &background_index,
                        //                                                foreground_colour, background_colour,
                        //                                                flashing, concealed,
                        //                                                FALSE, FALSE, video_mix_overridden);

                        // NB - the forcground colour is also set here in case a new flashing or
                        //      conceal attrubute has been genereated.
                        //                     destination_data_ptr->foreground_index = foreground_index;
                        //                     destination_data_ptr->background_index = background_index;
                        break;
                     }
                     case 0x0d: // New Background ("set at")
                     {
                        //                     background_colour = foreground_colour;
                        //
                        //                     GetCharacterPaletteIndexes(display_info_ptr,
                        //                                                &foreground_index, &background_index,
                        //                                                foreground_colour, background_colour,
                        //                                                flashing, concealed,
                        //                                                FALSE, FALSE, video_mix_overridden);
                        //
                        //                     // NB - the forcground colour is also set here in case a new flashing or
                        //                     //      conceal attrubute has been genereated.
                        //                     destination_data_ptr->foreground_index = foreground_index;
                        //                     destination_data_ptr->background_index = background_index;
                        break;
                     }
                     case 0x0e: // Hold Mosaics ("set at")
                     {
                        if (mosaic_enabled == FALSE)
                        {
                           if (hold_mosaic_enabled == FALSE)
                           {
                              destination_data_ptr->info &= ~(TELETEXT_CHARACTER_FONT_INDEX_MASK |
                                                              TELETEXT_CHARACTER_BITMAP_INDEX_MASK);
                              destination_data_ptr->info |= (FONT_INDEX_G1_BLOCK_MOSAICS_SET << 8) +
                                 hold_mosaics_char;

                              hold_mosaic_enabled = TRUE;
                           }
                        }
                        break;
                     }
                     case 0x0f: // Release Mosaics ("set after")
                     {
                        if (mosaic_enabled == TRUE)
                        {
                           hold_mosaic_enabled = FALSE;
                        }
                        break;
                     }
                     default: // 0x0 to 0x07 - Mosaic Black to Mosaic White ("set after")
                     {
                        //                     foreground_colour = current_source_char & 0x07;
                        //                     concealed = FALSE;
                        //                     mosaic_enabled = TRUE;
                        //
                        //                     GetCharacterPaletteIndexes(display_info_ptr,
                        //                                                &foreground_index, &background_index,
                        //                                                foreground_colour, background_colour,
                        //                                                flashing, concealed,
                        //                                                FALSE, FALSE, video_mix_overridden);
                        break;
                     }
                  }
                  break;
               }
               default: // 0x20 to 0x7f - Font character
               {
                  destination_data_ptr->info |= current_source_char - 0x20;

                  if (mosaic_enabled == TRUE)
                  {
                     if (((current_source_char & 0x70) == 0x40) ||
                         ((current_source_char & 0x70) == 0x50))
                     {
                        if ((IsNationalOptionSubsetIndex(current_source_char, &subset_offset) == TRUE) &&
                            (current_mapping_ptr->national_option_subset_index != NATIONALITY_INDEX_UNDEFINED))
                        {
                           destination_data_ptr->info &= ~(TELETEXT_CHARACTER_FONT_INDEX_MASK |
                                                           TELETEXT_CHARACTER_BITMAP_INDEX_MASK |
                                                           TELETEXT_CHARACTER_SEPERATED_MOSAIC);

                           destination_data_ptr->info |= FONT_INDEX_NATIONAL_OPTION_SUBSET << 8;

                           destination_data_ptr->info |= subset_offset +
                              current_mapping_ptr->national_option_subset_index;
                        }
                        else
                        {
                           destination_data_ptr->info &= ~(TELETEXT_CHARACTER_FONT_INDEX_MASK |
                                                           TELETEXT_CHARACTER_SEPERATED_MOSAIC);

                           if (current_mapping_ptr->G0_index == FONT_INDEX_UNDEFINED)
                           {
                              destination_data_ptr->info |= FONT_INDEX_LATIN_G0_SET << 8;
                           }
                           else
                           {
                              destination_data_ptr->info |= current_mapping_ptr->G0_index << 8;
                           }
                        }
                     }
                  }
                  else
                  {
                     if ((current_mapping_ptr->G0_index == FONT_INDEX_LATIN_G0_SET) ||
                         (current_mapping_ptr->G0_index == FONT_INDEX_UNDEFINED))
                     {
                        if ((IsNationalOptionSubsetIndex(current_source_char, &subset_offset) == TRUE) &&
                            (current_mapping_ptr->national_option_subset_index != NATIONALITY_INDEX_UNDEFINED))
                        {
                           destination_data_ptr->info &= ~(TELETEXT_CHARACTER_FONT_INDEX_MASK |
                                                           TELETEXT_CHARACTER_BITMAP_INDEX_MASK);

                           destination_data_ptr->info |= FONT_INDEX_NATIONAL_OPTION_SUBSET << 8;

                           destination_data_ptr->info |= subset_offset +
                              current_mapping_ptr->national_option_subset_index;
                        }
                     }
                  }
                  break;
               }
            }
         }

         source_data_ptr++;
         destination_data_ptr++;
      }
   }

   FUNCTION_FINISH(DecodeTeletextClock);
}

/**
 *

 *

 *
 * @param   S_TELETEXT_CHARACTER* character_ptr -
 * @param   U8BIT* bitmap_ptr,
 * @param   U8BIT* aliased_foreground_index -
 * @param   U8BIT* aliased_background_index -
 * @param   S_PAGE_DISPLAY_INFO* display_info_ptr -
 *

 *
 */
static void GetAliasedColourIndexes(S_TELETEXT_CHARACTER *character_ptr, U8BIT *bitmap_ptr,
   U8BIT *aliased_foreground_index, U8BIT *aliased_background_index,
   S_PAGE_DISPLAY_INFO *display_info_ptr)
{
   U16BIT effect_index, alias_index, max_aliases;

   FUNCTION_START(GetAliasedColourIndexes);

   *aliased_foreground_index = character_ptr->foreground_index;
   *aliased_background_index = character_ptr->background_index;

   if (*bitmap_ptr & 4)
   {
      for (effect_index = 0; effect_index != NUM_PALETTE_EFFECTS; effect_index++)
      {
         if (display_info_ptr->palette_attribute_array[effect_index].effect_mask ==
             PALETTE_EFFECT_UNDEFINED)
         {
            break;
         }
      }

      if (effect_index != NUM_PALETTE_EFFECTS)
      {
         max_aliases = (NUM_PALETTE_EFFECTS - effect_index) / 2;

         for (alias_index = 0; alias_index < max_aliases; alias_index++)
         {
            if (display_info_ptr->palette_alias_array[alias_index].fore_colour ==
                PALETTE_BACKGROUND_TRANSPARENT)
            {
               display_info_ptr->palette_alias_array[alias_index].fore_colour =
                  character_ptr->foreground_index;
               display_info_ptr->palette_alias_array[alias_index].back_colour =
                  character_ptr->background_index;

               if (alias_index < (max_aliases - 1))
               {
                  display_info_ptr->palette_alias_array[alias_index + 1].fore_colour =
                     PALETTE_BACKGROUND_TRANSPARENT;
               }

               *aliased_foreground_index = PALETTE_EFFECT_ORIGIN + effect_index + (alias_index * 2);
               *aliased_background_index =
                  PALETTE_EFFECT_ORIGIN + effect_index + (alias_index * 2) + 1;

               break;
            }

            if ((display_info_ptr->palette_alias_array[alias_index].fore_colour ==
                 character_ptr->foreground_index) &&
                (display_info_ptr->palette_alias_array[alias_index].back_colour ==
                 character_ptr->background_index))
            {
               *aliased_foreground_index = PALETTE_EFFECT_ORIGIN + effect_index + (alias_index * 2);
               *aliased_background_index =
                  PALETTE_EFFECT_ORIGIN + effect_index + (alias_index * 2) + 1;

               break;
            }
         }
      }
   }

   FUNCTION_FINISH(GetAliasedColourIndexes);
}

/**
 *

 *

 *
 * @param   U16BIT x -
 * @param   U16BIT y -
 * @param   U16BIT x_scalar -
 * @param   U16BIT y_scalar -
 * @param   S_TELETEXT_CHARACTER* character_ptr -
 * @param   S_PAGE_DISPLAY_INFO* display_info_ptr -
 *

 *
 */
static void RenderCharacter(U16BIT x, U16BIT y, U16BIT x_scalar, U16BIT y_scalar,
   S_TELETEXT_CHARACTER *character_ptr, S_PAGE_DISPLAY_INFO *display_info_ptr)
{
   U8BIT row_index, column_index, repeat_index;
   U8BIT aliased_foreground_index, aliased_background_index;
   U16BIT font_index, bitmap_index, output_row_increment, output_repeat_increment;
   BOOLEAN render_done;
   U8BIT *output_row_ptr;
   U8BIT *output_column_ptr;
   U8BIT *output_repeat_ptr;
   U8BIT *bitmap_ptr;
   U8BIT *diacritic_bitmap_ptr;
   U8BIT pixel_colour;

   FUNCTION_START(RenderCharacter);

   render_done = FALSE;

   if (character_ptr->foreground_index == PALETTE_FOREGROUND_UNBOXED)
   {
      if (character_ptr->info & TELETEXT_CHARACTER_DOUBLE_HEIGHT)
      {
         y_scalar *= 2;
      }
      DBGPRINT("x=%d,y=%d", x, y);
      STB_OSDRegionFillRect( ttxt_osd_region, x, y,
         ebutt_font_ptr->character_width * x_scalar,
         ebutt_font_ptr->character_height * y_scalar,
         PALETTE_BACKGROUND_TRANSPARENT );

      render_done = TRUE;
   }
   else
   {
      font_index = (character_ptr->info & TELETEXT_CHARACTER_FONT_INDEX_MASK) >> 8;

      if (font_index != FONT_INDEX_NO_BITMAP)
      {
         if (font_index != FONT_INDEX_NATIONAL_OPTION_SUBSET)
         {
            bitmap_ptr = ebutt_font_ptr->font_table_set_ptr[font_index];
         }
         else
         {
            bitmap_ptr = ebutt_font_ptr->font_table_option_subset_ptr;
         }

         if (bitmap_ptr != NULL)
         {
            bitmap_index = character_ptr->info & TELETEXT_CHARACTER_BITMAP_INDEX_MASK;

            bitmap_ptr += ebutt_font_ptr->character_width * ebutt_font_ptr->character_height *
               bitmap_index;

            GetAliasedColourIndexes(character_ptr, bitmap_ptr,
               &aliased_foreground_index, &aliased_background_index,
               display_info_ptr);

            if (character_ptr->diacritic_info)
            {
               font_index = (character_ptr->diacritic_info & TELETEXT_CHARACTER_FONT_INDEX_MASK) >> 8;

               diacritic_bitmap_ptr = ebutt_font_ptr->font_table_set_ptr[font_index];

               if (diacritic_bitmap_ptr != NULL)
               {
                  bitmap_index = character_ptr->diacritic_info & TELETEXT_CHARACTER_BITMAP_INDEX_MASK;

                  diacritic_bitmap_ptr += ebutt_font_ptr->character_width *
                     ebutt_font_ptr->character_height * bitmap_index;
               }
            }
            else
            {
               diacritic_bitmap_ptr = NULL;
            }

            memset(display_char_buffer, character_ptr->background_index, display_char_buffer_size);

            if (character_ptr->info & TELETEXT_CHARACTER_DOUBLE_HEIGHT)
            {
               y_scalar *= 2;
            }

            output_row_ptr = display_char_buffer;

            output_row_increment = ebutt_font_ptr->character_width * x_scalar * y_scalar;
            output_repeat_increment = ebutt_font_ptr->character_width * x_scalar;

            for (row_index = 0; row_index != ebutt_font_ptr->character_height; row_index++)
            {
               output_column_ptr = output_row_ptr;

               if (character_ptr->info & TELETEXT_CHARACTER_SEPERATED_MOSAIC)
               {
                  if (ebutt_font_ptr->seperator_row_mask & (1L << (U32BIT)row_index))
                  {
                     output_row_ptr += output_row_increment;
                     bitmap_ptr += ebutt_font_ptr->character_width;
                     if (diacritic_bitmap_ptr != NULL)
                     {
                        diacritic_bitmap_ptr += ebutt_font_ptr->character_width;
                     }
                     continue;
                  }
               }

               for (column_index = 0; column_index != ebutt_font_ptr->character_width;
                    column_index++)
               {
                  if (character_ptr->info & TELETEXT_CHARACTER_SEPERATED_MOSAIC)
                  {
                     if (ebutt_font_ptr->seperator_column_mask & (1L << (U32BIT)column_index))
                     {
                        output_column_ptr += x_scalar;
                        bitmap_ptr++;
                        if (diacritic_bitmap_ptr != NULL)
                        {
                           diacritic_bitmap_ptr++;
                        }
                        continue;
                     }
                  }

                  if ((*bitmap_ptr) & 1)
                  {
                     output_repeat_ptr = output_column_ptr;

                     if (*bitmap_ptr == 3)
                     {
                        for (repeat_index = 0; repeat_index != y_scalar; repeat_index++)
                        {
                           memset(output_repeat_ptr, character_ptr->foreground_index, x_scalar);

                           *output_repeat_ptr = aliased_foreground_index;

                           output_repeat_ptr += output_repeat_increment;
                        }
                     }
                     else
                     {
                        for (repeat_index = 0; repeat_index != y_scalar; repeat_index++)
                        {
                           memset(output_repeat_ptr, character_ptr->foreground_index, x_scalar);

                           output_repeat_ptr += output_repeat_increment;
                        }
                     }
                  }
                  else
                  {
                     if (*bitmap_ptr == 2)
                     {
                        output_repeat_ptr = output_column_ptr;
                        pixel_colour = aliased_background_index;

                        for (repeat_index = 0; repeat_index != y_scalar; repeat_index++)
                        {
                           *output_repeat_ptr = pixel_colour;

                           output_repeat_ptr += output_repeat_increment;
                        }
                     }
                  }

                  if (diacritic_bitmap_ptr != NULL)
                  {
                     if ((*diacritic_bitmap_ptr) & 1)
                     {
                        output_repeat_ptr = output_column_ptr;

                        if ((*diacritic_bitmap_ptr == 3) &&
                            (*output_repeat_ptr != character_ptr->foreground_index))
                        {
                           for (repeat_index = 0; repeat_index != y_scalar; repeat_index++)
                           {
                              memset(output_repeat_ptr, character_ptr->foreground_index, x_scalar);

                              *output_repeat_ptr = aliased_foreground_index;

                              output_repeat_ptr += output_repeat_increment;
                           }
                        }
                        else
                        {
                           for (repeat_index = 0; repeat_index != y_scalar; repeat_index++)
                           {
                              memset(output_repeat_ptr, character_ptr->foreground_index, x_scalar);

                              output_repeat_ptr += output_repeat_increment;
                           }
                        }
                     }
                     else
                     {
                        output_repeat_ptr = output_column_ptr;

                        if ((*diacritic_bitmap_ptr == 2) &&
                            (*output_repeat_ptr != character_ptr->foreground_index))
                        {
                           pixel_colour = aliased_background_index;

                           for (repeat_index = 0; repeat_index != y_scalar; repeat_index++)
                           {
                              *output_repeat_ptr = pixel_colour;

                              output_repeat_ptr += output_repeat_increment;
                           }
                        }
                     }
                  }

                  output_column_ptr += x_scalar;
                  bitmap_ptr++;
                  if (diacritic_bitmap_ptr != NULL)
                  {
                     diacritic_bitmap_ptr++;
                  }
               }

               output_row_ptr += output_row_increment;
            }
            DBGPRINT("x=%d y=%d c=%c", x, y, bitmap_index ? bitmap_index + 0x20 : 176);
            STB_OSDDrawBitmapInRegion(ttxt_osd_region, x, y, output_repeat_increment,
               ebutt_font_ptr->character_height * y_scalar, display_char_buffer, FALSE);

            render_done = TRUE;
         }
      }
   }

   if (render_done == FALSE)
   {
      if (character_ptr->info & TELETEXT_CHARACTER_DOUBLE_HEIGHT)
      {
         y_scalar *= 2;
      }
      DBGPRINT("x=%d,y=%d", x, y);
      STB_OSDRegionFillRect( ttxt_osd_region, x, y,
         ebutt_font_ptr->character_width * x_scalar,
         ebutt_font_ptr->character_height * y_scalar,
         character_ptr->background_index );
   }

   FUNCTION_FINISH(RenderCharacter);
}

/**
 *

 *

 *
 * @param   BOOLEAN tranparency_required -
 * @param   U8BIT transparency_level -
 * @param   BOOLEAN red_required -
 * @param   BOOLEAN green_required -
 * @param   BOOLEAN blue_required -
 * @param   U8BIT gun_intensity -
 *
 * @return   
 *
 */
static U32BIT GetRGBSetting(BOOLEAN tranparency_required, U8BIT transparency_level,
   BOOLEAN red_required, BOOLEAN green_required, BOOLEAN blue_required,
   U8BIT gun_intensity)
{
   U32BIT retval;

   FUNCTION_START(GetRGBSetting);

   if (tranparency_required == TRUE)
   {
      retval = (U32BIT)transparency_level << 24L;
   }
   else
   {
      retval = 0;
   }

   if (red_required == TRUE)
   {
      retval |= (U32BIT)gun_intensity << 16L;
   }

   if (green_required == TRUE)
   {
      retval |= (U32BIT)gun_intensity << 8L;
   }

   if (blue_required == TRUE)
   {
      retval |= (U32BIT)gun_intensity;
   }

   FUNCTION_FINISH(GetRGBSetting);

   return(retval);
}

/**
 *

 *

 *
 * @param   U32BIT fore_colour -
 * @param   U32BIT back_colour -
 * @param   U32BIT foreground_sixteenths -
 *
 * @return   
 *
 */
static U32BIT GetRGBRamping(U32BIT fore_colour, U32BIT back_colour, U32BIT foreground_sixteenths)
{
   U8BIT index;
   U32BIT retval, gun_mask, fore_gun_value, back_gun_value, temp;

   FUNCTION_START(GetRGBRamping);

   retval = 0;
   gun_mask = 0x00ff0000L;

   for (index = 0; index != 3; index++)
   {
      fore_gun_value = fore_colour & gun_mask;
      back_gun_value = back_colour & gun_mask;

      if (fore_gun_value >= back_gun_value)
      {
         temp = (fore_gun_value * foreground_sixteenths) + (back_gun_value *
                                                            (16 - foreground_sixteenths));
      }
      else
      {
         temp = (back_gun_value * (16 - foreground_sixteenths)) +
            (fore_gun_value * foreground_sixteenths);
      }
      temp >>= 4;

      retval |= temp & gun_mask;

      gun_mask >>= 8;
   }

   FUNCTION_FINISH(GetRGBRamping);

   return(retval);
}

/**
 *

 *

 *
 * @param   S_PAGE_DISPLAY_INFO* display_info_ptr -
 *

 *
 */
static void RenderDisplay(S_PAGE_DISPLAY_INFO *display_info_ptr)
{
   U8BIT fore_colour_index, back_colour_index;
   U16BIT row_index, y, y_scalar, column_index, x, character_width, character_height;
   U16BIT effect_index, alias_index, max_aliases;
   U16BIT display_zoom_scalar, display_start_row, display_num_rows;
   U32BIT alias_colour_value;
   S_TELETEXT_CHARACTER *character_ptr;
   U8BIT stc[5];
   U32BIT pts_timestamp;
   U32BIT stc_timestamp;
   BOOLEAN render_required;

   static U32BIT normal_palette_array[PALETTE_EFFECT_ORIGIN];
   static U32BIT mixed_palette_array[PALETTE_EFFECT_ORIGIN];
   static U8BIT gun_intensity;
   static U8BIT antialias_level;
   static U8BIT transparency_level;

   FUNCTION_START(RenderDisplay);

   display_zoom_scalar = display_info_ptr->page_scale_zoom_scalar;
   display_start_row = display_info_ptr->page_scale_start_row;
   display_num_rows = display_info_ptr->page_scale_num_rows;
   if (display_start_row == 0 && !show_header_row)
   {
      display_start_row++;
      display_num_rows--;
   }

   render_required = FALSE;

   if (display_info_ptr->render_required == TRUE)
   {
#ifdef EBU_DEBUG
      if (display_info_ptr->clear_screen || display_info_ptr->render_row[display_start_row + 1])
      {
         EBU_PRINT("RenderDisplay: start=%u, num=%u, clear=%u\n", display_start_row, display_num_rows,
            display_info_ptr->clear_screen);

         EBU_PRINT("  valid =");
         for (row_index = display_start_row; row_index < (display_start_row + display_num_rows);
              row_index++)
         {
            EBU_PRINT("%c", display_info_ptr->valid_row[row_index] ? '1' : '0');
         }
         EBU_PRINT("\n");
         EBU_PRINT("  render=");
         for (row_index = display_start_row; row_index < (display_start_row + display_num_rows);
              row_index++)
         {
            EBU_PRINT("%c", display_info_ptr->render_row[row_index] ? '1' : '0');
         }
         EBU_PRINT("\n");
      }
#endif

      if (display_info_ptr->subtitle && display_info_ptr->pts_valid)
      {
         /* Work out whether there's anything to render */
         if (!display_info_ptr->clear_set)
         {
            for (row_index = display_start_row;
                 (row_index != (display_start_row + display_num_rows)) && !render_required; row_index++)
            {
               if ((display_info_ptr->valid_row[row_index] == TRUE) &&
                   (display_info_ptr->render_row[row_index] == TRUE))
               {
                  render_required = TRUE;
               }
            }
         }

         if (render_required)
         {
            pts_timestamp = (display_info_ptr->page_pts[0] << 24) + (display_info_ptr->page_pts[1] << 16) +
               (display_info_ptr->page_pts[2] << 8) + display_info_ptr->page_pts[3];

            STB_AVGetSTC(STB_DPGetPathVideoDecoder(teletext_path), stc);
            stc_timestamp = (stc[0] << 24) + (stc[1] << 16) + (stc[2] << 8) + stc[3];

            if ((stc[0] == 0) && (display_info_ptr->page_pts[0] == 1))
            {
               /* Top bit of STC isn't present so make sure PTS doesn't have it set either */
               EBU_DBG("Clearing top bit of PTS 0x%08lx, STC=0x%08lx", pts_timestamp, stc_timestamp);
               pts_timestamp &= 0x00ffffff;
            }
            else if ((stc[0] == 1) && (display_info_ptr->page_pts[0] == 0))
            {
               /* Top bit of STC is present so make sure PTS has it set too */
               EBU_DBG("Setting top bit of PTS 0x%08lx, STC=0x%08lx", pts_timestamp, stc_timestamp);
               pts_timestamp |= 0x01000000;
            }

            if (display_info_ptr->time_offset == 0)
            {
               /* Some broadcasters present PTS values that are completely different to the STC,
                * so a large difference needs to be detected and an offset value calculated */
               if ((stc_timestamp != 0) && (pts_timestamp != 0))
               {
                  display_info_ptr->time_offset = (S32BIT)((pts_timestamp - stc_timestamp) & 0xffffff00);
                  if ((display_info_ptr->time_offset < -0x10000) || (display_info_ptr->time_offset > 0x10000))
                  {
                     EBU_DBG("Using offset 0x%08lx, PTS 0x%08lx, STC 0x%08lx",
                              display_info_ptr->time_offset, pts_timestamp, stc_timestamp);

                     pts_timestamp -= display_info_ptr->time_offset;
                  }
                  else
                  {
                     display_info_ptr->time_offset = 0;
                  }
               }
            }
            else
            {
               pts_timestamp -= display_info_ptr->time_offset;
            }

            /* Wait to update the display */
            do
            {
               STB_AVGetSTC(STB_DPGetPathVideoDecoder(teletext_path), stc);
               stc_timestamp = (stc[0] << 24) + (stc[1] << 16) + (stc[2] << 8) + stc[3];

               if ((stc_timestamp < pts_timestamp) && !stop_teletext)
               {
                  EBU_DBG("Wait to display..., PTS=0x%08lx, STC=0x%08lx", pts_timestamp, stc_timestamp);
                  STB_OSTaskDelay(100);
               }
            }
            while ((stc_timestamp < pts_timestamp) && !stop_teletext);

            EBU_DBG("Display now: PTS=0x%08lx, STC=0x%08lx", pts_timestamp, stc_timestamp);
         }
      }

      /* Hide the region before drawing into it */
      STB_OSDHideRegion(ttxt_osd_region);

      if (display_info_ptr->clear_screen == TRUE)
      {
         if (display_info_ptr->clear_set == TRUE)
         {
            STB_OSDFillRegion( ttxt_osd_region, PALETTE_BACKGROUND_TRANSPARENT );
         }
         else
         {
            STB_OSDFillRegion( ttxt_osd_region, display_info_ptr->screen_colour );
         }

         display_info_ptr->clear_screen = FALSE;
      }

      if (ebutt_font_ptr != NULL)
      {
         character_width = ebutt_font_ptr->character_width * display_x_scalar;

         y_scalar = display_y_scalar * display_zoom_scalar;

         character_height = ebutt_font_ptr->character_height * y_scalar;

         y = display_margin_top;

         if (display_info_ptr->clear_set == TRUE)
         {
            character_ptr = &display_info_ptr->character_map[0][2];

            x = display_margin_left;
            x += character_width * 2;

            for (column_index = 2; column_index != 6; column_index++)
            {
               RenderCharacter(x, y, display_x_scalar, y_scalar, character_ptr, display_info_ptr);
               x += character_width;
               character_ptr++;
            }
         }
         else
         {
#ifdef EBU_DEBUG
            EBU_PRINT("%s: ", __FUNCTION__);
#endif
            for (row_index = display_start_row; row_index != (display_start_row + display_num_rows);
                 row_index++)
            {
               if ((display_info_ptr->valid_row[row_index] == TRUE) &&
                   (display_info_ptr->render_row[row_index] == TRUE))
               {
                  character_ptr = &display_info_ptr->character_map[row_index][0];

                  x = display_margin_left;

                  for (column_index = 0; column_index != MAX_COLUMN; column_index++)
                  {
                     if (row_index > 0)
                     {
                        if ((display_info_ptr->valid_row[row_index - 1] == TRUE) &&
                            (display_info_ptr->character_map[row_index - 1][column_index].info &
                             TELETEXT_CHARACTER_DOUBLE_HEIGHT))
                        {
                           x += character_width;
                           character_ptr++;
                           continue;
                        }
                     }

#ifdef EBU_DEBUG
                     if ((character_ptr->info & 0xff) < 0x5f)
                        EBU_PRINT("%c", (character_ptr->info & 0xff) + 0x20);
#endif
                     RenderCharacter(x, y, display_x_scalar, y_scalar, character_ptr, display_info_ptr);
                     x += character_width;
                     character_ptr++;
                  }

                  display_info_ptr->render_row[row_index] = FALSE;
               }

               y += character_height;
            }
#ifdef EBU_DEBUG
            EBU_PRINT("\n");
#endif
         }
      }

      display_info_ptr->render_required = FALSE;
   }

   if (gun_intensity != current_gun_intensity)
   {
      gun_intensity = current_gun_intensity;

      normal_palette_array[PALETTE_BACKGROUND_TRANSPARENT] =
         GetRGBSetting(TRUE, current_transparency_level, FALSE, FALSE, FALSE, 0);
      normal_palette_array[PALETTE_BACKGROUND_BLACK_FIXED] =
         GetRGBSetting(FALSE, 0, FALSE, FALSE, FALSE, 0);

      normal_palette_array[PALETTE_BACKGROUND_BLACK] =
         GetRGBSetting(FALSE, 0, FALSE, FALSE, FALSE, 0);
      normal_palette_array[PALETTE_BACKGROUND_RED] =
         GetRGBSetting(FALSE, 0, TRUE, FALSE, FALSE, gun_intensity);
      normal_palette_array[PALETTE_BACKGROUND_GREEN] =
         GetRGBSetting(FALSE, 0, FALSE, TRUE, FALSE, gun_intensity);
      normal_palette_array[PALETTE_BACKGROUND_YELLOW] =
         GetRGBSetting(FALSE, 0, TRUE, TRUE, FALSE, gun_intensity);
      normal_palette_array[PALETTE_BACKGROUND_BLUE] =
         GetRGBSetting(FALSE, 0, FALSE, FALSE, TRUE, gun_intensity);
      normal_palette_array[PALETTE_BACKGROUND_MAGENTA] =
         GetRGBSetting(FALSE, 0, TRUE, FALSE, TRUE, gun_intensity);
      normal_palette_array[PALETTE_BACKGROUND_CYAN] =
         GetRGBSetting(FALSE, 0, FALSE, TRUE, TRUE, gun_intensity);
      normal_palette_array[PALETTE_BACKGROUND_WHITE] =
         GetRGBSetting(FALSE, 0, TRUE, TRUE, TRUE, gun_intensity);

      normal_palette_array[PALETTE_FOREGROUND_BLACK] =
         normal_palette_array[PALETTE_BACKGROUND_BLACK];
      normal_palette_array[PALETTE_FOREGROUND_RED] =
         normal_palette_array[PALETTE_BACKGROUND_RED];
      normal_palette_array[PALETTE_FOREGROUND_GREEN] =
         normal_palette_array[PALETTE_BACKGROUND_GREEN];
      normal_palette_array[PALETTE_FOREGROUND_YELLOW] =
         normal_palette_array[PALETTE_BACKGROUND_YELLOW];
      normal_palette_array[PALETTE_FOREGROUND_BLUE] =
         normal_palette_array[PALETTE_BACKGROUND_BLUE];
      normal_palette_array[PALETTE_FOREGROUND_MAGENTA] =
         normal_palette_array[PALETTE_BACKGROUND_MAGENTA];
      normal_palette_array[PALETTE_FOREGROUND_CYAN] =
         normal_palette_array[PALETTE_BACKGROUND_CYAN];
      normal_palette_array[PALETTE_FOREGROUND_WHITE] =
         normal_palette_array[PALETTE_BACKGROUND_WHITE];

      mixed_palette_array[PALETTE_BACKGROUND_TRANSPARENT] =
         GetRGBSetting(TRUE, current_transparency_level, FALSE, FALSE, FALSE, 0);
      mixed_palette_array[PALETTE_BACKGROUND_BLACK_FIXED] =
         GetRGBSetting(FALSE, 0, FALSE, FALSE, FALSE, 0);

      mixed_palette_array[PALETTE_BACKGROUND_BLACK] =
         mixed_palette_array[PALETTE_BACKGROUND_TRANSPARENT];
      mixed_palette_array[PALETTE_BACKGROUND_RED] =
         mixed_palette_array[PALETTE_BACKGROUND_TRANSPARENT];
      mixed_palette_array[PALETTE_BACKGROUND_GREEN] =
         mixed_palette_array[PALETTE_BACKGROUND_TRANSPARENT];
      mixed_palette_array[PALETTE_BACKGROUND_YELLOW] =
         mixed_palette_array[PALETTE_BACKGROUND_TRANSPARENT];
      mixed_palette_array[PALETTE_BACKGROUND_BLUE] =
         mixed_palette_array[PALETTE_BACKGROUND_TRANSPARENT];
      mixed_palette_array[PALETTE_BACKGROUND_MAGENTA] =
         mixed_palette_array[PALETTE_BACKGROUND_TRANSPARENT];
      mixed_palette_array[PALETTE_BACKGROUND_CYAN] =
         mixed_palette_array[PALETTE_BACKGROUND_TRANSPARENT];
      mixed_palette_array[PALETTE_BACKGROUND_WHITE] =
         mixed_palette_array[PALETTE_BACKGROUND_TRANSPARENT];

      mixed_palette_array[PALETTE_FOREGROUND_BLACK] =
         normal_palette_array[PALETTE_FOREGROUND_BLACK];
      mixed_palette_array[PALETTE_FOREGROUND_RED] =
         normal_palette_array[PALETTE_FOREGROUND_RED];
      mixed_palette_array[PALETTE_FOREGROUND_GREEN] =
         normal_palette_array[PALETTE_FOREGROUND_GREEN];
      mixed_palette_array[PALETTE_FOREGROUND_YELLOW] =
         normal_palette_array[PALETTE_FOREGROUND_YELLOW];
      mixed_palette_array[PALETTE_FOREGROUND_BLUE] =
         normal_palette_array[PALETTE_FOREGROUND_BLUE];
      mixed_palette_array[PALETTE_FOREGROUND_MAGENTA] =
         normal_palette_array[PALETTE_FOREGROUND_MAGENTA];
      mixed_palette_array[PALETTE_FOREGROUND_CYAN] =
         normal_palette_array[PALETTE_FOREGROUND_CYAN];
      mixed_palette_array[PALETTE_FOREGROUND_WHITE] =
         normal_palette_array[PALETTE_FOREGROUND_WHITE];

      display_info_ptr->palette_update_required = TRUE;
   }

   if (antialias_level != current_antialias_level)
   {
      antialias_level = current_antialias_level;

      display_info_ptr->palette_update_required = TRUE;
   }

   if (transparency_level != current_transparency_level)
   {
      transparency_level = current_transparency_level;

      display_info_ptr->palette_update_required = TRUE;
   }

   if (palette_reset_required == TRUE)
   {
      if (display_info_ptr->video_mix_set == TRUE)
      {
         memcpy(display_info_ptr->palette_array, mixed_palette_array,
            PALETTE_EFFECT_ORIGIN * sizeof(U32BIT));
      }
      else
      {
         memcpy(display_info_ptr->palette_array, normal_palette_array,
            PALETTE_EFFECT_ORIGIN * sizeof(U32BIT));
      }

      STB_OSDSetRGBPalette( ttxt_osd_region, display_info_ptr->palette_array );

      palette_reset_required = FALSE;
   }

   if (display_info_ptr->palette_update_required == TRUE)
   {
      if (display_info_ptr->video_mix_set == TRUE)
      {
         memcpy(display_info_ptr->palette_array, mixed_palette_array,
            PALETTE_EFFECT_ORIGIN * sizeof(U32BIT));
      }
      else
      {
         memcpy(display_info_ptr->palette_array, normal_palette_array,
            PALETTE_EFFECT_ORIGIN * sizeof(U32BIT));
      }

      for (effect_index = 0; effect_index < NUM_PALETTE_EFFECTS; effect_index++)
      {
         if (display_info_ptr->palette_attribute_array[effect_index].effect_mask ==
             PALETTE_EFFECT_UNDEFINED)
         {
            break;
         }

         display_info_ptr->palette_array[effect_index + PALETTE_EFFECT_ORIGIN] =
            display_info_ptr->palette_array[display_info_ptr->palette_attribute_array[effect_index].fore_colour];
         display_info_ptr->palette_attribute_array[effect_index].fore_colour_used = TRUE;

         if (display_info_ptr->palette_attribute_array[effect_index].effect_mask & PALETTE_EFFECT_CONCEAL)
         {
            if (display_info_ptr->reveal_set == TRUE)
            {
               if (display_info_ptr->palette_attribute_array[effect_index].effect_mask & PALETTE_EFFECT_FLASH)
               {
                  if (display_info_ptr->flash_set == FALSE)
                  {
                     display_info_ptr->palette_array[effect_index + PALETTE_EFFECT_ORIGIN] =
                        display_info_ptr->palette_array[display_info_ptr->palette_attribute_array[effect_index].back_colour];
                     display_info_ptr->palette_attribute_array[effect_index].fore_colour_used = FALSE;
                  }
               }
            }
            else
            {
               display_info_ptr->palette_array[effect_index + PALETTE_EFFECT_ORIGIN] =
                  display_info_ptr->palette_array[display_info_ptr->palette_attribute_array[effect_index].back_colour];
               display_info_ptr->palette_attribute_array[effect_index].fore_colour_used = FALSE;
            }
         }
         else if (display_info_ptr->palette_attribute_array[effect_index].effect_mask & PALETTE_EFFECT_FLASH)
         {
            if (display_info_ptr->flash_set == FALSE)
            {
               display_info_ptr->palette_array[effect_index + PALETTE_EFFECT_ORIGIN] =
                  display_info_ptr->palette_array[display_info_ptr->palette_attribute_array[effect_index].back_colour];
               display_info_ptr->palette_attribute_array[effect_index].fore_colour_used = FALSE;
            }
         }
      }

      if (effect_index != NUM_PALETTE_EFFECTS)
      {
         max_aliases = (NUM_PALETTE_EFFECTS - effect_index) / 2;

         for (alias_index = 0; alias_index < max_aliases; alias_index++)
         {
            if (display_info_ptr->palette_alias_array[alias_index].fore_colour ==
                PALETTE_BACKGROUND_TRANSPARENT)
            {
               break;
            }

            fore_colour_index = display_info_ptr->palette_alias_array[alias_index].fore_colour;
            back_colour_index = display_info_ptr->palette_alias_array[alias_index].back_colour;

            if (fore_colour_index >= PALETTE_EFFECT_ORIGIN)
            {
               if (display_info_ptr->palette_attribute_array[fore_colour_index - PALETTE_EFFECT_ORIGIN].fore_colour_used == TRUE)
               {
                  fore_colour_index = display_info_ptr->palette_attribute_array[fore_colour_index - PALETTE_EFFECT_ORIGIN].fore_colour;
               }
               else
               {
                  fore_colour_index = display_info_ptr->palette_attribute_array[fore_colour_index - PALETTE_EFFECT_ORIGIN].back_colour;
               }
            }

            if (display_info_ptr->palette_array[back_colour_index] & 0xff000000L)
            {
               display_info_ptr->palette_array[PALETTE_EFFECT_ORIGIN + effect_index + (alias_index * 2)] =
                  display_info_ptr->palette_array[fore_colour_index];
               display_info_ptr->palette_array[PALETTE_EFFECT_ORIGIN + effect_index + (alias_index * 2) + 1] =
                  display_info_ptr->palette_array[back_colour_index];
            }
            else
            {
               alias_colour_value = GetRGBRamping(display_info_ptr->palette_array[fore_colour_index],
                     display_info_ptr->palette_array[back_colour_index],
                     16L - (U32BIT)antialias_level);

               display_info_ptr->palette_array[PALETTE_EFFECT_ORIGIN + effect_index + (alias_index * 2)] = alias_colour_value;

               alias_colour_value = GetRGBRamping(display_info_ptr->palette_array[fore_colour_index],
                     display_info_ptr->palette_array[back_colour_index],
                     (U32BIT)antialias_level);

               display_info_ptr->palette_array[PALETTE_EFFECT_ORIGIN + effect_index + (alias_index * 2) + 1] = alias_colour_value;
            }
         }
      }


      STB_OSDSetRGBPalette( ttxt_osd_region, display_info_ptr->palette_array );

      display_info_ptr->palette_update_required = FALSE;
   }

   if (!is_visible)
   {
      STB_OSDDisableUIRegion();
      //STB_OSDSetRegionDisplaySize(SD_WIDTH, SD_HEIGHT);
      is_visible = TRUE;
   }

   if (!stop_teletext)
   {
      if (render_required)
      {
         /* Need to set a new time for the subtitle timeout as the display of it is being delayed */
         subtitle_timeout = STB_OSGetClockMilliseconds();

         EBU_DBG("Start subtitle timeout @ %lu", subtitle_timeout);
      }

      STB_OSDShowRegion(ttxt_osd_region);
      STB_OSDUpdateRegions();
   }

   FUNCTION_FINISH(RenderDisplay);
}

/**
 *

 *
 * @brief   This process is called to service and display updates for the current Teletext
 *                 page, as collated by the cache mechanism.
 *                 NB - Thus function is called repeatedly by the DisplayUpdateTask(), and only
 *                      serves to abstract the display update functionality into a function which
 *                      can also be called from a single-threaded test environment.
 *                 Here the process of analysing chnages each page component, and the subsequent
 *                 refreshing of the display buffer occurs.
 *

 *

 *
 */
static void PerformDisplayUpdate(void)
{
   U16BIT row_index;
   BOOLEAN enforce_header_display;
   BOOLEAN override_header_backcolor;

   static U16BIT flash_counter;

   FUNCTION_START(PerformDisplayUpdate);

   enforce_header_display = FALSE;

   /* If the drivers have been reset ensure that the copy is cleared to prevent obsolete data being displayed */
   if (page_display_info.video_mix_required != page_control_info.video_mix_required)
   {
      page_display_info.video_mix_required = page_control_info.video_mix_required;

      page_display_info.video_mix_set = !page_display_info.video_mix_set;
   }

   if (page_display_copy.video_mix_set != page_display_info.video_mix_set)
   {
      page_display_copy.video_mix_set = page_display_info.video_mix_set;

      page_display_copy.palette_update_required = TRUE;
   }

   if (page_display_info.clear_required != page_control_info.clear_required)
   {
      page_display_info.clear_required = page_control_info.clear_required;

      page_display_info.clear_set = !page_display_info.clear_set;
   }

   if (page_display_copy.clear_set != page_display_info.clear_set)
   {
      page_display_copy.clear_set = page_display_info.clear_set;

      if (page_display_copy.clear_set == TRUE)
      {
         page_display_copy.render_required = TRUE;
         page_display_copy.clear_screen = TRUE;
      }
      else
      {
         if (page_display_info.page_number)
         {
            page_display_info.page_updated = TRUE;
         }
      }
   }

   override_header_backcolor = FALSE;
   if (show_header_row)
   {
      STB_OSSemaphoreWait(page_display_info.carousel_page_semaphore);

      if (page_display_info.carousel_page_updated == TRUE)
      {
         page_display_copy.carousel_page_number = page_display_info.carousel_page_number;

         memcpy(&page_display_copy.carousel_page_header, &page_display_info.carousel_page_header, MAX_COLUMN);

         page_display_info.carousel_page_updated = FALSE;

         STB_OSSemaphoreSignal(page_display_info.carousel_page_semaphore);

         DecodeTeletextCarouselHeader(&page_display_copy);

         override_header_backcolor = TRUE;
         enforce_header_display = TRUE;
      }
      else
      {
         STB_OSSemaphoreSignal(page_display_info.carousel_page_semaphore);
      }
   }

   STB_OSSemaphoreWait(page_display_info.page_semaphore);

   if (page_display_info.page_updated == TRUE)
   {
      page_display_copy.page_number = page_display_info.page_number;
      page_display_copy.page_sub_code = page_display_info.page_sub_code;

      memcpy(&page_display_copy.page_source, &page_display_info.page_source, sizeof(S_MAGAZINE_PAGE));
      memcpy(&page_display_copy.mgzn_data, &page_display_info.mgzn_data, sizeof(S_SPECIFIC_DATA));

      page_display_copy.page_clock_valid = page_display_info.page_clock_valid;

      page_display_info.page_updated = FALSE;

      STB_OSSemaphoreSignal(page_display_info.page_semaphore);

      page_display_copy.flash_set = TRUE;
      page_display_copy.flash_required = FALSE;

      page_display_copy.reveal_available = FALSE;

      DecodeTeletextPageBody(&page_display_copy, enforce_header_display);

      override_header_backcolor = FALSE;
   }
   else
   {
      STB_OSSemaphoreSignal(page_display_info.page_semaphore);
   }

   STB_OSSemaphoreWait(page_display_info.page_index_semaphore);

   if (page_display_info.page_index_updated == TRUE)
   {
      memcpy(&page_display_copy.page_index_digit, &page_display_info.page_index_digit,
         sizeof(E_PAGE_INDEX_DIGIT) * 3);

      page_display_copy.page_index_held = page_display_info.page_index_held;

      page_display_info.page_index_updated = FALSE;

      STB_OSSemaphoreSignal(page_display_info.page_index_semaphore);

      if (show_header_row)
      {
         if ((override_header_backcolor == TRUE) ||
             (page_control_info.state == PAGE_CONTROL_STATE_INPUT))
         {
            DecodeTeletextPageNumber(&page_display_copy, TRUE);
         }
         else
         {
            DecodeTeletextPageNumber(&page_display_copy, FALSE);
         }
      }
   }
   else
   {
      STB_OSSemaphoreSignal(page_display_info.page_index_semaphore);
   }

   STB_OSSemaphoreWait(page_display_info.time_filler_header_data_semaphore);

   if (page_display_info.clock_updated == TRUE)
   {
      memcpy(&page_display_copy.time_filler_header_data, &page_display_info.time_filler_header_data, 32);

      page_display_info.clock_updated = FALSE;

      STB_OSSemaphoreSignal(page_display_info.time_filler_header_data_semaphore);

      DecodeTeletextClock(&page_display_copy, override_header_backcolor);
   }
   else
   {
      STB_OSSemaphoreSignal(page_display_info.time_filler_header_data_semaphore);
   }

   STB_OSSemaphoreWait(page_display_info.page_scale_semaphore);

   if (page_display_info.page_scale_params_defined == FALSE)
   {
      page_display_copy.page_scale_zoom_scalar = 1;
      page_display_copy.page_scale_start_row = 0;
      page_display_copy.page_scale_num_rows = MAX_ROWS;
   }
   else
   {
      page_display_copy.page_scale_zoom_scalar = page_display_info.page_scale_zoom_scalar;
      page_display_copy.page_scale_start_row = page_display_info.page_scale_start_row;
      page_display_copy.page_scale_num_rows = page_display_info.page_scale_num_rows;
   }

   if (page_display_info.page_rescaled == TRUE)
   {
      page_display_info.page_rescaled = FALSE;

      STB_OSSemaphoreSignal(page_display_info.page_scale_semaphore);

      for (row_index = page_display_copy.page_scale_start_row;
           row_index < (page_display_copy.page_scale_start_row + page_display_copy.page_scale_num_rows);
           row_index++)
      {
         page_display_copy.render_row[row_index] = page_display_copy.valid_row[row_index];
      }

      page_display_copy.render_required = TRUE;
      page_display_copy.clear_screen = TRUE;
   }
   else
   {
      STB_OSSemaphoreSignal(page_display_info.page_scale_semaphore);
   }

   if (page_display_copy.flash_required == TRUE)
   {
      flash_counter++;
      if (flash_counter == 3)
      {
         page_display_copy.flash_set = !page_display_copy.flash_set;
         page_display_copy.palette_update_required = TRUE;

         flash_counter = 0;
      }
   }

   if (page_display_copy.reveal_available == TRUE)
   {
      if (page_display_info.reveal_required != page_control_info.reveal_required)
      {
         page_display_info.reveal_required = page_control_info.reveal_required;

         page_display_info.reveal_set = !page_display_info.reveal_set;
      }

      if (page_display_copy.reveal_set != page_display_info.reveal_set)
      {
         page_display_copy.reveal_set = page_display_info.reveal_set;

         page_display_copy.palette_update_required = TRUE;
      }
   }

   page_display_copy.pts_valid = page_display_info.pts_valid;
   if (page_display_copy.pts_valid)
   {
      memcpy(page_display_copy.page_pts, page_display_info.page_pts, 5);
   }

   if (page_display_copy.render_required)
   {
      /* page_display_info is now free to accept the next page */
      STB_OSSemaphoreSignal(page_free_sem);

      RenderDisplay(&page_display_copy);
   }

   FUNCTION_FINISH(PerformDisplayUpdate);
}

static BOOLEAN InitialiseDisplay(void)
{
   U8BIT *bitmap_ptr;
   U16BIT ch_width, ch_height;
   BOOLEAN retval;

   FUNCTION_START(InitialiseDisplay);

   retval = FALSE;
   ch_width = (U16BIT)ebutt_font_ptr->character_width;
   ch_height = (U16BIT)ebutt_font_ptr->character_height;

   ttxt_osd_region = STB_OSDCreateRegion( SD_WIDTH, SD_HEIGHT, 8 /*bit colour*/ );

   if (ttxt_osd_region != NULL)
   {
      STB_OSDSetRegionDisplaySize(SD_WIDTH, SD_HEIGHT);

      display_margin_left = (SD_WIDTH - (ch_width * display_x_scalar * MAX_COLUMN)) / 2;
      display_margin_top = (SD_HEIGHT - (ch_height * display_y_scalar * MAX_ROWS)) / 2;

      // initialise region colour
      STB_OSDFillRegion( ttxt_osd_region, PALETTE_BACKGROUND_TRANSPARENT );

      display_char_buffer_size = ch_width * ch_height * display_x_scalar * display_y_scalar *
         MAX_DISPLAY_ZOOM_SCALAR * 2;

      if (display_char_buffer != NULL)
      {
         STB_FreeMemory(display_char_buffer_origin);

         display_char_buffer = NULL;
      }

      // NB - we have a character rendering buffer that has a hidden additional six bytes at
      //      the head of it's allocation. DO NOT TAMPER WITH THIS!!
      //      In a standard build these header bytes are unused, and their existance effects
      //      nothing.
      //      In a TTDEMO build, they are used to display the bitmap content in an emulation.
      display_char_buffer_origin = (U8BIT *)STB_GetMemory(display_char_buffer_size + 6);
      if (display_char_buffer_origin != NULL)
      {
         display_char_buffer = display_char_buffer_origin + 6;

         diacritic_substitutions_avaialable = FALSE;

         // Check to see of anything has been defined for G1 block mosaic character 0x40
         // (i.e. a table offset of 0x20).
         // If anything is detected at this reserved (and nominally empty) location, then we
         // will assume that there are character/diacritic substitutions defined here.

         bitmap_ptr = ebutt_font_ptr->font_table_set_ptr[FONT_INDEX_G1_BLOCK_MOSAICS_SET];

         if (bitmap_ptr != NULL)
         {
            bitmap_ptr += ch_width * ch_height * 0x20;

            memset(display_char_buffer, 0, display_char_buffer_size);

            if (memcmp(bitmap_ptr, display_char_buffer, ch_width * ch_height))
            {
               diacritic_substitutions_avaialable = TRUE;
            }
         }
         retval = TRUE;
      }
   }

   FUNCTION_FINISH(InitialiseDisplay);

   return(retval);
}

/**
 *

 *
 * @brief   This task is used to service and display updates for the current Teletext page,
 *                 as collated by the cache mechanism.
 *                 The process of recognising changes to the collated page content and effecting
 *                 changes is coninuous, and so for this reason is assigned it's own task within
 *                 the driver module.
 *
 * @param   void *unwanted_ptr - requisite of task API - not used.
 *

 *
 */
static void DisplayUpdateTask(void *unwanted_ptr)
{
   FUNCTION_START(DisplayUpdateTask);

   USE_UNWANTED_PARAM(unwanted_ptr);

   if (InitialiseDisplay())
   {
      // Run forever
      while (1)
      {
         STB_OSSemaphoreWait(display_task_semaphore);

         if ((subtitle_timeout != 0) && (STB_OSGetClockDiff(subtitle_timeout) > SUBTITLE_TIMEOUT))
         {
            EBU_DBG("%s: Subtitle timeout @ %lu", __FUNCTION__, STB_OSGetClockMilliseconds());

            DriverReset();

            STB_OSDFillRegion( ttxt_osd_region, PALETTE_BACKGROUND_TRANSPARENT );
            STB_OSDUpdateRegions();
         }

         PerformDisplayUpdate();

         STB_OSSemaphoreSignal(display_task_semaphore);

         STB_OSTaskDelay(100);
      }
   }

   FUNCTION_FINISH(DisplayUpdateTask);
}

//---global function definitions-----------------------------------------------

/**
 *

 *
 * @brief   This must be called to invoke the EBU TeleText driver module before all other
 *                 functionality can be accessed. This initiates the relevant resources used by the
 *                 driver module.
 *                 As a consequence of calling this function, no TeleText processing will occur until
 *                 other functions are called.
 *                 It is envisaged that this function may be called within the general initialisation
 *                 of the STB layer.
 *

 *
 * @return   TRUE if initialisation was successful, FALSE otherwise.
 *
 */
BOOLEAN STB_EBUTT_Initialise(void)
{
   BOOLEAN retval;

   FUNCTION_START(STB_EBUTT_Initialise);

   retval = FALSE;

   if (is_initialised == FALSE)
   {
      if (InitialiseCollation() == TRUE)
      {
         if (InitialiseDisplayTask() == TRUE)
         {
            pes_collection_callback_handle =
               STB_RegisterPesCollectionCallback(PESCollectionCallback, 0x10, 0x1f);
            if (pes_collection_callback_handle)
            {
               is_initialised = TRUE;

               retval = TRUE;
            }
            else
            {
               KillDisplayTask();
            }
         }
      }

      /* Create the resources for the collation task */
      page_free_sem = STB_OSCreateSemaphore();
      collation_start = STB_OSCreateSemaphore();
      collation_stopped = STB_OSCreateSemaphore();

      collation_queue = STB_OSCreateQueue(sizeof(S_COLLATION_QUEUE_ITEM), COLLATION_QUEUE_SIZE);

      if ((collation_start != NULL) && (collation_stopped != NULL) && (collation_queue != NULL))
      {
         /* Initially grab both semaphores */
         STB_OSSemaphoreWait(collation_start);
         STB_OSSemaphoreWait(collation_stopped);

         stop_collation = FALSE;
         queue_count = 0;
         max_queue_count = 0;
         collation_started = FALSE;

         STB_OSCreateTask(PESCollationTask, NULL, DISPLAY_TASK_STACK_SIZE, DISPLAY_TASK_PRIORITY,
            (U8BIT *)"EBUTT_Collate");
      }
   }

   FUNCTION_FINISH(STB_EBUTT_Initialise);

   return(retval);
}

/**
 *

 *
 * @brief   This is the accompanying function to STB_EBUTT_Initialise( ), and is called to
 *                 shut-down the EBU TeleText driver module and release the relevant resources used
 *                 by the driver module.
 *                 This may not be used in some implementations.
 *
 * @param   U8BIT path - the path of where to kill teletext.
 *

 *
 */
void STB_EBUTT_Kill(U8BIT path)
{
   FUNCTION_START(STB_EBUTT_Kill);

   if (is_initialised == TRUE)
   {
      STB_EBUTT_Hide();
      STB_EBUTT_Stop(path, TRUE);

      STB_UnregisterPesCollectionCallback(pes_collection_callback_handle);

      if (display_char_buffer != NULL)
      {
         STB_FreeMemory(display_char_buffer_origin);

         display_char_buffer = NULL;
      }

      is_initialised = FALSE;
   }

   FUNCTION_FINISH(STB_EBUTT_Kill);
}

/*!**************************************************************************
 * @brief   Allows teletext PES data packets to be injected by an external module,
 *          which will be decoded and displayed.
 * @param   data_ptr - pointer to first whole PES packet data
 * @param   data_length - number of bytes of data provided
 * @return  TRUE if the data is valid EBU teletext PES data, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_EBUTT_InjectData(U8BIT *data_ptr, U32BIT data_length)
{
   BOOLEAN retval;
   U16BIT pes_size;

   FUNCTION_START(STB_EBUTT_InjectData);

   retval = FALSE;

   if ((data_ptr != NULL) && (data_length > 5))
   {
      retval = TRUE;

      /* Handle each PES packet one at a time */
      while (retval && (data_length > 5))
      {
         /* Find the size of the PES packet, which is +6 to allow for the header and size fields */
         pes_size = (*(data_ptr + 4) << 8) + *(data_ptr + 5) + 6;

         if (data_length >= pes_size)
         {
            /* Enough data has been provided, so add it to the processing queue */
            retval = AddPESDataToQueue(data_ptr, pes_size);

            data_length -= pes_size;
            data_ptr += pes_size;
         }
         else
         {
            retval = FALSE;
         }
      }
   }

   FUNCTION_FINISH(STB_EBUTT_InjectData);

   return(retval);
}

/**
 *

 *
 * @brief   These functions are called to control whether received TeleText data is processed.
 *                 Using this functionality ensures that the process of collating page information
 *                 can be halted when processing resources are a consideration.
 *                 It is envisaged that some implementations may wish only to collate TeleText
 *                 content only when the display is being shown,
 *                 whilst other designs may which to cache significant page content as a background
 *                 process.
 *                 Abstracting this functionality from the initialisation routines gives the option
 *                 to support both solution types.
 *
 * @param   magazine - initial magazine to be loaded
 *                 page - initial page to show from the magazine
 *

 *
 */
void STB_EBUTT_Start(U8BIT path, U16BIT text_pid, U8BIT magazine, U8BIT page)
{
   FUNCTION_START(STB_EBUTT_Start);

   if (is_initialised == TRUE)
   {
      STB_ChangePesCollectionPID(path, text_pid);

      STB_OSSemaphoreWait(page_request_info.semaphore);

      page_request_info.type = REQUEST_TYPE_EXPLICIT;

      if (magazine == 0)
      {
         page_request_info.page_number = 0x800 + page;
      }
      else
      {
         page_request_info.page_number = (magazine << 8) + page;
      }

      page_request_info.page_sub_code = PAGE_SUB_CODE_UNDEFINED;

      page_request_info.pending = TRUE;

      STB_OSSemaphoreSignal(page_request_info.semaphore);

      teletext_path = path;
      broadcast_index_page_invocation_required = TRUE;
      show_header_row = FALSE;
      subtitle_timeout = 0;
      stop_teletext = FALSE;

      stream_processing_enabled = TRUE;

      stop_collation = FALSE;

      /* Start the collation task */
      STB_OSSemaphoreSignal(collation_start);
   }

   FUNCTION_FINISH(STB_EBUTT_Start);
}

void STB_EBUTT_Stop(U8BIT path, BOOLEAN reset_cache)
{
   S_COLLATION_QUEUE_ITEM queue_item;

   FUNCTION_START(STB_EBUTT_Stop);

   if (is_initialised == TRUE)
   {
      stop_teletext = TRUE;

      STB_ChangePesCollectionPID(path, 0);

      stream_processing_enabled = FALSE;
      if (reset_cache)
      {
         DriverReset();

         /* Don't wait if the collation task has already been requested to stop */
         if (!stop_collation && collation_started)
         {
            /* Signal the collation task to stop */
            stop_collation = TRUE;

            /* Write an empty item to the queue to ensure the task isn't blocked
             * from seeing the request to stop */
            queue_item.data = NULL;
            if (STB_OSWriteQueue(collation_queue, &queue_item, sizeof(queue_item), TIMEOUT_NOW))
            {
               queue_count++;
            }

            /* Wait for the task to indicate it's stopped */
            if (!STB_OSSemaphoreWaitTimeout(collation_stopped, 200))
            {
               /* Task hasn't stopped yet so is probably blocked waiting for page display info */
               STB_OSSemaphoreSignal(page_free_sem);

               STB_OSSemaphoreWait(collation_stopped);
            }
         }

         queue_count = 0;
         max_queue_count = 0;

         /* Now clear any items on the queue */
         while (STB_OSReadQueue(collation_queue, &queue_item, sizeof(queue_item), TIMEOUT_NOW))
         {
            if (queue_item.data != NULL)
            {
               STB_FreeMemory(queue_item.data);
            }
         }

         if (is_shown == TRUE && is_initialised == TRUE)
         {
            /* STB_EBUTT_Hide has NOT been called */
            STB_OSDFillRegion( ttxt_osd_region, PALETTE_BACKGROUND_TRANSPARENT );
            STB_OSDUpdateRegions();
         }
      }
   }

   FUNCTION_FINISH(STB_EBUTT_Stop);
}

/**
 *

 *
 * @brief   Called to initiate continuous display of TeleText page content.
 *                 This will result in either the index page defined in the service data stream
 *                 being selected, or the previously displayed page (prior to a call to
 *                 STB_EBUTT_Hide( ) being made) from being reinstated.
 *                 This function can be called irrespective of the use of the STB_EBUTT_Start( ) /
 *                 STB_EBUTT_Stop( ) functions, but of course may result in no TeleText data being
 *                 displayed if none already exists within the cache.
 *                 The return value indicates whether initialisation of the display mechanism has
 *                 been successful or not.
 *
 * @param   E_STB_EBUTT_CHARACTER_SET_DESIGNATION character_set_designation -
 *                      used to specifiy the default character set designation for a level 1.0
 *                      Teletext broadcast. This is useful to configure the implementation to work
 *                      correctly when used with regionalised broadcasts.
 * @param   BOOLEAN navigate_to_index_page - set it TRUE if the index page defined within the
 *                                                  Teletext service is used, or the last visited
 *                                                  page is re-displayed.
 *
 * @return   TRUE if initialisation of diaply mechanism was successful, FALSE otherwise.
 *
 */
BOOLEAN STB_EBUTT_Show(E_EBUTT_CHARACTER_SET_DESIGNATION character_set_designation,
   BOOLEAN navigate_to_index_page, BOOLEAN show_header)
{
   BOOLEAN retval;

   FUNCTION_START(STB_EBUTT_Show);

   retval = FALSE;

   if ((is_initialised == TRUE) &&
       (is_shown == FALSE))
   {
      broadcast_index_page_invocation_required = navigate_to_index_page;
      show_header_row = show_header;

      default_character_set_mapping = (U8BIT)character_set_designation;

      page_control_info.video_mix_required = FALSE;
      page_control_info.clear_required = FALSE;

      STB_OSSemaphoreSignal(display_task_semaphore);

      palette_reset_required = TRUE;

      is_shown = TRUE;
      is_visible = FALSE;
      stop_teletext = FALSE;
      retval = TRUE;
   }

   FUNCTION_FINISH(STB_EBUTT_Show);

   return(retval);
}

/**
 *

 *
 * @brief   Called to halt the continuous display of TeleText page content, and to clear the
 *                 relevant area of the application display.
 *

 *

 *
 */
void STB_EBUTT_Hide(void)
{
   FUNCTION_START(STB_EBUTT_Hide);

   if ((is_initialised == TRUE) &&
       (is_shown == TRUE))
   {
      stop_teletext = TRUE;

      STB_OSSemaphoreWait(display_task_semaphore);

      STB_OSDFillRegion( ttxt_osd_region, PALETTE_BACKGROUND_TRANSPARENT );

      if (is_visible)
      {
         STB_OSDHideRegion( ttxt_osd_region );
         STB_OSDEnableUIRegion();
         is_visible = FALSE;
      }

      STB_OSDUpdateRegions();

      is_shown = FALSE;
   }

   FUNCTION_FINISH(STB_EBUTT_Hide);
}

/**
 *

 *
 * @brief   Called to define the strategy used to determine what page content is to be
 * displayed form the TeleText carousel content.
 *
 * @param   E_EBUTT_CACHING_METHOD ebutt_caching_method -
 *

 *
 */
void STB_EBUTT_SetCacheMethod(E_EBUTT_CACHING_METHOD ebutt_caching_method)
{
   FUNCTION_START(STB_EBUTT_SetCacheMethod);

   switch (ebutt_caching_method)
   {
      case EBUTT_CACHING_METHOD_PREVIOUS_NEXT:
      {
         // An nominal number of pages in the carousel numbering sequence that are close to the
         // currently displayed page are cached.
         // The minimal amount of memory allocation is required for this simple caching.
         history_cache_reset_requested = TRUE;

         caching_method = EBUTT_CACHING_METHOD_PREVIOUS_NEXT;
         break;
      }

      case EBUTT_CACHING_METHOD_HISTORY:
      {
         // As for EBUTT_CACHING_METHOD_PREVIOUS_NEXT, but also...
         // Previously visited pages are cached, up to a nominal limit.
         // The allows for easy travelral of a chain of recently visited pages.
         // Some memory allocation is required for this caching.
         editorial_link_cache_reset_requested = TRUE;

         caching_method = EBUTT_CACHING_METHOD_HISTORY;
         break;
      }

      case EBUTT_CACHING_METHOD_NAVIGATION:
      {
         // As for EBUTT_CACHING_METHOD_HISTORY, but also...
         // The FLOF (or FastText) page navigation links are cached for the
         // currently displayed page.
         // More complex memory allocation is required for this caching.
         editorial_link_cache_reset_requested = TRUE;

         caching_method = EBUTT_CACHING_METHOD_NAVIGATION;
         break;
      }

      case EBUTT_CACHING_METHOD_NAVIGATION_TREE:
      {
         // As for EBUTT_CACHING_METHOD_NAVIGATION, but also...
         // The pages defined in a tree of navigation links to the currently
         // displayed page are cached.
         // Complex memory allocation is required for this caching.

         caching_method = EBUTT_CACHING_METHOD_NAVIGATION_TREE;
         break;
      }

      case EBUTT_CACHING_METHOD_ALL:
      {
         // All TeleText carousel content is cached - offers optimal performance.
         // Significant amounts of memory allocation are required for this caching.
         caching_method = EBUTT_CACHING_METHOD_ALL;
         break;
      }

      default: // EBUTT_CACHING_METHOD_NONE
      {
         // No caching is performed - only presently displayed page and optionally next requested
         // page information is held in the system.
         history_cache_reset_requested = TRUE;
         editorial_link_cache_reset_requested = TRUE;

         caching_method = EBUTT_CACHING_METHOD_NONE;
         break;
      }
   }

   FUNCTION_FINISH(STB_EBUTT_SetCacheMethod);
}

/**
 *

 *
 * @brief   Called whenever a TeleText-specific event being invoked. This is used to pass
 *                 relevant user input to the driver, and the enumerate type supplied is defined
 *                 independently of any IR handset / keypad.
 *
 * @param   E_EBUTT_EVENT event_type -
 *

 *
 */
void STB_EBUTT_NotifyEvent(E_EBUTT_EVENT event_type)
{
   U16BIT page_number, index;
   BOOLEAN reset_state;

   FUNCTION_START(STB_EBUTT_NotifyEvent);

   if (is_initialised == TRUE)
   {
      reset_state = TRUE;

      switch (event_type)
      {
         // These are the four FastText shortcuts, usually represented by red, green,
         // yellow and blue keys on the handset.
         case EBUTT_EVENT_QUICK_NAVIGATE_1:
         case EBUTT_EVENT_QUICK_NAVIGATE_2:
         case EBUTT_EVENT_QUICK_NAVIGATE_3:
         case EBUTT_EVENT_QUICK_NAVIGATE_4:
         {
            RequestEditorialLinkPage((U16BIT)(event_type - EBUTT_EVENT_QUICK_NAVIGATE_1));
            break;
         }
         // The ten numeric keys used to input page indexes.
         case EBUTT_EVENT_0:
         case EBUTT_EVENT_1:
         case EBUTT_EVENT_2:
         case EBUTT_EVENT_3:
         case EBUTT_EVENT_4:
         case EBUTT_EVENT_5:
         case EBUTT_EVENT_6:
         case EBUTT_EVENT_7:
         case EBUTT_EVENT_8:
         case EBUTT_EVENT_9:
         {
            if (page_control_info.state != PAGE_CONTROL_STATE_INPUT)
            {
               if ((event_type != EBUTT_EVENT_0) &&
                   (event_type != EBUTT_EVENT_9))
               {
                  page_control_info.page_digit[0] = (U8BIT)(event_type - EBUTT_EVENT_0);
                  page_control_info.page_digit_offset = 1;

                  STB_OSSemaphoreWait(page_display_info.page_index_semaphore);

                  page_display_info.page_index_digit[0] = page_control_info.page_digit[0] +
                     PAGE_INDEX_DIGIT_0;
                  page_display_info.page_index_digit[1] = PAGE_INDEX_DIGIT_UNDEFINED;
                  page_display_info.page_index_digit[2] = PAGE_INDEX_DIGIT_UNDEFINED;

                  page_display_info.page_index_updated = TRUE;
                  page_display_info.page_index_held = FALSE;

                  page_control_info.state = PAGE_CONTROL_STATE_INPUT;

                  STB_OSSemaphoreSignal(page_display_info.page_index_semaphore);

                  reset_state = FALSE;
               }
            }
            else
            {
               page_control_info.page_digit[page_control_info.page_digit_offset] =
                  (U8BIT)(event_type - EBUTT_EVENT_0);

               STB_OSSemaphoreWait(page_display_info.page_index_semaphore);

               page_display_info.page_index_digit[page_control_info.page_digit_offset] =
                  page_control_info.page_digit[page_control_info.page_digit_offset] +
                  PAGE_INDEX_DIGIT_0;

               page_display_info.page_index_updated = TRUE;

               page_control_info.page_digit_offset++;

               if (page_control_info.page_digit_offset == 3)
               {
                  page_control_info.state = PAGE_CONTROL_STATE_IDLE;

                  for (index = 0, page_number = 0; index < 3; index++)
                  {
                     page_number <<= 4;
                     page_number += (U16BIT)page_control_info.page_digit[index];
                  }

                  RequestPage(page_number, PAGE_SUB_CODE_UNDEFINED);
               }
               else
               {
                  reset_state = FALSE;
               }

               STB_OSSemaphoreSignal(page_display_info.page_index_semaphore);
            }
            break;
         }

         // This is the home key, which returns to the nominated index page for this
         //   service.
         case EBUTT_EVENT_INDEXPAGE:
         {
            RequestIndexPage();
            break;
         }
         // These are used to quickly increment/decrement the page index.
         case EBUTT_EVENT_NEXTPAGE:
         {
            RequestNextAvailablePage();
            break;
         }
         case EBUTT_EVENT_PREVIOUSPAGE:
         {
            RequestPreviousAvailablePage();
            break;
         }
         // These are used to navigate the sub-pages when in 'hold' mode.
         case EBUTT_EVENT_NEXTSUBPAGE:
         {
            if (page_control_info.state == PAGE_CONTROL_STATE_HOLD)
            {
               RequestNextSubPage();

               reset_state = FALSE;
            }
            break;
         }
         case EBUTT_EVENT_PREVIOUSSUBPAGE:
         {
            if (page_control_info.state == PAGE_CONTROL_STATE_HOLD)
            {
               RequestPreviousSubPage();

               reset_state = FALSE;
            }
            break;
         }
         // These are used to traverse the page history (if caching requested).
         case EBUTT_EVENT_BACKPAGE:
         {
            RequestPreviousVisitedPage();
            break;
         }
         case EBUTT_EVENT_FORWARDPAGE:
         {
            RequestNextVisitedPage();
            break;
         }
         // This is used to toggle hold on the current page.
         case EBUTT_EVENT_HOLD:
         {
            if (page_control_info.clear_required == FALSE)
            {
               if (page_control_info.state == PAGE_CONTROL_STATE_HOLD)
               {
                  page_control_info.state = PAGE_CONTROL_STATE_IDLE;

                  STB_OSSemaphoreWait(page_request_info.semaphore);

                  page_request_info.target_page_sub_code = PAGE_SUB_CODE_UNDEFINED;

                  page_request_info.pending = FALSE;

                  STB_OSSemaphoreSignal(page_request_info.semaphore);

                  STB_OSSemaphoreWait(page_display_info.page_index_semaphore);

                  page_number = page_display_info.page_number;
                  if (page_number == 0)
                  {
                     page_number = page_request_info.target_page_number;
                  }

                  page_display_info.page_index_digit[0] =
                     ((page_number & 0x0f00) >> 8) + PAGE_INDEX_DIGIT_0;
                  page_display_info.page_index_digit[1] =
                     ((page_number & 0x00f0) >> 4) + PAGE_INDEX_DIGIT_0;
                  page_display_info.page_index_digit[2] =
                     (page_number & 0x000f) + PAGE_INDEX_DIGIT_0;

                  page_display_info.page_index_updated = TRUE;
                  page_display_info.page_index_held = FALSE;

                  STB_OSSemaphoreSignal(page_display_info.page_index_semaphore);
               }
               else
               {
                  page_control_info.state = PAGE_CONTROL_STATE_HOLD;

                  STB_OSSemaphoreWait(page_request_info.semaphore);

                  // Cancel and pending page request
                  page_request_info.type = REQUEST_TYPE_CANCEL;

                  page_request_info.pending = TRUE;

                  STB_OSSemaphoreSignal(page_request_info.semaphore);


                  STB_OSSemaphoreWait(page_display_info.page_index_semaphore);

                  // Force display task to redisplay the page number - now we are in 'hold' mode, the
                  // display task will show an icon accordingly.
                  page_display_info.page_index_updated = TRUE;
                  page_display_info.page_index_held = TRUE;

                  STB_OSSemaphoreSignal(page_display_info.page_index_semaphore);
               }
            }

            reset_state = FALSE;

            break;
         }
         // Reveal hidden page content (as defined in EBU specification)
         case EBUTT_EVENT_REVEAL:
         {
            page_control_info.reveal_required = !page_control_info.reveal_required;
            reset_state = FALSE;
            break;
         }
         // This key toggles 'clear' mode (page hidden until updated)
         case EBUTT_EVENT_CLEAR:
         {
            page_control_info.clear_required = !page_control_info.clear_required;

            if (page_control_info.state == PAGE_CONTROL_STATE_HOLD)
            {
               page_control_info.state = PAGE_CONTROL_STATE_IDLE;

               STB_OSSemaphoreWait(page_request_info.semaphore);

               page_request_info.target_page_sub_code = PAGE_SUB_CODE_UNDEFINED;

               page_request_info.pending = FALSE;

               STB_OSSemaphoreSignal(page_request_info.semaphore);

               STB_OSSemaphoreWait(page_display_info.page_index_semaphore);

               page_display_info.page_index_updated = TRUE;
               page_display_info.page_index_held = FALSE;

               STB_OSSemaphoreSignal(page_display_info.page_index_semaphore);
            }

            reset_state = FALSE;
            break;
         }
         // Used to toggle transparent background ('video mix' mode)
         case EBUTT_EVENT_MIX_VIDEO:
         {
            page_control_info.video_mix_required = !page_control_info.video_mix_required;
            reset_state = FALSE;
            break;
         }
         // Used to toggle double height top / double-height bottom / normal height display.
         case EBUTT_EVENT_DOUBLE_HEIGHT:
         {
            if (page_display_info.page_scale_params_defined == FALSE)
            {
               STB_OSSemaphoreWait(page_display_info.page_scale_semaphore);
               page_display_info.page_scale_status = PAGE_SCALE_STATUS_ZOOM_TOP;
               page_display_info.page_scale_start_row = 0;
               page_display_info.page_scale_num_rows = 13;
               page_display_info.page_scale_zoom_scalar = 2;
               page_display_info.page_rescaled = TRUE;
               STB_OSSemaphoreSignal(page_display_info.page_scale_semaphore);

               page_display_info.page_scale_params_defined = TRUE;
            }
            else
            {
               STB_OSSemaphoreWait(page_display_info.page_scale_semaphore);
               switch (page_display_info.page_scale_status)
               {
                  case PAGE_SCALE_STATUS_NORMAL:
                  {
                     page_display_info.page_scale_status = PAGE_SCALE_STATUS_ZOOM_TOP;
                     page_display_info.page_scale_start_row = 0;
                     page_display_info.page_scale_num_rows = 13;
                     page_display_info.page_scale_zoom_scalar = 2;
                     page_display_info.page_rescaled = TRUE;
                     break;
                  }
                  case PAGE_SCALE_STATUS_ZOOM_TOP:
                  {
                     if (page_display_info.page_scale_start_row == 0)
                     {
                        page_display_info.page_scale_status = PAGE_SCALE_STATUS_ZOOM_BOTTOM;
                        page_display_info.page_scale_start_row = 13;
                        page_display_info.page_scale_num_rows = 12;
                        page_display_info.page_scale_zoom_scalar = 2;
                        page_display_info.page_rescaled = TRUE;
                     }
                     else
                     {
                        page_display_info.page_scale_status = PAGE_SCALE_STATUS_NORMAL;
                        page_display_info.page_scale_start_row = 0;
                        page_display_info.page_scale_num_rows = MAX_ROWS;
                        page_display_info.page_scale_zoom_scalar = 1;
                        page_display_info.page_rescaled = TRUE;
                     }
                     break;
                  }
                  case PAGE_SCALE_STATUS_ZOOM_BOTTOM:
                  {
                     page_display_info.page_scale_status = PAGE_SCALE_STATUS_NORMAL;
                     page_display_info.page_scale_start_row = 0;
                     page_display_info.page_scale_num_rows = MAX_ROWS;
                     page_display_info.page_scale_zoom_scalar = 1;
                     page_display_info.page_rescaled = TRUE;
                     break;
                  }
               }
               STB_OSSemaphoreSignal(page_display_info.page_scale_semaphore);
            }

            reset_state = FALSE;
            break;
         }
         // Functional enhancement may offer finer scrolling of double-height display.
         case EBUTT_EVENT_DOUBLE_SCROLL_UP:
         {
            if (page_display_info.page_scale_params_defined == FALSE)
            {
               STB_OSSemaphoreWait(page_display_info.page_scale_semaphore);
               page_display_info.page_scale_status = PAGE_SCALE_STATUS_NORMAL;
               page_display_info.page_scale_start_row = 0;
               page_display_info.page_scale_num_rows = MAX_ROWS;
               page_display_info.page_scale_zoom_scalar = 1;
               page_display_info.page_rescaled = FALSE;
               STB_OSSemaphoreSignal(page_display_info.page_scale_semaphore);

               page_display_info.page_scale_params_defined = TRUE;
            }
            else
            {
               STB_OSSemaphoreWait(page_display_info.page_scale_semaphore);
               switch (page_display_info.page_scale_status)
               {
                  case PAGE_SCALE_STATUS_NORMAL:
                  {
                     break;
                  }
                  case PAGE_SCALE_STATUS_ZOOM_TOP:
                  case PAGE_SCALE_STATUS_ZOOM_BOTTOM:
                  {
                     if (page_display_info.page_scale_start_row > 0)
                     {
                        page_display_info.page_scale_start_row--;
                        page_display_info.page_rescaled = TRUE;
                     }
                     break;
                  }
               }
               STB_OSSemaphoreSignal(page_display_info.page_scale_semaphore);
            }

            reset_state = FALSE;
            break;
         }
         case EBUTT_EVENT_DOUBLE_SCROLL_DOWN:
         {
            if (page_display_info.page_scale_params_defined == FALSE)
            {
               STB_OSSemaphoreWait(page_display_info.page_scale_semaphore);
               page_display_info.page_scale_status = PAGE_SCALE_STATUS_NORMAL;
               page_display_info.page_scale_start_row = 0;
               page_display_info.page_scale_num_rows = MAX_ROWS;
               page_display_info.page_scale_zoom_scalar = 1;
               page_display_info.page_rescaled = FALSE;
               STB_OSSemaphoreSignal(page_display_info.page_scale_semaphore);

               page_display_info.page_scale_params_defined = TRUE;
            }
            else
            {
               STB_OSSemaphoreWait(page_display_info.page_scale_semaphore);
               switch (page_display_info.page_scale_status)
               {
                  case PAGE_SCALE_STATUS_NORMAL:
                  {
                     break;
                  }
                  case PAGE_SCALE_STATUS_ZOOM_TOP:
                  case PAGE_SCALE_STATUS_ZOOM_BOTTOM:
                  {
                     if ((page_display_info.page_scale_start_row +
                          page_display_info.page_scale_num_rows) < MAX_ROWS)
                     {
                        page_display_info.page_scale_start_row++;
                        page_display_info.page_rescaled = TRUE;
                     }
                     break;
                  }
               }
               STB_OSSemaphoreSignal(page_display_info.page_scale_semaphore);
            }

            reset_state = FALSE;
            break;
         }
         // Used to initiate/cancel 'timer' mode (clear and re-display page at set time)
         case EBUTT_EVENT_TIMER:
         {
            reset_state = FALSE;
            break;
         }
      }

      if (reset_state == TRUE)
      {
         page_control_info.state = PAGE_CONTROL_STATE_IDLE;

         STB_OSSemaphoreWait(page_display_info.page_index_semaphore);

         // Force display task to redisplay the page number
         page_display_info.page_index_updated = TRUE;
         page_display_info.page_index_held = FALSE;

         STB_OSSemaphoreSignal(page_display_info.page_index_semaphore);
      }
   }

   FUNCTION_FINISH(STB_EBUTT_NotifyEvent);
}

/**
 *

 *
 * @brief   Called whenever the application permits a TV/Radio service change during a
 *                 TeleText display session. This is provided because an update of the display to
 *                 new page content is required.
 *

 *

 *
 */
void STB_EBUTT_NotifyServiceChange(void)
{
   FUNCTION_START(STB_EBUTT_NotifyServiceChange);

   DriverReset();

   FUNCTION_FINISH(STB_EBUTT_NotifyServiceChange);
}

/**
 *

 *
 * @brief   Called to adjust the display brightness (colour intensity) of the Teletext pages.
 *
 * @param   U8BIT gun_intensity - a RGB gun intensity, ranging from 255 (brightest) to
 *                                       127 (darkest)
 *

 *
 */
void STB_EBUTT_SetDisplayBrightness(U8BIT gun_intensity)
{
   FUNCTION_START(STB_EBUTT_SetDisplayBrightness);

   if (gun_intensity < 127)
   {
      current_gun_intensity = 127;
   }
   else
   {
      current_gun_intensity = gun_intensity;
   }

   FUNCTION_FINISH(STB_EBUTT_SetDisplayBrightness);
}

/**
 *

 *
 * @brief   Called to adjust the display brightness (colour intensity) of the Teletext pages.
 *

 *

 *
 */
void STB_EBUTT_IncreaseDisplayBrightness(void)
{
   FUNCTION_START(STB_EBUTT_IncreaseDisplayBrightness);

   if (current_gun_intensity <= 247)
   {
      current_gun_intensity += 8;
   }

   FUNCTION_FINISH(STB_EBUTT_IncreaseDisplayBrightness);
}

void STB_EBUTT_DecreaseDisplayBrightness(void)
{
   FUNCTION_START(STB_EBUTT_DecreaseDisplayBrightness);

   if (current_gun_intensity >= 135)
   {
      current_gun_intensity -= 8;
   }

   FUNCTION_FINISH(STB_EBUTT_DecreaseDisplayBrightness);
}

/**
 *

 *
 * @brief   Called to adjust the display anti-aliasing level of the Teletext pages.
 *                 This is used to control 'flicker' on TV displays.
 *
 * @param   U8BIT antialias_level - a factor between zero (no anti-alaising) to 8 (full
 *                                         anti-aliasing)
 *

 *
 */
void STB_EBUTT_SetDisplayAntiAliasing(U8BIT antialias_level)
{
   FUNCTION_START(STB_EBUTT_SetDisplayAntiAliasing);

   if (antialias_level > 8)
   {
      current_antialias_level = 8;
   }
   else
   {
      current_antialias_level = antialias_level;
   }

   FUNCTION_FINISH(STB_EBUTT_SetDisplayAntiAliasing);
}

/**
 *

 *
 * @brief   Called to adjust the display anti-aliasing level of the Teletext pages.
 *

 *

 *
 */
void STB_EBUTT_IncreaseDisplayAntiAliasing(void)
{
   FUNCTION_START(STB_EBUTT_IncreaseDisplayAntiAliasing);

   if (current_antialias_level < 8)
   {
      current_antialias_level++;
   }

   FUNCTION_FINISH(STB_EBUTT_IncreaseDisplayAntiAliasing);
}

void STB_EBUTT_DecreaseDisplayAntiAliasing(void)
{
   FUNCTION_START(STB_EBUTT_DecreaseDisplayAntiAliasing);

   if (current_antialias_level > 0)
   {
      current_antialias_level--;
   }

   FUNCTION_FINISH(STB_EBUTT_DecreaseDisplayAntiAliasing);
}

/**
 *

 *
 * @brief   Called to adjust the display video-mix transparency level of the Teletext pages.
 *
 * @param   U8BIT transparency_level - a factor between zero (no transparency) to 255 (full
 *                                         transparency)
 *

 *
 */
void STB_EBUTT_SetDisplayMixTransparency(U8BIT transparency_level)
{
   FUNCTION_START(STB_EBUTT_SetDisplayMixTransparency);

   if (transparency_level < 31)
   {
      current_transparency_level = 31;
   }
   else
   {
      current_transparency_level = ((transparency_level - 31) & 0xf8) + 31;
   }

   FUNCTION_FINISH(STB_EBUTT_SetDisplayMixTransparency);
}

/**
 *

 *
 * @brief   Called to adjust the display video-mix transparency level of the Teletext pages.
 *

 *

 *
 */
void STB_EBUTT_IncreaseDisplayMixTransparency(void)
{
   FUNCTION_START(STB_EBUTT_IncreaseDisplayMixTransparency);

   if (current_transparency_level < 248)
   {
      current_transparency_level += 8;
   }
   else
   {
      current_transparency_level = 255;
   }

   FUNCTION_FINISH(STB_EBUTT_IncreaseDisplayMixTransparency);
}

void STB_EBUTT_DecreaseDisplayMixTransparency(void)
{
   FUNCTION_START(STB_EBUTT_DecreaseDisplayMixTransparency);

   if (current_transparency_level > 38)
   {
      current_transparency_level -= 8;
   }
   else
   {
      current_transparency_level = 31;
   }

   FUNCTION_FINISH(STB_EBUTT_DecreaseDisplayMixTransparency);
}

/**
 *

 *
 * @brief   Called to ascertain whether the present page is in 'hold' mode or not.
 *                 Thus function is useful when allocating handet keys to events on the basis of
 *                 the current functional state of the Teletext driver.
 *

 *
 * @return   TRUE if the displayed page is held, FALSE otherwise.
 *
 */
BOOLEAN STB_EBUTT_IsDisplayHeld(void)
{
   BOOLEAN retval;

   FUNCTION_START(STB_EBUTT_IsDisplayHeld);

   if (page_control_info.state == PAGE_CONTROL_STATE_HOLD)
   {
      retval = TRUE;
   }
   else
   {
      retval = FALSE;
   }

   FUNCTION_FINISH(STB_EBUTT_IsDisplayHeld);

   return(retval);
}

/**
 *

 *
 * @brief   Called to ascertain whether the present page is in 'double height' mode or not.
 *                 Thus function is useful when allocating handet keys to events on the basis of
 *                 the current functional state of the Teletext driver.
 *

 *
 * @return   TRUE if the displayed page is double height, FALSE otherwise.
 *
 */
BOOLEAN STB_EBUTT_IsDisplayDoubleHeight(void)
{
   BOOLEAN retval;

   FUNCTION_START(STB_EBUTT_IsDisplayDoubleHeight);

   if (page_display_info.page_scale_status == PAGE_SCALE_STATUS_NORMAL)
   {
      retval = FALSE;
   }
   else
   {
      retval = TRUE;
   }

   FUNCTION_FINISH(STB_EBUTT_IsDisplayDoubleHeight);

   return(retval);
}

//*****************************************************************************
// End of file
//*****************************************************************************

