/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Event reporting control functions
 *
 * @file    stberc.c
 * @date    15/02/2001
 */

//#define DEBUG_PRINTING_ENABLED

//---includes for this file----------------------------------------------------
// compiler library header files
#include <stdio.h>

// third party header files

// Ocean Blue Software header files

#include <techtype.h>
#include <dbgfuncs.h>

#include "stberc.h"
#include "stbpvr.h"
#include "stbdpc.h"
#include "stbhwos.h"

#ifdef INCLUDE_CA
#include "ca_glue.h"
#endif

//---constant definitions for this file----------------------------------------

#define EVENT_LATCHED_MAX        10    // Max number of outstanding latched events

#define INTERACTION_STACK_SIZE      1024
#define INTERACTION_TASK_PRIORITY   8  /* Not a critical task so can have a low priority */


//---local typedef structs for this file---------------------------------------

typedef struct
{
   BOOLEAN (*app_func)(BOOLEAN latched, BOOLEAN repeat, U16BIT class, U16BIT type, void *data, U32BIT data_size);
} ER_STATUS;

typedef struct
{
   U8BIT class;
   U16BIT type;
} ER_LATCHED;

//---local (static) variable declarations for this file------------------------
//   (internal variables declared static to make them local)

static void *er_protect;
static ER_STATUS event_status;
static ER_LATCHED latched_events[EVENT_LATCHED_MAX];

static void *user_interaction_sem = NULL;
static void *user_interaction_task = NULL;
static U32BIT user_interaction_timeout = 0;
static U32BIT last_user_interaction_time = 0;

//---local function prototypes for this file-----------------------------------
//   (internal functions declared static to make them local)

static void EventCallback(BOOLEAN repeat, U16BIT path_class, U16BIT type, void *data, U32BIT data_size);
static void UserInteractionTask(void *param);

//---local function definitions------------------------------------------------

/**
 *

 *
 * @brief   Callback function for HW events.
 *
 * @param   
 *

 *
 */
static void EventCallback(BOOLEAN repeat, U16BIT path_class, U16BIT type, void *data, U32BIT data_size)
{
   U8BIT path, class;
   U8BIT tuner_num;
   U32BIT rec_handle;
#ifdef INCLUDE_CA
   U32BIT ca_handle;
#endif

//   FUNCTION_START(STB_HWEventCallback); FLOODS THE DEBUGGER !!

   class = (U8BIT)(path_class & 0xff);

   switch (class)
   {
      case HW_EV_CLASS_HANDSET:
      {
         DBGPRINT("handset: r=%d t=%x", repeat, type);
         STB_ERSendEvent(FALSE, repeat, EV_CLASS_HSET, type, data, data_size);
         last_user_interaction_time = STB_OSGetClockMilliseconds();
         break;
      }

      case HW_EV_CLASS_KEYPAD:
      {
         DBGPRINT("keypad: t=%x", type);
         STB_ERSendEvent(FALSE, repeat, EV_CLASS_KEYP, type, data, data_size);
         last_user_interaction_time = STB_OSGetClockMilliseconds();
         break;
      }

      case HW_EV_CLASS_SCART:
      {
         switch (type)
         {
            case HW_EV_TYPE_TRUE:
            {
               STB_ERSendEvent(FALSE, repeat, EV_CLASS_SCART, EV_TYPE_FORCE, data, data_size);
               break;
            }

            case HW_EV_TYPE_FALSE:
            {
               STB_ERSendEvent(FALSE, repeat, EV_CLASS_SCART, EV_TYPE_DISCONNECTED, data, data_size);
               break;
            }

            case HW_EV_TYPE_4_3:
            {
               STB_ERSendEvent(FALSE, repeat, EV_CLASS_SCART, EV_TYPE_4_3, data, data_size);
               break;
            }

            case HW_EV_TYPE_16_9:
            {
               STB_ERSendEvent(FALSE, repeat, EV_CLASS_SCART, EV_TYPE_16_9, data, data_size);
               break;
            }
         }
         break;
      }

      case HW_EV_CLASS_LNB:
      {
         if (type == HW_EV_TYPE_FALSE)
         {
            STB_ERSendEvent(TRUE, repeat, EV_CLASS_LNB, EV_TYPE_FAIL, data, data_size);
         }
         break;
      }

      case HW_EV_CLASS_TUNER:
      {
         if (data != NULL)
         {
            /* A tuner can be associated with multiple paths so send event
             * to all paths using this tuner */
            path = INVALID_RES_ID;
            tuner_num = *(U8BIT *)data;

            while ((path = STB_DPPathForTuner(path, tuner_num)) != INVALID_RES_ID)
            {
               switch (type)
               {
                  case HW_EV_TYPE_LOCKED:
                  {
                     /* Clear any additional frequencies that have been set now the tuner has locked
                      * to prevent them being used in situations where the tuner loses lock */
                     STB_DPSetAdditionalFrequencies(path, 0, NULL);
                     STB_DPSetTuneStatus(path, TUNE_LOCKED);
                     break;
                  }

                  case HW_EV_TYPE_NOTLOCKED:
                  {
                     STB_DPSetTuneStatus(path, TUNE_NO_LOCK);
                     break;
                  }

                  default:
                  {
                     break;
                  }
               }
            }

            switch (type)
            {
               case HW_EV_TYPE_LOCKED:
               {
                  STB_ERSendEvent(FALSE, repeat, EV_CLASS_TUNE, EV_TYPE_LOCKED, data, data_size);
                  break;
               }

               case HW_EV_TYPE_NOTLOCKED:
               {
                  if (STB_DPTryAdditionalFrequency(tuner_num))
                  {
                     /* A possible alternative frequency has been set so try tuning again on all
                      * decode paths using the tuner and don't pass the not locked event on yet */
                     path = INVALID_RES_ID;
                     while ((path = STB_DPPathForTuner(path, tuner_num)) != INVALID_RES_ID)
                     {
                        STB_DPStartTune(path);
                     }
                  }
                  else
                  {
                     STB_ERSendEvent(FALSE, repeat, EV_CLASS_TUNE, EV_TYPE_NOTLOCKED, data, data_size);
                  }
                  break;
               }

               case HW_EV_TYPE_SIGNAL_DATA_BAD:
               {
                  STB_ERSendEvent(FALSE, repeat, EV_CLASS_TUNE, EV_TYPE_SIGNAL_DATA_BAD, data, data_size);
                  break;
               }

               case HW_EV_TYPE_SIGNAL_DATA_OK:
               {
                  STB_ERSendEvent(FALSE, repeat, EV_CLASS_TUNE, EV_TYPE_SIGNAL_DATA_OK, data, data_size);
                  break;
               }
            }
         }
         break;
      }

      case HW_EV_CLASS_DECODE:
      {
         if (data != NULL)
         {
            /* Map decoder to decode path */
            if ((type == HW_EV_TYPE_AUDIO_STARTED) || (type == HW_EV_TYPE_AUDIO_STOPPED) ||
               (type == HW_EV_TYPE_SAMPLE_STOPPED))
            {
               path = STB_DPPathForAudioDecoder(*(U8BIT *)data);
            }
            else if ((type == HW_EV_TYPE_VIDEO_STARTED) || (type == HW_EV_TYPE_VIDEO_STOPPED))
            {
               path = STB_DPPathForVideoDecoder(*(U8BIT *)data);
            }
            else if ((type == HW_EV_TYPE_AD_STARTED) || (type == HW_EV_TYPE_AD_STOPPED))
            {
               path = STB_DPPathForADDecoder(*(U8BIT *)data);
            }
         }
         else
         {
            path = INVALID_RES_ID;
         }

         if (path != INVALID_RES_ID)
         {
            if (type == HW_EV_TYPE_AUDIO_STARTED)
            {
               STB_DPSetAudioStatus(path, DECODE_RUNNING);
               STB_ERSendEvent(FALSE, repeat, EV_CLASS_DECODE, EV_TYPE_AUDIO_STARTED, data, data_size);
            }
            if (type == HW_EV_TYPE_VIDEO_STARTED)
            {
               STB_DPSetVideoStatus(path, DECODE_RUNNING);
               STB_ERSendEvent(FALSE, repeat, EV_CLASS_DECODE, EV_TYPE_VIDEO_STARTED, data, data_size);
            }
            if (type == HW_EV_TYPE_AD_STARTED)
            {
               STB_DPSetADStatus(path, DECODE_RUNNING);
               STB_ERSendEvent(FALSE, repeat, EV_CLASS_DECODE, EV_TYPE_AD_STARTED, data, data_size);
            }
            if (type == HW_EV_TYPE_AUDIO_STOPPED)
            {
               STB_DPSetAudioStatus(path, DECODE_STOPPED);
               STB_ERSendEvent(FALSE, repeat, EV_CLASS_DECODE, EV_TYPE_AUDIO_STOPPED, data, data_size);
            }
            if (type == HW_EV_TYPE_VIDEO_STOPPED)
            {
               if (STB_DPGetVideoStatus(path) == DECODE_STOPPING)
               {
                  STB_DPSetVideoStatus(path, DECODE_STOPPED);
                  STB_ERSendEvent(FALSE, repeat, EV_CLASS_DECODE, EV_TYPE_VIDEO_STOPPED, data, data_size);
               }
               else
               {
                  STB_DPSetVideoStatus(path, DECODE_STOPPED);
                  STB_ERSendEvent(FALSE, repeat, EV_CLASS_DECODE, EV_TYPE_FAIL, data, data_size);
               }
            }
            if (type == HW_EV_TYPE_AD_STOPPED)
            {
               STB_DPSetADStatus(path, DECODE_STOPPED);
               STB_ERSendEvent(FALSE, repeat, EV_CLASS_DECODE, EV_TYPE_AD_STOPPED, data, data_size);
            }
            if (type == HW_EV_TYPE_VIDEO_UNDERFLOW)
            {
               STB_ERSendEvent(FALSE, repeat, EV_CLASS_DECODE, EV_TYPE_VIDEO_UNDERFLOW, data, data_size);
            }
            if (type == HW_EV_TYPE_AUDIO_UNDERFLOW)
            {
               STB_ERSendEvent(FALSE, repeat, EV_CLASS_DECODE, EV_TYPE_AUDIO_UNDERFLOW, data, data_size);
            }
            if (type == HW_EV_TYPE_SAMPLE_STOPPED)
            {
               STB_DPSetAudioStatus(path, DECODE_STOPPED);
               STB_ERSendEvent(FALSE, repeat, EV_CLASS_DECODE, EV_TYPE_SAMPLE_STOPPED, data, data_size);
            }
         }
         break;
      }

      case HW_EV_CLASS_DVD:
      {
         if (type == HW_EV_TYPE_DVD_DISK_INSERTED)
         {
            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_DVD, EV_TYPE_DVD_DISK_INSERTED, data, data_size);
         }
         else if (type == HW_EV_TYPE_DVD_DISK_REMOVED)
         {
            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_DVD, EV_TYPE_DVD_DISK_REMOVED, data, data_size);
         }
         break;
      }

      case HW_EV_CLASS_DISK:
      {
         if (type == HW_EV_TYPE_DISK_CONNECTED)
         {
            if (STB_PVRIsInitialised())
            {
               /* PVR functions are interested in this event before it's passed on */
               STB_PVRUpdateRecordings(FALSE);
            }

            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_DISK, EV_TYPE_DISK_CONNECTED, data, data_size);
         }
         else if (type == HW_EV_TYPE_DISK_REMOVED)
         {
            if (STB_PVRIsInitialised())
            {
               /* PVR functions are interested in this event before it's passed on */
               STB_PVRUpdateRecordings(FALSE);
            }

            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_DISK, EV_TYPE_DISK_REMOVED, data, data_size);
         }
         else if (type == HW_EV_TYPE_DISK_FULL)
         {
            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_DISK, EV_TYPE_DISK_FULL, data, data_size);
         }
         break;
      }

      case HW_EV_CLASS_PVR:
      {
         switch (type)
         {
            case HW_EV_TYPE_PVR_REC_START:
               if ((data != NULL) && (data_size == sizeof(U8BIT)))
               {
                  /* Data sent with the event should be the recording index, so change this to
                   * the recording handle before passing it on to the app */
                  rec_handle = STB_PVRGetHandleForRecordingIndex(*(U8BIT *)data);
#ifdef INCLUDE_CA
                  /* Find the decode path being used for the recording */
                  path = STB_PVRGetPathForRecordingIndex(*(U8BIT*)data);
                  if ((path != INVALID_RES_ID) && STB_DPGetPathCADescrambler(path, &ca_handle))
                  {
                     STB_CANotifyRecordingStatus(ca_handle, TRUE);
                  }
#endif
               }
               else
               {
                  rec_handle = STB_PVR_INVALID_HANDLE;
               }
               STB_ERSendEvent(FALSE, repeat, EV_CLASS_PVR, EV_TYPE_PVR_REC_START, &rec_handle, sizeof(U32BIT));
               break;
            case HW_EV_TYPE_PVR_REC_STOP:
#ifdef INCLUDE_CA
               if ((data != NULL) && (data_size == sizeof(U8BIT)))
               {
                  /* Data sent with the event should be the recording index.
                   * Find the decode path from this */
                  path = STB_PVRGetPathForRecordingIndex(*(U8BIT*)data);
                  if ((path != INVALID_RES_ID) && STB_DPGetPathCADescrambler(path, &ca_handle))
                  {
                     STB_CANotifyRecordingStatus(ca_handle, FALSE);
                  }
               }
#endif
               STB_ERSendEvent(FALSE, repeat, EV_CLASS_PVR, EV_TYPE_PVR_REC_STOP, NULL, 0);
               break;
            case HW_EV_TYPE_PVR_PLAY_START:
               path = STB_DPGetPlaybackPath();
               STB_ERSendEvent(FALSE, repeat, EV_CLASS_PVR, EV_TYPE_PVR_PLAY_START, &path, sizeof(path));
               break;
            case HW_EV_TYPE_PVR_PLAY_STOP:
               path = STB_DPGetPlaybackPath();
               STB_ERSendEvent(FALSE, repeat, EV_CLASS_PVR, EV_TYPE_PVR_PLAY_STOP, &path, sizeof(path));
               break;
            case HW_EV_TYPE_PVR_PLAY_BOF:
               path = STB_DPGetPlaybackPath();
               STB_ERSendEvent(FALSE, repeat, EV_CLASS_PVR, EV_TYPE_PVR_PLAY_BOF, &path, sizeof(path));
               break;
            case HW_EV_TYPE_PVR_PLAY_EOF:
               path = STB_DPGetPlaybackPath();
               STB_ERSendEvent(FALSE, repeat, EV_CLASS_PVR, EV_TYPE_PVR_PLAY_EOF, &path, sizeof(path));
               break;
            case HW_EV_TYPE_PVR_PLAY_NOTIFY_TIME:
               path = STB_DPGetPlaybackPath();
               STB_PVRPlaybackNotifyTime(path);
               break;
            default:
               break;
         }
         break;
      }

      case HW_EV_CLASS_HDMI:
      {
         switch (type)
         {
            case HW_EV_TYPE_HDMI_CONNECT:
            {
               STB_ERSendEvent(FALSE, repeat, EV_CLASS_HDMI, EV_TYPE_CONNECTED, data, data_size);
               break;
            }

            case HW_EV_TYPE_HDMI_DISCONNECT:
            {
               STB_ERSendEvent(FALSE, repeat, EV_CLASS_HDMI, EV_TYPE_DISCONNECTED, data, data_size);
               break;
            }

            default:
               break;
         }
         break;
      }

      case HW_EV_CLASS_PRIVATE:
      {
         /* Pass all private events on without any changes */
         STB_ERSendEvent(FALSE, repeat, EV_CLASS_PRIVATE, type, data, data_size);
         break;
      }

      default:
      {
         break;
      }
   }

//   FUNCTION_FINISH(STB_HWEventCallback); FLOODS THE DEBUGGER !!
}

//---global function definitions-----------------------------------------------
/**
 *

 *
 * @brief   Initialises STB layer Event Reporting control.
 *

 *

 *
 */
void STB_ERInitialise(void)
{
   FUNCTION_START(STB_ERInitialise);

   er_protect = STB_OSCreateSemaphore();

   // register callback func and init hw events
   STB_OSRegisterCallback(&EventCallback);

   FUNCTION_FINISH(STB_ERInitialise);
}

/**
 *

 *
 * @brief   Called by app to register callback function for events.
 *
 * @param   callback function pointer
 *

 *
 */
void STB_ERRegisterHandler(BOOLEAN (*func)(BOOLEAN latched, BOOLEAN repeat, U16BIT class, U16BIT type,
      void *data, U32BIT data_size))
{
   FUNCTION_START(STB_ERRegisterHandler);

   event_status.app_func = func;

   FUNCTION_FINISH(STB_ERRegisterHandler);
}

/**
 *

 *
 * @brief   Called by app to notify consumption of latched event.
 *
 * @param   U8BIT class - event class
 * @param   U16BIT type - event type
 *

 *
 */
void STB_ERNotifyEvent(U8BIT event_class, U16BIT event_type)
{
   U8BIT latch_index;

   FUNCTION_START(STB_ERNotifyEvent);

   latch_index = 0;
   while (latch_index < EVENT_LATCHED_MAX)
   {
      // Clear flags and stop searching if this event is latched
      if ((latched_events[latch_index].class == event_class) &&
          (latched_events[latch_index].type == event_type))
      {
         // Clear the latched event
         latched_events[latch_index].class = 0;
         latched_events[latch_index].type = 0;
         break;
      }

      latch_index++;
   }

   FUNCTION_FINISH(STB_ERNotifyEvent);
}

/**
 *

 *
 * @brief   Sends an event to event reporting control module.
 *
 * @param   BOOLEAN latched - TRUE if event is to be latched
 * @param   BOOLEAN repeat - TRUE for handset repeat codes
 * @param   U8BIT class - event class
 * @param   U16BIT type - event type
 *

 *
 */
void STB_ERSendEvent(BOOLEAN latched, BOOLEAN repeat, U16BIT path_class, U16BIT type, void *data, U32BIT data_size)
{
   BOOLEAN do_queue_event;
   U8BIT latch_index;
   U8BIT free_index;
   U8BIT class;

   FUNCTION_START(STB_ERSendEvent);

   STB_OSSemaphoreWait(er_protect);

   // Gets set FALSE if there was already a latched event of same class and type
   do_queue_event = TRUE;

   if (latched == TRUE)
   {
      // Find an available slot in the latched events array
      class = (U8BIT)(path_class & 0xff);

      free_index = EVENT_LATCHED_MAX;
      for (latch_index = 0; latch_index < EVENT_LATCHED_MAX; latch_index++)
      {
         // Stop searching if this event is already latched
         if ((latched_events[latch_index].class == class) &&
             (latched_events[latch_index].type == type))
         {
            do_queue_event = FALSE;
            break;
         }

         if ((free_index == EVENT_LATCHED_MAX) && (latched_events[latch_index].class != 0))
         {
            free_index = latch_index;
         }
      }

      // If an available slot was found
      if (do_queue_event && (free_index != EVENT_LATCHED_MAX))
      {
         // Record the event details
         latched_events[free_index].class = class;
         latched_events[free_index].type = type;
      }
   }

   // Send the event to the queue if needed
   if (do_queue_event == TRUE)
   {
      // If an application callback is registered...
      if (event_status.app_func != NULL)
      {
         // ...make the callback with the event details
         (*event_status.app_func)(latched, repeat, path_class, type, data, data_size);
      }
   }

   STB_OSSemaphoreSignal(er_protect);

   FUNCTION_FINISH(STB_ERSendEvent);
}

/**
 * @brief   Sets the time, in seconds, after which the STB_EVENT_USER_INTERACTION_TIMEOUT event
 *          will be sent to indicate that there hasn't been any user interaction for the set
 *          period of time.
 * @param   timeout time in seconds after which an event will be sent
 */
void STB_ERSetUserInteractionTimeout(U32BIT timeout)
{
   FUNCTION_START(STB_ERSetUserInteractionTimeout);

   if (user_interaction_timeout != timeout * 1000)
   {
      user_interaction_timeout = timeout * 1000;

      if (user_interaction_task == NULL)
      {
         /* Auto standby time has been set, so start the monitoring task */
         if ((user_interaction_sem = STB_OSCreateSemaphore()) != NULL)
         {
            /* Grab the semaphore initially so the task won't run until required */
            STB_OSSemaphoreWait(user_interaction_sem);

            user_interaction_task = STB_OSCreateTask(UserInteractionTask, user_interaction_sem,
               INTERACTION_STACK_SIZE, INTERACTION_TASK_PRIORITY, (U8BIT*)"UserInteractionTask");
         }
      }
      else
      {
         /* Wake the task up so it can re-evaluate the timeout */
         STB_OSSemaphoreSignal(user_interaction_sem);
      }
   }

   FUNCTION_FINISH(STB_ERSetUserInteractionTimeout);
}

static void UserInteractionTask(void *sem)
{
   U32BIT timeout;

   while(TRUE)
   {
      /* Work out how long to wait */
      if (user_interaction_timeout == 0)
      {
         /* No user interaction timeout is set, so wait until one is set */
         STB_OSSemaphoreWait(sem);
      }
      else
      {
         timeout = user_interaction_timeout - STB_OSGetClockDiff(last_user_interaction_time);

         /* Wait for a max of 65 seconds (semaphore timeout is a 16 bit value) */
         if (timeout > 65000)
         {
            timeout = 65000;
         }

         STB_OSSemaphoreWaitTimeout(sem, (U16BIT)timeout);
      }

      if (user_interaction_timeout != 0)
      {
         if (STB_OSGetClockDiff(last_user_interaction_time) >= user_interaction_timeout)
         {
            /* Send event to indicate user interaction time has expired */
            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_TIMER, EV_TYPE_USER_INTERACTION_TIMEOUT, NULL, 0);

            /* Set last user interation time to stop the event from being sent continually */
            last_user_interaction_time = STB_OSGetClockMilliseconds();
         }
      }
   }
}

//*****************************************************************************
// End of file
//*****************************************************************************

