/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   General control functions
 *
 * @file    stbgc.c
 * @date    13/10/2000
 */

//---includes for this file----------------------------------------------------
// compiler library header files

#include <stdio.h>
#include <time.h>
#include <stdlib.h>
#include <string.h>

// third party header files

// Ocean Blue Software header files

#include <techtype.h>
#include <dbgfuncs.h>

#include "stbdpc.h"
#include "stbgc.h"
#include "stbhwos.h"
#include "stbhwfp.h"
#include "stbhwav.h"
#include "version.h"

#ifdef COMMON_INTERFACE
   #include "stbhwci.h" //   CI (for pre-standby shutdown)
#endif

//---constant definitions for this file----------------------------------------

#define LCODE_STRING_MAX         4
#define TIME_STRING_MAX          9
#define DATE_STRING_MAX          11
#define CLOCK_STRING_MAX         11
#define SERIAL_NO_MAX            50
#define VERSION_NO_MAX           9



//---local typedef structs for this file---------------------------------------

typedef struct
{
   U32BIT lang_code;
   BOOLEAN audio_signal;
   BOOLEAN time_set;
   BOOLEAN use_broadcast_time;
   U16BIT date_code;
   U8BIT time_hour;
   U8BIT time_min;
   U8BIT time_secs;
   U16BIT offset_change_date;
   U8BIT offset_change_hour;
   U8BIT offset_change_min;
   U8BIT offset_change_secs;
   S8BIT offset_old_hour;
   S8BIT offset_old_min;
   S8BIT offset_new_hour;
   S8BIT offset_new_min;
   S8BIT rtcoff_hour;
   S8BIT rtcoff_min;
} GC_STATUS;

//---local (static) variable declarations for this file------------------------
//   (internal variables declared static to make them local)

static GC_STATUS general_status;
static void *time_mutex;
static U8BIT lcode_string[LCODE_STRING_MAX];
static U8BIT time_string[TIME_STRING_MAX];
static U8BIT date_string[DATE_STRING_MAX];
static U8BIT clock_string[CLOCK_STRING_MAX];
static U8BIT serial_no[SERIAL_NO_MAX];
static U8BIT version_no[VERSION_NO_MAX];

//---local function prototypes for this file-----------------------------------
//   (internal functions declared static to make them local)

static void ExtractUTCDate(U16BIT code, U8BIT *wday, U8BIT *day, U8BIT *month, U16BIT *year);
static void AddTime(U16BIT code1, U8BIT hour1, U8BIT min1, U8BIT secs1,
   S16BIT ohour, S16BIT omin, S16BIT osecs,
   U16BIT *code, U8BIT *hour, U8BIT *min, U8BIT *secs);
static BOOLEAN FutureTime(U16BIT code1, U8BIT hour1, U8BIT min1, U8BIT secs1,
   U16BIT code2, U8BIT hour2, U8BIT min2, U8BIT secs2);
static void ConvertTimestamp(U32BIT timestamp, U16BIT *date, U8BIT *hour, U8BIT *mins, U8BIT *secs);
static U32BIT ConvertToTimestamp(U16BIT code, U8BIT hour, U8BIT min, U8BIT sec);
static void SetCTime( void );
static void GetCTime( void );
static void SetOffset(U16BIT code, U8BIT hour, U8BIT min, U8BIT secs,
   S8BIT ohour1, S8BIT omin1, S8BIT ohour2, S8BIT omin2);
static void GetOffset(U16BIT code, U8BIT hour, U8BIT min, U8BIT secs,
   S8BIT *ohour, S8BIT *omin);

//---local function definitions------------------------------------------------

/**
 *

 *
 * @brief   Extracts date information from UTC date code.
 *
 * @param   U16BIT code - UTC date code
 * @param   U8BIT* wday - returns weekday number (1-7)
 * @param   U8BIT* day - returns day number (1-31)
 * @param   U8BIT* month - returns month number (1-12)
 * @param   U16BIT* year - returns year number
 *

 *
 */
static void ExtractUTCDate(U16BIT code, U8BIT *wday, U8BIT *day, U8BIT *month, U16BIT *year)
{
   U32BIT t1, t2, t3, t4;

   FUNCTION_START(ExtractUTCDate);

   if (code != 0)
   {
      t1 = (U32BIT)code * 10L;            // MJD with 1 d.p.

      t2 = t1 - 150782L;                  // subtract 15078.2
      t2 *= 10L;                          // divide by 365.25
      t2 /= 36525L;                       // t2 is now equivalent to y'
      *year = (U16BIT)t2 + 1900;

      t3 = t1 - 149561L;                  // as before, MJD minus 14956.1

      t2 *= 36525L;                       // to be int( y' * 365.35)
      t2 /= 100L;

      t3 -= t2 * 10L;                     // subract int( y' * 365.35) with 1 d.p.
      t3 *= 1000L;                        // divide by 30.6001
      t3 /= 306001L;                      // t3 is now equivalent to m'

      t4 = t3;
      *month = (U8BIT)t3 - 1;

      t3 *= 306001L;                      // to be int( m' * 30.6001)
      t3 /= 10000L;
      *day = (U8BIT)((U32BIT)code - 14956UL - t2 - t3);

      if (t4 > 13)
      {
         *year = *year + 1;
         *month = *month - 12;
      }

      *wday = (U8BIT)((((U32BIT)code + 2L) % 7) + 1);
   }
   else
   {
      *wday = 0;
      *day = 0;
      *month = 0;
      *year = 0;
   }

   FUNCTION_FINISH(ExtractUTCDate);
}

/**
 *

 *
 * @brief   Adds/subtracts date/times
 *
 * @param   U16BIT code1 - value of first date / time.
 * @param   U8BIT hour1
 * @param   U8BIT min1
 * @param   U8BIT secs1
 * @param   S16BIT ohour - time to add / subtract
 * @param   S16BIT omin
 * @param   S16BIT osecs
 * @param   U16BIT* code - returns result of date / time.
 * @param   U8BIT* hour
 * @param   U8BIT* min
 * @param   U8BIT* secs
 *

 *
 */
static void AddTime(U16BIT code1, U8BIT hour1, U8BIT min1, U8BIT secs1,
   S16BIT ohour, S16BIT omin, S16BIT osecs,
   U16BIT *code, U8BIT *hour, U8BIT *min, U8BIT *secs)
{
   S32BIT rcode, rhour, rmin, rsecs;

   FUNCTION_START(AddTime);

   rcode = (S32BIT)code1;
   rhour = ((S32BIT)hour1 + (S32BIT)ohour);
   rmin = ((S32BIT)min1 + (S32BIT)omin);
   rsecs = ((S32BIT)secs1 + (S32BIT)osecs);

   while (rsecs < 0)
   {
      rsecs += 60;
      rmin -= 1;
   }
   while (rsecs > 59)
   {
      rsecs -= 60;
      rmin += 1;
   }

   while (rmin < 0)
   {
      rmin += 60;
      rhour -= 1;
   }
   while (rmin > 59)
   {
      rmin -= 60;
      rhour += 1;
   }

   while (rhour < 0)
   {
      rhour += 24;
      rcode -= 1;
   }
   while (rhour > 23)
   {
      rhour -= 24;
      rcode += 1;
   }

   if (code1 != 0)
      *code = (U16BIT)rcode;
   else
      *code = 0;
   *hour = (U8BIT)rhour;
   *min = (U8BIT)rmin;
   *secs = (U8BIT)rsecs;

   FUNCTION_FINISH(AddTime);
}

/**
 *

 *
 * @brief   Is first date/time past (or equal) second date/time?
 *
 * @param   U16BIT code1 - value of first date / time.
 * @param   U8BIT hour1
 * @param   U8BIT min1
 * @param   U8BIT secs1
 * @param   U16BIT code2 - value of second of date / time.
 * @param   U8BIT hour2
 * @param   U8BIT min2
 * @param   U8BIT secs2
 *
 * @return   BOOLEAN - TRUE if first time is now or in the future.
 *
 */
static BOOLEAN FutureTime(U16BIT code1, U8BIT hour1, U8BIT min1, U8BIT secs1,
   U16BIT code2, U8BIT hour2, U8BIT min2, U8BIT secs2)
{
   BOOLEAN ret_val = FALSE;

   FUNCTION_START(FutureTime);

   if (code1 > code2)
   {
      ret_val = TRUE;
   }
   if (code1 == code2)
   {
      if (hour1 > hour2)
      {
         ret_val = TRUE;
      }
      if (hour1 == hour2)
      {
         if (min1 > min2)
         {
            ret_val = TRUE;
         }
         if (min1 == min2)
         {
            if (secs1 >= secs2)
            {
               ret_val = TRUE;
            }
         }
      }
   }

   FUNCTION_FINISH(FutureTime);

   return(ret_val);
}

/**
 *

 *
 * @brief   Sets the real time clock copy of time from the current status
 *

 *

 *
 */
static void SetCTime( void )
{
   U32BIT now;
   U16BIT ecode;
   U8BIT ehour, emin, esecs;

   FUNCTION_START(SetCTime);

   now = ConvertToTimestamp(general_status.date_code, general_status.time_hour,
      general_status.time_min, general_status.time_secs);

   if (general_status.use_broadcast_time)
   {
      /* Set the GMT to the hardware */
      STB_OSSetClockGMT(now);
   }

   // calc local time
   GetOffset(general_status.date_code, general_status.time_hour,
      general_status.time_min, general_status.time_secs,
      &general_status.rtcoff_hour, &general_status.rtcoff_min);

   AddTime(general_status.date_code, general_status.time_hour, general_status.time_min, general_status.time_secs,
      (S16BIT)general_status.rtcoff_hour, (S16BIT)general_status.rtcoff_min, 0,
      &ecode, &ehour, &emin, &esecs);

   now = ConvertToTimestamp(ecode, ehour, emin, esecs);

   // Notify the hardware layer of the current local time
   STB_OSSetClockRTC(now);

   FUNCTION_FINISH(SetCTime);
}

/**
 * @brief   Converts a timestamp expressed in number of seconds since midnight (UTC) 1 January 1970
 * @param   timestamp number of seconds since midnight (UTC) 1 January 1970
 * @param   code pointer where the day (modified Julian dateline) will be saved
 * @param   hour pointer where the hour will be saved
 * @param   min pointer where the minute will be saved
 * @param   secs pointer where the second will be saved
 */
static void ConvertTimestamp(U32BIT timestamp, U16BIT *date, U8BIT *hour, U8BIT *mins, U8BIT *secs)
{
   time_t t;
   struct tm *tm_time;
   U32BIT date_code;

   FUNCTION_START(ConvertTimestamp);

   t = timestamp;
   tm_time = gmtime(&t);

   // Convert the year, month and date coefficients of 'C' compatible UTC format (days since 1970 00:00:00)
   // back into SI-specified MJD (modified Julian dateline) (days since 1st Jan 1900).
   // (this algorithm is derived from DVB (SI) specs - ETSI EN 300 468, Annex C)
   date_code = 14956L + (U32BIT)tm_time->tm_mday;
   // NB - algorithm is slightly different for January and Feburary, due to leap year synchronisation.
   if (tm_time->tm_mon < 2)
   {
      date_code += (((U32BIT)tm_time->tm_year - 1L) * 36525L) / 100L;
      date_code += (((U32BIT)tm_time->tm_mon + 14L) * 306001L) / 10000L;
   }
   else
   {
      date_code += ((U32BIT)tm_time->tm_year * 36525L) / 100L;
      date_code += (((U32BIT)tm_time->tm_mon + 2L) * 306001L) / 10000L;
   }

   *date = (U16BIT)date_code;
   *hour = (U8BIT)tm_time->tm_hour;
   *mins = (U8BIT)tm_time->tm_min;
   *secs = (U8BIT)tm_time->tm_sec;

   FUNCTION_FINISH(ConvertTimestamp);
}


/**
 * @brief   Convert the date code from SI-specified MJD (modified Julian dateline) (days since 1st
 *          Jan 1900) into 'C' compatible UTC format (seconds since 1970 00:00:00). In both cases,
 *          the time is measured from midnight, so no conversion on time quotient is required. MJD
 *          date code for 1st Jan 1970 is 40587 - this represents zero in UTC used by C libraries.
 * @param   code MJD date code
 * @param   hour Hours
 * @param   min Minutes
 * @param   sec Seconds
 * @return  Number of seconds since 1970 00:00:00
 */
static U32BIT ConvertToTimestamp(U16BIT code, U8BIT hour, U8BIT min, U8BIT sec)
{
   U32BIT timestamp;

   FUNCTION_START(ConvertToTimestamp);

   timestamp = (U32BIT)code;
   if (timestamp >= 40587)
      timestamp -= 40587;
   else
      timestamp = 0;

   // Convert date code (in days) and time into seconds
   timestamp *= 86400L;
   timestamp += (U32BIT)hour * 3600L;
   timestamp += (U32BIT)min * 60L;
   timestamp += (U32BIT)sec;

   FUNCTION_FINISH(ConvertToTimestamp);

   return timestamp;
}

/**
 *

 *
 * @brief   Sets the status copy of time from the real time clock
 *

 *

 *
 */
static void GetCTime( void )
{
   U32BIT t;
   U16BIT date;
   U8BIT hour, mins, secs;

   FUNCTION_START(GetCTime);

   // Read the current time from the hardware
   if (general_status.use_broadcast_time)
   {
      t = STB_OSGetClockRTC();
   }
   else
   {
      t = STB_OSGetClockGMT();
   }

   // If clock has been set, convert RTC values to correct values
   if (general_status.time_set == TRUE)
   {
      ConvertTimestamp(t, &date, &hour, &mins, &secs);

      #ifndef USE_SYSTEM_CLOCK
      AddTime(date, hour, mins, secs,
         (S16BIT)(general_status.rtcoff_hour * -1), (S16BIT)(general_status.rtcoff_min * -1), 0,
         &general_status.date_code, &general_status.time_hour, &general_status.time_min, &general_status.time_secs);
      #else
      general_status.time_hour = hour;
      general_status.time_min = mins;
      general_status.time_secs = secs;
      general_status.date_code = date;
      #endif //USE_SYSTEM_CLOCK
   }
   else // Clock has not yet been set, so store as zero
   {
      general_status.time_hour = 0;
      general_status.time_min = 0;
      general_status.time_secs = 0;
      general_status.date_code = 0;
   }

   FUNCTION_FINISH(GetCTime);
}

/**
 *

 *
 * @brief   Sets local time offsets
 *
 * @param   U16BIT code - date/time before which first offset applies
 * @param   U8BIT hour  - and after which second offset applies
 * @param   U8BIT min
 * @param   U8BIT secs
 * @param   S8BIT ohour1 - first local offset
 * @param   S8BIT omin1
 * @param   S8BIT ohour2 - second local offset
 * @param   S8BIT omin2
 *

 *
 */
static void SetOffset(U16BIT code, U8BIT hour, U8BIT min, U8BIT secs,
   S8BIT ohour1, S8BIT omin1, S8BIT ohour2, S8BIT omin2)
{
   FUNCTION_START(SetOffset);

   general_status.offset_change_date = code;
   general_status.offset_change_hour = hour;
   general_status.offset_change_min = min;
   general_status.offset_change_secs = secs;

   general_status.offset_old_hour = ohour1;
   general_status.offset_old_min = omin1;
   general_status.offset_new_hour = ohour2;
   general_status.offset_new_min = omin2;

   FUNCTION_FINISH(SetOffset);
}

/**
 *

 *
 * @brief   Returns correct local time offset for given date/time
 *
 * @param   U16BIT code - date/time for which offset is required
 * @param   U8BIT hour
 * @param   U8BIT min
 * @param   U8BIT secs
 * @param   S8BIT* ohour - returns local time offset
 * @param   S8BIT* omin
 *

 *
 */
static void GetOffset(U16BIT code, U8BIT hour, U8BIT min, U8BIT secs,
   S8BIT *ohour, S8BIT *omin)
{
   FUNCTION_START(GetOffset);

   if (FutureTime(code, hour, min, secs,
          general_status.offset_change_date, general_status.offset_change_hour,
          general_status.offset_change_min, general_status.offset_change_secs))
   {
      *ohour = general_status.offset_new_hour;
      *omin = general_status.offset_new_min;
   }
   else
   {
      *ohour = general_status.offset_old_hour;
      *omin = general_status.offset_old_min;
   }

   FUNCTION_FINISH(GetOffset);
}

//---global function definitions-----------------------------------------------

/**
 *

 *
 * @brief   Initialises general control.
 *

 *

 *
 */
void STB_GCInitialise(void)
{
   FUNCTION_START(STB_GCInitialise);

   general_status.lang_code = 0;
   general_status.audio_signal = FALSE;
   general_status.time_set = FALSE;
   general_status.use_broadcast_time = TRUE;
   general_status.date_code = 0;
   general_status.time_hour = 0;
   general_status.time_min = 0;
   general_status.time_secs = 0;
   general_status.offset_change_date = 0;
   general_status.offset_change_hour = 0;
   general_status.offset_change_min = 0;
   general_status.offset_change_secs = 0;
   general_status.offset_old_hour = 0;
   general_status.offset_old_min = 0;
   general_status.offset_new_hour = 0;
   general_status.offset_new_min = 0;
   general_status.rtcoff_hour = 0;
   general_status.rtcoff_min = 0;

   time_mutex = STB_OSCreateMutex();

   FUNCTION_FINISH(STB_GCInitialise);
}

/**
 *

 *
 * @brief   Writes SI search language code into general control store.
 *
 * @param   U32BIT lang - ISO language code.
 *

 *
 */
void STB_GCSetSearchLangCode(U32BIT lang)
{
   FUNCTION_START(STB_GCSetSearchLangCode);

   general_status.lang_code = lang;

   FUNCTION_FINISH(STB_GCSetSearchLangCode);
}

/**
 *

 *
 * @brief   Reads the SI search language code from general control store.
 *

 *
 * @return   U32BIT - ISO language code.
 *
 */
U32BIT STB_GCGetSearchLangCode(void)
{
   U32BIT ret_val;

   FUNCTION_START(STB_GCGetSearchLangCode);

   ret_val = general_status.lang_code;

   FUNCTION_FINISH(STB_GCGetSearchLangCode);

   return(ret_val);
}

/**
 *

 *
 * @brief   Converts the given 24bit ISO language code to a null-terminated string.
 *
 * @param   U32BIT lang - ISO language code.
 *
 * @return   U8BIT* - string pointer.
 *
 */
U8BIT* STB_GCGetLangCodeString(U32BIT lang)
{
   FUNCTION_START(STB_GCGetLangCodeString);

   lcode_string[0] = (U8BIT)((lang >> 16) & 0x000000ff);
   lcode_string[1] = (U8BIT)((lang >> 8) & 0x000000ff);
   lcode_string[2] = (U8BIT)((lang) & 0x000000ff);
   lcode_string[3] = '\0';

   FUNCTION_FINISH(STB_GCGetLangCodeString);

   return(lcode_string);
}

/**
 *

 *
 * @brief   Enables / disables CI standby state.
 *

 *

 *
 */
void STB_GCSetCIStandby(BOOLEAN standby)
{
   FUNCTION_START(STB_GCSetCIStandby);

#ifdef COMMON_INTERFACE
   if (standby == TRUE)
   {
      STB_CIStandbyOn();
   }
   else
   {
      STB_CIStandbyOff();
   }
#else
   // fool compiler into thinking these parameters are used - stops unnecessary warning
   USE_UNWANTED_PARAM(standby);
#endif

   FUNCTION_FINISH(STB_GCSetCIStandby);
}

/**
 *

 *
 * @brief   Retrieves and returns the fulls serial string, incorporating the hardware version
 *                 box serial number and application version.
 *

 *
 * @return   Pointer to the retrieved serial number.
 *
 */
U8BIT* STB_GCGetFullSerialString(void)
{
   FUNCTION_START(STB_GCGetFullSerialString);

   snprintf((char *)serial_no, SERIAL_NO_MAX, "%04x.%04x.%lu", STB_HWGetHwId(), STB_HWGetCustomerId(),
      (unsigned long)STB_HWGetBoxSerialNumber());

   FUNCTION_FINISH(STB_GCGetFullSerialString);

   return(serial_no);
}

/**
 *

 *
 * @brief   Retrieves and returns the library version number as a string.
 *

 *
 * @return   Pointer to the version number.
 *
 */
U8BIT* STB_GCGetVersionNumberString(void)
{
   FUNCTION_START(STB_GCGetVersionNumberString);

   snprintf((char *)version_no, VERSION_NO_MAX, "%02d.%02d.%02d", lib_ver_number[0], lib_ver_number[1], lib_ver_number[2]);

   FUNCTION_FINISH(STB_GCGetVersionNumberString);

   return(version_no);
}

/**
 *

 *
 * @brief   Enables/disables audio signal bleep and writes value into general control store.
 *
 * @param   BOOLEAN state - TRUE to turn on, else FALSE.
 *

 *
 */
void STB_GCSetAudioSignal(BOOLEAN state)
{
   FUNCTION_START(STB_GCSetAudioSignal);

   // jrg - add HI call in here
   if (state == TRUE)
   {
   }
   else
   {
   }

   general_status.audio_signal = state;

   FUNCTION_FINISH(STB_GCSetAudioSignal);
}

/**
 *

 *
 * @brief   Reads the current audio signal bleep state from general control store.
 *

 *
 * @return   BOOLEAN - TRUE if on, else FALSE.
 *
 */
BOOLEAN STB_GCGetAudioSignal(void)
{
   BOOLEAN ret_val;

   FUNCTION_START(STB_GCGetAudioSignal);

   ret_val = general_status.audio_signal;

   FUNCTION_FINISH(STB_GCGetAudioSignal);

   return(ret_val);
}

/**
 *

 *
 * @brief   Sets local time offset from GMT.
 *
 * @param   U8BIT ohour - hour offset (0-23).
 * @param   U8BIT omin - minute offset (0-59).
 * @param   BOOLEAN neg - TRUE if offset is negative, else FALSE.
 *

 *
 */
void STB_GCSetLocalTimeOffset(U8BIT ohour, U8BIT omin, BOOLEAN neg)
{
   S8BIT oh, om;

   FUNCTION_START(STB_GCSetLocalTimeOffset);

   ASSERT(ohour <= 23);
   ASSERT(omin <= 59);

   STB_OSMutexLock(time_mutex);

   // Update the local store from the hardware RTC
   GetCTime();

   if (neg)
   {
      oh = ((S8BIT)ohour * -1);
      om = ((S8BIT)omin * -1);
   }
   else
   {
      oh = (S8BIT)ohour;
      om = (S8BIT)omin;
   }
   SetOffset(0, 0, 0, 0, oh, om, oh, om);

   // Ensure 'C' format RTC is updated
   SetCTime();

   STB_OSMutexUnlock(time_mutex);

   FUNCTION_FINISH(STB_GCSetLocalTimeOffset);
}

/**
 *

 *
 * @brief   Reads local time offset from GMT.
 *
 * @param   U8BIT* ohour - returns hour offset (0-23).
 * @param   U8BIT* omin - returns minute offset (0-59).
 * @param   BOOLEAN* neg - returns TRUE if offset is negative, else FALSE.
 *

 *
 */
void STB_GCGetLocalTimeOffset(U8BIT *ohour, U8BIT *omin, BOOLEAN *neg)
{
   S8BIT oh, om;

   FUNCTION_START(STB_GCGetLocalTimeOffset);

   ASSERT(ohour != NULL);
   ASSERT(omin != NULL);
   ASSERT(neg != NULL);

   STB_OSMutexLock(time_mutex);

   // Update the local store from the hardware RTC
   GetCTime();

   // Get correct offset for now
   GetOffset(general_status.date_code, general_status.time_hour,
      general_status.time_min, general_status.time_secs,
      &oh, &om);

   STB_OSMutexUnlock(time_mutex);

   // return offset
   *ohour = (U8BIT)abs(oh);
   *omin = (U8BIT)abs(om);
   if ((oh < 0) || (om < 0))
   {
      *neg = TRUE;
   }
   else
   {
      *neg = FALSE;
   }

   FUNCTION_FINISH(STB_GCGetLocalTimeOffset);
}

/**
 *

 *
 * @brief   Sets new and old local time offset from GMT.
 *
 * @param   U16BIT code - date of change.
 * @param   U8BIT hour - hour of change (0-23).
 * @param   U8BIT min - minute of change (0-59).
 * @param   U8BIT secs - seconds of change (0-59).
 * @param   U8BIT ohour1 - old hour offset (0-23).
 * @param   U8BIT omin1 - old minute offset (0-59).
 * @param   U8BIT ohour2 - new hour offset (0-23).
 * @param   U8BIT omin2 - new minute offset (0-59).
 * @param   BOOLEAN neg - TRUE if offsets are negative, else FALSE.
 *

 *
 */
void STB_GCSetLocalTimeChange(U16BIT code, U8BIT hour, U8BIT min, U8BIT secs,
   U8BIT ohour1, U8BIT omin1, U8BIT ohour2, U8BIT omin2, BOOLEAN neg)
{
   S8BIT oh1, om1, oh2, om2;

   FUNCTION_START(STB_GCSetLocalTimeChange);

   ASSERT(hour <= 23);
   ASSERT(min <= 59);
   ASSERT(secs <= 59);
   ASSERT(ohour1 <= 23);
   ASSERT(omin1 <= 59);
   ASSERT(ohour2 <= 23);
   ASSERT(omin2 <= 59);

   STB_OSMutexLock(time_mutex);

   // Update the local store from the hardware RTC
   GetCTime();

   if (neg)
   {
      oh1 = ((S8BIT)ohour1 * -1);
      om1 = ((S8BIT)omin1 * -1);
      oh2 = ((S8BIT)ohour2 * -1);
      om2 = ((S8BIT)omin2 * -1);
   }
   else
   {
      oh1 = (S8BIT)ohour1;
      om1 = (S8BIT)omin1;
      oh2 = (S8BIT)ohour2;
      om2 = (S8BIT)omin2;
   }
   SetOffset(code, hour, min, secs, oh1, om1, oh2, om2);

   // Ensure 'C' format RTC is updated
   SetCTime();

   STB_OSMutexUnlock(time_mutex);

   FUNCTION_FINISH(STB_GCSetLocalTimeChange);
}

/**
 *

 *
 * @brief   Reads local time offset from GMT.
 *
 * @param   U16BIT code - date of change.
 * @param   U8BIT hour - hour of change (0-23).
 * @param   U8BIT min - minute of change (0-59).
 * @param   U8BIT secs - seconds of change (0-59).
 * @param   U8BIT* ohour - returns hour offset (0-23).
 * @param   U8BIT* omin - returns minute offset (0-59).
 * @param   BOOLEAN* neg - returns TRUE if offset is negative, else FALSE.
 *

 *
 */
void STB_GCGetLocalTimeChange(U16BIT code, U8BIT hour, U8BIT min, U8BIT secs,
   U8BIT *ohour, U8BIT *omin, BOOLEAN *neg)
{
   S8BIT oh, om;

   FUNCTION_START(STB_GCGetLocalTimeChange);

   ASSERT(hour <= 23);
   ASSERT(min <= 59);
   ASSERT(secs <= 59);
   ASSERT(ohour != NULL);
   ASSERT(omin != NULL);
   ASSERT(neg != NULL);

   USE_UNWANTED_PARAM(code);
   USE_UNWANTED_PARAM(hour);
   USE_UNWANTED_PARAM(min);
   USE_UNWANTED_PARAM(secs);

   STB_OSMutexLock(time_mutex);

   // Get correct offset for now
   GetCTime();
   GetOffset(general_status.date_code, general_status.time_hour,
      general_status.time_min, general_status.time_secs, &oh, &om);

   STB_OSMutexUnlock(time_mutex);

   // return offset
   *ohour = (U8BIT)abs(oh);
   *omin = (U8BIT)abs(om);
   if ((oh < 0) || (om < 0))
   {
      *neg = TRUE;
   }
   else
   {
      *neg = FALSE;
   }

   FUNCTION_FINISH(STB_GCGetLocalTimeChange);
}

/**
 *

 *
 * @brief   Sets current GMT time.
 *
 * @param   U8BIT hour - value of hour (0-23).
 * @param   U8BIT min -  value of minute (0-59).
 * @param   U8BIT secs - value of seconds (0-59).
 *

 *
 */
void STB_GCSetGMTTime(U8BIT hour, U8BIT min, U8BIT secs)
{
   FUNCTION_START(STB_GCSetGMTTime);

   ASSERT(hour <= 23);
   ASSERT(min <= 59);
   ASSERT(secs <= 59);

   STB_OSMutexLock(time_mutex);

   // Update the local store from the hardware RTC
   GetCTime();

   general_status.time_hour = hour;
   general_status.time_min = min;
   general_status.time_secs = secs;

   // Ensure 'C' format RTC is updated
   SetCTime();

   // Record that the time has now been set
   general_status.time_set = TRUE;

   STB_OSMutexUnlock(time_mutex);

   FUNCTION_FINISH(STB_GCSetGMTTime);
}

/**
 *

 *
 * @brief   Reads the current GMT hour.
 *

 *
 * @return   U8BIT - value of hour (0-23).
 *
 */
U8BIT STB_GCGetGMTHour(void)
{
   U8BIT ret_val;

   FUNCTION_START(STB_GCGetGMTHour);

   STB_OSMutexLock(time_mutex);

   // Update the local store from the hardware RTC
   GetCTime();

   ret_val = general_status.time_hour;

   STB_OSMutexUnlock(time_mutex);

   FUNCTION_FINISH(STB_GCGetGMTHour);

   return(ret_val);
}

/**
 *

 *
 * @brief   Reads the current GMT minute.
 *

 *
 * @return   U8BIT - value of minute (0-59).
 *
 */
U8BIT STB_GCGetGMTMin(void)
{
   U8BIT ret_val;

   FUNCTION_START(STB_GCGetGMTMin);

   STB_OSMutexLock(time_mutex);

   // Update the local store from the hardware RTC
   GetCTime();

   ret_val = general_status.time_min;

   STB_OSMutexUnlock(time_mutex);

   FUNCTION_FINISH(STB_GCGetGMTMin);

   return(ret_val);
}

/**
 *

 *
 * @brief   Reads the current GMT seconds.
 *

 *
 * @return   U8BIT - value of seconds (0-59).
 *
 */
U8BIT STB_GCGetGMTSecs(void)
{
   U8BIT ret_val;

   FUNCTION_START(STB_GCGetGMTSecs);

   STB_OSMutexLock(time_mutex);

   // Update the local store from the hardware RTC
   GetCTime();

   ret_val = general_status.time_secs;

   STB_OSMutexUnlock(time_mutex);

   FUNCTION_FINISH(STB_GCGetGMTSecs);

   return(ret_val);
}

/**
 *

 *
 * @brief   Sets current GMT date.
 *
 * @param   U16BIT code - date code in UTC.
 *

 *
 */
void STB_GCSetGMTDate(U16BIT code)
{
   FUNCTION_START(STB_GCSetGMTDate);

   STB_OSMutexLock(time_mutex);

   // Update the local store from the hardware RTC
   GetCTime();

   general_status.date_code = code;

   // Ensure 'C' format RTC is updated
   SetCTime();

   // Record that the time has now been set
   general_status.time_set = TRUE;

   STB_OSMutexUnlock(time_mutex);

   FUNCTION_FINISH(STB_GCSetGMTDate);
}

/**
 *

 *
 * @brief   Reads the current GMT date code
 *

 *
 * @return   U16BIT - value of date code.
 *
 */
U16BIT STB_GCGetGMTDate(void)
{
   U16BIT ret_val;

   FUNCTION_START(STB_GCGetGMTDate);

   STB_OSMutexLock(time_mutex);

   // Update the local store from the hardware RTC
   GetCTime();

   ret_val = general_status.date_code;

   STB_OSMutexUnlock(time_mutex);

   FUNCTION_FINISH(STB_GCGetGMTDate);

   return(ret_val);
}

/**
 *

 *
 * @brief   Reads the current GMT weekday number.
 *

 *
 * @return   E_STB_GC_WEEKDAY - weekday number (1-7).
 *
 */
E_STB_GC_WEEKDAY STB_GCGetGMTWeekDay(void)
{
   U8BIT wday, day, month;
   U16BIT year;
   E_STB_GC_WEEKDAY ret_val;

   FUNCTION_START(STB_GCGetGMTWeekDay);

   STB_OSMutexLock(time_mutex);

   // Update the local store from the hardware RTC
   GetCTime();

   ExtractUTCDate(general_status.date_code, &wday, &day, &month, &year);

   STB_OSMutexUnlock(time_mutex);

   ret_val = (E_STB_GC_WEEKDAY)wday;

   FUNCTION_FINISH(STB_GCGetGMTWeekDay);

   return(ret_val);
}

/**
 *

 *
 * @brief   Reads the current GMT day number.
 *

 *
 * @return   U8BIT - day number (1-31).
 *
 */
U8BIT STB_GCGetGMTDay(void)
{
   U8BIT wday, day, month;
   U16BIT year;

   FUNCTION_START(STB_GCGetGMTDay);

   STB_OSMutexLock(time_mutex);

   // Update the local store from the hardware RTC
   GetCTime();

   ExtractUTCDate(general_status.date_code, &wday, &day, &month, &year);

   STB_OSMutexUnlock(time_mutex);

   FUNCTION_FINISH(STB_GCGetGMTDay);

   return(day);
}

/**
 *

 *
 * @brief   Reads the current GMT month number.
 *

 *
 * @return   U8BIT - month number (1-12).
 *
 */
U8BIT STB_GCGetGMTMonth(void)
{
   U8BIT wday, day, month;
   U16BIT year;

   FUNCTION_START(STB_GCGetGMTMonth);

   STB_OSMutexLock(time_mutex);

   // Update the local store from the hardware RTC
   GetCTime();

   ExtractUTCDate(general_status.date_code, &wday, &day, &month, &year);

   STB_OSMutexUnlock(time_mutex);

   FUNCTION_FINISH(STB_GCGetGMTMonth);

   return(month);
}

/**
 *

 *
 * @brief   Reads the current GMT year number.
 *

 *
 * @return   U16BIT - value of year.
 *
 */
U16BIT STB_GCGetGMTYear(void)
{
   U8BIT wday, day, month;
   U16BIT year;

   FUNCTION_START(STB_GCGetGMTYear);

   STB_OSMutexLock(time_mutex);

   // Update the local store from the hardware RTC
   GetCTime();

   ExtractUTCDate(general_status.date_code, &wday, &day, &month, &year);

   STB_OSMutexUnlock(time_mutex);

   FUNCTION_FINISH(STB_GCGetGMTYear);

   return(year);
}

/**
 *

 *
 * @brief   Reads the current GMT date code and time
 *
 * @param   U16BIT* code - returns value of date code.
 * @param   U8BIT* hour - returns value of hour (0-23).
 * @param   U8BIT* min - returns value of minute (0-59).
 * @param   U8BIT* secs - returns value of seconds (0-59).
 *

 *
 */
void STB_GCGetGMTDateTime(U16BIT *code, U8BIT *hour, U8BIT *min, U8BIT *secs)
{
   FUNCTION_START(STB_GCGetGMTDateTime);

   ASSERT(code != NULL);
   ASSERT(hour != NULL);
   ASSERT(min != NULL);
   ASSERT(secs != NULL);

   STB_OSMutexLock(time_mutex);

   // Update the local store from the hardware RTC
   GetCTime();

   *code = general_status.date_code;
   *hour = general_status.time_hour;
   *min = general_status.time_min;
   *secs = general_status.time_secs;

   STB_OSMutexUnlock(time_mutex);

   FUNCTION_FINISH(STB_GCGetGMTDateTime);
}

/**
 *

 *
 * @brief   Tests given date and time against current GMT date and time
 *
 * @param   U16BIT code - value of date code.
 * @param   U8BIT hour - value of hour (0-23).
 * @param   U8BIT min - value of minute (0-59).
 * @param   U8BIT secs - value of seconds (0-59).
 *
 * @return   BOOLEAN - TRUE of given date/time is in the future (strictly greater than) else
 *                 FALSE.
 *
 */
BOOLEAN STB_GCIsFutureDateTime(U16BIT code, U8BIT hour, U8BIT min, U8BIT secs)
{
   BOOLEAN ret_val;

   FUNCTION_START(STB_GCIsFutureDateTime);

   ASSERT(hour <= 23);
   ASSERT(min <= 59);
   ASSERT(secs <= 59);

   STB_OSMutexLock(time_mutex);

   // Update the local store from the hardware RTC
   GetCTime();

   ret_val = !FutureTime(general_status.date_code, general_status.time_hour,
         general_status.time_min, general_status.time_secs,
         code, hour, min, secs);

   STB_OSMutexUnlock(time_mutex);

   FUNCTION_FINISH(STB_GCIsFutureDateTime);

   return(ret_val);
}

/**
 *

 *
 * @brief   Returns the weekday number of the specified date code.
 *
 * @param   U16BIT code - date code.
 *
 * @return   E_STB_GC_WEEKDAY - weekday number (1-7).
 *
 */
E_STB_GC_WEEKDAY STB_GCGetDateWeekDay(U16BIT code)
{
   E_STB_GC_WEEKDAY ret_val;

   FUNCTION_START(STB_GCGetDateWeekDay);

   ret_val = (E_STB_GC_WEEKDAY)(((code + 2L) % 7) + 1);

   FUNCTION_FINISH(STB_GCGetDateWeekDay);

   return(ret_val);
}

/**
 *

 *
 * @brief   Tests whether the weekday of the specified date code is during the week.
 *
 * @param   U16BIT code - date code.
 *
 * @return   BOOLEAN - TRUE if week day (1-5).
 *
 */
BOOLEAN STB_GCIsDateDayWeek(U16BIT code)
{
   E_STB_GC_WEEKDAY wday;
   BOOLEAN ret_val;

   FUNCTION_START(STB_GCIsDateDayWeek);

   wday = STB_GCGetDateWeekDay(code);
   if ((wday >= WEEKDAY_MONDAY) && (wday <= WEEKDAY_FRIDAY))
      ret_val = TRUE;
   else
      ret_val = FALSE;

   FUNCTION_FINISH(STB_GCIsDateDayWeek);

   return(ret_val);
}

/**
 *

 *
 * @brief   Tests whether the weekday of the specified date code is during the weekend.
 *
 * @param   U16BIT code - date code.
 *
 * @return   BOOLEAN - TRUE if week day (6-7).
 *
 */
BOOLEAN STB_GCIsDateDayWeekend(U16BIT code)
{
   E_STB_GC_WEEKDAY wday;
   BOOLEAN ret_val;

   FUNCTION_START(STB_GCIsDateDayWeekend);

   wday = STB_GCGetDateWeekDay(code);
   if ((wday >= WEEKDAY_SATURDAY) && (wday <= WEEKDAY_SUNDAY))
      ret_val = TRUE;
   else
      ret_val = FALSE;

   FUNCTION_FINISH(STB_GCIsDateDayWeekend);

   return(ret_val);
}

/**
 *

 *
 * @brief   Adds or subtracts offset from a date/time.
 *
 * @param   U16BIT code - date code.
 * @param   U8BIT hour - hour (0-23).
 * @param   U8BIT min - minute (0-59).
 * @param   U8BIT secs - seconds (0-59).
 * @param   U8BIT ohour - offset hour (0-23).
 * @param   U8BIT omin - offset minute (0-59).
 * @param   U8BIT osecs - offset seconds (0-59).
 * @param   U16BIT* rcode - returns result date code.
 * @param   U8BIT* rhour - returns result hour (0-23).
 * @param   U8BIT* rmin - returns result minute (0-59).
 * @param   U8BIT* rsecs - returns result seconds (0-59).
 *

 *
 */
void STB_GCCalculateDateTime(U16BIT code, U8BIT hour, U8BIT min, U8BIT secs,
   U8BIT ohour, U8BIT omin, U8BIT osecs,
   U16BIT *rcode, U8BIT *rhour, U8BIT *rmin, U8BIT *rsecs,
   E_STB_GC_CALCTYPE calc)
{
   FUNCTION_START(STB_GCCalculateDateTime);

   ASSERT(rcode != NULL);
   ASSERT(rhour != NULL);
   ASSERT(rmin != NULL);
   ASSERT(rsecs != NULL);

   switch (calc)
   {
      case CALC_ADD:
         AddTime(code, hour, min, secs,
         (S16BIT)ohour, (S16BIT)omin, (S16BIT)osecs,
         rcode, rhour, rmin, rsecs);
         break;

      case CALC_SUB:
         AddTime(code, hour, min, secs,
         ((S16BIT)ohour * -1), ((S16BIT)omin * -1), ((S16BIT)osecs * -1),
         rcode, rhour, rmin, rsecs);
         break;

      default:
         break;
   }

   FUNCTION_FINISH(STB_GCCalculateDateTime);
}

/**
 *

 *
 * @brief   Converts specified date / time to another.
 *
 * @param   U16BIT code - date code.
 * @param   U8BIT hour - hour (0-23).
 * @param   U8BIT min - minute (0-59).
 * @param   U8BIT secs - seconds (0-59).
 * @param   U16BIT* rcode - returns result date code.
 * @param   U8BIT* rhour - returns result hour (0-23).
 * @param   U8BIT* rmin - returns result minute (0-59).
 * @param   U8BIT* rsecs - returns result seconds (0-59).
 *

 *
 */
void STB_GCConvertDateTime(U16BIT code, U8BIT hour, U8BIT min, U8BIT secs,
   U16BIT *rcode, U8BIT *rhour, U8BIT *rmin, U8BIT *rsecs,
   E_STB_GC_CONVTYPE conv)
{
   S8BIT ohour, omin;

   FUNCTION_START(STB_GCConvertDateTime);

   ASSERT(rcode != NULL);
   ASSERT(rhour != NULL);
   ASSERT(rmin != NULL);
   ASSERT(rsecs != NULL);

   STB_OSMutexLock(time_mutex);

   GetCTime();
   GetOffset(code, hour, min, secs, &ohour, &omin);

   switch (conv)
   {
      case CONV_LOCAL:
         AddTime(code, hour, min, secs, (S16BIT)ohour, (S16BIT)omin, 0, rcode, rhour, rmin, rsecs);
         break;

      case CONV_GMT:
         AddTime(code, hour, min, secs, (S16BIT)(ohour * -1), (S16BIT)(omin * -1), 0,
         rcode, rhour, rmin, rsecs);
         break;

      default:
         break;
   }

   STB_OSMutexUnlock(time_mutex);

   FUNCTION_FINISH(STB_GCConvertDateTime);
}

/**
 *

 *
 * @brief   Compares one date/time to another.
 *
 * @param   U16BIT code1 - first date code.
 * @param   U8BIT hour1 - first hour (0-23).
 * @param   U8BIT min1 - first minute (0-59).
 * @param   U8BIT secs1 - first seconds (0-59).
 * @param   U16BIT code2 - second date code.
 * @param   U8BIT hour2 - second hour (0-23).
 * @param   U8BIT min2 - second minute (0-59).
 * @param   U8BIT secs2 - second seconds (0-59).
 *
 * @return   TRUE- if comparison is valid, else FALSE.
 *
 */
BOOLEAN STB_GCCompareDateTime(U16BIT code1, U8BIT hour1, U8BIT min1, U8BIT secs1,
   U16BIT code2, U8BIT hour2, U8BIT min2, U8BIT secs2,
   E_STB_GC_COMPTYPE comp)
{
   BOOLEAN ret_val = FALSE;

   FUNCTION_START(STB_GCCompareDateTime);


   /********************************************************/
   /* CQ99999:PRM:06/02/28                                 */
   /********************************************************/
   /* Changed COMP_EQUAL to COMP_MATCH multiple Definition */
   /********************************************************/
   switch (comp)
   {
      case COMP_MATCH:
         if ((code1 == code2) && (hour1 == hour2) && (min1 == min2) && (secs1 == secs2))
         {
            ret_val = TRUE;
         }
         break;

      case COMP_1GT2:
      case COMP_2LT1:
         ret_val = FutureTime(code2, hour2, min2, secs2, code1, hour1, min1, secs1);
         ret_val = !ret_val;
         break;

      case COMP_1GE2:
      case COMP_2LE1:
         ret_val = FutureTime(code1, hour1, min1, secs1, code2, hour2, min2, secs2);
         break;

      case COMP_1LT2:
      case COMP_2GT1:
         ret_val = FutureTime(code1, hour1, min1, secs1, code2, hour2, min2, secs2);
         ret_val = !ret_val;
         break;

      case COMP_1LE2:
      case COMP_2GE1:
         ret_val = FutureTime(code2, hour2, min2, secs2, code1, hour1, min1, secs1);
         break;

      default:
         break;
   }

   FUNCTION_FINISH(STB_GCCompareDateTime);

   return(ret_val);
}

/*!**************************************************************************
 * @brief   Returns the difference in seconds between the two dates/times, as time1 - time2,
 *          so the result will be negative if time1 is earlier than time2, and positive otherwise
 * @param   code1 - MJD date code
 * @param   hour1 - MJD hour
 * @param   min1 - MJD minutes
 * @param   secs1 - MJD seconds
 * @param   code2 - MJD date code
 * @param   hour2 - MJD hour
 * @param   min2 - MJD minutes
 * @param   secs2 - MJD seconds
 * @return  number of seconds between the two dates/times
 ****************************************************************************/
S32BIT STB_GCDateTimeDiff(U16BIT code1, U8BIT hour1, U8BIT min1, U8BIT secs1,
   U16BIT code2, U8BIT hour2, U8BIT min2, U8BIT secs2)
{
   S32BIT time1, time2;

   FUNCTION_START(STB_GCDateTimeDiff);

   time1 = ((code1 * 24 + hour1) * 60 + min1) * 60 + secs1;
   time2 = ((code2 * 24 + hour2) * 60 + min2) * 60 + secs2;

   FUNCTION_FINISH(STB_GCDateTimeDiff);

   return(time1 - time2);
}

/**
 *

 *
 * @brief   Supplies specified time as local time string, in format requested.
 *
 * @param   U16BIT code - date code.
 * @param   U8BIT hour - value of hour (0-23).
 * @param   U8BIT min -  value of minute (0-59).
 * @param   E_STB_GC_TIMETYPE format - string format required.
 *
 * @return   U8BIT* - string pointer.
 *
 */
U8BIT* STB_GCGetTimeString(U16BIT code, U8BIT hour, U8BIT min, E_STB_GC_TIMETYPE format)
{
   U16BIT lcode;
   U8BIT lhour, lmin, lsecs;
   S8BIT ohour, omin;

   FUNCTION_START(STB_GCGetTimeString);

   ASSERT(hour <= 23);
   ASSERT(min <= 59);

   STB_OSMutexLock(time_mutex);

   // calculate local time
   GetCTime();
   GetOffset(code, hour, min, 0, &ohour, &omin);
   AddTime(code, hour, min, 0, (S16BIT)ohour, (S16BIT)omin, 0,
      &lcode, &lhour, &lmin, &lsecs);

   STB_OSMutexUnlock(time_mutex);

   switch (format)
   {
      case TIME_12H:
         if (lhour > 11)
         {
            lhour -= 12;
            if (lhour == 0)
               lhour = 12;
            snprintf((char *)time_string, TIME_STRING_MAX, "%d:%02d pm", lhour, lmin);
         }
         else
         {
            if (lhour == 0)
               lhour = 12;
            snprintf((char *)time_string, TIME_STRING_MAX, "%d:%02d am", lhour, lmin);
         }
         break;

      case TIME_24H:
         snprintf((char *)time_string, TIME_STRING_MAX, "%02d:%02d", lhour, lmin);
         break;

      default:
         break;
   }

   FUNCTION_FINISH(STB_GCGetTimeString);

   return(time_string);
}

/**
 *

 *
 * @brief   Supplies specifed date as local date string, in format requested.
 *
 * @param   U16BIT code - date code.
 * @param   U8BIT hour - value of hour (0-23).
 * @param   U8BIT min -  value of minute (0-59).
 * @param   E_STB_GC_DATETYPE format - string format required.
 *
 * @return   U8BIT* - string pointer.
 *
 */
U8BIT* STB_GCGetDateString(U16BIT code, U8BIT hour, U8BIT min, E_STB_GC_DATETYPE format)
{
   U16BIT lcode, year;
   U8BIT lhour, lmin, lsecs, wday, day, month;
   S8BIT ohour, omin;

   FUNCTION_START(STB_GCGetDateString);

   ASSERT(hour <= 23);
   ASSERT(min <= 59);

   STB_OSMutexLock(time_mutex);

   // calculate local time and date
   GetCTime();
   GetOffset(code, hour, min, 0, &ohour, &omin);
   AddTime(code, hour, min, 0, (S16BIT)ohour, (S16BIT)omin, 0,
      &lcode, &lhour, &lmin, &lsecs);
   ExtractUTCDate(lcode, &wday, &day, &month, &year);

   STB_OSMutexUnlock(time_mutex);

   switch (format)
   {
      case DATE_DMY:
         snprintf((char *)date_string, DATE_STRING_MAX, "%02d/%02d/%04d", day, month, year);
         break;

      case DATE_YMD:
         snprintf((char *)date_string, DATE_STRING_MAX, "%04d/%02d/%02d", year, month, day);
         break;

      default:
         break;
   }

   FUNCTION_FINISH(STB_GCGetDateString);

   return(date_string);
}

/**
 *

 *
 * @brief   Get the day, weekday, month and the year from a date code and
 *                 hour / minute offset.
 *
 * @param   U16BIT code - date code.
 * @param   U8BIT hour - value of hour (0-23).
 * @param   U8BIT min -  value of minute (0-59).
 *
 * @param   U8BIT* day -  Day of the month returned value
 * @param   U8BIT* wday - Week day returned value
 * @param   U8BIT* month - Month number returned value
 * @param   U16BIT* year - Year returned value
 *
 */
void STB_GCGetDateInfo(U16BIT code, U8BIT hour, U8BIT min, U8BIT *day, U8BIT *wday,
   U8BIT *month, U16BIT *year)
{
   U16BIT lcode;
   U8BIT lhour, lmin, lsecs;
   S8BIT ohour, omin;

   FUNCTION_START(STB_GCGetDateInfo);

   ASSERT(hour <= 23);
   ASSERT(min <= 59);

   STB_OSMutexLock(time_mutex);

   // calculate local time and date
   GetCTime();
   GetOffset(code, hour, min, 0, &ohour, &omin);
   AddTime(code, hour, min, 0, (S16BIT)ohour, (S16BIT)omin, 0,
      &lcode, &lhour, &lmin, &lsecs);

   STB_OSMutexUnlock(time_mutex);

   ExtractUTCDate(lcode, wday, day, month, year);

   FUNCTION_FINISH(STB_GCGetDateInfo);
}

/*!**************************************************************************
 * @brief   Returns the date info from the given MJD date code
 * @param   code - date code as a Modified Julian Date value
 * @param   day - returns with day of the month
 * @param   wday - returns with day of the week
 * @param   month - returns with month number
 * @param   year - returns with the year
 ****************************************************************************/
void STB_GCGetMJDDateInfo(U16BIT code, U8BIT *day, U8BIT *wday, U8BIT *month, U16BIT *year)
{
   FUNCTION_START(STB_GCGetMJDDateInfo);

   ExtractUTCDate(code, wday, day, month, year);

   FUNCTION_FINISH(STB_GCGetMJDDateInfo);
}

/**
 *

 *
 * @brief   Supplies specified time as clock string, in format requested.
 *
 * @param   U8BIT hour - value of hour (0-23).
 * @param   U8BIT min -  value of minute (0-59).
 * @param   U8BIT secs -  value of secs (0-59).
 * @param   E_STB_GC_CLOCKTYPE format - string format required.
 *
 * @return   U8BIT* - string pointer.
 *
 */
U8BIT* STB_GCGetClockString(U8BIT hour, U8BIT min, U8BIT secs, E_STB_GC_CLOCKTYPE format)
{
   FUNCTION_START(STB_GCGetClockString);

   switch (format)
   {
      case CLOCK_HMS:
         snprintf((char *)clock_string, CLOCK_STRING_MAX, "%02d:%02d:%02d", hour, min, secs);
         break;

      case CLOCK_SMH:
         snprintf((char *)clock_string, CLOCK_STRING_MAX, "%02d:%02d:%02d", secs, min, hour);
         break;

      case CLOCK_HM:
         snprintf((char *)clock_string, CLOCK_STRING_MAX,  "%02d:%02d", hour, min);
         break;

      case CLOCK_MH:
         snprintf((char *)clock_string, CLOCK_STRING_MAX, "%02d:%02d", min, hour);
         break;

      default:
         break;
   }

   FUNCTION_FINISH(STB_GCGetClockString);

   return(clock_string);
}

/**
 *

 *
 * @brief   Get the local year, month, day, hour and minute
 *
 * @param   U16BIT* year - Year returned value
 * @param   U8BIT* month - Month number returned value  (1-12)
 * @param   U8BIT* day   - Day of the month returned value  (1-31)
 * @param   U8BIT* hour  - value of hour (0-23).
 * @param   U8BIT* min   - value of minute (0-59).
 *
 */
void STB_GCGetLocalDateTime(U16BIT *year, U8BIT *month, U8BIT *day, U8BIT *hour, U8BIT *min)
{
   U16BIT gcode, lcode;
   U8BIT ghour, gmin, secs, wday;
   S8BIT ohour, omin;

   FUNCTION_START(STB_GCGetLocalDateTime);

   ASSERT(year != NULL);
   ASSERT(month != NULL);
   ASSERT(day != NULL);
   ASSERT(hour != NULL);
   ASSERT(min != NULL);

   STB_OSMutexLock(time_mutex);

   GetCTime();

   gcode = general_status.date_code;
   ghour = general_status.time_hour;
   gmin = general_status.time_min;

   GetOffset(gcode, ghour, gmin, 0, &ohour, &omin);
   AddTime(gcode, ghour, gmin, 0, (S16BIT)ohour, (S16BIT)omin, 0,
      &lcode, hour, min, &secs);

   STB_OSMutexUnlock(time_mutex);

   ExtractUTCDate(lcode, &wday, day, month, year);

   FUNCTION_FINISH(STB_GCGetLocalDateTime);
}

/*!**************************************************************************
 * @brief   Sets whether the date/time are taken from the broadcast or the system.
 *          The default is to use the broadcast for date/time.
 * @param   state - FALSE to use date/time from the system
 ****************************************************************************/
void STB_GCUseBroadcastTime(BOOLEAN state)
{
   FUNCTION_START(STB_GCUseBroadcastTime);

   if (state != general_status.use_broadcast_time)
   {
      STB_OSMutexLock(time_mutex);

      general_status.use_broadcast_time = state;

      if (!general_status.use_broadcast_time)
      {
         /* Time is assumed to be always available if it isn't being taken
          * from the broadcast */
         general_status.time_set = TRUE;
      }

      /* Force the time to be re-read with the change in state */
      GetCTime();

      STB_OSMutexUnlock(time_mutex);
   }

   FUNCTION_FINISH(STB_GCUseBroadcastTime);
}

/**
 *

 *
 * @brief   Makes U32DHMS formated date/time from date code, hour, minutes, seconds
 * @note    This function is used when DEBUG_ASSERT is on, and used in place of macro
 *                 DHMS_CREATE, so that the inputs of the macro can be checked.
 *
 * @param   U32BIT date - date code.
 * @param   U32BIT hour - value of hour (0-23).
 * @param   U32BIT mins - value of minute (0-59).
 * @param   U32BIT secs - value of seconds (0-59).
 *
 * @return   U32DHMS
 *
 */
U32DHMS STB_GCCreateDebugDHMS( U32BIT date, U32BIT hour, U32BIT mins, U32BIT secs )
{
   U32DHMS dtm;
   FUNCTION_START(STB_GCCreateDebugDHMS);
   ASSERT( hour < 24 );
   ASSERT( mins < 60 );
   ASSERT( secs < 60 );
   dtm = (U32DHMS)date << D_DSHFT |
      (U32DHMS)hour << D_HSHFT |
      (U32DHMS)mins << D_MSHFT |
      (U32DHMS)secs;
   FUNCTION_FINISH(STB_GCCreateDebugDHMS);
   return dtm;
}

/**
 *

 *
 * @brief   Makes U32DHMS formated date/time from date code, hour, minutes, seconds
 * @note    This function may be used instead of macro DHMS_CREATE, when it is known
 *                 that ASSERTs in STB_GCCreateDebugDHMS may fail.
 *
 * @param   U16BIT date
 * @param   U8BIT hour
 * @param   U8BIT mins
 * @param   U8BIT secs
 *
 * @return   U32DHMS
 *
 */
U32DHMS STB_GCCreateDHMS( U16BIT date, U8BIT hour, U8BIT mins, U8BIT secs )
{
   FUNCTION_START(STB_GCCreateDHMS);

   while (secs > 59)
   {
      secs -= 60;
      mins++;
   }
   while (mins > 59)
   {
      mins -= 60;
      hour++;
   }
   while (hour > 23)
   {
      hour -= 24;
      date++;
   }

   FUNCTION_FINISH(STB_GCCreateDHMS);

   return DHMS_CREATE(date, hour, mins, secs);
}

/**
 * @brief   Creates a DHMS value consisting of hours, minutes and seconds from a number of seconds
 * @param   num_seconds number of seconds to be converted
 * @return  converted value in hours, minutes and seconds as a U32DHMS value
 */
U32DHMS STB_GCCreateDHMSFromSeconds(U32BIT num_seconds)
{
   U32BIT hours, mins;

   FUNCTION_START(STB_GCCreateDHMSFromSeconds);

   hours = num_seconds / 3600;
   num_seconds %= 3600;

   mins = num_seconds / 60;
   num_seconds %= 60;

   FUNCTION_FINISH(STB_GCCreateDHMSFromSeconds);

   return(DHMS_CREATE(0, hours, mins, num_seconds));
}

/**
 * @brief   Calculates the date/time when the period is added/subtracted to/from dhms
 * @param   dhms base date/time
 * @param   period days/hours/mins/secs to be added or subtracted
 * @param   calc calculation to be performed
 *               CALC_ADD adds period to dhms
 *               CALC_SUB subtracts period from dhms
 * @return resulting date/time following the calculation
 */
U32DHMS STB_GCCalculateDHMS(U32DHMS dhms, U32DHMS period, E_STB_GC_CALCTYPE calc)
{
   S32BIT rcode, rhour, rmins, rsecs;

   FUNCTION_START(STB_GCCalculateDHMS);

   if (calc == CALC_ADD)
   {
      // top bit of date is not needed here for 'rcode', as returned U32DHMS value will not have it anyway,
      // so just use DHMS_DAYS macro, rather than DHMS_DATE.
      rcode = DHMS_DAYS(dhms) + DHMS_DAYS(period);
      rhour = DHMS_HOUR32(dhms) + DHMS_HOUR32(period);
      rmins = DHMS_MINS32(dhms) + DHMS_MINS32(period);
      rsecs = DHMS_SECS32(dhms) + DHMS_SECS32(period);

      while (rsecs > 59)
      {
         rsecs -= 60;
         rmins++;
      }
      while (rmins > 59)
      {
         rmins -= 60;
         rhour++;
      }
      while (rhour > 23)
      {
         rhour -= 24;
         rcode++;
      }
   }
   else if (calc == CALC_SUB)
   {
      // top bit of date is not needed here for 'rcode', as returned U32DHMS value will not have it anyway,
      // so just use DHMS_DAYS macro, rather than DHMS_DATE.
      rcode = (S32BIT)DHMS_DAYS(dhms) - (S32BIT)DHMS_DAYS(period);
      rhour = (S32BIT)DHMS_HOUR32(dhms) - (S32BIT)DHMS_HOUR32(period);
      rmins = (S32BIT)DHMS_MINS32(dhms) - (S32BIT)DHMS_MINS32(period);
      rsecs = (S32BIT)DHMS_SECS32(dhms) - (S32BIT)DHMS_SECS32(period);

      while (rsecs < 0)
      {
         rsecs += 60;
         rmins--;
      }
      while (rmins < 0)
      {
         rmins += 60;
         rhour--;
      }
      while (rhour < 0)
      {
         rhour += 24;
         rcode--;
      }
   }

   FUNCTION_FINISH(STB_GCCalculateDHMS);

   return DHMS_CREATE(rcode, rhour, rmins, rsecs);
}

/**
 * @brief   Converts the given date/time to local or GMT
 * @param   dhms date/time to be converted
 * @param   conv conversion to be performed
 *               CONV_LOCAL = convert to local
 *               CONV_GMT = convert to GMT
 * @return  converted date/time
 */
U32DHMS STB_GCConvertDHMS(U32DHMS dhms, E_STB_GC_CONVTYPE conv)
{
   S8BIT ohour, omin;

   FUNCTION_START(STB_GCConvertDHMS);

   STB_OSMutexLock(time_mutex);

   GetCTime();
   GetOffset(DHMS_DATE(dhms), DHMS_HOUR(dhms), DHMS_MINS(dhms), 0, &ohour, &omin);

   STB_OSMutexUnlock(time_mutex);

   if (conv == CONV_GMT)
   {
      /* Convert local time to GMT */
      if (ohour < 0)
      {
         ohour *= -1;
         omin *= -1;
         dhms = STB_GCCalculateDHMS(dhms, DHMS_CREATE(0, ohour, omin, 0), CALC_ADD);
      }
      else if (ohour > 0)
      {
         dhms = STB_GCCalculateDHMS(dhms, DHMS_CREATE(0, ohour, omin, 0), CALC_SUB);
      }
   }
   else if (conv == CONV_LOCAL)
   {
      /* Convert GMT time to local */
      if (ohour > 0)
      {
         dhms = STB_GCCalculateDHMS(dhms, DHMS_CREATE(0, ohour, omin, 0), CALC_ADD);
      }
      else if (ohour < 0)
      {
         ohour *= -1;
         omin *= -1;
         dhms = STB_GCCalculateDHMS(dhms, DHMS_CREATE(0, ohour, omin, 0), CALC_SUB);
      }
   }

   FUNCTION_FINISH(STB_GCConvertDHMS);

   return dhms;
}

/**
 *

 *
 * @brief   Reads the current GMT date code and time
 *
 * @param   void
 *
 * @return   U32DHMS - now in GMT
 *
 */
U32DHMS STB_GCNowDHMSGmt(void)
{
   U32DHMS now;

   FUNCTION_START(STB_GCNowDHMSGmt);

   STB_OSMutexLock(time_mutex);

   GetCTime();
   now = DHMS_CREATE(general_status.date_code,
         general_status.time_hour,
         general_status.time_min,
         general_status.time_secs);

   STB_OSMutexUnlock(time_mutex);

   FUNCTION_FINISH(STB_GCNowDHMSGmt);

   return now;
}

/**
 *

 *
 * @brief   Reads the current Local date code and time
 *
 * @param   void
 *
 * @return   U32DHMS - now in local time
 *
 */
U32DHMS STB_GCNowDHMSLocal(void)
{
   U32DHMS local;
   S8BIT ohour, omin;

   FUNCTION_START(STB_GCNowDHMSLocal);

   STB_OSMutexLock(time_mutex);

   GetCTime();
   GetOffset( general_status.date_code,
      general_status.time_hour,
      general_status.time_min, 0, &ohour, &omin);
   if (ohour < 0)
   {
      ohour *= -1;
      omin *= -1;
      local = STB_GCCalculateDHMS(
            DHMS_CREATE(general_status.date_code,
               general_status.time_hour,
               general_status.time_min,
               general_status.time_secs),
            DHMS_CREATE(0, ohour, omin, 0), CALC_SUB);
   }
   else
   {
      local = STB_GCCalculateDHMS(
            DHMS_CREATE(general_status.date_code,
               general_status.time_hour,
               general_status.time_min,
               general_status.time_secs),
            DHMS_CREATE(0, ohour, omin, 0), CALC_ADD);
   }

   STB_OSMutexUnlock(time_mutex);

   FUNCTION_FINISH(STB_GCNowDHMSLocal);

   return local;
}

/**
 *

 *
 * @brief   Supplies specifed date as local date string, in format requested.
 *
 * @param   U32DHMS dhms - date time (gmt)
 * @param   E_STB_GC_DATETYPE format - string format required.
 *
 * @return   U8BIT* - string pointer.
 *
 */
U8BIT* STB_GCGetDateStringDHMS(U32DHMS dhms, E_STB_GC_DATETYPE format)
{
   U16BIT year;
   U8BIT wday, day, month;

   FUNCTION_START(STB_GCGetDateStringDHMS);

   dhms = STB_GCConvertDHMS(dhms, CONV_LOCAL);

   ExtractUTCDate(DHMS_DATE(dhms), &wday, &day, &month, &year);
   switch (format)
   {
      case DATE_DMY:
         snprintf((char *)date_string, DATE_STRING_MAX, "%02d/%02d/%04d", day, month, year);
         break;

      case DATE_YMD:
         snprintf((char *)date_string, DATE_STRING_MAX, "%04d/%02d/%02d", year, month, day);
         break;

      default:
         break;
   }
   FUNCTION_FINISH(STB_GCGetDateStringDHMS);
   return(date_string);
}

/**
 *

 *
 * @brief   Supplies specified time as local time string, in format requested.
 *
 * @param   U32DHMS dhms - date time (gmt)
 * @param   E_STB_GC_TIMETYPE format - string format required.
 *
 * @return   U8BIT* - string pointer.
 *
 */
U8BIT* STB_GCGetTimeStringDHMS(U32DHMS dhms, E_STB_GC_TIMETYPE format)
{
   U8BIT lhour, lmins;

   FUNCTION_START(STB_GCGetTimeStringDHMS);

   dhms = STB_GCConvertDHMS(dhms, CONV_LOCAL);

   lhour = DHMS_HOUR(dhms);
   lmins = DHMS_MINS(dhms);
   switch (format)
   {
      case TIME_12H:
         if (lhour > 11)
         {
            lhour -= 12;
            if (lhour == 0)
               lhour = 12;
            snprintf((char *)time_string, TIME_STRING_MAX, "%d:%02d pm", lhour, lmins);
         }
         else
         {
            if (lhour == 0)
               lhour = 12;
            snprintf((char *)time_string, TIME_STRING_MAX, "%d:%02d am", lhour, lmins);
         }
         break;
      case TIME_24H:
         snprintf((char *)time_string, TIME_STRING_MAX, "%02d:%02d", lhour, lmins);
         break;
      default:
         break;
   }

   FUNCTION_FINISH(STB_GCGetTimeStringDHMS);

   return(time_string);
}

/**
 * @brief   Converts a timestamp expressed in number of seconds since midnight (UTC) 1 January 1970
 * @param   timestamp number of seconds since midnight (UTC) 1 January 1970
 * @return  time expressed as U32DHMS
 */
U32DHMS STB_GCConvertTimestamp(U32BIT timestamp)
{
   U16BIT date;
   U8BIT h, m, s;
   U32DHMS time;

   FUNCTION_START(STB_GCConvertTimestamp);

   ConvertTimestamp(timestamp, &date, &h, &m, &s);
   time = STB_GCCreateDHMS(date, h, m, s);

   FUNCTION_FINISH(STB_GCConvertTimestamp);

   return time;
}

/**
 * @brief   Returns the number of seconds from midnight (UTC) 1 January 1970 to the specified
 *          U32DHMS time
 * @param   time time in U32DHMS format
 */
U32BIT STB_GCConvertToTimestamp(U32DHMS time)
{
   U32BIT timestamp;
   U16BIT code;
   U8BIT h, m, s;

   FUNCTION_START(STB_GCConvertToTimestamp);

   code = DHMS_DATE(time);
   h = DHMS_HOUR(time);
   m = DHMS_MINS(time);
   s = DHMS_SECS(time);

   timestamp = ConvertToTimestamp(code, h, m, s);

   FUNCTION_FINISH(STB_GCConvertToTimestamp);

   return timestamp;
}

//*****************************************************************************
// End of file
//*****************************************************************************

