/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Heap memory management routines
 *
 * @file    stbheap.c
 * @date    06/09/2000
 */

#define _STBHEAP_C

// uncomment this line to give direct COM port access
/* #define STB_DEBUG */

// uncomment these lines to give heap debugging
/* #define DEBUG_HEAP */
/* #define DEBUG_HEAP_STATS */
/* #define DEBUG_APP_STATS */
/* #define DEBUG_HEAP_FULL */
/* #define DEBUG_APP_FULL */

// uncomment these lines for additional heap debugging functionality
/* #define DEBUG_MEMCOPY */
/* #define DEBUG_TEST_APPHEAP */
/* #define DEBUG_TEST_STBHEAP */
// uncomment this line to disable memcpy debugs for function calls within this source file
/* #define IGNORE_MEMCOPY_IN_SOURCEFILE */


//---includes for this file----------------------------------------------------
// compiler library header files
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stddef.h>

// third party header files

// Ocean Blue Software header files

#include <techtype.h>
#include <dbgfuncs.h>

#include "stbheap.h"
#include "stbhwmem.h"

#ifdef PGW_ENABLED
   #undef DEBUG_HEAP
   #undef DEBUG_MEMCOPY
   #undef DEBUG_TEST_APPHEAP
   #undef DEBUG_TEST_STBHEAP
   #define STB_DEBUG
/*#define PGW_DUMP_ON*/
/*#define PGW_LAST_ON*/
/* #include "prgwatch.h" */
#endif

//---constant definitions for this file----------------------------------------
#ifdef STB_HEAP_PRINT_REQUIRED
   #define STB_HEAP_PRINT(x) DEBUG_PRINTX_CONDITIONAL(DEBUG_STB_HEAP) x
#else
   #ifdef STB_DEBUG
      #define STB_HEAP_PRINT(x) STB_SPDebugWrite x
   #else
      #define STB_HEAP_PRINT(x)
   #endif
#endif

#ifdef DEBUG_MEMCOPY
   #ifdef IGNORE_MEMCOPY_IN_SOURCEFILE
      #ifdef memcpy
         #undef memcpy
      #endif
   #endif
#endif

#ifdef PGW_ENABLED
#define PGW_GOOD_MAGIC 0xdeadbeef
#define PGW_BAD_MAGIC 0xfeebdaed
#define LAST_FREED_MSK 0x7
#endif

//---local typedef structs for this file---------------------------------------

typedef struct s_stat_mem
{
   U32BIT sum;
   U32BIT total;
   U32BIT maximum;
} S_STAT_MEM;

#ifdef DEBUG_HEAP
   #define HEAP_HDR_SIZE 16
   #define HEAP_LIST_SIZE 10000
   #define TEST_BUFF_SIZE 8
typedef struct
{
   U8BIT *addr[HEAP_LIST_SIZE];
   U32BIT size[HEAP_LIST_SIZE];
   U16BIT sum;
   U32BIT total;
   U32BIT max;
} DEBUG_HEAP_STRUCT;
#endif

#ifdef DEBUG_HEAP_STATS
   #define SIZEOF_SIZE_FIELD  4
#endif

typedef struct s_cache_registration
{
   BOOLEAN (*callback_function)(void);
   struct s_cache_registration *next_ptr;
}
S_CACHE_REGISTRATION;

#ifdef PGW_ENABLED
typedef struct mem_info S_MEM_INFO;
struct mem_info
{
   unsigned long magic;
   size_t size;
   char *filename;
   int lineno;
   #ifdef PGW_DUMP_ON
   S_MEM_INFO *next;
   S_MEM_INFO *prev;
   #endif /*PGW_DUMP_ON*/
   unsigned long align;
};
#endif

//---local (static) variable declarations for this file------------------------
//   (internal variables declared static to make them local)

#ifdef DEBUG_HEAP
static DEBUG_HEAP_STRUCT heap_list;
static DEBUG_HEAP_STRUCT app_list;
static U8BIT test_buffer[TEST_BUFF_SIZE] = {0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef};
#endif
#ifdef DEBUG_HEAP_STATS
static S_STAT_MEM heap_stats;
#endif
#ifdef DEBUG_APP_STATS
static S_STAT_MEM app_stats;
#endif


static S_CACHE_REGISTRATION *cache_registration_ptr;
static S_CACHE_REGISTRATION *cache_marker_ptr;

#ifdef PGW_ENABLED
#ifdef PGW_DUMP_ON
static S_MEM_INFO *sys_head = NULL;
static S_MEM_INFO *sys_tail = NULL;
static S_MEM_INFO *app_head = NULL;
static S_MEM_INFO *app_tail = NULL;
static void *pgw_mutex = NULL;
#elif defined(PGW_LAST_ON)
static S_MEM_INFO sys_last_alloc[LAST_FREED_MSK + 1];
static S_MEM_INFO app_last_alloc[LAST_FREED_MSK + 1];
static unsigned int sys_alloc_cnt = 0;
static unsigned int app_alloc_cnt = 0;
static S_MEM_INFO sys_last_freed[LAST_FREED_MSK + 1];
static S_MEM_INFO app_last_freed[LAST_FREED_MSK + 1];
static unsigned int sys_freed_cnt = 0;
static unsigned int app_freed_cnt = 0;
#endif /*PGW_DUMP_ON*/
#endif

//---local function prototypes for this file-----------------------------------
//   (internal functions declared static to make them local)
#ifdef DEBUG_MEMCOPY
static void CheckHeapMemory(U8BIT *destn, U16BIT nbytes);
#endif

//---local function definitions------------------------------------------------

//---global function definitions-----------------------------------------------

void STB_HeapInitialise(void)
{
#ifdef PGW_ENABLED
   #ifdef PGW_DUMP_ON
   if (pgw_mutex == NULL)
   {
      pgw_mutex = STB_OSCreateMutex();
      ASSERT( pgw_mutex != NULL );
   }
   #endif /*PGW_DUMP_ON*/
#endif
}

/**
 *

 *
 * @brief   Attempts to allocate memory from the heap.
 *
 * @param   bytes - block size required in bytes
 *
 * @return   Pointer to block allocated or NULL on failure.
 *
 */
#ifndef PGW_ENABLED
void* STB_GetMemory(U32BIT bytes)
{
   void *ret_addr;

//   FUNCTION_START(STB_GetMemory);

   ASSERT(bytes > 0);

   #ifndef DEBUG_HEAP
   {
      #ifndef DEBUG_HEAP_STATS
      ret_addr = STB_MEMGetSysRAM(bytes);
         #ifdef DEBUG_HEAP_FULL
      if (ret_addr == NULL)
      {
         STB_HEAP_PRINT(("Heap Malloc FAILED"));
      }
         #endif
      #else
      {
         U8BIT *blk_addr;
         U32BIT blk_size;

         blk_size = bytes + SIZEOF_SIZE_FIELD;
         blk_addr = STB_MEMGetSysRAM(blk_size);
         if (blk_addr != NULL)
         {
            *(U32BIT *)blk_addr = blk_size;
            ret_addr = blk_addr + SIZEOF_SIZE_FIELD;

            heap_stats.sum++;
            heap_stats.total += blk_size;
            if (heap_stats.total > heap_stats.maximum)
            {
               heap_stats.maximum = heap_stats.total;
            }
         }
         else
         {
            ret_addr = NULL;
         }
         STB_HEAP_PRINT(("M(h): % 6d @ %p (num=% 5d tot=% 7d max=% 7d)", bytes, ret_addr, heap_stats.sum, heap_stats.total, heap_stats.maximum));
      }
      #endif
   }
   #else
   {
      U16BIT count;
      U8BIT *blk_ptr;
      U32BIT blk_size;

      // malloc bigger block
      blk_size = (bytes + (TEST_BUFF_SIZE * 2));
      blk_ptr = (U8BIT *)STB_MEMGetSysRAM(blk_size);
      if (blk_ptr != NULL)
      {
         // insert test pattern and calc return address
         memcpy((void *)blk_ptr, (void *)test_buffer, TEST_BUFF_SIZE);
         memcpy((void *)(blk_ptr + bytes + TEST_BUFF_SIZE), (void *)test_buffer, TEST_BUFF_SIZE);
         ret_addr = (void *)(blk_ptr + TEST_BUFF_SIZE);

         // update heap stats
         for (count = 0; count < HEAP_LIST_SIZE; count++)
         {
            if (heap_list.addr[count] == NULL)
            {
               heap_list.addr[count] = ret_addr;
               heap_list.size[count] = bytes;
               heap_list.sum++;
               heap_list.total += (bytes + (TEST_BUFF_SIZE * 2));
               if (heap_list.total > heap_list.max)
                  heap_list.max = heap_list.total;

               #ifdef DEBUG_HEAP_STATS
               STB_HEAP_PRINT(("Malloc(heap) : %6lu @ %p - num=%3u tot=%6lu hi=%6lu",
                               bytes, ret_addr, heap_list.sum, heap_list.total, heap_list.max));
               #endif

               break;
            }
         } //eof for(count...)

         if (count >= HEAP_LIST_SIZE)
         {
            STB_HEAP_PRINT(("Malloc(heap) : ERROR: list full"));
            return(NULL);
         }
      }
      else
      {
         // malloc failed!
         STB_HEAP_PRINT(("Malloc(heap) : ERROR: refused %lu bytes", bytes));
         ret_addr = NULL;
      }
   }
   #endif

   ASSERT(ret_addr != NULL);

//   FUNCTION_FINISH(STB_GetMemory);

   return(ret_addr);
}

/**
 *

 *
 * @brief   Releases previously allocated heap memory.
 *
 * @param   addr - pointer to block to release.
 *

 *
 */
void STB_FreeMemory(void *addr)
{
   //   FUNCTION_START(STB_FreeMemory);

   #ifndef DEBUG_HEAP
   {
      #ifndef DEBUG_HEAP_STATS
      STB_MEMFreeSysRAM(addr);
      #else
      if (addr != NULL)
      {
         U8BIT *blk_addr;
         U32BIT blk_size;
#if defined (STB_HEAP_PRINT_REQUIRED) || defined (STB_DEBUG)
         U32BIT bytes;
#endif

         blk_addr = (U8BIT *)addr - SIZEOF_SIZE_FIELD;
         blk_size = *(U32BIT *)blk_addr;
#if defined (STB_HEAP_PRINT_REQUIRED) || defined (STB_DEBUG)
         bytes = blk_size - SIZEOF_SIZE_FIELD;
#endif

         STB_MEMFreeSysRAM(blk_addr);

         if (heap_stats.sum > 0)
         {
            heap_stats.sum--;
         }
         if (heap_stats.total > blk_size)
         {
            heap_stats.total -= blk_size;
         }
         else
         {
            heap_stats.total = 0;
         }
         STB_HEAP_PRINT(("F(h): % 6d @ %p (num=% 5d tot=% 7d max=% 7d)", bytes, addr, heap_stats.sum, heap_stats.total, heap_stats.maximum));
      }
      #endif
   }
   #else
   {
      U16BIT count;
      U8BIT *blk_ptr;

      if (addr != NULL)
      {
         // update heap stats
         for (count = 0; count < HEAP_LIST_SIZE; count++)
         {
            if (heap_list.addr[count] == addr)
            {
               // found it - calculate real block address
               blk_ptr = ((U8BIT *)addr - TEST_BUFF_SIZE);

               // check for corruption
               if ((memcmp((void *)blk_ptr, (void *)test_buffer, TEST_BUFF_SIZE) != 0) ||
                   (memcmp((void *)(blk_ptr + heap_list.size[count] + TEST_BUFF_SIZE), (void *)test_buffer, TEST_BUFF_SIZE) != 0))
               {
                  // corrupted!
                  STB_HEAP_PRINT(("Free(heap)   : ERROR: this block corrupt..."));
                  return;
               }

               heap_list.sum--;
               heap_list.total -= (heap_list.size[count] + (TEST_BUFF_SIZE * 2));

               #ifdef DEBUG_HEAP_STATS
               STB_HEAP_PRINT(("Free(heap)   : %6lu @ %p - num=%3u tot=%6lu hi=%6lu",
                               heap_list.size[count], addr, heap_list.sum, heap_list.total,
                               heap_list.max));
               #endif

               // this must be last because it releases the entry in the heap list for re-use
               heap_list.addr[count] = NULL;
               break;
            }
         } //eof for(count...)

         if (count >= HEAP_LIST_SIZE)
         {
            // not malloced!
            STB_HEAP_PRINT(("Free(heap)   : ERROR: unallocated pointer: %p", addr));
            return;
         }
      }
      else
      {
         // NULL address!
         STB_HEAP_PRINT(("Free(heap)   : ERROR: NULL pointer"));
         blk_ptr = NULL;
      }

      // free block
      STB_MEMFreeSysRAM((void *)blk_ptr);
   }
   #endif
//   FUNCTION_FINISH(STB_FreeMemory);
}

#ifdef DEBUG_HEAP
static void* DbgResizeMemory(void *ptr, U32BIT new_num_bytes, DEBUG_HEAP_STRUCT *hp_ptr )
{
   U8BIT *blk_ptr;
   U16BIT count;

   // update heap stats
   for (count = 0; count != HEAP_LIST_SIZE; count++)
   {
      if (hp_ptr->addr[count] == ptr)
      {
         // found it - calculate real block address
         blk_ptr = ((U8BIT *)ptr - TEST_BUFF_SIZE);

         // check for corruption
         if ((memcmp((void *)blk_ptr, (void *)test_buffer, TEST_BUFF_SIZE) != 0) ||
             (memcmp((void *)(blk_ptr + hp_ptr->size[count] + TEST_BUFF_SIZE), (void *)test_buffer, TEST_BUFF_SIZE) != 0))
         {
            // corrupted!
            STB_HEAP_PRINT(("Realloc(heap)   : ERROR: this block corrupt.sz=%d, new_sz=%d", hp_ptr->size[count], new_num_bytes));
         }

         if (new_num_bytes >= hp_ptr->size[count])
         {
            hp_ptr->total += new_num_bytes - hp_ptr->size[count];
            if (hp_ptr->total > hp_ptr->max)
               hp_ptr->max = hp_ptr->total;
         }
         else
         {
            hp_ptr->total -= hp_ptr->size[count] - new_num_bytes;
         }

         #ifdef DEBUG_HEAP_STATS
         STB_HEAP_PRINT(("Realloc(heap)   : %6lu @ %p - num=%3u tot=%6lu hi=%6lu",
                         hp_ptr->size[count], addr, hp_ptr->sum, hp_ptr->total,
                         hp_ptr->max));
         #endif

         blk_ptr = STB_MEMResizeSysRAM(blk_ptr, new_num_bytes + (TEST_BUFF_SIZE * 2));
         if (blk_ptr != NULL)
         {
            if (memcmp((void *)blk_ptr, (void *)test_buffer, TEST_BUFF_SIZE) != 0)
            {
               STB_HEAP_PRINT(("Realloc(heap)   : ERROR: Lost header : %p", blk_ptr));
            }
            // place test block at new end position
            memcpy((void *)(blk_ptr + new_num_bytes + TEST_BUFF_SIZE), (void *)test_buffer, TEST_BUFF_SIZE);
            hp_ptr->size[count] = new_num_bytes;
            blk_ptr += TEST_BUFF_SIZE;
         }
         hp_ptr->addr[count] = blk_ptr;
         break;
      }
   } //eof for(count...)

   if (count == HEAP_LIST_SIZE)
   {
      // not malloced!
      STB_HEAP_PRINT(("Realloc(heap)   : ERROR: unallocated pointer: %p", ptr));
      blk_ptr = STB_GetMemory(new_num_bytes);
   }
   return blk_ptr;
}

#endif /*DEBUG_HEAP*/

#ifdef DEBUG_HEAP_STATS
static void* StatResizeMemory(void *ptr, U32BIT new_num_bytes, S_STAT_MEM *stat_ptr )
{
   U8BIT *blk_ptr;
   U32BIT blk_size;
   blk_ptr = (U8BIT *)ptr - SIZEOF_SIZE_FIELD;
   blk_size = *(U32BIT *)blk_ptr;
   blk_size -= SIZEOF_SIZE_FIELD;
   if (blk_size < new_num_bytes)
   {
      stat_ptr->total += new_num_bytes - blk_size;
      if (stat_ptr->total > stat_ptr->maximum)
         stat_ptr->maximum = stat_ptr->total;
   }
   else
   {
      if (stat_ptr->total > (blk_size - new_num_bytes))
      {
         stat_ptr->total -= blk_size - new_num_bytes;
      }
      else
      {
         stat_ptr->total = 0;
      }
   }
   blk_ptr = STB_MEMResizeSysRAM(blk_ptr, new_num_bytes + SIZEOF_SIZE_FIELD);
   *(U32BIT *)blk_ptr = new_num_bytes + SIZEOF_SIZE_FIELD;
   blk_ptr += SIZEOF_SIZE_FIELD;
   return blk_ptr;
}

#endif /*DEBUG_HEAP_STATS*/

/**
 *

 *
 * @brief   Re-allocates a given memory area to the new size, ensuring data contained
 *                 within the original memory block is maintained.
 *
 * @param   ptr           - pointer to block to re-allocate.
 * @param   new_num_bytes - size of new memory block
 *
 * @return   Pointer to the new (re-sized) memory block.
 *
 */
void* STB_ResizeMemory(void *ptr, U32BIT new_num_bytes)
{
   void *retval;

//   FUNCTION_START(STB_ResizeMemory);

   retval = NULL;

   if (new_num_bytes == 0)
   {
      if (ptr != NULL)
      {
         STB_FreeMemory(ptr);
      }
   }
   else
   {
      if (ptr == NULL)
      {
         retval = STB_GetMemory(new_num_bytes);
      }
      else
      {
         #ifndef DEBUG_HEAP
            #ifndef DEBUG_HEAP_STATS
         retval = STB_MEMResizeSysRAM(ptr, new_num_bytes);
            #else
         retval = StatResizeMemory(ptr, new_num_bytes, &heap_stats );
            #endif
         #else
         retval = DbgResizeMemory(ptr, new_num_bytes, &heap_list);
         #endif
      }
   }
//   FUNCTION_FINISH(STB_ResizeMemory);
   return(retval);
}

#endif /*ifndef PGW_ENABLED*/


//**************************************************************************************************
// THE FOLLOWING FUNCTIONS PROVIDE MEMORY MANAGEMENT FOR THE APPLICATION TASK.  THEY USE A
// DIFFERENT HEAP TO THE STB & HW LAYERS TO AVOID THE POSSIBILITY OF THE APPLICATION TAKING TOO MUCH
// MEMORY AND CRASHING THE LOWER LAYERS
//**************************************************************************************************

static BOOLEAN AppFreeCachedMemory(void)
{
   BOOLEAN retval;
   S_CACHE_REGISTRATION *ptr;

   retval = FALSE;

   // Do we have any caches registered?
   if (cache_marker_ptr != NULL)
   {
      // If so then set rolling pointer to the next cache registration
      ptr = cache_marker_ptr;

      // This loop will attempt to free memory from each registered cache, stopping when one of
      // them has succeeded, or when all of them have failed.
      do
      {
         // Access the callback function - it will return TRUE if memory was freed
         retval = (*ptr->callback_function)();

         // Advance the rolling pointer...
         ptr = ptr->next_ptr;
         // ... and roll back to start of registration link-list if at end.
         if (ptr == NULL)
         {
            ptr = cache_registration_ptr;
         }

         // If the callback function managed to free some memory, then adjust the marker pointer -
         // the next call to this function will access the next callback function.
         if (retval == TRUE)
         {
            // NB - we will inherently now leave this do-while loop
            cache_marker_ptr = ptr;
         }
      }
      while (ptr != cache_marker_ptr);
   }

   return(retval);
}

/**
 *

 *
 * @brief   Attempts to allocate memory from the application heap.
 *
 * @param   bytes - block size required in bytes
 *
 * @return   Pointer to block allocated or NULL on failure.
 *
 */
#ifndef PGW_ENABLED
void* STB_AppGetMemory(U32BIT bytes)
{
   void *ret_addr;

//   FUNCTION_START(STB_AppGetMemory);

   ASSERT(bytes > 0);

   #ifndef DEBUG_HEAP
   {
      #ifndef DEBUG_APP_STATS
      ret_addr = STB_MEMGetAppRAM(bytes);
         #ifdef DEBUG_APP_FULL
      if (ret_addr == NULL)
      {
         STB_HEAP_PRINT(("APP Heap Malloc FAILED"));
      }
         #endif
      #else
      {
         U8BIT *blk_addr;
         U32BIT blk_size;

         blk_size = bytes + SIZEOF_SIZE_FIELD;
         blk_addr = STB_MEMGetAppRAM(blk_size);
         if (blk_addr != NULL)
         {
            *(U32BIT *)blk_addr = blk_size;
            ret_addr = blk_addr + SIZEOF_SIZE_FIELD;

            app_stats.sum++;
            app_stats.total += blk_size;
            if (app_stats.total > app_stats.maximum)
            {
               app_stats.maximum = app_stats.total;
            }
         }
         else
         {
            ret_addr = NULL;
         }
//            STB_HEAP_PRINT(("M(a): % 6d @ %p (num=% 5d tot=% 7d max=% 7d)", bytes, ret_addr, app_stats.sum, app_stats.total, app_stats.maximum));
      }
      #endif
      while ((ret_addr == NULL) && (bytes > 0))
      {
         if (AppFreeCachedMemory() == FALSE)
         {
            break;
         }
         ret_addr = STB_MEMGetAppRAM(bytes);
      }
   }
   #else
   {
      U16BIT count;
      U8BIT *blk_ptr;
      U32BIT blk_size;

      // malloc bigger block
      blk_size = (bytes + (TEST_BUFF_SIZE * 2));
      blk_ptr = (U8BIT *)STB_MEMGetAppRAM(blk_size);
      while ((blk_ptr == NULL) && (bytes > 0))
      {
         if (AppFreeCachedMemory() == FALSE)
         {
            break;
         }
         blk_ptr = (U8BIT *)STB_MEMGetAppRAM(blk_size);
      }
      if (blk_ptr != NULL)
      {
         // insert test pattern and calc return address
         memcpy((void *)blk_ptr, (void *)test_buffer, TEST_BUFF_SIZE);
         memcpy((void *)(blk_ptr + bytes + TEST_BUFF_SIZE), (void *)test_buffer, TEST_BUFF_SIZE);
         ret_addr = (void *)(blk_ptr + TEST_BUFF_SIZE);

         // update heap stats
         for (count = 0; count < HEAP_LIST_SIZE; count++)
         {
            if (app_list.addr[count] == NULL)
            {
               app_list.addr[count] = ret_addr;
               app_list.size[count] = bytes;
               app_list.sum++;
               app_list.total += (bytes + (TEST_BUFF_SIZE * 2));
               if (app_list.total > app_list.max)
                  app_list.max = app_list.total;

               #ifdef DEBUG_APP_STATS
               STB_HEAP_PRINT(("Malloc(app)  : %6lu @ %p - num=%3u tot=%6lu hi=%6lu",
                               bytes, ret_addr, app_list.sum, app_list.total, app_list.max));
               #endif
               break;
            }
         } //eof for(count...)

         if (count >= HEAP_LIST_SIZE)
         {
            STB_HEAP_PRINT(("Malloc(app)  : ERROR: list full"));
            return(NULL);
         }
      }
      else
      {
         // malloc failed!
         STB_HEAP_PRINT(("Malloc(app)  : ERROR: refused %lu bytes", bytes));
         ret_addr = NULL;
      }
   }
   #endif

   ASSERT(ret_addr != NULL);

//   FUNCTION_FINISH(STB_AppGetMemory);

   return(ret_addr);
}

/**
 *

 *
 * @brief   Releases previously allocated application heap memory.
 *
 * @param   addr - pointer to block to release.
 *

 *
 */
void STB_AppFreeMemory(void *addr)
{
//   FUNCTION_START(STB_AppFreeMemory);

   #ifndef DEBUG_HEAP
   {
      #ifndef DEBUG_APP_STATS
      STB_MEMFreeAppRAM(addr);
      #else
      if (addr != NULL)
      {
         U8BIT *blk_addr;
         U32BIT blk_size;
//         U32BIT bytes;

         blk_addr = (U8BIT *)addr - SIZEOF_SIZE_FIELD;
         blk_size = *(U32BIT *)blk_addr;
//         bytes = blk_size - SIZEOF_SIZE_FIELD;

         STB_MEMFreeAppRAM(blk_addr);

         if (app_stats.sum > 0)
         {
            app_stats.sum--;
         }
         if (app_stats.total > blk_size)
         {
            app_stats.total -= blk_size;
         }
         else
         {
            app_stats.total = 0;
         }
//            STB_HEAP_PRINT(("F(a): % 6d @ %p (num=% 5d tot=% 7d max=% 7d)", bytes, addr, app_stats.sum, app_stats.total, app_stats.maximum));
      }
      #endif
   }
   #else
   {
      U16BIT count;
      U8BIT *blk_ptr;

      if (addr != NULL)
      {
         // update heap stats
         for (count = 0; count < HEAP_LIST_SIZE; count++)
         {
            if (app_list.addr[count] == addr)
            {
               // found it - calculate real block address
               blk_ptr = ((U8BIT *)addr - TEST_BUFF_SIZE);

               // check for corruption
               if ((memcmp((void *)blk_ptr, (void *)test_buffer, TEST_BUFF_SIZE) != 0) ||
                   (memcmp((void *)(blk_ptr + app_list.size[count] + TEST_BUFF_SIZE), (void *)test_buffer, TEST_BUFF_SIZE) != 0))
               {
                  // corrupted!
                  STB_HEAP_PRINT(("Free(app)    : ERROR: this block corrupt..."));
                  return;
               }

               app_list.sum--;
               app_list.total -= (app_list.size[count] + (TEST_BUFF_SIZE * 2));

               #ifdef DEBUG_APP_STATS
               STB_HEAP_PRINT(("Free(app)    : %6lu @ %p - num=%3u tot=%6lu hi=%6lu",
                               app_list.size[count], addr, app_list.sum, app_list.total,
                               app_list.max));
               #endif

               // this must be last because it releases the entry in the heap list for re-use
               app_list.addr[count] = NULL;
               break;
            }
         } //eof for(count...)

         if (count >= HEAP_LIST_SIZE)
         {
            // not malloced!
            STB_HEAP_PRINT(("Free(app)    : ERROR: unallocated pointer: %p", addr));
            return;
         }
      }
      else
      {
         // NULL address!
         STB_HEAP_PRINT(("Free(app)    : ERROR: NULL pointer"));
         blk_ptr = NULL;
      }

      // free block
      STB_MEMFreeAppRAM((void *)blk_ptr);
   }
   #endif
//   FUNCTION_FINISH(STB_AppFreeMemory);
}

/**
 *

 *
 * @brief   Re-allocates a given memory area to the new size, ensuring data contained
 *                 within the original memory block is maintained.
 *
 * @param   ptr           - pointer to block to re-allocate.
 * @param   new_num_bytes - size of new memory block
 *
 * @return   Pointer to the new (re-sized) memory block.
 *
 */
void* STB_AppResizeMemory(void *ptr, U32BIT new_num_bytes)
{
   void *retval;

//   FUNCTION_START(STB_AppResizeMemory);

   retval = NULL;

   if (new_num_bytes == 0)
   {
      if (ptr != NULL)
      {
         STB_AppFreeMemory(ptr);
      }
   }
   else
   {
      if (ptr == NULL)
      {
         retval = STB_AppGetMemory(new_num_bytes);
      }
      else
      {
         #ifndef DEBUG_HEAP
            #ifndef DEBUG_APP_STATS
         retval = STB_MEMResizeSysRAM(ptr, new_num_bytes);
            #else
         retval = StatResizeMemory(ptr, new_num_bytes, &app_stats );
            #endif
         #else
         retval = DbgResizeMemory(ptr, new_num_bytes, &app_list);
         #endif
      }
   }
//   FUNCTION_FINISH(STB_AppResizeMemory);
   return(retval);
}

#endif /*ifndef PGW_ENABLED*/

void STB_AppRegisterCacheFreeFunction(BOOLEAN (*callback_function)(void))
{
   S_CACHE_REGISTRATION *ptr;

   FUNCTION_START(STB_AppRegisterCacheFreeFunction);

   if (callback_function != NULL)
   {
      // Go through cache registrations ensuring that the callback function has not already
      // been registered
      ptr = cache_registration_ptr;

      while (ptr != NULL)
      {
         if (ptr->callback_function == callback_function)
         {
            break;
         }
      }

      // Pointer is NULL only if this is a new function (address)
      if (ptr == NULL)
      {
         // Generate a new registration record, populate it, ans add it to the link-list.
         #ifndef PGW_ENABLED
         ptr = (S_CACHE_REGISTRATION *)STB_AppGetMemory(sizeof(S_CACHE_REGISTRATION));
         #else
         ptr = (S_CACHE_REGISTRATION *)STB_AppGetMemoryPGW(sizeof(S_CACHE_REGISTRATION), __FILE__, __LINE__);
         #endif

         if (ptr != NULL)
         {
            ptr->callback_function = callback_function;
            ptr->next_ptr = cache_registration_ptr;

            cache_registration_ptr = ptr;

            // If this is the first registration, then set up the rolling pointer (used for access)
            if (cache_marker_ptr == NULL)
            {
               cache_marker_ptr = cache_registration_ptr;
            }
         }
      }
   }

   FUNCTION_FINISH(STB_AppRegisterCacheFreeFunction);
}

//**************************************************************************************************
//
//          PGW (PROGRAM WATCH) - For Tracking Memory Allocation
//
//**************************************************************************************************

#ifdef PGW_ENABLED
/*!**************************************************************************
 * @brief    Allocate a new system memory buffer (with tracking).
 * @param    bytes      Size of buffer in bytes
 * @param    filename   Name of file where allocation takes place
 * @param    lineno     Line number in file
 * @return   A pointer to the newly allocated buffer, NULL if failed
 ****************************************************************************/
void* STB_GetMemoryPGW(U32BIT size, char *filename, int lineno)
{
   S_MEM_INFO *mem_info;
   void *buffer = NULL;

   //FUNCTION_START(STB_GetMemoryPGW);

   mem_info = (S_MEM_INFO *)STB_MEMGetSysRAM(sizeof(S_MEM_INFO) + size - sizeof(unsigned long));
   if (mem_info != NULL)
   {
      mem_info->magic = PGW_GOOD_MAGIC;
      mem_info->size = size;
      mem_info->filename = filename;
      mem_info->lineno = lineno;

   #ifdef PGW_DUMP_ON
      ASSERT( pgw_mutex != NULL );
      mem_info->next = NULL;
      STB_OSMutexLock(pgw_mutex);
      mem_info->prev = sys_tail;
      if (sys_tail != NULL)
      {
         sys_tail->next = mem_info;
      }
      sys_tail = mem_info;
      if (sys_head == NULL)
      {
         sys_head = sys_tail;
      }
      STB_OSMutexUnlock(pgw_mutex);
   #elif defined(PGW_LAST_ON)
      sys_last_alloc[sys_alloc_cnt & LAST_FREED_MSK] = *mem_info;
      sys_alloc_cnt++;
   #endif /*PGW_DUMP_ON*/

      buffer = &mem_info->align;
   }
   else
   {
      STB_HEAP_PRINT(("PGW: Cannot allocate %d bytes at %s:%d\n", size,
                      filename, lineno));
   }

   //FUNCTION_FINISH(STB_GetMemoryPGW);

   return buffer;
}

/*!**************************************************************************
 * @brief   Reallocates a system memory buffer (with tracking).
 * @param   buffer     Pointer to current buffer
 * @param   bytes      Size of buffer in bytes
 * @param   filename   Name of file where allocation takes place
 * @param   lineno     Line number in file
 * @return  A pointer to the newly allocated buffer, NULL if failed
 ****************************************************************************/
void* STB_ResizeMemoryPGW(void *buffer, U32BIT size, char *filename, int lineno)
{
   S_MEM_INFO *mem_info;

   //FUNCTION_START(STB_ResizeMemoryPGW);

   if (size == 0)
   {
      if (buffer != NULL)
      {
         STB_FreeMemoryPGW(buffer, filename, lineno);
         buffer = NULL;
      }
   }
   else if (buffer == NULL)
   {
      buffer = STB_GetMemoryPGW(size, filename, lineno);
   }
   else
   {
      mem_info = (S_MEM_INFO *)((char *)buffer - offsetof(S_MEM_INFO, align));
      if (mem_info->magic != PGW_GOOD_MAGIC)
      {
         STB_HEAP_PRINT(("PGW: Reallocating invalid sys pointer %p at %s:%d\n", buffer,
                         filename, lineno));
         STB_MemoryDumpPGW(0);
      }
      else
      {
         mem_info->magic = PGW_BAD_MAGIC;

      #ifdef PGW_DUMP_ON
         /* Remove the memory from the list */
         ASSERT( pgw_mutex != NULL );
         STB_OSMutexLock(pgw_mutex);
         if (sys_tail == mem_info)
         {
            sys_tail = sys_tail->prev;
         }
         if (sys_head == mem_info)
         {
            sys_head = sys_head->next;
         }
         if (mem_info->next != NULL)
         {
            mem_info->next->prev = mem_info->prev;
         }

         if (mem_info->prev != NULL)
         {
            mem_info->prev->next = mem_info->next;
         }
         STB_OSMutexUnlock(pgw_mutex);
      #endif /*PGW_DUMP_ON*/

         /* Realloc the memory and add it back into the list */
         mem_info = STB_MEMResizeSysRAM(mem_info, sizeof(S_MEM_INFO) + size - sizeof(unsigned long));

         mem_info->magic = PGW_GOOD_MAGIC;
         mem_info->size = size;
         mem_info->filename = filename;
         mem_info->lineno = lineno;

      #ifdef PGW_DUMP_ON
         mem_info->next = NULL;
         STB_OSMutexLock(pgw_mutex);
         mem_info->prev = sys_tail;
         if (sys_tail != NULL)
         {
            sys_tail->next = mem_info;
         }
         sys_tail = mem_info;
         if (sys_head == NULL)
         {
            sys_head = sys_tail;
         }
         STB_OSMutexUnlock(pgw_mutex);
      #elif defined(PGW_LAST_ON)
         sys_last_alloc[sys_alloc_cnt & LAST_FREED_MSK] = *mem_info;
         sys_alloc_cnt++;
      #endif /*PGW_DUMP_ON*/

         buffer = &mem_info->align;
      }
   }

   //FUNCTION_FINISH(STB_ResizeMemoryPGW);

   return buffer;
}

/*!**************************************************************************
 * @brief    Free buffer that was allocated using PGW_sys_malloc or STB_ResizeMemoryPGW
 * @param    buffer     A pointer to the buffer
 * @return   None.
 ****************************************************************************/
void STB_FreeMemoryPGW(void *buffer, char *filename, int lineno)
{
   S_MEM_INFO *mem_info;

   //FUNCTION_START(STB_FreeMemoryPGW);

   if (buffer != NULL)
   {
      mem_info = (S_MEM_INFO *)((char *)buffer - offsetof(S_MEM_INFO, align));
      if (mem_info->magic != PGW_GOOD_MAGIC)
      {
         STB_HEAP_PRINT(("PGW: Freeing invalid pointer %p at %s:%d\n", buffer,
                         filename, lineno));
         STB_MemoryDumpPGW(0);
      }
      else
      {
      #ifdef PGW_DUMP_ON
         mem_info->magic = PGW_BAD_MAGIC;

         ASSERT( pgw_mutex != NULL );
         STB_OSMutexLock(pgw_mutex);
         if (sys_tail == mem_info)
         {
            sys_tail = sys_tail->prev;
         }
         if (sys_head == mem_info)
         {
            sys_head = sys_head->next;
         }
         if (mem_info->next != NULL)
         {
            mem_info->next->prev = mem_info->prev;
         }

         if (mem_info->prev != NULL)
         {
            mem_info->prev->next = mem_info->next;
         }
         STB_OSMutexUnlock(pgw_mutex);
      #elif defined(PGW_LAST_ON)
         sys_last_freed[sys_freed_cnt & LAST_FREED_MSK] = *mem_info;
         sys_freed_cnt++;
         mem_info->magic = PGW_BAD_MAGIC;
      #endif /*PGW_DUMP_ON*/

         STB_MEMFreeSysRAM(mem_info);
      }
   }

   //FUNCTION_FINISH(STB_FreeMemoryPGW);
}

/*!**************************************************************************
 * @brief    Allocate a new app memory buffer (with tracking).
 * @param    bytes      Size of buffer in bytes
 * @param    filename   Name of file where allocation takes place
 * @param    lineno     Line number in file
 * @return   A pointer to the newly allocated buffer, NULL if failed
 ****************************************************************************/
void* STB_AppGetMemoryPGW(U32BIT size, char *filename, int lineno)
{
   S_MEM_INFO *mem_info;
   void *buffer = NULL;

   //FUNCTION_START(STB_AppGetMemoryPGW);

   ASSERT(size > 0);

   mem_info = (S_MEM_INFO *)STB_MEMGetAppRAM(sizeof(S_MEM_INFO) + size - sizeof(unsigned long));
   while (mem_info == NULL && AppFreeCachedMemory())
   {
      mem_info = (S_MEM_INFO *)STB_MEMGetAppRAM(sizeof(S_MEM_INFO) + size - sizeof(unsigned long));
   }
   if (mem_info != NULL)
   {
      mem_info->magic = PGW_GOOD_MAGIC;
      mem_info->size = size;
      mem_info->filename = filename;
      mem_info->lineno = lineno;

   #ifdef PGW_DUMP_ON
      ASSERT( pgw_mutex != NULL );
      mem_info->next = NULL;
      STB_OSMutexLock(pgw_mutex);
      mem_info->prev = app_tail;
      if (app_tail != NULL)
      {
         app_tail->next = mem_info;
      }
      app_tail = mem_info;
      if (app_head == NULL)
      {
         app_head = app_tail;
      }
      STB_OSMutexUnlock(pgw_mutex);
   #elif defined(PGW_LAST_ON)
      app_last_alloc[app_alloc_cnt & LAST_FREED_MSK] = *mem_info;
      app_alloc_cnt++;
   #endif /*PGW_DUMP_ON*/

      buffer = &mem_info->align;
   }
   else
   {
      STB_HEAP_PRINT(("PGW: Cannot allocate %d bytes at %s:%d\n", size,
                      filename, lineno));
   }

   //FUNCTION_FINISH(STB_AppGetMemoryPGW);

   return buffer;
}

/*!**************************************************************************
 * @brief   Reallocates an app memory buffer (with tracking).
 * @param   buffer     Pointer to current buffer
 * @param   bytes      Size of buffer in bytes
 * @param   filename   Name of file where allocation takes place
 * @param   lineno     Line number in file
 * @return  A pointer to the newly allocated buffer, NULL if failed
 ****************************************************************************/
void* STB_AppResizeMemoryPGW(void *buffer, U32BIT size, char *filename, int lineno)
{
   S_MEM_INFO *mem_info;

   //FUNCTION_START(STB_AppResizeMemoryPGW);


   if (size == 0)
   {
      if (buffer != NULL)
      {
         STB_AppFreeMemoryPGW(buffer, filename, lineno);
         buffer = NULL;
      }
   }
   else if (buffer == NULL)
   {
      buffer = STB_AppGetMemoryPGW(size, filename, lineno);
   }
   else
   {
      mem_info = (S_MEM_INFO *)((char *)buffer - offsetof(S_MEM_INFO, align));
      if (mem_info->magic != PGW_GOOD_MAGIC)
      {
         STB_HEAP_PRINT(("PGW: Reallocating invalid app pointer %p at %s:%d\n", buffer,
                         filename, lineno));
         STB_MemoryDumpPGW(0);
      }
      else
      {
         mem_info->magic = PGW_BAD_MAGIC;

      #ifdef PGW_DUMP_ON
         /* Remove the memory from the list */
         ASSERT( pgw_mutex != NULL );
         STB_OSMutexLock(pgw_mutex);
         if (app_tail == mem_info)
         {
            app_tail = app_tail->prev;
         }
         if (app_head == mem_info)
         {
            app_head = app_head->next;
         }
         if (mem_info->next != NULL)
         {
            mem_info->next->prev = mem_info->prev;
         }

         if (mem_info->prev != NULL)
         {
            mem_info->prev->next = mem_info->next;
         }
         STB_OSMutexUnlock(pgw_mutex);
      #endif /*PGW_DUMP_ON*/

         /* Realloc the memory and add it back into the list */
         mem_info = STB_MEMResizeAppRAM(mem_info, sizeof(S_MEM_INFO) + size - sizeof(unsigned long));

         mem_info->magic = PGW_GOOD_MAGIC;
         mem_info->size = size;
         mem_info->filename = filename;
         mem_info->lineno = lineno;

      #ifdef PGW_DUMP_ON
         mem_info->next = NULL;
         STB_OSMutexLock(pgw_mutex);
         mem_info->prev = app_tail;
         if (app_tail != NULL)
         {
            app_tail->next = mem_info;
         }
         app_tail = mem_info;
         if (app_head == NULL)
         {
            app_head = app_tail;
         }
         STB_OSMutexUnlock(pgw_mutex);
      #elif defined(PGW_LAST_ON)
         app_last_alloc[app_alloc_cnt & LAST_FREED_MSK] = *mem_info;
         app_alloc_cnt++;
      #endif /*PGW_DUMP_ON*/

         buffer = &mem_info->align;
      }
   }

   //FUNCTION_FINISH(STB_AppResizeMemoryPGW);

   return buffer;
}

/*!**************************************************************************
 * @brief    Free buffer that was allocated using STB_AppGetMemoryPGW or STB_AppResizeMemoryPGW
 * @param    buffer     A pointer to the buffer
 * @return   None.
 ****************************************************************************/
void STB_AppFreeMemoryPGW(void *buffer, char *filename, int lineno)
{
   S_MEM_INFO *mem_info;

   //FUNCTION_START(STB_AppFreeMemoryPGW);

   if (buffer != NULL)
   {
      mem_info = (S_MEM_INFO *)((char *)buffer - offsetof(S_MEM_INFO, align));
      if (mem_info->magic != PGW_GOOD_MAGIC)
      {
         STB_HEAP_PRINT(("PGW: Freeing invalid pointer %p at %s:%d\n", buffer,
                         filename, lineno));
         STB_MemoryDumpPGW(0);
      }
      else
      {
      #ifdef PGW_DUMP_ON
         mem_info->magic = PGW_BAD_MAGIC;

         ASSERT( pgw_mutex != NULL );
         STB_OSMutexLock(pgw_mutex);
         if (app_tail == mem_info)
         {
            app_tail = app_tail->prev;
         }
         if (app_head == mem_info)
         {
            app_head = app_head->next;
         }
         if (mem_info->next != NULL)
         {
            mem_info->next->prev = mem_info->prev;
         }

         if (mem_info->prev != NULL)
         {
            mem_info->prev->next = mem_info->next;
         }
         STB_OSMutexUnlock(pgw_mutex);
      #elif defined(PGW_LAST_ON)
         app_last_freed[app_freed_cnt & LAST_FREED_MSK] = *mem_info;
         app_freed_cnt++;
         mem_info->magic = PGW_BAD_MAGIC;
      #endif /*PGW_DUMP_ON*/

         STB_MEMFreeAppRAM(mem_info);
      }
   }

   //FUNCTION_FINISH(STB_AppFreeMemoryPGW);
}

#ifdef PGW_LAST_ON
static void DumpLast( const char *str, S_MEM_INFO *info, unsigned int cnt )
{
   S_MEM_INFO *mem_info;
   unsigned int c;
   STB_HEAP_PRINT(("%s Count=%d", str, cnt));
   if (cnt > (LAST_FREED_MSK + 1))
   {
      c = cnt - (LAST_FREED_MSK + 1);
   }
   else
   {
      c = 0;
   }
   while (cnt != c)
   {
      cnt--;
      mem_info = &info[ cnt & LAST_FREED_MSK ];
      STB_HEAP_PRINT(("%s:%d\t%d bytes \tmgc=0x%x", mem_info->filename, mem_info->lineno, mem_info->size, mem_info->magic));
   }
}

#endif

/*!**************************************************************************
 * @brief    Dump list of current memory buffers
 * @param    full       Whether to print the full list or just the total
 * @return   None.
 * @warning  None.
 * @bug      None.
 ****************************************************************************/
void STB_MemoryDumpPGW(BOOLEAN full)
{
   S_MEM_INFO *mem_info;
   size_t total = 0;

   FUNCTION_START(STB_MemoryDumpPGW);

#ifdef PGW_DUMP_ON

   ASSERT( pgw_mutex != NULL );

   STB_OSMutexLock(pgw_mutex);
   STB_HEAP_PRINT(("*** SYSTEM MEMORY DUMP BEGIN ***\n"));
   for (mem_info = sys_head; mem_info != NULL; mem_info = mem_info->next)
   {
      if (full)
      {
         STB_HEAP_PRINT(("%s:%d\t%d bytes", mem_info->filename, mem_info->lineno, mem_info->size));
      }
      total += mem_info->size;
   }
   STB_HEAP_PRINT(("TOTAL\t%d bytes\n", total));
   total = 0;
   STB_HEAP_PRINT(("*** APP MEMORY DUMP BEGIN ***\n"));
   for (mem_info = app_head; mem_info != NULL; mem_info = mem_info->next)
   {
      if (full)
      {
         STB_HEAP_PRINT(("%s:%d\t%d bytes", mem_info->filename, mem_info->lineno, mem_info->size));
      }
      total += mem_info->size;
   }
   STB_HEAP_PRINT(("TOTAL\t%d bytes\n", total));
   STB_HEAP_PRINT(("*** MEMORY DUMP END ***\n"));
   STB_OSMutexUnlock(pgw_mutex);

#else

   USE_UNWANTED_PARAM(full);
   USE_UNWANTED_PARAM(total);
   USE_UNWANTED_PARAM(mem_info);

#ifdef PGW_LAST_ON
   DumpLast( "SYS ALLOC", sys_last_alloc, sys_alloc_cnt );
   DumpLast( "SYS FREED", sys_last_freed, sys_freed_cnt );
   DumpLast( "APP ALLOC", app_last_alloc, app_alloc_cnt );
   DumpLast( "APP FREED", app_last_freed, app_freed_cnt );
#endif

#endif /*ifdef PGW_DUMP_ON*/

   FUNCTION_FINISH(STB_MemoryDumpPGW);
}

#endif  /* ifdef PGW_ENABLED */


//**************************************************************************************************
//
//          MEMORY COPY DEBUG FUNCTIONS -- For Detecting Memory Corruption
//
//**************************************************************************************************
#ifdef DEBUG_MEMCOPY
static void CheckHeapMemory(U8BIT *destn, U16BIT nbytes)
{
   U16BIT count;
   U16BIT sum;
   U8BIT *blk_start;
   U8BIT *blk_end;

   // check for app heap corruption
   for (sum = 0, count = 0; ((count < HEAP_LIST_SIZE) && (sum < app_list.sum)); count++)
   {
      if (*(app_list.addr + count) != NULL)
      {
         sum++;
         blk_start = *(app_list.addr + count) - TEST_BUFF_SIZE;
         blk_end = *(app_list.addr + count) + (*(app_list.size + count));

         if (destn <= (blk_end + TEST_BUFF_SIZE))
         {
            if ((destn + nbytes) >= blk_start)
            {
               if (destn < (blk_start + TEST_BUFF_SIZE))
               {
                  STB_HEAP_PRINT(("MemCopy  : ERROR: App Heap Corruption"));
               }
               else if ((destn + nbytes) > blk_end)
               {
                  STB_HEAP_PRINT(("MemCopy  : ERROR: App Heap Corruption"));
               }
            }
         }
      }
   }

   // check for stb heap corruption
   for (sum = 0, count = 0; ((count < HEAP_LIST_SIZE) && (sum < heap_list.sum)); count++)
   {
      if (*(heap_list.addr + count) != NULL)
      {
         sum++;
         blk_start = *(heap_list.addr + count) - TEST_BUFF_SIZE;
         blk_end = *(heap_list.addr + count) + (*(heap_list.size + count));

         if (destn <= (blk_end + TEST_BUFF_SIZE))
         {
            if ((destn + nbytes) >= blk_start)
            {
               if (destn < (blk_start + TEST_BUFF_SIZE))
               {
                  STB_HEAP_PRINT(("MemCopy  : ERROR: Stb Heap Corruption"));
               }
               else if ((destn + nbytes) > blk_end)
               {
                  STB_HEAP_PRINT(("MemCopy  : ERROR: Stb Heap Corruption"));
               }
            }
         }
      }
   }
}

#endif //#ifdef DEBUG_MEMCOPY


#ifdef DEBUG_TEST_APPHEAP
void STB_CheckAppHeap(void)
{
   U16BIT count;
   U16BIT sum;
   U8BIT *blk_start;
   U8BIT *blk_end;

   // check for app heap corruption
   for (sum = 0, count = 0; ((count < HEAP_LIST_SIZE) && (sum < app_list.sum)); count++)
   {
      if (*(app_list.addr + count) != NULL)
      {
         sum++;
         blk_start = *(app_list.addr + count) - TEST_BUFF_SIZE;
         blk_end = *(app_list.addr + count) + (*(app_list.size + count));

         if ((memcmp((void *)blk_start, (void *)test_buffer, TEST_BUFF_SIZE) != 0) ||
             (memcmp((void *)blk_end, (void *)test_buffer, TEST_BUFF_SIZE) != 0))
         {
            STB_HEAP_PRINT(("HeapTest(App) : ERROR: Corrupt Block: start = %p end = %p", blk_start, blk_end));
            break;
         }
      }
   }
}

#endif

#ifdef DEBUG_TEST_STBHEAP
void STB_CheckStbHeap(void)
{
   U16BIT count;
   U16BIT sum;
   U8BIT *blk_start;
   U8BIT *blk_end;

   // check for app heap corruption
   for (sum = 0, count = 0; ((count < HEAP_LIST_SIZE) && (sum < heap_list.sum)); count++)
   {
      if (*(heap_list.addr + count) != NULL)
      {
         sum++;
         blk_start = *(heap_list.addr + count) - TEST_BUFF_SIZE;
         blk_end = *(heap_list.addr + count) + (*(heap_list.size + count));

         if ((memcmp((void *)blk_start, (void *)test_buffer, TEST_BUFF_SIZE) != 0) ||
             (memcmp((void *)blk_end, (void *)test_buffer, TEST_BUFF_SIZE) != 0))
         {
            STB_HEAP_PRINT(("HeapTest(Stb) : ERROR: Corrupt Block: start = %p end = %p", blk_start, blk_end));
            break;
         }
      }
   }
}

#endif


//**************************************************************************************************
//**************************************************************************************************
//**************************************************************************************************
//
// IMPORTANT NOTE: These Functions Must remain at the End of this File in order for them
//                 to function correctly.
//
//**************************************************************************************************
#ifdef DEBUG_MEMCOPY
void* STB_MemCopy(void *s1, const void *s2, U16BIT n)
{
   CheckHeapMemory((U8BIT *)s1, n);

   #ifdef memcpy
      #undef memcpy
   #endif
   return memcpy(s1, s2, n);
}

void* STB_MemMove(void *s1, const void *s2, U16BIT n)
{
   CheckHeapMemory((U8BIT *)s1, n);

   #undef memmove
   return memmove(s1, s2, n);
}

void* STB_MemSet(void *s, int c, U16BIT n)
{
   CheckHeapMemory((U8BIT *)s, n);

   #undef memset
   return memset(s, c, n);
}

char* STB_StrCopy(char *s1, const char *s2)
{
   CheckHeapMemory((U8BIT *)s1, (U16BIT)strlen(s2));

   #undef strcpy
   return strcpy(s1, s2);
}

char* STB_StrNCopy(char *s1, const char *s2, U16BIT n)
{
   CheckHeapMemory((U8BIT *)s1, n);

   #undef strncpy
   return strncpy(s1, s2, n);
}

char* STB_StrCat(char *s1, const char *s2)
{
   CheckHeapMemory((U8BIT *)s1 + strlen(s1), strlen(s2) + 1);

   #undef strcat
   return strcat(s1, s2);
}

char* STB_StrNCat(char *s1, const char *s2, U16BIT n)
{
   //n == max copy characters (not including terminating null)
   #define min_number(a, b)  (((a) < (b)) ? (a) : (b))
   CheckHeapMemory((U8BIT *)s1 + strlen(s1), min_number(n, strlen(s2)) + 1);

   #undef strncat
   return strncat(s1, s2, n);
}

#undef vsprintf
int STB_VSPrintF(char *s1, const char *fmt, va_list va)
{
   int count;

   count = vsprintf(s1, fmt, va);

   //count = num copy characters (not including terminating null)
   CheckHeapMemory((U8BIT *)s1, (U16BIT)(count + 1));
   return count;
}

int STB_SPrintF(char *s1, const char *fmt, ...)
{
   int count;
   va_list va_ptr;

   va_start(va_ptr, fmt);
   count = vsprintf(s1, fmt, va_ptr);
   va_end(va_ptr);

   //count = num copy characters (not including terminating null)
   CheckHeapMemory((U8BIT *)s1, (U16BIT)(count + 1));
   return count;
}

int STB_SNPrintF(char *s1, U16BIT n, const char *fmt, ...)
{
   int count;
   char *tmpbuff;
   va_list va_ptr;

   CheckHeapMemory((U8BIT *)s1, n);

   tmpbuff = (char *)STB_AppGetMemory(16000);

   va_start(va_ptr, fmt);
   count = vsprintf(tmpbuff, fmt, va_ptr);
   va_end(va_ptr);

   //n == max copy characters (inclusive of terminating null)
   //count = num copy characters (not including terminating null)
   if (count >= n)
   {
      count = n - 1;
   }

   #ifdef memcpy
      #undef memcpy
   #endif
   memcpy(s1, tmpbuff, count);
   *(s1 + count) = '\0';

   STB_AppFreeMemory((void *)tmpbuff);

   return count;
}

#endif //#ifdef DEBUG_MEMCOPY

#ifdef DEBUG_HEAP_STATS
#ifdef DEBUG_APP_STATS
void STB_GetHeapStats(U32BIT *total_app, U32BIT *max_app, U32BIT *num_app,
   U32BIT *total_mem, U32BIT *max_mem, U32BIT *num_mem)
{
   *total_app = app_stats.total;
   *max_app = app_stats.maximum;
   *num_app = app_stats.sum;

   *total_mem = heap_stats.total;
   *max_mem = heap_stats.maximum;
   *num_mem = heap_stats.sum;
}

#endif
#endif

//*****************************************************************************
// End of file
//*****************************************************************************

