/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2010 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   STB midware Huffman decompression routines defined by the BBC
 * @file    stbhuffman.c
 * @date    17 February 2010
 * @author  Steve Ford
 */

/*#define ENABLE_DEBUG*/

/*---includes for this file--------------------------------------------------*/

/* compiler library header files */

/* third party header files */

/* DVBCore header files */
#include "techtype.h"
#include "dbgfuncs.h"

#include <huffman_table1.h>
#include <huffman_table2.h>

/*---constant definitions for this file--------------------------------------*/

#ifdef ENABLE_DEBUG
#define DBG(x)       STB_SPDebugWrite x
#else
#define DBG(x)
#endif

/*---local typedef structs for this file-------------------------------------*/

/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local) */

/*---local function prototypes for this file---------------------------------*/
/*   (internal functions declared static to make them local) */

/*---global function definitions---------------------------------------------*/

/*!**************************************************************************
 * @brief   Decompresses the input buffer according to the BBC's Huffman algorithm
 *          as defined in the DTG D-Book 6.1 and using the licensed tables provided
 *          by the BBC that relate to the algorithm.
 * @param   encoding_type - Defines which table should be used
 * @param   input - start of buffer containing the encoded data
 * @param   output - allocated buffer into which the decoded data will be placed
 * @param   output_size - size of the supplied output buffer
 * @return  Number of bytes decompressed into the output buffer, 0 when the
 *          decompression failes.
 ****************************************************************************/
U16BIT STB_HuffmanDecompress(U8BIT encoding_type, U8BIT *input, U8BIT *output, U16BIT output_size)
{
   U8BIT *table = NULL;
   U8BIT prev_char;
   U8BIT value;
   U8BIT mask;
   U8BIT i, mask_pos;
   U16BIT byte_offset;
   U16BIT num_output_bytes = 0;
   BOOLEAN finished;
   BOOLEAN char_decoded;
   BOOLEAN compressed_char;

   FUNCTION_START(STB_HuffmanDecompress);

   if ((encoding_type == 1) || (encoding_type == 2))
   {
      if (encoding_type == 1)
      {
#ifndef HUFFMAN_TABLE_1_MISSING
         table = huffman_table_1;
#endif
      }
      else
      {
#ifndef HUFFMAN_TABLE_2_MISSING
         table = huffman_table_2;
#endif
      }

      if (table != NULL)
      {
         /* When starting, the previous byte is always 0 */
         prev_char = 0;
         finished = FALSE;
         mask = 0x80;
         mask_pos = 7;
         compressed_char = FALSE;

         for (finished = FALSE; !finished && (num_output_bytes < output_size); )
         {
            if (((prev_char == 0x1b) && compressed_char) || ((prev_char >= 0x80) && !compressed_char))
            {
               compressed_char = FALSE;
               for (i = 0, prev_char = 0; i < 8; i++)
               {
                  prev_char <<= 1;
                  prev_char |= ((*input & mask) >> mask_pos);

                  mask >>= 1;
                  mask_pos--;
                  if (mask == 0)
                  {
                     mask = 0x80;
                     mask_pos = 7;
                     input++;
                  }
               }

               *output = prev_char;

               if (prev_char == 0)
               {
                  finished = TRUE;
               }
               else
               {
                  output++;
               }

               num_output_bytes++;
            }
            else
            {
               compressed_char = TRUE;
               byte_offset = (table[2 * prev_char] << 8) + table[2 * prev_char + 1];

               if ((*input & mask) == 0)
               {
                  value = table[byte_offset];
               }
               else
               {
                  value = table[byte_offset + 1];
               }

               mask >>= 1;
               mask_pos--;
               if (mask == 0)
               {
                  mask = 0x80;
                  mask_pos = 7;
                  input++;
               }

               char_decoded = FALSE;

               while (!char_decoded)
               {
                  if ((value & 0x80) == 0)
                  {
                     if ((*input & mask) == 0)
                     {
                        value = table[byte_offset + 2 * (value & 0x7f)];
                     }
                     else
                     {
                        value = table[byte_offset + 2 * (value & 0x7f) + 1];
                     }

                     mask >>= 1;
                     mask_pos--;
                     if (mask == 0)
                     {
                        mask = 0x80;
                        mask_pos = 7;
                        input++;
                     }
                  }
                  else
                  {
                     prev_char = value & 0x7f;

                     if (prev_char != 0x1b)
                     {
                        *output = prev_char;

                        if (prev_char == 0)
                        {
                           finished = TRUE;
                        }
                        else
                        {
                           output++;
                        }

                        num_output_bytes++;
                     }

                     char_decoded = TRUE;
                  }
               }
            }
         }

         if (!finished)
         {
            /* Failed to decompress the whole string. Put a NULL char on the end */
            *(output - 1) = '\0';
         }
      }
   }

   FUNCTION_FINISH(STB_HuffmanDecompress);

   return(num_output_bytes);
}

