/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   PES collection task, pieces together PES fragments to form
 *          completed packet and place in an appropriate queue.
 *
 * @file    stbpes.c
 * @date    04/02/2004
 * @author  Ocean Blue
 */
/* #define STB_PES_COLLECTION_PRINT_REQUIRED */

//---includes for this file----------------------------------------------------
// compiler library header files

#include <string.h>

// third party header files

// Ocean Blue Software header files

#include <techtype.h>
#include <dbgfuncs.h>

#include "stbhwos.h"
#include "stbhwdmx.h"
#include "stbheap.h"
#include "stbpes.h"
#include "stbdpc.h"


//---constant definitions for this file----------------------------------------

#define PES_COLL_TASK_STACK_SIZE       1024 * 4
#define PES_COLLECTION_PRIORITY        STB_TPID_CBUFF_PRIORITY - 1

// pes_hdr_data_len byte location withinthe PES packet structure
#define PES_HDR_DATA_LEN_BYTE 0x08

typedef enum e_pes_state
{
   LOOKING_PREFIX_BYTE_ONE,
   LOOKING_PREFIX_BYTE_TWO,
   LOOKING_PREFIX_CODE_THREE,
   LOOKING_STREAM_ID,

   LOOKING_LENGTH_BYTE_ONE,
   LOOKING_LENGTH_BYTE_TWO,

   LOOKING_FOR_END
} E_PES_STATE;

#define FIXED_BUFFER_LENGTH 8192L

// Debug message macros

#ifdef STB_PES_COLLECTION_PRINT_REQUIRED
   #define STB_PRINT_PES_ERROR
   #ifdef STB_PES_COLLECTION_CHANNEL_DEBUG_REQUIRED
      #define STB_PES_COLLECTION_PRINT(x) DEBUG_PRINTX_CONDITIONAL(DEBUG_STB_OSD) x
   #else
      #define STB_PES_COLLECTION_PRINT(x) STB_SPDebugWrite x
   #endif
#else
   #define STB_PES_COLLECTION_PRINT(x)
#endif

//---local typedef structs for this file---------------------------------------

typedef struct s_pes_request
{
   U32BIT handle;
   U8BIT lowest_data_identifier;
   U8BIT highest_data_identifier;
   void (*callback_function)(U32BIT, U8BIT, void *, U32BIT);  // (handle, data_identifier, pes_data_ptr, pes_data_length)

   struct s_pes_request *next_ptr;
} S_PES_REQUEST;

//---local (static) variable declarations for this file------------------------
//   (internal variables declared static to make them local)

static BOOLEAN initialised;

static void *pes_collection_task;

static U8BIT pes_collection_path = INVALID_RES_ID;
static BOOLEAN reset_pes_collection_task;

static S_PES_REQUEST *pes_request_link_list;
static void *pes_request_semaphore;

static U32BIT next_avaialable_handle;

static U8BIT *pes_fixed_buffer;
static BOOLEAN task_running;

//---local function prototypes for this file-----------------------------------
//   (internal functions declared static to make them local)

static void PesCollectionTask(void *unused);

//---local function definitions------------------------------------------------

/**
 *

 *
 * @brief   Task for assembling PES Packets from output of hardware layer
 *                 DMX circular buffer.
 *

 *

 *
 */
static void PesCollectionTask(void *unused)
{
   U8BIT data_identifier;
   U32BIT m_buf_size, pes_len, num_bytes_obtained;
   U32BIT nbytes, available_bytes, remaining_bytes, i;
   E_PES_STATE state;
   U8BIT *malloc_buf;
   U8BIT *buffer;
   S_PES_REQUEST *pes_request_ptr;
   U8BIT demux;
#ifdef STB_PRINT_PES_ERROR
   BOOLEAN pes_error;
#endif

   FUNCTION_START(PesCollectionTask);

   USE_UNWANTED_PARAM(unused);

   malloc_buf = NULL;
   pes_len = 0;

   state = LOOKING_PREFIX_BYTE_ONE;
   num_bytes_obtained = 0;
#ifdef STB_PRINT_PES_ERROR
   pes_error = FALSE;
#endif
   demux = INVALID_RES_ID;

   while (1)
   {
      if (reset_pes_collection_task)
      {
         reset_pes_collection_task = FALSE;
         state = LOOKING_PREFIX_BYTE_ONE;

         if (pes_collection_path != INVALID_RES_ID)
         {
            demux = STB_DPGetPathDemux(pes_collection_path);
         }
         else
         {
            demux = INVALID_RES_ID;
         }
      }

      if (demux != INVALID_RES_ID)
      {
         STB_DMXReadTextPES(demux, &buffer, &nbytes);
         STB_PES_COLLECTION_PRINT(("DMX PES returns %d", nbytes));
         if (nbytes == 0)
         {
            /* Give the OS a chance to deschedule task */
            STB_OSTaskDelay(25);
         }
      }
      else
      {
         /* Give the OS a chance to deschedule task */
         STB_OSTaskDelay(25);
         nbytes = 0;
      }

      // Find begin of a PES; PES Prefix 0x00 0x00 0x01 Then Stream ID 0xbd
      i = 0;
      while (i < nbytes)
      {
         switch (state)
         {
            case LOOKING_PREFIX_BYTE_ONE:
            {
               if (buffer[i] == 0x00)
               {
                  state = LOOKING_PREFIX_BYTE_TWO;
                  num_bytes_obtained = 0;

               #ifdef STB_PRINT_PES_ERROR
                  if (pes_error)
                  {
                     STB_SPDebugWrite("pes err@ %d", num_bytes_obtained);
                  }
                  pes_error = TRUE;
               #endif
               }
               #ifdef STB_PES_COLLECTION_PRINT_REQUIRED
               else
               {
                  STB_PES_COLLECTION_PRINT(("PES %d: %u=0x%02x", __LINE__, i, buffer[i]));
               }
               #endif
               break;
            }

            case LOOKING_PREFIX_BYTE_TWO:
            {
               if (buffer[i] == 0x00)
               {
                  state = LOOKING_PREFIX_CODE_THREE;
               }
               else
               {
                  state = LOOKING_PREFIX_BYTE_ONE;
                  STB_PES_COLLECTION_PRINT(("PES %d: %u=0x%02x\n", __LINE__, i, buffer[i]));
               }
               break;
            }

            case LOOKING_PREFIX_CODE_THREE:
            {
               if (buffer[i] == 0x01)
               {
                  state = LOOKING_STREAM_ID;
               }
               else
               {
                  if (!(buffer[i] == 0x00))
                  {
                     state = LOOKING_PREFIX_BYTE_ONE;
                     STB_PES_COLLECTION_PRINT(("PES %d: %u=0x%02x", __LINE__, i, buffer[i]));
                  }
               }
               break;
            }

            case LOOKING_STREAM_ID:
            {
               if (buffer[i] == 0xbd)
               {
                  state = LOOKING_LENGTH_BYTE_ONE;
               }
               else
               {
                  state = LOOKING_PREFIX_BYTE_ONE;
                  STB_PES_COLLECTION_PRINT(("PES %d: %u=0x%02x", __LINE__, i, buffer[i]));
               }
               break;
            }

            case LOOKING_LENGTH_BYTE_ONE:
            {
               // Plus six because length values are form the length bytes onward.
               m_buf_size = (buffer[i] << 8) + 0xff + 6;

               if (malloc_buf != NULL)
               {
                  if (malloc_buf != pes_fixed_buffer)
                  {
                     STB_FreeMemory(malloc_buf);
                  }
                  malloc_buf = NULL;
               }

               if ((m_buf_size <= FIXED_BUFFER_LENGTH) && (pes_fixed_buffer != NULL))
               {
                  malloc_buf = pes_fixed_buffer;
               }
               else
               {
                  malloc_buf = (U8BIT *)STB_GetMemory((U32BIT)m_buf_size);
               }

               if (malloc_buf == NULL)
               {
                  state = LOOKING_PREFIX_BYTE_ONE;
               }
               else
               {
                  malloc_buf[0] = 0x00;
                  malloc_buf[1] = 0x00;
                  malloc_buf[2] = 0x01;
                  malloc_buf[3] = 0xbd;
                  malloc_buf[4] = buffer[i];
                  pes_len = (buffer[i] << 8);
                  state = LOOKING_LENGTH_BYTE_TWO;
               }
               break;
            }

            case LOOKING_LENGTH_BYTE_TWO:
            {
               malloc_buf[5] = buffer[i];
               pes_len += buffer[i];

               state = LOOKING_FOR_END;

               // Six because length values are form the length bytes onward.
               num_bytes_obtained = 6;
               break;
            }

            case LOOKING_FOR_END:
            {
               // PES continues into this buffer
               // Plus six because length values are form the length bytes onward.
               remaining_bytes = (pes_len + 6) - num_bytes_obtained;

               if (remaining_bytes <= (nbytes - i))
               {
                  // Remainder of PES present in this buffer
               #ifdef STB_PRINT_PES_ERROR
                  pes_error = FALSE;
               #endif
                  memcpy((void *)&malloc_buf[num_bytes_obtained], (void *)&buffer[i], (size_t)remaining_bytes);

                  data_identifier = malloc_buf[PES_HDR_DATA_LEN_BYTE + malloc_buf[PES_HDR_DATA_LEN_BYTE] + 1];

                  if (!reset_pes_collection_task)
                  {
                     STB_PES_COLLECTION_PRINT(("PES size %d", pes_len + 6));

                     // We are going to traverse a link-list wich can be altered by other threads.
                     STB_OSSemaphoreWait(pes_request_semaphore);

                     pes_request_ptr = pes_request_link_list;

                     while (pes_request_ptr != NULL)
                     {
                        if ((data_identifier >= pes_request_ptr->lowest_data_identifier) &&
                            (data_identifier <= pes_request_ptr->highest_data_identifier))
                        {
                           (*pes_request_ptr->callback_function)(pes_request_ptr->handle,
                              data_identifier,
                              (void *)malloc_buf,
                              num_bytes_obtained + remaining_bytes);
                        }

                        pes_request_ptr = pes_request_ptr->next_ptr;
                     }

                     STB_OSSemaphoreSignal(pes_request_semaphore);

                     if (malloc_buf != pes_fixed_buffer)
                     {
                        STB_FreeMemory(malloc_buf);
                     }
                     malloc_buf = NULL;
                  }

                  state = LOOKING_PREFIX_BYTE_ONE;

                  // Because the remainder of this PES is in this buffer, there may be another PES
                  // starting at the end of this buffer.
                  // NB. added minus one because of the i++ later.
                  i += remaining_bytes - 1;
               }
               else
               {
                  // PES continues into another buffer
                  available_bytes = nbytes - i;
                  memcpy((void *)&malloc_buf[num_bytes_obtained], (void *)&buffer[i], (size_t)available_bytes);
                  num_bytes_obtained += available_bytes;
                  i += available_bytes;
               }
               break;
            }

            default:
            {
               state = LOOKING_PREFIX_BYTE_ONE;
               STB_PES_COLLECTION_PRINT(("PES %d: %u=0x%02x", __LINE__, i, buffer[i]));
               break;
            }
         }
         i++;
      }
   }

   STB_PES_COLLECTION_PRINT(("PesCollectionTask Terminated"));

   FUNCTION_FINISH(PesCollectionTask);
}

//---global function definitions-----------------------------------------------

/**
 *

 *
 * @brief   Initialises the PES collection task.
 *

 *

 *
 */
void STB_PesCollectionTaskInitialise(void)
{
   FUNCTION_START(STB_PesCollectionTaskInitialise);

   if (!initialised)
   {
      pes_fixed_buffer = (U8BIT *)STB_GetMemory(FIXED_BUFFER_LENGTH);

      pes_request_semaphore = STB_OSCreateSemaphore();

      pes_collection_task = STB_OSCreateTask(PesCollectionTask, NULL, PES_COLL_TASK_STACK_SIZE,
            PES_COLLECTION_PRIORITY, (U8BIT *)"PesColl");

      reset_pes_collection_task = FALSE;
      initialised = TRUE;
   }

   FUNCTION_FINISH(STB_PesCollectionTaskInitialise);
}

/**
 *

 *
 * @brief   Flushes the PES collection task.
 *

 *

 *
 */
void STB_FlushPesCollectionTask(void)
{
   FUNCTION_START(STB_FlushPesCollectionTask);

   reset_pes_collection_task = TRUE;

   FUNCTION_FINISH(STB_FlushPesCollectionTask);
}

/**
 *

 *
 * @brief   Changes the PID that the DMX is using to gather data on.
 *

 *

 *
 */
void STB_ChangePesCollectionPID(U8BIT path, U16BIT text_pid)
{
   FUNCTION_START(STB_ChangePesCollectionPID);

   if ((text_pid != STB_DPGetTextPID(path)) || (path != pes_collection_path))
   {
      if ((pes_collection_path != INVALID_RES_ID) && (pes_collection_path != path))
      {
         /* Stop the current PES collection */
         STB_DPSetTextPID(pes_collection_path, 0);
      }

      /* Set the new PID */
      STB_DPSetTextPID(path, text_pid);

      reset_pes_collection_task = TRUE;

      if (text_pid == 0)
      {
         /* PES collection has been stopped */
         pes_collection_path = INVALID_RES_ID;
      }
      else
      {
         pes_collection_path = path;
         task_running = TRUE;
      }
   }

   FUNCTION_FINISH(STB_ChangePesCollectionPID);
}

/**
 *

 *
 * @brief   Used to register a callback function that will receive specific PES data packets.
 *                 In theory, an unlimited number of callbacks can be registered, and each will be
 *                 sent data irrespective of other callback requests.
 *                 The returned handle can be used in conjunction with
 *                 STB_UnregisterPesCollectionCallback to remove the callback
 *                 from this task's functionality.
 *
 *                 This function is anticipated only to be called from other STB functionality.
 *
 * @param   callback_function - a pointer to a correspondingly profiled function that will
 *                                     recieve specific PES data packets as they arrive.
 *                                     The parameters returned to the callback are as follows:
 * @param   U32BIT - the handle associated with this initialisation request.
 *                                     This can be used to call () from within the callback if
 *                                     required.
 * @param   U8BIT  - the data identifier of the supplied packet.
 *                                     This is useful to differentiate packets within the callback
 *                                     if a range of data identifier types is requested.
 * @param   void*  - a pointer to the PES data packet.
 * @param   U32BIT - the width (in bytes) of the data packet supplied.
 *
 *                            NB - the callback MUST NOT free the memory supplied by the void*
 *                                 reference, and MUST allocate it's own copy if it requires data
 *                                 persistance.
 *
 * @param   U8BIT lowest_data_identifier -
 * @param   U8BIT highest_data_identifier - used to specify the PES data content being
 *                                                 passed to the callback funtion.
 *                                                 These paramaters can be set to the same value if
 *                                                 data from a single identifier is required.
 *
 * @return   U32BIT - a unique handle to the PES callback request, or zero of failed.
 *
 */
U32BIT STB_RegisterPesCollectionCallback(void (*callback_function)(U32BIT, U8BIT, void *, U32BIT),
   U8BIT lowest_data_identifier, U8BIT highest_data_identifier)
{
   U32BIT retval;
   S_PES_REQUEST *pes_request_ptr;

   FUNCTION_START(STB_RegisterPesCollectionCallback);

   retval = 0L;

   if (initialised == TRUE)
   {
      // If we have valid parameters....
      if ((callback_function) &&
          (lowest_data_identifier <= highest_data_identifier))
      {
         // ...and we can add allocate a PES reqyest structure
         pes_request_ptr = (S_PES_REQUEST *)STB_GetMemory(sizeof(S_PES_REQUEST));
         if (pes_request_ptr != NULL)
         {
            // Contrive a unique handle to be returned - can be used to cancel the callback request,
            next_avaialable_handle++;
            pes_request_ptr->handle = next_avaialable_handle;

            // Make record of PES request details.
            pes_request_ptr->callback_function = callback_function;
            pes_request_ptr->lowest_data_identifier = lowest_data_identifier;
            pes_request_ptr->highest_data_identifier = highest_data_identifier;

            // We are going to alter a link-list traversed by several threads.
            STB_OSSemaphoreWait(pes_request_semaphore);

            // Attach PES request record to head of lin-list.
            pes_request_ptr->next_ptr = pes_request_link_list;
            pes_request_link_list = pes_request_ptr;

            // Return value is request record handle.
            retval = pes_request_ptr->handle;

            STB_OSSemaphoreSignal(pes_request_semaphore);
         }
      }
   }

   FUNCTION_FINISH(STB_RegisterPesCollectionCallback);

   return(retval);
}

/**
 *

 *
 * @brief   Used to un-register a callback function that will receive specific PES data
 *                 packets.
 *
 *                 This function is anticipated only to be called from other STB functionality.
 *
 * @param   U32BIT handle - a unique handle to the PES callback request made in a prior call
 *                                 to STB_RegisterPesCollectionCallback( ) that is to be removed
 *                                 from the PES collection functionality.
 *

 *
 */
void STB_UnregisterPesCollectionCallback(U32BIT handle)
{
   S_PES_REQUEST *pes_request_ptr;
   S_PES_REQUEST *prev_pes_request_ptr;

   FUNCTION_START(STB_UnregisterPesCollectionCallback);

   if (initialised == TRUE)
   {
      // We are going to alter a link-list traversed by several threads.
      STB_OSSemaphoreWait(pes_request_semaphore);

      // Initialise pointers used to traverse the link-list.
      pes_request_ptr = pes_request_link_list;
      prev_pes_request_ptr = NULL;

      // Go through link-list looking for handle match
      while (pes_request_ptr != NULL)
      {
         if (handle == pes_request_ptr->handle)
         {
            // If this is the first record on the link list...
            if (prev_pes_request_ptr == NULL)
            {
               // ... then just modify the link-list header pointer.
               pes_request_link_list = pes_request_link_list->next_ptr;
            }
            else
            {
               // ... then modify the previos link-list record to point to the next.
               prev_pes_request_ptr->next_ptr = pes_request_ptr->next_ptr;
            }

            // The record has now been removed from the link-list, so can be safely freed.
            STB_FreeMemory(pes_request_ptr);

            // Once matching handle has been delat with, quit.
            break;
         }

         // Move to next link-list item
         prev_pes_request_ptr = pes_request_ptr;
         pes_request_ptr = pes_request_ptr->next_ptr;
      }

      STB_OSSemaphoreSignal(pes_request_semaphore);
   }

   FUNCTION_FINISH(STB_UnregisterPesCollectionCallback);
}

//*****************************************************************************
// End of file
//*****************************************************************************

