/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   PVR control functions
 *
 * @file    stbpvr.c
 * @date    07/02/2003
 */


// gives direct COM port access
/*#define STB_DEBUG*/
/*#define DEBUG_CIPLUS*/

//---includes for this file----------------------------------------------------
// compiler library header files

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* third party header files */

/* DVBCore header files */

#include <techtype.h>
#include <dbgfuncs.h>

#include "stbhwos.h"
#include "stbhwav.h"
#include "stbpvrpr.h"
#include "stbhwdsk.h"
#include "stbhwmem.h"
#include "stbhwcrypt.h"

#include "stbpvr.h"
#include "stbheap.h"
#include "stbheap.h"
#include "stbgc.h"
#include "stbuni.h"
#include "stbdpc.h"
#include "stbpvrmsg.h"
#include "stberc.h"
#include "stbllist.h"

#ifdef COMMON_INTERFACE
#include "stbhwdmx.h"
#include "stbsitab.h"
#include "stbcicc.h"
#endif

/*---constant definitions for this file----------------------------------------*/
#ifdef STB_DEBUG
   #define STB_PVR_PRINT(x) STB_SPDebugWrite x
#else
   #define STB_PVR_PRINT(x)
#endif

#ifdef DEBUG_CIPLUS
   #define DBG_CIP(x, ...)   {U32DHMS gmt = STB_GCNowDHMSGmt(); \
      STB_SPDebugWrite("%02u:%02u:%02u: %s" x, DHMS_HOUR(gmt), DHMS_MINS(gmt), DHMS_SECS(gmt), __FUNCTION__, ##__VA_ARGS__); }
#else
   #define DBG_CIP(x, ...)
#endif

#if 0
#ifdef FUNCTION_START
#undef FUNCTION_START
#endif
#define FUNCTION_START(X)  STB_SPDebugWrite(">>> %s\n", # X)
#ifdef FUNCTION_FINISH
#undef FUNCTION_FINISH
#endif
#define FUNCTION_FINISH(X)  STB_SPDebugWrite("<<< %s\n", # X)
#endif

#define HANDLE_BASE           0x00000001

#define DB_FILENAME_LEN       32
#define BASENAME_LEN          16

#define DB_FILE_EXTENSION     ".odb"
#define DB_FILE_EXTENTION_LEN 4

#define ODB_MAGIC_1           'O'
#define ODB_MAGIC_2           'd'
#define ODB_MAGIC_3           'B'

#define ODB_VERSION_1         1
#define ODB_VERSION_2         2
#define ODB_CUR_VERSION       ODB_VERSION_2

#define BOOKMARK_FILENAME_SIZE   10
#define BOOKMARK_TOLERANCE    30    /* +/- number of seconds allowed when finding an existing bookmark */

#define REC_AD_RECEIVER_MIX   0
#define REC_AD_BROADCAST_MIX  1

#define REC_INFO_BUFFER_SIZE  ((sizeof(S_REC_INFO) + (AES128_KEY_SIZE - 1)) & ~(AES128_KEY_SIZE - 1))

/* Flags stored with a recording */
#define REC_LOCKED         0x0001
#define REC_ENCRYPTED      0x0002
#define REC_HAS_GUIDANCE   0x0004
#define REC_SERIES         0x0008
#define REC_RECOMMENDATION 0x0010
#define REC_HAS_VIDEO      0x0020
#define REC_PARENTAL_LOCK  0x0040

/*---local typedef structs for this file---------------------------------------*/

#ifdef COMMON_INTERFACE
typedef struct
{
   BOOLEAN used;
   U16BIT length;
   U8BIT *data;
} S_REC_CIPLUS_LICENCE;

typedef struct
{
   U8BIT age_rating;
   U8BIT private_data[CIP_PIN_PRIVATE_DATA_SIZE];
} S_REC_CIPLUS_PIN;

typedef struct s_rec_ciplus_item
{
   U32BIT timestamp;
   U16BIT item_type;
   union
   {
      U8BIT uri[CIP_URI_LEN];
      S_REC_CIPLUS_LICENCE licence;
      S_REC_CIPLUS_PIN pin;
   } u;
   struct s_rec_ciplus_item *prev;
   struct s_rec_ciplus_item *next;
} S_REC_CIPLUS_ITEM;
#endif

typedef struct
{
   U32BIT handle;
#ifdef COMMON_INTERFACE
   U8BIT slot_id;
   void *play_sem;
   S_REC_CIPLUS_ITEM *item_list;
   S_REC_CIPLUS_ITEM *uri_item;
   S_REC_CIPLUS_ITEM *licence_item;
   S_REC_CIPLUS_ITEM *pin_item;
#endif
} S_PVRPLAY_STATUS;

#ifdef COMMON_INTERFACE
typedef struct
{
   BOOLEAN valid;
   U8BIT key[16];
   U8BIT iv[16];
} S_KEY_INFO;
#endif

typedef struct
{
   U32BIT handle;
   U32BIT start_time;
   U32BIT duration;
   BOOLEAN paused;
   BOOLEAN encrypted;
   U16BIT num_pids;
   S_PVR_PID_INFO *rec_pids_array;
   U16BIT serv_id;
   U16BIT ts_id;
   U16BIT orig_net_id;
   U8BIT rec_index;
   E_STB_PVR_START_MODE smode;
   U32BIT timeshift_seconds;
#ifdef COMMON_INTERFACE
   E_STB_DMX_DESC_TYPE desc_type;
   S_KEY_INFO keys[2];
#endif
} S_PVRRECORD_STATUS;

typedef struct
{
   U16BIT disk_id;
   BOOLEAN valid;
} PVR_DB_DISK;

typedef struct
{
   U8BIT magic[3];
   U8BIT version;
} S_REC_HEADER;

typedef struct
{
   U16BIT rec_date;
   U8BIT rec_hour;
   U8BIT rec_min;
   U8BIT rec_sec;
   U8BIT len_hour;
   U8BIT len_min;
   U8BIT len_sec;
   U8BIT name[STB_PVR_NAME_LEN];
   U8BIT prog_crid[STB_PVR_MAX_CRID_LEN];
   U8BIT other_crid[STB_PVR_MAX_CRID_LEN];
   U8BIT additional_info[STB_PVR_ADDITIONAL_INFO_LEN];
   U16BIT serv_id;
   U16BIT ts_id;
   U16BIT orig_net_id;
   U32BIT flags;
   U32BIT parental_rating;
   S32BIT start_padding;
   S32BIT end_padding;
   U8BIT status;
} S_REC_INFO;

enum
{
   EXT_INFO_SERVICE_NAME,
   EXT_INFO_SHORT_DESC,
   EXT_INFO_GUIDANCE,
   EXT_INFO_EXTENDED_DESC,
   EXT_INFO_CIPLUS_LICENCE,
   EXT_INFO_CIPLUS_URI,
   EXT_INFO_CIPLUS_CICAM_ID,
   EXT_INFO_CIPLUS_PIN
};

typedef struct s_recording
{
   U32BIT handle;
   U16BIT disk_id;
   BOOLEAN recording;
   BOOLEAN selected;
   S_REC_INFO rec_info;
   U8BIT basename[BASENAME_LEN];
   struct s_recording *prev;
   struct s_recording *next;
} S_RECORDING;

typedef enum
{
   BOOKMARK_USER,
   BOOKMARK_PLAYBACK_POSITION
} E_BOOKMARK_TYPE;

typedef struct
{
   LINK_LIST_PTR_BLK ptrs;

   U32BIT time;
   U8BIT *name;
} S_BOOKMARK;


//---local (static) variable declarations for this file------------------------
//   (internal variables declared static to make them local)

static BOOLEAN initialised_flag = FALSE;
static BOOLEAN formatting_flag = FALSE;
static BOOLEAN repairing_flag = FALSE;

static U8BIT num_paths;

static S_PVRPLAY_STATUS *play_status;
static S_PVRRECORD_STATUS *record_status;

static void *disk_mutex;
static PVR_DB_DISK *disks_in_db;
static U16BIT num_disks_in_db;

static U16BIT default_pvr_disk_id = INVALID_DISK_ID;

static void *list_mutex;
static S_RECORDING *rec_list;

static U8BIT *enc_dec_key;
static U8BIT *enc_dec_iv;
static U32BIT enc_dec_key_len;
static U8BIT *rec_info_buffer;

//---local function prototypes for this file-----------------------------------
//   (internal functions declared static to make them local)

static void ReadRecordings(U16BIT disk_id);
static void RemoveRecordings(U16BIT disk_id);

static U32BIT GetCurrentRecordTime(U32BIT handle);

static BOOLEAN WriteODBFile(U16BIT disk_id, U8BIT *basename, BOOLEAN write_header,
   S_REC_INFO *rec_info);
static BOOLEAN WriteExtendedInfo(U16BIT disk_id, U8BIT *basename, U16BIT ext_id,
   U32BIT data_len, U8BIT *data);
static BOOLEAN ReadExtendedInfo(U16BIT disk_id, U8BIT *basename, U16BIT ext_id, U8BIT *data,
   U32BIT data_len);
static U32BIT GetExtendedInfoSize(U16BIT disk_id, U8BIT *basename, U16BIT ext_id);
static void InsertRecordingInList(S_RECORDING *rec_ptr);
static void RemoveRecordingFromList(S_RECORDING *rec_ptr);
static U32BIT GetNextFreeHandle(void);
static S_RECORDING* GetRecording(U32BIT handle);
static void GetDBRecordBasename(U32BIT file_number, U8BIT *filename);
static void GetDBRecordFilename(U32BIT file_number, U8BIT *filename);
static BOOLEAN IncludesVideoPid(S_PVR_PID_INFO *pid_array, U16BIT num_pids);

#ifdef COMMON_INTERFACE
static void FreeCIPlusItemList(U8BIT path);
static void FreeCIPlusItem(S_REC_CIPLUS_ITEM *item);
static void ReadCIPlusItems(U8BIT path, S_RECORDING *rec_ptr);
static void ApplyCIPlusItems(U8BIT path, S_RECORDING *rec_ptr, U32BIT position_in_seconds,
   S16BIT speed);
static void FindNextCIPlusItem(U8BIT path, S16BIT speed);
static BOOLEAN WriteCIPlusInfo(U16BIT disk_id, U8BIT *basename, U16BIT ext_id, U32BIT timestamp,
   U32BIT data_len, U8BIT *data);
static BOOLEAN UpdateCIPlusInfo(U16BIT disk_id, U8BIT *basename, U16BIT ext_id, U32BIT timestamp,
   U32BIT data_len, U8BIT *data);
#endif

static void GetBookmarkFolderName(U8BIT *basename, U8BIT *bookmark_folder_name);
static void GetBookmarkFileName(U32BIT time, E_BOOKMARK_TYPE type, U8BIT *name);
static E_BOOKMARK_TYPE GetBookmarkType(U8BIT *file_name);
static U32BIT GetBookmarkTime(U8BIT *file_name);
static BOOLEAN AddBookmark(U16BIT disk_id, U8BIT *basename, U32BIT time, U8BIT *name,
   E_BOOKMARK_TYPE type);
static void RemoveBookmark(U16BIT disk_id, U8BIT *basename, U32BIT time, U8BIT *name,
   E_BOOKMARK_TYPE type);
static void RemoveAllBookmarks(U16BIT disk_id, U8BIT *basename);
static U32BIT GetBookmarks(U16BIT disk_id, U8BIT *basename, LINK_LIST_HEADER *bookmarks,
   E_BOOKMARK_TYPE type, BOOLEAN names);
static void EmptyBookmarks(LINK_LIST_HEADER *bookmarks, BOOLEAN release);
static S16BIT CompareBookmarks(LINK_LIST_PTR_BLK **bookmark1, LINK_LIST_PTR_BLK **bookmark2);


//---global function definitions-----------------------------------------------

/**
 *

 *
 * @brief   Initialises PVR path control.
 *

 *
 * @return   TRUE if initialised, else FALSE
 *
 */
BOOLEAN STB_PVRInitialise(void)
{
   U16BIT i;
   U8BIT num_recorders, num_players;
   U8BIT num_audio_decoders, num_video_decoders;

   FUNCTION_START(STB_PVRInitialise);

   num_paths = STB_DPGetNumPaths();
   num_recorders = STB_HWGetNumRecorders();
   num_audio_decoders = STB_HWGetAudioDecodePaths();
   num_video_decoders = STB_HWGetVideoDecodePaths();

   STB_PVR_PRINT(("STB_PVRInitialise: tuners = %d, audio/video decoders = %d/%d", num_recorders,
                  num_audio_decoders, num_video_decoders));

   num_players = STB_PVRInitPlayback(num_audio_decoders, num_video_decoders);
   num_recorders = STB_PVRInitRecording(num_recorders);

   STB_PVR_PRINT(("STB_PVRInitialise: num players = %d, num recorders = %d", num_players, num_recorders));

   if ((num_players > 0) && (num_recorders > 0))
   {
      play_status = (S_PVRPLAY_STATUS *)STB_GetMemory((U32BIT)(sizeof(S_PVRPLAY_STATUS) * num_paths));
      if (play_status != NULL)
      {
         for (i = 0; i < num_paths; i++)
         {
            play_status[i].handle = 0;
#ifdef COMMON_INTERFACE
            play_status[i].play_sem = STB_OSCreateSemaphore();
            play_status[i].item_list = NULL;
            play_status[i].uri_item = NULL;
            play_status[i].licence_item = NULL;
            play_status[i].pin_item = NULL;
#endif
         }
      }

      record_status = (S_PVRRECORD_STATUS *)STB_GetMemory((U32BIT)(sizeof(S_PVRRECORD_STATUS) * num_paths));
      if (record_status != NULL)
      {
         for (i = 0; i < num_paths; i++)
         {
            record_status[i].handle = 0;
            record_status[i].start_time = 0;
            record_status[i].duration = 0;
            record_status[i].paused = FALSE;
            record_status[i].encrypted = FALSE;
            record_status[i].rec_index = INVALID_RES_ID;
            record_status[i].num_pids = 0;
            record_status[i].rec_pids_array = NULL;
#ifdef COMMON_INTERFACE
            record_status[i].keys[KEY_PARITY_EVEN].valid = FALSE;
            record_status[i].keys[KEY_PARITY_ODD].valid = FALSE;
#endif
         }
      }

      list_mutex = STB_OSCreateMutex();

      STB_PVRInitialiseMessages();

      initialised_flag = TRUE;
   }

   disk_mutex = STB_OSCreateMutex();

   disks_in_db = NULL;
   num_disks_in_db = 0;
   default_pvr_disk_id = INVALID_DISK_ID;

   /* Initialise buffers to be used for encrypting/decrypting data */
   enc_dec_key = (U8BIT *)STB_MEMReadSecureConstant(SECURE_NVM_DEFAULT_ENCRYPTION_KEY, &enc_dec_key_len);
   enc_dec_iv = (U8BIT *)STB_MEMReadSecureConstant(SECURE_NVM_DEFAULT_ENC_INIT_VECTOR, &enc_dec_key_len);

   rec_info_buffer = STB_GetMemory(REC_INFO_BUFFER_SIZE);

   FUNCTION_FINISH(STB_PVRInitialise);

   return(initialised_flag);
}

/**
 *

 *
 * @brief   Sets the standby state of the PVR path control
 *
 * @param   state - TRUE if PVR is being put into standby, FALSE otherwise
 *

 *
 */
void STB_PVRSetStandbyState(BOOLEAN state)
{
   FUNCTION_START(STB_PVRSetStandbyState);

   if (initialised_flag == TRUE)
   {
      STB_DSKSetStandby(state);
   }

   FUNCTION_FINISH(STB_PVRSetStandbyState);
}

/**
 *

 *
 * @brief   Returns Initialised status.
 *

 *
 * @return   TRUE if initialised, else FALSE
 *
 */
BOOLEAN STB_PVRIsInitialised(void)
{
   BOOLEAN ret_val;

   FUNCTION_START(STB_PVRIsInitialised);

   ret_val = initialised_flag;

   FUNCTION_FINISH(STB_PVRIsInitialised);

   return(ret_val);
}

/**
 * @brief   Sets the disk ID to be used as the default disk for PVR recordings.
 *          If this is set to INVALID_DISK_ID then the first mounted disk will be used.
 * @param   disk_id ID of the disk to use as the default
 */
void STB_PVRSetDefaultDisk(U16BIT disk_id)
{
   FUNCTION_START(STB_PVRSetDefaultDisk);

   default_pvr_disk_id = disk_id;

   FUNCTION_FINISH(STB_PVRSetDefaultDisk);
}

/**
 * @brief   Returns the set default disk, or finds the first mounted (usable) disk
 *          if a default hasn't been set
 * @return  ID of the disk, or INVALID_DISK_ID if none found
 */
U16BIT STB_PVRGetDefaultDisk(void)
{
   U16BIT num_disks;
   U16BIT index;
   U16BIT disk_id;

   FUNCTION_START(STB_PVRGetDefaultDisk);

   disk_id = default_pvr_disk_id;
   if (disk_id == INVALID_DISK_ID)
   {
      num_disks = STB_DSKGetNumDisks();

      for (index = 0; (index < num_disks) && (disk_id == INVALID_DISK_ID); index++)
      {
         disk_id = STB_DSKGetDiskIdByIndex(index);
         if ((disk_id != INVALID_DISK_ID) && !STB_DSKIsMounted(disk_id))
         {
            disk_id = INVALID_DISK_ID;
         }
      }
   }

   FUNCTION_FINISH(STB_PVRGetDefaultDisk);

   return(disk_id);
}

/**
 *

 *
 * @brief   Checks whether the given disk can be used for PVR functions
 *
 * @param   U16BIT disk_id - disk to be checked
 *
 * @return   TRUE if the disk can be used, FALSE otherwise
 *
 */
BOOLEAN STB_PVRCanDiskBeUsed(U16BIT disk_id)
{
   BOOLEAN result = FALSE;

   FUNCTION_START(STB_PVRCanDiskBeUsed);

   if (STB_DSKIsMounted(disk_id))
   {
      result = TRUE;
   }

   FUNCTION_FINISH(STB_PVRCanDiskBeUsed);

   return(result);
}

/**
 *

 *
 * @brief   Formats disk for PVR. Call this for each mode.
 *
 * @param   U16BIT disk_id - ID of the disk to be, or being, formatted
 * @param   E_STB_PVR_FORMATMODE mode - format mode
 * @param   U8BIT* prog - returns progress (0-100%).
 *
 * @return   TRUE if successful, else FALSE
 *
 */
BOOLEAN STB_PVRFormat(U16BIT disk_id, E_STB_PVR_FORMATMODE mode, U8BIT *prog)
{
   BOOLEAN ret_val = FALSE;

   FUNCTION_START(STB_PVRFormat);

   STB_PVR_PRINT(("STB_PVRFormat (mode = %d)", mode));

   switch (mode)
   {
      case FORMAT_START:
         if (formatting_flag == FALSE)
         {
            // start format if not in progress
            STB_DSKFormat(disk_id);
            formatting_flag = TRUE;
            *prog = 0;
            ret_val = TRUE;
            RemoveRecordings(disk_id);
         }
         break;

      case FORMAT_PROGRESS:
         if (formatting_flag == TRUE)
         {
            // return progress if formatting
            *prog = STB_DSKGetFormatProgress(disk_id);
            ret_val = TRUE;
         }
         break;

      case FORMAT_END:
         if (formatting_flag == TRUE)
         {
            if (STB_DSKGetFormatProgress(disk_id) == 100)
            {
               STB_PVRInitialiseMessages();
               STB_PVRUpdateRecordings(FALSE);
               formatting_flag = FALSE;
               *prog = 100;
               ret_val = TRUE;
            }
         }
         break;

      default:
         break;
   }

   FUNCTION_FINISH(STB_PVRFormat);

   return(ret_val);
}

/**
 *

 *
 * @brief   Repairs disk for PVR. Call this for each mode.
 *
 * @param   U16BIT disk_id - ID of the disk to be, or being, repaired
 * @param   E_STB_PVR_REPAIRMODE mode - repair mode
 * @param   U8BIT* prog - returns progress (0-100%).
 *
 * @return   TRUE if successful, else FALSE
 *
 */
BOOLEAN STB_PVRRepair(U16BIT disk_id, E_STB_PVR_REPAIRMODE mode, U8BIT *prog)
{
   BOOLEAN ret_val = FALSE;

   FUNCTION_START(STB_PVRRepair);

   STB_PVR_PRINT(("STB_PVRRepair (mode = %d)", mode));

   if (initialised_flag == TRUE)
   {
      switch (mode)
      {
         case REPAIR_ENQUIRE:
            if (repairing_flag == FALSE)
            {
               // return integrity if not scanning
               *prog = 0;
               ret_val = STB_DSKGetIntegrity(disk_id);
            }
            break;

         case REPAIR_START:
            if (repairing_flag == FALSE)
            {
               // start scan if not scanning
               STB_DSKRepair(disk_id);
               repairing_flag = TRUE;
               *prog = 0;
               ret_val = TRUE;
            }
            break;

         case REPAIR_PROGRESS:
            if (repairing_flag == TRUE)
            {
               // return progress if scanning
               *prog = STB_DSKGetRepairProgress(disk_id);
               ret_val = TRUE;
            }
            break;

         case REPAIR_END:
            if (repairing_flag == TRUE)
            {
               if (STB_DSKGetRepairProgress(disk_id) == 100)
               {
                  repairing_flag = FALSE;
                  *prog = 100;
                  ret_val = TRUE;
               }
            }
            break;

         default:
            break;
      }
   }

   FUNCTION_FINISH(STB_PVRRepair);

   return(ret_val);
}

/*!**************************************************************************
 * @brief   Called when a disk is added or removed and updates the disk database
 *          and the recordings that are now available
 * @param   force_load - forces the recordings database to be updated
 ****************************************************************************/
void STB_PVRUpdateRecordings(BOOLEAN force_load)
{
   U16BIT num_disks;
   U16BIT index, i;
   U16BIT disk_id;
   BOOLEAN disk_found;

   FUNCTION_START(STB_PVRUpdateRecordings);

   if (initialised_flag)
   {
      STB_OSMutexLock(disk_mutex);

      /* Mark all disks currently in the database as invalid */
      for (index = 0; index < num_disks_in_db; index++)
      {
         disks_in_db[index].valid = FALSE;
      }

      num_disks = STB_DSKGetNumDisks();

      /* Check all disks to see if there are any new ones */
      for (index = 0; index < num_disks; index++)
      {
         disk_id = STB_DSKGetDiskIdByIndex(index);
         if (STB_DSKIsMounted(disk_id))
         {
            /* Check whether the contents of this disk have already been loaded into the database */
            disk_found = FALSE;

            for (i = 0; (i < num_disks_in_db) && !disk_found; i++)
            {
               if (disks_in_db[i].disk_id == disk_id)
               {
                  disks_in_db[i].valid = TRUE;
                  disk_found = TRUE;
               }
            }

            if (!disk_found)
            {
               /* This is a new disk */
               disks_in_db = STB_ResizeMemory(disks_in_db, (num_disks_in_db + 1) * sizeof(PVR_DB_DISK));
               if (disks_in_db != NULL)
               {
                  ReadRecordings(disk_id);

                  disks_in_db[num_disks_in_db].disk_id = disk_id;
                  disks_in_db[num_disks_in_db].valid = TRUE;

                  num_disks_in_db++;
               }
            }
            else if (force_load)
            {
               ReadRecordings(disk_id);
            }
         }
      }

      /* Check to see if any disks have been removed */
      for (index = 0; index < num_disks_in_db; index++)
      {
         if (!disks_in_db[index].valid)
         {
            /* This disk is no longer present. Remove it's contents from the database */
            RemoveRecordings(disks_in_db[index].disk_id);

            for (i = index + 1; i < num_disks_in_db; i++)
            {
               disks_in_db[i - 1].disk_id = disks_in_db[i].disk_id;
               disks_in_db[i - 1].valid = disks_in_db[i].valid;
            }

            num_disks_in_db--;

            if (num_disks_in_db == 0)
            {
               STB_FreeMemory(disks_in_db);
               disks_in_db = NULL;
            }
            else
            {
               disks_in_db = STB_ResizeMemory(disks_in_db, num_disks_in_db * sizeof(PVR_DB_DISK));
            }
         }
      }

      STB_OSMutexUnlock(disk_mutex);
   }

   FUNCTION_FINISH(STB_PVRUpdateRecordings);
}

/*!**************************************************************************
 * @brief   Determines whether the given handle is in the recording database
 * @param   handle - recording handle to be checked
 * @return  TRUE if the handle is found, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRIsValidHandle(U32BIT handle)
{
   BOOLEAN valid;

   FUNCTION_START(STB_PVRIsValidHandle);

   valid = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      if (GetRecording(handle) != NULL)
      {
         valid = TRUE;
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRIsValidHandle);

   return(valid);
}

/*!**************************************************************************
 * @brief   Create a new recording and return the handle
 * @param   disk_id - ID of disk to be used for the recording
 * @param   name - name of recording
 * @param   handle - returned recording handle
 * @return  TRUE if successful, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRCreateRecording(U16BIT disk_id, U8BIT *name, U32BIT *handle)
{
   S_RECORDING *rec_ptr;
   U8BIT db_filename[DB_FILENAME_LEN];
   U8BIT basename[BASENAME_LEN];
   U8BIT tmp_str[STB_PVR_NAME_LEN];
   U32BIT file_number;
   U32BIT num_bytes;
   U16BIT date;
   U8BIT hour, min, sec;
   BOOLEAN retval;

   FUNCTION_START(STB_PVRCreateRecording);

   retval = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the next available database handle for use by a recording */
      *handle = GetNextFreeHandle();

      STB_PVR_PRINT(("STB_PVRCreateRecording(disk=0x%04x, name=\"%s\"): handle=0x%08lx", disk_id, name, *handle));

      /* Find a filename that can be used for recording on the given disk */
      file_number = 0;
      do
      {
         file_number++;

         GetDBRecordBasename(file_number, basename);
         GetDBRecordFilename(file_number, db_filename);
      }
      while (STB_DSKFileExists(disk_id, db_filename) || !STB_PVRCanBeUsedForRecording(disk_id, basename));

      /* Create a new entry for the recording */
      rec_ptr = (S_RECORDING *)STB_GetMemory(sizeof(S_RECORDING));
      if (rec_ptr != NULL)
      {
         memset(rec_ptr, 0, sizeof(S_RECORDING));

         rec_ptr->handle = *handle;
         rec_ptr->disk_id = disk_id;

         strncpy((char *)rec_ptr->basename, (char *)basename, BASENAME_LEN);

         STB_GCGetGMTDateTime(&date, &hour, &min, &sec);
         STB_GCConvertDateTime(date, hour, min, sec,
            &rec_ptr->rec_info.rec_date,
            &rec_ptr->rec_info.rec_hour,
            &rec_ptr->rec_info.rec_min,
            &rec_ptr->rec_info.rec_sec,
            CONV_LOCAL);

         if (name == NULL)
         {
            snprintf((char *)tmp_str, STB_PVR_NAME_LEN, "Recording %lu", (unsigned long)*handle);
            strncpy((char *)rec_ptr->rec_info.name, (char *)tmp_str, STB_PVR_NAME_LEN);
         }
         else
         {
            num_bytes = STB_GetNumBytesInString(name);
            if (num_bytes > sizeof(rec_ptr->rec_info.name))
            {
               /* If the name supplied is longer than the space to store it then it can't really
                * be stored because we don't know how to terminate the string correctly
                * (e.g. it might be compressed), so just have to store a default name to be safe */
               snprintf((char *)tmp_str, STB_PVR_NAME_LEN, "Recording %lu", (unsigned long)*handle);
               strncpy((char *)rec_ptr->rec_info.name, (char *)tmp_str, STB_PVR_NAME_LEN);
            }
            else
            {
               memcpy(rec_ptr->rec_info.name, name, num_bytes);
            }
         }

         if (WriteODBFile(rec_ptr->disk_id, rec_ptr->basename, TRUE, &rec_ptr->rec_info))
         {
            InsertRecordingInList(rec_ptr);
            retval = TRUE;
         }
         else
         {
            STB_FreeMemory(rec_ptr);
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRCreateRecording);

   return(retval);
}

/*!**************************************************************************
 * @brief   Completely destroys a recording by deleting all files and removing
 *          it from the list of recordings
 * @param   handle - handle of recording to be destroyed
 * @return  TRUE if recording found and deleted, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRDestroyRecording(U32BIT handle)
{
   S_RECORDING *rec_ptr;
   U8BIT db_filename[DB_FILENAME_LEN];
   BOOLEAN retval;

   FUNCTION_START(STB_PVRDestroyRecording);

   STB_PVR_PRINT(("STB_PVRDestroyRecording(0x%lx)", handle));

   retval = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         if (strlen((char *)rec_ptr->basename) > 0)
         {
            strncpy((char *)db_filename, (char *)rec_ptr->basename, DB_FILENAME_LEN);
            strncat((char *)db_filename, DB_FILE_EXTENSION, DB_FILENAME_LEN - strlen((char *)rec_ptr->basename));

            STB_DSKDeleteFile(rec_ptr->disk_id, db_filename);

            RemoveAllBookmarks(rec_ptr->disk_id, rec_ptr->basename);

            /* Delete all files associated with the recording */
            STB_PVRDeleteRecording(rec_ptr->disk_id, rec_ptr->basename);
         }

         RemoveRecordingFromList(rec_ptr);

         retval = TRUE;
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRDestroyRecording);

   return(retval);
}

/*!**************************************************************************
 * @brief   Save all info on the given recording. The filename and disk should already
 *          have been setup, either when the recording is first created or when it's
 *          read from a disk.
 * @param   handle - recording handle
 ****************************************************************************/
void STB_PVRSaveRecording(U32BIT handle)
{
   S_RECORDING *rec_ptr;

   FUNCTION_START(STB_PVRSaveRecording);

   STB_PVR_PRINT(("STB_PVRSaveRecording(handle=0x%lx)", handle));

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         if (strlen((char *)rec_ptr->basename) > 0)
         {
            WriteODBFile(rec_ptr->disk_id, rec_ptr->basename, TRUE, &rec_ptr->rec_info);
         }
      }
#ifdef STB_DEBUG
      else
      {
         STB_PVR_PRINT(("STB_PVRSaveRecording(0x%08lx): Failed to find recording", handle));
      }
#endif

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRSaveRecording);
}

/*!**************************************************************************
 * @brief   Allocates and returns an array of all the recording handles
 * @param   handle_array - pointer to array of handles, must be freed using
 *                         STB_PVRReleaseRecordingHandles
 * @return  Number of handles returned in the array
 ****************************************************************************/
U16BIT STB_PVRGetRecordingHandles(U32BIT **handle_array)
{
   S_RECORDING *rec_ptr;
   U16BIT num_recordings;
   U16BIT index;

   FUNCTION_START(STB_PVRGetRecordingHandles);

   num_recordings = 0;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Count the number of recordings available */
      for (num_recordings = 0, rec_ptr = rec_list; rec_ptr != NULL;
           num_recordings++, rec_ptr = rec_ptr->next)
         ;

      if (num_recordings > 0)
      {
         /* Allocate an array to be returned */
         *handle_array = (U32BIT *)STB_GetMemory(num_recordings * sizeof(U32BIT));
         if (*handle_array != NULL)
         {
            for (index = 0, rec_ptr = rec_list; rec_ptr != NULL; index++, rec_ptr = rec_ptr->next)
            {
               (*handle_array)[index] = rec_ptr->handle;
            }
         }
         else
         {
            num_recordings = 0;
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRGetRecordingHandles);

   return(num_recordings);
}

/*!**************************************************************************
 * @brief   Frees the given array of handles allocated by the preceding function
 * @param   handle_array - array to be freed
 ****************************************************************************/
void STB_PVRReleaseRecordingHandles(U32BIT *handle_array)
{
   FUNCTION_START(STB_PVRReleaseRecordingHandles);

   if (handle_array != NULL)
   {
      STB_FreeMemory(handle_array);
   }

   FUNCTION_FINISH(STB_PVRReleaseRecordingHandles);
}

/*!**************************************************************************
 * @brief   Returns the handle of the recording with the given programme CRID
 * @param   prog_crid - CRID to be searched for
 * @param   handle - handle of recording found, returned
 * @return  TRUE if a recording is found, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRFindRecordingFromCrid(U8BIT *prog_crid, U32BIT *handle)
{
   S_RECORDING *rec_ptr;
   BOOLEAN found;

   FUNCTION_START(STB_PVRFindRecordingFromCrid);

   found = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      for (rec_ptr = rec_list; !found && (rec_ptr != NULL); rec_ptr = rec_ptr->next)
      {
         if (STB_CompareStringsIgnoreCase(rec_ptr->rec_info.prog_crid, prog_crid) == 0)
         {
            *handle = rec_ptr->handle;
            found = TRUE;
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRFindRecordingFromCrid);

   return(found);
}

/*!**************************************************************************
 * @brief   Returns the handle of a split event recording that follows on from
 *          curr_handle
 * @param   curr_handle - handle of the recording
 * @param   next_handle - handle of recording found, returned
 * @return  TRUE if a recording is found, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRFindNextSplitRecording(U32BIT curr_handle, U32BIT *next_handle)
{
   S_RECORDING *curr_rec;
   S_RECORDING *rec_ptr;
   U16BIT end_date;
   U8BIT end_hour, end_min, end_secs;
   U16BIT earliest_date;
   U8BIT earliest_hour, earliest_min;
   BOOLEAN found;

   FUNCTION_START(STB_PVRFindNextSplitRecording);

   found = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      curr_rec = GetRecording(curr_handle);
      if ((curr_rec != NULL) && (STB_GetNumBytesInString(curr_rec->rec_info.prog_crid) > 1))
      {
         /* Calculate the end date/time so that the earliest recording following this can be found */
         STB_GCCalculateDateTime(curr_rec->rec_info.rec_date, curr_rec->rec_info.rec_hour,
            curr_rec->rec_info.rec_min, curr_rec->rec_info.rec_sec, curr_rec->rec_info.len_hour,
            curr_rec->rec_info.len_min, curr_rec->rec_info.len_sec, &end_date, &end_hour, &end_min,
            &end_secs, CALC_ADD);

         /* Search through all the recordings looking for one that follows this one */
         for (rec_ptr = rec_list; !found && (rec_ptr != NULL); rec_ptr = rec_ptr->next)
         {
            if (rec_ptr->handle != curr_handle)
            {
               if (STB_CompareStringsIgnoreCase(curr_rec->rec_info.prog_crid, rec_ptr->rec_info.prog_crid) == 0)
               {
                  if (STB_GCCompareDateTime(end_date, end_hour, end_min, 0, rec_ptr->rec_info.rec_date,
                         rec_ptr->rec_info.rec_hour, rec_ptr->rec_info.rec_min, 0, COMP_2GE1))
                  {
                     /* This part of the split occurs after the end of the currently playing part */
                     if (found)
                     {
                        /* A later recording has already been found so check whether this part
                         * is earlier than the one already found */
                        if (STB_GCCompareDateTime(earliest_date, earliest_hour, earliest_min, 0,
                               rec_ptr->rec_info.rec_date, rec_ptr->rec_info.rec_hour,
                               rec_ptr->rec_info.rec_min, 0, COMP_2LT1))
                        {
                           /* This part is earlier than the previously found part */
                           earliest_date = rec_ptr->rec_info.rec_date;
                           earliest_hour = rec_ptr->rec_info.rec_hour;
                           earliest_min = rec_ptr->rec_info.rec_min;
                           *next_handle = rec_ptr->handle;
                        }
                     }
                     else
                     {
                        /* This is the first recording found that occurs after the current part */
                        earliest_date = rec_ptr->rec_info.rec_date;
                        earliest_hour = rec_ptr->rec_info.rec_hour;
                        earliest_min = rec_ptr->rec_info.rec_min;
                        *next_handle = rec_ptr->handle;

                        found = TRUE;
                     }
                  }
               }
            }
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRFindNextSplitRecording);

   return(found);
}

#if 0
/**
 *

 *
 * @brief   Import new recording if files exist.
 *

 *
 * @return   BOOLEAN - TRUE on success, else FALSE.
 *
 */
BOOLEAN STB_PVRImportRecording(void)
{
   BOOLEAN ret_val = FALSE;
   E_STB_DMX_DEMUX_SOURCE dmx_source;
   U8BIT dmx_param;
   U32BIT handle;
   BOOLEAN ts, video, audio;

   FUNCTION_START(STB_PVRImportRecording);

   STB_PVR_PRINT(("STB_PVRImportRecording"));

   if (initialised_flag == TRUE)
   {
      if (STB_PVRIsRecordStarted(0) == FALSE)
      {
         handle = STB_PVRGrabHandle("import");
         ts = STB_PVRTestDataImporter(0, handle, STREAM_TRANS);
         video = STB_PVRTestDataImporter(0, handle, STREAM_VIDEO);
         audio = STB_PVRTestDataImporter(0, handle, STREAM_AUDIO);

         if ((ts == TRUE) || (video == TRUE) || (audio == TRUE))
         {
            if (ts == TRUE)
            {
               STB_PVR_PRINT(("PVR - Start TS Import"));
               STB_DMXGetDemuxSource(0, &dmx_source, &dmx_param);
               STB_DMXSetDemuxSource(0, DMX_MEMORY, 0);
               STB_DMXChangeDecodePIDs(0, 0, 0x44, 0x45, 0, 0);
               STB_AVStartVideoDecoding(0);
               STB_AVStartAudioDecoding(0);
               STB_AVBlankVideo(0, FALSE);
               STB_PVRSetRecordStartMode(0, START_IMPORT_TS, 0);
               STB_PVRStartRecording(0, handle);
            }
            else
            {
               STB_PVR_PRINT(("PVR - Start PES Import"));
               STB_PVRSetRecordStartMode(0, START_IMPORT_PES, 0);
               STB_PVRStartRecording(0, handle);
            }

            do
            {
               STB_OSTaskDelay(100);
               STB_PVRRecordEnabled(0, &video, &audio);
            }
            while ((video == TRUE) || (audio == TRUE));

            if (ts == TRUE)
            {
               STB_AVBlankVideo(0, TRUE);
               STB_AVStopVideoDecoding(0);
               STB_AVStopAudioDecoding(0);
               STB_DMXSetDemuxSource(0, dmx_source, dmx_param);
            }
            STB_PVR_PRINT(("PVR - Finished Import"));
            STB_PVRStopRecording(0);
            STB_PVRSetRecordStartMode(0, START_RUNNING, 0);
            ret_val = TRUE;
         }
         else
         {
            STB_PVRReleaseHandle(handle);
            STB_PVR_PRINT(("PVR - No Import"));
         }
      }
   }
   FUNCTION_FINISH(STB_PVRImportRecording);

   return(ret_val);
}

#endif

#if 0
/**
 *

 *
 * @brief   Export existing recording for specified handle.
 *
 * @param   U32BIT handle - handle
 *
 * @return   BOOLEAN - TRUE on success, else FALSE.
 *
 */
BOOLEAN STB_PVRExportRecording(U32BIT handle)
{
   BOOLEAN ret_val = FALSE;
   USE_UNWANTED_PARAM(handle);
   BOOLEAN video, audio;

   FUNCTION_START(STB_PVRExportRecording);

   STB_PVR_PRINT(("STB_PVRExportRecording(%lx)", handle));

   if (initialised_flag == TRUE)
   {
      if (STB_PVRIsPlayStarted(0) == FALSE)
      {
         STB_PVR_PRINT(("PVR - Start Export"));
         STB_PVRSetPlayStartMode(0, START_EXPORT_PES);
         STB_PVRStartPlaying(0, handle);

         do
         {
            STB_OSTaskDelay(100);
            STB_PVRPlayEnabled(0, &video, &audio);
         }
         while ((video == TRUE) || (audio == TRUE));

         STB_PVR_PRINT(("PVR - Finished Export"));
         STB_PVRStopPlaying(0);
         STB_PVRSetPlayStartMode(0, START_RUNNING);
         ret_val = TRUE;
      }
   }

   FUNCTION_FINISH(STB_PVRExportRecording);

   return(ret_val);
}

#endif

/*!**************************************************************************
 * @brief   Returns the handle for the given recording index
 * @param   rec_index - recording index
 * @return  handle, or 0 if recording isn't found
 ****************************************************************************/
U32BIT STB_PVRGetHandleForRecordingIndex(U8BIT rec_index)
{
   U8BIT i;
   U32BIT handle;

   FUNCTION_START(STB_PVRGetHandleForRecordingIndex);

   handle = 0;

   for (i = 0; (i < num_paths) && (handle == 0); i++)
   {
      if (record_status[i].rec_index == rec_index)
      {
         handle = record_status[i].handle;
      }
   }

   FUNCTION_FINISH(STB_PVRGetHandleForRecordingIndex);

   return(handle);
}

/*!**************************************************************************
 * @brief   Returns the path for the given recording index
 * @param   rec_index - recording index
 * @return  path, or INVALID_RES_ID if recording isn't found
 ****************************************************************************/
U8BIT STB_PVRGetPathForRecordingIndex(U8BIT rec_index)
{
   U8BIT i;
   U8BIT path;

   FUNCTION_START(STB_PVRGetPathForRecordingIndex);

   path = INVALID_RES_ID;

   for (i = 0; (i < num_paths) && (path == INVALID_RES_ID); i++)
   {
      if (record_status[i].rec_index == rec_index)
      {
         path = i;
      }
   }

   FUNCTION_FINISH(STB_PVRGetPathForRecordingIndex);

   return(path);
}

/*!**************************************************************************
 * @brief   Sets the name for a recording
 * @param   handle - recording handle
 * @param   name - name to be set
 ****************************************************************************/
void STB_PVRRecordingSetName(U32BIT handle, U8BIT *name)
{
   S_RECORDING *rec_ptr;
   U32BIT num_bytes;

   FUNCTION_START(STB_PVRRecordingSetName);

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         if (strlen((char *)rec_ptr->basename) > 0)
         {
            num_bytes = STB_GetNumBytesInString(name);

            /* Only update the name if it's short enough */
            if (num_bytes <= sizeof(rec_ptr->rec_info.name))
            {
               memcpy(rec_ptr->rec_info.name, name, num_bytes);
            }

            /* Update the file */
            WriteODBFile(rec_ptr->disk_id, rec_ptr->basename, FALSE, &rec_ptr->rec_info);
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingSetName);
}

/*!**************************************************************************
 * @brief   Gets the name for a recording
 * @param   handle - recording handle
 * @return  Pointer to the name
 ****************************************************************************/
U8BIT* STB_PVRRecordingGetName(U32BIT handle)
{
   S_RECORDING *rec_ptr;
   U8BIT *name_ptr;

   FUNCTION_START(STB_PVRRecordingGetName);

   name_ptr = NULL;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         name_ptr = &rec_ptr->rec_info.name[0];
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingGetName);

   return(name_ptr);
}

/*!**************************************************************************
 * @brief   Gets the date and time for a recording
 * @param   handle - recording handle
 * @param   date - returned date value
 * @param   hours - returned time in hours
 * @param   mins - returned time in minutes
 * @param   secs - returned time in seconds
 * @return  TRUE if the recording is found
 ****************************************************************************/
BOOLEAN STB_PVRRecordingGetDateTime(U32BIT handle, U16BIT *date, U8BIT *hours, U8BIT *mins, U8BIT *secs)
{
   S_RECORDING *rec_ptr;
   BOOLEAN retval;

   FUNCTION_START(STB_PVRRecordingGetDateTime);

   retval = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         *date = rec_ptr->rec_info.rec_date;
         *hours = rec_ptr->rec_info.rec_hour;
         *mins = rec_ptr->rec_info.rec_min;
         *secs = rec_ptr->rec_info.rec_sec;
         retval = TRUE;
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingGetDateTime);

   return(retval);
}

/*!**************************************************************************
 * @brief   Returns the length in time and size in KB of the recording with the given handle
 * @param   handle - handle of recording being queried
 * @param   length_hours - length of the recording in hours, returned
 * @param   length_mins - length of the recording in minutes, returned
 * @param   length_secs - length of the recording in seconds, returned
 * @param   rec_size_kb - size of the recording in KB, returned
 * @return  TRUE if the length and size are being returned, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRRecordingGetLength(U32BIT handle, U8BIT *length_hours, U8BIT *length_mins,
   U8BIT *length_secs, U32BIT *rec_size_kb)
{
   S_RECORDING *rec_ptr;
   BOOLEAN retval;
   U32BIT rec_time;

   FUNCTION_START(STB_PVRRecordingGetLength);

   retval = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         if (rec_ptr->recording)
         {
            /* Recording is still taking place so the time needs to be calculated */
            rec_time = GetCurrentRecordTime(handle);
            *length_hours = rec_time / 3600;
            *length_mins = (rec_time % 3600) / 60;
            *length_secs = (rec_time % 3600) % 60;
            retval = TRUE;
         }
         else
         {
            *length_hours = rec_ptr->rec_info.len_hour;
            *length_mins = rec_ptr->rec_info.len_min;
            *length_secs = rec_ptr->rec_info.len_sec;
            retval = TRUE;
         }

         if (retval)
         {
            /* Get the size of the recording */
            if (!STB_PVRGetRecordingSize(rec_ptr->disk_id, rec_ptr->basename, rec_size_kb))
            {
               retval = FALSE;
            }
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingGetLength);

   return(retval);
}

/*!**************************************************************************
 * @brief   Gets the disk id for a recording
 * @param   handle - recording handle
 * @return  Disk id
 ****************************************************************************/
U16BIT STB_PVRRecordingGetDiskId(U32BIT handle)
{
   S_RECORDING *rec_ptr;
   U16BIT disk_id;

   FUNCTION_START(STB_PVRRecordingGetDiskId);

   disk_id = INVALID_DISK_ID;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         disk_id = rec_ptr->disk_id;
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingGetDiskId);

   return(disk_id);
}

/*!**************************************************************************
 * @brief   Sets the series/recommendation CRID for a recording and saves the database file
 * @param   handle - recording handle
 * @param   crid - crid string
 ****************************************************************************/
void STB_PVRRecordingSetOtherCrid(U32BIT handle, U8BIT *crid)
{
   S_RECORDING *rec_ptr;
   U32BIT name_len;

   FUNCTION_START(STB_PVRRecordingSetOtherCrid);

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         if (crid != NULL)
         {
            /* Can only save the crid if there's enough space for it */
            name_len = STB_GetNumBytesInString(crid);
            if (name_len <= sizeof(rec_ptr->rec_info.other_crid))
            {
               memcpy(rec_ptr->rec_info.other_crid, crid, name_len);
            }
            else
            {
               rec_ptr->rec_info.other_crid[0] = '\0';
            }
         }
         else
         {
            rec_ptr->rec_info.other_crid[0] = '\0';
         }

         if (strlen((char *)rec_ptr->basename) > 0)
         {
            WriteODBFile(rec_ptr->disk_id, rec_ptr->basename, FALSE, &rec_ptr->rec_info);
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingSetOtherCrid);
}

/*!**************************************************************************
 * @brief   Gets the series/recommendation CRID for a given recording
 * @param   crid - array into which the crid will be copied
 * @param   name_len - size of the crid array
 * @return  TRUE if the recording is found and the crid copied
 ****************************************************************************/
BOOLEAN STB_PVRRecordingGetOtherCrid(U32BIT handle, U8BIT *crid, U16BIT name_len)
{
   S_RECORDING *rec_ptr;
   U32BIT len;
   BOOLEAN result;

   FUNCTION_START(STB_PVRRecordingGetOtherCrid);

   result = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         /* The string can only be returned if it will fully fit in the buffer provided */
         len = STB_GetNumBytesInString(rec_ptr->rec_info.other_crid);
         if ((len > 0) && (len < name_len))
         {
            memcpy(crid, rec_ptr->rec_info.other_crid, len);
            result = TRUE;
         }
         else
         {
            crid[0] = '\0';
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingGetOtherCrid);

   return(result);
}

/**
 * @brief   Sets the additional info string for a recording
 * @param   handle recording handle
 * @param   additional_info additional info to be set
 */
void STB_PVRRecordingSetAdditionalInfo(U32BIT handle, U8BIT *additional_info)
{
   S_RECORDING *rec_ptr;
   U32BIT num_bytes;

   FUNCTION_START(STB_PVRRecordingSetAdditionalInfo);

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         if (strlen((char *)rec_ptr->basename) > 0)
         {
            num_bytes = STB_GetNumBytesInString(additional_info);

            if (num_bytes <= STB_PVR_ADDITIONAL_INFO_LEN)
            {
               memcpy(rec_ptr->rec_info.additional_info, additional_info, num_bytes);
            }
            else
            {
               memcpy(rec_ptr->rec_info.additional_info, additional_info, STB_PVR_ADDITIONAL_INFO_LEN);
            }

            /* Update the file */
            WriteODBFile(rec_ptr->disk_id, rec_ptr->basename, FALSE, &rec_ptr->rec_info);
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingSetAdditionalInfo);
}

/**
 * @brief   Gets the additional info string pointer for a recording
 * @param   handle recording handle
 * @return  Pointer to the additional info
 */
U8BIT* STB_PVRRecordingGetAdditionalInfo(U32BIT handle)
{
   S_RECORDING *rec_ptr;
   U8BIT *name_ptr;

   FUNCTION_START(STB_PVRRecordingGetAdditionalInfo);

   name_ptr = NULL;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         name_ptr = &rec_ptr->rec_info.additional_info[0];
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingGetAdditionalInfo);

   return(name_ptr);
}

/**
 * @brief   Sets the parental rating age for the specified recording
 * @param   handle recording handle
 * @param   parental_rating parental rating age
 */
void STB_PVRRecordingSetParentalRatingAge(U32BIT handle, U32BIT parental_rating)
{
   S_RECORDING *rec_ptr;

   FUNCTION_START(STB_PVRRecordingSetParentalRating);

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         rec_ptr->rec_info.parental_rating = parental_rating;

         /* Update the file */
         WriteODBFile(rec_ptr->disk_id, rec_ptr->basename, FALSE, &rec_ptr->rec_info);
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingSetParentalRating);
}

/**
 * @brief   Returns the parental rating age for the specified recording as set by
 *          STB_PVRRecordingSetParentalRatingAge
 * @param   handle recording handle
 * @return  Parental rating age
 */
U32BIT STB_PVRRecordingGetParentalRating(U32BIT handle)
{
   S_RECORDING *rec_ptr;
   U32BIT parental_rating = 0;

   FUNCTION_START(STB_PVRRecordingGetParentalRating);

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         parental_rating = rec_ptr->rec_info.parental_rating;
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingGetParentalRating);

   return parental_rating;
}

/**
 * @brief   Sets the start padding value for the specified recording
 * @param   handle recording handle
 * @param   start_padding start padding value
 */
void STB_PVRRecordingSetStartPadding(U32BIT handle, S32BIT start_padding)
{
   S_RECORDING *rec_ptr;

   FUNCTION_START(STB_PVRRecordingSetStartPadding);

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         rec_ptr->rec_info.start_padding = start_padding;

         /* Update the file */
         WriteODBFile(rec_ptr->disk_id, rec_ptr->basename, FALSE, &rec_ptr->rec_info);
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingSetStartPadding);
}

/**
 * @brief   Gets the start padding value for the specified recording as set by
 *          STB_PVRRecordingSetStartPadding
 * @param   handle recording handle
 * @return  Start padding value
 */
S32BIT STB_PVRRecordingGetStartPadding(U32BIT handle)
{
   S_RECORDING *rec_ptr;
   S32BIT start_padding = 0;

   FUNCTION_START(STB_PVRRecordingGetStartPadding);

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         start_padding = rec_ptr->rec_info.start_padding;
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingGetStartPadding);

   return start_padding;
}

/**
 * @brief   Sets the end padding value for the specified recording
 * @param   handle recording handle
 * @param   start_padding end padding value
 */
void STB_PVRRecordingSetEndPadding(U32BIT handle, S32BIT end_padding)
{
   S_RECORDING *rec_ptr;

   FUNCTION_START(STB_PVRRecordingSetEndPadding);

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         rec_ptr->rec_info.end_padding = end_padding;

         /* Update the file */
         WriteODBFile(rec_ptr->disk_id, rec_ptr->basename, FALSE, &rec_ptr->rec_info);
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingSetEndPadding);
}

/**
 * @brief   Gets the end padding value for the specified recording as set by
 *          STB_PVRRecordingSetEndPadding
 * @param   handle recording handle
 * @return  End padding value
 */
S32BIT STB_PVRRecordingGetEndPadding(U32BIT handle)
{
   S_RECORDING *rec_ptr;
   S32BIT end_padding = 0;

   FUNCTION_START(STB_PVRRecordingGetEndPadding);

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         end_padding = rec_ptr->rec_info.end_padding;
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingGetEndPadding);

   return end_padding;
}

/*!**************************************************************************
 * @brief   Sets the service name for a recording and saves the database file
 * @param   handle - recording handle
 * @param   service_name - service name to be set
 ****************************************************************************/
void STB_PVRRecordingSetServiceName(U32BIT handle, U8BIT *service_name)
{
   S_RECORDING *rec_ptr;
   U32BIT name_len;

   FUNCTION_START(STB_PVRRecordingSetServiceName);

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if ((rec_ptr != NULL) && (service_name != NULL))
      {
         if (strlen((char *)rec_ptr->basename) > 0)
         {
            name_len = STB_GetNumBytesInString(service_name);
            if (name_len > 0)
            {
               WriteExtendedInfo(rec_ptr->disk_id, rec_ptr->basename, EXT_INFO_SERVICE_NAME,
                  name_len, service_name);
            }
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingSetServiceName);
}

/*!**************************************************************************
 * @brief   Gets the service name for a given recording
 * @param   service_name - array into which the name will be copied
 * @param   name_len - size of the name array
 * @return  TRUE if the recording is found and it's database file is successfully opened
 ****************************************************************************/
BOOLEAN STB_PVRRecordingGetServiceName(U32BIT handle, U8BIT *service_name, U16BIT name_len)
{
   S_RECORDING *rec_ptr;
   BOOLEAN result;

   FUNCTION_START(STB_PVRRecordingGetServiceName);

   result = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         result = ReadExtendedInfo(rec_ptr->disk_id, rec_ptr->basename, EXT_INFO_SERVICE_NAME,
               service_name, name_len);
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingGetServiceName);

   return(result);
}

/*!**************************************************************************
 * @brief   Sets the short description for a recorded programme
 * @param   handle - recording handle
 * @param   description - text string
 ****************************************************************************/
void STB_PVRRecordingSetDescription(U32BIT handle, U8BIT *description)
{
   S_RECORDING *rec_ptr;
   U16BIT desc_len;

   FUNCTION_START(STB_PVRRecordingSetDescription);

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if ((rec_ptr != NULL) && (description != NULL))
      {
         if (strlen((char *)rec_ptr->basename) > 0)
         {
            desc_len = STB_GetNumBytesInString(description);
            if (desc_len > 0)
            {
               WriteExtendedInfo(rec_ptr->disk_id, rec_ptr->basename, EXT_INFO_SHORT_DESC,
                  desc_len, description);
            }
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingSetDescription);
}

/*!**************************************************************************
 * @brief   Gets the short description for a given recording
 * @param   handle - recording handle
 * @param   description - array into which the description is copied
 * @param   desc_len - size of the description array
 ****************************************************************************/
BOOLEAN STB_PVRRecordingGetDescription(U32BIT handle, U8BIT *description, U16BIT desc_len)
{
   S_RECORDING *rec_ptr;
   BOOLEAN result;

   FUNCTION_START(STB_PVRRecordingGetDescription);

   result = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         result = ReadExtendedInfo(rec_ptr->disk_id, rec_ptr->basename, EXT_INFO_SHORT_DESC,
               description, desc_len);
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingGetDescription);

   return(result);
}

/*!**************************************************************************
 * @brief   Gets the length of the short description for a given recording
 * @param   handle - recording handle
 * @return  length of the short description text, or 0 if none available
 ****************************************************************************/
U16BIT STB_PVRRecordingGetDescriptionLen(U32BIT handle)
{
   S_RECORDING *rec_ptr;
   U32BIT length;

   FUNCTION_START(STB_PVRRecordingGetDescriptionLen);

   length = 0;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         length = GetExtendedInfoSize(rec_ptr->disk_id, rec_ptr->basename, EXT_INFO_SHORT_DESC);
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingGetDescriptionLen);

   return((U16BIT)length);
}

/*!**************************************************************************
 * @brief   Sets the extended description for a recorded programme
 * @param   handle - recording handle
 * @param   description - text string
 ****************************************************************************/
void STB_PVRRecordingSetExtendedDescription(U32BIT handle, U8BIT *description)
{
   S_RECORDING *rec_ptr;
   U16BIT desc_len;

   FUNCTION_START(STB_PVRRecordingSetExtendedDescription);

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if ((rec_ptr != NULL) && (description != NULL))
      {
         if (strlen((char *)rec_ptr->basename) > 0)
         {
            desc_len = STB_GetNumBytesInString(description);
            if (desc_len > 0)
            {
               WriteExtendedInfo(rec_ptr->disk_id, rec_ptr->basename, EXT_INFO_EXTENDED_DESC,
                  desc_len, description);
            }
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingSetExtendedDescription);
}

/*!**************************************************************************
 * @brief   Gets the extended description for a given recording
 * @param   handle - recording handle
 * @param   description - array into which the description is copied
 * @param   desc_len - size of the description array
 ****************************************************************************/
BOOLEAN STB_PVRRecordingGetExtendedDescription(U32BIT handle, U8BIT *description, U16BIT desc_len)
{
   S_RECORDING *rec_ptr;
   BOOLEAN result;

   FUNCTION_START(STB_PVRRecordingGetExtendedDescription);

   result = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         result = ReadExtendedInfo(rec_ptr->disk_id, rec_ptr->basename, EXT_INFO_EXTENDED_DESC,
               description, desc_len);
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingGetExtendedDescription);

   return(result);
}

/*!**************************************************************************
 * @brief   Gets the length of the extended description for a given recording
 * @param   handle - recording handle
 * @return  length of the extended description text, or 0 if none available
 ****************************************************************************/
U16BIT STB_PVRRecordingGetExtendedDescriptionLen(U32BIT handle)
{
   S_RECORDING *rec_ptr;
   U32BIT length;

   FUNCTION_START(STB_PVRRecordingGetExtendedDescriptionLen);

   length = 0;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         length = GetExtendedInfoSize(rec_ptr->disk_id, rec_ptr->basename, EXT_INFO_EXTENDED_DESC);
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingGetExtendedDescriptionLen);

   return((U16BIT)length);
}

/*!**************************************************************************
 * @brief   Sets the programme CRID for a recording and saves the database file
 * @param   handle - recording handle
 * @param   crid - programme crid string
 ****************************************************************************/
void STB_PVRRecordingSetCrid(U32BIT handle, U8BIT *crid)
{
   S_RECORDING *rec_ptr;
   U32BIT name_len;

   FUNCTION_START(STB_PVRRecordingSetCrid);

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         if (crid != NULL)
         {
            /* Can only save the crid if there's enough space for it */
            name_len = STB_GetNumBytesInString(crid);
            if (name_len <= sizeof(rec_ptr->rec_info.prog_crid))
            {
               memcpy(rec_ptr->rec_info.prog_crid, crid, name_len);
            }
            else
            {
               rec_ptr->rec_info.prog_crid[0] = '\0';
            }
         }
         else
         {
            rec_ptr->rec_info.prog_crid[0] = '\0';
         }

         if (strlen((char *)rec_ptr->basename) > 0)
         {
            WriteODBFile(rec_ptr->disk_id, rec_ptr->basename, FALSE, &rec_ptr->rec_info);
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingSetCrid);
}

/*!**************************************************************************
 * @brief   Gets the programme CRID for a given recording
 * @param   crid - array into which the crid will be copied
 * @param   name_len - size of the crid array
 * @return  TRUE if the recording is found and the crid copied
 ****************************************************************************/
BOOLEAN STB_PVRRecordingGetCrid(U32BIT handle, U8BIT *crid, U16BIT name_len)
{
   S_RECORDING *rec_ptr;
   U32BIT len;
   BOOLEAN result;

   FUNCTION_START(STB_PVRRecordingGetCrid);

   result = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         /* The string can only be returned if it will fully fit in the buffer provided */
         len = STB_GetNumBytesInString(rec_ptr->rec_info.prog_crid);
         if ((len > 0) && (len < name_len))
         {
            memcpy(crid, rec_ptr->rec_info.prog_crid, len);
            result = TRUE;
         }
         else
         {
            crid[0] = '\0';
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingGetCrid);

   return(result);
}

/*!**************************************************************************
 * @brief   Returns whether the recording with the given handle is currently being recorded
 * @param   handle - recording being queried
 * @return  TRUE if recording, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRIsBeingRecorded(U32BIT handle)
{
   S_RECORDING *rec_ptr;
   BOOLEAN is_recording;

   FUNCTION_START(STB_PVRIsBeingRecorded);

   is_recording = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         is_recording = rec_ptr->recording;
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRIsBeingRecorded);

   return(is_recording);
}

/*!**************************************************************************
 * @brief   Sets a recording to a series recording.
 * @param   handle - recording handle
 * @param   crid - crid string
 ****************************************************************************/
void STB_PVRRecordingSetSeries(U32BIT handle)
{
   S_RECORDING *rec_ptr;

   FUNCTION_START(STB_PVRRecordingSetSeries);

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         rec_ptr->rec_info.flags |= REC_SERIES;

         if (strlen((char *)rec_ptr->basename) > 0)
         {
            WriteODBFile(rec_ptr->disk_id, rec_ptr->basename, FALSE, &rec_ptr->rec_info);
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingSetSeries);
}

/*!**************************************************************************
 * @brief   Sets a recording to a recommendation recording.
 * @param   handle - recording handle
 * @param   crid - crid string
 ****************************************************************************/
void STB_PVRRecordingSetRecommendation(U32BIT handle)
{
   S_RECORDING *rec_ptr;

   FUNCTION_START(STB_PVRRecordingSetRecommendation);

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         rec_ptr->rec_info.flags |= REC_RECOMMENDATION;

         if (strlen((char *)rec_ptr->basename) > 0)
         {
            WriteODBFile(rec_ptr->disk_id, rec_ptr->basename, FALSE, &rec_ptr->rec_info);
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingSetRecommendation);
}

/*!**************************************************************************
 * @brief   Returns whether the given recording is a series
 * @param   handle - recording handle
 * @return  TRUE if recording is part of a series, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRRecordingGetSeries(U32BIT handle)
{
   S_RECORDING *rec_ptr;
   BOOLEAN series;

   FUNCTION_START(STB_PVRRecordingGetSeries);

   series = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         if ((rec_ptr->rec_info.flags & REC_SERIES) != 0)
         {
            series = TRUE;
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingGetSeries);

   return(series);
}

/*!**************************************************************************
 * @brief   Returns whether the given recording is a recommendation
 * @param   handle - recording handle
 * @return  TRUE if recording is a recommendation, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRRecordingGetRecommendation(U32BIT handle)
{
   S_RECORDING *rec_ptr;
   BOOLEAN recommendation;

   FUNCTION_START(STB_PVRRecordingGetRecommendation);

   recommendation = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         if ((rec_ptr->rec_info.flags & REC_RECOMMENDATION) != 0)
         {
            recommendation = TRUE;
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingGetRecommendation);

   return(recommendation);
}

/*!**************************************************************************
 * @brief   Sets the locked state of a recording
 * @param   handle - recording handle
 * @param   state - TRUE for locked, FALSE for unlocked
 * @return  TRUE if recording is valid, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRRecordingSetLocked(U32BIT handle, BOOLEAN state)
{
   S_RECORDING *rec_ptr;
   BOOLEAN retval;

   FUNCTION_START(STB_PVRRecordingSetLocked);

   retval = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         if (state)
         {
            rec_ptr->rec_info.flags |= REC_LOCKED;
         }
         else
         {
            rec_ptr->rec_info.flags &= ~REC_LOCKED;
         }

         if (strlen((char *)rec_ptr->basename) > 0)
         {
            WriteODBFile(rec_ptr->disk_id, rec_ptr->basename, FALSE, &rec_ptr->rec_info);
         }

         retval = TRUE;
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingSetLocked);

   return(retval);
}

/*!**************************************************************************
 * @brief   Returns whether the given recording is locked
 * @param   handle - recording handle
 * @return  TRUE if recording is locked, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRRecordingGetLocked(U32BIT handle)
{
   S_RECORDING *rec_ptr;
   BOOLEAN locked;

   FUNCTION_START(STB_PVRRecordingGetLocked);

   locked = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         if ((rec_ptr->rec_info.flags & REC_LOCKED) != 0)
         {
            locked = TRUE;
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingGetLocked);

   return(locked);
}

/*!**************************************************************************
 * @brief   Sets the selected state of a recording
 * @param   handle - recording handle
 * @param   state - TRUE for selected, FALSE for unselected
 * @return  TRUE if recording is valid, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRRecordingSetSelected(U32BIT handle, BOOLEAN state)
{
   S_RECORDING *rec_ptr;
   BOOLEAN retval;

   FUNCTION_START(STB_PVRRecordingSetSelected);

   retval = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         rec_ptr->selected = state;
         retval = TRUE;
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingSetSelected);

   return(retval);
}

/*!**************************************************************************
 * @brief   Returns whether the given recording is selected
 * @param   handle - recording handle
 * @return  TRUE if recording is selected, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRRecordingGetSelected(U32BIT handle)
{
   S_RECORDING *rec_ptr;
   BOOLEAN selected;

   FUNCTION_START(STB_PVRRecordingGetSelected);

   selected = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         selected = rec_ptr->selected;
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingGetSelected);

   return(selected);
}

/*!**************************************************************************
 * @brief   Returns whether the given recording is encrypted
 * @param   handle - recording handle
 * @return  TRUE if recording is encrypted, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRRecordingIsEncrypted(U32BIT handle)
{
   S_RECORDING *rec_ptr;
   BOOLEAN encrypted;

   FUNCTION_START(STB_PVRRecordingIsEncrypted);

   encrypted = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         if ((rec_ptr->rec_info.flags & REC_ENCRYPTED) != 0)
         {
            encrypted = TRUE;
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingIsEncrypted);

   return(encrypted);
}

/*!**************************************************************************
 * @brief   Sets the parental lock state of a recording
 * @param   handle - recording handle
 * @param   state - TRUE for locked, FALSE for unlocked
 * @return  TRUE if recording is valid, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRRecordingSetParentalLock(U32BIT handle, BOOLEAN state)
{
   S_RECORDING *rec_ptr;
   BOOLEAN retval;

   FUNCTION_START(STB_PVRRecordingSetParentalLock);

   retval = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         if (state)
         {
            rec_ptr->rec_info.flags |= REC_PARENTAL_LOCK;
         }
         else
         {
            rec_ptr->rec_info.flags &= ~REC_PARENTAL_LOCK;
         }

         if (strlen((char *)rec_ptr->basename) > 0)
         {
            WriteODBFile(rec_ptr->disk_id, rec_ptr->basename, FALSE, &rec_ptr->rec_info);
         }

         retval = TRUE;
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingSetParentalLock);

   return(retval);
}

/*!**************************************************************************
 * @brief   Returns the parental lock state of the given recording
 * @param   handle - recording handle
 * @return  TRUE if recording is locked, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRRecordingGetParentalLock(U32BIT handle)
{
   S_RECORDING *rec_ptr;
   BOOLEAN locked;

   FUNCTION_START(STB_PVRRecordingGetParentalLock);

   locked = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         if ((rec_ptr->rec_info.flags & REC_PARENTAL_LOCK) != 0)
         {
            locked = TRUE;
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingGetParentalLock);

   return(locked);
}

/*!**************************************************************************
 * @brief   Sets the guidance text of a recording
 * @param   handle - recording handle
 * @param   text - guidance text
 * @return  TRUE if recording is valid, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRRecordingSetGuidance(U32BIT handle, U8BIT *text)
{
   S_RECORDING *rec_ptr;
   U16BIT text_len;
   BOOLEAN retval;

   FUNCTION_START(STB_PVRRecordingSetGuidance);

   retval = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(handle);
      if ((rec_ptr != NULL) && (text != NULL))
      {
         if (strlen((char *)rec_ptr->basename) > 0)
         {
            text_len = STB_GetNumBytesInString(text);
            if (text_len > 0)
            {
               if (WriteExtendedInfo(rec_ptr->disk_id, rec_ptr->basename, EXT_INFO_GUIDANCE,
                      text_len, text))
               {
                  rec_ptr->rec_info.flags |= REC_HAS_GUIDANCE;
                  WriteODBFile(rec_ptr->disk_id, rec_ptr->basename, FALSE, &rec_ptr->rec_info);
               }
            }
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingSetGuidance);

   return(retval);
}

/*!**************************************************************************
 * @brief   Returns whether the given recording has guidance
 * @param   handle - recording handle
 * @return  TRUE if recording has guidance, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRRecordingHasGuidance(U32BIT handle)
{
   S_RECORDING *rec_ptr;
   BOOLEAN has_guidance;

   FUNCTION_START(STB_PVRRecordingHasGuidance);

   has_guidance = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         if ((rec_ptr->rec_info.flags & REC_HAS_GUIDANCE) != 0)
         {
            has_guidance = TRUE;
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingHasGuidance);

   return(has_guidance);
}

/*!**************************************************************************
 * @brief   Returns the whether a recording has guidance and the guidance text if available
 * @param   handle - recording handle
 * @param   text - array into which the guidance text is copied, can be NULL if text isn't required
 * @param   text_len - size of the array passed in, ignored if text is NULL
 * @return  TRUE if recording has guidance, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRRecordingGetGuidance(U32BIT handle, U8BIT *text, U16BIT text_len)
{
   S_RECORDING *rec_ptr;
   BOOLEAN result;

   FUNCTION_START(STB_PVRRecordingGetGuidance);

   result = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         /* Don't bother reading the text if guidance isn't available */
         if ((rec_ptr->rec_info.flags & REC_HAS_GUIDANCE) != 0)
         {
            result = ReadExtendedInfo(rec_ptr->disk_id, rec_ptr->basename, EXT_INFO_GUIDANCE,
                  text, text_len);
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingGetGuidance);

   return(result);
}

/*!**************************************************************************
 * @brief   Gets the length of the guidance text for a given recording
 * @param   handle - recording handle
 * @return  length of the guidance text, or 0 if none available
 ****************************************************************************/
U16BIT STB_PVRRecordingGetGuidanceLen(U32BIT handle)
{
   S_RECORDING *rec_ptr;
   U32BIT length;

   FUNCTION_START(STB_PVRRecordingGetGuidanceLen);

   length = 0;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         /* Don't bother reading the text if guidance isn't available */
         if ((rec_ptr->rec_info.flags & REC_HAS_GUIDANCE) != 0)
         {
            length = GetExtendedInfoSize(rec_ptr->disk_id, rec_ptr->basename, EXT_INFO_GUIDANCE);
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingGetGuidanceLen);

   return((U16BIT)length);
}

/**
 * @brief   Set the status to be saved with the recording
 * @param   handle recording handle
 * @param   status status of the recording
 * @return  TRUE if the status is successfully saved with the given recording, FALSE otherwise
 */
BOOLEAN STB_PVRRecordingSetStatus(U32BIT handle, E_PVR_RECORDING_STATUS status)
{
   BOOLEAN retval;
   S_RECORDING *rec_ptr;

   FUNCTION_START(STB_PVRRecordingSetStatus);

   retval = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         rec_ptr->rec_info.status = (U8BIT)status;

         retval = WriteODBFile(rec_ptr->disk_id, rec_ptr->basename, FALSE, &rec_ptr->rec_info);
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingSetStatus);

   return(retval);
}

/**
 * @brief   Returns the status saved with the given recording
 * @param   handle recording handle
 * @param   status pointer to recording status if read successfully
 * @return  TRUE if status is returned, FALSE otherwise
 */
BOOLEAN STB_PVRRecordingGetStatus(U32BIT handle, E_PVR_RECORDING_STATUS *status)
{
   BOOLEAN retval;
   S_RECORDING *rec_ptr;

   FUNCTION_START(STB_PVRRecordingGetStatus);

   retval = FALSE;

   if (initialised_flag && (status != NULL))
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         *status = (E_PVR_RECORDING_STATUS)rec_ptr->rec_info.status;
         retval = TRUE;
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingGetStatus);

   return(retval);
}

#ifdef COMMON_INTERFACE
/*!**************************************************************************
 * @brief   Saves the given CI+ CICAM id with a recording
 * @param   handle - recording handle
 * @param   cicam_id - an array of bytes defining the cicam id
 * @param   id_len - number of bytes in the array
 * @return  TRUE if value is saved, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRRecordingSetCicamId(U32BIT handle, U8BIT *cicam_id, U8BIT id_len)
{
   S_RECORDING *rec_ptr;
   BOOLEAN retval;

   FUNCTION_START(STB_PVRRecordingSetCicamId);

   retval = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         if (strlen((char *)rec_ptr->basename) > 0)
         {
            retval = WriteExtendedInfo(rec_ptr->disk_id, rec_ptr->basename,
                  EXT_INFO_CIPLUS_CICAM_ID, id_len, cicam_id);
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingSetCicamId);

   return(retval);
}

/*!**************************************************************************
 * @brief   Reads the CI+ CICAM id for the given recording
 * @param   handle - recording handle
 * @param   cicam_id - an array of bytes to read into
 * @param   id_len - number of bytes in the array
 * @return  TRUE if value is read, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRRecordingGetCicamId(U32BIT handle, U8BIT *cicam_id, U8BIT id_len)
{
   BOOLEAN retval;
   S_RECORDING *rec_ptr;

   FUNCTION_START(STB_PVRRecordingGetCicamId);

   retval = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         retval = ReadExtendedInfo(rec_ptr->disk_id, rec_ptr->basename, EXT_INFO_CIPLUS_CICAM_ID,
               cicam_id, id_len);
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingGetCicamId);

   return(retval);
}

/*!**************************************************************************
 * @brief   Stores the given CI+ URI with a recording, along with a timestamp
 * @param   handle - recording handle
 * @param   uri - usage rules
 * @return  TRUE if URI is stored, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRRecordingAddURI(U32BIT handle, U8BIT *uri)
{
   S_RECORDING *rec_ptr;
   BOOLEAN retval;
   U32BIT timestamp;
   U8BIT play_path;
   S_REC_CIPLUS_ITEM *item;
   S_REC_CIPLUS_ITEM *list_item;
   U8BIT audio_decoder;
   U8BIT video_decoder;
   S16BIT speed;

   FUNCTION_START(STB_PVRRecordingAddURI);

   retval = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         if (strlen((char *)rec_ptr->basename) > 0)
         {
            timestamp = GetCurrentRecordTime(handle);
            retval = WriteCIPlusInfo(rec_ptr->disk_id, rec_ptr->basename, EXT_INFO_CIPLUS_URI,
                  timestamp, CIP_URI_LEN, uri);

            DBG_CIP("(0x%08lx): Added URI @ %lu secs", handle, timestamp)

            /* If this recording is also being played, such as timeshift or chase playback, then
             * the playback status needs to be updated with this new URI so that it can be applied
             * at the appropriate time */
            for (play_path = 0; play_path < num_paths; play_path++)
            {
               if (play_status[play_path].handle == handle)
               {
                  STB_OSSemaphoreWait(play_status[play_path].play_sem);

                  /* Recording is being played, so add an item for this URI to the end of its list */
                  if ((item = (S_REC_CIPLUS_ITEM *)STB_GetMemory(sizeof(S_REC_CIPLUS_ITEM))) != NULL)
                  {
                     memset(item, 0, sizeof(S_REC_CIPLUS_ITEM));

                     item->item_type = EXT_INFO_CIPLUS_URI;
                     item->timestamp = timestamp;
                     memcpy(item->u.uri, uri, CIP_URI_LEN);

                     if (play_status[play_path].item_list == NULL)
                     {
                        /* First item in the list */
                        play_status[play_path].item_list = item;
                     }
                     else
                     {
                        /* Find the end of the list */
                        for (list_item = play_status[play_path].item_list; list_item->next != NULL;
                             list_item = list_item->next)
                           ;

                        list_item->next = item;
                        item->prev = list_item;
                     }

                     /* Playback will always be behind this item that has just been recorded,
                      * so re-evaluate the next item to be applied during playback */
                     audio_decoder = STB_DPGetPathAudioDecoder(play_path);
                     video_decoder = STB_DPGetPathVideoDecoder(play_path);
                     speed = STB_PVRGetPlaySpeed(audio_decoder, video_decoder);

                     FindNextCIPlusItem(play_path, speed);
                  }

                  STB_OSSemaphoreSignal(play_status[play_path].play_sem);
                  break;
               }
            }
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingAddURI);

   return(retval);
}

/*!**************************************************************************
 * @brief   Update during playback the CI+ URI that's stored with a recording
 * @param   handle - recording handle
 * @param   uri - usage rules
 * @return  TRUE if URI is stored, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRRecordingUpdateURI(U32BIT handle, U8BIT *uri)
{
   S_RECORDING *rec_ptr;
   U8BIT path;
   BOOLEAN retval;

   FUNCTION_START(STB_PVRRecordingUpdateURI);

   retval = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         if (strlen((char *)rec_ptr->basename) > 0)
         {
            /* Find the playback path */
            for (path = 0; (path < num_paths) && (play_status[path].handle != handle); path++)
               ;

            if (path < num_paths)
            {
               /* The URI item being updated should be the last one that was passed to the CAM */
               if (play_status[path].uri_item != NULL)
               {
                  /* Update this URI and save it back to the recording */
                  memcpy(play_status[path].uri_item->u.uri, uri, CIP_URI_LEN);

                  retval = UpdateCIPlusInfo(rec_ptr->disk_id, rec_ptr->basename, EXT_INFO_CIPLUS_URI,
                        play_status[path].uri_item->timestamp, CIP_URI_LEN, uri);

                  DBG_CIP("(0x%08lx): Updated URI @ %lu secs", handle,
                           play_status[path].uri_item->timestamp)
               }
               else
               {
                  /* No URI item saved with the recording, so can't be updated */
                  retval = TRUE;
               }
            }
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingUpdateURI);

   return(retval);
}

/*!**************************************************************************
 * @brief   Stores the given CI+ licence with the recording along with a timestamp
 * @param   handle - recording handle
 * @param   licence - licence data
 * @param   licence_len - number of bytes
 * @return  TRUE if licence is stored, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRRecordingAddLicence(U32BIT handle, U8BIT *licence, U16BIT licence_len)
{
   S_RECORDING *rec_ptr;
   BOOLEAN retval;
   U8BIT play_path;
   U32BIT timestamp;
   S_REC_CIPLUS_ITEM *item;
   S_REC_CIPLUS_ITEM *list_item;
   U8BIT audio_decoder;
   U8BIT video_decoder;
   S16BIT speed;

   FUNCTION_START(STB_PVRRecordingAddLicence);

   retval = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(handle);
      if ((rec_ptr != NULL) && (licence != NULL) && (licence_len > 0))
      {
         if (strlen((char *)rec_ptr->basename) > 0)
         {
            timestamp = GetCurrentRecordTime(handle);

            retval = WriteCIPlusInfo(rec_ptr->disk_id, rec_ptr->basename, EXT_INFO_CIPLUS_LICENCE,
                  timestamp, licence_len, licence);

            DBG_CIP("(0x%08lx): Added licence, len %u, @ %lu secs", handle,
                     licence_len, timestamp)

            /* If this recording is also being played, such as timeshift or chase playback, then
             * the playback status needs to be updated with this new licence so that it can be applied
             * at the appropriate time */
            for (play_path = 0; play_path < num_paths; play_path++)
            {
               if (play_status[play_path].handle == handle)
               {
                  STB_OSSemaphoreWait(play_status[play_path].play_sem);

                  /* Recording is being played, so add an item for this licence to the end of its list */
                  if ((item = (S_REC_CIPLUS_ITEM *)STB_GetMemory(sizeof(S_REC_CIPLUS_ITEM))) != NULL)
                  {
                     memset(item, 0, sizeof(S_REC_CIPLUS_ITEM));

                     item->item_type = EXT_INFO_CIPLUS_LICENCE;
                     item->timestamp = timestamp;
                     item->u.licence.used = FALSE;

                     if ((item->u.licence.data = STB_GetMemory(licence_len)) != NULL)
                     {
                        memcpy(item->u.licence.data, licence, licence_len);
                        item->u.licence.length = licence_len;
                     }

                     if (play_status[play_path].item_list == NULL)
                     {
                        /* First item in the list */
                        play_status[play_path].item_list = item;
                     }
                     else
                     {
                        /* Find the end of the list */
                        for (list_item = play_status[play_path].item_list; list_item->next != NULL;
                             list_item = list_item->next)
                           ;

                        list_item->next = item;
                        item->prev = list_item;
                     }

                     /* Playback will always be behind this item that has just been recorded,
                      * so re-evaluate the next item to be applied during playback */
                     audio_decoder = STB_DPGetPathAudioDecoder(play_path);
                     video_decoder = STB_DPGetPathVideoDecoder(play_path);
                     speed = STB_PVRGetPlaySpeed(audio_decoder, video_decoder);

                     FindNextCIPlusItem(play_path, speed);
                  }

                  STB_OSSemaphoreSignal(play_status[play_path].play_sem);
                  break;
               }
            }
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingAddLicence);

   return(retval);
}

/*!**************************************************************************
 * @brief   Update during playback the CI+ licence that's stored with a recording
 * @param   handle - recording handle
 * @param   licence - licence data
 * @param   licence_len - number of bytes
 * @return  TRUE if licence is stored, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRRecordingUpdateLicence(U32BIT handle, U8BIT *licence, U16BIT licence_len)
{
   S_RECORDING *rec_ptr;
   U8BIT path;
   BOOLEAN retval;

   FUNCTION_START(STB_PVRRecordingUpdateLicence);

   retval = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      /* Find the recording */
      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         if (strlen((char *)rec_ptr->basename) > 0)
         {
            /* Find the playback path */
            for (path = 0; (path < num_paths) && (play_status[path].handle != handle); path++)
               ;

            if (path < num_paths)
            {
               /* The licence item being updated should be the last one that was passed to the CAM */
               if (play_status[path].licence_item != NULL)
               {
                  /* Update this licence and save it back to the recording.
                   * Licences don't change in size during playback */
                  memcpy(play_status[path].licence_item->u.licence.data, licence, licence_len);

                  retval = UpdateCIPlusInfo(rec_ptr->disk_id, rec_ptr->basename, EXT_INFO_CIPLUS_LICENCE,
                        play_status[path].licence_item->timestamp, licence_len, licence);

                  DBG_CIP("(0x%08lx): Updated licence @ %lu secs", handle,
                           play_status[path].licence_item->timestamp)
               }
            }
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingUpdateLicence);

   return(retval);
}

/*!**************************************************************************
 * @brief   Stores a CI+ pin event with the recording along with a timestamp
 * @param   handle - recording handle
 * @param   age_rating - age rating introduced by the pin, 0 = no rating
 * @param   private_data - pin private data - this is a fixed size, defined by CI+
 * @param   date_code - UTC MJD date code when pin is to be applied
 * @param   hour - UTC hour when pin is to be applied
 * @param   min - UTC minute when pin is to be applied
 * @param   secs - UTC seconds when pin is to be applied
 * @return  TRUE if pin is stored, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRRecordingAddPin(U32BIT handle, U8BIT age_rating, U8BIT *private_data,
   U16BIT date_code, U8BIT hour, U8BIT min, U8BIT secs)
{
   S_RECORDING *rec_ptr;
   BOOLEAN retval;
   S_REC_CIPLUS_PIN pin_item;
   U32BIT timestamp;
   U16BIT gmt_date;
   U8BIT gmt_hour, gmt_min, gmt_secs;
   S32BIT time_diff;
   U8BIT play_path;
   S_REC_CIPLUS_ITEM *item;
   S_REC_CIPLUS_ITEM *list_item;
   U8BIT audio_decoder;
   U8BIT video_decoder;
   S16BIT speed;

   FUNCTION_START(STB_PVRRecordingAddPin);

   retval = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         if (strlen((char *)rec_ptr->basename) > 0)
         {
            pin_item.age_rating = age_rating;
            memcpy(pin_item.private_data, private_data, CIP_PIN_PRIVATE_DATA_SIZE);

            /* Convert the time included with the pin event to a timestamp relative to the
             * time the recording started */
            timestamp = GetCurrentRecordTime(handle);
            if (date_code != 0)
            {
               STB_GCGetGMTDateTime(&gmt_date, &gmt_hour, &gmt_min, &gmt_secs);

               if ((time_diff = STB_GCDateTimeDiff(date_code, hour, min, secs,
                          gmt_date, gmt_hour, gmt_min, gmt_secs)) < 0)
               {
                  if ((U32BIT)-time_diff > timestamp)
                  {
                     /* Pin is defined to be applied before the recording started,
                      * so change the time so it will start at the beginning */
                     time_diff = -(S32BIT)timestamp;
                  }
               }

               timestamp = (S32BIT)timestamp + time_diff;
            }

            retval = WriteCIPlusInfo(rec_ptr->disk_id, rec_ptr->basename, EXT_INFO_CIPLUS_PIN,
                  timestamp, sizeof(pin_item), (U8BIT *)&pin_item);

            DBG_CIP("(0x%08lx): Added pin rating %u, @ %lu secs", handle,
                     age_rating, timestamp)

            /* If this recording is also being played, such as timeshift or chase playback, then
             * the playback status needs to be updated with this new pin so that it can be applied
             * at the appropriate time */
            for (play_path = 0; play_path < num_paths; play_path++)
            {
               if (play_status[play_path].handle == handle)
               {
                  STB_OSSemaphoreWait(play_status[play_path].play_sem);

                  /* Recording is being played, so add an item for this pin to the end of its list */
                  if ((item = (S_REC_CIPLUS_ITEM *)STB_GetMemory(sizeof(S_REC_CIPLUS_ITEM))) != NULL)
                  {
                     memset(item, 0, sizeof(S_REC_CIPLUS_ITEM));

                     item->item_type = EXT_INFO_CIPLUS_PIN;
                     item->timestamp = timestamp;
                     item->u.pin.age_rating = age_rating;
                     memcpy(item->u.pin.private_data, private_data, CIP_PIN_PRIVATE_DATA_SIZE);

                     if (play_status[play_path].item_list == NULL)
                     {
                        /* First item in the list */
                        play_status[play_path].item_list = item;
                     }
                     else
                     {
                        /* Find the end of the list */
                        for (list_item = play_status[play_path].item_list; list_item->next != NULL;
                             list_item = list_item->next)
                           ;

                        list_item->next = item;
                        item->prev = list_item;
                     }

                     /* Playback will always be behind this item that has just been recorded,
                      * so re-evaluate the next item to be applied during playback */
                     audio_decoder = STB_DPGetPathAudioDecoder(play_path);
                     video_decoder = STB_DPGetPathVideoDecoder(play_path);
                     speed = STB_PVRGetPlaySpeed(audio_decoder, video_decoder);

                     FindNextCIPlusItem(play_path, speed);
                  }

                  STB_OSSemaphoreSignal(play_status[play_path].play_sem);
                  break;
               }
            }
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRRecordingAddPin);

   return(retval);
}

#endif /* COMMON_INTERFACE */

/*!**************************************************************************
 * @brief   Calculates the total disk space used in minutes
 * @param   disk_id - disk to be used
 * @return  Total size of all recordings in minutes
 ****************************************************************************/
U32BIT STB_PVRGetTimeOfAllRecordings(U16BIT disk_id)
{
   S_RECORDING *rec_ptr;
   U32BIT retval;

   FUNCTION_START(STB_PVRGetTimeOfAllRecordings);

   retval = 0;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      for (rec_ptr = rec_list; rec_ptr != NULL; rec_ptr = rec_ptr->next)
      {
         /* Only get info for recordings on the specified disk */
         if (rec_ptr->disk_id == disk_id)
         {
            if (rec_ptr->recording)
            {
               /* The final length of the recording won't be known yet, so use the time taken so far */
               retval += ((GetCurrentRecordTime(rec_ptr->handle) + 29) / 60);
            }
            else
            {
               retval += (rec_ptr->rec_info.len_hour * 60) + rec_ptr->rec_info.len_min;
            }
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRGetTimeOfAllRecordings);

   return(retval);
}

/*!**************************************************************************
 * @brief   Calculates the total disk space used in KB for all recordings
 * @param   disk_id - disk to be used
 * @return  Total size of all recordings in KB
 ****************************************************************************/
U32BIT STB_PVRGetSizeOfAllRecordings(U16BIT disk_id)
{
   S_RECORDING *rec_ptr;
   U32BIT rec_size;
   U32BIT total_size;

   FUNCTION_START(STB_PVRGetSizeOfAllRecordings);

   total_size = 0;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      for (rec_ptr = rec_list; rec_ptr != NULL; rec_ptr = rec_ptr->next)
      {
         /* Only get info for recordings on the specified disk */
         if (rec_ptr->disk_id == disk_id)
         {
            if (STB_PVRGetRecordingSize(disk_id, rec_ptr->basename, &rec_size))
            {
               total_size += rec_size;
            }
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRGetSizeOfAllRecordings);

   return(total_size);
}

/**
 *

 *
 * @brief   Start the PVR playing.
 *
 * @param   U8BIT path - player number
 * @param   U32BIT handle - handle to use for play file
 *
 * @return   TRUE is playback starts, FALSE otherwise
 *
 */
BOOLEAN STB_PVRStartPlaying(U8BIT path, U32BIT handle, BOOLEAN resume)
{
   S_RECORDING *rec_ptr;
   U8BIT audio_decoder, video_decoder;
   BOOLEAN retval;
   CREATE_LINK_LIST_HEADER(bookmarks);
   U32BIT position_in_seconds;
   BOOLEAN start_playback;
#ifdef COMMON_INTERFACE
   U8BIT cicam_id[8];
#endif
   S_BOOKMARK *b;

   FUNCTION_START(STB_PVRStartPlaying);

   retval = FALSE;

   STB_PVR_PRINT(("PVR - StartPlaying(%d): handle=0x%lx", path, handle));

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         audio_decoder = STB_DPGetPathAudioDecoder(path);
         video_decoder = STB_DPGetPathVideoDecoder(path);

         if (strlen((char *)rec_ptr->basename) > 0)
         {
            start_playback = TRUE;

#ifdef COMMON_INTERFACE
            if (ReadExtendedInfo(rec_ptr->disk_id, rec_ptr->basename, EXT_INFO_CIPLUS_CICAM_ID,
                   cicam_id, sizeof(cicam_id)))
            {
               /* This recording is protected by CI+, check that the CAM is present */
               if ((play_status[path].slot_id = STB_CiCcFindSlotForCicamId(cicam_id)) != INVALID_RES_ID)
               {
                  /* Read all URI, licence and PIN items so they can be applied
                   * at the appropriate times during playback */
                  STB_OSSemaphoreWait(play_status[path].play_sem);
                  ReadCIPlusItems(path, rec_ptr);
                  STB_OSSemaphoreSignal(play_status[path].play_sem);
               }
               else
               {
                  /* CAM isn't available so can't start playback */
                  start_playback = FALSE;
               }
            }
#endif

            if (start_playback)
            {
               play_status[path].handle = handle;
               position_in_seconds = 0;

               if (resume)
               {
                  /* Read the playback position boookmark */
                  if (GetBookmarks(rec_ptr->disk_id, rec_ptr->basename, &bookmarks,
                     BOOKMARK_PLAYBACK_POSITION, FALSE) > 0)
                  {
                     b = (S_BOOKMARK *)STB_LLGetFirstBlock(&bookmarks);
                     if (b != NULL)
                     {
                        position_in_seconds = b->time;
                     }
                     EmptyBookmarks(&bookmarks, TRUE);
                  }
                  STB_PVR_PRINT(("PVR - StartPlaying(%d): resuming from %d seconds", path, position_in_seconds));
               }

               if ((rec_ptr->rec_info.flags & REC_ENCRYPTED) != 0)
               {
                  /* Get the encryption key to be used */
                  if ((enc_dec_key != NULL) && (enc_dec_iv != NULL))
                  {
                     STB_PVRSetPlaybackDecryptionKey(audio_decoder, video_decoder, TRUE, enc_dec_key,
                        enc_dec_iv, enc_dec_key_len);
                  }
#ifdef STB_DEBUG
                  else
                  {
                     STB_PVR_PRINT(("STB_PVRStartPlaying(%u 0x%08lx): Playback failed, encryption key not available",
                                    path, handle));
                  }
#endif
               }
               else
               {
                  STB_PVRSetPlaybackDecryptionKey(audio_decoder, video_decoder, FALSE, NULL, NULL, 0);
               }

#ifdef COMMON_INTERFACE
               if ((play_status[path].slot_id != INVALID_RES_ID) && (play_status[path].item_list != NULL))
               {
                  STB_OSSemaphoreWait(play_status[path].play_sem);

                  /* Find the URI/licence/pin that should be applied at the start of playback */
                  ApplyCIPlusItems(path, rec_ptr, position_in_seconds, 100);

                  /* Find the next URI/licence/pin to be applied */
                  FindNextCIPlusItem(path, 100);

                  STB_OSSemaphoreSignal(play_status[path].play_sem);
               }
#endif

               if ((rec_ptr->rec_info.flags & REC_HAS_VIDEO) != 0)
               {
                  STB_PVRPlayHasVideo(audio_decoder, video_decoder, TRUE);
               }
               else
               {
                  STB_PVRPlayHasVideo(audio_decoder, video_decoder, FALSE);
               }

               if (STB_PVRPlayStart(rec_ptr->disk_id, audio_decoder, video_decoder,
                      STB_DPGetPathDemux(path), rec_ptr->basename))
               {
                  if (resume)
                  {
                     STB_PVRPlaySetPosition(audio_decoder, video_decoder, position_in_seconds);
                  }

                  retval = TRUE;
               }
            }
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRStartPlaying);

   return(retval);
}

/**
 *

 *
 * @brief   Reads the PVR play status.
 *
 * @param   U8BIT path - player number
 * @param   U32BIT* handle - returns handle in use
 *
 * @return   BOOLEAN - TRUE if playing, else FALSE.
 *
 */
BOOLEAN STB_PVRIsPlaying(U8BIT path, U32BIT *handle)
{
   BOOLEAN ret_val;

   FUNCTION_START(STB_PVRIsPlaying);

   *handle = play_status[path].handle;
   ret_val = STB_PVRIsPlayStarted(STB_DPGetPathAudioDecoder(path), STB_DPGetPathVideoDecoder(path));

   FUNCTION_FINISH(STB_PVRIsPlaying);

   return(ret_val);
}

/**
 * @brief   Saves the bookmark holding the playback position. This bookmark is used to resume a
 *          playback when STB_PVRStartPlaying is called with resume = TRUE. This function should
 *          be called while decoders are still running, to be sure the position in the playback
 *          is accurate.
 * @param   U8BIT path Playback path
 */
void STB_PVRSavePlayPosition(U8BIT path)
{
   U8BIT audio_decoder;
   U8BIT video_decoder;
   S_RECORDING *rec_ptr;
   CREATE_LINK_LIST_HEADER(bookmarks);
   U8BIT hours, mins, secs;
   U32BIT position_in_seconds;
   S_BOOKMARK *b;

   FUNCTION_START(STB_PVRSavePlayPosition);

   if ((path < num_paths) && (play_status[path].handle != 0))
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(play_status[path].handle);
      if ((rec_ptr != NULL) && (strlen((char *)rec_ptr->basename) > 0))
      {
         /* Read the old playback position boookmark */
         if (GetBookmarks(rec_ptr->disk_id, rec_ptr->basename, &bookmarks,
            BOOKMARK_PLAYBACK_POSITION, FALSE) > 0)
         {
            b = (S_BOOKMARK *)STB_LLGetFirstBlock(&bookmarks);
            if (b != NULL)
            {
               position_in_seconds = b->time;

               /* Remove it */
               STB_PVR_PRINT(("PVR - STB_PVRSavePlayPosition(%d): removing bookmark at %d seconds",
                  path, position_in_seconds));

               RemoveBookmark(rec_ptr->disk_id, rec_ptr->basename, position_in_seconds, NULL,
                  BOOKMARK_PLAYBACK_POSITION);
            }
            EmptyBookmarks(&bookmarks, TRUE);
         }

         audio_decoder = STB_DPGetPathAudioDecoder(path);
         video_decoder = STB_DPGetPathVideoDecoder(path);

         /* Get the elapsed playback time and save it so playback can be resumed from current
            position */
         if (STB_PVRGetElapsedTime(audio_decoder, video_decoder, &hours, &mins, &secs))
         {
            position_in_seconds = (hours * 3600) + (mins * 60) + secs;
         }
         else
         {
            STB_PVR_PRINT(("PVR - STB_PVRSavePlayPosition(%d): failed retrieving elapsed time", path));
            position_in_seconds = 0;
         }

         STB_PVR_PRINT(("PVR - STB_PVRSavePlayPosition(%d): adding bookmark at %d seconds", path,
            position_in_seconds));

         AddBookmark(rec_ptr->disk_id, rec_ptr->basename, position_in_seconds, NULL,
            BOOKMARK_PLAYBACK_POSITION);
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRSavePlayPosition);
}

/**
 * @brief   Returns the bookmarked play position for the given recording
 * @param   handle handle of recording
 * @return  play position in seconds; will be 0 if the recording hasn't been played yet
 */
U32BIT STB_PVRGetPlayPosition(U32BIT handle)
{
   S_RECORDING *rec_ptr;
   CREATE_LINK_LIST_HEADER(bookmarks);
   U32BIT position;
   S_BOOKMARK *b;

   FUNCTION_START(STB_PVRGetPlayPosition);

   position = 0;

   STB_OSMutexLock(list_mutex);

   rec_ptr = GetRecording(handle);
   if ((rec_ptr != NULL) && (strlen((char *)rec_ptr->basename) > 0))
   {
      /* Read the playback position boookmark */
      if (GetBookmarks(rec_ptr->disk_id, rec_ptr->basename, &bookmarks,
         BOOKMARK_PLAYBACK_POSITION, FALSE) > 0)
      {
         b = (S_BOOKMARK *)STB_LLGetFirstBlock(&bookmarks);
         if (b != NULL)
         {
            position = b->time;
         }
         EmptyBookmarks(&bookmarks, TRUE);
      }
   }

   STB_OSMutexUnlock(list_mutex);

   FUNCTION_FINISH(STB_PVRGetPlayPosition);

   return(position);
}

/**
 *

 *
 * @brief   Stop the PVR play back.
 *
 * @param   U8BIT path - player number
 *

 *
 */
void STB_PVRStopPlaying(U8BIT path)
{
   U8BIT audio_decoder;
   U8BIT video_decoder;
   S_RECORDING *rec_ptr;

   FUNCTION_START(STB_PVRStopPlaying);

   STB_PVR_PRINT(("PVR - StopPlaying (%d)", path));

   if ((path < num_paths) && (play_status[path].handle != 0))
   {
      audio_decoder = STB_DPGetPathAudioDecoder(path);
      video_decoder = STB_DPGetPathVideoDecoder(path);

      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(play_status[path].handle);
      if ((rec_ptr != NULL) && (strlen((char *)rec_ptr->basename) > 0))
      {
#ifdef COMMON_INTERFACE
         /* Free all CI+ items */
         FreeCIPlusItemList(path);
#endif
      }

      STB_OSMutexUnlock(list_mutex);

      STB_PVRPlayStop(audio_decoder, video_decoder);

      play_status[path].handle = 0;
   }

   FUNCTION_FINISH(STB_PVRStopPlaying);
}

/*!**************************************************************************
 * @brief   Read the DVB triplet (service id, transport id, orig net id) for a recording
 * @param   handle - handle of recording
 * @param   serv_id - pointer to return service id
 * @param   ts_id - pointer to return transport id
 * @param   orig_net_id - pointer to return original network id
 * @return  TRUE if values are returned
 ****************************************************************************/
BOOLEAN STB_PVRRecordingGetTriplet(U32BIT handle, U16BIT *serv_id, U16BIT *ts_id, U16BIT *orig_net_id)
{
   BOOLEAN ret_val;
   S_RECORDING *rec_ptr;

   FUNCTION_START(STB_PVRGetTriplet);

   ret_val = FALSE;

   rec_ptr = GetRecording(handle);
   if (rec_ptr != NULL)
   {
      *serv_id = rec_ptr->rec_info.serv_id;
      *ts_id = rec_ptr->rec_info.ts_id;
      *orig_net_id = rec_ptr->rec_info.orig_net_id;
      ret_val = TRUE;
   }
   FUNCTION_FINISH(STB_PVRGetTriplet);

   return(ret_val);
}

/**
 *

 *
 * @brief   Set to start playback in running mode.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *

 *
 */
void STB_PVRStartPlayRunning(U8BIT path)
{
   FUNCTION_START(STB_PVRStartPlayRunning);

   STB_PVR_PRINT(("STB_PVRStartPlayRunning(%d)", path));

   STB_PVRSetPlayStartMode(STB_DPGetPathAudioDecoder(path), STB_DPGetPathVideoDecoder(path), START_RUNNING);

   FUNCTION_FINISH(STB_PVRStartPlayRunning);
}

/**
 *

 *
 * @brief   Set to start playback in paused mode.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *

 *
 */
void STB_PVRStartPlayPaused(U8BIT path)
{
   FUNCTION_START(STB_PVRStartPlayPaused);

   STB_PVR_PRINT(("STB_PVRStartPlayPaused(%d)", path));

   STB_PVRSetPlayStartMode(STB_DPGetPathAudioDecoder(path), STB_DPGetPathVideoDecoder(path), START_PAUSED);

   FUNCTION_FINISH(STB_PVRStartPlayPaused);
}

/**
 *

 *
 * @brief   Set to start playback synced to AV decoder.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *

 *
 */
void STB_PVRStartPlaySync(U8BIT path)
{
   FUNCTION_START(STB_PVRStartPlaySync);

   STB_PVR_PRINT(("STB_PVRStartPlaySync(%d)", path));

   STB_PVRSetPlayStartMode(STB_DPGetPathAudioDecoder(path), STB_DPGetPathVideoDecoder(path), START_AVSYNC);

   FUNCTION_FINISH(STB_PVRStartPlaySync);
}

/**
 *

 *
 * @brief   Returns TRUE if play contains video data.
 *
 * @param   U32BIT handle - the ID of the decode path to use
 *
 * @return   BOOLEAN.
 *
 */
BOOLEAN STB_PVRIsPlayVideo(U32BIT handle)
{
   S_RECORDING *rec_ptr;
   BOOLEAN video;

   FUNCTION_START(STB_PVRIsPlayVideo);

   video = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(handle);
      if (rec_ptr != NULL)
      {
         if ((rec_ptr->rec_info.flags & REC_HAS_VIDEO) != 0)
         {
            video = TRUE;
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRIsPlayVideo);

   return(video);
}

/**
 *

 *
 * @brief   Returns TRUE if play contains audio data.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   BOOLEAN.
 *
 */
BOOLEAN STB_PVRIsPlayAudio(U8BIT path)
{
   BOOLEAN video, audio;

   FUNCTION_START(STB_PVRIsPlayAudio);

   audio = FALSE;

   if (STB_PVRIsPlayStarted(STB_DPGetPathAudioDecoder(path), STB_DPGetPathVideoDecoder(path)))
   {
      STB_PVRPlayEnabled(STB_DPGetPathAudioDecoder(path), STB_DPGetPathVideoDecoder(path), &video, &audio);
   }

   FUNCTION_FINISH(STB_PVRIsPlayAudio);

   return(audio);
}

/*!**************************************************************************
 * @brief   Saves (takes a copy of) the array of pids to be recorded
 * @param   path - decode path that will be used for the recording
 * @param   num_pids - number of pids in the array
 * @param   pid_array - array of structures describing PIDs
 * @return  TRUE if the PIDs are saved successfully, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRSetRecordingPids(U8BIT path, U16BIT num_pids, S_PVR_PID_INFO *pid_array)
{
   BOOLEAN retval;

   FUNCTION_START(STB_PVRSetRecordingPids);

   retval = FALSE;

   if (path < num_paths)
   {
      if (record_status[path].rec_pids_array != NULL)
      {
         STB_FreeMemory(record_status[path].rec_pids_array);
         record_status[path].rec_pids_array = NULL;
         record_status[path].num_pids = 0;
      }

      if ((num_pids > 0) && (pid_array != NULL))
      {
         record_status[path].rec_pids_array = (S_PVR_PID_INFO *)STB_GetMemory(num_pids * sizeof(S_PVR_PID_INFO));
         if (record_status[path].rec_pids_array != NULL)
         {
            memcpy(record_status[path].rec_pids_array, pid_array, num_pids * sizeof(S_PVR_PID_INFO));
            record_status[path].num_pids = num_pids;
            retval = TRUE;
         }
      }
   }

   FUNCTION_FINISH(STB_PVRSetRecordingPids);

   return(retval);
}

/*!**************************************************************************
 * @brief   Saves (takes a copy of) the array of pids to be recorded and
            pass them to the platform
 * @param   path - decode path that used for the recording
 * @param   num_pids - number of pids in the array
 * @param   pid_array - array of structures describing PIDs
 * @return  TRUE if the PIDs are updated successfully, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRUpdateRecordingPids(U8BIT path, U16BIT num_pids, S_PVR_PID_INFO *pid_array)
{
   BOOLEAN retval;
   S_RECORDING *rec_ptr;

   FUNCTION_START(STB_PVRUpdateRecordingPids);

   retval = FALSE;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(record_status[path].handle);
      if (rec_ptr != NULL)
      {
         retval = STB_PVRSetRecordingPids(path, num_pids, pid_array);
         if (retval)
         {
            retval = STB_PVRRecordChangePids(record_status[path].rec_index, record_status[path].num_pids,
                  record_status[path].rec_pids_array);
         }
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRUpdateRecordingPids);

   return(retval);
}

/*!**************************************************************************
 * @brief   Saves the DVB triplet of the service to be recorded
 * @param   path - decode path that will be used for the recording
 * @param   serv_id - service id
 * @param   ts_id - transport stream
 * @param   orig_net_id - original network id
 * @return  TRUE if the triplet is saved successfully, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRSetRecoringTriplet(U8BIT path, U16BIT serv_id, U16BIT ts_id, U16BIT orig_net_id)
{
   BOOLEAN retval;

   FUNCTION_START(STB_PVRSetRecoringTriplet);

   retval = FALSE;

   if (path < num_paths)
   {
      record_status[path].serv_id = serv_id;
      record_status[path].ts_id = ts_id;
      record_status[path].orig_net_id = orig_net_id;
      retval = TRUE;
   }

   FUNCTION_FINISH(STB_PVRSetRecoringTriplet);

   return(retval);
}

/*!**************************************************************************
 * @brief   Creates or deletes a bookmark at the current position during playback.
 *          If there's a bookmark within a defined number of seconds of the current
 *          position then the bookmark is deleted, otherwise one is created.
 * @param   path - playback decode path
 * @return  TRUE if a bookmark is created or deleted, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRToggleBookmark(U8BIT path)
{
   S_RECORDING *rec_ptr;
   CREATE_LINK_LIST_HEADER(bookmarks);
   U8BIT audio_decoder;
   U8BIT video_decoder;
   U8BIT hours, mins, secs;
   U32BIT position_in_seconds;
   S_BOOKMARK *b;
   BOOLEAN retval;

   FUNCTION_START(STB_PVRToggleBookmark);

   retval = FALSE;

   if ((path < num_paths) && (play_status[path].handle != 0))
   {
      audio_decoder = STB_DPGetPathAudioDecoder(path);
      video_decoder = STB_DPGetPathVideoDecoder(path);

      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(play_status[path].handle);
      if ((rec_ptr != NULL) && (strlen((char *)rec_ptr->basename) > 0))
      {
         /* Read the boookmarks */
         GetBookmarks(rec_ptr->disk_id, rec_ptr->basename, &bookmarks,
            BOOKMARK_USER, FALSE);

         /* The elapsed playback time is where the bookmark needs to be set */
         if (STB_PVRGetElapsedTime(audio_decoder, video_decoder, &hours, &mins, &secs))
         {
            position_in_seconds = (hours * 3600) + (mins * 60) + secs;

            /* Check whether this position coincides with an existing bookmark.
             * The first bookmark is always used to save the position for resume playback  */
            b = (S_BOOKMARK *)STB_LLGetFirstBlock(&bookmarks);
            while (b != NULL)
            {
               if ((position_in_seconds <= BOOKMARK_TOLERANCE) &&
                   (b->time <= BOOKMARK_TOLERANCE))
               {
                  break;
               }
               else if ((position_in_seconds >= b->time - BOOKMARK_TOLERANCE) &&
                        (position_in_seconds <= b->time + BOOKMARK_TOLERANCE))
               {
                  break;
               }

               b = (S_BOOKMARK *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)b);
            }

            if (b != NULL)
            {
               /* Existing bookmark has been found so delete it */
               RemoveBookmark(rec_ptr->disk_id, rec_ptr->basename, b->time, NULL,
                  BOOKMARK_USER);
            }
            else
            {
               /* This is a new bookmark */
               AddBookmark(rec_ptr->disk_id, rec_ptr->basename, position_in_seconds, NULL,
                  BOOKMARK_USER);
            }
         }

         EmptyBookmarks(&bookmarks, TRUE);
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRToggleBookmark);

   return(retval);
}

/*!**************************************************************************
 * @brief   Jumps playback to the position of the bookmark after the current play position
 * @param   path - playback decode path
 * @return  TRUE if a bookmark exists and play position is changed, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRGotoNextBookmark(U8BIT path)
{
   S_RECORDING *rec_ptr;
   CREATE_LINK_LIST_HEADER(bookmarks);
   U32BIT num_bookmarks;
   U8BIT audio_decoder;
   U8BIT video_decoder;
   U8BIT hours, mins, secs;
   U32BIT position_in_seconds;
   S_BOOKMARK *b;
   BOOLEAN retval;

   FUNCTION_START(STB_PVRGotoNextBookmark);

   retval = FALSE;

   if ((path < num_paths) && (play_status[path].handle != 0))
   {
      audio_decoder = STB_DPGetPathAudioDecoder(path);
      video_decoder = STB_DPGetPathVideoDecoder(path);

      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(play_status[path].handle);
      if ((rec_ptr != NULL) && (strlen((char *)rec_ptr->basename) > 0))
      {
         /* Read the boookmarks */
         num_bookmarks = GetBookmarks(rec_ptr->disk_id, rec_ptr->basename, &bookmarks,
            BOOKMARK_USER, FALSE);

         if (num_bookmarks > 0)
         {
            STB_LLSort(&bookmarks, CompareBookmarks);

            /* Need to find the bookmark that occurs after the current playback position */
            if (STB_PVRGetElapsedTime(audio_decoder, video_decoder, &hours, &mins, &secs))
            {
               position_in_seconds = (hours * 3600) + (mins * 60) + secs;

               b = (S_BOOKMARK *)STB_LLGetFirstBlock(&bookmarks);
               while (b != NULL)
               {
                  if (b->time > position_in_seconds)
                  {
                     break;
                  }
                  b = (S_BOOKMARK *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)b);
               }

               if (b != NULL)
               {
                  retval = STB_PVRPlaySetPosition(audio_decoder, video_decoder, b->time);
               }
            }
         }

         EmptyBookmarks(&bookmarks, TRUE);
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRGotoNextBookmark);

   return(retval);
}

/**
 * @brief   Allocates and returns an array containing the handles of the bookmarks for
 *          the recording currently being played.
 * @param   path Playback decode path
 * @param   bookmarks Pointer to array of returned bookmark handles
 * @return  The number of handles in the allocated array. If this is 0 then array will be NULL.
 */
U16BIT STB_PVRGetBookmarks(U8BIT path, void ***bookmarks)
{
   U16BIT num_bookmarks;

   FUNCTION_START(STB_PVRGetBookmarks);

   num_bookmarks = 0;

   if (path < num_paths)
   {
      num_bookmarks = STB_PVRGetBookmarksForRecording(play_status[path].handle, bookmarks);
   }

   FUNCTION_FINISH(STB_PVRGetBookmarks);

   return(num_bookmarks);
}

/**
 * @brief   Allocates and returns an array containing the handles of the bookmarks for
 *          the recording specified by the given handle
 * @param   handle Recording handle
 * @param   bookmarks Pointer to array of returned bookmark handles
 * @return  The number of handles in the allocated array. If this is 0 then array will be NULL.
 */
U16BIT STB_PVRGetBookmarksForRecording(U32BIT handle, void ***bookmarks)
{
   CREATE_LINK_LIST_HEADER(bookmark_list);
   U32BIT num_bookmarks;
   S_RECORDING *rec_ptr;
   S_BOOKMARK *b;
   U32BIT i;

   FUNCTION_START(STB_PVRGetBookmarksForRecording);

   num_bookmarks = 0;

   if (bookmarks != NULL)
   {
      *bookmarks = NULL;
   }

   if (initialised_flag && (handle != 0))
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(handle);
      if ((rec_ptr != NULL) && (strlen((char *)rec_ptr->basename) > 0))
      {
         /* Read the boookmarks */
         num_bookmarks = GetBookmarks(rec_ptr->disk_id, rec_ptr->basename, &bookmark_list,
            BOOKMARK_USER, TRUE);
         if ((num_bookmarks > 0) && (bookmarks != NULL))
         {
            *bookmarks = STB_GetMemory(num_bookmarks * sizeof(void *));
            if (*bookmarks != NULL)
            {
               i = 0;
               b = (S_BOOKMARK *)STB_LLGetFirstBlock(&bookmark_list);
               while (b != NULL)
               {
                  (*bookmarks)[i] = b;
                  b = (S_BOOKMARK *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)b);
                  i++;
               }
            }
            else
            {
               num_bookmarks = 0;
            }
         }

         EmptyBookmarks(&bookmark_list, FALSE);
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRGetBookmarksForRecording);

   return(num_bookmarks);
}

/**
 * @brief   Frees a previously allocated array of bookmark handles
 * @param   bookmarks Array of bookmark handles to be freed
 * @param   num Number of handles in the array.
 */
void STB_PVRReleaseBookmarks(void **bookmarks, U16BIT num)
{
   U16BIT i;
   S_BOOKMARK *b

   FUNCTION_START(STB_PVRReleaseBookmarks);

   if (bookmarks != NULL)
   {
      for (i = 0; i < num; i++)
      {
         b = bookmarks[i];
         if (b->name != NULL)
         {
            STB_FreeMemory(b->name);
         }
         STB_FreeMemory(b);
      }

      STB_FreeMemory(bookmarks);
   }

   FUNCTION_FINISH(STB_PVRReleaseBookmarks);
}

/**
 * @brief   Returns the time associated with a bookmark
 * @param   bookmark_handle Bookmark handle
 * @return  Bookmark time
 */
U32BIT STB_PVRGetBookmarkTime(void *bookmark_handle)
{
   U32BIT time;

   FUNCTION_START(STB_PVRGetBookmarkTime);

   time = ((S_BOOKMARK *)bookmark_handle)->time;

   FUNCTION_FINISH(STB_PVRGetBookmarkTime);

   return time;
}

/**
 * @brief   Allocates and returns the name associated with a bookmark
 * @param   bookmark_handle Bookmark handle
 * @return  Pointer to the name string. This pointer must freed with STB_FreeMemory
 */
U8BIT *STB_PVRGetBookmarkName(void *bookmark_handle)
{
   U8BIT *name, *n;
   U32BIT len;

   FUNCTION_START(STB_PVRGetBookmarkName);

   n = ((S_BOOKMARK *)bookmark_handle)->name;
   if (n != NULL)
   {
      len = strlen((char *)n);
      if (len >= STB_PVR_NAME_LEN)
      {
         len = STB_PVR_NAME_LEN - 1;
      }

      name = STB_GetMemory(len + 1);
      if (name != NULL)
      {
         memcpy(name, n, len);
         name[len] = '\0';
      }
   }
   else
   {
      name = NULL;
   }

   FUNCTION_FINISH(STB_PVRGetBookmarkName);

   return name;
}

/*!**************************************************************************
 * @brief   Handles the notification received from the platform code when playback
 *          reaches a set position. For CI+, this works out the next URI/licence/pin
 *          that needs to be applied, applies them and sets the next notification time
 * @param   path - decode path
 ****************************************************************************/
void STB_PVRPlaybackNotifyTime(U8BIT path)
{
#ifdef COMMON_INTERFACE
   S_RECORDING *rec_ptr;
   U8BIT audio_decoder;
   U8BIT video_decoder;
   U8BIT hours, mins, secs;
   U32BIT position_in_seconds;
   S16BIT speed;
#endif

   FUNCTION_START(STB_PVRPlaybackNotifyTime);

#ifdef COMMON_INTERFACE
   if ((path < num_paths) && (play_status[path].handle != 0))
   {
      if ((play_status[path].slot_id != INVALID_RES_ID) && (play_status[path].item_list != NULL))
      {
         STB_OSMutexLock(list_mutex);

         if ((rec_ptr = GetRecording(play_status[path].handle)) != NULL)
         {
            audio_decoder = STB_DPGetPathAudioDecoder(path);
            video_decoder = STB_DPGetPathVideoDecoder(path);

            if (STB_PVRGetElapsedTime(audio_decoder, video_decoder, &hours, &mins, &secs))
            {
               position_in_seconds = (hours * 3600) + (mins * 60) + secs;
               speed = STB_PVRGetPlaySpeed(audio_decoder, video_decoder);

               /* Get the current playback position and apply the next URI/licence/pin for this position */
               ApplyCIPlusItems(path, rec_ptr, position_in_seconds, speed);

               /* Find the next URI/licence/pin to be applied */
               FindNextCIPlusItem(path, speed);
            }
         }

         STB_OSMutexUnlock(list_mutex);
      }
   }
#else
   USE_UNWANTED_PARAM(path);
#endif

   FUNCTION_FINISH(STB_PVRPlaybackNotifyTime);
}

/**
 *

 *
 * @brief   Start the PVR record tasks.
 *
 * @param   U8BIT path - recorder number
 * @param   U32BIT handle - handle to use for play file
 *

 *
 */
BOOLEAN STB_PVRStartRecording(U8BIT path, U32BIT handle)
{
   S_RECORDING *rec_ptr;
   BOOLEAN start_recording;
   BOOLEAN retval;

   FUNCTION_START(STB_PVRStartRecording);

   retval = FALSE;

   ASSERT(path < num_paths);

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(handle);
      if ((rec_ptr != NULL) && (strlen((char *)rec_ptr->basename) > 0))
      {
         /* Copy the triplet into the info */
         rec_ptr->rec_info.serv_id = record_status[path].serv_id;
         rec_ptr->rec_info.ts_id = record_status[path].ts_id;
         rec_ptr->rec_info.orig_net_id = record_status[path].orig_net_id;

         if (record_status[path].encrypted && (enc_dec_key != NULL) && (enc_dec_iv != NULL))
         {
            rec_ptr->rec_info.flags |= REC_ENCRYPTED;
         }
         else
         {
            rec_ptr->rec_info.flags &= ~REC_ENCRYPTED;
         }

         /* Remember whether the recording has video */
         if (IncludesVideoPid(record_status[path].rec_pids_array, record_status[path].num_pids))
         {
            rec_ptr->rec_info.flags |= REC_HAS_VIDEO;
         }
         else
         {
            rec_ptr->rec_info.flags &= ~REC_HAS_VIDEO;
         }

         rec_ptr->rec_info.status = RECORDING_STATUS_STARTED;

         /* Write the updated info to disk */
         WriteODBFile(rec_ptr->disk_id, rec_ptr->basename, FALSE, &rec_ptr->rec_info);

         record_status[path].rec_index = STB_PVRAcquireRecorderIndex(STB_DPGetPathTuner(path), STB_DPGetPathDemux(path));

         start_recording = FALSE;

         if (record_status[path].encrypted)
         {
            /* Get the encryption key to be used */
            if ((enc_dec_key != NULL) && (enc_dec_iv != NULL))
            {
               STB_PVRSetRecordEncryptionKey(record_status[path].rec_index, TRUE, enc_dec_key, enc_dec_iv, enc_dec_key_len);
               start_recording = TRUE;
            }
            else
            {
               STB_PVRReleaseRecorderIndex(record_status[path].rec_index);
               record_status[path].rec_index = INVALID_RES_ID;
               STB_PVR_PRINT(("STB_PVRStartRecording(%u 0x%08lx): Recording failed, encryption key not available",
                              path, handle));
            }
         }
         else
         {
            STB_PVRSetRecordEncryptionKey(record_status[path].rec_index, FALSE, NULL, NULL, 0);
            start_recording = TRUE;
         }

         if (start_recording)
         {
            STB_PVRSetRecordStartMode(record_status[path].rec_index, record_status[path].smode,
               record_status[path].timeshift_seconds);

            record_status[path].handle = handle;

#ifdef COMMON_INTERFACE
            /* Apply descrambler key if it's been provided */
            if (record_status[path].keys[KEY_PARITY_EVEN].valid)
            {
               STB_PVRApplyDescramblerKey(record_status[path].rec_index, record_status[path].desc_type,
                  KEY_PARITY_EVEN, record_status[path].keys[KEY_PARITY_EVEN].key,
                  record_status[path].keys[KEY_PARITY_EVEN].iv);
            }
            if (record_status[path].keys[KEY_PARITY_ODD].valid)
            {
               STB_PVRApplyDescramblerKey(record_status[path].rec_index, record_status[path].desc_type,
                  KEY_PARITY_ODD, record_status[path].keys[KEY_PARITY_ODD].key,
                  record_status[path].keys[KEY_PARITY_ODD].iv);
            }
#endif

            if (STB_PVRRecordStart(rec_ptr->disk_id, record_status[path].rec_index, rec_ptr->basename,
                   record_status[path].num_pids, record_status[path].rec_pids_array))
            {
               record_status[path].start_time = STB_OSGetClockMilliseconds();
               record_status[path].paused = FALSE;
               record_status[path].duration = 0;

               rec_ptr->recording = TRUE;
               retval = TRUE;
            }
            else
            {
               record_status[path].handle = 0;
               STB_PVRReleaseRecorderIndex(record_status[path].rec_index);
               record_status[path].rec_index = INVALID_RES_ID;
               STB_PVR_PRINT(("STB_PVRStartRecording(%u, 0x%08lx): Failed to start recording", path, handle));
            }
         }
         else
         {
            STB_PVRReleaseRecorderIndex(record_status[path].rec_index);
            record_status[path].rec_index = INVALID_RES_ID;
         }
      }
#ifdef STB_DEBUG
      else
      {
         if (rec_ptr == NULL)
         {
            STB_PVR_PRINT(("STB_PVRStartRecording(%u, 0x%08lx): Couldn't find recording to be started", path, handle));
         }
         else
         {
            STB_PVR_PRINT(("STB_PVRStartRecording(%u, 0x%08lx): Basename not set for recording", path, handle));
         }
      }
#endif

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRStartRecording);

   return(retval);
}

/*!**************************************************************************
 * @brief   Pauses recording after it's been started
 * @param   path - decode path of the recording
 * @return  TRUE if recording is paused or is already paused, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRPauseRecording(U8BIT path)
{
   BOOLEAN retval;

   FUNCTION_START(STB_PVRPauseRecording);

   if (!record_status[path].paused)
   {
      retval = STB_PVRRecordPause(record_status[path].rec_index);
      if (retval)
      {
         /* Update the recording duration upto this point */
         record_status[path].duration += STB_OSGetClockDiff(record_status[path].start_time);
         record_status[path].paused = TRUE;
      }
   }
   else
   {
      retval = TRUE;
   }

   FUNCTION_FINISH(STB_PVRPauseRecording);

   return(retval);
}

/*!**************************************************************************
 * @brief   Resumes a recording after it's been paused
 * @param   path - decode path of the recording
 * @return  TRUE if the recording is resumed or isn't currently paused, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRResumeRecording(U8BIT path)
{
   BOOLEAN retval;

   FUNCTION_START(STB_PVRResumeRecording);

   if (record_status[path].paused)
   {
      retval = STB_PVRRecordResume(record_status[path].rec_index);
      if (retval)
      {
         /* Save the time the recording has been started again */
         record_status[path].start_time = STB_OSGetClockMilliseconds();
         record_status[path].paused = FALSE;
      }
   }
   else
   {
      retval = TRUE;
   }

   FUNCTION_FINISH(STB_PVRResumeRecording);

   return(retval);
}

/**
 *

 *
 * @brief   Stop the PVR recording.
 *
 * @param   U8BIT path - recorder number
 *

 *
 */
void STB_PVRStopRecording(U8BIT path)
{
   S_RECORDING *rec_ptr;
   U32BIT recording_time;
   U8BIT hours, mins, secs;

   FUNCTION_START(STB_PVRStopRecording);

   ASSERT(path < num_paths);

   STB_PVRRecordStop(record_status[path].rec_index);

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      rec_ptr = GetRecording(record_status[path].handle);
      if ((rec_ptr != NULL) && (strlen((char *)rec_ptr->basename) > 0))
      {
         /* Work out the duration of the recording and save to the DB */
         if (record_status[path].paused)
         {
            recording_time = record_status[path].duration;
         }
         else
         {
            recording_time = STB_OSGetClockDiff(record_status[path].start_time);
         }

         if (recording_time >= 4000)
            recording_time -= 4000;  //FIXME: a workaround for the recording start delay (1500 + 2500)ms

         recording_time/= 1000;

         hours = recording_time / 3600;
         mins = (recording_time / 60) - (hours * 60);
         secs = recording_time % 60;

         rec_ptr->rec_info.len_hour = hours;
         rec_ptr->rec_info.len_min = mins;
         rec_ptr->rec_info.len_sec = secs;
         rec_ptr->rec_info.status = RECORDING_STATUS_COMPLETED;

         rec_ptr->recording = FALSE;
         WriteODBFile(rec_ptr->disk_id, rec_ptr->basename, TRUE, &rec_ptr->rec_info);
      }

      if (record_status[path].rec_pids_array != NULL)
      {
         STB_FreeMemory(record_status[path].rec_pids_array);
         record_status[path].rec_pids_array = NULL;
         record_status[path].num_pids = 0;
      }

      record_status[path].handle = 0; /* Recording handle not valid any more */

#ifdef COMMON_INTERFACE
      /* Make sure the keys aren't used for the next recording */
      record_status[path].keys[KEY_PARITY_EVEN].valid = FALSE;
      record_status[path].keys[KEY_PARITY_ODD].valid = FALSE;
#endif

      STB_PVRReleaseRecorderIndex(record_status[path].rec_index);
      record_status[path].rec_index = INVALID_RES_ID;

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(STB_PVRStopRecording);
}

/**
 *

 *
 * @brief   Reads the PVR record status.
 *
 * @param   U8BIT path - decoder number
 * @param   U32BIT* handle - returns handle in use
 *
 * @return   BOOLEAN - TRUE if playing, else FALSE.
 *
 */
BOOLEAN STB_PVRIsRecording(U8BIT path, U32BIT *handle)
{
   BOOLEAN ret_val;
   S_RECORDING *rec_ptr;

   FUNCTION_START(STB_PVRIsRecording);

   ASSERT(path < num_paths);

   ret_val = FALSE;

   if (initialised_flag)
   {
      if (STB_DPIsRecordingPath(path) == TRUE)
      {
         *handle = record_status[path].handle;
         rec_ptr = GetRecording(*handle);
         if (rec_ptr != NULL)
         {
            ret_val = TRUE;
         }
      }
   }

   FUNCTION_FINISH(STB_PVRIsRecording);

   return(ret_val);
}

/**
 *

 *
 * @brief   Set to start recording in running mode.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *

 *
 */
void STB_PVRStartRecordRunning(U8BIT path)
{
   FUNCTION_START(STB_PVRStartRecordRunning);

   ASSERT(path < num_paths);

   STB_PVR_PRINT(("STB_PVRStartRecordRunning(%d)", path));

   record_status[path].smode = START_RUNNING;
   record_status[path].timeshift_seconds = 0;

   FUNCTION_FINISH(STB_PVRStartRecordRunning);
}

/**
 *

 *
 * @brief   Set to start recording in paused mode for timeshift recording.
 *
 * @param   U8BIT path - the ID of the decode path to use
 * @param   U32BIT timeshift_seconds - the size of the timeshift buffer in seconds
 *

 *
 */
void STB_PVRStartRecordPaused(U8BIT path, U32BIT timeshift_seconds)
{
   FUNCTION_START(STB_PVRStartRecordPaused);

   ASSERT(path < num_paths);

   STB_PVR_PRINT(("STB_PVRStartRecordPaused(%d)", path));

   record_status[path].smode = START_PAUSED;
   record_status[path].timeshift_seconds = timeshift_seconds;

   FUNCTION_FINISH(STB_PVRStartRecordPaused);
}

/**
 *

 *
 * @brief   Returns TRUE if record contains video data.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   BOOLEAN.
 *
 */
BOOLEAN STB_PVRIsRecordVideo(U8BIT path)
{
   BOOLEAN audio, video;

   FUNCTION_START(STB_PVRIsRecordVideo);

   ASSERT(path < num_paths);

   video = FALSE;

   if (STB_PVRIsRecordStarted(record_status[path].rec_index))
   {
      STB_PVRRecordEnabled(record_status[path].rec_index, &video, &audio);
   }

   FUNCTION_FINISH(STB_PVRIsRecordVideo);

   return(video);
}

/**
 *

 *
 * @brief   Returns TRUE if record contains audio data.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   BOOLEAN.
 *
 */
BOOLEAN STB_PVRIsRecordAudio(U8BIT path)
{
   BOOLEAN video, audio;

   FUNCTION_START(STB_PVRIsRecordAudio);

   ASSERT(path < num_paths);

   audio = FALSE;

   if (STB_PVRIsRecordStarted(record_status[path].rec_index))
   {
      STB_PVRRecordEnabled(record_status[path].rec_index, &video, &audio);
   }

   FUNCTION_FINISH(STB_PVRIsRecordAudio);

   return(audio);
}

/*!**************************************************************************
 * @brief   Sets whether a recording should be encrypted.
 *          Must be set before the recording is started
 * @param   path - decode path that will be used for the recording
 * @param   state - TRUE if it's to be encrypted
 ****************************************************************************/
void STB_PVREncryptRecording(U8BIT path, BOOLEAN state)
{
   FUNCTION_START(STB_PVREncryptRecording);

   record_status[path].encrypted = state;

   FUNCTION_FINISH(STB_PVREncryptRecording);
}

#ifdef COMMON_INTERFACE
/*!**************************************************************************
 * @brief   Sets the descrambler key data to be applied with a recording
 * @param   path - decode path that will be/is being used for the recording
 * @param   desc_type - type of descrambling to be applied, normally DES or AES
 * @param   parity - key parity
 * @param   key - key data
 * @param   iv - initialisation vector for AES
 * @return  TRUE if the key data is saved successfully, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRSetDescramblerKey(U8BIT path, E_STB_DMX_DESC_TYPE desc_type,
   E_STB_DMX_DESC_KEY_PARITY parity, U8BIT *key, U8BIT *iv)
{
   BOOLEAN retval;

   FUNCTION_START(STB_PVRSetDescramblerKey);

   retval = FALSE;

   if (path < num_paths)
   {
      if (record_status[path].rec_index != INVALID_RES_ID)
      {
         /* Recording has been started so apply the updated descrambler key immediately */
         retval = STB_PVRApplyDescramblerKey(record_status[path].rec_index, desc_type,
               parity, key, iv);
      }
      else
      {
         /* Save the descrambler info for use when the recording is started */
         retval = TRUE;

         switch (desc_type)
         {
            case DESC_TYPE_DES:
               record_status[path].desc_type = desc_type;
               memcpy(record_status[path].keys[parity].key, key, 8);
               record_status[path].keys[parity].valid = TRUE;
               break;

            case DESC_TYPE_AES:
               record_status[path].desc_type = desc_type;
               memcpy(record_status[path].keys[parity].key, key, 16);
               memcpy(record_status[path].keys[parity].iv, iv, 16);
               record_status[path].keys[parity].valid = TRUE;
               break;

            default:
               record_status[path].keys[parity].valid = FALSE;
               retval = FALSE;
               break;
         }
      }
   }

   FUNCTION_FINISH(STB_PVRSetDescramblerKey);

   return(retval);
}

#endif /* COMMON_INTERFACE */

#if 0
/**
 *

 *
 * @brief   Export current frame of playback.
 *
 * @param   U8BIT decoder - the ID of the decode path to use
 *

 *
 */
void STB_PVRExportFrame(U8BIT path)
{
   FUNCTION_START(STB_PVRExportFrame);

   ASSERT(path < num_paths);

   STB_PVR_PRINT(("STB_PVRExportFrame(%d)", path));

   if (STB_PVRIsPlayStarted(STB_DPGetPathAudioDecoder(path), STB_DPGetPathVideoDecoder(path)) == TRUE)
   {
      STB_PVRSaveFrame(STB_DPGetPathAudioDecoder(path), STB_DPGetPathVideoDecoder(path));
   }

   FUNCTION_FINISH(STB_PVRExportFrame);
}

#endif

/**
 *

 *
 * @brief   Restarts playback (no trick mode) - has no effect when decoding from demux.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *

 *
 */
void STB_PVRPlayNormal(U8BIT path)
{
   U8BIT audio_decoder, video_decoder;
   S16BIT play_speed;

   FUNCTION_START(STB_PVRPlayNormal);

   ASSERT(path < num_paths);

   STB_PVR_PRINT(("STB_PVRPlayNormal(%u)", path));

   audio_decoder = STB_DPGetPathAudioDecoder(path);
   video_decoder = STB_DPGetPathVideoDecoder(path);

   if (STB_PVRIsPlayStarted(audio_decoder, video_decoder))
   {
      play_speed = STB_PVRGetPlaySpeed(audio_decoder, video_decoder);
      if (play_speed != 100)
      {
         /* Request normal playback */
         STB_PVRSetPlaySpeed(audio_decoder, video_decoder, 100);

#ifdef COMMON_INTERFACE
         if (play_speed < 0)
         {
            /* Change of direction, so re-evaluate the next URI/licence/pin that needs to be applied */
            FindNextCIPlusItem(path, play_speed);
         }
#endif
      }
   }

   FUNCTION_FINISH(STB_PVRPlayNormal);
}

/**
 *

 *
 * @brief   Returns TRUE if play is not in trick mode.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   BOOLEAN.
 *
 */
BOOLEAN STB_PVRIsPlayNormal(U8BIT path)
{
   U8BIT audio_decoder, video_decoder;
   BOOLEAN ret_val = FALSE;

   FUNCTION_START(STB_PVRIsPlayNormal);

   ASSERT(path < num_paths);

   audio_decoder = STB_DPGetPathAudioDecoder(path);
   video_decoder = STB_DPGetPathVideoDecoder(path);

   if (STB_PVRIsPlayStarted(audio_decoder, video_decoder))
   {
      if (STB_PVRGetPlaySpeed(audio_decoder, video_decoder) == 100)
      {
         ret_val = TRUE;
      }
   }

   FUNCTION_FINISH(STB_PVRIsPlayNormal);

   return(ret_val);
}

/**
 *

 *
 * @brief   Pauses play - has no effect when decoding from demux.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *

 *
 */
void STB_PVRPlayPause(U8BIT path)
{
   U8BIT audio_decoder, video_decoder;

   FUNCTION_START(STB_PVRPlayPause);

   ASSERT(path < num_paths);

   STB_PVR_PRINT(("STB_PVRPlayPause(%d)", path));

   audio_decoder = STB_DPGetPathAudioDecoder(path);
   video_decoder = STB_DPGetPathVideoDecoder(path);

   if (STB_PVRIsPlayStarted(audio_decoder, video_decoder) == TRUE)
   {
      STB_PVRSetPlaySpeed(audio_decoder, video_decoder, 0);
   }

   FUNCTION_FINISH(STB_PVRPlayPause);
}

/**
 *

 *
 * @brief   Returns TRUE if play is paused trick mode.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   BOOLEAN.
 *
 */
BOOLEAN STB_PVRIsPlayPause(U8BIT path)
{
   U8BIT audio_decoder, video_decoder;
   BOOLEAN ret_val = FALSE;

   FUNCTION_START(STB_PVRIsPlayPause);

   ASSERT(path < num_paths);

   audio_decoder = STB_DPGetPathAudioDecoder(path);
   video_decoder = STB_DPGetPathVideoDecoder(path);

   if (STB_PVRIsPlayStarted(audio_decoder, video_decoder) == TRUE)
   {
      if (STB_PVRGetPlaySpeed(audio_decoder, video_decoder) == 0)
      {
         ret_val = TRUE;
      }
   }

   FUNCTION_FINISH(STB_PVRIsPlayPause);

   return(ret_val);
}

/**
 *

 *
 * @brief   Set play forwards - has no effect when decoding from demux.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *

 *
 */
void STB_PVRPlayForward(U8BIT path)
{
   U8BIT audio_decoder, video_decoder;
   S16BIT play_speed;

   FUNCTION_START(STB_PVRPlayForward);

   ASSERT(path < num_paths);

   STB_PVR_PRINT(("STB_PVRPlayForward(%d)", path));

   audio_decoder = STB_DPGetPathAudioDecoder(path);
   video_decoder = STB_DPGetPathVideoDecoder(path);

   if (STB_PVRIsPlayStarted(audio_decoder, video_decoder))
   {
      /* Request forwards PVR trick mode */
      play_speed = STB_PVRGetPlaySpeed(audio_decoder, video_decoder);
      if (play_speed <= 0)
      {
         play_speed = -play_speed;

         STB_PVR_PRINT(("STB_PVRPlayForward(%d): requesting FF play speed %d", path, play_speed));

         STB_PVRSetPlaySpeed(audio_decoder, video_decoder, play_speed);

#ifdef COMMON_INTERFACE
         /* Change of direction, so re-evaluate the next URI/licence/pin that needs to be applied */
         FindNextCIPlusItem(path, play_speed);
#endif
      }
   }

   FUNCTION_FINISH(STB_PVRPlayForward);
}

/**
 *

 *
 * @brief   Returns TRUE if play is forwards trick mode.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   BOOLEAN.
 *
 */
BOOLEAN STB_PVRIsPlayForward(U8BIT path)
{
   U8BIT audio_decoder, video_decoder;
   BOOLEAN ret_val = FALSE;

   FUNCTION_START(STB_PVRIsPlayForward);

   ASSERT(path < num_paths);

   audio_decoder = STB_DPGetPathAudioDecoder(path);
   video_decoder = STB_DPGetPathVideoDecoder(path);

   if (STB_PVRIsPlayStarted(audio_decoder, video_decoder))
   {
      if (STB_PVRGetPlaySpeed(audio_decoder, video_decoder) > 0)
      {
         ret_val = TRUE;
      }
   }

   FUNCTION_FINISH(STB_PVRIsPlayForward);

   return(ret_val);
}

/**
 *

 *
 * @brief   Set play backwards - has no effect when decoding from demux.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *

 *
 */
void STB_PVRPlayReverse(U8BIT path)
{
   U8BIT audio_decoder, video_decoder;
   S16BIT play_speed;

   FUNCTION_START(STB_PVRPlayReverse);

   ASSERT(path < num_paths);

   STB_PVR_PRINT(("STB_PVRPlayReverse(%d)", path));

   audio_decoder = STB_DPGetPathAudioDecoder(path);
   video_decoder = STB_DPGetPathVideoDecoder(path);

   if (STB_PVRIsPlayStarted(audio_decoder, video_decoder))
   {
      /* Request backwards PVR trick mode */
      play_speed = STB_PVRGetPlaySpeed(audio_decoder, video_decoder);
      if (play_speed > 0)
      {
         play_speed = -play_speed;

         STB_PVR_PRINT(("STB_PVRPlayReverse(%d): requesting FR play speed %d", path, play_speed));

         STB_PVRSetPlaySpeed(audio_decoder, video_decoder, play_speed);

#ifdef COMMON_INTERFACE
         /* Change of direction, so re-evaluate the next URI/licence/pin that needs to be applied */
         FindNextCIPlusItem(path, play_speed);
#endif
      }
   }

   FUNCTION_FINISH(STB_PVRPlayReverse);
}

/**
 *

 *
 * @brief   Returns TRUE if play is backwards trick mode.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   BOOLEAN.
 *
 */
BOOLEAN STB_PVRIsPlayReverse(U8BIT path)
{
   U8BIT audio_decoder, video_decoder;
   BOOLEAN ret_val = FALSE;

   FUNCTION_START(STB_PVRIsPlayReverse);

   ASSERT(path < num_paths);

   audio_decoder = STB_DPGetPathAudioDecoder(path);
   video_decoder = STB_DPGetPathVideoDecoder(path);

   if (STB_PVRIsPlayStarted(audio_decoder, video_decoder))
   {
      if (STB_PVRGetPlaySpeed(audio_decoder, video_decoder) < 0)
      {
         ret_val = TRUE;
      }
   }

   FUNCTION_FINISH(STB_PVRIsPlayReverse);

   return(ret_val);
}

/**
 *

 *
 * @brief   Increments play by one frame - has no effect when decoding from demux.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *

 *
 */
void STB_PVRPlayFrameInc(U8BIT path)
{
   FUNCTION_START(STB_PVRPlayFrameInc);

   ASSERT(path < num_paths);

   STB_PVR_PRINT(("STB_PVRPlayFrameInc(%d)", path));

   if (STB_PVRIsPlayStarted(STB_DPGetPathAudioDecoder(path), STB_DPGetPathVideoDecoder(path)) == TRUE)
   {
      // request frame PVR trick mode
      STB_PVRPlayTrickMode(STB_DPGetPathAudioDecoder(path), STB_DPGetPathVideoDecoder(path), PLAY_TRICK_FRAME, 1);
   }

   FUNCTION_FINISH(STB_PVRPlayFrameInc);
}

/**
 *

 *
 * @brief   Decrements play by one frame - has no effect when decoding from demux.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *

 *
 */
void STB_PVRPlayFrameDec(U8BIT path)
{
   FUNCTION_START(STB_PVRPlayFrameDec);

   ASSERT(path < num_paths);

   STB_PVR_PRINT(("STB_PVRPlayFrameDec(%d)", path));

   if (STB_PVRIsPlayStarted(STB_DPGetPathAudioDecoder(path), STB_DPGetPathVideoDecoder(path)) == TRUE)
   {
      // request frame PVR trick mode
      STB_PVRPlayTrickMode(STB_DPGetPathAudioDecoder(path), STB_DPGetPathVideoDecoder(path), PLAY_TRICK_FRAME, -1);
   }

   FUNCTION_FINISH(STB_PVRPlayFrameDec);
}

/**
 *

 *
 * @brief   Sets play speed to medium (1x) - has no effect when decoding from demux.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *

 *
 */
void STB_PVRPlayMedium(U8BIT path)
{
   FUNCTION_START(STB_PVRPlayMedium);

   ASSERT(path < num_paths);

   STB_PVRPlayNormal(path);

   FUNCTION_FINISH(STB_PVRPlayMedium);
}

/**
 *

 *
 * @brief   Returns TRUE if play speed is medium (1x).
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   BOOLEAN.
 *
 */
BOOLEAN STB_PVRIsPlayMedium(U8BIT path)
{
   BOOLEAN ret_val;

   FUNCTION_START(STB_PVRIsPlayMedium);

   ASSERT(path < num_paths);

   ret_val = STB_PVRIsPlayNormal(path);

   FUNCTION_FINISH(STB_PVRIsPlayMedium);

   return(ret_val);
}

/*!**************************************************************************
 * @brief    Decrease play speed - has no effect when decoding from demux.
 * @param    path - ID of the decode path to use
 * @param    include_slow_speeds - selects whether slow motion speeds, >-100% and <100%, are included
 ****************************************************************************/
void STB_PVRPlaySlower(U8BIT path, BOOLEAN include_slow_speeds)
{
   S16BIT speed;
   U8BIT audio_decoder, video_decoder;

   FUNCTION_START(STB_PVRPlaySlower);

   ASSERT(path < num_paths);

   STB_PVR_PRINT(("STB_PVRPlaySlower(%d)", path));

   audio_decoder = STB_DPGetPathAudioDecoder(path);
   video_decoder = STB_DPGetPathVideoDecoder(path);

   if (STB_PVRIsPlayStarted(audio_decoder, video_decoder))
   {
      speed = STB_PVRGetPlaySpeed(audio_decoder, video_decoder);
      speed = STB_AVGetNextPlaySpeed(video_decoder, speed, -1, include_slow_speeds);
      STB_PVRSetPlaySpeed(audio_decoder, video_decoder, speed);
   }

   FUNCTION_FINISH(STB_PVRPlaySlower);
}

/**
 *

 *
 * @brief   Returns TRUE if play speed is slowest possible.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   BOOLEAN.
 *
 */
BOOLEAN STB_PVRIsPlaySlowest(U8BIT path)
{
   BOOLEAN ret_val = FALSE;
   U8BIT audio_decoder, video_decoder;

   FUNCTION_START(STB_PVRIsPlaySlowest);

   ASSERT(path < num_paths);

   audio_decoder = STB_DPGetPathAudioDecoder(path);
   video_decoder = STB_DPGetPathVideoDecoder(path);

   if (STB_PVRIsPlayStarted(audio_decoder, video_decoder))
   {
      if (STB_PVRGetPlaySpeed(audio_decoder, video_decoder) == STB_PVRGetMinPlaySpeed(path))
      {
         ret_val = TRUE;
      }
   }

   FUNCTION_FINISH(STB_PVRIsPlaySlowest);

   return(ret_val);
}

/*!**************************************************************************
 * @brief    Increase play speed - has no effect when decoding from demux.
 * @param    path - ID of the decode path to use
 * @param    include_slow_speeds - selects whether slow motion speeds, >-100% and <100%, are included
 ****************************************************************************/
void STB_PVRPlayFaster(U8BIT path, BOOLEAN include_slow_speeds)
{
   S16BIT speed;
   U8BIT audio_decoder, video_decoder;

   FUNCTION_START(STB_PVRPlayFaster);

   ASSERT(path < num_paths);

   STB_PVR_PRINT(("STB_PVRPlayFaster(%d)", path));

   audio_decoder = STB_DPGetPathAudioDecoder(path);
   video_decoder = STB_DPGetPathVideoDecoder(path);

   if (STB_PVRIsPlayStarted(audio_decoder, video_decoder))
   {
      /* Set play speed to the next fastest speed */
      speed = STB_PVRGetPlaySpeed(audio_decoder, video_decoder);
      speed = STB_AVGetNextPlaySpeed(video_decoder, speed, 1, include_slow_speeds);
      STB_PVRSetPlaySpeed(audio_decoder, video_decoder, speed);
   }

   FUNCTION_FINISH(STB_PVRPlayFaster);
}

/**
 *

 *
 * @brief   Returns TRUE if play speed is fastest possible.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   BOOLEAN.
 *
 */
BOOLEAN STB_PVRIsPlayFastest(U8BIT path)
{
   BOOLEAN ret_val = FALSE;
   U8BIT audio_decoder, video_decoder;

   FUNCTION_START(STB_PVRIsPlayFastest);

   ASSERT(path < num_paths);

   audio_decoder = STB_DPGetPathAudioDecoder(path);
   video_decoder = STB_DPGetPathVideoDecoder(path);

   if (STB_PVRIsPlayStarted(audio_decoder, video_decoder))
   {
      if (STB_PVRGetPlaySpeed(audio_decoder, video_decoder) == STB_PVRGetMaxPlaySpeed(path))
      {
         ret_val = TRUE;
      }
   }

   FUNCTION_FINISH(STB_PVRIsPlayFastest);

   return(ret_val);
}

/**
 *

 *
 * @brief   Reads the PVR min play speed.
 *
 * @param   U8BIT path - player number
 *
 * @return   S8BIT - speed.
 *
 */
S16BIT STB_PVRGetMinPlaySpeed(U8BIT path)
{
   S16BIT ret_val;

   FUNCTION_START(STB_PVRGetMinPlaySpeed);

   ASSERT(path < num_paths);

   ret_val = STB_AVGetMinPlaySpeed(STB_DPGetPathVideoDecoder(path));

   FUNCTION_FINISH(STB_PVRGetMinPlaySpeed);

   return(ret_val);
}

/**
 *

 *
 * @brief   Reads the PVR max play speed.
 *
 * @param   U8BIT path - player number
 *
 * @return   S8BIT - speed.
 *
 */
S16BIT STB_PVRGetMaxPlaySpeed(U8BIT path)
{
   S16BIT ret_val;

   FUNCTION_START(STB_PVRGetMaxPlaySpeed);

   ASSERT(path < num_paths);

   ret_val = STB_AVGetMaxPlaySpeed(STB_DPGetPathVideoDecoder(path));

   FUNCTION_FINISH(STB_PVRGetMaxPlaySpeed);

   return(ret_val);
}

/**
 *

 *
 * @brief   Returns disk space used in hours and minutes.
 *
 * @param   U8BIT* hours - returns num of hours
 * @param   U8BIT* mins - returns num of mins
 *

 *
 */
void STB_PVRDiskUsed(U16BIT disk_id, U8BIT *hours, U8BIT *mins)
{
   U32BIT temp_val;

   FUNCTION_START(STB_PVRDiskUsed);

   temp_val = STB_PVRGetTimeOfAllRecordings(disk_id);
   *hours = (U8BIT)(temp_val / 60);
   *mins = (U8BIT)(temp_val % 60);

   FUNCTION_FINISH(STB_PVRDiskUsed);
}

/**
 *

 *
 * @brief   Returns disk space free in hours and minutes.
 *
 * @param   U16BIT disk_id - ID of the disk being queried
 * @param   U8BIT* hours - returns num of hours
 * @param   U8BIT* mins - returns num of mins
 *

 *
 */
void STB_PVRDiskFree(U16BIT disk_id, U8BIT *hours, U8BIT *mins)
{
   U32BIT recording_time;
   U32BIT disk_used;
   U32BIT disk_free;
   U32BIT space_in_mins;
   U32BIT kb_per_min;

   FUNCTION_START(STB_PVRDiskFree);

   disk_used = STB_PVRGetSizeOfAllRecordings(disk_id);
   recording_time = STB_PVRGetTimeOfAllRecordings(disk_id);

   if ((disk_used == 0) || (recording_time == 0))
   {
      /* Estimate for the KB per minute */
      kb_per_min = 85 * 1024;
   }
   else
   {
      /* Calculate the KB/min for all current recordings */
      kb_per_min = disk_used / recording_time;
   }

   disk_free = STB_DSKGetSize(disk_id) - STB_DSKGetUsed(disk_id);
   space_in_mins = disk_free / kb_per_min;

   *hours = (U8BIT)(space_in_mins / 60);
   *mins = (U8BIT)(space_in_mins % 60);

   FUNCTION_FINISH(STB_PVRDiskFree);
}

/**
 *

 *
 * @brief   Returns disk size in MBytes.
 *
 * @param   U16BIT disk_id - ID of the disk being queried
 *
 * @return   U32BIT - MB.
 *
 */
U32BIT STB_PVRDiskSize(U16BIT disk_id)
{
   U32BIT ret_val;

   FUNCTION_START(STB_PVRDiskSize);

   ret_val = STB_DSKGetSize(disk_id) / 1024;

   FUNCTION_FINISH(STB_PVRDiskSize);

   return(ret_val);
}

/**
 * @brief   Creates a bookmark associated with the a recording.
 * @param   handle Recording handle
 * @param   time Time in seconds since the beginning of the recording
 * @param   name NULL terminated string representing the bookmark name, maximum length
 *          STB_PVR_NAME_LEN. If name is NULL, a string is formed to represent the bookmark time in
 *          the format hh:mm:ss
 * @return  TRUE if the bookmark was successfully created, FALSE otherwise.
 */
BOOLEAN STB_PVRCreateBookmark(U32BIT handle, U32BIT time, U8BIT *name)
{
   BOOLEAN retval;
   S_RECORDING *rec_ptr;

   FUNCTION_START(STB_PVRCreateBookmark);

   STB_PVR_PRINT(("STB_PVRCreateBookmark(handle = 0x%x, time = %d, name = %p", handle, time, name));
   rec_ptr = GetRecording(handle);
   if ((rec_ptr != NULL) && (strlen((char *)rec_ptr->basename) > 0))
   {
      retval = AddBookmark(rec_ptr->disk_id, rec_ptr->basename, time, name, BOOKMARK_USER);
   }
   else
   {
      STB_PVR_PRINT(("STB_PVRCreateBookmark: failed finding the recording"));
      retval = FALSE;
   }

   FUNCTION_FINISH(STB_PVRCreateBookmark);

   return retval;
}

/**
 * @brief   Deletes a bookmark associated with the a recording.
 * @param   handle Recording handle
 * @param   time Time in seconds since the beginning of the recording
 * @param   name NULL terminated string representing the bookmark name, maximum length
 *          STB_PVR_NAME_LEN. If name is NULL, a string is formed to represent the bookmark time in
 *          the format hh:mm:ss
 * @return  TRUE if the bookmark was successfully deleted, FALSE otherwise.
 */
BOOLEAN STB_PVRDeleteBookmark(U32BIT handle, U32BIT time, U8BIT *name)
{
   BOOLEAN retval;
   S_RECORDING *rec_ptr;

   FUNCTION_START(STB_PVRDeleteBookmark);

   STB_PVR_PRINT(("STB_PVRDeleteBookmark(handle = 0x%x, time = %d, name = %p", handle, time, name));
   rec_ptr = GetRecording(handle);
   if ((rec_ptr != NULL) && (strlen((char *)rec_ptr->basename) > 0))
   {
      RemoveBookmark(rec_ptr->disk_id, rec_ptr->basename, time, name, BOOKMARK_USER);
      retval = TRUE;
   }
   else
   {
      STB_PVR_PRINT(("STB_PVRDeleteBookmark: failed finding the recording"));
      retval = FALSE;
   }

   FUNCTION_FINISH(STB_PVRDeleteBookmark);

   return retval;
}


//---local function definitions------------------------------------------------

/*!**************************************************************************
 * @brief   Read all recording info from a given disk and create an entry in the recording list
 *          for each one. Recordings are only recognised in the disk's base/root directory.
 * @param   disk_id - disk to be read
 ****************************************************************************/
static void ReadRecordings(U16BIT disk_id)
{
   void *dir;
   U8BIT filename[DB_FILENAME_LEN];
   E_STB_DIR_ENTRY_TYPE entry_type;
   void *dbfile;
   S_RECORDING *rec_ptr;
   S_REC_HEADER header;
   S_REC_INFO rec_info;
   BOOLEAN result;

   FUNCTION_START(ReadRecordings);

   STB_PVR_PRINT(("ReadRecordings(0x%04x)", disk_id));

   /* Get a list of all the filenames in the base directory */
   dir = STB_DSKOpenDirectory(disk_id, (U8BIT *)"/");
   if (dir != NULL)
   {
      while (STB_DSKReadDirectory(dir, filename, sizeof(filename), &entry_type))
      {
         if (entry_type == DIR_ENTRY_FILE)
         {
            /* Look for recording database files */
            if ((strlen((char *)filename) > strlen(DB_FILE_EXTENSION)) &&
                (strlen((char *)filename) <= (STB_PVR_BASENAME_LEN + strlen(DB_FILE_EXTENSION))) &&
                (strcmp((char *)&filename[strlen((char *)filename) - strlen(DB_FILE_EXTENSION)], DB_FILE_EXTENSION) == 0))
            {
               /* Check the file is valid */
               dbfile = STB_DSKOpenFile(disk_id, filename, FILE_MODE_READ);
               if (dbfile != NULL)
               {
                  rec_ptr = NULL;

                  if (STB_DSKReadFile(dbfile, (U8BIT *)&header, (U32BIT)sizeof(S_REC_HEADER)) ==
                      (U32BIT)sizeof(S_REC_HEADER))
                  {
                     /* Check header */
                     if ((header.magic[0] == ODB_MAGIC_1) && (header.magic[1] == ODB_MAGIC_2) &&
                         (header.magic[2] == ODB_MAGIC_3))
                     {
                        result = FALSE;

                        /* Now read the basic recording info that is held in memory */
                        if ((header.version == ODB_VERSION_1) || (header.version == ODB_VERSION_2))
                        {
                           /* Data will be encrypted, so read and decrypt it */
                           if (STB_DSKReadFile(dbfile, rec_info_buffer, REC_INFO_BUFFER_SIZE) == REC_INFO_BUFFER_SIZE)
                           {
                              if ((enc_dec_key != NULL) && (enc_dec_iv != NULL))
                              {
                                 STB_CRYPTAesCbcDecrypt(rec_info_buffer, rec_info_buffer, REC_INFO_BUFFER_SIZE,
                                    enc_dec_key, enc_dec_iv);
                              }

                              /* Copy the decrypted data */
                              memcpy(&rec_info, rec_info_buffer, sizeof(S_REC_INFO));

                              if (header.version == ODB_VERSION_1)
                              {
                                 /* Version 1 files don't have status to indicate whether the
                                  * recording completed successfully or not, so check the recording
                                  * duration and set the status accordingly */
                                 if ((rec_info.len_hour == 0) && (rec_info.len_min == 0) &&
                                    (rec_info.len_sec == 0))
                                 {
                                    rec_info.status = RECORDING_STATUS_FAILED;
                                 }
                                 else
                                 {
                                    rec_info.status = RECORDING_STATUS_COMPLETED;
                                 }
                              }
                              else if (header.version == ODB_VERSION_2)
                              {
                                 if (rec_info.status == RECORDING_STATUS_STARTED)
                                 {
                                    /* Recording didn't complete */
                                    rec_info.status = RECORDING_STATUS_FAILED;
                                 }
                              }

                              result = TRUE;
                           }
                        }

                        if (result)
                        {
                           /* Get the basename for the recording by terminating the filename
                            * before the extension string */
                           filename[strlen((char *)filename) - strlen(DB_FILE_EXTENSION)] = '\0';

                           /* Check the platform to make sure this is a valid recording */
                           if (STB_PVRIsValidRecording(disk_id, filename))
                           {
                              /* Create a list entry for this recording */
                              rec_ptr = STB_GetMemory(sizeof(S_RECORDING));
                              if (rec_ptr != NULL)
                              {
                                 memset(rec_ptr, 0, sizeof(S_RECORDING));

                                 /* Copy the recording info into the list entry */
                                 memcpy(&rec_ptr->rec_info, &rec_info, sizeof(S_REC_INFO));

                                 /* Store the disk ID with the recording */
                                 rec_ptr->disk_id = disk_id;

                                 /* Store the base filename for this recording */
                                 strncpy((char *)rec_ptr->basename, (char *)filename, BASENAME_LEN);

                                 /* Add into the recording list according to the recording date/time */
                                 STB_OSMutexLock(list_mutex);

                                 /* Assign a unique handle to the recording */
                                 /* restore the recording handle from the basename */
                                 rec_ptr->handle = strtoul(filename, NULL, 16);
                                 if (NULL != GetRecording(rec_ptr->handle))
                                    rec_ptr->handle = GetNextFreeHandle();

                                 /* Find where this recording should be inserted into the recording list
                                  * based on the date & time of the recording */
                                 InsertRecordingInList(rec_ptr);

                                 STB_OSMutexUnlock(list_mutex);
                              }
                           }
                        }
                     }
                  }

                  STB_DSKCloseFile(dbfile);
               }
            }
         }
      }

      STB_DSKCloseDirectory(dir);
   }

   FUNCTION_FINISH(ReadRecordings);
}

/*!**************************************************************************
 * @brief   Remove all recordings from the database that are on a given disk id.
 *          This would be due to the disk being removed.
 * @param   disk_id - ID of disk
 ****************************************************************************/
static void RemoveRecordings(U16BIT disk_id)
{
   S_RECORDING *rec_ptr;
   S_RECORDING *next_rec;

   FUNCTION_START(RemoveRecordings);

   STB_OSMutexLock(list_mutex);

   /* Iterate through the database looking for any entries with the given disk ID */
   rec_ptr = rec_list;
   while (rec_ptr != NULL)
   {
      next_rec = rec_ptr->next;

      if (rec_ptr->disk_id == disk_id)
      {
         RemoveRecordingFromList(rec_ptr);
      }

      rec_ptr = next_rec;
   }

   STB_OSMutexUnlock(list_mutex);

   FUNCTION_FINISH(RemoveRecordings);
}

/*!**************************************************************************
 * @brief   Returns the current recording duration, in seconds, of the given recording handle
 * @param   handle - recording handle
 * @return  Recording time in seconds, 0 if not found.
 ****************************************************************************/
static U32BIT GetCurrentRecordTime(U32BIT handle)
{
   U16BIT path;
   U32BIT record_time;

   FUNCTION_START(GetCurrentRecordTime);

   record_time = 0;

   /* Find the recording status for the given recording handle */
   for (path = 0; path < num_paths; path++)
   {
      if (record_status[path].handle == handle)
      {
         if (record_status[path].paused)
         {
            record_time = record_status[path].duration;
         }
         else
         {
            record_time = record_status[path].duration + STB_OSGetClockDiff(record_status[path].start_time);
         }

         record_time /= 1000;
         break;
      }
   }

   FUNCTION_FINISH(GetCurrentRecordTime);

   return(record_time);
}

/*!**************************************************************************
 * @brief   Returns the next available handle value that isn't already used in the DB
 * @return  handle value
 ****************************************************************************/
static U32BIT GetNextFreeHandle(void)
{
   U32BIT handle = HANDLE_BASE;

   FUNCTION_START(GetNextFreeHandle);

   while (GetRecording(handle) != NULL)
   {
      handle++;
   }

   FUNCTION_FINISH(GetNextFreeHandle);

   return(handle);
}

/*!**************************************************************************
 * @brief   Finds recording entry for given handle
 * @param   handle - recording handle
 * @return  pointer to recording
 ****************************************************************************/
static S_RECORDING* GetRecording(U32BIT handle)
{
   S_RECORDING *rec_ptr;

   FUNCTION_START(GetRecording);

   rec_ptr = NULL;

   if (initialised_flag)
   {
      STB_OSMutexLock(list_mutex);

      for (rec_ptr = rec_list; (rec_ptr != NULL) && (rec_ptr->handle != handle); )
      {
         rec_ptr = rec_ptr->next;
      }

      STB_OSMutexUnlock(list_mutex);
   }

   FUNCTION_FINISH(GetRecording);

   return(rec_ptr);
}

/*!**************************************************************************
 * @brief   Constructs the base filename for a recording
 * @param   file_number - number to be used to create the filename
 * @param   filename - array in which the filename is returned
 ****************************************************************************/
static void GetDBRecordBasename(U32BIT file_number, U8BIT *filename)
{
   FUNCTION_START(GetDBRecordBasename);

   snprintf((char *)filename, BASENAME_LEN, "%08lx", (long)file_number);

   FUNCTION_FINISH(GetDBRecordBasename);
}

/*!**************************************************************************
 * @brief   Constructs the filename used to store the database info for a recording
 * @param   file_number - number to be used to create the filename
 * @param   filename - array in which the filename is returned
 ****************************************************************************/
static void GetDBRecordFilename(U32BIT file_number, U8BIT *filename)
{
   FUNCTION_START(GetDBRecordFilename);

   GetDBRecordBasename(file_number, filename);
   strncat((char *)filename, DB_FILE_EXTENSION, DB_FILE_EXTENTION_LEN);

   FUNCTION_FINISH(GetDBRecordFilename);
}

/*!**************************************************************************
 * @brief   Writes the various parts of the ODB file
 * @param   disk_id - ID of disk file is to be written to
 * @param   basename - name of file ODB name is to be based on
 * @param   write_header - TRUE if ODB header is to be written
 * @param   rec_info - standard recording info, can be NULL if not to be written
 * @return  TRUE if file is written successfully, FALSE otherwise
 ****************************************************************************/
static BOOLEAN WriteODBFile(U16BIT disk_id, U8BIT *basename, BOOLEAN write_header, S_REC_INFO *rec_info)
{
   U8BIT db_filename[DB_FILENAME_LEN];
   void *dbfile;
   S_REC_HEADER header;
   BOOLEAN retval;

   FUNCTION_START(WriteODBFile);

   retval = TRUE;

   strncpy((char *)db_filename, (char *)basename, DB_FILENAME_LEN);
   strncat((char *)db_filename, DB_FILE_EXTENSION, DB_FILENAME_LEN - (strlen((char *)basename)));

   if (STB_DSKFileExists(disk_id, db_filename))
   {
      dbfile = STB_DSKOpenFile(disk_id, db_filename, FILE_MODE_WRITE);
   }
   else
   {
      dbfile = STB_DSKOpenFile(disk_id, db_filename, FILE_MODE_OVERWRITE);
   }

   if (dbfile != NULL)
   {
      if (write_header)
      {
         /* Ensure the header is set */
         header.magic[0] = ODB_MAGIC_1;
         header.magic[1] = ODB_MAGIC_2;
         header.magic[2] = ODB_MAGIC_3;
         header.version = ODB_CUR_VERSION;

         if (STB_DSKWriteFile(dbfile, (U8BIT *)&header, (U32BIT)sizeof(S_REC_HEADER)) != sizeof(S_REC_HEADER))
         {
            retval = FALSE;
         }
      }
      else
      {
         /* Skip the header */
         retval = STB_DSKSeekFile(dbfile, FILE_POSITION_START, sizeof(S_REC_HEADER));
      }

      if (retval)
      {
         /* Encrypt the info before storing it */
         memcpy(rec_info_buffer, (U8BIT *)rec_info, sizeof(S_REC_INFO));
         memset(rec_info_buffer + sizeof(S_REC_INFO), 0, REC_INFO_BUFFER_SIZE - sizeof(S_REC_INFO));

         if ((enc_dec_key != NULL) && (enc_dec_iv != NULL))
         {
            STB_CRYPTAesCbcEncrypt(rec_info_buffer, rec_info_buffer, REC_INFO_BUFFER_SIZE, enc_dec_key, enc_dec_iv);
         }

         if (STB_DSKWriteFile(dbfile, rec_info_buffer, REC_INFO_BUFFER_SIZE) != REC_INFO_BUFFER_SIZE)
         {
            retval = FALSE;
         }
      }

      STB_DSKCloseFile(dbfile);
   }
#ifdef STB_DEBUG
   else
   {
      STB_PVR_PRINT(("WriteODBFile: Failed to open file \"%s\" on disk 0x%04x", db_filename, disk_id));
   }
#endif

   FUNCTION_FINISH(WriteODBFile);

   return(retval);
}

/*!**************************************************************************
 * @brief   Writes the extended info data to the end of the ODB file
 * @param   disk_id - ID of disk containing the file
 * @param   basename - base name of the ODB file
 * @param   ext_id - ID of the extended info
 * @param   data_len - number of bytes of data to be written
 * @param   data - extended info data
 * @return  TRUE if the data is written successfully, FALSE otherwise
 ****************************************************************************/
static BOOLEAN WriteExtendedInfo(U16BIT disk_id, U8BIT *basename, U16BIT ext_id,
   U32BIT data_len, U8BIT *data)
{
   U8BIT db_filename[DB_FILENAME_LEN];
   void *dbfile;
   S32BIT offset;
   U16BIT id;
   U32BIT size;
   BOOLEAN id_found;
   BOOLEAN retval;

   FUNCTION_START(WriteExtendedInfo);

   retval = FALSE;

   strncpy((char *)db_filename, (char *)basename, DB_FILENAME_LEN);
   strncat((char *)db_filename, DB_FILE_EXTENSION, DB_FILENAME_LEN - (strlen((char *)basename)));

   dbfile = STB_DSKOpenFile(disk_id, db_filename, FILE_MODE_WRITE);
   if (dbfile != NULL)
   {
      /* Move to the start of the extended info */
      offset = sizeof(S_REC_HEADER) + REC_INFO_BUFFER_SIZE;

      if (STB_DSKSeekFile(dbfile, FILE_POSITION_START, offset))
      {
         id_found = FALSE;

         /* Check to see if the extended info has already been written */
         while (STB_DSKReadFile(dbfile, (U8BIT *)&id, sizeof(id)) == sizeof(id))
         {
            if (STB_DSKReadFile(dbfile, (U8BIT *)&size, sizeof(size)) == sizeof(size))
            {
               if (id == ext_id)
               {
                  /* The extended info already exists */
                  id_found = TRUE;

                  if (size == data_len)
                  {
                     /* The data size hasn't changed so can be updated */
                     if (STB_DSKWriteFile(dbfile, data, data_len) == data_len)
                     {
                        retval = TRUE;
                     }
                  }
               }
               else
               {
                  /* Seek to the start of the next info id */
                  STB_DSKSeekFile(dbfile, FILE_POSITION_CURRENT, size);
               }
            }
         }

         if (!id_found)
         {
            /* This extended info doesn't exist or shouldn't be updated,
             * so write it to the end of the file */

            /* Write the ext id first */
            if (STB_DSKWriteFile(dbfile, (U8BIT *)&ext_id, sizeof(ext_id)) == sizeof(ext_id))
            {
               /* Now write the size of the extended item */
               if (STB_DSKWriteFile(dbfile, (U8BIT *)&data_len, sizeof(data_len)) == sizeof(data_len))
               {
                  /* Now write the actual data */
                  if (STB_DSKWriteFile(dbfile, data, data_len) == data_len)
                  {
                     retval = TRUE;
                  }
               }
            }
         }
      }

      STB_DSKCloseFile(dbfile);
   }

   FUNCTION_FINISH(WriteExtendedInfo);

   return(retval);
}

/*!**************************************************************************
 * @brief   Reads extended info from the ODB file
 * @param   disk_id - ID of disk containing the file to be read
 * @param   basename - base name of the ODB file
 * @param   ext_id - ID of the extended info to be read
 * @param   data - buffer into which data is to be read
 * @param   data_len - size of the provided buffer
 * @return  TRUE if the info is read, FALSE otherwise
 ****************************************************************************/
static BOOLEAN ReadExtendedInfo(U16BIT disk_id, U8BIT *basename, U16BIT ext_id, U8BIT *data,
   U32BIT data_len)
{
   U8BIT db_filename[DB_FILENAME_LEN];
   void *dbfile;
   S32BIT offset;
   U16BIT id;
   U32BIT size;
   BOOLEAN retval;

   FUNCTION_START(ReadExtendedInfo);

   retval = FALSE;

   strncpy((char *)db_filename, (char *)basename, DB_FILENAME_LEN);
   strncat((char *)db_filename, DB_FILE_EXTENSION, DB_FILENAME_LEN - strlen((char *)basename));

   dbfile = STB_DSKOpenFile(disk_id, db_filename, FILE_MODE_READ);
   if (dbfile != NULL)
   {
      /* Move to the start of the extended info */
      offset = sizeof(S_REC_HEADER) + REC_INFO_BUFFER_SIZE;

      if (STB_DSKSeekFile(dbfile, FILE_POSITION_START, offset))
      {
         /* Read each extended info item until the required one is found or there's no more */
         while (STB_DSKReadFile(dbfile, (U8BIT *)&id, sizeof(id)) == sizeof(id))
         {
            if (STB_DSKReadFile(dbfile, (U8BIT *)&size, sizeof(size)) == sizeof(size))
            {
               if (id == ext_id)
               {
                  if (data_len >= size)
                  {
                     /* Found the item we're looking for and there's enough space to read the data */
                     if (STB_DSKReadFile(dbfile, data, size) == size)
                     {
                        retval = TRUE;
                     }
                  }
                  break;
               }
               else
               {
                  /* Seek to the start of the next info id */
                  STB_DSKSeekFile(dbfile, FILE_POSITION_CURRENT, size);
               }
            }
         }
      }

      STB_DSKCloseFile(dbfile);
   }

   FUNCTION_FINISH(ReadExtendedInfo);

   return(retval);
}

/*!**************************************************************************
 * @brief   Finds the extended info in the ODB file and returns the data size
 * @param   disk_id - ID of disk containing the file to be read
 * @param   basename - base name of the ODB file
 * @param   ext_id - ID of the extended info to be read
 * @return  Size of the data for the extended info, or 0 if not found
 ****************************************************************************/
static U32BIT GetExtendedInfoSize(U16BIT disk_id, U8BIT *basename, U16BIT ext_id)
{
   U8BIT db_filename[DB_FILENAME_LEN];
   void *dbfile;
   S32BIT offset;
   U16BIT id;
   U32BIT size;
   U32BIT retval;

   FUNCTION_START(GetExtendedInfoSize);

   retval = 0;

   strncpy((char *)db_filename, (char *)basename, DB_FILENAME_LEN);
   strncat((char *)db_filename, DB_FILE_EXTENSION, DB_FILENAME_LEN - strlen((char *)basename));

   dbfile = STB_DSKOpenFile(disk_id, db_filename, FILE_MODE_READ);
   if (dbfile != NULL)
   {
      /* Move to the start of the extended info */
      offset = sizeof(S_REC_HEADER) + REC_INFO_BUFFER_SIZE;

      if (STB_DSKSeekFile(dbfile, FILE_POSITION_START, offset))
      {
         /* Read each extended info item until the required one is found or there's no more */
         while (STB_DSKReadFile(dbfile, (U8BIT *)&id, sizeof(id)) == sizeof(id))
         {
            if (STB_DSKReadFile(dbfile, (U8BIT *)&size, sizeof(size)) == sizeof(size))
            {
               if (id == ext_id)
               {
                  retval = size;
                  break;
               }
               else
               {
                  /* Seek to the start of the next info id */
                  STB_DSKSeekFile(dbfile, FILE_POSITION_CURRENT, size);
               }
            }
         }
      }

      STB_DSKCloseFile(dbfile);
   }

   FUNCTION_FINISH(GetExtendedInfoSize);

   return(retval);
}

/*!**************************************************************************
 * @brief   Inserts a recording into the recording list based on its date & time
 * @param   rec_ptr - recording to be inserted
 ****************************************************************************/
static void InsertRecordingInList(S_RECORDING *rec_ptr)
{
   S_RECORDING *prev;
   S_RECORDING *list_ptr;
   BOOLEAN found;

   FUNCTION_START(InsertRecordingInList);

   prev = NULL;
   found = FALSE;

   for (list_ptr = rec_list; (list_ptr != NULL) && !found; )
   {
      if (STB_GCCompareDateTime(list_ptr->rec_info.rec_date, list_ptr->rec_info.rec_hour,
             list_ptr->rec_info.rec_min, list_ptr->rec_info.rec_sec,
             rec_ptr->rec_info.rec_date, rec_ptr->rec_info.rec_hour,
             rec_ptr->rec_info.rec_min, rec_ptr->rec_info.rec_sec, COMP_1GT2))
      {
         prev = list_ptr;
         list_ptr = list_ptr->next;
      }
      else
      {
         found = TRUE;
      }
   }

   if (prev == NULL)
   {
      if (rec_list != NULL)
      {
         rec_list->prev = rec_ptr;
      }

      rec_ptr->prev = NULL;
      rec_ptr->next = rec_list;
      rec_list = rec_ptr;
   }
   else
   {
      rec_ptr->prev = prev;
      rec_ptr->next = list_ptr;
      prev->next = rec_ptr;

      if (list_ptr != NULL)
      {
         list_ptr->prev = rec_ptr;
      }
   }

   FUNCTION_FINISH(InsertRecordingInList);
}

/*!**************************************************************************
 * @brief   Unlinks a recording fro the list and frees all its memory
 * @param   rec_ptr - recording to be removed
 ****************************************************************************/
static void RemoveRecordingFromList(S_RECORDING *rec_ptr)
{
   FUNCTION_START(RemoveRecordingFromList);

   if (rec_list == rec_ptr)
   {
      rec_list = rec_ptr->next;
   }

   if (rec_ptr->prev != NULL)
   {
      rec_ptr->prev->next = rec_ptr->next;
   }

   if (rec_ptr->next != NULL)
   {
      rec_ptr->next->prev = rec_ptr->prev;
   }

   STB_FreeMemory(rec_ptr);

   FUNCTION_FINISH(RemoveRecordingFromList);
}

/*!**************************************************************************
 * @brief   Searches the given pid array for any video type pids
 * @param   pid_array - PID array to be searched
 * @param   num_pids - number of PIDs in the given array
 * @return  TRUE if a video PID is found, FALSE otherwise
 ****************************************************************************/
static BOOLEAN IncludesVideoPid(S_PVR_PID_INFO *pid_array, U16BIT num_pids)
{
   BOOLEAN found;
   U16BIT i;

   FUNCTION_START(IncludesVideoPid);

   found = FALSE;

   for (i = 0; (i < num_pids) && !found; i++)
   {
      if (pid_array[i].type == PVR_PID_TYPE_VIDEO)
      {
         found = TRUE;
      }
   }

   FUNCTION_FINISH(IncludesVideoPid);

   return(found);
}

#ifdef COMMON_INTERFACE
/*!**************************************************************************
 * @brief   Frees the list of CI+ items for a recording
 * @param   rec_ptr - recording
 ****************************************************************************/
static void FreeCIPlusItemList(U8BIT path)
{
   S_REC_CIPLUS_ITEM *item;
   S_REC_CIPLUS_ITEM *next_item;

   FUNCTION_START(FreeCIPlusItemList);

   for (item = play_status[path].item_list; item != NULL; item = next_item)
   {
      next_item = item->next;

      FreeCIPlusItem(item);
   }

   play_status[path].item_list = NULL;
   play_status[path].uri_item = NULL;
   play_status[path].licence_item = NULL;
   play_status[path].pin_item = NULL;

   FUNCTION_FINISH(FreeCIPlusItemList);
}

/*!**************************************************************************
 * @brief   Frees the list of CI+ items for a recording
 * @param   item - item to be freed
 ****************************************************************************/
static void FreeCIPlusItem(S_REC_CIPLUS_ITEM *item)
{
   FUNCTION_START(FreeCIPlusItem);

   if (item->item_type == EXT_INFO_CIPLUS_LICENCE)
   {
      if (item->u.licence.data != NULL)
      {
         STB_FreeMemory(item->u.licence.data);
      }
   }

   STB_FreeMemory(item);

   FUNCTION_FINISH(FreeCIPlusItem);
}

/*!**************************************************************************
 * @brief   Reads the CI+ related extedned info from an odb file and creates
 *          a list in memory for a given recording
 * @param   path - decode path
 * @param   rec_ptr - recording
 ****************************************************************************/
static void ReadCIPlusItems(U8BIT path, S_RECORDING *rec_ptr)
{
   U8BIT db_filename[DB_FILENAME_LEN];
   void *dbfile;
   S32BIT offset;
   U16BIT id;
   U32BIT size;
   S_REC_CIPLUS_ITEM *item;
   S_REC_CIPLUS_ITEM *prev_item;

   FUNCTION_START(ReadCIPlusItems);

   /* Free any existing items */
   FreeCIPlusItemList(path);

   strcpy((char *)db_filename, (char *)rec_ptr->basename);
   strcat((char *)db_filename, DB_FILE_EXTENSION);

   dbfile = STB_DSKOpenFile(rec_ptr->disk_id, db_filename, FILE_MODE_READ);
   if (dbfile != NULL)
   {
      /* Move to the start of the extended info */
      offset = sizeof(S_REC_HEADER) + REC_INFO_BUFFER_SIZE;

      if (STB_DSKSeekFile(dbfile, FILE_POSITION_START, offset))
      {
         prev_item = NULL;

         /* Check each extended info item to find the ones related to CI+ */
         while (STB_DSKReadFile(dbfile, (U8BIT *)&id, sizeof(id)) == sizeof(id))
         {
            item = NULL;

            if (STB_DSKReadFile(dbfile, (U8BIT *)&size, sizeof(size)) == sizeof(size))
            {
               if (id == EXT_INFO_CIPLUS_URI)
               {
                  /* Create a new item for the URI */
                  if ((item = (S_REC_CIPLUS_ITEM *)STB_GetMemory(sizeof(S_REC_CIPLUS_ITEM))) != NULL)
                  {
                     memset(item, 0, sizeof(S_REC_CIPLUS_ITEM));

                     item->item_type = id;

                     STB_DSKReadFile(dbfile, (U8BIT *)&item->timestamp, sizeof(U32BIT));
                     STB_DSKReadFile(dbfile, item->u.uri, CIP_URI_LEN);
                  }
               }
               else if (id == EXT_INFO_CIPLUS_LICENCE)
               {
                  /* Create a new item for the licence */
                  if ((item = (S_REC_CIPLUS_ITEM *)STB_GetMemory(sizeof(S_REC_CIPLUS_ITEM))) != NULL)
                  {
                     memset(item, 0, sizeof(S_REC_CIPLUS_ITEM));

                     item->item_type = id;
                     item->u.licence.used = FALSE;

                     STB_DSKReadFile(dbfile, (U8BIT *)&item->timestamp, sizeof(U32BIT));

                     /* Length of the licence is all the data excluing the timestamp */
                     item->u.licence.length = size - sizeof(U32BIT);

                     if ((item->u.licence.data = STB_GetMemory(item->u.licence.length)) != NULL)
                     {
                        STB_DSKReadFile(dbfile, item->u.licence.data, item->u.licence.length);
                     }
                  }
               }
               else if (id == EXT_INFO_CIPLUS_PIN)
               {
                  /* Create a new item for the pin event */
                  if ((item = (S_REC_CIPLUS_ITEM *)STB_GetMemory(sizeof(S_REC_CIPLUS_ITEM))) != NULL)
                  {
                     memset(item, 0, sizeof(S_REC_CIPLUS_ITEM));

                     item->item_type = id;

                     STB_DSKReadFile(dbfile, (U8BIT *)&item->timestamp, sizeof(U32BIT));
                     STB_DSKReadFile(dbfile, (U8BIT *)&item->u.pin, sizeof(item->u.pin));
                  }
               }
               else
               {
                  /* Seek to the start of the next info id */
                  STB_DSKSeekFile(dbfile, FILE_POSITION_CURRENT, size);
               }

               if (item != NULL)
               {
                  /* Add the item to the recording's item list */
                  item->prev = prev_item;

                  if (prev_item == NULL)
                  {
                     play_status[path].item_list = item;
                  }
                  else
                  {
                     prev_item->next = item;
                  }

                  prev_item = item;
               }
            }
         }
      }

      STB_DSKCloseFile(dbfile);
   }

   FUNCTION_FINISH(ReadCIPlusItems);
}

/*!**************************************************************************
 * @brief   Checks the CI+ items for a recording and applies any that should be used
 *          at the given position. It's assumed that items are held in timestamp order,
 *          such that current timestamp <= next timestamp.
 * @param   path - decode path
 * @param   rec_ptr - recording
 * @param   position_in_seconds - position in the recording
 * @param   speed - playback speed, only used to determine direction of playback
 ****************************************************************************/
static void ApplyCIPlusItems(U8BIT path, S_RECORDING *rec_ptr, U32BIT position_in_seconds,
   S16BIT speed)
{
   S_REC_CIPLUS_ITEM *item;
   S_REC_CIPLUS_ITEM *found_item;
   U32BIT retention_limit;
   U8BIT uri[CIP_URI_LEN];

   FUNCTION_START(ApplyCIPlusItems);

   /* Find the URI that should be applied for the given position in the recording */
   found_item = NULL;

   if (((item = play_status[path].uri_item) == NULL) || (speed >= 0))
   {
      if (item == NULL)
      {
         /* No current URI, so work forwards through the item list to find the URI that
          * precedes the position and so should be applied. This will be true irrespective
          * of the direction of playback at the start */
         item = play_status[path].item_list;
      }

      for (; item != NULL; item = item->next)
      {
         if (item->timestamp <= position_in_seconds)
         {
            if (item->item_type == EXT_INFO_CIPLUS_URI)
            {
               if ((found_item == NULL) || (item->timestamp > found_item->timestamp))
               {
                  found_item = item;
               }
            }
         }
         else
         {
            /* All items from now on will occur later in the recording so can be ignored */
            break;
         }
      }
   }
   else
   {
      /* Backwards */
      for (item = item->prev; item != NULL; item = item->prev)
      {
         if (item->item_type == EXT_INFO_CIPLUS_URI)
         {
            /* Found the previous URI */
            found_item = item;
            break;
         }
      }
   }

   /* Check if there's a URI item and if it's changed */
   if (found_item == NULL)
   {
      DBG_CIP(": No URI found, so applying default URI for playback")

      /* No URI, so need to apply the default URI */
      STB_CiCcGetDefaultUsageRulesInfo(uri);

      /* Set retention limit for playback */
      retention_limit = STB_CiCcGetRetentionLimit(uri);

      STB_PVRPlaySetRetentionLimit(STB_DPGetPathAudioDecoder(path), STB_DPGetPathVideoDecoder(path),
         retention_limit, rec_ptr->rec_info.rec_date, rec_ptr->rec_info.rec_hour,
         rec_ptr->rec_info.rec_min);

      /* Apply the URI */
      STB_CiCcApplyUsageRulesInfoForPlayback(uri);
   }
   else if ((found_item != NULL) && (found_item != play_status[path].uri_item))
   {
      DBG_CIP(": Applying URI at %lu seconds for playback", found_item->timestamp)

      /* Save the URI item so it can be updated during playback */
      play_status[path].uri_item = found_item;

      /* Set retention limit for playback */
      retention_limit = STB_CiCcGetRetentionLimit(found_item->u.uri);
      STB_PVRPlaySetRetentionLimit(STB_DPGetPathAudioDecoder(path), STB_DPGetPathVideoDecoder(path),
         retention_limit, rec_ptr->rec_info.rec_date, rec_ptr->rec_info.rec_hour,
         rec_ptr->rec_info.rec_min);

      /* Apply the URI */
      STB_CiCcApplyUsageRulesInfoForPlayback(found_item->u.uri);
   }

   /* Is there a licence that should be sent to the CAM for this position in the recording? */
   found_item = NULL;

   if (((item = play_status[path].licence_item) == NULL) || (speed >= 0))
   {
      if (item == NULL)
      {
         /* No current licence, so work forwards through the item list to find the licence that
          * precedes the position and so should be applied. This will be true irrespective
          * of the direction of playback at the start */
         item = play_status[path].item_list;
      }

      for (; item != NULL; item = item->next)
      {
         if (item->timestamp <= position_in_seconds)
         {
            if (item->item_type == EXT_INFO_CIPLUS_LICENCE)
            {
               if (!item->u.licence.used)
               {
                  if ((found_item == NULL) || (item->timestamp > found_item->timestamp))
                  {
                     found_item = item;
                  }
               }
            }
            else if ((found_item != NULL) && (item->item_type == EXT_INFO_CIPLUS_URI))
            {
               /* A licence is valid upto the next URI or licence - see 5.12 of CI+ 1.3.1 spec */
               if (item->timestamp > found_item->timestamp)
               {
                  /* This URI cancels the licence */
                  found_item = NULL;
               }
            }
         }
         else
         {
            /* All items from now on will occur later in the recording so can be ignored */
            break;
         }
      }
   }
   else
   {
      /* Backwards */
      for (item = item->prev; item != NULL; item = item->prev)
      {
         if ((item->item_type == EXT_INFO_CIPLUS_LICENCE) && !item->u.licence.used)
         {
            /* Found the previous licence */
            found_item = item;
            break;
         }
      }
   }

   /* Check if there's a licence item and if it's changed */
   if ((found_item != NULL) && (found_item != play_status[path].licence_item))
   {
      DBG_CIP(": Sending licence at %lu seconds to CAM in slot %u",
               found_item->timestamp, play_status[path].slot_id)

      /* Save the licence item so it can be updated during playback */
      play_status[path].licence_item = found_item;

      /* Send the licence to the CAM so it can be updated */
      if (STB_CiCcSendPlaybackLicence(play_status[path].slot_id, rec_ptr->rec_info.serv_id,
             found_item->u.licence.data, found_item->u.licence.length))
      {
         /* Licences should only be sent to the CAM once during playback, so mark as used */
         found_item->u.licence.used = TRUE;
      }
      else
      {
         play_status[path].licence_item = NULL;
         DBG_CIP(": Failed to send CI+ licence for recording 0x%08lx to slot %u", rec_ptr->handle, play_status[path].slot_id)
      }
   }

   /* Find the pin that should be applied for the given position in the recording */
   found_item = NULL;

   if (((item = play_status[path].pin_item) == NULL) || (speed >= 0))
   {
      if (item == NULL)
      {
         /* No current pin, so work forwards through the item list to find the pin that
          * precedes the position and so should be applied. This will be true irrespective
          * of the direction of playback at the start */
         item = play_status[path].item_list;
      }

      for (; item != NULL; item = item->next)
      {
         if (item->timestamp <= position_in_seconds)
         {
            if (item->item_type == EXT_INFO_CIPLUS_PIN)
            {
               if ((found_item == NULL) || (item->timestamp > found_item->timestamp))
               {
                  found_item = item;
               }
            }
         }
         else
         {
            /* All items from now on will occur later in the recording so can be ignored */
            break;
         }
      }
   }
   else
   {
      /* Backwards */
      for (item = item->prev; item != NULL; item = item->prev)
      {
         if (item->item_type == EXT_INFO_CIPLUS_PIN)
         {
            /* Found the previous URI */
            found_item = item;
            break;
         }
      }
   }

   /* Check if there's a pin item and if it's changed */
   if ((found_item != NULL) && (found_item != play_status[path].pin_item))
   {
      DBG_CIP(": Applying PIN at %lu seconds for playback", found_item->timestamp)

      /* Save the pin item so it can be updated during playback */
      play_status[path].pin_item = found_item;

      /* Send an event to blank the video and pass the pin event to the CAM.
       * The video will be unblanked when a reply is received for the pin */
      STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_PLAYBACK_BLANK_VIDEO, NULL, 0);
      STB_CiCcSendPinPlayback(play_status[path].slot_id, found_item->u.pin.age_rating,
         found_item->u.pin.private_data);
   }

   FUNCTION_FINISH(ApplyCIPlusItems);
}

/*!**************************************************************************
 * @brief   Finds the CI+ item after the given position in seconds that needs to be applied
 *          during the playback. This time is passed to the platform so that it can provide a
 *          notification when this position is reached and the item can be applied.
 * @param   path - decode path
 * @param   speed - playback speed, only used to determine direction of playback
 ****************************************************************************/
static void FindNextCIPlusItem(U8BIT path, S16BIT speed)
{
   S_REC_CIPLUS_ITEM *item;
   S_REC_CIPLUS_ITEM *next_item;
   U8BIT audio_decoder, video_decoder;

   FUNCTION_START(FindNextCIPlusItem);

   next_item = NULL;

   /* Find the next URI, licence and pin items to be applied */
   for (item = play_status[path].item_list; item != NULL; item = item->next)
   {
      if (item->item_type == EXT_INFO_CIPLUS_URI)
      {
         /* Check whether this is the next URI item */
         if (speed >= 0)
         {
            /* Play forwards */
            if (((play_status[path].uri_item == NULL) || (item->timestamp > play_status[path].uri_item->timestamp)) &&
                ((next_item == NULL) || (item->timestamp < next_item->timestamp)))
            {
               next_item = item;
            }
         }
         else
         {
            /* Play backwards */
            if (((play_status[path].uri_item == NULL) || (item->timestamp < play_status[path].uri_item->timestamp)) &&
                ((next_item == NULL) || (item->timestamp > next_item->timestamp)))
            {
               next_item = item;
            }
         }
      }

      if ((item->item_type == EXT_INFO_CIPLUS_LICENCE) && !item->u.licence.used)
      {
         /* Check whether this is the next licence item */
         if (speed >= 0)
         {
            /* Play forwards */
            if (((play_status[path].licence_item == NULL) || (item->timestamp > play_status[path].licence_item->timestamp)) &&
                ((next_item == NULL) || (item->timestamp < next_item->timestamp)))
            {
               next_item = item;
            }
         }
         else
         {
            /* Play backwards */
            if (((play_status[path].licence_item == NULL) || (item->timestamp < play_status[path].licence_item->timestamp)) &&
                ((next_item == NULL) || (item->timestamp > next_item->timestamp)))
            {
               next_item = item;
            }
         }
      }

      if (item->item_type == EXT_INFO_CIPLUS_PIN)
      {
         /* Check whether this is the next pin item */
         if (speed >= 0)
         {
            /* Play forwards */
            if (((play_status[path].pin_item == NULL) || (item->timestamp > play_status[path].pin_item->timestamp)) &&
                ((next_item == NULL) || (item->timestamp < next_item->timestamp)))
            {
               next_item = item;
            }
         }
         else
         {
            /* Play backwards */
            if (((play_status[path].pin_item == NULL) || (item->timestamp < play_status[path].pin_item->timestamp)) &&
                ((next_item == NULL) || (item->timestamp > next_item->timestamp)))
            {
               next_item = item;
            }
         }
      }
   }

   if (next_item != NULL)
   {
      DBG_CIP(": Next item is a change of %s @ %lu seconds",
               ((next_item->item_type == EXT_INFO_CIPLUS_URI) ? "URI" :
                (next_item->item_type == EXT_INFO_CIPLUS_LICENCE) ? "LICENCE" : "PIN"), next_item->timestamp)

      /* Set the time the next item needs to be applied */
      audio_decoder = STB_DPGetPathAudioDecoder(path);
      video_decoder = STB_DPGetPathVideoDecoder(path);
      STB_PVRSetPlaybackNotifyTime(audio_decoder, video_decoder, next_item->timestamp);
   }

   FUNCTION_FINISH(FindNextCIPlusItem);
}

/*!**************************************************************************
 * @brief   Writes the CI+ info as an extended info item to the end of the ODB file
 * @param   disk_id - ID of disk containing the file
 * @param   basename - base name of the ODB file
 * @param   ext_id - ID of the extended info
 * @param   timestamp - timestamp to be written as part of the data
 * @param   data_len - number of bytes of data to be written
 * @param   data - extended info data
 * @return  TRUE if the data is written successfully, FALSE otherwise
 ****************************************************************************/
static BOOLEAN WriteCIPlusInfo(U16BIT disk_id, U8BIT *basename, U16BIT ext_id, U32BIT timestamp,
   U32BIT data_len, U8BIT *data)
{
   U8BIT db_filename[DB_FILENAME_LEN];
   void *dbfile;
   S32BIT offset;
   U16BIT id;
   U32BIT size;
   BOOLEAN retval;

   FUNCTION_START(WriteCIPlusInfo);

   retval = FALSE;

   strcpy((char *)db_filename, (char *)basename);
   strcat((char *)db_filename, DB_FILE_EXTENSION);

   dbfile = STB_DSKOpenFile(disk_id, db_filename, FILE_MODE_WRITE);
   if (dbfile != NULL)
   {
      /* Move to the start of the extended info */
      offset = sizeof(S_REC_HEADER) + REC_INFO_BUFFER_SIZE;

      if (STB_DSKSeekFile(dbfile, FILE_POSITION_START, offset))
      {
         /* Move to the end of the last written item */
         while (STB_DSKReadFile(dbfile, (U8BIT *)&id, sizeof(id)) == sizeof(id))
         {
            if (STB_DSKReadFile(dbfile, (U8BIT *)&size, sizeof(size)) == sizeof(size))
            {
               /* Seek to the start of the next info id */
               STB_DSKSeekFile(dbfile, FILE_POSITION_CURRENT, size);
            }
         }

         /* Write the ext id first */
         if (STB_DSKWriteFile(dbfile, (U8BIT *)&ext_id, sizeof(ext_id)) == sizeof(ext_id))
         {
            /* Now write the size of the extended item */
            size = data_len + sizeof(timestamp);

            if (STB_DSKWriteFile(dbfile, (U8BIT *)&size, sizeof(size)) == sizeof(size))
            {
               /* Write the timestamp for the item */
               if (STB_DSKWriteFile(dbfile, (U8BIT *)&timestamp, sizeof(timestamp)) == sizeof(timestamp))
               {
                  /* Now write the actual data */
                  if (STB_DSKWriteFile(dbfile, data, data_len) == data_len)
                  {
                     retval = TRUE;
                  }
               }
            }
         }
      }

      STB_DSKCloseFile(dbfile);
   }

   FUNCTION_FINISH(WriteCIPlusInfo);

   return(retval);
}

/*!**************************************************************************
 * @brief   Updates and existing CI+ extended info item in place in the ODB file
 * @param   disk_id - ID of disk containing the file
 * @param   basename - base name of the ODB file
 * @param   ext_id - ID of the extended info
 * @param   timestamp - timestamp of the item to be updated
 * @param   data_len - number of bytes of data to be written
 * @param   data - extended info data
 * @return  TRUE if the data is written successfully, FALSE otherwise
 ****************************************************************************/
static BOOLEAN UpdateCIPlusInfo(U16BIT disk_id, U8BIT *basename, U16BIT ext_id, U32BIT timestamp,
   U32BIT data_len, U8BIT *data)
{
   U8BIT db_filename[DB_FILENAME_LEN];
   void *dbfile;
   S32BIT offset;
   U16BIT id;
   U32BIT size;
   U32BIT file_time;
   BOOLEAN retval;

   FUNCTION_START(UpdateCIPlusInfo);

   retval = FALSE;

   strcpy((char *)db_filename, (char *)basename);
   strcat((char *)db_filename, DB_FILE_EXTENSION);

   dbfile = STB_DSKOpenFile(disk_id, db_filename, FILE_MODE_WRITE);
   if (dbfile != NULL)
   {
      /* Move to the start of the extended info */
      offset = sizeof(S_REC_HEADER) + REC_INFO_BUFFER_SIZE;

      if (STB_DSKSeekFile(dbfile, FILE_POSITION_START, offset))
      {
         /* Find the extended info item to be updated */
         while (STB_DSKReadFile(dbfile, (U8BIT *)&id, sizeof(id)) == sizeof(id))
         {
            if (STB_DSKReadFile(dbfile, (U8BIT *)&size, sizeof(size)) == sizeof(size))
            {
               if (id == ext_id)
               {
                  if (STB_DSKReadFile(dbfile, (U8BIT *)&file_time, sizeof(file_time)) == sizeof(file_time))
                  {
                     if (file_time == timestamp)
                     {
                        /* Found the item to be updated */
                        if (size == (data_len + sizeof(timestamp)))
                        {
                           /* The data size hasn't changed so can be updated */
                           if (STB_DSKWriteFile(dbfile, data, data_len) == data_len)
                           {
                              retval = TRUE;
                              break;
                           }
                        }
                     }
                     else
                     {
                        /* Seek to the start of the next info id - timestamp has already been read */
                        STB_DSKSeekFile(dbfile, FILE_POSITION_CURRENT, size - sizeof(timestamp));
                     }
                  }
               }
               else
               {
                  /* Seek to the start of the next info id */
                  STB_DSKSeekFile(dbfile, FILE_POSITION_CURRENT, size);
               }
            }
         }
      }

      STB_DSKCloseFile(dbfile);
   }

   FUNCTION_FINISH(UpdateCIPlusInfo);

   return(retval);
}

#endif /* COMMON_INTERFACE */

/**
 * @brief   Constructs the name of the bookmark folder for a recording, basing it on its ODB file
 *          basename.
 * @param   basename base name of the ODB file.
 * @param   bookmark_folder_name Array in which the folder name is returned. This array must be
 *          allocated with at least STB_PVR_BASENAME_LEN + 2 bytes
 */
static void GetBookmarkFolderName(U8BIT *basename, U8BIT *bookmark_folder_name)
{
   FUNCTION_START(GetBookmarkFolderName);

   sprintf((char *)bookmark_folder_name, "bm%s", (char *)basename);

   FUNCTION_FINISH(GetBookmarkFolderName);
}

/**
 * @brief   Constructs the file name for a bookmark with the given time and type. The first part of
 *          the file name is a hexadecimal number representing the bookmark time, the last
 *          character represents the bookmark type.
 * @param   time Bookmark time
 * @param   type Bookmark type
 * @param   name Buffer where the file name will be returned. It needs to be allocated with at least
 *          10 bytes: 8 bytes for the hexadecimal characters representing the time, 1 for the ending
 *          character representing the type and one for the string terminator
 */
static void GetBookmarkFileName(U32BIT time, E_BOOKMARK_TYPE type, U8BIT *name)
{
   char tail;

   FUNCTION_START(GetBookmarkFileName);

   switch (type)
   {
      case BOOKMARK_PLAYBACK_POSITION:
      {
         tail = 'p';
         break;
      }

      case BOOKMARK_USER:
      default:
      {
         tail = 'u';
         break;
      }
   }

   sprintf((char *)name, "%08x%c", (unsigned int)time, tail);

   FUNCTION_FINISH(GetBookmarkFileName);
}

/**
 * @brief   Returns the type of a bookmark given its file name
 * @brief   file_name Bookmark file name
 * @return  Bookmark type
 */
static E_BOOKMARK_TYPE GetBookmarkType(U8BIT *file_name)
{
   E_BOOKMARK_TYPE type;

   FUNCTION_START(GetBookmarkType);

   switch (file_name[BOOKMARK_FILENAME_SIZE - 2])
   {
      case 'p':
      {
         type = BOOKMARK_PLAYBACK_POSITION;
         break;
      }

      case 'u':
      default:
      {
         type = BOOKMARK_USER;
         break;
      }
   }

   FUNCTION_FINISH(GetBookmarkType);

   return type;
}

/**
 * @brief   Returns the time of a bookmark given its file name
 * @brief   file_name Bookmark file name
 * @return  Bookmark time
 */
static U32BIT GetBookmarkTime(U8BIT *file_name)
{
   U32BIT time;

   FUNCTION_START(GetBookmarkTime);

   sscanf((char *)file_name, "%08x%*c", (unsigned int *)&time);

   FUNCTION_FINISH(GetBookmarkTime);

   return time;
}

/**
 * @brief   Creates a bookmark associated with the a recording. A bookmark name is constructed
 *          from its time and type, see GetBookmarkFileName
 * @param   disk_id ID of disk containing the file
 * @param   basename base name of the ODB file. The name of the folder containing the bookmarks is
 *          formed using the ODB file base name, see GetBookmarkFolderName.
 * @param   time Time in seconds since the beginning of the recording
 * @param   name NULL terminated string representing the bookmark name, maximum length
 *          STB_PVR_NAME_LEN. If name is NULL, a string is formed to represent the bookmark time in
 *          the format hh:mm:ss
 * @param   type Bookmark type
 * @return  TRUE if the bookmark has been created, FALSE otherwise
 */
static BOOLEAN AddBookmark(U16BIT disk_id, U8BIT *basename, U32BIT time, U8BIT *name, E_BOOKMARK_TYPE type)
{
   BOOLEAN success = TRUE, check_folder = TRUE;
   void *dir;
   U8BIT folder_name[STB_PVR_BASENAME_LEN + 2];
   U8BIT file_name[BOOKMARK_FILENAME_SIZE];
   U8BIT file_str[STB_PVR_BASENAME_LEN + 2 + BOOKMARK_FILENAME_SIZE];
   U8BIT default_name[9];
   void *file;
   U32BIT size;

   FUNCTION_START(AddBookmark);

   GetBookmarkFolderName(basename, folder_name);
   GetBookmarkFileName(time, type, file_name);
   sprintf((char *)file_str, "%s/%.*s", (char *)folder_name, BOOKMARK_FILENAME_SIZE - 1,
      (char *)file_name);

   if (STB_DSKFileExists(disk_id, file_str))
   {
      check_folder = FALSE; /* If file exists, folder exists, no need to check */
      if (type == BOOKMARK_USER)
      {
         /* User bookmarks will not be overwritten */
         STB_PVR_PRINT(("AddBookmark: bookmark %s already exists", file_str));
         success = FALSE;
      }
   }

   if (check_folder)
   {
      /* Create the bookmark folder if it does not exist */
      dir = STB_DSKOpenDirectory(disk_id, folder_name);
      if (dir == NULL)
      {
         STB_PVR_PRINT(("AddBookmark: creating folder %s", folder_name));
         if (!STB_DSKCreateDirectory(disk_id, folder_name))
         {
            STB_PVR_PRINT(("AddBookmark: failed creating folder %s", folder_name));
            success = FALSE;
         }
      }
      else
      {
         STB_DSKCloseDirectory(dir);
      }
   }

   if (success)
   {
      STB_PVR_PRINT(("AddBookmark: creating bookmark %s", file_str));
      file = STB_DSKOpenFile(disk_id, file_str, FILE_MODE_OVERWRITE);
      if (file != NULL)
      {
         if (name == NULL)
         {
            U8BIT h, m, s;

            m = time / 60;
            s = time % 60;
            h = m / 60;
            m = m % 60;

            sprintf((char *)default_name, "%02d:%02d:%02d", h, m, s);
            name = default_name;
            size = 8;
         }
         else
         {
            size = strlen((char *)name);
            if (size > STB_PVR_NAME_LEN - 1)
            {
               STB_PVR_PRINT(("AddBookmark: name is too long, limiting it to %d characters", STB_PVR_NAME_LEN));
               size = STB_PVR_NAME_LEN - 1;
            }
         }

         if (STB_DSKWriteFile(file, name, size) != size)
         {
            success = FALSE;
         }

         STB_DSKCloseFile(file);

         if (!success)
         {
            STB_PVR_PRINT(("AddBookmark: failed writing bookmark %s", file_str));
            STB_DSKDeleteFile(disk_id, file_str);
         }
      }
#ifdef STB_DEBUG
      else
      {
         STB_PVR_PRINT(("AddBookmark: could not create file %s", file_str));
      }
#endif
   }

   FUNCTION_FINISH(AddBookmark);

   return success;
}

/**
 * @brief   Removes a bookmark associated with the a recording
 * @param   disk_id ID of disk containing the file
 * @param   basename Base name of the ODB file. The name of the folder containing the bookmarks is
 *          formed using the ODB file base name, see GetBookmarkFolderName.
 * @param   time Time in seconds since the beginning of the recording
 * @param   name NULL terminated string representing the bookmark name. Not used for now, but
 *          might be useful in the future.
 * @param   type Bookmark type
 */
static void RemoveBookmark(U16BIT disk_id, U8BIT *basename, U32BIT time, U8BIT *name, E_BOOKMARK_TYPE type)
{
   U8BIT folder_name[STB_PVR_BASENAME_LEN + 2];
   U8BIT file_name[BOOKMARK_FILENAME_SIZE];
   U8BIT file_str[STB_PVR_BASENAME_LEN + 2 + BOOKMARK_FILENAME_SIZE];

   FUNCTION_START(RemoveBookmark);
   USE_UNWANTED_PARAM(name);

   GetBookmarkFolderName(basename, folder_name);
   GetBookmarkFileName(time, type, file_name);
   sprintf((char *)file_str, "%s/%.*s", (char *)folder_name, BOOKMARK_FILENAME_SIZE - 1,
      (char *)file_name);

   STB_PVR_PRINT(("Deleting bookmark %s", file_str));
   STB_DSKDeleteFile(disk_id, file_str);

   FUNCTION_FINISH(RemoveBookmark);
}

/**
 * @brief   Deletes all bookmarks from a recording
 * @param   disk_id ID of disk containing the recording
 * @param   basename base name of the ODB file. The name of the folder containing the bookmarks is
 *          formed using the ODB file base name, see GetBookmarkFolderName.
 */
static void RemoveAllBookmarks(U16BIT disk_id, U8BIT *basename)
{
   CREATE_LINK_LIST_HEADER(bookmarks);
   S_BOOKMARK *b_ptr;
   U8BIT folder_name[STB_PVR_BASENAME_LEN + 2];

   FUNCTION_START(RemoveAllBookmarks);

   if (GetBookmarks(disk_id, basename, &bookmarks, BOOKMARK_USER, FALSE) != 0)
   {
      b_ptr = (S_BOOKMARK *)STB_LLGetFirstBlock(&bookmarks);
      while (b_ptr != NULL)
      {
         RemoveBookmark(disk_id, basename, b_ptr->time, NULL, BOOKMARK_USER);
         b_ptr = (S_BOOKMARK*)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)b_ptr);
      }

      EmptyBookmarks(&bookmarks, TRUE);
   }

   if (GetBookmarks(disk_id, basename, &bookmarks, BOOKMARK_PLAYBACK_POSITION, FALSE) != 0)
   {
      b_ptr = (S_BOOKMARK *)STB_LLGetFirstBlock(&bookmarks);
      while (b_ptr != NULL)
      {
         RemoveBookmark(disk_id, basename, b_ptr->time, NULL, BOOKMARK_PLAYBACK_POSITION);
         b_ptr = (S_BOOKMARK*)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)b_ptr);
      }

      EmptyBookmarks(&bookmarks, TRUE);
   }

   GetBookmarkFolderName(basename, folder_name);
   STB_DSKDeleteDirectory(disk_id, folder_name);

   FUNCTION_FINISH(RemoveAllBookmarks);
}

/**
 * @brief   For a given pair of disk ID and basename, this function returns an array of bookmarks
 * @param   disk_id ID of disk containing the file
 * @param   basename base name of the ODB file. The name of the folder containing the bookmarks is
 *          formed using the ODB file base name, see GetBookmarkFolderName.
 * @param   bookmarks Linked list of returned S_BOOKMARK items, this needs to be emptied with
 *          EmptyBookmarks.
 * @param   type Bookmark type
 * @param   names If TRUE the string name in the S_BOOKMARK structure will be populated, otherwise
 *          its value will be NULL
 * @return  Number of elements in the returned arrays.
 */
static U32BIT GetBookmarks(U16BIT disk_id, U8BIT *basename, LINK_LIST_HEADER *bookmarks,
   E_BOOKMARK_TYPE type, BOOLEAN names)
{
   U32BIT num;
   void *dir;
   U8BIT folder_name[STB_PVR_BASENAME_LEN + 2];
   U8BIT file_name[BOOKMARK_FILENAME_SIZE];
   U8BIT file_str[STB_PVR_BASENAME_LEN + 2 + BOOKMARK_FILENAME_SIZE];
   void *file;
   U32BIT read_bytes, offset;
   S_BOOKMARK *b;
   E_STB_DIR_ENTRY_TYPE entry_type;

   FUNCTION_START(GetBookmarks);

   num = 0;
   GetBookmarkFolderName(basename, folder_name);
   dir = STB_DSKOpenDirectory(disk_id, folder_name);
   if (dir != NULL)
   {
      read_bytes = 0;
      while (STB_DSKReadDirectory(dir, file_name, BOOKMARK_FILENAME_SIZE, &entry_type))
      {
         if ((entry_type == DIR_ENTRY_FILE) && (GetBookmarkType(file_name) == type))
         {
            b = STB_GetMemory(sizeof(S_BOOKMARK));
            if (b != NULL)
            {
               num++;
               b->time = GetBookmarkTime(file_name);
               b->name = NULL;
               if (names)
               {
                  sprintf((char *)file_str, "%s/%.*s", (char *)folder_name,
                     BOOKMARK_FILENAME_SIZE - 1, (char *)file_name);
                  file = STB_DSKOpenFile(disk_id, file_str, FILE_MODE_READ);
                  if (file != NULL)
                  {
                     STB_DSKSeekFile(file, FILE_POSITION_END, 0);
                     if (STB_DSKTellFile(file, &offset))
                     {
                        if (offset < STB_PVR_NAME_LEN)
                        {
                           read_bytes = offset;
                        }
                        else
                        {
                           read_bytes = STB_PVR_NAME_LEN - 1;
                        }

                        STB_DSKSeekFile(file, FILE_POSITION_START, 0);

                        b->name = STB_GetMemory(read_bytes + 1);
                        STB_DSKReadFile(file, b->name, read_bytes);
                        b->name[read_bytes] = '\0';
                     }

                     STB_DSKCloseFile(file);
                  }
               }

               STB_LLAddBlockToEnd(bookmarks, (LINK_LIST_PTR_BLK *)b);
            }
         }
      }

      STB_DSKCloseDirectory(dir);
   }
#ifdef STB_DEBUG
   else
   {
      STB_PVR_PRINT(("GetBookmarks: could not open folder %s", folder_name));
   }
#endif

   FUNCTION_FINISH(GetBookmarks);

   return num;
}

/**
 * @brief   Empties the bookmark linked list, and releases the memory if required
 * @param   bookmarks Linked list of bookmarks
 * @param   release If true the memory allocated for the structure and the name will be released
 */
static void EmptyBookmarks(LINK_LIST_HEADER *bookmarks, BOOLEAN release)
{
   S_BOOKMARK *b, *next;

   FUNCTION_START(EmptyBookmarks);

   b = (S_BOOKMARK *)STB_LLGetFirstBlock(bookmarks);
   while (b != NULL)
   {
      next = (S_BOOKMARK *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)b);
      STB_LLRemoveBlock((LINK_LIST_PTR_BLK *)b);

      if (release)
      {
         if (b->name != NULL)
         {
            STB_FreeMemory(b->name);
         }
         STB_FreeMemory(b);
      }
      b = next;
   }

   FUNCTION_FINISH(EmptyBookmarks);
}

/**
 * @brief   Function passed to STB_LLSort to sort the bookmarks in time order.
 * @param   bookmark1 Link list block representing the first bookmark
 * @param   bookmark2 Link list block represengint the second bookmark
 * @return  0 when the bookmarks are at the same time, -1 when the first bookmark is earlier than the
 *          second, +1 when the first bookmark is later than the second
 */
static S16BIT CompareBookmarks(LINK_LIST_PTR_BLK **bookmark1, LINK_LIST_PTR_BLK **bookmark2)
{
   S_BOOKMARK *b1, *b2;
   S16BIT retval;

   FUNCTION_START(CompareBookmarks);

   b1 = (S_BOOKMARK *)*bookmark1;
   b2 = (S_BOOKMARK *)*bookmark2;

   if ((b1 != NULL) && (b2 != NULL))
   {
      if (b1->time < b2->time)
      {
         retval = -1;
      }
      else if (b1->time > b2->time)
      {
         retval = 1;
      }
      else
      {
         retval = 0;
      }
   }
   else if (b1 != NULL)
   {
      retval = 1;
   }
   else if (b2 != NULL)
   {
      retval = -1;
   }
   else
   {
      retval = 0;
   }

   FUNCTION_FINISH(CompareBookmarks);

   return retval;
}

//*****************************************************************************
// End of file
//*****************************************************************************
