/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2007 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   PVR message database access functions.
 * @file    stbpvrmsg.c
 * @date    Nov 2007
 * @author  Chris Aldworth
 */

/* Enable debug output */
/* #define STB_DEBUG*/

/*---includes for this file---------------------------------------------------- */
/* compiler library header files */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* third party header files */

/* DVBCore header files */

#include <techtype.h>
#include <dbgfuncs.h>

#include "stbhwos.h"
#include "stbheap.h"
#include "stbgc.h"
#include "stbuni.h"

/*---constant definitions for this file---------------------------------------- */
#ifdef STB_DEBUG
#define STB_PVR_MSG_PRINT(x) STB_SPDebugWrite x
#else
#define STB_PVR_MSG_PRINT(x)
#endif

/*---local typedef structs for this file--------------------------------------- */

typedef struct s_pvr_message
{
   struct s_pvr_message *prev;
   struct s_pvr_message *next;
   U16BIT handle;
   U16BIT date;
   U8BIT hour;
   U8BIT min;
   U8BIT message;
} S_PVR_MESSAGE;

/*---local (static) variable declarations for this file---------------------------------------------
   (internal variables declared static to make them local)*/
static S_PVR_MESSAGE *msg_head;
static S_PVR_MESSAGE *msg_tail;
static U16BIT last_handle_value;
static BOOLEAN new_messages;
static void *msg_mutex;

/*---local function prototypes for this file--------------------------------------------------------
   (internal functions declared static to make them local) */
static S_PVR_MESSAGE* GetHandleStatus(U16BIT handle);


/*!**************************************************************************
 * @brief   Initialises file that provides messages for recording er
 * @param   None
 * @return  TRUE if successful, otherwise FALSE
 ****************************************************************************/
BOOLEAN STB_PVRInitialiseMessages(void)
{
   FUNCTION_START(STB_PVRInitialiseMessages);

   STB_PVR_MSG_PRINT(("PVR MSG - InitialiseMessages"));

   if (msg_mutex == NULL)
   {
      msg_head = NULL;
      msg_tail = NULL;
      last_handle_value = 0;
      new_messages = FALSE;

      msg_mutex = STB_OSCreateMutex();
   }

   FUNCTION_FINISH(STB_PVRInitialiseMessages);

   return(TRUE);
}

/*!**************************************************************************
 * @brief   Adds a message to the end of the list of existing messages
 * @param   message - The message to be written
 * @return  TRUE if successful, otherwise FALSE
 ****************************************************************************/
BOOLEAN STB_PVRAddMessage(U8BIT *message)
{
   BOOLEAN retval;
   S_PVR_MESSAGE *msg_ptr;
   U32BIT num_bytes_in_string;
   U16BIT handle;
   U16BIT cur_date;
   U8BIT cur_hour;
   U8BIT cur_min;
   U8BIT cur_sec;

   FUNCTION_START(STB_PVRAddMessage);

   retval = FALSE;

   if (msg_mutex != NULL)
   {
      num_bytes_in_string = STB_GetNumBytesInString(message);
      if (num_bytes_in_string > 0)
      {
         /* Lock access to the messages while the new message is added */
         STB_OSMutexLock(msg_mutex);

         /* Find an unused handle*/
         handle = last_handle_value;

         do
         {
            handle++;
            msg_ptr = GetHandleStatus(handle);
         }
         while (msg_ptr != NULL);

         last_handle_value = handle;

         STB_GCGetGMTDateTime(&cur_date, &cur_hour, &cur_min, &cur_sec);

         /* Create a new entry for the message */
         msg_ptr = (S_PVR_MESSAGE *)STB_GetMemory(sizeof(S_PVR_MESSAGE) + num_bytes_in_string);
         if (msg_ptr != NULL)
         {
            memset(msg_ptr, 0, sizeof(S_PVR_MESSAGE));

            msg_ptr->handle = handle;
            msg_ptr->date = cur_date;
            msg_ptr->hour = cur_hour;
            msg_ptr->min = cur_min;

            memcpy(&msg_ptr->message, message, num_bytes_in_string);

            /* Add the message to the list */
            if (msg_head == NULL)
            {
               msg_head = msg_ptr;
               msg_tail = msg_ptr;
            }
            else
            {
               msg_ptr->prev = msg_tail;
               msg_tail->next = msg_ptr;
               msg_tail = msg_ptr;
            }

            new_messages = TRUE;
            retval = TRUE;
         }

         STB_OSMutexUnlock(msg_mutex);
      }
   }

   FUNCTION_FINISH(STB_PVRAddMessage);

   return(retval);
}

/*!**************************************************************************
 * @brief   Check for new messages and clear the new message flag
 * @return  TRUE if there is a new message, otherwise FALSE
 ****************************************************************************/
BOOLEAN STB_PVRCheckMessages(void)
{
   BOOLEAN ret_val;

   FUNCTION_START(STB_PVRCheckMessages);

   ret_val = new_messages;
   new_messages = FALSE;

   FUNCTION_FINISH(STB_PVRCheckMessages);

   return(ret_val);
}

/*!**************************************************************************
 * @brief   Returns the number of messages
 * @return  number of messages
 ****************************************************************************/
U16BIT STB_PVRGetNumMessages(void)
{
   U16BIT num_messages;
   S_PVR_MESSAGE *msg_ptr;

   FUNCTION_START(STB_PVRGetNumMessages);

   num_messages = 0;

   if (msg_mutex != NULL)
   {
      STB_OSMutexLock(msg_mutex);

      msg_ptr = msg_head;
      while (msg_ptr != NULL)
      {
         num_messages++;
         msg_ptr = msg_ptr->next;
      }

      STB_OSMutexUnlock(msg_mutex);
   }

   FUNCTION_FINISH(STB_PVRGetNumMessages);

   return(num_messages);
}

/*!**************************************************************************
 * @brief   Deletes a message from the list
 * @param   handle - message handle
 * @return  TRUE if the message is found and deleted, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRDeleteMessage(U16BIT handle)
{
   S_PVR_MESSAGE *msg_ptr;
   BOOLEAN retval;

   FUNCTION_START(STB_PVRDeleteHandle);

   retval = FALSE;

   if (msg_mutex != NULL)
   {
      STB_OSMutexLock(msg_mutex);

      /* Find the message */
      msg_ptr = GetHandleStatus(handle);
      if (msg_ptr != NULL)
      {
         /* Remove it from the list */
         if (msg_ptr->prev != NULL)
         {
            /* Message isn't at the head of the list */
            msg_ptr->prev->next = msg_ptr->next;
         }
         else
         {
            /* This message is at the head of the list */
            msg_head = msg_ptr->next;
            if (msg_head != NULL)
            {
               msg_head->prev = NULL;
            }
         }

         if (msg_ptr->next != NULL)
         {
            /* Message isn't at the tail of the list */
            msg_ptr->next->prev = msg_ptr->prev;
         }
         else
         {
            /* This message is at the tail of the list */
            msg_tail = msg_tail->prev;
            if (msg_tail != NULL)
            {
               msg_tail->next = NULL;
            }
         }

         /* Destroy the message */
         STB_FreeMemory(msg_ptr);

         retval = TRUE;
      }

      STB_OSMutexUnlock(msg_mutex);
   }

   FUNCTION_FINISH(STB_PVRDeleteMessage);

   return(retval);
}

/*!**************************************************************************
 * @brief    Returns an allocated array of the message handles and number of
 *           messages currently in the list
 * @param    handle_array - pointer to an array that's allocated by the function
 *                          and is filled with the handles of all messages
 * @return   Number of messages
 ****************************************************************************/
U16BIT STB_PVRGetMessages(U16BIT **handle_array)
{
   U16BIT i, num_messages;
   S_PVR_MESSAGE *msg_ptr;

   FUNCTION_START(STB_PVRGetMessages);

   if (msg_mutex != NULL)
   {
      STB_OSMutexLock(msg_mutex);

      num_messages = STB_PVRGetNumMessages();
      if (num_messages > 0)
      {
         *handle_array = (U16BIT *)STB_GetMemory(num_messages * sizeof(U16BIT));
         if (*handle_array != NULL)
         {
            msg_ptr = msg_head;
            for (i = 0; (i < num_messages) && (msg_ptr != NULL); i++)
            {
               (*handle_array)[i] = msg_ptr->handle;
               msg_ptr = msg_ptr->next;
            }
         }
         else
         {
            num_messages = 0;
         }
      }

      STB_OSMutexUnlock(msg_mutex);
   }
   else
   {
      num_messages = 0;
   }

   FUNCTION_FINISH(STB_PVRGetMessages);

   return(num_messages);
}

/*!**************************************************************************
 * @brief    Gets date/time for the message with the given handle
 * @param    handle - message handle
 * @param    date - pointer to value in which to return the date of the message
 * @param    hour - pointer to value in which to return the hour of the message
 * @param    min - pointer to value in which to return the min of the message
 * @return   TRUE if the message is found, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_PVRGetMessageInfo(U16BIT handle, U16BIT *date, U8BIT *hour, U8BIT *min)
{
   BOOLEAN retval;
   S_PVR_MESSAGE *msg_ptr;

   FUNCTION_START(STB_PVRGetMessageInfo);

   retval = FALSE;

   if (msg_mutex != NULL)
   {
      STB_OSMutexLock(msg_mutex);

      msg_ptr = GetHandleStatus(handle);
      if (msg_ptr != NULL)
      {
         *date = msg_ptr->date;
         *hour = msg_ptr->hour;
         *min = msg_ptr->min;

         retval = TRUE;
      }

      STB_OSMutexUnlock(msg_mutex);
   }

   FUNCTION_FINISH(STB_PVRGetMessageInfo);

   return(retval);
}

/*!**************************************************************************
 * @brief   Returns a pointer to the text for the message with the given handle
 * @param   handle - message handle
 * @return  pointer to message text (this is not a copy and shouldn't be deleted)
 ****************************************************************************/
U8BIT* STB_PVRGetMessageText(U16BIT handle)
{
   U8BIT *text_ptr;
   S_PVR_MESSAGE *msg_ptr;

   FUNCTION_START(STB_PVRGetMessageText);

   text_ptr = NULL;

   if (msg_mutex != NULL)
   {
      STB_OSMutexLock(msg_mutex);

      msg_ptr = GetHandleStatus(handle);
      if (msg_ptr != NULL)
      {
         text_ptr = &msg_ptr->message;
      }

      STB_OSMutexUnlock(msg_mutex);
   }

   FUNCTION_FINISH(STB_PVRGetMessageText);

   return(text_ptr);
}

/*!**************************************************************************
 * @brief    Gets a handle
 * @param    handle - message handle
 * @return   pointer to message entry
 ****************************************************************************/
static S_PVR_MESSAGE* GetHandleStatus(U16BIT handle)
{
   S_PVR_MESSAGE *msg_ptr;

   FUNCTION_START(GetHandleStatus);

   msg_ptr = msg_head;
   while (msg_ptr != NULL)
   {
      /* Go through each handle to find match */
      if (msg_ptr->handle == handle)
      {
         break;
      }

      msg_ptr = msg_ptr->next;
   }

   FUNCTION_FINISH(GetHandleStatus);

   return(msg_ptr);
}

