/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2007 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   STB middleware resource management module source file.
 * @file    stbresmgr.c
 * @date    June 2007
 * @author  Steve Ford
 */

/*---includes for this file--------------------------------------------------*/
/* compiler library header files */
#include <string.h>

/* third party header files */

/* DVBCore header files */

#include <techtype.h>
#include <dbgfuncs.h>

#include "stbdpc.h"
#include "stbresmgr.h"
#include "stbsitab.h"
#include "stbheap.h"
#include "stbhwtun.h"
#include "stbhwdmx.h"

#ifdef COMMON_INTERFACE
#include "stbhwci.h"
#include "stbcicc.h"
#endif


/*---constant definitions for this file--------------------------------------*/
/*#define STB_RES_DEBUG*/

#ifdef STB_RES_DEBUG
#define STB_RES_DBG(x) STB_SPDebugWrite x
#else
#define STB_RES_DBG(x)
#endif

/*---local typedef structs for this file---------------------------------------*/

typedef struct
{
   BOOLEAN disabled;
   U8BIT usage_count;               /* Number of times the tuner has been acquired */
   U8BIT high_count;                /* Number of times the tuner has been acquired with a high priority */
   U16BIT tuner_type;               /* Type of tuner (terrestrial / cable / etc) */
   void *tuned_transport;           /* Transport that this tuner is tuned to */
   E_STB_DP_RES_OWNER owner;        /* Owner of the resource */
} S_TUNER_STATUS;

typedef struct
{
   U8BIT usage_count;               /* Number of times the demux has been acquired */
   U16BIT caps;                     /* Demux capability flags */
} S_DEMUX_STATUS;

typedef struct
{
   BOOLEAN is_acquired;             /* TRUE if the decoder has been acquired */
} S_DECODER_STATUS;

typedef struct
{
   U8BIT usage_count;               /* Number of times the slot has been acquired */
   void *service;                   /* Service this CI slot is associated with */
} S_CI_SLOT_STATUS;


/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local) */

static BOOLEAN res_initialised = FALSE;

static U8BIT num_tuners;                         /* Number of tuners in the system */
static S_TUNER_STATUS *tuner_status;             /* Status of each tuner in the system */
static U8BIT num_demuxes;                        /* Number of demux paths */
static S_DEMUX_STATUS *demux_status;             /* Status of each demux path */
static U8BIT num_audio_decoders;                 /* Number of audio decoders in the system */
static S_DECODER_STATUS *audio_decoder_status;   /* Status of each audio decoder in the system */
static S_DECODER_STATUS *ad_decoder_status;      /* Status of each AD decoder in the system */
static U8BIT num_video_decoders;                 /* Number of decoders in the system */
static S_DECODER_STATUS *video_decoder_status;   /* Status of each video decoder in the system */
static U8BIT num_ci_slots;                       /* Number of CI slots in the system */
static S_CI_SLOT_STATUS *ci_slot_status;         /* Status of each CI slot in the system */


/*---local function prototypes for this file---------------------------------*/
/*   (internal functions declared static to make them local) */


/*---global function definitions---------------------------------------------*/

/*!**************************************************************************
 * @brief   Creates and initialises control structures for the resources to be managed
 * @param   None
 * @return  TRUE if successful
 ****************************************************************************/
BOOLEAN STB_RESInitialise(void)
{
   U8BIT i;
   S_TUNER_STATUS *tuner;
   S_DECODER_STATUS *decoder;
   S_DEMUX_STATUS *demux;
   E_STB_TUNE_SIGNAL_TYPE signal_type;
#ifdef COMMON_INTERFACE
   S_CI_SLOT_STATUS *slot;
#endif

   FUNCTION_START(STB_RESInitialise);

   if (!res_initialised)
   {
      /* Initialise the tuner management */
      num_tuners = STB_HWGetTunerPaths();
      if (num_tuners > 0)
      {
         tuner_status = (S_TUNER_STATUS *)STB_GetMemory((U32BIT)(sizeof(S_TUNER_STATUS) * num_tuners));
         if (tuner_status != NULL)
         {
            STB_RES_DBG(("STB_RESInitialise: Initialising %d tuners", num_tuners));
            for (i = 0, tuner = &tuner_status[0]; i < num_tuners; tuner++, i++)
            {
               memset(tuner, 0, sizeof(*tuner));

               tuner->tuner_type = SIGNAL_NONE;

               signal_type = STB_TuneGetSignalType(i);

               if ((signal_type & TUNE_SIGNAL_ANALOG) != 0)
               {
                  tuner->tuner_type |= SIGNAL_ANALOG;
               }
               if ((signal_type & TUNE_SIGNAL_COFDM) != 0)
               {
                  tuner->tuner_type |= SIGNAL_COFDM;
               }
               if ((signal_type & TUNE_SIGNAL_QAM) != 0)
               {
                  tuner->tuner_type |= SIGNAL_QAM;
               }
               if ((signal_type & TUNE_SIGNAL_QPSK) != 0)
               {
                  tuner->tuner_type |= SIGNAL_QPSK;
               }

               tuner->disabled = FALSE;
               tuner->tuned_transport = NULL;
               tuner->owner = RES_OWNER_NONE;
            }
         }
      }
      else
      {
         STB_RES_DBG(("STB_RESInitialise: No tuners!"));
         tuner_status = NULL;
      }

      num_demuxes = STB_HWGetDemuxPaths();
      if (num_demuxes > 0)
      {
         demux_status = (S_DEMUX_STATUS *)STB_GetMemory((U32BIT)(sizeof(S_DEMUX_STATUS) * num_demuxes));
         if (demux_status != NULL)
         {
            STB_RES_DBG(("STB_RESInitialise: Initialising %d demuxes", num_demuxes));
            for (i = 0, demux = &demux_status[0]; i < num_demuxes; demux++, i++)
            {
               memset(demux, 0, sizeof(*demux));

               demux->caps = STB_DMXGetCapabilities(i);
            }
         }
      }
      else
      {
         STB_RES_DBG(("STB_RESInitialise: No demuxes!"));
         demux_status = NULL;
      }

      num_audio_decoders = STB_HWGetAudioDecodePaths();
      if (num_audio_decoders > 0)
      {
         audio_decoder_status = (S_DECODER_STATUS *)STB_GetMemory((U32BIT)(sizeof(S_DECODER_STATUS) * num_audio_decoders));
         if (audio_decoder_status != NULL)
         {
            STB_RES_DBG(("STB_RESInitialise: Initialising %d audio decoders", num_audio_decoders));
            for (i = 0, decoder = &audio_decoder_status[0]; i < num_audio_decoders; decoder++, i++)
            {
               decoder->is_acquired = FALSE;
            }
         }
         ad_decoder_status = (S_DECODER_STATUS *)STB_GetMemory((U32BIT)(sizeof(S_DECODER_STATUS) * num_audio_decoders));
         if (ad_decoder_status != NULL)
         {
            STB_RES_DBG(("STB_RESInitialise: Initialising %d AD decoders", num_audio_decoders));
            for (i = 0, decoder = &ad_decoder_status[0]; i < num_audio_decoders; decoder++, i++)
            {
               decoder->is_acquired = FALSE;
            }
         }
      }
      else
      {
         STB_RES_DBG(("STB_RESInitialise: No audio decoders!"));
         audio_decoder_status = NULL;
         ad_decoder_status = NULL;
      }

      num_video_decoders = STB_HWGetVideoDecodePaths();
      if (num_video_decoders > 0)
      {
         video_decoder_status = (S_DECODER_STATUS *)STB_GetMemory((U32BIT)(sizeof(S_DECODER_STATUS) * num_video_decoders));
         if (video_decoder_status != NULL)
         {
            STB_RES_DBG(("STB_RESInitialise: Initialising %d video decoders", num_video_decoders));
            for (i = 0, decoder = &video_decoder_status[0]; i < num_video_decoders; decoder++, i++)
            {
               decoder->is_acquired = FALSE;
            }
         }
      }
      else
      {
         STB_RES_DBG(("STB_RESInitialise: No video decoders!"));
         video_decoder_status = NULL;
      }

#ifdef COMMON_INTERFACE
      num_ci_slots = STB_CIGetSlotCount();
      if (num_ci_slots > 0)
      {
         ci_slot_status = (S_CI_SLOT_STATUS *)STB_GetMemory((U32BIT)(sizeof(S_CI_SLOT_STATUS) * num_ci_slots));
         if (ci_slot_status != NULL)
         {
            STB_RES_DBG(("STB_RESInitialise: Initialising %d CI slots", num_ci_slots));
            for (i = 0, slot = &ci_slot_status[0]; i < num_ci_slots; slot++, i++)
            {
               slot->usage_count = 0;
               slot->service = NULL;
            }
         }
      }
      else
      {
         ci_slot_status = NULL;
      }
#endif

      res_initialised = TRUE;
   }

   FUNCTION_FINISH(STB_RESInitialise);

   return(res_initialised);
}

/*!**************************************************************************
 * @brief   Returns the number of tuners
 * @return  Number of tuners on the system
 ****************************************************************************/
U8BIT STB_RESNumTuners(void)
{
   FUNCTION_START(STB_RESNumTuners);
   FUNCTION_FINISH(STB_RESNumTuners);
   return(num_tuners);
}

/*!**************************************************************************
 * @brief   Acquires a tuner. Tuners will be reused if possible
 * @param   tuner_type - type of tuner to be acquired
 * @param   transport - initial transport the tuner will be tuned to
 * @param   owner - module acquiring the tuner
 * @param   high_priority - TRUE if the request is a high priority, meaning a tuner acquired with
 *                          a lower priority can be taken. A tuner acquired with a high priority
 *                          will be locked, preventing it being retuned.
 * @param   tuner_taken - returned as TRUE if the acquired tuner was in use but is reacquired.
 * @return  The ID of the tuner acquired or INVALID_RES_ID if none are available
 ****************************************************************************/
U8BIT STB_RESAcquireTuner(E_STB_DP_SIGNAL_TYPE tuner_type, void *transport, E_STB_DP_RES_OWNER owner,
   BOOLEAN high_priority, BOOLEAN *tuner_taken)
{
   U8BIT i;
   U8BIT tuner_id = INVALID_RES_ID;

   FUNCTION_START(STB_RESAcquireTuner);

   *tuner_taken = FALSE;

   if (res_initialised)
   {
      if (transport != NULL)
      {
         /* See if a tuner is currently tuned to this transport and the owner is the same */
         for (i = 0; (i < num_tuners) && (tuner_id == INVALID_RES_ID); i++)
         {
            if ((tuner_status[i].tuned_transport == transport))
            {
               tuner_status[i].usage_count++;

               if ((tuner_status[i].owner == RES_OWNER_NONE) && (owner != RES_OWNER_NONE))
               {
                  /* Tuner is now owned by a module */
                  tuner_status[i].owner = owner;
               }

               if (high_priority)
               {
                  /* This tuner can't be tuned to another transport until it's released */
                  tuner_status[i].high_count++;
               }

               tuner_id = i;
            }
         }
      }

      if ((tuner_id == INVALID_RES_ID) || (transport == NULL))
      {
         /* Look for a tuner of the given type that hasn't been acquired yet */
         for (i = 0; (i < num_tuners) && (tuner_id == INVALID_RES_ID); i++)
         {
            if (!tuner_status[i].disabled && (tuner_status[i].usage_count == 0) &&
               ((tuner_status[i].tuner_type & tuner_type) != 0))
            {
               if (high_priority)
               {
                  tuner_status[i].high_count++;
               }

               tuner_status[i].usage_count++;
               tuner_status[i].owner = owner;
               tuner_id = i;

               STB_TuneSetSignalType(tuner_id, tuner_type);
            }
         }

         if ((tuner_id == INVALID_RES_ID) && high_priority)
         {
            /* All tuners are being used but this request has a higher priority so find one that
             * isn't currently being used with a high priority and isn't owned by another module */
            for (i = 0; (i < num_tuners) && (tuner_id == INVALID_RES_ID); i++)
            {
               if (!tuner_status[i].disabled && ((tuner_status[i].tuner_type & tuner_type) != 0) &&
                  (tuner_status[i].high_count == 0) && (tuner_status[i].owner == RES_OWNER_NONE))
               {
                  tuner_status[i].tuned_transport = transport;
                  tuner_status[i].high_count++;
                  tuner_status[i].usage_count++;
                  tuner_status[i].owner = owner;
                  tuner_id = i;
                  *tuner_taken = TRUE;

                  STB_TuneSetSignalType(tuner_id, tuner_type);
               }
            }
         }
      }
   }

#ifdef STB_RES_DEBUG
   if (tuner_id == INVALID_RES_ID)
   {
      STB_RES_DBG(("STB_RESAcquireTuner(%u, %p, %u): Failed to acquire a tuner for owner %u", tuner_type,
                   transport, high_priority, owner));
   }
   else
   {
      STB_RES_DBG(("STB_RESAcquireTuner(%u, %p, %u): Tuner %u acquired, usage_count=%u, high_count=%u, owner=%u",
                   tuner_type, transport, high_priority, tuner_id, tuner_status[tuner_id].usage_count,
                   tuner_status[tuner_id].high_count, owner));
   }
#endif

   FUNCTION_FINISH(STB_RESAcquireTuner);

   return(tuner_id);
}

/*!**************************************************************************
 * @brief   Releases a tuner that has previously been acquired
 * @param   tuner_id - ID of a tuner previously acquired
 * @param   high_priority - TRUE if the tuner was acquired with a high priority
 * @param   owner - module releasing the tuner
 ****************************************************************************/
void STB_RESReleaseTuner(U8BIT tuner_id, BOOLEAN high_priority, E_STB_DP_RES_OWNER owner)
{
   FUNCTION_START(STB_RESReleaseTuner);

   if (res_initialised)
   {
      ASSERT(tuner_id < num_tuners);

      if (tuner_id < num_tuners)
      {
         STB_RES_DBG(("STB_RESReleaseTuner(%u, %u, %u): usage_count=%u, high_count=%u, owner=%u",
                      tuner_id, high_priority, owner, tuner_status[tuner_id].usage_count,
                      tuner_status[tuner_id].high_count, tuner_status[tuner_id].owner));

         if (high_priority)
         {
            tuner_status[tuner_id].high_count--;
         }

         if (tuner_status[tuner_id].usage_count != 0)
         {
            tuner_status[tuner_id].usage_count--;
            if (tuner_status[tuner_id].usage_count == 0)
            {
               /* Tuner is no longer in use */
               tuner_status[tuner_id].tuned_transport = NULL;
               tuner_status[tuner_id].owner = RES_OWNER_NONE;

               STB_TuneSetSignalType(tuner_id, TUNE_SIGNAL_NONE);
            }
            else if (tuner_status[tuner_id].owner == owner)
            {
               tuner_status[tuner_id].owner = RES_OWNER_NONE;
            }
         }
#ifdef STB_RES_DEBUG
         else
         {
            STB_RES_DBG(("STB_RESReleaseTuner(%u): Tuner released too many times!", tuner_id));
         }
#endif
      }
   }

   FUNCTION_FINISH(STB_RESReleaseTuner);
}

/**
 * @brief   Set the disable state for a tuner. When disabled, a tuner will be ignored when
 *          acquiring a new tuner resource
 * @param   tuner_id ID of tuner
 * @param   disable TRUE if the tuner is to be disabled, FALSE otherwise
 */
void STB_RESSetTunerDisabled(U8BIT tuner_id, BOOLEAN disable)
{
   FUNCTION_START(STB_RESSetTunerDisabled);

   if (res_initialised)
   {
      ASSERT(tuner_id < num_tuners);

      if (tuner_id < num_tuners)
      {
         tuner_status[tuner_id].disabled = disable;
      }
   }

   FUNCTION_FINISH(STB_RESSetTunerDisabled);
}

/**
 * @brief   Returns whether a tuner has been disabled or not
 * @param   tuner_id ID of tuner
 * @return  TRUE if the tuner is disabled, FALSE otherwise
 */
BOOLEAN STB_RESIsTunerDisabled(U8BIT tuner_id)
{
   BOOLEAN disabled;

   FUNCTION_START(STB_RESIsTunerDisabled);

   disabled = TRUE;

   if (res_initialised)
   {
      ASSERT(tuner_id < num_tuners);

      if (tuner_id < num_tuners)
      {
         disabled = tuner_status[tuner_id].disabled;
      }
   }

   FUNCTION_FINISH(STB_RESIsTunerDisabled);

   return(disabled);
}

/**
 * @brief   Returns the number of tuners that aren't disabled
 * @return  number of enabled tuners
 */
U8BIT STB_RESNumEnabledTuners(void)
{
   U8BIT id, enabled_tuners;

   FUNCTION_START(STB_RESNumEnabledTuners);

   enabled_tuners = 0;

   for (id  = 0; id < num_tuners; id++)
   {
      if (!tuner_status[id].disabled)
      {
         enabled_tuners++;
      }
   }

   FUNCTION_FINISH(STB_RESNumEnabledTuners);

   return(enabled_tuners);
}

/*!**************************************************************************
 * @brief   Sets the owner of the tuner
 * @param   tuner_id - ID of a tuner
 * @param   owner - module acquiring the tuner
 ****************************************************************************/
void STB_RESSetTunerOwner(U8BIT tuner_id, E_STB_DP_RES_OWNER owner)
{
   FUNCTION_START(STB_RESSetTunerOwner);

   if (res_initialised)
   {
      ASSERT(tuner_id < num_tuners);

      if (tuner_id < num_tuners)
      {
         tuner_status[tuner_id].owner = owner;
      }
   }

   FUNCTION_FINISH(STB_RESSetTunerOwner);
}

/*!**************************************************************************
 * @brief   Returns the type of the given tuner
 * @param   tuner_id - ID of a tuner
 * @return  tuner type, SIGNAL_NONE on error
 ****************************************************************************/
E_STB_DP_SIGNAL_TYPE STB_RESGetTunerType(U8BIT tuner_id)
{
   E_STB_DP_SIGNAL_TYPE type;

   FUNCTION_START(STB_RESGetTunerType);

   type = SIGNAL_NONE;

   if (res_initialised)
   {
      ASSERT(tuner_id < num_tuners);

      if (tuner_id < num_tuners)
      {
         type = tuner_status[tuner_id].tuner_type;
      }
   }

   FUNCTION_FINISH(STB_RESGetTunerType);

   return(type);
}

/*!**************************************************************************
 * @brief   Saves the transport pointer with the specified tuner
 * @param   tuner_id - ID of a previously acquired tuner
 * @param   t_ptr - pointer to the transport record to be saved
 * @return  None
 ****************************************************************************/
void STB_RESSetTunedTransport(U8BIT tuner_id, void *t_ptr)
{
   FUNCTION_START(STB_RESSetTunedTransport);

   if (res_initialised)
   {
      ASSERT(tuner_id < num_tuners);

      /* Only set the transport if the tuner isn't being used by more than one decode path */
      if ((tuner_id < num_tuners) && (tuner_status[tuner_id].usage_count == 1))
      {
         tuner_status[tuner_id].tuned_transport = t_ptr;
      }
   }

   FUNCTION_FINISH(STB_RESSetTunedTransport);
}

/*!**************************************************************************
 * @brief   Returns the transport record saved with the specified tuner
 * @param   tuner_id - ID of a tuner
 * @return  Pointer to the transport record saved with the tuner
 ****************************************************************************/
void* STB_RESGetTunedTransport(U8BIT tuner_id)
{
   void *t_ptr = NULL;

   FUNCTION_START(STB_RESGetTunedTransport);

   if (res_initialised)
   {
      ASSERT(tuner_id < num_tuners);

      if (tuner_id < num_tuners)
      {
         t_ptr = tuner_status[tuner_id].tuned_transport;
      }
   }

   FUNCTION_FINISH(STB_RESGetTunedTransport);

   return(t_ptr);
}

/*!**************************************************************************
 * @brief   Returns whether there's a tuner available to tune to the given transport
 * @param   tuner_type - tuner type
 * @param   transport - transport to be tuned to
 * @return  TRUE if a tuner is available, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_RESCanTuneToTransport(E_STB_DP_SIGNAL_TYPE tuner_type, void *transport)
{
   U8BIT i;
   BOOLEAN can_tune;

   FUNCTION_START(STB_RESCanTuneToTransport);

   can_tune = FALSE;

   /* See if there's a tuner already tuned to this transport or one of the right type that can be tuned */
   for (i = 0; (i < num_tuners) && !can_tune; i++)
   {
      if (((tuner_status[i].usage_count > 0) && (tuner_status[i].tuned_transport == transport)) ||
          (((tuner_status[i].tuner_type & tuner_type) != 0) && (tuner_status[i].high_count == 0)))
      {
         can_tune = TRUE;
      }
   }

   FUNCTION_FINISH(STB_RESCanTuneToTransport);

   return(can_tune);
}

/*!**************************************************************************
 * @brief   Returns the number of paths using the tuner
 * @param   tuner_id - tuner
 * @return  Usage count
 ****************************************************************************/
U8BIT STB_RESTunerUsageCount(U8BIT tuner_id)
{
   U8BIT usage_count;

   FUNCTION_START(STB_RESTunerUsageCount);

   usage_count = 0;

   if (res_initialised)
   {
      ASSERT(tuner_id < num_tuners);

      if (tuner_id < num_tuners)
      {
         usage_count = tuner_status[tuner_id].usage_count;
      }
   }

   FUNCTION_FINISH(STB_RESTunerUsageCount);

   return(usage_count);
}

/*!**************************************************************************
 * @brief   Returns the number of demux paths under the control of the resource manager
 * @return  Number of demux paths
 ****************************************************************************/
U8BIT STB_RESNumDemuxes(void)
{
   FUNCTION_START(STB_RESNumDemuxes);
   FUNCTION_FINISH(STB_RESNumDemuxes);
   return(num_demuxes);
}

/*!**************************************************************************
 * @brief   Acquires a demux path and marks it as being used
 * @param   demux_id - id of existing demux, or INVALID_RES_ID if a new one is to be acquired
 * @param   caps - required capabilities, or 0 if an existing demux is being acquired
 * @return  ID of the demux path acquired, or INVALID_RES_ID if none are available
 ****************************************************************************/
U8BIT STB_RESAcquireDemux(U8BIT demux_id, U16BIT caps)
{
   U8BIT i;

   FUNCTION_START(STB_RESAcquireDemux);

   if (res_initialised)
   {
      STB_RES_DBG(("STB_RESAcquireDemux(%u, %u)", demux_id, caps));

      if (demux_id == INVALID_RES_ID)
      {
         for (i = 0; (i < num_demuxes) && (demux_id == INVALID_RES_ID); i++)
         {
            if ((demux_status[i].usage_count == 0) && ((demux_status[i].caps & caps) == caps))
            {
               demux_status[i].usage_count = 1;
               demux_id = i;
            }
         }

         if ((demux_id == INVALID_RES_ID) && (caps == DMX_CAPS_MONITOR_SI))
         {
            /* No demux available for monitoring SI data, so see if a live demux is available
             * as this will be able to do the same task */
            for (i = 0; (i < num_demuxes) && (demux_id == INVALID_RES_ID); i++)
            {
               if ((demux_status[i].usage_count == 0) && ((demux_status[i].caps & DMX_CAPS_LIVE) != 0))
               {
                  demux_status[i].usage_count = 1;
                  demux_id = i;
               }
            }
         }
      }
      else
      {
         /* Acquiring a specific demux, increment its usage count */
         demux_status[demux_id].usage_count++;
      }
   }
   else
   {
      demux_id = INVALID_RES_ID;
   }

#ifdef STB_RES_DEBUG
   if (demux_id == INVALID_RES_ID)
   {
      STB_RES_DBG(("STB_RESAcquireDemux: Failed to acquire a demux"));
   }
   else
   {
      STB_RES_DBG(("STB_RESAcquireDemux: Acquired %u, usage_count=%u", demux_id,
                   demux_status[demux_id].usage_count));
   }
#endif

   FUNCTION_FINISH(STB_RESAcquireDemux);

   return(demux_id);
}

/*!**************************************************************************
 * @brief   Releases a previously acquired demux path
 * @param   demux_id - ID of the demux path to be released
 * @return  None
 ****************************************************************************/
void STB_RESReleaseDemux(U8BIT demux_id)
{
   FUNCTION_START(STB_RESReleaseDemux);

   if (res_initialised)
   {
      ASSERT(demux_id < num_demuxes);

      if (demux_id < num_demuxes)
      {
         STB_RES_DBG(("STB_RESReleaseDemux(%u): usage_count=%u", demux_id,
                      demux_status[demux_id].usage_count));

         if (demux_status[demux_id].usage_count > 0)
         {
            demux_status[demux_id].usage_count--;
            STB_RES_DBG(("STB_RESReleaseDemux: Released demux %d", demux_id));
         }
#ifdef STB_RES_DEBUG
         else
         {
            STB_RES_DBG(("STB_RESReleaseDemux: Demux %d released too many times!", demux_id));
         }
#endif
      }
   }

   FUNCTION_FINISH(STB_RESReleaseDemux);
}

/*!**************************************************************************
 * @brief   Returns the capability flags for the given demux
 * @param   demux_id - demux
 * @return  Capability flags, 0 for invalid demux
 ****************************************************************************/
U16BIT STB_RESGetDemuxCaps(U8BIT demux_id)
{
   U16BIT caps;

   FUNCTION_START(STB_RESGetDemuxCaps);

   if (demux_id != INVALID_RES_ID)
   {
      caps = demux_status[demux_id].caps;
   }
   else
   {
      caps = 0;
   }

   FUNCTION_FINISH(STB_RESGetDemuxCaps);

   return(caps);
}

/*!**************************************************************************
 * @brief   Returns the number of audio decoders in the system
 * @return  Number of audio decoders
 ****************************************************************************/
U8BIT STB_RESNumAudioDecoders(void)
{
   FUNCTION_START(STB_RESNumAudioDecoders);
   FUNCTION_FINISH(STB_RESNumAudioDecoders);
   return(num_audio_decoders);
}

/*!**************************************************************************
 * @brief   Acquires an AD decoder and marks it as used
 * @param   None
 * @return  ID of the decoder, or INVALID_RES_ID if none are available
 ****************************************************************************/
U8BIT STB_RESAcquireADDecoder(void)
{
   U8BIT i;
   U8BIT decoder_id = INVALID_RES_ID;
   FUNCTION_START(STB_RESAcquireADDecoder);
   if (res_initialised)
   {
      for (i = 0; (i < num_audio_decoders) && (decoder_id == INVALID_RES_ID); i++)
      {
         if (!ad_decoder_status[i].is_acquired)
         {
            ad_decoder_status[i].is_acquired = TRUE;
            decoder_id = i;
            STB_RES_DBG(("STB_RESAcquireADDecoder: Decoder %d acquired", decoder_id));
         }
      }
   }
#ifdef STB_RES_DEBUG
   if (decoder_id == INVALID_RES_ID)
   {
      STB_RES_DBG(("STB_RESAcquireADDecoder: Failed to acquire a decoder"));
   }
#endif

   FUNCTION_FINISH(STB_RESAcquireADDecoder);

   return(decoder_id);
}

/*!**************************************************************************
 * @brief   Releases a previously acquired AD decoder
 * @param   decoder_id - ID of the decoder
 * @return  None
 ****************************************************************************/
void STB_RESReleaseADDecoder(U8BIT decoder_id)
{
   FUNCTION_START(STB_RESReleaseADDecoder);

   if (res_initialised)
   {
      if (decoder_id < num_audio_decoders)
      {
         ad_decoder_status[decoder_id].is_acquired = FALSE;
         STB_RES_DBG(("STB_RESReleaseADDecoder: Released decoder %d", decoder_id));
      }
   }

   FUNCTION_FINISH(STB_RESReleaseADDecoder);
}

/*!**************************************************************************
 * @brief   Acquires an audio decoder and marks it as used
 * @param   None
 * @return  ID of the decoder, or INVALID_RES_ID if none are available
 ****************************************************************************/
U8BIT STB_RESAcquireAudioDecoder(void)
{
   U8BIT i;
   U8BIT decoder_id = INVALID_RES_ID;

   FUNCTION_START(STB_RESAcquireAudioDecoder);

   if (res_initialised)
   {
      for (i = 0; (i < num_audio_decoders) && (decoder_id == INVALID_RES_ID); i++)
      {
         if (!audio_decoder_status[i].is_acquired)
         {
            audio_decoder_status[i].is_acquired = TRUE;
            decoder_id = i;
            STB_RES_DBG(("STB_RESAcquireAudioDecoder: Decoder %d acquired", decoder_id));
         }
      }
   }

#ifdef STB_RES_DEBUG
   if (decoder_id == INVALID_RES_ID)
   {
      STB_RES_DBG(("STB_RESAcquireAudioDecoder: Failed to acquire a decoder"));
   }
#endif

   FUNCTION_FINISH(STB_RESAcquireAudioDecoder);

   return(decoder_id);
}

/*!**************************************************************************
 * @brief   Releases a previously acquired audio decoder
 * @param   decoder_id - ID of the decoder
 * @return  None
 ****************************************************************************/
void STB_RESReleaseAudioDecoder(U8BIT decoder_id)
{
   FUNCTION_START(STB_RESReleaseAudioDecoder);

   if (res_initialised)
   {
      ASSERT(decoder_id < num_audio_decoders);

      if (decoder_id < num_audio_decoders)
      {
         audio_decoder_status[decoder_id].is_acquired = FALSE;
         STB_RES_DBG(("STB_RESReleaseAudioDecoder: Released decoder %d", decoder_id));
      }
   }

   FUNCTION_FINISH(STB_RESReleaseAudioDecoder);
}

/*!**************************************************************************
 * @brief   Returns the number of video decoders in the system
 * @param   None
 * @return  Number of video decoders
 ****************************************************************************/
U8BIT STB_RESNumVideoDecoders(void)
{
   FUNCTION_START(STB_RESNumVideoDecoders);
   FUNCTION_FINISH(STB_RESNumVideoDecoders);
   return(num_video_decoders);
}

/*!**************************************************************************
 * @brief   Acquires a video decoder and marks it as used
 * @param   None
 * @return  ID of the decoder, or INVALID_RES_ID if none are available
 ****************************************************************************/
U8BIT STB_RESAcquireVideoDecoder(void)
{
   U8BIT i;
   U8BIT decoder_id = INVALID_RES_ID;

   FUNCTION_START(STB_RESAcquireVideoDecoder);

   if (res_initialised)
   {
      for (i = 0; (i < num_video_decoders) && (decoder_id == INVALID_RES_ID); i++)
      {
         if (!video_decoder_status[i].is_acquired)
         {
            video_decoder_status[i].is_acquired = TRUE;
            decoder_id = i;
            STB_RES_DBG(("STB_RESAcquireVideoDecoder: Decoder %d acquired", decoder_id));
         }
      }
   }

#ifdef STB_RES_DEBUG
   if (decoder_id == INVALID_RES_ID)
   {
      STB_RES_DBG(("STB_RESAcquireVideoDecoder: Failed to acquire a decoder"));
   }
#endif

   FUNCTION_FINISH(STB_RESAcquireVideoDecoder);

   return(decoder_id);
}

/*!**************************************************************************
 * @brief   Releases a previously acquired video decoder
 * @param   decoder_id - ID of decoder
 * @return  None
 ****************************************************************************/
void STB_RESReleaseVideoDecoder(U8BIT decoder_id)
{
   FUNCTION_START(STB_RESReleaseVideoDecoder);

   if (res_initialised)
   {
      ASSERT(decoder_id < num_video_decoders);

      if (decoder_id < num_video_decoders)
      {
         video_decoder_status[decoder_id].is_acquired = FALSE;
         STB_RES_DBG(("STB_RESReleaseVideoDecoder: Released decoder %d", decoder_id));
      }
   }

   FUNCTION_FINISH(STB_RESReleaseVideoDecoder);
}

/*!**************************************************************************
 * @brief   Returns the number of CI slots under the control of the resource manager
 * @return  Number of CI slots
 ****************************************************************************/
U8BIT STB_RESNumCISlots(void)
{
   FUNCTION_START(STB_RESNumCISlots);
   FUNCTION_FINISH(STB_RESNumCISlots);
   return(num_ci_slots);
}

#ifdef COMMON_INTERFACE
/*!**************************************************************************
 * @brief   Acquires a CI slot
 * @param   service - service the slot will be used for
 * @param   pmt_data - raw PMT data for the service
 * @param   ci_protection_desc - raw CI protection descriptor for the service
 * @return  ID of the CI slot acquired, or INVALID_RES_ID if none are available
 ****************************************************************************/
U8BIT STB_RESAcquireCISlot(void *service, U8BIT *pmt_data, U8BIT *ci_protection_desc)
{
   U8BIT i;
   U8BIT slot_id = INVALID_RES_ID;

   FUNCTION_START(STB_RESAcquireCISlot);

   if (res_initialised)
   {
      /* See if a CI slot is already being used for this service */
      for (i = 0; (i < num_ci_slots) && (slot_id == INVALID_RES_ID); i++)
      {
         if ((ci_slot_status[i].usage_count > 0) && (ci_slot_status[i].service == service))
         {
            /* This slot is already being used for this service, so reuse it */
            ci_slot_status[i].usage_count++;
            slot_id = i;
            STB_RES_DBG(("STB_RESAcquireCISlot: CI slot %u acquired, usage_count=%u", slot_id,
                         ci_slot_status[i].usage_count));
         }
      }

      /* Acquire a slot if one hasn't already been found */
      for (i = 0; (i < num_ci_slots) && (slot_id == INVALID_RES_ID); i++)
      {
         if (ci_slot_status[i].usage_count == 0)
         {
            if (STB_CiCcIsServiceAllowed(i, ci_protection_desc))
            {
               ci_slot_status[i].usage_count = 1;
               ci_slot_status[i].service = service;
               slot_id = i;
               STB_RES_DBG(("STB_RESAcquireCISlot: CI slot %u acquired for service %p", slot_id, service));
            }
         }
      }
   }

#ifdef STB_RES_DEBUG
   if (slot_id == INVALID_RES_ID)
   {
      STB_RES_DBG(("STB_RESAcquireCISlot: Failed to acquire a CI slot for service %p", service));
   }
#endif

   FUNCTION_FINISH(STB_RESAcquireCISlot);

   return(slot_id);
}

/*!**************************************************************************
 * @brief   Set the given CI slot as acquired and in use
 * @param   slot_id - ID of the CI slot
 * @return  TRUE if the slot isn't currently being used, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_RESUseCISlot(U8BIT slot_id)
{
   BOOLEAN retval;

   FUNCTION_START(STB_RESUseCISlot);

   retval = FALSE;

   if (res_initialised && (slot_id < num_ci_slots))
   {
      STB_RES_DBG(("STB_RESUseCISlot(%u): usage count=%u", slot_id,
                   ci_slot_status[slot_id].usage_count));

      if (ci_slot_status[slot_id].usage_count == 0)
      {
         ci_slot_status[slot_id].usage_count = 1;
         ci_slot_status[slot_id].service = NULL;
         retval = TRUE;
      }
   }

   FUNCTION_FINISH(STB_RESUseCISlot);

   return(retval);
}

#endif /*COMMON_INTERFACE*/

/*!**************************************************************************
 * @brief   Returns the number of times the given slot is in use
 * @param   slot_id - ID of the CI slot
 * @return  Slot usage count
 ****************************************************************************/
U8BIT STB_RESGetCISlotUsageCount(U8BIT slot_id)
{
   U8BIT retval;

   FUNCTION_START(STB_RESGetCISlotUsageCount);

   retval = 0;

   if (res_initialised)
   {
      ASSERT(slot_id < num_ci_slots);

      if (slot_id < num_ci_slots)
      {
         retval = ci_slot_status[slot_id].usage_count;
      }
   }

   FUNCTION_FINISH(STB_RESGetCISlotUsageCount);

   return(retval);
}

/*!**************************************************************************
 * @brief   Releases a previously acquired CI slot
 * @param   slot_id - ID of the CI slot to be released
 ****************************************************************************/
void STB_RESReleaseCISlot(U8BIT slot_id)
{
   FUNCTION_START(STB_RESReleaseCISlot);

   if (res_initialised)
   {
      ASSERT(slot_id < num_ci_slots);

      if (slot_id < num_ci_slots)
      {
         STB_RES_DBG(("STB_RESReleaseCISlot(%u): usage_count=%u", slot_id,
                      ci_slot_status[slot_id].usage_count));

         if (ci_slot_status[slot_id].usage_count > 0)
         {
            ci_slot_status[slot_id].usage_count--;
         }

         if (ci_slot_status[slot_id].usage_count == 0)
         {
            ci_slot_status[slot_id].service = NULL;
         }
      }
   }

   FUNCTION_FINISH(STB_RESReleaseCISlot);
}

