/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   STB layer SI filter handling
 *
 * @file    stbsiflt.c
 * @date    06/03/2003
 */

// gives direct COM port access
/* #define STB_DEBUG */

// prints filter use
/* #define DEBUG_SI_FILTER */
/* #define DEBUG_SI_FILTER_FAILURES */
/* #define DEBUG_SI_FILTER_INT */
/* #define DEBUG_FILTER_PERFORMANCE */

// prints table requests
/* #define DEBUG_TABLE_REQUESTS */

/* #define DEBUG_PMT_REPORTING */
/* #define DEBUG_CAT_REPORTING */

//---includes for this file----------------------------------------------------
// compiler library header files

#include <stdio.h>
#include <string.h>

// third party header files

// Ocean Blue Software header files

#include <techtype.h>
#include <dbgfuncs.h>

#include "stberc.h"
#include "stbhwdmx.h"
#include "stbhwos.h"
#include "stbllist.h"
#include "stbheap.h"
//#include "stbresmgr.h"
#include "stbdpc.h"

#include "stbsic.h"
#include "stbsiflt.h"
#include "stbsipmt.h"
#include "ca_glue.h"
#include "stbsitab.h"


//---constant definitions for this file----------------------------------------
#ifdef STB_SI_PRINT_REQUIRED
   #define STB_SI_PRINT(x) DEBUG_PRINTX_CONDITIONAL(DEBUG_STB_SI) x
#else
   #ifdef STB_DEBUG
      #define STB_SI_PRINT(x) STB_SPDebugWrite x
   #else
      #define STB_SI_PRINT(x)
   #endif
#endif

#define STB_SI_ERROR(x)

#define MAX_ENTRIES_FILT_TASK_QUEUE      100
#define MAX_ENTRIES_CNTRL_TASK_QUEUE      50
#define MAX_ENTRIES_LOW_PRI_TASK_QUEUE   100

#define FILT_TASK_PRIORITY                13    // 1 higher than UI
#define CNTRL_TASK_PRIORITY               11    // 1 lower than UI
#define LOW_PRI_TASK_PRIORITY             10    // 1 lower than cntrl task

#define FILT_TASK_STACK_SIZE            4096
#define CNTRL_TASK_STACK_SIZE           4096
#define LOW_PRI_TASK_STACK_SIZE         4096


// other stuff
#define SECT_FILTER_MASK_SIZE              8
#define NUM_SECT_RECEIVED_BYTES      (256 / 8)

#define NUM_PMT_OBSERVERS                  4


//---local typedef structs for this file---------------------------------------

// message queue structures

// filter structures, enums etc
typedef struct table_list_entry
{
   struct table_list_entry *next;
   U8BIT tid;
   U16BIT xtid;
   U8BIT version;
   U16BIT num_sect_reqd;
   U16BIT num_sect_received;
   U8BIT sect_received[NUM_SECT_RECEIVED_BYTES];
   U8BIT ev_sched_sect_reqd[NUM_SECT_RECEIVED_BYTES];
   BOOLEAN complete;
   BOOLEAN new_table;
   U8BIT atre_count;
   SI_SECTION_RECORD *section_list;
   SI_SECTION_RECORD *segment_list[NUM_SECT_RECEIVED_BYTES];
   BOOLEAN ev_sched_segment_notified[NUM_SECT_RECEIVED_BYTES];
   enum {VALID, DELETED} state;
} TABLE_LIST_ENTRY;

typedef struct filter_status
{
   struct filter_status *next;
   U32BIT id;
   U8BIT path;
   BOOLEAN low_priority;
   BOOLEAN crc;
   E_SI_TABLE_FORMAT_TYPE table_format;
   E_SI_REQUEST_TYPE req_type;
   void (*return_fn)(void *, U32BIT, SI_TABLE_RECORD *);
   U32BIT return_param;
   U16BIT pid;
   U16BIT pfid;
   U16BIT mfid;
   U8BIT match_array[SECT_FILTER_MASK_SIZE];
   U8BIT mask_array[SECT_FILTER_MASK_SIZE];
   U16BIT buff_len;
   U8BIT *data_start;
   U8BIT *data_end;
   U16BIT num_tables_reqd;
   U16BIT num_tables_received;
   TABLE_LIST_ENTRY *table_list;
   TABLE_LIST_ENTRY *last_rxed_table;
   #ifdef DEBUG_FILTER_PERFORMANCE
   U32BIT start_time;
   #endif
   BOOLEAN traversing;
} FILTER_STATUS;

typedef struct cntrl_task_msg
{
   U8BIT path;
   enum {SI_EVENT, TABLE_READY, SECTION_RECEIVED, SEGMENT_READY} type;
   FILTER_STATUS *flt_ptr;
   TABLE_LIST_ENTRY *tbl_entry;
   SI_SECTION_RECORD *sec_rec;
   U32BIT numval;
} CNTRL_TASK_MSG;

//---local (static) variable declarations for this file------------------------
//   (internal variables declared static to make them local)

static void *cntrl_task_ptr;
static void *low_pri_task_ptr;
static void *filt_task_ptr;

static void *cntrl_task_msg_queue;
static void *low_pri_task_msg_queue;
static void *filt_task_msg_queue;

static void *table_mutex;
static void *section_sema;
static void *filter_mutex;

// status variable
static U8BIT num_paths;
static E_STB_SI_STATUS *si_status;
static BOOLEAN cntrl_task_stop_pending;
static BOOLEAN low_pri_task_stop_pending;

// linked list of filters
static FILTER_STATUS *filter_list;

// pointer to application si interface
static void (*app_si_interface_ptr)(U8BIT, E_APP_SI_EVENT_TYPE);

// function to be called to pass pmt to mheg interface
static F_PmtObserver pmt_observers[NUM_PMT_OBSERVERS] = {0, 0, 0, 0};

//---local function prototypes for this file-----------------------------------
//   (internal functions declared static to make them local)

static void SiControlTask(void *notused);
static void SiFilterTask(void *notused);
static void SiLowPriTableTask(void *notused);

static FILTER_STATUS* GetFilter(U8BIT path, U16BIT pid, U8BIT tid_match, U8BIT tid_mask,
   E_SI_TABLE_FORMAT_TYPE format, U16BIT expected_tables,
   U16BIT xtid_match, U16BIT xtid_mask, E_SI_REQUEST_TYPE req_type,
   U16BIT buff_size, BOOLEAN crc, BOOLEAN low_priority,
   void (*ret_fn)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param);
static void StartFilter(FILTER_STATUS *filter_ptr);
static void StopFilter(FILTER_STATUS *filter_ptr);
static void ModifyFilter(FILTER_STATUS *filter_ptr, U8BIT tid_match, U8BIT tid_mask,
   U16BIT xtid_match, U16BIT xtid_mask, U16BIT expected_tables);
static void ReleaseFilter(FILTER_STATUS *filter_ptr);
static BOOLEAN ValidateFilterPtr(FILTER_STATUS *filter_ptr);

static void HandleSiInterrupt(U8BIT path, U16BIT bytes, U16BIT pfid);

static void ActionTableReadyEvent(FILTER_STATUS *filter_ptr, BOOLEAN *stop_flag_ptr);
static void ActionSectionReceivedEvent(FILTER_STATUS *filter_ptr, TABLE_LIST_ENTRY *table_entry,
   SI_SECTION_RECORD *section_rec, BOOLEAN *stop_flag_ptr);
static void ActionSegmentReadyEvent(FILTER_STATUS *filter_ptr, TABLE_LIST_ENTRY *table_entry,
   U32BIT sect_num, BOOLEAN *stop_flag_ptr);

static TABLE_LIST_ENTRY* AddTableListEntry(FILTER_STATUS *filter_ptr, U8BIT tid, U16BIT xtid,
   U8BIT ver, U16BIT num_sect);
static void ResetTableListEntry(TABLE_LIST_ENTRY *table_entry, U8BIT ver, U16BIT last_sect);
static TABLE_LIST_ENTRY* FindTableListEntry(FILTER_STATUS *filter_ptr, U8BIT tid, U16BIT xtid);
static void FreeTableList(FILTER_STATUS *filter_ptr);
static void ProcessTableEntries(FILTER_STATUS *filter_ptr);
static SI_SECTION_RECORD* AddSectionRecord(TABLE_LIST_ENTRY *table_entry, U8BIT sect_num, U16BIT data_len,
   U8BIT *data_ptr, BOOLEAN add_to_table, BOOLEAN add_to_segment_list);
static void FreeSectionList(TABLE_LIST_ENTRY *table_entry);



//--- structures, variables etc associated with the control of private data-------------------------



//---local function definitions------------------------------------------------

/**
 *

 *
 * @brief   STB layer SI module control task.
 *

 *

 *
 */
static void SiControlTask(void *notused)
{
   CNTRL_TASK_MSG msg;
   BOOLEAN msg_ready;
   FILTER_STATUS *filter_ptr;

   FUNCTION_START(SiControlTask);

   // fool compiler into thinking these parameters are used - stops unnecessary warning
   USE_UNWANTED_PARAM(notused);

   while (1)
   {
      // Waits for an event to be reported, or the next 100 ms to elapse
      msg_ready = STB_OSReadQueue(cntrl_task_msg_queue, (void *)&msg, sizeof(CNTRL_TASK_MSG), 100);
      if (msg_ready == TRUE)
      {
         if ((cntrl_task_stop_pending == TRUE) &&
             ((msg.type != SI_EVENT) || (msg.numval != SI_EVENT_STOP)))
         {
            // ignore all events other than stop when stop event is pending
            STB_SI_PRINT(("SICtask - s.e.p. ignore"));
         }
         else
         {
            if (msg.type == SI_EVENT)
            {
               // message is SI event - maintain status and pass event to application
               switch (msg.numval)
               {
                  case SI_EVENT_STOP:
                  {
                     STB_SI_PRINT(("SITask(%d) stop", msg.path));
                     if (app_si_interface_ptr != NULL)
                     {
                        (app_si_interface_ptr)(msg.path, STOP_SI);
                     }
                     si_status[msg.path] = SI_STOPPED;
                     cntrl_task_stop_pending = FALSE;

                     // wait for low priority task to get its stop event...
                     STB_SI_PRINT(("SITask(%d) wait for low priority stop", msg.path));
                     while (low_pri_task_stop_pending == TRUE)
                     {
                        STB_OSTaskDelay(10);
                     }
                     STB_SI_PRINT(("SITask(%d) finished waiting for low priority stop", msg.path));
                  }
                  break;

                  case SI_EVENT_SEARCH:
                  {
                     STB_SI_PRINT(("SITask(%d) starting search - %dMHz",
                                   msg.path, STB_DPGetFrequency(msg.path)));
                     if (app_si_interface_ptr != NULL)
                     {
                        (app_si_interface_ptr)(msg.path, START_SI_SEARCHING);
                     }
                     si_status[msg.path] = SI_SEARCHING;
                  }
                  break;

                  case SI_EVENT_UPDATE:
                  {
                     STB_SI_PRINT(("SITask(%d) starting update - %dMHz",
                                   msg.path, STB_DPGetFrequency(msg.path)));
                     if (app_si_interface_ptr != NULL)
                     {
                        (app_si_interface_ptr)(msg.path, START_SI_UPDATING_NEW_TRANSPORT);
                     }
                     si_status[msg.path] = SI_UPDATING;
                  }
                  break;

                  case SI_EVENT_SERVICE_CHANGE:
                  {
                     STB_SI_PRINT(("SITask(%d) channel change", msg.path));
                     if (app_si_interface_ptr != NULL)
                     {
                        (app_si_interface_ptr)(msg.path, START_SI_UPDATING_SAME_TRANSPORT);
                     }
                  }
                  break;
               }
            }
            else if (msg.type == TABLE_READY)
            {
               // TABLE READY event - msg.flt_ptr is a filter_ptr
               filter_ptr = msg.flt_ptr;
               #ifdef DEBUG_SI_FILTER
               STB_SI_PRINT(("SITask(%d) table ready filter %p", msg.path, filter_ptr));
               #endif
               ActionTableReadyEvent(filter_ptr, &cntrl_task_stop_pending);
            }
            else if (msg.type == SECTION_RECEIVED)
            {
               /* Section received event */
               filter_ptr = msg.flt_ptr;
               #ifdef DEBUG_SI_FILTER
               STB_SI_PRINT(("SITask(%d) section received filter %p", msg.path, filter_ptr));
               #endif
               ActionSectionReceivedEvent(filter_ptr, msg.tbl_entry,
                  msg.sec_rec, &cntrl_task_stop_pending);
            }
            else if (msg.type == SEGMENT_READY)
            {
               // TABLE READY event - msg.flt_ptr is a filter_ptr
               filter_ptr = msg.flt_ptr;
               #ifdef DEBUG_SI_FILTER
               STB_SI_PRINT(("SITask(%d) segment ready filter %p", msg.path, filter_ptr));
               #endif
               ActionSegmentReadyEvent(filter_ptr, msg.tbl_entry, msg.numval,
                  &cntrl_task_stop_pending);
            }
         }
      }
      else
      {
         // timeout - report it to the application
         if (app_si_interface_ptr != NULL)
         {
            (app_si_interface_ptr)(INVALID_RES_ID, SI_TIMEOUT);
         }
      }
   }

   FUNCTION_FINISH(SiControlTask);
}

/**
 *

 *
 * @brief   STB layer SI module low priority table handling task
 *

 *

 *
 */
static void SiLowPriTableTask(void *notused)
{
   CNTRL_TASK_MSG msg;
   BOOLEAN msg_ready;
   FILTER_STATUS *filter_ptr;

   FUNCTION_START(SiLowPriTableTask);

   // fool compiler into thinking these parameters are used - stops unnecessary warning
   USE_UNWANTED_PARAM(notused);
   while (1)
   {
      // Waits for an event to be reported
      msg_ready = STB_OSReadQueue(low_pri_task_msg_queue, (void *)&msg, sizeof(CNTRL_TASK_MSG),
            TIMEOUT_NEVER);
      if (msg_ready == TRUE)
      {
         if ((low_pri_task_stop_pending == TRUE) &&
             ((msg.type != SI_EVENT) || (msg.numval != SI_EVENT_STOP)))
         {
            // ignore all events other than stop when stop event is pending
            STB_SI_PRINT(("SI Low Pri task - s.e.p. ignore"));
         }
         else
         {
            if ((msg.type == SI_EVENT) && (msg.numval == SI_EVENT_STOP))
            {
               STB_SI_PRINT(("SI low pri task(%d) stop", msg.path));
               low_pri_task_stop_pending = FALSE;
            }
            else if (msg.type == TABLE_READY)
            {
               // TABLE READY event - msg.flt_ptr is a filter_ptr
               filter_ptr = msg.flt_ptr;
               #ifdef DEBUG_SI_FILTER
               STB_SI_PRINT(("SI low priority(%d) table ready filter %p", msg.path, filter_ptr));
               #endif
               ActionTableReadyEvent(filter_ptr, &low_pri_task_stop_pending);
            }
            else if (msg.type == SECTION_RECEIVED)
            {
               // TABLE READY event - msg.flt_ptr is a filter_ptr
               filter_ptr = msg.flt_ptr;
               #ifdef DEBUG_SI_FILTER
               STB_SI_PRINT(("SI low priority(%d) section received filter %p", msg.path, filter_ptr));
               #endif
               ActionSectionReceivedEvent(filter_ptr, msg.tbl_entry,
                  msg.sec_rec, &low_pri_task_stop_pending);
            }
            else if (msg.type == SEGMENT_READY)
            {
               // TABLE READY event - msg.flt_ptr is a filter_ptr
               filter_ptr = msg.flt_ptr;
               #ifdef DEBUG_SI_FILTER
               STB_SI_PRINT(("SI low priority(%d) segment ready filter %p", msg.path, filter_ptr));
               #endif
               ActionSegmentReadyEvent(filter_ptr, msg.tbl_entry, msg.numval,
                  &low_pri_task_stop_pending);
            }
         }
      }
   }

   FUNCTION_FINISH(SiLowPriTableTask);
}

/**
 * @brief   Actions segment ready event for Control Task and Low Priority Table Task. Sends
 *                Table records to the application
 * @param   filter_ptr pointer to the filter status block
 * @param   table_entry table that has the segment that is complete
 * @param   sect_num section number that completed the segment
 * @param   stop_flag_ptr pointer to the appropriate stop pending flag
 */
static void ActionSegmentReadyEvent(FILTER_STATUS *filter_ptr, TABLE_LIST_ENTRY *table_entry,
   U32BIT sect_num, BOOLEAN *stop_flag_ptr)
{
   static U8BIT actn_tbl_rdy_evt_cnt = 0;
   SI_TABLE_RECORD *table_rec;
   BOOLEAN filter_ok;
   BOOLEAN filter_changed;
   U32BIT saved_id;
   U8BIT save_atre;
   U8BIT segment_num;

   FUNCTION_START(ActionSegmentReadyEvent);

   STB_OSMutexLock(filter_mutex);

   filter_ok = ValidateFilterPtr(filter_ptr);
   if (filter_ok == TRUE)
   {
      save_atre = actn_tbl_rdy_evt_cnt++;

      filter_ptr->traversing = TRUE;
      filter_changed = FALSE;

      STB_OSMutexLock(table_mutex);

      if ((table_entry != NULL) && (*stop_flag_ptr == FALSE))
      {
         table_entry->atre_count = save_atre;

         // make a table record to pass to callback function
         table_rec = STB_GetMemory(sizeof(SI_TABLE_RECORD));
         if (table_rec != NULL)
         {
            // set up the table record
            table_rec->path = filter_ptr->path;
            table_rec->tid = table_entry->tid;
            table_rec->xtid = table_entry->xtid;
            table_rec->pid = filter_ptr->pid;
            table_rec->version = table_entry->version;
            table_rec->num_sect = table_entry->num_sect_received;

            // swap the section list from the table entry to the table record
            segment_num = sect_num / 8;
            STB_OSSemaphoreWait(section_sema);
            table_rec->section_list = table_entry->segment_list[segment_num];
            table_entry->segment_list[segment_num] = NULL;
            STB_OSSemaphoreSignal(section_sema);

            // pass table record to return function - it will call STB_SIReleaseTableRec
            // when it has finished with it
            #ifdef DEBUG_TABLE_REQUESTS
            STB_SI_PRINT(("report table  - rec %p", table_rec));
            #endif
            saved_id = filter_ptr->id;

            (filter_ptr->return_fn)(filter_ptr, filter_ptr->return_param, table_rec);

            // action function may have cancelled or restarted the filter so we must check here
            // that it is valid to continue through the table list...
            //
            // ...first check that the filter_ptr is still valid - if not then exit the loop
            filter_ok = ValidateFilterPtr(filter_ptr);
            if (filter_ok == FALSE)
            {
               filter_changed = TRUE;
            }
            else
            {
               // filter_ptr is still in the list but...
               //
               // the action function may have restarted or modified the filter (using
               // STB_SIRestartTableRequest or STB_SIModifyTableRequest respectively) both of
               // which would have deleted the table list which we are traversing.
               //
               // in addition it is possible for the action function to cancel one
               // filter and then get another which happens to be allocated the same address in
               // memory. In this case ValidateFilterPtr() would have correctly returned true
               // because the same filter_ptr is in the list of filters.
               //
               // so, we protect against these problems by checking the filter id. The id is set
               // each time StartFilter() is called - i.e. when a new filter is requested, or an
               // existing filter is modified or restarted. Before the call to the action
               // function the filter id was copied into saved_id - if it is not the same now
               // the filter has been changed. In this case filter_ptr and/or table_entry are no
               // longer valid.
               if (filter_ptr->id != saved_id)
               {
                  filter_changed = TRUE;
               }
               else
               {
                  table_entry->complete = FALSE;
               }
            }
         }
      }

      if (filter_changed == FALSE)
      {
         filter_ptr->traversing = FALSE;
         ProcessTableEntries(filter_ptr);
      }

      STB_OSMutexUnlock(table_mutex);
   }

   STB_OSMutexUnlock(filter_mutex);

   FUNCTION_FINISH(ActionSegmentReadyEvent);
}

/**
 *

 *
 * @brief   Actions table ready event for Control Task and Low Priority Table Task. Sends
 *                Table records to the application
 *
 * @param   filter_ptr    - pointer to the filter status block
 * @param   stop_flag_ptr - pointer to the appropriate stop pending flag
 *

 *
 */
static void ActionTableReadyEvent(FILTER_STATUS *filter_ptr, BOOLEAN *stop_flag_ptr)
{
   static U8BIT actn_tbl_rdy_evt_cnt = 0;
   TABLE_LIST_ENTRY *table_entry;
   SI_TABLE_RECORD *table_rec;
   BOOLEAN filter_ok;
   BOOLEAN filter_changed;
   U32BIT saved_id;
   U8BIT save_atre;

   FUNCTION_START(ActionTableReadyEvent);

   STB_OSMutexLock(filter_mutex);

   filter_ok = ValidateFilterPtr(filter_ptr);
   if (filter_ok == TRUE)
   {
      save_atre = actn_tbl_rdy_evt_cnt++;
      // check filter for any complete tables
      filter_ptr->traversing = TRUE;
      filter_changed = FALSE;

      STB_OSMutexLock(table_mutex);

      /* Find first valid table entry */
      table_entry = filter_ptr->table_list;
      while ((table_entry != NULL) && (table_entry->state != VALID))
      {
         table_entry = table_entry->next;
      }

      while ((table_entry != NULL) && (*stop_flag_ptr == FALSE))
      {
         table_entry->atre_count = save_atre;
         if (table_entry->complete == TRUE)
         {
            // make a table record to pass to callback function
            table_rec = STB_GetMemory(sizeof(SI_TABLE_RECORD));
            if (table_rec != NULL)
            {
               // set up the table record
               table_rec->path = filter_ptr->path;
               table_rec->tid = table_entry->tid;
               table_rec->xtid = table_entry->xtid;
               table_rec->pid = filter_ptr->pid;
               table_rec->version = table_entry->version;
               table_rec->num_sect = table_entry->num_sect_received;

               // swap the section list from the table entry to the table record
               STB_OSSemaphoreWait(section_sema);
               table_rec->section_list = table_entry->section_list;
               table_entry->section_list = NULL;
               STB_OSSemaphoreSignal(section_sema);

               // pass table record to return function - it will call STB_SIReleaseTableRec
               // when it has finished with it
               #ifdef DEBUG_TABLE_REQUESTS
               STB_SI_PRINT(("report table  - rec %p", table_rec));
               #endif
               saved_id = filter_ptr->id;

               (filter_ptr->return_fn)(filter_ptr, filter_ptr->return_param, table_rec);

               // action function may have cancelled or restarted the filter so we must check here
               // that it is valid to continue through the table list...
               //
               // ...first check that the filter_ptr is still valid - if not then exit the loop
               filter_ok = ValidateFilterPtr(filter_ptr);
               if (filter_ok == FALSE)
               {
                  filter_changed = TRUE;
                  break;
               }
               else
               {
                  // filter_ptr is still in the list but...
                  //
                  // the action function may have restarted or modified the filter (using
                  // STB_SIRestartTableRequest or STB_SIModifyTableRequest respectively) both of
                  // which would have deleted the table list which we are traversing.
                  //
                  // in addition it is possible for the action function to cancel one
                  // filter and then get another which happens to be allocated the same address in
                  // memory. In this case ValidateFilterPtr() would have correctly returned true
                  // because the same filter_ptr is in the list of filters.
                  //
                  // so, we protect against these problems by checking the filter id. The id is set
                  // each time StartFilter() is called - i.e. when a new filter is requested, or an
                  // existing filter is modified or restarted. Before the call to the action
                  // function the filter id was copied into saved_id - if it is not the same now
                  // the filter has been changed. In this case filter_ptr and/or table_entry are no
                  // longer valid.
                  if (filter_ptr->id != saved_id)
                  {
                     filter_changed = TRUE;
                     break;
                  }
                  else
                  {
                     table_entry->complete = FALSE;
                  }
               }
            }
         }
         // move to next table
         table_entry = filter_ptr->table_list;
         while ((table_entry != NULL) &&
                ((table_entry->state != VALID) || (table_entry->atre_count == save_atre)))
         {
            table_entry = table_entry->next;
         }
      }

      if (filter_changed == FALSE)
      {
         filter_ptr->traversing = FALSE;
         ProcessTableEntries(filter_ptr);
      }

      STB_OSMutexUnlock(table_mutex);
   }

   STB_OSMutexUnlock(filter_mutex);

   FUNCTION_FINISH(ActionTableReadyEvent);
}

/**
 *

 *
 * @brief   Actions table ready event for Control Task and Low Priority Table Task. Sends
 *                Table records to the application
 *
 * @param   filter_ptr    - pointer to the filter status block
 * @param   stop_flag_ptr - pointer to the appropriate stop pending flag
 *

 *
 */
static void ActionSectionReceivedEvent(FILTER_STATUS *filter_ptr, TABLE_LIST_ENTRY *table_entry,
   SI_SECTION_RECORD *section_rec, BOOLEAN *stop_flag_ptr)
{
   SI_TABLE_RECORD *table_rec;
   BOOLEAN filter_ok;
   BOOLEAN filter_changed;
   U32BIT saved_id;

   FUNCTION_START(ActionSectionReceivedEvent);

   STB_OSMutexLock(filter_mutex);

   filter_ok = ValidateFilterPtr(filter_ptr);
   if (filter_ok == TRUE)
   {
      // check filter for any complete tables
      filter_ptr->traversing = TRUE;
      filter_changed = FALSE;

      if ((table_entry != NULL) && (*stop_flag_ptr == FALSE))
      {
         // make a table record to pass to callback function
         table_rec = STB_GetMemory(sizeof(SI_TABLE_RECORD));
         if (table_rec != NULL)
         {
            // set up the table record
            table_rec->path = filter_ptr->path;
            table_rec->tid = table_entry->tid;
            table_rec->xtid = table_entry->xtid;
            table_rec->pid = filter_ptr->pid;
            table_rec->version = table_entry->version;
            table_rec->num_sect = 1;
            table_rec->section_list = section_rec;

            // pass table record to return function - it will call STB_SIReleaseTableRec
            // when it has finished with it
            #ifdef DEBUG_TABLE_REQUESTS
            STB_SI_PRINT(("report table  - rec %p", table_rec));
            #endif
            saved_id = filter_ptr->id;
            (filter_ptr->return_fn)(filter_ptr, filter_ptr->return_param, table_rec);

            // action function may have cancelled or restarted the filter so we must check here
            // that it is valid to continue through the table list...
            //
            // ...first check that the filter_ptr is still valid - if not then exit the loop
            filter_ok = ValidateFilterPtr(filter_ptr);
            if (filter_ok == FALSE)
            {
               filter_changed = TRUE;
            }
            else
            {
               // filter_ptr is still in the list but...
               //
               // the action function may have restarted or modified the filter (using
               // STB_SIRestartTableRequest or STB_SIModifyTableRequest respectively) both of
               // which would have deleted the table list which we are traversing.
               //
               // in addition it is possible for the action function to cancel one
               // filter and then get another which happens to be allocated the same address in
               // memory. In this case ValidateFilterPtr() would have correctly returned true
               // because the same filter_ptr is in the list of filters.
               //
               // so, we protect against these problems by checking the filter id. The id is set
               // each time StartFilter() is called - i.e. when a new filter is requested, or an
               // existing filter is modified or restarted. Before the call to the action
               // function the filter id was copied into saved_id - if it is not the same now
               // the filter has been changed. In this case filter_ptr and/or table_entry are no
               // longer valid.
               if (filter_ptr->id != saved_id)
               {
                  filter_changed = TRUE;
               }
               else
               {
                  table_entry->complete = FALSE;
               }
            }
         }
      }

      if (filter_changed == FALSE)
      {
         filter_ptr->traversing = FALSE;
         ProcessTableEntries(filter_ptr);
      }
   }

   STB_OSMutexUnlock(filter_mutex);

   FUNCTION_FINISH(ActionSectionReceivedEvent);
}

/**
 *

 *
 * @brief   STB layer SI module filter task - handles output from pid/section filters
 *

 *

 *
 */
static void SiFilterTask(void *notused)
{
   BOOLEAN msg_ready;
   FILTER_STATUS *filter_ptr;
   BOOLEAN filter_ok;
   U8BIT *data_ptr;
   U16BIT sect_len;
   U8BIT tid;
   U16BIT xtid;
   U8BIT version;
   U8BIT sect;
   TABLE_LIST_ENTRY *table_entry;
   BOOLEAN sect_reqd;
   SI_SECTION_RECORD *section_rec;
   BOOLEAN finished;
   BOOLEAN table_ready;
   U16BIT last_sect;
   CNTRL_TASK_MSG msg;
   BOOLEAN all_sect_recvd;
   BOOLEAN use_data;
   U16BIT i;

   FUNCTION_START(SiFilterTask);

   // fool compiler into thinking these parameters are used - stops unnecessary warning
   USE_UNWANTED_PARAM(notused);

   while (1)
   {
      // Waits for a section received to be reported
      msg_ready = STB_OSReadQueue(filt_task_msg_queue, (void *)&filter_ptr, sizeof(FILTER_STATUS *),
            TIMEOUT_NEVER);
      if (msg_ready == TRUE)
      {
         STB_OSMutexLock(filter_mutex);

         filter_ok = ValidateFilterPtr(filter_ptr);
         if (filter_ok == FALSE)
         {
            #ifdef DEBUG_SI_FILTER
            STB_SI_PRINT(("ActionSectRxed: filter %p - invalid filter", filter_ptr));
            #endif
         }
         else
         {
            table_ready = FALSE;

            // process all the sections in the buffer at the moment if any
            finished = FALSE;
            data_ptr = filter_ptr->data_start;
            while ((data_ptr < filter_ptr->data_end) && (finished == FALSE))
            {
               use_data = TRUE;

               tid = data_ptr[0];
               sect_len = (((data_ptr[1] & 0x0f) << 8) | data_ptr[2]) + 3;

               // section number is only relevant for multi-section formats
               if (filter_ptr->table_format == MULTI_SECT)
               {
                  // first check if it is a current section - only accept currents
                  if ((data_ptr[5] & 0x01) == 0)
                  {
                     // not a current table
                     use_data = FALSE;
#ifdef DEBUG_SI_FILTER_INT
                     STB_SI_PRINT(("SiFilterTask: Section isn't for a current table"));
#endif
                  }
                  else
                  {
                     // find the xtid, version and section number from the data
                     xtid = (data_ptr[3] << 8) | data_ptr[4];
                     version = (data_ptr[5] >> 1) & 0x1f;
                     sect = data_ptr[6];

                     STB_OSMutexLock(table_mutex);

                     table_entry = FindTableListEntry(filter_ptr, tid, xtid);
                     if (table_entry != NULL)
                     {
                        if ((filter_ptr->req_type == ONE_SHOT_REQUEST) && (table_entry->complete == TRUE))
                        {
                           // if this is a one shot filter and the table is already complete then we
                           // don't need this section even if it is a new version. The filter might
                           // still be running because there are other tables expected
                           use_data = FALSE;
#ifdef DEBUG_SI_FILTER_INT
                           STB_SI_PRINT(("SiFilterTask: Section is for an already complete table"));
#endif
                        }
                        else
                        {
                           if (table_entry->version == version)
                           {
                              if (((table_entry->sect_received[sect >> 3] >> (sect & 0x07)) & 0x01) != 0)
                              {
                                 // already got it - return false
                                 use_data = FALSE;
#ifdef DEBUG_SI_FILTER_INT
                                 STB_SI_PRINT(("SiFilterTask: Section has already been received"));
#endif
                              }
                           }
                        }
                     }

                     STB_OSMutexUnlock(table_mutex);
                  }
               }

               if (use_data == TRUE)
               {
                  #ifdef DEBUG_SI_FILTER
                  STB_SI_PRINT(("ActionSectRxed: filter %p dptr %p (data %p to %p)", filter_ptr,
                                data_ptr, filter_ptr->data_start, filter_ptr->data_end));
                  #endif
                  if (filter_ptr->table_format == SINGLE_SECT)
                  {
                     xtid = 0;
                     version = 0;
                     sect = 0;
                     last_sect = 0;
                     #ifdef DEBUG_SI_FILTER
                     STB_SI_PRINT(("single-sect format: len %d tab 0x%02x", sect_len, tid));
                     #endif
                  }
                  else
                  {
                     // it is a multi section format...
                     // find the xtid, version and section number from the data
                     xtid = (data_ptr[3] << 8) | data_ptr[4];
                     version = (data_ptr[5] >> 1) & 0x1f;
                     sect = data_ptr[6];
                     last_sect = data_ptr[7];

                     #ifdef DEBUG_SI_FILTER
                     STB_SI_PRINT(("multi-sect format: len %d tab 0x%02x/0x%04x ver %d sect %d",
                                   sect_len, tid, xtid, version, sect));
                     #endif
                  }

                  sect_reqd = FALSE;

                  STB_OSMutexLock(table_mutex);

                  table_entry = FindTableListEntry(filter_ptr, tid, xtid);
                  if (table_entry == NULL)
                  {
                     table_entry = AddTableListEntry(filter_ptr, tid, xtid, version, last_sect);
                     if (table_entry != NULL)
                     {
                        sect_reqd = TRUE;
                     }
                  }
                  else
                  {
                     // update last_rxed_table
                     filter_ptr->last_rxed_table = table_entry;

                     // if the table is not complete check if this section is required
                     if (table_entry->complete == FALSE)
                     {
                        if (filter_ptr->table_format == SINGLE_SECT)
                        {
                           // single section format
                           sect_reqd = TRUE;
                        }
                        else
                        {
                           // multi-section format - check for version change
                           if (table_entry->version != version)
                           {
                              #ifdef DEBUG_SI_FILTER
                              STB_SI_PRINT(("Free existing table for new version"));
                              #endif
                              // free any existing sections because they are out of date
                              ResetTableListEntry(table_entry, version, last_sect);
                              sect_reqd = TRUE;
                           }
                           else
                           {
                              // same version - have we already got this section
                              if (((table_entry->sect_received[sect >> 3] >> (sect & 0x07)) & 0x01) == 0)
                              {
                                 // not got the section - add it
                                 sect_reqd = TRUE;
                              }
                           }
                        }
                     }
                  }

                  STB_OSMutexUnlock(table_mutex);

                  if (sect_reqd == TRUE)
                  {
                     all_sect_recvd = FALSE;

#ifndef EIT_REPORT_SEGMENTS
                     /* Pass each section from an EIT table to the app as it arrives */
                     if ((0x50 <= tid) && (tid <= 0x6f))
                     {
                        // add the section - copy the whole section data incl header and crc
                        section_rec = AddSectionRecord(table_entry, sect, sect_len, data_ptr, FALSE, FALSE);
                        if (section_rec != NULL)
                        {
                           msg.path = filter_ptr->path;
                           msg.type = SECTION_RECEIVED;
                           msg.flt_ptr = filter_ptr;
                           msg.tbl_entry = table_entry;
                           msg.sec_rec = section_rec;
                           if (filter_ptr->low_priority == TRUE)
                           {
                              /* Allow writes to this queue to timeout to avoid lockups */
                              if (STB_OSWriteQueue(low_pri_task_msg_queue, (void *)&msg,
                                 sizeof(CNTRL_TASK_MSG), TIMEOUT_NOW) == FALSE)
                              {
                                 /* Mark the section as not received, and try to notify it later */
                                 table_entry->sect_received[sect >> 3] &= ~(1 << (sect & 0x07));

                                 /* section_rec pointer hasn't been added to the list in table_entry
                                  * (AddSectionRecord called with FALSE), so we only need to free it */
                                 STB_FreeMemory(section_rec);
                              }
                           }
                           else
                           {
                              STB_OSWriteQueue(cntrl_task_msg_queue, (void *)&msg, sizeof(CNTRL_TASK_MSG), TIMEOUT_NEVER);
                           }
                        }
                     }
                     else
#endif
                     {
                        /* Check if table is complete */
                        if ((0x50 <= tid) && (tid <= 0x6f))
                        {
                           /* Add the section - copy the whole section data incl header and crc */
                           section_rec = AddSectionRecord(table_entry, sect, sect_len, data_ptr, TRUE, TRUE);
                           if (section_rec != NULL)
                           {
                              /* Check if all sections received for this segment */
                              for (i = 0; i < NUM_SECT_RECEIVED_BYTES; i++)
                              {
                                 if (!table_entry->ev_sched_segment_notified[i] &&
                                    (table_entry->ev_sched_sect_reqd[i] == table_entry->sect_received[i]))
                                 {
                                    /* All sections for this segment have been received and it
                                     * hasn't yet been notified */
                                    msg.path = filter_ptr->path;
                                    msg.type = SEGMENT_READY;
                                    msg.flt_ptr = filter_ptr;
                                    msg.tbl_entry = table_entry;
                                    msg.numval = sect;
                                    if (filter_ptr->low_priority)
                                    {
                                       if (STB_OSWriteQueue(low_pri_task_msg_queue, (void *)&msg,
                                          sizeof(CNTRL_TASK_MSG), TIMEOUT_NOW))
                                       {
                                          table_entry->ev_sched_segment_notified[i] = TRUE;
                                       }
                                    }
                                    else
                                    {
                                       STB_OSWriteQueue(cntrl_task_msg_queue, (void *)&msg,
                                          sizeof(CNTRL_TASK_MSG), TIMEOUT_NEVER);

                                       table_entry->ev_sched_segment_notified[i] = TRUE;
                                    }
                                 }
                              }
                           }
                        }
                        else
                        {
                           /* Add the section - copy the whole section data incl header and crc */
                           section_rec = AddSectionRecord(table_entry, sect, sect_len, data_ptr, TRUE, FALSE);
                           if (section_rec != NULL)
                           {
                              /* For non-event schedule tables check if all sections arrived */
                              all_sect_recvd = (table_entry->num_sect_received >= table_entry->num_sect_reqd);
                           }
                        }
                     }

                     // for non-event schedule tables check if all sections arrived
                     if (all_sect_recvd == TRUE)
                     {
                        // mark table record complete
                        #ifdef DEBUG_SI_FILTER
                        STB_SI_PRINT(("Table complete"));
                        #endif
                        #ifdef DEBUG_FILTER_PERFORMANCE
                        STB_SI_PRINT(("SI filter %p table complete @ %dms", filter_ptr, STB_OSGetClockDiff(filter_ptr->start_time)));
                        #endif
                        table_entry->complete = TRUE;
                        table_ready = TRUE;
                        if (table_entry->new_table == TRUE)
                        {
                           filter_ptr->num_tables_received++;
                           table_entry->new_table = FALSE;
                        }

                        // check if filter is done
                        if (filter_ptr->req_type == ONE_SHOT_REQUEST)
                        {
                           // this is a one-shot filter - if all tables have arrived then stop the filter
                           if (filter_ptr->num_tables_received >= filter_ptr->num_tables_reqd)
                           {
                              StopFilter(filter_ptr);
                              finished = TRUE;
                           }
                        }
                     }
                  }
               }

               // move data_ptr past this section
               data_ptr += sect_len;
            }

            // done all relevant sections in the buffer - reset the buffer pointer to the start
            filter_ptr->data_end = filter_ptr->data_start;

            // if a table is ready (could be more than one table) send a message to the control task
            if (table_ready)
            {
               msg.path = filter_ptr->path;
               msg.type = TABLE_READY;
               msg.flt_ptr = filter_ptr;
               if (filter_ptr->low_priority == TRUE)
               {
                  /* Allow writes to this queue to timeout to avoid lockups */
                  STB_OSWriteQueue(low_pri_task_msg_queue, (void *)&msg, sizeof(CNTRL_TASK_MSG), TIMEOUT_NOW);
               }
               else
               {
                  STB_OSWriteQueue(cntrl_task_msg_queue, (void *)&msg, sizeof(CNTRL_TASK_MSG), TIMEOUT_NEVER);
               }
            }
         }

         STB_OSMutexUnlock(filter_mutex);
      }
   }
   FUNCTION_FINISH(SiFilterTask);
}

/**
 *

 *
 * @brief   Actions DMUX dispatcher interrupt on pids used for SI gathering. Must only copy
 *                one section, even if <bytes> indicates that there is more than one section
 *                available. If more sections are copied the link driver gets confused. This
 *                function will be called again if there are more sections to be read.
 *
 * @param   U8BIT demux_path - path of the demux originating the data
 * @param   U16BIT nbytes - number of bytes in buffer (note this may be more than one
 *                                section's worth)
 * @param   U16BIT pfid   - id of PID filter which has caused interrupt
 *

 *
 */
static void HandleSiInterrupt(U8BIT demux_path, U16BIT nbytes, U16BIT pfid)
{
   U16BIT sect_len;
   U16BIT free_len;
   BOOLEAN sect_valid;
   FILTER_STATUS *filter_ptr;
   U8BIT *data_ptr;
   U8BIT demux_num;

   STB_OSMutexLock(filter_mutex);

   // find si filter on which interrupt has occurred
   filter_ptr = filter_list;
   while (filter_ptr != NULL)
   {
      if ((filter_ptr->pfid == pfid) &&
          (STB_DPGetPathDemux(filter_ptr->path) == demux_path))
      {
         break;
      }
      filter_ptr = filter_ptr->next;
   }

   if (filter_ptr == NULL)
   {
      // filter not found - skip section to keep link driver happy
      STB_SI_ERROR(("HandleSiInterrupt(%u): Failed to find filter, pfid=%d!", demux_path, pfid));
      #ifdef DEBUG_SI_FILTER_INT
      STB_SI_PRINT(("Int - filt? skip"));
      #endif
   }
   else
   {
      // valid filter...

      #ifdef DEBUG_SI_FILTER_INT
      STB_SI_PRINT(("Int - filt %p", filter_ptr));
      #endif

      // check buffer space. We don't know how long the section is - nbytes may cover more than
      // one section. The best we can do is check that we have room for nbytes bytes - there may
      // only be one section, but if there is more than one the length of the first section will
      // be less than nbytes
      free_len = filter_ptr->buff_len - (filter_ptr->data_end - filter_ptr->data_start);
      if (nbytes > free_len)
      {
         // not enough room - skip the section
         #ifdef DEBUG_SI_FILTER_INT
         STB_SI_PRINT(("Int - no room skip"));
         #endif
         STB_DMXSkipPIDFilterSect(STB_DPGetPathDemux(filter_ptr->path), pfid);
      }
      else
      {
         // setup pointer to data for faster access
         data_ptr = filter_ptr->data_end;

         // copy single section
         demux_num = STB_DPGetPathDemux(filter_ptr->path);
         if (demux_num != INVALID_RES_ID)
         {
            sect_valid = STB_DMXCopyPIDFilterSect(demux_num, data_ptr, free_len, pfid);
            if (sect_valid == FALSE)
            {
               #ifdef DEBUG_SI_FILTER_INT
               STB_SI_PRINT(("Int - sect err"));
               #endif
            }
            else
            {
               // get section length (+3 is for table id and section length bytes
               sect_len = (((data_ptr[1] & 0x0f) << 8) | data_ptr[2]) + 3;

               // Move end pointer to end of section just received
               filter_ptr->data_end += sect_len;

               // trigger task
               if (!STB_OSWriteQueue(filt_task_msg_queue, (void *)&filter_ptr, sizeof(FILTER_STATUS *), TIMEOUT_NOW))
               {
                  /* Failed to write to the queue */
                  filter_ptr->data_end -= sect_len;
#ifdef DEBUG_SI_FILTER_INT
                  STB_SI_PRINT(("HandleSiInterrupt: Failed to write to the queue!"));
#endif
               }
            }
         }
      }
   }

   STB_OSMutexUnlock(filter_mutex);
}

/**
 *

 *
 * @brief   Sets up filter to get all sections of the si tables specified
 *
 * @param   path            - the demux path to use
 * @param   pid             - required pid
 * @param   tid_match       - required match byte for the tid in the match/mask filter
 * @param   tid_mask        - required mask byte for the tid in the match/mask filter
 * @param   format          - format of the table i.e. single or multiple section
 * @param   expected_tables - number of tables expected. Used to decide when multi-table filter is complete
 * @param   xtid_match      - required match byte for the extended tid in the match/mask filter
 * @param   xtid_mask       - required mask byte for the extended tid in the match/mask filter
 * @param   req_type        - ONE_SHOT or CONTINUOUS request
 * @param   buff_size       - size of buffer to be used
 * @param   crc             - TRUE if crc checking required, FALSE otherwise
 * @param   low_priority    - TRUE if request is lower priority
 * @param   ret_fn          - pointer to callback function for return of table
 * @param   ret_param       - parameter to be returned to callback function
 *
 * @return   pointer to filter status block or NULL if not setup
 *
 */
static FILTER_STATUS* GetFilter(U8BIT path, U16BIT pid, U8BIT tid_match, U8BIT tid_mask,
   E_SI_TABLE_FORMAT_TYPE format, U16BIT expected_tables,
   U16BIT xtid_match, U16BIT xtid_mask, E_SI_REQUEST_TYPE req_type,
   U16BIT buff_size, BOOLEAN crc, BOOLEAN low_priority,
   void (*ret_fn)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param)
{
   U16BIT pfid;
   U16BIT mfid;
   U8BIT *match_ptr;
   U8BIT *mask_ptr;
   FILTER_STATUS *filter_ptr;
   BOOLEAN success;
   U8BIT demux_num;

   FUNCTION_START(GetFilter);

   success = FALSE;

   demux_num = STB_DPGetPathDemux(path);

   if (demux_num == INVALID_RES_ID)
   {
      filter_ptr = NULL;
   }
   else
   {
      // get memory for a filter and its buffer
      filter_ptr = (FILTER_STATUS *)STB_GetMemory(sizeof(FILTER_STATUS) + buff_size);
      if (filter_ptr != NULL)
      {
         // create the match/mask arrays and setup section filter...
         match_ptr = filter_ptr->match_array;
         mask_ptr = filter_ptr->mask_array;
         memset(match_ptr, 0, SECT_FILTER_MASK_SIZE);
         memset(mask_ptr, 0, SECT_FILTER_MASK_SIZE);
         match_ptr[0] = tid_match;
         mask_ptr[0] = tid_mask;

         if (format == MULTI_SECT)
         {
            // always look for current
            match_ptr[3] = 0x01;
            mask_ptr[3] = 0x01;

            // set xtid match/mask
            match_ptr[1] = (U8BIT)(xtid_match >> 8);
            match_ptr[2] = (U8BIT)(xtid_match & 0x00ff);
            mask_ptr[1] = (U8BIT)(xtid_mask >> 8);
            mask_ptr[2] = (U8BIT)(xtid_mask & 0x00ff);
         }

         pfid = STB_DMXGrabPIDFilter(demux_num, pid, HandleSiInterrupt);

         if (pfid != STB_DMX_PID_FILTER_INVALID)
         {
            mfid = STB_DMXGrabSectFilter(demux_num, pfid);
            if (mfid == STB_DMX_SECT_FILTER_INVALID)
            {
               STB_DMXReleasePIDFilter(demux_num, pfid);
            }
            else
            {
               // setup the filter status block
               filter_ptr->next = NULL;
               filter_ptr->path = path;
               filter_ptr->table_format = format;
               filter_ptr->req_type = req_type;
               filter_ptr->return_fn = ret_fn;
               filter_ptr->return_param = ret_param;
               filter_ptr->pid = pid;
               filter_ptr->pfid = pfid;
               filter_ptr->mfid = mfid;
               filter_ptr->buff_len = buff_size;
               filter_ptr->data_start = ((U8BIT *)filter_ptr) + sizeof(FILTER_STATUS);
               filter_ptr->data_end = filter_ptr->data_start;
               filter_ptr->num_tables_reqd = expected_tables;
               filter_ptr->num_tables_received = 0;
               filter_ptr->table_list = NULL;
               filter_ptr->last_rxed_table = NULL;
               filter_ptr->low_priority = low_priority;
               filter_ptr->traversing = FALSE;

               filter_ptr->crc = crc;
               STB_DMXSetupSectFilter(demux_num, mfid, match_ptr, mask_ptr, 0, crc);

               // add the filter status block to the top of the linked list
               STB_OSMutexLock(filter_mutex);
               filter_ptr->next = filter_list;
               filter_list = filter_ptr;
               STB_OSMutexUnlock(filter_mutex);

               // indicate success
               success = TRUE;
               #ifdef DEBUG_SI_FILTER
               STB_SI_PRINT(("GetFilter: filter %p", filter_ptr));
               #endif
            }
         }
      }
   }

   if (success == FALSE)
   {
      if (filter_ptr != NULL)
      {
         STB_FreeMemory(filter_ptr);
         filter_ptr = NULL;
      }
      #ifdef DEBUG_SI_FILTER_FAILURES
      STB_SI_PRINT(("GetFilter: FAILED"));
      #endif
   }

   FUNCTION_FINISH(GetFilter);
   return(filter_ptr);
}

/**
 *

 *
 * @brief   Modify the section filtering on the specified filter (note pid cannot change)
 *
 * @param   filter_ptr      - pointer to the filter status block
 * @param   tid_match       - required match byte for the tid in the match/mask filter
 * @param   tid_mask        - required mask byte for the tid in the match/mask filter
 * @param   xtid_match      - required match byte for the extended tid in the match/mask filter
 * @param   xtid_mask       - required mask byte for the extended tid in the match/mask filter
 * @param   expected_tables - number of tables expected. Used to decide when multi-table filter is complete
 *
 * @return   pointer to filter status block or NULL if not setup
 *
 */
static void ModifyFilter(FILTER_STATUS *filter_ptr, U8BIT tid_match, U8BIT tid_mask,
   U16BIT xtid_match, U16BIT xtid_mask, U16BIT expected_tables)
{
   U8BIT *match_ptr;
   U8BIT *mask_ptr;

   FUNCTION_START(ModifyFilter);

   #ifdef DEBUG_SI_FILTER
   STB_SI_PRINT(("ModifyFilter: filter %p", filter_ptr));
   #endif

   filter_ptr->data_start = ((U8BIT *)filter_ptr) + sizeof(FILTER_STATUS);
   filter_ptr->data_end = filter_ptr->data_start;
   filter_ptr->num_tables_reqd = expected_tables;
   filter_ptr->num_tables_received = 0;

   // create the match/mask arrays and setup section filter...
   match_ptr = filter_ptr->match_array;
   mask_ptr = filter_ptr->mask_array;
   memset(match_ptr, 0, SECT_FILTER_MASK_SIZE);
   memset(mask_ptr, 0, SECT_FILTER_MASK_SIZE);
   match_ptr[0] = tid_match;
   mask_ptr[0] = tid_mask;

   if (filter_ptr->table_format == MULTI_SECT)
   {
      // always look for current
      match_ptr[3] = 0x01;
      mask_ptr[3] = 0x01;

      // set xtid match/mask
      match_ptr[1] = (U8BIT)(xtid_match >> 8);
      match_ptr[2] = (U8BIT)(xtid_match & 0x00ff);
      mask_ptr[1] = (U8BIT)(xtid_mask >> 8);
      mask_ptr[2] = (U8BIT)(xtid_mask & 0x00ff);
   }

   STB_DMXSetupSectFilter(STB_DPGetPathDemux(filter_ptr->path), filter_ptr->mfid, match_ptr,
      mask_ptr, 0, filter_ptr->crc);

   FUNCTION_FINISH(ModifyFilter);
}

/**
 *

 *
 * @brief   Starts a SI PID filter
 *
 * @param   filter_ptr - pointer to the filter status block
 *

 *
 */
static void StartFilter(FILTER_STATUS *filter_ptr)
{
   FUNCTION_START(StartFilter);

   #ifdef DEBUG_SI_FILTER
   STB_SI_PRINT(("StartFilter: %p", filter_ptr));
   #endif

   // initialise data pointer
   filter_ptr->data_end = filter_ptr->data_start;

   // update the id - used in control task to uniquely identify filter run
   filter_ptr->id = STB_OSGetClockMilliseconds();

   #ifdef DEBUG_FILTER_PERFORMANCE
   filter_ptr->start_time = STB_OSGetClockMilliseconds();
   #endif

   // start the pid filter
   STB_DMXStartPIDFilter(STB_DPGetPathDemux(filter_ptr->path), filter_ptr->pfid);

   FUNCTION_FINISH(StartFilter);
}

/**
 *

 *
 * @brief   Stops a SI PID filter
 *
 * @param   filter_ptr - pointer to the filter status block
 *

 *
 */
static void StopFilter(FILTER_STATUS *filter_ptr)
{
   U8BIT demux;

   FUNCTION_START(StopFilter);

   #ifdef DEBUG_SI_FILTER
   STB_SI_PRINT(("StopFilter: %p", filter_ptr));
   #endif

   // stop the pid filter
   demux = STB_DPGetPathDemux(filter_ptr->path);
   if (demux != INVALID_RES_ID)
   {
      STB_DMXStopPIDFilter(demux, filter_ptr->pfid);
   }

   FUNCTION_FINISH(StopFilter);
}

/**
 *

 *
 * @brief   Releases a SI PID filter and clear status entry
 *
 * @param   filter_ptr - pointer to the filter status block
 *

 *
 */
static void ReleaseFilter(FILTER_STATUS *filter_ptr)
{
   FILTER_STATUS *temp_ptr;
   FILTER_STATUS **prev_ptr;
   U8BIT demux_num;

   FUNCTION_START(ReleaseFilter);

   // find filter in linked list
   temp_ptr = filter_list;
   prev_ptr = &filter_list;
   while (temp_ptr != NULL)
   {
      if (temp_ptr == filter_ptr)
      {
         #ifdef DEBUG_SI_FILTER
         STB_SI_PRINT(("ReleaseFilter: %p", filter_ptr));
         #endif

         // found it - remove it from the linked list
         *prev_ptr = filter_ptr->next;

         // free pid and section filters
         demux_num = STB_DPGetPathDemux(filter_ptr->path);
         if (demux_num != INVALID_RES_ID)
         {
            STB_DMXReleaseSectFilter(demux_num, filter_ptr->mfid);
            STB_DMXReleasePIDFilter(demux_num, filter_ptr->pfid);
         }

         // free any memory used in table records.
         FreeTableList(filter_ptr);
         ProcessTableEntries(filter_ptr);

         // free the filter status block itself
         STB_FreeMemory((void *)filter_ptr);
         break;
      }
      prev_ptr = &temp_ptr->next;
      temp_ptr = temp_ptr->next;
   }

   FUNCTION_FINISH(ReleaseFilter);
}

/**
 *

 *
 * @brief   Checks filter_ptr is in the linked-list of filters
 *
 * @param   filter_ptr - pointer to the filter status block
 *
 * @return   TRUE if found, FALSE otherwise
 *
 */
static BOOLEAN ValidateFilterPtr(FILTER_STATUS *filter_ptr)
{
   FILTER_STATUS *temp_ptr;
   BOOLEAN ret_val;

   FUNCTION_START(ValidateFilterPtr);

   ret_val = FALSE;

   // find filter in linked list
   temp_ptr = filter_list;
   while (temp_ptr != NULL)
   {
      if (temp_ptr == filter_ptr)
      {
         ret_val = TRUE;
         break;
      }
      temp_ptr = temp_ptr->next;
   }

   FUNCTION_FINISH(ValidateFilterPtr);
   return(ret_val);
}

/**
 *

 *
 * @brief   Creates a table record, initialises it and adds it to the table list
 *
 * @param   filter_ptr - the relevant filter status block
 * @param   tid        - tid of the new table
 * @param   xtid       - xtid of the new table
 * @param   ver        - version
 * @param   last_sect  - last section in table
 *

 *
 */
static TABLE_LIST_ENTRY* AddTableListEntry(FILTER_STATUS *filter_ptr, U8BIT tid, U16BIT xtid, U8BIT ver, U16BIT last_sect)
{
   TABLE_LIST_ENTRY *table_entry;
   U16BIT i;
   U8BIT last_sect_byte;

   FUNCTION_START(AddTableListEntry);

   ASSERT(filter_ptr != NULL);

   table_entry = STB_GetMemory(sizeof(TABLE_LIST_ENTRY));
   if (table_entry != NULL)
   {
      #ifdef DEBUG_SI_FILTER
      STB_SI_PRINT(("AddTableRec: tab 0x%02x/0x%04x v%d %d sections ", tid, xtid, ver, (last_sect + 1)));
      #endif
      memset(table_entry, 0, sizeof(TABLE_LIST_ENTRY));

      table_entry->new_table = TRUE;
      table_entry->tid = tid;
      table_entry->xtid = xtid;
      table_entry->version = ver;
      table_entry->num_sect_reqd = last_sect + 1;
      table_entry->state = VALID;

      // for event schedule tables setup reqd section map
      if ((0x50 <= tid) && (tid <= 0x6f))
      {
         last_sect_byte = last_sect >> 3;
         for (i = 0; i <= last_sect_byte; i++)
         {
            table_entry->ev_sched_sect_reqd[i] = 0xff;
            table_entry->ev_sched_segment_notified[i] = FALSE;
         }
      }

      // add new table after last_rxed_table in the table list, unless last_rxed_table is NULL in
      // which case add it to the beginning of the list
      if (filter_ptr->last_rxed_table == NULL)
      {
         // add to beginning of list
         table_entry->next = filter_ptr->table_list;
         filter_ptr->table_list = table_entry;
      }
      else
      {
         // add after last_rxed_table
         table_entry->next = filter_ptr->last_rxed_table->next;
         filter_ptr->last_rxed_table->next = table_entry;
      }

      // make last_rxed_table point to latest table
      filter_ptr->last_rxed_table = table_entry;
   }
   #ifdef DEBUG_SI_FILTER_FAILURES
   else
   {
      STB_SI_PRINT(("AddTableRec: FAILED"));
   }
   #endif

   FUNCTION_FINISH(AddTableListEntry);
   return(table_entry);
}

/**
 *

 *
 * @brief   Resets an existing table list entry because the version has changed - clears out
 *                any existing sections and resets number of sections required etc.
 *
 * @param   table_entry - the relevant table entry
 * @param   ver         - version
 * @param   last_sect   - last section in table
 *

 *
 */
static void ResetTableListEntry(TABLE_LIST_ENTRY *table_entry, U8BIT ver, U16BIT last_sect)
{
   U16BIT i;
   U8BIT last_sect_byte;

   FUNCTION_START(ResetTableListEntry);

   ASSERT(table_entry != NULL);

   FreeSectionList(table_entry);

   table_entry->version = ver;
   table_entry->num_sect_reqd = last_sect + 1;
   table_entry->num_sect_received = 0;
   memset(table_entry->sect_received, 0, NUM_SECT_RECEIVED_BYTES);
   memset(table_entry->ev_sched_sect_reqd, 0, NUM_SECT_RECEIVED_BYTES);

   if ((0x50 <= table_entry->tid) && (table_entry->tid <= 0x6f))
   {
      last_sect_byte = last_sect >> 3;
      for (i = 0; i <= last_sect_byte; i++)
      {
         table_entry->ev_sched_sect_reqd[i] = 0xff;
         table_entry->ev_sched_segment_notified[i] = FALSE;
      }
   }

   FUNCTION_FINISH(ResetTableListEntry);
}

/**
 *

 *
 * @brief   Searches the table list for the specified table record
 *                THIS FUNCTION IS CALLED AT INTERRUPT LEVEL BY HANDLE SI INTERRUPT as well as
 *                at task level so it MUST be re-entrant and as fast as possible
 *
 * @param   filter_ptr - the relevant filter status block
 * @param   tid        - tid of the required table
 * @param   xtid       - xtid of the required table
 *

 *
 */
static TABLE_LIST_ENTRY* FindTableListEntry(FILTER_STATUS *filter_ptr, U8BIT tid, U16BIT xtid)
{
   TABLE_LIST_ENTRY *table_entry;
   TABLE_LIST_ENTRY *first_entry;

   // use the last_rxed_table as the starting point for searching the list. This should make it
   // quicker because tables are generally sent in order with all sections of one table followed by
   // all sections of the next etc
   table_entry = filter_ptr->last_rxed_table;
   if (table_entry == NULL)
   {
      // if last_rxed_table is null get first in list
      table_entry = filter_ptr->table_list;
   }
   first_entry = table_entry;

   // search list
   while (table_entry != NULL)
   {
      if ((table_entry->tid == tid) && (table_entry->xtid == xtid))
      {
         if (table_entry->state == VALID)
         {
            // found
            break;
         }
      }
      // move on to next in list - if reached the end wrap round to the beginning. Keep going until
      // we reach the start again
      table_entry = table_entry->next;
      if (table_entry == NULL)
      {
         table_entry = filter_ptr->table_list;
      }
      if (table_entry == first_entry)
      {
         // done all of them without finding a match - set table_entry to NULL to indicate not found
         table_entry = NULL;
         break;
      }
   }

   return(table_entry);
}

/**
 *

 *
 * @brief   Frees the memory stored in a table record and associated records
 *
 * @param   table_entry - pointer to the table record to be freed
 *

 *
 */
static void FreeTableList(FILTER_STATUS *filter_ptr)
{
   TABLE_LIST_ENTRY *table_entry;
   TABLE_LIST_ENTRY *next_entry;

   FUNCTION_START(FreeTableList);

   ASSERT(filter_ptr != NULL);

   STB_OSMutexLock(table_mutex);

   table_entry = filter_ptr->table_list;
   if (filter_ptr->traversing)
   {
      /* Mark entries as DELETED */
      while (table_entry != NULL)
      {
         next_entry = table_entry->next;
         FreeSectionList(table_entry);
         table_entry->state = DELETED;
         table_entry = next_entry;
      }
   }
   else
   {
      /* Delete entries */
      while (table_entry != NULL)
      {
         next_entry = table_entry->next;
         FreeSectionList(table_entry);
         STB_FreeMemory(table_entry);
         table_entry = next_entry;
      }
      filter_ptr->table_list = NULL;
      filter_ptr->last_rxed_table = NULL;
   }

   STB_OSMutexUnlock(table_mutex);

   FUNCTION_FINISH(FreeTableList);
}

/**
 *

 *
 * @brief   Clear all DELETED table entries from the filter.
 *                Table entries are marked as DELETED when the list is freed while it's
 *                being traversed.
 *
 * @param   filter_ptr - the relevant filter status block
 *

 *
 */
static void ProcessTableEntries(FILTER_STATUS *filter_ptr)
{
   TABLE_LIST_ENTRY **table_entry_mod;
   TABLE_LIST_ENTRY *table_entry;

   FUNCTION_START(ProcessTableEntries);

   ASSERT(filter_ptr != NULL);

   STB_OSMutexLock(table_mutex);

   table_entry_mod = &filter_ptr->table_list;
   while (*table_entry_mod != NULL)
   {
      if ((*table_entry_mod)->state == DELETED)
      {
         table_entry = *table_entry_mod;
         *table_entry_mod = table_entry->next;
         if (filter_ptr->last_rxed_table == table_entry)
         {
            filter_ptr->last_rxed_table = table_entry->next;
         }
         STB_FreeMemory(table_entry);
      }
      else
      {
         table_entry_mod = &((*table_entry_mod)->next);
      }
   }

   STB_OSMutexUnlock(table_mutex);

   FUNCTION_FINISH(ProcessTableEntries);
}

/**
 *

 *
 * @brief   Adds a section record to the section list in the specified table record, sets
 *                the section received flag and increments num_sect_received
 *
 * @param   table_entry - pointer to the table record
 * @param   sect_num  - number of the section to be added
 * @param   data_len  - length of data
 * @param   data_ptr  - pointer to the data to be stored
 *
 * @return   section record pointer or NULL if not successful
 *
 */
static SI_SECTION_RECORD* AddSectionRecord(TABLE_LIST_ENTRY *table_entry, U8BIT sect_num, U16BIT data_len,
   U8BIT *data_ptr, BOOLEAN add_to_table, BOOLEAN add_to_segment_list)
{
   SI_SECTION_RECORD *section_rec;
   SI_SECTION_RECORD *next_sect_rec;
   SI_SECTION_RECORD **prev_rec_next_ptr;
   U8BIT segment_last_sect;

   FUNCTION_START(AddSectionRecord);

   section_rec = STB_GetMemory(sizeof(SI_SECTION_RECORD) + data_len);
   if (section_rec != NULL)
   {
      section_rec->sect_num = sect_num;
      section_rec->data_len = data_len;
      section_rec->next = NULL;
      memcpy(&section_rec->data_start, data_ptr, data_len);

      STB_OSSemaphoreWait(section_sema);

      if (add_to_table)
      {
         // add to the section list in section order...
         // ...find the right place in the list
         if (add_to_segment_list)
         {
            next_sect_rec = table_entry->segment_list[sect_num >> 3];
            prev_rec_next_ptr = &(table_entry->segment_list[sect_num >> 3]);
         }
         else
         {
            next_sect_rec = table_entry->section_list;
            prev_rec_next_ptr = &(table_entry->section_list);
         }

         while (next_sect_rec != NULL)
         {
            if (sect_num < next_sect_rec->sect_num)
            {
               break;
            }
            prev_rec_next_ptr = &(next_sect_rec->next);
            next_sect_rec = next_sect_rec->next;
         }

         // ...insert the record - this could be at the front, in the middle or on the end.
         section_rec->next = next_sect_rec;
         *prev_rec_next_ptr = section_rec;
      }

      // add this section to the record of sections received
      table_entry->sect_received[sect_num >> 3] |= (0x01 << (sect_num & 0x07));
      table_entry->num_sect_received++;

      STB_OSSemaphoreSignal(section_sema);

      // if this is an event schedule table update the sections required map from the
      // segment_last_section field
      if ((0x50 <= table_entry->tid) && (table_entry->tid <= 0x6f))
      {
         // event schedule tables - update ev_sched_sect_reqd array
         // (bitmap, 1 = sect reqd, 0 = sect not used)
         segment_last_sect = data_ptr[12];
         table_entry->ev_sched_sect_reqd[sect_num >> 3] = ~(0xff << ((segment_last_sect & 0x07) + 1));
      }

      #ifdef DEBUG_SI_FILTER
      STB_SI_PRINT(("AddSectionRec: sect %d [of %d]", sect_num, table_entry->num_sect_reqd));
      #endif
   }
   #ifdef DEBUG_SI_FILTER_FAILURES
   else
   {
      STB_SI_PRINT(("AddSectionRec: FAILED"));
   }
   #endif
   FUNCTION_FINISH(AddSectionRecord);
   return(section_rec);
}

/**
 *

 *
 * @brief   Frees the memory stored in a section list
 *
 * @param   table_entry - pointer to the table record
 *

 *
 */
static void FreeSectionList(TABLE_LIST_ENTRY *table_entry)
{
   SI_SECTION_RECORD *section_rec;
   SI_SECTION_RECORD *next_rec;
   U16BIT i;

   FUNCTION_START(FreeSectionList);

   ASSERT(table_entry != NULL);

   STB_OSSemaphoreWait(section_sema);

   section_rec = table_entry->section_list;
   while (section_rec != NULL)
   {
      next_rec = section_rec->next;
      STB_FreeMemory(section_rec);
      section_rec = next_rec;
   }
   table_entry->section_list = NULL;

   for (i = 0; i < NUM_SECT_RECEIVED_BYTES; i++)
   {
      section_rec = table_entry->segment_list[i];
      while (section_rec != NULL)
      {
         next_rec = section_rec->next;
         STB_FreeMemory(section_rec);
         section_rec = next_rec;
      }
      table_entry->segment_list[i] = NULL;
   }

   STB_OSSemaphoreSignal(section_sema);

   FUNCTION_FINISH(FreeSectionList);
}

//---global function definitions-----------------------------------------------

/**
 *

 *
 * @brief   Initialises STB layer SI engine - terrestrial version
 *

 *

 *
 */
void STB_SITerrInitialise(void)
{
   U8BIT i;

   FUNCTION_START(STB_SITerrInitialise);

   num_paths = STB_DPGetNumPaths(); //STB_OSGetTunerPaths();
   STB_SI_PRINT(("STB_SITerrInitialise (paths = %d)", num_paths));

   // grab enough memory for status for each path
   si_status = STB_GetMemory(sizeof(E_STB_SI_STATUS) * num_paths);
   ASSERT(si_status != NULL);

   // Create the message queues
   filt_task_msg_queue = STB_OSCreateQueue(sizeof(FILTER_STATUS *), MAX_ENTRIES_FILT_TASK_QUEUE);
   cntrl_task_msg_queue = STB_OSCreateQueue(sizeof(CNTRL_TASK_MSG), MAX_ENTRIES_CNTRL_TASK_QUEUE);
   low_pri_task_msg_queue = STB_OSCreateQueue(sizeof(CNTRL_TASK_MSG), MAX_ENTRIES_LOW_PRI_TASK_QUEUE);

   // Create the Tasks
   filt_task_ptr = STB_OSCreateTask(SiFilterTask, NULL, FILT_TASK_STACK_SIZE, FILT_TASK_PRIORITY, (U8BIT *)"SiFilt");
   cntrl_task_ptr = STB_OSCreateTask(SiControlTask, NULL, CNTRL_TASK_STACK_SIZE, CNTRL_TASK_PRIORITY, (U8BIT *)"SiCntrl");
   low_pri_task_ptr = STB_OSCreateTask(SiLowPriTableTask, NULL, LOW_PRI_TASK_STACK_SIZE, LOW_PRI_TASK_PRIORITY, (U8BIT *)"SiLowPri");

   // Create semaphores to control access to table list and section lists
   table_mutex = STB_OSCreateMutex();
   section_sema = STB_OSCreateSemaphore();
   filter_mutex = STB_OSCreateMutex();

   // init status vars
   for (i = 0; i < num_paths; i++)
   {
      si_status[i] = SI_STOPPED;
   }

   FUNCTION_FINISH(STB_SITerrInitialise);
}

/**
 *

 *
 * @brief   Sends an event to STB layer SI Terrestrial engine.
 *
 * @param   U32BIT events - event mask
 *

 *
 */
void STB_SITerrSendEvent(U8BIT path, U32BIT events)
{
   CNTRL_TASK_MSG msg;

   FUNCTION_START(STB_SITerrSendEvent);

   msg.path = path;
   msg.type = SI_EVENT;
   msg.numval = events;

   if (events == SI_EVENT_STOP)
   {
      // set flags to tell cntrl task and low priority task there is a stop event in the queue,
      // so that the queue will be flushed up to stop event
      low_pri_task_stop_pending = TRUE;
      cntrl_task_stop_pending = TRUE;

      STB_OSWriteQueue(low_pri_task_msg_queue, (void *)&msg, sizeof(CNTRL_TASK_MSG), TIMEOUT_NEVER);
   }
   STB_OSWriteQueue(cntrl_task_msg_queue, (void *)&msg, sizeof(CNTRL_TASK_MSG), TIMEOUT_NEVER);

   FUNCTION_FINISH(STB_SITerrSendEvent);
}

/**
 *

 *
 * @brief   Reads the SI status.
 *
 * @param   U8BIT path - the ID of the decode path to use
 *
 * @return   E_STB_SI_STATUS - status value.
 *
 */
E_STB_SI_STATUS STB_SITerrGetStatus(U8BIT path)
{
   E_STB_SI_STATUS ret_val;

   FUNCTION_START(STB_SITerrGetStatus);

   ASSERT(path < num_paths);
   ret_val = si_status[path];

   FUNCTION_FINISH(STB_SITerrGetStatus);

   return(ret_val);
}

/**
 *

 *
 * @brief   Registers a function to be called when the PMT changes
 *
 * @param   func_ptr - pointer to the function to be called
 *
 * @return   ID used to reference the observer in all future functions.
 *
 */
void STB_SIRegisterPmtObserver(F_PmtObserver func_ptr)
{
   U8BIT index;
   FUNCTION_START(STB_SIRegisterPmtObserver);
   for (index = 0; index != NUM_PMT_OBSERVERS; index++)
   {
      if (pmt_observers[index] == func_ptr)
      {
         break;
      }
   }
   if (index == NUM_PMT_OBSERVERS)
   {
      for (index = 0; index != NUM_PMT_OBSERVERS; index++)
      {
         if (pmt_observers[index] == NULL)
         {
            pmt_observers[index] = func_ptr;
            break;
         }
      }
   }
   FUNCTION_FINISH(STB_SIRegisterPmtObserver);
}

/**
 *

 *
 * @brief   Registers a function to be called when the PMT changes
 *
 * @param   func_ptr - pointer to the function to be called
 *
 * @return   ID used to reference the observer in all future functions.
 *
 */
void STB_SIUnregisterPmtObserver(F_PmtObserver func_ptr)
{
   U8BIT index;

   FUNCTION_START(STB_SIUnregisterPmtObserver);

   /* Find a blank slot and save the callback function */
   for (index = 0; index != NUM_PMT_OBSERVERS; index++)
   {
      if (pmt_observers[index] == func_ptr)
      {
         pmt_observers[index] = NULL;
         break;
      }
   }
   FUNCTION_FINISH(STB_SIUnregisterPmtObserver);
}

/**
 * @brief   Registers a function to be called to handle SI events
 * @param   func_ptr function being registered to handle SI events
 * @return  previously registered function
 */
F_AppSiEventHandler STB_SIRegisterAppSiEventHandler(void (*func_ptr)(U8BIT, E_APP_SI_EVENT_TYPE))
{
   F_AppSiEventHandler old_func;

   FUNCTION_START(STB_SIRegisterAppSiEventHandler);

   old_func = app_si_interface_ptr;
   app_si_interface_ptr = func_ptr;

   FUNCTION_FINISH(STB_SIRegisterAppSiEventHandler);

   return(old_func);
}

/**
 *

 *
 * @brief   Sets up filter for SI table
 *
 * @param   path            - the demux path to use
 * @param   pid             - required pid
 * @param   tid_match       - required match byte for the tid in the match/mask filter
 * @param   tid_mask        - required mask byte for the tid in the match/mask filter
 * @param   format          - format of the table i.e. single or multiple section
 * @param   expected_tables - number of tables expected. Used to decide when multi-table filter is complete
 * @param   xtid_match      - required match byte for the extended tid in the match/mask filter
 * @param   xtid_mask       - required mask byte for the extended tid in the match/mask filter
 * @param   req_type        - ONE_SHOT or CONTINUOUS request
 * @param   buff_size       - size of buffer to be used
 * @param   crc             - TRUE if crc checking required, FALSE otherwise
 * @param   low_priority    - TRUE if request is lower priority
 * @param   callback        - pointer to callback function for return of table
 * @param   ret_param       - parameter to be returned to callback function
 *
 * @return   filter handle or NULL if not setup
 *
 */
void* STB_SIRequestTable(U8BIT path, U16BIT pid, U8BIT tid_match, U8BIT tid_mask,
   E_SI_TABLE_FORMAT_TYPE format, U16BIT expected_tables,
   U16BIT xtid_match, U16BIT xtid_mask, E_SI_REQUEST_TYPE req_type,
   U16BIT buff_size, BOOLEAN crc, BOOLEAN low_priority,
   void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param)
{
   FILTER_STATUS *filter_ptr;

   FUNCTION_START(STB_SIRequestTable);

   #ifdef DEBUG_TABLE_REQUESTS
   STB_SI_PRINT(("STB_SIRequestTable - path %d, pid 0x%04x, tid 0x%02x[0x%02x], format %d",
                 path, pid, tid_match, tid_mask, format));
   STB_SI_PRINT(("                     xtid 0x%04x[0x%04x], exp %d, rtype %d, buff %d, crc %d, lp %d",
                 xtid_match, xtid_mask, expected_tables, req_type, buff_size, crc, low_priority));
   #endif

   filter_ptr = GetFilter(path, pid, tid_match, tid_mask, format, expected_tables, xtid_match,
         xtid_mask, req_type, buff_size, crc, low_priority, callback, ret_param);
   if (filter_ptr != NULL)
   {
      StartFilter(filter_ptr);
   }

   FUNCTION_FINISH(STB_SIRequestTable);
   return((void *)filter_ptr);
}

/**
 *

 *
 * @brief   Stops filtering for SI table
 *
 * @param   filter_handle
 *

 *
 */
void STB_SICancelTableRequest(void *filter_handle)
{
   FILTER_STATUS *filter_ptr;
   BOOLEAN filter_ok;

   FUNCTION_START(STB_SICancelTableRequest);

   #ifdef DEBUG_TABLE_REQUESTS
   STB_SI_PRINT(("STB_SICancelTableRequest - filter %p", filter_handle));
   #endif

   STB_OSMutexLock(filter_mutex);

   filter_ptr = (FILTER_STATUS *)filter_handle;
   filter_ok = ValidateFilterPtr(filter_ptr);
   if (filter_ok == TRUE)
   {
      StopFilter(filter_ptr);
      ReleaseFilter(filter_ptr);
   }

   STB_OSMutexUnlock(filter_mutex);

   FUNCTION_FINISH(STB_SICancelTableRequest);
}

/**
 *

 *
 * @brief   modifies the section filtering on an existing filter
 *
 * @param   filter_handle   - handle of the filter to be modified
 * @param   tid_match       - required match byte for the tid in the match/mask filter
 * @param   tid_mask        - required mask byte for the tid in the match/mask filter
 * @param   xtid_match      - required match byte for the extended tid in the match/mask filter
 * @param   xtid_mask       - required mask byte for the extended tid in the match/mask filter
 * @param   expected_tables - number of tables expected. Used to decide when multi-table filter is complete
 *

 *
 */
void STB_SIModifyTableRequest(void *filter_handle, U8BIT tid_match, U8BIT tid_mask,
   U16BIT xtid_match, U16BIT xtid_mask, U16BIT expected_tables)
{
   FILTER_STATUS *filter_ptr;
   BOOLEAN filter_ok;

   FUNCTION_START(STB_SIModifyTableRequest);

   #ifdef DEBUG_TABLE_REQUESTS
   STB_SI_PRINT(("STB_SIModifyTableRequest - filter %p", filter_handle));
   #endif

   STB_OSMutexLock(filter_mutex);

   filter_ptr = (FILTER_STATUS *)filter_handle;
   filter_ok = ValidateFilterPtr(filter_ptr);
   if (filter_ok == TRUE)
   {
      StopFilter(filter_ptr);
      FreeTableList(filter_ptr);
      ModifyFilter(filter_ptr, tid_match, tid_mask, xtid_match, xtid_mask, expected_tables);
      StartFilter(filter_ptr);
   }

   STB_OSMutexUnlock(filter_mutex);

   FUNCTION_FINISH(STB_SIModifyTableRequest);
}

/**
 *

 *
 * @brief   restarts the section filtering on an existing filter without changing and pid or
 *                match/mask filter settings
 *
 * @param   filter_handle - handle of the filter to be modified
 *

 *
 */
void STB_SIRestartTableRequest(void *filter_handle)
{
   FILTER_STATUS *filter_ptr;
   BOOLEAN filter_ok;

   FUNCTION_START(STB_SIRestartTableRequest);

   #ifdef DEBUG_TABLE_REQUESTS
   STB_SI_PRINT(("STB_SIRestartTableRequest - filter %p", filter_handle));
   #endif

   STB_OSMutexLock(filter_mutex);

   filter_ptr = (FILTER_STATUS *)filter_handle;
   filter_ok = ValidateFilterPtr(filter_ptr);
   if (filter_ok == TRUE)
   {
      StopFilter(filter_ptr);
      FreeTableList(filter_ptr);

      // reset buffer and table received
      filter_ptr->data_start = ((U8BIT *)filter_ptr) + sizeof(FILTER_STATUS);
      filter_ptr->data_end = filter_ptr->data_start;
      filter_ptr->num_tables_received = 0;

      StartFilter(filter_ptr);
   }

   STB_OSMutexUnlock(filter_mutex);

   FUNCTION_FINISH(STB_SIRestartTableRequest);
}

/**
 *

 *
 * @brief   Frees the memory used in a table record passed to the application
 *
 * @param   table_rec - pointer to the table record to be freed
 *

 *
 */
void STB_SIReleaseTableRecord(SI_TABLE_RECORD *table_rec)
{
   SI_SECTION_RECORD *section_rec;
   SI_SECTION_RECORD *next_rec;

   FUNCTION_START(STB_SIReleaseTableRecord);

   ASSERT(table_rec != NULL);

   #ifdef DEBUG_TABLE_REQUESTS
   STB_SI_PRINT(("STB_SIReleaseTableRecord - table_rec %p", table_rec));
   #endif

   if (table_rec != NULL)
   {
      STB_OSSemaphoreWait(section_sema);

      // first free section list
      section_rec = table_rec->section_list;
      while (section_rec != NULL)
      {
         next_rec = section_rec->next;
         STB_FreeMemory(section_rec);
         section_rec = next_rec;
      }

      STB_OSSemaphoreSignal(section_sema);

      // now free table record itself
      STB_FreeMemory(table_rec);
   }

   FUNCTION_FINISH(STB_SIReleaseTableRecord);
}

/**
 *

 *
 * @brief   Indicates SI search is complete
 *
 * @param   path - the path for which searching is complete
 *

 *
 */
void STB_SISearchComplete(U8BIT path, BOOLEAN success, void *event_data, U32BIT data_size)
{
   FUNCTION_START(STB_SISearchComplete);

   si_status[path] = SI_STOPPED;

   if (success)
   {
      STB_ERSendEvent(FALSE, FALSE, EV_CLASS_SEARCH, EV_TYPE_SUCCESS, event_data, data_size);
   }
   else
   {
      STB_ERSendEvent(FALSE, FALSE, EV_CLASS_SEARCH, EV_TYPE_FAIL, event_data, data_size);
   }

   STB_SI_PRINT(("SITask(%d) finished search", path));

   FUNCTION_FINISH(STB_SISearchComplete);
}

/**
 *

 *
 * @brief   Reports current pmt has arrived so that it can be passed on to interested parties
 *
 * @param   table_rec - table record containing the pmt details
 * @param   new_serv  - if TRUE this pmt is for a new service, if FALSE it is a new version
 *                            for the same service as last time.
 *

 *
 */
void STB_SIReportCurrentPmt(U16BIT service_id, SI_TABLE_RECORD *table_rec, BOOLEAN new_serv,
   BOOLEAN new_pmt_version)
{
   SI_SECTION_RECORD *sect_rec;
   U8BIT index;
   U32BIT ca_handle;

   FUNCTION_START(STB_SIReportCurrentPmt);

   if (table_rec != NULL)
   {
      #ifdef DEBUG_PMT_REPORTING
      STB_SI_PRINT(("STB_SIReportCurrentPmt: serv_id 0x%04x new_serv %d", table_rec->xtid, new_serv));
      #endif

      // pmts should only have one section so get the first section record from the table_rec
      sect_rec = table_rec->section_list;

      if (sect_rec != NULL)
      {
         /* Report the PMT to all registered observers */
         for (index = 0; index != NUM_PMT_OBSERVERS; index++)
         {
            if (pmt_observers[index] != 0)
            {
               (pmt_observers[index])(service_id, &(sect_rec->data_start));
            }
         }

         if (new_serv || new_pmt_version)
         {
            if (STB_DPGetPathCADescrambler(table_rec->path, &ca_handle))
            {
               /* Report the PMT to the CA */
               STB_CAReportPMT(ca_handle, &(sect_rec->data_start), sect_rec->data_len);
            }
         }
      }
   }
   else
   {
      #ifdef DEBUG_PMT_REPORTING
      STB_SI_PRINT(("STB_SIReportCurrentPmt: NULL table_rec"));
      #endif
      /* Report removal of service to all registered observers */
      for (index = 0; index != NUM_PMT_OBSERVERS; index++)
      {
         if (pmt_observers[index] != 0)
         {
            (pmt_observers[index])(service_id, NULL);
         }
      }
   }
   FUNCTION_FINISH(STB_SIReportCurrentPmt);
}

/**
 * @brief   Reports the CAT has been received so it can be passed on to other modules
 * @param   table_rec - table record containing the CAT section data
 * @return  TRUE if the CAT is passed to the CA module, FALSE otherwise
 */
BOOLEAN STB_SIReportCat(SI_TABLE_RECORD *table_rec)
{
   SI_SECTION_RECORD *sect_rec;
   U8BIT i;
   U32BIT ca_handle;
   BOOLEAN reported;

   FUNCTION_START(STB_SIReportCat);

   reported = FALSE;

   if (table_rec != NULL)
   {
      #ifdef DEBUG_CAT_REPORTING
      STB_SI_PRINT(("STB_SIReportCat: version %d num_sects %d", table_rec->version, table_rec->num_sect));
      #endif
      //if (STB_DPGetPathCADescrambler(table_rec->path, &ca_handle))
      if (TRUE)
      {
#ifdef CUSTOMIZED_FOR_CNS
         SI_CAT_TABLE *cat_table = STB_SIParseCatTable(table_rec);
         if (NULL != cat_table)
         {
            U8BIT path = table_rec->path;
            if (NULL != cat_table->ca_desc_array)
            {
               for (int i=0; i < cat_table->num_ca_entries; i++)
               {
                  U16BIT ca_system_id = cat_table->ca_desc_array[i].ca_id;
                  U16BIT emm_pid = cat_table->ca_desc_array[i].ca_pid;
                  STB_CAReportEmmPid(path, ca_handle, ca_system_id, emm_pid);     
               }
            }         
            STB_SIReleaseCatTable(cat_table);
            cat_table = NULL;
         }
#endif

         // work through the table reporting a section at a time
         sect_rec = table_rec->section_list;
         for (i = 0; ((i < table_rec->num_sect) && (sect_rec != NULL)); i++)
         {
            STB_CAReportCAT(ca_handle, &(sect_rec->data_start), sect_rec->data_len);
            sect_rec = sect_rec->next;
         }

         reported = TRUE;
      }
   }
   else
   {
      #ifdef DEBUG_CAT_REPORTING
      STB_SI_PRINT(("STB_SIReportCat: NULL table_rec"));
      #endif
   }

   FUNCTION_FINISH(STB_SIReportCat);

   return(reported);
}

/**
 * @brief   Reports the BAT has been received so it can be passed on to other modules
 * @param   table_rec - table record containing the BAT section data
 * @return  TRUE if the BAT is passed to the CA module, FALSE otherwise
 */
BOOLEAN STB_SIReportBat(SI_TABLE_RECORD *table_rec)
{
   SI_SECTION_RECORD *sect_rec;
   U8BIT i;
   U32BIT ca_handle;
   BOOLEAN reported;

   FUNCTION_START(STB_SIReportBat);

   reported = FALSE;

   if (table_rec != NULL)
   {
      if (STB_DPGetPathCADescrambler(table_rec->path, &ca_handle))
      {
         /* Work through the table reporting a section at a time */
         sect_rec = table_rec->section_list;
         for (i = 0; ((i < table_rec->num_sect) && (sect_rec != NULL)); i++)
         {
            STB_CAReportBAT(ca_handle, &(sect_rec->data_start), sect_rec->data_len);
            sect_rec = sect_rec->next;
         }

         reported = TRUE;
      }
   }

   FUNCTION_FINISH(STB_SIReportBat);

   return(reported);
}

/**
 * @brief   Reports the NIT has been received so it can be passed on to other modules
 * @param   table_rec - table record containing the NIT section data
 * @return  TRUE if the NIT is passed to the CA module, FALSE otherwise
 */
BOOLEAN STB_SIReportNit(SI_TABLE_RECORD *table_rec)
{
   SI_SECTION_RECORD *sect_rec;
   U8BIT i;
   U32BIT ca_handle;
   BOOLEAN reported;

   FUNCTION_START(STB_SIReportNit);

   reported = FALSE;

   if (table_rec != NULL)
   {
      if (STB_DPGetPathCADescrambler(table_rec->path, &ca_handle))
      {
         /* Work through the table reporting a section at a time */
         sect_rec = table_rec->section_list;
         for (i = 0; ((i < table_rec->num_sect) && (sect_rec != NULL)); i++)
         {
            STB_CAReportNIT(ca_handle, &(sect_rec->data_start), sect_rec->data_len);
            sect_rec = sect_rec->next;
         }

         reported = TRUE;
      }
   }

   FUNCTION_FINISH(STB_SIReportNit);

   return(reported);
}

//*****************************************************************************
// End of file
//*****************************************************************************

