/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   STB layer SI table parsing
 *
 * @file    stbsitab.c
 * @date    06/03/2003
 */

// gives direct COM port access
#define STB_DEBUG

// prints table requests
#define DEBUG_SI_PAT_REQUEST
#define DEBUG_SI_PMT_REQUEST
/* #define DEBUG_SI_NIT_REQUEST */
/* #define DEBUG_SI_SDT_REQUEST */
/* #define DEBUG_SI_BAT_REQUEST */
/* #define DEBUG_SI_EIT_REQUEST */
/* #define DEBUG_SI_TDT_REQUEST */
/* #define DEBUG_SI_TOT_REQUEST */
#define DEBUG_SI_CAT_REQUEST

// prints table parsing
#define DEBUG_SI_PAT_SUMMARY
#define DEBUG_SI_PMT_SUMMARY
/* #define DEBUG_SI_NIT_SUMMARY */
/* #define DEBUG_SI_SDT_SUMMARY */
/* #define DEBUG_SI_BAT_SUMMARY */
/* #define DEBUG_SI_EIT_SUMMARY */
/* #define DEBUG_SI_TDT_TOT_SUMMARY */
/* #define DEBUG_SI_RCT_SUMMARY */
#define DEBUG_SI_CAT_SUMMARY

#define DEBUG_SI_PAT_CONTENT
#define DEBUG_SI_PMT_CONTENT
/* #define DEBUG_SI_NIT_CONTENT */
/* #define DEBUG_SI_SDT_CONTENT */
/* #define DEBUG_SI_BAT_CONTENT */
/* #define DEBUG_SI_EIT_CONTENT */
/* #define DEBUG_SI_TDT_TOT_CONTENT */
/* #define DEBUG_SI_RCT_CONTENT */
#define DEBUG_SI_CAT_CONTENT

// prints descriptor parsing
/* #define DEBUG_AC3_DESC */
/* #define DEBUG_AAC_DESC */
/* #define DEBUG_APP_SIG_DESC*/
/* #define DEBUG_CA_DESC */
/* #define DEBUG_CA_ID_DESC */
/* #define DEBUG_CAROUSEL_ID_DESC */
/* #define DEBUG_CID_DESC */
/* #define DEBUG_COMPONENT_DESC */
/* #define DEBUG_CONTENT_DESC */
/* #define DEBUG_DEF_AUTH_DESC */
/* #define DEBUG_EXTENDED_EVENT_DESC */
/* #define DEBUG_FREQ_LIST_DESC */
/* #define DEBUG_FTA_CONTENT_DESC */
/* #define DEBUG_GUIDANCE_DESC */
/* #define DEBUG_IMAGE_ICON_DESC */
/* #define DEBUG_ISO_LANG_DESC */
/* #define DEBUG_LCN_DESC */
/* #define DEBUG_LINKAGE_DESC */
/* #define DEBUG_LOCAL_TIME_OFFSET_DESC */
/* #define DEBUG_MULTILING_COMPONENT_DESC */
/* #define DEBUG_MULTILING_NET_NAME_DESC */
/* #define DEBUG_MULTILING_SERV_NAME_DESC */
/* #define DEBUG_NET_NAME_DESC */
/* #define DEBUG_PARENTAL_RATING_DESC */
/* #define DEBUG_PREF_NAME_ID_DESC */
/* #define DEBUG_PREF_NAME_LIST_DESC */
/* #define DEBUG_PRIV_DATA_SPEC_DESC */
/* #define DEBUG_RCT_LINK_INFO */
/* #define DEBUG_SAT_DEL_SYS_DESC */
/* #define DEBUG_SERVICE_DESC */
/* #define DEBUG_SERVICE_MOVE_DESC*/
/* #define DEBUG_SERV_LIST_DESC */
/* #define DEBUG_SHORT_SERVICE_NAME */
/* #define DEBUG_SHORT_EVENT_DESC */
/* #define DEBUG_STREAM_ID_DESC */
/* #define DEBUG_SUBTITLE_DESC */
/* #define DEBUG_TELETEXT_DESC*/
/* #define DEBUG_TERR_DEL_SYS_DESC */
/* #define DEBUG_CABLE_DEL_SYS_DESC */
/* #define DEBUG_TARGET_REGION_DESC */
/* #define DEBUG_FREESAT_LCN_DESC */
/* #define DEBUG_FREESAT_LINKAGE_DESC */
/* #define DEBUG_FREESAT_TUNNELLED_DATA_DESC */
/* #define DEBUG_FREESAT_INFO_LOCATION_DESC */
/* #define DEBUG_FREESAT_GROUP_NAME_DESC */
/* #define DEBUG_FREESAT_PREFIX_DESC */
/* #define DEBUG_SUPP_AUDIO_DESC */
/* #define DEBUG_URI_LINKAGE_DESC */
/* #define DEBUG_NORDIG_CONTENT_PROTECTION_DESC */
/* #define DEBUG_SDT_CODE_DESC */

// includes dump of skipped descriptors if debug messages are turned on for the table
/* #define DEBUG_SKIPPED_DESC */


//---includes for this file----------------------------------------------------
// compiler library header files

#include <stdio.h>
#include <string.h>

// third party header files

// Ocean Blue Software header files

#include <techtype.h>
#include <dbgfuncs.h>

#include "stbhwc.h"
#include "stbheap.h"
#include "stbuni.h"

#include "stbsiflt.h"
#include "stbdpc.h"
#include "stbsitab.h"

//---constant definitions for this file----------------------------------------
#ifdef STB_SI_PRINT_REQUIRED
   #define STB_SI_PRINT(x) DEBUG_PRINTX_CONDITIONAL(DEBUG_STB_SI) x
#else
   #ifdef STB_DEBUG
      #define STB_SI_PRINT(x) STB_SPDebugWrite x
   #else
      #define STB_SI_PRINT(x)
   #endif
#endif

#ifdef DEBUG_SI_EIT_CONTENT
  #define DEBUG_SI_EIT_CONTENT_VALUE TRUE
#else
  #define DEBUG_SI_EIT_CONTENT_VALUE FALSE
#endif

#ifdef DEBUG_SI_SDT_CONTENT
  #define DEBUG_SI_SDT_CONTENT_VALUE TRUE
#else
  #define DEBUG_SI_SDT_CONTENT_VALUE FALSE
#endif

#ifdef DEBUG_SI_PMT_CONTENT
  #define DEBUG_SI_PMT_CONTENT_VALUE TRUE
#else
  #define DEBUG_SI_PMT_CONTENT_VALUE FALSE
#endif

#ifdef DEBUG_SI_NIT_CONTENT
  #define DEBUG_SI_NIT_CONTENT_VALUE TRUE
#else
  #define DEBUG_SI_NIT_CONTENT_VALUE FALSE
#endif

#ifdef DEBUG_SI_BAT_CONTENT
  #define DEBUG_SI_BAT_CONTENT_VALUE TRUE
#else
  #define DEBUG_SI_BAT_CONTENT_VALUE FALSE
#endif

#ifdef DEBUG_RCT_LINK_INFO
  #define DEBUG_RCT_LINK_INFO_VALUE TRUE
#else
  #define DEBUG_RCT_LINK_INFO_VALUE FALSE
#endif

#ifdef DEBUG_SI_RCT_CONTENT
  #define DEBUG_SI_RCT_CONTENT_VALUE TRUE
#else
  #define DEBUG_SI_RCT_CONTENT_VALUE FALSE
#endif

#ifdef DEBUG_SI_TDT_TOT_CONTENT
  #define DEBUG_SI_TDT_TOT_CONTENT_VALUE TRUE
#else
  #define DEBUG_SI_TDT_TOT_CONTENT_VALUE FALSE
#endif

#ifdef DEBUG_SI_UNT_CONTENT
  #define DEBUG_SI_UNT_CONTENT_VALUE TRUE
#else
  #define DEBUG_SI_UNT_CONTENT_VALUE FALSE
#endif

#ifdef DEBUG_SI_CAT_CONTENT
  #define DEBUG_SI_CAT_CONTENT_VALUE TRUE
#else
  #define DEBUG_SI_CAT_CONTENT_VALUE FALSE
#endif

#ifdef DEBUG_SERVICE_DESC
  #define DEBUG_SERVICE_DESC_VALUE TRUE
#else
  #define DEBUG_SERVICE_DESC_VALUE FALSE
#endif

#ifdef DEBUG_SHORT_EVENT_DESC
  #define DEBUG_SHORT_EVENT_DESC_VALUE TRUE
#else
  #define DEBUG_SHORT_EVENT_DESC_VALUE FALSE
#endif

#ifdef DEBUG_TERR_DEL_SYS_DESC
  #define DEBUG_TERR_DEL_SYS_DESC_VALUE TRUE
#else
  #define DEBUG_TERR_DEL_SYS_DESC_VALUE FALSE
#endif

#ifdef DEBUG_SAT_DEL_SYS_DESC
  #define DEBUG_SAT_DEL_SYS_DESC_VALUE TRUE
#else
  #define DEBUG_SAT_DEL_SYS_DESC_VALUE FALSE
#endif

#ifdef DEBUG_CABLE_DEL_SYS_DESC
  #define DEBUG_CABLE_DEL_SYS_DESC_VALUE TRUE
#else
  #define DEBUG_CABLE_DEL_SYS_DESC_VALUE FALSE
#endif

// si table PIDs
#define SI_PAT_PID         0x0000
#define SI_CAT_PID         0x0001
#define SI_NIT_PID         0x0010
#define SI_SDT_PID         0x0011
#define SI_BAT_PID         0x0011
#define SI_EIT_PID         0x0012
#define SI_TDT_PID         0x0014
#define SI_TOT_PID         0x0014
#define SI_UNT_PID         0x1EDC    // 7900

// si table IDs
#define SI_PAT_TID            0x00
#define SI_CAT_TID            0x01
#define SI_PMT_TID            0x02
#define SI_NIT_ACTUAL_TID     0x40
#define SI_NIT_OTHER_TID      0x41
#define SI_SDT_ACTUAL_TID     0x42
#define SI_SDT_OTHER_TID      0x46
#define SI_BAT_TID            0x4A
#define SI_UNT_TID            0x4B
#define SI_EITPF_ACTUAL_TID   0x4E
#define SI_EITPF_OTHER_TID    0x4F
#define SI_EITSC_ACTUAL_TID   0x50   // range 0x50 to 0x5f
#define SI_EITSC_OTHER_TID    0x60   // range 0x60 to 0x6f
#define SI_TDT_TID            0x70
#define SI_TOT_TID            0x73
#define SI_AIT_TID            0x74
#define SI_RCT_TID            0x76
#define SI_EIT_PLUS_TID       0xD1


// si table descriptor tags...
// MPEG descriptors
#define CA_DTAG                     0x09
#define ISO_LANG_DTAG               0x0a
#define CAROUSEL_ID_DTAG            0x13

// DVB descriptors
#define PRIV_DATA_INDICATOR_DTAG    0x0f
#define NET_NAME_DTAG               0x40
#define SERV_LIST_DTAG              0x41
#define SAT_DEL_SYS_DTAG            0x43
#define CABLE_DEL_SYS_DTAG          0x44
#define SERVICE_DTAG                0x48
#define LINKAGE_DTAG                0x4a
#define SHORT_EVENT_DTAG            0x4d
#define EXTENDED_EVENT_DTAG         0x4e
#define COMPONENT_DTAG              0x50
#define STREAM_ID_DTAG              0x52
#define CA_ID_DTAG                  0x53
#define CONTENT_DTAG                0x54
#define PARENTAL_RATING_DTAG        0x55
#define TELETEXT_DTAG               0x56
#define LOCAL_TIME_OFFSET_DTAG      0x58
#define SUBTITLE_DTAG               0x59
#define TERR_DEL_SYS_DTAG           0x5a
#define MULTILING_NET_NAME_DTAG     0x5b
#define MULTILING_SERV_NAME_DTAG    0x5d
#define MULTILING_COMPONENT_DTAG    0x5e
#define PRIV_DATA_SPEC_DTAG         0x5f
#define SERVICE_MOVE_DTAG           0x60
#define FREQ_LIST_DTAG              0x62
#define AC3_DTAG                    0x6a
#define APP_SIG_DTAG                0x6f     /* Application signalling descriptor tag value */
#define SERV_AVAIL_DESC_DTAG        0x72     /* Service Availability Descriptor */
#define DEF_AUTH_DTAG               0x73     /* Default authority descriptor tag value */
#define RELATED_CONTENT_DTAG        0x74     /* Related content descriptor tag value */
#define CONT_ID_DTAG                0x76     /* Content identifier descriptor tag value */
#define EAC3_DTAG                   0x7a
#define AAC_DTAG                    0x7c
#define FTA_CONTENT_DTAG            0x7e
#define EXT_DTAG                    0x7f     /* Extension descriptor */

// SSU descriptors
#define SSU_LOCATION_DTAG           0x03     /* SSU location descriptor */

// User defined descriptors
#define USER_DEFINED_DTAG_0x83      0x83
#define USER_DEFINED_DTAG_0x84      0x84    /* CNS: SDT code descriptor */
#define USER_DEFINED_DTAG_0x85      0x85
#define USER_DEFINED_DTAG_0x86      0x86
#define USER_DEFINED_DTAG_0x87      0x87
#define USER_DEFINED_DTAG_0x88      0x88
#define USER_DEFINED_DTAG_0x89      0x89
#define USER_DEFINED_DTAG_0x93      0x93    /* CNS: logical channel numbers */
#define USER_DEFINED_DTAG_0xA0      0xa0
#define USER_DEFINED_DTAG_0xD0      0xd0    /* CNS: series (linking) descriptor */

/* Satellite descriptor tag values */
#define BOUQUET_NAME_DTAG           0x47
#define LOGICAL_CHANNEL_DTAG        0x83

/* CI+ descriptor tag values */
#define CIPLUS_SERVICE_DTAG         0xcc /* NIT 2nd*/
#define CIPLUS_PROTECTION_DTAG      0xce /* SDT */
#define CIPLUS_VIRTUAL_CHAN_DTAG    0xd2 /* NIT 1st */


/* Freesat specific descriptor tag values */
#define FREESAT_TUNNELLED_DTAG         0xd0  /* PMT 1st */
#define FREESAT_ALT_TUNNELLED_DTAG     0xd1  /* PMT 2nd */
#define FREESAT_LINK_DTAG              0xd2  /* NIT 1st */
#define FREESAT_REGION_LCN_DTAG        0xd3  /* BAT 2nd */
#define FREESAT_REGION_NAME_DTAG       0xd4  /* BAT 1st */
#define FREESAT_SERV_GROUP_DTAG        0xd5  /* BAT 1st */
#define FREESAT_IACTIVE_STORAGE_DTAG   0xd6  /* BAT 1st */
#define FREESAT_INFO_LOCATION_DTAG     0xd7  /* BAT 1st */
#define FREESAT_SERV_GROUP_NAME_DTAG   0xd8  /* BAT 1st */
#define FREESAT_SERV_NAME_DTAG         0xd9  /* SDT */
#define FREESAT_GUIDANCE_DTAG          0xda  /* SDT, EIT */
#define FREESAT_IACTIVE_RESTRICT_DTAG  0xdb  /* BAT 2nd */
#define FREESAT_CONTENT_MANAGE_DTAG    0xdc  /* BAT 1st+2nd, SDT, EIT */
#define FREESAT_PREFIX_DTAG            0xdf  /* NIT 1st, BAT 1st, SDT */
#define FREESAT_MEDIA_LOCATOR_DTAG     0xe0  /* BAT 1st, SDT, EIT */


/* Extension descriptor tag values */
#define IMAGE_ICON_DTAG             0x00
#define T2_DELIVERY_SYS_DTAG        0x04
#define SUPPLEMENTARY_AUDIO_DTAG    0x06
#define NETWORK_CHANGE_NOTIFY_DTAG  0x07
#define MESSAGE_DTAG                0x08
#define TARGET_REGION_DTAG          0x09
#define TARGET_REGION_NAME_DTAG     0x0a
#define URI_LINKAGE_DTAG            0x13


/* Private data specifier values - somewhat duplicated in country_data in ap_cfdat.h */
#define UK_DTT_PRIVATE_DATA_SPECIFIER  0x0000233a
#define CIPLUS_PRIVATE_DATA_SPECIFIER  0x00000040
#define FREESAT_PRIVATE_DATA_SPECIFIER 0x46534154
#define NORDIG_PRIVATE_DATA_SPECIFIER  0x00000029
#define EACEM_PRIVATE_DATA_SPECIFIER   0x00000028  /* E-Book */
#define NZ_DTT_PRIVATE_DATA_SPECIFIER  0x00000037  /* New Zealand Freeview DTT */
#define NZ_SAT_PRIVATE_DATA_SPECIFIER  0x00000029  /* New Zealand Freeview Satellite */
#define ZAF_DTT_PRIVATE_DATA_SPECIFIER 0x000022c6  /* South Africa (SABC) */

/* Audio desc lang code */
#define LANG_CODE_NAR                 0x6e6172


//---local typedef structs for this file---------------------------------------


//---local (static) variable declarations for this file------------------------
//   (internal variables declared static to make them local)


static const U8BIT eit_tid_match[] =
{
   SI_EITPF_ACTUAL_TID,                                  // EIT_NOW_NEXT_ACT                    = 0,
   SI_EITPF_OTHER_TID,                                   // EIT_NOW_NEXT_OTHER                  = 1,
   SI_EITPF_OTHER_TID,                                   // EIT_NOW_NEXT_ALL                    = 2,
   SI_EIT_PLUS_TID                                       // EIT_PF_PLUS                         = 3
};

static const U8BIT eit_tid_mask[] =
{
   0xff,                                                 // EIT_NOW_NEXT_ACT                    = 0,
   0xff,                                                 // EIT_NOW_NEXT_OTHER                  = 1,
   (~(SI_EITPF_ACTUAL_TID ^ SI_EITPF_OTHER_TID) & 0xff), // EIT_NOW_NEXT_ALL                    = 2,
   0Xff                                                  // SI_EITPF_PLUS_TID                   = 3
};

static const U8BIT sched_tid_match[] =
{
   SI_EITSC_ACTUAL_TID,                                  // EIT_SCHED_ACT           = 0,
   SI_EITSC_OTHER_TID,                                   // EIT_SCHED_OTHER         = 1,
   SI_EITSC_ACTUAL_TID,                                  // EIT_SCHED_ACT_4DAY      = 2,
   SI_EITSC_OTHER_TID,                                   // EIT_SCHED_OTHER_4DAY    = 3,
   (SI_EITSC_ACTUAL_TID | SI_EITSC_OTHER_TID),           // EIT_SCHED_ALL_4DAY      = 4,
   (SI_EITSC_ACTUAL_TID | SI_EITSC_OTHER_TID),           // EIT_SCHED_ALL_8DAY      = 5,
   (SI_EITSC_ACTUAL_TID | SI_EITSC_OTHER_TID)            // EIT_SCHED_ALL           = 6
};

static const U8BIT sched_tid_mask[] =
{
   0xf0,                                                 // EIT_SCHED_ACT           = 0,
   0xf0,                                                 // EIT_SCHED_OTHER         = 1,
   0xff,                                                 // EIT_SCHED_ACT_4DAY      = 2,
   0xff,                                                 // EIT_SCHED_OTHER_4DAY    = 3,
   (~(SI_EITSC_ACTUAL_TID ^ SI_EITSC_OTHER_TID) & 0xff), // EIT_SCHED_ALL_4DAY      = 4,
   (~(SI_EITSC_ACTUAL_TID ^ SI_EITSC_OTHER_TID) & 0xfe), // EIT_SCHED_ALL_8DAY      = 5,
   (~(SI_EITSC_ACTUAL_TID ^ SI_EITSC_OTHER_TID) & 0xf0)  // EIT_SCHED_ALL           = 6
};


//---local function prototypes for this file-----------------------------------
//   (internal functions declared static to make them local)

static STB_SI_USER_DEF_DESCRIP_FUNCTION GetUserDefinedDescriptorFunction(U8BIT dtag);

static U8BIT* ReadLanguageCode(U8BIT *dptr, U32BIT *code_ptr);
static U8BIT* SkipDescriptor(U8BIT *dptr, U8BIT dtag, BOOLEAN db_print);

static U8BIT* ParseAACDescriptor(U8BIT *dptr, SI_AAC_DESC **desc_ptr, BOOLEAN db_print);
static U8BIT* ParseAC3Descriptor(U8BIT *dptr, U8BIT dtag, SI_AC3_DESC **desc_ptr, BOOLEAN db_print);
static U8BIT* ParseAppSignallingDescriptor(U8BIT *dptr, U16BIT *num_ptr, SI_APP_SIG_DESC **array_ptr,
   BOOLEAN db_print);
static U8BIT* ParseCaDescriptor(U8BIT *dptr, U16BIT *num_ptr, SI_CA_DESC **array_ptr, BOOLEAN db_print);
static U8BIT* ParseCaIdentifierDescriptor(U8BIT *dptr, U8BIT *num_ptr, U16BIT **array_ptr, BOOLEAN db_print);
static U8BIT* ParseCarouselIdDescriptor(U8BIT *dptr, U32BIT *carousel_id, BOOLEAN db_print);
static U8BIT* ParseComponentDescriptor(U8BIT *dptr, U8BIT *num_ptr, SI_COMPONENT_DESC **array_ptr,
   BOOLEAN db_print);
static U8BIT* ParseContentDescriptor(U8BIT *dptr, U8BIT *num_ptr, SI_CONTENT_DESC **array_ptr, BOOLEAN db_print);
static U8BIT* ParseFrequencyListDescriptor(U8BIT *dptr, SI_NIT_FREQUENCY_LIST_DESC **freq_list, BOOLEAN db_print);
static U8BIT* ParseFTAContentDescriptor(U8BIT *dptr, SI_FTA_CONTENT_DESC **desc_ptr, BOOLEAN db_print);
static U8BIT* ParseGuidanceDescriptor(U8BIT *dptr, SI_GUIDANCE_DESC **desc_ptr, BOOLEAN db_print);
static U8BIT* ParseIsoLangDescriptor(U8BIT *dptr, U16BIT *num_ptr, SI_ISO_LANG_DESC **array_ptr, BOOLEAN db_print);
static U8BIT* ParseLocalTimeOffsetDescriptor(U8BIT *dptr, U16BIT *num_ptr, SI_LTO_DESC **array_ptr,
   BOOLEAN db_print);
static U8BIT* ParseMultilingComponentDescriptor(U8BIT *dptr, U8BIT *num_ptr,
   SI_MULTILING_COMPONENT_DESC **array_ptr, BOOLEAN db_print);
static U8BIT* ParseMultilingNetNameDescriptor(U8BIT *dptr, U16BIT *num_ptr,
   SI_MULTILING_NET_NAME_DESC **array_ptr, BOOLEAN db_print);
static U8BIT* ParseMultilingServNameDescriptor(U8BIT *dptr, U16BIT *num_ptr,
   SI_MULTILING_SERV_NAME_DESC **array_ptr, BOOLEAN db_print);
static U8BIT* ParseParentalRatingDescriptor(U8BIT *dptr, U8BIT *num_ptr,
   SI_PARENTAL_RATING_DESC **array_ptr, BOOLEAN db_print);
static U8BIT* ParseServiceListDescriptor(U8BIT *dptr, U16BIT *num_ptr, SI_SERV_LIST_DESC **array_ptr,
   BOOLEAN db_print);
static U8BIT* ParseShortServiceNameDescriptor(U8BIT *dptr, SI_STRING_DESC **name_ptr, BOOLEAN db_print);
static U8BIT* ParseShortEventDescriptor(U8BIT *dptr, U8BIT *num_ptr, SI_SHORT_EVENT_DESC **array_ptr,
   BOOLEAN db_print);
static U8BIT* ParseExtendedEventDescriptor(U8BIT *dptr, U8BIT *num_ptr,
   SI_EXTENDED_EVENT_DESC **array_ptr, BOOLEAN db_print);
static U8BIT* ParseSubtitleDescriptor(U8BIT *dptr, U16BIT *num_ptr, SI_SUBTITLE_DESC **array_ptr,
   BOOLEAN db_print);
static U8BIT* ParseTeletextDescriptor(U8BIT *dptr, U16BIT *num_ptr, SI_TELETEXT_DESC **array_ptr,
   BOOLEAN db_print);
static U8BIT* ParseLogicalChannelDescriptor(U8BIT *dptr, U8BIT dtag, U16BIT *num_ptr,
   SI_LCN_DESC **array_ptr, BOOLEAN db_print);
static U8BIT* ParseCnsLogicalChannelDescriptor(U8BIT *dptr, U8BIT dtag, U16BIT *num_ptr,
   SI_LCN_DESC **array_ptr, BOOLEAN db_print);
static U8BIT* ParseServiceAttributeDescriptor(U8BIT *dptr, U16BIT *num_ptr,
   SI_SERV_ATTRIBUTE_DESC **array_ptr, BOOLEAN db_print);
static U8BIT* ParseNordigLCN2Descriptor(U8BIT *dptr, U16BIT *num_ptr, SI_NORDIG_LCN_DESC **array_ptr,
   BOOLEAN db_print);
static U8BIT* ParsePreferredNameListDescriptor(U8BIT *dptr, U16BIT *num_ptr,
   SI_PREFERRED_NAME_DESC **array_ptr, BOOLEAN db_print);

static U8BIT* ParseLinkageDescriptor(U8BIT *dptr, U16BIT *num_ptr, SI_LINKAGE_DESC_ENTRY **list_ptr,
   SI_LINKAGE_DESC_ENTRY **last_entry_ptr, U32BIT private_data_code, BOOLEAN db_print);

static U8BIT* ParseNetNameDescriptor(U8BIT *dptr, SI_STRING_DESC **name_ptr, BOOLEAN db_print);
static U8BIT* ParsePrivateDataSpecifierDescriptor(U8BIT *dptr, U32BIT *code_ptr, BOOLEAN db_print);
static U8BIT* ParseServiceDescriptor(U8BIT *dptr, U8BIT *type_ptr, SI_STRING_DESC **provider_ptr,
   SI_STRING_DESC **name_ptr, BOOLEAN db_print);
static U8BIT* ParseStreamIdDescriptor(U8BIT *dptr, U8BIT **tag_array_ptr, U8BIT *num_entries_ptr, BOOLEAN db_print);
static U8BIT* ParseTerrestrialDeliverySysDescriptor(U8BIT *dptr, SI_DELIVERY_SYS_DESC **desc_ptr,
   BOOLEAN db_print);
static U8BIT* ParseT2DeliverySysDescriptor(U8BIT *dptr, SI_DELIVERY_SYS_DESC **desc_ptr, BOOLEAN db_print);
static U8BIT* ParseSatelliteDeliverySysDescriptor(U8BIT *dptr, SI_DELIVERY_SYS_DESC **desc_ptr, BOOLEAN db_print);
static U8BIT* ParseFreesatLinkageDescriptor(U8BIT *dptr, SI_FREESAT_LINKAGE_DESC **desc, BOOLEAN db_print);
static U8BIT* ParseFreesatPrefixDescriptor(U8BIT *dptr, SI_FREESAT_PREFIX_DESC **list, BOOLEAN db_print);
static U8BIT* ParsePreferredNameIdDescriptor(U8BIT *dptr, U8BIT *id_ptr, BOOLEAN db_print);

static U8BIT* ParseDefaultAuthorityDescriptor(U8BIT *dptr, SI_STRING_DESC **authority_ptr, BOOLEAN db_print);
static U8BIT* ParseContentIdentifierDescriptor(U8BIT *dptr, U8BIT *num_ptr, SI_CRID_DESC **list_ptr,
   SI_CRID_DESC **last_entry_ptr, BOOLEAN db_print);
static U8BIT* ParseNetworkChangeNotifyDescriptor(U8BIT *dptr, U16BIT *num_change_notifies,
   SI_NIT_CHANGE_NOTIFY_DESC **desc_ptr,
   BOOLEAN db_print);
static U8BIT* ParseMessageDescriptor(U8BIT *dptr, U16BIT *num_messages,
   SI_NIT_MESSAGE_ENTRY **message_array, BOOLEAN db_print);
static U8BIT* ParseTargetRegionNameDescriptor(U8BIT *dptr, SI_NIT_TARGET_REGION_NAME_DESC **desc_list,
   BOOLEAN db_print);
static U8BIT* ParseTargetRegionDescriptor(U8BIT *dptr, SI_TARGET_REGION_DESC **desc_list,
   BOOLEAN db_print);
static U8BIT* ParseSupplementaryAudioDescriptor(U8BIT *dptr, SI_AD_DESC **audio_desc, BOOLEAN db_print);
static U8BIT* ParseRctLinkInfo(U8BIT *dptr, SI_RCT_LINK_INFO *link_info, BOOLEAN db_print);
static U8BIT* ParseImageIconDesc(U8BIT *dptr, U8BIT *num_icons, SI_IMAGE_ICON_DESC **icon_array, BOOLEAN db_print);
static U8BIT* ParseURILinkageDesc(U8BIT *dptr, SI_URI_LINKAGE_DESC **desc_list, U32BIT private_data_code,
   BOOLEAN db_print);
static U8BIT* ParseNordigContentProtectionDesc(U8BIT *dptr, U8BIT *protection_level, BOOLEAN db_print);
#if 0
static U8BIT* ParseServiceMoveDescriptor(U8BIT *dptr, SI_SERVICE_MOVE_DESC **desc_ptr, BOOLEAN db_print);
#endif

static U8BIT* ParseFreesatTunnelledDataDescriptor(U8BIT *dptr, U8BIT *num_entries,
   SI_FREESAT_TUNNELLED_DATA_DESC **desc_array, BOOLEAN alt_descriptor, BOOLEAN db_print);
static U8BIT* ParseFreesatServiceGroupNameDescriptor(U8BIT *dptr,
   SI_BAT_FREESAT_GROUP_NAME_ENTRY **name_list, BOOLEAN db_print);
static U8BIT* ParseFreesatServiceGroupDescriptor(U8BIT *dptr, SI_BAT_FREESAT_SERV_GROUP_ENTRY **serv_array,
   U16BIT *num_entries, BOOLEAN db_print);

static U8BIT* ParseFreesatInteractiveRestrictionDescriptor(U8BIT *dptr, U16BIT **int_res_array, U8BIT *num_entries, BOOLEAN db_print);

static U8BIT* ParseFreesatShortServiceNameDescriptor(U8BIT *dptr, U16BIT *num_ptr,
   SI_MULTILING_SHORT_NAME_DESC **array_ptr, BOOLEAN db_print);

static U8BIT* ParseRegionNameDescriptor(U8BIT *dptr, SI_BAT_FREESAT_REGION **region_list, BOOLEAN db_print);
static U8BIT* ParseIActiveStorageDescriptor(U8BIT *dptr,
   SI_BAT_FREESAT_IACTIVE_STORAGE_DESC **desc_list, BOOLEAN db_print);
static U8BIT* ParseFreesatInfoLocationDescriptor(U8BIT *dptr,
   SI_BAT_FREESAT_INFO_LOCATION **location_list, BOOLEAN db_print);

static U8BIT* SaveCIProtectionDescriptor(U8BIT *dptr, U8BIT **desc, BOOLEAN db_print);
static U8BIT* ParseCIPlusServiceDescriptor(U8BIT *dptr, U16BIT *num_services,
   SI_CIPLUS_SERVICE **service_list, SI_CIPLUS_SERVICE **last_service, BOOLEAN db_print);

static U8BIT* ParseSdtCodeDescriptor(U8BIT *dptr, SI_SDT_CODE_DESC **sdt_code, BOOLEAN db_print);
static U8BIT* ParseSeriesDescriptor(U8BIT *dptr, U8BIT dtag, SI_SERIES_DESC **series, BOOLEAN db_print);

// SSU descriptors
static U8BIT* ParseSsuLocationDescriptor(U8BIT *dptr, SI_SSU_LOCATION_DESC **ssu_location, BOOLEAN db_print);

static float BCDToFloat(U8BIT *data_ptr, U8BIT len, U8BIT point);


//--- structures, variables etc associated with the control of private data-------------------------
static U32BIT country_private_data_specifier_code = 0;
static BOOLEAN freesat_private_data_specifier = FALSE;
static BOOLEAN ciplus_private_data_specifier = FALSE;
static BOOLEAN eacem_private_data_specifier = FALSE;
static BOOLEAN nzsat_private_data_specifier = FALSE;
static BOOLEAN nordig_private_data_specifier = FALSE;


// not everybody uses private data specifiers - some want the user defined descriptors but don't
// send private data specifiers. So individual user-defined descriptors can be enabled by the
// application using STB_SISetUserDefinedDescriptor().
#define FIRST_USER_DEFINED_DTAG  0x80
#define LAST_USER_DEFINED_DTAG   0xfe
#define NUM_USER_DEFINED_DTAGS   (LAST_USER_DEFINED_DTAG - FIRST_USER_DEFINED_DTAG + 1)
static STB_SI_USER_DEF_DESCRIP_FUNCTION user_defined_dtag_function[NUM_USER_DEFINED_DTAGS];


//---local function definitions------------------------------------------------



//--------------------------------------------------------------------------------------------------
// Descriptor handling functions
//--------------------------------------------------------------------------------------------------

/**
 *

 *
 * @brief   Checks if the specified descriptor tag is in the user defined range (0x80 - 0xfe)
 *                and if so has it been enabled by the application
 *
 * @param   dtag - the descriptor tag to be checked
 *
 * @return   TRUE if tag has been enabled by the application
 *
 */
static STB_SI_USER_DEF_DESCRIP_FUNCTION GetUserDefinedDescriptorFunction(U8BIT dtag)
{
   STB_SI_USER_DEF_DESCRIP_FUNCTION retval;

   FUNCTION_START(GetUserDefinedDescriptorFunction);
   retval = FALSE;
   if ((FIRST_USER_DEFINED_DTAG <= dtag) && (dtag <= LAST_USER_DEFINED_DTAG))
   {
      retval = user_defined_dtag_function[dtag - FIRST_USER_DEFINED_DTAG];
   }
   FUNCTION_FINISH(GetUserDefinedDescriptorFunction);
   return(retval);
}

/**
 *

 *
 * @brief   Reads a 3-byte language code and returns it as a U32BIT
 *
 * @param   dptr     - pointer to first byte of language code
 * @param   code_ptr - pointer for the return of the language code
 *
 * @return   Updated dptr i.e. pointing to byte following the 3-byte language code
 *
 */
static U8BIT* ReadLanguageCode(U8BIT *dptr, U32BIT *code_ptr)
{
   U32BIT lcode;
   U8BIT i;
   U8BIT code_char;

   FUNCTION_START(ReadLanguageCode);

   lcode = 0;
   for (i = 0; i < 3; i++)
   {
      code_char = *dptr;
      dptr++;
      if ((code_char >= 'A') && (code_char <= 'Z'))
      {
         code_char += 0x20; // convert to lower case
      }
      lcode = (lcode << 8) | code_char;
   }
   *code_ptr = lcode;
   FUNCTION_FINISH(ReadLanguageCode);
   return(dptr);
}

/**
 *

 *
 * @brief   Skips the next descriptor
 *
 * @param   dptr     - pointer to length byte of descriptor
 * @param   dtag     - descriptor tag value (for debug only)
 * @param   db_print - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated dptr i.e. pointing to byte following descriptor
 *
 */
static U8BIT* SkipDescriptor(U8BIT *dptr, U8BIT dtag, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;

   FUNCTION_START(SkipDescriptor);

#ifndef DEBUG_SKIPPED_DESC
   USE_UNWANTED_PARAM(dtag);
   USE_UNWANTED_PARAM(db_print);
#endif

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   #ifdef DEBUG_SKIPPED_DESC
   {
      U16BIT i;
      U8BIT j;
      U8BIT msg_buff[54];
      U8BIT *msg_ptr;

      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   dtag=0x%02x, len=%d", dtag, dlen));
         msg_buff[0] = ' ';
         msg_buff[1] = ' ';
         msg_buff[2] = ' ';
         msg_ptr = &msg_buff[3];
         i = 0;
         while (i < dlen)
         {
            j = 0;
            while ((j < 16) && (i < dlen))
            {
               sprintf((char *)msg_ptr, " %02x", *dptr);
               dptr++;
               i++;
               j++;
               msg_ptr += 3;
            }
            *msg_ptr = '\0';
            STB_SI_PRINT(("%s", msg_buff));
            msg_ptr = &msg_buff[3];
         }
      }
   }
   #endif

   FUNCTION_FINISH(SkipDescriptor);
   return(end_ptr);
}

/*!**************************************************************************
 * @brief   Reads the AAC descriptor, of which only one can be present in an AAC/HE-AAC stream
 * @param   dptr - pointer to length byte of descriptor
 * @param   desc_ptr - pointer for returned descriptor
 * @param   db_print - if TRUE debug prints the contents of the descriptor (for debug only)
 * @return  Updated value of current data byte address i.e. at end of descriptor
 ****************************************************************************/
static U8BIT* ParseAACDescriptor(U8BIT *dptr, SI_AAC_DESC **desc_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;

   FUNCTION_START(ParseAACDescriptor);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

#ifdef DEBUG_AAC_DESC
   if (db_print == TRUE)
   {
      STB_SI_PRINT(("   AAC desc: (%d bytes)", dlen));
   }
#else
   USE_UNWANTED_PARAM(db_print);
#endif

   /* There should be only one AAC descriptor in a PMT
    * so skip this one if it already exists */
   if (*desc_ptr == NULL)
   {
      *desc_ptr = (SI_AAC_DESC *)STB_GetMemory(sizeof(SI_AAC_DESC));
      if (*desc_ptr != NULL)
      {
         /* Profile and level */
         (*desc_ptr)->profile_level = *dptr;
         dptr++;

         if ((dlen > 1) && ((*dptr & 0x80) != 0))
         {
            /* Component type */
            (*desc_ptr)->type_present = TRUE;
            dptr++;
            (*desc_ptr)->aac_type = *dptr;
#ifdef DEBUG_AAC_DESC
            if (db_print == TRUE)
            {
               STB_SI_PRINT(("    profile=0x%02x, type=0x%02x", (*desc_ptr)->profile_level,
                  (*desc_ptr)->aac_type));
            }
#endif
         }
         else
         {
            (*desc_ptr)->type_present = FALSE;
#ifdef DEBUG_AAC_DESC
            if (db_print == TRUE)
            {
               STB_SI_PRINT(("    profile=0x%02x", (*desc_ptr)->profile_level));
            }
#endif
         }
      }
   }

   FUNCTION_FINISH(ParseAACDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads the AC3 and EAC3 descriptors but just stores the descriptor tag so that
 *                the audio type can be determined, but skips the rest of the data
 *
 * @param   dptr     - pointer to length byte of descriptor
 * @param   dtag     - descriptor tag value
 * @param   desc_ptr - pointer for returned descriptor
 * @param   db_print - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated dptr i.e. pointing to byte following descriptor
 *
 */
static U8BIT* ParseAC3Descriptor(U8BIT *dptr, U8BIT dtag, SI_AC3_DESC **desc_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;

   FUNCTION_START(ParseAC3Descriptor);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

#ifdef DEBUG_AC3_DESC
   if (db_print)
   {
      if (dtag == AC3_DTAG)
      {
         STB_SI_PRINT(("   AC-3 desc: (%u bytes)", dlen));
      }
      else if (dtag == EAC3_DTAG)
      {
         STB_SI_PRINT(("   E-AC-3 desc: (%u bytes)", dlen));
      }
      else
      {
         STB_SI_PRINT(("   AC-3 desc: (%u bytes) descriptor_tag=0x%02x", dlen, dtag));
      }
   }
#else
   USE_UNWANTED_PARAM(db_print);
#endif

   /* There should be only one AC-3 or E-AC3 descriptor in a PMT
    * so skip this one if it already exists */
   if ((*desc_ptr == NULL) && (dlen >= 1))
   {
      dptr++;

      *desc_ptr = (SI_AC3_DESC *)STB_GetMemory(sizeof(SI_AC3_DESC));
      if (*desc_ptr != NULL)
      {
         memset(*desc_ptr, 0, sizeof(SI_AC3_DESC));

         /* Store the descriptor_tag as a means of differentiating between AC-3 and E-AC3 */
         (*desc_ptr)->dtag = dtag;

         if ((*dptr & 0x80) != 0)
         {
            (*desc_ptr)->component_type_flag = TRUE;
         }

         if ((*dptr & 0x40) != 0)
         {
            (*desc_ptr)->bsid_flag = TRUE;
         }

         if ((*dptr & 0x20) != 0)
         {
            (*desc_ptr)->mainid_flag = TRUE;
         }

         if ((*dptr & 0x10) != 0)
         {
            (*desc_ptr)->asvc_flag = TRUE;
         }

         if (dtag == EAC3_DTAG)
         {
            if ((*dptr & 0x08) != 0)
            {
               (*desc_ptr)->mixinfoexists = TRUE;
            }

            if ((*dptr & 0x04) != 0)
            {
               (*desc_ptr)->substream1_flag = TRUE;
            }

            if ((*dptr & 0x02) != 0)
            {
               (*desc_ptr)->substream2_flag = TRUE;
            }

            if ((*dptr & 0x01) != 0)
            {
               (*desc_ptr)->substream3_flag = TRUE;
            }
         }

         dptr++;

         if ((*desc_ptr)->component_type_flag)
         {
            (*desc_ptr)->component_type = *dptr;
#ifdef DEBUG_AC3_DESC
            if (db_print)
            {
               STB_SI_PRINT(("    component_type=0x%02x", (*desc_ptr)->component_type));
            }
#endif
            dptr++;
         }

         if ((*desc_ptr)->bsid_flag)
         {
            (*desc_ptr)->bsid = *dptr;
#ifdef DEBUG_AC3_DESC
            if (db_print)
            {
               STB_SI_PRINT(("    bsid=0x%02x", (*desc_ptr)->bsid));
            }
#endif
            dptr++;
         }

         if ((*desc_ptr)->mainid_flag)
         {
            (*desc_ptr)->mainid = *dptr;
#ifdef DEBUG_AC3_DESC
            if (db_print)
            {
               STB_SI_PRINT(("    mainid=0x%02x", (*desc_ptr)->mainid));
            }
#endif
            dptr++;
         }

         if ((*desc_ptr)->asvc_flag)
         {
            (*desc_ptr)->asvc = *dptr;
#ifdef DEBUG_AC3_DESC
            if (db_print)
            {
               STB_SI_PRINT(("    asvc=0x%02x", (*desc_ptr)->asvc));
            }
#endif
            dptr++;
         }

         if ((*desc_ptr)->substream1_flag)
         {
            (*desc_ptr)->substream1 = *dptr;
#ifdef DEBUG_AC3_DESC
            if (db_print)
            {
               STB_SI_PRINT(("    substream1=0x%02x", (*desc_ptr)->substream1));
            }
#endif
            dptr++;
         }

         if ((*desc_ptr)->substream2_flag)
         {
            (*desc_ptr)->substream2 = *dptr;
#ifdef DEBUG_AC3_DESC
            if (db_print)
            {
               STB_SI_PRINT(("    substream2=0x%02x", (*desc_ptr)->substream2));
            }
#endif
            dptr++;
         }

         if ((*desc_ptr)->substream3_flag)
         {
            (*desc_ptr)->substream3 = *dptr;
#ifdef DEBUG_AC3_DESC
            if (db_print)
            {
               STB_SI_PRINT(("    substream3=0x%02x", (*desc_ptr)->substream3));
            }
#endif
         }
      }
   }

   FUNCTION_FINISH(ParseAC3Descriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads the application signalling descriptor and stores any entries that exist
 *
 * @param   dptr     - pointer to length byte of descriptor
 * @param   num_ptr   - pointer for return of number of entries in the array
 * @param   array_ptr - pointer for return of pointer to array
 * @param   db_print - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated dptr i.e. pointing to byte following descriptor
 *
 */
static U8BIT* ParseAppSignallingDescriptor(U8BIT *dptr, U16BIT *num_ptr, SI_APP_SIG_DESC **array_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U16BIT num_entries;
   SI_APP_SIG_DESC *array;
   U16BIT app_type;
   U8BIT ait_version;

   FUNCTION_START(ParseAppSignallingDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(num_ptr != NULL);
   ASSERT(array_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

#ifdef DEBUG_APP_SIG_DESC
   if (db_print)
   {
      STB_SI_PRINT(("   App sig desc: (%d bytes)", dlen));
   }
#else
   USE_UNWANTED_PARAM(db_print);
#endif

   while (dlen >= 3)
   {
      app_type = ((dptr[0] & 0x7f) << 8) + dptr[1];
      ait_version = dptr[2] & 0x1f;

#ifdef DEBUG_APP_SIG_DESC
      if (db_print)
      {
         STB_SI_PRINT(("     app_type=0x%02x, ait_version=%u", app_type, ait_version));
      }
#endif

      if (*array_ptr == NULL)
      {
         /* No entries already - create new array */
         num_entries = 1;
         array = (SI_APP_SIG_DESC *)STB_GetMemory(sizeof(SI_APP_SIG_DESC));
      }
      else
      {
         /* Already got entries - make array bigger */
         num_entries = *num_ptr + 1;
         array = (SI_APP_SIG_DESC *)STB_GetMemory(num_entries * sizeof(SI_APP_SIG_DESC));
         if (array != NULL)
         {
            /* Copy over previous entries and free old array */
            memcpy(array, *array_ptr, (*num_ptr * sizeof(SI_APP_SIG_DESC)));
            STB_FreeMemory(*array_ptr);
         }
      }

      /* Add new entry to array */
      if (array != NULL)
      {
         array[num_entries - 1].app_type = app_type;
         array[num_entries - 1].ait_version = ait_version;
         *array_ptr = array;
         *num_ptr = num_entries;
      }
      else
      {
#ifdef DEBUG_APP_SIG_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("CAN'T ALLOCATE MEMORY FOR DESCRIPTOR ARRAY ENTRY"));
         }
#endif
      }

      dlen -= 3;
      dptr += 3;
   }

   FUNCTION_FINISH(ParseAppSignallingDescriptor);

   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads ca system descriptor - adds entry to ca descriptor list, checking for
 *                duplicate entries
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   num_ptr   - pointer for return of number of entries in the array
 * @param   array_ptr - pointer for return of pointer to array
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseCaDescriptor(U8BIT *dptr, U16BIT *num_ptr, SI_CA_DESC **array_ptr,
   BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U16BIT num_entries;
   SI_CA_DESC *array;
   U16BIT ca_id;
   U16BIT ca_pid;

   FUNCTION_START(ParseCaDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(num_ptr != NULL);
   ASSERT(array_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 4)
   {
      ca_id = (dptr[0] << 8) | dptr[1];
      ca_pid = ((dptr[2] & 0x1f) << 8) | dptr[3];
      #ifdef DEBUG_CA_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   CA desc: (%d bytes) id=0x%04x, pid=0x%04x", dlen, ca_id, ca_pid));
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif

      // check if there are already entries in the array (i.e. already received a descriptor)
      // if so add to the existing array, otherwise create new array
      if (*array_ptr == NULL)
      {
         // no entries already - create new array
         num_entries = 1;
         array = (SI_CA_DESC *)STB_GetMemory(sizeof(SI_CA_DESC));
      }
      else
      {
         // already got entries - make array bigger
         num_entries = *num_ptr + 1;
         array = (SI_CA_DESC *)STB_GetMemory(num_entries * sizeof(SI_CA_DESC));
         if (array != NULL)
         {
            // copy over previous entries and free old array
            memcpy(array, *array_ptr, (*num_ptr * sizeof(SI_CA_DESC)));
            STB_FreeMemory(*array_ptr);
         }
      }

      // add new entry to array
      if (array != NULL)
      {
         array[num_entries - 1].ca_id = ca_id;
         array[num_entries - 1].ca_pid = ca_pid;
         *array_ptr = array;
         *num_ptr = num_entries;
      }
      else
      {
         #ifdef DEBUG_CA_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   CAN'T ALLOCATE MEMORY FOR DESCRIPTOR ARRAY ENTRY"));
         }
         #endif
      }
   }
   else
   {
      #ifdef DEBUG_CA_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid CA desc: (%d bytes)", dlen));
      }
      #endif
   }

   FUNCTION_FINISH(ParseCaDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads ca identifier descriptor - adds entries to ca id array
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   num_ptr   - pointer for return of number of entries in the array
 * @param   array_ptr - pointer for return of pointer to array
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseCaIdentifierDescriptor(U8BIT *dptr, U8BIT *num_ptr, U16BIT **array_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U8BIT num_entries;
   U16BIT *array;
   U16BIT i;

   FUNCTION_START(ParseCaIdentifierDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(num_ptr != NULL);
   ASSERT(array_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 2)
   {
      num_entries = dlen / 2; // each entry in the descriptor is 2 bytes long
      #ifdef DEBUG_CA_ID_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   CA identifier desc: %d entries", num_entries));
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif

      // check if there are already entries in the array (i.e. already received a descriptor)
      // if so add to the existing array, otherwise create new array
      if (*array_ptr == NULL)
      {
         // no entries already - create new array
         array = (U16BIT *)STB_GetMemory(num_entries * sizeof(U16BIT));
      }
      else
      {
         // already got entries - make array bigger
         num_entries += *num_ptr;
         array = (U16BIT *)STB_GetMemory(num_entries * sizeof(U16BIT));
         if (array != NULL)
         {
            // copy over previous entries and free old array
            memcpy(array, *array_ptr, (*num_ptr * sizeof(U16BIT)));
            STB_FreeMemory(*array_ptr);
         }
      }

      // add new entries to array
      if (array != NULL)
      {
         for (i = *num_ptr; i < num_entries; i++)
         {
            array[i] = (dptr[0] << 8) | dptr[1];
            dptr += 2;
            #ifdef DEBUG_CA_ID_DESC
            if (db_print == TRUE)
            {
               STB_SI_PRINT(("    id=0x%04x", array[i]));
            }
            #endif
         }
         *array_ptr = array;
         *num_ptr = num_entries;
      }
      else
      {
         #ifdef DEBUG_CA_ID_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   CAN'T ALLOCATE MEMORY FOR DESCRIPTOR ARRAY ENTRY"));
         }
         #endif
      }
   }
   else
   {
      #ifdef DEBUG_CA_ID_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid CA identifier desc: (%d bytes)", dlen));
      }
      #endif
   }

   FUNCTION_FINISH(ParseCaIdentifierDescriptor);
   return(end_ptr);
}

static U8BIT* ParseCarouselIdDescriptor(U8BIT *dptr, U32BIT *carousel_id, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;

   FUNCTION_START(ParseCarouselIdDescriptor);

   ASSERT(dptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen == 5)
   {
      *carousel_id = (dptr[0] << 24) | (dptr[1] << 16) | (dptr[2] << 8) | dptr[3];

#ifdef DEBUG_CAROUSEL_ID_DESC
      if (db_print)
      {
         STB_SI_PRINT(("   Carousel ID: 0x%08x", *carousel_id));
      }
#else
      USE_UNWANTED_PARAM(db_print);
#endif
   }
#ifdef DEBUG_CAROUSEL_ID_DESC
   else if (db_print)
   {
      STB_SI_PRINT(("   Invalid carousel ID desc: %u bytes", dlen));
   }
#endif

   FUNCTION_FINISH(ParseCarouselIdDescriptor);

   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads component descriptor - adds entries to component desc array
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   num_ptr   - pointer for return of number of entries in the array
 * @param   array_ptr - pointer for return of pointer to array
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseComponentDescriptor(U8BIT *dptr, U8BIT *num_ptr, SI_COMPONENT_DESC **array_ptr,
   BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   SI_COMPONENT_DESC *array;
   U8BIT num_entries;
   U8BIT tag;
   U8BIT content;
   U8BIT type;
   U32BIT lang_code;
   SI_STRING_DESC *str_desc;

   FUNCTION_START(ParseComponentDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(num_ptr != NULL);
   ASSERT(array_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 6)
   {
      content = dptr[0] & 0x0f;
      type = dptr[1];
      tag = dptr[2];
      dptr += 3;
      dptr = ReadLanguageCode(dptr, &lang_code);
      dptr = STB_SIReadString(dlen - 6, dptr, &str_desc);

      #ifdef DEBUG_COMPONENT_DESC
      {
         char *tmp_str = "NULL";
         if (db_print == TRUE)
         {
            if (str_desc != NULL)
            {
               if (str_desc->str_ptr != NULL)
               {
                  tmp_str = (char *)str_desc->str_ptr;
               }
            }
            STB_SI_PRINT(("   Component desc: tag=%d, cont=%d, type=0x%02x, lang=%c%c%c, str=%s",
                          tag, content, type, (lang_code >> 16), (lang_code >> 8), lang_code, tmp_str));
         }
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif

      // check if there are already entries in the array (i.e. already received a descriptor)
      // if so add to the existing array, otherwise create new array
      if (*array_ptr == NULL)
      {
         // no entries already - create new array
         num_entries = 1;
         array = (SI_COMPONENT_DESC *)STB_GetMemory(sizeof(SI_COMPONENT_DESC));
      }
      else
      {
         // already got entries - make array bigger
         num_entries = *num_ptr + 1;
         array = (SI_COMPONENT_DESC *)STB_GetMemory(num_entries * sizeof(SI_COMPONENT_DESC));
         if (array != NULL)
         {
            // copy over previous entries and free old array
            memcpy(array, *array_ptr, (*num_ptr * sizeof(SI_COMPONENT_DESC)));
            STB_FreeMemory(*array_ptr);
         }
      }

      // add new entry to array
      if (array != NULL)
      {
         array[num_entries - 1].tag = tag;
         array[num_entries - 1].content = content;
         array[num_entries - 1].type = type;
         array[num_entries - 1].lang_code = lang_code;
         array[num_entries - 1].desc_str = str_desc;
         *array_ptr = array;
         *num_ptr = num_entries;
      }
      else
      {
         #ifdef DEBUG_COMPONENT_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   CAN'T ALLOCATE MEMORY FOR DESCRIPTOR ARRAY ENTRY"));
         }
         #endif
      }
   }
   else
   {
      #ifdef DEBUG_COMPONENT_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid Component desc: (%d bytes)", dlen));
      }
      #endif
   }

   USE_UNWANTED_PARAM(dptr);
   FUNCTION_FINISH(ParseComponentDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads content descriptor - adds entries to content desc array
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   num_ptr   - pointer for return of number of entries in the array
 * @param   array_ptr - pointer for return of pointer to array
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseContentDescriptor(U8BIT *dptr, U8BIT *num_ptr, SI_CONTENT_DESC **array_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U8BIT num_entries;
   SI_CONTENT_DESC *array;
   U16BIT i;

   FUNCTION_START(ParseContentDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(num_ptr != NULL);
   ASSERT(array_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 2)
   {
      num_entries = dlen / 2; // each entry in the descriptor is 2 bytes long
      #ifdef DEBUG_CONTENT_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Content desc: %d entries", num_entries));
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif

      // check if there are already entries in the array (i.e. already received a descriptor)
      // if so add to the existing array, otherwise create new array
      if (*array_ptr == NULL)
      {
         // no entries already - create new array
         array = (SI_CONTENT_DESC *)STB_GetMemory(num_entries * sizeof(SI_CONTENT_DESC));
      }
      else
      {
         // already got entries - make array bigger
         num_entries += *num_ptr;
         array = (SI_CONTENT_DESC *)STB_GetMemory(num_entries * sizeof(SI_CONTENT_DESC));
         if (array != NULL)
         {
            // copy over previous entries and free old array
            memcpy(array, *array_ptr, (*num_ptr * sizeof(SI_CONTENT_DESC)));
            STB_FreeMemory(*array_ptr);
         }
      }

      // add new entry to array
      if (array != NULL)
      {
         for (i = *num_ptr; i < num_entries; i++)
         {
            array[i].level_1 = dptr[0] >> 4;
            array[i].level_2 = dptr[0] & 0x0f;
            array[i].user_1 = dptr[1] >> 4;
            array[i].user_2 = dptr[1] & 0x0f;
            dptr += 2;
            #ifdef DEBUG_CONTENT_DESC
            if (db_print == TRUE)
            {
               STB_SI_PRINT(("    L1=0x%02x, L2=0x%02x, U1=0x%02x, U2=0x%02x",
                             array[i].level_1, array[i].level_2, array[i].user_1, array[i].user_2));
            }
            #endif
         }
         *array_ptr = array;
         *num_ptr = num_entries;
      }
      else
      {
         #ifdef DEBUG_CONTENT_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   CAN'T ALLOCATE MEMORY FOR DESCRIPTOR ARRAY ENTRY"));
         }
         #endif
      }
   }
   else
   {
      #ifdef DEBUG_CONTENT_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid Content desc: (%d bytes)", dlen));
      }
      #endif
   }

   FUNCTION_FINISH(ParseContentDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads frequency list descriptor - sets up array of U32BITs
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   freq_list - pointer to a list of frequency lists
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseFrequencyListDescriptor(U8BIT *dptr, SI_NIT_FREQUENCY_LIST_DESC **freq_list, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U8BIT coding_type;
   U8BIT i, num_freqs;
   SI_NIT_FREQUENCY_LIST_DESC *list_entry;

   FUNCTION_START(ParseFrequencyListDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(freq_list != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 5)
   {
      coding_type = *dptr & 0x03;
      dptr++;
      dlen--;

      if (coding_type != 0)
      {
         /* Each frequency is 4 bytes in length */
         num_freqs = dlen / 4;

         list_entry = STB_GetMemory(sizeof(SI_NIT_FREQUENCY_LIST_DESC));
         if (list_entry != NULL)
         {
            memset(list_entry, 0, sizeof(SI_NIT_FREQUENCY_LIST_DESC));

            list_entry->frequency_array = STB_GetMemory(num_freqs * sizeof(U32BIT));
            if (list_entry->frequency_array != NULL)
            {
               list_entry->coding_type = coding_type;
               list_entry->num_frequencies = num_freqs;

               #ifdef DEBUG_FREQ_LIST_DESC
               if (db_print)
               {
                  STB_SI_PRINT(("   Frequency list desc: coding_type %u with %u frequencies", coding_type, num_freqs));
               }
               #endif

               for (i = 0; i < num_freqs; dptr += 4, i++)
               {
                  switch (coding_type)
                  {
                     case 1:  /* satellite */
                        list_entry->frequency_array[i] = (U32BIT)BCDToFloat(dptr, 8, 6);
                        break;
                     case 2:  /* cable */
                        list_entry->frequency_array[i] = (U32BIT)(BCDToFloat(dptr, 8, 4) * 1000000);
                        break;
                     case 3:  /* terrestrial */
                        list_entry->frequency_array[i] = ((dptr[0] << 24) | (dptr[1] << 16) | (dptr[2] << 8) | dptr[3]) * 10;
                        break;
                  }

                  #ifdef DEBUG_FREQ_LIST_DESC
                  if (db_print)
                  {
                     STB_SI_PRINT(("     %u=%lu Hz", i, list_entry->frequency_array[i]));
                  }
                  #endif
               }

               /* Add the new frequency list entry to the start of the lists */
               list_entry->next = *freq_list;
               *freq_list = list_entry;
            }
            else
            {
               STB_FreeMemory(list_entry);
            }
         }
      }
      else
      {
         #ifdef DEBUG_FREQ_LIST_DESC
         if (db_print)
         {
            STB_SI_PRINT(("   Frequency list desc: coding_type 0 is not defined"));
         }
         #endif
      }
   }
   else
   {
      #ifdef DEBUG_FREQ_LIST_DESC
      if (db_print)
      {
         STB_SI_PRINT(("   Frequency list desc: invalid length, %u bytes", dlen));
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif
   }

   FUNCTION_FINISH(ParseFrequencyListDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads FTA content management descriptor
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   desc_ptr  - pointer for return of the descriptor
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseFTAContentDescriptor(U8BIT *dptr, SI_FTA_CONTENT_DESC **desc_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;

   FUNCTION_START(ParseFTAContentDescriptor);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   /* There should be only one descriptor in whichever table it's being parsed from,
    * so skip this one if it already exists */
   if (*desc_ptr == NULL)
   {
      *desc_ptr = (SI_FTA_CONTENT_DESC *)STB_GetMemory(sizeof(SI_FTA_CONTENT_DESC));
      if (*desc_ptr != NULL)
      {
         (*desc_ptr)->do_not_scramble = ((dptr[0] & 0x08) == 0x08) ? TRUE : FALSE;
         (*desc_ptr)->access_over_internet = (dptr[0] & 0x06) >> 1;
         (*desc_ptr)->do_not_apply_revocation = ((dptr[0] & 0x01) == 1) ? TRUE : FALSE;

#ifdef DEBUG_FTA_CONTENT_DESC
         if (db_print)
         {
            STB_SI_PRINT(("   FTA content desc: do_not_scramble=%s",
                          ((*desc_ptr)->do_not_scramble ? "TRUE" : "FALSE")));
            STB_SI_PRINT(("                   : do_not_apply_revocation=%s",
                          ((*desc_ptr)->do_not_apply_revocation ? "TRUE" : "FALSE")));
         }
#else
         USE_UNWANTED_PARAM(db_print);
#endif
      }
   }

   FUNCTION_FINISH(ParseFTAContentDescriptor);

   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads guidance descriptor
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   desc_ptr  - pointer for return of the guidance descriptor
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseGuidanceDescriptor(U8BIT *dptr, SI_GUIDANCE_DESC **desc_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U8BIT index;
   U32BIT *lang_codes;
   SI_STRING_DESC **strings;

   FUNCTION_START(ParseGuidanceDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(desc_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   /* This descriptor should only be received once in per service or event */
   if ((*desc_ptr == NULL) && (dlen >= 4))
   {
      *desc_ptr = (SI_GUIDANCE_DESC *)STB_GetMemory(sizeof(SI_GUIDANCE_DESC));
      memset(*desc_ptr, 0, sizeof(SI_GUIDANCE_DESC));
      (*desc_ptr)->guidance_type = 0xff;
   }

   if ((*desc_ptr != NULL) && (dlen >= 4))
   {
      /* The guidance type should be the same for all descriptors because
       * it's only the lang and text that should be changing, but type and
       * mode are read each time so if they are different it will take the
       * values of the last descriptor read */
      (*desc_ptr)->guidance_type = *dptr & 0x03;
      dptr++;
      dlen--;

      if ((*desc_ptr)->guidance_type < 0x02)
      {
         (*desc_ptr)->guidance_mode = FALSE;

         /* guidance type 0x1 has extra byte for 'mode' */
         if ((*desc_ptr)->guidance_type == 0x01)
         {
            if ((*dptr & 0x01) != 0)
            {
               (*desc_ptr)->guidance_mode = TRUE;
            }
            dptr++;
            dlen--;
         }

         if ((*desc_ptr)->num_langs == 0)
         {
            index = 0;
            lang_codes = (U32BIT *)STB_GetMemory(sizeof(U32BIT));
            strings = (SI_STRING_DESC **)STB_GetMemory(sizeof(SI_STRING_DESC *));
         }
         else
         {
            index = (*desc_ptr)->num_langs;
            lang_codes = (U32BIT *)STB_ResizeMemory((*desc_ptr)->lang_codes,
                  (index + 1) * sizeof(U32BIT));
            strings = (SI_STRING_DESC **)STB_ResizeMemory((*desc_ptr)->strings,
                  (index + 1) * sizeof(SI_STRING_DESC *));
         }

         if ((lang_codes != NULL) && (strings != NULL))
         {
            dptr = ReadLanguageCode(dptr, &lang_codes[index]);
            dlen -= 3;
            dptr = STB_SIReadString(dlen, dptr, &strings[index]);

            (*desc_ptr)->lang_codes = lang_codes;
            (*desc_ptr)->strings = strings;
            (*desc_ptr)->num_langs++;
         }
         else
         {
            if (lang_codes != NULL)
            {
               STB_FreeMemory(lang_codes);
            }
            if (strings != NULL)
            {
               STB_FreeMemory(strings);
            }

            (*desc_ptr)->num_langs = 0;

            if (db_print)
            {
               STB_SI_PRINT(("    Guidance desc: Out of memory!"));
            }
         }
      }

#ifdef DEBUG_GUIDANCE_DESC
      {
         if ((*desc_ptr)->guidance_type < 0x02)
         {
            char *tmp_str = "NULL";
            U8BIT index = (*desc_ptr)->num_langs - 1;
            U32BIT lang_code;

            if (((*desc_ptr)->strings != NULL) && ((*desc_ptr)->strings[index] != NULL))
            {
               if ((*desc_ptr)->strings[index]->str_ptr != NULL)
               {
                  tmp_str = (char *)(*desc_ptr)->strings[index]->str_ptr;
               }
            }

            lang_code = (*desc_ptr)->lang_codes[index];

            STB_SI_PRINT(("   Guidance desc: type=%u, mode=%u, lang=0x%02x%02x%02x, text=\"%s\"",
                          (*desc_ptr)->guidance_type, (*desc_ptr)->guidance_mode, ((lang_code >> 16) & 0xff),
                          ((lang_code >> 8) & 0xff), (lang_code & 0xff), tmp_str));
         }
         else
         {
            STB_SI_PRINT(("   Guidance desc: unsupported type=%u", (*desc_ptr)->guidance_type));
         }
      }
#endif
   }
   else
   {
#ifdef DEBUG_GUIDANCE_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Guidance desc: invalid length, %u", dlen));
      }
#endif
   }

   USE_UNWANTED_PARAM(dptr);
   FUNCTION_FINISH(ParseGuidanceDescriptor);

   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads Iso 639 language descriptor - adds entries to iso lang desc array
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   num_ptr   - pointer for return of number of entries in the array
 * @param   array_ptr - pointer for return of pointer to array
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseIsoLangDescriptor(U8BIT *dptr, U16BIT *num_ptr, SI_ISO_LANG_DESC **array_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U16BIT num_entries;
   SI_ISO_LANG_DESC *array;
   U16BIT i;

   FUNCTION_START(ParseIsoLangDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(num_ptr != NULL);
   ASSERT(array_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 4)
   {
      num_entries = dlen / 4; // each entry in the descriptor is 4 bytes long
      #ifdef DEBUG_ISO_LANG_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   ISO lang desc: %d entries", num_entries));
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif

      // check if there are already entries in the array (i.e. already received a descriptor)
      // if so add to the existing array, otherwise create new array
      if (*array_ptr == NULL)
      {
         // no entries already - create new array
         array = (SI_ISO_LANG_DESC *)STB_GetMemory(num_entries * sizeof(SI_ISO_LANG_DESC));
      }
      else
      {
         // already got entries - make array bigger
         num_entries += *num_ptr;
         array = (SI_ISO_LANG_DESC *)STB_GetMemory(num_entries * sizeof(SI_ISO_LANG_DESC));
         if (array != NULL)
         {
            // copy over previous entries and free old array
            memcpy(array, *array_ptr, (*num_ptr * sizeof(SI_ISO_LANG_DESC)));
            STB_FreeMemory(*array_ptr);
         }
      }

      // add new entries to array
      if (array != NULL)
      {
         for (i = *num_ptr; i < num_entries; i++)
         {
            dptr = ReadLanguageCode(dptr, &array[i].lang_code);
            array[i].audio_type = dptr[0];
            dptr += 1;
            #ifdef DEBUG_ISO_LANG_DESC
            if (db_print == TRUE)
            {
               STB_SI_PRINT(("    lang=%c%c%c, type=%d", (array[i].lang_code >> 16),
                             (array[i].lang_code >> 8), array[i].lang_code, array[i].audio_type));
            }
            #endif
         }
         *array_ptr = array;
         *num_ptr = num_entries;
      }
      else
      {
         #ifdef DEBUG_ISO_LANG_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   CAN'T ALLOCATE MEMORY FOR DESCRIPTOR ARRAY ENTRY"));
         }
         #endif
      }
   }
   else
   {
      #ifdef DEBUG_ISO_LANG_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid ISO lang desc: (%d bytes)", dlen));
      }
      #endif
   }

   FUNCTION_FINISH(ParseIsoLangDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads linkage descriptor - adds entries to linkage desc list
 *
 * @param   dptr           - pointer to length byte of descriptor
 * @param   num_ptr        - pointer for return of number of entries in the array
 * @param   list_ptr       - address of the top of list pointer
 * @param   last_entry_ptr - address of the last entry in list pointer
 * @param   private_data_code private data specifier value when the linkage descriptor is being read
 * @param   db_print       - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseLinkageDescriptor(U8BIT *dptr, U16BIT *num_ptr, SI_LINKAGE_DESC_ENTRY **list_ptr,
   SI_LINKAGE_DESC_ENTRY **last_entry_ptr, U32BIT private_data_code, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   SI_LINKAGE_DESC_ENTRY *desc_ptr;
   U8BIT dsize;

   FUNCTION_START(ParseLinkageDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(num_ptr != NULL);
   ASSERT(list_ptr != NULL);

   #ifndef DEBUG_LINKAGE_DESC
   USE_UNWANTED_PARAM(db_print);
   #endif

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 7)
   {
      dsize = dlen - 7;
      desc_ptr = (SI_LINKAGE_DESC_ENTRY *)STB_GetMemory(sizeof(SI_LINKAGE_DESC_ENTRY) + dsize);
      if (desc_ptr != NULL)
      {
         desc_ptr->next = NULL;
         desc_ptr->private_data_code = private_data_code;
         desc_ptr->tran_id = (dptr[0] << 8) | dptr[1];
         desc_ptr->orig_net_id = (dptr[2] << 8) | dptr[3];
         desc_ptr->serv_id = (dptr[4] << 8) | dptr[5];
         desc_ptr->link_type = dptr[6];
         desc_ptr->data_length = dsize;

         if (dsize != 0)
         {
            memcpy(&(desc_ptr->data), &dptr[7], dsize);
         }

         // now add descriptor to linked list
         if (*last_entry_ptr == NULL)
         {
            // first entry in the list
            *list_ptr = desc_ptr;
         }
         else
         {
            // not the first entry
            (*last_entry_ptr)->next = desc_ptr;
         }
         *last_entry_ptr = desc_ptr;
         (*num_ptr)++;


         #ifdef DEBUG_LINKAGE_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   Linkage desc: tid=0x%04x, onid=0x%04x, sid=0x%04x, type=0x%02x (private_data_code=0x%02x), dlen=%d",
                          desc_ptr->tran_id, desc_ptr->orig_net_id, desc_ptr->serv_id,
                          desc_ptr->link_type, private_data_code, dsize));

            {
               U8BIT i;

               for (i = 0; i < dsize; i++)
               {
                  if ((i & 0x07) == 0)
                  {
                     STB_SPDebugNoCnWrite("               : 0x%02x", dptr[7 + i]);
                  }
                  else if (((i & 0x07) == 7) || (i == (dsize - 1)))
                  {
                     STB_SPDebugWrite(" 0x%02x", dptr[7 + i]);   // includes cr
                  }
                  else
                  {
                     STB_SPDebugNoCnWrite(" 0x%02x", dptr[7 + i]);
                  }
               }
            }
         }
         #endif
      }
      else
      {
         #ifdef DEBUG_LINKAGE_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   CAN'T ALLOCATE MEMORY FOR DESCRIPTOR ARRAY ENTRY"));
         }
         #endif
      }
   }
   else
   {
      #ifdef DEBUG_LINKAGE_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid linkage desc: (%d bytes)", dlen));
      }
      #endif
   }

   FUNCTION_FINISH(ParseLinkageDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads teletext descriptor - sets up array of SI_LTO_DESC
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   num_ptr   - pointer for return of number of entries in the array
 * @param   array_ptr - pointer for return of pointer to array
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseLocalTimeOffsetDescriptor(U8BIT *dptr, U16BIT *num_ptr, SI_LTO_DESC **array_ptr,
   BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U16BIT num_entries;
   SI_LTO_DESC *array;
   U16BIT i;

   FUNCTION_START(ParseLocalTimeOffsetDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(num_ptr != NULL);
   ASSERT(array_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 13)
   {
      num_entries = dlen / 13;   // each entry in the descriptor is 13 bytes long
      #ifdef DEBUG_LOCAL_TIME_OFFSET_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Local time offset desc: %d entries", num_entries));
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif

      // check if there are already entries in the array (i.e. already received a descriptor)
      // if so add to the existing array, otherwise create new array
      if (*array_ptr == NULL)
      {
         // no entries already - create new array
         array = (SI_LTO_DESC *)STB_GetMemory(num_entries * sizeof(SI_LTO_DESC));
      }
      else
      {
         // already got entries - make array bigger
         num_entries += *num_ptr;
         array = (SI_LTO_DESC *)STB_GetMemory(num_entries * sizeof(SI_LTO_DESC));
         if (array != NULL)
         {
            // copy over previous entries and free old array
            memcpy(array, *array_ptr, (*num_ptr * sizeof(SI_LTO_DESC)));
            STB_FreeMemory(*array_ptr);
         }
      }

      // add new entries to array
      if (array != NULL)
      {
         for (i = *num_ptr; i < num_entries; i++)
         {
            dptr = ReadLanguageCode(dptr, &(array[i].country_code));
            array[i].region = (dptr[0] >> 2);
            array[i].offset_negative = ((dptr[0] & 0x01) == 1);
            array[i].offset_hrs = ((dptr[1] >> 4) * 10) + (dptr[1] & 0x0f);
            array[i].offset_mins = ((dptr[2] >> 4) * 10) + (dptr[2] & 0x0f);
            array[i].change_date = (dptr[3] << 8) | dptr[4];
            array[i].change_hrs = ((dptr[5] >> 4) * 10) + (dptr[5] & 0x0f);
            array[i].change_mins = ((dptr[6] >> 4) * 10) + (dptr[6] & 0x0f);
            array[i].change_secs = ((dptr[7] >> 4) * 10) + (dptr[7] & 0x0f);
            array[i].next_offset_hrs = ((dptr[8] >> 4) * 10) + (dptr[8] & 0x0f);
            array[i].next_offset_mins = ((dptr[9] >> 4) * 10) + (dptr[9] & 0x0f);
            dptr += 10;

            #ifdef DEBUG_LOCAL_TIME_OFFSET_DESC
            {
               U8BIT pol_char;

               if (db_print == TRUE)
               {
                  if (array[i].offset_negative == FALSE)
                  {
                     pol_char = '+';
                  }
                  else
                  {
                     pol_char = '-';
                  }
                  STB_SI_PRINT(("    %c%c%c: %c%d:%02d until %d %02d:%02d:%02d then %c%d:%02d",
                                (array[i].country_code >> 16), (array[i].country_code >> 8), array[i].country_code,
                                pol_char, array[i].offset_hrs, array[i].offset_mins,
                                array[i].change_date, array[i].change_hrs, array[i].change_mins, array[i].change_secs,
                                pol_char, array[i].next_offset_hrs, array[i].next_offset_mins));
               }
            }
            #endif
         }
         *array_ptr = array;
         *num_ptr = num_entries;
      }
      else
      {
         #ifdef DEBUG_LOCAL_TIME_OFFSET_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   CAN'T ALLOCATE MEMORY FOR DESCRIPTOR ARRAY ENTRY"));
         }
         #endif
      }
   }
   else
   {
      #ifdef DEBUG_LOCAL_TIME_OFFSET_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid local time offset desc: (%d bytes)", dlen));
      }
      #endif
   }

   FUNCTION_FINISH(ParseLocalTimeOffsetDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads multilingual component descriptor - sets up array of
 *                SI_MULTILING_COMPONENT_DESC
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   num_ptr   - pointer for return of number of entries in the array
 * @param   array_ptr - pointer for return of pointer to array
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseMultilingComponentDescriptor(U8BIT *dptr, U8BIT *num_ptr,
   SI_MULTILING_COMPONENT_DESC **array_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U8BIT num_entries;
   SI_MULTILING_COMPONENT_DESC *array;
   U16BIT i;
   U8BIT *tmp_ptr;
   U8BIT name_len;
   U8BIT tag;

   FUNCTION_START(ParseMultilingComponentDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(num_ptr != NULL);
   ASSERT(array_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 4)
   {
      // work out number of entries
      num_entries = 0;
      tmp_ptr = dptr + 1;  // skip component tag byte
      while (tmp_ptr < end_ptr)
      {
         name_len = tmp_ptr[3];     // read name length
         tmp_ptr += (name_len + 4); // skip language code, name length and name
         num_entries++;
      }

      #ifdef DEBUG_MULTILING_COMPONENT_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Multilingual component desc: %d entries", num_entries));
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif

      // check if there are already entries in the array (i.e. already received a descriptor)
      // if so add to the existing array, otherwise create new array
      if (*array_ptr == NULL)
      {
         // no entries already - create new array
         array = (SI_MULTILING_COMPONENT_DESC *)STB_GetMemory(num_entries * sizeof(SI_MULTILING_COMPONENT_DESC));
      }
      else
      {
         // already got entries - make array bigger
         num_entries += *num_ptr;
         array = (SI_MULTILING_COMPONENT_DESC *)STB_GetMemory(num_entries * sizeof(SI_MULTILING_COMPONENT_DESC));
         if (array != NULL)
         {
            // copy over previous entries and free old array
            memcpy(array, *array_ptr, (*num_ptr * sizeof(SI_MULTILING_COMPONENT_DESC)));
            STB_FreeMemory(*array_ptr);
         }
      }

      // add new entries to array
      if (array != NULL)
      {
         tag = dptr[0];
         dptr++;
         for (i = *num_ptr; i < num_entries; i++)
         {
            array[i].tag = tag;
            dptr = ReadLanguageCode(dptr, &array[i].lang_code);
            dptr = STB_SIReadString(dptr[0], dptr + 1, &array[i].desc_str);

            #ifdef DEBUG_MULTILING_COMPONENT_DESC
            {
               char *tmp_str = "NULL";
               if (db_print == TRUE)
               {
                  if (array[i].desc_str != NULL)
                  {
                     if (array[i].desc_str->str_ptr != NULL)
                     {
                        tmp_str = (char *)array[i].desc_str->str_ptr;
                     }
                  }
                  STB_SI_PRINT(("    tag=%d, lang=%c%c%c, string=%s",
                                array[i].tag, (array[i].lang_code >> 16), (array[i].lang_code >> 8),
                                array[i].lang_code, tmp_str));
               }
            }
            #endif
         }
         *array_ptr = array;
         *num_ptr = num_entries;
      }
      else
      {
         #ifdef DEBUG_MULTILING_COMPONENT_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   CAN'T ALLOCATE MEMORY FOR DESCRIPTOR ARRAY ENTRY"));
         }
         #endif
      }
   }
   else
   {
      #ifdef DEBUG_MULTILING_COMPONENT_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid multilingual component desc: (%d bytes)", dlen));
      }
      #endif
   }

   FUNCTION_FINISH(ParseMultilingComponentDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads multilingual netowrk name descriptor - sets up array of
 *                SI_MULTILING_NET_NAME_DESC
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   num_ptr   - pointer for return of number of entries in the array
 * @param   array_ptr - pointer for return of pointer to array
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseMultilingNetNameDescriptor(U8BIT *dptr, U16BIT *num_ptr,
   SI_MULTILING_NET_NAME_DESC **array_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U16BIT num_entries;
   SI_MULTILING_NET_NAME_DESC *array;
   U16BIT i;
   U8BIT *tmp_ptr;
   U8BIT name_len;

   FUNCTION_START(ParseMultilingNetNameDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(num_ptr != NULL);
   ASSERT(array_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 4)
   {
      // work out number of entries
      num_entries = 0;
      tmp_ptr = dptr;
      while (tmp_ptr < end_ptr)
      {
         name_len = tmp_ptr[3];     // read name length
         tmp_ptr += (name_len + 4); // skip language code, name length and name
         num_entries++;
      }

      #ifdef DEBUG_MULTILING_NET_NAME_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Multilingual net name desc: %d entries", num_entries));
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif

      // check if there are already entries in the array (i.e. already received a descriptor)
      // if so add to the existing array, otherwise create new array
      if (*array_ptr == NULL)
      {
         // no entries already - create new array
         array = (SI_MULTILING_NET_NAME_DESC *)STB_GetMemory(num_entries * sizeof(SI_MULTILING_NET_NAME_DESC));
      }
      else
      {
         // already got entries - make array bigger
         num_entries += *num_ptr;
         array = (SI_MULTILING_NET_NAME_DESC *)STB_GetMemory(num_entries * sizeof(SI_MULTILING_NET_NAME_DESC));
         if (array != NULL)
         {
            // copy over previous entries and free old array
            memcpy(array, *array_ptr, (*num_ptr * sizeof(SI_MULTILING_NET_NAME_DESC)));
            STB_FreeMemory(*array_ptr);
         }
      }

      // add new entries to array
      if (array != NULL)
      {
         for (i = *num_ptr; i < num_entries; i++)
         {
            dptr = ReadLanguageCode(dptr, &array[i].lang_code);
            dptr = STB_SIReadString(dptr[0], dptr + 1, &array[i].name_str);

            #ifdef DEBUG_MULTILING_NET_NAME_DESC
            {
               char *tmp_str = "NULL";
               if (db_print == TRUE)
               {
                  if (array[i].name_str != NULL)
                  {
                     if (array[i].name_str->str_ptr != NULL)
                     {
                        tmp_str = (char *)array[i].name_str->str_ptr;
                     }
                  }
                  STB_SI_PRINT(("    lang=%c%c%c, name=%s", (array[i].lang_code >> 16),
                                (array[i].lang_code >> 8), array[i].lang_code, tmp_str));
               }
            }
            #endif
         }
         *array_ptr = array;
         *num_ptr = num_entries;
      }
      else
      {
         #ifdef DEBUG_MULTILING_NET_NAME_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   CAN'T ALLOCATE MEMORY FOR DESCRIPTOR ARRAY ENTRY"));
         }
         #endif
      }
   }
   else
   {
      #ifdef DEBUG_MULTILING_NET_NAME_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid multilingual net name desc: (%d bytes)", dlen));
      }
      #endif
   }

   FUNCTION_FINISH(ParseMultilingNetNameDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads multilingual netowrk name descriptor - sets up array of
 *                SI_MULTILING_SERV_NAME_DESC
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   num_ptr   - pointer for return of number of entries in the array
 * @param   array_ptr - pointer for return of pointer to array
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseMultilingServNameDescriptor(U8BIT *dptr, U16BIT *num_ptr,
   SI_MULTILING_SERV_NAME_DESC **array_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U16BIT num_entries;
   SI_MULTILING_SERV_NAME_DESC *array;
   U16BIT i;
   U8BIT *tmp_ptr;
   U8BIT name_len;

   FUNCTION_START(ParseMultilingServNameDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(num_ptr != NULL);
   ASSERT(array_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 4)
   {
      // work out number of entries
      num_entries = 0;
      tmp_ptr = dptr;
      while (tmp_ptr < end_ptr)
      {
         name_len = tmp_ptr[3];     // read provider name length
         tmp_ptr += (name_len + 4); // skip language code, name length and name
         name_len = tmp_ptr[0];     // read service name length
         tmp_ptr += (name_len + 1); // skip name and length
         num_entries++;
      }

      #ifdef DEBUG_MULTILING_SERV_NAME_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Multilingual serv name desc: %d entries", num_entries));
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif

      // check if there are already entries in the array (i.e. already received a descriptor)
      // if so add to the existing array, otherwise create new array
      if (*array_ptr == NULL)
      {
         // no entries already - create new array
         array = (SI_MULTILING_SERV_NAME_DESC *)STB_GetMemory(num_entries * sizeof(SI_MULTILING_SERV_NAME_DESC));
      }
      else
      {
         // already got entries - make array bigger
         num_entries += *num_ptr;
         array = (SI_MULTILING_SERV_NAME_DESC *)STB_GetMemory(num_entries * sizeof(SI_MULTILING_SERV_NAME_DESC));
         if (array != NULL)
         {
            // copy over previous entries and free old array
            memcpy(array, *array_ptr, (*num_ptr * sizeof(SI_MULTILING_SERV_NAME_DESC)));
            STB_FreeMemory(*array_ptr);
         }
      }

      // add new entries to array
      if (array != NULL)
      {
         for (i = *num_ptr; i < num_entries; i++)
         {
            dptr = ReadLanguageCode(dptr, &array[i].lang_code);
            dptr = STB_SIReadString(dptr[0], dptr + 1, &array[i].provider_str);
            dptr = STB_SIReadString(dptr[0], dptr + 1, &array[i].name_str);

            #ifdef DEBUG_MULTILING_SERV_NAME_DESC
            {
               char *tmp_prov_str = "NULL";
               char *tmp_name_str = "NULL";

               if (db_print == TRUE)
               {
                  if (array[i].provider_str != NULL)
                  {
                     if (array[i].provider_str->str_ptr != NULL)
                     {
                        tmp_prov_str = (char *)array[i].provider_str->str_ptr;
                     }
                  }
                  if (array[i].name_str != NULL)
                  {
                     if (array[i].name_str->str_ptr != NULL)
                     {
                        tmp_name_str = (char *)array[i].name_str->str_ptr;
                     }
                  }

                  STB_SI_PRINT(("    lang=%c%c%c, provider=%s, name=%s",
                                (array[i].lang_code >> 16), (array[i].lang_code >> 8), array[i].lang_code,
                                tmp_prov_str, tmp_name_str));
               }
            }
            #endif
         }
         *array_ptr = array;
         *num_ptr = num_entries;
      }
      else
      {
         #ifdef DEBUG_MULTILING_SERV_NAME_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   CAN'T ALLOCATE MEMORY FOR DESCRIPTOR ARRAY ENTRY"));
         }
         #endif
      }
   }
   else
   {
      #ifdef DEBUG_MULTILING_SERV_NAME_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid multilingual serv name desc: (%d bytes)", dlen));
      }
      #endif
   }

   FUNCTION_FINISH(ParseMultilingServNameDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads network name descriptor
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   str_ptr   - pointer for return of the name string
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseNetNameDescriptor(U8BIT *dptr, SI_STRING_DESC **name_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;

   FUNCTION_START(ParseNetNameDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(name_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   // this descriptor should only be received once in the NIT - if a name string already exists
   // then this must be a duplicate entry so ignore it
   if (*name_ptr == NULL)
   {
      dptr = STB_SIReadString(dlen, dptr, name_ptr);
      #ifdef DEBUG_NET_NAME_DESC
      {
         char *tmp_str = "NULL";
         if (db_print == TRUE)
         {
            if (*name_ptr != NULL)
            {
               if ((*name_ptr)->str_ptr != NULL)
               {
                  tmp_str = (char *)(*name_ptr)->str_ptr;
               }
            }
            STB_SI_PRINT(("   Net name desc=%s", tmp_str));
         }
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif
   }
   else
   {
      #ifdef DEBUG_NET_NAME_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Net name desc: duplicate entry ignored"));
      }
      #endif
   }

   USE_UNWANTED_PARAM(dptr);
   FUNCTION_FINISH(ParseNetNameDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads parental rating descriptor - sets up array of SI_PARENTAL_RATING_DESC
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   num_ptr   - pointer for return of number of entries in the array
 * @param   array_ptr - pointer for return of pointer to array
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseParentalRatingDescriptor(U8BIT *dptr, U8BIT *num_ptr,
   SI_PARENTAL_RATING_DESC **array_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U8BIT num_entries;
   SI_PARENTAL_RATING_DESC *array;
   U16BIT i;

   FUNCTION_START(ParseParentalRatingDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(num_ptr != NULL);
   ASSERT(array_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 4)
   {
      num_entries = dlen / 4; // each entry in the descriptor is 4 bytes long
      #ifdef DEBUG_PARENTAL_RATING_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Parental rating desc: %d entries", num_entries));
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif

      // check if there are already entries in the array (i.e. already received a descriptor)
      // if so add to the existing array, otherwise create new array
      if (*array_ptr == NULL)
      {
         // no entries already - create new array
         array = (SI_PARENTAL_RATING_DESC *)STB_GetMemory(num_entries * sizeof(SI_PARENTAL_RATING_DESC));
      }
      else
      {
         // already got entries - make array bigger
         num_entries += *num_ptr;
         array = (SI_PARENTAL_RATING_DESC *)STB_GetMemory(num_entries * sizeof(SI_PARENTAL_RATING_DESC));
         if (array != NULL)
         {
            // copy over previous entries and free old array
            memcpy(array, *array_ptr, (*num_ptr * sizeof(SI_PARENTAL_RATING_DESC)));
            STB_FreeMemory(*array_ptr);
         }
      }

      // add new entries to array
      if (array != NULL)
      {
         for (i = *num_ptr; i < num_entries; i++)
         {
            dptr = ReadLanguageCode(dptr, &(array[i].country_code));
            array[i].rating = dptr[0];
            dptr++;
            #ifdef DEBUG_PARENTAL_RATING_DESC
            if (db_print == TRUE)
            {
               STB_SI_PRINT(("    country=%c%c%c, rating=0x%02x", (array[i].country_code >> 16),
                             (array[i].country_code >> 8), array[i].country_code, array[i].rating));
            }
            #endif
         }
         *array_ptr = array;
         *num_ptr = num_entries;
      }
      else
      {
         #ifdef DEBUG_PARENTAL_RATING_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   CAN'T ALLOCATE MEMORY FOR DESCRIPTOR ARRAY ENTRY"));
         }
         #endif
      }
   }
   else
   {
      #ifdef DEBUG_PARENTAL_RATING_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid parental rating desc: (%d bytes)", dlen));
      }
      #endif
   }

   FUNCTION_FINISH(ParseParentalRatingDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads private data specifier descriptor and returns code or 0 if the descriptor
 *                can't be parsed. The returned code can be used in the table parsing functions
 *                to determine if subsequent private descriptors should be handled or not.
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   code_ptr  - pointer for return of private code
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParsePrivateDataSpecifierDescriptor(U8BIT *dptr, U32BIT *code_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;

   FUNCTION_START(ParsePrivateDataSpecifierDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(code_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen == 4)
   {
      *code_ptr = (dptr[0] << 24) | (dptr[1] << 16) | (dptr[2] << 8) | dptr[3];

      #ifdef DEBUG_PRIV_DATA_SPEC_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Private data specifier desc: (returned 0x%08x)", *code_ptr));
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif
   }
   else
   {
      *code_ptr = 0;
      #ifdef DEBUG_PRIV_DATA_SPEC_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid private data specifier desc: (%d bytes)", dlen));
      }
      #endif
   }

   FUNCTION_FINISH(ParsePrivateDataSpecifierDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads service descriptor
 *
 * @param   dptr         - pointer to length byte of descriptor
 * @param   type_ptr     - pointer for return of the service type
 * @param   provider_ptr - pointer for return of the service provider string
 * @param   name_ptr     - pointer for return of the service name string
 * @param   db_print     - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseServiceDescriptor(U8BIT *dptr, U8BIT *type_ptr, SI_STRING_DESC **provider_ptr,
   SI_STRING_DESC **name_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;

   FUNCTION_START(ParseServiceDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(type_ptr != NULL);
   ASSERT(provider_ptr != NULL);
   ASSERT(name_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 3)
   {
      if (*provider_ptr != NULL)
      {
         STB_SIReleaseStringDesc(*provider_ptr);
      }
      if (*name_ptr != NULL)
      {
         STB_SIReleaseStringDesc(*name_ptr);
      }

      *type_ptr = dptr[0];
      dptr++;
      dptr = STB_SIReadString(dptr[0], dptr + 1, provider_ptr);
      dptr = STB_SIReadString(dptr[0], dptr + 1, name_ptr);

      #ifdef DEBUG_SERVICE_DESC
      {
         char *tmp_prov_str = "NULL";
         char *tmp_name_str = "NULL";

         if (db_print == TRUE)
         {
            if (*provider_ptr != NULL)
            {
               if ((*provider_ptr)->str_ptr != NULL)
               {
                  tmp_prov_str = (char *)(*provider_ptr)->str_ptr;
               }
            }
            if (*name_ptr != NULL)
            {
               if ((*name_ptr)->str_ptr != NULL)
               {
                  tmp_name_str = (char *)(*name_ptr)->str_ptr;
               }
            }

            STB_SI_PRINT(("   Service desc: type=%d, prov=%s, name=%s", *type_ptr,
                          tmp_prov_str, tmp_name_str));
         }
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif
   }
   else
   {
      #ifdef DEBUG_SERVICE_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid service desc: (%d bytes)", dlen));
      }
      #endif
   }

   USE_UNWANTED_PARAM(dptr);
   FUNCTION_FINISH(ParseServiceDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads service list descriptor - sets up array of SI_SERV_LIST_DESC
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   num_ptr   - pointer for return of number of entries in the array
 * @param   array_ptr - pointer for return of pointer to array
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseServiceListDescriptor(U8BIT *dptr, U16BIT *num_ptr,
   SI_SERV_LIST_DESC **array_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U16BIT num_entries;
   SI_SERV_LIST_DESC *array;
   U16BIT i;

   FUNCTION_START(ParseServiceListDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(num_ptr != NULL);
   ASSERT(array_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 3)
   {
      num_entries = dlen / 3; // each entry in the descriptor is 3 bytes long
      #ifdef DEBUG_SERV_LIST_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Service list desc: %d entries", num_entries));
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif

      // check if there are already entries in the array (i.e. already received a descriptor)
      // if so add to the existing array, otherwise create new array
      if (*array_ptr == NULL)
      {
         // no entries already - create new array
         array = (SI_SERV_LIST_DESC *)STB_GetMemory(num_entries * sizeof(SI_SERV_LIST_DESC));
      }
      else
      {
         // already got entries - make array bigger
         num_entries += *num_ptr;
         array = (SI_SERV_LIST_DESC *)STB_GetMemory(num_entries * sizeof(SI_SERV_LIST_DESC));
         if (array != NULL)
         {
            // copy over previous entries and free old array
            memcpy(array, *array_ptr, (*num_ptr * sizeof(SI_SERV_LIST_DESC)));
            STB_FreeMemory(*array_ptr);
         }
      }

      // add new entries to array
      if (array != NULL)
      {
         for (i = *num_ptr; i < num_entries; i++)
         {
            array[i].serv_id = (dptr[0] << 8) | dptr[1];
            array[i].serv_type = dptr[2];
            dptr += 3;
            #ifdef DEBUG_SERV_LIST_DESC
            if (db_print == TRUE)
            {
               STB_SI_PRINT(("    id=0x%04x, type=0x%02x", array[i].serv_id, array[i].serv_type));
            }
            #endif
         }
         *array_ptr = array;
         *num_ptr = num_entries;
      }
      else
      {
         #ifdef DEBUG_SERV_LIST_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   CAN'T ALLOCATE MEMORY FOR DESCRIPTOR ARRAY ENTRY"));
         }
         #endif
      }
   }
   else
   {
      #ifdef DEBUG_SERV_LIST_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid service list desc: (%d bytes)", dlen));
      }
      #endif
   }

   FUNCTION_FINISH(ParseServiceListDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads short service name descriptor
 *
 * @param   dptr         - pointer to length byte of descriptor
 * @param   name_ptr     - pointer for return of the service name string
 * @param   db_print     - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseShortServiceNameDescriptor(U8BIT *dptr, SI_STRING_DESC **name_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;

   FUNCTION_START(ParseShortServiceNameDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(name_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen > 1)
   {
      if (*name_ptr != NULL)
      {
         STB_SIReleaseStringDesc(*name_ptr);
      }

      dptr = STB_SIReadString(dlen, dptr, name_ptr);

      #ifdef DEBUG_SHORT_SERVICE_NAME
      {
         char *tmp_name_str = "NULL";

         if (db_print == TRUE)
         {
            if (*name_ptr != NULL)
            {
               if ((*name_ptr)->str_ptr != NULL)
               {
                  tmp_name_str = (char *)(*name_ptr)->str_ptr;
               }
            }

            STB_SI_PRINT(("   Short service name: name=%s", tmp_name_str));
         }
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif
   }
   else
   {
      #ifdef DEBUG_SHORT_SERVICE_NAME
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid short service name: (%d bytes)", dlen));
      }
      #endif
   }

   USE_UNWANTED_PARAM(dptr);
   FUNCTION_FINISH(ParseShortServiceNameDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads short event descriptor - adds entries to short event desc array
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   num_ptr   - pointer for return of number of entries in the array
 * @param   array_ptr - pointer for return of pointer to array
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseShortEventDescriptor(U8BIT *dptr, U8BIT *num_ptr,
   SI_SHORT_EVENT_DESC **array_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U8BIT num_entries;
   SI_SHORT_EVENT_DESC *array;
   U32BIT lang_code;
   SI_STRING_DESC *name_str;
   SI_STRING_DESC *desc_str;

   FUNCTION_START(ParseShortEventDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(num_ptr != NULL);
   ASSERT(array_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 5)
   {
      dptr = ReadLanguageCode(dptr, &lang_code);
      dptr = STB_SIReadString(dptr[0], dptr + 1, &name_str);
      dptr = STB_SIReadString(dptr[0], dptr + 1, &desc_str);

      #ifdef DEBUG_SHORT_EVENT_DESC
      {
         char *tmp_name_str = "NULL";
         char *tmp_desc_str = "NULL";

         if (db_print == TRUE)
         {
            if (name_str != NULL)
            {
               if (name_str->str_ptr != NULL)
               {
                  tmp_name_str = (char *)name_str->str_ptr;
               }
            }
            if (desc_str != NULL)
            {
               if (desc_str->str_ptr != NULL)
               {
                  tmp_desc_str = (char *)desc_str->str_ptr;
               }
            }

            STB_SI_PRINT(("   Short event desc: lang=%c%c%c",
                          (lang_code >> 16), (lang_code >> 8), lang_code));
            STB_SI_PRINT(("    name: %s", tmp_name_str));
            STB_SI_PRINT(("    text: %s", tmp_desc_str));
         }
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif

      // check if there are already entries in the array (i.e. already received a descriptor)
      // if so add to the existing array, otherwise create new array
      if (*array_ptr == NULL)
      {
         // no entries already - create new array
         num_entries = 1;
         array = (SI_SHORT_EVENT_DESC *)STB_GetMemory(sizeof(SI_SHORT_EVENT_DESC));
      }
      else
      {
         // already got entries - make array bigger
         num_entries = *num_ptr + 1;
         array = (SI_SHORT_EVENT_DESC *)STB_GetMemory(num_entries * sizeof(SI_SHORT_EVENT_DESC));
         if (array != NULL)
         {
            // copy over previous entries and free old array
            memcpy(array, *array_ptr, (*num_ptr * sizeof(SI_SHORT_EVENT_DESC)));
            STB_FreeMemory(*array_ptr);
         }
      }

      // add new entry to array
      if (array != NULL)
      {
         array[num_entries - 1].lang_code = lang_code;
         array[num_entries - 1].name_str = name_str;
         array[num_entries - 1].desc_str = desc_str;

         *array_ptr = array;
         *num_ptr = num_entries;
      }
      else
      {
         #ifdef DEBUG_SHORT_EVENT_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   CAN'T ALLOCATE MEMORY FOR DESCRIPTOR ARRAY ENTRY"));
         }
         #endif
      }
   }
   else
   {
      #ifdef DEBUG_SHORT_EVENT_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid short event desc: (%d bytes)", dlen));
      }
      #endif
   }

   USE_UNWANTED_PARAM(dptr);
   FUNCTION_FINISH(ParseShortEventDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads extended event descriptor - adds entries to extended event desc array
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   num_ptr   - pointer for return of number of entries in the array
 * @param   array_ptr - pointer for return of pointer to array
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseExtendedEventDescriptor(U8BIT *dptr, U8BIT *num_ptr,
   SI_EXTENDED_EVENT_DESC **array_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U8BIT num_entries;
   SI_EXTENDED_EVENT_DESC *array;
   U8BIT desc_num, last_desc_num;
   U32BIT lang_code;
   U8BIT num_items;
   U8BIT *item_end;
   SI_STRING_DESC **item_desc_array;
   SI_STRING_DESC **item_text_array;
   SI_STRING_DESC *text_str;

   FUNCTION_START(ParseExtendedEventDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(num_ptr != NULL);
   ASSERT(array_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 5)
   {
      desc_num = *dptr >> 4;
      last_desc_num = *dptr & 0x0f;
      dptr++;

      dptr = ReadLanguageCode(dptr, &lang_code);

      item_end = dptr + *dptr + 1;
      dptr++;

      num_items = 0;
      item_desc_array = NULL;
      item_text_array = NULL;

      while ((dptr < item_end) && (dptr < end_ptr))
      {
         if (num_items == 0)
         {
            item_desc_array = (SI_STRING_DESC **)STB_GetMemory(sizeof(SI_STRING_DESC *));
            item_text_array = (SI_STRING_DESC **)STB_GetMemory(sizeof(SI_STRING_DESC *));
         }
         else
         {
            item_desc_array = (SI_STRING_DESC **)STB_ResizeMemory(item_desc_array,
                  (num_items + 1) * sizeof(SI_STRING_DESC *));
            item_text_array = (SI_STRING_DESC **)STB_ResizeMemory(item_text_array,
                  (num_items + 1) * sizeof(SI_STRING_DESC *));
         }

         dptr = STB_SIReadString(*dptr, dptr + 1, &item_desc_array[num_items]);
         dptr = STB_SIReadString(*dptr, dptr + 1, &item_text_array[num_items]);

         num_items++;
      }

      if (dptr < end_ptr)
      {
         dptr = STB_SIReadString(*dptr, dptr + 1, &text_str);
      }
      else
      {
         text_str = NULL;
      }

      #ifdef DEBUG_EXTENDED_EVENT_DESC
      {
         U8BIT i;

         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   Extended event desc: lang=%c%c%c",
                          (lang_code >> 16), (lang_code >> 8), lang_code));
            STB_SI_PRINT(("    desc num     : %u", desc_num));
            STB_SI_PRINT(("    last desc num: %u", last_desc_num));
            STB_SI_PRINT(("    num items: %u", num_items));

            for (i = 0; i < num_items; i++)
            {
               STB_SI_PRINT(("     %u: item_desc=%s, item=%s", i,
                             ((item_desc_array[i]->str_ptr != NULL) ? (char *)item_desc_array[i]->str_ptr : "NULL"),
                             ((item_text_array[i]->str_ptr != NULL) ? (char *)item_text_array[i]->str_ptr : "NULL")));
            }

            if (text_str != NULL)
            {
               STB_SI_PRINT(("    text=%s", ((text_str->str_ptr != NULL) ? (char *)text_str->str_ptr : "NULL")));
            }
         }
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif

      /* Check if there are already entries in the array (i.e. already received a descriptor),
       * if so add to the existing array, otherwise create new array */
      if (*array_ptr == NULL)
      {
         /* No entries already - create new array */
         num_entries = 1;
         array = (SI_EXTENDED_EVENT_DESC *)STB_GetMemory(sizeof(SI_EXTENDED_EVENT_DESC));
      }
      else
      {
         /* Already got entries - make array bigger */
         num_entries = *num_ptr + 1;
         array = (SI_EXTENDED_EVENT_DESC *)STB_ResizeMemory(*array_ptr,
               num_entries * sizeof(SI_EXTENDED_EVENT_DESC));
      }

      /* Add new entry to array */
      if (array != NULL)
      {
         array[num_entries - 1].desc_number = desc_num;
         array[num_entries - 1].last_desc_number = last_desc_num;
         array[num_entries - 1].lang_code = lang_code;
         array[num_entries - 1].num_items = num_items;
         array[num_entries - 1].item_desc_array = item_desc_array;
         array[num_entries - 1].item_text_array = item_text_array;
         array[num_entries - 1].text_str = text_str;

         *array_ptr = array;
         *num_ptr = num_entries;
      }
      else
      {
         #ifdef DEBUG_EXTENDED_EVENT_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   CAN'T ALLOCATE MEMORY FOR DESCRIPTOR ARRAY ENTRY"));
         }
         #endif
      }
   }
   else
   {
      #ifdef DEBUG_EXTENDED_EVENT_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid extended event desc: (%d bytes)", dlen));
      }
      #endif
   }

   USE_UNWANTED_PARAM(dptr);
   FUNCTION_FINISH(ParseExtendedEventDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads stream id descriptor - returns component tag
 *
 * @param   dptr     - pointer to length byte of descriptor
 * @param   tag_ptr  - pointer for return of component tag
 * @param   db_print - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseStreamIdDescriptor(U8BIT *dptr, U8BIT **tag_array_ptr, U8BIT *num_entries_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U8BIT num_entries;

   FUNCTION_START(ParseStreamIdDescriptor);

   ASSERT(dptr != NULL);

#ifndef DEBUG_STREAM_ID_DESC
   USE_UNWANTED_PARAM(db_print);
#endif

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen == 1)
   {
      if (*tag_array_ptr == NULL)
      {
         /* No entries already - create new array */
         num_entries = 1;
         *tag_array_ptr = (U8BIT *)STB_GetMemory(sizeof(U8BIT));
      }
      else
      {
         /* Already got entries - make array bigger */
         num_entries = *num_entries_ptr + 1;
         *tag_array_ptr = (U8BIT *)STB_ResizeMemory(*tag_array_ptr,
               num_entries * sizeof(U8BIT));
      }

      /* Add new entry to array */
      if (*tag_array_ptr != NULL)
      {
         (*tag_array_ptr)[num_entries - 1] = dptr[0];
         *num_entries_ptr = num_entries;
      }
      else
      {
         #ifdef DEBUG_STREAM_ID_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   CAN'T ALLOCATE MEMORY FOR DESCRIPTOR ARRAY ENTRY"));
         }
         #endif
      }
   }
   else
   {
      #ifdef DEBUG_STREAM_ID_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid stream id desc: (%d bytes)", dlen));
      }
      #endif
   }

   FUNCTION_FINISH(ParseStreamIdDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads subtitle descriptor - sets up array of SI_SUBTITLE_DESC
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   num_ptr   - pointer for return of number of entries in the array
 * @param   array_ptr - pointer for return of pointer to array
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseSubtitleDescriptor(U8BIT *dptr, U16BIT *num_ptr,
   SI_SUBTITLE_DESC **array_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U16BIT num_entries;
   SI_SUBTITLE_DESC *array;
   U16BIT i;

   FUNCTION_START(ParseSubtitleDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(num_ptr != NULL);
   ASSERT(array_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 8)
   {
      num_entries = dlen / 8; // each entry in the descriptor is 8 bytes long
      #ifdef DEBUG_SUBTITLE_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Subtitle desc: %d entries", num_entries));
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif

      // check if there are already entries in the array (i.e. already received a descriptor)
      // if so add to the existing array, otherwise create new array
      if (*array_ptr == NULL)
      {
         // no entries already - create new array
         array = (SI_SUBTITLE_DESC *)STB_GetMemory(num_entries * sizeof(SI_SUBTITLE_DESC));
      }
      else
      {
         // already got entries - make array bigger
         num_entries += *num_ptr;
         array = (SI_SUBTITLE_DESC *)STB_GetMemory(num_entries * sizeof(SI_SUBTITLE_DESC));
         if (array != NULL)
         {
            // copy over previous entries and free old array
            memcpy(array, *array_ptr, (*num_ptr * sizeof(SI_SUBTITLE_DESC)));
            STB_FreeMemory(*array_ptr);
         }
      }

      // add new entries to array
      if (array != NULL)
      {
         for (i = *num_ptr; i < num_entries; i++)
         {
            dptr = ReadLanguageCode(dptr, &(array[i].lang_code));
            array[i].type = dptr[0];
            array[i].composition_page = (dptr[1] << 8) | dptr[2];
            array[i].ancillary_page = (dptr[3] << 8) | dptr[4];
            dptr += 5;
            #ifdef DEBUG_SUBTITLE_DESC
            if (db_print == TRUE)
            {
               STB_SI_PRINT(("    lang=%c%c%c, type=0x%02x, comp pg=%d, anc pg=%d",
                             (array[i].lang_code >> 16), (array[i].lang_code >> 8), array[i].lang_code,
                             array[i].type, array[i].composition_page, array[i].ancillary_page));
            }
            #endif
         }
         *array_ptr = array;
         *num_ptr = num_entries;
      }
      else
      {
         #ifdef DEBUG_SUBTITLE_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   CAN'T ALLOCATE MEMORY FOR DESCRIPTOR ARRAY ENTRY"));
         }
         #endif
      }
   }
   else
   {
      #ifdef DEBUG_SUBTITLE_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid subtitle desc: (%d bytes)", dlen));
      }
      #endif
   }

   FUNCTION_FINISH(ParseSubtitleDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads teletext descriptor - sets up array of SI_TELETEXT_DESC
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   num_ptr   - pointer for return of number of entries in the array
 * @param   array_ptr - pointer for return of pointer to array
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseTeletextDescriptor(U8BIT *dptr, U16BIT *num_ptr,
   SI_TELETEXT_DESC **array_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U16BIT num_entries;
   SI_TELETEXT_DESC *array;
   U16BIT i;

   FUNCTION_START(ParseTeletextDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(num_ptr != NULL);
   ASSERT(array_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 5)
   {
      num_entries = dlen / 5; // each entry in the descriptor is 5 bytes long
      #ifdef DEBUG_TELETEXT_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Teletext desc: %d entries", num_entries));
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif

      // check if there are already entries in the array (i.e. already received a descriptor)
      // if so add to the existing array, otherwise create new array
      if (*array_ptr == NULL)
      {
         // no entries already - create new array
         array = (SI_TELETEXT_DESC *)STB_GetMemory(num_entries * sizeof(SI_TELETEXT_DESC));
      }
      else
      {
         // already got entries - make array bigger
         num_entries += *num_ptr;
         array = (SI_TELETEXT_DESC *)STB_GetMemory(num_entries * sizeof(SI_TELETEXT_DESC));
         if (array != NULL)
         {
            // copy over previous entries and free old array
            memcpy(array, *array_ptr, (*num_ptr * sizeof(SI_TELETEXT_DESC)));
            STB_FreeMemory(*array_ptr);
         }
      }

      // add new entries to array
      if (array != NULL)
      {
         for (i = *num_ptr; i < num_entries; i++)
         {
            dptr = ReadLanguageCode(dptr, &(array[i].lang_code));
            array[i].type = (dptr[0] >> 3);
            array[i].magazine = (dptr[0] & 0x07);
            array[i].page = dptr[1];
            dptr += 2;
            #ifdef DEBUG_TELETEXT_DESC
            if (db_print == TRUE)
            {
               STB_SI_PRINT(("    lang=%c%c%c, type=%d, mag=%d, page=0x%02x",
                             (array[i].lang_code >> 16), (array[i].lang_code >> 8), array[i].lang_code,
                             array[i].type, array[i].magazine, array[i].page));
            }
            #endif
         }
         *array_ptr = array;
         *num_ptr = num_entries;
      }
      else
      {
         #ifdef DEBUG_TELETEXT_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   CAN'T ALLOCATE MEMORY FOR DESCRIPTOR ARRAY ENTRY"));
         }
         #endif
      }
   }
   else
   {
      #ifdef DEBUG_TELETEXT_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid teletext desc: (%d bytes)", dlen));
      }
      #endif
   }

   FUNCTION_FINISH(ParseTeletextDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads terrestrial delivery sys descriptor
 *
 * @param   dptr     - pointer to length byte of descriptor
 * @param   desc_ptr - pointer for the return of delivery system descriptor
 * @param   db_print - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseTerrestrialDeliverySysDescriptor(U8BIT *dptr, SI_DELIVERY_SYS_DESC **desc_ptr,
   BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   SI_DELIVERY_SYS_DESC *del_sys;
   #ifdef DEBUG_TERR_DEL_SYS_DESC
   U8BIT constellation;
   U8BIT hierarchy;
   U8BIT code_rate_hp;
   U8BIT code_rate_lp;
   U8BIT guard;
   U8BIT alt_freq;
   #endif

   FUNCTION_START(ParseTerrestrialDeliverySysDescriptor);

   ASSERT(dptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   #ifdef DEBUG_TERR_DEL_SYS_DESC
   if (db_print == TRUE)
   {
      STB_SI_PRINT(("   Terrestrial delivery system desc:"));
   }
   #else
   USE_UNWANTED_PARAM(db_print);
   #endif

   if (*desc_ptr == NULL)
   {
      *desc_ptr = STB_GetMemory(sizeof(SI_DELIVERY_SYS_DESC));
   }
   del_sys = *desc_ptr;

   if (del_sys != NULL)
   {
      del_sys->terr.is_t2 = FALSE;
      del_sys->terr.u.t1.freq_hz = ((dptr[0] << 24) | (dptr[1] << 16) | (dptr[2] << 8) | dptr[3]) * 10;

      switch (dptr[4] >> 5)
      {
         case 0x00:
            del_sys->terr.u.t1.bwidth = TBWIDTH_8MHZ;
            break;

         case 0x01:
            del_sys->terr.u.t1.bwidth = TBWIDTH_7MHZ;
            break;

         case 0x02:
            del_sys->terr.u.t1.bwidth = TBWIDTH_6MHZ;
            break;

         case 0x03:
            del_sys->terr.u.t1.bwidth = TBWIDTH_5MHZ;
            break;

         default:
            del_sys->terr.u.t1.bwidth = TBWIDTH_UNDEFINED;
            break;
      }

      #ifdef DEBUG_TERR_DEL_SYS_DESC
      constellation = (dptr[5] >> 6);
      hierarchy = ((dptr[5] >> 3) & 0x07);
      code_rate_hp = (dptr[5] & 0x07);
      code_rate_lp = (dptr[6] >> 5);
      guard = ((dptr[6] >> 3) & 0x03);
      alt_freq = (dptr[6] & 0x01);
      #endif

      switch ((dptr[6] >> 1) & 0x03)
      {
         case 0x00:
            del_sys->terr.u.t1.mode = MODE_COFDM_2K;
            break;

         case 0x01:
            del_sys->terr.u.t1.mode = MODE_COFDM_8K;
            break;

         case 0x02:
            del_sys->terr.u.t1.mode = MODE_COFDM_4K;
            break;

         default:
            del_sys->terr.u.t1.mode = MODE_COFDM_UNDEFINED;
            break;
      }

      #ifdef DEBUG_TERR_DEL_SYS_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("    f=%dHz, bw=%d, m=%d, c=%d, h=%d, rhp=%d, rlp=%d, g=%d, alt=%d",
                       del_sys->terr.u.t1.freq_hz, del_sys->terr.u.t1.bwidth, del_sys->terr.u.t1.mode,
                       constellation, hierarchy, code_rate_hp, code_rate_lp, guard, alt_freq));
      }
      #endif
   }

   FUNCTION_FINISH(ParseTerrestrialDeliverySysDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads T2 delivery sys descriptor
 *
 * @param   dptr     - pointer to length byte of descriptor
 * @param   desc_ptr - pointer for the return of delivery system descriptor
 * @param   db_print - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseT2DeliverySysDescriptor(U8BIT *dptr, SI_DELIVERY_SYS_DESC **desc_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   SI_DELIVERY_SYS_DESC *del_sys;
   U8BIT tfs;
   U8BIT num_cells;
   SI_T2_DEL_SYS_CELL *cell;
   U8BIT i;
   U8BIT *freq_ptr;
   #ifdef DEBUG_TERR_DEL_SYS_DESC
   U8BIT j;
   #endif

   FUNCTION_START(ParseT2DeliverySysDescriptor);

   ASSERT(dptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   /* Skip the desc tag extension value */
   dptr++;

   #ifdef DEBUG_TERR_DEL_SYS_DESC
   if (db_print == TRUE)
   {
      STB_SI_PRINT(("   T2 delivery system desc:"));
   }
   #else
   USE_UNWANTED_PARAM(db_print);
   #endif

   if (*desc_ptr == NULL)
   {
      *desc_ptr = (SI_DELIVERY_SYS_DESC *)STB_GetMemory(sizeof(SI_DELIVERY_SYS_DESC));
   }

   del_sys = *desc_ptr;

   if (del_sys != NULL)
   {
      del_sys->terr.is_t2 = TRUE;

      /* Only interested in the first 2 fields of this descriptor, anything else is skipped */
      del_sys->terr.u.t2.plp_id = dptr[0];
      del_sys->terr.u.t2.t2_system_id = (dptr[1] << 8) + dptr[2];

      if (dlen > 4)
      {
         switch ((dptr[3] & 0x3c) >> 2)
         {
            case 0x00:
               del_sys->terr.u.t2.bwidth = TBWIDTH_8MHZ;
               break;

            case 0x01:
               del_sys->terr.u.t2.bwidth = TBWIDTH_7MHZ;
               break;

            case 0x02:
               del_sys->terr.u.t2.bwidth = TBWIDTH_6MHZ;
               break;

            case 0x03:
               del_sys->terr.u.t2.bwidth = TBWIDTH_5MHZ;
               break;

            case 0x04:
               del_sys->terr.u.t2.bwidth = TBWIDTH_10MHZ;
               break;

            default:
               del_sys->terr.u.t2.bwidth = TBWIDTH_UNDEFINED;
               break;
         }

         switch ((dptr[4] & 0x1c) >> 2)
         {
            case 0x00:
               del_sys->terr.u.t2.mode = MODE_COFDM_2K;
               break;

            case 0x01:
               del_sys->terr.u.t2.mode = MODE_COFDM_8K;
               break;

            case 0x02:
               del_sys->terr.u.t2.mode = MODE_COFDM_4K;
               break;

            case 0x03:
               del_sys->terr.u.t2.mode = MODE_COFDM_1K;
               break;

            case 0x04:
               del_sys->terr.u.t2.mode = MODE_COFDM_16K;
               break;

            case 0x05:
               del_sys->terr.u.t2.mode = MODE_COFDM_32K;
               break;

            default:
               del_sys->terr.u.t2.mode = MODE_COFDM_UNDEFINED;
               break;
         }

         tfs = dptr[4] & 0x01;

         del_sys->terr.u.t2.num_cells = 0;
         del_sys->terr.u.t2.cell = NULL;

         dptr += 5;

         num_cells = 0;
         cell = NULL;

         while (dptr < end_ptr)
         {
            num_cells++;
            if (cell == NULL)
            {
               cell = (SI_T2_DEL_SYS_CELL *)STB_GetMemory(sizeof(SI_T2_DEL_SYS_CELL));
            }
            else
            {
               cell = (SI_T2_DEL_SYS_CELL *)STB_ResizeMemory(cell, num_cells * sizeof(SI_T2_DEL_SYS_CELL));
            }

            if (cell != NULL)
            {
               cell[num_cells - 1].cell_id = (dptr[0] << 8) + dptr[1];

               if (tfs == 1)
               {
                  cell[num_cells - 1].num_freqs = dptr[2] / 4;
                  dptr += 3;

                  for (i = 0, freq_ptr = dptr; i < cell[num_cells - 1].num_freqs; freq_ptr += 4, i++)
                  {
                     cell[num_cells - 1].freq_hz[i] = ((freq_ptr[0] << 24) | (freq_ptr[1] << 16) |
                                                       (freq_ptr[2] << 8) | freq_ptr[3]) * 10;
                     dptr += 4;
                  }
               }
               else
               {
                  cell[num_cells - 1].num_freqs = 1;
                  cell[num_cells - 1].freq_hz[0] = ((dptr[2] << 24) | (dptr[3] << 16) | (dptr[4] << 8) | dptr[5]) * 10;

                  dptr += 6;
               }

               /* Next come the subcells, but these are currently being skipped - not sure how we'd use them */
               dptr += dptr[0];
            }
         }

         del_sys->terr.u.t2.num_cells = num_cells;
         del_sys->terr.u.t2.cell = cell;
      }
      else
      {
         del_sys->terr.u.t2.bwidth = TBWIDTH_UNDEFINED;
         del_sys->terr.u.t2.mode = MODE_COFDM_UNDEFINED;
         del_sys->terr.u.t2.num_cells = 0;
         del_sys->terr.u.t2.cell = NULL;
      }

      #ifdef DEBUG_TERR_DEL_SYS_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("    plp=%u, t2_system_id=0x%04x, bw=%d, m=%d", del_sys->terr.u.t2.plp_id,
                       del_sys->terr.u.t2.t2_system_id, del_sys->terr.u.t2.bwidth, del_sys->terr.u.t2.mode));
         if (del_sys->terr.u.t2.num_cells > 0)
         {
            for (i = 0; i < del_sys->terr.u.t2.num_cells; i++)
            {
               STB_SI_PRINT(("     cell=0x%04x", del_sys->terr.u.t2.cell[i].cell_id));
               for (j = 0; j < del_sys->terr.u.t2.cell[i].num_freqs; j++)
               {
                  STB_SI_PRINT(("      freq=%luHz", del_sys->terr.u.t2.cell[i].freq_hz[j]));
               }
            }
         }
      }
      #endif
   }

   FUNCTION_FINISH(ParseT2DeliverySysDescriptor);

   return(end_ptr);
}

static U8BIT* ParseSatelliteDeliverySysDescriptor(U8BIT *dptr, SI_DELIVERY_SYS_DESC **desc_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   SI_DELIVERY_SYS_DESC *del_sys;
   FUNCTION_START(ParseSatelliteDeliverySysDescriptor);

   ASSERT(dptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   #ifdef DEBUG_SAT_DEL_SYS_DESC
   if (db_print == TRUE)
   {
      STB_SI_PRINT(("   Satellite delivery system desc:"));
   }
   #else
   USE_UNWANTED_PARAM(db_print);
   #endif

   if (*desc_ptr == NULL)
   {
      *desc_ptr = STB_GetMemory(sizeof(SI_DELIVERY_SYS_DESC));
   }
   del_sys = *desc_ptr;

   if (del_sys != NULL)
   {
      del_sys->sat.freq_hz = (U32BIT)BCDToFloat(dptr, 8, 6);
      dptr += 4;

      /* Save the orbital position in 1/10ths degree */
      del_sys->sat.position = (U16BIT)(BCDToFloat(dptr, 4, 3) * 10);
      dptr += 2;

      del_sys->sat.east_west = ((((*dptr & 0x80) >> 7) == 1) ? TRUE : FALSE);
      del_sys->sat.polarity = (E_STB_DP_POLARITY)((*dptr & 0x60) >> 5);
      del_sys->sat.dvb_s2 = (((*dptr & 0x04) != 0) ? TRUE : FALSE);

      switch (*dptr & 0x03)
      {
         case 0x00: /* Auto */
            del_sys->sat.modulation = MOD_AUTO;
            break;

         case 0x01:
            del_sys->sat.modulation = MOD_QPSK;
            break;

         case 0x02:
            del_sys->sat.modulation = MOD_8PSK;
            break;

         case 0x03:
            del_sys->sat.modulation = MOD_16QAM;
            break;
      }
      dptr++;

      del_sys->sat.sym_rate = (U16BIT)((BCDToFloat(dptr, 7, 3) * 1000) + 0.5);
      dptr += 3;
      del_sys->sat.fec_code = (E_STB_DP_FEC)(*dptr & 0x0f);

      #ifdef DEBUG_SAT_DEL_SYS_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("    f=%dHz, sym=%d, fec=%d, pol=%d, [pos=%d, we=%s, s2=%u, mod=%s]\n",
                       del_sys->sat.freq_hz,
                       del_sys->sat.sym_rate,
                       del_sys->sat.fec_code,
                       del_sys->sat.polarity,
                       del_sys->sat.position,
                       (del_sys->sat.east_west ? "E" : "W"),
                       del_sys->sat.dvb_s2,
                       ((del_sys->sat.modulation == MOD_QPSK) ? "QPSK" :
                        ((del_sys->sat.modulation == MOD_8PSK) ? "8PSK" :
                         ((del_sys->sat.modulation == MOD_16QAM) ? "16QAM" : "Auto")))));
      }
      #endif
   }

   FUNCTION_FINISH(ParseSatelliteDeliverySysDescriptor);

   return(end_ptr);
}

static U8BIT* ParseCableDeliverySysDescriptor(U8BIT *dptr, SI_DELIVERY_SYS_DESC **desc_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   SI_DELIVERY_SYS_DESC *del_sys;

   FUNCTION_START(ParseCableDeliverySysDescriptor);

   ASSERT(dptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   #ifdef DEBUG_CABLE_DEL_SYS_DESC
   if (db_print == TRUE)
   {
      STB_SI_PRINT(("   Cable delivery system desc:"));
   }
   #else
   USE_UNWANTED_PARAM(db_print);
   #endif

   if (*desc_ptr == NULL)
   {
      *desc_ptr = (SI_DELIVERY_SYS_DESC *)STB_GetMemory(sizeof(SI_DELIVERY_SYS_DESC));
   }

   del_sys = *desc_ptr;

   if (del_sys != NULL)
   {
      /* Read the value and convert to MHz */
      del_sys->cable.freq_hz = (U32BIT)(BCDToFloat(dptr, 8, 4) * 1000000);
      dptr += 4;

      /* Skip reserved field */
      dptr++;

      del_sys->cable.fec_outer = *dptr & 0x0f;
      dptr++;

      del_sys->cable.modulation = *dptr;
      dptr++;

      del_sys->cable.symbol_rate = (U32BIT)(BCDToFloat(dptr, 7, 3) * 1000);
      dptr += 3;

      del_sys->cable.fec_inner = *dptr & 0x0f;

      #ifdef DEBUG_CABLE_DEL_SYS_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("    freq=%u, fec_outer=0x%x, modulation=0x%02x, symbol_rate=%u, fec_inner=0x%x",
                       del_sys->cable.freq_hz, del_sys->cable.fec_outer, del_sys->cable.modulation,
                       del_sys->cable.symbol_rate, del_sys->cable.fec_inner));
      }
      #endif
   }

   FUNCTION_FINISH(ParseCableDeliverySysDescriptor);

   return(end_ptr);
}

static U8BIT* ParseFreesatLinkageDescriptor(U8BIT *dptr, SI_FREESAT_LINKAGE_DESC **desc, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   SI_FREESAT_LINKAGE_DESC *new_item;
   SI_FREESAT_LINKAGE_DESC *last_item;
   U8BIT tun_loop_length;
   U8BIT i;

   FUNCTION_START(ParseFreesatLinkageDescriptor);

   ASSERT(dptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   #ifdef DEBUG_FREESAT_LINKAGE_DESC
   if (db_print == TRUE)
   {
      STB_SI_PRINT(("   Free sat linkage desc: len=%u", dlen));
   }
   #else
   USE_UNWANTED_PARAM(db_print);
   #endif

   /* There should only be one of these descriptors, so ignore this if there's already parsed data */
   if (*desc == NULL)
   {
      last_item = NULL;

      while (dptr < end_ptr)
      {
         new_item = (SI_FREESAT_LINKAGE_DESC *)STB_GetMemory(sizeof(SI_FREESAT_LINKAGE_DESC));
         if (new_item != NULL)
         {
            new_item->next = NULL;
            new_item->trans_id = (dptr[0] << 8) | dptr[1];
            new_item->onet_id = (dptr[2] << 8) | dptr[3];
            dptr += 4;

            /* According to Freesat spec, the standard loop length should be 0, but skip it anyway */
            dptr += (*dptr) + 1;

            new_item->serv_id = (dptr[0] << 8) | dptr[1];

            #ifdef DEBUG_FREESAT_LINKAGE_DESC
            STB_SI_PRINT(("     onid=0x%04x, tid=0x%04x, sid=0x%04x",
                          new_item->onet_id, new_item->trans_id, new_item->serv_id));
            #endif

            tun_loop_length = dptr[2];
            dptr += 3;

            if (tun_loop_length > 0)
            {
               new_item->num_data_types = tun_loop_length;

               /* Allocate memory to hold the tunnelled data types */
               new_item->data_types = (U8BIT *)STB_GetMemory(tun_loop_length * sizeof(U8BIT));
               if (new_item->data_types != NULL)
               {
                  for (i = 0; i < tun_loop_length; i++)
                  {
                     new_item->data_types[i] = dptr[i];

                     #ifdef DEBUG_FREESAT_LINKAGE_DESC
                     STB_SI_PRINT(("      tunnelled data type 0x%02x", dptr[i]));
                     #endif
                  }
               }
               else
               {
                  /* Descriptor is of no use without the types */
                  STB_FreeMemory(new_item);
                  new_item = NULL;
                  #ifdef DEBUG_FREESAT_LINKAGE_DESC
                  STB_SI_PRINT(("     Failed to allocate memory for tunnelled data types, descriptor ignored"));
                  #endif
               }
            }
            else
            {
               /* If no tunnelled data is defined then this descriptor isn't any use */
               STB_FreeMemory(new_item);
               new_item = NULL;
               #ifdef DEBUG_FREESAT_LINKAGE_DESC
               STB_SI_PRINT(("     No tunnelled data, descriptor ignored"));
               #endif
            }

            dptr += tun_loop_length;

            if (new_item != NULL)
            {
               /* Add this linkage descriptor to the list */
               if (*desc == NULL)
               {
                  *desc = new_item;
               }
               else
               {
                  last_item->next = new_item;
               }

               last_item = new_item;
            }
         }
         #ifdef DEBUG_FREESAT_LINKAGE_DESC
         else
         {
            STB_SI_PRINT(("   Failed to allocate memory for freesat linkage desc"));
         }
         #endif
      }
   }

   FUNCTION_FINISH(ParseFreesatLinkageDescriptor);

   return(end_ptr);
}

static U8BIT* ParseFreesatPrefixDescriptor(U8BIT *dptr, SI_FREESAT_PREFIX_DESC **list, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   SI_FREESAT_PREFIX_DESC *prefix;

   FUNCTION_START(ParseFreesatPrefixDescriptor);

   ASSERT(dptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   #ifdef DEBUG_FREESAT_PREFIX_DESC
   if (db_print)
   {
      STB_SI_PRINT(("   Freesat prefix desc: len=%u", dlen));
   }
   #else
   USE_UNWANTED_PARAM(db_print);
   #endif

   while (dptr < end_ptr)
   {
      prefix = (SI_FREESAT_PREFIX_DESC *)STB_GetMemory(sizeof(SI_FREESAT_PREFIX_DESC));
      if (prefix != NULL)
      {
         prefix->prefix_index = *dptr & 0x7f;
         dptr++;

         dptr = STB_SIReadString(*dptr, dptr + 1, &prefix->uri_prefix);

         #ifdef DEBUG_FREESAT_PREFIX_DESC
         STB_SI_PRINT(("     index=%u, prefix=\"%s\"", prefix->prefix_index, prefix->uri_prefix->str_ptr));
         #endif

         /* Prepend this descriptor to the list */
         prefix->next = *list;
         *list = prefix;
      }
      #ifdef DEBUG_FREESAT_PREFIX_DESC
      else
      {
         STB_SI_PRINT(("   Failed to allocate memory for freesat prefix desc"));
      }
      #endif
   }

   FUNCTION_FINISH(ParseFreesatPrefixDescriptor);

   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads logical channel descriptor - sets up array of SI_LCN_DESC
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   dtag      - the descriptor ID
 * @param   num_ptr   - pointer for return of number of entries in the array
 * @param   array_ptr - pointer for return of pointer to array
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseLogicalChannelDescriptor(U8BIT *dptr, U8BIT dtag, U16BIT *num_ptr,
   SI_LCN_DESC **array_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U16BIT num_entries;
   SI_LCN_DESC *lcns;
   U16BIT i;

   FUNCTION_START(ParseLogicalChannelDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(num_ptr != NULL);
   ASSERT(array_ptr != NULL);

   USE_UNWANTED_PARAM(dtag);

#ifndef DEBUG_LCN_DESC
   USE_UNWANTED_PARAM(db_print);
#endif

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 4)
   {
      num_entries = dlen / 4; // each entry in the descriptor is 4 bytes long
      #ifdef DEBUG_LCN_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Logical channel desc 0x%02x: %d entries", dtag, num_entries));
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif

      // check if there are already entries in the array (i.e. already received a descriptor)
      // if so add to the existing array, otherwise create new array
      if (*array_ptr == NULL)
      {
         // no entries already - create new array
         lcns = (SI_LCN_DESC *)STB_GetMemory(num_entries * sizeof(SI_LCN_DESC));
      }
      else
      {
         // already got entries - make array bigger
         num_entries += *num_ptr;
         lcns = (SI_LCN_DESC *)STB_GetMemory(num_entries * sizeof(SI_LCN_DESC));
         if (lcns != NULL)
         {
            // copy over previous entries and free old array
            memcpy(lcns, *array_ptr, (*num_ptr * sizeof(SI_LCN_DESC)));
            STB_FreeMemory(*array_ptr);
         }
      }

      // add new entries to array
      if (lcns != NULL)
      {
         for (i = *num_ptr; i < num_entries; i++)
         {
            lcns[i].serv_id = (dptr[0] << 8) | dptr[1];
            lcns[i].serv_lcn = ((dptr[2] & 0x03) << 8) | dptr[3];

            /* The 1-bit field 'service visible' flag is defined in the E-Book
             * but not in the UK's D-Book, where this bit is a reserved bit. However,
             * all reserved bits are defined as being set to '1' */
            if ((dptr[2] & 0x80) != 0)
            {
               lcns[i].visible = TRUE;
            }
            else
            {
               lcns[i].visible = FALSE;
            }

            dptr += 4;
            #ifdef DEBUG_LCN_DESC
            if (db_print == TRUE)
            {
               STB_SI_PRINT(("    id=0x%04x, lcn=%d, visible=%s", lcns[i].serv_id, lcns[i].serv_lcn,
                             (lcns[i].visible ? "TRUE" : "FALSE")));
            }
            #endif
         }
         *array_ptr = lcns;
         *num_ptr = num_entries;
      }
      else
      {
         #ifdef DEBUG_LCN_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   CAN'T ALLOCATE MEMORY FOR DESCRIPTOR ARRAY ENTRY"));
         }
         #endif
      }
   }
   else
   {
      #ifdef DEBUG_LCN_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid Logical channel desc: (%d bytes)", dlen));
      }
      #endif
   }

   FUNCTION_FINISH(ParseLogicalChannelDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads CNS logical channel descriptor - sets up array of SI_LCN_DESC
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   dtag      - the descriptor ID
 * @param   num_ptr   - pointer for return of number of entries in the array
 * @param   array_ptr - pointer for return of pointer to array
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseCnsLogicalChannelDescriptor(U8BIT *dptr, U8BIT dtag, U16BIT *num_ptr,
   SI_LCN_DESC **array_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U16BIT num_entries;
   SI_LCN_DESC *lcns;
   U16BIT i;

   FUNCTION_START(ParseLogicalChannelDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(num_ptr != NULL);
   ASSERT(array_ptr != NULL);

   USE_UNWANTED_PARAM(dtag);

#ifndef DEBUG_LCN_DESC
   USE_UNWANTED_PARAM(db_print);
#endif

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 4)
   {
      num_entries = dlen / 4; // each entry in the descriptor is 4 bytes long
      #ifdef DEBUG_LCN_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   CNS Logical channel desc 0x%02x: %d entries", dtag, num_entries));
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif

      // check if there are already entries in the array (i.e. already received a descriptor)
      // if so add to the existing array, otherwise create new array
      if (*array_ptr == NULL)
      {
         // no entries already - create new array
         lcns = (SI_LCN_DESC *)STB_GetMemory(num_entries * sizeof(SI_LCN_DESC));
      }
      else
      {
         // already got entries - make array bigger
         num_entries += *num_ptr;
         lcns = (SI_LCN_DESC *)STB_GetMemory(num_entries * sizeof(SI_LCN_DESC));
         if (lcns != NULL)
         {
            // copy over previous entries and free old array
            memcpy(lcns, *array_ptr, (*num_ptr * sizeof(SI_LCN_DESC)));
            STB_FreeMemory(*array_ptr);
         }
      }

      // add new entries to array
      if (lcns != NULL)
      {
         for (i = *num_ptr; i < num_entries; i++)
         {
            lcns[i].serv_id = (dptr[0] << 8) | dptr[1];
            lcns[i].serv_lcn = (dptr[2] << 8) | dptr[3];
            lcns[i].visible = TRUE;

            dptr += 4;
            #ifdef DEBUG_LCN_DESC
            if (db_print == TRUE)
            {
               STB_SI_PRINT(("    id=0x%04x, lcn=%d, visible=%s", lcns[i].serv_id, lcns[i].serv_lcn,
                             (lcns[i].visible ? "TRUE" : "FALSE")));
            }
            #endif
         }
         *array_ptr = lcns;
         *num_ptr = num_entries;
      }
      else
      {
         #ifdef DEBUG_LCN_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   CAN'T ALLOCATE MEMORY FOR DESCRIPTOR ARRAY ENTRY"));
         }
         #endif
      }
   }
   else
   {
      #ifdef DEBUG_LCN_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid Logical channel desc: (%d bytes)", dlen));
      }
      #endif
   }

   FUNCTION_FINISH(ParseLogicalChannelDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads a service attribute descriptor - sets up array of SI_SERV_ATTRIBUTE_DESC
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   num_ptr   - pointer for return of number of entries in the array
 * @param   array_ptr - pointer for return of pointer to array
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseServiceAttributeDescriptor(U8BIT *dptr, U16BIT *num_ptr,
   SI_SERV_ATTRIBUTE_DESC **array_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U16BIT num_entries;
   SI_SERV_ATTRIBUTE_DESC *array;
   U16BIT i;

   FUNCTION_START(ParseServiceAttributeDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(num_ptr != NULL);
   ASSERT(array_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 3)
   {
      num_entries = dlen / 3; // each entry in the descriptor is 3 bytes long
#ifdef DEBUG_LCN_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Service attribute desc: %d entries", num_entries));
      }
      #else
      USE_UNWANTED_PARAM(db_print);
#endif

      // check if there are already entries in the array (i.e. already received a descriptor)
      // if so add to the existing array, otherwise create new array
      if (*array_ptr == NULL)
      {
         // no entries already - create new array
         array = (SI_SERV_ATTRIBUTE_DESC *)STB_GetMemory(num_entries * sizeof(SI_SERV_ATTRIBUTE_DESC));
      }
      else
      {
         // already got entries - make array bigger
         num_entries += *num_ptr;
         array = (SI_SERV_ATTRIBUTE_DESC *)STB_GetMemory(num_entries * sizeof(SI_SERV_ATTRIBUTE_DESC));
         if (array != NULL)
         {
            // copy over previous entries and free old array
            memcpy(array, *array_ptr, (*num_ptr * sizeof(SI_SERV_ATTRIBUTE_DESC)));
            STB_FreeMemory(*array_ptr);
         }
      }

      // add new entries to array
      if (array != NULL)
      {
         for (i = *num_ptr; i < num_entries; i++)
         {
            array[i].serv_id = (dptr[0] << 8) | dptr[1];
            /* If the service is visible then it is also selectable */
            if ((dptr[2] & 0x01) != 0)
            {
               array[i].service_visible = TRUE;
               array[i].service_selectable = TRUE;
            }
            else
            {
               array[i].service_visible = FALSE;
               if ((dptr[2] & 0x02) != 0)
               {
                  array[i].service_selectable = TRUE;
               }
               else
               {
                  array[i].service_selectable = FALSE;
               }
            }

            dptr += 3;
#ifdef DEBUG_LCN_DESC
            if (db_print == TRUE)
            {
               STB_SI_PRINT(("    id=0x%04x, visible=%d, selectable=%d", array[i].serv_id,
                             array[i].service_visible, array[i].service_selectable));
            }
#endif
         }
         *array_ptr = array;
         *num_ptr = num_entries;
      }
      else
      {
#ifdef DEBUG_LCN_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   CAN'T ALLOCATE MEMORY FOR DESCRIPTOR ARRAY ENTRY"));
         }
#endif
      }
   }
   else
   {
#ifdef DEBUG_LCN_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid service attribute desc: (%d bytes)", dlen));
      }
#endif
   }

   FUNCTION_FINISH(ParseServiceAttributeDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads the Nordig logical channel descriptor - sets up array of SI_NORDIG_LCN_DESC
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   num_ptr   - pointer for return of number of entries in the array
 * @param   array_ptr - pointer for return of pointer to array
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseNordigLCN2Descriptor(U8BIT *dptr, U16BIT *num_ptr, SI_NORDIG_LCN_DESC **array_ptr,
   BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U16BIT num_entries;
   SI_NORDIG_LCN_DESC *array;
   U16BIT i, j;

   FUNCTION_START(ParseNordigLCN2Descriptor);

   ASSERT(dptr != NULL);
   ASSERT(num_ptr != NULL);
   ASSERT(array_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   num_entries = *num_ptr;

   #ifdef DEBUG_LCN_DESC
   if (db_print == TRUE)
   {
      STB_SI_PRINT(("   Nordig logical channel desc:"));
   }
   #else
   USE_UNWANTED_PARAM(db_print);
   #endif

   for (i = num_entries; dlen > 6; i++)
   {
      /* Check if there are already entries in the array (i.e. already received a descriptor)
       * and if so add to the existing array, otherwise create new array */
      array = (SI_NORDIG_LCN_DESC *)STB_GetMemory((num_entries + 1) * sizeof(SI_NORDIG_LCN_DESC));

      num_entries++;
      if ((array != NULL) && (*array_ptr != NULL))
      {
         /* already got entries - copy over previous entries and free old array */
         memcpy(array, *array_ptr, (num_entries * sizeof(SI_NORDIG_LCN_DESC)));
         STB_FreeMemory(*array_ptr);
      }

      /* add new entries to array */
      if (array != NULL)
      {
         array[i].chan_list_id = *dptr;
         dptr++;

         /* Read the channel list name */
         dptr = STB_SIReadString(*dptr, dptr + 1, &array[i].chan_list_name);

         dptr = ReadLanguageCode(dptr, &array[i].country_code);

         /* Each entry in serv/LCN list is 4 bytes in size */
         array[i].num_services = *dptr / 4;

         #ifdef DEBUG_LCN_DESC
         if (db_print)
         {
            char *tmp_str = "";
            U32BIT lang = array[i].country_code;

            if (array[i].chan_list_name != NULL)
            {
               if (array[i].chan_list_name->str_ptr != NULL)
               {
                  tmp_str = (char *)array[i].chan_list_name->str_ptr;
               }
            }

            STB_SI_PRINT(("   list_id=%u, list_name=\"%s\", lang=%c%c%c, num servs=%u",
                          array[i].chan_list_id, tmp_str, (lang >> 16), (lang >> 8), lang,
                          array[i].num_services));
         }
         #endif

         if (array[i].num_services > 0)
         {
            array[i].serv_array = (SI_NORDIG_SERV_LCN *)STB_GetMemory(array[i].num_services * sizeof(SI_NORDIG_SERV_LCN));
            if (array[i].serv_array != NULL)
            {
               dptr++;

               for (j = 0; j < array[i].num_services; j++)
               {
                  array[i].serv_array[j].serv_id = (dptr[0] << 8) | dptr[1];
                  array[i].serv_array[j].visible = ((dptr[2] & 0x80) != 0) ? TRUE : FALSE;
                  array[i].serv_array[j].serv_lcn = ((dptr[2] & 0x03) << 8) | dptr[3];
                  dptr += 4;

                  #ifdef DEBUG_LCN_DESC
                  if (db_print == TRUE)
                  {
                     STB_SI_PRINT(("    id=0x%04x, lcn=%d, visible=%s", array[i].serv_array[j].serv_id,
                                   array[i].serv_array[j].serv_lcn,
                                   (array[i].serv_array[j].visible ? "TRUE" : "FALSE")));
                  }
                  #endif
               }
            }
            else
            {
               array[i].num_services = 0;
               dptr += *dptr;
               dptr++;
            }
         }
         else
         {
            array[i].serv_array = NULL;
            dptr += *dptr;
            dptr++;
         }

         *array_ptr = array;
      }
      else
      {
         #ifdef DEBUG_LCN_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   CAN'T ALLOCATE MEMORY FOR DESCRIPTOR ARRAY ENTRY"));
         }
         #endif
      }

      /* Calculate the number of bytes left to be processed */
      dlen = end_ptr - dptr;
   }

   *num_ptr = num_entries;

   FUNCTION_FINISH(ParseNordigLCN2Descriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads terrestrial delivery sys descriptor
 *
 * @param   dptr     - pointer to length byte of descriptor
 * @param   desc_ptr - pointer for the return of delivery system descriptor
 * @param   db_print - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParsePreferredNameIdDescriptor(U8BIT *dptr, U8BIT *id_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;

   FUNCTION_START(ParsePreferredNameIdDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(id_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen == 1)
   {
      *id_ptr = dptr[0];
      #ifdef DEBUG_PREF_NAME_ID_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Preferred name id desc: id=%d", *id_ptr));
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif
   }
   else
   {
      *id_ptr = 0;
      #ifdef DEBUG_PREF_NAME_ID_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid Preferred name id desc: (%d bytes)", dlen));
      }
      #endif
   }

   FUNCTION_FINISH(ParsePreferredNameIdDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads preferred name list descriptor - sets up array of SI_PREFERRED_NAME_DESC
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   num_ptr   - pointer for return of number of entries in the array
 * @param   array_ptr - pointer for return of pointer to array
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParsePreferredNameListDescriptor(U8BIT *dptr, U16BIT *num_ptr,
   SI_PREFERRED_NAME_DESC **array_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U16BIT num_entries;
   SI_PREFERRED_NAME_DESC *array;
   U16BIT i;
   U8BIT *tmp_ptr;
   U8BIT name_len;
   U16BIT name_count;
   U16BIT entry_no;
   U32BIT lang_code;

   FUNCTION_START(ParsePreferredNameListDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(num_ptr != NULL);
   ASSERT(array_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 4)
   {
      // work out number of entries
      num_entries = 0;
      tmp_ptr = dptr;
      while (tmp_ptr < end_ptr)
      {
         name_count = tmp_ptr[3];         // read name count
         tmp_ptr += 4;                    // skip language code and name count
         for (i = 0; i < name_count; i++)
         {
            name_len = tmp_ptr[1];        // read name length
            tmp_ptr += (name_len + 2);    // skip name id, name length and name
         }
         num_entries += name_count;
      }

      #ifdef DEBUG_PREF_NAME_LIST_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Preferred name list desc: %d entries", num_entries));
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif

      // check if there are already entries in the array (i.e. already received a descriptor)
      // if so add to the existing array, otherwise create new array
      if (*array_ptr == NULL)
      {
         // no entries already - create new array
         array = (SI_PREFERRED_NAME_DESC *)STB_GetMemory(num_entries * sizeof(SI_PREFERRED_NAME_DESC));
      }
      else
      {
         // already got entries - make array bigger
         num_entries += *num_ptr;
         array = (SI_PREFERRED_NAME_DESC *)STB_GetMemory(num_entries * sizeof(SI_PREFERRED_NAME_DESC));
         if (array != NULL)
         {
            // copy over previous entries and free old array
            memcpy(array, *array_ptr, (*num_ptr * sizeof(SI_PREFERRED_NAME_DESC)));
            STB_FreeMemory(*array_ptr);
         }
      }

      // add new entries to array
      if (array != NULL)
      {
         entry_no = *num_ptr;
         while ((dptr < end_ptr) && (entry_no < num_entries))
         {
            dptr = ReadLanguageCode(dptr, &lang_code);
            name_count = dptr[0];
            dptr++;
            for (i = 0; i < name_count; i++, entry_no++)
            {
               array[entry_no].lang_code = lang_code;
               array[entry_no].name_id = dptr[0];
               dptr++;
               dptr = STB_SIReadString(dptr[0], dptr + 1, &array[entry_no].name_str);

               #ifdef DEBUG_PREF_NAME_LIST_DESC
               {
                  char *tmp_str = "NULL";

                  if (db_print == TRUE)
                  {
                     if (array[entry_no].name_str != NULL)
                     {
                        if (array[entry_no].name_str->str_ptr != NULL)
                        {
                           tmp_str = (char *)array[entry_no].name_str->str_ptr;
                        }
                     }

                     STB_SI_PRINT(("    lang=%c%c%c, id=%d, name=%s",
                                   (lang_code >> 16), (lang_code >> 8), lang_code, array[entry_no].name_id, tmp_str));
                  }
               }
               #endif
            }
         }
         *array_ptr = array;
         *num_ptr = num_entries;
      }
      else
      {
         #ifdef DEBUG_PREF_NAME_LIST_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   CAN'T ALLOCATE MEMORY FOR DESCRIPTOR ARRAY ENTRY"));
         }
         #endif
      }
   }
   else
   {
      #ifdef DEBUG_PREF_NAME_LIST_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid Preferred name list desc: (%d bytes)", dlen));
      }
      #endif
   }

   FUNCTION_FINISH(ParsePreferredNameListDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads network name descriptor
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   auth_ptr   - pointer for return of the authority string
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseDefaultAuthorityDescriptor(U8BIT *dptr, SI_STRING_DESC **auth_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;

   FUNCTION_START(ParseDefaultAuthorityDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(auth_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   // this descriptor should only be received once in the NIT - if a name string already exists
   // then this must be a duplicate entry so ignore it
   if (*auth_ptr == NULL)
   {
      dptr = STB_SIReadString(dlen, dptr, auth_ptr);
#ifdef DEBUG_DEF_AUTH_DESC
      {
         char *tmp_str = "NULL";
         if (db_print == TRUE)
         {
            if (*auth_ptr != NULL)
            {
               if ((*auth_ptr)->str_ptr != NULL)
               {
                  tmp_str = (char *)(*auth_ptr)->str_ptr;
               }
            }
            STB_SI_PRINT(("   Default authority desc=\"%s\"", tmp_str));
         }
      }
      #else
      USE_UNWANTED_PARAM(db_print);
#endif
   }
   else
   {
#ifdef DEBUG_DEF_AUTH_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Default authority desc: duplicate entry ignored"));
      }
#endif
   }

   USE_UNWANTED_PARAM(dptr);
   FUNCTION_FINISH(ParseDefaultAuthorityDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads content identifier descriptor - adds entries to crid desc list
 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   num_ptr   - pointer for return of number of entries in the list
 * @param   list_ptr  - pointer for return of pointer to list
 * @param   last_entry_ptr - pointer for return of pointer to last entry in the list
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseContentIdentifierDescriptor(U8BIT *dptr, U8BIT *num_ptr,
   SI_CRID_DESC **list_ptr, SI_CRID_DESC **last_entry_ptr,
   BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U8BIT crid_type;
   U8BIT location;
   SI_CRID_DESC *crid_entry;

   FUNCTION_START(ParseContentIdentifierDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(num_ptr != NULL);
   ASSERT(list_ptr != NULL);
   ASSERT(last_entry_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 3)
   {
      while (dptr < end_ptr)
      {
         crid_type = *dptr;
         dptr++;

         /* Top 2 bits of the first byte define the 'location'. The current profile
          * only supports location='00' */
         location = (crid_type & CRID_LOCATION_MASK);
         if (location == CRID_LOCATION_0)
         {
            /* Location is valid so it can now be ignored. The rest of the byte defines the CRID type */
            crid_type &= CRID_TYPE_MASK;
            crid_type >>= CRID_TYPE_SHIFT;

            crid_entry = (SI_CRID_DESC *)STB_GetMemory(sizeof(SI_CRID_DESC));

            crid_entry->next = NULL;
            crid_entry->type = crid_type;

            dptr = STB_SIReadString(dptr[0], dptr + 1, &crid_entry->crid_str);

#ifdef DEBUG_CID_DESC
            if (db_print == TRUE)
            {
               STB_SI_PRINT(("   EIT CRID type: 0x%x, CRID=\"%s\"", crid_type, crid_entry->crid_str->str_ptr));
            }
            #else
            USE_UNWANTED_PARAM(db_print);
#endif

            if (*last_entry_ptr == NULL)
            {
               /* First entry in the list */
               *list_ptr = crid_entry;
            }
            else
            {
               /* Not the first entry */
               (*last_entry_ptr)->next = crid_entry;
            }

            *last_entry_ptr = crid_entry;
            (*num_ptr)++;
         }
         else
         {
#ifdef DEBUG_CID_DESC
            if (db_print == TRUE)
            {
               STB_SI_PRINT(("   Invalid CRID location: 0x%x", (U32BIT)location));
            }
#endif
            if (location == CRID_LOCATION_1)
            {
               dptr += 2;
            }
         }
      }
   }
   else
   {
#ifdef DEBUG_CID_DESC
      if ((db_print == TRUE) && (dlen != 0))
      {
         STB_SI_PRINT(("   Invalid content identifier desc: (%d bytes)", dlen));
      }
#endif
   }

   FUNCTION_FINISH(ParseContentIdentifierDescriptor);
   return(end_ptr);
}

/*!**************************************************************************
 * @brief   Reads the network change notify descriptor that is being defined for use
 *          in broadcasts to indicate when automatic rescans should be performed. Only
 *          one change notify entry can be present, so if one already exists then this
 *          one won't be parsed.
 * @param   dptr - pointer to length byte of descriptor
 * @param   num_change_notifies - pointer to return new number of entries in the array
 * @param   desc_ptr - address of pointer so that the parsed structure can be returned
 * @param   db_print - if TRUE debug prints the contents of the descriptor (for debug only)
 * @return  Updated value of current data byte address i.e. at end of descriptor
 ****************************************************************************/
static U8BIT* ParseNetworkChangeNotifyDescriptor(U8BIT *dptr, U16BIT *num_change_notifies,
   SI_NIT_CHANGE_NOTIFY_DESC **desc_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U8BIT loop_length;
   U8BIT i;
   SI_NIT_CHANGE_NOTIFY_DESC *desc;
   SI_NIT_CHANGE_ENTRY *entry;

   FUNCTION_START(ParseNetworkChangeNotifyDescriptor);

   #ifndef DEBUG_SI_NIT_CONTENT
   USE_UNWANTED_PARAM(db_print);
   #endif

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   /* Skip the extension descriptor tag as this has already been checked */
   dptr++;

   if (dlen >= 3)
   {
      /* There can be more than one descriptor per NIT */
      if (*num_change_notifies == 0)
      {
         *desc_ptr = (SI_NIT_CHANGE_NOTIFY_DESC *)STB_GetMemory(sizeof(SI_NIT_CHANGE_NOTIFY_DESC));
         *num_change_notifies = 1;
         desc = *desc_ptr;
      }
      else
      {
         *num_change_notifies += 1;
         *desc_ptr = (SI_NIT_CHANGE_NOTIFY_DESC *)STB_ResizeMemory(*desc_ptr,
               *num_change_notifies * sizeof(SI_NIT_CHANGE_NOTIFY_DESC));
         desc = &(*desc_ptr)[*num_change_notifies - 1];
      }

      if (desc != NULL)
      {
         desc->cell_id = (dptr[0] << 8) + dptr[1];
         dptr += 2;

         #ifdef DEBUG_SI_NIT_CONTENT
         if (db_print)
         {
            STB_SI_PRINT(("   Change notify for cell_id 0x%04x", desc->cell_id));
         }
         #endif

         loop_length = *dptr;
         dptr++;

         desc->num_changes = 0;
         desc->change_array = NULL;

         /* As the size of an entry in the array can vary depending on whether it includes
          * the invariant IDs, the array has to be built as each entry is decoded */
         for (i = 0; (loop_length >= 12) && (dptr < end_ptr); i++)
         {
            if (desc->change_array == NULL)
            {
               desc->change_array = (SI_NIT_CHANGE_ENTRY *)STB_GetMemory(sizeof(SI_NIT_CHANGE_ENTRY));
            }
            else
            {
               desc->change_array = (SI_NIT_CHANGE_ENTRY *)STB_ResizeMemory(desc->change_array,
                     (desc->num_changes + 1) * sizeof(SI_NIT_CHANGE_ENTRY));
            }

            entry = &desc->change_array[desc->num_changes];

            desc->num_changes++;

            memset(entry, 0, sizeof(SI_NIT_CHANGE_ENTRY));

            entry->change_id = *dptr;
            dptr++;

            entry->version = *dptr;
            dptr++;

            #ifdef DEBUG_SI_NIT_CONTENT
            if (db_print)
            {
               STB_SI_PRINT(("    Change id %u, version %u", entry->change_id, entry->version));
            }
            #endif

            entry->start_date = (dptr[0] << 8) | dptr[1];
            entry->start_hours = ((dptr[2] >> 4) * 10) + (dptr[2] & 0x0f);
            entry->start_mins = ((dptr[3] >> 4) * 10) + (dptr[3] & 0x0f);
            entry->start_secs = ((dptr[4] >> 4) * 10) + (dptr[4] & 0x0f);
            dptr += 5;

            entry->dur_hours = ((dptr[0] >> 4) * 10) + (dptr[0] & 0x0f);
            entry->dur_mins = ((dptr[1] >> 4) * 10) + (dptr[1] & 0x0f);
            entry->dur_secs = ((dptr[2] >> 4) * 10) + (dptr[2] & 0x0f);
            dptr += 3;

            entry->receiver_category = (*dptr >> 5);

            if ((*dptr & 0x10) != 0)
            {
               entry->invariant_ts_present = TRUE;
            }

            entry->change_type = *dptr & 0x0f;
            dptr++;

            entry->message_id = *dptr;
            dptr++;

            loop_length -= 12;

            #ifdef DEBUG_SI_NIT_CONTENT
            if (db_print)
            {
               STB_SI_PRINT(("     Starts on %u @ %02u:%02u:%02u, duration %02u:%02u:%02u",
                             entry->start_date, entry->start_hours, entry->start_mins, entry->start_secs,
                             entry->dur_hours, entry->dur_mins, entry->dur_secs));
               STB_SI_PRINT(("     Receiver type %u, change type %u", entry->receiver_category,
                             entry->change_type));
               STB_SI_PRINT(("     Message id %u", entry->message_id));
            }
            #endif

            if (entry->invariant_ts_present && (loop_length >= 4))
            {
               /* Read invariant transport stream ids */
               entry->invariant_ts_tsid = (U16BIT)(dptr[0] << 8) + dptr[1];
               entry->invariant_ts_onid = (U16BIT)(dptr[2] << 8) + dptr[3];

               dptr += 4;
               loop_length -= 4;

               #ifdef DEBUG_SI_NIT_CONTENT
               if (db_print)
               {
                  STB_SI_PRINT(("     Invariant TS ids: tsid=0x%04x, onid=0x%04x", entry->invariant_ts_tsid,
                                entry->invariant_ts_onid));
               }
               #endif
            }
         }
      }
   }

   FUNCTION_FINISH(ParseNetworkChangeNotifyDescriptor);

   return(end_ptr);
}

/*!**************************************************************************
 * @brief   Reads the network change info descriptor that contains messages to be
 *          presented to the user. More than one of these can exist in the NIT.
 * @param   dptr - pointer to length byte of descriptor
 * @param   num_messages - pointer to the count of entries already in the following array.
 *                        The value will be updated in this function.
 * @param   message_array - address of array containing info descriptors already found in the NIT.
 *                       The array may be increased in this function, so the array pointer
 *                       may be different on exit from this function.
 * @param   db_print - if TRUE debug prints the contents of the descriptor (for debug only)
 * @return  Updated value of current data byte address i.e. at end of descriptor
 ****************************************************************************/
static U8BIT* ParseMessageDescriptor(U8BIT *dptr, U16BIT *num_messages,
   SI_NIT_MESSAGE_ENTRY **message_array, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   SI_NIT_MESSAGE_ENTRY *entry;

   FUNCTION_START(ParseMessageDescriptor);

   #ifndef DEBUG_SI_NIT_CONTENT
   USE_UNWANTED_PARAM(db_print);
   #endif

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   /* Skip the extension descriptor tag as this has already been checked */
   dptr++;

   if (dlen > 5)
   {
      if (*message_array == NULL)
      {
         /* No entries in the array yet */
         *message_array = (SI_NIT_MESSAGE_ENTRY *)STB_GetMemory(sizeof(SI_NIT_MESSAGE_ENTRY));
         *num_messages = 1;
         entry = *message_array;
      }
      else
      {
         *num_messages += 1;
         *message_array = (SI_NIT_MESSAGE_ENTRY *)STB_ResizeMemory(*message_array,
               *num_messages * sizeof(SI_NIT_MESSAGE_ENTRY));
         entry = &(*message_array)[*num_messages - 1];
      }

      if (entry != NULL)
      {
         entry->message_id = *dptr;
         dptr++;

         dptr = ReadLanguageCode(dptr, &entry->lang_code);

         dptr = STB_SIReadString(dlen - 5, dptr, &entry->message);

         #ifdef DEBUG_SI_NIT_CONTENT
         if (db_print)
         {
            STB_SI_PRINT(("   Message id %u, lang \"%c%c%c\"", entry->message_id,
                          (entry->lang_code >> 16) & 0xff, (entry->lang_code >> 8) & 0xff, entry->lang_code & 0xff));
            if ((*entry->message->str_ptr >= 0x20) && (*entry->message->str_ptr <= 0x7f))
            {
               STB_SI_PRINT(("    \"%s\"", entry->message->str_ptr));
            }
            else
            {
               STB_SI_PRINT(("    message is non-ascii"));
            }
         }
         #endif
      }
   }

   USE_UNWANTED_PARAM(dptr);
   FUNCTION_FINISH(ParseMessageDescriptor);

   return(end_ptr);
}

/*!**************************************************************************
 * @brief   Reads the target region name descriptor. More than one of these can exist in an NIT
 * @param   dptr - pointer to length byte of descriptor
 * @param   desc_list - address of pointer to the first entry in the list
 * @param   db_print - if TRUE debug prints the contents of the descriptor (for debug only)
 * @return  Updated value of current data byte address i.e. at end of descriptor
 ****************************************************************************/
static U8BIT* ParseTargetRegionNameDescriptor(U8BIT *dptr, SI_NIT_TARGET_REGION_NAME_DESC **desc_list,
   BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U8BIT *eptr;
   U8BIT region_depth;
   U8BIT name_len;
   U8BIT i;
   SI_NIT_TARGET_REGION_NAME_DESC *entry;
   SI_NIT_REGION_NAME *name_entry;

   FUNCTION_START(ParseTargetRegionNameDescriptor);

   #ifndef DEBUG_TARGET_REGION_DESC
   USE_UNWANTED_PARAM(db_print);
   #endif

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   /* Skip the extension descriptor tag as this has already been checked */
   dptr++;

   if (dlen > 7)
   {
      entry = (SI_NIT_TARGET_REGION_NAME_DESC *)STB_GetMemory(sizeof(SI_NIT_TARGET_REGION_NAME_DESC));
      if (entry != NULL)
      {
         dptr = ReadLanguageCode(dptr, &entry->country_code);
         dptr = ReadLanguageCode(dptr, &entry->lang_code);

         dlen -= 7;

         entry->num_names = 0;
         entry->name_array = NULL;

         /* Calculate the number of names in the descriptor */
         for (eptr = dptr; dlen > 0; )
         {
            region_depth = (*eptr >> 6) & 0x03;
            name_len = *eptr & 0x3f;
            eptr++;
            dlen--;
            if (name_len > 0)
            {
               entry->num_names++;
            }
            eptr += name_len;
            dlen -= name_len;
            eptr++;
            dlen--;
            if (region_depth >= 2)
            {
               eptr++;
               dlen--;
               if (region_depth == 3)
               {
                  eptr += 2;
                  dlen -= 2;
               }
            }
         }

#ifdef DEBUG_TARGET_REGION_DESC
         if (db_print)
         {
            STB_SI_PRINT(("   Target region name: country=%c%c%c, lang=%c%c%c, num_names=%u",
                          entry->country_code >> 16, entry->country_code >> 8, entry->country_code & 0xff,
                          entry->lang_code >> 16, entry->lang_code >> 8, entry->lang_code & 0xff,
                          entry->num_names));
         }
#endif

         if (entry->num_names > 0)
         {
            entry->name_array = (SI_NIT_REGION_NAME *)STB_GetMemory(entry->num_names * sizeof(SI_NIT_REGION_NAME));
            if (entry->name_array != NULL)
            {
               for (i = 0; i < entry->num_names; i++)
               {
                  name_entry = &entry->name_array[i];
                  name_entry->region_depth = (*dptr >> 6) & 0x03;

                  dptr = STB_SIReadString((dptr[0] & 0x3f), dptr + 1, &name_entry->region_name);

                  name_entry->primary_region_code = *dptr;
                  dptr++;
                  if (name_entry->region_depth >= 2)
                  {
                     name_entry->secondary_region_code = *dptr;
                     dptr++;
                     if (name_entry->region_depth == 3)
                     {
                        name_entry->tertiary_region_code = (dptr[0] << 8) | dptr[1];
                        dptr += 2;
                     }
                  }
#ifdef DEBUG_TARGET_REGION_DESC
                  if (db_print)
                  {
                     if (name_entry->region_depth == 3)
                     {
                        STB_SI_PRINT(("     Tertiary region name: %u:%u:%u = \"%s\"",
                                      name_entry->primary_region_code, name_entry->secondary_region_code,
                                      name_entry->tertiary_region_code, name_entry->region_name->str_ptr));
                     }
                     else if (name_entry->region_depth == 2)
                     {
                        STB_SI_PRINT(("     Secondary region name: %u:%u = \"%s\"",
                                      name_entry->primary_region_code, name_entry->secondary_region_code,
                                      name_entry->region_name->str_ptr));
                     }
                     else
                     {
                        STB_SI_PRINT(("     Primary region name: %u = \"%s\"",
                                      name_entry->primary_region_code, name_entry->region_name->str_ptr));
                     }
                  }
#endif
               }
            }
         }

         /* Add the entry to the start of the existing list */
         entry->next = *desc_list;
         *desc_list = entry;
      }
   }
#ifdef DEBUG_TARGET_REGION_DESC
   else if (db_print)
   {
      STB_SI_PRINT(("   Target region name desc, skipped, invalid size: %u", dlen));
   }
#endif

   FUNCTION_FINISH(ParseTargetRegionNameDescriptor);

   return(end_ptr);
}

/*!**************************************************************************
 * @brief   Reads the target region descriptor. More than one of these can exist in an NIT
 * @param   dptr - pointer to length byte of descriptor
 * @param   desc_list - address of pointer to the first entry in the list
 * @param   db_print - if TRUE debug prints the contents of the descriptor (for debug only)
 * @return  Updated value of current data byte address i.e. at end of descriptor
 ****************************************************************************/
static U8BIT* ParseTargetRegionDescriptor(U8BIT *dptr, SI_TARGET_REGION_DESC **desc_list,
   BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U32BIT country_code;
   U8BIT region_depth;
   SI_TARGET_REGION_DESC *cptr;
   SI_TARGET_REGION *rptr;
   BOOLEAN read_country;

   FUNCTION_START(ParseTargetRegionDescriptor);

   #ifndef DEBUG_TARGET_REGION_DESC
   USE_UNWANTED_PARAM(db_print);
   #endif

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   /* Skip the extension descriptor tag as this has already been checked */
   dptr++;
   dlen--;

   read_country = TRUE;

   while (dlen >= 3)
   {
      cptr = NULL;

      if (read_country)
      {
         /* Descriptor starts with a country code */
         dptr = ReadLanguageCode(dptr, &country_code);
         dlen -= 3;

#ifdef DEBUG_TARGET_REGION_DESC
         if (db_print)
         {
            STB_SI_PRINT(("   Target region: country=%c%c%c",
                          country_code >> 16, country_code >> 8, country_code & 0xff));
         }
#endif

         /* Check whether there's an entry in the list for this country already */
         for (cptr = *desc_list; (cptr != NULL) && (cptr->country_code != country_code); )
         {
            cptr = cptr->next;
         }

         if (cptr == NULL)
         {
            /* This is a new country code */
            cptr = (SI_TARGET_REGION_DESC *)STB_GetMemory(sizeof(SI_TARGET_REGION_DESC));
            if (cptr != NULL)
            {
               cptr->country_code = country_code;
               cptr->region_list = NULL;

               /* Add the new entry to the start of the list */
               cptr->next = *desc_list;
               *desc_list = cptr;
            }
         }

         read_country = FALSE;
      }

      if (cptr != NULL)
      {
         while (!read_country && (dlen > 0))
         {
            if ((*dptr & 0x04) != 0)
            {
               read_country = TRUE;
            }
            else
            {
               /* Read the region codes */
               region_depth = *dptr & 0x03;
               dptr++;
               dlen--;

               if (region_depth > 0)
               {
                  /* Create a new entry in the region list */
                  rptr = (SI_TARGET_REGION *)STB_GetMemory(sizeof(SI_TARGET_REGION));
                  if (rptr != NULL)
                  {
                     rptr->region_depth = region_depth;
                     rptr->primary_region_code = *dptr;
                     dptr++;
                     dlen--;

                     if (region_depth > 1)
                     {
                        rptr->secondary_region_code = *dptr;
                        dptr++;
                        dlen--;

                        if (region_depth > 2)
                        {
                           rptr->tertiary_region_code = (dptr[0] << 8) | dptr[1];
                           dptr += 2;
                           dlen -= 2;
                        }
                     }

#ifdef DEBUG_TARGET_REGION_DESC
                     if (db_print)
                     {
                        if (rptr->region_depth == 3)
                        {
                           STB_SI_PRINT(("     Tertiary region: %u:%u:%u",
                                         rptr->primary_region_code, rptr->secondary_region_code,
                                         rptr->tertiary_region_code));
                        }
                        else if (rptr->region_depth == 2)
                        {
                           STB_SI_PRINT(("     Secondary region: %u:%u",
                                         rptr->primary_region_code, rptr->secondary_region_code));
                        }
                        else
                        {
                           STB_SI_PRINT(("     Primary region: %u",
                                         rptr->primary_region_code));
                        }
                     }
#endif

                     /* Add the region to the start of the country's list */
                     rptr->next = cptr->region_list;
                     cptr->region_list = rptr;
                  }
               }
            }
         }
      }
   }

   FUNCTION_FINISH(ParseTargetRegionDescriptor);

   return(end_ptr);
}

/*!**************************************************************************
 * @brief   Reads the service availability descriptor.
 * @param   dptr - pointer to length byte of descriptor
 * @param   serv_avail_list - address of pointer to the first descriptor in
 *          the list
 * @param   db_print - if TRUE debug prints the contents of the descriptor
 *          (for debug only)
 * @return  Updated value of current data byte address i.e. at end of
 *          descriptor
 ****************************************************************************/
static U8BIT* ParseServiceAvailabilityDescriptor(U8BIT *dptr, SI_SERV_AVAIL_DESC **serv_avail_list, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U16BIT num_of_ids;
   U16BIT i;
   SI_SERV_AVAIL_DESC *desc;

   FUNCTION_START(ParseServiceAvailabilityDescriptor);

   #ifndef DEBUG_SI_SDT_CONTENT
   USE_UNWANTED_PARAM(db_print);
   #endif

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

#ifdef DEBUG_SERV_AVAIL_DESC
   STB_SI_PRINT(("ParseServiceAvailabilityDescriptor: descriptor length = %d", dlen));
#endif
   if (dlen >= 3)
   {
      num_of_ids = (dlen - 1) / 2;

      desc = (SI_SERV_AVAIL_DESC *)STB_GetMemory(sizeof(SI_SERV_AVAIL_DESC));
      if (desc != NULL)
      {
         desc->next = *serv_avail_list;
         *serv_avail_list = desc;

#ifdef DEBUG_SERV_AVAIL_DESC
         STB_SI_PRINT(("ParseServiceAvailabilityDescriptor: availability flag = %s", ((*dptr & 0x80) == 0) ? "FALSE" : "TRUE"));
#endif
         if ((*dptr & 0x80) == 0)
         {
            desc->availability_flag = FALSE;
         }
         else
         {
            desc->availability_flag = TRUE;
         }

         dptr++;

         desc->cell_ids = (U16BIT *)STB_GetMemory(num_of_ids * 2);
         if (desc->cell_ids != NULL)
         {
            desc->num_of_cell_ids = num_of_ids;
            for (i = 0; i < num_of_ids; i++)
            {
               desc->cell_ids[i] = (dptr[0] << 8) + dptr[1];
#ifdef DEBUG_SERV_AVAIL_DESC
               STB_SI_PRINT(("ParseServiceAvailabilityDescriptor: id[%d]=0x%x", i, desc->cell_ids[i]));
#endif
               dptr += 2;
            }
         }
      }
   }

#ifdef DEBUG_SERV_AVAIL_DESC
   STB_SI_PRINT(("ParseServiceAvailabilityDescriptor: end"));
#endif

   FUNCTION_FINISH(ParseServiceAvailabilityDescriptor);

   return(end_ptr);
}

/*!**************************************************************************
 * @brief   Reads the Supplementary Audio descriptor.
 * @param   dptr - pointer to length byte of descriptor
 * @param   db_print - if TRUE debug prints the contents of the descriptor (for debug only)
 * @return  Updated value of current data byte address i.e. at end of descriptor
 ****************************************************************************/
static U8BIT* ParseSupplementaryAudioDescriptor(U8BIT *dptr, SI_AD_DESC **audio_desc, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U8BIT lang_present;

   FUNCTION_START(ParseSupplementaryAudioDescriptor);

   ASSERT(dptr != NULL);

#ifndef DEBUG_SUPP_AUDIO_DESC
   USE_UNWANTED_PARAM(db_print);
#endif

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   /* Skip the extension desc tag */
   dptr++;

   *audio_desc = (SI_AD_DESC *)STB_GetMemory(sizeof(SI_AD_DESC));
   if (*audio_desc != NULL)
   {
      (*audio_desc)->mix_type = (*dptr >> 7) & 0x1;
      (*audio_desc)->edit_class = (*dptr >> 2) & 0x1f;
      /* (*dptr & 0x2) is reserved */
      lang_present = *dptr & 0x1;

      if ((dlen >= 2) && (lang_present != 0))
      {
         dptr = ReadLanguageCode(&dptr[1], &((*audio_desc)->lang_code));
      }
      #ifdef DEBUG_SUPP_AUDIO_DESC
      else if (db_print)
      {
         STB_SI_PRINT(("   Supplementary audio: invalid desc len %u", dlen));
      }
      #endif

#ifdef DEBUG_SUPP_AUDIO_DESC
      if (db_print)
      {
         STB_SI_PRINT(("    Supplementary audio: mix %s, class %u lang=%c%c%c",
                       ((*audio_desc)->mix_type == 1 ? "independent" : "supplementary"),
                       (*audio_desc)->edit_class,
                       (char)((*audio_desc)->lang_code >> 16),
                       (char)((*audio_desc)->lang_code >> 8),
                       (char)(*audio_desc)->lang_code));
         STB_SI_PRINT(("    data(%d):%*.*s", dlen - 5, dlen - 5, dlen - 5, dptr));
      }
#endif
   }
   else
   {
#ifdef DEBUG_SUPP_AUDIO_DESC
      if (db_print)
      {
         STB_SI_PRINT(("   CAN'T ALLOCATE MEMORY FOR DESCRIPTOR ARRAY ENTRY"));
      }
#endif
   }

   USE_UNWANTED_PARAM(dptr);
   FUNCTION_FINISH(ParseSupplementaryAudioDescriptor);
   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads the freesat types available of this stream
 *
 * @param   dptr        - pointer to length byte of descriptor
 * @param   num_entries - pointer to number of entries in array
 * @param   desc_array  - pointer to descriptor array
 * @param   alt_descriptor - TRUE if this is the alternative tunnelled data descriptor
 * @param   db_print - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated dptr i.e. pointing to byte following descriptor
 *
 */
static U8BIT* ParseFreesatTunnelledDataDescriptor(U8BIT *dptr, U8BIT *num_entries,
   SI_FREESAT_TUNNELLED_DATA_DESC **desc_array, BOOLEAN alt_descriptor, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U8BIT i, num;

   FUNCTION_START(ParseFreesatTunnelledDataDescriptor);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   /* Work out the number of entries defined in the descriptor */
   if (alt_descriptor)
   {
      num = dlen;
   }
   else
   {
      num = dlen / 2;
   }

   if (num > 0)
   {
#ifdef DEBUG_FREESAT_TUNNELLED_DATA_DESC
      if (db_print)
      {
         STB_SI_PRINT(("   Freesat tunnelled data desc: alternative=%u, %u entries", alt_descriptor, num));
      }
#else
      USE_UNWANTED_PARAM(db_print);
#endif
      if (*desc_array != NULL)
      {
         *desc_array = (SI_FREESAT_TUNNELLED_DATA_DESC *)STB_ResizeMemory(*desc_array,
               (num + *num_entries) * sizeof(SI_FREESAT_TUNNELLED_DATA_DESC));
      }
      else
      {
         *desc_array = (SI_FREESAT_TUNNELLED_DATA_DESC *)STB_GetMemory(num * sizeof(SI_FREESAT_TUNNELLED_DATA_DESC));
      }

      if (*desc_array != NULL)
      {
         for (i = 0; i < num; i++)
         {
            (*desc_array)[i + *num_entries].data_type = *dptr;
            dptr++;

            if (!alt_descriptor)
            {
               /* Read the component tag */
               (*desc_array)[i + *num_entries].component_tag = *dptr;
               dptr++;
            }
#ifdef DEBUG_FREESAT_TUNNELLED_DATA_DESC
            if (db_print)
            {
               if (alt_descriptor)
               {
                  STB_SI_PRINT(("     type=%u", (*desc_array)[i + *num_entries].data_type));
               }
               else
               {
                  STB_SI_PRINT(("     type=%u, tag=%u", (*desc_array)[i + *num_entries].data_type,
                                (*desc_array)[i + *num_entries].component_tag));
               }
            }
#endif
         }

         *num_entries += num;
      }
   }

   FUNCTION_FINISH(ParseFreesatTunnelledDataDescriptor);

   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads a region name decripter
 *
 * @param   dptr     - pointer to length byte of descriptor
 * @param   region_name - name of the current required region
 * @param   db_print - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated dptr i.e. pointing to byte following descriptor
 *
 */
static U8BIT* ParseRegionNameDescriptor(U8BIT *dptr, SI_BAT_FREESAT_REGION **region_list, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   SI_BAT_FREESAT_REGION *region;

   FUNCTION_START(ParseRegionNameDescriptor);

   USE_UNWANTED_PARAM(db_print);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   while (dptr < end_ptr)
   {
      if ((region = (SI_BAT_FREESAT_REGION *)STB_GetMemory(sizeof(SI_BAT_FREESAT_REGION))) != NULL)
      {
         region->region_id = (dptr[0] << 8) | (dptr[1]);
         dptr += 2;
         dptr = ReadLanguageCode(dptr, &region->lang_code);
         dptr = STB_SIReadString(dptr[0], dptr + 1, &region->region_name);

         /* Add the new region to the start of the list */
         region->next = *region_list;
         *region_list = region;
      }
   }

   FUNCTION_FINISH(ParseRegionNameDescriptor);

   return(end_ptr);
}

/**
 *

 *

 *
 * @param   dptr      - pointer to length byte of descriptor
 * @param   desc_list - pointer to the previous stored descriptor
 * @param   db_print  - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated value of current data byte address i.e. at end of descriptor.
 *
 */
static U8BIT* ParseIActiveStorageDescriptor(U8BIT *dptr,
   SI_BAT_FREESAT_IACTIVE_STORAGE_DESC **desc_list, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   SI_BAT_FREESAT_IACTIVE_STORAGE_DESC *new_entry;

   FUNCTION_START(ParseIActiveStorageDescriptor);

   ASSERT(dptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen > 0)
   {
      #ifdef DEBUG_PRIV_DATA_SPEC_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Interactive storage descriptor : (len = %d)", dlen));
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif

      new_entry = STB_GetMemory(sizeof(SI_BAT_FREESAT_IACTIVE_STORAGE_DESC));
      if (new_entry != NULL)
      {
         memset(new_entry, 0, sizeof(SI_BAT_FREESAT_IACTIVE_STORAGE_DESC));
         new_entry->data = STB_GetMemory(dlen);
         if (new_entry->data != NULL)
         {
            memcpy(new_entry->data, dptr, dlen);
            new_entry->next_desc = *desc_list;
            new_entry->length = dlen;
            *desc_list = new_entry;
         }
         else
         {
            STB_FreeMemory(new_entry);
         }
      }
   }
   else
   {
      #ifdef DEBUG_PRIV_DATA_SPEC_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid interactive storage descriptor: (%d bytes)", dlen));
      }
      #endif
   }

   FUNCTION_FINISH(ParseIActiveStorageDescriptor);
   return(end_ptr);
}

static U8BIT* ParseFreesatInfoLocationDescriptor(U8BIT *dptr,
   SI_BAT_FREESAT_INFO_LOCATION **location_list, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U8BIT loop_len;
   SI_BAT_FREESAT_INFO_LOCATION *new_entry;
   SI_BAT_FREESAT_INFO_LOCATION **last_pointer;

   FUNCTION_START(ParseFreesatInfoLocationDescriptor);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   /* find the last pointer in the list (i.e. pointing to NULL) */
   last_pointer = location_list;
   while (*last_pointer)
      last_pointer = &((*last_pointer)->next);

   if (dlen > 2)
   {
      #ifdef DEBUG_FREESAT_INFO_LOCATION_DESC
      if (db_print)
      {
         STB_SI_PRINT(("   %s:", __FUNCTION__));
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif

      while (dptr < end_ptr)
      {
         /* Home loop length is profiled to be 0 bytes long, but skip it anyway */
         dptr += *dptr + 1;

         /* Extract the tunnelled values */
         loop_len = *dptr;
         dptr++;
         #ifdef DEBUG_FREESAT_INFO_LOCATION_DESC
         if (db_print)
         {
            STB_SI_PRINT(("    tunnelled loop length = %u", loop_len));
         }
         #endif

         while ((loop_len >= 6) && (dptr < end_ptr))
         {
            new_entry = (SI_BAT_FREESAT_INFO_LOCATION *)STB_GetMemory(sizeof(SI_BAT_FREESAT_INFO_LOCATION));
            if (new_entry != NULL)
            {
               memset(new_entry, 0, sizeof(*new_entry));

               new_entry->transport_id = (dptr[0] << 8) | dptr[1];
               new_entry->orig_net_id = (dptr[2] << 8) | dptr[3];
               new_entry->service_id = (dptr[4] << 8) | dptr[5];

               dptr += 6;
               loop_len -= 6;

               #ifdef DEBUG_FREESAT_INFO_LOCATION_DESC
               if (db_print)
               {
                  STB_SI_PRINT(("     service = 0x%04x/0x%04x/0x%04x", new_entry->orig_net_id,
                                new_entry->transport_id, new_entry->service_id));
               }
               #endif

               /* Add the new location to the end of the list */
               new_entry->next = NULL;
               *last_pointer = new_entry;
               last_pointer = &(new_entry->next);
            }
            else
            {
               #ifdef DEBUG_FREESAT_INFO_LOCATION_DESC
               if (db_print)
               {
                  STB_SI_PRINT(("   Failed to allocate memory"));
               }
               #endif
               dptr = end_ptr;
            }
         }
      }
   }

   FUNCTION_FINISH(ParseFreesatInfoLocationDescriptor);

   return(end_ptr);
}

/**
 *

 *
 * @brief   Reads a regionalised lcn decripter
 *
 * @param   dptr     - pointer to length byte of descriptor
 * @param   region_name - name of the current required region
 * @param   db_print - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated dptr i.e. pointing to byte following descriptor
 *
 */
static U8BIT* ParseRegionLcnDescriptor(U8BIT *dptr, SI_BAT_FREESAT_REGION_LCN_ENTRY **lcn_array, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U8BIT *loop_ptr;
   U8BIT *end_loop;
   SI_BAT_FREESAT_REGION_LCN_ENTRY *region_entry;
   SI_BAT_FREESAT_REGION_LCN_ENTRY *last_region;
   SI_FREESAT_LCN *lcn_entry;
   SI_FREESAT_LCN *last_lcn;

   FUNCTION_START(ParseRegionLcnDescriptor);

#ifndef DEBUG_FREESAT_LCN_DESC
   USE_UNWANTED_PARAM(db_print);
#endif

   last_region = *lcn_array;
   while (last_region != NULL)
   {
      if (last_region->next != NULL)
      {
         last_region = last_region->next;
      }
      else
      {
         /* This is the last item in the list */
         break;
      }
   }

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   while (dptr < end_ptr)
   {
      region_entry = STB_GetMemory(sizeof(SI_BAT_FREESAT_REGION_LCN_ENTRY));
      if (region_entry != NULL)
      {
         memset(region_entry, 0, sizeof(*region_entry));

         region_entry->service_id = (dptr[0] << 8) | dptr[1];
         dptr += 2;

         region_entry->freesat_id = ((dptr[0] & 0x7F) << 8) | dptr[1];
         dptr += 2;

#ifdef DEBUG_FREESAT_LCN_DESC
         if (db_print)
         {
            STB_SI_PRINT(("   ParseRegionLcnDescriptor: serv_id = 0x%04x, freesat_id = 0x%04x",
                          region_entry->service_id, region_entry->freesat_id));
         }
#endif

         loop_ptr = dptr + 1;
         end_loop = loop_ptr + dptr[0];

         last_lcn = NULL;

         while (loop_ptr < end_loop)
         {
            if (end_loop - loop_ptr >= 4)
            {
               lcn_entry = (SI_FREESAT_LCN *)STB_GetMemory(sizeof(SI_FREESAT_LCN));
               if (lcn_entry != NULL)
               {
                  memset(lcn_entry, 0, sizeof(*lcn_entry));

                  lcn_entry->numeric_selection = ((loop_ptr[0] & 0x80) >> 7);
                  lcn_entry->visible_flag = ((loop_ptr[0] & 0x40) >> 6);
                  lcn_entry->user_cust = ((loop_ptr[0] & 0x20) >> 5);
                  lcn_entry->lcn = (((loop_ptr[0] & 0x0F) << 8) | (loop_ptr[1]));
                  lcn_entry->region_id = (loop_ptr[2] << 8) | (loop_ptr[3]);
                  loop_ptr += 4;

#ifdef DEBUG_FREESAT_LCN_DESC
                  STB_SI_PRINT(("      numeric_selection=%d, visible=%d, user_cust=%d, lcn=%d, region_id=0x%04x",
                                lcn_entry->numeric_selection, lcn_entry->visible_flag, lcn_entry->user_cust,
                                lcn_entry->lcn, lcn_entry->region_id));
#endif

                  if (last_lcn == NULL)
                  {
                     /* First item in the list */
                     region_entry->freesat_lcn_list = lcn_entry;
                  }
                  else
                  {
                     /* Add item to the end of the list */
                     last_lcn->next = lcn_entry;
                  }

                  last_lcn = lcn_entry;
               }
               else
               {
                  loop_ptr = end_loop;
               }
            }
            else
            {
               loop_ptr = end_loop;
            }
         }

         dptr = loop_ptr;

         if (last_region == NULL)
         {
            /* First item in the list */
            *lcn_array = region_entry;
         }
         else
         {
            /* Add the item to the end of the list */
            last_region->next = region_entry;
         }

         last_region = region_entry;
      }
      else
      {
         dptr = end_ptr;
      }
   }

   FUNCTION_FINISH(ParseRegionLcnDescriptor);

   return(end_ptr);
}

static U8BIT* ParseFreesatShortServiceNameDescriptor(U8BIT *dptr, U16BIT *num_ptr,
   SI_MULTILING_SHORT_NAME_DESC **array_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U16BIT num_entries;
   SI_MULTILING_SHORT_NAME_DESC *array;
   U16BIT i;
   U8BIT *tmp_ptr;
   U8BIT name_len;

   FUNCTION_START(ParseFreesatShortServiceNameDescriptor);

   ASSERT(dptr != NULL);
   ASSERT(num_ptr != NULL);
   ASSERT(array_ptr != NULL);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 4)
   {
      // work out number of entries
      num_entries = 0;
      tmp_ptr = dptr;
      while (tmp_ptr < end_ptr)
      {
         name_len = tmp_ptr[3];     // read name length
         tmp_ptr += (name_len + 4); // skip language code, name length and name
         num_entries++;
      }

      #ifdef DEBUG_MULTILING_SHORT_NAME_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Multilingual short name desc: %d entries", num_entries));
      }
      #else
      USE_UNWANTED_PARAM(db_print);
      #endif

      // check if there are already entries in the array (i.e. already received a descriptor)
      // if so add to the existing array, otherwise create new array
      if (*array_ptr == NULL)
      {
         // no entries already - create new array
         array = (SI_MULTILING_SHORT_NAME_DESC *)STB_GetMemory(num_entries * sizeof(SI_MULTILING_SHORT_NAME_DESC));
      }
      else
      {
         // already got entries - make array bigger
         num_entries += *num_ptr;
         array = (SI_MULTILING_SHORT_NAME_DESC *)STB_ResizeMemory(*array_ptr,
               num_entries * sizeof(SI_MULTILING_SHORT_NAME_DESC));
      }

      // add new entries to array
      if (array != NULL)
      {
         for (i = *num_ptr; i < num_entries; i++)
         {
            dptr = ReadLanguageCode(dptr, &array[i].lang_code);
            dptr = STB_SIReadString(dptr[0], dptr + 1, &array[i].name_str);

            #ifdef DEBUG_MULTILING_SHORT_NAME_DESC
            {
               char *tmp_name_str = "NULL";

               if (db_print == TRUE)
               {
                  if (array[i].name_str != NULL)
                  {
                     if (array[i].name_str->str_ptr != NULL)
                     {
                        tmp_name_str = (char *)array[i].name_str->str_ptr;
                     }
                  }

                  STB_SI_PRINT(("    lang=%c%c%c, name=%s",
                                (array[i].lang_code >> 16), (array[i].lang_code >> 8), array[i].lang_code,
                                tmp_name_str));
               }
            }
            #endif
         }
         *array_ptr = array;
         *num_ptr = num_entries;
      }
      else
      {
         #ifdef DEBUG_MULTILING_SHORT_NAME_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   CAN'T ALLOCATE MEMORY FOR DESCRIPTOR ARRAY ENTRY"));
         }
         #endif
      }
   }
   else
   {
      #ifdef DEBUG_MULTILING_SHORT_NAME_DESC
      if (db_print == TRUE)
      {
         STB_SI_PRINT(("   Invalid multilingual short name desc: (%d bytes)", dlen));
      }
      #endif
   }

   FUNCTION_FINISH(ParseFreesatShortServiceNameDescriptor);
   return(end_ptr);
}

/*!**************************************************************************
 * @brief   Reads the Freesat Service Group Name Descripter
 * @param   data_ptr - pointer to length byte of descriptor
 * @param   name_list - pointer to structure to parse data into
 * @param   db_print - if TRUE debug prints the contents of the descriptor (for debug only)
 * @return  Updated value of current data byte address i.e. at end of descriptor
 ****************************************************************************/
static U8BIT* ParseFreesatServiceGroupNameDescriptor(U8BIT *dptr,
   SI_BAT_FREESAT_GROUP_NAME_ENTRY **name_list, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U8BIT *end_name_loop_ptr;
   SI_BAT_FREESAT_GROUP_NAME_ENTRY *new_entry;
   U8BIT *temp_dptr;

   FUNCTION_START(ParseFreesatServiceGroupNameDescriptor);

#ifndef DEBUG_FREESAT_GROUP_NAME_DESC
   USE_UNWANTED_PARAM(db_print);
#endif

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   while (dptr < end_ptr)
   {
      new_entry = STB_GetMemory(sizeof(SI_BAT_FREESAT_GROUP_NAME_ENTRY));
      if (new_entry != NULL)
      {
         memset(new_entry, 0, sizeof(*new_entry));

         new_entry->group_type = (dptr[0] & 0xF8) >> 3;
         new_entry->group_id = (((dptr[0] & 0x07) << 8) | dptr[1]);
         dptr += 2;

#ifdef DEBUG_FREESAT_GROUP_NAME_DESC
         if (db_print)
         {
            STB_SI_PRINT(("   Freesat service group name: group type=%d, group ID=%u",
                          new_entry->group_type, new_entry->group_id));
         }
#endif
         dlen = dptr[0];
         dptr++;
         end_name_loop_ptr = dptr + dlen;

         /* Work out how many names are defined */
         new_entry->num_group_names = 0;
         temp_dptr = dptr;

         while (temp_dptr < end_name_loop_ptr)
         {
            temp_dptr += 3;          /* Language code */
            temp_dptr += *temp_dptr;  /* Length of name */
            temp_dptr++;
            new_entry->num_group_names++;
         }

         /* Allocate array for name strings */
         new_entry->string_array = STB_GetMemory(new_entry->num_group_names *
               sizeof(SI_MULTILANG_GROUP_NAME_DESC));
         if (new_entry->string_array != NULL)
         {
            new_entry->num_group_names = 0;

            while (dptr < end_name_loop_ptr)
            {
               dptr = ReadLanguageCode(dptr, &new_entry->string_array[new_entry->num_group_names].lang_code);
               dptr = STB_SIReadString(*dptr, dptr + 1,
                     &new_entry->string_array[new_entry->num_group_names].name_str);

#ifdef DEBUG_FREESAT_GROUP_NAME_DESC
               if (db_print)
               {
                  STB_SI_PRINT(("     lang_code = %c%c%c, name = \"%s\"",
                                new_entry->string_array[new_entry->num_group_names].lang_code >> 16,
                                new_entry->string_array[new_entry->num_group_names].lang_code >> 8,
                                new_entry->string_array[new_entry->num_group_names].lang_code & 0xff,
                                new_entry->string_array[new_entry->num_group_names].name_str->str_ptr));
               }
#endif
               new_entry->num_group_names++;
            }

            /* Add group name entry to start of the list */
            new_entry->next_group = *name_list;
            *name_list = new_entry;
         }
         else
         {
            new_entry->num_group_names = 0;
            dptr = end_ptr;
         }
      }
      else
      {
         dptr = end_ptr;
      }
   }

   FUNCTION_FINISH(ParseFreesatServiceGroupNameDescriptor);

   return(end_ptr);
}

/*!**************************************************************************
 * @brief   Reads the Freesat Interactive Restriction Descripter
 * @param   data_ptr - pointer to length byte of descriptor
 * @param   serv_array - pointer to structure to parse data into
 * @param   num_entries - number of name entries
 * @param   db_print - if TRUE debug prints the contents of the descriptor (for debug only)
 * @return  Updated value of current data byte address i.e. at end of descriptor
 ****************************************************************************/
static U8BIT* ParseFreesatInteractiveRestrictionDescriptor(U8BIT *dptr, U16BIT **int_res_array,
   U8BIT *num_entries, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U8BIT count;

   FUNCTION_START(ParseFreesatInteractiveRestrictionDescriptor);

   USE_UNWANTED_PARAM(db_print);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;
   count = 0;

   *int_res_array = STB_GetMemory(((dlen / 2) * sizeof(U16BIT)));
   if (*int_res_array != NULL)
   {
      while (dptr < end_ptr)
      {
         (*int_res_array)[count] = ((dptr[0] << 8) | dptr[1]);
         count++;
         dptr += 2;
      }
   }

   *num_entries = count;

   FUNCTION_FINISH(ParseFreesatInteractiveRestrictionDescriptor);

   return(end_ptr);
}

/*!**************************************************************************
 * @brief   Reads the Freesat Service Group Descripter
 * @param   data_ptr - pointer to length byte of descriptor
 * @param   serv_array - pointer to structure to parse data into
 * @param   num_entries - number of name entries
 * @param   db_print - if TRUE debug prints the contents of the descriptor (for debug only)
 * @return  Updated value of current data byte address i.e. at end of descriptor
 ****************************************************************************/
static U8BIT* ParseFreesatServiceGroupDescriptor(U8BIT *dptr, SI_BAT_FREESAT_SERV_GROUP_ENTRY **serv_array,
   U16BIT *num_entries, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   SI_BAT_FREESAT_SERV_GROUP_ENTRY *entry_ptr;
   U16BIT group_id;
   U16BIT index;
   U8BIT group_size;

   FUNCTION_START(ParseFreesatServiceGroupDescriptor);

   USE_UNWANTED_PARAM(db_print);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   while (dptr < end_ptr)
   {
      group_id = ((dptr[0] & 0x07) << 8) | dptr[1];

      /* See if there's an entry for this group already */
      for (entry_ptr = *serv_array; entry_ptr != NULL; entry_ptr = entry_ptr->next_group)
      {
         if (entry_ptr->group_id == group_id)
         {
            /* Already have an entry for this group */
            break;
         }
      }

      if (entry_ptr == NULL)
      {
         /* No existing entry for this group, so create one */
         entry_ptr = (SI_BAT_FREESAT_SERV_GROUP_ENTRY *)STB_GetMemory(sizeof(SI_BAT_FREESAT_SERV_GROUP_ENTRY));
         if (entry_ptr != NULL)
         {
            memset(entry_ptr, 0, sizeof(*entry_ptr));
            entry_ptr->next_group = *serv_array;
            *serv_array = entry_ptr;
            (*num_entries)++;
         }
      }

      if (entry_ptr != NULL)
      {
         entry_ptr->non_destructive_flag = ((dptr[0] & 0x80) >> 7);
         entry_ptr->return_channel_access_flag = ((dptr[0] & 0x40) >> 6);
         entry_ptr->g2_extension_flag = ((dptr[0] & 0x20) >> 5);
         entry_ptr->group_id = group_id;

         group_size = dptr[2] / 2;
         dptr += 3;

         if (group_size != 0)
         {
            /* Allocate space for the Freesat IDs of the services in the group */
            if (entry_ptr->freesat_id == NULL)
            {
               /* */
               entry_ptr->freesat_id = (U16BIT *)STB_GetMemory(group_size * sizeof(U16BIT));
               index = 0;
            }
            else
            {
               /* Increase the number of IDs */
               entry_ptr->freesat_id = (U16BIT *)STB_ResizeMemory(entry_ptr->freesat_id,
                     (entry_ptr->num_services + group_size) * sizeof(U16BIT));
               index = entry_ptr->num_services;
            }

            if (entry_ptr->freesat_id != NULL)
            {
               /* Copy the new IDs into the array */
               entry_ptr->num_services += group_size;

               for (; group_size > 0; group_size--, index++)
               {
                  entry_ptr->freesat_id[index] = ((dptr[0] & 0x7F) << 8) | dptr[1];
                  dptr += 2;
               }
            }
         }

         if (entry_ptr->g2_extension_flag != 0)
         {
            entry_ptr->g2_flags = *dptr;
            dptr++;
         }
      }
      else
      {
         dptr = end_ptr;
      }
   }

   FUNCTION_FINISH(ParseFreesatServiceGroupDescriptor);

   return(end_ptr);
}

#if 0
/**
 *

 *
 * @brief   Reads the service move descriptor broadcast in the PMT
 *
 * @param   dptr     - pointer to length byte of descriptor
 * @param   desc_ptr - pointer for returned descriptor
 * @param   db_print - if TRUE debug prints the contents of the descriptor (for debug only)
 *
 * @return   Updated dptr i.e. pointing to byte following descriptor
 *
 */
static U8BIT* ParseServiceMoveDescriptor(U8BIT *dptr, SI_SERVICE_MOVE_DESC **desc_ptr, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;

   FUNCTION_START(ParseServiceMoveDescriptor);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   /* There should be only one service move descriptor in a PMT
    * so skip this one if it already exists */
   if ((*desc_ptr == NULL) && (dlen == 6))
   {
      *desc_ptr = (SI_SERVICE_MOVE_DESC *)STB_GetMemory(sizeof(SI_SERVICE_MOVE_DESC));
      if (*desc_ptr != NULL)
      {
         (*desc_ptr)->onet_id = (dptr[0] << 8) | dptr[1];
         (*desc_ptr)->ts_id = (dptr[2] << 8) | dptr[3];
         (*desc_ptr)->service_id = (dptr[4] << 8) | dptr[5];

#ifdef DEBUG_SERVICE_MOVE_DESC
         if (db_print == TRUE)
         {
            STB_SI_PRINT(("   service move desc: onet=0x%04x, ts=0x%04x, service=0x%04x",
                          (*desc_ptr)->onet_id, (*desc_ptr)->ts_id, (*desc_ptr)->service_id));
         }
#else
         USE_UNWANTED_PARAM(db_print);
#endif
      }
   }

   FUNCTION_FINISH(ParseServiceMoveDescriptor);

   return(end_ptr);
}

#endif

/*!**************************************************************************
 * @brief   Reads the RCT link_info structure
 * @param   dptr - pointer to length byte of descriptor
 * @param   link_info - pointer to structure to parse data into
 * @param   db_print - if TRUE debug prints the contents of the descriptor (for debug only)
 * @return  Updated value of current data byte address i.e. at end of descriptor
 ****************************************************************************/
static U8BIT* ParseRctLinkInfo(U8BIT *dptr, SI_RCT_LINK_INFO *link_info, BOOLEAN db_print)
{
   U16BIT dlen;
   U8BIT *end_ptr;
   U8BIT text_len, i;
   U16BIT dloop_len;
   U8BIT *dloop_end;
   U8BIT dtag;

   FUNCTION_START(ParseRctLinkInfo);

#ifndef DEBUG_RCT_LINK_INFO
   USE_UNWANTED_PARAM(db_print);
#endif

   dlen = ((dptr[0] & 0x0f) << 8) | dptr[1];
   dptr += 2;
   end_ptr = dptr + dlen;

   memset(link_info, 0, sizeof(SI_RCT_LINK_INFO));

   if (dlen >= 7)
   {
      link_info->link_type = dptr[0] >> 4;

#ifdef DEBUG_RCT_LINK_INFO
      if (db_print)
      {
         STB_SI_PRINT(("   RCT link_info: link_type=0x%x", link_info->link_type));
      }
#endif

      /* Only link type 0x00 (URI) is supported by D-Book 6.2.1 */
      if (link_info->link_type == RCT_LINK_TYPE_URI)
      {
         link_info->how_related = ((dptr[0] & 0x03) << 4) | (dptr[1] >> 4);
         link_info->term_id = ((dptr[1] & 0x0f) << 8) | dptr[2];
         link_info->group_id = dptr[3] >> 4;
         link_info->precedence = dptr[3] & 0x0f;

#ifdef DEBUG_RCT_LINK_INFO
         if (db_print)
         {
            STB_SI_PRINT(("     how_related=0x%02x, term_id=%u, group_id=0x%02x, precedence=%u",
                          link_info->how_related, link_info->term_id, link_info->group_id, link_info->precedence));
         }
#endif
         text_len = dptr[4];
         dptr += 5;

         if (text_len > 0)
         {
            link_info->uri_string = (U8BIT *)STB_GetMemory(text_len + 1);
            if (link_info->uri_string != NULL)
            {
               memcpy(link_info->uri_string, dptr, text_len);
               link_info->uri_string[text_len] = '\0';
#ifdef DEBUG_RCT_LINK_INFO
               if (db_print)
               {
                  STB_SI_PRINT(("     len=%u, URI=\"%s\"", text_len, link_info->uri_string));
               }
#endif
            }

            dptr += text_len;
         }

         /* Process the promotional text items */
         link_info->num_items = dptr[0] & 0x3f;
         dptr++;

#ifdef DEBUG_RCT_LINK_INFO
         if (db_print)
         {
            STB_SI_PRINT(("     num promo items=%u", link_info->num_items));
         }
#endif

         if (link_info->num_items > 0)
         {
            link_info->promo_text_array =
               (SI_RCT_PROMO_TEXT *)STB_GetMemory(link_info->num_items * sizeof(SI_RCT_PROMO_TEXT));
            if (link_info->promo_text_array != NULL)
            {
               for (i = 0; i < link_info->num_items; i++)
               {
                  memset(&link_info->promo_text_array[i], 0, sizeof(SI_RCT_PROMO_TEXT));

                  dptr = ReadLanguageCode(dptr, &link_info->promo_text_array[i].lang_code);
                  dptr = STB_SIReadString(dptr[0], dptr + 1, &link_info->promo_text_array[i].string);

#ifdef DEBUG_RCT_LINK_INFO
                  if (db_print)
                  {
                     STB_SI_PRINT(("       lang=%c%c%c, promo text len=%u bytes",
                                   link_info->promo_text_array[i].lang_code >> 16,
                                   link_info->promo_text_array[i].lang_code >> 8,
                                   link_info->promo_text_array[i].lang_code & 0xff,
                                   link_info->promo_text_array[i].string->nbytes));

                     if ((*link_info->promo_text_array[i].string->str_ptr >= 0x20) &&
                         (*link_info->promo_text_array[i].string->str_ptr < 0x7f))
                     {
                        STB_SI_PRINT(("        \"%s\"", link_info->promo_text_array[i].string->str_ptr));
                     }
                  }
#endif
               }
            }
            else
            {
               link_info->num_items = 0;
            }
         }

         if ((dptr[0] & 0x80) != 0)
         {
            link_info->can_use_default_icon = TRUE;
         }
         else
         {
            link_info->can_use_default_icon = FALSE;
         }

         link_info->icon_id = (dptr[0] & 0x70) >> 4;

#ifdef DEBUG_RCT_LINK_INFO
         if (db_print)
         {
            STB_SI_PRINT(("     default_icon_flag=%u, icon_id=%u", link_info->can_use_default_icon,
                          link_info->icon_id));
         }
#endif

         /* Process the descriptor loop */
         dloop_len = ((dptr[0] & 0x0f) << 8) | dptr[1];
         dptr += 2;

         dloop_end = dptr + dloop_len;
         while (dptr < dloop_end)
         {
            dtag = dptr[0];
            dptr++;

            switch (dtag)
            {
               case SHORT_EVENT_DTAG:
               {
                  text_len = 0;
                  dptr = ParseShortEventDescriptor(dptr, &text_len, &link_info->event_desc, DEBUG_RCT_LINK_INFO_VALUE);
                  break;
               }

               case EXT_DTAG:
               {
                  /* Examine the extension tag for the table being signalled */
                  switch (dptr[1])
                  {
                     case IMAGE_ICON_DTAG:
                     {
                        dptr = ParseImageIconDesc(dptr, &link_info->num_icons, &link_info->icon_array, DEBUG_RCT_LINK_INFO_VALUE);
                        break;
                     }

                     default:
                     {
                        dptr = SkipDescriptor(dptr, dtag, DEBUG_RCT_LINK_INFO_VALUE);
                        break;
                     }
                  }
                  break;
               }

               default:
               {
                  dptr = SkipDescriptor(dptr, dtag, DEBUG_RCT_LINK_INFO_VALUE);
                  break;
               }
            }
         }
      }
   }
#ifdef DEBUG_RCT_LINK_INFO
   else if (db_print)
   {
      STB_SI_PRINT(("   Invalid RCT link_info, len=%u, descriptor skipped", dlen));
   }
#endif

   FUNCTION_FINISH(ParseRctLinkInfo);

   return(end_ptr);
}

/*!**************************************************************************
 * @brief   Reads the image icon descriptor
 * @param   dptr - pointer to length byte of descriptor
 * @param   num_icons - pointer to number of icons currently in array, number is updated on return
 * @param   icon_array - pointer to array of icon structures, array is updated on return
 * @param   db_print - if TRUE debug prints the contents of the descriptor (for debug only)
 * @return  Updated value of current data byte address i.e. at end of descriptor
 ****************************************************************************/
static U8BIT* ParseImageIconDesc(U8BIT *dptr, U8BIT *num_icons, SI_IMAGE_ICON_DESC **icon_array, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   U8BIT data_len;
   U8BIT desc_num, last_desc_num;
   U8BIT icon_id;
   SI_IMAGE_ICON_DESC *icon;
   U8BIT i;

   FUNCTION_START(ParseImageIconDesc);

#ifndef DEBUG_IMAGE_ICON_DESC
   USE_UNWANTED_PARAM(db_print);
#endif

   dlen = *dptr;

   /* Skip the length and tag extension */
   dptr += 2;

   end_ptr = dptr + (dlen - 1);

   if (dlen >= 4)
   {
      desc_num = dptr[0] >> 4;
      last_desc_num = dptr[0] & 0x0f;
      icon_id = dptr[1] & 0x07;

#ifdef DEBUG_IMAGE_ICON_DESC
      if (db_print)
      {
         STB_SI_PRINT(("   Image icon: desc_num=%u, last_desc_num=%u, icon_id=%u", desc_num,
                       last_desc_num, icon_id));
      }
#endif

      if (desc_num == 0)
      {
         /* This is a new icon, add an entry to the array */
         if (*num_icons > 0)
         {
            *icon_array = (SI_IMAGE_ICON_DESC *)STB_ResizeMemory(*icon_array, (*num_icons + 1) * sizeof(SI_IMAGE_ICON_DESC));
            if (*icon_array != NULL)
            {
               icon = &(*icon_array)[*num_icons];
               (*num_icons)++;
            }
            else
            {
               icon = NULL;
#ifdef DEBUG_IMAGE_ICON_DESC
               if (db_print)
               {
                  STB_SI_PRINT(("Failed to allocate memory for new image icon!"));
               }
#endif
            }
         }
         else
         {
            *icon_array = (SI_IMAGE_ICON_DESC *)STB_GetMemory(sizeof(SI_IMAGE_ICON_DESC));
            icon = *icon_array;
            *num_icons = 1;
         }

         if (icon != NULL)
         {
            memset(icon, 0, sizeof(SI_IMAGE_ICON_DESC));

            icon->desc_num = desc_num;
            icon->last_desc_num = last_desc_num;
            icon->icon_id = icon_id;

            icon->transport_mode = dptr[2] >> 6;

#ifdef DEBUG_IMAGE_ICON_DESC
            if (db_print)
            {
               STB_SI_PRINT(("     transport_mode=%u", icon->transport_mode));
            }
#endif

            if ((dptr[2] & 0x20) != 0)
            {
               icon->position_defined = TRUE;
               icon->coord_system = (dptr[2] >> 2) & 0x07;
               icon->x_pos = (dptr[3] << 4) | (dptr[4] >> 4);
               icon->y_pos = ((dptr[4] & 0x0f) << 8) | dptr[5];
               dptr += 6;
#ifdef DEBUG_IMAGE_ICON_DESC
               if (db_print)
               {
                  STB_SI_PRINT(("     coord system=%u, pos=%u,%u", icon->coord_system, icon->x_pos, icon->y_pos));
               }
#endif
            }
            else
            {
               icon->position_defined = FALSE;
               dptr += 3;
            }

            data_len = dptr[0];
            dptr++;

            if (data_len > 0)
            {
               icon->icon_type = (U8BIT *)STB_GetMemory(data_len + 1);
               memcpy(icon->icon_type, dptr, data_len);
               icon->icon_type[data_len] = '\0';
               dptr += data_len;
#ifdef DEBUG_IMAGE_ICON_DESC
               if (db_print)
               {
                  STB_SI_PRINT(("     icon_type=\"%s\"", icon->icon_type));
               }
#endif
            }

            if (icon->transport_mode == ICON_TRANS_LOCAL)
            {
               icon->data_len = dptr[0];

#ifdef DEBUG_IMAGE_ICON_DESC
               if (db_print)
               {
                  STB_SI_PRINT(("     %u icon data bytes", icon->data_len));
               }
#endif
               if (icon->data_len > 0)
               {
                  icon->icon_data = (U8BIT *)STB_GetMemory(icon->data_len);
                  if (icon->icon_data != NULL)
                  {
                     memcpy(icon->icon_data, &dptr[1], icon->data_len);
                  }
                  else
                  {
                     icon->data_len = 0;
                  }
               }
            }
            else if (icon->transport_mode == ICON_TRANS_URL)
            {
               icon->data_len = dptr[0];

               if (icon->data_len > 0)
               {
                  /* Allocate an extra byte to store a null string terminator */
                  icon->data_len++;

                  icon->icon_data = (U8BIT *)STB_GetMemory(icon->data_len);
                  if (icon->icon_data != NULL)
                  {
                     memcpy(icon->icon_data, &dptr[1], icon->data_len - 1);
                     icon->icon_data[icon->data_len - 1] = '\0';
#ifdef DEBUG_IMAGE_ICON_DESC
                     if (db_print)
                     {
                        STB_SI_PRINT(("     icon url=\"%s\"", icon->icon_data));
                     }
#endif
                  }
                  else
                  {
                     icon->data_len = 0;
                  }
               }
            }
         }
      }
      else
      {
         /* This is additional data for an icon that should already have been added */
         for (i = 0; i < *num_icons; i++)
         {
            if ((*icon_array)[i].icon_id == icon_id)
            {
               icon = &(*icon_array)[i];

               /* Icon found, check that the desc number follows on from the last one */
               if ((desc_num == icon->desc_num + 1) && (desc_num <= icon->last_desc_num))
               {
                  /* Update the last desc num seen */
                  icon->desc_num = desc_num;

                  data_len = dptr[2];

#ifdef DEBUG_IMAGE_ICON_DESC
                  if (db_print)
                  {
                     STB_SI_PRINT(("     %u icon data bytes", data_len));
                  }
#endif
                  if (data_len > 0)
                  {
                     /* Add the new data to the end of the existing data */
                     icon->icon_data = (U8BIT *)STB_ResizeMemory(icon->icon_data, data_len + icon->data_len);
                     if (icon->icon_data != NULL)
                     {
                        /* Copy new data into the buffer */
                        memcpy(icon->icon_data + icon->data_len, &dptr[3], data_len);
                        icon->data_len += data_len;
                     }
                     else
                     {
                        icon->data_len = 0;
                     }
                  }
                  break;
               }
            }
         }
      }
   }
#ifdef DEBUG_IMAGE_ICON_DESC
   else
   {
      if (db_print)
      {
         STB_SI_PRINT(("   Image icon: invalid desc len %u", dlen));
      }
   }
#endif

   FUNCTION_FINISH(ParseImageIconDesc);

   return(end_ptr);
}

/**
 * @brief   Reads the URI linkage descriptor
 * @param   dptr pointer to length byte of descriptor
 * @param   desc_list address of pointer to the first entry in the list
 * @param   private_data_code private data specifier value when the linkage descriptor is being read
 * @param   db_print if TRUE debug prints the contents of the descriptor (for debug only)
 * @return  Updated value of current data byte address i.e. at end of descriptor
 */
static U8BIT* ParseURILinkageDesc(U8BIT *dptr, SI_URI_LINKAGE_DESC **desc_list, U32BIT private_data_code,
   BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   SI_URI_LINKAGE_DESC *entry;

   FUNCTION_START(ParseURILinkageDesc);

   #ifndef DEBUG_URI_LINKAGE_DESC
   USE_UNWANTED_PARAM(db_print);
   #endif

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   /* Skip the extension descriptor tag as this has already been checked */
   dptr++;
   dlen--;

   if (dlen >= 2)
   {
      entry = (SI_URI_LINKAGE_DESC *)STB_GetMemory(sizeof(SI_URI_LINKAGE_DESC));
      if (entry != NULL)
      {
         memset(entry, 0, sizeof(SI_URI_LINKAGE_DESC));

         entry->private_data_code = private_data_code;
         entry->uri_linkage_type = *dptr;
         dptr++;

         /* For the uri linkage type and uri length */
         dlen -= 2;

         if (dlen != 0)
         {
            dlen -= *dptr;

            dptr = STB_SIReadString(dptr[0], dptr + 1, &entry->uri);

            if ((entry->uri_linkage_type == 0x00) || (entry->uri_linkage_type == 0x01))
            {
               if (dlen >= 2)
               {
                  entry->min_polling_interval = (dptr[0] << 8) | dptr[1];
                  dptr += 2;
                  dlen -= 2;
               }
               else
               {
                  /* Wrong number of bytes left so ignore the rest */
                  dlen = 0;
               }
            }

            if (dlen != 0)
            {
               /* The rest is private data */
               entry->private_data = STB_GetMemory(dlen);
               if (entry->private_data != NULL)
               {
                  entry->private_data_length = dlen;
                  memcpy(entry->private_data, dptr, dlen);
                  dlen = 0;
               }
            }
         }

         #ifdef DEBUG_URI_LINKAGE_DESC
         if (db_print)
         {
            STB_SI_PRINT(("   URI linkage desc: type=0x%02x (private_data_specifier 0x%02x)",
               entry->uri_linkage_type, entry->private_data_code));
            if (entry->uri != NULL)
            {
               STB_SI_PRINT(("     URI=\"%s\"", entry->uri->str_ptr));
            }
            if ((entry->uri_linkage_type == 0x00) || (entry->uri_linkage_type == 0x01))
            {
               STB_SI_PRINT(("     min_polling_interval=%u", entry->min_polling_interval));
            }
            STB_SI_PRINT(("     %u bytes of private data", entry->private_data_length));
         }
         #endif

         /* Add the new entry to the beginning of the uri linkage list */
         entry->next = *desc_list;
         *desc_list = entry;
      }
   }

   FUNCTION_FINISH(ParseURILinkageDesc);

   return(end_ptr);
}

/**
 * @brief   Reads the Nordig content protection descriptor
 * @param   dptr pointer to length byte of descriptor
 * @param   protection_level pointer to protection level
 * @param   db_print if TRUE debug prints the contents of the descriptor (for debug only)
 * @return  Updated value of current data byte address i.e. at end of descriptor
 */
static U8BIT* ParseNordigContentProtectionDesc(U8BIT *dptr, U8BIT *protection_level, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;

   FUNCTION_START(ParseNordigContentProtectionDesc);

#ifndef DEBUG_NORDIG_CONTENT_PROTECTION_DESC
   USE_UNWANTED_PARAM(db_print);
#endif

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   /* This descriptor contains just 1 byte */
   if (dlen == 1)
   {
      *protection_level = *dptr;
#ifdef DEBUG_NORDIG_CONTENT_PROTECTION_DESC
      if (db_print)
      {
         STB_SI_PRINT(("   Nordig content protection desc: level=0x%02x", *protection_level));
      }
#endif
   }
#ifdef DEBUG_NORDIG_CONTENT_PROTECTION_DESC
   else if (db_print)
   {
      STB_SI_PRINT(("   Nordig content protection desc: wrong length %u, expected 1", dlen));
   }
#endif

   FUNCTION_FINISH(ParseNordigContentProtectionDesc);

   return(end_ptr);
}

/*!**************************************************************************
 * @brief   Allocates memory for and copies the unparsed descriptor.
 *          Only one desc per service is allowed.
 * @param   dptr - pointer to length byte of descriptor
 * @param   desc - pointer to variable takign the descriptor data
 * @param   db_print - if TRUE debug prints the contents of the descriptor (for debug only)
 * @return  Updated value of current data byte address i.e. at end of descriptor
 ****************************************************************************/
static U8BIT* SaveCIProtectionDescriptor(U8BIT *dptr, U8BIT **desc, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;

   FUNCTION_START(SaveCIProtectionDescriptor);

#ifndef DEBUG_SI_SDT_CONTENT
   USE_UNWANTED_PARAM(db_print);
#endif

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (*desc == NULL)
   {
      /* No parsing is done, just save the raw data */
      *desc = STB_GetMemory(dlen + 2);
      if (*desc != NULL)
      {
         memcpy(*desc, dptr - 2, dlen + 2);
      }
#ifdef DEBUG_SI_SDT_CONTENT
      else if (db_print)
      {
         STB_SI_PRINT(("  Failed to allocate %u bytes for CI protection desc", dlen));
      }
#endif
   }
#ifdef DEBUG_SI_SDT_CONTENT
   else if (db_print)
   {
      STB_SI_PRINT(("  CI protection desc found but service already has one!"));
   }
#endif

   FUNCTION_FINISH(SaveCIProtectionDescriptor);

   return(end_ptr);
}

/*!**************************************************************************
 * @brief   Parses the CI+ service descriptor
 * @param   dptr - pointer to length byte of descriptor
 * @param   num_services - pointer to count of services, updated on return
 * @param   service_list - pointer to list of service descriptor structures
 * @param   last_service - pointer to last service descriptor structure in the list
 * @param   db_print - if TRUE debug prints the contents of the descriptor (for debug only)
 * @return  Updated value of current data byte address i.e. at end of descriptor
 ****************************************************************************/
static U8BIT* ParseCIPlusServiceDescriptor(U8BIT *dptr, U16BIT *num_services,
   SI_CIPLUS_SERVICE **service_list, SI_CIPLUS_SERVICE **last_service, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   SI_CIPLUS_SERVICE *new_serv;

   FUNCTION_START(ParseCIPlusServiceDescriptor);

   #ifndef DEBUG_SI_NIT_CONTENT
   USE_UNWANTED_PARAM(db_print);
   #endif

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 7)
   {
      if ((new_serv = (SI_CIPLUS_SERVICE *)STB_GetMemory(sizeof(SI_CIPLUS_SERVICE))) != NULL)
      {
         memset(new_serv, 0, sizeof(SI_CIPLUS_SERVICE));

         new_serv->id = (*dptr << 8) + *(dptr + 1);
         dptr += 2;

         new_serv->type = *dptr;
         dptr++;

         if ((*dptr & 0x80) != 0)
         {
            new_serv->visible = TRUE;
         }

         if ((*dptr & 0x40) != 0)
         {
            new_serv->selectable = TRUE;
         }

         new_serv->lcn = ((*dptr & 0x3f) << 8) + *(dptr + 1);
         dptr += 2;

         dptr = STB_SIReadString(*dptr, dptr + 1, &new_serv->provider_str);
         dptr = STB_SIReadString(*dptr, dptr + 1, &new_serv->name_str);

#ifdef DEBUG_SI_NIT_CONTENT
         if (db_print)
         {
            STB_SI_PRINT(("   CI+ service: id=%u, type=%u, visible=%u, selectable=%u, lcn=%u",
                          new_serv->id, new_serv->type, new_serv->visible, new_serv->selectable, new_serv->lcn));
            if (new_serv->name_str != NULL)
            {
               STB_SI_PRINT(("     Name=\"%s\"", new_serv->name_str->str_ptr));
            }
            if (new_serv->provider_str != NULL)
            {
               STB_SI_PRINT(("     Provider=\"%s\"", new_serv->provider_str->str_ptr));
            }
         }
#endif

         /* Append the service to the end of the list */
         if (*num_services == 0)
         {
            *service_list = new_serv;
         }
         else
         {
            (*last_service)->next = new_serv;
         }

         *last_service = new_serv;
         (*num_services)++;
      }
#ifdef DEBUG_SI_NIT_CONTENT
      else if (db_print)
      {
         STB_SI_PRINT(("  CI+ service: Failed to allocate %u bytes", sizeof(SI_CIPLUS_SERVICE)));
      }
#endif
   }
#ifdef DEBUG_SI_NIT_CONTENT
   else if (db_print)
   {
      STB_SI_PRINT(("  CI+ service: Expected a minimum of 7 bytes, but desc is %u bytes", dlen));
   }
#endif

   USE_UNWANTED_PARAM(dptr);
   FUNCTION_FINISH(ParseCIPlusServiceDescriptor);

   return(end_ptr);
}

/*!**************************************************************************
 * @brief   Parses the CI+ service descriptor
 * @param   dptr - pointer to length byte of descriptor
 * @param   virtual_chan - pointer for return for virtual channel descriptor
 * @param   db_print - if TRUE debug prints the contents of the descriptor (for debug only)
 * @return  Updated value of current data byte address i.e. at end of descriptor
 ****************************************************************************/
static U8BIT* ParseCIPlusVirtualChannelDescriptor(U8BIT *dptr,
   SI_CIPLUS_VIRTUAL_CHANNEL **virtual_chan, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   SI_CIPLUS_VIRTUAL_CHANNEL *v_chan;

   FUNCTION_START(ParseCIPlusVirtualChannelDescriptor);

   #ifndef DEBUG_SI_NIT_CONTENT
   USE_UNWANTED_PARAM(db_print);
   #endif

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   if (dlen >= 6)
   {
      if ((v_chan = (SI_CIPLUS_VIRTUAL_CHANNEL *)STB_GetMemory(sizeof(SI_CIPLUS_VIRTUAL_CHANNEL))) != NULL)
      {
         memset(v_chan, 0, sizeof(SI_CIPLUS_SERVICE));

         v_chan->lcn = ((*dptr & 0x3f) << 8) + *(dptr + 1);
         dptr += 2;

         dptr = STB_SIReadString(*dptr, dptr + 1, &v_chan->provider_str);
         dptr = STB_SIReadString(*dptr, dptr + 1, &v_chan->name_str);

         dlen = *dptr;
         dptr++;
         v_chan->event_info = (U8BIT *)STB_GetMemory(dlen * sizeof(U8BIT));
         if (v_chan->event_info != NULL)
         {
            memcpy(v_chan->event_info, dptr, dlen);
            v_chan->event_info_len = dlen;
         }
         dptr += dlen;

         dlen = *dptr;
         dptr++;
         v_chan->app_domain_id = (U8BIT *)STB_GetMemory((dlen+1) * sizeof(U8BIT));
         if (v_chan->app_domain_id != NULL)
         {
            memcpy(v_chan->app_domain_id, dptr, dlen);
            v_chan->app_domain_id[dlen] = 0;
            v_chan->app_domain_len = dlen;
         }

#ifdef DEBUG_SI_NIT_CONTENT
         if (db_print)
         {
            STB_SI_PRINT(("   CI+ virtual: lcn=%u", v_chan->lcn));
            if (v_chan->name_str != NULL)
            {
               STB_SI_PRINT(("     Name=\"%s\"", v_chan->name_str->str_ptr));
            }
            if (v_chan->provider_str != NULL)
            {
               STB_SI_PRINT(("     Provider=\"%s\"", v_chan->provider_str->str_ptr));
            }
            if (v_chan->event_info != NULL)
            {
               STB_SI_PRINT(("     Event Info=\"%s\"", v_chan->event_info));
            }
            if (v_chan->app_domain_id != NULL)
            {
               STB_SI_PRINT(("     AppDomainId=\"%s\"", v_chan->app_domain_id));
            }
         }
#endif

         *virtual_chan = v_chan;
      }
#ifdef DEBUG_SI_NIT_CONTENT
      else if (db_print)
      {
         STB_SI_PRINT(("  CI+ virtual: Failed to allocate %u bytes", sizeof(SI_CIPLUS_VIRTUAL_CHANNEL)));
      }
#endif
   }
#ifdef DEBUG_SI_NIT_CONTENT
   else if (db_print)
   {
      STB_SI_PRINT(("  CI+ virtual: Expected a minimum of 6 bytes, but desc is %u bytes", dlen));
   }
#endif

   FUNCTION_FINISH(ParseCIPlusVirtualChannelDescriptor);

   return(end_ptr);
}

/**
 * @brief   Reads the CNS SDT code descriptor
 * @param   dptr pointer to length byte of descriptor
 * @param   sdt_code address of pointer to the entry of SI_SDT_CODE_DESC
  * @param   db_print if TRUE debug prints the contents of the descriptor (for debug only)
 * @return  Updated value of current data byte address i.e. at end of descriptor
 */
static U8BIT* ParseSdtCodeDescriptor(U8BIT *dptr, SI_SDT_CODE_DESC **sdt_code, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   SI_SDT_CODE_DESC *entry;

   FUNCTION_START(ParseSdtCodeDescriptor);

   #ifndef DEBUG_SDT_CODE_DESC
   USE_UNWANTED_PARAM(db_print);
   #endif

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   entry = (SI_SDT_CODE_DESC *)STB_GetMemory(sizeof(SI_SDT_CODE_DESC)+ dlen);
   if (entry != NULL)
   {
      memset(entry, 0, sizeof(SI_SDT_CODE_DESC));
      entry->data_length = dlen;
      entry->data = (U8BIT *)(entry + 1); // data offset: sizeof(SI_SDT_CODE_DESC) bytes
      memcpy (entry->data, dptr, dlen); 

      #ifdef DEBUG_SDT_CODE_DESC
      if (db_print)
      {
         int i;
         STB_SPDebugNoCnWrite("   CNS SDT code desc: code=0x");
         for (i = 0; i < dlen; i++)
            STB_SPDebugNoCnWrite("%02X", entry->data[i]);
            
         STB_SPDebugNoCnWrite("\n\r");
      }
      #endif
   }
   *sdt_code = entry;

   FUNCTION_FINISH(ParseSdtCodeDescriptor);

   return(end_ptr);
}

static U8BIT* ParseSeriesDescriptor(U8BIT *dptr, U8BIT dtag, SI_SERIES_DESC **series, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   SI_SERIES_DESC *entry;

   FUNCTION_START(ParseSeriesDescriptor);

   USE_UNWANTED_PARAM(db_print);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   entry = (SI_SERIES_DESC *)STB_GetMemory(sizeof(SI_SERIES_DESC));
   if (entry != NULL)
   {
      memset(entry, 0, sizeof(SI_SERIES_DESC));
      if (dlen > 2) {
         dptr = STB_SIReadString(dlen - 2, dptr, &entry->episode_name);
      }
      entry->episode_number = ((uint16_t)dptr[0] << 5) + (dptr[1] >> 3);
      entry->content_status = (dptr[1] >> 1) & 0x03;
      entry->last_episode = (dptr[1] & 0x01) ? TRUE : FALSE;

      #ifdef DEBUG_SERIES_DESC
      if (db_print)
      {
         int i;
         STB_SPDebugNoCnWrite("   CNS series (linking) desc: code=0x");
         for (i = 0; i < dlen; i++)
            STB_SPDebugNoCnWrite("%02X", entry->data[i]);
            
         STB_SPDebugNoCnWrite("\n\r");
      }
      #endif
   }
   *series = entry;

   FUNCTION_FINISH(ParseSeriesDescriptor);

   return(end_ptr);
}

// SSU descriptors
static U8BIT* ParseSsuLocationDescriptor(U8BIT *dptr, SI_SSU_LOCATION_DESC **ssu_location, BOOLEAN db_print)
{
   U8BIT dlen;
   U8BIT *end_ptr;
   SI_SSU_LOCATION_DESC *entry;

   FUNCTION_START(ParseSsuLocationDescriptor);

   USE_UNWANTED_PARAM(db_print);

   dlen = *dptr;
   dptr++;
   end_ptr = dptr + dlen;

   entry = (SI_SSU_LOCATION_DESC *)STB_GetMemory(sizeof(SI_SSU_LOCATION_DESC));
   if (entry != NULL)
   {
      S16BIT private_data_length = dlen;
      memset(entry, 0, sizeof(SI_SSU_LOCATION_DESC));
      
      entry->data_broadcast_id = ((uint16_t)dptr[0] << 8) + dptr[1];
      dptr += 2;
      private_data_length -= 2;
      
      if (0x000A == entry->data_broadcast_id) {
         entry->association_tag = ((uint16_t)dptr[0] << 8) + dptr[1];
         dptr += 2;
         private_data_length -= 2;
      }
      if (private_data_length > 0) {
         entry->private_data_byte = STB_GetMemory(private_data_length);
         if (entry->private_data_byte != NULL)
         {
            entry->private_data_length = private_data_length;
            memcpy(entry->private_data_byte, dptr, private_data_length);
         }
      }
      
      #ifdef DEBUG_SSU_LOCATION_DESC
      if (db_print || TRUE)
      {
         int i;
         STB_SPDebugNoCnWrite("   SSU location desc: %u private_data_byte=", private_data_length);
         for (i = 0; i < private_data_length; i++)
            STB_SPDebugNoCnWrite(" %02X", entry->private_data_byte[i]);
            
         STB_SPDebugNoCnWrite("\n\r");
      }
      #endif
   }
   *ssu_location = entry;

   
   FUNCTION_FINISH(ParseSsuLocationDescriptor);

   return(end_ptr);
}

/**
 *

 *
 * @brief   Converts 4-bit BCD to floating point value
 *
 * @param   data_ptr - current data byte address
 * @param   len      - number of digits
 * @param   point    - digits before the decimal point
 *
 * @return   Floating point value
 *
 */
static float BCDToFloat(U8BIT *data_ptr, U8BIT len, U8BIT point)
{
   U8BIT i, ival;
   float mult = 0.1f;
   float fval = 0.0f;

   FUNCTION_START(BCDToFloat);

   ASSERT(data_ptr != NULL);

   for (i = 0; i < point; i++)
      mult *= 10.0f;

   for (i = 0; i < len; i++)
   {
      if ((i % 2) != 0)
         ival = (data_ptr[i / 2] & 0x0f);                      // odd digit
      else
         ival = ((data_ptr[i / 2] >> 4) & 0x0f);               // even digit
      fval += ((float)ival * mult);
      mult /= 10.0f;
   }

   FUNCTION_FINISH(BCDToFloat);

   return(fval);
}

//---global function definitions-----------------------------------------------

/**
 * @brief   Sets the country specific private data specifier code that will be used when parsing
 *          SI tables
 * @param   code country specific private data code to be enabled
 */
void STB_SISetCountryPrivateDataSpecifier(U32BIT code)
{
   FUNCTION_START(STB_SISetCountryPrivateDataSpecifier);

   country_private_data_specifier_code = code;
   STB_SI_PRINT(("Set country private data specifier 0x%08x", code));

   FUNCTION_FINISH(STB_SISetCountryPrivateDataSpecifier);
}

/**
 * @brief   Enables or disables use of the Freesat private data specifier when parsing SI tables
 * @param   mode TRUE to enable, FALSE to disable
 */
void STB_SISetFreesatPrivateDataSpecifierMode(BOOLEAN mode)
{
   FUNCTION_START(STB_SISetPrivateDataSpecifier);

   freesat_private_data_specifier = mode;

   FUNCTION_FINISH(STB_SISetPrivateDataSpecifier);
}

/**
 * @brief   Enables or disables use of the CI+ private data specifier when parsing SI tables
 * @param   mode TRUE to enable, FALSE to disable
 */
void STB_SISetCiplusPrivateDataSpecifierMode(BOOLEAN mode)
{
   FUNCTION_START(STB_SISetPrivateDataSpecifier);

   ciplus_private_data_specifier = mode;

   FUNCTION_FINISH(STB_SISetPrivateDataSpecifier);
}

/**
 * @brief   Enables or disables use of the EACEM private data specifier when parsing SI tables
 * @param   mode TRUE to enable, FALSE to disable
 */
void STB_SISetEacemPrivateDataSpecifierMode(BOOLEAN mode)
{
   FUNCTION_START(STB_SISetPrivateDataSpecifier);

   eacem_private_data_specifier = mode;

   FUNCTION_FINISH(STB_SISetPrivateDataSpecifier);
}

/**
 * @brief   Enables or disables use of the New Zealand satellite private data specifier when parsing SI tables
 * @param   mode TRUE to enable, FALSE to disable
 */
void STB_SISetNZSatPrivateDataSpecifierMode(BOOLEAN mode)
{
   FUNCTION_START(STB_SISetPrivateDataSpecifier);

   nzsat_private_data_specifier = mode;

   FUNCTION_FINISH(STB_SISetPrivateDataSpecifier);
}

/**
 * @brief   Enables or disables use of the Nordig private data specifier when parsing SI tables
 * @param   mode TRUE to enable, FALSE to disable
 */
void STB_SISetNordigPrivateDataSpecifierMode(BOOLEAN mode)
{
   FUNCTION_START(STB_SISetNordigPrivateDataSpecifierMode);

   nordig_private_data_specifier = mode;

   FUNCTION_FINISH(STB_SISetNordigPrivateDataSpecifierMode);
}

/**
 *

 *
 * @brief   Sets the function allocated to the user defined descriptor
 *
 * @param   dtag - the descriptor tag to be allocated
 * @param   func - the required function
 *

 *
 */
void STB_SISetUserDefinedDescriptorFunction(U8BIT dtag, STB_SI_USER_DEF_DESCRIP_FUNCTION func)
{
   FUNCTION_START(STB_SISetUserDefinedDescriptorFunction);

   if ((FIRST_USER_DEFINED_DTAG <= dtag) && (dtag <= LAST_USER_DEFINED_DTAG))
   {
      user_defined_dtag_function[dtag - FIRST_USER_DEFINED_DTAG] = func;
      #ifdef STB_DEBUG
      {
         U8BIT *func_str;
         switch (func)
         {
            case USER_DEF_DESCRIP_NOT_USED:           {func_str = (U8BIT *)"not used"; break; }
            case USER_DEF_DESCRIP_LOGICAL_CHAN_NUM:   {func_str = (U8BIT *)"logical chan num"; break; }
            case USER_DEF_DESCRIP_PREF_NAME_LIST:     {func_str = (U8BIT *)"pref name list"; break; }
            case USER_DEF_DESCRIP_PREF_NAME_ID:       {func_str = (U8BIT *)"pref name id"; break; }
            case USER_DEF_DESCRIP_SDT_CODE:           {func_str = (U8BIT *)"STD code"; break; }
            case USER_DEF_DESCRIP_SERIES:             {func_str = (U8BIT *)"series"; break; }
            default:                                  {func_str = (U8BIT *)"INVALID"; break; }
         }
         STB_SI_PRINT(("Set user defined descriptor: 0x%02x = %s", dtag, func_str));
      }
      #endif
   }
   FUNCTION_FINISH(STB_SISetUserDefinedDescriptorFunction);
}

/*!**************************************************************************
 * @brief   Clear all entries in the user defined SI descriptor table
 ****************************************************************************/
void STB_SIClearUserDefinedDescriptorFunctions(void)
{
   U8BIT index;

   FUNCTION_START(STB_SIClearUserDefinedDescriptorFunctions);

   for (index = FIRST_USER_DEFINED_DTAG; index <= LAST_USER_DEFINED_DTAG; index++)
   {
      user_defined_dtag_function[index - FIRST_USER_DEFINED_DTAG] = USER_DEF_DESCRIP_NOT_USED;
   }

   FUNCTION_FINISH(STB_SIClearUserDefinedDescriptorFunctions);
}

/**
 *

 *
 * @brief   Generates request for PAT
 *
 * @param   path      - the demux path to use
 * @param   req_type  - ONE_SHOT or CONTINUOUS request
 * @param   callback  - callback function to receive the parsed pat table
 * @param   ret_param - parameter to be returned to callback function
 *
 * @return   filter handle - NULL if filter not successfully setup
 *
 */
void* STB_SIRequestPat(U8BIT path, E_SI_REQUEST_TYPE req_type,
   void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param)
{
   void *filter_ptr;

   FUNCTION_START(STB_SIRequestPat);

   #ifdef DEBUG_SI_PAT_REQUEST
   STB_SI_PRINT(("STB_SIRequestPat: path %d", path));
   #endif

   filter_ptr = STB_SIRequestTable(path, SI_PAT_PID, SI_PAT_TID, 0xff, MULTI_SECT, 1,
         SI_XTID_MATCH_DONT_CARE, SI_XTID_MASK_DONT_CARE, req_type,
         SI_BUFFER_4K, TRUE, FALSE, callback, ret_param);

   FUNCTION_FINISH(STB_SIRequestPat);
   return((void *)filter_ptr);
}

/**
 *

 *
 * @brief   Generates request for PMT
 *
 * @param   path        - the demux path to use
 * @param   req_type    - ONE_SHOT or CONTINUOUS request
 * @param   pmt_pid     - pid for the pmt table
 * @param   sid_match   - service id to find
 * @param   sid_mask    - service id mask word for filter
 * @param   table_count - number of tables expected (only relevant for non-continuous)
 * @param   callback    - callback function to receive the parsed pmt table
 * @param   ret_param   - parameter to be returned to callback function
 *
 * @return   filter handle - NULL if filter not successfully setup
 *
 */
void* STB_SIRequestPmt(U8BIT path, E_SI_REQUEST_TYPE req_type, U16BIT pmt_pid,
   U16BIT sid_match, U16BIT sid_mask, U16BIT table_count,
   void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param)
{
   void *filter_ptr;

   FUNCTION_START(STB_SIRequestPmt);

   #ifdef DEBUG_SI_PMT_REQUEST
   STB_SI_PRINT(("STB_SIRequestPmt: path %d pid 0x%04x sid 0x%04x[0x%04x]",
                 path, pmt_pid, sid_match, sid_mask));
   #endif
   filter_ptr = STB_SIRequestTable(path, pmt_pid, SI_PMT_TID, 0xff, MULTI_SECT, table_count,
         sid_match, sid_mask, req_type, SI_BUFFER_4K, TRUE, FALSE,
         callback, ret_param);

   FUNCTION_FINISH(STB_SIRequestPmt);
   return((void *)filter_ptr);
}

/**
 *

 *
 * @brief   Modifies request for PMT to look for different service on SAME PID
 *
 * @param   fhandle     - handle of the filter to be modified
 * @param   sid_match   - service id to find
 * @param   sid_mask    - service id mask word for filter
 * @param   table_count - number of tables expected (only relevant for non-continuous)
 *

 *
 */
void STB_SIModifyPmtRequest(void *fhandle, U16BIT sid_match, U16BIT sid_mask, U16BIT table_count)
{
   FUNCTION_START(STB_SIModifyPmtRequest);

   #ifdef DEBUG_SI_PMT_REQUEST
   STB_SI_PRINT(("STB_SIModifyPmtRequest: sid 0x%04x[0x%04x]", sid_match, sid_mask));
   #endif
   STB_SIModifyTableRequest(fhandle, SI_PMT_TID, 0xff, sid_match, sid_mask, table_count);

   FUNCTION_FINISH(STB_SIModifyPmtRequest);
}

/**
 *

 *
 * @brief   Generates request for NIT(actual)
 *
 * @param   path      - the demux path to use
 * @param   req_type  - ONE_SHOT or CONTINUOUS request
 * @param   callback  - callback function to receive the parsed nit table
 * @param   ret_param - parameter to be returned to callback function
 *
 * @return   filter handle - NULL if filter not successfully setup
 *
 */
void* STB_SIRequestNit(U8BIT path, E_SI_REQUEST_TYPE req_type,
   void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param)
{
   void *filter_ptr;
   U16BIT expected_tables = 1;

   FUNCTION_START(STB_SIRequestNit);

   #ifdef DEBUG_SI_NIT_REQUEST
   STB_SI_PRINT(("STB_SIRequestNit: path %d pid 0x%04x", path, SI_NIT_PID));
   #endif

   if (CONTINUOUS_REQUEST == req_type)
      expected_tables = 0xFFFF;

   filter_ptr = STB_SIRequestTable(path, SI_NIT_PID, SI_NIT_ACTUAL_TID, 0xff, MULTI_SECT, expected_tables,
         SI_XTID_MATCH_DONT_CARE, SI_XTID_MASK_DONT_CARE, req_type,
         SI_BUFFER_4K, TRUE, FALSE, callback, ret_param);

   FUNCTION_FINISH(STB_SIRequestNit);
   return((void *)filter_ptr);
}

/**
 * @brief   Create an SI filter for an NIT table
 * @param   path demux path to be used
 * @param   pid PID the table will be available on - this allows the standard PID to be overridden
 * @param   actual TRUE if NITactual is to be retrieved, FALSE for NITother
 * @param   req_type ONE_SHOT_REQUEST or CONTINUOUS_REQUEST
 * @param   callback function to be called when table is received
 * @param   ret_param value that will be passed to the callback function
 * @return  filter handle, NULL if filter isn't created
 */
void* STB_SIRequestNitFromPid(U8BIT path, U16BIT pid, BOOLEAN actual, E_SI_REQUEST_TYPE req_type,
   void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param)
{
   void *filter_ptr;
   U8BIT tid;

   FUNCTION_START(STB_SIRequestNitFromPid);

   #ifdef DEBUG_SI_NIT_REQUEST
   STB_SI_PRINT(("STB_SIRequestNit: path %d pid 0x%04x", path, pid));
   #endif

   if (actual)
   {
      tid = SI_NIT_ACTUAL_TID;
   }
   else
   {
      tid = SI_NIT_OTHER_TID;
   }

   filter_ptr = STB_SIRequestTable(path, pid, tid, 0xff, MULTI_SECT, 1,
         SI_XTID_MATCH_DONT_CARE, SI_XTID_MASK_DONT_CARE, req_type,
         SI_BUFFER_4K, TRUE, FALSE, callback, ret_param);

   FUNCTION_FINISH(STB_SIRequestNitFromPid);

   return(filter_ptr);
}

/**
 * @brief   Generates request for NITactual and NITother for the given network ID
 * @param   path the decode path to use
 * @param   network_id network ID being requested
 * @param   req_type ONE_SHOT or CONTINUOUS request
 * @param   callback callback function to receive the parsed nit table
 * @param   ret_param parameter to be returned to callback function
 * @return  filter handle or NULL if filter not successfully setup
 */
void* STB_SIRequestNitWithId(U8BIT path, U16BIT network_id, E_SI_REQUEST_TYPE req_type,
   void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param)
{
   void *filter_ptr;

   FUNCTION_START(STB_SIRequestNitWithId);

   #ifdef DEBUG_SI_NIT_REQUEST
   STB_SI_PRINT(("STB_SIRequestNitWithId: path %d pid 0x%04x", path, SI_NIT_PID));
   #endif

   filter_ptr = STB_SIRequestTable(path, SI_NIT_PID, SI_NIT_ACTUAL_TID, 0xfe, MULTI_SECT, 1,
         network_id, 0xffff, req_type, SI_BUFFER_4K, TRUE, FALSE, callback, ret_param);

   FUNCTION_FINISH(STB_SIRequestNitWithId);

   return((void *)filter_ptr);
}

/**
 *

 *
 * @brief   Generates request for SDT
 *
 * @param   path           - the demux path to use
 * @param   req_type       - ONE_SHOT or CONTINUOUS request
 * @param   inc_sdt_actual - search for sdt actual
 * @param   inc_sdt_other  - search for sdt other
 * @param   tran_id_match  - transport id to find
 * @param   tran_id_mask   - transport id mask word for filter
 * @param   table_count    - number of tables expected (only relevant for non-continuous)
 * @param   callback       - callback function to receive the parsed sdt table
 * @param   ret_param      - parameter to be returned to callback function
 *
 * @return   filter handle - NULL if filter not successfully setup
 *
 */
void* STB_SIRequestSdt(U8BIT path, E_SI_REQUEST_TYPE req_type,
   BOOLEAN inc_sdt_actual, BOOLEAN inc_sdt_other,
   U16BIT tran_id_match, U16BIT tran_id_mask, U16BIT table_count,
   void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param)
{
   void *filter_ptr;
   U8BIT tid_match;
   U8BIT tid_mask;

   FUNCTION_START(STB_SIRequestSdt);


   tid_match = SI_SDT_ACTUAL_TID;
   tid_mask = 0xff;

   if (inc_sdt_other == TRUE)
   {
      tid_match = SI_SDT_OTHER_TID;
      if (inc_sdt_actual == TRUE)
      {
         tid_mask = (U8BIT)(~(SI_SDT_ACTUAL_TID ^ SI_SDT_OTHER_TID));
      }
   }

   #ifdef DEBUG_SI_SDT_REQUEST
   STB_SI_PRINT(("STB_SIRequestSdt: path %d pid 0x%04x tid 0x%02x[0x%02x] tran_id 0x%04x[0x%04x]",
                 path, SI_SDT_PID, tid_match, tid_mask, tran_id_match, tran_id_mask));
   #endif

   filter_ptr = STB_SIRequestTable(path, SI_SDT_PID, tid_match, tid_mask, MULTI_SECT, table_count,
         tran_id_match, tran_id_mask, req_type, SI_BUFFER_4K,
         TRUE, FALSE, callback, ret_param);

   FUNCTION_FINISH(STB_SIRequestSdt);
   return((void *)filter_ptr);
}

/**
 *

 *
 * @brief   Generates request for SDT
 *
 * @param   path           - the demux path to use
 * @param   pid            - the pid of where to reuest sdt
 * @param   req_type       - ONE_SHOT or CONTINUOUS request
 * @param   inc_sdt_actual - search for sdt actual
 * @param   inc_sdt_other  - search for sdt other
 * @param   tran_id_match  - transport id to find
 * @param   tran_id_mask   - transport id mask word for filter
 * @param   table_count    - number of tables expected (only relevant for non-continuous)
 * @param   callback       - callback function to receive the parsed sdt table
 * @param   ret_param      - parameter to be returned to callback function
 *
 * @return   filter handle - NULL if filter not successfully setup
 *
 */
void* STB_SIRequestSdtFromPid(U8BIT path, U16BIT pid, E_SI_REQUEST_TYPE req_type,
   BOOLEAN inc_sdt_actual, BOOLEAN inc_sdt_other,
   U16BIT tran_id_match, U16BIT tran_id_mask, U16BIT table_count,
   void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param)
{
   void *filter_ptr;
   U8BIT tid_match;
   U8BIT tid_mask;

   FUNCTION_START(STB_SIRequestSdtFromPid);


   tid_match = SI_SDT_ACTUAL_TID;
   tid_mask = 0xff;

   if (inc_sdt_other == TRUE)
   {
      tid_match = SI_SDT_OTHER_TID;
      if (inc_sdt_actual == TRUE)
      {
         tid_mask = (U8BIT)(~(SI_SDT_ACTUAL_TID ^ SI_SDT_OTHER_TID));
      }
   }

   #ifdef DEBUG_SI_SDT_REQUEST
   STB_SI_PRINT(("STB_SIRequestSdt: path %d pid 0x%04x tid 0x%02x[0x%02x] tran_id 0x%04x[0x%04x]",
                 path, pid, tid_match, tid_mask, tran_id_match, tran_id_mask));
   #endif

   filter_ptr = STB_SIRequestTable(path, pid, tid_match, tid_mask, MULTI_SECT, table_count,
         tran_id_match, tran_id_mask, req_type, SI_BUFFER_4K,
         TRUE, FALSE, callback, ret_param);

   FUNCTION_FINISH(STB_SIRequestSdtFromPid);
   return((void *)filter_ptr);
}

/**
 *

 *
 * @brief   Modifies request for SDT to look for different transport on SAME PID
 *
 * @param   fhandle        - handle of the filter to be modified
 * @param   inc_sdt_actual - search for sdt actual
 * @param   inc_sdt_other  - search for sdt other
 * @param   tran_id_match  - transport id to find
 * @param   tran_id_mask   - transport id mask word for filter
 * @param   table_count    - number of tables expected (only relevant for non-continuous)
 *

 *
 */
void STB_SIModifySdtRequest(void *fhandle, BOOLEAN inc_sdt_actual, BOOLEAN inc_sdt_other,
   U16BIT tran_id_match, U16BIT tran_id_mask, U16BIT table_count)
{
   U8BIT tid_match;
   U8BIT tid_mask;

   FUNCTION_START(STB_SIModifySdtRequest);

   tid_match = SI_SDT_ACTUAL_TID;
   tid_mask = 0xff;

   if (inc_sdt_other == TRUE)
   {
      tid_match = SI_SDT_OTHER_TID;
      if (inc_sdt_actual == TRUE)
      {
         tid_mask = (U8BIT)(~(SI_SDT_ACTUAL_TID ^ SI_SDT_OTHER_TID));
      }
   }

   #ifdef DEBUG_SI_SDT_REQUEST
   STB_SI_PRINT(("STB_SIModifySdtRequest: tid 0x%02x[0x%02x] tran_id 0x%04x[0x%04x]",
                 tid_match, tid_mask, tran_id_match, tran_id_mask));
   #endif

   STB_SIModifyTableRequest(fhandle, tid_match, tid_mask, tran_id_match, tran_id_mask, table_count);

   FUNCTION_FINISH(STB_SIModifySdtRequest);
}

/**
 *

 *
 * @brief   Generates request for Bat
 *
 * @param   path           - the demux path to use
 * @param   req_type       - ONE_SHOT or CONTINUOUS request
 * @param   bouquet_id_match - bouquet id to find
 * @param   bouquet_id_mask  - bouquet id mask word for filter
 * @param   table_count    - number of tables expected (only relevant for non-continuous)
 * @param   callback       - callback function to receive the parsed sdt table
 * @param   ret_param      - parameter to be returned to callback function
 *
 * @return   filter handle - NULL if filter not successfully setup
 *
 */
void* STB_SIRequestBat(U8BIT path, E_SI_REQUEST_TYPE req_type, U16BIT bouquet_id_match,
   U16BIT bouquet_id_mask, U16BIT table_count, void (*callback)(void *, U32BIT, SI_TABLE_RECORD *),
   U32BIT ret_param)
{
   void *filter_ptr;

   FUNCTION_START(STB_SIRequestBat);

   #ifdef DEBUG_SI_BAT_REQUEST
   STB_SI_PRINT(("STB_SIRequestBat: path %d pid 0x%04x tid 0x%02x[0xff] tran_id 0x%04x[0x%04x]",
                 path, SI_BAT_PID, SI_BAT_TID, bouquet_id_match, bouquet_id_mask));
   #endif

   filter_ptr = STB_SIRequestTable(path, SI_BAT_PID, SI_BAT_TID, 0xff, MULTI_SECT, table_count,
         bouquet_id_match, bouquet_id_mask, req_type, SI_BUFFER_4K,
         TRUE, FALSE, callback, ret_param);

   FUNCTION_FINISH(STB_SIRequestBat);
   return((void *)filter_ptr);
}

/**
 *

 *
 * @brief   Generates request for Bat
 *
 * @param   path           - the demux path to use
 * @param   pid            - the pid of where to reuest sdt
 * @param   req_type       - ONE_SHOT or CONTINUOUS request
 * @param   bouquet_id_match - bouquet id to find
 * @param   bouquet_id_mask  - bouquet id mask word for filter
 * @param   table_count    - number of tables expected (only relevant for non-continuous)
 * @param   callback       - callback function to receive the parsed sdt table
 * @param   ret_param      - parameter to be returned to callback function
 *
 * @return   filter handle - NULL if filter not successfully setup
 *
 */
void* STB_SIRequestBatFromPid(U8BIT path, U16BIT pid, E_SI_REQUEST_TYPE req_type,
   U16BIT bouquet_id_match, U16BIT bouquet_id_mask, U16BIT table_count,
   void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param)
{
   void *filter_ptr;
   U8BIT tid_match;

   FUNCTION_START(STB_SIRequestBatFromPid);

   tid_match = SI_BAT_TID;

   #ifdef DEBUG_SI_BAT_REQUEST
   STB_SI_PRINT(("STB_SIRequestBat: path %d pid 0x%04x tid 0x%02x[0xff] tran_id 0x%04x[0x%04x]",
                 path, pid, tid_match, bouquet_id_match, bouquet_id_mask));
   #endif

   filter_ptr = STB_SIRequestTable(path, pid, tid_match, 0xff, MULTI_SECT, table_count,
         bouquet_id_match, bouquet_id_mask, req_type, SI_BUFFER_4K,
         TRUE, FALSE, callback, ret_param);

   FUNCTION_FINISH(STB_SIRequestBatFromPid);
   return((void *)filter_ptr);
}

/**
 *

 *
 * @brief   Generates request for EIT
 *
 * @param   path            - the demux path to use
 * @param   req_type        - ONE_SHOT or CONTINUOUS request
 * @param   reqd_eit_tables - defines which tables are required
 * @param   sid_match       - service id to find
 * @param   sid_mask        - service id mask word for filter
 * @param   table_count     - number of tables expected (only relevant for non-continuous)
 * @param   callback        - callback function to receive the parsed eit table
 * @param   ret_param       - parameter to be returned to callback function
 *
 * @return   filter handle - NULL if filter not successfully setup
 *
 */
void* STB_SIRequestEit(U8BIT path, E_SI_REQUEST_TYPE req_type, E_SI_EIT_TABLE_REQ reqd_eit_tables,
   U16BIT sid_match, U16BIT sid_mask, U16BIT table_count,
   void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param)
{
   void *filter_ptr;

   FUNCTION_START(STB_SIRequestEit);

   #ifdef DEBUG_SI_EIT_REQUEST
   STB_SI_PRINT(("STB_SIRequestEit: path %d pid 0x%04x tid 0x%02x[0x%02x] tran_id 0x%04x[0x%04x]",
                 path, SI_EIT_PID, eit_tid_match[reqd_eit_tables], eit_tid_mask[reqd_eit_tables],
                 sid_match, sid_mask));
   #endif

   filter_ptr = STB_SIRequestTable(path, SI_EIT_PID, eit_tid_match[reqd_eit_tables],
         eit_tid_mask[reqd_eit_tables], MULTI_SECT, table_count,
         sid_match, sid_mask, req_type, SI_BUFFER_8K, TRUE, TRUE,
         callback, ret_param);

   FUNCTION_FINISH(STB_SIRequestEit);
   return((void *)filter_ptr);
}

/**
 *

 *
 * @brief   Generates request for EIT
 *
 * @param   path            - the demux path to use
 * @param   pid             - the pid of where to request the eit
 * @param   req_type        - ONE_SHOT or CONTINUOUS request
 * @param   reqd_eit_tables - defines which tables are required
 * @param   sid_match       - service id to find
 * @param   sid_mask        - service id mask word for filter
 * @param   table_count     - number of tables expected (only relevant for non-continuous)
 * @param   callback        - callback function to receive the parsed eit table
 * @param   ret_param       - parameter to be returned to callback function
 *
 * @return   filter handle - NULL if filter not successfully setup
 *
 */
void* STB_SIRequestEitFromPid(U8BIT path, U16BIT pid, E_SI_REQUEST_TYPE req_type, E_SI_EIT_TABLE_REQ reqd_eit_tables,
   U16BIT sid_match, U16BIT sid_mask, U16BIT table_count,
   void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param)
{
   void *filter_ptr;

   FUNCTION_START(STB_SIRequestEitFromPid);

   #ifdef DEBUG_SI_EIT_REQUEST
   STB_SI_PRINT(("STB_SIRequestEit: path %d pid 0x%04x tid 0x%02x[0x%02x] tran_id 0x%04x[0x%04x]",
                 path, pid, eit_tid_match[reqd_eit_tables], eit_tid_mask[reqd_eit_tables],
                 sid_match, sid_mask));
   #endif

   filter_ptr = STB_SIRequestTable(path, pid, eit_tid_match[reqd_eit_tables],
         eit_tid_mask[reqd_eit_tables], MULTI_SECT, table_count,
         sid_match, sid_mask, req_type, SI_BUFFER_8K, TRUE, TRUE,
         callback, ret_param);

   FUNCTION_FINISH(STB_SIRequestEitFromPid);
   return((void *)filter_ptr);
}

/**
 *

 *
 * @brief   Modifies request for SDT to look for different transport on SAME PID
 *
 * @param   fhandle         - handle of the filter to be modified
 * @param   reqd_eit_tables - defines which tables are required
 * @param   sid_match       - service id to find
 * @param   sid_mask        - service id mask word for filter
 * @param   table_count     - number of tables expected (only relevant for non-continuous)
 *

 *
 */
void STB_SIModifyEitRequest(void *fhandle, E_SI_EIT_TABLE_REQ reqd_eit_tables,
   U16BIT sid_match, U16BIT sid_mask, U16BIT table_count)
{
   FUNCTION_START(STB_SIModifyEitRequest);

   #ifdef DEBUG_SI_EIT_REQUEST
   STB_SI_PRINT(("STB_SIModifyEitRequest: tid 0x%02x[0x%02x] tran_id 0x%04x[0x%04x]",
                 eit_tid_match[reqd_eit_tables], eit_tid_mask[reqd_eit_tables],
                 sid_match, sid_mask));
   #endif

   STB_SIModifyTableRequest(fhandle, eit_tid_match[reqd_eit_tables], eit_tid_mask[reqd_eit_tables],
      sid_match, sid_mask, table_count);

   FUNCTION_FINISH(STB_SIModifyEitRequest);
}

/**
 *

 *
 * @brief   Generates request for EIT schedule table
 *
 * @param   path            - the demux path to use
 * @param   req_type        - ONE_SHOT or CONTINUOUS request
 * @param   reqd_eit_tables - defines which tables are required
 * @param   sid_match       - service id to find
 * @param   sid_mask        - service id mask word for filter
 * @param   table_count     - number of tables expected (only relevant for non-continuous)
 * @param   callback        - callback function to receive the parsed eit table
 * @param   ret_param       - parameter to be returned to callback function
 *
 * @return   filter handle - NULL if filter not successfully setup
 *
 */
void* STB_SIRequestSched(U8BIT path, E_SI_REQUEST_TYPE req_type, E_SI_SCHED_TABLE_REQ reqd_eit_tables,
   U16BIT sid_match, U16BIT sid_mask, U16BIT table_count,
   void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param)
{
   void *filter_ptr;

   FUNCTION_START(STB_SIRequestSched);

   #ifdef DEBUG_SI_EIT_REQUEST
   STB_SI_PRINT(("STB_SIRequestSched: path %d tid 0x%02x[0x%02x] tran_id 0x%04x[0x%04x]",
                 path, sched_tid_match[reqd_eit_tables], sched_tid_mask[reqd_eit_tables],
                 sid_match, sid_mask));
   #endif

   filter_ptr = STB_SIRequestTable(path, SI_EIT_PID, sched_tid_match[reqd_eit_tables],
         sched_tid_mask[reqd_eit_tables], MULTI_SECT, table_count,
         sid_match, sid_mask, req_type, SI_BUFFER_32K, TRUE, TRUE,
         callback, ret_param);

   FUNCTION_FINISH(STB_SIRequestSched);
   return((void *)filter_ptr);
}

/**
 *

 *
 * @brief   Generates request for EIT schedule table
 *
 * @param   path            - the demux path to use
 * @param   pid             - the pid of where to request the schedule
 * @param   req_type        - ONE_SHOT or CONTINUOUS request
 * @param   reqd_eit_tables - defines which tables are required
 * @param   sid_match       - service id to find
 * @param   sid_mask        - service id mask word for filter
 * @param   table_count     - number of tables expected (only relevant for non-continuous)
 * @param   callback        - callback function to receive the parsed eit table
 * @param   ret_param       - parameter to be returned to callback function
 *
 * @return   filter handle - NULL if filter not successfully setup
 *
 */
void* STB_SIRequestSchedFromPid(U8BIT path, U16BIT pid, E_SI_REQUEST_TYPE req_type, E_SI_SCHED_TABLE_REQ reqd_eit_tables,
   U16BIT sid_match, U16BIT sid_mask, U16BIT table_count,
   void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param)
{
   void *filter_ptr;

   FUNCTION_START(STB_SIRequestSchedFromPid);

   #ifdef DEBUG_SI_EIT_REQUEST
   STB_SI_PRINT(("STB_SIRequestSched: path %d tid 0x%02x[0x%02x] tran_id 0x%04x[0x%04x]",
                 path, sched_tid_match[reqd_eit_tables], sched_tid_mask[reqd_eit_tables],
                 sid_match, sid_mask));
   #endif

   filter_ptr = STB_SIRequestTable(path, pid, sched_tid_match[reqd_eit_tables],
         sched_tid_mask[reqd_eit_tables], MULTI_SECT, table_count,
         sid_match, sid_mask, req_type, SI_BUFFER_64K, TRUE, TRUE,
         callback, ret_param);

   FUNCTION_FINISH(STB_SIRequestSchedFromPid);
   return((void *)filter_ptr);
}

/**
 *

 *
 * @brief   Generates request for TDT
 *
 * @param   path      - the demux path to use
 * @param   req_type  - ONE_SHOT or CONTINUOUS request
 * @param   callback  - callback function to receive the parsed pat table
 * @param   ret_param - parameter to be returned to callback function
 *
 * @return   filter handle - NULL if filter not successfully setup
 *
 */
void* STB_SIRequestTdt(U8BIT path, E_SI_REQUEST_TYPE req_type,
   void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param)
{
   void *filter_ptr;

   FUNCTION_START(STB_SIRequestTdt);

   #ifdef DEBUG_SI_TDT_REQUEST
   STB_SI_PRINT(("STB_SIRequestTdt: path %d pid 0x%04x", path, SI_TDT_PID));
   #endif

   filter_ptr = STB_SIRequestTable(path, SI_TDT_PID, SI_TDT_TID, 0xff, SINGLE_SECT, 1,
         SI_XTID_MATCH_DONT_CARE, SI_XTID_MASK_DONT_CARE, req_type,
         SI_BUFFER_1K, FALSE, FALSE, callback, ret_param);

   FUNCTION_FINISH(STB_SIRequestTdt);
   return((void *)filter_ptr);
}

/**
 *

 *
 * @brief   Generates request for TDT
 *
 * @param   path      - the demux path to use
 * @param   pid       - the pid of where to request the tdt
 * @param   req_type  - ONE_SHOT or CONTINUOUS request
 * @param   callback  - callback function to receive the parsed pat table
 * @param   ret_param - parameter to be returned to callback function
 *
 * @return   filter handle - NULL if filter not successfully setup
 *
 */
void* STB_SIRequestTdtFromPid(U8BIT path, U16BIT pid, E_SI_REQUEST_TYPE req_type,
   void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param)
{
   void *filter_ptr;

   FUNCTION_START(STB_SIRequestTdtFromPid);

   #ifdef DEBUG_SI_TDT_REQUEST
   STB_SI_PRINT(("STB_SIRequestTdt: path %d pid 0x%04x", path, pid));
   #endif

   filter_ptr = STB_SIRequestTable(path, pid, SI_TDT_TID, 0xff, SINGLE_SECT, 1,
         SI_XTID_MATCH_DONT_CARE, SI_XTID_MASK_DONT_CARE, req_type,
         SI_BUFFER_1K, FALSE, FALSE, callback, ret_param);

   FUNCTION_FINISH(STB_SIRequestTdtFromPid);
   return((void *)filter_ptr);
}

/**
 *

 *
 * @brief   Generates request for TOT
 *
 * @param   path      - the demux path to use
 * @param   req_type  - ONE_SHOT or CONTINUOUS request
 * @param   callback  - callback function to receive the parsed pat table
 * @param   ret_param - parameter to be returned to callback function
 *
 * @return   filter handle - NULL if filter not successfully setup
 *
 */
void* STB_SIRequestTot(U8BIT path, E_SI_REQUEST_TYPE req_type,
   void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param)
{
   void *filter_ptr;

   FUNCTION_START(STB_SIRequestTot);

   #ifdef DEBUG_SI_TOT_REQUEST
   STB_SI_PRINT(("STB_SIRequestTot: path %d pid 0x%04x", path, SI_TOT_PID));
   #endif

   filter_ptr = STB_SIRequestTable(path, SI_TOT_PID, SI_TOT_TID, 0xff, SINGLE_SECT, 1,
         SI_XTID_MATCH_DONT_CARE, SI_XTID_MASK_DONT_CARE, req_type,
         SI_BUFFER_1K, TRUE, FALSE, callback, ret_param);

   FUNCTION_FINISH(STB_SIRequestTot);
   return((void *)filter_ptr);
}

/**
 *

 *
 * @brief   Generates request for TOT
 *
 * @param   path      - the demux path to use
 * @param   req_type  - ONE_SHOT or CONTINUOUS request
 * @param   callback  - callback function to receive the parsed pat table
 * @param   ret_param - parameter to be returned to callback function
 *
 * @return   filter handle - NULL if filter not successfully setup
 *
 */
void* STB_SIRequestTotFromPid(U8BIT path, U16BIT pid, E_SI_REQUEST_TYPE req_type,
   void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param)
{
   void *filter_ptr;

   FUNCTION_START(STB_SIRequestTotFromPid);

   #ifdef DEBUG_SI_TOT_REQUEST
   STB_SI_PRINT(("STB_SIRequestTot: path %d pid 0x%04x", path, pid));
   #endif

   filter_ptr = STB_SIRequestTable(path, pid, SI_TOT_TID, 0xff, SINGLE_SECT, 1,
         SI_XTID_MATCH_DONT_CARE, SI_XTID_MASK_DONT_CARE, req_type,
         SI_BUFFER_1K, TRUE, FALSE, callback, ret_param);

   FUNCTION_FINISH(STB_SIRequestTotFromPid);
   return((void *)filter_ptr);
}

/**
 *

 *
 * @brief   Generates request for CAT
 *
 * @param   path      - the demux path to use
 * @param   req_type  - ONE_SHOT or CONTINUOUS request
 * @param   callback  - callback function to receive the parsed pat table
 * @param   ret_param - parameter to be returned to callback function
 *
 * @return   filter handle - NULL if filter not successfully setup
 *
 */
void* STB_SIRequestCat(U8BIT path, E_SI_REQUEST_TYPE req_type,
   void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param)
{
   void *filter_ptr;

   FUNCTION_START(STB_SIRequestCat);

   #ifdef DEBUG_SI_CAT_REQUEST
   STB_SI_PRINT(("STB_SIRequestCat: path %d pid 0x%04x", path, SI_CAT_PID));
   #endif

   filter_ptr = STB_SIRequestTable(path, SI_CAT_PID, SI_CAT_TID, 0xff, MULTI_SECT, 1,
         SI_XTID_MATCH_DONT_CARE, SI_XTID_MASK_DONT_CARE, req_type,
         SI_BUFFER_4K, TRUE, FALSE, callback, ret_param);

   FUNCTION_FINISH(STB_SIRequestCat);
   return((void *)filter_ptr);
}

/**
 *

 *
 * @brief   Generates request for RCT on given PID
 *
 * @param   path        - the demux path to use
 * @param   req_type    - ONE_SHOT or CONTINUOUS request
 * @param   rct_pid     - pid for the rct table
 * @param   callback    - callback function to receive the parsed pmt table
 * @param   ret_param   - parameter to be returned to callback function
 *
 * @return   filter handle - NULL if filter not successfully setup
 *
 */
void* STB_SIRequestRct(U8BIT path, E_SI_REQUEST_TYPE req_type, U16BIT rct_pid,
   void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param)
{
   void *filter_ptr;

   FUNCTION_START(STB_SIRequestRct);

   #ifdef DEBUG_SI_RCT_REQUEST
   STB_SI_PRINT(("STB_SIRequestRct: path %d pid 0x%04x", path, rct_pid));
   #endif
   filter_ptr = STB_SIRequestTable(path, rct_pid, SI_RCT_TID, 0xff, MULTI_SECT, 1,
         SI_XTID_MATCH_DONT_CARE, SI_XTID_MASK_DONT_CARE, req_type,
         SI_BUFFER_4K, TRUE, FALSE, callback, ret_param);

   FUNCTION_FINISH(STB_SIRequestRct);

   return((void *)filter_ptr);
}

/**
 *

 *
 * @brief   Generates request for UNT
 *
 * @param   path      - the demux path to use
 * @param   req_type  - ONE_SHOT or CONTINUOUS request
 * @param   callback  - callback function to receive the parsed UNT table
 * @param   ret_param - parameter to be returned to callback function
 *
 * @return   filter handle - NULL if filter not successfully setup
 *
 */
void* STB_SIRequestUnt(U8BIT path, E_SI_REQUEST_TYPE req_type,
   void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param)
{
   void *filter_ptr;

   FUNCTION_START(STB_SIRequestUnt);

   #ifdef DEBUG_SI_UNT_REQUEST
   STB_SI_PRINT(("STB_SIRequestUnt: path %d pid 0x%04x", path, SI_UNT_PID));
   #endif

   filter_ptr = STB_SIRequestTable(path, SI_UNT_PID, SI_UNT_TID, 0xff, MULTI_SECT, 1,
         SI_XTID_MATCH_DONT_CARE, SI_XTID_MASK_DONT_CARE, req_type,
         SI_BUFFER_4K, TRUE, FALSE, callback, ret_param);

   FUNCTION_FINISH(STB_SIRequestUnt);
   return((void *)filter_ptr);
}


/**
 * @brief   Create an SI filter for an UNT table
 * @param   path demux path to be used
 * @param   pid PID the table will be available on - this allows the standard PID to be overridden
 * @param   actual TRUE if NITactual is to be retrieved, FALSE for NITother
 * @param   req_type ONE_SHOT_REQUEST or CONTINUOUS_REQUEST
 * @param   callback function to be called when table is received
 * @param   ret_param value that will be passed to the callback function
 * @return  filter handle, NULL if filter isn't created
 */
void* STB_SIRequestUntFromPid(U8BIT path, U16BIT pid, E_SI_REQUEST_TYPE req_type,
                             void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param )
{
   void *filter_ptr;

   FUNCTION_START(STB_SIRequestUntFromPid);

   #ifdef DEBUG_SI_UNT_REQUEST
   STB_SI_PRINT(("STB_SIRequestUntFromPid: path %d pid 0x%04x", path, pid));
   #endif

   filter_ptr = STB_SIRequestTable(path, pid, SI_UNT_TID, 0xff, MULTI_SECT, 1,
         SI_XTID_MATCH_DONT_CARE, SI_XTID_MASK_DONT_CARE, req_type,
         SI_BUFFER_4K, TRUE, FALSE, callback, ret_param);

   FUNCTION_FINISH(STB_SIRequestUntFromPid);

   return(filter_ptr);
}

//--------------------------------------------------------------------------------------------------
// Table parsing functions
//--------------------------------------------------------------------------------------------------

/**
 *

 *
 * @brief   Parses the Pat table supplied in TABLE_RECORD format to create a PAT_TABLE
 *                structure. Returns a pointer to the table. Application must call
 *                STB_SIReleasePatTable to free the data.
 *
 * @param   table_rec - pointer to the table record to be parsed
 *
 * @return   pointer to the parsed pat table
 *
 */
SI_PAT_TABLE* STB_SIParsePatTable(SI_TABLE_RECORD *table_rec)
{
   SI_PAT_TABLE *pat_table;
   SI_SECTION_RECORD *section_rec;
   U8BIT *data_ptr;
   U8BIT *data_end;
   U16BIT prog_no;
   U16BIT pmt_pid;
   SI_PAT_SERVICE_ENTRY *serv_entry;

   FUNCTION_START(STB_SIParsePatTable);

   ASSERT(table_rec != NULL);

   pat_table = NULL;
   if (table_rec != NULL)
   {
      if (table_rec->tid == SI_PAT_TID)
      {
         #ifdef DEBUG_SI_PAT_SUMMARY
         STB_SI_PRINT(("STB_SIParsePatTable: tid 0x%02x/0x%04x, v%d, nsect %d",
                       table_rec->tid, table_rec->xtid, table_rec->version,
                       table_rec->num_sect));
         #endif

         // grab memory for the table
         pat_table = STB_GetMemory(sizeof(SI_PAT_TABLE));
         if (pat_table != NULL)
         {
            memset(pat_table, 0, sizeof(SI_PAT_TABLE));

            pat_table->version = table_rec->version;
            pat_table->tran_id = table_rec->xtid;

            section_rec = table_rec->section_list;
            while (section_rec != NULL)
            {
               #ifdef DEBUG_SI_PAT_CONTENT
               STB_SI_PRINT((" section %d", section_rec->sect_num));
               #endif

               // get pointer to section data and end of section
               data_ptr = &(section_rec->data_start);
               data_end = data_ptr + section_rec->data_len - 4;   // -4 for crc

               // skip section header
               data_ptr += 8;

               while (data_ptr < data_end)
               {
                  // read next entry from section - if it is not 0 (nit pid) and it is not a
                  // duplicate of an entry we have already had add it to the pat table
                  prog_no = (data_ptr[0] << 8) | data_ptr[1];
                  pmt_pid = ((data_ptr[2] & 0x1f) << 8) | data_ptr[3];
                  data_ptr += 4;

                  if (prog_no != 0)
                  {
                     // check if this is a duplicate or new entry
                     serv_entry = pat_table->service_list;
                     while (serv_entry != NULL)
                     {
                        if ((serv_entry->serv_id == prog_no) && (serv_entry->pmt_pid == pmt_pid))
                        {
                           #ifdef DEBUG_SI_PAT_CONTENT
                           STB_SI_PRINT(("  sid=0x%04x, pmt=0x%04x - duplicated", prog_no,
                                         pmt_pid));
                           #endif
                           break;
                        }
                        serv_entry = serv_entry->next;
                     }
                     if (serv_entry == NULL)
                     {
                        // not found - create a new entry
                        serv_entry = STB_GetMemory(sizeof(SI_PAT_SERVICE_ENTRY));
                        if (serv_entry != NULL)
                        {
                           serv_entry->next = NULL;
                           serv_entry->serv_id = prog_no;
                           serv_entry->pmt_pid = pmt_pid;

                           // add to the end of the entry list in the pat table
                           if (pat_table->last_service_entry == NULL)
                           {
                              // first entry in the list
                              pat_table->service_list = serv_entry;
                           }
                           else
                           {
                              // not the first entry
                              pat_table->last_service_entry->next = serv_entry;
                           }
                           pat_table->last_service_entry = serv_entry;
                           pat_table->num_services++;

                           #ifdef DEBUG_SI_PAT_CONTENT
                           STB_SI_PRINT(("  sid=0x%04x, pmt=0x%04x", prog_no, pmt_pid));
                           #endif
                        }
                     }
                  }
               }
               // move on to next section
               section_rec = section_rec->next;
            }
         }
      }
   }

   FUNCTION_FINISH(STB_SIParsePatTable);
   return(pat_table);
}

/**
 *

 *
 * @brief   Parses the Pmt table supplied in TABLE_RECORD format to create a PMT_TABLE
 *                structure. Returns a pointer to the table. Application must call
 *                STB_SIReleasePmtTable to free the data.
 *
 * @param   table_rec - pointer to the table record to be parsed
 *
 * @return   pointer to the parsed pmt table
 *
 */
SI_PMT_TABLE* STB_SIParsePmtTable(SI_TABLE_RECORD *table_rec)
{
   SI_PMT_TABLE *pmt_table;
   SI_SECTION_RECORD *section_rec;
   U8BIT *data_ptr;
   U8BIT *data_end;
   U16BIT dloop_len;
   U8BIT *dloop_end;
   U8BIT dtag;
   SI_PMT_STREAM_ENTRY *stream_entry;
   U32BIT priv_data_code;

   FUNCTION_START(STB_SIParsePmtTable);

   ASSERT(table_rec != NULL);

   pmt_table = NULL;
   if (table_rec != NULL)
   {
      if (table_rec->tid == SI_PMT_TID)
      {
         #ifdef DEBUG_SI_PMT_SUMMARY
         STB_SI_PRINT(("STB_SIParsePmtTable: tid 0x%02x/0x%04x, v%d, nsect %d",
                       table_rec->tid, table_rec->xtid, table_rec->version,
                       table_rec->num_sect));
         #endif

         pmt_table = STB_GetMemory(sizeof(SI_PMT_TABLE));
         if (pmt_table != NULL)
         {
            memset(pmt_table, 0, sizeof(SI_PMT_TABLE));

            pmt_table->version = table_rec->version;
            pmt_table->serv_id = table_rec->xtid;

            section_rec = table_rec->section_list;
            while (section_rec != NULL)
            {
               #ifdef DEBUG_SI_PMT_CONTENT
               STB_SI_PRINT((" section %d", section_rec->sect_num));
               #endif

               // get pointer to section data and end of section
               data_ptr = &(section_rec->data_start);
               data_end = data_ptr + section_rec->data_len - 4;   // -4 for crc

               // skip section header
               data_ptr += 8;

               // get pcr pid and descriptor loop length
               pmt_table->pcr_pid = ((data_ptr[0] & 0x1f) << 8) | data_ptr[1];
               dloop_len = ((data_ptr[2] & 0x0f) << 8) | data_ptr[3];
               data_ptr += 4;

               #ifdef DEBUG_SI_PMT_CONTENT
               STB_SI_PRINT(("  pcr pid=0x%04x (first loop len %d)", pmt_table->pcr_pid,
                             dloop_len));
               STB_SI_PRINT(("  {"));
               #endif

               /* According to Nordig v2.5.1, the default setting for content protection is 0x01,
                * which means content protection isn't required and the content protectionxi
                * mechanism, e.g. HDCP, can be either on or off */
               pmt_table->content_protection_level = 0x01;

               // process first descriptor loop
               priv_data_code = 0;
               dloop_end = data_ptr + dloop_len;
               while (data_ptr < dloop_end)
               {
                  dtag = data_ptr[0];
                  data_ptr++;
                  #ifdef DEBUG_SI_PMT_CONTENT
                  if (dtag != 00)
                  {
                     STB_SI_PRINT(("   tag %02x len %04x", dtag, *data_ptr));
                  }
                  #endif

                  switch (dtag)
                  {
                     case CA_DTAG:
                     {
                        data_ptr = ParseCaDescriptor(data_ptr, &pmt_table->num_ca_entries,
                              &pmt_table->ca_desc_array, DEBUG_SI_PMT_CONTENT_VALUE);
                        break;
                     }

                     case PRIV_DATA_INDICATOR_DTAG:
                     {
                        /* In Freesat when there is an indicator it should end the scope of the
                         * private data specifier */
                        data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_PMT_CONTENT_VALUE);
                        priv_data_code = 0;
                        break;
                     }

                     case PRIV_DATA_SPEC_DTAG:
                     {
                        data_ptr = ParsePrivateDataSpecifierDescriptor(data_ptr, &priv_data_code, DEBUG_SI_PMT_CONTENT_VALUE);
                        break;
                     }

                     case FREESAT_TUNNELLED_DTAG:
                     {
                        if ((priv_data_code == FREESAT_PRIVATE_DATA_SPECIFIER) &&
                            (freesat_private_data_specifier == TRUE))
                        {
                           data_ptr = ParseFreesatTunnelledDataDescriptor(data_ptr,
                                 &pmt_table->num_tunnelled_entries,
                                 &pmt_table->tunnelled_desc_array,
                                 FALSE, DEBUG_SI_PMT_CONTENT_VALUE);
                        }
                        else
                        {
                           data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_PMT_CONTENT_VALUE);
                        }
                        break;
                     }

                     case USER_DEFINED_DTAG_0xA0:
                     {
                        if (nordig_private_data_specifier)
                        {
                           /* This is the Nordig content protection descriptor */
                           data_ptr = ParseNordigContentProtectionDesc(data_ptr,
                              &pmt_table->content_protection_level, DEBUG_SI_PMT_CONTENT_VALUE);
                        }
                        else
                        {
                           data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_PMT_CONTENT_VALUE);
                        }
                        break;
                     }

                     default:
                     {
                        data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_PMT_CONTENT_VALUE);
                        break;
                     }
                  }
               }
               #ifdef DEBUG_SI_PMT_CONTENT
               STB_SI_PRINT(("  }"));
               #endif

               // read entry for each stream
               while (data_ptr < data_end)
               {
                  stream_entry = (SI_PMT_STREAM_ENTRY *)STB_GetMemory(sizeof(SI_PMT_STREAM_ENTRY));
                  if (stream_entry != NULL)
                  {
                     // initialise new stream structure
                     memset(stream_entry, 0, sizeof(SI_PMT_STREAM_ENTRY));

                     // add to the end of the stream list in the pmt table
                     if (pmt_table->last_stream_entry == NULL)
                     {
                        // first entry in the list
                        pmt_table->stream_list = stream_entry;
                     }
                     else
                     {
                        // not the first entry
                        pmt_table->last_stream_entry->next = stream_entry;
                     }
                     pmt_table->last_stream_entry = stream_entry;
                     pmt_table->num_streams++;

                     // read stream data and descriptor loop length
                     stream_entry->type = data_ptr[0];
                     stream_entry->pid = ((data_ptr[1] & 0x1f) << 8) | data_ptr[2];
                     dloop_len = ((data_ptr[3] & 0x0f) << 8) | data_ptr[4];
                     data_ptr += 5;

                     stream_entry->carousel_id = DVB_INVALID_CAROUSEL_ID;

                     #ifdef DEBUG_SI_PMT_CONTENT
                     STB_SI_PRINT(("  type=0x%02x, pid=0x%02x (desc loop len %d)",
                                   stream_entry->type, stream_entry->pid, dloop_len));
                     STB_SI_PRINT(("  {"));
                     #endif

                     // process stream descriptor loop
                     priv_data_code = 0;
                     dloop_end = data_ptr + dloop_len;
                     while ((data_ptr < dloop_end) && (data_ptr < data_end))
                     {
                        dtag = data_ptr[0];
                        data_ptr++;
                        #ifdef DEBUG_SI_PMT_CONTENT
                        if (dtag > 0)
                        {
                           STB_SI_PRINT(("   tag %02x len %04x", dtag, *data_ptr));
                        }
                        #endif

                        switch (dtag)
                        {
                           case CAROUSEL_ID_DTAG:
                           {
                              if ((stream_entry->type == SI_STREAM_TYPE_DATA_B) ||
                                  (stream_entry->type == SI_STREAM_TYPE_DATA_C) ||
                                  (stream_entry->type == SI_STREAM_TYPE_DATA_D))
                              {
                                 data_ptr = ParseCarouselIdDescriptor(data_ptr,
                                    &stream_entry->carousel_id, DEBUG_SI_PMT_CONTENT_VALUE);
                              }
                              else
                              {
                                 data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_PMT_CONTENT_VALUE);
                              }
                              break;
                           }

                           case CA_DTAG:
                           {
                              data_ptr = ParseCaDescriptor(data_ptr, &stream_entry->num_ca_entries,
                                    &stream_entry->ca_desc_array, DEBUG_SI_PMT_CONTENT_VALUE);
                              break;
                           }

                           case ISO_LANG_DTAG:
                           {
                              data_ptr = ParseIsoLangDescriptor(data_ptr,
                                    &stream_entry->num_iso_lang_entries,
                                    &stream_entry->iso_lang_desc_array, DEBUG_SI_PMT_CONTENT_VALUE);
                              break;
                           }

                           case STREAM_ID_DTAG:
                           {
                              data_ptr = ParseStreamIdDescriptor(data_ptr, &stream_entry->tag_array_ptr,
                                    &stream_entry->num_tag_entries, DEBUG_SI_PMT_CONTENT_VALUE);
                              break;
                           }

                           case SUBTITLE_DTAG:
                           {
                              data_ptr = ParseSubtitleDescriptor(data_ptr,
                                    &stream_entry->num_subtitle_entries,
                                    &stream_entry->subtitle_desc_array, DEBUG_SI_PMT_CONTENT_VALUE);
                              break;
                           }

                           case TELETEXT_DTAG:
                           {
                              data_ptr = ParseTeletextDescriptor(data_ptr,
                                    &stream_entry->num_teletext_entries,
                                    &stream_entry->teletext_desc_array, DEBUG_SI_PMT_CONTENT_VALUE);
                              break;
                           }

                           case PRIV_DATA_INDICATOR_DTAG:
                           {
                              /* In Freesat when there is an indicator it should end the scope of the
                               * private data specifier */
                              data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_PMT_CONTENT_VALUE);
                              priv_data_code = 0;
                              break;
                           }

                           case PRIV_DATA_SPEC_DTAG:
                           {
                              data_ptr = ParsePrivateDataSpecifierDescriptor(data_ptr,
                                    &priv_data_code, DEBUG_SI_PMT_CONTENT_VALUE);
                              break;
                           }

                           case AC3_DTAG:
                           case EAC3_DTAG:
                           {
                              stream_entry->tag_array_ptr = (U8BIT *)STB_GetMemory(sizeof(U8BIT));
                              stream_entry->tag_array_ptr[0] = dtag;
                              stream_entry->num_tag_entries = 1;
                              data_ptr = ParseAC3Descriptor(data_ptr, dtag,
                                    &stream_entry->ac3_descriptor, DEBUG_SI_PMT_CONTENT_VALUE);
                              break;
                           }

                           case AAC_DTAG:
                           {
                              data_ptr = ParseAACDescriptor(data_ptr,
                                    &stream_entry->aac_descriptor, DEBUG_SI_PMT_CONTENT_VALUE);
                              break;
                           }

                           case EXT_DTAG:
                           {
                              /* Examine the extension tag for the table being signalled */
                              switch (data_ptr[1])
                              {
                                 case SUPPLEMENTARY_AUDIO_DTAG:
                                    data_ptr = ParseSupplementaryAudioDescriptor(data_ptr,
                                          &stream_entry->audio_desc, DEBUG_SI_PMT_CONTENT_VALUE);
                                    break;

                                 default:
                                    data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_PMT_CONTENT_VALUE);
                                    break;
                              }
                              break;
                           }

                           case FREESAT_ALT_TUNNELLED_DTAG:
                           {
                              if ((priv_data_code == FREESAT_PRIVATE_DATA_SPECIFIER) &&
                                  (freesat_private_data_specifier == TRUE) &&
                                  ((stream_entry->type == SI_STREAM_TYPE_PRIVATE) ||
                                   (stream_entry->type == SI_STREAM_TYPE_DATA_B)))
                              {
                                 data_ptr = ParseFreesatTunnelledDataDescriptor(data_ptr,
                                       &stream_entry->num_tunnelled_entries,
                                       &stream_entry->tunnelled_desc_array,
                                       TRUE, DEBUG_SI_PMT_CONTENT_VALUE);
                              }
                              else
                              {
                                 data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_PMT_CONTENT_VALUE);
                              }
                              break;
                           }

#if 0
                           case SERVICE_MOVE_DTAG:
                           {
                              data_ptr = ParseServiceMoveDescriptor(data_ptr,
                                    &stream_entry->service_move, DEBUG_SI_PMT_CONTENT_VALUE);
                              break;
                           }
#endif

                           case APP_SIG_DTAG:
                           {
                              /* The application signalling descriptor is only
                               * valid for private stream types */
                              if (stream_entry->type == SI_STREAM_TYPE_PRIVATE)
                              {
                                 /* Even if the descriptor doesn't actually contain any info, it indicates
                                  * that this is the PID to be used to collect the AIT table */
                                 stream_entry->has_ait = TRUE;

                                 data_ptr = ParseAppSignallingDescriptor(data_ptr,
                                       &stream_entry->num_app_sig_entries, &stream_entry->app_sig_desc_array, DEBUG_SI_PMT_CONTENT_VALUE);
                              }
                              else
                              {
                                 /* Descriptor shouldn't be here so skip it */
                                 data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_PMT_CONTENT_VALUE);
                              }
                              break;
                           }

                           case RELATED_CONTENT_DTAG:
                           {
                              /* The related content descriptor is only
                               * valid for private stream types */
                              if (stream_entry->type == SI_STREAM_TYPE_PRIVATE)
                              {
                                 /* The descriptor doesn't actually contain any info, it just indicates
                                  * that this is the PID to be used to collect the RCT table */
                                 stream_entry->has_rct = TRUE;
                              }

                              /* Ensure the descriptor is skipped */
                              data_ptr = SkipDescriptor(data_ptr, dtag, FALSE);
                              break;
                           }

                           default:
                           {
                              #ifdef DEBUG_SI_PMT_CONTENT
                              if (dtag == 0x6f)
                              {
                                 U8BIT *dptr = data_ptr + 1;
                                 U8BIT len;

                                 STB_SI_PRINT(("  loop2: dtag 0x%02x, len %u\n", dtag, *data_ptr));
                                 for (len = 0; len < *data_ptr; dptr++, len++)
                                 {
                                    STB_SI_PRINT(("    %02x", *dptr));
                                 }
                              }
                              #endif
                              data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_PMT_CONTENT_VALUE);
                              break;
                           }
                        }
                     }
                     #ifdef DEBUG_SI_PMT_CONTENT
                     STB_SI_PRINT(("  }"));
                     #endif
                  }
               }
               // move on to next section
               section_rec = section_rec->next;
            }
         }
      }
   }
   FUNCTION_FINISH(STB_SIParsePmtTable);
   return(pmt_table);
}

/**
 *

 *
 * @brief   Parses the Nit table supplied in TABLE_RECORD format to create a NIT_TABLE
 *                structure. Returns a pointer to the table. Application must call
 *                STB_SIReleaseNitTable to free the data.
 *
 * @param   table_rec - pointer to the table record to be parsed
 *
 * @return   pointer to the parsed nit table
 *
 */
SI_NIT_TABLE* STB_SIParseNitTable(SI_TABLE_RECORD *table_rec)
{
   SI_NIT_TABLE *nit_table;
   SI_SECTION_RECORD *section_rec;
   U8BIT *data_ptr;
   U8BIT *data_end;
   U16BIT dloop_len;
   U8BIT *dloop_end;
   U8BIT dtag;
   U32BIT priv_data_code;
   SI_NIT_TRANSPORT_ENTRY *trans_entry;

   FUNCTION_START(STB_SIParseNitTable);

   ASSERT(table_rec != NULL);

   nit_table = NULL;
   if (table_rec != NULL)
   {
      if ((table_rec->tid == SI_NIT_ACTUAL_TID) || (table_rec->tid == SI_NIT_OTHER_TID))
      {
         #ifdef DEBUG_SI_NIT_SUMMARY
         STB_SI_PRINT(("STB_SIParseNitTable: tid 0x%02x/0x%04x, v%d, nsect %d",
                       table_rec->tid, table_rec->xtid, table_rec->version, table_rec->num_sect));
         #endif

         nit_table = STB_GetMemory(sizeof(SI_NIT_TABLE));
         if (nit_table != NULL)
         {
            memset(nit_table, 0, sizeof(SI_NIT_TABLE));

            nit_table->version = table_rec->version;
            nit_table->net_id = table_rec->xtid;

            section_rec = table_rec->section_list;
            while (section_rec != NULL)
            {
               #ifdef DEBUG_SI_NIT_CONTENT
               STB_SI_PRINT((" section %d", section_rec->sect_num));
               #endif

               // get pointer to section data and end of section
               data_ptr = &(section_rec->data_start);
               data_end = data_ptr + section_rec->data_len - 4;   // -4 for crc

               // skip section header
               data_ptr += 8;

               // read first descriptor loop length
               dloop_len = ((data_ptr[0] & 0x0f) << 8) | data_ptr[1];
               data_ptr += 2;

               #ifdef DEBUG_SI_NIT_CONTENT
               STB_SI_PRINT(("  (first loop len %d)", dloop_len));
               #endif

               // process first (network) descriptor loop
               priv_data_code = 0;
               dloop_end = data_ptr + dloop_len;
               while (data_ptr < dloop_end)
               {
                  dtag = data_ptr[0];

                  #ifdef DEBUG_SI_NIT_CONTENT
                  STB_SI_PRINT(("  dtag=0x%02x", (int)dtag));
                  #endif

                  data_ptr++;
                  switch (dtag)
                  {
                     case NET_NAME_DTAG:
                     {
                        data_ptr = ParseNetNameDescriptor(data_ptr, &nit_table->name_str, DEBUG_SI_NIT_CONTENT_VALUE);
                        break;
                     }

                     case MULTILING_NET_NAME_DTAG:
                     {
                        data_ptr = ParseMultilingNetNameDescriptor(data_ptr,
                              &nit_table->num_multiling_net_names,
                              &nit_table->multiling_net_name_desc_array, DEBUG_SI_NIT_CONTENT_VALUE);
                        break;
                     }

                     case LINKAGE_DTAG:
                     {
                        data_ptr = ParseLinkageDescriptor(data_ptr,
                              &nit_table->num_linkage_entries,
                              &nit_table->linkage_desc_list,
                              &nit_table->last_linkage_entry, priv_data_code,
                              DEBUG_SI_NIT_CONTENT_VALUE);
                        break;
                     }

                     case PRIV_DATA_SPEC_DTAG:
                     {
                        data_ptr = ParsePrivateDataSpecifierDescriptor(data_ptr, &priv_data_code, DEBUG_SI_NIT_CONTENT_VALUE);
                        break;
                     }

                     case DEF_AUTH_DTAG:
                     {
                        data_ptr = ParseDefaultAuthorityDescriptor(data_ptr, &nit_table->def_authority, DEBUG_SI_NIT_CONTENT_VALUE);
                        break;
                     }

                     case FTA_CONTENT_DTAG:
                     {
                        data_ptr = ParseFTAContentDescriptor(data_ptr, &nit_table->fta_content_desc, DEBUG_SI_NIT_CONTENT_VALUE);
                        break;
                     }

                     case FREESAT_LINK_DTAG: /* or CIPLUS_VIRTUAL_CHAN_DTAG */
                     {
                        if ((priv_data_code == FREESAT_PRIVATE_DATA_SPECIFIER) &&
                            (freesat_private_data_specifier == TRUE))
                        {
                           data_ptr = ParseFreesatLinkageDescriptor(data_ptr,
                                 &nit_table->freesat_linkage_desc, DEBUG_SI_NIT_CONTENT_VALUE);
                        }
                        else if ((priv_data_code == CIPLUS_PRIVATE_DATA_SPECIFIER) &&
                                  (ciplus_private_data_specifier == TRUE))
                        {
                           data_ptr = ParseCIPlusVirtualChannelDescriptor(data_ptr,
                                 &nit_table->ciplus_virtual_channel, DEBUG_SI_NIT_CONTENT_VALUE);
                        }
                        else
                        {
                           data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_NIT_CONTENT_VALUE);
                        }
                        break;
                     }

                     case FREESAT_PREFIX_DTAG:
                     {
                        if ((priv_data_code == FREESAT_PRIVATE_DATA_SPECIFIER) &&
                            (freesat_private_data_specifier == TRUE))
                        {
                           data_ptr = ParseFreesatPrefixDescriptor(data_ptr,
                                 &nit_table->freesat_prefix_list, DEBUG_SI_NIT_CONTENT_VALUE);
                        }
                        else
                        {
                           data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_NIT_CONTENT_VALUE);
                        }
                        break;
                     }

                     case EXT_DTAG:
                     {
                        /* Examine the extension tag for the table being signalled */
                        switch (data_ptr[1])
                        {
                           case NETWORK_CHANGE_NOTIFY_DTAG:
                              data_ptr = ParseNetworkChangeNotifyDescriptor(data_ptr,
                                    &nit_table->num_change_notifies,
                                    &nit_table->change_notify_array, DEBUG_SI_NIT_CONTENT_VALUE);
                              break;

                           case MESSAGE_DTAG:
                              data_ptr = ParseMessageDescriptor(data_ptr,
                                    &nit_table->num_messages,
                                    &nit_table->message_array, DEBUG_SI_NIT_CONTENT_VALUE);
                              break;

                           case TARGET_REGION_NAME_DTAG:
                              data_ptr = ParseTargetRegionNameDescriptor(data_ptr,
                                    &nit_table->target_region_name_list, DEBUG_SI_NIT_CONTENT_VALUE);
                              break;

                           case TARGET_REGION_DTAG:
                              data_ptr = ParseTargetRegionDescriptor(data_ptr,
                                    &nit_table->target_region_list, DEBUG_SI_NIT_CONTENT_VALUE);
                              break;

                           case URI_LINKAGE_DTAG:
                              data_ptr = ParseURILinkageDesc(data_ptr,
                                 &nit_table->uri_linkage_list, priv_data_code, DEBUG_SI_NIT_CONTENT_VALUE);
                              break;

                           default:
                              data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_NIT_CONTENT_VALUE);
                              break;
                        }
                        break;
                     }

                     default:
                     {
                        data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_NIT_CONTENT_VALUE);
                        break;
                     }
                  }
               }

               // skip transport stream loop length - not needed because we keep reading transports
               // until we reach the end of the table
               data_ptr += 2;

               // read entry for each transport
               while (data_ptr < data_end)
               {
                  trans_entry = (SI_NIT_TRANSPORT_ENTRY *)STB_GetMemory(sizeof(SI_NIT_TRANSPORT_ENTRY));
                  if (trans_entry != NULL)
                  {
                     // initialise new transport structure
                     memset(trans_entry, 0, sizeof(SI_NIT_TRANSPORT_ENTRY));

                     // add to the end of the stream list in the nit table
                     if (nit_table->last_transport_entry == NULL)
                     {
                        // first entry in the list
                        nit_table->transport_list = trans_entry;
                     }
                     else
                     {
                        // not the first entry
                        nit_table->last_transport_entry->next = trans_entry;
                     }
                     nit_table->last_transport_entry = trans_entry;
                     nit_table->num_transports++;

                     // read stream data and descriptor loop length
                     trans_entry->tran_id = (data_ptr[0] << 8) | data_ptr[1];
                     trans_entry->orig_net_id = (data_ptr[2] << 8) | data_ptr[3];
                     dloop_len = ((data_ptr[4] & 0x0f) << 8) | data_ptr[5];
                     data_ptr += 6;

                     #ifdef DEBUG_SI_NIT_CONTENT
                     STB_SI_PRINT(("  tid=0x%04x, onid=0x%04x (desc loop len %d)",
                                   trans_entry->tran_id, trans_entry->orig_net_id, dloop_len));
                     #endif

                     // process transport descriptor loop
                     priv_data_code = 0;
                     dloop_end = data_ptr + dloop_len;
                     while (data_ptr < dloop_end)
                     {
                        dtag = data_ptr[0];

                        #ifdef DEBUG_SI_NIT_CONTENT
                        STB_SI_PRINT(("  dtag=0x%02x", (int)dtag));
                        #endif

                        data_ptr++;
                        switch (dtag)
                        {
                           case FREQ_LIST_DTAG:
                           {
                              data_ptr = ParseFrequencyListDescriptor(data_ptr,
                                    &trans_entry->freq_list, DEBUG_SI_NIT_CONTENT_VALUE);
                              break;
                           }

                           case SERV_LIST_DTAG:
                           {
                              data_ptr = ParseServiceListDescriptor(data_ptr,
                                    &trans_entry->num_serv_list_entries,
                                    &trans_entry->serv_list_desc_array, DEBUG_SI_NIT_CONTENT_VALUE);
                              break;
                           }

                           case TERR_DEL_SYS_DTAG:
                           {
                              trans_entry->del_sys_desc_type = SI_DEL_SYS_DESC_TYPE_TERR;

                              data_ptr = ParseTerrestrialDeliverySysDescriptor(data_ptr,
                                    &trans_entry->del_sys_desc, DEBUG_SI_NIT_CONTENT_VALUE);
                              break;
                           }

                           case SAT_DEL_SYS_DTAG:
                           {
                              trans_entry->del_sys_desc_type = SI_DEL_SYS_DESC_TYPE_SAT;

                              data_ptr = ParseSatelliteDeliverySysDescriptor(data_ptr,
                                    &trans_entry->del_sys_desc, DEBUG_SI_NIT_CONTENT_VALUE);
                              break;
                           }

                           case CABLE_DEL_SYS_DTAG:
                           {
                              trans_entry->del_sys_desc_type = SI_DEL_SYS_DESC_TYPE_CABLE;

                              data_ptr = ParseCableDeliverySysDescriptor(data_ptr,
                                    &trans_entry->del_sys_desc, DEBUG_SI_NIT_CONTENT_VALUE);
                              break;
                           }

                           case PRIV_DATA_SPEC_DTAG:
                           {
                              data_ptr = ParsePrivateDataSpecifierDescriptor(data_ptr,
                                    &priv_data_code, DEBUG_SI_NIT_CONTENT_VALUE);
                              break;
                           }

                           case DEF_AUTH_DTAG:
                           {
                              data_ptr = ParseDefaultAuthorityDescriptor(data_ptr,
                                    &trans_entry->def_authority, DEBUG_SI_NIT_CONTENT_VALUE);
                              break;
                           }

                           case FTA_CONTENT_DTAG:
                           {
                              data_ptr = ParseFTAContentDescriptor(data_ptr, &trans_entry->fta_content_desc, DEBUG_SI_NIT_CONTENT_VALUE);
                              break;
                           }

                           case EXT_DTAG:
                           {
                              /* Examine the extension tag for the table being signalled */
                              switch (data_ptr[1])
                              {
                                 case T2_DELIVERY_SYS_DTAG:
                                    trans_entry->del_sys_desc_type = SI_DEL_SYS_DESC_TYPE_TERR;
                                    data_ptr = ParseT2DeliverySysDescriptor(data_ptr,
                                          &trans_entry->del_sys_desc, DEBUG_SI_NIT_CONTENT_VALUE);
                                    break;

                                 case TARGET_REGION_DTAG:
                                    data_ptr = ParseTargetRegionDescriptor(data_ptr,
                                          &trans_entry->target_region_list, DEBUG_SI_NIT_CONTENT_VALUE);
                                    break;

                                 default:
                                    data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_NIT_CONTENT_VALUE);
                                    break;
                              }
                              break;
                           }

                           case USER_DEFINED_DTAG_0x83:
                           {
                              /* For UK DTT this is logical chan number and private data
                               * descriptor will have been set. This will also be true for
                               * some other countries that comply with the E-Book, but in
                               * this case the descriptor is almost identical except for the
                               * inclusion of a 1-bit 'visible service' flag. And finally, some
                               * other countries use this descriptor tag but don't set a private
                               * data specifier, so in this case the app will have set the user
                               * defined descriptor function */
                              if (((priv_data_code == UK_DTT_PRIVATE_DATA_SPECIFIER) &&
                                   (priv_data_code == country_private_data_specifier_code)) ||
                                  ((priv_data_code == ZAF_DTT_PRIVATE_DATA_SPECIFIER) &&
                                   (priv_data_code == country_private_data_specifier_code)) ||
                                  ((priv_data_code == NORDIG_PRIVATE_DATA_SPECIFIER) &&
                                   (nordig_private_data_specifier == TRUE)) ||
                                  (GetUserDefinedDescriptorFunction(dtag) == USER_DEF_DESCRIP_LOGICAL_CHAN_NUM))
                              {
                                 data_ptr = ParseLogicalChannelDescriptor(data_ptr, dtag,
                                       &trans_entry->num_lcn_entries,
                                       &trans_entry->lcn_desc_array, DEBUG_SI_NIT_CONTENT_VALUE);
                              }
                              else
                              {
                                 data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_NIT_CONTENT_VALUE);
                              }
                              break;
                           }

                           case USER_DEFINED_DTAG_0x86:
                           {
                              if (((priv_data_code == UK_DTT_PRIVATE_DATA_SPECIFIER) &&
                                   (priv_data_code == country_private_data_specifier_code)) ||
                                  ((priv_data_code == ZAF_DTT_PRIVATE_DATA_SPECIFIER) &&
                                   (priv_data_code == country_private_data_specifier_code)))
                              {
                                 data_ptr = ParseServiceAttributeDescriptor(data_ptr,
                                       &trans_entry->num_serv_attr_entries,
                                       &trans_entry->serv_attr_array, DEBUG_SI_NIT_CONTENT_VALUE);
                              }
                              else
                              {
                                 data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_NIT_CONTENT_VALUE);
                              }
                              break;
                           }

                           case USER_DEFINED_DTAG_0x87:
                           {
                              if(((priv_data_code == NORDIG_PRIVATE_DATA_SPECIFIER) &&
                                  (nordig_private_data_specifier == TRUE)) ||
                                  (priv_data_code == country_private_data_specifier_code) ||
                                 ((priv_data_code == NZ_SAT_PRIVATE_DATA_SPECIFIER) &&
                                  (nzsat_private_data_specifier == TRUE)))
                              {
                                 /* Nordig LCN descriptor, version 2 */
                                 data_ptr = ParseNordigLCN2Descriptor(data_ptr,
                                       &trans_entry->num_nordig_lcn_entries,
                                       &trans_entry->nordig_lcn_desc_array, DEBUG_SI_NIT_CONTENT_VALUE);
                              }
                              else
                              {
                                 data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_NIT_CONTENT_VALUE);
                              }
                              break;
                           }

                           case USER_DEFINED_DTAG_0x88:
                           {
                              /* In the E-Book this is defined as the HD simulcast LCN descriptor,
                               * but the definition is identical to the normal LCN descriptor. */
                              if (((priv_data_code == UK_DTT_PRIVATE_DATA_SPECIFIER) &&
                                   (priv_data_code == country_private_data_specifier_code)) ||
                                  ((priv_data_code == EACEM_PRIVATE_DATA_SPECIFIER) &&
                                   (eacem_private_data_specifier == TRUE)))
                              {
                                 data_ptr = ParseLogicalChannelDescriptor(data_ptr, dtag,
                                       &trans_entry->num_hd_lcn_entries,
                                       &trans_entry->hd_lcn_desc_array, DEBUG_SI_NIT_CONTENT_VALUE);
                              }
                              else
                              {
                                 data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_NIT_CONTENT_VALUE);
                              }
                              break;
                           }

                           case CIPLUS_SERVICE_DTAG:
                           {
                              if ((priv_data_code == CIPLUS_PRIVATE_DATA_SPECIFIER) &&
                                  (ciplus_private_data_specifier == TRUE))
                              {
                                 data_ptr = ParseCIPlusServiceDescriptor(data_ptr,
                                       &trans_entry->num_ciplus_services,
                                       &trans_entry->ciplus_service_list,
                                       &trans_entry->last_ciplus_service, DEBUG_SI_NIT_CONTENT_VALUE);
                              }
                              else
                              {
                                 data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_NIT_CONTENT_VALUE);
                              }
                              break;
                           }

                           default:
                           {
                              data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_NIT_CONTENT_VALUE);
                              break;
                           }
                        }
                     }
                  }
               }
               // move on to next section
               section_rec = section_rec->next;
            }
         }
      }
   }
   FUNCTION_FINISH(STB_SIParseNitTable);
   return(nit_table);
}

/**
 *

 *
 * @brief   Parses the Sdt table supplied in TABLE_RECORD format to create a SDT_TABLE
 *                structure. Returns a pointer to the table. Application must call
 *                STB_SIReleaseSdtTable to free the data.
 *
 * @param   table_rec - pointer to the table record to be parsed
 *
 * @return   pointer to the parsed sdt table
 *
 */
SI_SDT_TABLE* STB_SIParseSdtTable(SI_TABLE_RECORD *table_rec)
{
   SI_SDT_TABLE *sdt_table;
   SI_SECTION_RECORD *section_rec;
   U8BIT *data_ptr;
   U8BIT *data_end;
   U16BIT dloop_len;
   U8BIT *dloop_end;
   U8BIT dtag;
   SI_SDT_SERVICE_ENTRY *serv_entry;
   U32BIT priv_data_code;


   FUNCTION_START(STB_SIParseSdtTable);

   ASSERT(table_rec != NULL);

   sdt_table = NULL;
   if (table_rec != NULL)
   {
      if ((table_rec->tid == SI_SDT_ACTUAL_TID) || (table_rec->tid == SI_SDT_OTHER_TID))
      {
         #ifdef DEBUG_SI_SDT_SUMMARY
         STB_SI_PRINT(("STB_SIParseSdtTable: tid 0x%02x/0x%04x, v%d, nsect %d",
                       table_rec->tid, table_rec->xtid, table_rec->version, table_rec->num_sect));
         #endif


         sdt_table = STB_GetMemory(sizeof(SI_SDT_TABLE));
         if (sdt_table != NULL)
         {
            memset(sdt_table, 0, sizeof(SI_SDT_TABLE));

            sdt_table->version = table_rec->version;
            sdt_table->tran_id = table_rec->xtid;

            section_rec = table_rec->section_list;
            while (section_rec != NULL)
            {
               // get pointer to section data and end of section
               data_ptr = &(section_rec->data_start);
               data_end = data_ptr + section_rec->data_len - 4;   // -4 for crc

               // skip section header
               data_ptr += 8;

               // read original network id. If there is more than omne section all sections should
               // contain the same value, so we will allow later sections to overwrite the value
               // i.e. the value in the last section will be used
               sdt_table->orig_net_id = (data_ptr[0] << 8) | data_ptr[1];
               data_ptr += 3;    // move past onid field and unused reserved byte

               #ifdef DEBUG_SI_SDT_CONTENT
               STB_SI_PRINT((" section %d, onid=0x%04x",
                             section_rec->sect_num, sdt_table->orig_net_id));
               #endif

               // read entry for each service
               while (data_ptr < data_end)
               {
                  serv_entry = (SI_SDT_SERVICE_ENTRY *)STB_GetMemory(sizeof(SI_SDT_SERVICE_ENTRY));
                  if (serv_entry != NULL)
                  {
                     // initialise new service structure
                     memset(serv_entry, 0, sizeof(SI_SDT_SERVICE_ENTRY));

                     // add to the end of the stream list in the sdt table
                     if (sdt_table->last_service_entry == NULL)
                     {
                        // first entry in the list
                        sdt_table->service_list = serv_entry;
                     }
                     else
                     {
                        // not the first entry
                        sdt_table->last_service_entry->next = serv_entry;
                     }
                     sdt_table->last_service_entry = serv_entry;
                     sdt_table->num_services++;

                     // read service data and descriptor loop length
                     serv_entry->serv_id = (data_ptr[0] << 8) | data_ptr[1];
                     serv_entry->eit_sched_avail = (((data_ptr[2] >> 1) & 0x01) == 0x01);
                     serv_entry->eit_now_next_avail = ((data_ptr[2] & 0x01) == 0x01);
                     serv_entry->running_status = data_ptr[3] >> 5;
                     serv_entry->all_streams_free = (((data_ptr[3] >> 4) & 0x01) == 0x00);

                     dloop_len = ((data_ptr[3] & 0x0f) << 8) | data_ptr[4];
                     data_ptr += 5;

                     #ifdef DEBUG_SI_SDT_CONTENT
                     STB_SI_PRINT(("  sid=0x%04x, eis=%d, eit=%d, rs=%d, free=%d (desc loop len %d)",
                                   serv_entry->serv_id, serv_entry->eit_sched_avail,
                                   serv_entry->eit_now_next_avail, serv_entry->running_status,
                                   serv_entry->all_streams_free, dloop_len));
                     #endif

                     // process descriptor loop
                     priv_data_code = 0;
                     dloop_end = data_ptr + dloop_len;
                     while (data_ptr < dloop_end)
                     {
                        dtag = data_ptr[0];
                        data_ptr++;
                        switch (dtag)
                        {
                           case CA_ID_DTAG:
                           {
                              data_ptr = ParseCaIdentifierDescriptor(data_ptr,
                                    &serv_entry->num_ca_id_entries,
                                    &serv_entry->ca_id_desc_array, DEBUG_SI_SDT_CONTENT_VALUE);
                              break;
                           }

                           case LINKAGE_DTAG:
                           {
                              data_ptr = ParseLinkageDescriptor(data_ptr,
                                    &serv_entry->num_linkage_entries,
                                    &serv_entry->linkage_desc_list,
                                    &serv_entry->last_linkage_entry, priv_data_code,
                                    DEBUG_SI_SDT_CONTENT_VALUE);
                              break;
                           }

                           case MULTILING_SERV_NAME_DTAG:
                           {
                              data_ptr = ParseMultilingServNameDescriptor(data_ptr,
                                    &serv_entry->num_multiling_names,
                                    &serv_entry->multiling_name_desc_array, DEBUG_SI_SDT_CONTENT_VALUE);
                              break;
                           }

                           case SERVICE_DTAG:
                           {
                              data_ptr = ParseServiceDescriptor(data_ptr, &serv_entry->serv_type,
                                    &serv_entry->provider_str,
                                    &serv_entry->name_str, DEBUG_SI_SDT_CONTENT_VALUE);
                              break;
                           }

                           case PRIV_DATA_SPEC_DTAG:
                           {
                              data_ptr = ParsePrivateDataSpecifierDescriptor(data_ptr,
                                    &priv_data_code, DEBUG_SI_SDT_CONTENT_VALUE);
                              break;
                           }

                           case FTA_CONTENT_DTAG:
                           {
                              data_ptr = ParseFTAContentDescriptor(data_ptr, &serv_entry->fta_content_desc, DEBUG_SI_SDT_CONTENT_VALUE);
                              break;
                           }

                           case FREESAT_CONTENT_MANAGE_DTAG:
                           {
                              if ((priv_data_code == FREESAT_PRIVATE_DATA_SPECIFIER) &&
                                  (freesat_private_data_specifier == TRUE))
                              {
                                 data_ptr = ParseFTAContentDescriptor(data_ptr, &serv_entry->fta_content_desc, DEBUG_SI_SDT_CONTENT_VALUE);
                              }
                              else
                              {
                                 data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_SDT_CONTENT_VALUE);
                              }
                              break;
                           }

                           case USER_DEFINED_DTAG_0x84:  /* CNS SDT code descriptor */
                           {
#if 0
                              // for uk dtt this is preferred name list and private data
                              // descriptor will have been sent. For Australia it is
                              // also preferred name list but there won't be a private data
                              // specifier so the application will have set the user defined
                              // descriptor function
                              if (((priv_data_code == UK_DTT_PRIVATE_DATA_SPECIFIER) &&
                                   (priv_data_code == country_private_data_specifier_code)) ||
                                  (GetUserDefinedDescriptorFunction(dtag) == USER_DEF_DESCRIP_PREF_NAME_LIST))
                              {
                                 data_ptr = ParsePreferredNameListDescriptor(data_ptr,
                                       &serv_entry->num_preferred_names,
                                       &serv_entry->preferred_name_desc_array, DEBUG_SI_SDT_CONTENT_VALUE);
                              }
#else
                              if (GetUserDefinedDescriptorFunction(dtag) == USER_DEF_DESCRIP_SDT_CODE)
                              {
                                 data_ptr = ParseSdtCodeDescriptor(data_ptr, &serv_entry->sdt_code_desc, DEBUG_SI_SDT_CONTENT_VALUE);
                              }
#endif
                              else
                              {
                                 data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_SDT_CONTENT_VALUE);
                              }
                              break;
                           }

                           case USER_DEFINED_DTAG_0x87:
                           {
                              // For UK DTT this is the short service name descriptor
                              if (((priv_data_code == UK_DTT_PRIVATE_DATA_SPECIFIER) &&
                                   (priv_data_code == country_private_data_specifier_code)))
                              {
                                 data_ptr = ParseShortServiceNameDescriptor(data_ptr,
                                       &serv_entry->short_name_str, DEBUG_SI_SDT_CONTENT_VALUE);
                              }
                              else
                              {
                                 data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_SDT_CONTENT_VALUE);
                              }
                              break;
                           }

                           case DEF_AUTH_DTAG:
                           {
                              data_ptr = ParseDefaultAuthorityDescriptor(data_ptr,
                                    &serv_entry->def_authority, DEBUG_SI_SDT_CONTENT_VALUE);
                              break;
                           }

                           case USER_DEFINED_DTAG_0x89:
                           case FREESAT_GUIDANCE_DTAG:
                           {
                              if (((priv_data_code == UK_DTT_PRIVATE_DATA_SPECIFIER) &&
                                   (priv_data_code == country_private_data_specifier_code) &&
                                   (freesat_private_data_specifier == FALSE)) ||
                                  ((priv_data_code == ZAF_DTT_PRIVATE_DATA_SPECIFIER) &&
                                   (priv_data_code == country_private_data_specifier_code) &&
                                   (freesat_private_data_specifier == FALSE)) ||
                                  ((priv_data_code == FREESAT_PRIVATE_DATA_SPECIFIER) &&
                                   (freesat_private_data_specifier == TRUE)))
                              {
                                 /* For the UK this is the guidance descriptor and it is
                                  * only valid after a private data specifier descriptor */
                                 data_ptr = ParseGuidanceDescriptor(data_ptr, &serv_entry->guidance, DEBUG_SI_SDT_CONTENT_VALUE);
                              }
                              else
                              {
                                 data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_SDT_CONTENT_VALUE);
                              }
                              break;
                           }

                           case FREESAT_SERV_NAME_DTAG:
                           {
                              if ((priv_data_code == FREESAT_PRIVATE_DATA_SPECIFIER) &&
                                  (freesat_private_data_specifier == TRUE))
                              {
                                 /* Freesat short service name descriptor */
                                 data_ptr = ParseFreesatShortServiceNameDescriptor(data_ptr,
                                       &serv_entry->num_multiling_short_names,
                                       &serv_entry->multiling_short_name_array, DEBUG_SI_SDT_CONTENT_VALUE);
                              }
                              else
                              {
                                 data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_SDT_CONTENT_VALUE);
                              }
                              break;
                           }

                           case EXT_DTAG:
                           {
                              switch (data_ptr[1])
                              {
                                 case TARGET_REGION_DTAG:
                                    data_ptr = ParseTargetRegionDescriptor(data_ptr,
                                          &serv_entry->target_region_list, DEBUG_SI_SDT_CONTENT_VALUE);
                                    break;

                                 case URI_LINKAGE_DTAG:
                                    data_ptr = ParseURILinkageDesc(data_ptr,
                                       &serv_entry->uri_linkage_list, priv_data_code,
                                       DEBUG_SI_SDT_CONTENT_VALUE);
                                    break;

                                 default:
                                    data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_SDT_CONTENT_VALUE);
                                    break;
                              }
                              break;
                           }

                           case CIPLUS_PROTECTION_DTAG:
                           {
                              /* This descriptor is only valid on SDT actual */
                              if ((table_rec->tid == SI_SDT_ACTUAL_TID) &&
                                  ((priv_data_code == CIPLUS_PRIVATE_DATA_SPECIFIER) &&
                                   (ciplus_private_data_specifier == TRUE)))
                              {
                                 data_ptr = SaveCIProtectionDescriptor(data_ptr,
                                       &serv_entry->ci_protection_desc, DEBUG_SI_SDT_CONTENT_VALUE);
                              }
                              else
                              {
                                 data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_SDT_CONTENT_VALUE);
                              }
                              break;
                           }

                           case SERV_AVAIL_DESC_DTAG:
                           {
                              data_ptr = ParseServiceAvailabilityDescriptor(data_ptr,
                                    &serv_entry->serv_avail_desc_list, DEBUG_SI_SDT_CONTENT_VALUE);
                              break;
                           }

                           case FREESAT_PREFIX_DTAG:
                           {
                              if ((priv_data_code == FREESAT_PRIVATE_DATA_SPECIFIER) &&
                                  (freesat_private_data_specifier == TRUE))
                              {
                                 data_ptr = ParseFreesatPrefixDescriptor(data_ptr,
                                       &serv_entry->freesat_prefix_list, DEBUG_SI_SDT_CONTENT_VALUE);
                              }
                              else
                              {
                                 data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_SDT_CONTENT_VALUE);
                              }
                              break;
                           }

                           default:
                           {
                              data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_SDT_CONTENT_VALUE);
                              break;
                           }
                        }
                     }
                  }
               }
               // move on to next section
               section_rec = section_rec->next;
            }
         }
      }
   }

   FUNCTION_FINISH(STB_SIParseSdtTable);
   return(sdt_table);
}

/**
 *

 *
 * @brief   Parses the BAT table supplied in TABLE_RECORD format to create a SI_BAT_TABLE
 *                structure. Returns a pointer to the table. Application must call
 *                STB_SIReleaseBatTable to free the data.
 *
 * @param   table_rec - pointer to the table record to be parsed
 *
 * @return   pointer to the parsed sdt table
 *
 */
SI_BAT_TABLE* STB_SIParseBatTable(SI_TABLE_RECORD *table_rec)
{
   SI_BAT_TABLE *bat_table;
   SI_SECTION_RECORD *section_rec;
   U8BIT *data_ptr;
   U8BIT *tran_dloop_end;
   U16BIT tran_dloop_len;
   U16BIT dloop_len;
   U8BIT *dloop_end;
   U8BIT dtag;
   U32BIT priv_data_code;
   SI_BAT_TRANSPORT_ENTRY *trans_entry;

   FUNCTION_START(STB_SIParseBatTable);

   ASSERT(table_rec != NULL);

   bat_table = NULL;
   if (table_rec != NULL)
   {
      if (table_rec->tid == SI_BAT_TID)
      {
         #ifdef DEBUG_SI_BAT_SUMMARY
         STB_SI_PRINT(("STB_SIParseBatTable: tid 0x%02x/0x%04x, v%d, nsect %d",
                       table_rec->tid, table_rec->xtid, table_rec->version, table_rec->num_sect));
         #endif

         bat_table = STB_GetMemory(sizeof(SI_BAT_TABLE));
         if (bat_table != NULL)
         {
            memset(bat_table, 0, sizeof(SI_BAT_TABLE));

            bat_table->version = table_rec->version;
            bat_table->bouquet_id = table_rec->xtid;

            section_rec = table_rec->section_list;
            while (section_rec != NULL)
            {
               // get pointer to section data and end of section
               data_ptr = &(section_rec->data_start);

               // skip section header
               data_ptr += 8;
               priv_data_code = 0;
               // read bouquet descriptor length
               dloop_len = (((data_ptr[0] & 0x0F) << 8) | (data_ptr[1]));
               data_ptr += 2;    // move past bouquet length

               // process descriptor loop
               dloop_end = data_ptr + dloop_len;
               while (data_ptr < dloop_end)
               {
                  dtag = data_ptr[0];
                  data_ptr++;

                  #ifdef DEBUG_SI_BAT_CONTENT
                  STB_SI_PRINT(("  dtag=0x%02x", (int)dtag));
                  #endif

                  switch (dtag)
                  {
                     case PRIV_DATA_SPEC_DTAG:
                     {
                        data_ptr = ParsePrivateDataSpecifierDescriptor(data_ptr,
                              &priv_data_code, DEBUG_SI_BAT_CONTENT_VALUE);
                        break;
                     }

                     case LINKAGE_DTAG:
                     {
                        data_ptr = ParseLinkageDescriptor(data_ptr,
                              &bat_table->num_linkage_entries,
                              &bat_table->linkage_desc_list,
                              &bat_table->last_linkage_entry, priv_data_code,
                              DEBUG_SI_BAT_CONTENT_VALUE);
                        break;
                     }

                     case DEF_AUTH_DTAG:
                     {
                        data_ptr = ParseDefaultAuthorityDescriptor(data_ptr, &bat_table->def_authority, DEBUG_SI_BAT_CONTENT_VALUE);
                        break;
                     }

                     case FREESAT_SERV_GROUP_NAME_DTAG:
                     {
                        if ((priv_data_code == FREESAT_PRIVATE_DATA_SPECIFIER) &&
                            (freesat_private_data_specifier == TRUE))
                        {
                           data_ptr = ParseFreesatServiceGroupNameDescriptor(data_ptr,
                                 &bat_table->group_name_list, DEBUG_SI_BAT_CONTENT_VALUE);
                        }
                        else
                        {
                           data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_BAT_CONTENT_VALUE);
                        }
                        break;
                     }

                     case FREESAT_SERV_GROUP_DTAG:
                     {
                        if ((priv_data_code == FREESAT_PRIVATE_DATA_SPECIFIER) &&
                            (freesat_private_data_specifier == TRUE))
                        {
                           data_ptr = ParseFreesatServiceGroupDescriptor(data_ptr, &bat_table->serv_group_array,
                                 &bat_table->num_serv_groups, DEBUG_SI_BAT_CONTENT_VALUE);
                        }
                        else
                        {
                           data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_BAT_CONTENT_VALUE);
                        }
                        break;
                     }

                     case FREESAT_IACTIVE_STORAGE_DTAG:
                     {
                        if ((priv_data_code == FREESAT_PRIVATE_DATA_SPECIFIER) &&
                            (freesat_private_data_specifier == TRUE))
                        {
                           data_ptr = ParseIActiveStorageDescriptor(data_ptr,
                                 &bat_table->iactive_storage_desc_list, DEBUG_SI_BAT_CONTENT_VALUE);
                        }
                        else
                        {
                           data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_BAT_CONTENT_VALUE);
                        }
                        break;
                     }

                     case FREESAT_INFO_LOCATION_DTAG:
                     {
                        if ((priv_data_code == FREESAT_PRIVATE_DATA_SPECIFIER) &&
                            (freesat_private_data_specifier == TRUE))
                        {
                           data_ptr = ParseFreesatInfoLocationDescriptor(data_ptr,
                                 &bat_table->info_location_list, DEBUG_SI_BAT_CONTENT_VALUE);
                        }
                        else
                        {
                           data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_BAT_CONTENT_VALUE);
                        }
                        break;
                     }

                     case FREESAT_REGION_NAME_DTAG:
                     {
                        if ((priv_data_code == FREESAT_PRIVATE_DATA_SPECIFIER) &&
                            (freesat_private_data_specifier == TRUE))
                        {
                           data_ptr = ParseRegionNameDescriptor(data_ptr, &bat_table->region_list, DEBUG_SI_BAT_CONTENT_VALUE);
                        }
                        else
                        {
                           data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_BAT_CONTENT_VALUE);
                        }
                        break;
                     }

                     case BOUQUET_NAME_DTAG:
                     {
                        data_ptr = ParseNetNameDescriptor(data_ptr, &bat_table->bouquet_name, DEBUG_SI_BAT_CONTENT_VALUE);
                        break;
                     }

                     case FREESAT_CONTENT_MANAGE_DTAG:
                     {
                        if ((priv_data_code == FREESAT_PRIVATE_DATA_SPECIFIER) &&
                            (freesat_private_data_specifier == TRUE))
                        {
                           data_ptr = ParseFTAContentDescriptor(data_ptr, &bat_table->fta_content_desc, DEBUG_SI_BAT_CONTENT_VALUE);
                        }
                        else
                        {
                           data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_BAT_CONTENT_VALUE);
                        }
                        break;
                     }

                     case FREESAT_PREFIX_DTAG:
                     {
                        if ((priv_data_code == FREESAT_PRIVATE_DATA_SPECIFIER) &&
                            (freesat_private_data_specifier == TRUE))
                        {
                           data_ptr = ParseFreesatPrefixDescriptor(data_ptr,
                                 &bat_table->freesat_prefix_list, DEBUG_SI_BAT_CONTENT_VALUE);
                        }
                        else
                        {
                           data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_BAT_CONTENT_VALUE);
                        }
                        break;
                     }

                     case EXT_DTAG:
                     {
                        /* Examine the extension tag for the table being signalled */
                        switch (data_ptr[1])
                        {
                           case URI_LINKAGE_DTAG:
                              data_ptr = ParseURILinkageDesc(data_ptr,
                                 &bat_table->uri_linkage_list, priv_data_code, DEBUG_SI_BAT_CONTENT_VALUE);
                              break;

                           default:
                              data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_BAT_CONTENT_VALUE);
                              break;
                        }
                        break;
                     }

                     default:
                     {
                        data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_BAT_CONTENT_VALUE);
                        break;
                     }
                  }
               }

               dloop_len = (((data_ptr[0] & 0x0F) << 8) | (data_ptr[1]));
               data_ptr += 2;    // move past bouquet length

               // process descriptor loop
               dloop_end = data_ptr + dloop_len;
               while (data_ptr < dloop_end)
               {
                  trans_entry = (SI_BAT_TRANSPORT_ENTRY *)STB_GetMemory(sizeof(SI_BAT_TRANSPORT_ENTRY));
                  if (trans_entry != NULL)
                  {
                     // initialise new transport structure
                     memset(trans_entry, 0, sizeof(SI_BAT_TRANSPORT_ENTRY));

                     // add to the end of the stream list in the nit table
                     if (bat_table->last_transport_entry == NULL)
                     {
                        // first entry in the list
                        bat_table->transport_list = trans_entry;
                     }
                     else
                     {
                        // not the first entry
                        bat_table->last_transport_entry->next = trans_entry;
                     }

                     bat_table->last_transport_entry = trans_entry;
                     bat_table->num_transports++;

                     trans_entry->tran_id = ((data_ptr[0]) << 8 | (data_ptr[1]));
                     data_ptr += 2;
                     trans_entry->orig_net_id = ((data_ptr[0] << 8) | (data_ptr[1]));
                     // FIXME: workaround for CNS BAT
                     // the original_network_id in CNS's BAT is incorrect
                     // the onid in NIT and SDT is set to 0xA012  
                     if (0x12 == trans_entry->orig_net_id) {
                            trans_entry->orig_net_id = 0xA012;
                     }
                     data_ptr += 2;
                     tran_dloop_len = (((data_ptr[0] & 0x0F) << 8) | (data_ptr[1]));
                     data_ptr += 2;

                     #ifdef DEBUG_SI_BAT_CONTENT
                     STB_SI_PRINT(("  tid 0x%04x, onet 0x%04x", trans_entry->tran_id, trans_entry->orig_net_id));
                     #endif

                     // process transport descriptor loop
                     priv_data_code = 0;
                     tran_dloop_end = data_ptr + tran_dloop_len;
                     while (data_ptr < tran_dloop_end)
                     {
                        dtag = data_ptr[0];
                        data_ptr++;

                        #ifdef DEBUG_SI_BAT_CONTENT
                        STB_SI_PRINT(("  dtag=0x%02x", (int)dtag));
                        #endif

                        switch (dtag)
                        {
                           case PRIV_DATA_SPEC_DTAG:
                           {
                              data_ptr = ParsePrivateDataSpecifierDescriptor(data_ptr,
                                    &priv_data_code, DEBUG_SI_BAT_CONTENT_VALUE);
                              break;
                           }

                           case SERV_LIST_DTAG:
                           {
                              data_ptr = ParseServiceListDescriptor(data_ptr,
                                    &trans_entry->num_serv_list_entries,
                                    &trans_entry->serv_list_desc_array, DEBUG_SI_BAT_CONTENT_VALUE);
                              break;
                           }

                           case LOGICAL_CHANNEL_DTAG:
                           {
                              /* NZ Freeview, free-to-air */
                              if ((priv_data_code == NZ_SAT_PRIVATE_DATA_SPECIFIER) &&
                                  (nzsat_private_data_specifier == TRUE))
                              {
                                 data_ptr = ParseLogicalChannelDescriptor(data_ptr, dtag,
                                       &trans_entry->num_lcn_entries,
                                       &trans_entry->lcn_desc_array, DEBUG_SI_BAT_CONTENT_VALUE);
                              }
                              else
                              {
                                 data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_BAT_CONTENT_VALUE);
                              }
                              break;
                           }

                           case DEF_AUTH_DTAG:
                           {
                              data_ptr = ParseDefaultAuthorityDescriptor(data_ptr, &trans_entry->def_authority, DEBUG_SI_BAT_CONTENT_VALUE);
                              break;
                           }

                           case FREESAT_REGION_LCN_DTAG:
                           {
                              if ((priv_data_code == FREESAT_PRIVATE_DATA_SPECIFIER) &&
                                  (freesat_private_data_specifier == TRUE))
                              {
                                 data_ptr = ParseRegionLcnDescriptor(data_ptr, &trans_entry->lcn_region_list, DEBUG_SI_BAT_CONTENT_VALUE);
                              }
                              else
                              {
                                 data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_BAT_CONTENT_VALUE);
                              }
                              break;
                           }

                           case FREESAT_CONTENT_MANAGE_DTAG:
                           {
                              if ((priv_data_code == FREESAT_PRIVATE_DATA_SPECIFIER) &&
                                  (freesat_private_data_specifier == TRUE))
                              {
                                 data_ptr = ParseFTAContentDescriptor(data_ptr, &trans_entry->fta_content_desc, DEBUG_SI_BAT_CONTENT_VALUE);
                              }
                              else
                              {
                                 data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_BAT_CONTENT_VALUE);
                              }
                              break;
                           }

                           case FREESAT_IACTIVE_RESTRICT_DTAG:
                           {
                              if ((priv_data_code == FREESAT_PRIVATE_DATA_SPECIFIER) &&
                                  (freesat_private_data_specifier == TRUE))
                              {
                                 data_ptr = ParseFreesatInteractiveRestrictionDescriptor(data_ptr,
                                       &trans_entry->int_rest_serv_array,
                                       &trans_entry->num_int_rest_serv, DEBUG_SI_BAT_CONTENT_VALUE);
                              }
                              else
                              {
                                 data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_BAT_CONTENT_VALUE);
                              }
                              break;
                           }

                           case USER_DEFINED_DTAG_0x93:  /* CNS logical channel number descriptor */
                           {
                              if (GetUserDefinedDescriptorFunction(dtag) == USER_DEF_DESCRIP_LOGICAL_CHAN_NUM)
                              {
                                 data_ptr = ParseCnsLogicalChannelDescriptor(data_ptr, dtag,
                                       &trans_entry->num_lcn_entries,
                                       &trans_entry->lcn_desc_array, DEBUG_SI_BAT_CONTENT_VALUE);
                              }
                              else
                              {
                                 data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_EIT_CONTENT_VALUE);
                              } 
                              break;
                           }

                           default:
                           {
                              data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_BAT_CONTENT_VALUE);
                              break;
                           }
                        }
                     }
                  }
               }
               // move on to next section
               section_rec = section_rec->next;
            }
         }
      }
   }

   FUNCTION_FINISH(STB_SIParseBatTable);
   return(bat_table);
}

/**
 *

 *
 * @brief   Parses the UNT table supplied in TABLE_RECORD format to create a SI_UNT_TABLE
 *                structure. Returns a pointer to the table. Application must call
 *                STB_SIReleaseUntTable to free the data.
 *
 * @param   table_rec - pointer to the table record to be parsed
 *
 * @return   pointer to the parsed UNT table
 *
 */
SI_UNT_TABLE* STB_SIParseUntTable(SI_TABLE_RECORD *table_rec)
{
   SI_UNT_TABLE *unt_table;
   SI_SECTION_RECORD *section_rec;
   U8BIT *data_ptr;
   U8BIT *data_end;
   U16BIT dloop_len;
   U8BIT *dloop_end;
   U8BIT dtag;

   FUNCTION_START(STB_SIParseUntTable);
   
   ASSERT(table_rec != NULL);

   unt_table = NULL;
   if (table_rec != NULL)
   {
      if (table_rec->tid == SI_UNT_TID)
      {
         #ifdef DEBUG_SI_UNT_SUMMARY
         STB_SI_PRINT(("STB_SIParseUntTable: tid 0x%02x/0x%04x, v%d, nsect %d",
                       table_rec->tid, table_rec->xtid, table_rec->version, table_rec->num_sect));
         #endif

         unt_table = STB_GetMemory(sizeof(SI_UNT_TABLE));
         if (unt_table != NULL)
         {
            memset(unt_table, 0, sizeof(SI_UNT_TABLE));

            unt_table->version = table_rec->version;
            unt_table->action_type = (U8BIT)((table_rec->xtid >> 8) & 0xFF);
            unt_table->del_sys_desc_type = SI_DEL_SYS_DESC_TYPE_NONE;

            section_rec = table_rec->section_list;
            while (section_rec != NULL)
            {
               // get pointer to section data and end of section
               data_ptr = &(section_rec->data_start);
               data_end = data_ptr + section_rec->data_len - 4;   // -4 for crc

               // skip section header
               data_ptr += 8;

               unt_table->oui = (((U32BIT)data_ptr[0]) << 16) + (((U32BIT)data_ptr[1]) << 8) + data_ptr[2];
               data_ptr += 3;
               #ifdef DEBUG_SI_UNT_CONTENT
               STB_SI_PRINT((" section %d, OUI=0x%08X",
                             section_rec->sect_num, unt_table->oui));
               #endif
               unt_table->processing_order = data_ptr[0];
               data_ptr ++;
      
               // common_descriptor_loop
               dloop_len = (((U16BIT)data_ptr[0] & 0x0F) << 8) + data_ptr[1];
		       data_ptr += 2;
               dloop_end = data_ptr + dloop_len;
               while (data_ptr < dloop_end)
               {
                  // skip
                  data_ptr = dloop_end;
               }

               // optional loop
               while (data_ptr < data_end)
               {
                  // compatibility descriptor loop
                  dloop_len = (((U16BIT)data_ptr[0] & 0x0F) << 8) + data_ptr[1];
		          data_ptr += 2;
                  dloop_end = data_ptr + dloop_len;
                  while (data_ptr < dloop_end)
                  {
                     // skip
                     data_ptr = dloop_end;
                  }
                  // platform loop
                  dloop_len = (((U16BIT)data_ptr[0] & 0x0F) << 8) + data_ptr[1];
		          data_ptr += 2;
                  dloop_end = data_ptr + dloop_len;

                  // optional loop
                  while (data_ptr < dloop_end)
                  {
                     U16BIT dloop_len2;
                     U8BIT *dloop_end2;

                     // target desctiptor loop
                     dloop_len2 = (((U16BIT)data_ptr[0] & 0x0F) << 8) + data_ptr[1];
                     data_ptr += 2;
                     dloop_end2 = data_ptr + dloop_len2;
                     while (data_ptr < dloop_end2)
                     {
                        // skip
                        data_ptr = dloop_end2;
                     }

                     // operational descriptor loop
                     dloop_len2 = (((U16BIT)data_ptr[0] & 0x0F) << 8) + data_ptr[1];
                     data_ptr += 2;
                     dloop_end2 = data_ptr + dloop_len2;
                     while (data_ptr < dloop_end2)
                     {
                        dtag = data_ptr[0];
                        #ifdef DEBUG_SI_UNT_CONTENT
                        STB_SI_PRINT(("  dtag=0x%02x", (int)dtag));
                        #endif

                        data_ptr++;
                        switch (dtag)
                        {
                           case SSU_LOCATION_DTAG:
                           {
                              data_ptr = ParseSsuLocationDescriptor(data_ptr,
                                    &unt_table->ssu_location_desc, DEBUG_SI_UNT_CONTENT_VALUE);
                              break;
                           }
                           case CABLE_DEL_SYS_DTAG:
                           {
                              unt_table->del_sys_desc_type = SI_DEL_SYS_DESC_TYPE_CABLE;
                              data_ptr = ParseCableDeliverySysDescriptor(data_ptr,
                                    &unt_table->del_sys_desc, DEBUG_SI_UNT_CONTENT_VALUE);
                              break;
                           }
                           default:
                           {
                              data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_UNT_CONTENT_VALUE);
                              break;
                           }
                        }
                     }  // while (data_ptr < dloop_end2)  
                  }  // while (data_ptr < dloop_end)
               }  // while (data_ptr < data_end)

               // move on to next section
               section_rec = section_rec->next;
            }  // while (section_rec != NULL)
         }  // if (unt_table != NULL)
      }  // if (table_rec->tid == SI_UNT_TID)
   }  // if (table_rec != NULL)
   
   FUNCTION_FINISH(STB_SIParseUntTable);
   return (unt_table);
}

SI_CAT_TABLE* STB_SIParseCatTable(SI_TABLE_RECORD *table_rec)
{
   SI_CAT_TABLE *cat_table = NULL;
   SI_SECTION_RECORD *section_rec;
   U8BIT *data_ptr;
   U8BIT *data_end;
   U16BIT dloop_len;
   U8BIT *dloop_end;
   U8BIT dtag;

   FUNCTION_START(STB_SIParseCatTable);
   
   ASSERT(table_rec != NULL);

   if (table_rec != NULL)
   {
      if (table_rec->tid == SI_CAT_TID)
      {
         #ifdef DEBUG_SI_CAT_SUMMARY
         STB_SI_PRINT(("STB_SIParseCatTable: tid 0x%02x/0x%04x, v%d, nsect %d",
                       table_rec->tid, table_rec->xtid, table_rec->version, table_rec->num_sect));
         #endif

         cat_table = STB_GetMemory(sizeof(SI_CAT_TABLE));
         if (cat_table != NULL)
         {
            memset(cat_table, 0, sizeof(SI_CAT_TABLE));
            cat_table->version = table_rec->version;
            
            section_rec = table_rec->section_list;
            while (section_rec != NULL)
            {
               // get pointer to section data and end of section
               data_ptr = &(section_rec->data_start);
               data_end = data_ptr + section_rec->data_len - 4;   // -4 for crc
        
               // skip the section header
               data_ptr += 8;

               #ifdef DEBUG_SI_CAT_CONTENT
               STB_SI_PRINT((" section %d", section_rec->sect_num));
               #endif
      
               // descriptor loop
               while (data_ptr < data_end)
               {                   
                  dtag = data_ptr[0];
                  data_ptr++;
                  #ifdef DEBUG_SI_CAT_CONTENT
                  if (dtag != 00)
                  {
                     STB_SI_PRINT(("   tag %02x len %04x", dtag, *data_ptr));
                  }
                  #endif

                  switch (dtag)
                  {
                     case CA_DTAG:
                     {
                        data_ptr = ParseCaDescriptor(data_ptr, &cat_table->num_ca_entries,
                              &cat_table->ca_desc_array, DEBUG_SI_CAT_CONTENT_VALUE);
                        break;
                     }

                     default:
                     {
                        data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_CAT_CONTENT_VALUE);
                        break;
                     }
                  }
               }

               // move on to next section
               section_rec = section_rec->next;
            }  // while (section_rec != NULL)
         }  // if (cat_table != NULL)
      }  // if (table_rec->tid == SI_CAT_TID)
   }  // if (table_rec != NULL)
   
   FUNCTION_FINISH(STB_SIParseCatTable);
   return (cat_table);
}

/**
 *

 *
 * @brief   Parses the Eit table supplied in TABLE_RECORD format to create a EIT_TABLE
 *                structure. Returns a pointer to the table. Application must call
 *                STB_SIReleaseEitTable to free the data.
 *
 * @param   table_rec - pointer to the table record to be parsed
 *
 * @return   pointer to the parsed eit table
 *
 */
SI_EIT_TABLE* STB_SIParseEitTable(SI_TABLE_RECORD *table_rec)
{
   U8BIT tid;
   SI_EIT_TABLE *eit_table;
   SI_SECTION_RECORD *section_rec;
   U8BIT *data_ptr;
   U8BIT *data_end;
   U16BIT dloop_len;
   U8BIT *dloop_end;
   U8BIT dtag;
   SI_EIT_EVENT_ENTRY *event_entry;
   U32BIT priv_data_code;
   BOOLEAN abort;

   FUNCTION_START(STB_SIParseEitTable);

   ASSERT(table_rec != NULL);

   abort = FALSE;
   eit_table = NULL;

   if (table_rec != NULL)
   {
      tid = table_rec->tid;
      if ((tid == SI_EITPF_ACTUAL_TID) || (tid == SI_EITPF_OTHER_TID) ||
          ((tid & 0xf0) == SI_EITSC_ACTUAL_TID) || ((tid & 0xf0) == SI_EITSC_OTHER_TID))
      {
         #ifdef DEBUG_SI_EIT_SUMMARY
         STB_SI_PRINT(("STB_SIParseEitTable: tid 0x%02x/0x%04x, v%d, nsect %d",
                       table_rec->tid, table_rec->xtid, table_rec->version, table_rec->num_sect));
         #endif

         eit_table = STB_GetMemory(sizeof(SI_EIT_TABLE));
         if (eit_table != NULL)
         {
            memset(eit_table, 0, sizeof(SI_EIT_TABLE));

            eit_table->version = table_rec->version;
            eit_table->table_id = table_rec->tid;
            eit_table->serv_id = table_rec->xtid;

            // loop through all sections in the list - they will be in section order
            section_rec = table_rec->section_list;
            while ((section_rec != NULL) && !abort)
            {
               // get pointer to section data and end of section
               data_ptr = &(section_rec->data_start);
               data_end = data_ptr + section_rec->data_len - 4;   // -4 for crc

               // skip section header
               data_ptr += 8;

               // read transport id, original network id and last table id. If there is more than
               // one section all sections should contain the same value, so we will allow later
               // sections to overwrite the values i.e. the values in the last section will be used
               eit_table->tran_id = (data_ptr[0] << 8) | data_ptr[1];
               eit_table->orig_net_id = (data_ptr[2] << 8) | data_ptr[3];
               eit_table->last_table_id = data_ptr[5];
               data_ptr += 6;

               #ifdef DEBUG_SI_EIT_CONTENT
               STB_SI_PRINT((" section %d, tranid=0x%04x, onid=0x%04x, last tid=0x%02x",
                             section_rec->sect_num, eit_table->tran_id, eit_table->orig_net_id,
                             eit_table->last_table_id));
               #endif

               // read entry for each event
               while ((data_ptr < data_end) && !abort)
               {
                  event_entry = (SI_EIT_EVENT_ENTRY *)STB_GetMemory(sizeof(SI_EIT_EVENT_ENTRY));
                  if (event_entry != NULL)
                  {
                     // initialise new event structure
                     memset(event_entry, 0, sizeof(SI_EIT_EVENT_ENTRY));

                     // add to the end of the event list in the eit table
                     if (eit_table->last_event_entry == NULL)
                     {
                        // first entry in the list
                        eit_table->event_list = event_entry;
                     }
                     else
                     {
                        // not the first entry
                        eit_table->last_event_entry->next = event_entry;
                     }
                     eit_table->last_event_entry = event_entry;
                     eit_table->num_events++;

                     // read event data and descriptor loop length
                     event_entry->sect_num = section_rec->sect_num;
                     event_entry->event_id = (data_ptr[0] << 8) | data_ptr[1];
                     event_entry->start_date = (data_ptr[2] << 8) | data_ptr[3];
                     event_entry->start_hrs = ((data_ptr[4] >> 4) * 10) + (data_ptr[4] & 0x0f);
                     event_entry->start_mins = ((data_ptr[5] >> 4) * 10) + (data_ptr[5] & 0x0f);
                     event_entry->start_secs = ((data_ptr[6] >> 4) * 10) + (data_ptr[6] & 0x0f);
                     event_entry->duration_hrs = ((data_ptr[7] >> 4) * 10) + (data_ptr[7] & 0x0f);
                     event_entry->duration_mins = ((data_ptr[8] >> 4) * 10) + (data_ptr[8] & 0x0f);
                     event_entry->duration_secs = ((data_ptr[9] >> 4) * 10) + (data_ptr[9] & 0x0f);
                     event_entry->running_status = data_ptr[10] >> 5;
                     event_entry->all_streams_free = (((data_ptr[10] >> 4) & 0x01) == 0x00);
                     dloop_len = ((data_ptr[10] & 0x0f) << 8) | data_ptr[11];
                     data_ptr += 12;

                     #ifdef DEBUG_SI_EIT_CONTENT
                     STB_SI_PRINT(("  eid=0x%04x on %d @ %02d:%02d:%02d for %02d:%02d:%02d, rs=%d, free=%d (desc loop len %d)",
                                   event_entry->event_id, event_entry->start_date,
                                   event_entry->start_hrs, event_entry->start_mins,
                                   event_entry->start_secs, event_entry->duration_hrs,
                                   event_entry->duration_mins, event_entry->duration_secs,
                                   event_entry->running_status, event_entry->all_streams_free,
                                   dloop_len));
                     #endif

                     // process descriptor loop
                     priv_data_code = 0;
                     dloop_end = data_ptr + dloop_len;
                     while (data_ptr < dloop_end)
                     {
                        dtag = data_ptr[0];
                        data_ptr++;
                        switch (dtag)
                        {
                           case CA_ID_DTAG:
                           {
                              data_ptr = ParseCaIdentifierDescriptor(data_ptr,
                                    &event_entry->num_ca_id_entries,
                                    &event_entry->ca_id_desc_array, DEBUG_SI_EIT_CONTENT_VALUE);
                              break;
                           }

                           case COMPONENT_DTAG:
                           {
                              data_ptr = ParseComponentDescriptor(data_ptr,
                                    &event_entry->num_component_entries,
                                    &event_entry->component_desc_array, DEBUG_SI_EIT_CONTENT_VALUE);
                              break;
                           }

                           case CONTENT_DTAG:
                           {
                              data_ptr = ParseContentDescriptor(data_ptr,
                                    &event_entry->num_content_entries,
                                    &event_entry->content_desc_array, DEBUG_SI_EIT_CONTENT_VALUE);
                              break;
                           }

                           case PARENTAL_RATING_DTAG:
                           {
                              data_ptr = ParseParentalRatingDescriptor(data_ptr,
                                    &event_entry->num_parental_rating_entries,
                                    &event_entry->parental_rating_desc_array, DEBUG_SI_EIT_CONTENT_VALUE);
                              break;
                           }

                           case MULTILING_COMPONENT_DTAG:
                           {
                              data_ptr = ParseMultilingComponentDescriptor(data_ptr,
                                    &event_entry->num_multiling_component_entries,
                                    &event_entry->multiling_component_desc_array, DEBUG_SI_EIT_CONTENT_VALUE);
                              break;
                           }

                           case SHORT_EVENT_DTAG:
                           {
                              data_ptr = ParseShortEventDescriptor(data_ptr,
                                    &event_entry->num_short_event_entries,
                                    &event_entry->short_event_desc_array, DEBUG_SI_EIT_CONTENT_VALUE);
                              break;
                           }

                           case EXTENDED_EVENT_DTAG:
                           {
                              data_ptr = ParseExtendedEventDescriptor(data_ptr,
                                    &event_entry->num_extended_event_entries,
                                    &event_entry->extended_event_desc_array, DEBUG_SI_EIT_CONTENT_VALUE);
                              break;
                           }

                           case PRIV_DATA_SPEC_DTAG:
                           {
                              data_ptr = ParsePrivateDataSpecifierDescriptor(data_ptr,
                                    &priv_data_code, DEBUG_SI_EIT_CONTENT_VALUE);
                              break;
                           }

                           case USER_DEFINED_DTAG_0x85:
                           {
                              // for uk dtt this is preferred name id and private data
                              // descriptor will have been sent. For Australia it is
                              // also preferred name id but there won't be a private data
                              // specifier so the application will have set the user defined
                              // descriptor function
                              if (((priv_data_code == UK_DTT_PRIVATE_DATA_SPECIFIER) &&
                                   (priv_data_code == country_private_data_specifier_code)) ||
                                  (GetUserDefinedDescriptorFunction(dtag) == USER_DEF_DESCRIP_PREF_NAME_ID))
                              {
                                 data_ptr = ParsePreferredNameIdDescriptor(data_ptr,
                                       &event_entry->preferred_name_id, DEBUG_SI_EIT_CONTENT_VALUE);
                              }
                              else
                              {
                                 data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_EIT_CONTENT_VALUE);
                              }
                              break;
                           }

                           case CONT_ID_DTAG:
                           {
                              data_ptr = ParseContentIdentifierDescriptor(data_ptr, &event_entry->num_crids,
                                    &event_entry->crid_list, &event_entry->last_crid_entry, DEBUG_SI_EIT_CONTENT_VALUE);
                              break;
                           }

                           case USER_DEFINED_DTAG_0x89:
                           case FREESAT_GUIDANCE_DTAG:
                           {
                              if (((priv_data_code == UK_DTT_PRIVATE_DATA_SPECIFIER) &&
                                   (priv_data_code == country_private_data_specifier_code) &&
                                   (freesat_private_data_specifier == FALSE)) ||
                                  ((priv_data_code == ZAF_DTT_PRIVATE_DATA_SPECIFIER) &&
                                   (priv_data_code == country_private_data_specifier_code) &&
                                   (freesat_private_data_specifier == FALSE)) ||
                                  ((priv_data_code == FREESAT_PRIVATE_DATA_SPECIFIER) &&
                                   (freesat_private_data_specifier == TRUE)))
                              {
                                 /* For the UK this is the guidance descriptor and it is
                                  * only valid after a private data specifier descriptor */
                                 data_ptr = ParseGuidanceDescriptor(data_ptr, &event_entry->guidance, DEBUG_SI_EIT_CONTENT_VALUE);
                              }
                              else
                              {
                                 data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_EIT_CONTENT_VALUE);
                              }
                              break;
                           }

                           case USER_DEFINED_DTAG_0xD0:  /* CNS series (linking) descriptor */
                           {
                              if (GetUserDefinedDescriptorFunction(dtag) == USER_DEF_DESCRIP_SERIES)
                              {
                                 data_ptr = ParseSeriesDescriptor(data_ptr, dtag, &event_entry->series_desc, DEBUG_SI_EIT_CONTENT_VALUE);
                              }
                              else
                              {
                                 data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_EIT_CONTENT_VALUE);
                              } 
                              break;
                           }

                           default:
                           {
                              data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_EIT_CONTENT_VALUE);
                              break;
                           }
                        }
                     }
                  }
                  else
                  {
                     /* Failed to allocate memory, so abort and return whatever data has been collected */
                     abort = TRUE;
                  }
               }

               // move on to next section
               section_rec = section_rec->next;
            }
         }
      }
   }

   FUNCTION_FINISH(STB_SIParseEitTable);
   return(eit_table);
}

/**
 *

 *
 * @brief   Parses the tdt or tot table supplied in TABLE_RECORD format to create a
 *                TIME_TABLE structure. Returns a pointer to the table. Application must call
 *                STB_SIReleaseTimeTable to free the data.
 *
 * @param   table_rec - pointer to the table record to be parsed
 *
 * @return   pointer to the parsed time table
 *
 */
SI_TIME_TABLE* STB_SIParseTimeTable(SI_TABLE_RECORD *table_rec)
{
   SI_TIME_TABLE *time_table;
   SI_SECTION_RECORD *section_rec;
   U8BIT *data_ptr;
   U16BIT dloop_len;
   U8BIT *dloop_end;
   U8BIT dtag;

   FUNCTION_START(STB_SIParseTimeTable);

   ASSERT(table_rec != NULL);

   time_table = NULL;
   if (table_rec != NULL)
   {
      if ((table_rec->tid == SI_TDT_TID) || (table_rec->tid == SI_TOT_TID))
      {
         #ifdef DEBUG_SI_TDT_TOT_SUMMARY
         STB_SI_PRINT(("STB_SIParseTimeTable: tid 0x%02x", table_rec->tid));
         #endif

         time_table = STB_GetMemory(sizeof(SI_TIME_TABLE));
         if (time_table != NULL)
         {
            memset(time_table, 0, sizeof(SI_TIME_TABLE));

            // get pointer to section data
            section_rec = table_rec->section_list;
            data_ptr = &(section_rec->data_start);
            data_ptr += 3;    // skip section header

            // read time
            time_table->date = (data_ptr[0] << 8) | data_ptr[1];
            time_table->hrs = ((data_ptr[2] >> 4) * 10) + (data_ptr[2] & 0x0f);
            time_table->mins = ((data_ptr[3] >> 4) * 10) + (data_ptr[3] & 0x0f);
            time_table->secs = ((data_ptr[4] >> 4) * 10) + (data_ptr[4] & 0x0f);
            data_ptr += 5;

            #ifdef DEBUG_SI_TDT_TOT_CONTENT
            STB_SI_PRINT((" %d %02d:%02d:%02d", time_table->date, time_table->hrs,
                          time_table->mins, time_table->secs));
            #endif

            if (table_rec->tid == SI_TOT_TID)
            {
               // process descriptor loop
               dloop_len = ((data_ptr[0] & 0x0f) << 8) | data_ptr[1];
               data_ptr += 2;
               dloop_end = data_ptr + dloop_len;
               #ifdef DEBUG_SI_TDT_TOT_CONTENT
               STB_SI_PRINT((" Tot desc loop (len %d)", dloop_len));
               #endif
               while (data_ptr < dloop_end)
               {
                  dtag = data_ptr[0];
                  data_ptr++;

                  switch (dtag)
                  {
                     case LOCAL_TIME_OFFSET_DTAG:
                     {
                        data_ptr = ParseLocalTimeOffsetDescriptor(data_ptr,
                              &time_table->num_lto_entries,
                              &time_table->lto_desc_array, DEBUG_SI_TDT_TOT_CONTENT_VALUE);
                        break;
                     }

                     default:
                     {
                        data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_TDT_TOT_CONTENT_VALUE);
                        break;
                     }
                  }
               }
            }
         }
      }
   }

   FUNCTION_FINISH(STB_SIParseTimeTable);
   return(time_table);
}

/*!**************************************************************************
 * @brief   Parses the related content table (RCT) to create an SI_RCT_TABLE structure.
 * @param   table_rec - pointer to the table record to be parsed
 * @return  Pointer to the parsed RCT table which must be freed using STB_SIReleaseRctTable
 ****************************************************************************/
SI_RCT_TABLE* STB_SIParseRctTable(SI_TABLE_RECORD *table_rec)
{
   SI_RCT_TABLE *rct_table;
   SI_SECTION_RECORD *section_rec;
   U8BIT *data_ptr;
   U16BIT dloop_len;
   U8BIT *dloop_end;
   U8BIT dtag;
   U16BIT service_id;
   SI_RCT_SUBTABLE *subtable;
   SI_RCT_SUBTABLE *sub_ptr;
   SI_RCT_SUBTABLE_DATA *sub_data;
   SI_RCT_SUBTABLE_DATA *next_data;
   U8BIT i;

   FUNCTION_START(STB_SIParseRctTable);

   ASSERT(table_rec != NULL);

   rct_table = NULL;

   if (table_rec != NULL)
   {
      if (table_rec->tid == SI_RCT_TID)
      {
         #ifdef DEBUG_SI_RCT_SUMMARY
         STB_SI_PRINT(("STB_SIParseRctTable: tid 0x%02x/0x%04x, v%d, nsect %d",
                       table_rec->tid, table_rec->xtid, table_rec->version, table_rec->num_sect));
         #endif

         rct_table = STB_GetMemory(sizeof(SI_RCT_TABLE));
         if (rct_table != NULL)
         {
            memset(rct_table, 0, sizeof(SI_RCT_TABLE));

            section_rec = table_rec->section_list;
            while (section_rec != NULL)
            {
               /* Get pointer to section data and end of section */
               data_ptr = &(section_rec->data_start);

               /* Check the table_id_extension flag that indicates whether the xtid defines the service ID */
               if ((data_ptr[1] & 0x40) == 0)
               {
                  service_id = table_rec->xtid;
               }
               else
               {
                  service_id = 0;
               }

               #ifdef DEBUG_SI_RCT_CONTENT
               STB_SI_PRINT(("  service_id=0x%04x", service_id));
               #endif

               /* Skip section header */
               data_ptr += 8;

               /* Does a subtable for this service already exist? */
               for (subtable = rct_table->subtables; subtable != NULL; subtable = subtable->next)
               {
                  if ((subtable->version == table_rec->version) && (subtable->service_id == service_id))
                  {
                     /* Subtable found */
                     break;
                  }
               }

               if (subtable == NULL)
               {
                  /* Need a new subtable */
                  subtable = (SI_RCT_SUBTABLE *)STB_GetMemory(sizeof(SI_RCT_SUBTABLE));
                  if (subtable != NULL)
                  {
                     memset(subtable, 0, sizeof(SI_RCT_SUBTABLE));

                     subtable->version = table_rec->version;
                     subtable->service_id = service_id;

                     /* Add the subtable to the RCT */
                     if (rct_table->subtables == NULL)
                     {
                        rct_table->subtables = subtable;
                     }
                     else
                     {
                        /* Find the end of the list */
                        for (sub_ptr = rct_table->subtables; sub_ptr->next != NULL; sub_ptr = sub_ptr->next)
                           ;

                        sub_ptr->next = subtable;
                     }
                  }
               }

               if (subtable != NULL)
               {
                  /* Create a new subtable data structure to hold the info contained in this section */
                  sub_data = (SI_RCT_SUBTABLE_DATA *)STB_GetMemory(sizeof(SI_RCT_SUBTABLE_DATA));
                  if (sub_data != NULL)
                  {
                     memset(sub_data, 0, sizeof(SI_RCT_SUBTABLE_DATA));

                     /* Add the subtable's data to the end of the list */
                     if (subtable->data == NULL)
                     {
                        subtable->data = sub_data;
                     }
                     else
                     {
                        for (next_data = subtable->data; next_data->next != NULL; next_data = next_data->next)
                           ;

                        next_data->next = sub_data;
                     }

                     /* Read the year offset to be used for all dates in this table */
                     sub_data->year_offset = (data_ptr[0] << 8) | data_ptr[1];

                     /* Process the links */
                     sub_data->link_count = data_ptr[2];
                     data_ptr += 3;

                     #ifdef DEBUG_SI_RCT_CONTENT
                     STB_SI_PRINT(("  year_offset=%u", sub_data->year_offset));
                     STB_SI_PRINT(("  link_count=%u", sub_data->link_count));
                     #endif

                     if (sub_data->link_count > 0)
                     {
                        /* Allocate memory for the links and read each one */
                        sub_data->link_array = (SI_RCT_LINK_INFO *)STB_GetMemory(sub_data->link_count * sizeof(SI_RCT_LINK_INFO));
                        if (sub_data->link_array != NULL)
                        {
                           for (i = 0; i < sub_data->link_count; i++)
                           {
                              #ifdef DEBUG_SI_RCT_CONTENT
                              STB_SI_PRINT(("  link=%u:", i));
                              #endif

                              data_ptr = ParseRctLinkInfo(data_ptr, &sub_data->link_array[i], DEBUG_SI_RCT_CONTENT_VALUE);
                           }
                        }
                        else
                        {
                           sub_data->link_count = 0;
                        }
                     }

                     /* Process the descriptor loop */
                     dloop_len = ((data_ptr[0] & 0x0f) << 8) | data_ptr[1];
                     data_ptr += 2;

                     dloop_end = data_ptr + dloop_len;
                     while (data_ptr < dloop_end)
                     {
                        dtag = data_ptr[0];
                        data_ptr++;
                        switch (dtag)
                        {
                           case EXT_DTAG:
                           {
                              /* Examine the extension tag for the table being signalled */
                              switch (data_ptr[1])
                              {
                                 case IMAGE_ICON_DTAG:
                                 {
                                    data_ptr = ParseImageIconDesc(data_ptr, &sub_data->num_icons, &sub_data->icon_array, DEBUG_SI_RCT_CONTENT_VALUE);
                                    break;
                                 }

                                 default:
                                 {
                                    data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_RCT_CONTENT_VALUE);
                                    break;
                                 }
                              }
                              break;
                           }
#if 0
                           case SHORT_EVENT_DTAG:
                           {
                              data_ptr = ParseShortEventDescriptor(data_ptr,
                                    &event_entry->num_short_event_entries,
                                    &event_entry->short_event_desc_array, DEBUG_SI_RCT_CONTENT_VALUE);
                              break;
                           }
#endif

                           default:
                           {
                              data_ptr = SkipDescriptor(data_ptr, dtag, DEBUG_SI_RCT_CONTENT_VALUE);
                              break;
                           }
                        }
                     }
                  }
               }

               /* Move on to next section */
               section_rec = section_rec->next;
            }
         }
      }
   }

   FUNCTION_FINISH(STB_SIParseRctTable);

   return(rct_table);
}

/**
 *

 *
 * @brief   Copies a string from a descriptor for the specified length. Converts the number
 *                of bytes specified into a null-terminated string, removing any embedded null
 *                codes. Does not affect the coding of the string, e.g. single byte ascii or
 *                unicode. Does not remove any control characters.
 *
 * @param   nbytes  - number of bytes in string
 * @param   dptr    - pointer to first byte of string
 * @param   str_ptr - pointer for the return of the null_termainated string
 *
 * @return   Updated dptr i.e. pointing to byte following the string
 *
 */
U8BIT* STB_SIReadString(U8BIT nbytes, U8BIT *dptr, SI_STRING_DESC **str_ptr)
{
   U8BIT *end_ptr;
   SI_STRING_DESC *str_desc;
   U8BIT *tmp_ptr;
   BOOLEAN unicode;
   U8BIT byte_val;
   U8BIT byte_count;

   S16BIT num_bytes;

   FUNCTION_START(STB_SIReadString);

   end_ptr = dptr + nbytes;

   str_desc = (SI_STRING_DESC *)STB_GetMemory(sizeof(SI_STRING_DESC) + nbytes + 2 + 1 /*recover the character encoding*/); // worst case need +2 for unicode null term
   if (str_desc != NULL)
   {
      num_bytes = nbytes;
      byte_count = 0;
      tmp_ptr = ((U8BIT *)str_desc) + sizeof(SI_STRING_DESC);
      str_desc->str_ptr = tmp_ptr;
      unicode = FALSE;

      if (nbytes > 0)
      {
         // check type of dvb string coding
         byte_val = *dptr;
         if ((byte_val >= 0x01) && (byte_val <= 0x0b))
         {
            // one of the 11 Latin tables specified in the DVB SI specification from ISO 8859.
            *tmp_ptr = byte_val;
            tmp_ptr++;
            dptr++;
            num_bytes--;
            byte_count++;
         }
         else if (byte_val == 0x10)
         {
            // next 2 bytes indicate the table from ISO 8859 - copy the 3 bytes header across
            memcpy(tmp_ptr, dptr, 3);
            tmp_ptr += 3;
            dptr += 3;
            num_bytes -= 3;
            byte_count += 3;
         }
         else if ((byte_val == 0x11) || (byte_val == 0x14))
         {
            // unicode coded (2 byte codes) - copy header across
            *tmp_ptr = byte_val;
            tmp_ptr++;
            dptr++;
            num_bytes--;
            byte_count++;
            unicode = TRUE;
         }
         else if (byte_val == 0x1f)
         {
            /* Compressed string, just read all available bytes */
            memcpy(tmp_ptr, dptr, num_bytes);
            tmp_ptr += num_bytes;
            dptr += num_bytes;
            byte_count += num_bytes;
            num_bytes = 0;
         }
         else {
            // recover the character encoding type
            // assume byte_val=0x15 (UTF-8),  ASCII compatible
            // workaround for CNS 
            tmp_ptr[0] = 0x15;
            tmp_ptr++;
            byte_count++;
         }

         // for remainder of string copy across removing null codes embedded in the string before
         // we reach the end
         while (num_bytes > 0)
         {
            if (unicode == TRUE)
            {
               // two bytes per character code
               if ((dptr[0] != 0) || (dptr[1] != 0))
               {
                  // not null - copy the character across
                  tmp_ptr[0] = dptr[0];
                  tmp_ptr[1] = dptr[1];
                  tmp_ptr += 2;
                  byte_count += 2;
               }
               dptr += 2;
               num_bytes -= 2;
            }
            else
            {
               // single byte ascii
               if (dptr[0] != 0)
               {
                  // not null - copy the character across
                  tmp_ptr[0] = dptr[0];
                  tmp_ptr++;
                  byte_count++;
               }
               dptr++;
               num_bytes--;
            }
         }
      }

      // add double null terminator (suitable for unicode)
      tmp_ptr[0] = 0;
      tmp_ptr[1] = 0;
      byte_count += 2;

      str_desc->nbytes = byte_count;
   }

   *str_ptr = str_desc;

   FUNCTION_FINISH(STB_SIReadString);

   return(end_ptr);
}

/*!**************************************************************************
 * @brief   Parses and allocates a system delivery descriptor which should be freed
 *          by calling STB_SIReleaseDelSysDesc. Currently supports DVB-T/T2, DVB-C and
 *          DVB-S/S2 descriptors.
 * @param   data - pointer to SI data, where the first byte is the desc tag
 * @param   type - returned with type of descriptor parsed
 * @param   desc - returned with pointer to parsed and descriptor
 * @return  TRUE if a descriptor is parsed, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_SIParseDelSysDesc(U8BIT *data, SI_DELIVERY_SYS_DESC_TYPE *type, SI_DELIVERY_SYS_DESC **desc)
{
   BOOLEAN retval;

   FUNCTION_START(STB_SIParseDelSysDesc);

   retval = TRUE;

   switch (data[0])
   {
      case TERR_DEL_SYS_DTAG:
         *type = SI_DEL_SYS_DESC_TYPE_TERR;
         ParseTerrestrialDeliverySysDescriptor(&data[1], desc, DEBUG_TERR_DEL_SYS_DESC_VALUE);
         if (*desc == NULL)
         {
            retval = FALSE;
         }
         break;

      case SAT_DEL_SYS_DTAG:
         *type = SI_DEL_SYS_DESC_TYPE_SAT;
         ParseSatelliteDeliverySysDescriptor(&data[1], desc, DEBUG_SAT_DEL_SYS_DESC_VALUE);
         if (*desc == NULL)
         {
            retval = FALSE;
         }
         break;

      case CABLE_DEL_SYS_DTAG:
         *type = SI_DEL_SYS_DESC_TYPE_CABLE;
         ParseCableDeliverySysDescriptor(&data[1], desc, DEBUG_CABLE_DEL_SYS_DESC_VALUE);
         if (*desc == NULL)
         {
            retval = FALSE;
         }
         break;

      case EXT_DTAG:
      {
         /* Extended tag uses the 3rd byte to identify the descriptor */
         switch (data[2])
         {
            case T2_DELIVERY_SYS_DTAG:
               *type = SI_DEL_SYS_DESC_TYPE_TERR;
               ParseT2DeliverySysDescriptor(&data[1], desc, DEBUG_TERR_DEL_SYS_DESC_VALUE);
               if (*desc == NULL)
               {
                  retval = FALSE;
               }
               break;

            default:
               retval = FALSE;
               break;
         }
         break;
      }

      default:
         retval = FALSE;
         break;
   }

   FUNCTION_FINISH(STB_SIParseDelSysDesc);

   return(retval);
}

/*!**************************************************************************
 * @brief   Parses a service descriptor, tag 0x48, allocating SI strings that must
 *          be freed using STB_SIReleaseStringDesc.
 * @param   data - pointer to SI data, where the first byte is the desc tag
 * @param   type - pointer to return service type
 * @param   provider - an SI_STRING_DESC will be allocated to return the provider string, if present
 * @param   name - an SI_STRING_DESC will be allocated to return the service name string, if present
 * @return  TRUE if a descriptor is parsed, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_SIParseServiceDescriptor(U8BIT *data, U8BIT *type, SI_STRING_DESC **provider,
   SI_STRING_DESC **name)
{
   BOOLEAN retval;

   FUNCTION_START(STB_SIParseServiceDescriptor);

   if (data[0] == SERVICE_DTAG)
   {
      *provider = NULL;
      *name = NULL;

      ParseServiceDescriptor(&data[1], type, provider, name, DEBUG_SERVICE_DESC_VALUE);
      retval = TRUE;
   }
   else
   {
      retval = FALSE;
   }

   FUNCTION_FINISH(STB_SIParseServiceDescriptor);

   return(retval);
}

/*!**************************************************************************
 * @brief   Parses a short event descriptor, tag 0x4d, that must be released by
 *          calling STB_SIReleaseShortEventDescArray
 * @param   data - pointer to SI data, where the first byte is the desc tag
 * @param   event_desc - pointer to return an array (always 1) of parsed descriptors
 * @return  TRUE if a descriptor is parsed, FALSE otherwise
 ****************************************************************************/
BOOLEAN STB_SIParseShortEventDescriptor(U8BIT *data, SI_SHORT_EVENT_DESC **event_desc)
{
   BOOLEAN retval;
   U8BIT num;

   FUNCTION_START(STB_SIParseShortEventDescriptor);

   retval = FALSE;

   if (data[0] == SHORT_EVENT_DTAG)
   {
      num = 0;
      *event_desc = NULL;

      ParseShortEventDescriptor(&data[1], &num, event_desc, DEBUG_SHORT_EVENT_DESC_VALUE);
      if ((num == 1) && (*event_desc != NULL))
      {
         retval = TRUE;
      }
   }

   FUNCTION_FINISH(STB_SIParseShortEventDescriptor);

   return(retval);
}

//--------------------------------------------------------------------------------------------------
// Table releasing functions
//--------------------------------------------------------------------------------------------------


/**
 *

 *
 * @brief   Frees the memory used by the pat table
 *
 * @param   pat_table - pointer to the SI_PAT_TABLE structure to be freed
 *

 *
 */
void STB_SIReleasePatTable(SI_PAT_TABLE *pat_table)
{
   SI_PAT_SERVICE_ENTRY *serv_entry;
   SI_PAT_SERVICE_ENTRY *next_entry;

   FUNCTION_START(STB_SIReleasePatTable);

   ASSERT(pat_table != NULL);

   // release the memory used by the pat table
   if (pat_table != NULL)
   {
      serv_entry = pat_table->service_list;
      while (serv_entry != NULL)
      {
         next_entry = serv_entry->next;
         STB_SIReleasePatStreamEntry(serv_entry);
         serv_entry = next_entry;
      }
      STB_FreeMemory(pat_table);
   }

   FUNCTION_FINISH(STB_SIReleasePatTable);
}

/**
 *

 *
 * @brief   Frees the memory used by the pmt table
 *
 * @param   pmt_table - pointer to the SI_PMT_TABLE structure to be freed
 *

 *
 */
void STB_SIReleasePmtTable(SI_PMT_TABLE *pmt_table)
{
   SI_PMT_STREAM_ENTRY *stream_entry;
   SI_PMT_STREAM_ENTRY *next_entry;

   FUNCTION_START(STB_SIReleasePmtTable);

   ASSERT(pmt_table != NULL);

   // release the memory used by the pmt table
   if (pmt_table != NULL)
   {
      stream_entry = pmt_table->stream_list;
      while (stream_entry != NULL)
      {
         next_entry = stream_entry->next;
         STB_SIReleasePmtStreamEntry(stream_entry);
         stream_entry = next_entry;
      }
      STB_SIReleaseCaDescArray(pmt_table->ca_desc_array, pmt_table->num_ca_entries);
      if (pmt_table->tunnelled_desc_array != NULL)
      {
         STB_FreeMemory(pmt_table->tunnelled_desc_array);
      }
      STB_FreeMemory(pmt_table);
   }

   FUNCTION_FINISH(STB_SIReleasePmtTable);
}

/**
 *

 *
 * @brief   Frees the memory used by the nit table
 *
 * @param   nit_table - pointer to the SI_NIT_TABLE structure to be freed
 *

 *
 */
void STB_SIReleaseNitTable(SI_NIT_TABLE *nit_table)
{
   SI_NIT_TRANSPORT_ENTRY *trans_entry;
   SI_NIT_TRANSPORT_ENTRY *next_entry;
   U16BIT i;

   FUNCTION_START(STB_SIReleaseNitTable);

   ASSERT(nit_table != NULL);

   // release the memory used by the nit table
   if (nit_table != NULL)
   {
      trans_entry = nit_table->transport_list;
      while (trans_entry != NULL)
      {
         next_entry = trans_entry->next;
         STB_SIReleaseNitTransportEntry(trans_entry);
         trans_entry = next_entry;
      }

      STB_SIReleaseStringDesc(nit_table->name_str);
      STB_SIReleaseStringDesc(nit_table->def_authority);
      STB_SIReleaseMultilingNetNameDescArray(nit_table->multiling_net_name_desc_array,
         nit_table->num_multiling_net_names);
      STB_SIReleaseLinkageDescList(nit_table->linkage_desc_list, nit_table->num_linkage_entries);

      if (nit_table->change_notify_array != NULL)
      {
         for (i = 0; i < nit_table->num_change_notifies; i++)
         {
            if (nit_table->change_notify_array[i].change_array != NULL)
            {
               STB_FreeMemory(nit_table->change_notify_array[i].change_array);
            }
         }

         STB_FreeMemory(nit_table->change_notify_array);
      }

      if ((nit_table->num_messages > 0) && (nit_table->message_array != NULL))
      {
         for (i = 0; i < nit_table->num_messages; i++)
         {
            if (nit_table->message_array[i].message != NULL)
            {
               STB_SIReleaseStringDesc(nit_table->message_array[i].message);
            }
         }

         STB_FreeMemory(nit_table->message_array);
      }

      if (nit_table->fta_content_desc != NULL)
      {
         STB_FreeMemory(nit_table->fta_content_desc);
      }

      if (nit_table->ciplus_virtual_channel != NULL)
      {
         STB_SIReleaseStringDesc(nit_table->ciplus_virtual_channel->provider_str);
         STB_SIReleaseStringDesc(nit_table->ciplus_virtual_channel->name_str);
         STB_FreeMemory(nit_table->ciplus_virtual_channel->event_info);
         STB_FreeMemory(nit_table->ciplus_virtual_channel->app_domain_id);
         STB_FreeMemory(nit_table->ciplus_virtual_channel);
      }

      STB_SIReleaseTargetRegionNameList(nit_table->target_region_name_list);
      STB_SIReleaseTargetRegionList(nit_table->target_region_list);

      STB_SIReleaseFreesatLinkageDesc(nit_table->freesat_linkage_desc);
      STB_SIReleaseFreesatPrefixList(nit_table->freesat_prefix_list);

      STB_SIReleaseURILinkageList(nit_table->uri_linkage_list);

      STB_FreeMemory(nit_table);
   }

   FUNCTION_FINISH(STB_SIReleaseNitTable);
}

/**
 *

 *
 * @brief   Frees the memory used by the sdt table
 *
 * @param   sdt_table - pointer to the SI_SDT_TABLE structure to be freed
 *

 *
 */
void STB_SIReleaseSdtTable(SI_SDT_TABLE *sdt_table)
{
   SI_SDT_SERVICE_ENTRY *serv_entry;
   SI_SDT_SERVICE_ENTRY *next_entry;

   FUNCTION_START(STB_SIReleaseSdtTable);

   ASSERT(sdt_table != NULL);

   // release the memory used by the sdt table
   if (sdt_table != NULL)
   {
      serv_entry = sdt_table->service_list;
      while (serv_entry != NULL)
      {
         next_entry = serv_entry->next;
         STB_SIReleaseSdtServiceEntry(serv_entry);
         serv_entry = next_entry;
      }
      STB_FreeMemory(sdt_table);
   }

   FUNCTION_FINISH(STB_SIReleaseSdtTable);
}

/**
 *

 *
 * @brief   Frees the memory used by the bat table
 *
 * @param   sdt_table - pointer to the SI_BAT_TABLE structure to be freed
 *

 *
 */
void STB_SIReleaseBatTable(SI_BAT_TABLE *bat_table)
{
   SI_BAT_TRANSPORT_ENTRY *trans_entry;
   SI_BAT_TRANSPORT_ENTRY *next_entry;
   SI_BAT_FREESAT_REGION *region_ptr;
   SI_BAT_FREESAT_GROUP_NAME_ENTRY *name_entry;
   SI_BAT_FREESAT_GROUP_NAME_ENTRY *next_name_entry;
   SI_BAT_FREESAT_SERV_GROUP_ENTRY *serv_entry;
   SI_BAT_FREESAT_SERV_GROUP_ENTRY *next_serv_entry;
   SI_BAT_FREESAT_IACTIVE_STORAGE_DESC *iactive_storage_desc;
   SI_BAT_FREESAT_IACTIVE_STORAGE_DESC *next_desc;
   SI_BAT_FREESAT_INFO_LOCATION *location_desc;
   U16BIT i;

   FUNCTION_START(STB_SIReleaseBatTable);

   ASSERT(bat_table != NULL);

   /* release the memory used by the bat table */
   if (bat_table != NULL)
   {
      trans_entry = bat_table->transport_list;
      while (trans_entry != NULL)
      {
         next_entry = trans_entry->next;
         STB_SIReleaseBatTransportEntry(trans_entry);
         trans_entry = next_entry;
      }

      STB_SIReleaseStringDesc(bat_table->bouquet_name);
      STB_SIReleaseLinkageDescList(bat_table->linkage_desc_list, bat_table->num_linkage_entries);

      while (bat_table->region_list != NULL)
      {
         region_ptr = bat_table->region_list;
         if (region_ptr->region_name != NULL)
         {
            STB_SIReleaseStringDesc(region_ptr->region_name);
         }

         bat_table->region_list = region_ptr->next;

         STB_FreeMemory(region_ptr);
      }

      /* Release Group names */
      name_entry = bat_table->group_name_list;
      while (name_entry != NULL)
      {
         next_name_entry = name_entry->next_group;
         for (i = 0; i < name_entry->num_group_names; i++)
         {
            STB_SIReleaseStringDesc(name_entry->string_array[i].name_str);
         }
         STB_FreeMemory(name_entry->string_array);
         STB_FreeMemory(name_entry);
         name_entry = next_name_entry;
      }

      /* Release group services */
      serv_entry = bat_table->serv_group_array;
      while (serv_entry != NULL)
      {
         next_serv_entry = serv_entry->next_group;
         //for (i=0; i<serv_entry->num_services; i++)
         {
            STB_FreeMemory(serv_entry->freesat_id);
         }
         STB_FreeMemory(serv_entry);
         serv_entry = next_serv_entry;
      }

      /* Release interactive storage descriptors */
      iactive_storage_desc = bat_table->iactive_storage_desc_list;
      while (iactive_storage_desc != NULL)
      {
         STB_FreeMemory(iactive_storage_desc->data);
         next_desc = iactive_storage_desc->next_desc;
         STB_FreeMemory(iactive_storage_desc);
         iactive_storage_desc = next_desc;
      }

      /* Release the info location list */
      location_desc = bat_table->info_location_list;
      while (location_desc != NULL)
      {
         bat_table->info_location_list = location_desc->next;
         STB_FreeMemory(location_desc);
         location_desc = bat_table->info_location_list;
      }

      STB_SIReleaseFreesatPrefixList(bat_table->freesat_prefix_list);

      STB_SIReleaseURILinkageList(bat_table->uri_linkage_list);

      STB_FreeMemory(bat_table);
   }

   FUNCTION_FINISH(STB_SIReleaseBatTable);
}

/**
 *

 *
 * @brief   Frees the memory used by the eit table
 *
 * @param   unt_table - pointer to the SI_UNT_TABLE structure to be freed
 *

 *
 */
void STB_SIReleaseUntTable(SI_UNT_TABLE *unt_table)
{
   FUNCTION_START(STB_SIReleaseEitTable);

   if (unt_table != NULL)
   {
      if (NULL != unt_table->ssu_location_desc)
      {
         STB_SIReleaseSsuLocationDesc(unt_table->ssu_location_desc);
      }

      if (NULL != unt_table->del_sys_desc)
      {
         STB_SIReleaseDelSysDesc(unt_table->del_sys_desc, unt_table->del_sys_desc_type);
      }

      STB_FreeMemory(unt_table);
   }
         
   FUNCTION_FINISH(STB_SIReleaseEitTable);
}

/**
 *

 *
 * @brief   Frees the memory used by the cat table
 *
 * @param   cat_table - pointer to the SI_CAT_TABLE structure to be freed
 *

 *
 */
void STB_SIReleaseCatTable(SI_CAT_TABLE *cat_table)
{
   FUNCTION_START(STB_SIReleaseCatTable);

   ASSERT(cat_table != NULL);

   // release the memory used by the pmt table
   if (cat_table != NULL)
   {
      STB_SIReleaseCaDescArray(cat_table->ca_desc_array, cat_table->num_ca_entries);
      STB_FreeMemory(cat_table);
   }

   FUNCTION_FINISH(STB_SIReleaseCatTable);
}

/**
 *

 *
 * @brief   Frees the memory used by the eit table
 *
 * @param   eit_table - pointer to the SI_EIT_TABLE structure to be freed
 *

 *
 */
void STB_SIReleaseEitTable(SI_EIT_TABLE *eit_table)
{
   SI_EIT_EVENT_ENTRY *event_entry;
   SI_EIT_EVENT_ENTRY *next_entry;

   FUNCTION_START(STB_SIReleaseEitTable);

   ASSERT(eit_table != NULL);

   // release the memory used by the eit table
   if (eit_table != NULL)
   {
      event_entry = eit_table->event_list;
      while (event_entry != NULL)
      {
         next_entry = event_entry->next;
         STB_SIReleaseEitEventEntry(event_entry);
         event_entry = next_entry;
      }

      STB_FreeMemory(eit_table);
   }

   FUNCTION_FINISH(STB_SIReleaseEitTable);
}

/**
 *

 *
 * @brief   Frees the memory used by the time table (tdt or tot)
 *
 * @param   time_table - pointer to the SI_TIME_TABLE structure to be freed
 *

 *
 */
void STB_SIReleaseTimeTable(SI_TIME_TABLE *time_table)
{
   FUNCTION_START(STB_SIReleaseTimeTable);

   ASSERT(time_table != NULL);

   if (time_table != NULL)
   {
      // release the memory used by the table
      STB_SIReleaseLtoDescArray(time_table->lto_desc_array, time_table->num_lto_entries);
      STB_FreeMemory(time_table);
   }

   FUNCTION_FINISH(STB_SIReleaseTimeTable);
}

/*!**************************************************************************
 * @brief   Frees memory used by an RCT table
 * @param   rct_table - pointer to the SI_RCT_TABLE structure to be freed
 ****************************************************************************/
void STB_SIReleaseRctTable(SI_RCT_TABLE *rct_table)
{
   SI_RCT_SUBTABLE *sub_ptr;
   SI_RCT_SUBTABLE *next_subtable;

   FUNCTION_START(STB_SIReleaseRctTable);

   ASSERT(rct_table != NULL);

   if (rct_table != NULL)
   {
      sub_ptr = rct_table->subtables;
      while (sub_ptr != NULL)
      {
         next_subtable = sub_ptr->next;
         STB_SIReleaseRctSubtable(sub_ptr);
         sub_ptr = next_subtable;
      }

      STB_FreeMemory(rct_table);
   }

   FUNCTION_FINISH(STB_SIReleaseRctTable);
}

//--------------------------------------------------------------------------------------------------
// Table entry releasing functions
//--------------------------------------------------------------------------------------------------
// the following functions can be used to release individual table entry structures from the parsed
// tables. They are mainly intended for releasing entries which have been transferred
// out of the parsed tables by the application SI handler and are therefore not released when the
// table is released.

/**
 *

 *
 * @brief   Frees the memory used by the table entry specified
 *
 * @param   entry_ptr - the entry to be released
 *

 *
 */
void STB_SIReleasePatStreamEntry(SI_PAT_SERVICE_ENTRY *entry_ptr)
{
   FUNCTION_START(STB_SIReleasePatStreamEntry);

   STB_FreeMemory(entry_ptr);

   FUNCTION_FINISH(STB_SIReleasePatStreamEntry);
}

/**
 *

 *
 * @brief   Frees the memory used by the table entry specified
 *
 * @param   entry_ptr - the entry to be released
 *

 *
 */
void STB_SIReleasePmtStreamEntry(SI_PMT_STREAM_ENTRY *entry_ptr)
{
   FUNCTION_START(STB_SIReleasePmtStreamEntry);

   STB_SIReleaseIsoLangDescArray(entry_ptr->iso_lang_desc_array,
      entry_ptr->num_iso_lang_entries);
   STB_SIReleaseCaDescArray(entry_ptr->ca_desc_array, entry_ptr->num_ca_entries);
   STB_SIReleaseSubtitleDescArray(entry_ptr->subtitle_desc_array,
      entry_ptr->num_subtitle_entries);
   STB_SIReleaseTeletextDescArray(entry_ptr->teletext_desc_array,
      entry_ptr->num_teletext_entries);

   if (entry_ptr->ac3_descriptor != NULL)
   {
      STB_FreeMemory(entry_ptr->ac3_descriptor);
   }

   if (entry_ptr->aac_descriptor != NULL)
   {
      STB_FreeMemory(entry_ptr->aac_descriptor);
   }

   if (entry_ptr->app_sig_desc_array != NULL)
   {
      STB_FreeMemory(entry_ptr->app_sig_desc_array);
   }

   if (entry_ptr->tag_array_ptr != NULL)
   {
      STB_FreeMemory(entry_ptr->tag_array_ptr);
   }

   if (entry_ptr->audio_desc != NULL)
   {
      STB_FreeMemory(entry_ptr->audio_desc);
   }

#if 0
   if (entry_ptr->service_move != NULL)
   {
      STB_FreeMemory(entry_ptr->service_move);
   }
#endif

   if (entry_ptr->tunnelled_desc_array != NULL)
   {
      STB_FreeMemory(entry_ptr->tunnelled_desc_array);
   }

   STB_FreeMemory(entry_ptr);

   FUNCTION_FINISH(STB_SIReleasePmtStreamEntry);
}

/**
 *

 *
 * @brief   Frees the memory used by the table entry specified
 *
 * @param   entry_ptr - the entry to be released
 *

 *
 */
void STB_SIReleaseNitTransportEntry(SI_NIT_TRANSPORT_ENTRY *entry_ptr)
{
   FUNCTION_START(STB_SIReleaseNitTransportEntry);

   STB_SIReleaseDelSysDesc(entry_ptr->del_sys_desc, entry_ptr->del_sys_desc_type);
   STB_SIReleaseFreqListDescArray(entry_ptr->freq_list);
   STB_SIReleaseServListDescArray(entry_ptr->serv_list_desc_array,
      entry_ptr->num_serv_list_entries);
   STB_SIReleaseLcnDescArray(entry_ptr->lcn_desc_array, entry_ptr->num_lcn_entries);
   STB_SIReleaseStringDesc(entry_ptr->def_authority);
   STB_SIReleaseLcnDescArray(entry_ptr->hd_lcn_desc_array, entry_ptr->num_hd_lcn_entries);
   STB_SIReleaseNordigLcn2DescArray(entry_ptr->nordig_lcn_desc_array, entry_ptr->num_nordig_lcn_entries);

   if (entry_ptr->serv_attr_array != NULL)
   {
      STB_FreeMemory(entry_ptr->serv_attr_array);
   }

   if (entry_ptr->fta_content_desc != NULL)
   {
      STB_FreeMemory(entry_ptr->fta_content_desc);
   }

   STB_SIReleaseTargetRegionList(entry_ptr->target_region_list);
   STB_SIReleaseCIPlusServiceList(entry_ptr->ciplus_service_list);

   STB_FreeMemory(entry_ptr);

   FUNCTION_FINISH(STB_SIReleaseNitTransportEntry);
}

/**
 *

 *
 * @brief   Frees the memory used by the table entry specified
 *
 * @param   entry_ptr - the entry to be released
 *

 *
 */
void STB_SIReleaseBatTransportEntry(SI_BAT_TRANSPORT_ENTRY *entry_ptr)
{
   SI_BAT_FREESAT_REGION_LCN_ENTRY *lcn_entry;
   SI_BAT_FREESAT_REGION_LCN_ENTRY *next_entry;

   FUNCTION_START(STB_SIReleaseBatTransportEntry);

   STB_SIReleaseLcnDescArray(entry_ptr->lcn_desc_array, entry_ptr->num_lcn_entries);

   lcn_entry = entry_ptr->lcn_region_list;
   while (lcn_entry != NULL)
   {
      next_entry = lcn_entry->next;
      STB_SIReleaseBatLcnEntry(lcn_entry);
      lcn_entry = next_entry;
   }

   STB_SIReleaseServListDescArray(entry_ptr->serv_list_desc_array, entry_ptr->num_serv_list_entries);

   STB_FreeMemory(entry_ptr);

   FUNCTION_FINISH(STB_SIReleaseBatTransportEntry);
}

/**
 *

 *
 * @brief   Frees the memory used by the table entry specified
 *
 * @param   entry_ptr - the entry to be released
 *

 *
 */
void STB_SIReleaseBatLcnEntry(SI_BAT_FREESAT_REGION_LCN_ENTRY *entry_ptr)
{
   SI_FREESAT_LCN *freesat_lcn;
   SI_FREESAT_LCN *next_entry;

   FUNCTION_START(STB_SIReleaseBatLcnEntry);

   freesat_lcn = entry_ptr->freesat_lcn_list;
   while (freesat_lcn != NULL)
   {
      next_entry = freesat_lcn->next;
      STB_FreeMemory((void *)freesat_lcn);
      freesat_lcn = next_entry;
   }

   STB_FreeMemory(entry_ptr);

   FUNCTION_FINISH(STB_SIReleaseBatLcnEntry);
}

/**
 *

 *
 * @brief   Frees the memory used by the table entry specified
 *
 * @param   entry_ptr - the entry to be released
 *

 *
 */
void STB_SIReleaseSdtServiceEntry(SI_SDT_SERVICE_ENTRY *entry_ptr)
{
   FUNCTION_START(STB_SIReleaseSdtServiceEntry);

   STB_SIReleaseStringDesc(entry_ptr->name_str);
   STB_SIReleaseStringDesc(entry_ptr->provider_str);
   STB_SIReleaseCaIdDescArray(entry_ptr->ca_id_desc_array, entry_ptr->num_ca_id_entries);
   STB_SIReleaseMultilingServNameDescArray(entry_ptr->multiling_name_desc_array,
      entry_ptr->num_multiling_names);
   STB_SIReleasePrefNameDescArray(entry_ptr->preferred_name_desc_array,
      entry_ptr->num_preferred_names);
   STB_SIReleaseLinkageDescList(entry_ptr->linkage_desc_list, entry_ptr->num_linkage_entries);
   STB_SIReleaseStringDesc(entry_ptr->def_authority);
   STB_SIReleaseStringDesc(entry_ptr->short_name_str);
   STB_SIReleaseMultilingShortNameArray(entry_ptr->multiling_short_name_array,
      entry_ptr->num_multiling_short_names);
   STB_SIReleaseGuidanceDesc(entry_ptr->guidance);

   if (entry_ptr->fta_content_desc != NULL)
   {
      STB_FreeMemory(entry_ptr->fta_content_desc);
   }

   STB_SIReleaseTargetRegionList(entry_ptr->target_region_list);

   if (entry_ptr->ci_protection_desc != NULL)
   {
      STB_FreeMemory(entry_ptr->ci_protection_desc);
   }
   STB_SIReleaseAvailabilityDescriptorList(entry_ptr->serv_avail_desc_list);
   STB_SIReleaseFreesatPrefixList(entry_ptr->freesat_prefix_list);
   STB_SIReleaseURILinkageList(entry_ptr->uri_linkage_list);

   STB_SIReleaseSdtCodeDesc(entry_ptr->sdt_code_desc);

   STB_FreeMemory(entry_ptr);

   FUNCTION_FINISH(STB_SIReleaseSdtServiceEntry);
}

/**
 *

 *
 * @brief   Frees the memory used by the table entry specified
 *
 * @param   entry_ptr - the entry to be released
 *

 *
 */
void STB_SIReleaseEitEventEntry(SI_EIT_EVENT_ENTRY *entry_ptr)
{
   FUNCTION_START(STB_SIReleaseEitEventEntry);

   STB_SIReleaseCaIdDescArray(entry_ptr->ca_id_desc_array, entry_ptr->num_ca_id_entries);
   STB_SIReleaseComponentDescArray(entry_ptr->component_desc_array,
      entry_ptr->num_component_entries);
   STB_SIReleaseMultilingComponentDescArray(entry_ptr->multiling_component_desc_array,
      entry_ptr->num_multiling_component_entries);

   STB_SIReleaseContentDescArray(entry_ptr->content_desc_array, entry_ptr->num_content_entries);
   STB_SIReleaseParentalRatingDescArray(entry_ptr->parental_rating_desc_array,
      entry_ptr->num_parental_rating_entries);
   STB_SIReleaseShortEventDescArray(entry_ptr->short_event_desc_array,
      entry_ptr->num_short_event_entries);
   STB_SIReleaseExtendedEventDescArray(entry_ptr->extended_event_desc_array,
      entry_ptr->num_extended_event_entries);
   STB_SIReleaseCRIDList(entry_ptr->crid_list);
   STB_SIReleaseGuidanceDesc(entry_ptr->guidance);

   if (entry_ptr->fta_content_desc != NULL)
   {
      STB_FreeMemory(entry_ptr->fta_content_desc);
   }
   
   STB_SIReleaseSeriesDesc(entry_ptr->series_desc);

   STB_FreeMemory(entry_ptr);

   FUNCTION_FINISH(STB_SIReleaseEitEventEntry);
}

/*!**************************************************************************
 * @brief   Frees memory used by an RCT subtable
 * @param   sub_ptr - pointer to the SI_RCT_SUBTABLE structure to be freed
 ****************************************************************************/
void STB_SIReleaseRctSubtable(SI_RCT_SUBTABLE *sub_ptr)
{
   SI_RCT_SUBTABLE_DATA *data_ptr;
   SI_RCT_SUBTABLE_DATA *next_data_ptr;

   FUNCTION_START(STB_SIReleaseRctSubtable);

   if (sub_ptr != NULL)
   {
      data_ptr = sub_ptr->data;
      while (data_ptr != NULL)
      {
         next_data_ptr = data_ptr->next;
         STB_SIReleaseRctSubtableData(data_ptr);
         data_ptr = next_data_ptr;
      }

      STB_FreeMemory(sub_ptr);
   }

   FUNCTION_FINISH(STB_SIReleaseRctSubtable);
}

/*!**************************************************************************
 * @brief   Frees memory used by an SI_RCT_SUBTABLE_DATA structure
 * @param   data_ptr - pointer to the SI_RCT_SUBTABLE_DATA structure to be freed
 ****************************************************************************/
void STB_SIReleaseRctSubtableData(SI_RCT_SUBTABLE_DATA *data_ptr)
{
   U8BIT i;

   FUNCTION_START(STB_SIReleaseRctSubtableData);

   if (data_ptr != NULL)
   {
      if ((data_ptr->link_count > 0) && (data_ptr->link_array != NULL))
      {
         for (i = 0; i < data_ptr->link_count; i++)
         {
            STB_SIReleaseRctLinkInfo(&data_ptr->link_array[i]);
         }

         STB_FreeMemory(data_ptr->link_array);
      }

      STB_SIReleaseImageIconDescArray(data_ptr->icon_array, data_ptr->num_icons);

      STB_FreeMemory(data_ptr);
   }

   FUNCTION_FINISH(STB_SIReleaseRctSubtableData);
}

//--------------------------------------------------------------------------------------------------
// Descriptor releasing functions
//--------------------------------------------------------------------------------------------------
// the following functions can be used to release individual descriptor arrays from the parsed
// tables. They are mainly intended for releasing descriptor arrays which have been transferred
// out of the parsed tables by the application SI handler (by setting the structure element pointing
// to the descriptor array to NULL), and are therefore not released when the table is released.

/**
 *

 *
 * @brief   Frees the memory used by the descriptor specified
 *
 * @param   desc - the descriptor to be released
 * @param   type - the type of descriptor to be released
 *

 *
 */
void STB_SIReleaseDelSysDesc(SI_DELIVERY_SYS_DESC *desc, SI_DELIVERY_SYS_DESC_TYPE type)
{
   FUNCTION_START(STB_SIReleaseDelSysDesc);

   if (desc != NULL)
   {
      switch (type)
      {
         case SI_DEL_SYS_DESC_TYPE_SAT:
         {
            break;
         }
         case SI_DEL_SYS_DESC_TYPE_CABLE:
         {
            break;
         }
         case SI_DEL_SYS_DESC_TYPE_TERR:
         {
            if (desc->terr.is_t2)
            {
               if ((desc->terr.u.t2.num_cells > 0) && (desc->terr.u.t2.cell != NULL))
               {
                  STB_FreeMemory(desc->terr.u.t2.cell);
               }
            }
            break;
         }
         default:
         {
            STB_SI_PRINT(("SI_DELIVERY_SYS_DESC_TYPE release not supported!"));
            break;
         }
      }

      STB_FreeMemory(desc);
   }

   FUNCTION_FINISH(STB_SIReleaseDelSysDesc);
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor specified
 *
 * @param   desc - the descriptor to be released
 *

 *
 */
void STB_SIReleaseStringDesc(SI_STRING_DESC *desc)
{
   FUNCTION_START(STB_SIReleaseStringDesc);

   if (desc != NULL)
   {
      STB_FreeMemory(desc);
   }

   FUNCTION_FINISH(STB_SIReleaseStringDesc);
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor array specified
 *
 * @param   desc_array  - the descriptor array to be released
 * @param   num_entries - number of entries in the array
 *

 *
 */
void STB_SIReleaseCaDescArray(SI_CA_DESC *desc_array, U16BIT num_entries)
{
   FUNCTION_START(STB_SIReleaseCaDescArray);

   USE_UNWANTED_PARAM(num_entries);

   if (desc_array != NULL)
   {
      STB_FreeMemory(desc_array);
   }

   FUNCTION_FINISH(STB_SIReleaseCaDescArray);
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor array specified
 *
 * @param   desc_array  - the descriptor array to be released
 * @param   num_entries - number of entries in the array
 *

 *
 */
void STB_SIReleaseCaIdDescArray(U16BIT *desc_array, U8BIT num_entries)
{
   FUNCTION_START(STB_SIReleaseCaIdDescArray);

   USE_UNWANTED_PARAM(num_entries);

   if (desc_array != NULL)
   {
      STB_FreeMemory(desc_array);
   }

   FUNCTION_FINISH(STB_SIReleaseCaIdDescArray);
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor array specified
 *
 * @param   desc_array  - the descriptor array to be released
 * @param   num_entries - number of entries in the array
 *

 *
 */
void STB_SIReleaseComponentDescArray(SI_COMPONENT_DESC *desc_array, U8BIT num_entries)
{
   SI_COMPONENT_DESC *comp_desc_ptr;
   U8BIT i;

   FUNCTION_START(STB_SIReleaseComponentDescArray);
   if (desc_array != NULL)
   {
      comp_desc_ptr = desc_array;
      for (i = 0; i < num_entries; i++, comp_desc_ptr++)
      {
         STB_SIReleaseStringDesc(comp_desc_ptr->desc_str);
      }
      STB_FreeMemory(desc_array);
   }
   FUNCTION_FINISH(STB_SIReleaseComponentDescArray);
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor array specified
 *
 * @param   desc_array  - the descriptor array to be released
 * @param   num_entries - number of entries in the array
 *

 *
 */
void STB_SIReleaseContentDescArray(SI_CONTENT_DESC *desc_array, U8BIT num_entries)
{
   FUNCTION_START(STB_SIReleaseContentDescArray);

   USE_UNWANTED_PARAM(num_entries);

   if (desc_array != NULL)
   {
      STB_FreeMemory(desc_array);
   }
   FUNCTION_FINISH(STB_SIReleaseContentDescArray);
}

/**
 * @brief   Frees the memory used by the frequency list descriptor
 * @param   freq_list - the frequency list to be released
 */
void STB_SIReleaseFreqListDescArray(SI_NIT_FREQUENCY_LIST_DESC *freq_list)
{
   SI_NIT_FREQUENCY_LIST_DESC *next;

   FUNCTION_START(STB_SIReleaseFreqListDescArray);

   if (freq_list != NULL)
   {
      for ( ; freq_list != NULL; freq_list = next)
      {
         next = freq_list->next;
         if (freq_list->frequency_array != NULL)
         {
            STB_FreeMemory(freq_list->frequency_array);
         }
         STB_FreeMemory(freq_list);
      }
   }

   FUNCTION_FINISH(STB_SIReleaseFreqListDescArray);
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor specified
 *
 * @param   desc_array  - the descriptor array to be released
 * @param   num_entries - number of entries in the array
 *

 *
 */
void STB_SIReleaseGuidanceDesc(SI_GUIDANCE_DESC *desc_ptr)
{
   U8BIT index;

   FUNCTION_START(STB_SIReleaseGuidanceDesc);

   if (desc_ptr != NULL)
   {
      if (desc_ptr->strings != NULL)
      {
         for (index = 0; index < desc_ptr->num_langs; index++)
         {
            if (desc_ptr->strings[index] != NULL)
            {
               STB_SIReleaseStringDesc(desc_ptr->strings[index]);
            }
         }

         STB_FreeMemory(desc_ptr->strings);
      }

      if (desc_ptr->lang_codes != NULL)
      {
         STB_FreeMemory(desc_ptr->lang_codes);
      }

      STB_FreeMemory(desc_ptr);
   }

   FUNCTION_FINISH(STB_SIReleaseGuidanceDesc);
}

/*!**************************************************************************
 * @brief   Frees an array of image icons, including the memory used by each one
 * @param   icon_array - image icon array
 * @param   num_icons - number of icons in the array
 ****************************************************************************/
void STB_SIReleaseImageIconDescArray(SI_IMAGE_ICON_DESC *icon_array, U8BIT num_icons)
{
   U8BIT i;

   FUNCTION_START(STB_SIReleaseImageIconDesc);

   if (icon_array != NULL)
   {
      for (i = 0; i < num_icons; i++)
      {
         if (icon_array[i].icon_type != NULL)
         {
            STB_FreeMemory(icon_array[i].icon_type);
         }

         if (icon_array[i].icon_data != NULL)
         {
            STB_FreeMemory(icon_array[i].icon_data);
         }
      }

      STB_FreeMemory(icon_array);
   }

   FUNCTION_FINISH(STB_SIReleaseImageIconDesc);
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor array specified
 *
 * @param   desc_array  - the descriptor array to be released
 * @param   num_entries - number of entries in the array
 *

 *
 */
void STB_SIReleaseIsoLangDescArray(SI_ISO_LANG_DESC *desc_array, U16BIT num_entries)
{
   FUNCTION_START(STB_SIReleaseIsoLangDescArray);

   USE_UNWANTED_PARAM(num_entries);

   if (desc_array != NULL)
   {
      STB_FreeMemory(desc_array);
   }
   FUNCTION_FINISH(STB_SIReleaseIsoLangDescArray);
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor array specified
 *
 * @param   desc_array  - the descriptor array to be released
 * @param   num_entries - number of entries in the array
 *

 *
 */
void STB_SIReleaseLinkageDescList(SI_LINKAGE_DESC_ENTRY *list_ptr, U16BIT num_entries)
{
   SI_LINKAGE_DESC_ENTRY *desc_ptr;
   SI_LINKAGE_DESC_ENTRY *tmp_ptr;

   FUNCTION_START(STB_SIReleaseLinkageDescArray);

   USE_UNWANTED_PARAM(num_entries);

   desc_ptr = list_ptr;
   while (desc_ptr != NULL)
   {
      tmp_ptr = desc_ptr->next;
      STB_FreeMemory(desc_ptr);
      desc_ptr = tmp_ptr;
   }
   FUNCTION_FINISH(STB_SIReleaseLinkageDescArray);
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor array specified
 *
 * @param   desc_array  - the descriptor array to be released
 * @param   num_entries - number of entries in the array
 *

 *
 */
void STB_SIReleaseLtoDescArray(SI_LTO_DESC *desc_array, U16BIT num_entries)
{
   FUNCTION_START(STB_SIReleaseLtoDescArray);

   USE_UNWANTED_PARAM(num_entries);

   if (desc_array != NULL)
   {
      STB_FreeMemory(desc_array);
   }
   FUNCTION_FINISH(STB_SIReleaseLtoDescArray);
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor array specified
 *
 * @param   desc_array  - the descriptor array to be released
 * @param   num_entries - number of entries in the array
 *

 *
 */
void STB_SIReleaseMultilingComponentDescArray(SI_MULTILING_COMPONENT_DESC *desc_array,
   U8BIT num_entries)
{
   U8BIT i;
   SI_MULTILING_COMPONENT_DESC *ml_comp_desc_ptr;

   FUNCTION_START(STB_SIReleaseMultilingComponentDescArray);

   if (desc_array != NULL)
   {
      ml_comp_desc_ptr = desc_array;
      for (i = 0; i < num_entries; i++, ml_comp_desc_ptr++)
      {
         STB_SIReleaseStringDesc(ml_comp_desc_ptr->desc_str);
      }
      STB_FreeMemory(desc_array);
   }

   FUNCTION_FINISH(STB_SIReleaseMultilingComponentDescArray);
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor array specified
 *
 * @param   desc_array  - the descriptor array to be released
 * @param   num_entries - number of entries in the array
 *

 *
 */
void STB_SIReleaseMultilingNetNameDescArray(SI_MULTILING_NET_NAME_DESC *desc_array,
   U16BIT num_entries)
{
   SI_MULTILING_NET_NAME_DESC *mlnn_desc_ptr;
   U16BIT i;

   FUNCTION_START(STB_SIReleaseMultilingNetNameDescArray);

   if (desc_array != NULL)
   {
      mlnn_desc_ptr = desc_array;
      for (i = 0; i < num_entries; i++, mlnn_desc_ptr++)
      {
         STB_SIReleaseStringDesc(mlnn_desc_ptr->name_str);
      }
      STB_FreeMemory(desc_array);
   }

   FUNCTION_FINISH(STB_SIReleaseMultilingNetNameDescArray);
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor array specified
 *
 * @param   desc_array  - the descriptor array to be released
 * @param   num_entries - number of entries in the array
 *

 *
 */
void STB_SIReleaseMultilingServNameDescArray(SI_MULTILING_SERV_NAME_DESC *desc_array,
   U16BIT num_entries)
{
   SI_MULTILING_SERV_NAME_DESC *ml_name_desc_ptr;
   U16BIT i;

   FUNCTION_START(STB_SIReleaseMultilingServNameDescArray);

   if (desc_array != NULL)
   {
      ml_name_desc_ptr = desc_array;
      for (i = 0; i < num_entries; i++, ml_name_desc_ptr++)
      {
         STB_SIReleaseStringDesc(ml_name_desc_ptr->name_str);
         STB_SIReleaseStringDesc(ml_name_desc_ptr->provider_str);
      }
      STB_FreeMemory(desc_array);
   }

   FUNCTION_FINISH(STB_SIReleaseMultilingServNameDescArray);
}

void STB_SIReleaseMultilingShortNameArray(SI_MULTILING_SHORT_NAME_DESC *desc_array,
   U16BIT num_entries)
{
   SI_MULTILING_SHORT_NAME_DESC *name_desc_ptr;
   U16BIT i;

   FUNCTION_START(STB_SIReleaseMultilingShortNameArray);

   if (desc_array != NULL)
   {
      name_desc_ptr = desc_array;
      for (i = 0; i < num_entries; i++, name_desc_ptr++)
      {
         STB_SIReleaseStringDesc(name_desc_ptr->name_str);
      }
      STB_FreeMemory(desc_array);
   }

   FUNCTION_FINISH(STB_SIReleaseMultilingShortNameArray);
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor array specified
 *
 * @param   desc_array  - the descriptor array to be released
 * @param   num_entries - number of entries in the array
 *

 *
 */
void STB_SIReleaseParentalRatingDescArray(SI_PARENTAL_RATING_DESC *desc_array, U8BIT num_entries)
{
   FUNCTION_START(STB_SIReleaseParentalRatingDescArray);

   USE_UNWANTED_PARAM(num_entries);

   if (desc_array != NULL)
   {
      STB_FreeMemory(desc_array);
   }
   FUNCTION_FINISH(STB_SIReleaseParentalRatingDescArray);
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor array specified
 *
 * @param   desc_array  - the descriptor array to be released
 * @param   num_entries - number of entries in the array
 *

 *
 */
void STB_SIReleaseServListDescArray(SI_SERV_LIST_DESC *desc_array, U16BIT num_entries)
{
   FUNCTION_START(STB_SIReleaseServListDescArray);

   USE_UNWANTED_PARAM(num_entries);

   if (desc_array != NULL)
   {
      STB_FreeMemory(desc_array);
   }
   FUNCTION_FINISH(STB_SIReleaseServListDescArray);
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor array specified
 *
 * @param   desc_array  - the descriptor array to be released
 * @param   num_entries - number of entries in the array
 *

 *
 */
void STB_SIReleaseShortEventDescArray(SI_SHORT_EVENT_DESC *desc_array, U8BIT num_entries)
{
   SI_SHORT_EVENT_DESC *sevnt_desc_ptr;
   U8BIT i;

   FUNCTION_START(STB_SIReleaseShortEventDescArray);

   if (desc_array != NULL)
   {
      sevnt_desc_ptr = desc_array;
      for (i = 0; i < num_entries; i++, sevnt_desc_ptr++)
      {
         STB_SIReleaseStringDesc(sevnt_desc_ptr->desc_str);
         STB_SIReleaseStringDesc(sevnt_desc_ptr->name_str);
      }
      STB_FreeMemory(desc_array);
   }

   FUNCTION_FINISH(STB_SIReleaseShortEventDescArray);
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor array specified
 *
 * @param   desc_array  - the descriptor array to be released
 * @param   num_entries - number of entries in the array
 *

 *
 */
void STB_SIReleaseExtendedEventDescArray(SI_EXTENDED_EVENT_DESC *desc_array, U8BIT num_entries)
{
   SI_EXTENDED_EVENT_DESC *evnt_desc_ptr;
   U8BIT i, j;

   FUNCTION_START(STB_SIReleaseExtendedEventDescArray);

   if (desc_array != NULL)
   {
      evnt_desc_ptr = desc_array;
      for (i = 0; i < num_entries; i++, evnt_desc_ptr++)
      {
         for (j = 0; j < evnt_desc_ptr->num_items; j++)
         {
            STB_SIReleaseStringDesc(evnt_desc_ptr->item_desc_array[j]);
            STB_SIReleaseStringDesc(evnt_desc_ptr->item_text_array[j]);
         }

         if (evnt_desc_ptr->item_desc_array != NULL)
         {
            STB_FreeMemory(evnt_desc_ptr->item_desc_array);
         }
         if (evnt_desc_ptr->item_text_array != NULL)
         {
            STB_FreeMemory(evnt_desc_ptr->item_text_array);
         }

         STB_SIReleaseStringDesc(evnt_desc_ptr->text_str);
      }

      STB_FreeMemory(desc_array);
   }

   FUNCTION_FINISH(STB_SIReleaseExtendedEventDescArray);
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor specified
 *
 * @param   desc_array  - the descriptor array to be released
 *

 *
 */
void STB_SIReleaseSsuLocationDesc(SI_SSU_LOCATION_DESC *desc_ptr)
{
   U8BIT index;

   FUNCTION_START(STB_SIReleaseSsuLocationDesc);

   if (desc_ptr != NULL)
   {
      if (desc_ptr->private_data_byte != NULL)
      {
         STB_FreeMemory(desc_ptr->private_data_byte);
      }

      STB_FreeMemory(desc_ptr);
   }

   FUNCTION_FINISH(STB_SIReleaseSsuLocationDesc);
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor array specified
 *
 * @param   desc_array  - the descriptor array to be released
 * @param   num_entries - number of entries in the array
 *

 *
 */
void STB_SIReleaseSubtitleDescArray(SI_SUBTITLE_DESC *desc_array, U16BIT num_entries)
{
   FUNCTION_START(STB_SIReleaseSubtitleDescArray);

   USE_UNWANTED_PARAM(num_entries);

   if (desc_array != NULL)
   {
      STB_FreeMemory(desc_array);
   }
   FUNCTION_FINISH(STB_SIReleaseSubtitleDescArray);
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor list specified
 *
 * @param   desc_list  - the descriptor list to be released
 *

 *
 */
void STB_SIReleaseTargetRegionNameList(SI_NIT_TARGET_REGION_NAME_DESC *desc_list)
{
   SI_NIT_TARGET_REGION_NAME_DESC *ptr;
   SI_NIT_TARGET_REGION_NAME_DESC *next;
   U8BIT i;

   FUNCTION_START(STB_SIReleaseTargetRegionNameList);

   for (ptr = desc_list; ptr != NULL; ptr = next)
   {
      next = ptr->next;

      if (ptr->name_array != NULL)
      {
         for (i = 0; i < ptr->num_names; i++)
         {
            STB_SIReleaseStringDesc(ptr->name_array[i].region_name);
         }

         STB_FreeMemory(ptr->name_array);
      }

      STB_FreeMemory(ptr);
   }

   FUNCTION_FINISH(STB_SIReleaseTargetRegionNameList);
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor list specified
 *
 * @param   desc_list  - the descriptor list to be released
 *

 *
 */
void STB_SIReleaseTargetRegionList(SI_TARGET_REGION_DESC *desc_list)
{
   SI_TARGET_REGION_DESC *cptr;
   SI_TARGET_REGION_DESC *cnext;
   SI_TARGET_REGION *rptr;
   SI_TARGET_REGION *rnext;

   FUNCTION_START(STB_SIReleaseTargetRegionList);

   for (cptr = desc_list; cptr != NULL; cptr = cnext)
   {
      cnext = cptr->next;

      for (rptr = cptr->region_list; rptr != NULL; rptr = rnext)
      {
         rnext = rptr->next;

         STB_FreeMemory(rptr);
      }

      STB_FreeMemory(cptr);
   }

   FUNCTION_FINISH(STB_SIReleaseTargetRegionList);
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor list specified
 *
 * @param   desc_list  - the descriptor list to be released
 *

 *
 */
void STB_SIReleaseAvailabilityDescriptorList(SI_SERV_AVAIL_DESC *desc_list)
{
   SI_SERV_AVAIL_DESC *desc, *next;

   desc = desc_list;
   while (desc != NULL)
   {
      if (desc->cell_ids != NULL)
      {
#ifdef DEBUG_SERV_AVAIL_DESC
         STB_SI_PRINT(("STB_SIReleaseSdtServiceEntry: Freeing cell_ids"));
#endif
         STB_FreeMemory(desc->cell_ids);
      }
      next = desc->next;
#ifdef DEBUG_SERV_AVAIL_DESC
      STB_SI_PRINT(("STB_SIReleaseSdtServiceEntry: Freeing serv_avail_desc"));
#endif
      STB_FreeMemory(desc);
      desc = next;
   }
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor array specified
 *
 * @param   desc_array  - the descriptor array to be released
 * @param   num_entries - number of entries in the array
 *

 *
 */
void STB_SIReleaseTeletextDescArray(SI_TELETEXT_DESC *desc_array, U16BIT num_entries)
{
   FUNCTION_START(STB_SIReleaseTeletextDescArray);

   USE_UNWANTED_PARAM(num_entries);

   if (desc_array != NULL)
   {
      STB_FreeMemory(desc_array);
   }
   FUNCTION_FINISH(STB_SIReleaseTeletextDescArray);
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor array specified
 *
 * @param   desc_array  - the descriptor array to be released
 * @param   num_entries - number of entries in the array
 *

 *
 */
void STB_SIReleaseLcnDescArray(SI_LCN_DESC *desc_array, U16BIT num_entries)
{
   FUNCTION_START(STB_SIReleaseLcnDescArray);

   USE_UNWANTED_PARAM(num_entries);

   if (desc_array != NULL)
   {
      STB_FreeMemory(desc_array);
   }
   FUNCTION_FINISH(STB_SIReleaseLcnDescArray);
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor array specified
 *
 * @param   desc_array  - the descriptor array to be released
 * @param   num_entries - number of entries in the array
 *

 *
 */
void STB_SIReleaseNordigLcn2DescArray(SI_NORDIG_LCN_DESC *desc_array, U16BIT num_entries)
{
   U16BIT i;

   FUNCTION_START(STB_SIReleaseNordigLcn2DescArray);

   if (desc_array != NULL)
   {
      for (i = 0; i < num_entries; i++)
      {
         STB_SIReleaseStringDesc(desc_array[i].chan_list_name);

         if (desc_array[i].serv_array != NULL)
         {
            STB_FreeMemory(desc_array[i].serv_array);
         }
      }

      STB_FreeMemory(desc_array);
   }
   FUNCTION_FINISH(STB_SIReleaseNordigLcn2DescArray);
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor array specified
 *
 * @param   desc_array  - the descriptor array to be released
 * @param   num_entries - number of entries in the array
 *

 *
 */
void STB_SIReleasePrefNameDescArray(SI_PREFERRED_NAME_DESC *desc_array, U16BIT num_entries)
{
   SI_PREFERRED_NAME_DESC *pref_name_desc_ptr;
   U16BIT i;

   FUNCTION_START(STB_SIReleasePrefNameDescArray);

   if (desc_array != NULL)
   {
      pref_name_desc_ptr = desc_array;
      for (i = 0; i < num_entries; i++, pref_name_desc_ptr++)
      {
         STB_SIReleaseStringDesc(pref_name_desc_ptr->name_str);
      }
      STB_FreeMemory(desc_array);
   }
   FUNCTION_FINISH(STB_SIReleasePrefNameDescArray);
}

/**
 *

 *
 * @brief   Frees the memory used by the descriptor list specified
 *
 * @param   crid_list  - the list to be released
 *

 *
 */
void STB_SIReleaseCRIDList(SI_CRID_DESC *crid_list)
{
   SI_CRID_DESC *next_crid_ptr;

   FUNCTION_START(STB_SIReleaseCRIDList);

   while (crid_list != NULL)
   {
      STB_SIReleaseStringDesc(crid_list->crid_str);
      next_crid_ptr = crid_list->next;
      STB_FreeMemory(crid_list);
      crid_list = next_crid_ptr;
   }

   FUNCTION_FINISH(STB_SIReleaseCRIDList);
}

/*!**************************************************************************
 * @brief   Frees memory used by an SI_RCT_LINK_INFO structure
 * @param   data_ptr - pointer to the SI_RCT_LINK_INFO structure
 ****************************************************************************/
void STB_SIReleaseRctLinkInfo(SI_RCT_LINK_INFO *link_info)
{
   U8BIT i;

   FUNCTION_START(STB_SIReleaseRctLinkInfo);

   if (link_info != NULL)
   {
      if (link_info->uri_string != NULL)
      {
         STB_FreeMemory(link_info->uri_string);
      }

      if ((link_info->num_items > 0) && (link_info->promo_text_array != NULL))
      {
         for (i = 0; i < link_info->num_items; i++)
         {
            STB_SIReleaseStringDesc(link_info->promo_text_array[i].string);
         }

         STB_FreeMemory(link_info->promo_text_array);
      }

      if (link_info->event_desc != NULL)
      {
         STB_SIReleaseShortEventDescArray(link_info->event_desc, 1);
      }

      STB_SIReleaseImageIconDescArray(link_info->icon_array, link_info->num_icons);
   }

   FUNCTION_FINISH(STB_SIReleaseRctLinkInfo);
}

/*!**************************************************************************
 * @brief   Frees memory used by an SI_FREESAT_LINKAGE_DESC structure
 * @param   desc - pointer to the SI_FREESAT_LINKAGE_DESC
 ****************************************************************************/
void STB_SIReleaseFreesatLinkageDesc(SI_FREESAT_LINKAGE_DESC *desc)
{
   SI_FREESAT_LINKAGE_DESC *next;

   FUNCTION_START(STB_SIReleaseFreesatLinkageDesc);

   while (desc != NULL)
   {
      next = desc->next;

      if (desc->data_types != NULL)
      {
         STB_FreeMemory(desc->data_types);
      }

      STB_FreeMemory(desc);

      desc = next;
   }

   FUNCTION_FINISH(STB_SIReleaseFreesatLinkageDesc);
}

/*!**************************************************************************
 * @brief   Frees memory used by an SI_FREESAT_PREFIX_DESC structure
 * @param   list - pointer to the SI_FREESAT_PREFIX_DESC
 ****************************************************************************/
void STB_SIReleaseFreesatPrefixList(SI_FREESAT_PREFIX_DESC *list)
{
   SI_FREESAT_PREFIX_DESC *next;

   FUNCTION_START(STB_SIReleaseFreesatPrefixList);

   while (list != NULL)
   {
      next = list->next;

      if (list->uri_prefix != NULL)
      {
         STB_SIReleaseStringDesc(list->uri_prefix);
      }

      STB_FreeMemory(list);

      list = next;
   }

   FUNCTION_FINISH(STB_SIReleaseFreesatPrefixList);
}

/*!**************************************************************************
 * @brief   Frees a CI+ service list
 * @param   service_list - pointer to the start of the CI+ service list
 ****************************************************************************/
void STB_SIReleaseCIPlusServiceList(SI_CIPLUS_SERVICE *service_list)
{
   SI_CIPLUS_SERVICE *next_service;

   FUNCTION_START(STB_SIReleaseCIPlusServiceList);

   while (service_list != NULL)
   {
      next_service = service_list->next;

      if (service_list->provider_str != NULL)
      {
         STB_SIReleaseStringDesc(service_list->provider_str);
      }

      if (service_list->name_str != NULL)
      {
         STB_SIReleaseStringDesc(service_list->name_str);
      }

      STB_FreeMemory(service_list);
      service_list = next_service;
   }

   FUNCTION_FINISH(STB_SIReleaseCIPlusServiceList);
}

/**
 * @brief   Frees a list of URI linkage descriptors
 * @param   list pointer to the start of the URI linkage descriptor list
 */
void STB_SIReleaseURILinkageList(SI_URI_LINKAGE_DESC *list)
{
   SI_URI_LINKAGE_DESC *next;

   FUNCTION_START(STB_SIReleaseURILinkageList);

   while (list != NULL)
   {
      next = list->next;

      if (list->uri != NULL)
      {
         STB_SIReleaseStringDesc(list->uri);
      }

      if (list->private_data != NULL)
      {
         STB_FreeMemory(list->private_data);
      }

      STB_FreeMemory(list);

      list = next;
   }

   FUNCTION_FINISH(STB_SIReleaseURILinkageList);
}

/**
 * @brief   Frees SDT code descriptor
 * @param   pointer to the SDT code descriptor
 */
void STB_SIReleaseSdtCodeDesc(SI_SDT_CODE_DESC *desc)
{
   FUNCTION_START(STB_SIReleaseSdtCodeDesc);

   if (desc != NULL)
   {
      STB_FreeMemory(desc);
   }

   FUNCTION_FINISH(STB_SIReleaseSdtCodeDesc);
}

/**
 * @brief   Frees CNS series (linking) descriptor
 * @param   pointer to the CNS series (linking) descriptor
 */
void STB_SIReleaseSeriesDesc(SI_SERIES_DESC *desc)
{
   FUNCTION_START(STB_SIReleaseSeriesDesc);

   if (desc != NULL)
   {
      if (desc->episode_name != NULL)
      {
         STB_SIReleaseStringDesc(desc->episode_name);
      }
      STB_FreeMemory(desc);
   }

   FUNCTION_FINISH(STB_SIReleaseSeriesDesc);
}

/*!**************************************************************************
 * @brief   Parses the given PMT to produce an array of the CA system IDs required
 *          by the service or streams on the service. The array of IDs will be
 *          allocated by this function and should be freed using STB_SIReleaseCAIds.
 * @param   pmt_data - raw PMT section data
 * @param   pmt_ca_ids - pointer to an array that will be allocated containing
 *                       the CA IDs found in the PMT
 * @return  Number of CA IDs found in the PMT and returned in the array,
 *          0 if none are found.
 ****************************************************************************/
U16BIT STB_SIGetPmtCaIdDescArray(U8BIT *pmt_data, U16BIT **pmt_ca_ids)
{
   U8BIT *data_ptr;
   U16BIT sec_len;
   U8BIT *data_end;
   U16BIT dloop_len;
   U8BIT *dloop_end;
   U8BIT dtag;
   U16BIT i, num_ca_entries;
   SI_CA_DESC *ca_desc_array;

   FUNCTION_START(STB_SIGetPmtCaIdDescArray);

   /* Get pointer to section data and end of section */
   data_ptr = pmt_data;
   sec_len = (((data_ptr[1] & 0x0f) << 8) | data_ptr[2]) + 3;
   data_end = data_ptr + sec_len - 4;   // -4 for crc

   /* Skip section header */
   data_ptr += 8;

   /* Get descriptor loop length */
   dloop_len = ((data_ptr[2] & 0x0f) << 8) | data_ptr[3];
   data_ptr += 4;

   num_ca_entries = 0;
   ca_desc_array = NULL;

   /* Process first descriptor loop */
   dloop_end = data_ptr + dloop_len;
   while (data_ptr < dloop_end)
   {
      dtag = data_ptr[0];
      data_ptr++;

      switch (dtag)
      {
         case CA_DTAG:
         {
            data_ptr = ParseCaDescriptor(data_ptr, &num_ca_entries, &ca_desc_array, FALSE);
            break;
         }

         default:
         {
            /* Skip the descriptor */
            data_ptr += (*data_ptr + 1);
            break;
         }
      }
   }

   /* Read entry for each stream */
   while (data_ptr < data_end)
   {
      dloop_len = ((data_ptr[3] & 0x0f) << 8) | data_ptr[4];
      data_ptr += 5;

      /* Process stream descriptor loop */
      dloop_end = data_ptr + dloop_len;
      while ((data_ptr < dloop_end) && (data_ptr < data_end))
      {
         dtag = data_ptr[0];
         data_ptr++;

         switch (dtag)
         {
            case CA_DTAG:
            {
               data_ptr = ParseCaDescriptor(data_ptr, &num_ca_entries, &ca_desc_array, FALSE);
               break;
            }

            default:
            {
               /* Skip the descriptor */
               data_ptr += (*data_ptr + 1);
               break;
            }
         }
      }
   }

   if (num_ca_entries > 0)
   {
      /* Allocate an array to just pass back the CA system IDs found */
      *pmt_ca_ids = (U16BIT *)STB_GetMemory(num_ca_entries * sizeof(U16BIT));
      if (*pmt_ca_ids != NULL)
      {
         for (i = 0; i < num_ca_entries; i++)
         {
            (*pmt_ca_ids)[i] = ca_desc_array[i].ca_id;
         }
      }
      else
      {
         num_ca_entries = 0;
      }

      STB_SIReleaseCaDescArray(ca_desc_array, num_ca_entries);
   }
   else
   {
      *pmt_ca_ids = NULL;
   }

   FUNCTION_FINISH(STB_SIGetPmtCaIdDescArray);

   return(num_ca_entries);
}

/**
 * @brief   Generates request for AIT on given PID
 * @param   path The demux path to use
 * @param   req_type ONE_SHOT or CONTINUOUS request
 * @param   ait_pid Pid for the AIT
 * @param   callback Callback function to receive the parsed pmt table
 * @param   ret_param Parameter to be returned to callback function
 * @return  Filter handle, NULL if filter not successfully setup
 */
void* STB_SIRequestAit(U8BIT path, E_SI_REQUEST_TYPE req_type, U16BIT ait_pid,
   void (*callback)(void *, U32BIT, SI_TABLE_RECORD *), U32BIT ret_param)
{
   void *filter_ptr;

   FUNCTION_START(STB_SIRequestAit);

   #ifdef DEBUG_SI_RCT_REQUEST
   STB_SI_PRINT(("STB_SIRequestRct: path %d pid 0x%04x", path, rct_pid));
   #endif
   filter_ptr = STB_SIRequestTable(path, ait_pid, SI_AIT_TID, 0xff, MULTI_SECT, 1,
         SI_XTID_MATCH_DONT_CARE, SI_XTID_MASK_DONT_CARE, req_type,
         SI_BUFFER_4K, TRUE, FALSE, callback, ret_param);

   FUNCTION_FINISH(STB_SIRequestAit);

   return((void *)filter_ptr);
}

/****************************************************************************
** End of file
*****************************************************************************/

