/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Contains Unicode string handling functions for STB usage
 *
 * @file    stbuni.c
 * @date    31/05/2001
 */
//---includes for this file-------------------------------------------------------------------------
// compiler library header files
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>

// third party header files

// Ocean Blue Software header files
#include <techtype.h>
#include <dbgfuncs.h>

#include "asciimap.h"
#include "stbheap.h"
#include "stbuni.h"

#include "stbhuffman.h"

//---constant definitions for this file-------------------------------------------------------------
#define UTF16_HEADER_VALUE       0x11     // Unicode header value
#define UCS2_HEADER_VALUE        0x14     // UCS-2BE header value
#define UTF8_HEADER_VALUE        0x15     // UTF-8 header value
#define UNICODE_HEADER_POS       0        // Unicode header, start position in string
#define LENGTH_LOOP_LIMIT        INVALID_UNICODE_CHAR  // Catch for non null terminated strings
// REM: LENGTH_LOOP_LIMIT Must be Less than INVALID_UNICODE_CHAR

#define STRINGS_EQUAL            0        // Return value when compared strings are the same
#define FIRST_STRING_GREATER     1        // Return value when 1st string is greater than 2nd
#define SECOND_STRING_GREATER    -1        // Return value when 2nd string is greater than 1st

#define MAX_NUMBER_DIGITS        11        // Maximum number digits in a S32BIT number (e.g. -2147483648 )


/* Unicode tokens for format strings */
#define UNI_PERCENT_CHAR         0x0025
#define UNI_SMALL_D_CHAR         0x0064
#define UNI_SMALL_H_CHAR         0x0068
#define UNI_SMALL_U_CHAR         0x0075
#define UNI_SMALL_L_CHAR         0x006C
#define UNI_SMALL_S_CHAR         0x0073
#define UNI_SMALL_X_CHAR         0x0078
#define UNI_LARGE_X_CHAR         0x0058
#define UNI_ZERO_CHAR            0x0030
#define UNI_NINE_CHAR            0x0039

#define UNI_SUR_HIGH_START       0xD800
#define UNI_SUR_HIGH_END         0xDBFF
#define UNI_SUR_LOW_START        0xDC00
#define UNI_SUR_LOW_END          0xDFFF
#define UNI_REPLACEMENT_CHAR     0x0000FFFD

/* Pairs of unicode tokens for format strings */
#define UNI_SMALL_L_SMALL_D_CHARS      ((UNI_SMALL_L_CHAR << 16) | UNI_SMALL_D_CHAR)
#define UNI_SMALL_L_SMALL_U_CHARS      ((UNI_SMALL_L_CHAR << 16) | UNI_SMALL_U_CHAR)
#define UNI_SMALL_L_SMALL_X_CHARS      ((UNI_SMALL_L_CHAR << 16) | UNI_SMALL_X_CHAR)
#define UNI_SMALL_L_LARGE_X_CHARS      ((UNI_SMALL_L_CHAR << 16) | UNI_LARGE_X_CHAR)
#define UNI_SMALL_H_SMALL_D_CHARS      ((UNI_SMALL_H_CHAR << 16) | UNI_SMALL_D_CHAR)
#define UNI_SMALL_H_SMALL_U_CHARS      ((UNI_SMALL_H_CHAR << 16) | UNI_SMALL_U_CHAR)
#define UNI_SMALL_H_SMALL_X_CHARS      ((UNI_SMALL_H_CHAR << 16) | UNI_SMALL_X_CHAR)
#define UNI_SMALL_H_LARGE_X_CHARS      ((UNI_SMALL_H_CHAR << 16) | UNI_LARGE_X_CHAR)

#define MAX_DECODE_BUFFER_SIZE         255      /* Buffer size used for decoding compressed strings */
#define MAX_NUM_FORMAT_SPEC_STR_SIZE   6        /* max size of numeric format specifier string e.g. "%011ld" */
#define MAX_NUM_WIDTH_DIGITS           3        /* max number of digits to specify number width
                                                   in numeric format specifier e.g. "011" in "%011ld" */

//---local typedefs, structs, enumerations for this file--------------------------------------------
typedef struct
{
   U32BIT lang_code;
   U8BIT table_id;
} S_LANG_CODE_ENTRY;

//---local (static) variable declarations for this file---------------------------------------------
//   (internal variables declared static to make them local)

/* This table is used to set the default 8859 char table that should be used for a particular
 * language code. The entries here are exceptions if the default table, which is 0, can't be
 * used and the broadcasters aren't encoding the strings correctly such that the table to be
 * used isn't being specified at the start of each string. So if a language isn't defined here
 * it doesn't mean it isn't supported! */
static const S_LANG_CODE_ENTRY lang_code_lookup_table[] =
{
   {(U32BIT)(('r' << 16) | ('u' << 8) | 'm'), 2}, /* Romanian */
   {(U32BIT)(('r' << 16) | ('o' << 8) | 'n'), 2}     /* Romanian */
};

// Default Latin table - defined as being table 0 in ETSI 300 468.
static U8BIT default_ascii_table = 0;

//---local function prototypes for this file--------------------------------------------------------
//   (internal functions declared static to make them local)

static BOOLEAN CheckUnicodeCharForReverseDirection(U16BIT unicode);
static void MakeUnicode( U8BIT **addr_string_ptr, BOOLEAN *new_string,
   U16BIT *length_ptr, BOOLEAN *reverse_dir, BOOLEAN strip_DVB_cntrl_char);
static U8BIT* OutputUTF8(U8BIT *buffer, U32BIT char_code);
static U32BIT ReadUTF8(U8BIT **buffer);
static U32BIT CharToLower(U32BIT char_code);

//--------------------------------------------------------------------------------------------------
// global function definitions
//--------------------------------------------------------------------------------------------------
/**
 *

 *
 * @brief   Determines the length, in characters, of the given unicode string
 *                by searching for NULL.  Count ignores the unicode header value.
 *
 * @param   string_ptr - Pointer to the U8BIT string.
 *
 * @return   Returns length of string
 *
 */
U32BIT STB_UnicodeStringLen(U8BIT *string_ptr)
{
   U32BIT string_length;            // Length of string (U16BIT counts)
   U32BIT string_offset;            // Counter for offset into string (U8BIT counts)
   U32BIT char_code;
   U8BIT *ptr;

   FUNCTION_START(STB_UnicodeStringLen);

   ASSERT(string_ptr != NULL);

   // Reset string length to zero
   string_length = 0;

   // only process if this is a valid Unicode string
   if (string_ptr != NULL)
   {
      if ((string_ptr[UNICODE_HEADER_POS] == UTF16_HEADER_VALUE) || (string_ptr[UNICODE_HEADER_POS] == UCS2_HEADER_VALUE))
      {
         // look beyond the header
         string_offset = 1;

         do
         {
            // if we have found a unicode NULL then exit loop else increment window to look at
            // next U16BIT value
            if (string_ptr[string_offset] == '\0' && string_ptr[string_offset + 1] == '\0')
            {
               break;
            }

            string_offset += 2;
            string_length++;
         }
         while (string_length < LENGTH_LOOP_LIMIT);
      }
      else if (string_ptr[UNICODE_HEADER_POS] == UTF8_HEADER_VALUE)
      {
         ptr = string_ptr + 1;

         while ((char_code = ReadUTF8(&ptr)) != 0)
         {
            string_length++;
         }
      }
   }

   FUNCTION_FINISH(STB_UnicodeStringLen);

   return(string_length);
}

/**
 *

 *
 * @brief   Checks to see if the supplied string is unicode and if it is reversed (arabic)
 *
 * @param   U8BIT* string_ptr - string to test if unicode
 *
 * @return   BOOLEAN is_reversed     TRUE if reversed
 *
 */
BOOLEAN STB_IsUnicodeStringReversed(U8BIT *string_ptr)
{
   BOOLEAN is_reversed;    // TRUE if reversed char found
   U32BIT num_bytes;       // Number of bytes in string
   U32BIT offset_count;    // Offset counter
   U16BIT unicode;         // Unicode value

   FUNCTION_START(STB_IsUnicodeStringReversed);

   is_reversed = FALSE;

   if ((string_ptr != NULL) && (string_ptr[UNICODE_HEADER_POS] == UTF16_HEADER_VALUE))
   {
      num_bytes = STB_UnicodeStringLen(string_ptr);
      string_ptr++;
      num_bytes *= 2;
      offset_count = 0;

      while ((offset_count < num_bytes) && (is_reversed == FALSE))
      {
         // Convert next two bytes to a 16 bit unicode value
         unicode = ((*(string_ptr + offset_count)) << 8) | ((*(string_ptr + offset_count + 1)));

         is_reversed = CheckUnicodeCharForReverseDirection(unicode );

         offset_count += 2;
      }
   }

   FUNCTION_FINISH(STB_IsUnicodeStringReversed);
   return(is_reversed);
}

/**
 *

 *
 * @brief   Tests for unicode string
 *
 * @param   U8BIT* string_ptr - string to test if unicode
 *
 * @return   BOOLEAN is_unicode
 *
 */
BOOLEAN STB_IsUnicodeString(U8BIT *string_ptr)
{
   BOOLEAN is_unicode;

   FUNCTION_START(STB_IsUnicodeString);

   if ((string_ptr != NULL) &&
       ((string_ptr[UNICODE_HEADER_POS] == UTF16_HEADER_VALUE) ||
        (string_ptr[UNICODE_HEADER_POS] == UTF8_HEADER_VALUE)))
   {
      is_unicode = TRUE;
   }
   else
   {
      is_unicode = FALSE;
   }

   FUNCTION_FINISH(STB_IsUnicodeString);

   return(is_unicode);
}

/**
 *

 *
 * @brief   Tests for normal ascii string
 *
 * @param   U8BIT* string_ptr - string to test if normal ascii
 *
 * @return   BOOLEAN is_normal
 *
 */
BOOLEAN STB_IsNormalString(U8BIT *string_ptr)
{
   BOOLEAN is_normal;

   FUNCTION_START(STB_IsNormalString);

   is_normal = FALSE;
   if (string_ptr != NULL)
   {
      if ((*string_ptr == 0) || (*string_ptr >= 0x20))
      {
         is_normal = TRUE;
      }
   }

   FUNCTION_FINISH(STB_IsNormalString);
   return(is_normal);
}

/**
 *

 *
 * @brief   Determines the no of bytes of the given string
 *
 * @param   string_ptr - Pointer to the U8BIT string.
 *
 * @return   Returns no of bytes of string including headers and terminators.
 *
 */
U32BIT STB_GetNumBytesInString(U8BIT *string_ptr)
{
   U32BIT num_chars;
   U32BIT num_bytes;
   U8BIT byte_val;
   BOOLEAN finished;
   BOOLEAN prev_char_null;

   FUNCTION_START(STB_GetNumBytesInString);

   num_bytes = 0;

   if (string_ptr != NULL)
   {
      byte_val = string_ptr[0];
      if (byte_val >= 0x20)
      {
         // normal ascii codes plus null terminator
         num_bytes = strlen((char *)string_ptr) + 1;
      }
      else if (byte_val == 0)
      {
         // empty ascii string contains null char only
         num_bytes = 1;
      }
      else if ((byte_val >= 1) && (byte_val <= 11))
      {
         // one byte header followed by normal ascii codes and null terminator
         num_bytes = strlen((char *)(string_ptr + 1)) + 2;
      }
      else if (byte_val == 0x10)
      {
         // three byte header followed by normal ascii codes and null terminator
         num_bytes = strlen((char *)(string_ptr + 3)) + 4;
      }
      else if ((byte_val == UTF16_HEADER_VALUE) || (byte_val == UCS2_HEADER_VALUE))
      {
         // unicode - one byte header followed by 16-bit character codes and 16-bit terminator
         num_chars = STB_UnicodeStringLen(string_ptr);
         if (num_chars != INVALID_UNICODE_CHAR)
         {
            num_bytes = STB_UTF16_LEN_TO_BYTES_IN_STRING(num_chars);
         }
      }
      else if (byte_val == UTF8_HEADER_VALUE)
      {
         /* UTF-8 encoded string, 1 byte header followed by 1 byte terminator */
         num_bytes = strlen((char *)(string_ptr + 1)) + 2;
      }
      else if (byte_val == 0x1f)
      {
         /* Compressed UTF-8 string.
          * This count relies on the fact that when the string is read it's terminated with a
          * two null chars as without this there's no way to determine the length other than
          * by decompressing it! */
         finished = FALSE;
         prev_char_null = FALSE;

         for (num_bytes = 0; !finished; )
         {
            if (string_ptr[num_bytes] == 0)
            {
               if (prev_char_null)
               {
                  /* Double 0 char found, end of string */
                  finished = TRUE;
               }
               else
               {
                  prev_char_null = TRUE;
                  num_bytes++;
               }
            }
            else
            {
               prev_char_null = FALSE;
               num_bytes++;
            }
         }
      }
   }
   FUNCTION_FINISH(STB_GetNumBytesInString);
   return(num_bytes);
}

/**
 *

 *
 * @brief   Takes a string and changes the requested location to a new value. This
 *                request may involve appending to the string in which case the string is
 *                extended (always extended by one character, independent of char_id).
 *
 * @param   string_ptr         Pointer to the string to be updated
 * @param   char_id            Character to be changed
 * @param   code               Unicode value to be inserted at position indicated by
 *                                   char_id
 *
 * @return   Pointer to the updated string or unchanged string if invalid request made.
 *
 */
U8BIT* STB_SetUnicodeStringChar(U8BIT *string_ptr, U16BIT char_id, U16BIT code)
{
   U32BIT string_length;          // Length of string passed in
   U32BIT byte_offset;            // Byte offset rather than unicode character offset

   FUNCTION_START(STB_SetUnicodeStringChar);

   ASSERT(string_ptr != NULL);
   ASSERT((string_ptr[0] == UTF16_HEADER_VALUE) || (string_ptr[0] == UTF8_HEADER_VALUE));

   // Only process if this is a valid pointer and a valid unicode string
   if (string_ptr != NULL && string_ptr[UNICODE_HEADER_POS] == UTF16_HEADER_VALUE)
   {
      string_length = STB_UnicodeStringLen(string_ptr);

      // check if we need to re-alloc string to append the code (expand by one)
      if (char_id >= string_length)
      {
         // Realloc enough memory to include string, null terminator(2) and unicode header(1)
         string_ptr = STB_ResizeMemory(string_ptr, STB_UTF16_LEN_TO_BYTES_IN_STRING(string_length));

         // re-align to end of string
         char_id = (U16BIT) string_length;
      } // if

      // If we still have a valid string then insert new character
      if (string_ptr != NULL)
      {
         // Actual byte offset of required unicode character (take header into account)
         byte_offset = (char_id * 2) + 1;

         string_ptr[byte_offset] = (U8BIT) (code >> 8);
         byte_offset++;

         string_ptr[byte_offset] = (U8BIT) (code & 0x00FF);
         byte_offset++;

         // if this is an append then terminate string with null
         if (char_id == string_length)
         {
            string_ptr[byte_offset++] = '\0';
            byte_offset++;

            string_ptr[byte_offset] = '\0';
         } // if
      } // if
   } // if

   FUNCTION_FINISH(STB_SetUnicodeStringChar);
   return(string_ptr);
} // STB_SetUnicodeStringChar

/**
 *

 *
 * @brief   Takes a string and removes the requested location, shuffling
 *                any following data down (thus removing gap)
 *
 * @param   string_ptr         Pointer to the string to be updated
 * @param   char_id            Character to be changed
 *
 * @return   Pointer to the updated string or unchanged string if invalid request made.
 *
 */
U8BIT* STB_DeleteUnicodeStringChar(U8BIT *string_ptr, U16BIT char_id)
{
   U32BIT string_length;          // Length of string passed in
   U32BIT byte_offset;            // Byte offset rather than unicode character offset

   FUNCTION_START(STB_DeleteUnicodeStringChar);

   ASSERT(string_ptr != NULL);
   ASSERT((string_ptr[0] == UTF16_HEADER_VALUE) || (string_ptr[0] == UTF8_HEADER_VALUE));

   // Only process if this is a valid pointer and a valid unicode string
   if (string_ptr != NULL && string_ptr[UNICODE_HEADER_POS] == UTF16_HEADER_VALUE)
   {
      string_length = STB_UnicodeStringLen(string_ptr);

      // check if we are asking to delete a valid character
      if (char_id < string_length)
      {
         // Actual byte offset of required unicode character (take header into account)
         byte_offset = (char_id * 2) + 1;

         // Compress string by removing requested data
         memcpy(&string_ptr[byte_offset], &string_ptr[byte_offset + 2],
            ((string_length - char_id) * 2));

         // re-size, allowing for unicode header and NULL
         string_ptr = STB_ResizeMemory(string_ptr, (string_length * 2) + 1);
      } // if
   } // if

   FUNCTION_FINISH(STB_DeleteUnicodeStringChar);
   return(string_ptr);
} // STB_DeleteUnicodeStringChar

/**
 *

 *
 * @brief   Retrieves the unicode value pointed to by char_id within the given string.
 *                If an invalid request occurs (ie char_id is beyond string limit) then 0 is
 *                returned.
 *
 * @param   string_ptr         Pointer to the string to be searched
 * @param   char_id            Character to be retrieved
 *
 * @return   unicode value contained at the requested string position (0xFFFF if invalid
 *                position or string).
 *
 */
U32BIT STB_GetUnicodeStringChar(U8BIT *string_ptr, U16BIT char_id)
{
   U32BIT string_length;         // Length of string passed in
   U32BIT byte_offset;           // Byte offset rather than unicode character offset
   U32BIT return_code;           // Value found at location, to be returned

   FUNCTION_START(STB_GetUnicodeStringChar);

   ASSERT(string_ptr != NULL);
   ASSERT((string_ptr[0] == UTF16_HEADER_VALUE) || (string_ptr[0] == UTF8_HEADER_VALUE));

   return_code = INVALID_UNICODE_CHAR;

   // Only process if this is a valid pointer and a valid unicode string
   if (string_ptr != NULL)
   {
      if (string_ptr[UNICODE_HEADER_POS] == UTF16_HEADER_VALUE)
      {
         string_length = STB_UnicodeStringLen(string_ptr);

         if (char_id <= string_length)
         {
            // Actual byte offset of required unicode character (take header into account)
            byte_offset = (char_id * 2) + 1;

            return_code = (U16BIT) (string_ptr[byte_offset] << 8);
            byte_offset++;

            return_code |= (U16BIT) string_ptr[byte_offset];
         }
      }
      else if (string_ptr[UNICODE_HEADER_POS] == UTF8_HEADER_VALUE)
      {
         /* Can't work out the byte offset of the nth char, so have to iterate through */
         string_ptr++;
         while (char_id > 0)
         {
            if ((*string_ptr & 0x80) == 0)
            {
               string_ptr++;
               char_id--;
            }
            else if ((*string_ptr & 0xE0) == 0xC0)
            {
               string_ptr += 2;
               char_id--;
            }
            else if ((*string_ptr & 0xF0) == 0xE0)
            {
               string_ptr += 3;
               char_id--;
            }
            else if ((*string_ptr & 0xF8) == 0xF0)
            {
               string_ptr += 4;
               char_id--;
            }
            else
            {
               /* Invalid UTF-8 string */
               char_id = 0;
               string_ptr = NULL;
            }
         }

         if (string_ptr != NULL)
         {
            return_code = ReadUTF8(&string_ptr);
         }
      }
   }

   FUNCTION_FINISH(STB_GetUnicodeStringChar);

   return(return_code);
}

/**
 *

 *
 * @brief   Appends the contents of string2_ptr to string1_ptr and returns a pointer
 *                to the newly created string.
 *
 * @param   string1_ptr         Pointer to the first string, and hence first part of
 *                                    concatinated string.
 * @param   string2_ptr         Pointer to the second string and hence last part of
 *                                    concatinated string.
 *
 * @return   Pointer to the concatinated string or NULL if failed
 *
 */
U8BIT* STB_ConcatUnicodeStrings(U8BIT *string1_ptr, U8BIT *string2_ptr)
{
   U16BIT char_count;
   U8BIT *unicode_str1;
   U8BIT *unicode_str2;
   U8BIT *concatinated_string_ptr; // Holder for the concatinated string
   U32BIT new_string_byte_count;   // Concatinated string length (in bytes)
   U32BIT num_bytes1;
   U32BIT num_bytes2;

   FUNCTION_START(STB_ConcatUnicodeStrings);

   ASSERT(string1_ptr != NULL);
   ASSERT(string2_ptr != NULL);

   concatinated_string_ptr = NULL;

   // Only process if strings are valid pointers and both have valid unicode strings
   if ((string1_ptr != NULL) && (string2_ptr != NULL))
   {
      unicode_str1 = NULL;
      unicode_str2 = NULL;

      if (((string1_ptr[UNICODE_HEADER_POS] == UTF16_HEADER_VALUE) &&
           (string2_ptr[UNICODE_HEADER_POS] == UTF16_HEADER_VALUE)) ||
          ((string1_ptr[UNICODE_HEADER_POS] == UTF8_HEADER_VALUE) &&
           (string2_ptr[UNICODE_HEADER_POS] == UTF8_HEADER_VALUE)))
      {
         /* Both strings are the same, so don't need to be converted */
         num_bytes1 = STB_GetNumBytesInString(string1_ptr);
         num_bytes2 = STB_GetNumBytesInString(string2_ptr);
      }
      else
      {
         /* Convert both strings before concatenating */
         unicode_str1 = STB_ConvertStringToUTF8(string1_ptr, &char_count, FALSE, 0);
         string1_ptr = unicode_str1;
         num_bytes1 = STB_GetNumBytesInString(string1_ptr);

         unicode_str2 = STB_ConvertStringToUTF8(string2_ptr, &char_count, FALSE, 0);
         string2_ptr = unicode_str2;
         num_bytes2 = STB_GetNumBytesInString(string2_ptr);
      }

      if ((string1_ptr != NULL) && (string2_ptr != NULL))
      {
         if (string1_ptr[UNICODE_HEADER_POS] == UTF16_HEADER_VALUE)
         {
            /* Length of concat string is sum of lengths, which include header and null bytes for each,
             * minus 3 as we only need one header and terminator on the new string */
            new_string_byte_count = num_bytes1 + num_bytes2 - 3;
         }
         else
         {
            /* Length of concat string is sum of lengths, which include header and null byte for each,
             * minus 2 as we only need one header and null on the new string */
            new_string_byte_count = num_bytes1 + num_bytes2 - 2;
         }

         concatinated_string_ptr = (U8BIT *)STB_GetMemory(new_string_byte_count);
         if (concatinated_string_ptr != NULL)
         {
            if (string1_ptr[UNICODE_HEADER_POS] == UTF16_HEADER_VALUE)
            {
               /* Concat UTF-16 strings */
               /* Copy string1, minus its null terminator */
               memcpy(&concatinated_string_ptr[0], string1_ptr, num_bytes1 - 2);

               /* Concat string2 starting from the first actual char, and including its null */
               memcpy((char *)&concatinated_string_ptr[num_bytes1 - 2], &string2_ptr[1], num_bytes2 - 1);
            }
            else
            {
               /* Concat UTF-8 strings */
               /* Copy string1, minus its null terminator */
               memcpy(&concatinated_string_ptr[0], string1_ptr, num_bytes1 - 1);

               /* Concat string2 starting from the first actual char, and including its null */
               memcpy((char *)&concatinated_string_ptr[num_bytes1 - 1], &string2_ptr[1], num_bytes2 - 1);
            }
         }
      }

      STB_ReleaseUnicodeString(unicode_str1);
      STB_ReleaseUnicodeString(unicode_str2);
   }

   FUNCTION_FINISH(STB_ConcatUnicodeStrings);

   return(concatinated_string_ptr);
}

/**
 *

 *
 * @brief   Divides the (space separated) string up into individual words and returns
 *                them one per call
 *
 * @param   str         The string to be tokenised (will be modified by this function)
 * @param   save_ptr    The caller's variable in which the current pointer can be saved
 *
 * @return   Pointer to the next token in the string, or NULL if no more
 *
 */
U8BIT* STB_UnicodeStringTokenise(U8BIT *string, U8BIT **save_ptr)
{
   U8BIT *return_ptr = NULL;
   U8BIT *next_char_ptr;
   U32BIT char_code;

   FUNCTION_START(STB_UnicodeStringTokenise);

   ASSERT(save_ptr != NULL);

   // Pick up the remembered pointer to the start of the next string
   // if this is a repeat call
   if ((string == NULL) && (*save_ptr != NULL))
   {
      string = *save_ptr;
      // Replace 2nd NULL terminator from previously found word with header for new word
      *string = UTF8_HEADER_VALUE;
   }

   if ((string != NULL) && (string[UNICODE_HEADER_POS] == UTF8_HEADER_VALUE))
   {
      return_ptr = string;
      next_char_ptr = string + 1;
      *save_ptr = NULL;

      while ((char_code = ReadUTF8(&next_char_ptr)) != 0)
      {
         if (char_code == 0x20)
         {
            /* Convert the separator (space) to a terminating NULL */
            *string = 0x00;

            /* Skip any further whitespace */
            while ((char_code = ReadUTF8(&next_char_ptr)) == 0x20)
            {
               string = next_char_ptr;
            }

            *save_ptr = string;
            break;
         }

         string = next_char_ptr;
      }
   }

   FUNCTION_FINISH(STB_UnicodeStringTokenise);

   return return_ptr;
}

/**
 *

 *
 * @brief   Finds the first occurence of str2 in str1 and returns a pointer to the substring
 *                (as per strstr)
 *
 * @param   str1        String being searched
 * @param   str2        String being searched for
 * @param   ignore_case If TRUE, ignores case when comparing characters
 *
 * @return   Pointer to the first occurence of substring in str1, or NULL if not found
 *
 */
U8BIT* STB_UnicodeStrStr(U8BIT *str1, U8BIT *str2, BOOLEAN ignore_case)
{
   U8BIT *subs_ptr = NULL;
   U32BIT str1_len;
   U32BIT str2_len;
   U32BIT str1_index;
   U32BIT str1_cmp_index;
   U32BIT str2_cmp_index;
   U8BIT *str1_ptr;
   U8BIT *next_str1_ptr;
   U8BIT *str1_cmp_ptr;
   U8BIT *str2_cmp_ptr;
   U32BIT str1_char, str2_char;

   FUNCTION_START(STB_UnicodeStrStr);

   // Only process if strings are valid pointers and both have valid unicode strings
   if ((str1 != NULL) && (str2 != NULL))
   {
      if ((str1[UNICODE_HEADER_POS] == UTF16_HEADER_VALUE) &&
          (str2[UNICODE_HEADER_POS] == UTF16_HEADER_VALUE))
      {
         /* Search procedure for UTF-16 strings */
         str1_len = STB_UnicodeStringLen(str1);
         str2_len = STB_UnicodeStringLen(str2);

         str1_index = 0;

         while (str1_index < str1_len)
         {
            str1_cmp_index = str1_index;
            str2_cmp_index = 0;

            while ((str1_cmp_index < str1_len) && (str2_cmp_index < str2_len))
            {
               str1_char = STB_GetUnicodeStringChar(str1, str1_cmp_index);
               str2_char = STB_GetUnicodeStringChar(str2, str2_cmp_index);

               if (ignore_case)
               {
                  str1_char = CharToLower(str1_char);
                  str2_char = CharToLower(str2_char);
               }

               if (str1_char != str2_char)
               {
                  break;
               }

               str1_cmp_index++;
               str2_cmp_index++;
            }

            if (str2_cmp_index >= str2_len)
            {
               subs_ptr = &str1[1 + ((str1_cmp_index - 1) * 2)];
               break;
            }
            str1_index++;
         }
      }
      else if ((str1[UNICODE_HEADER_POS] == UTF8_HEADER_VALUE) &&
               (str2[UNICODE_HEADER_POS] == UTF8_HEADER_VALUE))
      {
         /* Search procedure for UTF-8 strings */
         str1_ptr = &str1[1];
         next_str1_ptr = str1_ptr;
         str1_char = ReadUTF8(&next_str1_ptr);

         while (str1_char != 0)
         {
            str1_cmp_ptr = str1_ptr;
            str2_cmp_ptr = &str2[1];

            do
            {
               str1_char = ReadUTF8(&str1_cmp_ptr);
               str2_char = ReadUTF8(&str2_cmp_ptr);

               if (ignore_case)
               {
                  str1_char = CharToLower(str1_char);
                  str2_char = CharToLower(str2_char);
               }
            }
            while ((str1_char != 0) && (str2_char != 0) && (str1_char == str2_char));

            if (str2_char == 0)
            {
               /* Matched to the end of the search string */
               subs_ptr = str1_ptr;
               break;
            }

            str1_ptr = next_str1_ptr;
            str1_char = ReadUTF8(&next_str1_ptr);
         }
      }
   }

   FUNCTION_FINISH(STB_UnicodeStrStr);

   return(subs_ptr);
}

/**
 *

 *
 * @brief   Compares the contents of the two given unicode strings and returns the status
 *                (as per strcmp)
 *
 * @param   string1_ptr    Pointer to the 'master' string
 * @param   string2_ptr    Pointer to the 'slave' string
 * @param   exact_match    If TRUE, and the strings are the same upto the end of one of them,
 *                               the lengths of the strings must also be the same for the strings to be equal
 * @param   ignore_case    If TRUE, case is ignored when comparing chars, if appropriate.
 *
 * @return   Result of the comparison.  0 if equal, +ve if string1_ptr > string2_ptr,
 *                -ve if string1_ptr < string2_ptr.
 *
 */
S8BIT STB_CompareUnicodeStrings(U8BIT *string1_ptr, U8BIT *string2_ptr, BOOLEAN exact_match, BOOLEAN ignore_case)
{
   U32BIT string1_length;         // Length of 1st string passed in
   U32BIT string2_length;         // Length of 2nd string passed in
   U32BIT min_string_length;      // Smallest string length from the two
   U32BIT unicode_count;          // Counter value pointing at current unicode character
   U32BIT byte_offset;            // Counter value pointing at current byte offset
   U32BIT current_string1_code;   // Current unicode value from string1
   U32BIT current_string2_code;   // Current unicode value from string2
   S8BIT difference = STRINGS_EQUAL;  // Return value indicating difference between strings
   U8BIT *ptr1;
   U8BIT *ptr2;

   FUNCTION_START(STB_CompareUnicodeStrings);

   ASSERT(string1_ptr != NULL);
   ASSERT(string2_ptr != NULL);

   // Only process if strings are valid pointers and both have valid unicode strings
   if ((string1_ptr != NULL) && (string2_ptr != NULL))
   {
      if ((string1_ptr[UNICODE_HEADER_POS] == UTF16_HEADER_VALUE) &&
          (string2_ptr[UNICODE_HEADER_POS] == UTF16_HEADER_VALUE))
      {
         string1_length = STB_UnicodeStringLen(string1_ptr);
         string2_length = STB_UnicodeStringLen(string2_ptr);

         // Determine the shortest string
         if (string1_length < string2_length)
         {
            min_string_length = string1_length;
         }
         else
         {
            min_string_length = string2_length;
         }

         difference = STRINGS_EQUAL;
         unicode_count = 0;
         byte_offset = 1;

         // Loop through the strings
         while (unicode_count <= min_string_length && difference == STRINGS_EQUAL)
         {
            current_string1_code = (U16BIT) (string1_ptr[byte_offset] << 8);
            current_string2_code = (U16BIT) (string2_ptr[byte_offset] << 8);
            byte_offset++;
            current_string1_code |= (U16BIT) string1_ptr[byte_offset];
            current_string2_code |= (U16BIT) string2_ptr[byte_offset];
            byte_offset++;

            if (ignore_case)
            {
               current_string1_code = CharToLower(current_string1_code);
               current_string2_code = CharToLower(current_string2_code);
            }

            if (current_string1_code > current_string2_code)
            {
               difference = FIRST_STRING_GREATER;
            }
            else if (current_string1_code < current_string2_code)
            {
               difference = SECOND_STRING_GREATER;
            }

            unicode_count++;
         }
      }
      else if ((string1_ptr[UNICODE_HEADER_POS] == UTF8_HEADER_VALUE) &&
               (string2_ptr[UNICODE_HEADER_POS] == UTF8_HEADER_VALUE))
      {
         ptr1 = &string1_ptr[1];
         ptr2 = &string2_ptr[1];
         current_string1_code = ReadUTF8(&ptr1);
         current_string2_code = ReadUTF8(&ptr2);

         if (ignore_case)
         {
            current_string1_code = CharToLower(current_string1_code);
            current_string2_code = CharToLower(current_string2_code);
         }

         while ((current_string1_code != 0) && (current_string2_code != 0) &&
                (difference == STRINGS_EQUAL))
         {
            if (current_string1_code > current_string2_code)
            {
               difference = FIRST_STRING_GREATER;
            }
            else if (current_string1_code < current_string2_code)
            {
               difference = SECOND_STRING_GREATER;
            }
            else
            {
               current_string1_code = ReadUTF8(&ptr1);
               current_string2_code = ReadUTF8(&ptr2);

               if (ignore_case)
               {
                  current_string1_code = CharToLower(current_string1_code);
                  current_string2_code = CharToLower(current_string2_code);
               }
            }
         }

         if (exact_match && (difference == STRINGS_EQUAL))
         {
            /* Match will be adjusted according to string lengths */
            string1_length = STB_UnicodeStringLen(string1_ptr);
            string2_length = STB_UnicodeStringLen(string2_ptr);
         }
      }
      else
      {
         /* no useful way to report utf8/16 error - compare as C strings */
         difference = strcmp((char *) string1_ptr, (char *) string2_ptr);
         exact_match = FALSE;
      }

      // final check, if loop exits with strings set the same, but strings not same length
      // then modify difference as one string continues and hence is greater than the other
      if (exact_match == TRUE)
      {
         if ((difference == STRINGS_EQUAL) && (string1_length != string2_length))
         {
            if (string1_length < string2_length)
            {
               difference = SECOND_STRING_GREATER;
            }
            else
            {
               difference = FIRST_STRING_GREATER;
            }
         }
      }
   }

   FUNCTION_FINISH(STB_CompareUnicodeStrings);

   return(difference);
}

/**
 *

 *
 * @brief   Converts the specified DVB coded string into a unicode string, counting the number
 *              of characters and checking for right-to-left characters as it goes.
 *
 * @param   string               - pointer to the string to be converted
 * @param   reverse_dir          - reverse print direction (passed by ref and set by this function)
 * @param   nchar                - number of characters (passed by ref and set by this function)
 * @param   strip_DVB_cntrl_char - True if all DVB control chars are to be removed
 *
 * @return   A unicode string or NULL. NULL indicates error or NULL string pointer.
 *
 */
U8BIT* STB_ConvertStringToUnicode(U8BIT *string, BOOLEAN *reverse_dir, U16BIT *nchar,
   BOOLEAN strip_DVB_cntrl_char, U32BIT lang_code)
{
   U8BIT *char_ptr;
   U8BIT byte_val;
   BOOLEAN convert_reqd;
   U8BIT table;
   U8BIT *op_buff;
   U8BIT *op_ptr;
   U16BIT num_ip_char;
   U16BIT num_op_char;
   BOOLEAN reverse;
   BOOLEAN utf8;
   U8BIT *decoded_utf8;
   U16BIT unicode;
   U16BIT nbytes;
   U16BIT i;
   U16BIT prev_char;
   U8BIT table_id;
   U8BIT encoding_type_id;
   U16BIT actual_size;

   FUNCTION_START(STB_ConvertStringToUnicode);

   ASSERT(reverse_dir != NULL);
   ASSERT(nchar != NULL);

   // setup pointer to characters in string
   char_ptr = string;

   table = 0;
   num_op_char = 0;
   reverse = FALSE;
   utf8 = FALSE;
   decoded_utf8 = NULL;

   if (string != NULL)
   {
      /* Use the default table if there's no lang code or the code isn't found */
      table_id = default_ascii_table;

      if (lang_code != 0)
      {
         /* Lookup the lang code to find the table id to be used */
         for (i = 0; i < (sizeof(lang_code_lookup_table) / sizeof(S_LANG_CODE_ENTRY)); i++)
         {
            if (lang_code_lookup_table[i].lang_code == lang_code)
            {
               table_id = lang_code_lookup_table[i].table_id;
               break;
            }
         }
      }

      // determine type of character coding in string - first byte indicates coding (see DVB SI spec
      // for details)
      byte_val = *char_ptr;
      if ((byte_val >= 0x20) || (byte_val == '\0'))
      {
         /* Use the default ascii table */
         convert_reqd = TRUE;
         table = table_id;
         num_ip_char = (U16BIT) strlen((char *)char_ptr);
      }
      else if ((byte_val >= 1) && (byte_val <= 11))
      {
         // one of the Latin tables specified in the DVB SI specification from ISO 8859.
         // 1 => 8859-5, 2 => 8859-6, 3 => 8859-7, 4 => 8859-8, 5 => 8859-9
         // 6 => 8859-10, 7 => 8859-11, 8 => 8859-12, 9 => 8859-13, 10 => 8859-14, 11 => 8859-15
         convert_reqd = TRUE;
         table = byte_val + 4;
         char_ptr++;
         num_ip_char = (U16BIT) strlen((char *)char_ptr);
      }
      else if (byte_val == 0x10)
      {
         // next 2 bytes indicate the table from ISO 8859
         convert_reqd = TRUE;
         table = ((*(char_ptr + 1)) << 8) | *(char_ptr + 2);
         char_ptr += 3;
         num_ip_char = (U16BIT) strlen((char *)char_ptr);
      }
      else if (byte_val == 0x11)
      {
         // already unicode, no conversion required.
         convert_reqd = FALSE;
         char_ptr++;
         // count number of characters in the string
         num_ip_char = 0;
         do
         {
            // read next unicode character
            unicode = ((*char_ptr) << 8) | *(char_ptr + 1);
            char_ptr += 2;
            if (unicode != '\0')
            {
               num_ip_char++;
            }
         }
         while (unicode != '\0');
         char_ptr = string + 1;
      }
      else if ((byte_val == UTF8_HEADER_VALUE) || (byte_val == 0x1F))
      {
         if (byte_val == UTF8_HEADER_VALUE)
         {
            /* This is a normal uncompressed UTF-8 string */
            char_ptr++;
         }
         else
         {
            /* This is a compressed UTF-8 string with the next byte defining the table
             * to be used for decoding */
            char_ptr++;
            encoding_type_id = *char_ptr;
            char_ptr++;

            if ((encoding_type_id == 1) || (encoding_type_id == 2))
            {
               /* It's not possible to tell how long the string is without decoding it first,
                * so a sufficiently large buffer is allocated to decode into */
               decoded_utf8 = (U8BIT *)STB_GetMemory(MAX_DECODE_BUFFER_SIZE);
               if (decoded_utf8 != NULL)
               {
                  actual_size = STB_HuffmanDecompress(encoding_type_id, char_ptr, decoded_utf8,
                        MAX_DECODE_BUFFER_SIZE);

                  if (actual_size > 0)
                  {
                     /* The length is now known, the buffer can be resized */
                     decoded_utf8 = (U8BIT *)STB_ResizeMemory(decoded_utf8, actual_size);
                     char_ptr = decoded_utf8;
                  }
                  else
                  {
                     /* Decompression failed */
                     *nchar = 0;
                     *reverse_dir = FALSE;
                     STB_FreeMemory(decoded_utf8);
                     FUNCTION_FINISH(STB_ConvertStringToUnicode);
                     return(NULL);
                  }
               }
               else
               {
                  /* Can't decode the string */
                  *nchar = 0;
                  *reverse_dir = FALSE;
                  FUNCTION_FINISH(STB_ConvertStringToUnicode);
                  return(NULL);
               }
            }
            else
            {
               /* Don't know how to decode the string */
               *nchar = 0;
               *reverse_dir = FALSE;
               FUNCTION_FINISH(STB_ConvertStringToUnicode);
               return(NULL);
            }
         }

         /* UTF-8 string */
         convert_reqd = TRUE;
         utf8 = TRUE;

         /* Find the length of the string in chars.
          * As conversion is to 16-bit unicode, only char codes upto 0xFFFF are supported,
          * which means only chars upto 3-byte UTF-8 can be converted */
         num_ip_char = 0;
         do
         {
            if ((*char_ptr & 0x80) == 0)
            {
               char_ptr++;
            }
            else if ((*char_ptr & 0xE0) == 0xC0)
            {
               char_ptr += 2;
            }
            else if ((*char_ptr & 0xF0) == 0xE0)
            {
               char_ptr += 3;
            }
            else
            {
               /* Invalid string */
               if (decoded_utf8 != NULL)
               {
                  STB_FreeMemory(decoded_utf8);
               }

               *nchar = 0;
               *reverse_dir = FALSE;
               FUNCTION_FINISH(STB_ConvertStringToUnicode);
               return(NULL);
            }
            num_ip_char++;
         }
         while (*char_ptr != '\0');

         if (decoded_utf8 != NULL)
         {
            char_ptr = decoded_utf8;
         }
         else
         {
            char_ptr = string + 1;
         }
      }
      else
      {
         // invalid string - exit
         *nchar = 0;
         *reverse_dir = FALSE;
         FUNCTION_FINISH(STB_ConvertStringToUnicode);
         return(NULL);
      }


      // Now create output string
      nbytes = STB_UTF16_LEN_TO_BYTES_IN_STRING(num_ip_char);   // +3 for 0x11 code at the beginning and 0x0000 terminator
      op_buff = (U8BIT *)STB_GetMemory(nbytes);
      if (op_buff != NULL)
      {
         prev_char = 0;
         op_ptr = op_buff;

         // put in code indicating uincode string
         *op_ptr = 0x11;
         op_ptr++;

         // copy over each character, converting if necessary
         for (i = 0; i < num_ip_char; i++)
         {
            unicode = 0;
            if (convert_reqd == FALSE)
            {
               // already unicode - read next character
               unicode = ((*char_ptr) << 8) | *(char_ptr + 1);
               char_ptr += 2;
            }
            else
            {
               if (utf8)
               {
                  byte_val = *char_ptr;
                  if ((byte_val & 0x80) == 0)
                  {
                     unicode = byte_val;
                     char_ptr++;
                  }
                  else if ((byte_val & 0xE0) == 0xC0)
                  {
                     unicode = byte_val & 0x1F;
                     unicode <<= 6;
                     unicode += (*(char_ptr + 1) & 0x3F);
                     char_ptr += 2;
                  }
                  else if ((byte_val & 0xF0) == 0xE0)
                  {
                     unicode = byte_val & 0x0F;
                     unicode <<= 6;
                     unicode += (*(char_ptr + 1) & 0x3F);
                     unicode <<= 6;
                     unicode += (*(char_ptr + 2) & 0x3F);
                     char_ptr += 3;
                  }
                  if ((unicode >= CHAR_TABLE_START_ASCII_CODE) && (unicode <= 0xff))
                  {
                     /* Map this range of chars to ISO-6937 using table 0 */
                     unicode = utf8_char_map[unicode - CHAR_TABLE_START_ASCII_CODE];
                  }
               }
               else
               {
                  // ascii codes - convert character to unicode
                  byte_val = *char_ptr;
                  char_ptr++;
                  if (byte_val < CHAR_TABLE_START_ASCII_CODE)
                  {
                     unicode = (U16BIT)byte_val;
                  }
                  else
                  {
                     if (table < MAX_CHAR_MAP_TABLES)
                     {
                        unicode = char_map[table][byte_val - CHAR_TABLE_START_ASCII_CODE];
                     }
                  }
               }
            }

            // check for control codes
            if (((unicode > 0x0000) && (unicode <= 0x001f)) ||
                ((unicode >= 0x0080) && (unicode <= 0x009f)) ||
                ((unicode >= 0xe080) && (unicode <= 0xe09f))
                )
            {
               switch (unicode)
               {
                  // only accept LF and convert to 0xe08a
                  // DVB - CR/LF.
                  case 0x008a:
                  case 0xe08a:
                  {
                     if (strip_DVB_cntrl_char == FALSE)
                     {
                        unicode = 0xe08a;
                     }
                     else
                     {
                        unicode = 0;
                     }
                     break;
                  }
                  // DVB - character emphasis ON
                  case 0x0086:
                  case 0xe086:
                  {
                     if (strip_DVB_cntrl_char == FALSE)
                     {
                        unicode = 0xe086;
                     }
                     else
                     {
                        unicode = 0;
                     }
                     break;
                  }
                  // DVB - character emphasis OFF
                  case 0x0087:
                  case 0xe087:
                  {
                     if (strip_DVB_cntrl_char == FALSE)
                     {
                        unicode = 0xe087;
                     }
                     else
                     {
                        unicode = 0;
                     }
                     break;
                  }
                  // convert all other control codes to 0
                  default:
                  {
                     unicode = 0;
                     break;
                  }
               } // e.o. switch
            } //  e.o. control codes

            // if valid character copy to output string and check for reverse direction
            if (unicode != 0)
            {
               /* Check for diacritic character. In DVB strings the diacritic precedes the char
                * it combines with, but in Unicode a diacritic follows the char, so the char order needs
                * to be swapped. Only a single combining diacritic is valid in DVB so only need
                * to check previous char to see if a swap is required. */
               if (prev_char != 0)
               {
                  /* Don't want to swap if this char is also a diacritic */
                  if ((unicode < 0x300) || (unicode > 0x36f))
                  {
                     /* Swap chars - replace the last output char with the current char */
                     op_ptr -= 2;
                     *op_ptr = (U8BIT)((unicode >> 8) & 0xff);
                     op_ptr++;
                     *op_ptr = (U8BIT)(unicode & 0xff);
                     op_ptr++;

                     /* Set the char to be output to the previous (diacritic) char */
                     unicode = prev_char;
                     prev_char = 0;
                  }
               }
               else
               {
                  if ((unicode >= 0x300) && (unicode <= 0x36f))
                  {
                     /* This char is a diacritic */
                     prev_char = unicode;
                  }
               }

               *op_ptr = (U8BIT)((unicode >> 8) & 0xff);
               op_ptr++;
               *op_ptr = (U8BIT)(unicode & 0xff);
               op_ptr++;

               num_op_char++;

#if 0
               if (reverse == FALSE)
               {
                  reverse = CheckUnicodeCharForReverseDirection(unicode);
               }
#endif
            }
         }

         // add null terminator
         *op_ptr = 0x00;
         *(op_ptr + 1) = 0x00;
      }

      if (decoded_utf8 != NULL)
      {
         STB_FreeMemory(decoded_utf8);
      }
   }
   else
   {
      op_buff = NULL;
   }

   /* return num chars and reverse */
   *nchar = num_op_char;
   *reverse_dir = reverse;

   FUNCTION_FINISH(STB_ConvertStringToUnicode);
   return(op_buff);
}

/*!**************************************************************************
 * @brief   Converts the given DVB coded string into a UTF-8 unicode string.
 *          The returned string will be preceded by the DVB byte, 0x15, indicating
 *          the string is UTF-8 format.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   string - DVB string to be converted
 * @param   nchar - number of characters, not bytes, in the returned string
 * @param   strip_DVB_cntrl_char - TRUE if DVB control character codes aren't
 *                                 to be included in the converted string
 * @param   lang_code - language code of the string, which may affect the ETSI defined
 *                      character code table used when doing the conversion.
 *                      If the code is 0 then the default table will be used.
 * @return  UTF-8 format string
 ****************************************************************************/
U8BIT* STB_ConvertStringToUTF8(U8BIT *string, U16BIT *nchar, BOOLEAN strip_DVB_cntrl_char,
   U32BIT lang_code)
{
   U8BIT *char_ptr;
   U8BIT byte_val;
   U8BIT table;
   U8BIT *op_buff;
   U8BIT *op_ptr;
   U16BIT num_ip_char;
   U16BIT num_op_char;
   BOOLEAN utf8;
   BOOLEAN utf16;
   U8BIT *decoded_utf8;
   U32BIT unicode;
   U16BIT nbytes;
   U16BIT i;
   U32BIT prev_char;
   U32BIT temp_char;
   U8BIT table_id;
   U8BIT encoding_type_id;
   U16BIT actual_size;

   FUNCTION_START(STB_ConvertStringToUTF8);

   ASSERT(nchar != NULL);

   char_ptr = string;

   table = 0;
   num_op_char = 0;
   nbytes = 0;
   utf8 = FALSE;
   utf16 = FALSE;
   decoded_utf8 = NULL;

   if (string != NULL)
   {
      /* Use the default table if there's no lang code or the code isn't found */
      table_id = default_ascii_table;

      if (lang_code != 0)
      {
         /* Lookup the lang code to find the table id to be used */
         for (i = 0; i < (sizeof(lang_code_lookup_table) / sizeof(S_LANG_CODE_ENTRY)); i++)
         {
            if (lang_code_lookup_table[i].lang_code == lang_code)
            {
               table_id = lang_code_lookup_table[i].table_id;
               break;
            }
         }
      }

      /* Determine type of character coding in string -
       * first byte indicates coding (see DVB SI spec for details) */
      byte_val = *char_ptr;
      if ((byte_val >= 0x20) || (byte_val == '\0'))
      {
         /* Use the default ascii table */
         table = table_id;
         num_ip_char = (U16BIT)strlen((char *)char_ptr);
      }
      else if ((byte_val >= 1) && (byte_val <= 11))
      {
         // one of the Latin tables specified in the DVB SI specification from ISO 8859.
         // 1 => 8859-5, 2 => 8859-6, 3 => 8859-7, 4 => 8859-8, 5 => 8859-9
         // 6 => 8859-10, 7 => 8859-11, 8 => 8859-12, 9 => 8859-13, 10 => 8859-14, 11 => 8859-15
         table = byte_val + 4;
         char_ptr++;
         num_ip_char = (U16BIT)strlen((char *)char_ptr);
      }
      else if (byte_val == 0x10)
      {
         // next 2 bytes indicate the table from ISO 8859
         table = ((*(char_ptr + 1)) << 8) | *(char_ptr + 2);
         char_ptr += 3;
         num_ip_char = (U16BIT)strlen((char *)char_ptr);
      }
      else if ((byte_val == 0x11) || (byte_val == UCS2_HEADER_VALUE))
      {
         utf16 = TRUE;
         char_ptr++;

         /* Count number of characters in the string */
         num_ip_char = 0;
         do
         {
            unicode = ((*char_ptr) << 8) | *(char_ptr + 1);
            char_ptr += 2;
            if (unicode != '\0')
            {
               num_ip_char++;
            }
         }
         while (unicode != '\0');
         char_ptr = string + 1;
      }
      else if ((byte_val == UTF8_HEADER_VALUE) || (byte_val == 0x1F))
      {
         /* The string is already UTF-8 */
         utf8 = TRUE;
         char_ptr++;

         if (byte_val == 0x1F)
         {
            /* This is a compressed UTF-8 string with the next byte defining the table
             * to be used for decoding */
            encoding_type_id = *char_ptr;
            char_ptr++;

            if ((encoding_type_id == 1) || (encoding_type_id == 2))
            {
               /* It's not possible to tell how long the string is without decoding it first,
                * so a sufficiently large buffer is allocated to decode into */
               decoded_utf8 = (U8BIT *)STB_GetMemory(MAX_DECODE_BUFFER_SIZE);
               if (decoded_utf8 != NULL)
               {
                  actual_size = STB_HuffmanDecompress(encoding_type_id, char_ptr, decoded_utf8,
                        MAX_DECODE_BUFFER_SIZE);

                  if (actual_size > 0)
                  {
                     /* The length is now known, the buffer can be resized */
                     decoded_utf8 = (U8BIT *)STB_ResizeMemory(decoded_utf8, actual_size);
                     char_ptr = decoded_utf8;
                  }
                  else
                  {
                     /* Decompression failed */
                     *nchar = 0;
                     STB_FreeMemory(decoded_utf8);
                     FUNCTION_FINISH(STB_ConvertStringToUTF8);
                     return(NULL);
                  }
               }
               else
               {
                  /* Can't decode the string */
                  *nchar = 0;
                  FUNCTION_FINISH(STB_ConvertStringToUTF8);
                  return(NULL);
               }
            }
            else
            {
               /* Don't know how to decode the string */
               *nchar = 0;
               FUNCTION_FINISH(STB_ConvertStringToUTF8);
               return(NULL);
            }
         }

         /* Find the length of the string in chars */
         num_ip_char = 0;
         do
         {
            if ((*char_ptr & 0x80) == 0)
            {
               char_ptr++;
            }
            else if ((*char_ptr & 0xE0) == 0xC0)
            {
               char_ptr += 2;
            }
            else if ((*char_ptr & 0xF0) == 0xE0)
            {
               char_ptr += 3;
            }
            else if ((*char_ptr & 0xF8) == 0xF0)
            {
               char_ptr += 4;
            }
            else
            {
               /* Invalid byte sequence */
               if (decoded_utf8 != NULL)
               {
                  STB_FreeMemory(decoded_utf8);
               }

               *nchar = 0;
               FUNCTION_FINISH(STB_ConvertStringToUTF8);
               return(NULL);
            }
            num_ip_char++;
         }
         while (*char_ptr != '\0');

         if (decoded_utf8 != NULL)
         {
            char_ptr = decoded_utf8;
         }
         else
         {
            char_ptr = string + 1;
         }
      }
      else
      {
         // invalid string - exit
         *nchar = 0;
         FUNCTION_FINISH(STB_ConvertStringToUTF8);
         return(NULL);
      }


      /* Calculate the number of bytes required for the conversion.
       * Each character could result in 4 bytes being generated, which is the worst case */
      nbytes = num_ip_char * 4;

      /* Two additional bytes required for 0x15 at the start and 0x00 at the end */
      nbytes += 2;

      /* Now create output string */
      op_buff = (U8BIT *)STB_GetMemory(nbytes);
      if (op_buff != NULL)
      {
         prev_char = 0;
         op_ptr = op_buff;

         /* Prepend with code indicating it's a UTF-8 string */
         *op_ptr = UTF8_HEADER_VALUE;
         op_ptr++;

         /* Copy over each character, converting if necessary */
         for (i = 0; i < num_ip_char; i++)
         {
            unicode = 0;
            if (utf16)
            {
               unicode = ((*char_ptr) << 8) | *(char_ptr + 1);
               char_ptr += 2;
            }
            else if (utf8)
            {
               byte_val = *char_ptr;
               if ((byte_val & 0x80) == 0)
               {
                  unicode = byte_val;
                  char_ptr++;
               }
               else if ((byte_val & 0xE0) == 0xC0)
               {
                  unicode = byte_val & 0x1F;
                  unicode <<= 6;
                  unicode += (*(char_ptr + 1) & 0x3F);
                  char_ptr += 2;
               }
               else if ((byte_val & 0xF0) == 0xE0)
               {
                  unicode = byte_val & 0x0F;
                  unicode <<= 6;
                  unicode += (*(char_ptr + 1) & 0x3F);
                  unicode <<= 6;
                  unicode += (*(char_ptr + 2) & 0x3F);
                  char_ptr += 3;
               }
               else if ((byte_val & 0xF8) == 0xF0)
               {
                  unicode = byte_val & 0x07;
                  unicode <<= 6;
                  unicode += (*(char_ptr + 1) & 0x3F);
                  unicode <<= 6;
                  unicode += (*(char_ptr + 2) & 0x3F);
                  unicode <<= 6;
                  unicode += (*(char_ptr + 3) & 0x3F);
                  char_ptr += 4;
               }

               if ((unicode >= CHAR_TABLE_START_ASCII_CODE) && (unicode <= 0xff))
               {
                  /* Map this range of chars to ISO-6937 using table 0 */
                  unicode = utf8_char_map[unicode - CHAR_TABLE_START_ASCII_CODE];
               }
            }
            else
            {
               /* ASCII codes - convert character to unicode */
               byte_val = *char_ptr;
               char_ptr++;
               if (byte_val < CHAR_TABLE_START_ASCII_CODE)
               {
                  unicode = (U32BIT)byte_val;
               }
               else
               {
                  if (table < MAX_CHAR_MAP_TABLES)
                  {
                     unicode = char_map[table][byte_val - CHAR_TABLE_START_ASCII_CODE];
                  }
               }
            }

            /* Check for control codes */
            if (((unicode > 0x0000) && (unicode <= 0x001f)) ||
                ((unicode >= 0x0080) && (unicode <= 0x009f)) ||
                ((unicode >= 0xe080) && (unicode <= 0xe09f)))
            {
               switch (unicode)
               {
                  // only accept LF and convert to 0xe08a
                  // DVB - CR/LF.
                  case 0x008a:
                  case 0xe08a:
                  {
                     if (strip_DVB_cntrl_char == FALSE)
                     {
                        unicode = 0xe08a;
                     }
                     else
                     {
                        unicode = 0;
                     }
                     break;
                  }
                  // DVB - character emphasis ON
                  case 0x0086:
                  case 0xe086:
                  {
                     if (strip_DVB_cntrl_char == FALSE)
                     {
                        unicode = 0xe086;
                     }
                     else
                     {
                        unicode = 0;
                     }
                     break;
                  }
                  // DVB - character emphasis OFF
                  case 0x0087:
                  case 0xe087:
                  {
                     if (strip_DVB_cntrl_char == FALSE)
                     {
                        unicode = 0xe087;
                     }
                     else
                     {
                        unicode = 0;
                     }
                     break;
                  }
                  // convert all other control codes to 0
                  default:
                  {
                     unicode = 0;
                     break;
                  }
               }
            }

            /* If valid character copy to output string */
            if (unicode != 0)
            {
               /* Check for diacritic character. In DVB strings the diacritic precedes the char
                * it combines with, but in Unicode a diacritic follows the char, so the char order needs
                * to be swapped. Only a single combining diacritic is valid in DVB so only need
                * to check previous char to see if a swap is required. */
               if (prev_char != 0)
               {
                  if ((unicode >= 0x300) && (unicode <= 0x36f))
                  {
                     /* This char is also a diacritic which means there's two diacritics together,
                      * so output the previous diacritic and save this one to see if it needs to
                      * combine with the next char */
                     temp_char = unicode;
                     unicode = prev_char;
                     prev_char = temp_char;
                  }
                  else
                  {
                     /* Diacritic preceding a normal char, so output the normal char first */
                     op_ptr = OutputUTF8(op_ptr, unicode);
                     num_op_char++;

                     /* Set the char to be output to the previous (diacritic) char */
                     unicode = prev_char;
                     prev_char = 0;
                  }
               }
               else
               {
                  if ((unicode >= 0x300) && (unicode <= 0x36f))
                  {
                     /* This char is a diacritic */
                     prev_char = unicode;
                     unicode = 0;
                  }
               }

               if (unicode != 0)
               {
                  op_ptr = OutputUTF8(op_ptr, unicode);
                  num_op_char++;
               }
            }
         }

         if (prev_char != 0)
         {
            /* Last char was a diacritic so output it */
            op_ptr = OutputUTF8(op_ptr, prev_char);
            num_op_char++;
         }

         /* Add null terminator */
         *op_ptr++ = 0x00;

         if ((op_ptr - op_buff) < nbytes)
         {
            nbytes = op_ptr - op_buff;
            op_buff = STB_ResizeMemory(op_buff, nbytes);
         }
      }

      if (decoded_utf8 != NULL)
      {
         STB_FreeMemory(decoded_utf8);
      }
   }
   else
   {
      op_buff = NULL;
   }

   /* return num chars and reverse */
   *nchar = num_op_char;

   FUNCTION_FINISH(STB_ConvertStringToUTF8);

   return(op_buff);
}

/**
 *

 *
 * @brief   Releases the specified unicode string, freeing associated heap resources.
 *
 * @param   string      - pointer to the unicode string to be released.
 *                              NB - This will have been supplied by a previous call to
 *                                   STB_ConvertStringToUnicode()!
 *

 *
 */
void STB_ReleaseUnicodeString(U8BIT *string)
{
   FUNCTION_START(STB_ReleaseUnicodeString);

   if (string != NULL)
   {
      STB_FreeMemory(string);
   }

   FUNCTION_FINISH(STB_ReleaseUnicodeString);
}

/*!**************************************************************************
 * @brief   Creates the given string from UTF-16 to UTF-8 and returns a new string.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   src - UTF-16 string to be converted
 * @param   outlen - number of bytes in the returned string
 * @return  UTF-8 format string
 ****************************************************************************/
U8BIT* STB_ConvertUTF16toUTF8( U8BIT *src, U32BIT *outlen )
{
   const U32BIT byteMask = 0xBF;
   const U32BIT byteMark = 0x80;
   const int halfShift = 10;
   const U32BIT halfBase = 0x0010000UL;
   const U8BIT firstByteMark[7] = { 0x00, 0x00, 0xC0, 0xE0, 0xF0, 0xF8, 0xFC };
   U8BIT *result, *pch;
   U16BIT bytesToWrite;
   U32BIT ch1, ch2, length;

   FUNCTION_START(STB_ConvertUTF16toUTF8);

   if ((src == NULL) || (src[UNICODE_HEADER_POS] != UTF16_HEADER_VALUE))
   {
      DBGPRINT("ERR");
      result = NULL;
      *outlen = 0;
   }
   else
   {
      length = STB_UnicodeStringLen(src);
      if (length == 0)
      {
         DBGPRINT("ERR");
         result = NULL;
         *outlen = 0;
      }
      else
      {
         src++; // skip UTF16_HEADER_VALUE

         /* Allocate memory for the UTF-8 string allowing 4 bytes per char + 0x15 and null */
         result = STB_GetMemory(length * 4 + 2);
         if (result == NULL)
         {
            DBGPRINT("ERR getting 3 * %d bytes", length);
            *outlen = 0;
         }
         else
         {
            *result = UTF8_HEADER_VALUE;

            pch = result + 1;
            while (length--)
            {
               ch1 = *src++;
               ch1 <<= 8;
               ch1 += *src++;
               /* If we have a surrogate pair, convert to UTF32 first. */
               if (ch1 >= UNI_SUR_HIGH_START && ch1 <= UNI_SUR_HIGH_END)
               {
                  if (length)
                  {
                     ch2 = (*src << 8) + *(src + 1);
                     /* If it's a low surrogate, convert to UTF32. */
                     if (ch2 >= UNI_SUR_LOW_START && ch2 <= UNI_SUR_LOW_END)
                     {
                        ch1 = ((ch1 - UNI_SUR_HIGH_START) << halfShift) + (ch2 - UNI_SUR_LOW_START) + halfBase;
                        src += 2;
                     }
                     else
                     {
                        DBGPRINT("ERR");
                        break;
                     }
                  }
                  else
                  {
                     DBGPRINT("ERR");
                     break;
                  }
               }
               else if (ch1 >= UNI_SUR_LOW_START && ch1 <= UNI_SUR_LOW_END)
               {
                  /* UTF-16 surrogate values are illegal in UTF-32 */
                  DBGPRINT("ERR");
                  break;
               }
               /* Figure out how many bytes the result will require */
               if (ch1 < (U32BIT)0x80)
               {
                  bytesToWrite = 1;
               }
               else if (ch1 < (U32BIT)0x800)
               {
                  bytesToWrite = 2;
               }
               else if (ch1 < (U32BIT)0x10000)
               {
                  bytesToWrite = 3;
               }
               else if (ch1 < (U32BIT)0x110000)
               {
                  bytesToWrite = 4;
               }
               else
               {
                  bytesToWrite = 3;
                  ch1 = UNI_REPLACEMENT_CHAR;
               }
               pch += bytesToWrite;
               switch (bytesToWrite) /* note: everything falls through. */
               {
                  case 4: *--pch = (U8BIT)((ch1 | byteMark) & byteMask); ch1 >>= 6;
                  case 3: *--pch = (U8BIT)((ch1 | byteMark) & byteMask); ch1 >>= 6;
                  case 2: *--pch = (U8BIT)((ch1 | byteMark) & byteMask); ch1 >>= 6;
                  case 1: *--pch = (U8BIT) (ch1 | firstByteMark[bytesToWrite]);
               }
               pch += bytesToWrite;
            }

            /* Append null terminator */
            *pch = 0;
            pch++;

            *outlen = pch - result;

            result = STB_ResizeMemory(result, *outlen);
         }
      }
   }

   FUNCTION_FINISH(STB_ConvertUTF16toUTF8);

   return result;
}

/**
 *

 *
 * @brief   Checks for a string of only spaces
 *
 * @param   string_ptr - pointer to the string to be checked.
 *
 * @return   TRUE if empty, FALSE otherwise.
 *
 */
BOOLEAN STB_IsStringEmpty(U8BIT *string_ptr)
{
   BOOLEAN retval;
   U8BIT *ucode_string;
   BOOLEAN rev_string;
   U16BIT num_char;
   U16BIT i;
   U16BIT char_code;
   U8BIT *byte_ptr;

   FUNCTION_START(STB_IsStringEmpty);

   retval = TRUE;
   ucode_string = STB_ConvertStringToUnicode(string_ptr, &rev_string, &num_char, TRUE, 0);

   if (ucode_string != NULL)
   {
      byte_ptr = &ucode_string[1];
      for (i = 0; i < num_char; i++)
      {
         char_code = (U16BIT)((*byte_ptr) << 8);
         byte_ptr++;
         char_code |= (U16BIT)(*byte_ptr);
         byte_ptr++;

         if ((char_code != 0x0020) && (char_code != 0x00a0))
         {
            retval = FALSE;
            break;
         }
      }
      STB_ReleaseUnicodeString(ucode_string);
   }

   FUNCTION_FINISH(STB_IsStringEmpty);
   return(retval);
}

/**
 * @brief   Sets default ascii table to be used, if not overridden by the table index at the start
 *          of a string
 * @param   table - index of the table to be used (0 to 15), as defined in ETSI 300 468,
 *                  Annex A, table A.3
 */
void STB_SetDefaultAsciiTable(U8BIT table)
{
   FUNCTION_START(STB_SetDefaultAsciiTable);

   if (table < MAX_CHAR_MAP_TABLES)
   {
      default_ascii_table = table;
   }

   FUNCTION_FINISH(STB_SetDefaultAsciiTable);
}

/**
 *

 *
 * @brief   Compares the contents of the two given ASCII strings and returns the status
 *                (as per strcmp) but ignores case
 *
 * @param   string1_ptr         Pointer to the 'master' string
 * @param   string2_ptr         Pointer to the 'slave' string
 *
 * @return   Result of the comparison.  0 if equal, +ve if string1_ptr > string2_ptr,
 *                -ve if string1_ptr < string2_ptr.
 *
 */
S8BIT STB_CompareStringsIgnoreCase(U8BIT *string1_ptr, U8BIT *string2_ptr)
{
   U32BIT str1_len, str2_len;
   U32BIT min_str_len;
   U32BIT index;
   U8BIT char1, char2;
   S8BIT difference = STRINGS_EQUAL;  // Return value indicating difference between strings

   FUNCTION_START(STB_CompareStringsIgnoreCase);

   if (string1_ptr != NULL)
   {
      str1_len = strlen((char *)string1_ptr);
   }
   else
   {
      str1_len = 0;
   }
   if (string2_ptr != NULL)
   {
      str2_len = strlen((char *)string2_ptr);
   }
   else
   {
      str2_len = 0;
   }

   if (str2_len < str1_len)
   {
      min_str_len = str2_len;
   }
   else
   {
      min_str_len = str1_len;
   }

   for (index = 0; (index < min_str_len) && (difference == STRINGS_EQUAL); index++)
   {
      char1 = *string1_ptr;
      char2 = *string2_ptr;

      if ((char1 >= 'A') && (char1 <= 'Z'))
      {
         char1 = 'a' + (char1 - 'A');
      }

      if ((char2 >= 'A') && (char2 <= 'Z'))
      {
         char2 = 'a' + (char2 - 'A');
      }

      if (char1 > char2)
      {
         difference = FIRST_STRING_GREATER;
      }
      else if (char1 < char2)
      {
         difference = SECOND_STRING_GREATER;
      }

      string1_ptr++;
      string2_ptr++;
   }

   if ((difference == STRINGS_EQUAL) && (str1_len != str2_len))
   {
      if (str1_len < str2_len)
      {
         difference = SECOND_STRING_GREATER;
      }
      else
      {
         difference = FIRST_STRING_GREATER;
      }
   }

   FUNCTION_FINISH(STB_CompareStringsIgnoreCase);

   return(difference);
}

/**
 *

 *
 * @brief   Unicode version of sprintf.
 *
 * @param   strip_DVB_cntrl_char True if all DVB control chars are to be removed
 * @param   reverse_dir          reverse print direction (out)
 * @param   format_ptr           Takes a format string of any format, and recognises
 *                                     the following tokens:
 *                                     %hu (U8BIT), %u (U16BIT), %lu (U32BIT),
 *                                     %hd (S8BIT), %d (S16BIT), %ld (S32BIT),
 *                                     %hx (U8BIT), %x (U16BIT), %lx (U32BIT),
 *                                     %s (U8BIT*) for 8-bit Ascii or Unicode strings,
 *                                     %% (to output a % character).
 *                                     For the decimal and hexadecimal number tokens
 *                                     there is also limited support for number/precision
 *                                     flags up to a maximum width of 11 digits (e.g. %1 to
 *                                     %11 to pad the number with leading spaces, %01 to
 *                                     %011 to pad the number with leading zeroes) the
 *                                     maximum width of the number/precision flag is 3
 *                                     characters.
 *                ...                  Parameters to be subsituted into the format string.
 *
 * @return   NULL if there is any error, otherwise a newly allocated unicode string
 *                Note: The string must be freed with STB_ReleaseUnicodeString.
 *
 */
U8BIT* STB_FormatUnicodeString( BOOLEAN strip_DVB_cntrl_char,
   BOOLEAN *reverse_dir,
   const U8BIT *const format_ptr,
   ... )
{
   typedef struct format_param
   {
      U16BIT format_index; /* Number of bytes including DVB header */
      U16BIT format_adj; /* Number of bytes to adjucst for format tokens */
      U8BIT *param;
      U16BIT param_size; /* Number of Unicode characters */
      BOOLEAN free_param; /* String has been reallocated so must be freed */
      struct format_param *next_ptr;
   } E_FORMAT_PARAM;

   va_list args_ptr;
   U8BIT *uni_string_ptr;
   U8BIT *uni_format_ptr;
   BOOLEAN reverse;
   U16BIT format_length;
   U32BIT nchar;   /* Number of Characters */
   U16BIT total_length;   /* Number of U16BIT characters not bytes */
   BOOLEAN free_uni_format_ptr;
   BOOLEAN error_exit;
   U16BIT format_loop;
   U16BIT uni_cur_char;
   E_FORMAT_PARAM *uni_format_param_ptr;
   E_FORMAT_PARAM *uni_head_param_list_ptr;
   E_FORMAT_PARAM *uni_last_param_list_ptr;
   U8BIT *tmp_str_ptr;
   U16BIT cur_format_index;
   U8BIT u8_number;
   U16BIT u16_number;
   U32BIT u32_number;
   S8BIT s8_number;
   S16BIT s16_number;
   S32BIT s32_number;
   U8BIT ascii_string[MAX_NUMBER_DIGITS + 1];
   U16BIT uni_next_char;
   U16BIT length_adj;
   U8BIT *tmp_format_ptr;
   U8BIT num_form_spec_str[MAX_NUM_FORMAT_SPEC_STR_SIZE];
   U8BIT num_form_spec_str_index;

   FUNCTION_START(STB_FormatUnicodeString);

   ASSERT(reverse_dir != NULL);

   /* Initialise the argument pointer */
   va_start(args_ptr, format_ptr);
   *reverse_dir = FALSE;
   uni_string_ptr = NULL;
   uni_head_param_list_ptr = NULL;
   uni_format_param_ptr = NULL;
   uni_last_param_list_ptr = NULL;
   error_exit = FALSE;

   uni_format_ptr = (U8BIT *)format_ptr;

   MakeUnicode( &uni_format_ptr, &free_uni_format_ptr, &format_length, &reverse,
      strip_DVB_cntrl_char);

   total_length = format_length;
   *reverse_dir |= reverse;

   /*  If the format string is valid */
   if (uni_format_ptr != NULL)
   {
      /* Scan format loop for tokens.
       * Stop if there is no space for 2 tokens i.e %<x>. */
      for (format_loop = 0;
           (format_loop < (format_length - 1)) && (error_exit == FALSE);
           format_loop++)
      {
         /* Allow for the unicode header */
         uni_cur_char = STB_GetUnicodeStringChar(uni_format_ptr, format_loop);

         /* Look for the '%' skipping unicode header */
         if (uni_cur_char == UNI_PERCENT_CHAR)
         {
            cur_format_index = format_loop + 1;

            /* Allow for the unicode header */
            uni_cur_char = STB_GetUnicodeStringChar(uni_format_ptr, cur_format_index);

            /* Check for next token element */
            /* Note not a switch because unicode chars are not constants. */
            if (uni_cur_char == UNI_SMALL_S_CHAR)
            {
               uni_format_param_ptr = (E_FORMAT_PARAM *)STB_GetMemory(
                     sizeof(E_FORMAT_PARAM));
               if (uni_format_param_ptr != NULL)
               {
                  /* First allocation? */
                  if (uni_head_param_list_ptr == NULL)
                  {
                     uni_head_param_list_ptr = uni_format_param_ptr;
                     uni_last_param_list_ptr = uni_format_param_ptr;
                  }
                  else
                  {
                     uni_last_param_list_ptr->next_ptr = uni_format_param_ptr;
                     uni_last_param_list_ptr = uni_format_param_ptr;
                  }
                  uni_format_param_ptr->format_index = (2 * format_loop) + 1; /* start of '%' */
                  uni_format_param_ptr->format_adj = 2 * 2; /* Conver U16BIT to U8BIT */
                  uni_format_param_ptr->param = (U8BIT *) va_arg(args_ptr, char *);
                  MakeUnicode( &(uni_format_param_ptr->param),
                     &(uni_format_param_ptr->free_param), &(uni_format_param_ptr->param_size),
                     &reverse, strip_DVB_cntrl_char);
                  uni_format_param_ptr->next_ptr = NULL;
                  total_length = ((total_length - 2) +
                                  (uni_format_param_ptr->param_size));
                  uni_format_param_ptr = uni_format_param_ptr->next_ptr;
                  format_loop = cur_format_index; /* Skip parsed tokens */
                  *reverse_dir |= reverse;
               }
               else
               {
                  error_exit = TRUE;
               }
            } /* end UNI_SMALL_S_CHAR */
            else /* handle numeric format specifiers */
            {
               /* build up numeric format specifier on the fly */
               length_adj = 0;
               memset((void *)num_form_spec_str, 0, sizeof(num_form_spec_str));
               num_form_spec_str[0] = '%';
               num_form_spec_str_index = 1;

               /* handle zero padding and number width specifiers */
               while (((U8BIT)uni_cur_char >= '0' && (U8BIT)uni_cur_char <= '9') && (num_form_spec_str_index <= MAX_NUM_WIDTH_DIGITS))
               {
                  /* place the next character into the numeric format spec. string */
                  num_form_spec_str[num_form_spec_str_index] = (U8BIT)uni_cur_char;
                  num_form_spec_str_index++;

                  /* if we have'nt exceeded the maximum number of digits or reached the end of the
                     string read in the next character */
                  if ((atoi((char *)&num_form_spec_str[1]) <= MAX_NUMBER_DIGITS)
                      &&
                      ((cur_format_index + 1) <= format_length))
                  {
                     uni_cur_char = STB_GetUnicodeStringChar(uni_format_ptr, (cur_format_index + 1));
                     cur_format_index++;
                  }
                  /* otherwise force an error that gets us out of this loop and forces an exit from the main loop */
                  else
                  {
                     uni_cur_char = 0xffff;
                  }
               }

               switch (uni_cur_char)
               {
                  /* handle %%*/
                  case UNI_PERCENT_CHAR:
                  {
                     /* must be "%%" dont allow width specifiers here e.g. "%02%" etc */
                     if (num_form_spec_str_index == 1)
                     {
                        length_adj = 2;
                        snprintf((char *)ascii_string, MAX_NUMBER_DIGITS + 1, "%%");
                     }
                     break;
                  }

                  /* handle %u %d %x %X */
                  case UNI_SMALL_U_CHAR:
                  case UNI_SMALL_D_CHAR:
                  case UNI_SMALL_X_CHAR:
                  case UNI_LARGE_X_CHAR:
                  {
                     /* finish building the format spec string and then sprintf into the buffer */
                     length_adj = num_form_spec_str_index + 1;
                     num_form_spec_str[num_form_spec_str_index] = (U8BIT)uni_cur_char;
                     if (uni_cur_char == UNI_SMALL_D_CHAR)
                     {
                        s16_number = (S16BIT) va_arg(args_ptr, int);
                        snprintf((char *)ascii_string, MAX_NUMBER_DIGITS + 1, (char *)num_form_spec_str, s16_number );
                     }
                     else
                     {
                        u16_number = (U16BIT) va_arg(args_ptr, int);
                        snprintf((char *)ascii_string, MAX_NUMBER_DIGITS + 1, (char *)num_form_spec_str, u16_number );
                     }
                     break;
                  }

                  /* handle "%hu" "%hd" "%hx" "%hX" "%lu" "%ld" "%lx" "%lX" */
                  case UNI_SMALL_H_CHAR:
                  case UNI_SMALL_L_CHAR:
                  {
                     /* there needs to be at least one more character in the string to modify the h or u */
                     if ((cur_format_index + 1) <= format_length)
                     {
                        /* read it, and build up the rest of the format specifier string */
                        uni_next_char = STB_GetUnicodeStringChar(uni_format_ptr, (cur_format_index + 1));
                        cur_format_index++;
                        length_adj = num_form_spec_str_index + 2;
                        num_form_spec_str[num_form_spec_str_index] = (U8BIT)uni_cur_char;
                        num_form_spec_str_index++;
                        num_form_spec_str[num_form_spec_str_index] = (U8BIT)uni_next_char;

                        /* now process the pair of unicode characters together to read the
                           correct size and sprintf into local buffer ready for conversion to unicode */
                        switch ((U32BIT)((uni_cur_char << 16) | uni_next_char))
                        {
                           /* "%lu", "%lx", "%lX" */
                           case UNI_SMALL_L_SMALL_U_CHARS:
                           case UNI_SMALL_L_SMALL_X_CHARS:
                           case UNI_SMALL_L_LARGE_X_CHARS:
                           {
                              u32_number = (U32BIT) va_arg(args_ptr, int);
                              snprintf((char *)ascii_string, MAX_NUMBER_DIGITS + 1, (char *)num_form_spec_str, u32_number );
                              break;
                           }

                           /* "%hu", "%hx", "%hX" */
                           case UNI_SMALL_H_SMALL_U_CHARS:
                           case UNI_SMALL_H_SMALL_X_CHARS:
                           case UNI_SMALL_H_LARGE_X_CHARS:
                           {
                              u8_number = (U8BIT) va_arg(args_ptr, int);
                              snprintf((char *)ascii_string, MAX_NUMBER_DIGITS + 1, (char *)num_form_spec_str, u8_number );
                              break;
                           }

                           /* "%ld" */
                           case UNI_SMALL_L_SMALL_D_CHARS:
                           {
                              s32_number = (S32BIT) va_arg(args_ptr, int);
                              snprintf((char *)ascii_string, MAX_NUMBER_DIGITS + 1, (char *)num_form_spec_str, s32_number );
                              break;
                           }

                           /* "%hd" */
                           case UNI_SMALL_H_SMALL_D_CHARS:
                           {
                              s8_number = (S8BIT) va_arg(args_ptr, int);
                              snprintf((char *)ascii_string, MAX_NUMBER_DIGITS + 1, (char *)num_form_spec_str, s8_number );
                              break;
                           }

                           /* unsupported format specifier %h? or %l? */
                           default:
                           {
                              error_exit = TRUE;
                              break;
                           }
                        } /* end switch on uni_next_char */
                     } /* end if */

                     break;
                  } /* end case UNI_SMALL_L_CHAR, UNI_SMALL_H_CHAR */

                  /* unsupported format specifier "%?" */
                  default:
                  {
                     error_exit = TRUE;
                     break;
                  }
               } /* end Switch on uni_cur_char */

               /* Correct tokens found to consume */
               if (length_adj != 0)
               {
                  uni_format_param_ptr = (E_FORMAT_PARAM *)STB_GetMemory(
                        sizeof(E_FORMAT_PARAM));
                  if (uni_format_param_ptr != NULL)
                  {
                     /* First allocation? */
                     if (uni_head_param_list_ptr == NULL)
                     {
                        uni_head_param_list_ptr = uni_format_param_ptr;
                        uni_last_param_list_ptr = uni_format_param_ptr;
                     }
                     else
                     {
                        uni_last_param_list_ptr->next_ptr = uni_format_param_ptr;
                        uni_last_param_list_ptr = uni_format_param_ptr;
                     }
                     uni_format_param_ptr->format_index = (2 * format_loop) + 1; /* start of '%' */
                     uni_format_param_ptr->format_adj = 2 * length_adj;
                     uni_format_param_ptr->param = ascii_string;
                     MakeUnicode( &(uni_format_param_ptr->param),
                        &(uni_format_param_ptr->free_param), &(uni_format_param_ptr->param_size),
                        &reverse, FALSE);
                     uni_format_param_ptr->next_ptr = NULL;
                     total_length = ((total_length - length_adj) +
                                     (uni_format_param_ptr->param_size));
                     uni_format_param_ptr = uni_format_param_ptr->next_ptr;
                     *reverse_dir |= reverse;
                  }
                  else
                  {
                     error_exit = TRUE;
                  }
               }
               format_loop = cur_format_index; /* Skip parsed tokens */
            } /* End else check for numeric format specifiers */
         } /* End if '%' found */
      } /* End for to loop over the format string */

      if (error_exit == FALSE)
      {
         /* Allocate memory for new string plus space for null terminator and header */
         uni_string_ptr = (U8BIT *)STB_GetMemory(sizeof(U8BIT) * STB_UTF16_LEN_TO_BYTES_IN_STRING(total_length));
         tmp_str_ptr = uni_string_ptr;
         if (uni_string_ptr != NULL)
         {
            /* No need to add header as it will be added from the format string */
            uni_format_param_ptr = uni_head_param_list_ptr;
            tmp_format_ptr = uni_format_ptr;
            while (tmp_str_ptr < (uni_string_ptr + STB_UTF16_LEN_TO_BYTES_IN_STRING(total_length)))
            {
               if (uni_format_param_ptr != NULL)
               {
                  /* Need to copy some of the format string over? */
                  if ((tmp_format_ptr - uni_format_ptr) <
                      uni_format_param_ptr->format_index)
                  {
                     nchar = ((uni_format_param_ptr->format_index) - (tmp_format_ptr - uni_format_ptr));
                     memcpy((void *)tmp_str_ptr, (void *)tmp_format_ptr, nchar);
                     tmp_str_ptr += nchar;
                     tmp_format_ptr += nchar;
                  }
                  /* copy the parameter string skipping the unicode header */
                  if (uni_format_param_ptr->param != NULL)
                  {
                     memcpy((void *)tmp_str_ptr,
                        (void *)((uni_format_param_ptr->param) + 1),
                        2 * (uni_format_param_ptr->param_size));
                  }
                  tmp_str_ptr += (2 * (uni_format_param_ptr->param_size));
                  tmp_format_ptr += uni_format_param_ptr->format_adj;
                  uni_format_param_ptr = uni_format_param_ptr->next_ptr;
               }
               else
               {
                  /* Copy the end of format string */
                  nchar = (STB_UTF16_LEN_TO_BYTES_IN_STRING(format_length) - (tmp_format_ptr - uni_format_ptr));
                  memcpy((void *)tmp_str_ptr, (void *)tmp_format_ptr, nchar);
                  tmp_str_ptr += nchar;
                  tmp_format_ptr += nchar;
               } /* end of no more params to process */
            } /* loop to copy over string */
              /* No need to add null terminator as got from format string */
         }
      } /* End if no error */
   } /* End if Format string is valid */
   else
   {
      uni_string_ptr = NULL;
   }

   /* When constructing the string need to check for reverse_dir */
   /* Clean up and free resources */
   if (free_uni_format_ptr == TRUE)
   {
      STB_ReleaseUnicodeString(uni_format_ptr);
   }

   while (uni_head_param_list_ptr != NULL)
   {
      uni_format_param_ptr = uni_head_param_list_ptr;
      uni_head_param_list_ptr = uni_format_param_ptr->next_ptr;
      if (uni_format_param_ptr->free_param == TRUE)
      {
         STB_FreeMemory(uni_format_param_ptr->param);
      }
      STB_FreeMemory(uni_format_param_ptr);
   }

   va_end(args_ptr);

   if (uni_string_ptr != NULL)
   {
      /* Convert the resulting string to UTF-8 */
      tmp_str_ptr = STB_ConvertUTF16toUTF8(uni_string_ptr, &nchar);
      STB_ReleaseUnicodeString(uni_string_ptr);
      uni_string_ptr = tmp_str_ptr;
   }

   FUNCTION_FINISH(STB_FormatUnicodeString);

   return(uni_string_ptr);
}

/*!**************************************************************************
 * @brief   Creates a new string by inserting one string into another at a given position,
 *          with the option of replacing the char at the given position. Strings can be
 *          passed as DVB or unicode, but output will always be unicode and the resulting
 *          string must be freed.
 * @param   src_str - string into which the insertion will be made
 * @param   insert_pos - position in the source string to make the insertion, it will be after this position
 * @param   insert_str - string to be inserted
 * @param   replace_char - TRUE if the char at the insertion point is to be replaced by the insertion string
 * @return  new string with text inserted
 ****************************************************************************/
U8BIT* STB_UnicodeInsertString(U8BIT *src_str, U16BIT insert_pos, U8BIT *insert_str, BOOLEAN replace_char)
{
   U8BIT *uni_src;
   U8BIT *uni_insert;
   U8BIT *new_str;
   U8BIT *str_ptr;
   BOOLEAN reverse;
   U16BIT num_chars;
   U16BIT num_bytes;

   FUNCTION_START(STB_UnicodeInsertString);

   new_str = NULL;

   if ((src_str != NULL) && (insert_str != NULL))
   {
      if (STB_IsUnicodeString(src_str))
      {
         uni_src = src_str;
         num_bytes = STB_GetNumBytesInString(uni_src);
      }
      else
      {
         uni_src = STB_ConvertStringToUnicode(src_str, &reverse, &num_chars, FALSE, 0);
         num_bytes = STB_UTF16_LEN_TO_BYTES_IN_STRING(num_chars);
      }

      if (replace_char)
      {
         num_bytes -= 2;
      }

      if (STB_IsUnicodeString(insert_str))
      {
         uni_insert = insert_str;
      }
      else
      {
         uni_insert = STB_ConvertStringToUnicode(insert_str, &reverse, &num_chars, FALSE, 0);
      }

      if ((uni_src != NULL) && (uni_insert != NULL))
      {
         /* Subtract 3 for the unicode header byte, 0x11, and the trailing null bytes */
         num_bytes += (STB_GetNumBytesInString(uni_insert) - 3);

         /* Allocate memory for the new string and copy the data into it */
         new_str = STB_GetMemory(num_bytes);
         if (new_str != NULL)
         {
            /* Copy the unicode header byte and all chars upto the insertion point */
            num_bytes = (insert_pos * 2) + 1;
            memcpy(new_str, uni_src, num_bytes);
            str_ptr = new_str + num_bytes;

            /* Copy the string being inserted, without the unicode header byte */
            num_bytes = STB_GetNumBytesInString(uni_insert) - 3;
            memcpy(str_ptr, uni_insert + 1, num_bytes);
            str_ptr += num_bytes;

            if (replace_char)
            {
               /* Char is being replaced, so skip it when copying the remainder of the source string */
               insert_pos++;
            }

            /* Copy the remainder of the source string */
            num_bytes = STB_GetNumBytesInString(uni_src) - ((insert_pos * 2) + 1);
            memcpy(str_ptr, uni_src + (insert_pos * 2) + 1, num_bytes);
         }
      }

      if ((uni_insert != NULL) && (uni_insert != insert_str))
      {
         STB_ReleaseUnicodeString(uni_insert);
      }

      if ((uni_src != NULL) && (uni_src != src_str))
      {
         STB_ReleaseUnicodeString(uni_src);
      }
   }

   FUNCTION_FINISH(STB_UnicodeInsertString);

   return(new_str);
}

/**
 * @brief   Strips the DVB control characters from a string that's already in UTF-8 or UTF-16
 *          format. The control chars that are stripped are DVB emphasis on/off and DVB CR/LF.
 *          The input string isn't changed and the returned string must be freed by calling
 *          STB_ReleaseUnicodeString.
 * @param   string_ptr UTF-8 or UTF-16 string from which the control chars are to be stripped
 * @return  new string in the same format as the input string
 */
U8BIT* STB_UnicodeStripControlChars(U8BIT *string_ptr)
{
   U8BIT *out_string;
   U8BIT *out_ptr;
   U8BIT *str_ptr;
   U8BIT *end_ptr;
   U16BIT num_bytes;
   U16BIT unicode;

   FUNCTION_START(STB_UnicodeStripControlChars);

   out_string = NULL;

   if (STB_IsUnicodeString(string_ptr))
   {
      /* Allocate the return string */
      num_bytes = STB_GetNumBytesInString(string_ptr);
      if (num_bytes != 0)
      {
         out_string = STB_GetMemory(num_bytes);
         if (out_string != NULL)
         {
            out_ptr = out_string;
            str_ptr = string_ptr;
            end_ptr = string_ptr + num_bytes;

            *out_ptr = *str_ptr;
            out_ptr++;
            str_ptr++;

            if (string_ptr[UNICODE_HEADER_POS] == UTF16_HEADER_VALUE)
            {
               while (str_ptr < end_ptr)
               {
                  unicode = ((*str_ptr) << 8) | *(str_ptr + 1);
                  str_ptr += 2;

                  if ((unicode != 0x008a) && (unicode != 0xe08a) && (unicode != 0x0086) &&
                     (unicode != 0xe086) && (unicode != 0x0087) && (unicode != 0xe087))
                  {
                     *out_ptr = (U8BIT)((unicode >> 8) & 0xff);
                     out_ptr++;
                     *out_ptr = (U8BIT)(unicode & 0xff);
                     out_ptr++;
                  }
               }
            }
            else
            {
               while (str_ptr < end_ptr)
               {
                  unicode = ReadUTF8(&str_ptr);

                  if ((unicode != 0x008a) && (unicode != 0xe08a) && (unicode != 0x0086) &&
                     (unicode != 0xe086) && (unicode != 0x0087) && (unicode != 0xe087))
                  {
                     out_ptr = OutputUTF8(out_ptr, unicode);
                  }
               }
            }
         }
      }
   }

   FUNCTION_FINISH(STB_UnicodeStripControlChars);

   return(out_string);
}


//--------------------------------------------------------------------------------------------------
// local function definitions
//--------------------------------------------------------------------------------------------------


/**
 *

 *
 * @brief   Checks if character code specified is one requiring right-to-left printing. If
 *                it is then returns TRUE.
 *
 * @param   unicode - character code to be checked
 *
 * @return   TRUE if right-to-left character, FALSE otherwise
 *
 */
static BOOLEAN CheckUnicodeCharForReverseDirection(U16BIT unicode)
{
   BOOLEAN retval;

   FUNCTION_START(CheckUnicodeCharForReverseDirection);

   retval = FALSE;

   if (((unicode >= 0x0600) && (unicode <= 0x06ff)) ||
       ((unicode >= 0xfb50) && (unicode <= 0xfdff)) ||
       ((unicode >= 0xfe70) && (unicode <= 0xfeff))
       )
   {
      retval = TRUE;
   }

   FUNCTION_FINISH(CheckUnicodeCharForReverseDirection);
   return(retval);
}

/**
 *

 *
 * @brief   Convert string to unicode UTF-16 if it isn't already
 *
 * @param   addr_string_ptr      Address of the string pointer to check (in/out)
 * @param   new_string           TRUE if string has been reallocated (out)
 * @param   length_ptr           length of string (out)
 * @param   reverse_dir          reverse print direction (out) (only valid if new string)
 * @param   strip_DVB_cntrl_char True if all DVB control chars are to be removed
 *
 * @return   TRUE if there is any error, otherwise FALSE
 *
 */
static void MakeUnicode( U8BIT **addr_string_ptr, BOOLEAN *new_string, U16BIT *length_ptr,
   BOOLEAN *reverse_dir, BOOLEAN strip_DVB_cntrl_char)
{
   FUNCTION_START(MakeUnicode);

   *new_string = FALSE;
   if (*addr_string_ptr[UNICODE_HEADER_POS] != UTF16_HEADER_VALUE)
   {
      *addr_string_ptr = STB_ConvertStringToUnicode(*addr_string_ptr, reverse_dir, length_ptr,
            strip_DVB_cntrl_char, 0);
      if (*addr_string_ptr == NULL)
      {
         *new_string = FALSE;
      }
      else
      {
         *new_string = TRUE;
      }
   }
   else
   {
      *length_ptr = (U16BIT) STB_UnicodeStringLen( *addr_string_ptr );
   }

   FUNCTION_FINISH(MakeUnicode);
}

/*!**************************************************************************
 * @brief   Generates the UTF-8 byte sequence for the given character
 * @param   buffer - buffer to output the UTF-8 byte sequence to
 * @param   char_code - character to be output
 * @return  pointer to the first byte following the output sequence
 ****************************************************************************/
static U8BIT* OutputUTF8(U8BIT *buffer, U32BIT char_code)
{
   U8BIT num_bytes;
   const U8BIT firstByteMark[4] = { 0x00, 0xC0, 0xE0, 0xF0 };

   FUNCTION_START(OutputUTF8);

   if (char_code < 0x80)
   {
      num_bytes = 0;
   }
   else if (char_code < 0x0800)
   {
      num_bytes = 1;
   }
   else if (char_code < 0x00010000)
   {
      num_bytes = 2;
   }
   else if (char_code < 0x00110000)
   {
      num_bytes = 3;
   }
   else
   {
      num_bytes = 2;
      char_code = UNI_REPLACEMENT_CHAR;
   }

   buffer += num_bytes;

   switch (num_bytes) /* Note: everything falls through. */
   {
      case 3:
         *buffer = (U8BIT)((char_code | 0x80) & 0xBF);
         buffer--;
         char_code >>= 6;
      case 2:
         *buffer = (U8BIT)((char_code | 0x80) & 0xBF);
         buffer--;
         char_code >>= 6;
      case 1:
         *buffer = (U8BIT)((char_code | 0x80) & 0xBF);
         buffer--;
         char_code >>= 6;
      case 0:
         *buffer = (U8BIT)(char_code | firstByteMark[num_bytes]);
         break;
   }

   FUNCTION_FINISH(OutputUTF8);

   /* Return pointer to the next byte to be written */
   return(buffer + num_bytes + 1);
}

/*!**************************************************************************
 * @brief   Returns the character code from a UTF-8 byte sequence
 * @param   buffer - address of pointer to the character in the UTF-8 byte sequence,
 *                   this value is updated to point to the next char
 * @return  character code, or INVALID_UNICODE_CHAR
 ****************************************************************************/
static U32BIT ReadUTF8(U8BIT **buffer)
{
   U8BIT byte_val;
   U32BIT char_code;

   FUNCTION_START(ReadUTF8);

   byte_val = (*buffer)[0];
   char_code = INVALID_UNICODE_CHAR;

   if ((byte_val & 0x80) == 0)
   {
      char_code = byte_val;
      *buffer += 1;
   }
   else if ((byte_val & 0xE0) == 0xC0)
   {
      char_code = byte_val & 0x1F;
      char_code <<= 6;
      char_code += (((*buffer)[1]) & 0x3F);
      *buffer += 2;
   }
   else if ((byte_val & 0xF0) == 0xE0)
   {
      char_code = byte_val & 0x0F;
      char_code <<= 6;
      char_code += (((*buffer)[1]) & 0x3F);
      char_code <<= 6;
      char_code += (((*buffer)[2]) & 0x3F);
      *buffer += 3;
   }
   else if ((byte_val & 0xF8) == 0xF0)
   {
      char_code = byte_val & 0x07;
      char_code <<= 6;
      char_code += (((*buffer)[1]) & 0x3F);
      char_code <<= 6;
      char_code += (((*buffer)[2]) & 0x3F);
      char_code <<= 6;
      char_code += (((*buffer)[3]) & 0x3F);
      *buffer += 4;
   }

   FUNCTION_FINISH(ReadUTF8);

   return(char_code);
}

/*!**************************************************************************
 * @brief   Converts the given character to its lower case equivalent
 * @param   char_code - character to be converted
 * @return  Converted character code, or original code if no conversion is required or appropriate
 ****************************************************************************/
static U32BIT CharToLower(U32BIT char_code)
{
   FUNCTION_START(CharToLower);

   if (((char_code >= 'A') && (char_code <= 'Z')) ||
       ((char_code >= UC_LATIN_CAPITAL_LETTER_A_WITH_GRAVE) &&
        (char_code <= UC_LATIN_CAPITAL_LETTER_O_WITH_DIAERESIS)) ||
       ((char_code >= UC_LATIN_CAPITAL_LETTER_O_WITH_STROKE) &&
        (char_code <= UC_LATIN_CAPITAL_LETTER_THORN)))
   {
      char_code += 0x20;
   }
   else if ((char_code >= UC_LATIN_CAPITAL_LETTER_A_WITH_MACRON) &&
            (char_code <= UC_LATIN_CAPITAL_LETTER_Z_WITH_CARON))
   {
      /* Map individual chars through a mapping table */
      char_code = lowercase_chars_0x0100[char_code - UC_LATIN_CAPITAL_LETTER_A_WITH_MACRON];
   }
   else if (char_code == UC_GREEK_CAPITAL_LETTER_ALPHA_WITH_TONOS)
   {
      char_code = UC_GREEK_SMALL_LETTER_ALPHA_WITH_TONOS;
   }
   else if ((char_code >= UC_GREEK_CAPITAL_LETTER_EPSILON_WITH_TONOS) &&
            (char_code <= UC_GREEK_CAPITAL_LETTER_IOTA_WITH_TONOS))
   {
      char_code += 0x25;
   }
   else if (char_code == UC_GREEK_CAPITAL_LETTER_OMICRON_WITH_TONOS)
   {
      char_code = UC_GREEK_SMALL_LETTER_OMICRON_WITH_TONOS;
   }
   else if (char_code == UC_GREEK_CAPITAL_LETTER_UPSILON_WITH_TONOS)
   {
      char_code = UC_GREEK_SMALL_LETTER_UPSILON_WITH_TONOS;
   }
   else if (char_code == UC_GREEK_CAPITAL_LETTER_OMEGA_WITH_TONOS)
   {
      char_code = UC_GREEK_SMALL_LETTER_OMEGA_WITH_TONOS;
   }
   else if ((char_code >= UC_GREEK_CAPITAL_LETTER_ALPHA) &&
            (char_code <= UC_GREEK_CAPITAL_LETTER_UPSILON_WITH_DIALYTIKA))
   {
      char_code += 0x20;
   }
   else if ((char_code >= UC_CYRILLIC_CAPITAL_LETTER_IO) && (char_code <= UC_CYRILLIC_CAPITAL_LETTER_DZHE))
   {
      char_code += 0x50;
   }
   else if ((char_code >= UC_CYRILLIC_CAPITAL_LETTER_A) && (char_code <= UC_CYRILLIC_CAPITAL_LETTER_YA))
   {
      char_code += 0x20;
   }
   else if ((char_code >= UC_LATIN_CAPITAL_LETTER_A_WITH_RING_BELOW) &&
            (char_code <= UC_LATIN_CAPITAL_LETTER_Y_WITH_LOOP))
   {
      char_code = lowercase_chars_0x1E00[char_code - UC_LATIN_CAPITAL_LETTER_A_WITH_RING_BELOW];
   }

   FUNCTION_FINISH(CharToLower);

   return(char_code);
}

//**************************************************************************************************
// End of File
//**************************************************************************************************
