/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Registers and unregisters callback function for processing
 *          EBU teletext from PES data field into the VBI.
 *
 * @file    stbvbi.c
 * @date    22/04/2004
 * @author  Ocean Blue
 */

//---includes for this file----------------------------------------------------
// compiler library header files

// third party header files
#include <string.h>

// Ocean Blue Software header files
#include <techtype.h>
#include <dbgfuncs.h>

#include "stbhwc.h"
#include "stbpes.h"


//---constant definitions for this file----------------------------------------
#define PES_HDR_DATA_LEN_BYTE 0x08

//---local typedef structs for this file---------------------------------------

//---local (static) variable declarations for this file------------------------
//   (internal variables declared static to make them local)
static U32BIT handle;
static BOOLEAN initialised;

//---local function prototypes for this file-----------------------------------
//   (internal functions declared static to make them local)
static void VBIPesCallback(U32BIT handle, U8BIT data_identifier, void *buffer_ptr, U32BIT buffer_size);


//void STTTX_vbi_init(STTTX_InitParams_t* ttx_init_params);
//void STTTX_vbi_insert(U8* pes_data_field, U32 num_bytes);

//---local function definitions------------------------------------------------

/**
 *

 *
 * @brief   Callback function for use by the PES collection task. Checks PES is valid packet
 *                 and passes PES_data_field only to the hardware layer, en300 472 Section 4.3 .
 *                 There is no task for this process, as processing is very limited before passing
 *                 back into the hardware layer, which may choose to implement a task.
 *
 * @param   handle          - unique handle to the PES callback request.
 * @param   data_identifier - ID byte, first byte in PES_data_field.
 * @param   buffer_ptr      - complete PES packet.
 * @param   buffer_size     - number of bytes at buffer_ptr, in packet.
 *

 *
 */
static void VBIPesCallback(U32BIT handle, U8BIT data_identifier, void *buffer_ptr, U32BIT buffer_size)
{
   U16BIT pes_length;
   U16BIT processed_bytes;
   U8BIT pes_hdr_data_len;
   U8BIT segment_data;

   U8BIT *pes_ptr;

   FUNCTION_START(VBIPesCallback);

   USE_UNWANTED_PARAM(handle);
   USE_UNWANTED_PARAM(data_identifier);
   USE_UNWANTED_PARAM(buffer_size);

   pes_ptr = buffer_ptr;

   if ((pes_ptr[0] == 0x00) && (pes_ptr[1] == 0x00) && (pes_ptr[2] == 0x01))
   {
      // Check stream_id
      if (pes_ptr[3] == 0xbd)
      {
         // private_stream_1
         // This value is from this point forward, the size of the entire PES packet
         // is (pes_length + 6); as the bytes before and the length bytes are not included.
         pes_length = (pes_ptr[4] << 8) + pes_ptr[5] + 6;

         // PES_header_data_length - An 8bit field specifying the total number of bytes
         // occupied by the optional fields and any stuffing bytes in this PES packet header.
         pes_hdr_data_len = pes_ptr[PES_HDR_DATA_LEN_BYTE];

         // This should place data ptr at the beginning of the PES_packet_data_byte area
         segment_data = PES_HDR_DATA_LEN_BYTE + pes_hdr_data_len + 1;

         processed_bytes = segment_data;

         if ((pes_ptr[processed_bytes] >= 0x10) && (pes_ptr[processed_bytes] <= 0x1f))
         {
            STB_HWVBIInsert(&pes_ptr[processed_bytes], (pes_length - pes_hdr_data_len));
         }
      }
   }
   FUNCTION_FINISH(VBIPesCallback);
}

//---global function definitions-----------------------------------------------

/**
 *

 *
 * @brief   Initialises VBI insertion, and registers the callback function with PES
 *                 collection task.
 *

 *

 *
 */
void STB_VBIInitialise(void)
{
   FUNCTION_START(STB_VBIInitialise);

   if (!initialised)
   {
      STB_HWInitialiseVBI();

      STB_PesCollectionTaskInitialise();

      handle = STB_RegisterPesCollectionCallback(VBIPesCallback, 0x10, 0x1f);

      initialised = TRUE;
   }

   FUNCTION_FINISH(STB_VBIInitialise);
}

/**
 *

 *
 * @brief   Removes the VBI callback function from the PES collection task.
 *

 *

 *
 */
void STB_VBIRemoveCallbackFunction(void)
{
   FUNCTION_START(STB_VBIRemoveCallbackFunction);

   STB_UnregisterPesCollectionCallback(handle);

   initialised = FALSE;

   FUNCTION_FINISH(STB_VBIRemoveCallbackFunction);
}

//*****************************************************************************
// End of file
//*****************************************************************************

