/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2012 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Video Transformations functions
 *
 * @file    stbvtc.c
 * @date    28/02/2012
 * @author  Ocean Blue
 */

// gives direct COM port access
// #define STB_DEBUG

//---includes for this file----------------------------------------------------
// compiler library header files

// third party header files

// Ocean Blue Software header files

#include <techtype.h>
#ifdef STB_DEBUG
#define DEBUG_PRINTING_ENABLED
#endif
#include <dbgfuncs.h>

#include "stbvtc.h"
#include "stbhwav.h"
#include "stbheap.h"
#include "stbhwos.h"
#include "stbhwosd.h"

#include "vtc.h"

//---constant definitions for this file----------------------------------------
#ifdef STB_VT_PRINT_REQUIRED
   #define STB_VT_PRINT(x) DEBUG_PRINTX_CONDITIONAL(DEBUG_STB_VT) x
#else
   #ifdef STB_DEBUG
      #define STB_VT_PRINT(x) DBGPRINT(x)
   #else
      #define STB_VT_PRINT(x)
   #endif
#endif

//---local typedef structs for this file---------------------------------------

//---local (static) variable declarations for this file------------------------
//   (internal variables declared static to make them local)

static void *context = NULL;
static F_NOTIFY_VIDEO_AR vgr_video_ar = NULL;


//---local function prototypes for this file-----------------------------------
//   (internal functions declared static to make them local)

static void TransformationChangedCallback(void *user_data);
static void VideoChangedCallback(S_STB_AV_VIDEO_INFO *video_info,
   void *user_data);

//---global function definitions-----------------------------------------------

/*!**************************************************************************
 * @brief   Initialise the VTC module
 * @param   None
 * @return  None
 ****************************************************************************/
void STB_VTInitialise(void)
{
   S_VT_OPTIONS options;

   FUNCTION_START(STB_VTInitialise);

   options.mheg_required = FALSE;
   options.afd_required = TRUE;
   options.hbbtv_required = FALSE;

   context = VT_Open(&options);
   if (context != NULL)
   {
      VT_SetVideoChangedCallback(context, TransformationChangedCallback, NULL);
      STB_AVSetVideoCallback(0, VideoChangedCallback, context);
      VT_Enable(context, TRUE);
   }

   FUNCTION_FINISH(STB_VTInitialise);
}

/*!**************************************************************************
 * @brief   Set video alignment preference
 * @param   pref - video alignment
 * @return  None
 ****************************************************************************/
void STB_VTSetVideoAlignmentPref(E_VIDEO_ASPECT_MODE alignment)
{
   FUNCTION_START(STB_VTSetVideoAlignmentPref);
   if (context != NULL)
   {
      VT_SetVideoAlignmentPref(context, alignment);
   }
   else
   {
      STB_VT_PRINT(("STB_VTSetAlignmentMode: NULL context (was the module initialised?)"));
   }
   FUNCTION_FINISH(STB_VTSetVideoAlignmentPref);
}

/**
 * @brief   Sets the callback used to control the format conversion when the aspect mode set by
 *          STB_VTSetVideoAlignmentPref is ASPECT_MODE_CUSTOM
 * @param   callback Function used by the VTC module to determine what format conversion needs to
 *          be applied.
 */
void STB_VTCSetCustomModeCallback(F_VT_CUSTOM_MODE_CALLBACK callback)
{
   FUNCTION_START(STB_VTCSetCustomModeCallback);
   if (context != NULL)
   {
      VT_SetCustomModeCallback(context, callback);
   }
   else
   {
      STB_VT_PRINT(("STB_VTCSetCustomModeCallback: NULL context (was the module initialised?)"));
   }
   FUNCTION_FINISH(STB_VTCSetCustomModeCallback);
}

/*!**************************************************************************
 * @brief   DVBCore application output window for entire video
 * @param   output - output video rectangle. NULL means full-screen, unless
 *                   MHEG says duifferently.
 * @return  None
 ****************************************************************************/
void STB_VTSetVideoOutput(S_RECTANGLE *output)
{
   FUNCTION_START(STB_VTSetVideoOutput);

   if (context != NULL)
   {
   #ifdef DEBUG_PRINTING_ENABLED
      if (output != NULL)
      {
         DBGPRINT("(%d,%d,%d,%d)", output->left, output->top, output->width, output->height);
      }
      else
      {
         DBGPRINT("output is NOT specified");
      }
   #endif
      VT_SetAppScaling(context, output);
   }
   else
   {
      STB_VT_PRINT(("STB_VTSetVideoOutput: NULL context (was the module initialised?)"));
   }

   FUNCTION_FINISH(STB_VTSetVideoOutput);
}

/*!**************************************************************************
 * @brief   Turn on or off MHEG5 scaling calculation
 * @param   enable TRUE - turn on MHEG5 calculations
 * @return  None
 ****************************************************************************/
void STB_VTSetMhegEnable(BOOLEAN enable)
{
   FUNCTION_START(STB_VTSetMhegEnable);
   if (context != NULL)
   {
      VT_SetProfileMheg5(context, enable);
   }
   else
   {
      STB_VT_PRINT(("STB_VTSetMhegEnable: NULL context (was the module initialised?)"));
   }
   FUNCTION_FINISH(STB_VTSetMhegEnable);
}

/*!**************************************************************************
 * @brief   Set Mheg scaling resolution for video
 * @param   width - width resolution
 * @param   height - height resolution
 * @return  None
 ****************************************************************************/
void STB_VTSetMhegScalingResolution(U16BIT width, U16BIT height)
{
   FUNCTION_START(STB_VTSetMhegScalingResolution);

   if (context != NULL)
   {
      DBGPRINT("(%d,%d)", width, height);
      VT_SetMhegScalingResolution(context, width, height);
   }
   else
   {
      STB_VT_PRINT(("STB_VTSetVideoScalingResolution: NULL context (was the module initialised?)"));
   }
   FUNCTION_FINISH(STB_VTSetMhegScalingResolution);
}

/*!**************************************************************************
 * @brief   Set Mheg video scaling
 * @param   scaling - scaling transformation (offset, size)
 * @return  None
 ****************************************************************************/
void STB_VTSetMhegVideoScaling(S_RECTANGLE *scaling)
{
   FUNCTION_START(STB_VTSetVideoScaling);

   if (context != NULL)
   {
      VT_SetMhegScaling(context, scaling);
   }
   else
   {
      STB_VT_PRINT(("STB_VTSetVideoScaling: NULL context (was the module initialised?)"));
   }

   FUNCTION_FINISH(STB_VTSetVideoScaling);
}

/*!**************************************************************************
 * @brief   Get display aspect ratio
 * @param   aspect_ratio - display aspect ratio
 * @return  None
 ****************************************************************************/
E_ASPECT_RATIO STB_VTGetDisplayAspectRatio(void)
{
   E_ASPECT_RATIO aspect_ratio;
   FUNCTION_START(STB_VTSetDisplayAspectRatio);
   if (context != NULL)
   {
      aspect_ratio = VT_GetDisplayAspectRatio(context);
   }
   else
   {
      STB_VT_PRINT(("STB_VTSetDisplayAspectRatio: NULL context (was the module initialised?)"));
      aspect_ratio = ASPECT_UNDEFINED;
   }
   FUNCTION_FINISH(STB_VTSetDisplayAspectRatio);
   return aspect_ratio;
}

/*!**************************************************************************
 * @brief   Set display aspect ratio
 * @param   aspect_ratio - display aspect ratio
 * @return  None
 ****************************************************************************/
void STB_VTSetDisplayAspectRatio(E_ASPECT_RATIO aspect_ratio)
{
   FUNCTION_START(STB_VTSetDisplayAspectRatio);

   if (context != NULL)
   {
      VT_SetDisplayAspectRatio(context, aspect_ratio);
   }
   else
   {
      STB_VT_PRINT(("STB_VTSetDisplayAspectRatio: NULL context (was the module initialised?)"));
   }

   FUNCTION_FINISH(STB_VTSetDisplayAspectRatio);
}

/*!**************************************************************************
 * @brief    Set scene aspect ratio (MHEG-5 specific)
 * @param    aspect_ratio - scene aspect ratio
 * @return  None
 ****************************************************************************/
void STB_VTSetMhegAspectRatio(E_ASPECT_RATIO aspect_ratio)
{
   FUNCTION_START(STB_VTSetMhegAspectRatio);

   if (context != NULL)
   {
      VT_SetMhegAspectRatio(context, aspect_ratio);
   }
   else
   {
      STB_VT_PRINT(("STB_VTSetMhegAspectRatio: NULL context (was the module initialised?)"));
   }

   FUNCTION_FINISH(STB_VTSetMhegAspectRatio);
}

/*!**************************************************************************
 * @brief   Set MHEG5 widescreen alignment
 * @param   mode - widescreen alignment mode
 * @return  None
 ****************************************************************************/
void STB_VTSetMhegVideoAlignment(E_VIDEO_ASPECT_MODE mode)
{
   FUNCTION_START(STB_VTSetMhegVideoAlignment);

   if (context != NULL)
   {
      VT_SetMhegVideoAlignment(context, mode);
   }
   else
   {
      STB_VT_PRINT(("STB_VTSetWidescreenAlignmentMode: NULL context (was the module initialised?)"));
   }

   FUNCTION_FINISH(STB_VTSetMhegVideoAlignment);
}

/*!**************************************************************************
 * @brief   Turn on or off HBBTV scaling calculation
 * @param   enable TRUE - turn on MHEG5 calculations
 * @return  None
 ****************************************************************************/
void STB_VTSetHbbtvEnable(BOOLEAN enable)
{
   FUNCTION_START(STB_VTSetHbbtvEnable);
   if (context != NULL)
   {
      VT_SetProfileHbbtv(context, enable);
   }
   else
   {
      STB_VT_PRINT(("STB_VTSetHbbtvEnable: NULL context (was the module initialised?)"));
   }
   FUNCTION_FINISH(STB_VTSetHbbtvEnable);
}

/*!**************************************************************************
 * @brief   Set video scaling by HBBTV
 * @param   rect output window rectangle
 * @return  None
 ****************************************************************************/
void STB_VTSetHbbtvVideoWindow(S_RECTANGLE *rect)
{
   FUNCTION_START(STB_VTSetHbbtvVideoWindow);
   if (context != NULL)
   {
      VT_SetHbbtvWindow(context, rect);
   }
   else
   {
      STB_VT_PRINT(("STB_VTSetHbbtvVideoWindow: NULL context (was the module initialised?)"));
   }

   FUNCTION_FINISH(STB_VTSetHbbtvVideoWindow);
}

/*!**************************************************************************
 * @brief   Set video preferences change callback
 *
 * The callback function is called when video transformation is changed as a
 * result of a user preference change (only).
 *
 * @param   callback - callback for notification
 * @param   user_data - user data for the callback
 * @return  None
 ****************************************************************************/
void STB_VTSetVideoPrefChangedCallback(void (*callback)(void *),
   void *user_data)
{
   FUNCTION_START(STB_VTSetUserPreferenceChangedCallback);

   if (context != NULL)
   {
      VT_SetUserPreferenceChangedCallback(context, callback, user_data);
   }
   else
   {
      STB_VT_PRINT(("STB_VTSetUserPreferenceChangedCallback: NULL context (was the module initialised?)"));
   }

   FUNCTION_FINISH(STB_VTSetUserPreferenceChangedCallback);
}

/*!**************************************************************************
 * @brief   Set video preferences change callback
 *
 * The callback function is called when video transformation is changed as a
 * result of a user preference change (only).
 *
 * @param   callback - callback for notification
 * @param   user_data - user data for the callback
 * @return  None
 ****************************************************************************/
void STB_VTSetVideoRatioCallback(F_NOTIFY_VIDEO_AR ar)
{
   FUNCTION_START(STB_VTSetVideoRatioCallback);

   vgr_video_ar = ar;

   FUNCTION_FINISH(STB_VTSetVideoRatioCallback);
}

/*!**************************************************************************
 * @brief   Return the current decoder format conversion
 * @return  The current format conversion
 ****************************************************************************/
E_FORMAT_CONVERSION STB_VTGetDecoderFormatConversion(void)
{
   E_FORMAT_CONVERSION format_conversion;
   FUNCTION_START(STB_VTGetDecoderFormatConversion);
   if (context != NULL)
   {
      format_conversion = VT_GetDecoderFormatConversion(context);
   }
   else
   {
      format_conversion = FORMAT_CONVERSION_IGNORE;
      STB_VT_PRINT(("STB_VTGetDecoderFormatConversion: NULL context (was the module initialised?)"));
   }
   FUNCTION_FINISH(STB_VTGetDecoderFormatConversion);
   return format_conversion;
}

/*!**************************************************************************
 * @brief   Return the current video resolution
 * @param   width - video width
 * @param   height - video height
 * @return  None
 ****************************************************************************/
void STB_VTGetVideoResolution(U16BIT *width, U16BIT *height)
{
   FUNCTION_START(STB_VTGetVideoResolution);

   *width = 0;
   *height = 0;

   if (context != NULL)
   {
      VT_GetVideoResolution(context, width, height);
   }
   else
   {
      STB_VT_PRINT(("STB_VTGetVideoResolution: NULL context (was the module initialised?)"));
   }

   FUNCTION_FINISH(STB_VTGetVideoResolution);
}

/*!**************************************************************************
 * @brief   Return the current video aspect ratio
 * @return  E_ASPECT_RATIO aspect ratio
 ****************************************************************************/
E_ASPECT_RATIO STB_VTGetVideoAspectRatio(void)
{
   E_ASPECT_RATIO ar = ASPECT_UNDEFINED;

   FUNCTION_START(STB_VTGetVideoAspectRatio);

   if (context != NULL)
   {
      ar = VT_GetVideoAspectRatio(context);
   }
   else
   {
      STB_VT_PRINT(("STB_VTGetVideoAspectRatio: NULL context (was the module initialised?)"));
   }

   FUNCTION_FINISH(STB_VTGetVideoAspectRatio);

   return ar;
}

//---local function definitions------------------------------------------------

/*!**************************************************************************
 * @brief   Callback which is called when the video transformation changes
 * @param   user_data - user data for the callback
 * @return  None
 ****************************************************************************/
static void TransformationChangedCallback(void *user_data)
{
   S_RECTANGLE input_rect;
   S_RECTANGLE output_rect;
   U16BIT screen_width;
   U16BIT screen_height;
   BOOLEAN osd_scaled;
   E_STB_AV_DECODER_STATUS decoder_status;

   FUNCTION_START(TransformationChangedCallback);
   USE_UNWANTED_PARAM(user_data);

   if (context != NULL)
   {
      /* Apply video transformation (if relevant) */
      decoder_status = VT_GetDecoderStatus(context);
      if (decoder_status != DECODER_STATUS_NONE)
      {
         VT_GetVideoTransformation(context, &input_rect, &output_rect);
         STB_VT_PRINT(("STB_AVApplyVideoTransformation(0, [(%d,%d) %dx%d], [(%d,%d) %dx%d])",
                       input_rect.left, input_rect.top, input_rect.width, input_rect.height,
                       output_rect.left, output_rect.top, output_rect.width, output_rect.height));
         STB_AVApplyVideoTransformation(0, &input_rect, &output_rect);
      }
      #ifdef DEBUG_PRINTING_ENABLED
      else
      {
         VT_GetVideoTransformation(context, &input_rect, &output_rect);
         DBGPRINT("decoder_status is NONE! out=(%d,%d,%d,%d)",
            output_rect.left, output_rect.top, output_rect.width, output_rect.height);
      }
      #endif

      /* Apply OSD transformation */
      STB_OSDGetSize(&screen_width, &screen_height);
      osd_scaled = VT_IsOsdScaled(context);
      STB_OSDResize(osd_scaled, screen_width, screen_height, 0, 0);
   }
   else
   {
      STB_VT_PRINT(("TransformationChangedCallback: NULL context (was the module initialised?)"));
   }

   FUNCTION_FINISH(TransformationChangedCallback);
}

/*!**************************************************************************
 * @brief   Callback which is called when video properties change
 * @param   video_info - new video information
 * @param   context - transformation manager context
 * @return  None
 ****************************************************************************/
static void VideoChangedCallback(S_STB_AV_VIDEO_INFO *video_info,
   void *context)
{
   FUNCTION_START(VideoChangedCallback);

   VT_Enable(context, FALSE);

   if ((video_info->flags & VIDEO_INFO_VIDEO_RESOLUTION) != 0)
   {
      /* New video resolution */
      STB_VT_PRINT(("VideoChangedCallback: Video resolution %d %d",
                    video_info->video_width,
                    video_info->video_height));
      VT_SetVideoResolution(context, video_info->video_width,
         video_info->video_height);
   }

   if ((video_info->flags & VIDEO_INFO_SCREEN_RESOLUTION) != 0)
   {
      /* New screen resolution */
      STB_VT_PRINT(("VideoChangedCallback: Screen resolution %d %d",
                    video_info->screen_width,
                    video_info->screen_height));
      VT_SetScreenResolution(context, video_info->screen_width,
         video_info->screen_height);
   }

   if ((video_info->flags & VIDEO_INFO_VIDEO_ASPECT_RATIO) != 0)
   {
      /* New video aspect ratio */
      STB_VT_PRINT(("VideoChangedCallback: Video aspect ratio %s",
                    video_info->video_aspect_ratio == ASPECT_RATIO_4_3 ? "ASPECT_RATIO_4_3" :
                    video_info->video_aspect_ratio == ASPECT_RATIO_16_9 ? "ASPECT_RATIO_16_9" :
                    "*unknown*"));

      VT_SetVideoAspectRatio(context, video_info->video_aspect_ratio);
      if (vgr_video_ar != NULL)
      {
         vgr_video_ar(video_info->video_aspect_ratio);
      }
   }

   if ((video_info->flags & VIDEO_INFO_DISPLAY_ASPECT_RATIO) != 0)
   {
      /* New video aspect ratio */
      STB_VT_PRINT(("VideoChangedCallback: Display aspect ratio %s",
                    video_info->display_aspect_ratio == ASPECT_RATIO_4_3 ? "ASPECT_RATIO_4_3" :
                    video_info->display_aspect_ratio == ASPECT_RATIO_16_9 ? "ASPECT_RATIO_16_9" :
                    "*unknown*"));

      VT_SetDisplayAspectRatio(context, video_info->display_aspect_ratio);
   }

   if ((video_info->flags & VIDEO_INFO_AFD) != 0)
   {
      /* New AFD value */
      STB_VT_PRINT(("VideoChangedCallback: AFD value %d",
                    video_info->afd));
      VT_SetAfd(context, video_info->afd);
   }

   if ((video_info->flags & VIDEO_INFO_DECODER_STATUS) != 0)
   {
      /* New decoder status */
      STB_VT_PRINT(("VideoChangedCallback: Decoder status %s",
                    video_info->status == DECODER_STATUS_NONE ? "DECODER_STATUS_NONE" :
                    video_info->status == DECODER_STATUS_VIDEO ? "DECODER_STATUS_VIDEO" :
                    video_info->status == DECODER_STATUS_IFRAME ? "DECODER_STATUS_IFRAME" :
                    "*unknown*"));
      VT_SetDecoderStatus(context, video_info->status);
   }

   VT_Enable(context, TRUE);

   FUNCTION_FINISH(VideoChangedCallback);
}

//*****************************************************************************
// End of file
//*****************************************************************************

