/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Blank description
 * @file    vtc.h
 * @date    01/12/2004
 * @author  Ocean Blue
 */
#ifndef VTC_H
#define VTC_H

#include "techtype.h"
#include "stbhwav.h"
#include "vtctype.h"

#ifdef __cplusplus
extern "C" {
#endif

typedef struct s_vt_options
{
   BOOLEAN mheg_required;
   BOOLEAN afd_required;
   BOOLEAN hbbtv_required;
} S_VT_OPTIONS;

typedef void (*F_VT_NOTIFICATION_CALLBACK)(void *userdata);

/*!**************************************************************************
 * @brief    Open video transformation manager
 * @param    options - transformation manager options
 * @return   Pointer to manager context, NULL if cannot be created
 ****************************************************************************/
void* VT_Open(S_VT_OPTIONS *options);

/*!**************************************************************************
 * @brief    Close video transformation manager
 * @param    context - transformation calculator context
 ****************************************************************************/
void VT_Close(void *context);

/*!**************************************************************************
 * @brief    Enable or disable transformation calculations
 * @param    context - transformation calculator context
 * @param    enable - TRUE if calculations are enabled, FALSE otherwise
 ****************************************************************************/
void VT_Enable(void *context, BOOLEAN enable);

/*!**************************************************************************
 * @brief    Set current AFD (active format descriptor value)
 * @param    context - transformation calculator context
 * @param    afd_value - AFD value
 ****************************************************************************/
void VT_SetAfd(void *context, U8BIT afd_value);

/*!**************************************************************************
 * @brief    Set video aspect ratio
 * @param    context - transformation calculator context
 * @param    aspect_ratio - video aspect ratio
 ****************************************************************************/
void VT_SetVideoAspectRatio(void *context, E_ASPECT_RATIO aspect_ratio);

/*!**************************************************************************
 * @brief    Get video aspect ratio
 * @param    context - transformation calculator context
 * @return   aspect_ratio - video aspect ratio
 ****************************************************************************/
E_ASPECT_RATIO VT_GetVideoAspectRatio(void *context);

/*!**************************************************************************
 * @brief    Set MHEG5 scene aspect ratio
 * @param    context - transformation calculator context
 * @param    aspect_ratio - scene aspect ratio
 ****************************************************************************/
void VT_SetMhegAspectRatio(void *context, E_ASPECT_RATIO aspect_ratio);

/*!**************************************************************************
 * @brief    Get display aspect ratio
 * @param    context - transformation calculator context
 * @return   E_ASPECT_RATIO of display
 ****************************************************************************/
E_ASPECT_RATIO VT_GetDisplayAspectRatio(void *context);

/*!**************************************************************************
 * @brief    Set display aspect ratio
 * @param    context - transformation calculator context
 * @param    aspect_ratio - video aspect ratio
 ****************************************************************************/
void VT_SetDisplayAspectRatio(void *context, E_ASPECT_RATIO aspect_ratio);

/*!**************************************************************************
 * @brief    Set Voyager scaling resolution
 * @param    context - transformation calculator context
 * @param    width
 * @param    height
 ****************************************************************************/
void VT_SetMhegScalingResolution(void *context, U16BIT width, U16BIT height);

/*!**************************************************************************
 * @brief    Set MHEG-5 scaling information
 * @param    context - transformation calculator context
 * @param    scaling - scaling and positioning transformation
 * @note     When scaling is NULL, scaling is ignored and the behaviour will
 *           be as if full screen video is mapped to the full screen.
 ****************************************************************************/
void VT_SetMhegScaling(void *context, S_RECTANGLE *scaling);

/*!**************************************************************************
 * @brief    Set application scaling information
 * @param    context - transformation calculator context
 * @param    window - output window (screen CS)
 * @note     When window is NULL, application scaling is turned off
 ****************************************************************************/
void VT_SetAppScaling(void *context, S_RECTANGLE *window);

/*!**************************************************************************
 * @brief    Set video resolution
 * @param    context - transformation calculator context
 * @param    width - video width (in pixels)
 * @param    height - video height (in pixels)
 ****************************************************************************/
void VT_SetVideoResolution(void *context, U16BIT width, U16BIT height);

/*!**************************************************************************
 * @brief    Set screen resolution
 * @param    context - transformation calculator context
 * @param    width - screen width (in pixels)
 * @param    height - screen height (in pixels)
 ****************************************************************************/
void VT_SetScreenResolution(void *context, U16BIT width, U16BIT height);

/*!**************************************************************************
 * @brief    Set profile to apply MHEG5 option
 * @param    context - transformation calculator context
 * @param    enable - TRUE turns MHEG5 option on
 ****************************************************************************/
void VT_SetProfileMheg5(void *context, BOOLEAN enable);

/*!**************************************************************************
 * @brief    Set profile to apply HBBTV option
 * @param    context - transformation calculator context
 * @param    enable - TRUE turns MHEG5 option on
 ****************************************************************************/
void VT_SetProfileHbbtv(void *context, BOOLEAN enable);

/*!**************************************************************************
 * @brief    Set HBBTV output window
 * @param    context - transformation calculator context
 * @param    output - window
 ****************************************************************************/
void VT_SetHbbtvWindow(void *context, S_RECTANGLE *output);

/*!**************************************************************************
 * @brief    Set widescreeen alignment mode for MHEG-5
 * @param    context - transformation calculator context
 * @param    wam - widescreen alignment mode
 ****************************************************************************/
void VT_SetMhegVideoAlignment(void *context, E_VIDEO_ASPECT_MODE wam);

/*!**************************************************************************
 * @brief    Get the current video transfromation rectangles
 * @param    context - transformation calculator context
 * @param    input_rect - input rectangle for transformation
 * @param    output_rect - output rectangle for transformation
 ****************************************************************************/
void VT_GetVideoTransformation(void *context, S_RECTANGLE *input_rect,
   S_RECTANGLE *output_rect);

/*!**************************************************************************
 * @brief    Return WSS (wide-screen signalling) value
 * @param    context - transformation calculator context
 * @param    wss - WSS value
 ****************************************************************************/
void VT_GetWss(void *context, U8BIT *wss);

/*!**************************************************************************
 * @brief    Set video change callback
 *
 * The callback is called whenever the video transformation is changed
 * for any reason.
 *
 * @param    context - transformation calculator context
 * @param    callback - the callback to call
 * @param    user_data - user data to pass to the callback
 ****************************************************************************/
void VT_SetVideoChangedCallback(void *context,
   F_VT_NOTIFICATION_CALLBACK callback,
   void *user_data);

/*!**************************************************************************
 * @brief    Set user preference change callback
 *
 * The callback is called whenever the video transformation changes as
 * a result of a change in user preferences.
 *
 * @param    context - transformation calculator context
 * @param    callback - the callback to call
 * @param    user_data - user data to pass to the callback
 ****************************************************************************/
void VT_SetUserPreferenceChangedCallback(void *context,
   F_VT_NOTIFICATION_CALLBACK callback,
   void *user_data);

/*!**************************************************************************
 * @brief    Return the current decoder format conversion
 * @param    context - transformation calculator context
 * @return   Decoder format conversion
 ****************************************************************************/
E_FORMAT_CONVERSION VT_GetDecoderFormatConversion(void *context);

/*!**************************************************************************
 * @brief    Return the current screen resolution
 * @param    context - transformation calculator context
 * @param    width - screen width
 * @param    height - screen height
 ****************************************************************************/
void VT_GetScreenResolution(void *context, U16BIT *width, U16BIT *height);

/*!**************************************************************************
 * @brief    Return the current video resolution
 * @param    context - transformation calculator context
 * @param    width - video width
 * @param    height - video height
 ****************************************************************************/
void VT_GetVideoResolution(void *context, U16BIT *width, U16BIT *height);

/*!**************************************************************************
 * @brief    Check if osd must be scaled due to MHEG scene aspect ratio
 *
 * @param    context - transformation calculator context
 * @return   TRUE if osd scaling is required, FALSE otherwise
 ****************************************************************************/
BOOLEAN VT_IsOsdScaled(void *context);

/*!**************************************************************************
 * @brief    Set user preference for video aspect ratio
 *
 * @param    context - transformation calculator context
 * @param    alignment - New video alignment preference:
 ****************************************************************************/
void VT_SetVideoAlignmentPref(void *context, E_VIDEO_ASPECT_MODE alignment);

/*!**************************************************************************
 * @brief    Set customer mode callback
 *
 * @param    context - transformation calculator context
 * @param    callback - callback
 ****************************************************************************/
void VT_SetCustomModeCallback(void *context, F_VT_CUSTOM_MODE_CALLBACK callback);

/*!**************************************************************************
 * @brief    Set the decoder status
 * @param    context - transformation calculator context
 * @param    status - New decoder status
 ****************************************************************************/
void VT_SetDecoderStatus(void *context, E_STB_AV_DECODER_STATUS status);

/*!**************************************************************************
 * @brief    Get the decoder status
 * @param    context - transformation calculator context
 * @return   The decoder status
 ****************************************************************************/
E_STB_AV_DECODER_STATUS VT_GetDecoderStatus(void *context);

#ifdef __cplusplus
}
#endif

#endif /*VTC_H*/
