/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Header file - Function prototypes for A/V control
 * @file    stbhwav.h
 * @date    06/02/2001
 */

// pre-processor mechanism so multiple inclusions don't cause compilation error

#ifndef _STBHWAV_H
#define _STBHWAV_H

#include "techtype.h"
#include "stbhwc.h"
#include "osdtype.h"

#ifdef __cplusplus
extern "C" {
#endif

//---Constant and macro definitions for public use-----------------------------

//---Enumerations for public use-----------------------------------------------

typedef enum e_stb_av_audio_mode
{
   AV_AUDIO_STEREO = 0,
   AV_AUDIO_LEFT = 1,
   AV_AUDIO_RIGHT = 2,
   AV_AUDIO_MONO = 3,
   AV_AUDIO_MULTICHANNEL = 4
} E_STB_AV_AUDIO_MODE;

typedef enum e_stb_av_decode_source
{
   AV_DEMUX = 0,
   AV_MEMORY = 1
} E_STB_AV_DECODE_SOURCE;

typedef enum e_stb_av_video_codec
{
   AV_VIDEO_CODEC_AUTO = 0,
   AV_VIDEO_CODEC_MPEG1 = 1,
   AV_VIDEO_CODEC_MPEG2 = 2,
   AV_VIDEO_CODEC_H264 = 3,
   AV_VIDEO_CODEC_H265 = 4,
} E_STB_AV_VIDEO_CODEC;

typedef enum e_stb_av_audio_codec
{
   AV_AUDIO_CODEC_AUTO = 0,
   AV_AUDIO_CODEC_MP2 = 1,
   AV_AUDIO_CODEC_MP3 = 2,
   AV_AUDIO_CODEC_AC3 = 3,
   AV_AUDIO_CODEC_EAC3 = 4,
   AV_AUDIO_CODEC_AAC = 5,
   AV_AUDIO_CODEC_HEAAC = 6,
   AV_AUDIO_CODEC_AAC_ADTS = 7
} E_STB_AV_AUDIO_CODEC;

typedef E_ASPECT_RATIO E_STB_AV_ASPECT_RATIO;

typedef enum e_stb_av_video_format
{
   VIDEO_FORMAT_UNDEFINED = 255,
   VIDEO_FORMAT_AUTO = 0,
   VIDEO_FORMAT_ORIGINAL,
   VIDEO_FORMAT_PALBGH,
   VIDEO_FORMAT_PALDKL,
   VIDEO_FORMAT_PALI,
   VIDEO_FORMAT_PALM,
   VIDEO_FORMAT_PALN,
   VIDEO_FORMAT_NTSC,
   VIDEO_FORMAT_SECAMBGH,
   VIDEO_FORMAT_SECAMDKL,
   VIDEO_FORMAT_576IHD,
   VIDEO_FORMAT_576PHD,
   VIDEO_FORMAT_720HD,     /* HD Format, 1280x720  progressive, 50 fps */
   VIDEO_FORMAT_720P50HD = VIDEO_FORMAT_720HD,
   VIDEO_FORMAT_720P60HD,
   VIDEO_FORMAT_1080IHD,   /* HD Format, 1920x1080 interlaced, 25 fps */
   VIDEO_FORMAT_1080P25HD, /* HD Format, 1920x1080 progressive, 25 fps */
   VIDEO_FORMAT_1080P30HD, /* HD Format, 1920x1080 progressive, 30 fps */
   VIDEO_FORMAT_1080I50HD, /* HD Format, 1920x1080 interlaced,  50 fps */
   VIDEO_FORMAT_1080P50HD, /* HD Format, 1920x1080 progressive, 50 fps */
   VIDEO_FORMAT_1080P60HD, /* HD Format, 1920x1080 progressive, 60 fps */
   VIDEO_FORMAT_2160P24UHD,
   VIDEO_FORMAT_2160P25UHD,
   VIDEO_FORMAT_2160P30UHD,
   VIDEO_FORMAT_2160P50UHD,
   VIDEO_FORMAT_2160P60UHD,
} E_STB_AV_VIDEO_FORMAT;

typedef enum e_stb_av_outputs
{
   AV_OUTPUT_TV_SCART = 0,
   AV_OUTPUT_VCR_SCART = 1,
   AV_OUTPUT_AUX_SCART = 2,
   AV_OUTPUT_HDMI = 3
} E_STB_AV_OUTPUTS;

typedef enum e_stb_av_sources
{
   AV_SOURCE_ENCODER_SVIDEO = 0,
   AV_SOURCE_ENCODER_RGB = 1,
   AV_SOURCE_ENCODER_COMPOSITE = 2,
   AV_SOURCE_VCR_SVIDEO = 3,
   AV_SOURCE_VCR_RGB = 4,
   AV_SOURCE_VCR_COMPOSITE = 5,
   AV_SOURCE_AUX_SVIDEO = 6,
   AV_SOURCE_AUX_RGB = 7,
   AV_SOURCE_AUX_COMPOSITE = 8,
   AV_SOURCE_DVD_SVIDEO = 9,
   AV_SOURCE_DVD_RGB = 10,
   AV_SOURCE_DVD_COMPOSITE = 11,
   AV_SOURCE_ANALOG_TUNER = 12,
   AV_SOURCE_ANALOG_TUNER_WITH_OSD = 13,
   AV_SOURCE_HDMI = 14,
   AV_SOURCE_CVBS = 15,
   AV_SOURCE_TUNER = 16,
   AV_SOURCE_NONE = 255
} E_STB_AV_SOURCES;

typedef enum e_stb_av_srm_reply
{
   SRM_OK = 0,
   SRM_BUSY = 1,
   SRM_NOT_REQUIRED = 2
} E_STB_AV_SRM_REPLY;

typedef enum
{
   VIDEO_INFO_VIDEO_RESOLUTION = 0x01,
   VIDEO_INFO_SCREEN_RESOLUTION = 0x02,
   VIDEO_INFO_VIDEO_ASPECT_RATIO = 0x04,
   VIDEO_INFO_DISPLAY_ASPECT_RATIO = 0x08,
   VIDEO_INFO_ASPECT_MODE = 0x10,
   VIDEO_INFO_AFD = 0x20,
   VIDEO_INFO_DECODER_STATUS = 0x40
} E_STB_AV_VIDEO_INFO_TYPE;

typedef enum
{
   DECODER_STATUS_NONE,
   DECODER_STATUS_VIDEO,
   DECODER_STATUS_IFRAME
} E_STB_AV_DECODER_STATUS;

typedef enum
{
   DIGITAL_AUDIO_PCM,
   DIGITAL_AUDIO_COMPRESSED,
   DIGITAL_AUDIO_AUTO
} E_STB_DIGITAL_AUDIO_TYPE;

typedef enum e_stb_av_audio_route
{
   SPEAKER_TV_N_AD_HEADPHONE_TV_N_AD,
   SPEAKER_TV_ONLY_HEADPHONE_AD_ONLY,
   SPEAKER_TV_N_AD_HEADPHONE_AD_ONLY,
   SPEAKER_TV_ONLY_HEADPHONE_TV_N_AD,
   AUTO_ROUTE_UNKNOWN,
} E_STB_AV_AUDIO_ROUTE;


//---Global type defs for public use-------------------------------------------

typedef struct
{
   BOOLEAN macrovision_set;
   U8BIT macrovision;

   BOOLEAN aps_set;
   U8BIT aps;

   BOOLEAN cgms_a_set;
   U8BIT cgms_a;

   BOOLEAN ict_set;
   U8BIT ict;

   BOOLEAN hdcp_set;
   BOOLEAN hdcp;

   BOOLEAN scms_set;
   U8BIT scms;

   BOOLEAN dot_set;
   U8BIT dot;
} S_STB_AV_COPY_PROTECTION;

typedef struct
{
   E_STB_AV_VIDEO_INFO_TYPE flags;
   U16BIT video_width;
   U16BIT video_height;
   U16BIT screen_width;
   U16BIT screen_height;
   E_STB_AV_ASPECT_RATIO video_aspect_ratio;
   E_STB_AV_ASPECT_RATIO display_aspect_ratio;
   U8BIT afd;
   E_STB_AV_DECODER_STATUS status;
} S_STB_AV_VIDEO_INFO;


//---Global Function prototypes for public use---------------------------------

/**
 * @brief   Initialises the AV components
 * @param   audio_paths number of audio decoders
 * @param   video_paths number of video decoders
 */
void STB_AVInitialise(U8BIT audio_paths, U8BIT video_paths);

/**
 * @brief   Blanks or unblanks the video display
 * @param   path video path
 * @param   blank TRUE to blank, FALSE to unblank
 */
void STB_AVBlankVideo(U8BIT path, BOOLEAN blank);

/**
 * @brief   Sets the source of the input to the given video decoder path
 * @param   path video path to configure
 * @param   source the source device to use
 * @param   param parameter set with the source, typically the tuner or demux number
 */
void STB_AVSetVideoSource(U8BIT path, E_STB_AV_DECODE_SOURCE source, U32BIT param);

/**
 * @brief   Sets the video codec to be used when decoding video with the given video decoder path
 * @param   path video path
 * @param   codec codec to be used
 * @return  TRUE if the codec is supported and is set correctly, FALSE otherwise
 */
BOOLEAN STB_AVSetVideoCodec(U8BIT path, E_STB_AV_VIDEO_CODEC codec);

/**
 * @brief   Returns the video codec previously set for the given video path.
 *          This function is currently unused within DVBCore.
 * @param   path video path
 * @return  codec previously set
 */
E_STB_AV_VIDEO_CODEC STB_AVGetVideoCodec(U8BIT path);

/**
 * @brief   Starts video decoding on the given video path
 * @param   path video decoder path to be started
 */
void STB_AVStartVideoDecoding(U8BIT path);

/**
 * @brief   Pauses video decoding on the given video path.
 *          The video should not be blanked.
 * @param   path video decoder path to be paused
 */
void STB_AVPauseVideoDecoding(U8BIT path);

/**
 * @brief   Resumes video decoding on the given video path that has previously been paused
 *          The video should not be unblanked.
 * @param   path video decoder path to be resumed
 */
void STB_AVResumeVideoDecoding(U8BIT path);

/**
 * @brief   Stops video decoding on the given video path.
 *          The video is not expected to be blanked.
 * @param   path video decoder path to be stopped
 */
void STB_AVStopVideoDecoding(U8BIT decoder);

/**
 * @brief   Sets the source of the input for the main audio on the given audio decoder path
 * @param   path audio path to configure
 * @param   source the source device to use
 * @param   param parameter set with the source, typically the tuner or demux number
 */
void STB_AVSetAudioSource(U8BIT path, E_STB_AV_DECODE_SOURCE source, U32BIT param);

/**
 * @brief   Sets the audio codec to be used when decoding audio with the given audio decoder path
 * @param   path audio path
 * @param   codec codec to be used
 * @return  TRUE if the codec is supported and is set correctly, FALSE otherwise
 */
BOOLEAN STB_AVSetAudioCodec(U8BIT path, E_STB_AV_AUDIO_CODEC codec);

/**
 * @brief   Returns the audio codec previously set for the given audio path.
 *          This function is currently unused within DVBCore.
 * @param   path audio path
 * @return  codec previously set
 */
E_STB_AV_AUDIO_CODEC STB_AVGetAudioCodec(U8BIT path);

/**
 * @brief   Configures the main audio channel mode (stereo/left/right) in the case where
 *          dual-mono audio is used, such that only the audio from one channel is heard.
 * @param   path audio path to be configured
 * @param   mode audio mode to use
 */
void STB_AVChangeAudioMode(U8BIT path, E_STB_AV_AUDIO_MODE mode);

/**
 * @brief   Starts audio decoding on the given audio path
 * @param   path audio decoder path to be started
 */
void STB_AVStartAudioDecoding(U8BIT decoder);

/**
 * @brief   Stops audio decoding on the given audio path.
 * @param   path audio decoder path to be stopped
 */
void STB_AVStopAudioDecoding(U8BIT decoder);

/**
 * @brief   Sets the volume of the main audio output
 * @param   path audio path to be configured
 * @param   vol audio volume (0-100%)
 */
void STB_AVSetAudioVolume(U8BIT path, U8BIT vol);

/**
 * @brief   Returns the current volume of the main audio output
 * @param   path audio path
 * @return  audio volume (0-100%)
 */
U8BIT STB_AVGetAudioVolume(U8BIT path);

/**
 * @brief   Mute or unmute the audio output on the given audio decoder path
 * @param   path audio path to be configured
 * @param   mute TRUE to mute, FALSE to unmute
 */
void STB_AVSetAudioMute(U8BIT path, BOOLEAN mute);

/**
 * @brief   Returns the current mute setting of the audio output on the given path
 * @param   path audio path
 * @return  TRUE if audio is muted, FALSE otherwise
 */
BOOLEAN STB_AVGetAudioMute(U8BIT path);

/**
 * @brief   Sets the source of the input for the audio description audio on the
 *          given audio decoder path
 * @param   path audio path to configure
 * @param   source the source device to use
 * @param   param parameter set with the source, typically the tuner or demux number
 */
void STB_AVSetADSource(U8BIT path, E_STB_AV_DECODE_SOURCE source, U32BIT param);

/**
 * @brief   Sets the codec to be used for audio description when decoding audio
 *          with the given audio decoder path
 * @param   path audio path
 * @param   codec codec to be used
 * @return  TRUE if the codec is supported and is set correctly, FALSE otherwise
 */
BOOLEAN STB_AVSetADCodec(U8BIT path, E_STB_AV_AUDIO_CODEC codec);

/**
 * @brief   Returns the codec previously set for audio description on the given audio path.
 *          This function is currently unused within DVBCore.
 * @param   path audio path
 * @return  codec previously set
 */
E_STB_AV_AUDIO_CODEC STB_AVGetADCodec(U8BIT path);

/**
 * @brief   Configures the audio description channel mode (stereo/left/right) in the case where
 *          dual-mono audio is used, such that only the audio from one channel is heard.
 * @param   path audio path to be configured
 * @param   mode audio mode to use
 */
void STB_AVChangeADMode(U8BIT path, E_STB_AV_AUDIO_MODE mode);

/**
 * @brief   Starts decoding audio description on the given audio path
 * @param   path audio decoder path to be started
 */
void STB_AVStartADDecoding(U8BIT decoder);

/**
 * @brief   Stops decoding audio description on the given audio path.
 * @param   path audio decoder path to be stopped
 */
void STB_AVStopADDecoding(U8BIT decoder);

/**
 * @brief   Sets the volume of the audio description output
 * @param   path audio path to be configured
 * @param   vol audio volume (0-100%)
 */
void STB_AVSetADVolume(U8BIT path, U8BIT vol);

/**
 * @brief   Returns the current volume of the audio description output
 * @param   path audio path
 * @return  audio volume (0-100%)
 */
U8BIT STB_AVGetADVolume(U8BIT path);

/**
 * @brief   Returns the current 33-bit System Time Clock from the PCR PES.
 *          On some systems, this information may need to be obtained from the associated demux,
 *          which will be contained in the 'param' value when STB_AVSetVideoSource is called.
 * @param   path video path
 * @param   stc an array in which the STC will be returned, ordered such that
 *                stc[0] contains the MS bit (33) of the STC value and stc[4]
 *                contains the LS bits (0-7).
 */
void STB_AVGetSTC(U8BIT path, U8BIT stc[5]);

/**
 * @brief   Sets the aspect ratio and signal format for the connected television
 * @param   path The video path to be set
 * @param   ratio The aspect ratio of the tv
 * @param   format The signal format of the tv
 */
void STB_AVSetTVType(U8BIT path, E_STB_AV_ASPECT_RATIO ratio, E_STB_AV_VIDEO_FORMAT format);

/**
 * @brief   Returns the current size of the screen in pixels
 * @param   path 0
 * @param   width returned width
 * @param   height returned height
 */
void STB_AVGetScreenSize(U8BIT path, U16BIT *width, U16BIT *height);

/**
 * @brief   Routes a specified AV source to a specified AV output
 * @param   output The output to be connected
 * @param   source The source signal to be connected
 * @param   param parameter associated with the source
 */
void STB_AVSetAVOutputSource(E_STB_AV_OUTPUTS output, E_STB_AV_SOURCES source, U32BIT param);

/**
 * @brief   Turns on/off all AV outputs (e.g. for standby mode)
 * @param   av_on TRUE to turn on, FALSE to turn off
 */
void STB_AVSetAVOutput(BOOLEAN av_on);

/**
 * @brief   Sets the standby state of the HDMI output
 * @param   standby TRUE to put the HDMI in standby, FALSE to come out of standby
 */
void STB_AVSetHDMIStandby(BOOLEAN standby);

/**
 * @brief   Returns the resolutions supported by the HDMI
 * @param   modes array of supported modes
 * @return  number of supported modes
 */
U8BIT STB_AVGetHDMISupportedModes(E_STB_AV_VIDEO_FORMAT **modes);

/**
 * @brief   Returns the native resolution, i.e. the resolution that is
 *          set with a call to STB_AVSetTVType(path, ratio, VIDEO_FORMAT_AUTO);
 * @param   width pointer to the variable where to store the width
 * @param   height pointer to the variable where to store the height
 */
void STB_AVGetHDMINativeResolution(U16BIT *width, U16BIT *height);

/**
 * @brief   Enables AV output to HDMI
 */
void STB_AVEnableHDMIDecoding(void);

/**
 * @brief   Disables AV output to HDMI
 */
void STB_AVDisableHDMIDecoding(void);

/**
 * @brief   Returns whether HDCP has authenticated
 * @return  TRUE if authenticated, FALSE otherwise
 */
BOOLEAN STB_AVIsHDCPAuthenticated(void);

/**
 * @brief   Sets the codec to be used when decoding the next i-frame from memory
 * @param   path video path
 * @param   codec codec to be used
 * @return  TRUE if the codec is supported and is set correctly, FALSE otherwise
 */
BOOLEAN STB_AVSetIFrameCodec(U8BIT path, E_STB_AV_VIDEO_CODEC codec);

/**
 * @brief   Provides the video data for an i-frame for subsequent decode and display
 * @param   path video decode path to be used
 * @param   data i-frame data to be saved for decoding
 * @param   size size of the data in bytes
 */
void STB_AVLoadIFrame(U8BIT path, U8BIT *data, U32BIT size);

/**
 * @brief   Decodes and displays the previously loaded i-frame data
 * @param   path video path to use
 */
void STB_AVShowIFrame(U8BIT path);

/**
 * @brief   Hides the i-frame currently being displayed
 * @param   path video path to use
 */
void STB_AVHideIFrame(U8BIT path);

/**
 * @brief   Plays back a previously loaded audio sample
 * @param   path the audio path to use for playback
 * @param   loop_count the number of times to play the sample, 0=forever
 * @return  HW_OK on success, error code otherwise
 */
E_HW_STATUS STB_AVPlayAudioSample(U8BIT path, U32BIT loop_count);

/**
 * @brief   Loads an audio sample for subsequent playback
 * @param   path the decoder path to use for playback
 * @param   data the audio sample data to be loaded
 * @param   size the size of the audio sample in bytes
 * @return  HW_OK on success, error code otherwise
 */
E_HW_STATUS STB_AVLoadAudioSample(U8BIT path, U8BIT *data, U32BIT size);

/**
 * @brief   Pauses playback of an audio sample
 * @param   path Audio path on which to pause
 * @return  HW_OK on success, error code otherwise
 */
E_HW_STATUS STB_AVPauseAudioSample(U8BIT path);

/**
 * @brief   Resumes playback of an audio sample
 * @param   path Audio path on which to resume
 * @return  HW_OK on success, error code otherwise
 */
E_HW_STATUS STB_AVResumeAudioSample(U8BIT path);

/**
 * @brief   Stops playback of an audio sample
 * @param   path Audio path on which to stop
 */
void STB_AVStopAudioSample(U8BIT path);

/**
 * @brief   Sets the SPDIF output mode, PCM or compressed audio
 * @param   path decoder path
 * @param   spdif_type PCM or compressed
 */
void STB_AVSetSpdifMode(U8BIT path, E_STB_DIGITAL_AUDIO_TYPE audio_type);

/**
 * @brief   Sets the HDMI audio output mode, PCM or compressed
 * @param   path decoder path
 * @param   audio_type PCM or compressed
 */
void STB_AVSetHDMIAudioMode(U8BIT path, E_STB_DIGITAL_AUDIO_TYPE audio_type);

/**
 * @brief   Sets the audio delay on the given path
 * @param   path decoder path
 * @param   millisecond audio delay to be applied
 */
void STB_AVSetAudioDelay(U8BIT path, U16BIT millisecond);

/**
 * @brief   Register callback for updated video information
 * @param   path decoder path
 * @param   callback the callback to call when video information is updated
 * @param   user_data user data to pass to the callback
 */
void STB_AVSetVideoCallback(U8BIT path, void (*callback)(S_STB_AV_VIDEO_INFO *, void *), void *user_data);

/**
 * @brief   Apply video transformation
 * @param   path decoder path
 * @param   input input video rectangle
 * @param   output output video rectangle
 */
void STB_AVApplyVideoTransformation(U8BIT path, S_RECTANGLE *input, S_RECTANGLE *output);

/**
 * @brief   Returns minimum video play speed as a percentage.
 * @param   video_decoder video decoder path
 * @return  Minimum play speed.
 */
S16BIT STB_AVGetMinPlaySpeed(U8BIT video_decoder);

/**
 * @brief   Returns maximum video play speed as a percentage.
 * @param   video_decoder video decoder path
 * @return  Maximum play speed.
 */
S16BIT STB_AVGetMaxPlaySpeed(U8BIT video_decoder);

/**
 * @brief   Returns the next valid speed that is +/- inc above or below the
 *          given speed. Slow motion speeds (>-100% and < 100%) can be included.
 * @param   path Decode path
 * @param   speed Percentage speed above/below which the new speed is calculated
 * @param   inc number of speeds above that specified to return
 * @param   include_slow_speeds selects whether speeds >-100% and <100% are included
 * @return  Speed as a percentage
 */
S16BIT STB_AVGetNextPlaySpeed(U8BIT video_decoder, S16BIT speed, S16BIT inc,
   BOOLEAN include_slow_speeds);

/**
 * @brief Apply the specified copy protection. This function is used for CI+
 * @param copy_protection
 */
void STB_AVSetCopyProtection(S_STB_AV_COPY_PROTECTION *copy_protection);

/**
 * @brief   Sets the output channel of the RF Modulator
 * @param   chan the UHF channel number
 */
void STB_AVSetUhfModulatorChannel(U8BIT chan);

/**
 * @brief   Gets the current RF modulator channel
 * @return  The RF frequency as a UHF channel number
 */
U8BIT STB_AVGetUhfModulatorChannel(void);

/**
 * @brief   Returns the frame rate of the video being decoded
 * @param   path video path
 * @return  video frame rate in frame per seconds
 */
U8BIT STB_AVGetVideoFrameRate(U8BIT path);

/**
 * @brief   Apply System Renewability Message (SRM) to HDCP function
 * @param   path output path
 * @param   data SRM data
 * @param   len length of SRM data in bytes
 * @return  SRM_OK, SRM_BUSY or SRM_NOT_REQUIRED
 */
E_STB_AV_SRM_REPLY STB_AVApplySRM(U8BIT path, U8BIT *data, U32BIT len);

void STB_AVCECOneTouchPlay(void);
E_STB_AV_VIDEO_FORMAT STB_AVGetHDMINativeFormat(void);
void STB_AVPlayMp3(U8BIT path, U8BIT *buffer, U32BIT buffer_size);
void STB_AVStopMp3(U8BIT path);

void STB_AVStartDecoding(U8BIT path);
void STB_AVSwitchLivePath(U8BIT decoder, U8BIT demux);

#ifdef __cplusplus
}
#endif

#endif //  _STBHWAV_H
