package org.dtvkit.inputsource;

import android.content.ComponentName;
import android.media.tv.TvContentRating;
import android.media.tv.TvContract;
import android.media.tv.TvContentRating;
import android.net.Uri;
import android.util.Log;

import org.dtvkit.companionlibrary.EpgSyncJobService;
import org.dtvkit.companionlibrary.model.Channel;
import org.dtvkit.companionlibrary.model.EventPeriod;
import org.dtvkit.companionlibrary.model.InternalProviderData;
import org.dtvkit.companionlibrary.model.Program;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

public class DtvkitEpgSync extends EpgSyncJobService
{
   private static final String TAG = "EpgSyncJobService";
   private static final boolean DEBUG = false;

   @Override
   public List<Channel> getChannels()
   {
      List<Channel> channels = new ArrayList<>();

      if (DEBUG) Log.i(TAG, "Get channels for epg sync");

      try
      {
         JSONObject obj =
            DtvkitGlueClient.getInstance().request("Dvb.getListOfServices", new JSONArray());

         if (DEBUG) Log.i(TAG, obj.toString());

         JSONArray services = obj.getJSONArray("data");
         String inputId = TvContract.buildInputId(new ComponentName(getApplicationContext(), DtvkitTvInput.class));
         if (DEBUG) Log.i(TAG, String.format("inputId: %s", inputId));

         for (int i = 0; i < services.length(); i++)
         {
            JSONObject service = services.getJSONObject(i);
            boolean selectable = service.getBoolean("selectable");
            boolean hidden = service.getBoolean("hidden");

            // EN 300 478, Table 87: Service type coding
            int serviceTypeCode = service.getInt("servicetype");
            String serviceType;
            switch (serviceTypeCode) {
                case 0x01:
                    serviceType = TvContract.Channels.SERVICE_TYPE_AUDIO_VIDEO;
                    break;
                case 0x02:
                    serviceType = TvContract.Channels.SERVICE_TYPE_AUDIO;
                    break;
                default:
                    serviceType = TvContract.Channels.SERVICE_TYPE_OTHER;
                    break;
            }
            String signalType = service.getString("type");
            String type;
            if (signalType == "dvbc")
                type = TvContract.Channels.TYPE_DVB_C;
            else if (signalType == "dvbc2")
                type = TvContract.Channels.TYPE_DVB_C2;
            else if (signalType == "dvbt")
                type = TvContract.Channels.TYPE_DVB_T;
            else if (signalType == "dvbt2")
                type = TvContract.Channels.TYPE_DVB_T2;
            else if (signalType == "dvbs")
                type = TvContract.Channels.TYPE_DVB_S;
            else if (signalType == "dvbs2")
                type = TvContract.Channels.TYPE_DVB_S2;
            else
                type = TvContract.Channels.TYPE_OTHER;

            String uri = service.getString("uri");
            InternalProviderData data = new InternalProviderData();
            data.put("dvbUri", uri);
            JSONArray groups = service.getJSONArray("groups");
            data.put("groups", groups);

            channels.add(new Channel.Builder()
               .setDisplayName(service.getString("name"))
               .setDisplayNumber(String.format(Locale.ENGLISH, "%d", service.getInt("lcn")))
               .setInputId(inputId)
               .setServiceType(serviceType)
               .setType(type)
               .setOriginalNetworkId(Integer.parseInt(uri.substring(6, 10), 16))
               .setTransportStreamId(Integer.parseInt(uri.substring(11, 15), 16))
               .setServiceId(Integer.parseInt(uri.substring(16, 20), 16))
               .setBrowsable(selectable)
               .setInternalProviderData(data)
               .build());
         }

      }
      catch (Exception e)
      {
         Log.e(TAG, e.getMessage());
      }

      return channels;
   }

   @Override
   public List<Program> getProgramsForChannel(Uri channelUri, Channel channel, long startMs,
      long endMs)
   {
      List<Program> programs = new ArrayList<>();

      try
      {
         String dvbUri = String.format("dvb://%04x.%04x.%04x", channel.getOriginalNetworkId(),
            channel.getTransportStreamId(), channel.getServiceId());

         if (DEBUG) Log.i(TAG, String.format("Get channel programs for epg sync. Uri %s, startMs %d, endMs %d",
            dvbUri, startMs, endMs));

         JSONArray args = new JSONArray();
         args.put(dvbUri); // uri
         args.put(startMs / 1000);
         args.put(endMs / 1000);
         JSONArray events = DtvkitGlueClient.getInstance().request("Dvb.getListOfEvents", args)
            .getJSONArray("data");

         for (int i = 0; i < events.length(); i++)
         {
            JSONObject event = events.getJSONObject(i);
            String uri = event.getString("uri");

            InternalProviderData data = new InternalProviderData();
            data.put("dvbUri", uri);

            long minimumAge = event.getLong("rating");
            minimumAge += 3; // DVB parental rating to the minimum age

            TvContentRating[] contentRatings = new TvContentRating[1];
            if (minimumAge > 3) {
                contentRatings[0] = TvContentRating.createRating(
                    "com.android.tv",
                    "DVB",
                    "DVB_" + String.valueOf(minimumAge)
                );
            }                
            else
                contentRatings[0] = TvContentRating.UNRATED;

            String episodeTitle = null;
            try {
               episodeTitle = event.getString("episode_title");
            }
            catch (JSONException e) {
            }
                
            int episodeNumber = 0;
            try {
               episodeNumber = event.getInt("episode_number");
            }
            catch (JSONException e) {
            }
                
            Program.Builder builder = new Program.Builder();
            if (null != builder) {
                builder.setChannelId(channel.getId());
                builder.setTitle(event.getString("name"));
                builder.setStartTimeUtcMillis(event.getLong("startutc") * 1000);
                builder.setEndTimeUtcMillis(event.getLong("endutc") * 1000);
                builder.setDescription(event.getString("description"));
                builder.setLongDescription(event.getString("long_description"));
                builder.setCanonicalGenres(getGenres(event.getString("genre")));
                builder.setContentRatings(contentRatings);
                builder.setInternalProviderData(data);
                // series info.                    
                if ((null != episodeTitle) && (0 != episodeNumber)) {
                    builder.setEpisodeTitle(episodeTitle);
                    builder.setEpisodeNumber(episodeNumber);
                }
                programs.add(builder.build());
            }
         }
      }
      catch (Exception e)
      {
         Log.e(TAG, e.getMessage());
      }

      return programs;
   }

   public static List<Program> getPresentFollowingProgramsForChannel(Uri channelUri, Channel channel) {
       List<Program> programs = new ArrayList<>();

       try {
           String dvbUri = String.format("dvb://%04x.%04x.%04x", channel.getOriginalNetworkId(), channel.getTransportStreamId(), channel.getServiceId());

           if (DEBUG) Log.i(TAG, "Get channel now next programs for epg sync. Uri " + dvbUri);

           JSONArray args = new JSONArray();
           args.put(dvbUri); // uri
           JSONObject events = DtvkitGlueClient.getInstance().request("Dvb.getNowNextEvents", args).getJSONObject("data");

           JSONObject now = events.optJSONObject("now");
           if (now != null) {
               InternalProviderData data = new InternalProviderData();
               data.put("dvbUri", now.getString("uri"));

               long minimumAge = now.getLong("rating");
               minimumAge += 3; // DVB parental rating to the minimum age

               TvContentRating[] contentRatings = new TvContentRating[1];
               if (minimumAge > 3) {
                   contentRatings[0] = TvContentRating.createRating(
                       "com.android.tv",
                       "DVB",
                       "DVB_" + String.valueOf(minimumAge)
                   );
               }                
               else
                   contentRatings[0] = TvContentRating.UNRATED;

               String episodeTitle = null;
               try {
                  episodeTitle = now.getString("episode_title");
               }
               catch (JSONException e) {
               }
                
               int episodeNumber = 0;
               try {
                  episodeNumber = now.getInt("episode_number");
               }
               catch (JSONException e) {
               }
                
               Program.Builder builder = new Program.Builder();
               if (null != builder) {
                   builder.setChannelId(channel.getId());
                   builder.setTitle(now.getString("name"));
                   builder.setStartTimeUtcMillis(now.getLong("startutc") * 1000);
                   builder.setEndTimeUtcMillis(now.getLong("endutc") * 1000);
                   builder.setDescription(now.getString("description"));
                   builder.setLongDescription(now.getString("long_description"));
                   builder.setCanonicalGenres(getGenres(now.getString("genre")));
                   builder.setContentRatings(contentRatings);
                   builder.setInternalProviderData(data);
                   // series info.                    
                   if ((null != episodeTitle) && (0 != episodeNumber)) {
                       builder.setEpisodeTitle(episodeTitle);
                       builder.setEpisodeNumber(episodeNumber);
                   }
                   programs.add(builder.build());
                }
           }

           JSONObject next = events.optJSONObject("next");
           if (next != null) {
               InternalProviderData data = new InternalProviderData();
               data.put("dvbUri", next.getString("uri"));

               long minimumAge = next.getLong("rating");
               minimumAge += 3; // DVB parental rating to the minimum age

               TvContentRating[] contentRatings = new TvContentRating[1];
               if (minimumAge > 3) {
                   contentRatings[0] = TvContentRating.createRating(
                       "com.android.tv",
                       "DVB",
                       "DVB_" + String.valueOf(minimumAge)
                   );
               }                
               else
                   contentRatings[0] = TvContentRating.UNRATED;

               String episodeTitle = null;
               try {
                  episodeTitle = next.getString("episode_title");
               }
               catch (JSONException e) {
               }
                
               int episodeNumber = 0;
               try {
                  episodeNumber = next.getInt("episode_number");
               }
               catch (JSONException e) {
               }
                
               Program.Builder builder = new Program.Builder();
               if (null != builder) {
                   builder.setChannelId(channel.getId());
                   builder.setTitle(next.getString("name"));
                   builder.setStartTimeUtcMillis(next.getLong("startutc") * 1000);
                   builder.setEndTimeUtcMillis(next.getLong("endutc") * 1000);
                   builder.setDescription(next.getString("description"));
                   builder.setLongDescription(next.getString("long_description"));
                   builder.setCanonicalGenres(getGenres(next.getString("genre")));
                   builder.setContentRatings(contentRatings);
                   builder.setInternalProviderData(data);
                   // series info.                    
                   if ((null != episodeTitle) && (0 != episodeNumber)) {
                       builder.setEpisodeTitle(episodeTitle);
                       builder.setEpisodeNumber(episodeNumber);
                   }
                   programs.add(builder.build());
               }
           }
       } catch (Exception e) {
           Log.e(TAG, e.getMessage());
       }

       return programs;
   }

   public List<Program> getNowNextProgramsForChannel(Uri channelUri, Channel channel)
   {
      return getPresentFollowingProgramsForChannel(channelUri, channel);
   }

   @Override
   public List<EventPeriod> getListOfUpdatedEventPeriods()
   {
      List<EventPeriod> eventPeriods = new ArrayList<>();

      if (DEBUG) Log.i(TAG, "getListOfUpdatedEventPeriods");

      try
      {
         JSONArray periods = DtvkitGlueClient.getInstance()
            .request("Dvb.getListOfUpdatedEventPeriods", new JSONArray()).getJSONArray("data");

         if (DEBUG) Log.i(TAG, periods.toString());

         for (int i = 0; i < periods.length(); i++)
         {
            JSONObject period = periods.getJSONObject(i);

            eventPeriods.add(new EventPeriod(period.getString("uri"), period.getLong("startutc"),
               period.getLong("endutc")));
         }
      }
      catch (Exception e)
      {
         Log.e(TAG, e.getMessage());
      }

      return eventPeriods;
   }

   private static String[] getGenres(String genre)
   {
      switch (genre)
      {
         case "movies":
            return new String[]{TvContract.Programs.Genres.MOVIES};
         case "news":
            return new String[]{TvContract.Programs.Genres.NEWS};
         case "entertainment":
            return new String[]{TvContract.Programs.Genres.ENTERTAINMENT};
         case "sport":
            return new String[]{TvContract.Programs.Genres.SPORTS};
         case "childrens":
            return new String[]{TvContract.Programs.Genres.FAMILY_KIDS};
         case "music":
            return new String[]{TvContract.Programs.Genres.MUSIC};
         case "arts":
            return new String[]{TvContract.Programs.Genres.ARTS};
         case "social":
            return new String[]{TvContract.Programs.Genres.LIFE_STYLE};
         case "education":
            return new String[]{TvContract.Programs.Genres.EDUCATION};
         case "leisure":
            return new String[]{TvContract.Programs.Genres.LIFE_STYLE};
         default:
            return new String[]{};
      }
   }

   private TvContentRating[] getContentRatings(int rating)
   {
      TvContentRating tvContentRatings[] = null;

      if ((rating >= 4) && (rating <= 18))
      {
         tvContentRatings = new TvContentRating[]
            {
               TvContentRating.createRating("com.android.tv", "DVB", "DVB_" + rating)
            };
      }

      return tvContentRatings;
   }
}
