package org.dtvkit.inputsource;

import android.os.HwBinder;
import android.os.RemoteException;
import android.util.Log;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.lang.reflect.Method;
import java.util.ArrayList;

import vendor.dtvkit.hardware.dtvkit.V1_0.IDTVKit;
import vendor.dtvkit.hardware.dtvkit.V1_0.IDTVKitSignalHandler;
import vendor.dtvkit.hardware.dtvkit.V1_0.IDTVKitOverlayTarget;

public class DtvkitGlueClient
{
   private static final String TAG = "DtvkitGlueClient";

   private static DtvkitGlueClient mSingleton = null;
   private IDTVKit mDtvkit = null;
   private ArrayList<SignalHandler> mHandlers = new ArrayList<>();
   private OverlayTarget mTarget = null;

   private final IDTVKitSignalHandler.Stub mSignalHandler = new IDTVKitSignalHandler.Stub()
   {
      public void signal(String signal, String json)
      {
         JSONObject object;
         try
         {
            object = new JSONObject(json);
         }
         catch (Exception e)
         {
            Log.e(TAG, e.getMessage());
            return;
         }
         for (SignalHandler handler : mHandlers)
         {
            handler.onSignal(signal, object);
         }
      }
   };

   private final IDTVKitOverlayTarget.Stub mOverlayTarget = new IDTVKitOverlayTarget.Stub()
   {
      public void regionCreate(int width, int height)
      {
         if (mTarget != null)
         {
            mTarget.regionCreate(width, height);
         }
      }

      public void regionDrawBitmap(int x, int y, int width, int height, byte[] data)
      {
         if (mTarget != null)
         {
            mTarget.regionDrawBitmap(x, y, width, height, data);
         }
      }

      public void overlayCreate(int width, int height)
      {
         if (mTarget != null)
         {
            mTarget.overlayCreate(width, height);
         }
      }

      public void overlayDrawRegion(int x, int y, int width, int height)
      {
         if (mTarget != null)
         {
            mTarget.overlayDrawRegion(x, y, width, height);
         }
      }

      public void overlayDisplay()
      {
         if (mTarget != null)
         {
            mTarget.overlayDisplay();
         }
      }
   };

   HwBinder.DeathRecipient mDeathRecipient = new HwBinder.DeathRecipient()
   {
      @Override
      public void serviceDied(long cookie)
      {
         mDtvkit = null;
      }
   };

   interface SignalHandler
   {
      void onSignal(String signal, JSONObject data);
   }

   interface OverlayTarget
   {
      void regionCreate(int width, int height);

      void regionDrawBitmap(int x, int y, int width, int height, byte[] data);

      void overlayCreate(int width, int height);

      void overlayDrawRegion(int x, int y, int width, int height);

      void overlayDisplay();
   }

   protected DtvkitGlueClient()
   {
      // Singleton
   }

   public static DtvkitGlueClient getInstance()
   {
      if (mSingleton == null)
      {
         mSingleton = new DtvkitGlueClient();
      }

      mSingleton.connectIfUnconnected();

      return mSingleton;
   }

   public void registerSignalHandler(SignalHandler handler)
   {
      if (!mHandlers.contains(handler))
      {
         mHandlers.add(handler);
      }
   }

   public void unregisterSignalHandler(SignalHandler handler)
   {
      if (mHandlers.contains(handler))
      {
         mHandlers.remove(handler);
      }
   }

   public void setOverlayTarget(OverlayTarget target)
   {
      mTarget = target;
   }

   public JSONObject request(String resource, JSONArray arguments) throws Exception
   {
      mSingleton.connectIfUnconnected();
      try
      {
         JSONObject object = new JSONObject(mDtvkit.request(resource, arguments.toString()));
         if (object.getBoolean("accepted"))
         {
            return object;
         }
         else
         {
            throw new Exception(object.getString("data"));
         }
      }
      catch (JSONException | RemoteException e)
      {
         throw new Exception(e.getMessage());
      }
   }

   private void connectIfUnconnected()
   {
      if (mDtvkit == null)
      {
         try
         {
            // throws NoSuchElementException if not available
            mDtvkit = IDTVKit.getService(true /* retry */);

            mDtvkit.registerSignalHandler(mSignalHandler);
            mDtvkit.registerOverlayTarget(mOverlayTarget);
            mDtvkit.linkToDeath(mDeathRecipient, 0);
         }
         catch (RemoteException e)
         {
            Log.wtf("DTVKIT", e.getMessage());
         }
      }
   }
}

