package org.dtvkit.inputsource;

import android.app.Notification;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.app.Service;
import android.content.BroadcastReceiver;
import android.content.ComponentName;
import android.content.ContentResolver;
import android.content.ContentUris;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.database.ContentObserver;
import android.database.Cursor;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.PorterDuff;
import android.graphics.Rect;
import android.hardware.usb.UsbManager;
import android.media.AudioManager;
import android.media.PlaybackParams;
import android.media.tv.TvContentRating;
import android.media.tv.TvContract;
import android.media.tv.TvInputInfo;
import android.media.tv.TvInputInfo.Builder;
import android.media.tv.TvInputManager;
import android.media.tv.TvInputService;
import android.media.tv.TvTrackInfo;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.ConditionVariable;
import android.os.Environment;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Message;
import android.os.Parcelable;
import android.os.storage.DiskInfo;
import android.os.storage.StorageEventListener;
import android.os.storage.StorageManager;
import android.os.storage.VolumeInfo;
import android.os.storage.VolumeRecord;
import android.os.SystemProperties;
import android.text.TextUtils;
import android.util.Log;
import android.util.LongSparseArray;
import android.view.LayoutInflater;
import android.view.Surface;
import android.view.SurfaceHolder;  
import android.view.SurfaceHolder.Callback;  
import android.view.SurfaceView;
import android.view.View;
import android.view.ViewGroup;
import android.widget.TextView;
import android.graphics.BitmapFactory;
import android.graphics.PixelFormat;

import org.dtvkit.companionlibrary.BaseTvInputService;
import org.dtvkit.companionlibrary.EpgSyncJobService;
import org.dtvkit.companionlibrary.model.Channel;
import org.dtvkit.companionlibrary.model.InternalProviderData;
import org.dtvkit.companionlibrary.model.ModelUtils;
import org.dtvkit.companionlibrary.model.Program;
import org.dtvkit.companionlibrary.model.RecordedProgram;
import org.dtvkit.companionlibrary.utils.TvContractUtils;

import org.dtvkit.companionlibrary.TvPlayer;
import org.dtvkit.inputsource.player.DtvkitTvPlayer;
import org.dtvkit.inputsource.TransportManager;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.lang.Long;
import java.lang.Runnable;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.Semaphore;
import java.util.concurrent.ThreadLocalRandom;
import java.util.Timer;
import java.util.TimerTask;

public class DtvkitTvInput extends BaseTvInputService
{
   private static final String TAG = "DtvkitTvInput";
   private static final boolean DEBUG = false;
   public static final String CHANNEL_ID = "DtvkitTvInputChannel";
   private static final boolean FAKE_SSU_DOWNLOAD = true;
   private static final String RECORDING_DATA_URI_SCHEME = "crid";
   private static final String RECORDING_DATA_URI_BASE = "crid://localhost/recordings/";
   private static final String DISK_PATH = "/storage/";
   private LongSparseArray<Channel> mChannels;
   private ContentResolver mContentResolver;
   private static ContentObserver mChannelObserver;
   private static ContentObserver mProgramObserver;
   private boolean mSyncEpg = false;
   private TvInputManager mTvInputManager;
   private final TransportManager mTransportManager = new TransportManager();
   private DtvkitTvInputSession mSession;
   private DtvkitTvInputRecordingSession mTimeShiftRecordingSession;
   private DtvkitTvInputRecordingSession mRecordingSession;
   private List<DtvkitTvInputRecordingSession> mRecordingSessions;
   private boolean mCanRecord;
   private String mDiskPath;
   private int mTunerCount;
   private TvInputInfo mTvInputInfo;
   private DtvkitGlueClient mDtvkitGlueClient;
   private StorageManager mStorageManager;
   private Context mContext;
   private int mOverlayViewWidth = -1;
   private int mOverlayViewHeight = -1;
   private DtvkitTvInputThread mMonitoringThread;
   private boolean mMonitoringThreadRunning = false;
   private ConditionVariable mSessionChanged;
   private boolean mDisableFcc = false;

   // Database calls
   private static HandlerThread mDbHandlerThread;

   // Content ratings
   private static final List<Session> mSessions = new ArrayList<>();

   private final StorageEventListener mStorageEventListener = new StorageEventListener() {
       @Override
       public void onUsbMassStorageConnectionChanged(boolean connected) {
           if (DEBUG) Log.d(TAG, "onUsbMassStorageConnectionChanged " + connected);
       }

       @Override
       public void onStorageStateChanged(String path, String oldState, String newState) {
           if (DEBUG || true) Log.d(TAG, "onStorageStateChanged " + path);
       }

       @Override
       public void onVolumeStateChanged(VolumeInfo vol, int oldState, int newState) {
           if (DEBUG) Log.d(TAG, "onVolumeStateChanged " + vol.toString());
           String path;
           File file = vol.getPath();
           if (null == file)
               return;

           path = file.toString();
           if (null == path)
               return;

           switch (newState) {
           case VolumeInfo.STATE_EJECTING:
               if (DEBUG) Log.d(TAG, "newState  ->  STATE_EJECTING");               
               if ((null != mDiskPath) && mDiskPath.equals(path)) {
                  mCanRecord = false;
                  mDiskPath = null;
                  // notify                     
                  mTvInputInfo = buildTvInputInfo(mContext);
                  mTvInputManager.updateTvInputInfo (mTvInputInfo);
                  Log.i(TAG, "TvInputInfo.canRecord(): " + mTvInputInfo.canRecord());
               }
               break;
           case VolumeInfo.STATE_MOUNTED:
               if (null == mDiskPath) {
                  if (path.startsWith("/storage/emulated/"))
                     break;

                  if (path.startsWith("/storage/self/"))
                     break;

                  if (path.startsWith(DISK_PATH)) {
                     mCanRecord = true;
                     mDiskPath = path;
                     int index = path.lastIndexOf("/Android/data");
                     if (index > 0) {
                        mDiskPath = path.substring(0, index);
                     }
                     // notify                     
                     mTvInputInfo = buildTvInputInfo(mContext);
                     mTvInputManager.updateTvInputInfo (mTvInputInfo);
                     Log.i(TAG, "TvInputInfo.canRecord(): " + mTvInputInfo.canRecord());
                  }
               }
               break;
           }
       }

       @Override
       public void onVolumeRecordChanged(VolumeRecord rec) {
           if (DEBUG) Log.d(TAG, "onVolumeRecordChanged ");
       }

       @Override
       public void onVolumeForgotten(String fsUuid) {
           if (DEBUG) Log.d(TAG, "onVolumeForgotten ");
       }

       @Override
       public void onDiskScanned(DiskInfo disk, int volumeCount) {
           if (DEBUG) Log.d(TAG, "onDiskScanned ");
       }

       @Override
       public void onDiskDestroyed(DiskInfo disk) {
           if (DEBUG) Log.d(TAG, "onDiskDestroyed ");
       }
   };

   private final BroadcastReceiver mParentalControlsBroadcastReceiver = new BroadcastReceiver()
   {
      @Override
      public void onReceive(Context context, Intent intent)
      {
         if (intent.getAction().equals(
                 TvInputManager.ACTION_PARENTAL_CONTROLS_ENABLED_CHANGED))
         {
             if (DEBUG || true) Log.d(TAG, "Received parental control settings change");
         }
         else if (intent.getAction().equals(
                 TvInputManager.ACTION_BLOCKED_RATINGS_CHANGED))
         {
             if (DEBUG || true) Log.d(TAG, "Received blocked ratings change");
         }
         else if (TextUtils.equals(intent.getAction(), UsbManager.ACTION_USB_DEVICE_ATTACHED)) {
             //Log.d(TAG, "ACTION_USB_DEVICE_ATTACHED");
         }
         else if (TextUtils.equals(intent.getAction(), UsbManager.ACTION_USB_DEVICE_DETACHED)) {
             //Log.d(TAG, "ACTION_USB_DEVICE_DETACHED");
         }
         else if (TextUtils.equals(intent.getAction(), VolumeInfo.ACTION_VOLUME_STATE_CHANGED)) {
             //Log.d(TAG, "ACTION_VOLUME_STATE_CHANGED");
         }

         for (Session session : mSessions)
         {
            TvInputManager tvInputManager = (TvInputManager) getApplicationContext()
               .getSystemService(Context.TV_INPUT_SERVICE);

            if (tvInputManager.isParentalControlsEnabled())
            {
               if (session instanceof DtvkitTvInputSession)
               {
                  ((DtvkitTvInputSession) session).checkCurrentProgramContent();
                  ((DtvkitTvInputSession) session).blockCurrentProgramIfNeeded();
               }
            }
            else
            {
               session.onUnblockContent(null);
            }
         }
      }
   };

   class DtvkitTvInputThread extends Thread {
      @Override
      public void run() {
         mMonitoringThreadRunning = true;
         boolean isMonitoring = true;

         String sBouquetId = SystemProperties.get("persist.vendor.dvb.bid");
         if ((sBouquetId != null) && sBouquetId.equals("99")) {
             return;
         }
         
         startDetection();
    
         while (mMonitoringThreadRunning) {
            mSessionChanged.block(1000 * 5);
            mSessionChanged.close();
            // ignore zapping events
            try {
               sleep(1000 * 1);
            } catch (InterruptedException e) {
               e.printStackTrace();
            }
            int sessions = mSessions.size();
            sessions += mRecordingSessions.size();
            
            if (sessions != 0) {
            Log.i(TAG, "sessions != zero");          
               if (isMonitoring) {               
                  // stop monitoring
                  Log.i(TAG, "STOP monitoring");
                  isMonitoring = false;
                  stopDetection();
               }
            }
            else {
               Log.i(TAG, "sessions == zero");
               if (isMonitoring == false) {
                  // start monitoring
                  Log.i(TAG, "START monitoring");
                  isMonitoring = true;
                  startDetection();
               } 
            }
         }  // while (mMonitoringThreadRunning)
         
         if (isMonitoring) {
            // stop monitoring
            Log.i(TAG, "STOP monitoring (exit)");
            isMonitoring = false;
            stopDetection();

         }
      } 
   }

   private void startDetection() {
     stopDetection();
     try {
         JSONArray args = new JSONArray();
         args.put(405000000);  //Hz
         args.put(String.valueOf("qam256"));
         args.put(Integer.valueOf(5217));
         args.put(String.valueOf("6MHz"));
         Log.i(TAG, args.toString());
         DtvkitGlueClient.getInstance().request("Dvbc.startSignalDetection", args);
     } catch (Exception e) {

     }
   }

   private void stopDetection() {
      try {
         JSONArray args = new JSONArray();
         DtvkitGlueClient.getInstance().request("Dvbc.stopSignalDetection", args);
      } catch (Exception e) {

      }
   }


   public DtvkitTvInput()
   {
      super();
      mSessionChanged = new ConditionVariable();
      mSession = null;
      mTimeShiftRecordingSession = null;
      mRecordingSession = null;
      mRecordingSessions = new ArrayList<>();
      mCanRecord = false;
      mDiskPath = null;
      mTunerCount = 0;
      mTvInputInfo = null;
      mDtvkitGlueClient = null;
      mContext = null;
      Log.i(TAG, "DtvkitTvInput");
   }

   @Override
   public void onCreate()
   {
      Log.i(TAG, "onCreate");
      super.onCreate();

      mContext = this;
      mTvInputManager = (TvInputManager) getSystemService(Context.TV_INPUT_SERVICE);
      mStorageManager = (StorageManager) getSystemService(Context.STORAGE_SERVICE);
      
      new Thread(new Runnable() {
         @Override
         public void run() {
            mDtvkitGlueClient = DtvkitGlueClient.getInstance();
            mDtvkitGlueClient.registerSignalHandler(mHandler);
            // Create database background thread
            mDbHandlerThread = new HandlerThread(mContext.getClass().getSimpleName());
            mDbHandlerThread.start();

            mContentResolver = getContentResolver();
            onChannelsChanged();
            setupChannelObserver();
            mContentResolver
               .registerContentObserver(TvContract.Channels.CONTENT_URI, true, mChannelObserver);

            // Listen for program updates and if current program is updated then check if it should be
            // blocked
            setupProgramObserver();
            mContentResolver
               .registerContentObserver(TvContract.Programs.CONTENT_URI, true, mProgramObserver);

            updateDefaultLanguage();

            // Listen for changes to blocked ratings and parental controls status
            setupParentalControlsBroadcastReceiver();

            IntentFilter intentFilter = new IntentFilter();
            intentFilter.addAction(TvInputManager.ACTION_PARENTAL_CONTROLS_ENABLED_CHANGED);
            intentFilter.addAction(TvInputManager.ACTION_BLOCKED_RATINGS_CHANGED);
            intentFilter.addAction(UsbManager.ACTION_USB_DEVICE_ATTACHED);
            intentFilter.addAction(UsbManager.ACTION_USB_DEVICE_DETACHED);
            intentFilter.addAction(VolumeInfo.ACTION_VOLUME_STATE_CHANGED);
            // number of tuners
            try {
                JSONObject obj = mDtvkitGlueClient.request("Dvb.getListOfFrontends", new JSONArray());
                JSONArray tuners = obj.getJSONArray("data");
                mTunerCount = tuners.length();
            }
            catch (Exception ignore) {
                Log.e(TAG, ignore.getMessage());
            }      

            if (null != mStorageManager) {
                mStorageManager.registerListener(mStorageEventListener);
            }

            for (File xfile : getExternalFilesDirs(Environment.MEDIA_MOUNTED)) {
               if (xfile != null) {
                  String absPath = xfile.getAbsolutePath();
                  if (absPath.startsWith("/storage/emulated/"))
                     continue;

                  if (absPath.startsWith("/storage/self/"))
                     continue;

                  if (absPath.startsWith(DISK_PATH)) {
                     int index = absPath.lastIndexOf("/Android/data");
                     if (index >= 0) {
                        String path = absPath.substring(0, index);
                        File file = new File(path);                 
                        if (file.exists()) {
                           mCanRecord = true;
                           mDiskPath = path;
                           Log.i(TAG, "mDiskPath: " + mDiskPath);
                           break;
                        }
                     }
                  }
               }
            }
            // check the external storage
            Log.i(TAG, "canRecord: " + mCanRecord);

            mTvInputInfo = buildTvInputInfo(mContext);
            mTvInputManager.updateTvInputInfo (mTvInputInfo);

            onChannelsChanged();
            
            // monitoring loop
            mMonitoringThread = new DtvkitTvInputThread();
            mMonitoringThread.start();
            
         }
      }).start();
   }

   @Override
   public void onDestroy()
   {
      Log.i(TAG, "onDestroy");
      mMonitoringThreadRunning = false;
      /*
      try {
        mMonitoringThread.join();
      } catch (InterruptedException e) {

      }
      */
      mMonitoringThread = null;
      unregisterReceiver(mParentalControlsBroadcastReceiver);
      mContentResolver.unregisterContentObserver(mProgramObserver);
      mContentResolver.unregisterContentObserver(mChannelObserver);
      mDbHandlerThread.quit();
      mDbHandlerThread = null;
      mSession = null;
      mTvInputInfo = null;
      mTvInputManager = null;
      mDtvkitGlueClient = null;
      if (null != mStorageManager) {
          mStorageManager.unregisterListener(mStorageEventListener);
          mStorageManager = null;
      } 
      super.onDestroy();
   }

   @Override
   public final Session onCreateSession(String inputId)
   {
      Log.i(TAG, "onCreateSession " + inputId);
      DtvkitTvInputSession session;
      if (null != mSession) {
          mSession.onRelease();
          mSession = null;
      }
      try {
          session = new DtvkitTvInputSession(this, inputId);
          mSession = session;
          session.setOverlayViewEnabled(true);
      } catch (RuntimeException e) {
          // There are no available DVB devices.
          Log.e(TAG, "Creating a session for " + inputId + " failed.", e);
          return null;
      }

      synchronized(mSessions) {
         mSessions.add(session);
      }
      return session;
   }

   @Override
   public RecordingSession onCreateRecordingSession(String inputId) {
       Log.i(TAG, "onCreateRecordingSession " + inputId);
       // disable FCC
       mDisableFcc = true;
       if (mSession != null) {
           Uri uri = mSession.getChannelUri();
           if (uri != null) {
              ArrayList<Uri> channelUris = new ArrayList<Uri>(1);  // [Live TV]
              channelUris.add(0, uri);
              playerPlay(channelUris);
           }
       }
       try {
           DtvkitTvInputRecordingSession session = new DtvkitTvInputRecordingSession(this, inputId);
           return session;
       } catch (RuntimeException e) {
           // There are no available DVB devices.
           Log.e(TAG, "Creating a RecordingSession for " + inputId + " failed.", e);
           return null;
       }
   }

   @Override
   public int onStartCommand(Intent intent, int flags, int startId) {
       Log.i(TAG, "onStartCommand");
       String input = intent.getStringExtra("inputExtra");
       createNotificationChannel();
       Intent notificationIntent = new Intent(this, MainActivity.class);
       PendingIntent pendingIntent = PendingIntent.getActivity(this,
               0, notificationIntent, 0);
 
       Notification notification = new Notification.Builder(this, CHANNEL_ID)
               .setContentTitle("DtvkitTvInput Foreground Service")
               .setContentText("inputsource")
               .setSmallIcon(R.drawable.dtvkit)
               .setContentIntent(pendingIntent)
               .build();
 
       startForeground(1, notification);
 
       //do heavy work on a background thread
       //stopSelf();
 
       return START_NOT_STICKY;
   }

   private void createNotificationChannel() {
       if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
           NotificationChannel serviceChannel = new NotificationChannel(
                   CHANNEL_ID,
                   TAG + " Foreground Service Channel",
                   NotificationManager.IMPORTANCE_DEFAULT
           );
 
           NotificationManager manager = getSystemService(NotificationManager.class);
           manager.createNotificationChannel(serviceChannel);
       }
   }

   public TvInputInfo buildTvInputInfo(Context context) {
       TvInputInfo tvInputInfo = null;
       ComponentName componentName = new ComponentName (context, DtvkitTvInput.class);
       try {
           TvInputInfo.Builder builder = new TvInputInfo.Builder(context, componentName);
           Bundle extrasBundle = new Bundle();
           if (null != extrasBundle) {
               Bundle moduleBundle = new Bundle();
               if (null != moduleBundle) {
                   String diskPath;
                   if (mCanRecord) {
                       diskPath = mDiskPath;
                   }
                   else {
                       diskPath = String.valueOf("");
                   }
                   moduleBundle.putString("disk-path", diskPath);
                   moduleBundle.putString("disk-serial-number", "TEST-1234567890");            
                   // an empty ssu bundle
                   Bundle ssuBundle = new Bundle();
                   if (null != ssuBundle) {
                       moduleBundle.putBundle("ssu", ssuBundle);
                   }
                   extrasBundle.putBundle(
                       TvContract.buildInputId(componentName),
                       moduleBundle
                   );
                   // an empty si bundle
                   Bundle siBundle = new Bundle();
                   if (null != siBundle) {
                       moduleBundle.putBundle("si", siBundle);
                   }
                   extrasBundle.putBundle(
                       TvContract.buildInputId(componentName),
                       moduleBundle
                   );
               }  // if (null != moduleBundle)
           }  // if (null != extrasBundle)
           builder.setCanRecord(mCanRecord)
               .setTunerCount(mTunerCount)
               .setExtras(extrasBundle);
           tvInputInfo = builder.build();
       } catch (NullPointerException e) {
           // TunerTvInputService is not enabled.
       }
        
       return tvInputInfo;
   }

   private final DtvkitGlueClient.SignalHandler mHandler = new DtvkitGlueClient.SignalHandler() {
       @Override
       public void onSignal(String signal, JSONObject data) {
           if (signal.equals("PlayerStatusChanged")) {
               String state = "off";
               try {
                   state = data.getString("state");
               } catch (JSONException ignore) {
               }
               switch (state) {
                   default:
                   Log.d(TAG, "onSignal   ");
                       break;
               }
           }
           else if (signal.equals("SsuUpdate")) {
               String ssuInfo;
               int percent;
               try {
                   ssuInfo = data.getString("ssu");
                   percent = data.getInt("progress");
                   Log.i(TAG, "ssuInfo: " + ssuInfo);
                   JSONObject jsonSsuInfo = new JSONObject(ssuInfo);
                   int onid = jsonSsuInfo.getInt("onid");
                   int tsid = jsonSsuInfo.getInt("tsid");
                   int sid = jsonSsuInfo.getInt("sid");
                   JSONObject jsonRelease = jsonSsuInfo.getJSONObject("info");
                   String release = jsonRelease.getString("update.zip");
                   if (false == Build.DISPLAY.equals(release)) {
                       // the version is different
                       if (null != mTvInputInfo) {
                           Bundle extrasBundle = mTvInputInfo.getExtras();
                           if (null != extrasBundle) {
                               Bundle moduleBundle = extrasBundle.getBundle(mTvInputInfo.getId());
                               if (null != moduleBundle) {
                                   String from = "ssu";
                                   moduleBundle.putString("from", from);
                                   Bundle ssuBundle = moduleBundle.getBundle(from);
                                   if (null != ssuBundle) {
                                       String event = "update";
                                       //ssuBundle.clear();
                                       ssuBundle.putString("event", event);
                                       Bundle eventBundle = new Bundle();
                                       if (null != eventBundle) {
                                           eventBundle.putInt("original-network-id", onid);
                                           eventBundle.putInt("transport-stream-id", tsid);
                                           eventBundle.putInt("service-id", sid);
                                           eventBundle.putString("release", release);
                                           eventBundle.putString("file", "update.zip");
                                           //
                                           ssuBundle.putBundle(event, eventBundle);
                                       }
                                   }  // if (null != ssuBundle)
                                }  // if (null != moduleBundle)
                            }  // if (null != extrasBundle)
                            mTvInputManager.updateTvInputInfo (mTvInputInfo);
                        }  //  if (null != mTvInputInfo)
                    }
               }
               catch (JSONException ignore) {

               }
           }
       }
   };

   class OverlayView extends View
   {
      Bitmap region = null;
      Bitmap overlay_draw = null;
      Bitmap overlay_update = null;
      Rect src, dst;

      Semaphore sem = new Semaphore(1);

      private final DtvkitGlueClient.OverlayTarget mTarget = new DtvkitGlueClient.OverlayTarget()
      {
         @Override
         public void regionCreate(int width, int height)
         {
            region = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888);
         }

         @Override
         public void regionDrawBitmap(int x, int y, int width, int height, byte[] data)
         {
            /* Build an array of ARGB_8888 pixels as signed ints and add this part to the region */
            int[] colors = new int[width * height];
            for (int i = 0, j = 0; i < width * height; i++, j += 4)
            {
               colors[i] = (((int) data[j] & 0xFF) << 24) | (((int) data[j + 1] & 0xFF) << 16) |
                  (((int) data[j + 2] & 0xFF) << 8) | ((int) data[j + 3] & 0xFF);
            }
            Bitmap part =
               Bitmap.createBitmap(colors, 0, width, width, height, Bitmap.Config.ARGB_8888);
            Canvas canvas = new Canvas(region);
            canvas.drawBitmap(part, x, y, null);
         }

         @Override
         public void overlayCreate(int width, int height)
         {
            if (overlay_draw == null)
            {
               /* Create 2 layers for double buffering */
               overlay_draw = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888);
               overlay_update = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888);

               /* Clear the overlay that will be drawn initially */
               Canvas canvas = new Canvas(overlay_draw);
               canvas.drawColor(0, PorterDuff.Mode.CLEAR);
            }

            Canvas canvas = new Canvas(overlay_update);
            canvas.drawColor(0, PorterDuff.Mode.CLEAR);
         }

         @Override
         public void overlayDrawRegion(int x, int y, int width, int height)
         {
            if (region != null)
            {
               Canvas canvas = new Canvas(overlay_update);
               Rect src = new Rect(0, 0, region.getWidth(), region.getHeight());
               Rect dst = new Rect(x, y, x + width, y + height);
               Paint paint = new Paint();
               paint.setAntiAlias(true);
               paint.setFilterBitmap(true);
               paint.setDither(true);
               canvas.drawBitmap(region, src, dst, paint);
               region = null;
            }
         }

         @Override
         public void overlayDisplay()
         {
                /* The update layer is now ready to be displayed so switch the overlays and use
                   the other one for the next update */
            sem.acquireUninterruptibly();
            Bitmap temp = overlay_draw;
            overlay_draw = overlay_update;
            src = new Rect(0, 0, overlay_draw.getWidth(), overlay_draw.getHeight());
            overlay_update = temp;
            sem.release();
            postInvalidate();
         }
      };

      public OverlayView(Context context)
      {
         super(context);
         DtvkitGlueClient.getInstance().setOverlayTarget(mTarget);
      }

      public void setSize(int width, int height)
      {
         dst = new Rect(0, 0, width, height);
      }

      @Override
      protected void onDraw(Canvas canvas)
      {
         super.onDraw(canvas);
         sem.acquireUninterruptibly();
         if (overlay_draw != null)
         {
            canvas.drawBitmap(overlay_draw, src, dst, null);
         }
         sem.release();
      }
   }

   // We do not indicate recording capabilities. TODO for recording.
   //public TvInputService.RecordingSession onCreateRecordingSession(String inputId)

   class DtvkitTvInputSession extends BaseTvInputService.Session
   {
      private static final String TAG = "DtvkitTvInputSession";
      private static final boolean DEBUG = false;
      private static final boolean DEBUG_SUBTITLE = true;
      Uri mChannelUri = null;
      private Channel mTunedChannel = null;
      private Uri mCurrentChannelUri;
      private List<TvTrackInfo> mTunedTracks = null;
      private TvTrackInfo mSubtitleTrack = null;
      private Context mContext;
      private final TvInputManager mTvInputManager;
      private Program mCurrentProgram;
      private TvContentRating mBlockedRating;
      private final PlayHandler mPlayHandler;
      private String retuneDvbUri;
      OverlayView mView = null;
      private Handler mUiHandler;
      private DtvkitTvPlayer mPlayer;
      private boolean mCaptionEnabled = false;
      private AudioManager mAudioManager;
      private Handler mTaskHandler;
      private RecordedProgram mRecordedProgram;
      private Uri mRecordedProgramUri;
      private long mTimeShiftResumePosition;
      private boolean mSeek;
      private Surface mSurface;
      private ArrayList<Uri> mChannelUris;  // [ Live TV Uri, FCC Uri [, FCC Uri] ]

      private class PlayHandler extends Handler
      {
         private static final int ACTION_PLAY = 100;
         private static final int ACTION_STOP = 101;
         private static final int ACTION_CHANGE_STATE = 102;
         private static final String KEY_DVB_URI = "DVB_URI";
         private static final String KEY_CHANNEL_URI = "CHANNEL_URI";
         private static final String KEY_STATE = "STATE";

         @Override
         public void handleMessage(Message msg)
         {
            switch (msg.what)
            {
               case ACTION_PLAY:
               {
                  handlePlay(mChannelUris);
                  break;
               }
               case ACTION_STOP:
               {
                  handleStop();
                  break;
               }
               case ACTION_CHANGE_STATE:
               {
                  String state = (String) msg.getData().get(KEY_STATE);
                  handleChangeState(state);
                  break;
               }
            }
         }

         private void handlePlay(ArrayList<Uri> channelUris)
         {
            notifyVideoUnavailable(TvInputManager.VIDEO_UNAVAILABLE_REASON_TUNING);
            mCurrentChannelUri = channelUris.get(0);
            mCurrentProgram = getCurrentProgram(mCurrentChannelUri);

            // Check if the current program should be blocked and cache the blocked rating
            if (mTvInputManager.isParentalControlsEnabled()) {
               checkCurrentProgramContent();
            }

            if (playerPlay(channelUris)) {
               mTunedChannel = getChannel(mCurrentChannelUri);
            }
            else {
               mTunedChannel = null;
               notifyVideoUnavailable(TvInputManager.VIDEO_UNAVAILABLE_REASON_UNKNOWN);
            }
         }

         private void handleStop()
         {
            playerStop();
            mTunedChannel = null;
         }

         private void handleChangeState(String state)
         {
            switch (state)
            {
               case "starting":
                  Platform platform = new Platform();
                  if (mSurface != null) {
                     platform.setVideoSurface(4 /*RSTBPlayer.PLAY_MODE.MODE_TUNER*/, mSurface, false);
                  }
                  break;
               case "playing":
                  // Block current program if the blocked rating cache exist
                  if (mTvInputManager.isParentalControlsEnabled())
                  {
                     blockCurrentProgramIfNeeded();
                  }
                  if ((null != mTunedChannel) && mTunedChannel.getServiceType().equals(TvContract.Channels.SERVICE_TYPE_AUDIO))
                  {
                     notifyVideoUnavailable(TvInputManager.VIDEO_UNAVAILABLE_REASON_AUDIO_ONLY);
                  }
                  else
                  {
                     notifyVideoAvailable();
                     mSessionChanged.open();
                     boolean timeShiftAvailable = false;
                     if (null != mTimeShiftRecordingSession) {
                         Uri recordingUri = mTimeShiftRecordingSession.getChannelUri();
                         Uri playbackUri = getChannelUri(); 
                         if (playbackUri.equals(recordingUri)) {
                             timeShiftAvailable = true;
                         }
                     }
                     else {
                         if (null != mRecordedProgramUri) {
                             String uri = mRecordedProgramUri.toString();
                             if (uri.startsWith("content://android.media.tv/recorded_program/")) {
                                 timeShiftAvailable = true;
                             }
                         }
                     }
                     if (timeShiftAvailable)
                         notifyTimeShiftStatusChanged(TvInputManager.TIME_SHIFT_STATUS_AVAILABLE);
                     else
                         notifyTimeShiftStatusChanged(TvInputManager.TIME_SHIFT_STATUS_UNAVAILABLE);
                  }
                  List<TvTrackInfo> tracks = playerGetTracks();
                  if (!tracks.equals(mTunedTracks))
                  {
                     mTunedTracks = tracks;
                     // TODO Also for service changed event
                     notifyTracksChanged(mTunedTracks);
                  }
                  notifyTrackSelected(TvTrackInfo.TYPE_AUDIO, playerGetSelectedAudioTrack());
                  //FIXME: for debugging
                  {
                      String pidString = playerGetSelectedSubtitleTrack();
                      String[] parts = pidString.split(":");
                      Iterator it = mTunedTracks.iterator();
                      mSubtitleTrack = null;
                      while(it.hasNext()){
                          TvTrackInfo trackInfo = (TvTrackInfo)it.next();
                          if(trackInfo.getId().equals(parts[0])) {
                              mSubtitleTrack = trackInfo;
                              break;
                          }
                      }
                  }
                  /* FIXME: no subtitle after tuning a new channel
                  if (playerGetSubtitlesOn())
                  {
                     notifyTrackSelected(TvTrackInfo.TYPE_SUBTITLE,
                        playerGetSelectedSubtitleTrack());
                  }
                  */
                  break;
               case "blocked":
                  notifyContentBlocked(
                     TvContentRating.createRating("com.android.tv", "DVB", "DVB_18"));
                  break;
               case "badsignal":
                  notifyVideoUnavailable(TvInputManager.VIDEO_UNAVAILABLE_REASON_WEAK_SIGNAL);
                  break;
               default:
                  break;
            }
         }
      }

      DtvkitTvInputSession(Context context, String inputId)
      {
         super(context, inputId);
         Log.i(TAG, "DtvkitTvInputSession");
         DtvkitGlueClient.getInstance().registerSignalHandler(mHandler);
         mContext = context;
         mTvInputManager = (TvInputManager) context.getSystemService(Context.TV_INPUT_SERVICE);
         mPlayHandler = new PlayHandler();
         retuneDvbUri = null;
         mAudioManager = (AudioManager) mContext.getSystemService(Context.AUDIO_SERVICE);
         mRecordedProgram = null;
         mRecordedProgramUri = null;
         mTimeShiftResumePosition = TvInputManager.TIME_SHIFT_INVALID_TIME;
         mSeek = false;
         mChannelUris = new ArrayList<Uri>(3);  // [Live TV, FCC preload, FCC preload]

         if (mSyncEpg)
         {
            ComponentName sync = new ComponentName(mContext, DtvkitEpgSync.class);

            EpgSyncJobService.requestImmediateSync(mContext, inputId,
               EpgSyncJobService.SYNC_MODE_FULL_UPDATE, false, sync);
            mSyncEpg = false;
         }
      }

      public void onRelease()
      {
         Log.i(TAG, "onRelease");
         mPlayHandler.removeMessages(PlayHandler.ACTION_PLAY);
         if (null != mRecordedProgram) {
             mRecordedProgram = null;
         }
         playerStop();
         if (mTunedChannel != null) {
             int onid = mTunedChannel.getOriginalNetworkId();
             int tsid = mTunedChannel.getTransportStreamId();
             boolean ret = mTransportManager.release(onid, tsid);
             mTunedChannel = null;
         }
         releasePlayer();
         DtvkitGlueClient.getInstance().unregisterSignalHandler(mHandler);
         synchronized(mSessions) {
             mSessions.remove(this);
         }
         super.onRelease();
      }

      @Override
      public boolean onSetSurface(Surface surface)
      {
         Log.i(TAG, "onSetSurface " + surface);
         mSurface = surface;
         return true;
      }

      @Override
      public void onSurfaceChanged(int format, int width, int height)
      {
         Log.i(TAG, "onSurfaceChanged " + format + ", " + width + ", " + height);
      }

      public View onCreateOverlayView()
      {
         mView = new OverlayView(mContext);
         if (mView != null)
         {
            if (mOverlayViewWidth >=0 && mOverlayViewHeight >= 0)
               onOverlayViewSizeChanged(mOverlayViewWidth, mOverlayViewHeight);
         }
         return mView;
      }

      @Override
      public void onOverlayViewSizeChanged(int width, int height)
      {
         Log.i(TAG, "onOverlayViewSizeChanged " + width + ", " + height);

         mOverlayViewWidth = width;
         mOverlayViewHeight = height;
         if (mView != null)
         {
            Platform platform = new Platform();
            playerSetRectangle(platform.getSurfaceX(), platform.getSurfaceY(), width, height);
            mView.setSize(width, height);
         }
      }

      public boolean onPlayRecordedProgram(RecordedProgram recordedProgram) {
          Log.i(TAG, "onPlayRecordedProgram");
          boolean ret = false;
          int videoType = TvContractUtils.SOURCE_TYPE_INVALID;
          String videoUrl;
          if (null == recordedProgram) {
              // time shifting
              videoUrl = "crid://localhost/timeshift";
          }
          else {
              try {
                  InternalProviderData internalProviderData = recordedProgram.getInternalProviderData();
                  videoType = internalProviderData.getVideoType();
                  videoUrl = internalProviderData.getVideoUrl();
                  if (null == videoUrl) {
                      videoUrl = recordedProgram.getRecordingDataUri();
                      videoType = TvContractUtils.SOURCE_TYPE_INVALID;
                  }
              }
              finally {
              }
          }

          createPlayer(videoType, Uri.parse(videoUrl));
          try {
              mPlayer.start();
              if (mSeek) {
                  if (TvInputManager.TIME_SHIFT_INVALID_TIME != mTimeShiftResumePosition) {
                      onTimeShiftSeekTo(mTimeShiftResumePosition);
                      mTimeShiftResumePosition = TvInputManager.TIME_SHIFT_INVALID_TIME;
                  }
                  mSeek = false;
              }
              ret = true;
          }
          catch (Exception e) {
              Log.w(TAG, "mPlayer.start() failed !!", e);
          }
          finally {
              if (true == ret)
                  notifyTimeShiftStatusChanged(TvInputManager.TIME_SHIFT_STATUS_AVAILABLE);
              else
                  notifyTimeShiftStatusChanged(TvInputManager.TIME_SHIFT_STATUS_UNAVAILABLE);
          }

          return ret;
      }

      @Override
      public Uri getChannelUri() {
          return mChannelUri;
      }

      public ArrayList<Uri> getChannelUris() {
          return mChannelUris;
      }

      public TvPlayer getTvPlayer() {
          return mPlayer;
      }

      @Override
      public boolean onTune(Uri channelUri)
      {
         Log.i(TAG, "onTune " + channelUri);
         boolean ret;
         int onid = 0;
         int tsid = 0;
         mTimeShiftResumePosition = TvInputManager.TIME_SHIFT_INVALID_TIME;
         mSeek = false;
         Channel tunedChannel = getChannel(channelUri);
         if (null == tunedChannel) {
             notifyVideoUnavailable(TvInputManager.VIDEO_UNAVAILABLE_REASON_UNKNOWN);
             return false;
         }
         if (mTunedChannel == tunedChannel) {
             return true;
         }
         mChannelUri = channelUri;
         if (mTunedChannel != null) {
             onid = mTunedChannel.getOriginalNetworkId();
             tsid = mTunedChannel.getTransportStreamId();
             ret = mTransportManager.release(onid, tsid);
         }
         ret = mTransportManager.allocate (
             tunedChannel.getOriginalNetworkId(),
             tunedChannel.getTransportStreamId()
         );
         if (false == ret) {
             if (mTunedChannel != null)
                 mTransportManager.allocate(onid, tsid);
               
             notifyVideoUnavailable(TvInputManager.VIDEO_UNAVAILABLE_REASON_UNKNOWN);
             return false;  
         }
 
         mTunedChannel = tunedChannel;
         mSubtitleTrack = null;

         boolean isParentalControlsEnabled = mTvInputManager.isParentalControlsEnabled();
         if (isParentalControlsEnabled == true) {
             boolean isContentBlocked = false;
                
             List<Program> programs;
             programs = DtvkitEpgSync.getPresentFollowingProgramsForChannel(channelUri, mTunedChannel);
             long nowMs = System.currentTimeMillis();
             for (Program program : programs) {
                 if (program.getStartTimeUtcMillis() <= nowMs && program.getEndTimeUtcMillis() > nowMs) {
                     TvContentRating[] ratings = program.getContentRatings();
                     for (TvContentRating rating : ratings) {
                         if (mTvInputManager.isRatingBlocked(rating)) {
                             notifyContentBlocked(rating);
                             isContentBlocked = true;
                         }
                     }
                     break;
                 }
             }
             if (isContentBlocked == false) {
                 notifyContentAllowed();
             }
         }


         if (tunedChannel != null)
         {
            mPlayHandler.removeMessages(PlayHandler.ACTION_PLAY);
            mPlayHandler.removeMessages(PlayHandler.ACTION_STOP);
            Message message = mPlayHandler.obtainMessage();
            message.what = PlayHandler.ACTION_PLAY;
            mPlayHandler.sendMessage(message);
         }
         return tunedChannel != null;
      }

      @Override
      public boolean onTune(Uri channelUri, Bundle params) {
          Log.i(TAG, "onTune(...) " + channelUri);
          Channel newChannel = getChannel(channelUri);
          if (mChannelUri != null) {
              Channel currentChannel = getChannel(mChannelUri);          
              if (newChannel == currentChannel) {
                 notifyVideoAvailable();
                 notifyContentAllowed();
                 mSessionChanged.open();
                 return true;
              }
          }
          // channel URI
          mChannelUris.clear();
          mChannelUris.add(0, channelUri);

          // preload channel URIs
          if (params != null) {
              Parcelable[] parcelables = params.getParcelableArray("preload_channel_uri");
              if (parcelables != null) {
                  for (Parcelable p : parcelables) {
                      Uri uri = (Uri)p;
                      mChannelUris.add(uri);
                      Log.i(TAG, "preload channel uri: " + uri);
                  }
              }
          }
          return super.onTune(channelUri, params);
      }

      @Override
      public void onSetStreamVolume(float volume)
      {
         Log.i(TAG, "onSetStreamVolume " + volume);
         int max = mAudioManager.getStreamMaxVolume (AudioManager.STREAM_MUSIC);
         mAudioManager.setStreamVolume(AudioManager.STREAM_MUSIC, (int)(max * volume), AudioManager.FLAG_REMOVE_SOUND_AND_VIBRATE);
         playerSetVolume((int) (volume * 100));
      }

      @Override
      public void onSetCaptionEnabled(boolean enabled)
      {
         Log.i(TAG, "onSetCaptionEnabled " + enabled);
         // TODO CaptioningManager.getLocale()
         playerSetSubtitlesOn(enabled);
      }

      @Override
      public boolean onSelectTrack(int type, String trackId)
      {
         Log.i(TAG, "onSelectTrack " + type + ", " + trackId);
         if (type == TvTrackInfo.TYPE_AUDIO)
         {
            if (playerSelectAudioTrack((null == trackId) ? "" : trackId))
            {
               notifyTrackSelected(type, trackId);
               return true;
            }
         }
         else if (type == TvTrackInfo.TYPE_SUBTITLE)
         {
            if (playerSelectSubtitleTrack((null == trackId) ? "" : trackId))
            {
               // FIXME: for debugging
               {
                   Iterator it = mTunedTracks.iterator();
                   mSubtitleTrack = null;
                   while(it.hasNext()){
                       TvTrackInfo trackInfo = (TvTrackInfo)it.next();
                       if(trackInfo.getId().equals(trackId)) {
                           mSubtitleTrack = trackInfo;
                           break;
                       }
                   }
                   if(mCaptionEnabled == true) {
                       onSetCaptionEnabled(true);
                   }
               }
               notifyTrackSelected(type, trackId);
               return true;
            }
         }
         return false;
      }

      @Override
      public void onUnblockContent(TvContentRating unblockedRating)
      {
         // If unblockedRating is null then parental controls are off
         super.onUnblockContent(unblockedRating);
         Log.i(TAG, "onUnblockContent " + unblockedRating);

         unblockContent();
      }

      @Override
      public void onAppPrivateCommand(String action, Bundle data)
      {
         Log.i(TAG, "onAppPrivateCommand " + action + ", " + data);
      }

      public long onTimeShiftGetCurrentPosition() {
          long t = super.onTimeShiftGetCurrentPosition();
          if (null == mRecordedProgramUri) {
              if (null != mTimeShiftRecordingSession) {
                  if (TvInputManager.TIME_SHIFT_INVALID_TIME != t) {
                      t += mTimeShiftRecordingSession.getRecordingStartTime();
                  }
                  else {
                      if (TvInputManager.TIME_SHIFT_INVALID_TIME == mTimeShiftResumePosition) {
                          t = System.currentTimeMillis();
                      }
                      else {
                          // Live channel is paused
                          t = mTimeShiftResumePosition;
                      }
                  }
              }
          }
           Log.i(TAG, "onTimeShiftGetCurrentPosition  " + t);
          return t;
      }
        
      public long onTimeShiftGetStartPosition() {
          long t = super.onTimeShiftGetStartPosition();
          if (null == mRecordedProgramUri) {
              if (null != mTimeShiftRecordingSession) {
                  t = mTimeShiftRecordingSession.getRecordingStartTime();
                  long recycleTime = mTimeShiftRecordingSession.getRecordingRecycleTime();
                  long currentTimeMs = System.currentTimeMillis();
                  if ((t + recycleTime) < currentTimeMs) {
                      // time-shift recording start-time is moving
                      t = currentTimeMs - recycleTime;
                  }
              }
          }
          return t;
      }
        
      public void onTimeShiftPause() {
          Log.i(TAG, "onTimeShiftPause");
          if (null != mPlayer)
              mPlayer.pause();
          else {
              if (null != mTimeShiftRecordingSession) {
                  mTimeShiftResumePosition = System.currentTimeMillis();
                  String dvbUri = getChannelInternalDvbUri(mTunedChannel);
                  Log.i(TAG, "onTimeShiftPause pause the live channel ......");
                  playerPause(dvbUri, true);
              }
          }
      }

      public void onTimeShiftPlay(Uri recordedProgramUri) {
          Log.i(TAG, "onTimeShiftPlay " + recordedProgramUri);
          playerStop();
          releasePlayer();

          mRecordedProgramUri = recordedProgramUri;
          if (null == recordedProgramUri) {
              if (null == mTimeShiftRecordingSession) {
                  notifyTimeShiftStatusChanged(TvInputManager.TIME_SHIFT_STATUS_UNAVAILABLE);
                  notifyVideoUnavailable(TvInputManager.VIDEO_UNAVAILABLE_REASON_UNKNOWN);
                  return;
              }
          }
          super.onTimeShiftPlay(recordedProgramUri);
      }
        
      public void onTimeShiftResume() {
          Log.i(TAG, "onTimeShiftResume");
          if (null != mPlayer)
              mPlayer.start();
          else {
              // start time-shift playback
              // seek
              mSeek = true;
              onTimeShiftPlay(null);
          }
      }
        
      public void onTimeShiftSeekTo(long timeMs) {
          Log.i(TAG, "onTimeShiftSeekTo timeMs " + (new Timestamp(timeMs)).toString());
          if (null != mPlayer) {
              if (null == mRecordedProgramUri) {
                  if (null != mTimeShiftRecordingSession) {
                      long t = timeMs - mTimeShiftRecordingSession.getRecordingStartTime();
                      if (t < 0) {
                          t = 0;
                      }
                      mPlayer.seekTo(t);
                  }
                  return;
              }
          }

          super.onTimeShiftSeekTo(timeMs);
      }

      public void onTimeShiftSetPlaybackParams(PlaybackParams params) {
          Log.i(TAG, "onTimeShiftSetPlaybackParams");
          super.onTimeShiftSetPlaybackParams(params);
      }

      @Override
      public void notifyTimeShiftStatusChanged (int status) {
          super.notifyTimeShiftStatusChanged(status);
      }

      private final DtvkitGlueClient.SignalHandler mHandler = new DtvkitGlueClient.SignalHandler()
      {
         @Override
         public void onSignal(String signal, JSONObject data)
         {
            boolean serviceUpdated = false;
            
            // TODO notifyChannelRetuned(Uri channelUri)
            // TODO notifyTracksChanged(List<TvTrackInfo> tracks)
            // TODO notifyTimeShiftStatusChanged(int status)
            if (signal.equals("PlayerStatusChanged"))
            {
               String state = "off";
               try
               {
                  state = data.getString("state");
               }
               catch (JSONException ignore)
               {
               }
               Log.d(TAG, "onSignal: state= " + state);

               Message message = mPlayHandler.obtainMessage();
               Bundle bundle = new Bundle();
               bundle.putString(PlayHandler.KEY_STATE, state);
               message.setData(bundle);
               message.what = PlayHandler.ACTION_CHANGE_STATE;
               mPlayHandler.sendMessage(message);
            }
            else if (signal.equals("DvbUpdatedEventPeriods"))
            {
               Log.i(TAG, "DvbUpdatedEventPeriods");
               String inputId =
                  TvContract.buildInputId(new ComponentName(mContext, DtvkitTvInput.class));
               ComponentName sync = new ComponentName(mContext, DtvkitEpgSync.class);
               EpgSyncJobService.requestImmediateSync(mContext, inputId,
                  EpgSyncJobService.SYNC_MODE_PERIOD_UPDATE, false, sync);
            }
            else if (signal.equals("DvbUpdatedEventNow"))
            {
               Log.i(TAG, "DvbUpdatedEventNow");
               String inputId =
                  TvContract.buildInputId(new ComponentName(mContext, DtvkitTvInput.class));
               ComponentName sync = new ComponentName(mContext, DtvkitEpgSync.class);
               EpgSyncJobService.requestImmediateSync(mContext, inputId,
                  EpgSyncJobService.SYNC_MODE_NOW_NEXT, false, sync);
            }
            else if (signal.equals("ServiceDeleted"))
            {
               Log.d(TAG, "ServiceDeleted");
               String inputId =
                  TvContract.buildInputId(new ComponentName(mContext, DtvkitTvInput.class));
               ComponentName sync = new ComponentName(mContext, DtvkitEpgSync.class);

               // Update channel list only as deleting a channel will remove all programs
               // associated with it
               EpgSyncJobService.requestImmediateSync(mContext, inputId,
                  EpgSyncJobService.SYNC_MODE_CHANNEL_LIST, true, sync);

               serviceUpdated = true;
            }
            else if (signal.equals("CurrentServiceDeleted"))
            {
               Log.d(TAG, "CurrentServiceDeleted");
               String inputId =
                  TvContract.buildInputId(new ComponentName(mContext, DtvkitTvInput.class));
               ComponentName sync = new ComponentName(mContext, DtvkitEpgSync.class);

               // Update channel list only as deleting a channel will remove all programs
               // associated with it. By default Android will automatically request to tune to
               // first channel in channel list, if current channel is removed from channel list
               EpgSyncJobService.requestImmediateSync(mContext, inputId,
                  EpgSyncJobService.SYNC_MODE_CHANNEL_LIST, true, sync);
                  
               serviceUpdated = true;
            }
            else if (signal.equals("CurrentServiceMoved"))
            {
               Log.d(TAG, "CurrentServiceMoved");

               // Get dvb uri for the new channel and cache it
               try
               {
                  retuneDvbUri = data.getString("uri");
               }
               catch (JSONException e)
               {
                  retuneDvbUri = "";
                  Log.e(TAG, "Unable to get uri " + e.getMessage());
               }

               String inputId =
                  TvContract.buildInputId(new ComponentName(mContext, DtvkitTvInput.class));
               ComponentName sync = new ComponentName(mContext, DtvkitEpgSync.class);

               // After the channel list has been updated it will re-tune to the new channel
               // with the cached dvb uri
               EpgSyncJobService.requestImmediateSync(mContext, inputId,
                  EpgSyncJobService.SYNC_MODE_FULL_UPDATE, true, sync);

               serviceUpdated = true;
            }
            else if (signal.equals("ServiceAdded"))
            {
               Log.d(TAG, "ServiceAdded");
               String inputId =
                  TvContract.buildInputId(new ComponentName(mContext, DtvkitTvInput.class));
               ComponentName sync = new ComponentName(mContext, DtvkitEpgSync.class);

               EpgSyncJobService.requestImmediateSync(mContext, inputId,
                  EpgSyncJobService.SYNC_MODE_FULL_UPDATE, true, sync);
               
               serviceUpdated = true;
            }
            else if (signal.equals("ServiceUpdated"))
            {
               Log.d(TAG, "ServiceUpdated");

               String inputId =
                  TvContract.buildInputId(new ComponentName(mContext, DtvkitTvInput.class));
               ComponentName sync = new ComponentName(mContext, DtvkitEpgSync.class);

               EpgSyncJobService.requestImmediateSync(mContext, inputId,
                  EpgSyncJobService.SYNC_MODE_CHANNEL_LIST, true, sync);
               
               serviceUpdated = true;
            }
            
            if (serviceUpdated) {
               //try {
                  if (null != mTvInputInfo) {
                     Bundle extrasBundle = mTvInputInfo.getExtras();
                     if (null != extrasBundle) {
                         Bundle moduleBundle = extrasBundle.getBundle(mTvInputInfo.getId());
                         if (null != moduleBundle) {
                             String from = "si";
                             moduleBundle.putString("from", from);
                             Bundle siBundle = moduleBundle.getBundle(from);
                             if (null != siBundle) {
                                 String event = "update";
                                 //siBundle.clear();
                                 siBundle.putString("event", event);
                                 Bundle eventBundle = new Bundle();
                                 if (null != eventBundle) {
                                     siBundle.putBundle(event, eventBundle);
                                 }
                             }  // if (null != siBundle)
                          }  // if (null != moduleBundle)
                      }  // if (null != extrasBundle)
                      mTvInputManager.updateTvInputInfo (mTvInputInfo);
                  }  //  if (null != mTvInputInfo)
               /*
               }
               catch (JSONException ignore) {

               }
               */
            }
         }
      };

      private void checkCurrentProgramContent()
      {
         // Get the content ratings for current program
         TvContentRating currentContentRatings[] = null;
         if ((mCurrentProgram != null) && (mCurrentProgram.getContentRatings() != null) &&
            (mCurrentProgram.getContentRatings().length > 0))
         {
            currentContentRatings = mCurrentProgram.getContentRatings();
         }

         // Check if any of the content ratings should be blocked
         if (currentContentRatings == null)
         {
            Log.d(TAG,
               "Content Ratings for current program are null or parental controls are disabled");
            mBlockedRating = null;
         }
         else
         {
            mBlockedRating = null;
            for (TvContentRating contentRating : currentContentRatings)
            {
               if (mTvInputManager.isRatingBlocked(contentRating))
               {
                  mBlockedRating = contentRating;
               }
            }

            if (mBlockedRating == null)
            {
               Log.d(TAG, "Do not need to block current content rating");
            }
            else
            {
               Log.d(TAG, "block content, rating: " + mBlockedRating.flattenToString());
            }
         }
      }

      private void blockCurrentProgramIfNeeded()
      {
         if (mBlockedRating == null)
         {
            unblockContent();
         }
         else
         {
            playerStopDecoding();
            notifyContentBlocked(mBlockedRating);
         }
      }

      private void unblockContent()
      {
         playerStartDecoding();
         notifyContentAllowed();
      }

      private void onCurrentProgramChange(Program updatedProgram)
      {
         Program cachedProgram = mCurrentProgram;
         mCurrentProgram = updatedProgram;

         if (mTvInputManager.isParentalControlsEnabled())
         {
            if ((cachedProgram == null) && (mCurrentProgram != null))
            {
               checkCurrentProgramContent();
               blockCurrentProgramIfNeeded();
            }
            else if (((cachedProgram != null) || (mCurrentProgram != null)) &&
               (!cachedProgram.equals(mCurrentProgram)))
            {
               checkCurrentProgramContent();
               blockCurrentProgramIfNeeded();
            }
         }
      }

      private void onProgramChange(final Uri uri)
      {
         if (uri != null)
         {
            if (mCurrentProgram == null)
            {
               onCurrentProgramChange(getCurrentProgram(mCurrentChannelUri));
            }
            else
            {
               Program program = getProgram(uri);

               if ((program != null) && (getProgramInternalDvbUri(program)
                  .equals(getProgramInternalDvbUri(mCurrentProgram))))
               {
                  onCurrentProgramChange(program);
               }
            }
         }
      }

      private void checkCurrentChannelRetuned()
      {
         // Check if retuneDvbUri is set and if it is then find the Uri for that channel
         if ((retuneDvbUri != null) && (!retuneDvbUri.isEmpty()) && (mChannels != null))
         {
            int size = mChannels.size();

            for (int i = 0; i < size; i++)
            {
               Channel channel;
               String dvbUri = "";

               channel = mChannels.valueAt(i);
               if (channel != null) {
                  try
                  {
                     dvbUri = channel.getInternalProviderData().get("dvbUri").toString();
                  }
                  catch (InternalProviderData.ParseException e)
                  {
                     Log.e(TAG, "Unable to get dvbUri for channel " + channel.toString() + ", " +
                        e.getMessage());
                  }

                  if (!dvbUri.isEmpty() && dvbUri.equals(retuneDvbUri))
                  {
                     notifyChannelRetuned(TvContract.buildChannelUri(channel.getId()));
                     break;
                  }
               }
            }

            retuneDvbUri = null;
         }
      }

      private void onChannelListChange()
      {
         // Check if current channel has been re-tuned and notify Android if it has
         checkCurrentChannelRetuned();
      }

      private void createPlayer(int videoType, Uri videoUrl) {
          releasePlayer();
          mPlayer = DtvkitTvPlayer.create(videoType, mContext, videoUrl);
          //mPlayer.prepare();
      }

      private void releasePlayer() {
          if (mPlayer != null) {
              mPlayer.setSurface(null);
              mPlayer.stop();
              mPlayer.reset();
              mPlayer = null;
          }
      }
   }

   public class DtvkitTvInputRecordingSession extends BaseTvInputService.RecordingSession {
       private static final String TAG = "DtvkitTvInputRecordingSession";  //RecordingSession.class.getCanonicalName();
       private static final boolean DEBUG = true;
       private static final String RECYCLE_TIME = "recycle-time";
       private static final String SSU = "ssu";
       private Context mContext;
       private String mInputId;
       private long mStartTimeMs;
       private long mRecordingDuration;  // unit: millisecond
       private long mRecordingSize;  // unit: byte
       private Channel mTunedChannel;
       private Uri mChannelUri;
       private Uri mProgramUri;
       private long mRecordingHandle;
       private long mRecycleTime;  // unit: millisecond, <greater than zero>: time shifting recording, others: normal recording
       private String mSsu;
       private boolean mIsRecording;
       private boolean mRecordingStoped;
       // fake SSU download session {
       private Timer mSsuTimer;
       private SsuTimerTask mSsuTimerTask;
       private int mPercent;

       public class SsuTimerTask extends TimerTask {
           public void run() {
               if (null != mTvInputInfo) {
                   Bundle extrasBundle = mTvInputInfo.getExtras();
                   if (null != extrasBundle) {
                       Bundle moduleBundle = extrasBundle.getBundle(mTvInputInfo.getId());
                       if (null != moduleBundle) {
                           String from = "ssu";
                           moduleBundle.putString("from", from);
                           Bundle ssuBundle = moduleBundle.getBundle(from);
                           if (null != ssuBundle) {
                               //ssuBundle.clear();
                               ssuBundle.putString("event", "download");
                               Bundle eventBundle = new Bundle();
                               if (null != eventBundle) {
                                   try {
                                       JSONArray args = new JSONArray();
                                       JSONObject data = DtvkitGlueClient.getInstance().request("Recorder.getSsuStatus", args).getJSONObject("data");
                                       String ssuInfo;
                                       try {
                                           ssuInfo = data.getString("ssu");
                                           mPercent = data.getInt("progress");
                                       }
                                       catch (JSONException e) {
                                           throw new RuntimeException(e);
                                       }
                                   }
                                   catch (Exception e) {
                                       Log.e(TAG, e.getMessage());
                                   }

                                   if (mPercent >= 100) {
                                       mPercent = 100;
                                       mSsuTimer.cancel();
                                   }
                                   eventBundle.putInt("percent", mPercent);
                                   ssuBundle.putBundle("download", eventBundle);
                               }
                           }
                        }
                    }
                    mTvInputManager.updateTvInputInfo (mTvInputInfo);
                }  // if (null != mTvInputInfo)
           }
       };
       // fake SSU download session }

       public DtvkitTvInputRecordingSession(Context context, String inputId) {
           super(context, inputId);
           Log.i(TAG, "DtvkitTvInputRecordingSession");
           mContext = context;
           mInputId = inputId;
           mStartTimeMs = 0;
           mRecordingDuration = 0;
           mRecordingSize = 0;
           mRecordingHandle = 0;
           mRecycleTime = 0;
           mSsu = String.valueOf("");
           mIsRecording = false;
           mRecordingStoped = true;
           synchronized(mRecordingSessions) {
               mRecordingSessions.add(this);
           }
           mSessionChanged.open();
           try {
             Thread.sleep(500);
           } catch (InterruptedException e) {
           }
           
           if (FAKE_SSU_DOWNLOAD) {
               // fake SSU download session
               mSsuTimer = null;
               mSsuTimerTask = null;
               mPercent = 0;
           }
       }

       public long getRecordingRecycleTime() {
           return mRecycleTime;
       }

       public long getRecordingStartTime() {
           return mStartTimeMs;
       }

       public void notifyTuned(Uri channelUri) {
	       super.notifyTuned(channelUri);
           if (DEBUG) Log.i(TAG, "notifyTuned");
       }
       public void notifyRecordingStopped(final Uri recordedProgramUri) {
           super.notifyRecordingStopped(recordedProgramUri);
           if (DEBUG) Log.i(TAG, "notifyRecordingStopped");
       }
       public void notifyError(final int error) {
           super.notifyError(error);
           Log.i(TAG, "notifyError");
       }
       public void notifySessionEvent(final String eventType, final Bundle eventArgs) {
         super.notifySessionEvent(eventType, eventArgs);
         if (DEBUG) Log.i(TAG, "notifySessionEvent");
       }

       @Override
       public void onTune(Uri channelUri) {
           Log.i(TAG, "onTune() " + channelUri);
           boolean ret;
           int onid = 0;
           int tsid = 0;
           Channel channel = getChannel(channelUri);
           if (null == channel) {
               notifyError(TvInputManager.RECORDING_ERROR_UNKNOWN);
               return;
           }
           if (mTunedChannel == channel) {
               notifyTuned(channelUri);
               return;
           }
           if (mTunedChannel != null) {
               onid = mTunedChannel.getOriginalNetworkId();
               tsid = mTunedChannel.getTransportStreamId();
               ret = mTransportManager.release(onid, tsid);
           }
           ret = mTransportManager.allocate (
               channel.getOriginalNetworkId(),
               channel.getTransportStreamId()
           );
           if (false == ret) {
               if (mTunedChannel != null)
                   mTransportManager.allocate(onid, tsid);
             
               notifyError(TvInputManager.RECORDING_ERROR_RESOURCE_BUSY);
               return;  
           }

           String dvbUri = getChannelInternalDvbUri(channel);
           recorderTune(dvbUri);

           mChannelUri = channelUri;
           mTunedChannel = channel;
           if (mRecycleTime > 0) {
               // time-shift recording
               if ((null != mTimeShiftRecordingSession) && (this != mTimeShiftRecordingSession)) {
                   Log.e(TAG, "A time-shift recording session exists for " + mChannelUri + " !!");
               }
               mTimeShiftRecordingSession = this;                
           }
           else {
               // normal recording
               if ((null != mRecordingSession) && (this != mRecordingSession)) {
                   Log.w(TAG, "A normal recording session exists for " + mChannelUri + " !!");
               }
               mRecordingSession = this;
           }
           super.onTune(channelUri);
           if (DEBUG) Log.d(TAG, "Tune recording session to " + channelUri);
           // By default, the number of tuners for this service is one. When a channel is being
           // recorded, no other channel from this TvInputService will be accessible. Developers
           // should call notifyError(TvInputManager.RECORDING_ERROR_RESOURCE_BUSY) to alert
           // the framework that this recording cannot be completed.
           // Developers can update the tuner count in xml/richtvinputservice or programmatically
           // by adding it to TvInputInfo.updateTvInputInfo.
           notifyTuned(channelUri);
       }

       @Override
       public void onTune(Uri channelUri, Bundle params) {
           if (null != params) {
               mRecycleTime = params.getLong(RECYCLE_TIME, (short)0);
               mSsu = params.getString(SSU);
           }
           Log.i(TAG, "onTune(...) " + channelUri + ", recycle-time: " + mRecycleTime + " ms, ssu: " + mSsu);
           try {
               Thread.sleep(1000);
           } catch (Exception e) {
                Log.e(TAG, e.getMessage());
           }
           super.onTune(channelUri, params);
       }
       @Override
       public void onStartRecording(Uri programUri) {
           if (DEBUG) Log.d(TAG, "onStartRecording " + programUri);
           if (mIsRecording)
               return;

           if (null == programUri) {
               if (DEBUG) Log.i(TAG, "onStartRecording channel recording");
           }
           String dvbUri = getChannelInternalDvbUri(mTunedChannel);
           mRecordingHandle = recorderStartRecording(dvbUri);
           Log.d(TAG, "mRecordingHandle = " + mRecordingHandle);
           if (0 == mRecordingHandle) {
               if ((null != mSsu) && mSsu.equals("download")) {
                   if (FAKE_SSU_DOWNLOAD) {
                       // fake SSU download session
                       mPercent = 0;
                       mSsuTimer = new Timer();
                       if (null != mSsuTimer) {
                           mSsuTimer.scheduleAtFixedRate(new SsuTimerTask(), 0, 2000);
                       }
                   } // if (FAKE_SSU_DOWNLOAD)
                   // ignore this in SSU download
               }
               else {
                   notifyError(TvInputManager.RECORDING_ERROR_UNKNOWN);
                   return;
               }
           }
           mStartTimeMs = System.currentTimeMillis();
           mRecordingDuration = 0;
           mRecordingSize = 0;
           mProgramUri = programUri;
           if (mRecycleTime > 0) {
               mProgramUri = null;
           }
           super.onStartRecording(mProgramUri);
	       if (mRecycleTime > 0) {
               if (0 != mRecordingHandle) {
                    if (null != mSession) {
                        Uri playbackChannelUri = mSession.getChannelUri();
                        if (mChannelUri.equals(playbackChannelUri)) {
                            mSession.notifyTimeShiftStatusChanged(TvInputManager.TIME_SHIFT_STATUS_AVAILABLE);
                        }
                    }
                }
           }
           mIsRecording = true;
           mRecordingStoped = false;
       }
       @Override
       public void onStopRecording() {
           if (DEBUG) Log.i(TAG, "onStopRecording");
           if (false == mIsRecording)
               return;

           mIsRecording = false;
           if ((0 == mRecordingHandle) && (0 == mRecycleTime)) {
               if ((null != mSsu) && mSsu.equals("download")) {
                   if (FAKE_SSU_DOWNLOAD) {
                       if (null != mSsuTimer) {
                           mSsuTimer.cancel();
                           mSsuTimer = null;
                       }
                   } // if (FAKE_SSU_DOWNLOAD)
                   // ignore this in SSU download
               }
               else {
                   notifyError(TvInputManager.RECORDING_ERROR_UNKNOWN);
                   return;
               }
           }

           String programUri = null;
           if (null != mProgramUri) {
               programUri = mProgramUri.toString();
           }
           recorderStopRecording(programUri);
           super.onStopRecording();
           mProgramUri = null;
       }
       @Override
       public void onStopRecording(Program programToRecord) {
           if (DEBUG) Log.d(TAG, "onStopRecording");
           // In this sample app, since all of the content is VOD, the video URL is stored.
           // If the video was live, the start and stop times should be noted using
           // RecordedProgram.Builder.setStartTimeUtcMillis and .setEndTimeUtcMillis.
           // The recordingstart time will be saved in the InternalProviderData.
           // Additionally, the stream should be recorded and saved as
           // a new file.
           long currentTime = System.currentTimeMillis();
           InternalProviderData internalProviderData = programToRecord.getInternalProviderData();
           String recordingDataUri = RECORDING_DATA_URI_BASE;
           recordingDataUri += String.valueOf(mRecordingHandle);
           internalProviderData.setRecordingStartTime(mStartTimeMs);
           RecordedProgram recordedProgram = new RecordedProgram.Builder(programToRecord)
               .setInputId(mInputId)
               .setRecordingDataUri(recordingDataUri)
               .setRecordingDataBytes(mRecordingSize)
               .setRecordingDurationMillis(mRecordingDuration)
               .setInternalProviderData(internalProviderData)
               .build();
           notifyRecordingStopped(recordedProgram);
           mRecordingStoped = true;
           mRecordingHandle = 0;
       }
       @Override
       public void onStopRecordingChannel(Channel channelToRecord) {
           if (DEBUG) Log.d(TAG, "onStopRecordingChannel");
           if (null == channelToRecord) {
               if (DEBUG) Log.d(TAG, "onStopRecordingChannel unknown channel !!");
               return;
           }

           RecordedProgram recordedProgram = null;
           if ((mRecycleTime <= 0) && ("" == mSsu)) {
               long currentTime = System.currentTimeMillis();
               InternalProviderData internalProviderData = new InternalProviderData();
               try {
                   String dvbUri = channelToRecord.getInternalProviderData().get("dvbUri").toString();
                   internalProviderData.put("dvbUri", dvbUri);
               }
               catch (InternalProviderData.ParseException e) {
         
               }
               String recordingDataUri = RECORDING_DATA_URI_BASE;
               recordingDataUri += String.valueOf(mRecordingHandle);
               internalProviderData.setRecordingStartTime(mStartTimeMs);

               TvContentRating[] contentRatings = new TvContentRating[1];
               contentRatings[0] = TvContentRating.UNRATED;
         
               recordedProgram = new RecordedProgram.Builder()
                   .setInputId(mInputId)
                   .setRecordingDataUri(recordingDataUri)
                   .setRecordingDataBytes(mRecordingSize)
                   .setRecordingDurationMillis(mRecordingDuration)
                   .setInternalProviderData(internalProviderData)                        
                   .setChannelId(channelToRecord.getId())
                   .setTitle(channelToRecord.getDisplayName())
                   .setDescription(channelToRecord.getDescription())
                   .setLongDescription(channelToRecord.getDescription())
                   .setContentRatings(contentRatings)
                   .setStartTimeUtcMillis(mStartTimeMs)
                   .setEndTimeUtcMillis(mStartTimeMs + mRecordingDuration)
                   .build();
           }
           notifyRecordingStopped(recordedProgram);
           mRecordingStoped = true;
           mRecordingHandle = 0;
       }
       public void onRelease() {
           if (DEBUG) Log.i(TAG, "onRelease");
           onStopRecording();
           while (false == mRecordingStoped) {
               // wait for the completion of stopRecording() 
               try {
                   // must not exceeding EXECUTE_MESSAGE_TIMEOUT_SHORT_MILLIS (50ms)
                   Thread.sleep(10);
               } catch (InterruptedException e) {
               }
           }
         
           if (mTunedChannel != null) {
               int onid = mTunedChannel.getOriginalNetworkId();
               int tsid = mTunedChannel.getTransportStreamId();
               boolean ret = mTransportManager.release(onid, tsid);
               mTunedChannel = null;
           }
           synchronized(mRecordingSessions) {
               mRecordingSessions.remove(this);
           }
           if (this == mTimeShiftRecordingSession) {
               if (null != mSession) {
                   mSession.notifyTimeShiftStatusChanged(TvInputManager.TIME_SHIFT_STATUS_UNAVAILABLE);
               }
               mTimeShiftRecordingSession = null;
           }
           if (this == mRecordingSession) {
               mRecordingSession = null;
           }
           mRecycleTime = 0;

           mDisableFcc = false;
           if (mSession != null) {
              ArrayList<Uri> channelUris = mSession.getChannelUris();
              playerPlay(channelUris);
           }
       }
       @Override
       public void onAppPrivateCommand(String action, Bundle data) {
           if (DEBUG) Log.i(TAG, "onAppPrivateCommand");
       }

       private final DtvkitGlueClient.SignalHandler mHandler = new DtvkitGlueClient.SignalHandler() {
           @Override
           public void onSignal(String signal, JSONObject data) {
               // TODO notifyTimeShiftStatusChanged(int status)
               if (signal.equals("RecorderStatusChanged")) {
                   String state = "off";
                   try {
                       state = data.getString("state");
                   } catch (JSONException ignore) {
                   }
                   switch (state) {
                       case "tuned":
                           break;
                       case "recording":
                           break;
                       case "blocked":
                           //notifyContentBlocked(TvContentRating.createRating("com.android.tv", "DVB", "DVB_18"));
                           break;
                       case "badsignal":
                           //notifyError(TvInputManager.VIDEO_UNAVAILABLE_REASON_WEAK_SIGNAL);
                           break;
                       default:
                           break;
                   }
               }
           }
       };

       // recorder
       private boolean recorderTune(String dvbUri) {
           if (DEBUG) Log.d(TAG, "recorderTune " + dvbUri);
           if (mRecycleTime > 0) {
               // ignore this for time-shifting
               return true;
           }

           if (null != mSsu) {
               if (mSsu.equals("download")) {
                   // ignore this for SSU data download
                   return true;
               }
           } 

           try {
               JSONArray args = new JSONArray();
               args.put(dvbUri);
               DtvkitGlueClient.getInstance().request("Recorder.tune", args);
               return true;
           } catch (Exception e) {
               Log.e(TAG, e.getMessage());
               return false;
           }
       }

       private long recorderStartRecording(String dvbUri) {
           if (DEBUG) Log.d(TAG, "recorderStartRecording " + dvbUri);
           long recordingHandle = 0;
           try {
               JSONArray args = new JSONArray();
               args.put(dvbUri);
               args.put(Long.valueOf(mRecycleTime));
               args.put(String.valueOf(mSsu));
               JSONObject response = DtvkitGlueClient.getInstance().request("Recorder.startRecording", args).getJSONObject("data");
               try {
                   recordingHandle = response.getLong("handle");
               }
               catch (JSONException e) {
                   throw new RuntimeException(e);
               }
           }
           catch (Exception e) {
               Log.e(TAG, e.getMessage());
           }

           return recordingHandle;
       }
 
       private boolean recorderStopRecording(String dvbUri) {
           if (DEBUG) Log.d(TAG, "recorderStopRecording " + dvbUri);
           try {
               JSONArray args = new JSONArray();
               args.put(dvbUri);
               args.put(Long.valueOf(mRecycleTime));
               args.put(String.valueOf(mSsu));
               JSONObject response = DtvkitGlueClient.getInstance().request("Recorder.stopRecording", args).getJSONObject("data");
               try {
                   mRecordingDuration = response.getLong("duration");  // second
                   mRecordingDuration *= 1000;  // millisecond
                   mRecordingSize = response.getLong("size");  // KB
                   mRecordingSize *= 1000;  // byte
               }
               catch (JSONException e) {
                   throw new RuntimeException(e);
               }
               return true;
           } catch (Exception e) {
               Log.e(TAG, e.getMessage());
               return false;
           }
       }
   }  // RecordingSession

   private void onChannelsChanged()
   {
      mChannels = TvContractUtils.buildChannelMap(mContentResolver,
         TvContract.buildInputId(new ComponentName(getApplicationContext(), DtvkitTvInput.class)));
   }

   private void setupChannelObserver()
   {
      // Query the Android database in the database thread
      mChannelObserver = new ContentObserver(new Handler(mDbHandlerThread.getLooper()))
      {
         @Override
         public void onChange(boolean selfChange)
         {
            onChannelsChanged();

            for (Session session : mSessions)
            {
               if (session instanceof DtvkitTvInputSession)
               {
                  ((DtvkitTvInputSession) session).onChannelListChange();
               }
            }
         }
      };
   }

   private void setupProgramObserver()
   {
      // Query the Android database in the database thread
      mProgramObserver = new ContentObserver(new Handler(mDbHandlerThread.getLooper()))
      {
         @Override
         public void onChange(boolean selfChange)
         {
            onChange(selfChange, null);
         }

         @Override
         public void onChange(boolean selfChange, Uri uri)
         {
            synchronized(mSessions) {
               for (Session session : mSessions)
               {
                  if (session instanceof DtvkitTvInputSession)
                  {
                     ((DtvkitTvInputSession) session).onProgramChange(uri);
                  }
               }
            }
         }
      };
   }

   private void setupParentalControlsBroadcastReceiver()
   {
      IntentFilter intentFilter = new IntentFilter();
      intentFilter.addAction(TvInputManager.ACTION_BLOCKED_RATINGS_CHANGED);
      intentFilter.addAction(TvInputManager.ACTION_PARENTAL_CONTROLS_ENABLED_CHANGED);
      registerReceiver(mParentalControlsBroadcastReceiver, intentFilter);
   }

   private void updateDefaultLanguage()
   {
      String langCode = Locale.getDefault().getISO3Language();
      String defaultAudioLangCode = playerGetDefaultAudioLanguageCode();
      String defaultSubtitlesLangCode = playerGetDefaultSubtitlesLanguageCode();
      Log.d(TAG, "System lang code: " + langCode);

      if ("zho".equals(langCode)) {
          langCode = "chi";
      }
 
      //FIXME: workaround for selinux failure
      //String audioLangCode = SystemProperties.get("persist.vendor.dvb.audio_language");
      String audioLangCode = "";
      if (audioLangCode.equals("")) {
          audioLangCode = "chi";
      }
      if (playerSetDefaultAudioLanguage(audioLangCode))
      {
         Log.d(TAG, "Default audio language changed from " + defaultAudioLangCode +
            " to " + audioLangCode);
      }

      if (playerSetDefaultSubtitlesLanguage(langCode))
      {
         Log.d(TAG, "Default subtitles language changed from " + defaultSubtitlesLangCode +
            " to " + langCode);
         mSyncEpg = true;
      }
   }

   private Channel getChannel(Uri channelUri)
   {
      Channel channel = null;

      if (mChannels != null)
         channel = mChannels.get(ContentUris.parseId(channelUri));
      
      return channel;
   }

   private String getChannelInternalDvbUri(Channel channel)
   {
      if (channel != null) {
         try
         {
            return channel.getInternalProviderData().get("dvbUri").toString();
         }
         catch (Exception e)
         {
            Log.e(TAG, e.getMessage());
            return "dvb://0000.0000.0000";
         }
      }
      return "dvb://0000.0000.0000";
   }

   private Program getCurrentProgram(Uri channelUri)
   {
      return TvContractUtils
         .getCurrentProgram(getApplicationContext().getContentResolver(), channelUri);
   }

   private Program getProgram(Uri programUri)
   {
      Program program = null;

      if (programUri != null)
      {
         Cursor cursor = null;
         try {
            cursor = mContentResolver.query(programUri, Program.PROJECTION, null, null, null);
            if ((cursor != null) && (cursor.moveToNext())) {
               program = Program.fromCursor(cursor);
            }
         }
         catch (Exception e) {
            Log.w(TAG, "Unable to get the program for " + programUri, e);
         }
         finally {
            if (cursor != null)
               cursor.close();
         }
      }

      return program;
   }

   private String getProgramInternalDvbUri(Program program)
   {
      String dvbUri;

      try
      {
         InternalProviderData internalProviderData = program.getInternalProviderData();
         if (internalProviderData != null)
         {
            dvbUri = internalProviderData.get("dvbUri").toString();
         }
         else
         {
            dvbUri = "dvb://0000.0000.0000.0000";
         }
      }
      catch (Exception exception)
      {
         Log.e(TAG, exception.getMessage());
         dvbUri = "dvb://0000.0000.0000.0000";
      }

      return dvbUri;
   }

   private void playerSetVolume(int volume)
   {
      try
      {
         JSONArray args = new JSONArray();
         args.put(volume);
         DtvkitGlueClient.getInstance().request("Player.setVolume", args);
      }
      catch (Exception e)
      {
         Log.e(TAG, e.getMessage());
      }
   }

   private void playerSetSubtitlesOn(boolean on)
   {
      try
      {
         JSONArray args = new JSONArray();
         args.put(on);
         DtvkitGlueClient.getInstance().request("Player.setSubtitlesOn", args);
      }
      catch (Exception e)
      {
         Log.e(TAG, e.getMessage());
      }
   }

   private boolean playerPause(String dvbUri, boolean pause) {
       try {
           JSONArray args = new JSONArray();
           args.put(dvbUri);
           args.put(pause);
           DtvkitGlueClient.getInstance().request("Player.pause", args);
           return true;
       } catch (Exception e) {
           Log.e(TAG, e.getMessage());
           return false;
       }
   }

   private boolean playerPlay(ArrayList<Uri> channelUris)
   {
      int size = channelUris.size();
      Uri[] uris = new Uri[channelUris.size()];
      channelUris.toArray(uris);
      Log.i(TAG, "channelUris   " + Arrays.toString(uris));

      boolean fcc = true;
      String sBouquetId = SystemProperties.get("persist.vendor.dvb.bid");
      if ((sBouquetId != null) && sBouquetId.equals("99")) {
         fcc = false;
      }
       
      //if (mCanRecord)
        // fcc = false;

       if (mDisableFcc) {
          fcc = false;
       }

      try
      {
         JSONArray args = new JSONArray();
         for(int i = 0; i < size; i++) {
            Channel channel = getChannel(channelUris.get(i));
            String dvbUri = getChannelInternalDvbUri(channel);
            args.put(dvbUri);
            if (fcc == false)
               break;
         }
         Log.i(TAG, "JSONArray   " + args.toString());
         DtvkitGlueClient.getInstance().request("Player.play", args);
         return true;
      }
      catch (Exception e)
      {
         Log.e(TAG, e.getMessage());
         return false;
      }
   }

   private void playerStop()
   {
      try
      {
         JSONArray args = new JSONArray();
         DtvkitGlueClient.getInstance().request("Player.stop", args);
      }
      catch (Exception e)
      {
         Log.e(TAG, e.getMessage());
      }
   }

   private void playerSetRectangle(int x, int y, int width, int height)
   {
      try
      {
         JSONArray args = new JSONArray();
         args.put(x);
         args.put(y);
         args.put(width);
         args.put(height);
         DtvkitGlueClient.getInstance().request("Player.setRectangle", args);
      }
      catch (Exception e)
      {
         Log.e(TAG, e.getMessage());
      }
   }

   private List<TvTrackInfo> playerGetTracks()
   {
      List<TvTrackInfo> tracks = new ArrayList<>();
      try
      {
         JSONArray args = new JSONArray();
         JSONArray audioStreams =
            DtvkitGlueClient.getInstance().request("Player.getListOfAudioStreams", args)
               .getJSONArray("data");
         for (int i = 0; i < audioStreams.length(); i++)
         {
            JSONObject audioStream = audioStreams.getJSONObject(i);
            String language = audioStream.getString("language");
            String trackId = (Integer.toString(audioStream.getInt("pid")) + ":" + language);
            Log.d(TAG, "playerGetTracks trackId[" + i + "] " + trackId);

            TvTrackInfo.Builder track = new TvTrackInfo.Builder(TvTrackInfo.TYPE_AUDIO, trackId);
            track.setLanguage(language);
            if (audioStream.getBoolean("ad"))
            {
               if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M)
               {
                  track.setDescription("AD");
               }
            }
            tracks.add(track.build());
         }
      }
      catch (Exception e)
      {
         Log.e(TAG, e.getMessage());
      }
      try
      {
         JSONArray args = new JSONArray();
         JSONArray subtitleStreams =
            DtvkitGlueClient.getInstance().request("Player.getListOfSubtitleStreams", args)
               .getJSONArray("data");
         for (int i = 0; i < subtitleStreams.length(); i++)
         {
            JSONObject subtitleStream = subtitleStreams.getJSONObject(i);
            String language = subtitleStream.getString("language");
            String trackId = (Integer.toString(subtitleStream.getInt("pid")) + ":" + language);
            Log.d(TAG, "playerGetTracks trackId[" + i + "] " + trackId);

            TvTrackInfo.Builder track = new TvTrackInfo.Builder(TvTrackInfo.TYPE_SUBTITLE, trackId);
            track.setLanguage(language);
            tracks.add(track.build());
         }
      }
      catch (Exception e)
      {
         Log.e(TAG, e.getMessage());
      }
      return tracks;
   }

   private boolean playerSelectAudioTrack(String trackId)
   {
      try
      {
         int pid = 0xFFFF;
         String lang_code = "";
         if (!trackId.isEmpty())
         {
            String arg_list[] = trackId.split(":", 2);
            pid = Integer.parseInt(arg_list[0]);
            lang_code = arg_list[1];
         }
         JSONArray args = new JSONArray();
         args.put(pid);
         args.put(lang_code);
         DtvkitGlueClient.getInstance().request("Player.setAudioStream", args);
      }
      catch (Exception e)
      {
         Log.e(TAG, e.getMessage());
         return false;
      }
      return true;
   }

   private boolean playerSelectSubtitleTrack(String trackId)
   {
      try
      {
         int pid = 0xFFFF;
         String lang_code = "";
         if (!trackId.isEmpty())
         {
            String arg_list[] = trackId.split(":", 2);
            pid = Integer.parseInt(arg_list[0]);
            lang_code = arg_list[1];
         }
         JSONArray args = new JSONArray();
         args.put(pid);
         args.put(lang_code);
         DtvkitGlueClient.getInstance().request("Player.setSubtitleStream", args);
      }
      catch (Exception e)
      {
         Log.e(TAG, e.getMessage());
         return false;
      }
      return true;
   }

   private String playerGetSelectedSubtitleTrack()
   {
      String trackId = "invalid";
      try
      {
         JSONArray args = new JSONArray();
         JSONArray subtitleStreams =
            DtvkitGlueClient.getInstance().request("Player.getListOfSubtitleStreams", args)
               .getJSONArray("data");
         for (int i = 0; i < subtitleStreams.length(); i++)
         {
            JSONObject subtitleStream = subtitleStreams.getJSONObject(i);
            if (subtitleStream.getBoolean("selected"))
            {
               int pid = subtitleStream.getInt("pid");
               String lang_code = subtitleStream.getString("language");

               trackId = Integer.toString(pid) + ":" + lang_code;
               break;
            }
         }
      }
      catch (Exception e)
      {
         Log.e(TAG, e.getMessage());
      }
      return trackId;
   }

   private String playerGetSelectedAudioTrack()
   {
      String trackId = "invalid";
      try
      {
         JSONArray args = new JSONArray();
         JSONArray audioStreams =
            DtvkitGlueClient.getInstance().request("Player.getListOfAudioStreams", args)
               .getJSONArray("data");
         for (int i = 0; i < audioStreams.length(); i++)
         {
            JSONObject audioStream = audioStreams.getJSONObject(i);
            if (audioStream.getBoolean("selected"))
            {
               int pid = audioStream.getInt("pid");
               String lang_code = audioStream.getString("language");

               trackId = Integer.toString(pid) + ":" + lang_code;
               break;
            }
         }
      }
      catch (Exception e)
      {
         Log.e(TAG, e.getMessage());
      }
      return trackId;
   }

   private boolean playerGetSubtitlesOn()
   {
      boolean on = false;
      try
      {
         JSONArray args = new JSONArray();
         on = DtvkitGlueClient.getInstance().request("Player.getSubtitlesOn", args)
            .getBoolean("data");
      }
      catch (Exception e)
      {
         Log.e(TAG, e.getMessage());
      }
      return on;
   }

   private String playerGetDefaultAudioLanguageCode()
   {
      String code = "";
      JSONArray args = new JSONArray();
      try
      {
         code =
            DtvkitGlueClient.getInstance().request("Player.getDefaultAudioLanguage", args)
               .getJSONObject("data").getString("code");
      }
      catch (Exception e)
      {
         Log.e(TAG, e.getMessage());
      }

      return code;
   }

   private boolean playerSetDefaultAudioLanguage(String code)
   {
      boolean success = false;
      JSONArray args = new JSONArray();
      args.put(code);
      try
      {
         success = DtvkitGlueClient.getInstance().request("Player.setDefaultAudioLanguage", args)
            .getBoolean("data");
      }
      catch (Exception e)
      {
         Log.e(TAG, e.getMessage());
      }

      return success;
   }

   private String playerGetDefaultSubtitlesLanguageCode()
   {
      String code = "";
      JSONArray args = new JSONArray();
      try
      {
         code =
            DtvkitGlueClient.getInstance().request("Player.getDefaultSubtitlesLanguage", args)
               .getJSONObject("data").getString("code");
      }
      catch (Exception e)
      {
         Log.e(TAG, e.getMessage());
      }

      return code;
   }

   private boolean playerSetDefaultSubtitlesLanguage(String code)
   {
      boolean success = false;
      JSONArray args = new JSONArray();
      args.put(code);
      try
      {
         success =
            DtvkitGlueClient.getInstance().request("Player.setDefaultSubtitlesLanguage", args)
               .getBoolean("data");
      }
      catch (Exception e)
      {
         Log.e(TAG, e.getMessage());
      }

      return success;
   }

   private void playerStartDecoding()
   {
      JSONArray args = new JSONArray();
      try
      {
         DtvkitGlueClient.getInstance().request("Player.startDecoding", args);
      }
      catch (Exception e)
      {
         Log.e(TAG, e.getMessage());
      }
   }

   private void playerStopDecoding()
   {
      JSONArray args = new JSONArray();
      try
      {
         DtvkitGlueClient.getInstance().request("Player.stopDecoding", args);
      }
      catch (Exception e)
      {
         Log.e(TAG, e.getMessage());
      }
   }

   private long playerGetVideoWindowHandle()
   {
      long handle = 0;
      try
      {
         JSONArray args = new JSONArray();
         handle = DtvkitGlueClient.getInstance().request("Player.getVideoWindowHandle", args)
            .getLong("data");
      }
      catch (Exception e)
      {
         Log.e(TAG, e.getMessage());
      }
      return handle;
   }
}
